/*
 * ZForce touch screen driver sample code for the I2C bus.
 *
 * Copyright (c) 2011 Neonode Technologies AB
 * Copyright (c) 2012 Robert Bosch Car Multimedia GmbH
 *
 * This source code is provided as a sample only and has not gone
 * through full internal testing. This sample code is provided "as is",
 * with no  warranties for the functionality of the code, nor
 * any warranties concerning side effects when using the code. The
 * intention of this code is reference example for internal evaluation only.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the term of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/uaccess.h>
#include <linux/i2c.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <linux/completion.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/jiffies.h>
#include "zforce_hwcheck.h"

#include <linux/input/mt.h>

#ifdef DEBUG
#define debug_printk(f, arg...) \
	pr_debug("%s:%d: " f "\n", __func__, __LINE__, ## arg)
#else
#define debug_printk(f, arg...)
#endif

#define DEBUG_ZF_DUMP_I2C

/* The command id's used by the ZForce */

#define ZF_DEACTIVATE 0x00
#define ZF_ACTIVATE 0x01
#define ZF_SET_RESOLUTION 0x02
#define ZF_CONFIGURE 0x03
#define ZF_REQ_COORDINATES 0x04
#define ZF_REQ_LOW_SIGNALS 0x0D
#define ZF_REQ_LED_LEVELS 0x1C
#define ZF_REQ_ACTIVE_LEDS 0x1D
#define ZF_REQ_OPENSHORT 0x21
#define ZF_REQ_PRODUCT_ID 0x29
#define ZF_REQ_BOOTMODE 0x2B
#define ZF_REQ_STATUS 0x1E
#define ZF_SET_SCANNIG_FREQ 0x08
#define ZF_SET_TOUCH_SIZE_LIMIT 0x09
#define ZF_BOOT_COMPLETE 0x07

/* zForce notifications */
#define ZF_NOTIFY_LOW_SIGNAL 0x30

/*
 * Bit flags for the startup state machinery. Response flags are set at
 * response from the ZForce, but ignored by the program logic
 */

#define ZF_STATUS_REQUESTED 0x0001
#define ZF_STATUS_RESPONDED 0x0002
#define ZF_DEACTIVATE_REQUESTED 0x0004
#define ZF_DEACTIVATE_RESPONDED 0x0008
#define ZF_ACTIVATE_REQUESTED 0x0010
#define ZF_ACTIVATE_RESPONDED 0x0020
#define ZF_RESOLUTION_REQUESTED 0x0040
#define ZF_RESOLUTION_RESPONDED 0x0080
#define ZF_CONFIGURE_REQUESTED 0x0100
#define ZF_CONFIGURE_RESPONDED 0x0200
#define ZF_FIRSTTOUCH_REQUESTED 0x0400
#define ZF_FIRSTTOUCH_RESPONDED 0x0800
#define ZF_SCANNINGFREQ_REQUESTED 0x1000
#define ZF_SCANNINGFREQ_RESPONDED 0x2000
#define ZF_PRODUCTID_REQUESTED 0x4000
#define ZF_PRODUCTID_RESPONDED 0x8000
#define ZF_TOUCHSIZELIMIT_REQUESTED 0x10000
#define ZF_TOUCHSIZELIMIT_RESPONDED 0x20000
#define ZF_STATUS_BOOT_COMPLETE	0x80000000

#define ZF_MAX_X 0xfff
#define ZF_MAX_Y 0xfff

#define ZF_MAXSCAN_FREQ 255
#define ZF_IDLESCAN_FREQ 50 /* Hz */
#define ZF_FULLSCAN_FREQ ZF_MAXSCAN_FREQ

#define ZF_MAX_TOUCH_SIZE_LIMIT 0xFF

#define ZF_FRAME_START 0xEE

#define ZF_MAX_RX_LEN	255	/* Considering Status response */
#define ZF_MAX_CMD_LEN	20
#define ZF_MAX_SUPPORT_POINTS 2

#define ZF_PROTOCOL_V30		0x30
#define ZF_PROTOCOL_V50		0x50
#define ZF_PROTOCOL_MASK	0xFF

#define ZF_FINGER_PRESS		0
#define ZF_FINGER_MOVE		1
#define ZF_FINGER_RELEASE	2

enum zf_touch_mode {
	ZF_SINGLE_TOUCH_MODE = 0,
	ZF_DUAL_TOUCH_MODE = 1,
};

enum {
	ZF_CMD_DEACTIVATE = 0x00,
	ZF_CMD_ACTIVATE = 0x01,
	ZF_CMD_SET_RESOLUTION = 0x02,
	ZF_CMD_CONFIGURE = 0x03,
	ZF_CMD_REQ_COORDINATES = 0x04,
	ZF_CMD_SET_SCANNIG_FREQ  = 0x08,
	ZF_CMD_SET_TOUCH_SIZE_LIMIT = 0x09,
	ZF_CMD_REQ_LOW_SIGNALS = 0x0D,
	ZF_CMD_REQ_LED_LEVELS = 0x1C,
	ZF_CMD_SET_ACTIVE_LEDS = 0x1D,
	ZF_CMD_REQ_STATUS  = 0x1E,
	ZF_CMD_OPENSHORT = 0x21,
	ZF_CMD_REQ_PRODUCT_ID = 0x29,
	ZF_CMD_REQ_BOOTMODE = 0x2B,
	ZF_CMD_MAX,
};

struct zf_cmd_node {
	unsigned char *head;
	unsigned int size;
	unsigned char data[ZF_MAX_CMD_LEN];
};

union zforce_calibration {
	struct data {
		int corr_xx;
		int corr_xy;
		int offset_x;
		int corr_yx;
		int corr_yy;
		int offset_y;
		int divider;
	} data;
	int array[sizeof(struct data)/sizeof(int)];
};

/* default calibration data*/
/*{12330, -8, 1104680, -147, 7197, 1312672, 65536}*/
/* defalut calibration data which will not change raw data */
static union zforce_calibration  cal = {
	.data = {
		.corr_xx =  1,
		.corr_xy =  0,
		.offset_x = 0,
		.corr_yx =  0,
		.corr_yy =  1,
		.offset_y = 0,
		.divider =  1,
	},
};

struct zforce_finger {
	int status;
	int id;
	int x;
	int y;
	int area;		/* Touch area */
	int pressure;	/* For v5 protocol */
};

/* Per- ZForce-device data */
struct zforce_ts_priv {
	/* I2C client to communicate with the ZF chip */
	struct i2c_client *client;
	/* The registered input device to which events are reported */
	struct input_dev *input;
	/* IRQ on which ZF signals data available */
	int irq;
	/* hardware reset pin */
	int reset_pin;
	enum of_gpio_flags reset_active;
	/* Bit flags according to ZF_xx_REQUESTED/RESPONDED definitions */
	int startup_state;
	/* protocol version */
	int protocol_version;
	enum zf_touch_mode mode;
	struct zforce_finger finger[ZF_MAX_SUPPORT_POINTS];
	int scan_freq;
	int low_signal;
	unsigned long time_driver_start;
	unsigned long time_of_last_touch;
	int touched[ZF_MAX_SUPPORT_POINTS];
	int min_touch_size;
	int max_touch_size;
};

/* switch on/off do calibrating in kernel */
static int calibrate_state;
/* Config touch with module param "zf_mode" */

/* boot loader active flag */
int zforce_bootloader = 0;

static	int (*ack_root_irq)(void);

module_param_named(corr_xx, cal.data.corr_xx, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(corr_xx, "Import corr_xx");

module_param_named(corr_xy, cal.data.corr_xy, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(corr_xy, "Import corr_xy");

module_param_named(offset_x, cal.data.offset_x, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(offset_x, "Import offset_x");

module_param_named(corr_yx, cal.data.corr_yx, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(corr_yx, "Import corr_yx");

module_param_named(corr_yy, cal.data.corr_yy, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(corr_yy, "Import corr_yy");

module_param_named(offset_y, cal.data.offset_y, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(offset_y, "Import offset_y");

module_param_named(divider, cal.data.divider, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(divider, "Import divider");

ssize_t zf_calibration_state_show(struct device *dev,
struct device_attribute *attr, char *buf)
{
	return scnprintf(buf, PAGE_SIZE, "%s\n",
	calibrate_state ? "enabled" : "disabled");
}

ssize_t zf_calibration_state_store(struct device *dev,
struct device_attribute *attr, const char *buf, size_t count)
{
	if (buf[0] == 'e')
		calibrate_state = 1;
	else
		calibrate_state = 0;
	return count;
}

ssize_t zf_calibration_data_show(struct device *dev,
struct device_attribute *attr, char *buf)
{
	int i, len;

	for (i = 0, len = 0; i < ARRAY_SIZE(cal.array); ++i)
		len += scnprintf(buf+len, PAGE_SIZE, "%d ", cal.array[i]);

	len += scnprintf(buf+len, PAGE_SIZE, "\n");
	return len;
}

ssize_t zf_calibration_data_store(struct device *dev,
struct device_attribute *attr, const char *buf, size_t count)
{
	int i;
	char *token;
	char *cur = (char *)buf;
	for (i = 0, token = strsep(&cur, " ");
	(token != NULL) && (i < sizeof(cal.array));
	i++, token = strsep(&cur, " ")) {
		sscanf(token, "%d", &cal.array[i]);
	}
	return count;
}

DEVICE_ATTR(zf_calibration, S_IRUGO | S_IWUSR,
			zf_calibration_data_show, zf_calibration_data_store);

DEVICE_ATTR(zf_cal_state, S_IRUGO | S_IWUSR,
			zf_calibration_state_show, zf_calibration_state_store);

/* Workaround to define the maximum coordinate value
 * suitable for higher input layer */
static int zf_max_x = ZF_MAX_X;

ssize_t zf_max_x_show(struct device *dev,
struct device_attribute *attr, char *buf)
{
	return scnprintf(buf, PAGE_SIZE, "%d\n", zf_max_x);
}

ssize_t zf_max_x_store(struct device *dev,
struct device_attribute *attr, const char *buf, size_t count)
{
	long val;

	if (kstrtol(buf, 10, &val))
		return count;

	if (val > 0)
		zf_max_x = val;

	if (val > ZF_MAX_X)
		zf_max_x = ZF_MAX_X;

	return count;
}

DEVICE_ATTR(zf_max_x, S_IRUGO | S_IWUSR, zf_max_x_show, zf_max_x_store);

static int zf_max_y = ZF_MAX_Y;

ssize_t zf_max_y_show(struct device *dev,
struct device_attribute *attr, char *buf)
{
	return scnprintf(buf, PAGE_SIZE, "%d\n", zf_max_y);
}

ssize_t zf_max_y_store(struct device *dev,
struct device_attribute *attr, const char *buf, size_t count)
{
	long val;

	if (kstrtol(buf, 10, &val))
		return count;

	if (val > 0)
		zf_max_y = val;

	if (val > ZF_MAX_Y)
		zf_max_y = ZF_MAX_Y;

	return count;
}

DEVICE_ATTR(zf_max_y, S_IRUGO | S_IWUSR, zf_max_y_show, zf_max_y_store);

#ifdef DEBUG_ZF_DUMP_I2C
static int zf_dump;

ssize_t zf_dump_show(struct device *dev,
struct device_attribute *attr, char *buf)
{
	return scnprintf(buf, PAGE_SIZE, "%d\n", zf_dump);
}

ssize_t zf_dump_store(struct device *dev,
struct device_attribute *attr, const char *buf, size_t count)
{
	long val;

	if (kstrtol(buf, 10, &val))
		return count;

	zf_dump = val;

	return count;
}

DEVICE_ATTR(zf_dump, S_IRUGO | S_IWUSR, zf_dump_show, zf_dump_store);

static void zf_dump_protocol(const char *title,
	const unsigned char *head, const unsigned char *buf, int count)
{
	#define DUMP_TITLE_LEN 16
	char dump_string[(ZF_MAX_RX_LEN*3)+DUMP_TITLE_LEN+1];

	if (buf) {
		int i, max_len = ZF_MAX_RX_LEN;
		char *ptr = dump_string;

		ptr += snprintf(ptr, DUMP_TITLE_LEN-2, "%s: ", title);

		if (head) {
			max_len -= 2;
			for (i = 0; i < 2; ++i)
				ptr += snprintf(ptr, 4, "%02X ",
							(unsigned char)head[i]);
		}

		if (count > max_len)
			count = max_len;

		for (i = 0; i < count; ++i)
			ptr += snprintf(ptr, 4, "%02X ", (unsigned char)buf[i]);

		pr_info("%s\n", dump_string);
	}
}
#endif /* DEBUG_ZF_DUMP_I2C */

static int zf_i2c_send(const struct i2c_client *client,
		const char *buf, int count)
{
#ifdef DEBUG_ZF_DUMP_I2C
	if (zf_dump)
		zf_dump_protocol("to zforce", NULL, buf, count);
#endif
	return i2c_master_send(client, buf, count);
}

static ssize_t zf_scan_freq_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct zforce_ts_priv *priv = dev_get_drvdata(&client->dev);

	return scnprintf(buf, PAGE_SIZE, "%d\n", priv->scan_freq);
}

static int zforce_ts_send_scanfreq_req(struct zforce_ts_priv *priv);

static ssize_t zf_scan_freq_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct zforce_ts_priv *priv = dev_get_drvdata(&client->dev);
	long val;

	if (kstrtol(buf, 10, &val))
		return count;

	if ((val > 0) && (val <= ZF_MAXSCAN_FREQ)) {
		priv->scan_freq = val;
		zforce_ts_send_scanfreq_req(priv);
	} else {
		dev_err(&priv->client->dev,
			"Scanning frequency %d Hz invalid ( 0 < f <= %d)\n",
			(int)val, ZF_MAXSCAN_FREQ);
	}

	return count;
}

DEVICE_ATTR(scan_freq, S_IRUGO | S_IWUSR, zf_scan_freq_show,
		zf_scan_freq_store);

static ssize_t zf_max_touch_size_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct zforce_ts_priv *priv = dev_get_drvdata(&client->dev);

	return scnprintf(buf, PAGE_SIZE, "%d\n", priv->max_touch_size);
}

static int zforce_ts_send_touch_size_limit_req(struct zforce_ts_priv *priv);

static ssize_t zf_max_touch_size_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct zforce_ts_priv *priv = dev_get_drvdata(&client->dev);
	long val;

	if (kstrtol(buf, 10, &val))
		return count;

	if ((val > 0) && (val <= ZF_MAX_TOUCH_SIZE_LIMIT)) {
		priv->max_touch_size = val;
		zforce_ts_send_touch_size_limit_req(priv);
	} else {
		dev_err(&priv->client->dev,
			"Maximum touch size limit %dmm invalid (0 < max <= %d)\n",
			(int)val, ZF_MAX_TOUCH_SIZE_LIMIT);
	}

	return count;
}

DEVICE_ATTR(max_touch_size, S_IRUGO | S_IWUSR, zf_max_touch_size_show,
		zf_max_touch_size_store);

static ssize_t zf_min_touch_size_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct zforce_ts_priv *priv = dev_get_drvdata(&client->dev);

	return scnprintf(buf, PAGE_SIZE, "%d\n", priv->min_touch_size);
}

static ssize_t zf_min_touch_size_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct zforce_ts_priv *priv = dev_get_drvdata(&client->dev);
	long val;

	if (kstrtol(buf, 10, &val))
		return count;

	if ((val > 0) && (val <= ZF_MAX_TOUCH_SIZE_LIMIT)) {
		priv->min_touch_size = val;
		zforce_ts_send_touch_size_limit_req(priv);
	} else {
		dev_err(&priv->client->dev,
			"Minimum touch size limit %dmm invalid (0 < min <= %d)\n",
			(int)val, ZF_MAX_TOUCH_SIZE_LIMIT);
	}

	return count;
}

DEVICE_ATTR(min_touch_size, S_IRUGO | S_IWUSR, zf_min_touch_size_show,
		zf_min_touch_size_store);


static int add_sysfs_files(struct device *dev)
{
	int ret;
	ret = device_create_file(dev, &dev_attr_zf_calibration);
	if (ret)
		return ret;
	ret = device_create_file(dev, &dev_attr_zf_cal_state);
	if (ret)
		return ret;
	ret = device_create_file(dev, &dev_attr_zf_max_x);
	if (ret)
		return ret;
	ret = device_create_file(dev, &dev_attr_zf_max_y);
	if (ret)
		return ret;
	ret = device_create_file(dev, &dev_attr_scan_freq);
	if (ret)
		return ret;
	ret = device_create_file(dev, &dev_attr_max_touch_size);
	if (ret)
		return ret;
	ret = device_create_file(dev, &dev_attr_min_touch_size);
#ifdef DEBUG_ZF_DUMP_I2C
	if (ret)
		return ret;
	ret = device_create_file(dev, &dev_attr_zf_dump);
#endif
	return ret;
}

static void remove_sysfs_files(struct device *dev)
{
	device_remove_file(dev, &dev_attr_zf_calibration);
	device_remove_file(dev, &dev_attr_zf_cal_state);
	device_remove_file(dev, &dev_attr_zf_max_x);
	device_remove_file(dev, &dev_attr_zf_max_y);
	device_remove_file(dev, &dev_attr_scan_freq);
	device_remove_file(dev, &dev_attr_max_touch_size);
	device_remove_file(dev, &dev_attr_min_touch_size);
#ifdef DEBUG_ZF_DUMP_I2C
	device_remove_file(dev, &dev_attr_zf_dump);
#endif
}

static struct zf_cmd_node zf_cmd_table[ZF_CMD_MAX] = {
	[ZF_CMD_DEACTIVATE] = {
		.head = zf_cmd_table[ZF_CMD_DEACTIVATE].data,
		.size = 3,
		.data = {ZF_FRAME_START, 1, ZF_DEACTIVATE},
		},
	[ZF_CMD_ACTIVATE] = {
		.head = zf_cmd_table[ZF_CMD_ACTIVATE].data,
		.size = 3,
		.data = {ZF_FRAME_START, 1, ZF_ACTIVATE},
		},
	[ZF_CMD_SET_RESOLUTION] = {
		.head = zf_cmd_table[ZF_CMD_SET_RESOLUTION].data,
		.size = 7,
		.data = {ZF_FRAME_START, 5, ZF_SET_RESOLUTION, 0, 0, 0, 0},
		},
	[ZF_CMD_CONFIGURE] = {
		.head = zf_cmd_table[ZF_CMD_CONFIGURE].data,
		.size = 7,
		.data = {ZF_FRAME_START, 5, ZF_CONFIGURE, 0, 0, 0, 0},
		},
	[ZF_CMD_REQ_COORDINATES] = {
		.head = zf_cmd_table[ZF_CMD_REQ_COORDINATES].data,
		.size = 3,
		.data = {ZF_FRAME_START, 1, ZF_REQ_COORDINATES},
		},
	[ZF_CMD_SET_SCANNIG_FREQ] = {
		.head = zf_cmd_table[ZF_CMD_SET_SCANNIG_FREQ].data,
		.size = 9,
		.data = {ZF_FRAME_START, 7, ZF_SET_SCANNIG_FREQ,
			ZF_IDLESCAN_FREQ, 0, ZF_FULLSCAN_FREQ, 0, 0xFF, 0},
		},
	[ZF_CMD_SET_TOUCH_SIZE_LIMIT] = {
			.head = zf_cmd_table[ZF_CMD_SET_TOUCH_SIZE_LIMIT].data,
			.size = 7,
			.data = {ZF_FRAME_START, 5, ZF_SET_TOUCH_SIZE_LIMIT, 0, 0, 0, 0},
			},
	[ZF_CMD_REQ_LOW_SIGNALS] = {
		.head = zf_cmd_table[ZF_CMD_REQ_LOW_SIGNALS].data,
		.size = 4,
		.data = {ZF_FRAME_START, 2, ZF_REQ_LOW_SIGNALS, 0},
		},
	[ZF_CMD_REQ_LED_LEVELS] = {
		.head = zf_cmd_table[ZF_CMD_REQ_LED_LEVELS].data,
		.size = 4,
		.data = {ZF_FRAME_START, 2, ZF_REQ_LED_LEVELS, 0},
		},
	[ZF_CMD_OPENSHORT] = {
		.head = zf_cmd_table[ZF_CMD_OPENSHORT].data,
		.size = 4,
		.data = {ZF_FRAME_START, 2, ZF_REQ_OPENSHORT, 0},
		},
	[ZF_CMD_REQ_STATUS] = {
		.head = zf_cmd_table[ZF_CMD_REQ_STATUS].data,
		.size = 3,
		.data = {ZF_FRAME_START, 1, ZF_REQ_STATUS},
		},
	[ZF_CMD_REQ_PRODUCT_ID] = {
			.head = zf_cmd_table[ZF_CMD_REQ_PRODUCT_ID].data,
			.size = 3,
			.data = {ZF_FRAME_START, 1, ZF_REQ_PRODUCT_ID},
			},
	[ZF_CMD_REQ_BOOTMODE] = {
			.head = zf_cmd_table[ZF_CMD_REQ_BOOTMODE].data,
			.size = 3,
			.data = {ZF_FRAME_START, 1, ZF_REQ_BOOTMODE},
			},
};

static void zf_cmd_table_v3_fixup(void)
{
	int i;
	struct zf_cmd_node *pcmd = zf_cmd_table;
	unsigned char scan_freq_data[] = { ZF_SET_SCANNIG_FREQ,
						ZF_IDLESCAN_FREQ,
						ZF_FULLSCAN_FREQ, 0 };

	for (i = 0; i < ZF_CMD_MAX; i++, pcmd++) {
		if (pcmd->size == 0)
			continue;
		if (i == ZF_CMD_SET_SCANNIG_FREQ) {
			pcmd->head += 2;
			pcmd->size = sizeof(scan_freq_data);
			memcpy(pcmd->head, scan_freq_data, pcmd->size);
		} else {
			pcmd->head += 2;
			pcmd->size -= 2;
		}
	}
}

static void zf_cmd_table_init(int version)
{
	switch (version) {
	case 0x30:
		zf_cmd_table_v3_fixup();
		break;
	case 0x50:
	default:
		break;
	}
}

static void zforce_ts_handle_deactivate(u_int8_t *data, int len)
{
	debug_printk("zforce deactivation response, status=%d", data[0]);
}

static void zforce_ts_handle_activate(u_int8_t *data, int len)
{
	debug_printk("zforce activation response, status=%d", data[0]);
}

static void zforce_ts_handle_configure(u_int8_t *data, int len)
{
	debug_printk("zforce configuration response, status=%d", data[0]);
}

static void zforce_ts_handle_setres(u_int8_t *data, int len)
{
	debug_printk("zforce resolution setting response, status=%d", data[0]);
}

static void zforce_ts_handle_scanfreq(u_int8_t *data, int len)
{
	debug_printk("zforce scanning frequency response, status=%d", data[0]);
}

static void zforce_ts_handle_touch_size_limit(u_int8_t *data, int len)
{
	debug_printk("zforce touch size limit response, status=%d", data[0]);
}

static void zforce_ts_handle_low_signals(u_int8_t *data, int len)
{
	debug_printk("zforce low signals response, status=%d", data[0]);
	zf_hwcheck_handle_response(data, len);
}

static void zforce_ts_handle_low_signal_notification(
		struct zforce_ts_priv *priv, u_int8_t *data, int len)
{
	priv->low_signal = data[0];
	debug_printk("zforce low signal notification, status=%d", data[0]);
}

static void zforce_ts_handle_led_levels(u_int8_t *data,
		int len)
{
	debug_printk("zforce get LED level response, status=%d", data[0]);
	zf_hwcheck_handle_response(data, len);
}

static void zforce_ts_handle_openshort(u_int8_t *data,
		int len)
{
	debug_printk("zforce get open/short response, status=%d", data[0]);
	zf_hwcheck_handle_response(data, len);
}

static void zforce_ts_handle_bootmode(u_int8_t *data, int len)
{
	debug_printk("zforce get bootmode response, status=%d", data[0]);
	zf_hwcheck_handle_response(data, len);
	zforce_bootloader = 1;
}

static int zforce_ts_send_deactivate(struct zforce_ts_priv *priv)
{

	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_DEACTIVATE;

	if (pcmd->size == 0)
		return -EINVAL;

	debug_printk("Deactivating zforce");

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}


static int zforce_ts_set_reset(struct zforce_ts_priv *priv, int reset)
{

	debug_printk("Set reset: %d", reset);

	if (gpio_is_valid(priv->reset_pin)){
		if (reset) {
			/* reset active */
			gpio_set_value(priv->reset_pin,
				(priv->reset_active == OF_GPIO_ACTIVE_LOW) ?
				0 : 1);
		} else {
			/* reset inactive */
			gpio_set_value(priv->reset_pin,
				(priv->reset_active == OF_GPIO_ACTIVE_LOW) ?
				1 : 0);
		}
		return 0;
	}
	return -EPERM;
}

static int zforce_ts_send_activate(struct zforce_ts_priv *priv)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_ACTIVATE;

	if (pcmd->size == 0)
		return -EINVAL;

	debug_printk("Activating zforce");

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}

static int zforce_ts_send_configure(struct zforce_ts_priv *priv)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_CONFIGURE;

	if (pcmd->size == 0)
		return -EINVAL;

	debug_printk("Configuring zforce, using %s touch",
			(priv->mode) ? "dual" : "single");

	pcmd->data[3] = (priv->mode) ? 0x01 : 0x00;

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}

static int zforce_ts_send_scanfreq_req(struct zforce_ts_priv *priv)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_SET_SCANNIG_FREQ;

	if (pcmd->size == 0)
		return -EINVAL;

	pr_info("Setting zforce scanning frequency to %d Hz\n",
		priv->scan_freq);

	pcmd->data[5] = priv->scan_freq & 0x00FF;
	pcmd->data[6] = (priv->scan_freq >> 8) & 0x00FF;

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}

static int zforce_ts_send_touch_size_limit_req(struct zforce_ts_priv *priv)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_SET_TOUCH_SIZE_LIMIT;

	if (pcmd->size == 0)
		return -EINVAL;

	if (priv->max_touch_size) {
		pr_info("Setting zforce touch size limit max. %dmm\n",
			priv->max_touch_size);
		pcmd->data[3] = 1;
		pcmd->data[4] = priv->max_touch_size;
	}

	if (priv->min_touch_size) {
		pr_info("Setting zforce touch size limit min. %dmm\n",
			priv->min_touch_size);
		pcmd->data[5] = 1;
		pcmd->data[6] = priv->min_touch_size;
	}

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}

static int zforce_ts_send_setres(struct zforce_ts_priv *priv, int width,
		int height)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_SET_RESOLUTION;

	if (pcmd->size == 0)
		return -EINVAL;

	debug_printk("Setting zforce resolution, width=%d, height=%d",
		width, height);

	pcmd->data[3] = width & 0xff;
	pcmd->data[4] = width >> 8;
	pcmd->data[5] = height & 0xff;
	pcmd->data[6] = height >> 8;

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}

static int zforce_ts_send_touchdata_req(struct zforce_ts_priv *priv)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_REQ_COORDINATES;

	if (pcmd->size == 0)
		return -EINVAL;

	if (!(priv->startup_state & ZF_FIRSTTOUCH_RESPONDED))
		debug_printk("Requesting touch coordinates");

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}

static int zforce_ts_send_status_req(struct zforce_ts_priv *priv)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_REQ_STATUS;

	if (pcmd->size == 0)
		return -EINVAL;

	debug_printk("Getting zforce status");

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}

static int zforce_ts_send_product_id_req(struct zforce_ts_priv *priv)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_REQ_PRODUCT_ID;

	if (pcmd->size == 0)
		return -EINVAL;

	debug_printk("Getting zforce product id");

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}

static int zforce_ts_send_bootmode_req(struct zforce_ts_priv *priv)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_REQ_BOOTMODE;
	int ret;

	debug_printk("Request bootmode to start firmware download");

	if (pcmd->size == 0)
		return -EINVAL;

	debug_printk("Request bootmode to start firmware download");

	ret = zf_i2c_send(priv->client, pcmd->head, pcmd->size);
	debug_printk("Request bootmode to start firmware download ret=%d", ret);
	return ret;
}

/* Timeout if touch active; send low signal request anyway */
#define TOUCH_ACTIVE_LOW_SIGNAL (21*HZ) /* 21s */
/* Timeout if touch inactive; wait after last touch event before sending low
 * signal request */
#define TOUCH_INACTIVE_LOW_SIGNAL (3*HZ) /* 3s */

static int zforce_ts_send_low_signals_req(struct zforce_ts_priv *priv, int axis)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_REQ_LOW_SIGNALS;
	int count;

	if (pcmd->size == 0)
		return -EINVAL;

	/* Low signal request calibrates the touch panel and disturbs active
	 * touch events. Send only if low signal notification was received before
	 * or no touch event occurred till now
	 */
	if (!priv->low_signal && priv->time_of_last_touch) {
		/* Sometimes an incomplete inserted SD card do not trigger a low
		 * signal notification and cause a permanent touch event. This timeout
		 * will send low signal request anyway. The timeout must be longer than
		 * the maximum long press time (>20s) */
		if (!time_after(jiffies,
			priv->time_of_last_touch + TOUCH_ACTIVE_LOW_SIGNAL)) {
			/* Allow low signal request if no touch event active */
			for (count = 0; count < ZF_MAX_SUPPORT_POINTS; count++) {
				if (priv->touched[count])
					return -EAGAIN; /* try again */
			}
			/* Wait 3 s after last release event */
			if (!time_after(jiffies,
				priv->time_of_last_touch + TOUCH_INACTIVE_LOW_SIGNAL))
				return -EAGAIN; /* try again */
		}
	}

	debug_printk("Getting zforce low signals %s", axis ? "Y" : "X");

	pcmd->head[3] = axis ? 1 : 0;

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}

static int zforce_ts_send_led_levels_req(struct zforce_ts_priv *priv, int axis)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_REQ_LED_LEVELS;

	if (pcmd->size == 0)
		return -EINVAL;
	debug_printk("Getting zforce LED levels %s", axis ? "Y" : "X");

	pcmd->head[3] = axis ? 1 : 0;

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}

static int zforce_ts_send_openshort(struct zforce_ts_priv *priv, int axis)
{
	struct zf_cmd_node *pcmd = zf_cmd_table + ZF_CMD_OPENSHORT;

	if (pcmd->size == 0)
		return -EINVAL;

	debug_printk("OPENSHORT TEST on %s", axis ? "Y" : "X");

	pcmd->head[3] = axis ? 1 : 0;

	return zf_i2c_send(priv->client, pcmd->head, pcmd->size);
}

int zforce_ts_hwcheck_send_low_signals_req(void *zforce_dev, int axis)
{
	return zforce_ts_send_low_signals_req(zforce_dev, axis);
}

int zforce_ts_hwcheck_send_led_levels_req(void *zforce_dev, int axis)
{
	return zforce_ts_send_led_levels_req(zforce_dev, axis);
}

int zforce_ts_hwcheck_send_send_openshort_req(void *zforce_dev, int axis)
{
	return zforce_ts_send_openshort(zforce_dev, axis);
}

int zforce_ts_hwcheck_send_status_req(void *zforce_dev)
{
	return zforce_ts_send_status_req(zforce_dev);
}

int zforce_ts_hwcheck_send_bootmode_req(void *zforce_dev)
{
	return zforce_ts_send_bootmode_req(zforce_dev);
}

int zforce_ts_hwcheck_send_bootloader_req(void *zforce_dev,
		u_int8_t *data, int len)
{
	struct zforce_ts_priv *priv = zforce_dev;
	return zf_i2c_send(priv->client, data, len);
}

int zforce_ts_hwcheck_set_reset(void *zforce_dev, int reset)
{
	return zforce_ts_set_reset(zforce_dev, reset);
}


static void zforce_ts_handle_status(struct zforce_ts_priv *priv,
		u_int8_t *data, int len)
{

	/* Version */
	u_int16_t major = data[0] + (data[1] << 8);
	u_int16_t minor = data[2] + (data[3] << 8);
	u_int16_t build = data[4] + (data[5] << 8);
	u_int16_t revision = data[6] + (data[7] << 8);

	/* Full scan mode scanning frequency */
	u_int16_t full_freq = data[23] + (data[24] << 8);

	/* Soft Resolution */
	u_int16_t width = data[27] + (data[28] << 8);
	u_int16_t height = data[29] + (data[30] << 8);

#ifdef DEBUG
	/* Scanning status */
	u_int8_t scan_status = ((data[8] & 0x80) >> 7);

	/* Current detected contacts */
	u_int8_t curr_contacts = (data[8] & 0x7F);

	/* Scanning counter */
	u_int16_t scan_counter = data[9] + (data[10] << 8);

	/* Touch data prepared to send */
	u_int16_t prep_touch_pack = data[11] + (data[12] << 8);

	/* Already sent touch data */
	u_int16_t sent_touch_pack = data[13] + (data[14] << 8);

	/* Invalid touch data */
	u_int16_t invalid_touch_pack = data[15] + (data[16] << 8);

	/* Configuration data bit 0: Single? Dual touch */
	u_int16_t config_data1 = data[17] + (data[18] << 8);
	/* u_int16_t configData2 = data[20] + (data[21] << 8); */

	/* Idle mode scanning frequencey */
	u_int16_t idle_freq = data[21] + (data[22] << 8);

	/* Pen Frequency */
	u_int16_t pen_freq = data[25] + (data[26] << 8);

	/* Physical area of touch */
	u_int16_t phy_width = data[31] + (data[32] << 8);
	u_int16_t phy_height = data[33] + (data[34] << 8);

	/* Active LED range x axis */
	u_int8_t x_led_start = data[35];
	u_int8_t x_led_end = data[36];

	/* Active LED range y axis */
	u_int8_t y_led_start = data[37];
	u_int8_t y_led_end = data[38];

	/* Touch object dimension configurations */
	u_int8_t max_touch_en = data[40];
	u_int8_t max_touch_area = data[41];

	/* max Strength */
	u_int8_t max_strength = data[39];

	/* touch size limit */
	u_int8_t max_touch_size_limit_en = data[40];
	u_int8_t max_touch_size_limit = data[41];
	u_int8_t min_touch_size_limit_en = data[42];
	u_int8_t min_touch_size_limit = data[43];

	debug_printk("zforce: Scanning%sactivated",
		scan_status ? " " : " is not ");
	debug_printk("zforce: There are%scurrent contacts",
		curr_contacts ? " " : " no ");
	debug_printk("zforce: Scanning Counter = 0x%04X",
		scan_counter);
	debug_printk("zforce: Prepared touch packages = 0x%04X",
		prep_touch_pack);
	debug_printk("zforce: Sent touch packages = 0x%04X",
		sent_touch_pack);
	debug_printk("zforce: Invalid touch packages = 0x%04X",
		invalid_touch_pack);
	debug_printk("zforce: Configured for %s touch",
		(config_data1 & 0x0001) ? "multi" : "single");
	debug_printk("zforce: Idle Scan mode frequency= 0x%02X", idle_freq);
	debug_printk("zforce: Full Scan mode frequency= 0x%02X", full_freq);
	debug_printk("zforce: Pen frequency= 0x%02X", pen_freq);
	debug_printk("zforce: Resolution --> width %u, height %u",
		width, height);
	debug_printk("zforce: Physical Area --> width %umm, height %umm",
		phy_width, phy_height);
	debug_printk("zforce: Active LED range X-Axis Start: %d, End %d",
		x_led_start, x_led_end);
	debug_printk("zforce: Active LED range Y-Axis Start: %d, End %d",
		y_led_start, y_led_end);
	debug_printk("zforce: Touch object dimension configuration%senabled",
		max_touch_en ? " " : " is not ");
	if (max_touch_en)
		debug_printk("zforce: Touch object dimension is %dmm",
		max_touch_area);
	debug_printk("zforce: Maximum strength 0x%02X", max_strength);
	if (max_touch_size_limit_en)
		debug_printk("zforce: Maximum touch size limit: %dmm",
			max_touch_size_limit);
	if (min_touch_size_limit_en)
		debug_printk("zforce: Minimum touch size limit: %dmm",
				min_touch_size_limit);

#endif /* DEBUG*/

	/* Check resolution if already set */
	if ((priv->startup_state & ZF_RESOLUTION_RESPONDED) &&
		((width != ZF_MAX_X) || (height != ZF_MAX_Y))) {

		dev_err(&priv->client->dev,
			"zforce: Resolution mismatch (x/y:%d/%d): %d/%d",
			ZF_MAX_X, ZF_MAX_Y, width, height);

		/* restart initialization */
		priv->startup_state =
			ZF_STATUS_REQUESTED | ZF_STATUS_BOOT_COMPLETE;
		zforce_ts_send_status_req(priv);

		return;
	}

	/* Check scanning frequency if already set */
	if ((priv->startup_state & ZF_SCANNINGFREQ_RESPONDED) &&
		(full_freq != priv->scan_freq)) {

		dev_err(&priv->client->dev,
			"zforce: Scanning frequency mismatch (%d): %d",
			priv->scan_freq, full_freq);

		/* restart initialization */
		priv->startup_state =
			ZF_STATUS_REQUESTED | ZF_STATUS_BOOT_COMPLETE;
		zforce_ts_send_status_req(priv);

		return;
	}

	zf_hwcheck_handle_response(data, len);

	printk(KERN_INFO
		"zforce: Firmware version : %d.%d, build %d, rev %d\n",
		major, minor, build, revision);
}

static void tslib_liner_process(struct zforce_ts_priv *priv, int *x, int *y)
{
	int xtemp = *x;
	int ytemp = *y;
	if (calibrate_state) {
		*x = (cal.data.offset_x +
			cal.data.corr_xx * xtemp +
			cal.data.corr_xy * ytemp) / cal.data.divider;
		*y = (cal.data.offset_y +
			cal.data.corr_yx * xtemp +
			cal.data.corr_yy * ytemp) / cal.data.divider;
	}
	debug_printk("x: %d to %d, y: %d to %d", xtemp, *x,
	ytemp, *y);
}

static void zforce_check_coordinate(struct zforce_ts_priv *priv,
		struct zforce_finger *finger)
{
	/* check coordinate range */
	if ((finger->x < 0) || (finger->x > ZF_MAX_X) ||
		(finger->y < 0) || (finger->y > ZF_MAX_Y)) {

		dev_err(&priv->client->dev,
			"zforce: Touch coordinate mismatch (x/y:%d/%d): %d/%d",
			ZF_MAX_X, ZF_MAX_Y, finger->x, finger->y);

		/* restart initialization */
		priv->startup_state =
			ZF_STATUS_REQUESTED | ZF_STATUS_BOOT_COMPLETE;
		zforce_ts_send_status_req(priv);
	}
}

static void zforce_multi_report(struct zforce_ts_priv *priv)
{
	struct zforce_finger *finger = priv->finger;
	struct input_dev *input_dev = priv->input;
	int finger_num = 0;
	int count;

	for (count = 0; count < ZF_MAX_SUPPORT_POINTS; count++) {

		priv->touched[count] = 0;

		if (finger[count].status < 0)
			continue;

		zforce_check_coordinate(priv, &finger[count]);

		/* Use mutli-touch protocol type B */
		input_mt_slot(input_dev, finger[count].id);
		/* Finger press or move */
		if ((finger[count].status == ZF_FINGER_PRESS) ||
			(finger[count].status == ZF_FINGER_MOVE)) {
			finger_num++;

			input_mt_report_slot_state(input_dev,
					MT_TOOL_FINGER, 1);
			input_report_abs(input_dev, ABS_MT_POSITION_X,
					finger[count].x);
			input_report_abs(input_dev, ABS_MT_POSITION_Y,
					finger[count].y);
			input_report_abs(input_dev, ABS_MT_TOUCH_MAJOR,
					finger[count].area);
			/* Only availabe in Protocol V50 & later */
			if (finger[count].pressure >= 0)
				input_report_abs(input_dev, ABS_MT_PRESSURE,
					finger[count].pressure);

			priv->touched[count] = 1;

		} else { /* Finger release */
			input_mt_report_slot_state(input_dev,
					MT_TOOL_FINGER, 0);
		}

		/* invalid the processed finger data */
		finger[count].status = -1;
	}
	input_mt_sync_frame(input_dev);
	input_mt_report_finger_count(input_dev, finger_num);
	input_sync(input_dev);
}

/* For single touch, the single_id is 0 */
static void zforce_single_report(struct zforce_ts_priv *priv)
{
	/* Only get the priv->finger[0] */
	struct zforce_finger *finger = priv->finger;
	struct input_dev *input_dev = priv->input;

	if (finger->status < 0)
		return;

	zforce_check_coordinate(priv, finger);

	input_report_abs(input_dev, ABS_X, finger->x);
	input_report_abs(input_dev, ABS_Y, finger->y);

	if ((finger->status == ZF_FINGER_PRESS) ||
		(finger->status == ZF_FINGER_MOVE)) {
		input_report_key(input_dev, BTN_TOUCH, 1);
		priv->touched[0] = 1;
	} else {
		input_report_key(input_dev, BTN_TOUCH, 0);
		priv->touched[0] = 0;
	}

	finger->status = -1;
	input_sync(input_dev);
}

/* Handle touch coordinate events form the ZF */
static void zforce_ts_handle_touchdata(struct zforce_ts_priv *priv,
		u_int8_t *payload, int len)
{

	/* Be more verbose for the very first touch coordinate we get */
	int more_verbose =
		priv->startup_state & ZF_FIRSTTOUCH_RESPONDED ? 0 : 1;

	/* The number of touch coordinates reported by the ZForce */
	int coordinate_count = payload[0];

	/* Used to automatically determine protocol version */
	int protocol_variant = 0;

	/* Index within the coordinate sub-data blocks */

	int status;
	u_int8_t *ptsdata;
	int i;
	struct zforce_finger *finger = priv->finger;
	int area_x, area_y;

	/* This is a QaD trick to determine which of the two protocol flavors
	 * we got (one uses 5 bytes per coordinate, the other uses 7 bytes per
	 * coordinate). The "protocol_variant" is set to the coordinate block
	 * length
	 */

	if (len == coordinate_count * 9 + 1) {
		if (more_verbose)
			debug_printk("Using 9 bytes per coordinate protocol");

		/* Latest ZForce with 9 bytes per coordinate */
		protocol_variant = 9;
	} else if (len == coordinate_count * 5 + 1) {

		if (more_verbose)
			debug_printk("Using 5 bytes per coordinate protocol");

		/* Older ZForce with 5 bytes per coordinate */
		protocol_variant = 5;
	} else if (len == coordinate_count * 7 + 1) {

		if (more_verbose)
			debug_printk("Using 7 bytes per coordinate protocol");

		/* Newer ZForce with 7 bytes per coordinate */
		protocol_variant = 7;
	}

	/* Did we figure out the protocol ? */
	if (!protocol_variant) {

		/* Not the expected packet length */

		dev_err(&priv->client->dev,
			"Could not match zforce block"
			 "length to any protocol\n");

		return;
	}

	/* point to real touch data */
	ptsdata = payload + 1;
	for (i = 0; i < coordinate_count; i++) {
		/*  Only parse the x,y,status and area
		 *  is enough for multiTouch events.
		 */
		ptsdata = i*protocol_variant + ptsdata;
		finger[i].x = ptsdata[0] | (ptsdata[1] << 8);
		finger[i].y = ptsdata[2] | (ptsdata[3] << 8);
		tslib_liner_process(priv, &finger[i].x, &finger[i].y);
		status = ptsdata[4];

		switch (protocol_variant) {
		case 9:
			finger[i].status = status & 0xF;
			finger[i].id = ((status>>4) & 0xF)-1;
			area_x = ptsdata[5];
			area_y = ptsdata[6];
			finger[i].area = max(area_x, area_y);
			finger[i].pressure = ptsdata[7];
			break;
		case 7:
			finger[i].status = status >> 6;
			area_x =
			((ptsdata[4] & 3) << 3) | ((ptsdata[4+1] & 0xe0) >> 5);
			area_y =
			ptsdata[4+1] & 0x1f;
			finger[i].area = max(area_x, area_y);
			finger[i].pressure = -1;
			break;

		case 5:
		default:
			finger[i].status = status & 3;
			finger[i].area = -1;
			finger[i].pressure = -1;
			break;

		}
		if (more_verbose)
			debug_printk("Status=0x0%x x=%x y=%x",
				finger[i].status, finger[i].x, finger[i].y);
	}

	priv->time_of_last_touch = jiffies;

	/* report all events follow the touch mode */
	if (priv->mode == ZF_SINGLE_TOUCH_MODE)
		zforce_single_report(priv);
	else
		zforce_multi_report(priv);

	if (more_verbose)
		debug_printk("Syncing multi touch event");
}

static irqreturn_t zforce_ts_handle_data(int irq, void *dev_id)
{

	/*	The container struct of the work_struct passed to the schedule
	 *  call happens to be the private data associated with this driver,
	 *  use the container_of macro to acquire it
	 */
	struct zforce_ts_priv *priv = dev_id;

	/* Current command received */
	int zforce_command;

	/* Length of payload part of received block */
	int payload_length;

	/* Data buffer (large enough to hold biggest expected block) */
	u_int8_t tmp_buf[ZF_MAX_RX_LEN];
#ifdef DEBUG_ZF_DUMP_I2C
	u_int8_t header_buf[2];
#endif

	int more_verbose;

	int i;

	if (ack_root_irq && (!ack_root_irq()))
		return 0;

	memset(tmp_buf, 0, sizeof(tmp_buf));

	/* handle bootloader response; only one byte */
	if (zforce_bootloader){
		if (i2c_master_recv(priv->client, tmp_buf, 1) != 1) {
			dev_err(&priv->client->dev,
					"Unable to read zforce boot loader response\n");
			goto out;
		}
		zf_hwcheck_handle_response(tmp_buf, 1);
		goto out;
	}

	/* Be more verbose until one touch coordinate was handled */
	more_verbose = priv->startup_state & ZF_FIRSTTOUCH_RESPONDED ? 0 : 1;

	/* Read the first three bytes to get the command id and the size of the
	 * rest of the message
	 */
	if (i2c_master_recv(priv->client, tmp_buf, 2) != 2) {
		dev_err(&priv->client->dev,
				"Unable to read zforce data header\n");
		goto out;
	}

	/* Check the start byte */
	if (tmp_buf[0] != ZF_FRAME_START) {

		dev_err(&priv->client->dev,
				"Invalid initial byte of zforce data block\n");
		goto out;
	}

	/* Get the length of the payload */
	payload_length = tmp_buf[1]; /* From zForce communication
					interface */

	/* The block is too long to handle */
	if (payload_length > sizeof(tmp_buf)) {

		dev_err(&priv->client->dev,
				"Block from Zforce was too long\n");

		/* Read byte by byte to flush the buffer */
		for (i = 0; i < payload_length; i++)
			i2c_master_recv(priv->client, tmp_buf, 1);

		goto out;
	}

#ifdef DEBUG_ZF_DUMP_I2C
	/* save first two bytes for dump */
	if (zf_dump) {
		header_buf[0] = tmp_buf[0];
		header_buf[1] = tmp_buf[1];
	}
#endif

	/* Reuse the data buffer and read the payload part of the block */
	if (i2c_master_recv(priv->client, tmp_buf, payload_length)
			!= payload_length) {
		dev_err(&priv->client->dev,
			"Unable to get zforce data header\n");
		goto out;
	}

	/* Get the command ID from the payload */
	zforce_command = tmp_buf[0];

#ifdef DEBUG_ZF_DUMP_I2C
	if (zf_dump)
		zf_dump_protocol("from zforce", header_buf, tmp_buf,
				payload_length);
#endif

	if (more_verbose)
		debug_printk("Data from zforce, command=%d, length=%d",
			zforce_command, payload_length);

	/* subtract lentgh by 1 to eliminate the command ID byte */
	payload_length -= 1;

	switch (zforce_command) { /* Attend to the command */

	case ZF_DEACTIVATE: /* Got response from deactivate request */

		priv->startup_state |= ZF_DEACTIVATE_RESPONDED;
		/* Device need active again */
		priv->startup_state &= ~ZF_ACTIVATE_RESPONDED;
		zforce_ts_handle_deactivate(&tmp_buf[1], payload_length);

		break;

	case ZF_ACTIVATE: /* Got response from activate request */

		priv->startup_state |= ZF_ACTIVATE_RESPONDED;
		zforce_ts_handle_activate(&tmp_buf[1], payload_length);

		break;

	case ZF_CONFIGURE: /* Got response from configuration request */

		priv->startup_state |= ZF_CONFIGURE_RESPONDED;
		zforce_ts_handle_configure(&tmp_buf[1], payload_length);

		break;

	case ZF_SET_RESOLUTION: /* Got response from resolution
				setting request */

		priv->startup_state |= ZF_RESOLUTION_RESPONDED;
		zforce_ts_handle_setres(&tmp_buf[1], payload_length);

		break;

	case ZF_REQ_COORDINATES: /* Got touch event */

		zforce_ts_handle_touchdata(priv, &tmp_buf[1],
					   payload_length);
		priv->startup_state |= ZF_FIRSTTOUCH_RESPONDED;

		break;
	case ZF_REQ_LOW_SIGNALS: /* Got LOW_SIGNALS */
		zforce_ts_handle_low_signals(&tmp_buf[1], payload_length);
		break;

	case ZF_REQ_LED_LEVELS: /* Got LED_LEVELS */
		zforce_ts_handle_led_levels(&tmp_buf[1], payload_length);
		break;

	case ZF_REQ_OPENSHORT: /* Openshort test */
		zforce_ts_handle_openshort(&tmp_buf[1], payload_length);
		break;

	case ZF_REQ_BOOTMODE: /* Boot mode response */
		zforce_ts_handle_bootmode(&tmp_buf[1], payload_length);
		break;

	case ZF_REQ_STATUS: /* Got status */

		priv->startup_state |= ZF_STATUS_RESPONDED;
		zforce_ts_handle_status(priv, &tmp_buf[1], payload_length);

		break;

	case ZF_REQ_PRODUCT_ID: /* Got Product ID */
		priv->startup_state |= ZF_PRODUCTID_RESPONDED;

		if ((payload_length + 1) < sizeof(tmp_buf))
			tmp_buf[payload_length+1] = 0;
		else
			tmp_buf[sizeof(tmp_buf)-1] = 0;

		pr_info("zforce: Product ID : %s\n", &tmp_buf[1]);
		break;
	case ZF_SET_SCANNIG_FREQ: /* Got scanning frequency */
		priv->startup_state |= ZF_SCANNINGFREQ_RESPONDED;
		zforce_ts_handle_scanfreq(&tmp_buf[1], payload_length);

		break;
	case ZF_SET_TOUCH_SIZE_LIMIT: /* Got touch size limit response */
		priv->startup_state |= ZF_TOUCHSIZELIMIT_RESPONDED;
		zforce_ts_handle_touch_size_limit(&tmp_buf[1], payload_length);
		break;
	case ZF_NOTIFY_LOW_SIGNAL:
		zforce_ts_handle_low_signal_notification(priv,
			&tmp_buf[1], payload_length);
		break;
	case ZF_BOOT_COMPLETE:
		/* Only V50 and later has BOOT_COMPLETE.
		 * Whenever get the BOOT_COMPLETE, the startup_state
		 * need be clear to trigger the initialization(again).
		 */
		/* check if driver state is "initialized" */
		if (priv->startup_state) {
			/* trigger reinitialization */
			priv->startup_state =
				ZF_STATUS_REQUESTED | ZF_STATUS_BOOT_COMPLETE;
			zforce_ts_send_status_req(priv);
		} else
			priv->startup_state = ZF_STATUS_BOOT_COMPLETE;

		return IRQ_HANDLED;
		break;
	}

	/* All command must send after get boot complete event
	 * for firmware with protocol version higher than 5.0.
	 * Old firmware will be set boot complete flags during
	 * driver initialization
	 */
	if (!(priv->startup_state & ZF_STATUS_BOOT_COMPLETE))
		return IRQ_HANDLED;

	if (!(priv->startup_state & ZF_DEACTIVATE_REQUESTED)) {

		/*	Still not deactivated, send a deactivation request,
			this is necessary as the ZForce will potentially
			stop sending coordinates if an activation is
			issued without prior deactivation
		*/

		priv->startup_state |= ZF_DEACTIVATE_REQUESTED;
		zforce_ts_send_deactivate(priv);
	} else if (!(priv->startup_state & ZF_ACTIVATE_REQUESTED)) {

		/* Still not activated, send an activation request */

		priv->startup_state |= ZF_ACTIVATE_REQUESTED;
		zforce_ts_send_activate(priv);
	} else if (!(priv->startup_state & ZF_SCANNINGFREQ_REQUESTED)) {

		/* Set scanning frequency after activating */
		priv->startup_state |= ZF_SCANNINGFREQ_REQUESTED;
		zforce_ts_send_scanfreq_req(priv);

	} else if (!(priv->startup_state & ZF_TOUCHSIZELIMIT_REQUESTED)) {

		/* Set touch size limit after activating */
		priv->startup_state |= ZF_TOUCHSIZELIMIT_REQUESTED;
		zforce_ts_send_touch_size_limit_req(priv);

	} else if (!(priv->startup_state & ZF_RESOLUTION_REQUESTED)) {

		/* Send a resolution setting request */

		priv->startup_state |= ZF_RESOLUTION_REQUESTED;
		zforce_ts_send_setres(priv, ZF_MAX_X, ZF_MAX_Y);
	} else if (!(priv->startup_state & ZF_CONFIGURE_REQUESTED)) {

		/* Still not configured, send a configuration request */

		priv->startup_state |= ZF_CONFIGURE_REQUESTED;
		zforce_ts_send_configure(priv);

		/* Get the status after applying all settings to confirm */
		priv->startup_state &= ~ZF_STATUS_REQUESTED;

	} else if (!(priv->startup_state & ZF_STATUS_REQUESTED)) {

		/* Get the latest status from zForce */
		priv->startup_state |= ZF_STATUS_REQUESTED;
		zforce_ts_send_status_req(priv);

	} else if (!(priv->startup_state & ZF_PRODUCTID_REQUESTED)) {

		/* Get the Product ID from zForce */
		priv->startup_state |= ZF_PRODUCTID_REQUESTED;
		zforce_ts_send_product_id_req(priv);

	} else {

		/* All setup done, request some coordinates
		 * if not requested before */
		if (!(priv->startup_state & ZF_FIRSTTOUCH_REQUESTED)) {
			zforce_ts_send_touchdata_req(priv);
			priv->startup_state |= ZF_FIRSTTOUCH_REQUESTED;
		}
	}

out:

	/* Re-enable IRQ so that we can handle the next ZForce event */
	return IRQ_HANDLED;
}

#define ZF_BOOT_COMPLETE_TIMEOUT 1500 /* ms */

static int zforce_ts_open(struct input_dev *dev)
{
	struct zforce_ts_priv *priv = input_get_drvdata(dev);
	struct i2c_client *client = priv->client;

	/* clear the finger status to -1 */
	memset((char *)priv->finger, 0xFF,
	sizeof(struct zforce_finger)*ZF_MAX_SUPPORT_POINTS);

	/* check BOOT_COMPLETE timeout */
	if (!(priv->startup_state & ZF_STATUS_BOOT_COMPLETE) &&
		!time_after(jiffies,
				priv->time_driver_start +
				msecs_to_jiffies(ZF_BOOT_COMPLETE_TIMEOUT))) {
		debug_printk("await boot complete; try again later");
		return -EAGAIN;
	}

	/* Boot complete received or assumed by timeout
	 * reset status to enter new initialization sequence */
	priv->startup_state = ZF_STATUS_BOOT_COMPLETE;

	/* Fire off a status request to get the state machinery
	 * up and running, the rest is done by the ISR handling
	 * routine
	 */
	priv->startup_state |= ZF_STATUS_REQUESTED;
	if (zforce_ts_send_status_req(priv) < 0) {

		dev_err(&client->dev,
			"Unable to request status from zforce touchscreen.\n");
		return -ENXIO;
	}

	return 0;
}

static void zforce_ts_close(struct input_dev *dev)
{
	struct zforce_ts_priv *priv = input_get_drvdata(dev);

	disable_irq(priv->irq);

	zforce_ts_send_deactivate(priv);

	enable_irq(priv->irq);
}

static void zforce_ts_input_config(struct input_dev *input,
		enum zf_touch_mode mode)
{
	__set_bit(EV_ABS, input->evbit);

	if (mode == ZF_DUAL_TOUCH_MODE) {
		__set_bit(ABS_MT_SLOT, input->absbit);
		__set_bit(ABS_PRESSURE, input->absbit);

		input_set_abs_params(input, ABS_MT_POSITION_X,
						0, ZF_MAX_X, 0, 0);
		input_set_abs_params(input, ABS_MT_POSITION_Y,
						0, ZF_MAX_Y, 0, 0);
		input_set_abs_params(input, ABS_MT_TOUCH_MAJOR,
						0, 255, 0, 0);
		input_set_abs_params(input, ABS_MT_WIDTH_MAJOR,
						0, 255, 0, 0);
		input_set_abs_params(input, ABS_MT_PRESSURE,
						0, 255, 0, 0);
		input_mt_init_slots(input,
			ZF_MAX_SUPPORT_POINTS, 0);
	} else {
		/* for ZF_SINGLE_TOUCH */
		__set_bit(EV_KEY, input->evbit);
		__set_bit(BTN_TOUCH, input->keybit);
		__set_bit(ABS_X, input->absbit);
		__set_bit(ABS_Y, input->absbit);

		/* Setup the coordinate system
		 * (span from 0 to MAX in both axis)*/
		input_set_abs_params(input, ABS_X, 0, ZF_MAX_X, 0, 0);
		input_set_abs_params(input, ABS_Y, 0, ZF_MAX_Y, 0, 0);
	}
}

static int zforce_ts_probe(struct i2c_client *client,
		const struct i2c_device_id *idp)
{
	struct device_node *np = client->dev.of_node;
	const __be32 *val;
	const char *s;
	struct zforce_ts_priv *priv;
	struct input_dev *input;
	struct device *dev;
	int error, freq, max_touch_size, min_touch_size;

	priv = kzalloc(sizeof(*priv), GFP_KERNEL);
	if (!priv) {
		dev_err(&client->dev, "Failed to allocate driver data\n");
		error = -ENOMEM;
		goto err0;
	}

	val = of_get_property(np, "reversion", NULL);
	if (val)
		priv->protocol_version = be32_to_cpup(val) & ZF_PROTOCOL_MASK;
	else
		priv->protocol_version = ZF_PROTOCOL_V50;

	s = of_get_property(np, "ack_root_irq", NULL);
	if (s) {
		if (!strcmp(s, "lvds-serializer")) {
			extern int lvds_serializer_ack_irq(void);
			ack_root_irq = lvds_serializer_ack_irq;
		}
	}

	priv->reset_pin = of_get_named_gpio_flags(np, "reset-gpio", 0,
							&priv->reset_active);
	if ((priv->reset_pin != -EPROBE_DEFER) &&
		gpio_is_valid(priv->reset_pin)) {
		if (gpio_request_one(priv->reset_pin, GPIOF_DIR_OUT |
			((priv->reset_active == OF_GPIO_ACTIVE_LOW) ?
			GPIOF_INIT_HIGH : GPIOF_INIT_LOW), "zforce-reset") < 0)
			priv->reset_pin = -EINVAL;
	}

	/* store driver starting time for BOOT_COMPLETE timeout */
	priv->time_driver_start = jiffies;

	if (priv->protocol_version < ZF_PROTOCOL_V50)
		priv->startup_state |= ZF_STATUS_BOOT_COMPLETE;

	zf_cmd_table_init(priv->protocol_version);

	dev_set_drvdata(&client->dev, priv);

	input = input_allocate_device();
	if (!input) {
		dev_err(&client->dev, "Failed to allocate input device.\n");
		error = -ENOMEM;
		goto err1;
	}

	/* select dual or single touch mode */
	s = of_get_property(np, "touch-mode", NULL);

	if (s && (strncmp(s, "dual", sizeof("dual")) == 0)) {
		/* select dual touch mode */
		priv->mode = ZF_DUAL_TOUCH_MODE;
	} else {
		/* if not define mode in DT, use single touch default */
		priv->mode = ZF_SINGLE_TOUCH_MODE;
	}
	zforce_ts_input_config(input, priv->mode);

	/* set scan frequency */
	priv->scan_freq = ZF_FULLSCAN_FREQ;
	if (!of_property_read_u32(np, "scan-freq", &freq)) {
		if ((freq > 0) && (freq <= ZF_MAXSCAN_FREQ))
			priv->scan_freq = freq;
		else
			dev_err(&priv->client->dev,
				"Scanning frequency %d Hz invalid (0 < f <= %d Hz)\n",
				freq, ZF_MAXSCAN_FREQ);
	}

	/* set touch size limits */
	if (!of_property_read_u32(np, "min-touch-size", &min_touch_size)) {
		if ((min_touch_size > 0) &&
			(min_touch_size <= ZF_MAX_TOUCH_SIZE_LIMIT))
			priv->min_touch_size = min_touch_size;
		else
			dev_err(&priv->client->dev,
				"Minimum touch size limit %dmm invalid (0 < min <= %d)\n",
				min_touch_size, ZF_MAX_TOUCH_SIZE_LIMIT);
	}
	if (!of_property_read_u32(np, "max-touch-size", &max_touch_size)) {
		if ((max_touch_size > 0) &&
			(max_touch_size <= ZF_MAX_TOUCH_SIZE_LIMIT))
			priv->max_touch_size = max_touch_size;
		else
			dev_err(&priv->client->dev,
				"Maximum touch size limit %dmm invalid (0 < max <= %d)\n",
				max_touch_size, ZF_MAX_TOUCH_SIZE_LIMIT);
	}

	input->name = client->name;
	input->id.bustype = BUS_I2C;
	input->dev.parent = &client->dev;

	input->open = zforce_ts_open;
	input->close = zforce_ts_close;

	input_set_drvdata(input, priv);

	priv->client = client;
	priv->input = input;
	priv->irq = client->irq;
	priv->startup_state = 0;

	error = input_register_device(input);
	if (error)
		goto err1;

	calibrate_state = 1;
	dev = &input->dev;
	error = add_sysfs_files(dev);
	if (error)
		goto err1;

	if (zf_hwcheck_init(priv))
		dev_err(&client->dev, "HW check device init failed\n");

	error = request_threaded_irq(client->irq, NULL, zforce_ts_handle_data,
		IRQF_TRIGGER_LOW | IRQF_ONESHOT, "zforcei2c", priv);

	if (error) {
		dev_err(&client->dev, "Unable to request touchscreen IRQ.\n");
		goto err2;
	}

	device_init_wakeup(&client->dev, 1);
	return 0;

err2:
	input_unregister_device(input);
	input = NULL; /* so we dont try to free it below */
err1:
	if (gpio_is_valid(priv->reset_pin))
		gpio_free(priv->reset_pin);

	input_free_device(input);
	kfree(priv);
err0:
	dev_set_drvdata(&client->dev, NULL);
	return error;
}

static int zforce_ts_remove(struct i2c_client *client)
{
	struct zforce_ts_priv *priv = dev_get_drvdata(&client->dev);
	struct device *dev = &priv->input->dev;

	remove_sysfs_files(dev);
	if (gpio_is_valid(priv->reset_pin))
		gpio_free(priv->reset_pin);

	free_irq(priv->irq, priv);
	input_unregister_device(priv->input);
	kfree(priv);

	dev_set_drvdata(&client->dev, NULL);

	zf_hwcheck_deinit();

	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int zforce_ts_suspend(struct device *dev)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct zforce_ts_priv *priv = dev_get_drvdata(&client->dev);

	if (device_may_wakeup(&client->dev))
		enable_irq_wake(priv->irq);

	return 0;
}

static int zforce_ts_resume(struct device *dev)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct zforce_ts_priv *priv = dev_get_drvdata(&client->dev);

	if (device_may_wakeup(&client->dev))
		disable_irq_wake(priv->irq);

	return 0;
}
#endif /* CONFIG_PM_SLEEP */

static const struct i2c_device_id zforce_ts_id[] = {
	{ "zforcei2c", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, zforce_ts_id);

static SIMPLE_DEV_PM_OPS(zforce_ts_pm_ops, zforce_ts_suspend, zforce_ts_resume);

static struct i2c_driver zforce_ts_driver = {
	.driver = {
		.name = "zforcei2c",
		.pm	= &zforce_ts_pm_ops,
	},
	.probe = zforce_ts_probe,
	.remove = zforce_ts_remove,
	.id_table = zforce_ts_id,
};


/* Register driver with I2C */
static int __init zforce_ts_init(void)
{
	/* Module entry point */
	return i2c_add_driver(&zforce_ts_driver);
};

/* Unregister driver with I2C */
static void __exit zforce_ts_exit(void)
{
	/* Module exit point */
	i2c_del_driver(&zforce_ts_driver);
};

module_init(zforce_ts_init);
module_exit(zforce_ts_exit);

MODULE_AUTHOR("Robert Bosch Car Multimedia GmbH");
MODULE_DESCRIPTION("Neonode ZForce touchscreen driver for I2C bus");
MODULE_LICENSE("GPL v2");
