/*
 * i.MX drm driver
 *
 * Copyright (C) 2012 Sascha Hauer, Pengutronix
 *
 * Based on Samsung Exynos code
 *
 * Copyright (c) 2011 Samsung Electronics Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */
#include <linux/module.h>
#include <drm/drmP.h>
#include <drm/drm_crtc.h>
#include <drm/drm_crtc_helper.h>
#include <drm/drm_fb_cma_helper.h>

#include "imx-drm.h"

#define MAX_CONNECTOR		4

static int preferred_bpp = 32;
MODULE_PARM_DESC(bpp, "fbdev bits-per-pixel (default: 32)\n");
module_param_named(bpp, preferred_bpp, int, 0400);

static struct drm_fbdev_cma *fbdev_cma;

static int __init imx_fb_helper_init(void)
{
	struct drm_device *drm;

	/*
	 * The CRTC/Plane and display drivers must be loaded, probed, and
	 * registered with imx-drm-core _before_ we proceed to call
	 * drm_fbdev_cma_init() and drm_fbdev_cma_restore_mode(),
	 * because the full set of crtc's, connectors, and encoders must
	 * be registered so that fbdev can setup framebuffers and video
	 * modes on them.
	 *
	 * This was the reason for the imx-drm "references" lock, which
	 * would refuse to register any more crtcs/planes/encoders/connectors
	 * if imx-fbdev was already loaded. This is not needed anymore
	 * since we are enforcing the proper load order below.
	 */
	request_module(IMX_CRTC_DRIVER);
	request_module(IMX_PLANE_DRIVER);
	/*
	 * Now request the display drivers (encoders and connectors).
	 *
	 * NOTE!: If new display drivers are added for imx, their
	 * request_module() call needs to be added here.
	 */
	request_module(IMX_LDB_DRIVER);
	request_module(IMX_HDMI_DRIVER);
	request_module(IMX_PD_DRIVER);

	drm = imx_drm_device_get();
	if (!drm)
		return -EINVAL;

	fbdev_cma = drm_fbdev_cma_init(drm, preferred_bpp,
			drm->mode_config.num_crtc, MAX_CONNECTOR);

	if (IS_ERR(fbdev_cma)) {
		imx_drm_device_put();
		return PTR_ERR(fbdev_cma);
	}

	imx_drm_fb_helper_set(fbdev_cma);
	drm_fbdev_cma_restore_mode(fbdev_cma);

	return 0;
}

static void __exit imx_fb_helper_exit(void)
{
	imx_drm_fb_helper_set(NULL);
	drm_fbdev_cma_fini(fbdev_cma);
	imx_drm_device_put();
}

late_initcall(imx_fb_helper_init);
module_exit(imx_fb_helper_exit);

MODULE_DESCRIPTION("Freescale i.MX legacy fb driver");
MODULE_AUTHOR("Sascha Hauer, Pengutronix");
MODULE_LICENSE("GPL");
