/*
 * Freescale i.MX drm driver
 *
 * Copyright (C) 2011 Sascha Hauer, Pengutronix
 * Copyright (C) 2012 Mentor Graphics
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/device.h>
#include <linux/platform_device.h>
#include <drm/drmP.h>
#include <drm/drm_fb_helper.h>
#include <drm/drm_crtc_helper.h>
#include <linux/fb.h>
#include <linux/module.h>
#include <drm/drm_gem_cma_helper.h>
#include <drm/drm_fb_cma_helper.h>
#include <linux/videodev2.h>

#include "imx-drm.h"

#define MAX_CRTC	4
#define MAX_PLANES	2

struct crtc_cookie {
	void *cookie;
	struct list_head list;
};

struct imx_drm_device {
	struct drm_device			*drm;
	struct device				*dev;
	struct list_head			crtc_list;
	struct list_head			encoder_list;
	struct list_head			connector_list;
	struct list_head			plane_list;
	struct mutex				mutex;
	struct drm_fbdev_cma			*fbhelper;
};

struct imx_drm_crtc {
	struct drm_crtc				*crtc;
	struct list_head			list;
	struct imx_drm_device			*imxdrm;
	int					id;
	struct imx_drm_crtc_helper_funcs	imx_drm_helper_funcs;
	struct module				*owner;
	struct crtc_cookie			cookie;
};

struct imx_drm_encoder {
	struct drm_encoder			*encoder;
	struct list_head			list;
	struct module				*owner;
	struct list_head			possible_crtcs;
};

struct imx_drm_connector {
	struct drm_connector			*connector;
	struct list_head			list;
	struct module				*owner;
};

struct imx_drm_plane {
	struct drm_plane			*plane;
	struct list_head			list;
	struct imx_drm_device			*imxdrm;
	int					plane_id;
	struct imx_drm_plane_helper_funcs	imx_drm_helper_funcs;
	struct module				*owner;
	struct list_head			possible_crtcs;
};

static struct imx_drm_device *imx_drm_device;

static struct imx_drm_device *__imx_drm_device(void)
{
	return imx_drm_device;
}

static int imx_drm_driver_firstopen(struct drm_device *drm)
{
	if (!imx_drm_device_get())
		return -EINVAL;

	return 0;
}

static void imx_drm_driver_lastclose(struct drm_device *drm)
{
	struct imx_drm_device *imxdrm = drm->dev_private;

	if (imxdrm->fbhelper)
		drm_fbdev_cma_restore_mode(imxdrm->fbhelper);

	imx_drm_device_put();
}

static int imx_drm_driver_unload(struct drm_device *drm)
{
	struct imx_drm_device *imxdrm = drm->dev_private;

	drm_mode_config_cleanup(imxdrm->drm);
	drm_kms_helper_poll_fini(imxdrm->drm);

	return 0;
}

/*
 * We don't care at all for crtc numbers, but the core expects the
 * crtcs to be numbered. Hold imxdrm->mutex before calling.
 */
static struct imx_drm_crtc *imx_drm_crtc_by_num(struct imx_drm_device *imxdrm,
		int num)
{
	struct imx_drm_crtc *imx_crtc;

	list_for_each_entry(imx_crtc, &imxdrm->crtc_list, list)
		if (imx_crtc->id == num)
			return imx_crtc;
	return NULL;
}

/* hold imxdrm->mutex before calling */
static struct imx_drm_crtc *imx_drm_crtc_by_crtc(struct imx_drm_device *imxdrm,
						 struct drm_crtc *crtc)
{
	struct imx_drm_crtc *imx_crtc;

	list_for_each_entry(imx_crtc, &imxdrm->crtc_list, list)
		if (imx_crtc->crtc == crtc)
			return imx_crtc;
	return NULL;
}

/*
 * hold dev->mode_config.mutex and imxdrm->mutex, in that order,
 * before calling.
 */
static struct imx_drm_crtc *imx_drm_crtc_by_id(struct drm_device *dev,
					       u32 crtc_id)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct drm_mode_object *obj;
	struct drm_crtc *crtc;

	obj = drm_mode_object_find(dev, crtc_id, DRM_MODE_OBJECT_CRTC);
	if (!obj)
		return NULL;
	crtc = obj_to_crtc(obj);

	return imx_drm_crtc_by_crtc(imxdrm, crtc);
}

/* hold imxdrm->mutex before calling */
static struct imx_drm_plane *imx_drm_plane_by_num(struct imx_drm_device *imxdrm,
		int num)
{
	struct imx_drm_plane *imx_plane;

	list_for_each_entry(imx_plane, &imxdrm->plane_list, list)
		if (imx_plane->plane_id == num)
			return imx_plane;
	return NULL;
}

/* hold imxdrm->mutex before calling */
static struct imx_drm_plane *
imx_drm_plane_by_plane(struct imx_drm_device *imxdrm, struct drm_plane *plane)
{
	struct imx_drm_plane *imx_plane;

	list_for_each_entry(imx_plane, &imxdrm->plane_list, list)
		if (imx_plane->plane == plane)
			return imx_plane;
	return NULL;
}

/*
 * hold dev->mode_config.mutex and imxdrm->mutex, in that order,
 * before calling.
 */
static struct imx_drm_plane *imx_drm_plane_by_id(struct drm_device *dev,
						 u32 plane_id)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct drm_mode_object *obj;
	struct drm_plane *plane;

	obj = drm_mode_object_find(dev, plane_id, DRM_MODE_OBJECT_PLANE);
	if (!obj)
		return NULL;

	plane = obj_to_plane(obj);

	return imx_drm_plane_by_plane(imxdrm, plane);
}

static int __imx_drm_crtc_get_ipu_di(struct imx_drm_device *imxdrm,
				     struct drm_crtc *crtc,
				     int *ipu, int *di)
{
	struct imx_drm_crtc_helper_funcs *helper;
	struct imx_drm_crtc *imx_crtc;

	imx_crtc = imx_drm_crtc_by_crtc(imxdrm, crtc);
	if (!imx_crtc)
		return -EINVAL;
	helper = &imx_crtc->imx_drm_helper_funcs;

	return helper->get_ipu_di(crtc, ipu, di);
}

int imx_drm_crtc_get_ipu_di(struct drm_crtc *crtc, int *ipu, int *di)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	int ret;

	mutex_lock(&imxdrm->mutex);
	ret = __imx_drm_crtc_get_ipu_di(imxdrm, crtc, ipu, di);
	mutex_unlock(&imxdrm->mutex);

	return ret;
}
EXPORT_SYMBOL_GPL(imx_drm_crtc_get_ipu_di);

int imx_drm_crtc_get_id(struct drm_crtc *crtc)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct imx_drm_crtc *imx_crtc;

	mutex_lock(&imxdrm->mutex);
	imx_crtc = imx_drm_crtc_by_crtc(imxdrm, crtc);
	mutex_unlock(&imxdrm->mutex);

	return imx_crtc ? imx_crtc->id : -EINVAL;
}
EXPORT_SYMBOL_GPL(imx_drm_crtc_get_id);

int imx_drm_crtc_panel_format(struct drm_crtc *crtc, u32 encoder_type,
			      u32 interface_pix_fmt, u32 *interface_pix_map)
{
	struct imx_drm_device *imxdrm = crtc->dev->dev_private;
	struct imx_drm_crtc *imx_crtc;
	struct imx_drm_crtc_helper_funcs *helper;

	mutex_lock(&imxdrm->mutex);
	imx_crtc = imx_drm_crtc_by_crtc(imxdrm, crtc);
	mutex_unlock(&imxdrm->mutex);
	if (!imx_crtc)
		return -EINVAL;

	helper = &imx_crtc->imx_drm_helper_funcs;
	if (helper->set_interface_pix_fmt)
		return helper->set_interface_pix_fmt(
			crtc, encoder_type, interface_pix_fmt,
			interface_pix_map);
	return 0;
}
EXPORT_SYMBOL_GPL(imx_drm_crtc_panel_format);

int imx_drm_crtc_vblank_get(struct imx_drm_crtc *imx_drm_crtc)
{
	return drm_vblank_get(imx_drm_crtc->imxdrm->drm, imx_drm_crtc->id);
}
EXPORT_SYMBOL_GPL(imx_drm_crtc_vblank_get);

void imx_drm_crtc_vblank_put(struct imx_drm_crtc *imx_drm_crtc)
{
	drm_vblank_put(imx_drm_crtc->imxdrm->drm, imx_drm_crtc->id);
}
EXPORT_SYMBOL_GPL(imx_drm_crtc_vblank_put);

void imx_drm_crtc_handle_vblank(struct imx_drm_crtc *imx_drm_crtc)
{
	drm_handle_vblank(imx_drm_crtc->imxdrm->drm, imx_drm_crtc->id);
}
EXPORT_SYMBOL_GPL(imx_drm_crtc_handle_vblank);

int imx_drm_plane_vblank_get(struct imx_drm_plane *imx_drm_plane)
{
	return drm_vblank_get(imx_drm_plane->imxdrm->drm,
			MAX_CRTC + imx_drm_plane->plane_id);
}
EXPORT_SYMBOL_GPL(imx_drm_plane_vblank_get);

void imx_drm_plane_vblank_put(struct imx_drm_plane *imx_drm_plane)
{
	drm_vblank_put(imx_drm_plane->imxdrm->drm,
			MAX_CRTC + imx_drm_plane->plane_id);
}
EXPORT_SYMBOL_GPL(imx_drm_plane_vblank_put);

void imx_drm_plane_handle_vblank(struct imx_drm_plane *imx_drm_plane)
{
	drm_handle_vblank(imx_drm_plane->imxdrm->drm,
			MAX_CRTC + imx_drm_plane->plane_id);
}
EXPORT_SYMBOL_GPL(imx_drm_plane_handle_vblank);

static int imx_drm_crtc_enable_vblank(struct drm_device *drm, int crtc)
{
	struct imx_drm_device *imxdrm = drm->dev_private;
	struct imx_drm_crtc *imx_drm_crtc;
	int ret;

	imx_drm_crtc = imx_drm_crtc_by_num(imxdrm, crtc);
	if (!imx_drm_crtc)
		return -EINVAL;

	if (!imx_drm_crtc->imx_drm_helper_funcs.enable_vblank)
		return -ENOSYS;

	ret = imx_drm_crtc->imx_drm_helper_funcs.enable_vblank(
			imx_drm_crtc->crtc);

	return ret;
}

static int imx_drm_plane_enable_vblank(struct drm_device *drm, int crtc)
{
	struct imx_drm_device *imxdrm = drm->dev_private;
	struct imx_drm_plane *imx_drm_plane;
	int ret;

	imx_drm_plane = imx_drm_plane_by_num(imxdrm, crtc);
	if (!imx_drm_plane)
		return -EINVAL;

	if (!imx_drm_plane->imx_drm_helper_funcs.enable_vblank)
		return -ENOSYS;

	ret = imx_drm_plane->imx_drm_helper_funcs.enable_vblank(
			imx_drm_plane->plane);

	return ret;
}

static int imx_drm_enable_vblank(struct drm_device *drm, int crtc)
{
	if (crtc < MAX_CRTC)
		return imx_drm_crtc_enable_vblank(drm, crtc);
	else
		return imx_drm_plane_enable_vblank(drm, crtc - MAX_CRTC);
}

static void imx_drm_crtc_disable_vblank(struct drm_device *drm, int crtc)
{
	struct imx_drm_device *imxdrm = drm->dev_private;
	struct imx_drm_crtc *imx_drm_crtc;

	imx_drm_crtc = imx_drm_crtc_by_num(imxdrm, crtc);
	if (!imx_drm_crtc)
		return;

	if (!imx_drm_crtc->imx_drm_helper_funcs.disable_vblank)
		return;

	imx_drm_crtc->imx_drm_helper_funcs.disable_vblank(imx_drm_crtc->crtc);
}

static void imx_drm_plane_disable_vblank(struct drm_device *drm, int crtc)
{
	struct imx_drm_device *imxdrm = drm->dev_private;
	struct imx_drm_plane *imx_drm_plane;

	imx_drm_plane = imx_drm_plane_by_num(imxdrm, crtc);
	if (!imx_drm_plane)
		return;

	if (!imx_drm_plane->imx_drm_helper_funcs.disable_vblank)
		return;

	imx_drm_plane->imx_drm_helper_funcs.disable_vblank(imx_drm_plane->plane);
}

static void imx_drm_disable_vblank(struct drm_device *drm, int crtc)
{
	if (crtc < MAX_CRTC)
		imx_drm_crtc_disable_vblank(drm, crtc);
	else
		imx_drm_plane_disable_vblank(drm, crtc - MAX_CRTC);
}

static const struct file_operations imx_drm_driver_fops = {
	.owner = THIS_MODULE,
	.open = drm_open,
	.release = drm_release,
	.unlocked_ioctl = drm_ioctl,
	.mmap = drm_gem_cma_mmap,
	.poll = drm_poll,
	.fasync = drm_fasync,
	.read = drm_read,
	.llseek = noop_llseek,
};

struct drm_device *imx_drm_device_get(void)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct imx_drm_encoder *enc;
	struct imx_drm_connector *con;
	struct imx_drm_crtc *crtc;

	mutex_lock(&imxdrm->mutex);

	list_for_each_entry(enc, &imxdrm->encoder_list, list) {
		if (!try_module_get(enc->owner)) {
			dev_err(imxdrm->dev, "could not get module %s\n",
					module_name(enc->owner));
			goto unwind_enc;
		}
	}

	list_for_each_entry(con, &imxdrm->connector_list, list) {
		if (!try_module_get(con->owner)) {
			dev_err(imxdrm->dev, "could not get module %s\n",
					module_name(con->owner));
			goto unwind_con;
		}
	}

	list_for_each_entry(crtc, &imxdrm->crtc_list, list) {
		if (!try_module_get(crtc->owner)) {
			dev_err(imxdrm->dev, "could not get module %s\n",
					module_name(crtc->owner));
			goto unwind_crtc;
		}
	}

	mutex_unlock(&imxdrm->mutex);

	return imxdrm->drm;

unwind_crtc:
	list_for_each_entry_continue_reverse(crtc, &imxdrm->crtc_list, list)
		module_put(crtc->owner);
unwind_con:
	list_for_each_entry_continue_reverse(con, &imxdrm->connector_list, list)
		module_put(con->owner);
unwind_enc:
	list_for_each_entry_continue_reverse(enc, &imxdrm->encoder_list, list)
		module_put(enc->owner);

	mutex_unlock(&imxdrm->mutex);

	return NULL;

}
EXPORT_SYMBOL_GPL(imx_drm_device_get);

void imx_drm_device_put(void)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct imx_drm_encoder *enc;
	struct imx_drm_connector *con;
	struct imx_drm_crtc *crtc;

	mutex_lock(&imxdrm->mutex);

	list_for_each_entry(crtc, &imxdrm->crtc_list, list)
		module_put(crtc->owner);

	list_for_each_entry(con, &imxdrm->connector_list, list)
		module_put(con->owner);

	list_for_each_entry(enc, &imxdrm->encoder_list, list)
		module_put(enc->owner);

	mutex_unlock(&imxdrm->mutex);
}
EXPORT_SYMBOL_GPL(imx_drm_device_put);

static int drm_mode_group_reinit(struct drm_device *dev)
{
	struct drm_mode_group *group = &dev->primary->mode_group;
	uint32_t *id_list = group->id_list;
	int ret;

	ret = drm_mode_group_init_legacy_group(dev, group);
	if (ret < 0)
		return ret;

	kfree(id_list);
	return 0;
}

/*
 * register an encoder to the drm core
 */
static int imx_drm_encoder_register(struct imx_drm_encoder *imx_drm_encoder)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();

	INIT_LIST_HEAD(&imx_drm_encoder->possible_crtcs);

	drm_encoder_init(imxdrm->drm, imx_drm_encoder->encoder,
			imx_drm_encoder->encoder->funcs,
			imx_drm_encoder->encoder->encoder_type);

	drm_mode_group_reinit(imxdrm->drm);

	return 0;
}

/*
 * register a connector to the drm core
 */
static int imx_drm_connector_register(
		struct imx_drm_connector *imx_drm_connector)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();

	drm_connector_init(imxdrm->drm, imx_drm_connector->connector,
			imx_drm_connector->connector->funcs,
			imx_drm_connector->connector->connector_type);
	drm_mode_group_reinit(imxdrm->drm);

	return drm_sysfs_connector_add(imx_drm_connector->connector);
}

/*
 * register a crtc to the drm core
 */
static int imx_drm_crtc_register(struct imx_drm_crtc *imx_drm_crtc)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();

	drm_crtc_init(imxdrm->drm, imx_drm_crtc->crtc,
			imx_drm_crtc->imx_drm_helper_funcs.crtc_funcs);

	drm_crtc_helper_add(imx_drm_crtc->crtc,
			imx_drm_crtc->imx_drm_helper_funcs.crtc_helper_funcs);

	drm_mode_group_reinit(imxdrm->drm);

	return 0;
}

/*
 * Called by the CRTC driver when all CRTCs are registered. This
 * puts all the pieces together and initializes the driver.
 * Once this is called no more CRTCs can be registered since
 * the drm core has hardcoded the number of crtcs in several
 * places.
 */
static int imx_drm_driver_load(struct drm_device *drm, unsigned long flags)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	int ret;

	imxdrm->drm = drm;

	drm->dev_private = imxdrm;

	/*
	 * enable drm irq mode.
	 * - with irq_enabled = 1, we can use the vblank feature.
	 *
	 * P.S. note that we wouldn't use drm irq handler but
	 *      just specific driver own one instead because
	 *      drm framework supports only one irq handler and
	 *      drivers can well take care of their interrupts
	 */
	drm->irq_enabled = 1;

	drm_mode_config_init(drm);
	imx_drm_mode_config_init(drm);

	mutex_lock(&imxdrm->mutex);

	drm_kms_helper_poll_init(imxdrm->drm);

	/* setup the grouping for the legacy output */
	ret = drm_mode_group_init_legacy_group(imxdrm->drm,
			&imxdrm->drm->primary->mode_group);
	if (ret)
		goto err_init;

	ret = drm_vblank_init(imxdrm->drm, MAX_CRTC + MAX_PLANES);
	if (ret)
		goto err_init;

	/*
	 * with vblank_disable_allowed = 1, vblank interrupt will be disabled
	 * by drm timer once a current process gives up ownership of
	 * vblank event.(after drm_vblank_put function is called)
	 */
	imxdrm->drm->vblank_disable_allowed = 1;

	ret = 0;

err_init:
	mutex_unlock(&imxdrm->mutex);

	return ret;
}

static void imx_drm_update_plane_possible_crtcs(struct imx_drm_plane *plane)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct imx_drm_crtc *imx_drm_crtc;
	struct crtc_cookie *cookie;
	u32 possible_crtcs = 0;

	list_for_each_entry(cookie, &plane->possible_crtcs, list) {
		list_for_each_entry(imx_drm_crtc, &imxdrm->crtc_list, list) {
			if (imx_drm_crtc->cookie.cookie == cookie->cookie)
				possible_crtcs |= 1 << imx_drm_crtc->id;
		}
	}
	plane->plane->possible_crtcs = possible_crtcs;
}

static void imx_drm_update_possible_crtcs(void)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct imx_drm_crtc *imx_drm_crtc;
	struct imx_drm_encoder *enc;
	struct imx_drm_plane *plane;
	struct crtc_cookie *cookie;

	list_for_each_entry(enc, &imxdrm->encoder_list, list) {
		u32 possible_crtcs = 0;

		list_for_each_entry(cookie, &enc->possible_crtcs, list) {
			list_for_each_entry(imx_drm_crtc, &imxdrm->crtc_list, list) {
				if (imx_drm_crtc->cookie.cookie ==
				    cookie->cookie)
					possible_crtcs |= 1 << imx_drm_crtc->id;
			}
		}
		enc->encoder->possible_crtcs = possible_crtcs;
		enc->encoder->possible_clones = possible_crtcs;
	}

	list_for_each_entry(plane, &imxdrm->plane_list, list)
		imx_drm_update_plane_possible_crtcs(plane);
}

/*
 * imx_drm_add_crtc - add a new crtc
 *
 * The return value if !NULL is a cookie for the caller to pass to
 * imx_drm_remove_crtc later.
 */
int imx_drm_add_crtc(
	struct drm_crtc *crtc,
	struct imx_drm_crtc **new_crtc,
	const struct imx_drm_crtc_helper_funcs *imx_drm_helper_funcs,
	struct module *owner, void *cookie, int crtc_id)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct imx_drm_crtc *imx_drm_crtc;
	const struct drm_crtc_funcs *crtc_funcs;
	int ret;

	imx_drm_crtc = kzalloc(sizeof(*imx_drm_crtc), GFP_KERNEL);
	if (!imx_drm_crtc) {
		ret = -ENOMEM;
		goto err_alloc;
	}

	imx_drm_crtc->imx_drm_helper_funcs = *imx_drm_helper_funcs;
	imx_drm_crtc->id = crtc_id;
	imx_drm_crtc->cookie.cookie = cookie;

	crtc_funcs = imx_drm_helper_funcs->crtc_funcs;

	imx_drm_crtc->crtc = crtc;
	imx_drm_crtc->imxdrm = imxdrm;

	imx_drm_crtc->owner = owner;

	mutex_lock(&imxdrm->mutex);
	list_add_tail(&imx_drm_crtc->list, &imxdrm->crtc_list);

	*new_crtc = imx_drm_crtc;
	mutex_unlock(&imxdrm->mutex);

	ret = imx_drm_crtc_register(imx_drm_crtc);
	if (ret)
		goto err_register;

	mutex_lock(&imxdrm->mutex);

	imx_drm_update_possible_crtcs();

	mutex_unlock(&imxdrm->mutex);

	return 0;

err_register:
	kfree(imx_drm_crtc);
err_alloc:
	return ret;
}
EXPORT_SYMBOL_GPL(imx_drm_add_crtc);

/*
 * imx_drm_remove_crtc - remove a crtc
 */
int imx_drm_remove_crtc(struct imx_drm_crtc *imx_drm_crtc)
{
	struct imx_drm_device *imxdrm = imx_drm_crtc->imxdrm;

	drm_crtc_cleanup(imx_drm_crtc->crtc);

	mutex_lock(&imxdrm->mutex);

	list_del(&imx_drm_crtc->list);

	drm_mode_group_reinit(imxdrm->drm);

	mutex_unlock(&imxdrm->mutex);

	kfree(imx_drm_crtc);

	return 0;
}
EXPORT_SYMBOL_GPL(imx_drm_remove_crtc);

/*
 * Set an imx_crtc, given by its id, to DPMS mode 'mode'.
 */
void imx_drm_crtc_dpms(int crtc_id, int mode)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct drm_crtc_helper_funcs *crtc_funcs;
	struct imx_drm_crtc *imx_crtc;
	struct drm_crtc *crtc;

	mutex_lock(&imxdrm->mutex);
	imx_crtc = imx_drm_crtc_by_num(imxdrm, crtc_id);
	mutex_unlock(&imxdrm->mutex);

	if (!imx_crtc)
		return;

	crtc = imx_crtc->crtc;
	crtc_funcs = crtc->helper_private;

	if (!crtc_funcs || !crtc_funcs->dpms)
		return;

	if (mode == DRM_MODE_DPMS_ON) {
		if (crtc->enabled)
			(*crtc_funcs->dpms)(crtc, DRM_MODE_DPMS_ON);
	} else
		(*crtc_funcs->dpms)(crtc, DRM_MODE_DPMS_OFF);
}
EXPORT_SYMBOL_GPL(imx_drm_crtc_dpms);

/*
 * Find and return a plane that is enabled and bound to this crtc.
 */
struct imx_drm_plane *
imx_drm_crtc_find_active_plane(struct imx_drm_crtc *imx_crtc)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct imx_drm_plane *imx_plane;
	struct imx_drm_plane *ret = NULL;

	mutex_lock(&imxdrm->mutex);

	list_for_each_entry(imx_plane, &imxdrm->plane_list, list) {
		if (imx_plane->plane->crtc == imx_crtc->crtc) {
			ret = imx_plane;
			goto out;
		}
	}

out:
	mutex_unlock(&imxdrm->mutex);
	return ret;
}
EXPORT_SYMBOL_GPL(imx_drm_crtc_find_active_plane);

/*
 * imx_drm_add_encoder - add a new encoder
 */
int imx_drm_add_encoder(struct drm_encoder *encoder,
		struct imx_drm_encoder **newenc, struct module *owner)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct imx_drm_encoder *imx_drm_encoder;
	int ret;

	imx_drm_encoder = kzalloc(sizeof(*imx_drm_encoder), GFP_KERNEL);
	if (!imx_drm_encoder) {
		ret = -ENOMEM;
		goto err_alloc;
	}

	imx_drm_encoder->encoder = encoder;
	imx_drm_encoder->owner = owner;

	ret = imx_drm_encoder_register(imx_drm_encoder);
	if (ret) {
		kfree(imx_drm_encoder);
		ret = -ENOMEM;
		goto err_register;
	}

	mutex_lock(&imxdrm->mutex);
	list_add_tail(&imx_drm_encoder->list, &imxdrm->encoder_list);

	*newenc = imx_drm_encoder;

	mutex_unlock(&imxdrm->mutex);

	return 0;

err_register:
	kfree(imx_drm_encoder);
err_alloc:
	return ret;
}
EXPORT_SYMBOL_GPL(imx_drm_add_encoder);

static int imx_drm_add_possible_crtcs(struct list_head *possible_crtcs,
				      struct device_node *np)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct crtc_cookie *c;
	struct device_node *crtc_np;
	int i, ret = 0;

	mutex_lock(&imxdrm->mutex);

	if (!list_empty(possible_crtcs)) {
		ret = -EBUSY;
		goto out_unlock;
	}

	for (i = 0;; i++) {
		crtc_np = of_parse_phandle(np, "crtcs", i);
		if (!crtc_np)
			break;

		c = kzalloc(sizeof(*c), GFP_KERNEL);
		if (!c) {
			of_node_put(crtc_np);
			ret = -ENOMEM;
			goto out_unlock;
		}

		c->cookie = crtc_np;

		of_node_put(crtc_np);

		list_add_tail(&c->list, possible_crtcs);
	}

	imx_drm_update_possible_crtcs();

out_unlock:
	mutex_unlock(&imxdrm->mutex);
	return ret;
}

int imx_drm_encoder_add_possible_crtcs(struct imx_drm_encoder *imx_drm_encoder,
				       struct device_node *np)
{
	return imx_drm_add_possible_crtcs(&imx_drm_encoder->possible_crtcs, np);
}
EXPORT_SYMBOL_GPL(imx_drm_encoder_add_possible_crtcs);

/*
 * imx_drm_remove_encoder - remove an encoder
 */
int imx_drm_remove_encoder(struct imx_drm_encoder *imx_drm_encoder)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct crtc_cookie *c, *tmp;

	drm_encoder_cleanup(imx_drm_encoder->encoder);

	mutex_lock(&imxdrm->mutex);

	drm_mode_group_reinit(imxdrm->drm);

	list_del(&imx_drm_encoder->list);

	list_for_each_entry_safe(c, tmp, &imx_drm_encoder->possible_crtcs,
			list)
		kfree(c);

	mutex_unlock(&imxdrm->mutex);

	kfree(imx_drm_encoder);

	return 0;
}
EXPORT_SYMBOL_GPL(imx_drm_remove_encoder);

/*
 * imx_drm_add_connector - add a connector
 */
int imx_drm_add_connector(struct drm_connector *connector,
		struct imx_drm_connector **new_con,
		struct module *owner)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct imx_drm_connector *imx_drm_connector;
	int ret;

	imx_drm_connector = kzalloc(sizeof(*imx_drm_connector), GFP_KERNEL);
	if (!imx_drm_connector) {
		ret = -ENOMEM;
		goto err_alloc;
	}

	imx_drm_connector->connector = connector;
	imx_drm_connector->owner = owner;

	ret = imx_drm_connector_register(imx_drm_connector);
	if (ret)
		goto err_register;

	mutex_lock(&imxdrm->mutex);
	list_add_tail(&imx_drm_connector->list, &imxdrm->connector_list);

	*new_con = imx_drm_connector;

	mutex_unlock(&imxdrm->mutex);

	return 0;

err_register:
	kfree(imx_drm_connector);
err_alloc:
	return ret;
}
EXPORT_SYMBOL_GPL(imx_drm_add_connector);

void imx_drm_fb_helper_set(struct drm_fbdev_cma *fbdev_helper)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();

	imxdrm->fbhelper = fbdev_helper;
}
EXPORT_SYMBOL_GPL(imx_drm_fb_helper_set);

void imx_drm_fb_output_poll_changed(struct drm_device *dev)
{
	struct imx_drm_device *imxdrm = dev->dev_private;

	if (imxdrm->fbhelper)
		drm_fbdev_cma_hotplug_event(imxdrm->fbhelper);
}
EXPORT_SYMBOL_GPL(imx_drm_fb_output_poll_changed);

/*
 * imx_drm_remove_connector - remove a connector
 */
int imx_drm_remove_connector(struct imx_drm_connector *imx_drm_connector)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();

	drm_sysfs_connector_remove(imx_drm_connector->connector);
	drm_connector_cleanup(imx_drm_connector->connector);

	mutex_lock(&imxdrm->mutex);

	drm_mode_group_reinit(imxdrm->drm);

	list_del(&imx_drm_connector->list);

	mutex_unlock(&imxdrm->mutex);

	kfree(imx_drm_connector);

	return 0;
}
EXPORT_SYMBOL_GPL(imx_drm_remove_connector);

/*
 * imx_drm_add_plane - add a plane
 */
int imx_drm_add_plane(struct drm_plane *plane,
		struct imx_drm_plane **new_plane,
		const struct imx_drm_plane_helper_funcs *imx_drm_helper_funcs,
		struct module *owner,
		const uint32_t *formats, uint32_t format_count,
		int plane_id)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct imx_drm_plane *imx_drm_plane;
	const struct drm_plane_funcs *funcs = imx_drm_helper_funcs->plane_funcs;
	int ret;

	imx_drm_plane = kzalloc(sizeof(*imx_drm_plane), GFP_KERNEL);
	if (!imx_drm_plane) {
		ret = -ENOMEM;
		goto err_alloc;
	}

	imx_drm_plane->plane = plane;
	imx_drm_plane->owner = owner;
	imx_drm_plane->imx_drm_helper_funcs = *imx_drm_helper_funcs;
	imx_drm_plane->plane_id = plane_id;
	imx_drm_plane->imxdrm = imxdrm;

	INIT_LIST_HEAD(&imx_drm_plane->possible_crtcs);

	ret = drm_plane_init(imxdrm->drm, plane, 0, funcs,
			formats, format_count, false);
	if (ret)
		goto err_register;

	mutex_lock(&imxdrm->mutex);
	list_add_tail(&imx_drm_plane->list, &imxdrm->plane_list);

	*new_plane = imx_drm_plane;

	mutex_unlock(&imxdrm->mutex);

	return 0;

err_register:
	kfree(imx_drm_plane);
err_alloc:
	return ret;
}
EXPORT_SYMBOL_GPL(imx_drm_add_plane);

int imx_drm_plane_add_possible_crtcs(struct imx_drm_plane *imx_drm_plane,
				     struct device_node *np)
{
	return imx_drm_add_possible_crtcs(&imx_drm_plane->possible_crtcs, np);
}
EXPORT_SYMBOL_GPL(imx_drm_plane_add_possible_crtcs);

/*
 * imx_drm_plane_get_config - retrieve the plane's current config.
 * Caller must hold the dev mode config lock.
 */
void imx_drm_plane_get_config(struct imx_drm_plane *imx_drm_plane,
			      int *crtc_x, int *crtc_y,
			      uint32_t *crtc_w, uint32_t *crtc_h,
			      uint32_t *src_x,  uint32_t *src_y,
			      uint32_t *src_w,  uint32_t *src_h)
{
	struct drm_plane *plane = imx_drm_plane->plane;
	struct imx_drm_plane_helper_funcs *helper;

	helper = &imx_drm_plane->imx_drm_helper_funcs;
	helper->get_config(plane, crtc_x, crtc_y, crtc_w, crtc_h,
			   src_x, src_y, src_w, src_h);
}
EXPORT_SYMBOL_GPL(imx_drm_plane_get_config);

/*
 * imx_drm_plane_disable - disable a plane. Caller must hold the
 * dev mode config lock.
 */
void imx_drm_plane_disable(struct imx_drm_plane *imx_drm_plane)
{
	struct drm_plane *plane = imx_drm_plane->plane;

	plane->funcs->disable_plane(plane);
}
EXPORT_SYMBOL_GPL(imx_drm_plane_disable);

/*
 * imx_drm_plane_enable - enable a plane after a disable. Caller must
 * hold the dev mode config lock.
 */
int imx_drm_plane_enable(struct imx_drm_plane *imx_drm_plane)
{
	struct drm_plane *plane = imx_drm_plane->plane;
	uint32_t crtc_w, crtc_h, src_x, src_y, src_w, src_h;
	int crtc_x, crtc_y;

	if (!plane->crtc || !plane->fb)
		return -EINVAL;

	imx_drm_plane_get_config(imx_drm_plane,
				 &crtc_x, &crtc_y, &crtc_w, &crtc_h,
				 &src_x, &src_y, &src_w, &src_h);

	return plane->funcs->update_plane(plane, plane->crtc, plane->fb,
					  crtc_x, crtc_y, crtc_w, crtc_h,
					  src_x, src_y, src_w, src_h);
}
EXPORT_SYMBOL_GPL(imx_drm_plane_enable);

/*
 * imx_drm_remove_plane - remove a plane
 */
int imx_drm_remove_plane(struct imx_drm_plane *imx_drm_plane)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct crtc_cookie *c, *tmp;

	drm_plane_cleanup(imx_drm_plane->plane);

	mutex_lock(&imxdrm->mutex);

	list_del(&imx_drm_plane->list);

	list_for_each_entry_safe(c, tmp, &imx_drm_plane->possible_crtcs,
			list)
		kfree(c);

	mutex_unlock(&imxdrm->mutex);

	kfree(imx_drm_plane);

	return 0;
}
EXPORT_SYMBOL_GPL(imx_drm_remove_plane);

int imx_drm_format_to_v4l(uint32_t drm_format, u32 *v4l_format)
{
	switch (drm_format) {
	case DRM_FORMAT_XRGB8888:
	case DRM_FORMAT_ARGB8888:
		*v4l_format = V4L2_PIX_FMT_RGB32;
		break;
	case DRM_FORMAT_RGB565:
		*v4l_format = V4L2_PIX_FMT_RGB565;
		break;
	case DRM_FORMAT_RGB888:
		*v4l_format = V4L2_PIX_FMT_RGB24;
		break;
	case DRM_FORMAT_BGR888:
		*v4l_format = V4L2_PIX_FMT_BGR24;
		break;
	case DRM_FORMAT_UYVY:
		*v4l_format = V4L2_PIX_FMT_UYVY;
		break;
	case DRM_FORMAT_YUYV:
		*v4l_format = V4L2_PIX_FMT_YUYV;
		break;
	case DRM_FORMAT_YUV420:
		*v4l_format = V4L2_PIX_FMT_YUV420;
		break;
	case DRM_FORMAT_YVU420:
		*v4l_format = V4L2_PIX_FMT_YVU420;
		break;
	case DRM_FORMAT_YUV422:
		*v4l_format = V4L2_PIX_FMT_YUV422P;
		break;
	case DRM_FORMAT_YVU422:
		*v4l_format = V4L2_PIX_FMT_YVU422P;
		break;
	case DRM_FORMAT_NV12:
		*v4l_format = V4L2_PIX_FMT_NV12;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}
EXPORT_SYMBOL_GPL(imx_drm_format_to_v4l);

static int drm_imx_get_phys_ioctl(struct drm_device *drm,
				  void *data, struct drm_file *file_priv)
{
	struct drm_imx_get_phys *args = data;
	struct drm_gem_object *gem_obj;
	struct drm_gem_cma_object *cma_obj;

	mutex_lock(&drm->struct_mutex);

	gem_obj = drm_gem_object_lookup(drm, file_priv, args->handle);
	if (!gem_obj) {
		dev_err(drm->dev, "failed to lookup gem object\n");
		mutex_unlock(&drm->struct_mutex);
		return -EINVAL;
	}

	cma_obj = to_drm_gem_cma_obj(gem_obj);
	args->paddr = cma_obj->paddr;

	drm_gem_object_unreference(gem_obj);

	mutex_unlock(&drm->struct_mutex);

	return 0;
}

static int drm_imx_plane_get_global_alpha_ioctl(struct drm_device *dev,
						void *data,
						struct drm_file *file_priv)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct drm_imx_global_alpha *ga = data;
	struct imx_drm_plane_helper_funcs *helper;
	struct imx_drm_plane *imx_plane;
	int ret;

	if (!drm_core_check_feature(dev, DRIVER_MODESET))
		return -ENODEV;

	mutex_lock(&dev->mode_config.mutex);
	mutex_lock(&imxdrm->mutex);

	imx_plane = imx_drm_plane_by_id(dev, ga->plane_id);
	if (!imx_plane) {
		ret = -EINVAL;
		goto out_unlock;
	}

	helper = &imx_plane->imx_drm_helper_funcs;
	ret = helper->get_global_alpha(imx_plane->plane, ga);

out_unlock:
	mutex_unlock(&imxdrm->mutex);
	mutex_unlock(&dev->mode_config.mutex);
	return ret;
}

static int drm_imx_plane_set_global_alpha_ioctl(struct drm_device *dev,
						void *data,
						struct drm_file *file_priv)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct drm_imx_global_alpha *ga = data;
	struct imx_drm_plane_helper_funcs *helper;
	struct imx_drm_plane *imx_plane;
	int ret;

	if (!drm_core_check_feature(dev, DRIVER_MODESET))
		return -ENODEV;

	mutex_lock(&dev->mode_config.mutex);
	mutex_lock(&imxdrm->mutex);

	imx_plane = imx_drm_plane_by_id(dev, ga->plane_id);
	if (!imx_plane) {
		ret = -EINVAL;
		goto out_unlock;
	}

	helper = &imx_plane->imx_drm_helper_funcs;
	ret = helper->set_global_alpha(imx_plane->plane, ga);

out_unlock:
	mutex_unlock(&imxdrm->mutex);
	mutex_unlock(&dev->mode_config.mutex);
	return ret;
}

static int drm_imx_plane_get_chromakey_ioctl(struct drm_device *dev,
					     void *data,
					     struct drm_file *file_priv)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct drm_imx_chromakey *ck = data;
	struct imx_drm_plane_helper_funcs *helper;
	struct imx_drm_plane *imx_plane;
	int ret;

	if (!drm_core_check_feature(dev, DRIVER_MODESET))
		return -ENODEV;

	mutex_lock(&dev->mode_config.mutex);
	mutex_lock(&imxdrm->mutex);

	imx_plane = imx_drm_plane_by_id(dev, ck->plane_id);
	if (!imx_plane) {
		ret = -EINVAL;
		goto out_unlock;
	}

	helper = &imx_plane->imx_drm_helper_funcs;
	ret = helper->get_chromakey(imx_plane->plane, ck);

out_unlock:
	mutex_unlock(&imxdrm->mutex);
	mutex_unlock(&dev->mode_config.mutex);
	return ret;
}

static int drm_imx_plane_set_chromakey_ioctl(struct drm_device *dev,
					     void *data,
					     struct drm_file *file_priv)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct drm_imx_chromakey *ck = data;
	struct imx_drm_plane_helper_funcs *helper;
	struct imx_drm_plane *imx_plane;
	int ret;

	if (!drm_core_check_feature(dev, DRIVER_MODESET))
		return -ENODEV;

	mutex_lock(&dev->mode_config.mutex);
	mutex_lock(&imxdrm->mutex);

	imx_plane = imx_drm_plane_by_id(dev, ck->plane_id);
	if (!imx_plane) {
		ret = -EINVAL;
		goto out_unlock;
	}

	helper = &imx_plane->imx_drm_helper_funcs;
	ret = helper->set_chromakey(imx_plane->plane, ck);

out_unlock:
	mutex_unlock(&imxdrm->mutex);
	mutex_unlock(&dev->mode_config.mutex);
	return ret;
}

static int drm_imx_set_gamma_ioctl(struct drm_device *dev,
				   void *data,
				   struct drm_file *file_priv)
{
	struct imx_drm_device *imxdrm = __imx_drm_device();
	struct drm_imx_gamma *g = data;
	struct imx_drm_crtc_helper_funcs *helper;
	struct imx_drm_crtc *imx_crtc;
	int ret;

	if (!drm_core_check_feature(dev, DRIVER_MODESET))
		return -ENODEV;

	mutex_lock(&dev->mode_config.mutex);
	mutex_lock(&imxdrm->mutex);

	imx_crtc = imx_drm_crtc_by_id(dev, g->crtc_id);
	if (!imx_crtc) {
		ret = -EINVAL;
		goto out_unlock;
	}

	helper = &imx_crtc->imx_drm_helper_funcs;
	ret = helper->gamma_set(imx_crtc->crtc, g->enable, g->m, g->b);

out_unlock:
	mutex_unlock(&imxdrm->mutex);
	mutex_unlock(&dev->mode_config.mutex);
	return ret;
}

static struct drm_ioctl_desc imx_drm_ioctls[] = {
	DRM_IOCTL_DEF_DRV(IMX_GET_PHYS,         drm_imx_get_phys_ioctl,               DRM_AUTH),
	DRM_IOCTL_DEF_DRV(IMX_GET_GLOBAL_ALPHA, drm_imx_plane_get_global_alpha_ioctl, DRM_AUTH),
	DRM_IOCTL_DEF_DRV(IMX_SET_GLOBAL_ALPHA, drm_imx_plane_set_global_alpha_ioctl, DRM_AUTH),
	DRM_IOCTL_DEF_DRV(IMX_GET_CHROMAKEY,    drm_imx_plane_get_chromakey_ioctl,    DRM_AUTH),
	DRM_IOCTL_DEF_DRV(IMX_SET_CHROMAKEY,    drm_imx_plane_set_chromakey_ioctl,    DRM_AUTH),
	DRM_IOCTL_DEF_DRV(IMX_SET_GAMMA,        drm_imx_set_gamma_ioctl,              DRM_AUTH),
};

static struct drm_driver imx_drm_driver = {
	.driver_features	= DRIVER_MODESET | DRIVER_GEM,
	.load			= imx_drm_driver_load,
	.unload			= imx_drm_driver_unload,
	.firstopen		= imx_drm_driver_firstopen,
	.lastclose		= imx_drm_driver_lastclose,
	.gem_free_object	= drm_gem_cma_free_object,
	.gem_vm_ops		= &drm_gem_cma_vm_ops,
	.dumb_create		= drm_gem_cma_dumb_create,
	.dumb_map_offset	= drm_gem_cma_dumb_map_offset,
	.dumb_destroy		= drm_gem_cma_dumb_destroy,

	.get_vblank_counter	= drm_vblank_count,
	.enable_vblank		= imx_drm_enable_vblank,
	.disable_vblank		= imx_drm_disable_vblank,
	.ioctls			= imx_drm_ioctls,
	.num_ioctls		= ARRAY_SIZE(imx_drm_ioctls),
	.fops			= &imx_drm_driver_fops,
	.name			= "imx-drm",
	.desc			= "i.MX DRM graphics",
	.date			= "20120507",
	.major			= 1,
	.minor			= 0,
	.patchlevel		= 0,
};

static int imx_drm_platform_probe(struct platform_device *pdev)
{
	imx_drm_device->dev = &pdev->dev;

	return drm_platform_init(&imx_drm_driver, pdev);
}

static int imx_drm_platform_remove(struct platform_device *pdev)
{
	drm_platform_exit(&imx_drm_driver, pdev);

	return 0;
}

static struct platform_driver imx_drm_pdrv = {
	.probe		= imx_drm_platform_probe,
	.remove		= imx_drm_platform_remove,
	.driver		= {
		.owner	= THIS_MODULE,
		.name	= "imx-drm",
	},
};

static struct platform_device *imx_drm_pdev;

static int __init imx_drm_init(void)
{
	int ret;

	imx_drm_device = kzalloc(sizeof(*imx_drm_device), GFP_KERNEL);
	if (!imx_drm_device)
		return -ENOMEM;

	mutex_init(&imx_drm_device->mutex);
	INIT_LIST_HEAD(&imx_drm_device->crtc_list);
	INIT_LIST_HEAD(&imx_drm_device->connector_list);
	INIT_LIST_HEAD(&imx_drm_device->encoder_list);
	INIT_LIST_HEAD(&imx_drm_device->plane_list);

	imx_drm_pdev = platform_device_register_simple("imx-drm", -1, NULL, 0);
	if (!imx_drm_pdev) {
		ret = -EINVAL;
		goto err_pdev;
	}

	imx_drm_pdev->dev.coherent_dma_mask = DMA_BIT_MASK(32),

	ret = platform_driver_register(&imx_drm_pdrv);
	if (ret)
		goto err_pdrv;

	return 0;

err_pdrv:
	platform_device_unregister(imx_drm_pdev);
err_pdev:
	kfree(imx_drm_device);

	return ret;
}

static void __exit imx_drm_exit(void)
{
	platform_device_unregister(imx_drm_pdev);
	platform_driver_unregister(&imx_drm_pdrv);

	kfree(imx_drm_device);
}

module_init(imx_drm_init);
module_exit(imx_drm_exit);

MODULE_AUTHOR("Sascha Hauer <s.hauer@pengutronix.de>");
MODULE_DESCRIPTION("i.MX drm driver core");
MODULE_LICENSE("GPL");
