/*
 * drivers/gpio/gpio-inc.c
 *
 * GPIO port extender over INC
 *
 *   Copyright (c) 2012 Robert Bosch GmbH, Hildesheim
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/gpio.h>
#include <linux/slab.h>
#include <linux/irq.h>
#include <linux/irqdomain.h>
#include <linux/interrupt.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/inc.h>
#include <linux/inc_ports.h>
#include <linux/inc_scc_port_extender_gpio.h>
#include <linux/dgram_service.h>
#include <linux/kthread.h>

MODULE_DESCRIPTION("GPIO port extender for INC");
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Matthias Thomae <matthias.thomae@de.bosch.com>");

/*
 * for testing purposes the remote port may be set differently
 * from the local port
 */
static u16 remote_port = 0x0000;
module_param(remote_port, ushort, 0);
MODULE_PARM_DESC(remote_port, "INC remote port");

#define GPIO_INC_DRV_NAME		"gpio-inc"
#define GPIO_INC_STATUS_ACTIVE		0x01
#define GPIO_INC_STATUS_INACTIVE	0x02
#define GPIO_INC_RX_BUFSIZE		64
#define GPIO_INC_MAX_NGPIO		256
#define GPIO_INC_DGRAM_MAX		GPIO_INC_RX_BUFSIZE
#define GPIO_INC_PEER_RESPONSE_TIMEOUT	(1*(HZ)) /* 1 second */

struct gpio_inc {
	struct device		*dev;
	struct gpio_chip	chip;
	struct irq_domain	*irq_domain;
	struct task_struct	*rx_task;
	wait_queue_head_t	waitq;
	struct mutex		lock;
	struct socket		*sock;
	struct sk_dgram		*dgram;
	struct sockaddr_in	local;
	struct sockaddr_in	remote;
	int			is_active;
	int			err_stat;
	u8			cache[(GPIO_INC_MAX_NGPIO/BITS_PER_BYTE)*2];
	int			rsp_status_active;
	int			rsp_get_state;
	int			rx_flag[GPIO_INC_MAX_NGPIO];
};

static inline struct gpio_inc *to_gpio_inc(struct gpio_chip *chip)
{
	return container_of(chip, struct gpio_inc, chip);
}

static unsigned int inet_addr(char *str)
{
	int a, b, c, d;
	char arr[4];
	sscanf(str, "%d.%d.%d.%d", &a, &b, &c, &d);
	arr[0] = a; arr[1] = b; arr[2] = c; arr[3] = d;
	return *(unsigned int *) arr;
}

static int gpio_inc_get_dt_properties(struct gpio_inc *gpio_inc)
{
	struct device_node *gpio_node = gpio_inc->dev->of_node;
	struct device_node *inc_node;
	const char *prop_str, *inc_node_path;
	const void *prop;

	gpio_inc->local.sin_family = AF_INET;
	gpio_inc->remote.sin_family = AF_INET;

	prop_str = "node";
	prop = of_get_property(gpio_node, prop_str, NULL);
	if (prop)
		inc_node_path = (char *) prop;
	else
		goto err;

	inc_node = of_find_node_by_path(inc_node_path);
	if (!inc_node) {
		dev_err(gpio_inc->dev, "can't find %s node in device tree\n",
				inc_node_path);
		return -1;
	}

	prop_str = "local-addr";
	prop = of_get_property(inc_node, prop_str, NULL);
	if (prop)
		gpio_inc->local.sin_addr.s_addr = inet_addr((char *) prop);
	else
		goto err;

	prop_str = "remote-addr";
	prop = of_get_property(inc_node, prop_str, NULL);
	if (prop)
		gpio_inc->remote.sin_addr.s_addr = inet_addr((char *) prop);
	else
		goto err;

	return 0;

err:
	dev_err(gpio_inc->dev, "could not read dt property: %s\n", prop_str);
	return -1;
}

static int gpio_inc_irq_domain_map(struct irq_domain *domain,
		unsigned int virq, irq_hw_number_t hwirq)
{
	pr_debug("%s: linux irq %d, hw irq %lu\n", __func__, virq, hwirq);

	irq_set_chip(virq, &dummy_irq_chip);
	irq_set_nested_thread(virq, true);

	return 0;
}

static struct irq_domain_ops gpio_inc_irq_domain_ops = {
	.map	= gpio_inc_irq_domain_map,
};

static void gpio_inc_irq_domain_cleanup(struct gpio_inc *gpio_inc)
{
	int hwirq, virq;

	if (!gpio_inc->irq_domain)
		return;

	for (hwirq = 0; hwirq < gpio_inc->chip.ngpio; hwirq++) {
		virq = irq_find_mapping(gpio_inc->irq_domain, hwirq);
		if (virq > 0)
			dev_dbg(gpio_inc->dev,
					"%s: irq_dispose_mapping virq %d\n",
					__func__, virq);
			irq_dispose_mapping(virq);
	}

	dev_dbg(gpio_inc->dev, "%s: irq_domain_remove\n", __func__);
	irq_domain_remove(gpio_inc->irq_domain);
	gpio_inc->irq_domain = NULL;
}

static int gpio_inc_irq_domain_init(struct gpio_inc *gpio_inc)
{
	gpio_inc->irq_domain = irq_domain_add_linear(gpio_inc->dev->of_node,
			gpio_inc->chip.ngpio, &gpio_inc_irq_domain_ops, NULL);
	if (!gpio_inc->irq_domain)
		return -EINVAL;
	return 0;
}

static int gpio_inc_sock_open(struct gpio_inc *gpio_inc)
{
	struct socket *sock = NULL;
	struct sk_dgram *dgram;
	int ret;

	dev_dbg(gpio_inc->dev, "%s: local %pI4:%d remote %pI4:%d\n", __func__,
			&gpio_inc->local.sin_addr.s_addr,
			ntohs(gpio_inc->local.sin_port),
			&gpio_inc->remote.sin_addr.s_addr,
			ntohs(gpio_inc->remote.sin_port));

	ret = sock_create_kern(AF_INC, SOCK_STREAM, 0, &sock);
	if (ret < 0) {
		dev_err(gpio_inc->dev, "%s: sock_create_kern failed: %d\n",
				__func__, ret);
		goto out;
	}

	dgram = dgram_init(sock, GPIO_INC_DGRAM_MAX, NULL);
	if (dgram == NULL) {
		dev_err(gpio_inc->dev, "%s: dgram_init failed\n", __func__);
		goto out;
	}

	ret = kernel_bind(sock, (struct sockaddr *) &gpio_inc->local,
			sizeof(gpio_inc->local));
	if (ret < 0) {
		dev_err(gpio_inc->dev, "%s: kernel_bind failed: %d\n",
				__func__, ret);
		goto out_release;
	}

	ret = kernel_connect(sock, (struct sockaddr *) &gpio_inc->remote,
			sizeof(gpio_inc->remote), 0);
	if (ret < 0) {
		dev_err(gpio_inc->dev, "%s: kernel_connect failed: %d\n",
				__func__, ret);
		goto out_release;
	}

	gpio_inc->sock = sock;
	gpio_inc->dgram = dgram;

	return 0;

out_release:
	sock_release(sock);
out:
	return ret;
}

static void gpio_inc_sock_close(struct gpio_inc *gpio_inc)
{
	dev_dbg(gpio_inc->dev, "%s: local %pI4:%d remote %pI4:%d\n", __func__,
			&gpio_inc->local.sin_addr.s_addr,
			ntohs(gpio_inc->local.sin_port),
			&gpio_inc->remote.sin_addr.s_addr,
			ntohs(gpio_inc->remote.sin_port));

	if (gpio_inc->dgram) {
		dev_dbg(gpio_inc->dev, "%s: dgram_exit\n", __func__);
		dgram_exit(gpio_inc->dgram);
	}

	if (gpio_inc->sock) {
		dev_dbg(gpio_inc->dev, "%s: sock_shutdown\n", __func__);
		kernel_sock_shutdown(gpio_inc->sock, SHUT_RDWR);
	}

	if (gpio_inc->sock) {
		dev_dbg(gpio_inc->dev, "%s: sock_release\n", __func__);
		sock_release(gpio_inc->sock);
	}

	return;
}

static int gpio_inc_sendmsg(struct gpio_inc *gpio_inc, u8 *data, int len)
{
	int ret, i;

	dev_dbg(gpio_inc->dev, "%s: dgram_send to %pI4:%d\n", __func__,
			&gpio_inc->remote.sin_addr.s_addr,
			ntohs(gpio_inc->remote.sin_port));

	ret = dgram_send(gpio_inc->dgram, data, len);
	if (ret < 0)
		dev_alert(gpio_inc->dev, "%s: could not send msg: %d\n",
				__func__, ret);

	for (i = 0; i < len; i++)
		dev_dbg(gpio_inc->dev, "%s: data[%d]=0x%02X\n", __func__,
				i, ((u8 *)data)[i]);

	return ret;
}

static int gpio_inc_recvmsg(struct gpio_inc *gpio_inc, void *data, int len)
{
	int ret, i;

	do {
		ret = dgram_recv(gpio_inc->dgram, data, len);
		if ((ret < 0) && (ret != -EAGAIN))
			dev_alert(gpio_inc->dev, "%s: dgram_recv failed: %d\n",
					__func__, ret);

		dev_dbg(gpio_inc->dev, "%s: dgram_recv: ret=%d\n",
				__func__, ret);
		for (i = 0; i < ret; i++)
			dev_dbg(gpio_inc->dev, "%s: data[%d]=0x%02X\n",
					__func__, i, ((u8 *)data)[i]);
	} while (ret == -EAGAIN);

	return ret;
}

static int gpio_inc_cmd_status_active(struct gpio_inc *gpio_inc)
{
	u8 data[2] = {SCC_PORT_EXTENDER_GPIO_C_COMPONENT_STATUS_MSGID,
			GPIO_INC_STATUS_ACTIVE};

	return gpio_inc_sendmsg(gpio_inc, data, 2);
}

static int gpio_inc_cmd_get_state(struct gpio_inc *gpio_inc)
{
	u8 data[1] = {SCC_PORT_EXTENDER_GPIO_C_GET_STATE_MSGID};

	return gpio_inc_sendmsg(gpio_inc, data, 1);
}

static int gpio_inc_cmd_set_state(struct gpio_inc *gpio_inc,
		unsigned offset, int value)
{
	u8 data[3] = {SCC_PORT_EXTENDER_GPIO_C_SET_STATE_MSGID,
			offset, value};

	return gpio_inc_sendmsg(gpio_inc, data, 3);
}

static int gpio_inc_to_irq(struct gpio_chip *chip, unsigned offset)
{
	struct gpio_inc *gpio_inc = to_gpio_inc(chip);

	dev_dbg(gpio_inc->dev, "%s: gpio%d\n", __func__, chip->base + offset);

	if (gpio_inc->err_stat) {
		dev_err(gpio_inc->dev, "%s: gpio%d: driver error status %d\n",
				__func__, chip->base + offset,
				gpio_inc->err_stat);
		return gpio_inc->err_stat;
	}

	if (gpio_inc->irq_domain && offset < gpio_inc->chip.ngpio)
		return irq_create_mapping(gpio_inc->irq_domain, offset);
	else
		return -ENXIO;
}

static int gpio_inc_trigger_irq(struct gpio_inc *gpio_inc, unsigned offset,
		int status)
{
	int irq = gpio_inc_to_irq(&gpio_inc->chip, offset);
	struct irq_desc *desc = irq_to_desc(irq);
	struct irqaction *action = desc->action;
	unsigned long flags;
	int ret = 0;

	if (!action)
		return 0;

	flags = action->flags & IRQF_TRIGGER_MASK;
	if ((status && IRQF_TRIGGER_RISING & flags)
			|| (!status && IRQF_TRIGGER_FALLING & flags)) {
		dev_dbg(gpio_inc->dev, "%s: handle irq %d\n", __func__, irq);
		handle_nested_irq(irq);
	}

	return ret;
}

static int gpio_inc_cache_update(struct gpio_inc *gpio_inc,
		u8 ngpio, u8 *state, int msg_size)
{
	u8 pos, pos_stat, pos_dir, gpio_mod;
	unsigned long bits_mod, cache, status, direction;
	int bit, bits_left, nbits, ret;
	u8 ngpio_bytes = ngpio / BITS_PER_BYTE +
			(ngpio % BITS_PER_BYTE > 0 ? 1 : 0);

	if (ngpio > GPIO_INC_MAX_NGPIO) {
		dev_alert(gpio_inc->dev, "%s: ngpio exceeds cache size: %d>%d\n",
				__func__, ngpio, GPIO_INC_MAX_NGPIO);
		return -EPROTO;
	}

	/* TBD: change this back to
	 * msg_size != 2 + ngpio_bytes * 2
	 * once msg size is configurable on peer
	 */
	if (msg_size < 2 + ngpio_bytes * 2) {
		dev_alert(gpio_inc->dev, "%s: mismatch: ngpio%d msg size %d\n",
				__func__, ngpio, msg_size);
		return -EPROTO;
	}

	if (!gpio_inc->is_active) {
		memcpy(gpio_inc->cache, state, ngpio_bytes * 2);
		return 0;
	}

	if (ngpio != gpio_inc->chip.ngpio) {
		dev_alert(gpio_inc->dev, "%s: ngpio modified: %d != %d\n",
				__func__, ngpio, gpio_inc->chip.ngpio);
		return -EPROTO;
	}

	bits_left = ngpio;
	for (pos = 0; pos < ngpio_bytes; pos++) {
		pos_stat = pos * 2;
		pos_dir = pos_stat + 1;
		nbits = bits_left > BITS_PER_BYTE - 1 ?
				BITS_PER_BYTE : bits_left;
		bits_left -= BITS_PER_BYTE;

		if (state[pos_dir] != gpio_inc->cache[pos_dir]) {
			dev_alert(gpio_inc->dev, "%s: direction modified: %d\n",
					__func__, pos);
			return -EPROTO;
		}

		cache = gpio_inc->cache[pos_stat];
		status = state[pos_stat];

		if (status == cache)
			continue;

		gpio_inc->cache[pos_stat] = status;
		bits_mod = status ^ cache;
		direction = gpio_inc->cache[pos_dir];

		for (bit = 0; bit < nbits; bit++) {
			if (test_bit(bit, &direction))
				continue; /* direction output -> ignore */

			if (test_bit(bit, &bits_mod)) {
				gpio_mod = bit + pos * BITS_PER_BYTE;
				dev_dbg(gpio_inc->dev, "%s: bit %d gpio %d\n",
						__func__, bit, gpio_mod);
				ret = gpio_inc_trigger_irq(gpio_inc, gpio_mod,
						test_bit(bit, &status));
				if (ret)
					return ret;
			}
		}
	}

	return 0;
}

static int gpio_inc_cache_get_value(struct gpio_inc *gpio_inc,
		unsigned offset)
{
	unsigned offset_byte = 2 * (offset / BITS_PER_BYTE);
	unsigned offset_bit  = offset % BITS_PER_BYTE;
	unsigned long cache = gpio_inc->cache[offset_byte];

	dev_dbg(gpio_inc->dev, "%s: gpio%d\n", __func__,
			gpio_inc->chip.base + offset);

	return test_bit(offset_bit, &cache);
}

static int gpio_inc_cache_set_value(struct gpio_inc *gpio_inc,
		unsigned offset, int value)
{
	unsigned offset_byte = 2 * (offset / BITS_PER_BYTE);
	unsigned offset_bit  = offset % BITS_PER_BYTE;

	dev_dbg(gpio_inc->dev, "%s: gpio%d\n", __func__,
			gpio_inc->chip.base + offset);

	/*
	 * no need to protect cache as long as it is only written from rx_task
	 * via gpio_inc_cache_update and gpio_inc_cache_set_value
	 * and hence all access is serialized
	 */
	if (value)
		gpio_inc->cache[offset_byte] |= 1U << offset_bit;
	else
		gpio_inc->cache[offset_byte] &= ~(1U << offset_bit);

	return 0;
}

static int gpio_inc_cache_get_direction(struct gpio_inc *gpio_inc,
		unsigned offset)
{
	unsigned offset_byte = 2 * (offset / BITS_PER_BYTE) + 1;
	unsigned offset_bit  = offset % BITS_PER_BYTE;
	unsigned long cache = gpio_inc->cache[offset_byte];

	dev_dbg(gpio_inc->dev, "%s: gpio%d\n", __func__,
			gpio_inc->chip.base + offset);

	return test_bit(offset_bit, &cache);
}

static int gpio_inc_set_state(struct gpio_inc *gpio_inc, unsigned offset,
		int value)
{
	int ret;

	dev_dbg(gpio_inc->dev, "%s: gpio%d\n", __func__,
			gpio_inc->chip.base + offset);

	gpio_inc->rx_flag[offset] = false;

	ret = gpio_inc_cmd_set_state(gpio_inc, offset, value);
	if (ret < 0)
		return ret;

	ret = wait_event_interruptible_timeout(gpio_inc->waitq,
			gpio_inc->rx_flag[offset],
			GPIO_INC_PEER_RESPONSE_TIMEOUT);

	gpio_inc->rx_flag[offset] = false;

	if (ret < 0)
		return ret;
	if (ret == 0) {
		dev_alert(gpio_inc->dev, "%s: gpio%d: peer not responding\n",
				__func__, gpio_inc->chip.base + offset);
		return -ETIME;
	}

	return 0;
}

static int gpio_inc_direction_input(struct gpio_chip *chip, unsigned offset)
{
	struct gpio_inc *gpio_inc = to_gpio_inc(chip);

	dev_dbg(gpio_inc->dev, "%s: gpio%d\n", __func__,
			chip->base + offset);

	if (gpio_inc->err_stat) {
		dev_err(gpio_inc->dev, "%s: gpio%d: driver error status %d\n",
				__func__, chip->base + offset,
				gpio_inc->err_stat);
		return gpio_inc->err_stat;
	}

	/* direction change is not supported (yet) */
	if (gpio_inc_cache_get_direction(gpio_inc, offset)) {
		dev_err(gpio_inc->dev, "%s: gpio%d: change not supported\n",
				__func__, chip->base + offset);
		return -EINVAL;
	}

	return 0;
}

static int gpio_inc_get(struct gpio_chip *chip, unsigned offset)
{
	struct gpio_inc *gpio_inc = to_gpio_inc(chip);

	dev_dbg(gpio_inc->dev, "%s: gpio%d\n", __func__, chip->base + offset);

	if (gpio_inc->err_stat) {
		dev_err(gpio_inc->dev, "%s: gpio%d: driver error status %d\n",
				__func__, chip->base + offset,
				gpio_inc->err_stat);
		return 0;
	}

	return gpio_inc_cache_get_value(gpio_inc, offset);
}

static int gpio_inc_direction_output(struct gpio_chip *chip, unsigned offset,
		int value)
{
	int ret = 0;
	struct gpio_inc *gpio_inc = to_gpio_inc(chip);

	dev_dbg(gpio_inc->dev, "%s: gpio%d: %d\n", __func__,
			chip->base + offset, value);

	if (gpio_inc->err_stat) {
		dev_err(gpio_inc->dev, "%s: gpio%d: driver error status %d\n",
				__func__, chip->base + offset,
				gpio_inc->err_stat);
		return gpio_inc->err_stat;
	}

	/* direction change is not supported (yet) */
	if (!gpio_inc_cache_get_direction(gpio_inc, offset)) {
		dev_err(gpio_inc->dev, "%s: gpio%d: change not supported\n",
				__func__, chip->base + offset);
		return -EINVAL;
	}

	ret = gpio_inc_set_state(gpio_inc, offset, value);

	return ret;
}

static void gpio_inc_set(struct gpio_chip *chip, unsigned offset, int value)
{
	struct gpio_inc *gpio_inc = to_gpio_inc(chip);

	dev_dbg(gpio_inc->dev, "%s: gpio%d value %d\n", __func__,
			chip->base + offset, value);

	if (gpio_inc->err_stat) {
		dev_err(gpio_inc->dev, "%s: gpio%d: driver error status %d\n",
				__func__, chip->base + offset,
				gpio_inc->err_stat);
		return;
	}

	if (!gpio_inc_cache_get_direction(gpio_inc, offset)) {
		dev_err(gpio_inc->dev, "%s: gpio%d: input mode\n",
				__func__, chip->base + offset);
		return;
	}

	gpio_inc_set_state(gpio_inc, offset, value);

	return;
}

static int gpio_inc_activate(struct gpio_inc *gpio_inc)
{
	struct gpio_chip *chip;
	int ret;

	chip = &gpio_inc->chip;
	chip->label		= dev_name(gpio_inc->dev);
	chip->dev		= gpio_inc->dev;
	chip->owner		= THIS_MODULE;
	chip->direction_input	= gpio_inc_direction_input;
	chip->get		= gpio_inc_get;
	chip->direction_output	= gpio_inc_direction_output;
	chip->set		= gpio_inc_set;
	chip->to_irq		= gpio_inc_to_irq;
	chip->base		= -1;
	chip->can_sleep		= 1;

	ret = gpio_inc_irq_domain_init(gpio_inc);
	if (ret) {
		dev_err(gpio_inc->dev, "%s: could not init irqdomain: %d\n",
				__func__, ret);
		return ret;
	}

	ret = gpiochip_add(chip);
	if (ret < 0) {
		dev_err(gpio_inc->dev, "%s: could not register gpiochip: %d\n",
				__func__, ret);
		return ret;
	}

	gpio_inc->is_active = 1;

	dev_info(gpio_inc->dev, "%s: %d GPIOs\n", __func__, chip->ngpio);

	return 0;
}

static int gpio_inc_handle_msg(struct gpio_inc *gpio_inc, u8 *data, int size)
{
	int ret = 0;
	struct device *dev = gpio_inc->dev;

	if (size <= 0) {
		dev_err(dev, "%s: invalid msg size: %d\n",
				__func__, size);
		return -EPROTO;
	}

	/* before activation discard all non-activation messages */
	if (data[0] != SCC_PORT_EXTENDER_GPIO_R_COMPONENT_STATUS_MSGID &&
	    !gpio_inc->rsp_status_active) {
		dev_dbg(dev, "%s: discarding msgid: 0x%02X\n",
			__func__, data[0]);
		return 0;
	}

	switch (data[0]) {
	case SCC_PORT_EXTENDER_GPIO_R_COMPONENT_STATUS_MSGID:
		if (size != 2) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
					__func__, data[0], size);
			return -EPROTO;
		}
		if (data[1] != GPIO_INC_STATUS_ACTIVE) {
			dev_alert(dev, "%s: invalid status: %d\n",
					__func__, data[1]);
			return -EPROTO;
		}
		gpio_inc->rsp_status_active = true;
		wake_up_interruptible(&gpio_inc->waitq);
		break;
	case SCC_PORT_EXTENDER_GPIO_R_GET_STATE_MSGID:
		if (size < 2) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
					__func__, data[0], size);
			return -EPROTO;
		}
		if (data[1] == 0) {
			dev_alert(dev, "%s: invalid number of gpios: %d\n",
					__func__, data[1]);
			return -EPROTO;
		}

		ret = gpio_inc_cache_update(gpio_inc, data[1], &data[2], size);
		if (ret)
			break;
		if (!gpio_inc->is_active) {
			gpio_inc->chip.ngpio = data[1];
			gpio_inc->rsp_get_state = true;
			wake_up_interruptible(&gpio_inc->waitq);
		}
		break;
	case SCC_PORT_EXTENDER_GPIO_R_SET_STATE_MSGID:
		if (!gpio_inc->is_active) {
			dev_alert(dev, "%s: inactive: 0x%02X %d\n",
					__func__, data[0], size);
			return -EPROTO;
		}
		if (size != 3) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
					__func__, data[0], size);
			return -EPROTO;
		}
		if (data[1] > gpio_inc->chip.ngpio) {
			dev_alert(dev, "%s: invalid gpio id: %d\n",
					__func__, data[1]);
			return -EPROTO;
		}
		if (data[2] > 1) {
			dev_alert(dev, "%s: invalid value: %d\n",
					__func__, data[2]);
			return -EPROTO;
		}

		ret = gpio_inc_cache_set_value(gpio_inc, data[1], data[2]);
		if (ret)
			break;

		gpio_inc->rx_flag[data[1]] = true;
		wake_up_interruptible(&gpio_inc->waitq);
		break;
	case SCC_PORT_EXTENDER_GPIO_R_REJECT_MSGID:
		if (size != 3) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
					__func__, data[0], size);
			return -EPROTO;
		}
		dev_alert(dev, "%s: msg rejected: id 0x%02X reason 0x%02X\n",
				__func__, data[2], data[1]);
		return -EPROTO;
	default:
		dev_alert(dev, "%s: invalid msgid: 0x%02X\n",
				__func__, data[0]);
		return -EPROTO;
	}

	return ret;
}

static int gpio_inc_rx_task(void *arg)
{
	u8 data[GPIO_INC_RX_BUFSIZE];
	int ret;

	struct gpio_inc *gpio_inc = arg;

	dev_dbg(gpio_inc->dev, "%s: enter\n", __func__);

	do {
		ret = gpio_inc_recvmsg(gpio_inc, data, GPIO_INC_RX_BUFSIZE);
		if (ret < 0)
			break;

		if (!gpio_inc->err_stat)
			gpio_inc->err_stat =
				gpio_inc_handle_msg(gpio_inc, data, ret);

		/* in error case wake up pending system calls */
		if (gpio_inc->err_stat && gpio_inc->is_active)
			wake_up_interruptible(&gpio_inc->waitq);
	} while (1);

	dev_dbg(gpio_inc->dev, "%s: exit: %d\n", __func__, ret);

	mutex_lock(&gpio_inc->lock);
	gpio_inc->rx_task = NULL;
	mutex_unlock(&gpio_inc->lock);

	return 0;
}

static int gpio_inc_remove(struct platform_device *pdev)
{
	struct gpio_inc *gpio_inc = platform_get_drvdata(pdev);
	struct device *dev = &pdev->dev;
	int ret = 0;

	if (!gpio_inc)
		goto out;

	dev_dbg(dev, "%s: base %d\n", __func__, gpio_inc->chip.base);

	if (gpio_inc->is_active) {
		ret = gpiochip_remove(&gpio_inc->chip);
		if (ret < 0) {
			dev_err(dev, "%s: could not remove gpiochip: %d\n",
					__func__, ret);
			goto out;
		}

		gpio_inc_irq_domain_cleanup(gpio_inc);
	}

	mutex_lock(&gpio_inc->lock);
	if (gpio_inc->rx_task)
		force_sig(SIGKILL, gpio_inc->rx_task);
	else
		dev_dbg(dev, "%s: rx_task already terminated\n",
				__func__);
	mutex_unlock(&gpio_inc->lock);

	if (gpio_inc->rx_task)
		kthread_stop(gpio_inc->rx_task);

	gpio_inc_sock_close(gpio_inc);

	platform_set_drvdata(pdev, NULL);
	kfree(gpio_inc);

out:
	return ret;
}

static const struct of_device_id inc_gpiochip_match[] = {
	{ .compatible = "rbcm,gpio-inc", },
	{}
};

static int gpio_inc_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct gpio_inc *gpio_inc;
	struct task_struct *task;
	int ret;
	const struct of_device_id *id =
			of_match_device(inc_gpiochip_match, dev);

	if (!id) {
		dev_err(dev, "%s: failed to find gpio controller\n", __func__);
		ret = -EFAULT;
		goto out;
	}

	dev_info(dev, "%s: compatible: %s\n", __func__, id->compatible);

	gpio_inc = kzalloc(sizeof(*gpio_inc), GFP_KERNEL);
	if (!gpio_inc) {
		ret = -ENOMEM;
		goto out;
	}

	gpio_inc->dev = dev;
	gpio_inc->irq_domain = NULL;
	gpio_inc->rx_task = NULL;
	gpio_inc->sock = NULL;
	gpio_inc->dgram = NULL;
	gpio_inc->is_active = 0;
	gpio_inc->err_stat = 0;
	gpio_inc->rsp_status_active = false;
	gpio_inc->rsp_get_state = false;

	gpio_inc->local.sin_port = htons(PORT_EXTENDER_GPIO_PORT);
	if (remote_port) {
		gpio_inc->remote.sin_port = htons(remote_port);
		dev_dbg(gpio_inc->dev, "%s: set remote_port %d\n",
				__func__, remote_port);
	} else {
		gpio_inc->remote.sin_port = gpio_inc->local.sin_port;
	}

	mutex_init(&gpio_inc->lock);

	init_waitqueue_head(&gpio_inc->waitq);

	platform_set_drvdata(pdev, gpio_inc);

	ret = gpio_inc_get_dt_properties(gpio_inc);
	if (ret < 0)
		goto out_remove;

	ret = gpio_inc_sock_open(gpio_inc);
	if (ret < 0)
		goto out_remove;

	/* create and start thread */
	task = kthread_run(gpio_inc_rx_task, gpio_inc, GPIO_INC_DRV_NAME);
	if (IS_ERR(task)) {
		ret = PTR_ERR(task);
		dev_err(dev, "%s: could not run thread: %d\n", __func__, ret);
		goto out_remove;
	}
	gpio_inc->rx_task = task;

	ret = gpio_inc_cmd_status_active(gpio_inc);
	if (ret < 0)
		goto out_remove;

	ret = wait_event_interruptible_timeout(gpio_inc->waitq,
			gpio_inc->rsp_status_active,
			GPIO_INC_PEER_RESPONSE_TIMEOUT);
	if (ret < 0)
		goto out_remove;
	if (ret == 0) {
		dev_alert(gpio_inc->dev,
				"%s: peer not responding to cmd_status_active",
				__func__);
		ret = -ETIME;
		goto out_remove;
	}

	ret = gpio_inc_cmd_get_state(gpio_inc);
	if (ret < 0)
		goto out_remove;

	ret = wait_event_interruptible_timeout(gpio_inc->waitq,
			gpio_inc->rsp_get_state,
			GPIO_INC_PEER_RESPONSE_TIMEOUT);
	if (ret < 0)
		goto out_remove;
	if (ret == 0) {
		dev_alert(gpio_inc->dev,
				"%s: peer not responding to cmd_get_state",
				__func__);
		ret = -ETIME;
		goto out_remove;
	}

	ret = gpio_inc_activate(gpio_inc);
	if (ret < 0)
		goto out_remove;

	return 0;

out_remove:
	gpio_inc_remove(pdev);
out:
	platform_set_drvdata(pdev, NULL);
	return ret;
}

static struct platform_driver gpio_inc_driver = {
	.driver = {
		.name	= GPIO_INC_DRV_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = inc_gpiochip_match,
	},
	.probe		= gpio_inc_probe,
	.remove		= gpio_inc_remove,
};

static int __init gpio_inc_init(void)
{
	pr_debug("%s: enter\n", __func__);

	return platform_driver_register(&gpio_inc_driver);
}
late_initcall(gpio_inc_init);

static void __exit gpio_inc_exit(void)
{
	platform_driver_unregister(&gpio_inc_driver);
}
module_exit(gpio_inc_exit);
