/*
 * Author: Adnan Ali<adnan.ali@codethink.co.uk>
 * This file is licensed under the GPL2 license.
 *
 *#############################################################################
 * GPL
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 *#############################################################################
 */

#include <linux/spinlock.h>
#include <linux/cred.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include "compat.h"
#include "intern.h"
#include "desc.h"
#include "error.h"
#include "jr.h"
#include "sm.h"
#include "desc_constr.h"

#define KSM "keyman"
static dev_t ksm_dev_number;
static struct cdev *ksm_object;
struct class *ksm_class;
#define UNIT_BANK	1
static u8 skeymod[] = {
	0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08,
	0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00
};

static struct device *gksdev;
static struct keyidmapper *gkeyidmapper;
static DEFINE_SPINLOCK(maptable_lock);

/* encrypt/decrypt desc */
static void mk_job_desc(u32 *desc, dma_addr_t key, u16 keysz, dma_addr_t indata,
		       dma_addr_t outdata, u16 sz, u32 cipherdir, u32 keymode)
{
	desc[0] = CMD_DESC_HDR | HDR_ONE | (8 & HDR_DESCLEN_MASK);
	desc[1] = CMD_KEY | CLASS_1 | (keysz & KEY_LENGTH_MASK) | keymode;
	desc[2] = (u32)key;
	desc[3] = CMD_OPERATION | OP_TYPE_CLASS1_ALG | OP_ALG_AAI_ECB |
		  cipherdir;
	desc[4] = CMD_FIFO_LOAD | FIFOLD_CLASS_CLASS1 |
		  FIFOLD_TYPE_MSG | FIFOLD_TYPE_LAST1 | sz;
	desc[5] = (u32)indata;
	desc[6] = CMD_FIFO_STORE | FIFOST_TYPE_MESSAGE_DATA | sz;
	desc[7] = (u32)outdata;
}

/*Hashing functionality*/
int mk_hash_op_desc(u32 *desc, u32 alg, dma_addr_t blob, u32 bblen,
			dma_addr_t hashbuff, u32 hlen)
{
	init_job_desc(desc, 0);
	append_operation(desc, alg);
	/* load data into fifo-in stack */
	append_fifo_load(desc, (dma_addr_t)blob, bblen,
		CLASS_2+FIFOLD_TYPE_MSG + FIFOLD_TYPE_LAST2);
	append_jump(desc, JUMP_CLASS_CLASS2 + 1);
	/* store contents of context register class 2 into result */
	append_store(desc, (dma_addr_t)hashbuff,
		hlen, CLASS_2 + LDST_SRCDST_BYTE_CONTEXT);
	return 0;
}

struct exec_test_result {
	int error;
	struct completion completion;
};

static void exec_operation_done(struct device *dev, u32 *desc,
					u32 err, void *context)
{
	struct exec_test_result *res = context;

	if (err) {
		char tmp[CAAM_ERROR_STR_MAX];
		dev_err(dev, "caam_sm_test ERROR %08x: %s\n",
				err, caam_jr_strstatus(tmp, err));
	}

	res->error = err;
	complete(&res->completion);
}

static int exec_test_job(struct device *ksdev, u32 *jobdesc)
{
	struct exec_test_result testres;
	struct caam_drv_private_sm *kspriv;
	int rtn = 0;

	kspriv = dev_get_drvdata(ksdev);

	init_completion(&testres.completion);

	rtn = caam_jr_enqueue(kspriv->smringdev, jobdesc, exec_operation_done,
			      &testres);
	if (!rtn) {
		wait_for_completion_interruptible(&testres.completion);
		rtn = testres.error;
	}
	return rtn;
}

/* get the slot of key id passed from mapper table */
static int getslot_from_table(u32 kid)
{
	int i, ret = -ENOKEY;

	spin_lock(&maptable_lock);

	for (i = 0; i < KID_TABLE_LEN; i++) {
		if (gkeyidmapper[i].kid == kid && gkeyidmapper[i].state == 1) {
			ret = gkeyidmapper[i].slot;
			break;
		}
	}

	spin_unlock(&maptable_lock);

	return ret;
}

/* write slot to key id mapping to table */
static int write_slot_to_table(u32 kid, u32 slot)
{
	int i, ret = -ENOSPC;

	spin_lock(&maptable_lock);

	for (i = 0; i < KID_TABLE_LEN; i++) {
		if (gkeyidmapper[i].state == 0) {
			gkeyidmapper[i].slot = slot;
			gkeyidmapper[i].kid = kid;
			gkeyidmapper[i].state = 1;
			ret = 0;
			break;
		}
	}

	spin_unlock(&maptable_lock);

	return ret;
}

/* release slot from mapping table */
static int remove_slot_to_table(u32 kid, u32 slot)
{
	int i, ret = -ENOKEY;

	spin_lock(&maptable_lock);

	for (i = 0; i < KID_TABLE_LEN; i++) {
		if (gkeyidmapper[i].kid == kid && gkeyidmapper[i].state == 1) {
			gkeyidmapper[i].state = 0;
			ret = 0;
			break;
		}
	}

	spin_unlock(&maptable_lock);

	return ret;
}

/*API to read key for operation
 *This function needs to replaced
 *with address of slot that holds the key
 */
u8 *caam_kha_readkey(u32 keyid)
{
	int ret;
	u8 blob[KEY32_BB_SIZE];
	u8 *key;

	key = kzalloc(KEY32_BB_SIZE, GFP_KERNEL);
	if (!key)
		goto fail;

	/* Read key that will be use to decrypt */
	ret = caam_sm_ksm_readblob(keyid, blob);
	if (ret)
		goto fail_free;

	/* But key is in black blob format so unwrap */
	ret = caam_sm_ksm_unprepareblob(blob);
	if (ret)
		goto fail_free;

	memcpy(key, blob, KEY32_BB_SIZE);

	return key;

fail_free:
	kfree(key);
fail:
	return NULL;
}
EXPORT_SYMBOL(caam_kha_readkey);


/* Descrypt encrypted key */
int caam_create_deckey(struct skey_prepare *keyprepare, u8 *keybuff)
{
	int rtnval = 0;
	u32 __iomem *jdesc, jstat = 0;
	dma_addr_t enckey_dma, deckey_dma, key_dma;
	struct device *dev = gksdev;
	u8 *key = NULL;

	pr_debug("decrypt key function\n");

	if (keyprepare->in_len != KEY_SIZE) {
		rtnval = -EOPNOTSUPP;
		goto fail;
	}

	/* Allocate storage for job descriptor */
	jdesc = kzalloc(9 * sizeof(u32), GFP_KERNEL | GFP_DMA);
	if (!jdesc) {
		rtnval = -ENOMEM;
		goto fail;
	}

	/*Read key that will be use to decrypt*/
	key = caam_kha_readkey(keyprepare->enc_keyid);
	if (!key) {
		rtnval = -ENOKEY;
		goto failmem;
	}

	/*We have the red key now*/
	rtnval = copy_from_user(keybuff, keyprepare->in_buf,
							keyprepare->in_len);
	if (rtnval)
		goto failmem;

	enckey_dma = dma_map_single(dev, keybuff, keyprepare->in_len,
					      DMA_TO_DEVICE);

	dma_sync_single_for_device(dev, enckey_dma, KEY_SIZE, DMA_TO_DEVICE);
	key_dma = dma_map_single(dev, key, KEY_SIZE,
					      DMA_TO_DEVICE);
	dma_sync_single_for_device(dev, key_dma, KEY_SIZE, DMA_TO_DEVICE);

	deckey_dma = dma_map_single(dev, keybuff, KEY_SIZE,
					      DMA_FROM_DEVICE);
	/*Make decrypt job desc*/
	mk_job_desc(jdesc, key_dma, KEY_SIZE, enckey_dma,
		       deckey_dma, KEY_SIZE,
			(OP_ALG_DECRYPT | OP_ALG_ALGSEL_AES), 0);

	/*Now hand job over to CAAM and wait for it to finish*/
	jstat = exec_test_job(dev, jdesc);
	if (jstat != 0) {
		rtnval = -EFAULT;
		goto faildma;
	}

	/*Read the red key back*/
	dma_sync_single_for_cpu(dev, deckey_dma, KEY_SIZE, DMA_FROM_DEVICE);

faildma:
	dma_unmap_single(dev, enckey_dma, KEY_SIZE, DMA_TO_DEVICE);
	dma_unmap_single(dev, key_dma, KEY_SIZE, DMA_TO_DEVICE);
	dma_unmap_single(dev, deckey_dma, KEY_SIZE, DMA_FROM_DEVICE);
failmem:
	/*Free desc memory*/
	kfree(key);
	kfree(jdesc);
fail:
	return rtnval;
}
int caam_create_hash(u8 *bbk, u32 len, u8 *hash)
{
	u32 __iomem *jdesc, jstat = 0;
	int rtnval = 0;
	dma_addr_t black_blob_dma, hash_dma;
	u32 alg = ((OP_TYPE_CLASS2_ALG + OP_ALG_ALGSEL_SHA256) |
			OP_ALG_AS_INITFINAL | OP_ALG_AAI_HASH);
	struct device *dev = gksdev;

	/* Allocate storage for job descriptor */
	jdesc = kzalloc(9 * sizeof(u32), GFP_KERNEL | GFP_DMA);
	if (jdesc == NULL) {
		rtnval = -ENOMEM;
		goto fail;
	}
	/* dma load key black blob to be hashed */
	black_blob_dma = dma_map_single(dev, bbk, len,
					      DMA_TO_DEVICE);
	dma_sync_single_for_device(dev, black_blob_dma, len,
				   DMA_TO_DEVICE);
	/* dma the output buffer where we will get hash*/
	hash_dma = dma_map_single(dev, hash, KEY_SIZE, DMA_FROM_DEVICE);

	/* Make hash descriptor */
	mk_hash_op_desc(jdesc, alg, black_blob_dma, len, hash_dma, KEY_SIZE);

	/* Execute hash job */
	jstat = exec_test_job(dev, jdesc);
	if (jstat != 0) {
		rtnval = -EFAULT;
		goto faildma;
	}

#ifdef SM_DEBUG
	for (i = 0; i < 9; i++)
		dev_info(dev, "%08x ", jdesc[i]);
	dev_info(dev, "\n");
#endif
	/* Get hash from caam */
	dma_sync_single_for_cpu(dev, hash_dma, KEY_SIZE, DMA_FROM_DEVICE);

faildma:
	/* unmap the dma handles */
	dma_unmap_single(dev, hash_dma, KEY_SIZE, DMA_FROM_DEVICE);
	dma_unmap_single(dev, black_blob_dma, len, DMA_TO_DEVICE);

fail:
	kfree(jdesc);
	return rtnval;
}
EXPORT_SYMBOL(caam_create_hash);
/*Hash functions ends*/

int caam_sm_ksm_init(struct platform_device *pdev)
{
	struct device *ctrldev, *ksdev;
	struct caam_drv_private *ctrlpriv;
	struct caam_drv_private_sm *kspriv;
	int units, ret;

	ctrldev = &pdev->dev;
	ctrlpriv = dev_get_drvdata(ctrldev);
	ksdev = ctrlpriv->smdev;
	kspriv = dev_get_drvdata(ksdev);
	if (kspriv == NULL)
		return -ENODEV;

	/* Now that we have the dev for the single SM instance, connect */
#ifdef SM_TEST_DETAIL
	dev_info(ksdev, "caam_sm_test_init() running\n");
#endif
	/* Probe to see what keystores are available to us */
	units = sm_detect_keystore_units(ksdev);
#ifdef SM_DEBUG
	dev_info(ksdev, "sm logical pages %d\n", units);
#endif
	if (!units)
		dev_err(ksdev, "caam_sm_test: no keystore units available\n");
	/*
	* MX6 bootloader stores some stuff in unit 0, so let's
	* use 1 or above
	*/
	if (units < 2) {
		dev_err(ksdev, "caam_sm_test: insufficient keystore units\n");
		return -ENODEV;
	}

#ifdef SM_TEST_DETAIL
	dev_info(ksdev, "caam_sm_test: %d keystore units available\n", units);
#endif

	/* Initialize/Establish Keystore */
	sm_establish_keystore(ksdev, UNIT_BANK);     /* Initalize store in #1*/

	/* Allocate memory for key id to slot mapping */
	gkeyidmapper = kmalloc(KID_TABLE_LEN * sizeof(struct keyidmapper),
				GFP_KERNEL);
	if (!gkeyidmapper) {
		ret = -ENOMEM;
		goto out;
	}

	memset(gkeyidmapper, 0, KID_TABLE_LEN * sizeof(struct keyidmapper));

	gksdev = ksdev;				/*for future refrence */

out:

	return ret;
}
EXPORT_SYMBOL(caam_sm_ksm_init);


int caam_sm_ksm_prepareblob(u8 *sm_key)
{
	u32 keyslot_aes256 = 0;	/*unit bank #1*/
	u32 keyslot_aes256temp = 0xFF;
	u8 __iomem *clear_key_aes256;
	u8 __iomem *tmpbuf;
	int stat, rtnval = 0;
	struct device *dev = gksdev;

	clear_key_aes256 = tmpbuf = NULL;

	clear_key_aes256 = kmalloc(32, GFP_KERNEL | GFP_DMA);
	if (clear_key_aes256 == NULL) {
		rtnval = -ENOMEM;
		goto fail;
	}

	tmpbuf = kmalloc(32, GFP_KERNEL | GFP_DMA);
	if (tmpbuf == NULL) {
		rtnval = -ENOMEM;
		goto freemem;
	}

	/* Load up clear keys */
	memcpy(clear_key_aes256, sm_key, KEY_SIZE);
	memset(tmpbuf, 0, KEY_SIZE);

	stat = sm_keystore_slot_alloc(dev, UNIT_BANK, KEY_SIZE,
							&keyslot_aes256);

	if (stat) {
		dev_err(dev, "unable to allocate slote\n");
		rtnval = -ENAVAIL;
		goto freemem;
	}
	stat = sm_keystore_slot_alloc(dev, UNIT_BANK, KEY_SIZE,
							&keyslot_aes256temp);
	if (stat) {
		dev_err(dev, "unable to allocate slote\n");
		rtnval = -ENAVAIL;
		goto freemem;
	}
#ifdef SM_DEBUG
	dev_info(dev, "caam_sm_test: KEY_SIZE byte key slot in %d\n",
		 keyslot_aes256);
#endif

	/*just for setting up special kid slot used for key prepare */
	stat = sm_keystore_slot_load(dev, UNIT_BANK, keyslot_aes256,
				     clear_key_aes256, KEY_SIZE, 0, 0, 0);
	stat = sm_keystore_slot_load(dev, UNIT_BANK, keyslot_aes256temp,
				     tmpbuf, KEY_SIZE, 0, 0, 0);

	/* Encapsulate all keys as SM blobs */
	stat = sm_keystore_slot_encapsulate(dev, UNIT_BANK, keyslot_aes256,
					    keyslot_aes256temp,
						KEY_SIZE, skeymod, 8);
	if (stat) {
		dev_err(dev, "caam_sm_test: can't encapsulate AES256 key\n");
		rtnval = -EFAULT;
		goto freemem;
	}

	memset(sm_key, 0, KEY32_BB_SIZE); /* Clear the buffer */

	/* Extract 80 byte black blob from slot*/
	sm_keystore_slot_read(dev, UNIT_BANK, keyslot_aes256temp, KEY32_BB_SIZE,
						sm_key);

	caam_sm_ksm_relslot_key(keyslot_aes256);
	caam_sm_ksm_relslot_key(keyslot_aes256temp);
freemem:
	kfree(clear_key_aes256);
	kfree(tmpbuf);

fail:
	return rtnval;
}
EXPORT_SYMBOL(caam_sm_ksm_prepareblob);


int caam_sm_ksm_readblob(u32 kid, u8 *smbkey)
{
	u8 __iomem  *black_blob;
	int rtnval = 0;
	u32 slot, uid, gid;
	struct device *dev = gksdev;

	black_blob = NULL;
	/* Allocate storage black blob */
	black_blob = kmalloc(KEY32_BB_SIZE, GFP_KERNEL | GFP_DMA);
	if (black_blob == NULL) {
		rtnval = -ENOMEM;
		dev_err(dev, "caam_sm_test: can't black key buffers\n");
		goto fail;
	}
	/*Get the slot number of given kid from mapper table*/
	slot = getslot_from_table(kid);
	if ((int)slot == -ENOKEY) {
		rtnval = -ENOKEY;
		goto freemem;
	}

	sm_keystore_get_permission(dev, UNIT_BANK, slot, &uid, &gid);
	if (gid != current_gid()) {
		rtnval = -EPERM;
		goto freemem;
	}

#ifdef SM_DEBUG
	dev_info(dev, "Slot from table %d\n", slot);
#endif
	/* Extract 80 byte black blob from slot*/
	sm_keystore_slot_read(dev, UNIT_BANK, slot, KEY32_BB_SIZE,
						black_blob);
	/* Load up clear keys */
	memcpy(smbkey, black_blob, KEY32_BB_SIZE);

#ifdef SM_DEBUG
	for (i = 0; i < KEY32_BB_SIZE; i++)
		dev_info(dev, "%02x ", black_blob[i]);
	dev_info(dev, "\n");
#endif
freemem:
	kfree(black_blob);
fail:
	return rtnval;
}
EXPORT_SYMBOL(caam_sm_ksm_readblob);

/*
*	Release a slot
*/
int caam_sm_ksm_relslot_key(u32 slot)
{
	int rtnval = 0;
	struct device *dev = gksdev;

	/* Release slot of given kid */
	rtnval = sm_keystore_slot_dealloc(dev, UNIT_BANK, slot);
	if (rtnval)
		dev_err(dev, "caam_sm_test: can't release slot %d\n",
								slot);
	return rtnval;
}
EXPORT_SYMBOL(caam_sm_ksm_relslot_key);


int caam_sm_ksm_prepare(struct skey_prepare *keyprepare)
{
	int rtnval = 0;
	u8 bbkey[KEY32_BB_SIZE];
	u8 hash[32], info[KEY32_BB_SIZE];

	/*Create black blob of key*/
	if (keyprepare->flags == KEYMAN_PREPARE_RED_KEY) {
		/*key buffer from user land*/
		rtnval = copy_from_user(bbkey, keyprepare->in_buf,
							keyprepare->in_len);
		if (rtnval)
			goto fail;

	} else if (keyprepare->flags == KEYMAN_PREPARE_ENCRYPTED_KEY) {
		/* Decrypt the encrypted key */
		if (keyprepare->enc_keyid != 0) {
			rtnval = caam_create_deckey(keyprepare, bbkey);
			if (rtnval != 0)
				goto fail;
		} else {
			rtnval = -ENOKEY;
			goto fail;
		}
	} else { /*Already a black blob*/
		rtnval = copy_from_user(bbkey, keyprepare->in_buf,
								KEY32_BB_SIZE);
		if (rtnval)
			goto fail;
	}

	/* setup for hash of info*/
	memcpy(info, &(keyprepare->in_kid), 4); /*key id*/
	memcpy(info+4, &keyprepare->uid, 4); /*Uid*/
	memcpy(info+8, &keyprepare->gid, 4); /*Gid*/
	memcpy(info+12, bbkey, keyprepare->in_len); /*Red key*/

	/* Create hash of bbkey
	 * hash of [kid+uid+gid+RK]
	 */
	caam_create_hash(info, (keyprepare->in_len+12), hash);

	/* copy hash to info to be save to disk*/
	memcpy(info+12, hash, KEY_SIZE); /*256 bit hash*/

	/*process/make black blob*/
	rtnval = caam_sm_ksm_prepareblob(bbkey);
	if (rtnval != 0)
		goto fail;

	/*Concate the long string like its on disk
	 *[kid,uid,gid,h(kid,uid,gid,rk)]+[black blob of Rkey]=(44+80)=124bytes
	 */
	rtnval = copy_to_user(keyprepare->out_buf, info,
						(keyprepare->in_len+12));
	if (rtnval)
		goto fail;

	rtnval = copy_to_user(keyprepare->out_buf+(keyprepare->in_len+12),
						bbkey, KEY32_BB_SIZE);
	if (rtnval)
		goto fail;

fail:
	return rtnval;
}
EXPORT_SYMBOL(caam_sm_ksm_prepare);

int caam_sm_ksm_loadblob(u32 kid, u32 uid, u32 gid, u8 *blob)
{
	int rtnval = 0;
	u32 slot, stat;
	u8 __iomem *tmpbuf;
	struct device *dev = gksdev;

	tmpbuf = kmalloc(KEY32_BB_SIZE, GFP_KERNEL | GFP_DMA);
	if (tmpbuf == NULL) {
		rtnval = -ENOMEM;
		goto fail;
	}

	/*copy blob to buffer */
	memcpy(tmpbuf, blob, KEY32_BB_SIZE);

	/*Alloc and load data into slot*/
	stat = sm_keystore_slot_alloc(dev, UNIT_BANK, KEY32_BB_SIZE, &slot);
	if (stat) {
		dev_err(dev, "unable to allocate slot\n");
		rtnval = -ENAVAIL;
		goto freemem;
	}

	stat = sm_keystore_slot_load(dev, UNIT_BANK, slot,
				     tmpbuf, KEY32_BB_SIZE, kid, uid, gid);

	/* add kid to slot mapping in the table*/
	rtnval  = write_slot_to_table(kid, slot);
	if (rtnval)
		goto freemem;

#ifdef SM_DEBUG
	dev_info(dev, "Key loaded to slot 0x%08x with key id 0x%08x\n",
					slot, kid);
#endif
freemem:
	kfree(tmpbuf);
fail:
	return rtnval;
}
EXPORT_SYMBOL(caam_sm_ksm_loadblob);

int caam_sm_ksm_unprepareblob(u8 *sm_blob)
{
	u32 keyslot_aes256 = 0;	/*unit bank #1*/
	u32 keyslot_aes256temp = 0xFF;
	u8 __iomem *blob;
	u8 __iomem *tmpbuf;
	int stat, rtnval = 0;
	struct device *dev = gksdev;

	blob = NULL;
	blob = kmalloc(KEY32_BB_SIZE, GFP_KERNEL | GFP_DMA);
	if (blob == NULL) {
		rtnval = -ENOMEM;
		goto fail;
	}
	tmpbuf = kmalloc(KEY_SIZE, GFP_KERNEL | GFP_DMA);
	if (tmpbuf == NULL) {
		rtnval = -ENOMEM;
		goto fail;
	}

	/* Load up clear keys */
	memcpy(blob, sm_blob, KEY32_BB_SIZE);
	memset(tmpbuf, 0, KEY_SIZE);

	stat = sm_keystore_slot_alloc(dev, UNIT_BANK, KEY32_BB_SIZE,
			&keyslot_aes256);

	if (stat) {
		dev_err(dev, "unable to allocate slot\n");
		rtnval = -ENAVAIL;
		goto freemem;
	}
	stat = sm_keystore_slot_alloc(dev, UNIT_BANK, KEY_SIZE,
							&keyslot_aes256temp);
	if (stat) {
		dev_err(dev, "unable to allocate slot\n");
		rtnval = -ENAVAIL;
		goto freemem;
	}
#ifdef SM_DEBUG
	dev_info(dev, "caam_sm_test: KEY_SIZE byte key slot in %d\n",
		 keyslot_aes256);
#endif

	/*just for setting up special kid slot used for key prepare */
	stat = sm_keystore_slot_load(dev, UNIT_BANK, keyslot_aes256,
				     blob, KEY32_BB_SIZE, 0, 0, 0);

	stat = sm_keystore_slot_load(dev, UNIT_BANK, keyslot_aes256temp,
				     tmpbuf, KEY_SIZE, 0, 0, 0);

	/* Now decapsulate as black key blobs */
	stat = sm_keystore_slot_decapsulate(dev, UNIT_BANK, keyslot_aes256,
				keyslot_aes256temp, KEY_SIZE, skeymod, 8);
	if (stat) {
		dev_err(dev, "caam_sm_test: can't decapsulate AES256 key\n");
		rtnval = -EFAULT;
		goto freemem;
	}

	/* Extract 80 byte black blob from slot*/
	sm_keystore_slot_read(dev, UNIT_BANK, keyslot_aes256temp, 32,
						sm_blob);

	caam_sm_ksm_relslot_key(keyslot_aes256);
	caam_sm_ksm_relslot_key(keyslot_aes256temp);

freemem:
	kfree(blob);
	kfree(tmpbuf);
fail:
	return rtnval;
}
EXPORT_SYMBOL(caam_sm_ksm_unprepareblob);

/*Load record from KHA and load it in memory*/
int caam_sm_ksm_load(struct skey_load *skeyload)
{
	int rtnval = 0, i;
	u8 bbkey[KEY32_BB_SIZE], info[KEY32_BB_SIZE];
	u8 rkey[KEY32_BB_SIZE];
	u8 hash[32], prvhash[32];

	/*split the blobs*/
	rtnval = copy_from_user(prvhash, skeyload->buf, KEY_SIZE); /*hash data*/
	if (rtnval)
		goto fail;

	rtnval = copy_from_user(bbkey, skeyload->buf + KEY_SIZE, KEY32_BB_SIZE);
	if (rtnval)
		goto fail;

	memcpy(rkey, bbkey, KEY32_BB_SIZE);
	/*unwrap and verify key with prev hash*/
	rtnval = caam_sm_ksm_unprepareblob(rkey);
	if (rtnval != 0)
		goto fail;

	for (i = 0; i < KEY_SIZE; i++)
		pr_debug("%02x ", rkey[i]);
	pr_debug("\n");

	/* setup for hash of info*/
	memcpy(info, &(skeyload->kid), 4); /*key id*/
	memcpy(info+4, &(skeyload->uid), 4); /*Uid*/
	memcpy(info+8, &(skeyload->gid), 4); /*Gid*/
	memcpy(info+12, rkey, KEY_SIZE); /*Red key*/


	/* Create hash of bbkey */
	caam_create_hash(info, (KEY_SIZE+12), hash);

	pr_debug("Kid %d, uid %d, gid %d\n", skeyload->kid,
				skeyload->uid, skeyload->gid);
	if (memcmp(hash, prvhash, KEY_SIZE) != 0) {
		pr_err("Record tampered\n");
		rtnval = -EKEYREJECTED;
		goto fail;
	}

	/*Making sure kid is unique*/
	if (getslot_from_table(skeyload->kid) != -ENOKEY) {
		pr_err("Key exists\n");
		rtnval = -EEXIST;
		goto fail;
	}

	rtnval = caam_sm_ksm_loadblob(skeyload->kid, skeyload->uid,
						skeyload->gid, bbkey);
	if (rtnval != 0)
		goto fail;

#ifdef SM_DEBUG
	pr_info("Key successfully loaded\n");
#endif

fail:
	return rtnval;
}
EXPORT_SYMBOL(caam_sm_ksm_load);

/*
 * Release key from key store and its mapping struct skey_remove
 */
static int caam_sm_ksm_releasekey(struct skey_remove *relkey)
{
	struct device *dev = gksdev;
	u32 uid, gid;
	u32 keyid = relkey->keyid;
	int ret = 0, slot = 0;

#ifdef SM_DEBUG
	pr_info("This keyid needs to be removed %d\n", keyid);
#endif
	/*Get the slot number of given kid from mapper table*/
	slot = getslot_from_table(keyid);
	if (slot == -ENOKEY) {
		ret = -ENOKEY;
		goto fail;
	}

	sm_keystore_get_permission(dev, UNIT_BANK, slot, &uid, &gid);
	if (uid != relkey->uid) {
		ret = -EACCES;
		goto fail;
	}

	/* Remove slot mapping from table */
	ret = remove_slot_to_table(keyid, slot);
	if (ret)
		goto fail;

	/* Release the slot */
	ret = caam_sm_ksm_relslot_key(slot);
	if (ret)
		goto fail;
#ifdef SM_DEBUG
	pr_info("key removed from store\n");
#endif

fail:
	return ret;
}

/*
 *  Retrive the job ring dev
 */
int get_jr_device(struct device **ksdev)
{
	*ksdev = gksdev;
	return 0;
}
EXPORT_SYMBOL(get_jr_device);

/*ioctl*/
static long
kha_ioctl(struct file *file, unsigned int cmd, unsigned long arg_)
{
	void __user *arg = (void __user *)arg_;
	int ret = 0;
	struct skey_prepare skeyprepare;
	struct skey_load skeyload;
	struct skey_remove rel_key;

	pr_debug("in keysec ioctl %d\n", cmd);

	switch (cmd) {
	case SM_PREPARE_KEY:
		/* make blob from info in structure */
		pr_debug("keysec prepare key cmd called\n");
		ret = copy_from_user(&skeyprepare, arg,
				sizeof(skeyprepare));
		/*Now prepare the whole blob*/
		ret = caam_sm_ksm_prepare(&skeyprepare);
		if (ret)
			break;
		ret = copy_to_user(arg, &skeyprepare,
				sizeof(skeyprepare));
		pr_debug("Return %d\n", ret);
		break;

	case SM_LOAD_KEY:
		ret = copy_from_user(&skeyload, arg, sizeof(skeyload));
		/*Now prepare the whole blob*/
		ret = caam_sm_ksm_load(&skeyload);
		if (ret)
			break;
		ret = copy_to_user(arg, &skeyload, sizeof(skeyload));
		pr_debug("Return %d\n", ret);
		break;

	case SM_REL_KEY:
		ret = copy_from_user(&rel_key, arg,
				sizeof(struct skey_remove));
		if (ret)
			break;

		ret = caam_sm_ksm_releasekey(&rel_key);
		break;
	default:
		pr_err("sms wrong command\n");
		break;
	}

	return ret;
}

ssize_t kharead_key(struct file *filp, char *buf,
		size_t count, loff_t *f_pos) {

	return 0;
}

ssize_t khawrite_key(struct file *filp, const char *buf,
		size_t count, loff_t *f_pos) {
	return 0;
}

/* Structure that declares the usual file */
/* access functions */
static const struct file_operations securememory_fops = {
	.unlocked_ioctl = kha_ioctl,
};

static void __exit sm_kha_exit(void)
{
	/* FIXME: caam_sm_ksm_exit */
	device_destroy(ksm_class, ksm_dev_number);
	class_destroy(ksm_class);
	cdev_del(ksm_object);
	unregister_chrdev_region(ksm_dev_number, 1);
}

static int __init sm_kha_init(void)
{
	struct device_node *dev_node;
	struct platform_device *pdev;
	struct device *dev;
	int result;

	result = alloc_chrdev_region(&ksm_dev_number, 0, 1, KSM);

	if (result < 0) {
		pr_err("Fail to register ksm char dev\n");
		return -EIO;
	}

	ksm_object = cdev_alloc();
	if (ksm_object == NULL)
		goto unregister_chrdev_region;

	ksm_object->owner = THIS_MODULE;
	ksm_object->ops = &securememory_fops;

	if (cdev_add(ksm_object, ksm_dev_number, 1))
		goto cdev_del;

	ksm_class = class_create(THIS_MODULE, KSM);
	if (IS_ERR(ksm_class))
		goto cdev_del;

	dev = device_create(ksm_class, NULL, ksm_dev_number, NULL,
		      "%s", KSM);
	if (IS_ERR(dev))
		goto class_destroy;


	/*
	 * Do of_find_compatible_node() then of_find_device_by_node()
	 * once a functional device tree is available
	 */
	dev_node = of_find_compatible_node(NULL, NULL, "fsl,sec-v4.0");
	if (!dev_node) {
		dev_node = of_find_compatible_node(NULL, NULL, "fsl,secv4.0");
		if (!dev_node)
			return -ENODEV;
	}
	pdev = of_find_device_by_node(dev_node);
	if (!pdev)
		return -ENODEV;
	of_node_put(dev_node);


	caam_sm_ksm_init(pdev);

	return 0;

class_destroy:
	class_destroy(ksm_class);
cdev_del:
	cdev_del(ksm_object);
unregister_chrdev_region:
	unregister_chrdev_region(ksm_dev_number, 1);
	return -EIO;
}

module_init(sm_kha_init);
module_exit(sm_kha_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("KHA driver API interface");
MODULE_AUTHOR("Adnan Ali <adnan.ali@codethink.co.uk>");
