/*
 * node_device.c
 * Device node handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <glib-object.h>
#include <glib/gprintf.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <string.h>

#include "bt_appl/common_def.h"

#include "main.h"
#include "utils_async_queue.h"
#include "device_info.h"
#include "callbacks.h"
#include "node_device.h"
#include "node_adapter.h"
#include "handle_phonebook.h"
#include "handle_messaging.h"
#include "btd_audio.h"

#include "debug.h"
#include "marshal.h"

#include "bt_appl/api/bt_appl_bt_setting_block.h"
#include "bt_appl/api/bt_appl_phone_call_block.h"
#include "bt_appl/api/bt_appl_audio_block.h"
#include "bt_appl/api/bt_appl_phone_book_block.h"
#include "bt_appl/api/bt_appl_messaging_block.h"
#include "bt_appl_dummy.h"

#include "dbus_error.h"

/***********************/
/* DEVICE DEFINITIONS */
/***********************/

/* OUR CLASS DEFINITION */
typedef struct {
	/* The parent class state. */
	GObjectClass parent;

	/* put device signal stuff here, (if any) */

	/* holds the allocated signal numbers */
	/* device signals */
	guint RoleChange_sig_id;
	guint Disconnection_sig_id;

	/* handsfree signals */
	guint CallStatus_sig_id;
	guint PhoneStatus_sig_id;
	guint HfSetupAudio_sig_id;
	guint HfSetupAudioWbs_sig_id;
	guint HfReleaseAudio_sig_id;
	guint HfRing_sig_id;
	guint HfVoiceRecognitionActivationInd_sig_id;
	guint Siri_Nrstat_sig_id;
	guint Siri_AplSiri_sig_id;

	/* audio visual signals */
	guint AvpVersion_sig_id;
	guint AvpCtrlCmdInd_sig_id;
	guint AvpPlayStatus_sig_id;
        guint AvpNotifySetComplete_sig_id;
	guint AvpNotifyChg_sig_id;
	guint AvpCodecAndBitrate_sig_id;

	/* pbdl signals */
	guint PbDlData_sig_id;
	guint PbDlConnectionLost_sig_id;
	guint PbDlStatusChg_sig_id;

	/* map signals */
	guint MapActiveInstance_sig_id;
	guint MapNotifyStatus_sig_id;
	guint MapNotifyRegistStatus_sig_id;
	guint MapListFolderData_sig_id;
	guint MapListFolderComp_sig_id;
	guint MapListMsgData_sig_id;
	guint MapListMsgComp_sig_id;
	guint MapReadMsgData_sig_id;
	guint MapReadMsgComp_sig_id;
	guint MapUploadComp_sig_id;
	guint MapNotify_sig_id;
	guint MapErrorReport_sig_id;

} DeviceObjectClass;

/*
 * MAGIC here to define our device object functions etc. in GLib
 * declares device_object_init(), device_object_class_init(),
 * device_object_get_type()
 */
G_DEFINE_TYPE(DeviceObject, device_object, G_TYPE_OBJECT)

/*
 * Forward declaration of the function that will return the GType of
 * the Value implementation. Not used
 */
GType device_object_get_type(void);

/*
 * Macro for the above. It is common to define macros using the
 * naming convention (seen below) for all GType implementations.
 */
#define DEVICE_TYPE_OBJECT	(device_object_get_type())

#define DEVICE_TYPE_INSTANCE_GET_CLASS(Instance,\
		GTypeObject,StructType)\
		((StructType*) (void *)(((GTypeInstance*) Instance)->g_class))

/* macro to get from the class to object */
#define DEVICE_OBJECT_GET_CLASS(obj) \
	(DEVICE_TYPE_INSTANCE_GET_CLASS ((obj), \
	DEVICE_TYPE_OBJECT, DeviceObjectClass))

/********************/
/* DEVICE CALLBACKS */
/********************/

/*
 * callback handling, parameters needs to match marshalling signatures
 *
 * callbacks for the auto generated method protoypes
 * Forward declare the prototypes of the callback method handlers for the
 * dbus_glib_device_object_methods[]
 */

/********************/
/* Device Interface */
/********************/

gboolean device_object_get_device_address(DeviceObject *object,
						GByteArray **remote_address_out,
						GError **error);

gboolean device_object_disconnect(DeviceObject *object, guchar service_in,
							GError **error);

gboolean device_object_get_local_role(DeviceObject *object,
							guint8 *local_role_out,
							GError **error);

gboolean device_object_set_local_role(DeviceObject *object,
							guint8 local_role_in,
							GError **error);

gboolean device_object_get_device_name(DeviceObject *object,
					DBusGMethodInvocation *context);

/***********************/
/* Handsfree Interface */
/***********************/

gboolean device_object_get_call_status(DeviceObject *object,
							guint8 spec_call_in,
							GError **error);

gboolean device_object_get_phone_status(DeviceObject *object, GError **error);

gboolean device_object_hf_dial(DeviceObject *object,
					gchar *phone_number,
					DBusGMethodInvocation *context);

gboolean device_object_hf_redial(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_mem_dial(DeviceObject *object,
					guint32 mem_index,
					DBusGMethodInvocation *context);

gboolean device_object_hf_terminate_call(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_terminate_all(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_audio_transfer(DeviceObject *object,
					gboolean audio_trans_dir,
					DBusGMethodInvocation *context);

gboolean device_object_hf_send_dtmf(DeviceObject *object,
					guint8 dtmf_index,
					DBusGMethodInvocation *context);

gboolean device_object_hf_incoming_call_act(DeviceObject *object,
					guint8 action,
					DBusGMethodInvocation *context);

gboolean device_object_hf_release_inactive_call(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_release_active_call(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_swap_call(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_add_call(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_release_spec_call(DeviceObject *object,
					guint8 idx,
					DBusGMethodInvocation *context);

gboolean device_object_hf_hold_excep_call(DeviceObject *object,
					guint8 idx,
					DBusGMethodInvocation *context);

gboolean device_object_hf_exp_call_trans(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_subscriber(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_manufacturer_id(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_model_id(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_revision_id(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_hf_voice_recognition_activation(DeviceObject *object,
					guint8 activation,
					DBusGMethodInvocation *context);

gboolean device_object_siri_xapl_req(DeviceObject *object,
					gchar *parameter,
					DBusGMethodInvocation *context);

gboolean device_object_siri_nr_stat_req(DeviceObject *object,
							gchar aplNrStat,
							DBusGMethodInvocation *context);

gboolean device_object_siri_efm_req(DeviceObject *object,
							gchar ucAplEfm,
							DBusGMethodInvocation *context);


/**************************/
/* Auido Visual Interface */
/**************************/

gboolean device_object_avp_ctrl_cmd(DeviceObject *object,
					guint8 command,
					guint8 status_flag,
					DBusGMethodInvocation *context);

gboolean device_object_avp_get_capabilities(DeviceObject *object,
					guint8 capabilities_id,
					DBusGMethodInvocation *context);

gboolean device_object_avp_list_attr(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_avp_list_val(DeviceObject *object,
					guint8 setting_attribute_id_in,
					DBusGMethodInvocation *context);

gboolean device_object_avp_set_val(DeviceObject *object,
					guint8 mode_in,
					guint8 value_in,
					DBusGMethodInvocation *context);

gboolean device_object_avp_config_notify(DeviceObject *object,
							guint8 event_id_in,
							GError **error);

gboolean device_object_avp_set_player(DeviceObject *object,
					guint32 player_id,
					guint8 player,
					DBusGMethodInvocation *context);

gboolean device_object_avp_get_metadata_attr(DeviceObject *object,
					guint8 scope,
					GByteArray *uid,
					GArray *attributes,
					DBusGMethodInvocation *context);

gboolean device_object_avp_get_folder_item(DeviceObject *object,
					guint8 scope,
					guint32 start_item,
					guint32 end_item,
					guint8 num_attributes,
					GArray *attributes,
					DBusGMethodInvocation *context);

gboolean device_object_avp_get_media_player_item(DeviceObject *object,
					guint8 max_number,
					DBusGMethodInvocation *context);

gboolean device_object_avp_play_item(DeviceObject *object,
					guint8 scope,
					GByteArray *uid,
					DBusGMethodInvocation *context);

gboolean device_object_avp_change_path(DeviceObject *object,
					guint8 direction,
					GByteArray *folder_uid,
					DBusGMethodInvocation *context);

gboolean device_object_avp_audio_buffer_status(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_avp_get_audio_stream(DeviceObject *object,
					DBusGMethodInvocation *context);

/***********************/
/* Phonebook Interface */
/***********************/

gboolean device_object_pb_dl_get_capabilities(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_pb_dl_prepare(DeviceObject *object,
					DBusGMethodInvocation *context);

gboolean device_object_pb_dl_start(DeviceObject *object,
					guint32 filter,
					guint8 pb_type,
					guint8 version,
					guint32 max_total_recs,
					guint8 max_recs_per_trans,
					gboolean use_agent,
					DBusGMethodInvocation *context);

gboolean device_object_pb_dl_complete(DeviceObject *object, guint8 comp_mode,
					DBusGMethodInvocation *context);

gboolean device_object_pb_dl_stop(DeviceObject *object,
               guint8 action,
               DBusGMethodInvocation *context);

/***********************/
/* MAP Interface       */
/***********************/

gboolean device_object_map_get_capabilities(DeviceObject *object,
	DBusGMethodInvocation *context);

gboolean device_object_map_start(DeviceObject *object,
	GPtrArray *instance_info, DBusGMethodInvocation *context);

gboolean device_object_map_stop(DeviceObject *object,
	DBusGMethodInvocation *context);

gboolean device_object_map_chg_instance(DeviceObject *object,
	gchar *service_name, guint8 msg_type, DBusGMethodInvocation *context);

gboolean device_object_map_update_inbox(DeviceObject *object,
	gchar *service_name, DBusGMethodInvocation *context);

gboolean device_object_map_list_folder(DeviceObject *object,
	gchar *service_name, gchar *folder_name, guint32 max_list_count,
	guint32 list_start_offset, guint32 max_recs_per_trans,
	gboolean use_agent, DBusGMethodInvocation *context);

gboolean device_object_map_list_msg(DeviceObject *object,
	gchar *service_name, gchar *folder_name, guint32 max_list_count,
	guint32 list_start_offset, guint8 subject_len, guint32 param_mask,
	guint8 status_filter, guint8 type_filter, guint32 max_recs_per_trans,
	gboolean use_agent, DBusGMethodInvocation *context);

gboolean device_object_map_read_msg(DeviceObject *object, gchar *service_name,
	GByteArray *handle, guint8 attachment, DBusGMethodInvocation *context);
	
gboolean device_object_map_set_msg_status(DeviceObject *object,
	gchar *service_name, GByteArray *handle, guint8 msg_status,
        DBusGMethodInvocation *context);

gboolean device_object_map_upload(DeviceObject *object, gchar *service_name,
	guint8 transparent, guint8 retry, guint8 read_status, guint8 type,
	gchar *folder_name, GValueArray *sender, GPtrArray *recipient,
	guint32 msg_len, GPtrArray *attachment_list, guint8 language,
	guint8 encoding, DBusGMethodInvocation *context);

gboolean device_object_map_upload_data(DeviceObject *object,
	gchar *service_name, guint8 data_flag, GByteArray *data,
	 GError **error);

gboolean device_object_map_abort(DeviceObject *object,
	gchar *service_name, DBusGMethodInvocation *context);

/* async method handling functions */
static void register_device_async_queue_list(DeviceObject *object);
static void unregister_device_async_queue_list(DeviceObject *object);



/* include auto-generated binding information */
#include "alps-bt-device-introspection.h"


/***********************************/
/* OBJECT AND CLASS INITIALIZATION */
/***********************************/

/*
 * DeviceObject object initialiser function
 *
 * Used to initialise each object instance via g_object_new()
 */
static void device_object_init(DeviceObject *object)
{
	DEBUG_FUNC("Called");
	g_assert(object != NULL);

	/* add code to initialise created object contents */

	/* make sure that the phonebook download state is in idle */
	object->pbdl_state = PBDL_IDLE;
	/* make sure the info pointer is NULL */
	object->phonebook = NULL;
	/* initialise the agent method reply list */
	object->phonebook_agent_reply_list = NULL;

	/* initialise the MAP list msg download state to idle */
	object->map_list_msg_state = MAP_LIST_MSG_IDLE;
	/* initialise the MAP list message info pointer to NULL */
	object->map_list_msg_info = NULL;

	/* initialise AvpGetMediaPlayerItem state */
	object->get_media_player_item_state = AVP_GET_MEDIA_PLAYER_ITEM_IDLE;
	/* initialise AvpGetMediaPlayerItem data */
	object->get_media_player_item_info = NULL;

	/* initialise AvpGetFolderItem state */
	object->get_folder_item_state = AVP_GET_FOLDER_ITEM_IDLE;
	/* initialise AvpGetFolderItem data */
	object->get_folder_item_info = NULL;

	/* initialise A2DP info */
	object->a2dp_info.connected = FALSE;
	object->a2dp_info.streaming = FALSE;
	object->a2dp_info.codec_valid = FALSE;
	object->a2dp_info.codec = CODEC_SBC;
	object->a2dp_info.bitrate = 0;
}

/*
 * DeviceObject class initialiser function
 *
 * Used to initialise each class via the first call of g_object_new()
 */
static void device_object_class_init(DeviceObjectClass *klass)
{
#ifdef NOT_USED
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
#endif

	DEBUG_FUNC("Called");
	g_assert(klass != NULL);

	guint signalId;

	/* register new signals here */

	/* Signal: RoleChange */
	signalId = g_signal_new("role_change", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__UCHAR_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. */
			G_TYPE_BOOLEAN,
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->RoleChange_sig_id = signalId;


	/* Signal: Disconnection */
	signalId = g_signal_new("disconnection", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__UCHAR_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR,
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->Disconnection_sig_id = signalId;

	/* register new signals here */

	/* Signal: CallStatus */
	signalId = g_signal_new("call_status", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__UCHAR_UCHAR_UCHAR_UCHAR_UCHAR_STRING_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			7,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR,
			G_TYPE_UCHAR,
			G_TYPE_UCHAR,
			G_TYPE_UCHAR,
			G_TYPE_UCHAR,
			G_TYPE_STRING,
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->CallStatus_sig_id = signalId;


	/* Signal: PhoneStatus */
	signalId = g_signal_new("phone_status", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__UCHAR_UCHAR_UCHAR_UCHAR_STRING_UINT_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			7,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR,
			G_TYPE_UCHAR,
			G_TYPE_UCHAR,
			G_TYPE_UCHAR,
			G_TYPE_STRING,
            G_TYPE_UINT,
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->PhoneStatus_sig_id = signalId;


	/* Signal: HfSetupAudio */
	signalId = g_signal_new("hf_setup_audio", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__VOID,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			0
			/* GType(s) of the parameters. We have none. */
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->HfSetupAudio_sig_id = signalId;

	/* Signal: HfSetupAudioWbs */
	signalId = g_signal_new("hf_setup_audio_wbs", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__UINT,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. We have none. */
			G_TYPE_UINT
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->HfSetupAudioWbs_sig_id = signalId;




	/* Signal: HfReleaseAudio */
	signalId = g_signal_new("hf_release_audio", /* name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__VOID,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			0
			/* GType(s) of the parameters. We have none. */
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->HfReleaseAudio_sig_id = signalId;


	/* Signal: HfRing */
	signalId = g_signal_new("hf_ring", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->HfRing_sig_id = signalId;


	/* Signal: HfVoiceRecognitionActivationInd */
	signalId = g_signal_new("hf_voice_recognition_activation_ind",
			/* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->HfVoiceRecognitionActivationInd_sig_id = signalId;


	/* Signal: SiriNrStatInd */
	signalId = g_signal_new("siri_nr_stat_ind",
			/* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->Siri_Nrstat_sig_id = signalId;


	/* Signal: SiriAplsiriInd */
	signalId = g_signal_new("siri_aplsiri_ind",
			/* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->Siri_AplSiri_sig_id = signalId;


	/* Signal: AvpVersion */
	signalId = g_signal_new("avp_version", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->AvpVersion_sig_id = signalId;


	/* Signal: AvpCtrlCmdInd */
	signalId = g_signal_new("avp_ctrl_cmd_ind", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->AvpCtrlCmdInd_sig_id = signalId;


	/* Signal: AvpPlayStatus */
	signalId = g_signal_new("avp_play_status", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__UCHAR_UINT_UINT_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			4,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR,
			G_TYPE_UINT,
			G_TYPE_UINT,
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->AvpPlayStatus_sig_id = signalId;


	/* Signal: AvpNotifySetComplete */
	signalId = g_signal_new("avp_notify_set_complete", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__UINT,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. */
			G_TYPE_UINT);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->AvpNotifySetComplete_sig_id = signalId;


	/* Signal: AvpNotifyChg */
	signalId = g_signal_new("avp_notify_chg", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__UCHAR_BOOLEAN_BOXED,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			3,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR,
			G_TYPE_BOOLEAN,
			DBUS_TYPE_G_UCHAR_ARRAY);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->AvpNotifyChg_sig_id = signalId;


	/* Signal: AvpCodecAndBitrate */
	signalId = g_signal_new("avp_codec_and_bitrate", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			marshal_VOID__UCHAR_UINT,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR,
			G_TYPE_UINT);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->AvpCodecAndBitrate_sig_id = signalId;


	/* Signal: PbDlDataSignal */
	signalId = g_signal_new("pb_dl_data_signal", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__UCHAR_BOOLEAN_BOXED_BOXED_BOOLEAN_UCHAR_UINT,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			7,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR,
			G_TYPE_BOOLEAN,
			DBUS_TYPE_G_UINT_ARRAY,
			DBUS_TYPE_G_UCHAR_ARRAY,
			G_TYPE_BOOLEAN,
			G_TYPE_UCHAR,
			G_TYPE_UINT);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->PbDlData_sig_id = signalId;


	/* Signal: PbDlConnectionLost */
	signalId = g_signal_new("pb_dl_connection_lost", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__VOID,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			0
			/* GType(s) of the parameters. We have none */
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->PbDlConnectionLost_sig_id = signalId;


	/* Signal: PbDlStatusChg */
	signalId = g_signal_new("pb_dl_status_chg", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. We have none */
			G_TYPE_UCHAR
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->PbDlStatusChg_sig_id = signalId;


	/* Signal: MapActiveInstance */
	signalId = g_signal_new("map_active_instance", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			g_cclosure_marshal_VOID__VOID,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. */
			G_TYPE_STRING,
			G_TYPE_STRING
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapActiveInstance_sig_id = signalId;

	/* Signal: MapNotifyStatus */
	signalId = g_signal_new("map_notify_status", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			g_cclosure_marshal_VOID__UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapNotifyStatus_sig_id = signalId;

	/* Signal: MapNotifyRegistStatus */
	signalId = g_signal_new("map_notify_regist_status", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			marshal_VOID__STRING_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. */
			G_TYPE_STRING,
			G_TYPE_UCHAR
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapNotifyRegistStatus_sig_id = signalId;

	/* Signal: MapListFolderData */
	signalId = g_signal_new("map_list_folder_data", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			marshal_VOID__STRING_BOXED,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. */
			G_TYPE_STRING,
			G_TYPE_STRV
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapListFolderData_sig_id = signalId;

	/* Signal: MapListFolderComp */
	signalId = g_signal_new("map_list_folder_comp", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			marshal_VOID__STRING_UINT_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			3,
			/* GType(s) of the parameters. */
			G_TYPE_STRING,
			G_TYPE_UINT,
			G_TYPE_UCHAR
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapListFolderComp_sig_id = signalId;

	/* Signal: MapListMsgData */
	signalId = g_signal_new("map_list_msg_data", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			marshal_VOID__STRING_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED_BOXED,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			18,
			/* GType(s) of the parameters. */
			G_TYPE_STRING,
			DBUS_UCHAR_ARRAY_ARRAY,
			G_TYPE_STRV,
			G_TYPE_STRV,
			G_TYPE_STRV,
			G_TYPE_STRV,
			G_TYPE_STRV,
			G_TYPE_STRV,
			G_TYPE_STRV,
			DBUS_TYPE_G_UCHAR_ARRAY,
			DBUS_TYPE_G_UINT_ARRAY,
			DBUS_TYPE_G_UCHAR_ARRAY,
			DBUS_TYPE_G_UCHAR_ARRAY,
			DBUS_TYPE_G_UINT_ARRAY,
			DBUS_TYPE_G_UCHAR_ARRAY,
			DBUS_TYPE_G_UCHAR_ARRAY,
			DBUS_TYPE_G_UCHAR_ARRAY,
			DBUS_TYPE_G_UCHAR_ARRAY
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapListMsgData_sig_id = signalId;

	/* Signal: MapListMsgComp */
	signalId = g_signal_new("map_list_msg_comp", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			marshal_VOID__STRING_UINT_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			3,
			/* GType(s) of the parameters. */
			G_TYPE_STRING,
			G_TYPE_UINT,
			G_TYPE_UCHAR
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapListMsgComp_sig_id = signalId;

	/* Signal: MapReadMsgData */
	signalId = g_signal_new("map_read_msg_data", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			marshal_VOID__STRING_UINT_BOXED,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			3,
			/* GType(s) of the parameters. */
			G_TYPE_STRING,
			G_TYPE_UINT,
			DBUS_TYPE_G_UCHAR_ARRAY
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapReadMsgData_sig_id = signalId;

	/* Signal: MapReadMsgComp */
	signalId = g_signal_new("map_read_msg_comp", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			g_cclosure_marshal_VOID__STRING,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. */
			G_TYPE_STRING
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapReadMsgComp_sig_id = signalId;

	/* Signal: MapUploadComp */
	signalId = g_signal_new("map_upload_comp", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			marshal_VOID__STRING_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. */
			G_TYPE_STRING,
			G_TYPE_UCHAR
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapUploadComp_sig_id = signalId;

	/* Signal: MapNotify */
	signalId = g_signal_new("map_notify", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			marshal_VOID__STRING_UCHAR_BOXED_STRING_STRING_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			6,
			/* GType(s) of the parameters. */
			G_TYPE_STRING,
			G_TYPE_UCHAR,
			DBUS_TYPE_G_UCHAR_ARRAY,
			G_TYPE_STRING,
			G_TYPE_STRING,
			G_TYPE_UCHAR
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapNotify_sig_id = signalId;

	/* Signal: MapErrorReport */
	signalId = g_signal_new("map_error_report", /* str name of the signal */
			/* GType to which signal is bound */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call.
			 */
			marshal_VOID__STRING_UCHAR_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			3,
			/* GType(s) of the parameters. */
			G_TYPE_STRING,
			G_TYPE_UCHAR,
			G_TYPE_UCHAR
			);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->MapErrorReport_sig_id = signalId;

	DEBUG_HIGH("Binding our GObject with D-bus...");
	/* Uses auto-generated methods and signal structure */
	dbus_g_object_type_install_info(DEVICE_TYPE_OBJECT,
		&dbus_glib_device_object_object_info);

}

/****************************/
/* signal sending functions */
/****************************/

/*
 * exported function
 *
 * Emit the "Disconnection" signal.
 */
gboolean emit_disconnection(BD_ADDRESS device_address, guint8 service,
							guint8 conn_loss_type)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/* alert the a2dp audio streaming thread of the change of state */
	if (service == 0x01) {	/* AVP service */
		struct btd_audio *aud;

		aud = adapter_object_get_btd_audio();
		object->a2dp_info.connected = FALSE;
		object->a2dp_info.codec_valid = FALSE;
		object->a2dp_info.streaming = FALSE;
		btd_audio_avp_disconnect(aud, device_address);
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->Disconnection_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			service,
			conn_loss_type);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "RoleChange" signal.
 */
gboolean emit_role_change(BD_ADDRESS device_address, gboolean status,
							guint8 role)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->RoleChange_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			status,
			role);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "CallStatus" signal.
 */
gboolean emit_call_status(BD_ADDRESS device_address, guint8 call_handle,
							guint8 call_total,
							guint8 call_direction,
							guint8 call_state,
							guint8 call_mode,
							gchar *call_number,
							guint8 call_type)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->CallStatus_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			call_handle,
			call_total,
			call_direction,
			call_state,
			call_mode,
			call_number,
			call_type);

	DEBUG("Preparing to free used data");

	/* finished with the call_number string so free it */
	g_free(call_number);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "PhoneStatus" signal.
 */
gboolean emit_phone_status(BD_ADDRESS device_address, guint8 phone_service,
						guint8 phone_signal,
						guint8 phone_roaming,
						guint8 phone_battery,
						gchar *phone_operator_name,
						guint32 phone_supp_feature,
						guint8 phone_AgThreeWayCallSuppFeatures)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Inside emit_phone_status\n");
	DEBUG_FUNC("The value of phone_AgThreeWayCallSuppFeatures(Hex) %x \n",phone_AgThreeWayCallSuppFeatures);

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->PhoneStatus_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			phone_service,
			phone_signal,
			phone_roaming,
			phone_battery,
			phone_operator_name,
			phone_supp_feature,
			phone_AgThreeWayCallSuppFeatures);

	DEBUG("Preparing to free used data");

	/* finished with the phone_operator_name string so free it */
	g_free(phone_operator_name);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "HfRing" signal.
 */
gboolean emit_hf_ring(BD_ADDRESS device_address, guint8 ring_tone)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->HfRing_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			ring_tone);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "HfSetupAudio" signal.
 */
gboolean emit_hf_setup_audio(BD_ADDRESS device_address)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->HfSetupAudio_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0
			/*
			 * Data to marshal into the signal.
			 */
			);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 *  exported function
 *
 *  Emit "HfSetupAudioWbsInd" signal
 */
gboolean emit_hf_setup_audio_wbs_ind(BD_ADDRESS device_address, guint8 audiocodec)
{
	DEBUG("Called");
	DeviceObject *object = NULL;
	g_assert(NULL != device_address);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);
	if (object == NULL)
	{
		DEBUG_ERROR("Bluetooth device address: %s "
				"not found so cannot send signal",
				bd_addr_to_str(device_address));
		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);
	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
	 * Instance of the object that is generating this
	 * signal. This will be passed as the first parameter
	 * to the signal handler (eventually). But obviously
	 * when speaking about D-Bus, a signal caught on the
	 * other side of D-Bus will be first processed by
	 * the GLib-wrappers (the object proxy) and only then
	 * processed by the signal handler.
	 */
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->HfSetupAudioWbs_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			audiocodec
	);

	DEBUG_FUNC("Exited");
	return TRUE;
}





/*
 * exported function
 *
 * Emit the "HfReleaseAudio" signal.
 */
gboolean emit_hf_release_audio(BD_ADDRESS device_address)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->HfReleaseAudio_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0
			/*
			 * Data to marshal into the signal.
			 */
			);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "HfVoiceRecognitionActivationInd" signal.
 */
gboolean emit_hf_voice_recognition_activation_ind(BD_ADDRESS device_address,
							guint8 activation)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->HfVoiceRecognitionActivationInd_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			activation);

	DEBUG_FUNC("Exited");
	return TRUE;
}


/*
 * exported function
 *
 * Emit the "applxaplsiri" signal.
 */
gboolean emit_siri_aplsiri_ind(BD_ADDRESS device_address,
								u8 status)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("emit_siri_aplsiri_ind Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("emit_siri_aplsiri_ind Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->Siri_AplSiri_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			status);

	DEBUG_FUNC("emit_siri_aplsiri_ind Exited");
	return TRUE;
}


/*
 * exported function
 *
 * Emit the "sirinrstatInd" signal.
 */
gboolean emit_siri_nrstat_ind(BD_ADDRESS device_address,
								u8 ucAplNrStat)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("emit_siri_nrstat_ind Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("emit_siri_aplsiri_ind Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->Siri_Nrstat_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			ucAplNrStat);

	DEBUG_FUNC("emit_siri_nrstat_ind Exited");
	return TRUE;
}


/*
 * exported function
 *
 * Emit the "AvpVersion" signal.
 */
gboolean emit_avp_version(BD_ADDRESS device_address, guint8 version)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->AvpVersion_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			version);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "AvpCtrlCmdInd" signal.
 */
gboolean emit_avp_ctrl_cmd_ind(BD_ADDRESS device_address, guint8 command)
{
	DeviceObject *object = NULL;
	struct btd_audio *aud;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/* alert the a2dp audio streaming thread of the change of state */
	aud = adapter_object_get_btd_audio();
	if (command == 0x0B) {
		object->a2dp_info.streaming = TRUE;
		btd_audio_media_resume(aud, device_address,
			object->a2dp_info.codec, object->a2dp_info.bitrate);
	} else if (command == 0x0A) {
		object->a2dp_info.streaming = FALSE;
		btd_audio_media_suspend(aud, device_address);
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->AvpCtrlCmdInd_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			command);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "AvpPlayStatus" signal.
 */
gboolean emit_avp_play_status(BD_ADDRESS device_address, guint8 status,
							guint32 song_length,
							guint32 song_position,
							guint8 play_status)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->AvpPlayStatus_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			status,
			song_length,
			song_position,
			play_status);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "AvpNotifySetComplete" signal.
 */
gboolean emit_avp_notify_set_complete(BD_ADDRESS device_address,
							guint32 event_id)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->AvpNotifySetComplete_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			event_id);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "AvpNotifyChg" signal.
 */
gboolean emit_avp_notify_chg(BD_ADDRESS device_address, guint8 event_id,
							gboolean network_order,
							GByteArray *data)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);
	g_assert(data != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->AvpNotifyChg_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			event_id,
			network_order,
			data);

	/* finshed so need to free the data array */
	g_byte_array_unref(data);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "AvpCodecAndBitrate" signal.
 */
gboolean emit_avp_codec_and_bitrate(BD_ADDRESS device_address,
				    guint8 codec_type, guint bitrate)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/* track the a2dp audio state */
	if ((codec_type < MAX_CODECS) && (bitrate > 0)) {
		object->a2dp_info.codec = codec_type;
		object->a2dp_info.bitrate = bitrate;
		object->a2dp_info.codec_valid = TRUE;
	} else {
		if (codec_type < MAX_CODECS)
			object->a2dp_info.codec = codec_type;
		else
			object->a2dp_info.codec = CODEC_SBC;
		object->a2dp_info.bitrate = bitrate;
		object->a2dp_info.codec_valid = FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->AvpCodecAndBitrate_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			codec_type,
			bitrate);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "PbDlDataSignal" signal.
 */
gboolean emit_pb_dl_data_signal(DeviceObject *object)
{
	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(object->phonebook != NULL);
	g_assert(object->phonebook->data != NULL);
	g_assert(object->phonebook->index != NULL);

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->PbDlData_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			object->phonebook->data_type,
			object->phonebook->data_present,
			object->phonebook->index,
			object->phonebook->data,
			object->phonebook->completed,
			object->phonebook->status,
			object->phonebook->pb_number);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "PbDlConnectionLost" signal.
 */
gboolean emit_pb_dl_connection_lost(BD_ADDRESS device_address)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->PbDlConnectionLost_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0
			/*
			 * Data to marshal into the signal.
			 */
			);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "PbDlStatusChg" signal.
 */
gboolean emit_pb_dl_status_chg(BD_ADDRESS device_address, guint8 status)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->PbDlStatusChg_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			status);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapActiveInstance" signal.
 */
gboolean emit_map_active_instance(BD_ADDRESS device_address,
					gchar *sms_name,
					gchar *email_name)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapActiveInstance_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			sms_name,
			email_name);

	/* finished with the strings so free them */
	g_free(sms_name);
	g_free(email_name);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapNotifyStatus" signal.
 */
gboolean emit_map_notify_status(BD_ADDRESS device_address,
				guint8 status)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapNotifyStatus_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			status);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapNotifyRegistStatus" signal.
 */
gboolean emit_map_notify_regist_status(BD_ADDRESS device_address,
					gchar *service_name, guint8 status)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapNotifyRegistStatus_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			service_name,
			status);

	/* finished with the service name so free it */
	g_free(service_name);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapListFolderData" signal.
 */
gboolean emit_map_list_folder_data(BD_ADDRESS device_address,
					gchar *service_name,
					gchar **folder_listing)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapListFolderData_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			service_name,
			folder_listing);

	/* finished with the service name so free it */
	g_free(service_name);

	/* finished with the folder listing string so free it */
	if (folder_listing[0] != NULL) {
		g_free(folder_listing[0]);
	}
	g_free(folder_listing);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapListFolderComp" signal.
 */
gboolean emit_map_list_folder_comp(BD_ADDRESS device_address,
					gchar *service_name,
					guint32 total_size,
					guint8 status)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapListFolderComp_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			service_name,
			total_size,
			status);

	/* finished with the service name so free it */
	g_free(service_name);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapListMsgData" signal.
 */
gboolean emit_map_list_msg_data(DeviceObject *object)
{
	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(object->map_list_msg_info != NULL);

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapListMsgData_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			object->map_list_msg_info->service_name,
			object->map_list_msg_info->handle_list,
			object->map_list_msg_info->subject_list,
			object->map_list_msg_info->date_time_list,
			object->map_list_msg_info->sender_name_list,
			object->map_list_msg_info->sender_address_list,
			object->map_list_msg_info->replyto_address_list,
			object->map_list_msg_info->recipient_name_list,
			object->map_list_msg_info->recipient_address_list,
			object->map_list_msg_info->type_list,
			object->map_list_msg_info->size_list,
			object->map_list_msg_info->text_list,
			object->map_list_msg_info->reception_status_list,
			object->map_list_msg_info->attach_size_list,
			object->map_list_msg_info->priority_list,
			object->map_list_msg_info->read_list,
			object->map_list_msg_info->sent_list,
			object->map_list_msg_info->protect_list);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapListMsgComp" signal.
 */
gboolean emit_map_list_msg_comp(BD_ADDRESS device_address,
	gchar *service_name, guint32 total_size, guint8 status)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapListMsgComp_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			service_name,
			total_size,
			status);

	/* finished with the service name so free it */
	g_free(service_name);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapReadMsgData" signal.
 */
gboolean emit_map_read_msg_data(BD_ADDRESS device_address,
				gchar *service_name,
				guint8 data_flag,
				GByteArray *msg_data)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapReadMsgData_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			service_name,
			data_flag,
			msg_data);

	/* finished with the service name so free it */
	g_free(service_name);

	/* free the data array as no longer needed */
	g_byte_array_unref(msg_data);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapReadMsgComp" signal.
 */
gboolean emit_map_read_msg_comp(BD_ADDRESS device_address, gchar *service_name)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapReadMsgComp_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			service_name);

	/* finished with the service name so free it */
	g_free(service_name);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapUploadComp" signal.
 */
gboolean emit_map_upload_comp(BD_ADDRESS device_address,
				gchar *service_name,
				guint8 status)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapUploadComp_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			service_name,
			status);

	/* finished with the service name so free it */
	g_free(service_name);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapNotify" signal.
 */
gboolean emit_map_notify(map_notify_ind_t *parms)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(parms != NULL);
	g_assert(parms->bd_addr != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(parms->bd_addr);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(parms->bd_addr));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapNotify_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			parms->service_name,
			parms->type,
			parms->handle,
			parms->folder,
			parms->old_folder,
			parms->msg_type);

	/* finished with the service name so free it */
	g_free(parms->service_name);

	/* finished with the handle so free it */
	g_byte_array_unref(parms->handle);

	/* finished with the folder names so free them */
	g_free(parms->folder);
	g_free(parms->old_folder);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "MapErrorReport" signal.
 */
gboolean emit_map_error_report(BD_ADDRESS device_address,
				gchar *service_name,
				guint8 operation,
				guint8 error_code)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot send signal",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	/*
	 * Track the object to its class information
	 */
	DeviceObjectClass *klass = DEVICE_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->MapErrorReport_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			service_name,
			operation,
			error_code);

	/* finished with the service name so free it */
	g_free(service_name);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * create a new device object and registers it onto D-bus
 *
 * Inputs:
 * bus = point to the bus proxy
 */
gboolean device_object_creation(DBusGConnection *bus, gchar *node_name,
						BD_ADDRESS *our_address,
						DeviceObject **device_object)
{
	DEBUG_HIGH("Creating an instance of the DeviceObject"
				" and calling initialise functions...");

	g_assert(bus != NULL);
	g_assert(node_name != NULL);
	g_assert(our_address != NULL);
	g_assert(device_object != NULL);
	g_assert(*device_object == NULL);

	*device_object = g_object_new(DEVICE_TYPE_OBJECT, NULL);
	if(*device_object == NULL) {
		DEBUG_ERROR("Failed to create an instance of a DeviceObject");
		return FALSE;
	}

	DEBUG_HIGH("Registering the device GObject onto the D-Bus");
	/* the object's path does not come from the XML file */
	dbus_g_connection_register_g_object(bus,
		node_name,	/* Object's path for the device node */
		G_OBJECT(*device_object));

	/* record our device address */
	(*device_object)->device_address = our_address;

	/* register the async method queue list */
	register_device_async_queue_list(*device_object);

	return TRUE;
}


/*
 * exported function
 *
 * destroys a device object and unregisters the device off D-bus
 *
 * Inputs:
 * bus = point to the bus proxy
 */
void device_object_destroy(DBusGConnection *bus, DeviceObject **device_object)
{
	DEBUG("Destroying an instance of the DeviceObject...");

	g_assert(bus != NULL);
	g_assert(device_object != NULL);
	g_assert(*device_object != NULL);

	/* unregister from async method queue list */
	unregister_device_async_queue_list(*device_object);

	DEBUG_HIGH("Unregistering the device GObject off D-Bus");
	dbus_g_connection_unregister_g_object(bus,
		G_OBJECT(*device_object));

	if ((*device_object)->phonebook != NULL) {

		if ((*device_object)->phonebook->index != NULL) {
			/* free the index array */
			g_array_unref((*device_object)->phonebook->index);
		}

		if ((*device_object)->phonebook->data != NULL) {
			/* free the data array */
			g_byte_array_unref((*device_object)->phonebook->data);
		}		

		/* free the phonebook info */
		g_free((*device_object)->phonebook);
	}

	if ((*device_object)->map_list_msg_info != NULL) {

		/* free any MAP list message info */
		handle_map_list_msg_info_free((*device_object)->map_list_msg_info,
			(*device_object)->map_list_msg_info->max_recs_per_trans);

		/* free old service name */
		g_free((*device_object)->map_list_msg_info->service_name);

		/* free the MAP list message info */
		g_free((*device_object)->map_list_msg_info);
	}

	if ((*device_object)->get_media_player_item_info != NULL) {

		if ((*device_object)->get_media_player_item_info->data !=
									NULL) {
			/* free the data array */
			g_byte_array_unref(
			(*device_object)->get_media_player_item_info->data);
		}		

		/* free the get_media_player_item_info */
		g_free((*device_object)->get_media_player_item_info);
	}

	if ((*device_object)->get_folder_item_info != NULL) {

		if ((*device_object)->get_folder_item_info->data != NULL) {
			/* free the data array */
			g_byte_array_unref(
			(*device_object)->get_folder_item_info->data);
		}		

		/* free the get_folder_item_info */
		g_free((*device_object)->get_folder_item_info);
	}

	/* free the device object */
	g_object_unref(*device_object);

	/* remove the reference as object is now deleted */
	*device_object = NULL;
}


/*****************************/
/* METHOD CALLBACK FUNCTIONS */
/*****************************/

/************************/
/* ASYNC METHOD QUEUING */
/************************/

/*
 * initialise the async method queue
 */
static void register_device_async_queue_list(DeviceObject *object)
{
	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	utils_async_queue_initialise(&object->aql_device);

	DEBUG_FUNC("Exited");
}

/*
 * destroy the async method queue by first cleaning up entries
 */
static void unregister_device_async_queue_list(DeviceObject *object)
{
	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	utils_async_queue_destroy(&object->aql_device);

	DEBUG_FUNC("Exited");
}

/*********/
/* Utils */
/*********/

static BTHRESULT check_uid_length(GByteArray *uid, const char *function_name)
{
	BTHRESULT result = BT_APPL_SUCCESS;

	/*
	 * need to trap the situation that the input uid length
	 * does not match 8
	 */
	if (uid->len != 8) {

		DEBUG_ERROR("\'%s\' Uid length is %d, should be 8",
			function_name, uid->len);

		/* spoof an ALPS error code */
		result = BT_APPL_ERROR_PARAMETERS;

	} else {

#ifdef DBG_ENABLED
		gchar uid_str[2*8+1];

		g_sprintf(uid_str, "%02X%02X%02X%02X%02X%02X%02X%02X",
			uid->data[0], uid->data[1], uid->data[2], uid->data[3],
			uid->data[4], uid->data[5], uid->data[6], uid->data[7]);

		DEBUG("UID data %s", uid_str); 	
#endif
	}

	return result;
}


/********************/
/* Device interface */
/********************/
/*
 * Method call "GetDeviceAddress"
 *
 * Note: This function has no ALPS API function
 */
gboolean device_object_get_device_address(DeviceObject *object,
						GByteArray **remote_address_out,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(remote_address_out != NULL);
	g_assert(*remote_address_out == NULL);

	/* create a new byte array for the BT address */
	*remote_address_out = g_byte_array_new();

#if (defined USE_DUMMY_API) && (!defined USE_TEST_NODE)
	{
		/* TEST CODE emit the disconnect signal */
		emit_disconnection(*object->device_address, 21, 93);

		/* TEST CODE emit the CallStatus signal */
		emit_call_status(*object->device_address, 1, 2, 3, 4, 5,
								"call name",
								6);

		/* TEST CODE emit the PhoneStatus signal */
		emit_phone_status(*object->device_address, 10, 11, 12, 13,
								"operator name",
								0xFFFF, 0x7F);

		/* TEST CODE emit the HfRing signal */
		emit_hf_ring(*object->device_address, 22);
	}
#endif

	/* save the BT address into the byte array */
	*remote_address_out = g_byte_array_append(*remote_address_out,
						&(*object->device_address)[0],
						sizeof(BD_ADDRESS));

	return_code = report_any_error_to_dbus(result,
					"device_object_get_device_address",
					error);

	return return_code;
}

gboolean device_object_disconnect(DeviceObject *object, guchar service_in,
								GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	DEBUG("Service = %d", service_in);

	/*
	 * Call BTHRESULT BT_APPL_Disconnection_REQ(
	 *					IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u8 ucService);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Disconnection_REQ(*object->device_address,
								service_in);
#else
	result = BT_APPL_Disconnection_REQ_DUMMY(*object->device_address,
								service_in);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Disconnection_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "GetLocalRole"
 */
gboolean device_object_get_local_role(DeviceObject *object,
					guint8 *local_role_out,
					GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(local_role_out != NULL);

	/*
	 * Call BTHRESULT BT_APPL_Get_Local_Role_REQ(
	 *					IN const BD_ADDRESS aucBD_ADDR,
	 *					OUT u8 *pucRole);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Get_Local_Role_REQ(*object->device_address,
						local_role_out);
#else
	result = BT_APPL_Get_Local_Role_REQ_DUMMY(*object->device_address,
							local_role_out);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Get_Local_Role_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "SetLocalRole"
 */
gboolean device_object_set_local_role(DeviceObject *object,
					guint8 local_role_in,
					GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	DEBUG("LocalRole = %d", local_role_in);

	/*
	 * Call BTHRESULT BT_APPL_Set_Local_Role_REQ(
	 *					IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u8 ucRole);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Set_Local_Role_REQ(*object->device_address,
						local_role_in);
#else
	result = BT_APPL_Set_Local_Role_REQ_DUMMY(*object->device_address,
							local_role_in);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Set_Local_Role_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "GetDeviceName"
 *
 * asynchronous method: call phase
 */
gboolean device_object_get_device_name(DeviceObject *object,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_Get_Device_Name_REQ(
	 * IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Get_Device_Name_REQ(*object->device_address);
#else
	result = BT_APPL_Get_Device_Name_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Get_Device_Name_REQ",
						&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_Get_Device_Name_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "GetDeviceName"
 *
 * asynchronous method: return phase
 * device_name is a null-terminated string
 */
gboolean device_object_get_device_name_RTN(BD_ADDRESS device_address,
							gchar *device_name)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);
	g_assert(device_name != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_Get_Device_Name_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, device_name);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	/* finished with the device name so free it */
	g_free(device_name);

	DEBUG_FUNC("Exited");

	return return_code;
}

/***********************/
/* Handsfree interface */
/***********************/

/*
 * Method call "GetCallStatus"
 */
gboolean device_object_get_call_status(DeviceObject *object,
							guint8 spec_call_in,
							GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	DEBUG("SpecCall = %d", spec_call_in);

	/*
	 * Call BTHRESULT BT_APPL_Get_Call_Status_REQ(
	 *					IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u8 ucSpecCall);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Get_Call_Status_REQ(*object->device_address,
								spec_call_in);
#else
	result = BT_APPL_Get_Call_Status_REQ_DUMMY(*object->device_address,
								spec_call_in);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Get_Call_Status_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "GetPhoneStatus"
 */
gboolean device_object_get_phone_status(DeviceObject *object, GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/*
	 * Call BTHRESULT BT_APPL_Get_Phone_Status_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Get_Phone_Status_REQ(*object->device_address);
#else
	result = BT_APPL_Get_Phone_Status_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Get_Phone_Status_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfDial"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_dial(DeviceObject *object,
					gchar *phone_number,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);
	g_assert(phone_number != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Dial_REQ(IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u8 ucLength,
	 *					IN const u8* pucPHNNumber);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Dial_REQ(*object->device_address,
					strlen(phone_number),
					(u8 *) phone_number);
#else
	result = BT_APPL_HF_Dial_REQ_DUMMY(*object->device_address,
					strlen(phone_number),
					(u8 *) phone_number);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Dial_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Dial_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfDial"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_dial_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Dial_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfRedial"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_redial(DeviceObject *object,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Redial_REQ(IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Redial_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Redial_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Redial_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Redial_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfRedial"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_redial_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Redial_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfMemDial"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_mem_dial(DeviceObject *object,
					guint32 mem_index,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	DEBUG("mem_index %04X", mem_index);

	if (mem_index > 0xFFFF) {
		/* spoof an ALPS error code */
		result = BT_APPL_ERROR_PARAMETERS;
	}

	if (result == BT_APPL_SUCCESS) {

		/*
		 * Call BTHRESULT BT_APPL_HF_Mem_Dial_REQ(
		 *				IN const BD_ADDRESS aucBD_ADDR,
		 *				IN u16 usMemIndex);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_HF_Mem_Dial_REQ(*object->device_address,
							(guint16) mem_index);
#else
		result = BT_APPL_HF_Mem_Dial_REQ_DUMMY(*object->device_address,
							(guint16) mem_index);
#endif

		DEBUG("BT stack function returned");
	}

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Mem_Dial_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Mem_Dial_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfMemDial"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_mem_dial_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Mem_Dial_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfTerminateCall"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_terminate_call(DeviceObject *object,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Terminate_Call_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Terminate_Call_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Terminate_Call_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Terminate_Call_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Terminate_Call_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfTerminateCall"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_terminate_call_RTN(BD_ADDRESS device_address,
								guint8 status,
								guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Terminate_Call_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfTerminateAll"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_terminate_all(DeviceObject *object,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Terminate_All_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Terminate_All_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Terminate_All_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Terminate_All_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Terminate_All_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfTerminateAll"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_terminate_all_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Terminate_All_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfAudioTransfer"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_audio_transfer(DeviceObject *object,
						gboolean audio_trans_dir,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Audio_Transfer_REQ(
	 *					IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u8 ucAudioTransDir);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Audio_Transfer_REQ(*object->device_address,
						audio_trans_dir);
#else
	result = BT_APPL_HF_Audio_Transfer_REQ_DUMMY(*object->device_address,
						audio_trans_dir);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Audio_Transfer_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Audio_Transfer_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfAudioTransfer"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_audio_transfer_RTN(BD_ADDRESS device_address,
								guint8 status,
								guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Audio_Transfer_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfSendDtmf"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_send_dtmf(DeviceObject *object,
					guint8 dtmf_index,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Send_DTMF_REQ(
	 *					IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u8 ucDTMFIndex);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Send_DTMF_REQ(*object->device_address, dtmf_index);
#else
	result = BT_APPL_HF_Send_DTMF_REQ_DUMMY(*object->device_address,
						dtmf_index);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Send_DTMF_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Send_DTMF_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfSendDtmf"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_send_dtmf_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Send_DTMF_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfIncomingCallAct"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_incoming_call_act(DeviceObject *object,
						guint8 action,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Incoming_Call_Act_REQ(
	 *					IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u8 ucAction);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Incoming_Call_Act_REQ(*object->device_address,
								action);
#else
	result = BT_APPL_HF_Incoming_Call_Act_REQ_DUMMY(
						*object->device_address,
						action);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Incoming_Call_Act_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Incoming_Call_Act_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfIncomingCallAct"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_incoming_call_act_RTN(BD_ADDRESS device_address,
								guint8 status,
								guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Incoming_Call_Act_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfReleaseInactiveCall"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_release_inactive_call(DeviceObject *object,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Release_Inactive_Call_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Release_Inactive_Call_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Release_Inactive_Call_REQ_DUMMY(
						*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Release_Inactive_Call_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Release_Inactive_Call_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfReleaseInactiveCall"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_release_inactive_call_RTN(BD_ADDRESS device_address,
								guint8 status,
								guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
					ID_HF_Release_Inactive_Call_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfReleaseActiveCall"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_release_active_call(DeviceObject *object,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Release_Active_Call_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Release_Active_Call_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Release_Active_Call_REQ_DUMMY(
						*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Release_Active_Call_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Release_Active_Call_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfReleaseActiveCall"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_release_active_call_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Release_Active_Call_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfSwapCall"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_swap_call(DeviceObject *object,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Swap_Call_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Swap_Call_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Swap_Call_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Swap_Call_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Swap_Call_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfSwapCall"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_swap_call_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Swap_Call_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfAddCall"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_add_call(DeviceObject *object,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Add_Call_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Add_Call_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Add_Call_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Add_Call_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Add_Call_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfAddCall"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_add_call_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Add_Call_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfReleaseSpecCall"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_release_spec_call(DeviceObject *object,
						guint8 idx,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Release_Spec_Call_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR,
	 *				IN u8 ucIndex);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Release_Spec_Call_REQ(
						*object->device_address,
						idx);
#else
	result = BT_APPL_HF_Release_Spec_Call_REQ_DUMMY(
						*object->device_address,
						idx);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Release_Spec_Call_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Release_Spec_Call_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfReleaseSpecCall"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_release_spec_call_RTN(BD_ADDRESS device_address,
							guint8 status,
							guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Release_Spec_Call_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfHoldExcepCall"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_hold_excep_call(DeviceObject *object,
						guint8 idx,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Hold_Excep_Call_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR,
	 *				IN u8 ucIndex);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Hold_Excep_Call_REQ(
						*object->device_address,
						idx);
#else
	result = BT_APPL_HF_Hold_Excep_Call_REQ_DUMMY(
						*object->device_address,
						idx);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Hold_Excep_Call_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Hold_Excep_Call_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfHoldExcepCall"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_hold_excep_call_RTN(BD_ADDRESS device_address,
							guint8 status,
							guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Hold_Excep_Call_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfExpCallTrans"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_exp_call_trans(DeviceObject *object,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Exp_Call_Trans_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Exp_Call_Trans_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Exp_Call_Trans_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Exp_Call_Trans_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Exp_Call_Trans_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfExpCallTrans"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_exp_call_trans_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Exp_Call_Trans_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfSubscriber"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_subscriber(DeviceObject *object,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Subscriber_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Subscriber_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Subscriber_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Subscriber_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Subscriber_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfSubscriber"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_subscriber_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code,
						GPtrArray *subscriber_info)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Subscriber_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code,
							subscriber_info);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG("Preparing to free used data");

	/* finshed so free the subscriber_info array */
	g_ptr_array_unref(subscriber_info);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfManufacturerId"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_manufacturer_id(DeviceObject *object,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Manufacturer_ID_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Manufacturer_ID_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Manufacturer_ID_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Manufacturer_ID_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Manufacturer_ID_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfManufacturerId"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_manufacturer_id_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code,
						gchar *manufacturer)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Manufacturer_ID_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code,
						manufacturer);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG("Preparing to free used data");

	/* finished with the manufacturer string so free it */
	g_free(manufacturer);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfModelId"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_model_id(DeviceObject *object,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Model_ID_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Model_ID_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Model_ID_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Model_ID_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Model_ID_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfModelId"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_model_id_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code,
						gchar *model)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Model_ID_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code,
						model);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG("Preparing to free used data");

	/* finished with the model string so free it */
	g_free(model);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfRevisionId"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_revision_id(DeviceObject *object,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_HF_Revision_ID_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_HF_Revision_ID_REQ(*object->device_address);
#else
	result = BT_APPL_HF_Revision_ID_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_HF_Revision_ID_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_HF_Revision_ID_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfRevisionId"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_revision_id_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 code,
						gchar *revision)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_HF_Revision_ID_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, code,
						revision);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG("Preparing to free used data");

	/* finished with the revision string so free it */
	g_free(revision);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "HfVoiceRecognitionActivation"
 *
 * asynchronous method: call phase
 */
gboolean device_object_hf_voice_recognition_activation(DeviceObject *object,
						guint8 activation,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_Voice_Recognition_Activation_REQ(
	 *		IN const BD_ADDRESS aucBD_ADDR, IN u8 ucActivation);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Voice_Recognition_Activation_REQ(
						*object->device_address,
						activation);
#else
	result = BT_APPL_HF_Voice_Recognition_Activation_REQ_DUMMY(
						*object->device_address,
						activation);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
				"BT_APPL_Voice_Recognition_Activation_REQ",
				&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
				ID_HF_Voice_Recognition_Activation_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "HfVoiceRecognitionActivation"
 *
 * asynchronous method: return phase
 */
gboolean device_object_hf_voice_recognition_activation_RTN(
						BD_ADDRESS device_address,
						guint8 activation,
						guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
					&matched_device->aql_device,
					ID_HF_Voice_Recognition_Activation_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, activation, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");
	return return_code;
}


/*
 * Method call "SIRIXapl"
 *
 * asynchronous method: call phase
 */
gboolean device_object_siri_xapl_req(DeviceObject *object,
							gchar *parameter,
							DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;
	guint32 parameter_len;

	DEBUG_FUNC("siri_xapl req Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_SIRI_XAPL_REQ(IN const BD_ADDRESS aucBD_ADDR,
	 *										IN u8 ucParameterLen,
	 *										IN u8 *pucParameter);
	 */
	DEBUG("Product and vendorID's %s"
					, parameter);

	parameter_len = parameter ? strlen(parameter) : 0;
		if (parameter_len > 255)
			parameter_len = 255;

#ifndef USE_DUMMY_API
	result = BT_APPL_SIRI_Xapl_REQ(*object->device_address,parameter_len,(u8*)parameter);
#else
	result = BT_APPL_SIRI_XAPL_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_SIRI_XAPL_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_APPL_SIRI_XAPL_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}



/*
 * exported function
 *
 * Method call "applsirixapl"
 *
 * asynchronous method: return phase
 */
gboolean device_object_appl_siri_xapl_RTN(BD_ADDRESS device_address,
									guint8 status,
									gchar *parameter)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
					&matched_device->aql_device,
					ID_APPL_SIRI_XAPL_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, parameter);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}
	/* finished with the revision string so free it */
	g_free(parameter);

	DEBUG_FUNC("Exited");
	return return_code;
}


/*
 * Method call "SIRINrStat"
 *
 * asynchronous method: call phase
 */
gboolean device_object_siri_nr_stat_req(DeviceObject *object,
							gchar aplNrStat,
							DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("siri_nr_stat req Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_SIRI_NrStat_REQ(IN const BD_ADDRESS aucBD_ADDR, IN u8 ucAplNrStat);
	 */


#ifndef USE_DUMMY_API
	result = BT_APPL_SIRI_NrStat_REQ(*object->device_address, aplNrStat);
#else
	result = BT_APPL_SIRI_NrStat_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_SIRI_NrStat_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_APPL_SIRI_NrStat_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("siri_nr_stat Exited");

	return return_code;
}


/*
 * exported function
 *
 * Method call "sirinrstat"
 *
 * asynchronous method: return phase
 */
gboolean device_object_siri_nrstat_RTN(BD_ADDRESS device_address,
									guint8 status )
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC(" device_object_siri_nrstat_RTN Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
					&matched_device->aql_device,
					ID_APPL_SIRI_NrStat_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("device_object_siri_nrstat_RTN Exited");
	return return_code;
}



/*
 * Method call "SIRIEFM- Eys free mode"
 *
 * asynchronous method: call phase
 */
gboolean device_object_siri_efm_req(DeviceObject *object,
							gchar ucAplEfm,
							DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("siri_nr_stat req Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_SIRI_Efm_REQ(IN const BD_ADDRESS aucBD_ADDR, IN u8 ucAplEfm);
	 */


#ifndef USE_DUMMY_API
	result = BT_APPL_SIRI_Efm_REQ(*object->device_address, ucAplEfm);
#else
	result = BT_APPL_SIRI_Efm_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_SIRI_Efm_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
							ID_APPL_SIRI_EFM_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("siri_efm_req Exited");

	return return_code;
}



/*
 * exported function
 *
 * Method call "siriefmcfm"
 *
 * asynchronous method: return phase
 */
gboolean device_object_siri_efm_RTN(BD_ADDRESS device_address,
									guint8 status )
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC(" device_object_siri_efm_RTN Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
					&matched_device->aql_device,
					ID_APPL_SIRI_EFM_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("device_object_siri_efm_RTN Exited");
	return return_code;
}

/**************************/
/* Auido Visual Interface */
/**************************/

/*
 * Method call "AvpCtrlCmd"
 *
 * asynchronous method: call phase
 */
gboolean device_object_avp_ctrl_cmd(DeviceObject *object,
					guint8 command,
					guint8 status_flag,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 *	BTHRESULT BT_APPL_AVP_Ctrl_Cmd_REQ(
	 *					IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u8 ucCommand,
	 *					IN u8 ucStatusFlag);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_AVP_Ctrl_Cmd_REQ(*object->device_address, command,
								status_flag);
#else
	result = BT_APPL_AVP_Ctrl_Cmd_REQ_DUMMY(
						*object->device_address,
						command,
						status_flag);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_AVP_Ctrl_Cmd_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_AVP_Ctrl_Cmd_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "AvpCtrlCmd"
 *
 * asynchronous method: return phase
 */
gboolean device_object_avp_ctrl_cmd_RTN(BD_ADDRESS device_address,
							guint8 command,
							guint8 status_flag,
							guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_AVP_Ctrl_Cmd_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, command, status_flag,
								status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "AvpGetCapabilities"
 *
 * asynchronous method: call phase
 */
gboolean device_object_avp_get_capabilities(DeviceObject *object,
					guint8 capabilities_id,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * BTHRESULT BT_APPL_AVP_Get_Capabilities_REQ(
	 *					IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u8 ucCapabilityId);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_AVP_Get_Capabilities_REQ(*object->device_address,
							capabilities_id);
#else
	result = BT_APPL_AVP_Get_Capabilities_REQ_DUMMY(
						*object->device_address,
						capabilities_id);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_AVP_Get_Capabilities_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_AVP_Get_Capabilities_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "AvpGetCapabilities"
 *
 * asynchronous method: return phase
 */
gboolean device_object_avp_get_capabilities_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 capability_id,
						guint32 event_supported)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_AVP_Get_Capabilities_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, capability_id,
							event_supported);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "AvpListAttr"
 *
 * asynchronous method: call phase
 */
gboolean device_object_avp_list_attr(DeviceObject *object,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * BTHRESULT BT_APPL_AVP_List_Attr_REQ(IN const BD_ADDRESS aucBD_ADDR);
	 */	
#ifndef USE_DUMMY_API
	result = BT_APPL_AVP_List_Attr_REQ(*object->device_address);
#else
	result = BT_APPL_AVP_List_Attr_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_AVP_List_Attr_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_AVP_List_Attr_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "AvpListAttr"
 *
 * asynchronous method: return phase
 */
gboolean device_object_avp_list_attr_RTN(BD_ADDRESS device_address,
						guint8 status,
						guint8 attributes_bit)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_AVP_List_Attr_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, attributes_bit);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "AvpListVal"
 *
 * asynchronous method: call phase
 */
gboolean device_object_avp_list_val(DeviceObject *object,
					guint8 setting_attribute_id_in,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * BTHRESULT BT_APPL_AVP_List_Val_REQ(IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u8 ucSettingAttributeId);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_AVP_List_Val_REQ(*object->device_address,
						setting_attribute_id_in);
#else
	result = BT_APPL_AVP_List_Val_REQ_DUMMY(*object->device_address, 
						setting_attribute_id_in);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_AVP_List_Val_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
							ID_AVP_List_Val_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "AvpListVal"
 *
 * asynchronous method: return phase
 */
gboolean device_object_avp_list_val_RTN(BD_ADDRESS device_address,
						guint8 status,
						GByteArray *setting_values,
						guint8 current_value)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);
	g_assert(setting_values != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_AVP_List_Val_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, setting_values,
							current_value);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	/* free the settings array as no longer needed */
	g_byte_array_unref(setting_values);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "AvpSetVal"
 *
 * asynchronous method: call phase
 */
gboolean device_object_avp_set_val(DeviceObject *object,
					guint8 mode_in,
					guint8 value_in,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * BTHRESULT BT_APPL_AVP_Set_Val_REQ(IN BD_ADDRESS aucBD_ADDR,
	 *							IN u8 ucMode,
	 *							IN u8 ucValue);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_AVP_Set_Val_REQ(*object->device_address, mode_in, 
								value_in);
#else
	result = BT_APPL_AVP_Set_Val_REQ_DUMMY(*object->device_address, mode_in, 
								value_in);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_AVP_Set_Val_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
							ID_AVP_Set_Val_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "AvpSetVal"
 *
 * asynchronous method: return phase
 */
gboolean device_object_avp_set_val_RTN(BD_ADDRESS device_address, guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_AVP_Set_Val_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "AvpConfigNotify"
 */
gboolean device_object_avp_config_notify(DeviceObject *object,
							guint8 event_id_in,
							GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	DEBUG("event_id = 0x%04X", event_id_in);

	/*
	 * BTHRESULT BT_APPL_AVP_Config_Notify_REQ(IN u16 usEventId);
	 *
	 * Note that the ALPS function does not have a device address
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_AVP_Config_Notify_REQ(event_id_in);
#else
	result = BT_APPL_AVP_Config_Notify_REQ_DUMMY(event_id_in);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
				"BT_APPL via BT_APPL_AVP_Config_Notify_REQ",
				error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "AvpSetPlayer"
 *
 * asynchronous method: call phase
 */
gboolean device_object_avp_set_player(DeviceObject *object,
					guint32 player_id,
					guint8 player,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	DEBUG("player_id 0x%04X, player %u", player_id, player);

	/*
	 * BTHRESULT BT_APPL_AVP_Set_Player_REQ(IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u16 usPlayerId,
	 *					IN u8 ucPlayer);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_AVP_Set_Player_REQ(*object->device_address,
								player_id,
								player);
#else
	result = BT_APPL_AVP_Set_Player_REQ_DUMMY(
						*object->device_address,
								player_id,
								player);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_AVP_Set_Player_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
							ID_AVP_Set_Player_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "AvpSetPlayer"
 *
 * asynchronous method: return phase
 */
gboolean device_object_avp_set_player_RTN(BD_ADDRESS device_address,
							guint8 status,
							guint32 num_item,
							guint8 is_db_aware)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_AVP_Set_Player_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, num_item,
								is_db_aware);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "AvpGetMetadataAttr"
 *
 * asynchronous method: call phase
 */
gboolean device_object_avp_get_metadata_attr(DeviceObject *object,
					guint8 scope,
					GByteArray *uid,
					GArray *attributes,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(uid != NULL);
	g_assert(attributes != NULL);
	g_assert(context != NULL);

	DEBUG("scope %u, num attributes %u", scope, attributes->len);

	result = check_uid_length(uid, __func__);

	if (result == BT_APPL_SUCCESS) {

		/*
		 * BTHRESULT BT_APPL_AVP_Get_Metadata_Attr_REQ(
		 *				IN BD_ADDRESS aucBD_ADDR,
		 *				IN u8 ucScope,
		 *  				IN u8 aucUid[],
		 * 				IN u8 ucNumAttributes,
		 *				IN u32 aulAttributes[]);
		 */

#ifndef USE_DUMMY_API
		result = BT_APPL_AVP_Get_Metadata_Attr_REQ(
						*object->device_address,
						scope,
						&uid->data[0],
						attributes->len,
						(u32 *)(void*) &attributes->data[0]);
#else
		result = BT_APPL_AVP_Get_Metadata_Attr_REQ_DUMMY(
						*object->device_address,
						scope,
						&uid->data[0],
						attributes->len,
						(u32 *)(void *) &attributes->data[0]);
#endif
	}

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_AVP_Get_Metadata_Attr_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_AVP_Get_Metadata_Attr_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "AvpGetMetadataAttr"
 *
 * asynchronous method: return phase
 */
gboolean device_object_avp_get_metadata_attr_RTN(BD_ADDRESS device_address,
							guint8 status,
							guint8 num_attributes,
							gboolean network_order,
							GByteArray *data)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);
	g_assert(data != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_AVP_Get_Metadata_Attr_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, num_attributes,
								network_order,
								data);
			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	/* free the data array as no longer needed */
	g_byte_array_unref(data);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "AvpGetFolderItem"
 *
 * asynchronous method: call phase
 */
gboolean device_object_avp_get_folder_item(DeviceObject *object,
					guint8 scope,
					guint32 start_item,
					guint32 end_item,
					guint8 num_attributes,
					GArray *attributes,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;
	u32 *attribute_ptr = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	DEBUG("scope %u, start_item 0x%04X, end_item 0x%04X, num_attributes %u",
		 scope, start_item, end_item, num_attributes);

	/* use NULL pointer unless num_attributes is suitable */
	if ((num_attributes != 0) && (num_attributes != 0xFF)) {
		attribute_ptr = (u32 *)(void*) &attributes->data[0];

		if (num_attributes != attributes->len) {
			DEBUG("Error: num_attributes %d does not match array"
				" length %u", num_attributes, attributes->len);

			/* spoof an ALPS error code */
			result = BT_APPL_ERROR_PARAMETERS;
		}
	}

	if (result == BT_APPL_SUCCESS) {
		/*
		 * BTHRESULT BT_APPL_AVP_Get_Folder_Item_REQ(
		 *				IN const BD_ADDRESS aucBD_ADDR,
		 *				IN u8 ucScope,
		 *				IN u32 ulStartItem,
		 *				IN u32 ulEndItem,
		 *				IN u8 ucNumAttributes,
		 *				IN u32 aulAttributes[]);
		 */

#ifndef USE_DUMMY_API
		result = BT_APPL_AVP_Get_Folder_Item_REQ(
						*object->device_address,
						scope,
						start_item,
						end_item,
						num_attributes,
						attribute_ptr);
#else
		result = BT_APPL_AVP_Get_Folder_Item_REQ_DUMMY(
						*object->device_address,
						scope,
						start_item,
						end_item,
						num_attributes,
						attribute_ptr);
#endif
	}

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_AVP_Get_Folder_Item_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_AVP_Get_Folder_Item_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "AvpGetFolderItem"
 *
 * asynchronous method: return phase
 */
gboolean device_object_avp_get_folder_item_RTN(DeviceObject *object)
{
	DBusGMethodInvocation *context = NULL;
	gboolean return_code = FALSE;
	GetFolderItemInfo *folder_info = object->get_folder_item_info;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(object->get_folder_item_state ==
					AVP_GET_FOLDER_ITEM_SEND_REPLY);

	context = utils_async_queue_find_list_entry(&object->aql_device,
					ID_AVP_Get_Folder_Item_CFM);

	if (context != NULL) {
		/* found the context info, so send reply */
		dbus_g_method_return(context, folder_info->status,
						folder_info->scope,
						folder_info->network_order,
						folder_info->data);

		return_code = TRUE;
	} else {
		DEBUG_ERROR("No device context so cannot reply on D-bus");
	}

	/* now do tidy-up */
	/* free the used AVRCP data */
	g_byte_array_unref(folder_info->data);

	/* free the used folder info */
	g_free(folder_info);

	/* set the state back to idle */
	object->get_folder_item_state = AVP_GET_FOLDER_ITEM_IDLE;
	object->get_folder_item_info = NULL;

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "AvpGetMediaPlayerItem"
 *
 * asynchronous method: call phase
 */
gboolean device_object_avp_get_media_player_item(DeviceObject *object,
					guint8 max_number,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * BTHRESULT BT_APPL_AVP_Get_Media_Player_Item_REQ(
	 *					IN const BD_ADDRESS aucBD_ADDR,
	 *					IN u16 usMaxNumber);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_AVP_Get_Media_Player_Item_REQ(*object->device_address,
								max_number);
#else
	result = BT_APPL_AVP_Get_Media_Player_Item_REQ_DUMMY(
						*object->device_address,
						max_number);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_AVP_Get_Media_Player_Item_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
					ID_AVP_Get_Media_Player_Item_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "AvpGetMediaPlayerItem"
 *
 * asynchronous method: return phase
 */
gboolean device_object_avp_get_media_player_item_RTN(DeviceObject *object)
{
	DBusGMethodInvocation *context = NULL;
	gboolean return_code = FALSE;
	GetMediaPlayerItemInfo *player_info = object->get_media_player_item_info;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(object->get_media_player_item_state ==
					AVP_GET_MEDIA_PLAYER_ITEM_SEND_REPLY);

	context = utils_async_queue_find_list_entry(&object->aql_device,
					ID_AVP_Get_Media_Player_Item_CFM);

	if (context != NULL) {
		/* found the context info, so send reply */
		dbus_g_method_return(context, player_info->status,
						player_info->network_order,
						player_info->data);

		return_code = TRUE;
	} else {
		DEBUG_ERROR("No device context so cannot reply on D-bus");
	}

	/* now do tidy-up */
	/* free the used AVRCP data */
	g_byte_array_unref(player_info->data);

	/* free the used media player info */
	g_free(player_info);

	/* set the state back to idle */
	object->get_media_player_item_state = AVP_GET_MEDIA_PLAYER_ITEM_IDLE;
	object->get_media_player_item_info = NULL;

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "AvpPlayItem"
 *
 * asynchronous method: call phase
 */
gboolean device_object_avp_play_item(DeviceObject *object,
					guint8 scope,
					GByteArray *uid,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(uid != NULL);
	g_assert(context != NULL);

	result = check_uid_length(uid, __func__);

	if (result == BT_APPL_SUCCESS) {

		/*
		 * BTHRESULT BT_APPL_AVP_Play_Item_REQ(IN BD_ADDRESS aucBD_ADDR,
		 *				       IN u8 ucScope,
		 *				       IN u8 aucUid[]);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_AVP_Play_Item_REQ(*object->device_address,
								scope,
								&uid->data[0]);
#else
		result = BT_APPL_AVP_Play_Item_REQ_DUMMY(
							*object->device_address,
							scope,
							&uid->data[0]);
#endif
	}

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_AVP_Play_Item_REQ",
						&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
					ID_AVP_Play_Item_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "AvpPlayItem"
 *
 * asynchronous method: return phase
 */
gboolean device_object_avp_play_item_RTN(BD_ADDRESS device_address,
							guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_AVP_Play_Item_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "AvpChangePath"
 *
 * asynchronous method: call phase
 */
gboolean device_object_avp_change_path(DeviceObject *object,
					guint8 direction,
					GByteArray *folder_uid,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(folder_uid != NULL);
	g_assert(context != NULL);

	result = check_uid_length(folder_uid, __func__);

	if (result == BT_APPL_SUCCESS) {

		/*
		 * BTHRESULT BT_APPL_AVP_Change_Path_REQ(
		 *				IN BD_ADDRESS aucBD_ADDR,
		 *				IN u8 ucDirection,
		 *				IN u8 aucFolderUid[]);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_AVP_Change_Path_REQ(*object->device_address,
							direction,
							&folder_uid->data[0]);
#else
		result = BT_APPL_AVP_Change_Path_REQ_DUMMY(
							*object->device_address,
							direction,
							&folder_uid->data[0]);
#endif
	}

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_AVP_Change_Path_REQ",
						&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_AVP_Change_Path_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "AvpChangePath"
 *
 * asynchronous method: return phase
 */
gboolean device_object_avp_change_path_RTN(BD_ADDRESS device_address,
							guint8 status,
							guint32 num_items)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_AVP_Change_Path_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, num_items);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/***********************/
/* Phonebook interface */
/***********************/

/*
 * Method call "PbDlGetCapabilities"
 *
 * asynchronous method: call phase
 */
gboolean device_object_pb_dl_get_capabilities(DeviceObject *object,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_PBDL_Get_Capabilities_REQ(
	 *				IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_PBDL_Get_Capabilities_REQ(*object->device_address);
#else
	result = BT_APPL_PBDL_Get_Capabilities_REQ_DUMMY(
						*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_PBDL_Get_Capabilities_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
					ID_PBDL_Get_Capabilities_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "PbDlGetCapabilities"
 *
 * asynchronous method: return phase
 */
gboolean device_object_pb_dl_get_capabilities_RTN(BD_ADDRESS device_address,
								guint8 status,
								guint8 type)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_PBDL_Get_Capabilities_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, type);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "PbDlPrepare"
 *
 * asynchronous method: call phase
 */
gboolean device_object_pb_dl_prepare(DeviceObject *object,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/*
	 * Call BTHRESULT BT_APPL_PBDL_Prepare_REQ(
	 *					IN const BD_ADDRESS aucBD_ADDR);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_PBDL_Prepare_REQ(*object->device_address);
#else
	result = BT_APPL_PBDL_Prepare_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_PBDL_Prepare_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
					ID_PBDL_Prepare_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "PbDlPrepare"
 *
 * asynchronous method: return phase
 */
gboolean device_object_pb_dl_prepare_RTN(BD_ADDRESS device_address,
								guint8 type)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_PBDL_Prepare_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, type);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "PbDlStart"
 *
 * asynchronous method: call phase
 */
gboolean device_object_pb_dl_start(DeviceObject *object,
					guint32 filter,
					guint8 pb_type,
					guint8 version,
					guint32 max_total_recs,
					guint8 max_recs_per_trans,
					gboolean use_agent,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	/* grab the additional parameters */
	DEBUG("MaxTotalRecs = %u, MaxRecsPerTrans = %u, UseAgent = %d",
		max_total_recs, max_recs_per_trans, use_agent);

	if (max_recs_per_trans == 0) {
		DEBUG("Error: passed in a zero MaxRecsPerTrans");

		/* spoof an ALPS error code */
		result = BT_APPL_ERROR_PARAMETERS;
	}

	if (result == BT_APPL_SUCCESS) {
		/*
		 * BTHRESULT BT_APPL_PBDL_Start_REQ(
		 *				IN const BD_ADDRESS aucBD_ADDR,
		 *				IN u32 ulFilter,
		 *				IN u8 ucPBType,
		 *				IN u8 ucAction,
		 *				IN u8 ucVersion);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_PBDL_Start_REQ(*object->device_address,
								filter,
								pb_type,
								0, /* start */
								version);
#else
		result = BT_APPL_PBDL_Start_REQ_DUMMY(*object->device_address,
								filter,
								pb_type,
								0, /* start */
								version);
#endif
		DEBUG("BT stack function returned");
	}

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_PBDL_Start_REQ",
					&error);

	if (return_code == TRUE) {
		/*
		 * initialise phonebook downloand stuff.
		 *
		 * Note that if the state was PBDL_WAIT_DATA
		 * then data will be lost due to a missing
		 * complete IND
		 */
		handle_phonebook_set_wait_data(object, pb_type, max_total_recs, 
							0,
							max_recs_per_trans,
							use_agent);

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
							ID_PBDL_Start_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "PbDlStart"
 *
 * Asynchronous method: return phase
 */
gboolean device_object_pb_dl_start_RTN(BD_ADDRESS device_address,
							guint8 status,
							guint32 pb_number,
							guint8 new_missed_call)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
							ID_PBDL_Start_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, pb_number,
							new_missed_call);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "PbDlComplete"
 *
 * asynchronous method: call phase
 */
gboolean device_object_pb_dl_complete(DeviceObject *object, guint8 comp_mode,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	DEBUG("comp_mode = %u", comp_mode);

	/*
	 * BTHRESULT BT_APPL_PBDL_Complete_REQ(IN BD_ADDRESS aucBD_ADDR,
	 *						IN u8 ucCompMode);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_PBDL_Complete_REQ(*object->device_address, comp_mode);
#else
	result = BT_APPL_PBDL_Complete_REQ_DUMMY(*object->device_address,
								comp_mode);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_PBDL_Complete_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_PBDL_Complete_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "PbDlComplete"
 *
 * asynchronous method: return phase
 */
gboolean device_object_pb_dl_complete_RTN(BD_ADDRESS device_address,
							guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_PBDL_Complete_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "PbDlStop"
 *
 * asynchronous method: Call Phase
 *
 * */
gboolean device_object_pb_dl_stop(DeviceObject *object,
               guint8 action,
               DBusGMethodInvocation *context)
{
   GError *error = NULL;
   BTHRESULT result;
   gboolean return_code = FALSE;

   DEBUG_FUNC("Called");
   g_assert(NULL != object);
   g_assert(NULL != context);
   DEBUG("Action:%d",action);

#ifndef USE_DUMMY_API
   result = BT_APPL_PBDL_Stop_REQ(*object->device_address,(u8)0);
#else
   result = BT_APPL_PBDL_Stop_REQ_DUMMY(*object->device_address,0);
#endif

   DEBUG("BT stack function returned");

   return_code = report_any_error_to_dbus(result,
                                 "BT_APPL_PBDL_Stop_REQ",
                                 &error);

   if (return_code == TRUE)
   {
      /* put the method return in a queue pending the CFM */
      utils_async_queue_add_list_entry(&object->aql_device, context,
            ID_PBDL_Stop_CFM);
   }
   else
   {
      DEBUG("async call error");
      /* an error occurred so abort the method call */
      dbus_g_method_return_error(context, error);
      g_error_free(error);
   }

   DEBUG_FUNC("Exited");
   return return_code;
}

/*
 * exported function
 *
 * Method call "PbDlStop"
 *
 * asynchronous method: return phase
 */
gboolean device_object_pb_dl_stop_RTN(BD_ADDRESS device_address,
                     guint8 status)
{
   DBusGMethodInvocation *context = NULL;
   DeviceObject *matched_device = NULL;
   gboolean return_code = FALSE;

   DEBUG_FUNC("Called");

   g_assert(device_address != NULL);

   /* find the device object relating to the device address */
   matched_device = find_device_object_from_list(device_address);

   if (matched_device != NULL)
   {
      context = utils_async_queue_find_list_entry(
                     &matched_device->aql_device,
                     ID_PBDL_Stop_CFM);

      if (context != NULL)
      {
         /* found the context info, so send reply */
         dbus_g_method_return(context, status);
         return_code = TRUE;
      }
      else
      {
         DEBUG_ERROR("No device context so cannot reply on D-bus");
      }
   }
   else
   {
      DEBUG_ERROR("Bluetooth device address: %s "
         "not found so cannot reply on D-bus",
         bd_addr_to_str(device_address));
   }

   DEBUG_FUNC("Exited");

   return return_code;
}



/*
 * Method call "MapGetCapabilities"
 *
 * asynchronous method: call phase
 */
gboolean device_object_map_get_capabilities(DeviceObject *object,
	DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

#ifndef USE_DUMMY_API
	result = BT_APPL_MAP_Get_Capabilities_REQ(*object->device_address);
#else
	result = BT_APPL_MAP_Get_Capabilities_REQ_DUMMY(
						*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_Get_Capabilities_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
					ID_MAP_Get_Capabilities_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "MapGetCapabilities"
 *
 * asynchronous method: return phase
 */
gboolean device_object_map_get_capabilities_RTN( BD_ADDRESS device_address,
						guint8 status,
						GPtrArray *msg_capability)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_MAP_Get_Capabilities_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, msg_capability);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	/* finished with the msg_capability arrray so free it */
	g_ptr_array_unref(msg_capability);

	DEBUG_FUNC("Exited");

	return return_code;
}


/*
 * Method call "MapStart"
 *
 * asynchronous method: call phase
 */
gboolean device_object_map_start(DeviceObject *object,
	GPtrArray *instance_info, DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;
	guint instance_num = 0;
	InstanceInfoPtr_t info;
	guint i;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	if (instance_info)
		instance_num = instance_info->len;

	DEBUG("instance_num = %d", instance_num);

	info = g_new0(InstanceInfo, instance_num);
	for (i = 0; i < instance_num; i++) {
		GValue *value;

		value = g_new0(GValue, 1);
		g_value_init(value, DBUS_STRUCT_STRING_UCHAR);
		g_value_set_boxed(value,
			g_ptr_array_index(instance_info, i));
		dbus_g_type_struct_get(value,
			0, &info[i].pucSvcName,
			1, &info[i].ucMsgType,
			G_MAXUINT);
		info[i].ucSvcNameLen = strlen((gchar *) (info[i].pucSvcName));
		DEBUG("instance_info[%d]: service_name = \"%s\", "
			"service_name_len = %d, msg_type = %d", i,
			info[i].pucSvcName, info[i].ucSvcNameLen,
			info[i].ucMsgType);
		g_free(value);
	}

#ifndef USE_DUMMY_API
	result = BT_APPL_MAP_Start_REQ(*object->device_address, instance_num,
		info);
#else
	result = BT_APPL_MAP_Start_REQ_DUMMY(*object->device_address,
		instance_num, info);
#endif

	DEBUG("BT stack function returned");

	/* Free InstanceInfo */
	for (i = 0; i < instance_num; i++)
		g_free(info[i].pucSvcName);
	g_free(info);

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_Start_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_MAP_Start_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "MapStart"
 *
 * asynchronous method: return phase
 */
gboolean device_object_map_start_RTN(BD_ADDRESS device_address,
	guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_MAP_Start_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "MapStop"
 *
 * asynchronous method: call phase
 */
gboolean device_object_map_stop(DeviceObject *object,
	DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

#ifndef USE_DUMMY_API
	result = BT_APPL_MAP_Stop_REQ(*object->device_address);
#else
	result = BT_APPL_MAP_Stop_REQ_DUMMY(*object->device_address);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_Stop_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
					ID_MAP_Stop_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "MapStop"
 *
 * asynchronous method: return phase
 */
gboolean device_object_map_stop_RTN(BD_ADDRESS device_address,
	guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_MAP_Stop_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "MapChgInstance"
 *
 * asynchronous method: call phase
 */
gboolean device_object_map_chg_instance(DeviceObject *object,
	gchar *service_name, guint8 msg_type, DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

#ifndef USE_DUMMY_API
	result = BT_APPL_MAP_Chg_Instance_REQ(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name,
		msg_type);
#else
	result = BT_APPL_MAP_Chg_Instance_REQ_DUMMY(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name,
		msg_type);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_Chg_Instance_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_MAP_Chg_Instance_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "MapChgInstance"
 *
 * asynchronous method: return phase
 */
gboolean device_object_map_chg_instance_RTN(BD_ADDRESS device_address,
	guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_MAP_Chg_Instance_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "MapUpdateInbox"
 *
 * asynchronous method: call phase
 */
gboolean device_object_map_update_inbox(DeviceObject *object,
	gchar *service_name, DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

#ifndef USE_DUMMY_API
	result = BT_APPL_MAP_Update_Inbox_REQ(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name);
#else
	result = BT_APPL_MAP_Update_Inbox_REQ_DUMMY(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_Update_Inbox_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_MAP_Update_Inbox_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "MapUpdateInbox"
 *
 * asynchronous method: return phase
 */
gboolean device_object_map_update_inbox_RTN(BD_ADDRESS device_address,
	guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_MAP_Update_Inbox_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "MapListFolder"
 *
 * asynchronous method: call phase
 */
gboolean device_object_map_list_folder(DeviceObject *object,
	gchar *service_name, gchar *folder_name, guint32 max_list_count,
	guint32 list_start_offset, guint32 max_recs_per_trans,
	gboolean use_agent, DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;
	ListFolderParam_t fp;
	guint service_name_len;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	service_name_len = service_name ? strlen(service_name) : 0;
	if (service_name_len > 255)
		service_name_len = 255;
	fp.pucFolderName = (u8 *) folder_name;
	fp.usFolderNameLen = folder_name ? strlen(folder_name) : 0;
	fp.usMaxListCount = max_list_count;
	fp.usListStartOffset = list_start_offset;

	DEBUG("service_name=\"%s\", folder_name=\"%s\", max_list_count=%u, "
		"list_start_offset=%u, max_recs_per_trans=%u, use_agent=%d",
		service_name ? service_name : "",
		folder_name ? folder_name : "", max_list_count,
		list_start_offset, max_recs_per_trans, use_agent);

#ifndef USE_DUMMY_API
	result = BT_APPL_MAP_List_Folder_REQ(*object->device_address,
		service_name_len, (u8 *) service_name, &fp);
#else
	result = BT_APPL_MAP_List_Folder_REQ_DUMMY(*object->device_address,
		service_name_len, (u8 *) service_name, &fp);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_List_Folder_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_MAP_List_Folder_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "MapListFolder"
 *
 * asynchronous method: return phase
 */
gboolean device_object_map_list_folder_RTN(BD_ADDRESS device_address,
	guint8 status, guint8 accessible, guint32 folder_num)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	DEBUG("status=%u, accessible=%u, folder_num=%u",
		status, accessible, folder_num);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_MAP_List_Folder_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, accessible,
				folder_num);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "MapListMsg"
 *
 * asynchronous method: call phase
 */
gboolean device_object_map_list_msg(DeviceObject *object,
	gchar *service_name, gchar *folder_name, guint32 max_list_count,
	guint32 list_start_offset, guint8 subject_len, guint32 param_mask,
	guint8 status_filter, guint8 type_filter, guint32 max_recs_per_trans,
	gboolean use_agent, DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result= BT_APPL_SUCCESS;
	gboolean return_code = FALSE;
	ListMsgParam_t mp;
	guint32 service_name_len;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	service_name_len = service_name ? strlen(service_name) : 0;
	if (service_name_len > 255)
		service_name_len = 255;
	mp.pucFolderName = (u8 *) folder_name;
	mp.usFolderNameLen = folder_name ? strlen(folder_name) : 0;
	mp.usMaxListCount = max_list_count;
	mp.usListStartOffset = list_start_offset;
	mp.ucSubjectLen = subject_len;
	mp.ulParamMask = param_mask;
	mp.ucStatusFilter = status_filter;
	mp.ucTypeFilter = type_filter;

	DEBUG("service_name=\"%s\", folder_name=\"%s\", max_list_count=%u, "
		"list_start_offset=%u, subject_len=%u, param_mask=%u, "
		"status_filter=%u, type_filter=%u, max_recs_per_trans=%u, "
		"use_agent=%d",
		service_name ? service_name : "",
		folder_name ? folder_name : "", max_list_count,
		list_start_offset, subject_len, param_mask, status_filter,
		type_filter, max_recs_per_trans, use_agent);

	if (max_recs_per_trans == 0) {
		DEBUG("Error: passed in a zero MaxRecsPerTrans");

		/* spoof an ALPS error code */
		result = BT_APPL_ERROR_PARAMETERS;
	}

	if (result == BT_APPL_SUCCESS) {

#ifndef USE_DUMMY_API
		result = BT_APPL_MAP_List_Msg_REQ(*object->device_address,
						service_name_len,
						(u8 *) service_name, &mp);
#else
		result = BT_APPL_MAP_List_Msg_REQ_DUMMY(*object->device_address,
						service_name_len,
						(u8 *) service_name, &mp);
#endif

		DEBUG("BT stack function returned");
	}

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_List_Msg_REQ",
					&error);

	if (return_code == TRUE) {
		/*
		 * Note that if the state was MAP_LIST_MSG_WAIT_DATA
		 * then data will be lost as the previous
		 * session was incomplete due to a missing
		 * complete IND
		 */
		handle_map_list_msg_set_wait_data(object, service_name,
							service_name_len,
							max_recs_per_trans,
							use_agent);

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_MAP_List_Msg_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "MapListMsg"
 *
 * asynchronous method: return phase
 */
gboolean device_object_map_list_msg_RTN(BD_ADDRESS device_address, guint8 status,
	guint32 message_num)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	DEBUG("status=%u, message_num=%u", status, message_num);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_MAP_List_Msg_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status, message_num);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "MapReadMsg"
 *
 * asynchronous method: call phase
 */
gboolean device_object_map_read_msg(DeviceObject *object, gchar *service_name,
	GByteArray *handle, guint8 attachment, DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);
	g_assert (handle != NULL);

	DEBUG("service_name=\"%s\" attachment=%u",
		service_name ? service_name : "", attachment);

#ifndef USE_DUMMY_API
	result = BT_APPL_MAP_Read_Msg_REQ(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name,
		handle->len, (u8 *) handle->data, attachment);
#else
	result = BT_APPL_MAP_Read_Msg_REQ_DUMMY(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name,
		handle->len, (u8 *) handle->data, attachment);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_Read_Msg_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_MAP_Read_Msg_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "MapReadMsg"
 *
 * asynchronous method: return phase
 */
gboolean device_object_map_read_msg_RTN(map_read_msg_cfm_t *parms)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(parms != NULL);
	g_assert(parms->bd_addr != NULL);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(parms->bd_addr);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_MAP_Read_Msg_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context,
						parms->status,
						parms->read_status,
						parms->type,
						parms->folder_name,
						parms->sender,
						parms->recipient,
						parms->msg_len,
						parms->attachment_list,
						parms->language,
						parms->encoding);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(parms->bd_addr));
	}

	DEBUG("Preparing to free used data");

	/* finished with the folder name so free it */
	g_free(parms->folder_name);

	/* finished with the sender info so free it */
	g_value_array_free(parms->sender);

	/* finished with the recipent info so free it */
	g_ptr_array_unref(parms->recipient);

	/* finished with the attachment list so free it */
	g_ptr_array_unref(parms->attachment_list);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "MapSetMsgStatus"
 *
 * asynchronous method: call phase
 */
gboolean device_object_map_set_msg_status(DeviceObject *object,
	gchar *service_name, GByteArray *handle, guint8 msg_status,
        DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);
	g_assert (handle != NULL);

	DEBUG("service_name=\"%s\" msg_status=%u",
		service_name ? service_name : "", msg_status);

#ifndef USE_DUMMY_API
	result = BT_APPL_MAP_Set_Msg_Status_REQ(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name,
		handle->len, (u8 *) handle->data, msg_status);
#else
	result = BT_APPL_MAP_Set_Msg_Status_REQ_DUMMY(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name,
		handle->len, (u8 *) handle->data, msg_status);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_Set_Msg_Status_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_MAP_Set_Msg_Status_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "MapSetMsgStatus"
 *
 * asynchronous method: return phase
 */
gboolean device_object_map_set_msg_status_RTN(BD_ADDRESS device_address,
	guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	DEBUG("status=%u", status);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_MAP_Set_Msg_Status_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "MapUpload"
 *
 * asynchronous method: call phase
 */
gboolean device_object_map_upload(DeviceObject *object, gchar *service_name,
	guint8 transparent, guint8 retry, guint8 read_status, guint8 type,
	gchar *folder_name, GValueArray *sender, GPtrArray *recipient,
	guint32 msg_len, GPtrArray *attachment_list, guint8 language,
	guint8 encoding, DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;
	UploadParam_t *up;
	MsgInfo_t *mi;
	guint service_name_len;
	guint num_recipients, num_attachments;
	guint i;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	up = g_new0(UploadParam_t, 1);
	mi = g_new0(MsgInfo_t, 1);
	up->pstMsgInfo = mi;

	service_name_len = service_name ? strlen(service_name) : 0;
	if (service_name_len > 255)
		service_name_len = 255;
	up->ucTransparent = transparent;
	up->ucRetry = retry;
	mi->ucReadStatus = read_status;
	mi->ucType = type;
	mi->stFolder.usFolderNameLen = folder_name ? strlen(folder_name) : 0;
	mi->stFolder.pucFolderName = (u8 *) folder_name;

	/* marshal the sender info */
	i = 0;
	mi->stOriginator.pucN = (u8 *)
		g_value_get_string(g_value_array_get_nth(sender, i++));
	mi->stOriginator.ucNLen =
		mi->stOriginator.pucN ? strlen((gchar *) mi->stOriginator.pucN) : 0;
	mi->stOriginator.pucFN = (u8 *)
		g_value_get_string(g_value_array_get_nth(sender, i++));
	mi->stOriginator.ucFNLen =
		mi->stOriginator.pucFN ? strlen((gchar *) mi->stOriginator.pucFN) : 0;
	mi->stOriginator.pucTEL = (u8 *)
		g_value_get_string(g_value_array_get_nth(sender, i++));
	mi->stOriginator.ucTELLen =
		mi->stOriginator.pucTEL ? strlen((gchar *) mi->stOriginator.pucTEL) : 0;
	mi->stOriginator.pucEMAIL = (u8 *)
		g_value_get_string(g_value_array_get_nth(sender, i++));
	mi->stOriginator.ucEMAILLen =
		mi->stOriginator.pucEMAIL ? strlen((gchar *) mi->stOriginator.pucEMAIL) : 0;

	/* marshal the recipient info */
	num_recipients = recipient ? recipient->len : 0;
	if (num_recipients > MAX_RECEIVER_NUM)
		num_recipients = MAX_RECEIVER_NUM;
	mi->ucReceiverNum = num_recipients;
	DEBUG("num_recipients=%u", num_recipients);
	for (i = 0; i < num_recipients; i++) {
		GValue *value;

		value = g_new0(GValue, 1);
		g_value_init(value, DBUS_STRUCT_STRING_STRING_STRING_STRING_UCHAR);
		g_value_set_boxed(value,
			g_ptr_array_index(recipient, i));
		dbus_g_type_struct_get(value,
			0, &mi->stRecipient[i].pucN,
			1, &mi->stRecipient[i].pucFN,
			2, &mi->stRecipient[i].pucTEL,
			3, &mi->stRecipient[i].pucEMAIL,
			4, &mi->stRecipient[i].ucBENVLevel,
			G_MAXUINT);
		mi->stRecipient[i].ucNLen =
			strlen((gchar *) (mi->stRecipient[i].pucN));
		mi->stRecipient[i].ucFNLen =
			strlen((gchar *) (mi->stRecipient[i].pucFN));
		mi->stRecipient[i].ucTELLen =
			strlen((gchar *) (mi->stRecipient[i].pucTEL));
		mi->stRecipient[i].ucEMAILLen =
			strlen((gchar *) (mi->stRecipient[i].pucEMAIL));
		DEBUG("recipient[%d]: N=\"%s\", NLen=%d, "
			"FN=\"%s\", FNLen=%d, "
			"TEL=\"%s\", TELLen=%d, "
			"EMAIL=\"%s\", EMAILLen=%d, BENVLevel=%u",
			i, mi->stRecipient[i].pucN, mi->stRecipient[i].ucNLen,
			mi->stRecipient[i].pucFN, mi->stRecipient[i].ucFNLen,
			mi->stRecipient[i].pucTEL, mi->stRecipient[i].ucTELLen,
			mi->stRecipient[i].pucEMAIL, mi->stRecipient[i].ucEMAILLen,
			mi->stRecipient[i].ucBENVLevel);
		g_free(value);
	}

	mi->ulMsgLen = msg_len;

	/* marshal the attachment list */
	num_attachments = attachment_list ? attachment_list->len : 0;
	if (num_attachments > MAX_ATTACHMENT_NUM)
		num_attachments = MAX_ATTACHMENT_NUM;
	mi->ucAttachNum = num_attachments;
	DEBUG("num_attachments=%u", num_attachments);
	for (i = 0; i < num_attachments; i++) {
		GValue *value;

		value = g_new0(GValue, 1);
		g_value_init(value, DBUS_STRUCT_STRING_UINT);
		g_value_set_boxed(value,
			g_ptr_array_index(attachment_list, i));
		dbus_g_type_struct_get(value,
			0, &mi->stAttachmentList[i].ucName,
			1, &mi->stAttachmentList[i].ulSize,
			G_MAXUINT);
		mi->stAttachmentList[i].usNameLen =
			strlen((gchar *) (mi->stAttachmentList[i].ucName));
		DEBUG("attachment_list[%d]: Name=\"%s\", NameLen=%d, "
			"Size=%lu", i, mi->stAttachmentList[i].ucName,
			mi->stAttachmentList[i].usNameLen,
			mi->stAttachmentList[i].ulSize);
		g_free(value);
	}

	mi->ucLanguage = language;
	mi->ucEncoding = encoding;

	DEBUG("service_name=\"%s\", transparent=%u, retry=%u, read_status=%u, "
		"type=%u, folder_name=\"%s\", msg_len=%u, language=%u, "
		"encoding=%u", service_name ? service_name : "",
		transparent, retry, read_status, type,
		folder_name ? folder_name : "", msg_len, language, encoding);

#ifndef USE_DUMMY_API
	result = BT_APPL_MAP_Upload_REQ(*object->device_address,
		service_name_len, (u8 *) service_name, up);
#else
	result = BT_APPL_MAP_Upload_REQ_DUMMY(*object->device_address,
		service_name_len, (u8 *) service_name, up);
#endif

	DEBUG("BT stack function returned");

	g_free(mi);
	g_free(up);

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_Upload_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_MAP_Upload_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "MapUpload"
 *
 * asynchronous method: return phase
 */
gboolean device_object_map_upload_RTN(BD_ADDRESS device_address,
	guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	DEBUG("status=%u", status);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_MAP_Upload_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}


gboolean device_object_map_upload_data(DeviceObject *object,
	gchar *service_name, guint8 data_flag, GByteArray *data,
	 GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	DEBUG("service_name=\"%s\" data_flag=%u",
		service_name ? service_name : "", data_flag);

#ifndef USE_DUMMY_API
	result = BT_APPL_MAP_Upload_RES(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name,
		data_flag, data->len, data->data);
#else
	result = BT_APPL_MAP_Upload_RES_DUMMY(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name,
		data_flag, data->len, data->data);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_MAP_Upload_RES",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "MapAbort"
 *
 * asynchronous method: call phase
 */
gboolean device_object_map_abort(DeviceObject *object,
	gchar *service_name, DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(context != NULL);

	DEBUG("service_name=\"%s\"", service_name ? service_name : "");

#ifndef USE_DUMMY_API
	result = BT_APPL_MAP_Abort_REQ(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name);
#else
	result = BT_APPL_MAP_Abort_REQ_DUMMY(*object->device_address,
		service_name ? strlen(service_name) : 0, (u8 *) service_name);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_MAP_Abort_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_device, context,
						ID_MAP_Abort_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "MapAbort"
 *
 * asynchronous method: return phase
 */
gboolean device_object_map_abort_RTN(BD_ADDRESS device_address,
	guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	DeviceObject *matched_device = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	DEBUG("status=%u", status);

	/* find the device object relating to the device address */
	matched_device = find_device_object_from_list(device_address);

	if (matched_device != NULL) {

		context = utils_async_queue_find_list_entry(
						&matched_device->aql_device,
						ID_MAP_Abort_CFM);

		if (context != NULL) {
			/* found the context info, so send reply */
			dbus_g_method_return(context, status);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("No device context so cannot reply on D-bus");
		}
	} else {
		DEBUG_ERROR("Bluetooth device address: %s "
			"not found so cannot reply on D-bus",
			bd_addr_to_str(device_address));
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

