/*
 * node_adapter.c
 * Adapter node handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <glib-object.h>
#include <glib/gprintf.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <string.h>
#include <sys/stat.h>

#include "bt_appl/common_def.h"

#include "main.h"
#include "utils_async_queue.h"
#include "device_info.h"
#include "node_device.h"
#include "node_adapter.h"
#include "btd_audio.h"
#define DO_NOT_INCLUDE_CALLBACKS (1)
#include "handle_spp_service.h"
#include "handle_pnp_info.h"
#include "debug.h"
#include "marshal.h"

#include "bt_appl/api/bt_appl_bt_setting_block.h"
#include "bt_appl/api/bt_appl_virtual_device_block.h"
#include "bt_appl_dummy.h"

#include "dbus_error.h"

/************/
/* #defines */
/************/

/***********************/
/* ADAPTER DEFINITIONS */
/***********************/

/* record of device list entry */
struct device_list_entry {
	BD_ADDRESS device_address;

	/* FIXME: Ensure this is big enough for the object path string */
	gchar device_object_path[50];
	DeviceObject *device_object_info;
};

/* OUR ADAPTER OBJECT DEFINITION */
typedef struct {
	/* The parent class object state. */
	GObject parent;

	/* holds the proxy to D-bus */
	DBusGConnection *bus;

	/* holds the device list*/
	GSList *device_list;

	/* async queue list */
	struct async_queue_list aql_adapter;

	/* multimedia audio streaming */
	struct btd_audio aud;

} AdapterObject;

/* OUR CLASS DEFINITION */
typedef struct {
	/* The parent class state. */
	GObjectClass parent;

	/* put adapter signal stuff here, (if any) */

	/* holds the allocated signal numbers */
	guint DeviceCreated_sig_id;
	guint DeviceRemoved_sig_id;
	guint ConnectionComplete_sig_id;
	guint ServiceType_sig_id;
	guint Versions_sig_id;
	guint SspNumericConfirmInd_sig_id;
	guint ConnectionRequestInd_sig_id;

	/* internet access signals */
	guint NapDestroyDeviceInd_sig_id;
	guint NapSppCapabilities_sig_id;
	guint PnPInfoInd_sig_id;
	guint InquiryInd_sig_id;
	guint InquiryCompleteInd_sig_id;
} AdapterObjectClass;

/* LOCAL GLOBAL */
/* adapter object pointer */
static AdapterObject *adapter_object = NULL;


/*
 * MAGIC here to define our adapter object functions etc. in GLib
 * declares adapter_object_init(), adapter_object_class_init(),
 * adapter_object_get_type()
 */
G_DEFINE_TYPE(AdapterObject, adapter_object, G_TYPE_OBJECT)

/*
 * Forward declaration of the function that will return the GType of
 * the Value implementation. Not used
 */
GType adapter_object_get_type(void);

/*
 * Macro for the above. It is common to define macros using the
 * naming convention (seen below) for all GType implementations.
 */
#define ADAPTER_TYPE_OBJECT	(adapter_object_get_type())

/* macro to get from the class to object */
#define ADAPTER_TYPE_INSTANCE_GET_CLASS(Instance,\
		GTypeObject,StructType)\
		((StructType*) (void *)(((GTypeInstance*) Instance)->g_class))

#define ADAPTER_OBJECT_GET_CLASS(obj) \
	(ADAPTER_TYPE_INSTANCE_GET_CLASS ((obj), \
	ADAPTER_TYPE_OBJECT, AdapterObjectClass))

/*********************/
/* ADAPTER CALLBACKS */
/*********************/

/*
 * callback handling, parameters needs to match marshalling signatures
 *
 * callbacks for the auto generated method protoypes
 * Forward declare the prototypes of the callback method handlers for the
 * dbus_glib_adapter_object_methods[]
 */

gboolean adapter_object_find_device(AdapterObject *object,
						GByteArray *remote_addr_in,
						gchar **device_path_out,
						GError **error);

gboolean adapter_object_list_devices(AdapterObject *object,
						GPtrArray **devices_out,
 						GError **error);

gboolean adapter_object_start_connection(AdapterObject *object,
						GByteArray *remote_addr_in,
						guint8 service_in,
						GError **error);

gboolean adapter_object_start_wait_connection(AdapterObject *object,
							GError **error);

gboolean adapter_object_stop_wait_connection(AdapterObject *object,
							GError **error);

gboolean adapter_object_set_local_name(AdapterObject *object, gchar *name_in,
							GError **error);

gboolean adapter_object_set_mode(AdapterObject *object,
						gboolean discoverable_in,
						gboolean connectable_in,
						GError **error);

gboolean adapter_object_set_ssp_mode(AdapterObject *object,
						guint8 ssp_mode_in,
						GError **error);

gboolean adapter_object_start_pairing(AdapterObject *object,
							GError **error);

gboolean adapter_object_stop_pairing(AdapterObject *object, GError **error);

gboolean adapter_object_service_search(AdapterObject *object,
						GByteArray *remote_addr_in,
						guint8 service_type_in,
						GError **error);

gboolean adapter_object_ssp_numeric_confirm_res(AdapterObject *object,
						GByteArray *remote_addr_in,
						gboolean accept_in,
						GError **error);

gboolean adapter_object_connection_request_res(AdapterObject *object,
						GByteArray *remote_addr_in,
						guint8 service_in,
						gboolean accept_in,
						GError **error);

gboolean adapter_object_register_pnp_info_req(AdapterObject *object,
						const guint specificationID,
						const guint vendorID,const guint productID,
						const guint version,const gboolean primaryRecord,
						const guint vendorIDSource,const GValueArray* aucURLArray,
						const guint* recordHandle,GError **error);

gboolean adapter_object_unregister_pnp_info(AdapterObject *object,
						const guint recordHandle, GError **error);

gboolean adapter_object_write_inquiry_mode(AdapterObject *object,
						const guint8 mode,
						GError **error);

gboolean adapter_object_inquiry_request(AdapterObject *object,
						guint8 numResponses,
						guint8 inquiryLength,
						GError **error);

gboolean adapter_object_inquiry_cancel_request(AdapterObject *object,
						GError **error);

gboolean adapter_object_initiate_pair(AdapterObject *object,
						const GByteArray *remote_addr,
						GError **error);

gboolean adapter_object_cancel_pair(AdapterObject *object,
						const GByteArray *remote_addr,
						GError **error);

/*****************************/
/* Internet Access Interface */
/*****************************/

gboolean adapter_object_set_serial_device_param(AdapterObject *object,
						GByteArray *remote_addr_in,
						guint8 service_in,
						guint8 service_handle_in,
						GError **error);

gboolean adapter_object_nap_create_device(AdapterObject *object,
					GByteArray *remote_addr,
					guint8 service,
					DBusGMethodInvocation *context);

gboolean adapter_object_nap_destroy_device(AdapterObject *object,
					GByteArray *remote_addr,
					guint8 service,
					DBusGMethodInvocation *context);

/*******************/
/* Audio Interface */
/*******************/

gboolean adapter_object_start_audio_pipe(AdapterObject *object,
                                         GByteArray *addr,
					 gchar **pipe_name,
					 guint8 *codec_type,
					 guint *bitrate,
					 GError **error);

gboolean adapter_object_stop_audio_pipe(AdapterObject *object, GError **error);

gboolean adapter_object_get_audio_status(AdapterObject *object,
					 gboolean *active,
					 GByteArray **addr,
					 gchar **pipe_name,
					 guint8 *codec_type,
					 guint *bitrate,
					 guint *bytes_streamed,
					 GError **error);

/* adapter device management functions */
static void register_device_list(AdapterObject *object);
static gchar *find_object_path_from_list(BD_ADDRESS our_address);

/* async method handling functions */
static void register_adapter_async_queue_list(AdapterObject *object);
#ifdef NOT_USED
static void unregister_adapter_async_queue_list(AdapterObject *object);
#endif


/* include auto-generated binding information */
#include "alps-bt-adapter-introspection.h"

/***********************************/
/* OBJECT AND CLASS INITIALIZATION */
/***********************************/


/*
 * AdapterObject object initialiser function
 *
 * Used to initialise each object instance via g_object_new()
 */
static void adapter_object_init(AdapterObject* object)
{
	DEBUG_FUNC("Called");
	g_assert(object != NULL);

	/* add code to initialise created object contents */
}

/*
 * AdapterObject class initialiser function
 *
 * Used to initialise each class via the first call of g_object_new()
 */
static void adapter_object_class_init(AdapterObjectClass* klass)
{
#ifdef NOT_USED
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
#endif

	DEBUG_FUNC("Called");
	g_assert(klass != NULL);

	guint signalId;

	/* register new signals here */

	/* Signal: DeviceCreated */
	signalId = g_signal_new("device_created", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__OBJECT,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. We only have one. */
			DBUS_TYPE_G_OBJECT_PATH);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->DeviceCreated_sig_id = signalId;


	/* Signal: DeviceRemoved */
	signalId = g_signal_new("device_removed", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			g_cclosure_marshal_VOID__OBJECT,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. We only have one. */
			DBUS_TYPE_G_OBJECT_PATH);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->DeviceRemoved_sig_id = signalId;


	/* Signal: ConnectionComplete */
	signalId = g_signal_new("connection_complete", /* name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__BOXED_BOOLEAN_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			3,
			/* GType(s) of the parameters. We only have one. */
			DBUS_TYPE_G_UCHAR_ARRAY,
			G_TYPE_BOOLEAN,
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->ConnectionComplete_sig_id = signalId;


	/* Signal: ServiceType */
	signalId = g_signal_new("service_type", /* name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__BOXED_UINT,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. We only have one. */
			DBUS_TYPE_G_UCHAR_ARRAY,
			G_TYPE_UINT);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->ServiceType_sig_id = signalId;


	/* Signal: Versions */
	signalId = g_signal_new("versions", /* name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__STRING_STRING,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. We only have one. */
			G_TYPE_STRING,
			G_TYPE_STRING);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->Versions_sig_id = signalId;


	/* Signal: SspNumericConfirmInd */
	signalId = g_signal_new("ssp_numeric_confirm_ind", /* name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__BOXED_STRING_UINT,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			3,
			/* GType(s) of the parameters. We only have one. */
			DBUS_TYPE_G_UCHAR_ARRAY,
			G_TYPE_STRING,
			G_TYPE_UINT);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->SspNumericConfirmInd_sig_id = signalId;


	/* Signal: ConnectionRequestInd */
	signalId = g_signal_new("connection_request_ind", /* name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__BOXED_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. We only have one. */
			DBUS_TYPE_G_UCHAR_ARRAY,
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->ConnectionRequestInd_sig_id = signalId;


	/* Signal: NapDestroyDeviceInd */
	signalId = g_signal_new("nap_destroy_device_ind", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__BOXED_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. */
			DBUS_TYPE_G_UCHAR_ARRAY,
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->NapDestroyDeviceInd_sig_id = signalId;


	/* Signal: NapSppCapabilities */
	signalId = g_signal_new("nap_spp_capabilities", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__BOXED_BOXED,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. */
			DBUS_TYPE_G_UCHAR_ARRAY,
			DBUS_GPTRARRAY_STRUCT_AUCHAR_AUCHAR_UCHAR_STRING);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->NapSppCapabilities_sig_id = signalId;


	/* Signal: PnPInfoInd */
	signalId = g_signal_new("pnp_info_ind", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__BOXED_BOXED,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. */
			DBUS_TYPE_G_UCHAR_ARRAY,
			DBUS_GPTRARRAY_STRUCT_UINT_UINT_UINT_UINT_UCHAR_UINT_AUCHAR_AUCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->PnPInfoInd_sig_id = signalId;

	
	/* Signal: InquiryInd */
	signalId = g_signal_new("inquiry_ind", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			//marshal_VOID__UCHAR_UCHAR_UCHAR_UINT_UINT_INT_STRING,
			  marshal_VOID__BOXED_UCHAR_UCHAR_UCHAR_UINT_UINT_INT_STRING,
			//VOID:BOXED,UCHAR,UCHAR,UCHAR,UINT,UINT,INT,STRING
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			8,
			/* GType(s) of the parameters. */
			DBUS_TYPE_G_UCHAR_ARRAY,
			G_TYPE_UCHAR,
			G_TYPE_UCHAR,
			G_TYPE_UCHAR,
			G_TYPE_UINT,
			G_TYPE_UINT,
			G_TYPE_INT,
			G_TYPE_STRING
			);
	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->InquiryInd_sig_id = signalId;


	/* Signal: InquiryCompleteInd */
	signalId = g_signal_new("inquiry_complete_ind", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__BOOLEAN,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			1,
			/* GType(s) of the parameters. */
			G_TYPE_BOOLEAN);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->InquiryCompleteInd_sig_id = signalId;


	DEBUG_HIGH("Binding our GObject with D-bus...");
	/* Uses auto-generated methods and signal structure */
	dbus_g_object_type_install_info(ADAPTER_TYPE_OBJECT,
		&dbus_glib_adapter_object_object_info);

}


/****************************/
/* signal sending functions */
/****************************/

/* Emit the "DeviceCreated" signal. */
static void emit_device_created(AdapterObject* object, gchar *object_path)
{
	DEBUG_FUNC("Called");

	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* klass = ADAPTER_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->DeviceCreated_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			object_path);
}

/* Emit the "DeviceRemoved" signal. */
static void emit_device_removed(AdapterObject* object, gchar *object_path)
{
	DEBUG_FUNC("Called");

	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* klass = ADAPTER_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->DeviceRemoved_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			object_path);
}

/*
 * exported function
 *
 * Emit the "ConnectionComplete" signal.
 */
gboolean emit_connection_complete(BD_ADDRESS device_address, gboolean status,
								guint8 service)
{
	AdapterObject *object = adapter_object;	/* use local global */
	GArray *g_bd_addr;

	DEBUG_FUNC("Called");

	/* track the AVP service connection state */
	if (!status && (service == 0x01)) {
		DeviceObject *device_object;

		/* find the device object relating to the device address */
		device_object = find_device_object_from_list(device_address);
		if (device_object)
			device_object->a2dp_info.connected = TRUE;
	}

	g_bd_addr = g_array_sized_new(FALSE, FALSE, sizeof(guint8),
				       sizeof(BD_ADDRESS));
	g_bd_addr = g_array_append_vals(g_bd_addr, device_address,
				         sizeof(BD_ADDRESS));

	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* klass = ADAPTER_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->ConnectionComplete_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			g_bd_addr,
			status,
			service);

	g_array_unref(g_bd_addr);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "ServiceType" signal.
 */
gboolean emit_service_type(BD_ADDRESS device_address, guint32 service_type)
{
	AdapterObject *object = adapter_object;	/* use local global */
	GArray *g_bd_addr;

	DEBUG_FUNC("Called");

	g_bd_addr = g_array_sized_new(FALSE, FALSE, sizeof(guint8),
				       sizeof(BD_ADDRESS));
	g_bd_addr = g_array_append_vals(g_bd_addr, device_address,
				         sizeof(BD_ADDRESS));

	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* klass = ADAPTER_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->ServiceType_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			g_bd_addr,
			service_type);

	g_array_unref(g_bd_addr);

	DEBUG_FUNC("Exited");
	return TRUE;
}

#define MAX_BTD_ALPS_STRING	(50)
/*
 * exported function
 *
 * Emit the "Versions" signal.
 */
gboolean emit_versions(void)
{
	AdapterObject *object = adapter_object;	/* use local global */

	DEBUG_FUNC("Called");

	gchar btd_string[MAX_BTD_ALPS_STRING];
	gchar alps_string[MAX_BTD_ALPS_STRING];

	guint16 alps_version = get_alps_stack_version();
	guint8 alps_test_version = get_alps_stack_test_version();

	g_snprintf(&btd_string[0], MAX_BTD_ALPS_STRING, "%s", D_BUS_BT_VERSION);

	if(alps_test_version == 0) {
		g_snprintf(&alps_string[0], MAX_BTD_ALPS_STRING,
				"ALPS version: %u.%02u",
				alps_version/256, alps_version%256);
	} else {
		g_snprintf(&alps_string[0], MAX_BTD_ALPS_STRING,
				"ALPS version: %u.%02u: test build %u",
				alps_version/256,
				alps_version%256,
				alps_test_version);
	}

	/* Show the versions */
	DEBUG_WARN("%s, %s", &btd_string[0], &alps_string[0]);

	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* klass = ADAPTER_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->Versions_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			&btd_string[0],
			&alps_string[0]);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "SspNumericConfirmInd" signal.
 */
gboolean emit_ssp_numeric_confirm_ind(BD_ADDRESS device_address,
					gchar *name,
					guint32 numeric_value)
{
	AdapterObject *object = adapter_object;	/* use local global */
	GArray *g_bd_addr;

	DEBUG_FUNC("Called");

	g_bd_addr = g_array_sized_new(FALSE, FALSE, sizeof(guint8),
				       sizeof(BD_ADDRESS));
	g_bd_addr = g_array_append_vals(g_bd_addr, device_address,
				         sizeof(BD_ADDRESS));

	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* klass = ADAPTER_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->SspNumericConfirmInd_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			g_bd_addr,
			name,
			numeric_value);

	g_array_unref(g_bd_addr);
	g_free(name);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "ConnectionRequestInd" signal.
 */
gboolean emit_connection_request_ind(BD_ADDRESS device_address,
					guint8 service)
{
	AdapterObject *object = adapter_object;	/* use local global */
	GArray *g_bd_addr;

	DEBUG_FUNC("Called");

	g_bd_addr = g_array_sized_new(FALSE, FALSE, sizeof(guint8),
				       sizeof(BD_ADDRESS));
	g_bd_addr = g_array_append_vals(g_bd_addr, device_address,
				         sizeof(BD_ADDRESS));

	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* klass = ADAPTER_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->ConnectionRequestInd_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			g_bd_addr,
			service);

	g_array_unref(g_bd_addr);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "InquiryInd" signal.
 */
gboolean emit_inquiry_ind(BD_ADDRESS bd_addr,
						guint8 rep_mode,
						guint8 major_device_class,
						guint8 minor_device_class,
						guint32 major_service_class,
						guint32 clock_offset,
						gint32 rssi,
						gchar* eir_data)
{
	AdapterObject *object = adapter_object;
	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* klass = ADAPTER_OBJECT_GET_CLASS(object);
	GArray *g_bd_addr;
	g_bd_addr = g_array_sized_new(FALSE, FALSE, sizeof(guint8),
				       sizeof(BD_ADDRESS));
	g_bd_addr = g_array_append_vals(g_bd_addr, bd_addr,
				         sizeof(BD_ADDRESS));

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->InquiryInd_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			g_bd_addr,
			rep_mode,
			major_device_class,
			minor_device_class,
			major_service_class,
			clock_offset,
			rssi,
			eir_data);

	g_array_unref(g_bd_addr);
	g_free(eir_data);

	DEBUG_FUNC("Exited");
	return TRUE;
}


/*
 * exported function
 *
 * Emit the "InquiryCompleteInd" signal.
 */
gboolean emit_inquiry_complete_ind(gboolean status)
{
	AdapterObject *object = adapter_object;

	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* klass = ADAPTER_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->InquiryCompleteInd_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			status);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "NapDestroyDeviceInd" signal.
 */
gboolean emit_nap_destroy_device_ind(BD_ADDRESS device_address, guint8 service)
{
	AdapterObject *object = adapter_object;	/* use local global */
	GArray *g_bd_addr;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	g_bd_addr = g_array_sized_new(FALSE, FALSE, sizeof(guint8),
				       sizeof(BD_ADDRESS));
	g_bd_addr = g_array_append_vals(g_bd_addr, device_address,
				         sizeof(BD_ADDRESS));

	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* klass = ADAPTER_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->NapDestroyDeviceInd_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			g_bd_addr,
			service);

	g_array_unref(g_bd_addr);

	DEBUG_FUNC("Exited");
	return TRUE;
}

/*
 * exported function
 *
 * Emit the "NapSppCapabilities" signal.
 */
gboolean emit_nap_spp_capabilities(BD_ADDRESS *device_address,
					GPtrArray *spp_capabilities_payload)
{
	AdapterObject *object = adapter_object;	/* use local global */
	GByteArray *g_bd_addr;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);
	g_assert(spp_capabilities_payload != NULL);

	/* put the address into a GByteArray for the signal */
	g_bd_addr = g_byte_array_new();
	g_assert(g_bd_addr != NULL);

	g_bd_addr = g_byte_array_append(g_bd_addr,
					(const guint8 *) device_address,
				         sizeof(BD_ADDRESS));

	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* klass = ADAPTER_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->NapSppCapabilities_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			g_bd_addr,
			spp_capabilities_payload);

	g_byte_array_unref(g_bd_addr);

	DEBUG_FUNC("Exited");
	return TRUE;
}

//External function definition for emit "PnPInfoInd" signal
gboolean emit_pnp_info_ind(BD_ADDRESS *bd_addr,
							GPtrArray* all_pnp_info_ind_data)
{
	AdapterObject *object = adapter_object;
	GByteArray *g_bd_addr;

	DEBUG_FUNC("Called");

	g_assert(bd_addr != NULL);
	g_assert(all_pnp_info_ind_data != NULL);

	// put the address into a GByteArray for the signal
	g_bd_addr = g_byte_array_new();
	g_assert(g_bd_addr != NULL);
	g_bd_addr = g_byte_array_append(g_bd_addr,
					(const guint8 *) bd_addr,
				         sizeof(BD_ADDRESS));

	/*
	 * Track the object to its class information
	 */
	AdapterObjectClass* class = ADAPTER_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
		g_signal_emit( /*
				* Instance of the object that is generating this
				* signal. This will be passed as the first parameter
				* to the signal handler (eventually). But obviously
				* when speaking about D-Bus, a signal caught on the
				* other side of D-Bus will be first processed by
				* the GLib-wrappers (the object proxy) and only then
				* processed by the signal handler.
				*/
				object,
				/*
				 * Signal id for the signal to generate. These are
				 * stored inside the class state structure.
				 */
				class->PnPInfoInd_sig_id,
				/*
				 * Detail of signal. Since we are not using detailed
				 * signals, we leave this at zero (default).
				 */
				0,
				/*
				 * Data to marshal into the signal.
				 */
				g_bd_addr,
				all_pnp_info_ind_data);

		g_byte_array_unref(g_bd_addr);

		DEBUG_FUNC("Exited");
		//emit the signal
	return TRUE;
}
/************************/
/* ASYNC METHOD QUEUING */
/************************/

/*
 * initialise the async method queue
 */
static void register_adapter_async_queue_list(AdapterObject *object)
{
	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	utils_async_queue_initialise(&object->aql_adapter);

	DEBUG_FUNC("Exited");
}

/* FIXME: find somewhere to call this routine */
/*
 * destroy the async method queue by first cleaning up entries
 */
#ifdef NOT_USED
static void unregister_adapter_async_queue_list(AdapterObject *object)
{
	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	utils_async_queue_destroy(&object->aql_adapter);

	DEBUG_FUNC("Exited");
}
#endif


/********************************/
/* ADAPTER MANAGEMENT FUNCTIONS */
/********************************/

/*
 * Exported function
 *
 * create a new adapter object
 *
 * Inputs:
 * bus = point to the bus proxy
 */
gboolean adapter_object_creation(DBusGConnection *bus, gchar *node_name)
{
	DEBUG_HIGH("Creating an instance of the AdapterObject"
				" and calling initialise functions...");

	adapter_object = g_object_new(ADAPTER_TYPE_OBJECT, NULL);
	if(adapter_object == NULL) {
		DEBUG_ERROR("Failed to create an instance "
			"of a AdapterObject");
		return FALSE;
	}

	DEBUG_HIGH("Registering the adapter GObject onto the D-Bus");
	/* the object's path does not come from the XML file */
	dbus_g_connection_register_g_object(bus,
		node_name,	/* Object's path for the adapter node */
		G_OBJECT(adapter_object));

	/*
	 * save the proxy bus information to allow devices to be
	 * created and removed
	 */
	adapter_object->bus = bus;

	register_device_list(adapter_object);

	/* register the async method queue list */
	register_adapter_async_queue_list(adapter_object);

	/* initialise audio streaming */
	if (btd_audio_initialize(&adapter_object->aud) == FALSE) {
		return FALSE;
	}

#if (defined USE_DUMMY_API) && (!defined USE_TEST_NODE)
	/* TEST CODE */
	{
		BD_ADDRESS device1 = { 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff };
		BD_ADDRESS device2 = { 9, 8, 7, 6, 5, 4 };
		BD_ADDRESS device3 = { 0x35, 0x36, 0x37, 0x38, 0x39, 0x30 };

		adapter_add_device_to_list(device1);
		adapter_add_device_to_list(device2);
		adapter_add_device_to_list(device3);

		adapter_remove_device_from_list(device1);
		adapter_add_device_to_list(device1);
	}
#endif

	return TRUE;
}

/*
 * exported function
 *
 * return a pointer to the btd_audio structure
 */
struct btd_audio *adapter_object_get_btd_audio(void)
{
	return &adapter_object->aud;
}

/*
 * Initialise the device list
 */
static void register_device_list(AdapterObject *object)
{
	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/* initialise the list */
	object->device_list = NULL;
}


/*
 * create a device node and adds the details to the adapter's device list
 *
 * returns FALSE if an error occurs
 * will not create a device if it already exists
 */
gboolean adapter_add_device_to_list(BD_ADDRESS our_address)
{
	AdapterObject *object = adapter_object;	/* use local global */
	struct device_list_entry *list_record = NULL;
	gboolean return_code = TRUE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(our_address != NULL);

	/* first, check that the device does not already exist */
	if (find_device_object_from_list(our_address) == NULL) {

		/* create a record */
		list_record = (struct device_list_entry *)
				g_malloc0(sizeof(struct device_list_entry));

		g_assert(list_record != NULL);

		/* update the record by creating the object path (node name) */
		g_sprintf(list_record->device_object_path,
			"/com/alps/bt/dev_%02x_%02x_%02x_%02x_%02x_%02x",
			our_address[0], our_address[1], our_address[2],
			our_address[3], our_address[4], our_address[5]);

		DEBUG("creating device object path \'%s\'",
					list_record->device_object_path);

		/* update the record by copying the address into the record */
		memcpy(list_record->device_address, our_address,
							sizeof(BD_ADDRESS));

		if (device_object_creation(object->bus,
				&list_record->device_object_path[0],
				&list_record->device_address,
				&list_record->device_object_info) == TRUE) {
			/* device was successfully created */

			/* append the record to the device list */
			object->device_list = g_slist_append(
							object->device_list,
							list_record);

			/* send the DeviceCreated signal */
			emit_device_created(object,
					list_record->device_object_path);

		} else {
			DEBUG_ERROR("Failed to create a device: \'%s\'",
				list_record->device_object_path);

			g_free(list_record);

			return_code = FALSE;
		}
	} else {
		DEBUG_WARN("Device already exists");

		return_code = FALSE;
	}

	return return_code;
}

/*
 * This function removes the device node from D-bus and
 * destroys the device object.
 * Finally, the device is removed from the device list
 *
 * returns FALSE if no entry has been found
 */
gboolean adapter_remove_device_from_list(BD_ADDRESS our_address)
{
	AdapterObject *object = adapter_object;	/* use local global */
	GSList *list_entry;
	struct device_list_entry *list_record = NULL;
	DeviceObject *device_object = NULL;

	gboolean result_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(our_address != NULL);

	/* start with the first entry */
	list_entry = object->device_list;

	while (list_entry != NULL)
	{
		/* spin through the list */
		list_record = list_entry->data;

		/* now compare the device address with the registered devices */
		if ((list_record->device_address[0] == our_address[0]) &&
			(list_record->device_address[1] == our_address[1]) &&
			(list_record->device_address[2] == our_address[2]) &&
			(list_record->device_address[3] == our_address[3]) &&
			(list_record->device_address[4] == our_address[4]) &&
			(list_record->device_address[5] == our_address[5])) {

			DEBUG("Found matching device address");

			/* extract the device info from the record */
			device_object = list_record->device_object_info;

			DEBUG_HIGH("device object %p", device_object);

			g_assert(device_object != NULL);

			/* send the DeviceRemoved signal */
			emit_device_removed(object,
					list_record->device_object_path);

			/* destroy and remove the device node from D-bus */
			device_object_destroy(object->bus, &device_object);

			g_assert(device_object == NULL);

			/* remove the device entry from the list */
			object->device_list = g_slist_delete_link(
							object->device_list,
							list_entry);

			/* free the record as no longer needed */
			g_free(list_record);

			result_code = TRUE;

			/* quit the while loop */
			list_entry = NULL;
		} else {
			/* move to the next entry */
			list_entry = g_slist_next(list_entry);
		}
	}

	if (result_code == FALSE) {
		DEBUG_ERROR("Tried to delete a non-existant device");
	}

	return result_code;
}

/*
 * exported function
 *
 * find the device object of a given device address
 *
 * returns NULL if no entry has been found
 */
DeviceObject *find_device_object_from_list(BD_ADDRESS our_address)
{
	AdapterObject *object = adapter_object;	/* use local global */
	GSList *list_entry;
	struct device_list_entry *list_record = NULL;
	DeviceObject *device_object = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(our_address != NULL);

	/* start with the first entry */
	list_entry = object->device_list;

	while (list_entry != NULL)
	{
		/* spin through the list */
		list_record = list_entry->data;

		/* now compare the device address with the registered devices */
		if ((list_record->device_address[0] == our_address[0]) &&
			(list_record->device_address[1] == our_address[1]) &&
			(list_record->device_address[2] == our_address[2]) &&
			(list_record->device_address[3] == our_address[3]) &&
			(list_record->device_address[4] == our_address[4]) &&
			(list_record->device_address[5] == our_address[5])) {

			DEBUG("Found matching device address");

			/* extract the device info from the record */
			device_object = list_record->device_object_info;

			DEBUG_HIGH("device object %p", device_object);

			/* quit the while loop */
			list_entry = NULL;
		} else {
			/* move to the next entry */
			list_entry = g_slist_next(list_entry);
		}
	}

	return device_object;
}

/*
 * find the object path of a given devive address
 *
 * returns NULL if no entry has been found
 */
static gchar *find_object_path_from_list(BD_ADDRESS our_address)
{
	AdapterObject *object = adapter_object;	/* use local global */
	GSList *list_entry;
	struct device_list_entry *list_record = NULL;
	gchar *object_path = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(our_address != NULL);

	/* start with the first entry */
	list_entry = object->device_list;

	while (list_entry != NULL)
	{
		/* spin through the list */
		list_record = list_entry->data;

		/* now compare the device address with the registered devices */
		if ((list_record->device_address[0] == our_address[0]) &&
			(list_record->device_address[1] == our_address[1]) &&
			(list_record->device_address[2] == our_address[2]) &&
			(list_record->device_address[3] == our_address[3]) &&
			(list_record->device_address[4] == our_address[4]) &&
			(list_record->device_address[5] == our_address[5])) {

			DEBUG("Found matching device address");

			/* extract the device info from the record */
			object_path = list_record->device_object_path;

			DEBUG("object path \'%s\'", object_path);

			/* quit the while loop */
			list_entry = NULL;
		} else {
			/* move to the next entry */
			list_entry = g_slist_next(list_entry);
		}
	}

	return object_path;
}

/*****************************/
/* METHOD CALLBACK FUNCTIONS */
/*****************************/

/*
 * Method call for "FindDevice"
 *
 * Note: This function has no ALPS API function
 */
gboolean adapter_object_find_device(AdapterObject *object,
						GByteArray *remote_addr_in,
						gchar **device_path_out,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;

	gchar *object_path = NULL;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(remote_addr_in != NULL);
	g_assert(device_path_out != NULL);

	DEBUG_HIGH("address length %d", remote_addr_in->len);

	DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
		 remote_addr_in->data[0], remote_addr_in->data[1],
		 remote_addr_in->data[2], remote_addr_in->data[3],
		 remote_addr_in->data[4], remote_addr_in->data[5]);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (remote_addr_in->len == sizeof(BD_ADDRESS)) {

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *)(void *) &remote_addr_in->data[0];

		/* find the object path relating to the device address */
		object_path = find_object_path_from_list(*device_address);

		if (object_path != NULL) {
			*device_path_out = g_strdup(object_path);

			return_code = TRUE;
		} else {
			DEBUG_ERROR("Device address not found");
		}
	} else {
		DEBUG_ERROR("Device address is invalid");
	}

	if (return_code == FALSE) {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
						"adapter_object_find_device",
						error);

	return return_code;
}

/*
 * Method call for "ListDevices"
 *
 * lists the object paths of the remote devices
 *
 * Note: This function has no ALPS API function
 */
gboolean adapter_object_list_devices(AdapterObject *object,
					GPtrArray **devices_out,
					GError **error)
{
	guint num_devices;
	GSList *list_entry;
	struct device_list_entry *list_record;
	GPtrArray *paths;
	(void) error;
	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(devices_out != NULL);

	num_devices = g_slist_length(object->device_list);
	*devices_out = g_ptr_array_sized_new(num_devices);
	paths = *devices_out;

	list_entry = object->device_list;
	while (list_entry != NULL)
	{
		list_record = list_entry->data;
		g_ptr_array_add(paths, g_strdup(list_record->device_object_path));
		list_entry = g_slist_next(list_entry);
	}

	/* Return success. */
	return TRUE;
}

/*
 * Method call for "StartConnection"
 */
gboolean adapter_object_start_connection(AdapterObject *object,
						GByteArray *remote_addr_in,
						guint8 service_in,
						GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(remote_addr_in != NULL);

	DEBUG_HIGH("len = %d", remote_addr_in->len);
	DEBUG_HIGH("size = %d", sizeof(BD_ADDRESS));
	DEBUG("service = %d,", service_in);

	/*
	 * need to trap the situation that the input address length does not
	 * match the length of the BD_ADDRESS
	 */
	if (remote_addr_in->len == sizeof(BD_ADDRESS)) {

		/* point to the data in the GByteArray */
		address = (BD_ADDRESS *)(void *) &remote_addr_in->data[ 0 ];

		/*
		 * Call BTHRESULT BT_APPL_Start_Connection_REQ(
		 *				IN const BD_ADDRESS aucBD_ADDR,
		 *				IN u8 ucService);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_Start_Connection_REQ(*address, service_in);
#else
		result = BT_APPL_Start_Connection_REQ_DUMMY(*address,
								service_in);
#endif

		DEBUG("BT stack function returned");

	} else {
		/* the input address is invalid */
		DEBUG_ERROR("Input address length is invalid %d",
			remote_addr_in->len);

		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Start_Connection_REQ",
						error);

#if (defined USE_DUMMY_API) && (!defined USE_TEST_NODE)
	/* FIXME: */
	/* TEST CODE to send signals */
	{
		emit_connection_complete(*address, TRUE, service_in);
	}
#endif

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "StartWaitConnection"
 */
gboolean adapter_object_start_wait_connection(AdapterObject *object,
								GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/* Call BTHRESULT BT_APPL_Start_Wait_Connection_REQ(void); */
#ifndef USE_DUMMY_API
	result = BT_APPL_Start_Wait_Connection_REQ();
#else
	result = BT_APPL_Start_Wait_Connection_REQ_DUMMY();
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_Start_Wait_Connection_REQ",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "StopWaitConnection"
 */
gboolean adapter_object_stop_wait_connection(AdapterObject *object,
							GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/* Call BTHRESULT BT_APPL_Stop_Wait_Connection_REQ(void); */
#ifndef USE_DUMMY_API
	result = BT_APPL_Stop_Wait_Connection_REQ();
#else
	result = BT_APPL_Stop_Wait_Connection_REQ_DUMMY();
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_Stop_Wait_Connection_REQ",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "SetLocalName"
 */
gboolean adapter_object_set_local_name(AdapterObject *object,
							gchar *utf8_name_in,
							GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;
	guint32 name_len;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(utf8_name_in != NULL);

	/* assume UTF-8 string is null terminated */
	name_len = strlen(utf8_name_in);

	DEBUG_HIGH("len %d", name_len);

	/* check for exceeding the length's parameter u8 capability */
	if (name_len <= 255) {

		/*
		 * Call BTHRESULT BT_APPL_Set_Local_Name_REQ(IN u8 ucNameLen,
		 *					IN const u8 *pucName);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_Set_Local_Name_REQ(name_len,
							(u8 *) utf8_name_in);
#else
		result = BT_APPL_Set_Local_Name_REQ_DUMMY(name_len,
							(u8 *) utf8_name_in);
#endif

		DEBUG("BT stack function returned");

	} else {
		/* the string length is invalid */
		DEBUG_ERROR("Name length is invalid %d", name_len);

		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Set_Local_Name_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "SetMode"
 */
gboolean adapter_object_set_mode(AdapterObject *object,
						gboolean discoverable_in,
						gboolean connectable_in,
						GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/*
	 * Call BTHRESULT BT_APPL_Set_Mode_REQ(IN u8 ucDiscoverable,
	 *						IN u8 ucConnectable);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Set_Mode_REQ(discoverable_in, connectable_in);
#else
	result = BT_APPL_Set_Mode_REQ_DUMMY(discoverable_in, connectable_in);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result, "BT_APPL_Set_Mode_REQ",
									error);

	DEBUG_FUNC("Exited");

	return return_code;
}


/*
 * Method call for "SetSspMode"
 */
gboolean adapter_object_set_ssp_mode(AdapterObject *object,
						guint8 ssp_mode_in,
						GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/*
	 * Call BTHRESULT BT_APPL_Set_SSP_Mode_REQ(IN u8 ucSSPMode);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Set_SSP_Mode_REQ(ssp_mode_in);
#else
	result = BT_APPL_Set_SSP_Mode_REQ_DUMMY(ssp_mode_in);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Set_SSP_Mode_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "StartPairing"
 */
gboolean adapter_object_start_pairing(AdapterObject *object, GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/* show the BTD and ALPS versions for debug purposes */
	emit_versions();

	/* Call BTHRESULT BT_APPL_Start_Pairing_REQ(void); */
#ifndef USE_DUMMY_API
	result = BT_APPL_Start_Pairing_REQ();
#else
	result = BT_APPL_Start_Pairing_REQ_DUMMY();
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Start_Pairing_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "StopPairing"
 */
gboolean adapter_object_stop_pairing(AdapterObject *object, GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/* Call BTHRESULT BT_APPL_Stop_Pairing_REQ(void); */
#ifndef USE_DUMMY_API
	result = BT_APPL_Stop_Pairing_REQ();
#else
	result = BT_APPL_Stop_Pairing_REQ_DUMMY();
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Stop_Pairing_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "ServiceSearch"
 */
gboolean adapter_object_service_search(AdapterObject *object,
						GByteArray *remote_addr_in,
						guint8 service_type_in,
						GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(remote_addr_in != NULL);

	DEBUG_HIGH("len = %d", remote_addr_in->len);
	DEBUG_HIGH("size = %d", sizeof(BD_ADDRESS));
	DEBUG("service_type = %d,", service_type_in);

	/*
	 * need to trap the situation that the input address length does not
	 * match the length of the BD_ADDRESS
	 */
	if (remote_addr_in->len == sizeof(BD_ADDRESS)) {

		/* point to the data in the GByteArray */
		address = (BD_ADDRESS *)(void *) &remote_addr_in->data[ 0 ];

		/*
		 * Call BTHRESULT BT_APPL_Service_Search_REQ(
		 *				IN BD_ADDRESS aucBD_ADDR,
		 *				IN u8 ucServiceType);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_Service_Search_REQ(*address, service_type_in);
#else
		result = BT_APPL_Service_Search_REQ_DUMMY(*address,
							service_type_in);
#endif

		DEBUG("BT stack function returned");

	} else {
		/* the input address is invalid */
		DEBUG_ERROR("Input address length is invalid %d",
			remote_addr_in->len);

		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Service_Search_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "SspNumericConfirmRes"
 */
gboolean adapter_object_ssp_numeric_confirm_res(AdapterObject *object,
						GByteArray *remote_addr_in,
						gboolean accept_in,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(remote_addr_in != NULL);

	DEBUG_HIGH("len = %d", remote_addr_in->len);
	DEBUG_HIGH("size = %d", sizeof(BD_ADDRESS));
	DEBUG("accept = %u", accept_in);

	/*
	 * need to trap the situation that the input address length does not
	 * match the length of the BD_ADDRESS
	 */
	if (remote_addr_in->len == sizeof(BD_ADDRESS)) {

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *)(void *) &remote_addr_in->data[0];

		/*
		 * Call BT_APPL_SSP_Numeric_Confirm_RES(
		 *				IN BD_ADDRESS aucBD_ADDR,
		 *				IN u8 ucAccept);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_SSP_Numeric_Confirm_RES(*device_address,
								accept_in);
#else
		result = BT_APPL_SSP_Numeric_Confirm_RES_DUMMY(*device_address,
								accept_in);
#endif

		DEBUG("BT stack function returned");
	} else {
		/* the input address is invalid */
		DEBUG_ERROR("Input address length is invalid %d",
			remote_addr_in->len);

		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_SSP_Numeric_Confirm_RES (Alt)",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "ConnectionRequestRes"
 */
gboolean adapter_object_connection_request_res(AdapterObject *object,
						GByteArray *remote_addr_in,
						guint8 service_in,
						gboolean accept_in,
						GError **error)
{
	BTHRESULT result = BT_APPL_SUCCESS;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(remote_addr_in != NULL);

	DEBUG_HIGH("len = %d", remote_addr_in->len);
	DEBUG_HIGH("size = %d", sizeof(BD_ADDRESS));
	DEBUG("service = %u", service_in);
	DEBUG("accept = %u", accept_in);

	/*
	 * need to trap the situation that the input address length does not
	 * match the length of the BD_ADDRESS
	 */
	if (remote_addr_in->len == sizeof(BD_ADDRESS)) {

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *)(void *) &remote_addr_in->data[0];

		/*
		 * Call BTHRESULT BT_APPL_Connection_Request_RES(
		 *				IN const BD_ADDRESS aucBD_ADDR,
		 *				IN u8 ucService,
		 *				IN u8 ucAccept);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_Connection_Request_RES(*device_address,
							service_in,
							accept_in);
#else
		result = BT_APPL_Connection_Request_RES_DUMMY(*device_address,
							service_in,
							accept_in);
#endif

		DEBUG("BT stack function returned");
	} else {
		/* the input address is invalid */
		DEBUG_ERROR("Input address length is invalid %d",
			remote_addr_in->len);

		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_Connection_Request_RES (Alt)",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "SetSerialDeviceParam"
 */
gboolean adapter_object_set_serial_device_param(AdapterObject *object,
						GByteArray *remote_addr_in,
						guint8 service_in,
						guint8 service_handle_in,
						GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(remote_addr_in != NULL);

	DEBUG_HIGH("len = %u", remote_addr_in->len);
	DEBUG_HIGH("size = %u", sizeof(BD_ADDRESS));
	DEBUG("service = %u,", service_in);
	DEBUG("service_handle = %u,", service_handle_in);

	/*
	 * need to trap the situation that the input address length does not
	 * match the length of the BD_ADDRESS
	 */
	if (remote_addr_in->len == sizeof(BD_ADDRESS)) {

		/* point to the data in the GByteArray */
		address = (BD_ADDRESS *)(void *) &remote_addr_in->data[ 0 ];

		/*
		 * Call BTHRESULT BT_APPL_Set_Serial_Device_Param_REQ(
		 *				IN const BD_ADDRESS aucBD_ADDR,
		 *				IN u8 ucService,
		 *				IN u8 ucServiceHandle);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_Set_Serial_Device_Param_REQ(
							*address, service_in,
							service_handle_in);
#else
		result = BT_APPL_Set_Serial_Device_Param_REQ_DUMMY(*address,
							service_in,
							service_handle_in);
#endif

		DEBUG("BT stack function returned");

	} else {
		/* the input address is invalid */
		DEBUG_ERROR("Input address length is invalid %d",
			remote_addr_in->len);

		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_Set_Serial_Device_Param_REQ",
					error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "NapCreateDevice"
 *
 * asynchronous method: call phase
 */
gboolean adapter_object_nap_create_device(AdapterObject *object,
					GByteArray *remote_addr,
					guint8 service,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(remote_addr != NULL);
	g_assert(context != NULL);

	DEBUG_HIGH("len = %u", remote_addr->len);
	DEBUG_HIGH("size = %u", sizeof(BD_ADDRESS));
	DEBUG("service = %u,", service);

	/*
	 * need to trap the situation that the input address length does not
	 * match the length of the BD_ADDRESS
	 */
	if (remote_addr->len == sizeof(BD_ADDRESS)) {

		/* point to the data in the GByteArray */
		address = (BD_ADDRESS *)(void *) &remote_addr->data[ 0 ];

		/*
		 * BTHRESULT BT_APPL_Create_Device_REQ(IN BD_ADDRESS aucBD_ADDR,
		 *					IN u8 ucService);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_Create_Device_REQ(*address, service);
#else
		result = BT_APPL_Create_Device_REQ_DUMMY(*address, service);
#endif

		DEBUG("BT stack function returned");

	} else {
		/* the input address is invalid */
		DEBUG_ERROR("Input address length is invalid %d",
			remote_addr->len);

		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_Create_Device_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_adapter, context,
						ID_NAP_Create_Device_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;

}

/*
 * exported function
 *
 * Method call "NapCreateDevice"
 *
 * asynchronous method: return phase
 */
gboolean adapter_object_nap_create_device_RTN(BD_ADDRESS device_address,
							guint8 service,
							guint8 status,
							gchar *device_name)
{
	DBusGMethodInvocation *context = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");
	DEBUG_FUNC("Device Name %s",device_name);

	g_assert(device_address != NULL);
	/* don't check the device_name being NULL as it might be NULL OK */

	context = utils_async_queue_find_list_entry(
			&adapter_object->aql_adapter,
			ID_NAP_Create_Device_CFM);
	if(status == 0)
	{
		DEBUG_FUNC("Changing the file permission is allowed");
		int ret;
		mode_t permission = S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH;
		ret = chmod(device_name,permission);
		if(!ret)
		{
			DEBUG_FUNC("Changing the file permission is success");
		}
		else
		{
			DEBUG_FUNC("Changing the file permission is failed");
		}

	}
	else
	{
		DEBUG_FUNC("Changing the file permission is not allowed");
	}

	if (context != NULL) {
		/* found the context info, so send reply */
		dbus_g_method_return(context, service, status, device_name);

		return_code = TRUE;
	} else {
		DEBUG_ERROR("No device context so cannot reply on D-bus");
	}

	g_free(device_name);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call "NapDestroyDevice"
 *
 * asynchronous method: call phase
 */
gboolean adapter_object_nap_destroy_device(AdapterObject *object,
					GByteArray *remote_addr,
					guint8 service,
					DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(remote_addr != NULL);
	g_assert(context != NULL);

	DEBUG_HIGH("len = %u", remote_addr->len);
	DEBUG_HIGH("size = %u", sizeof(BD_ADDRESS));
	DEBUG("service = %u,", service);

	/*
	 * need to trap the situation that the input address length does not
	 * match the length of the BD_ADDRESS
	 */
	if (remote_addr->len == sizeof(BD_ADDRESS)) {

		/* point to the data in the GByteArray */
		address = (BD_ADDRESS *)(void *) &remote_addr->data[ 0 ];

		/*
		 * BTHRESULT BT_APPL_Destroy_Device_REQ(IN BD_ADDRESS aucBD_ADDR,
		 *						IN u8 ucService);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_Destroy_Device_REQ(*address, service);
#else
		result = BT_APPL_Destroy_Device_REQ_DUMMY(*address, service);
#endif

		DEBUG("BT stack function returned");

	} else {
		/* the input address is invalid */
		DEBUG_ERROR("Input address length is invalid %d",
			remote_addr->len);

		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_Destroy_Device_REQ",
					&error);

	if (return_code == TRUE) {

		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(&object->aql_adapter, context,
						ID_NAP_Destroy_Device_CFM);

	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;

}

/*
 * exported function
 *
 * Method call "NapDestroyDevice"
 *
 * asynchronous method: return phase
 */
gboolean adapter_object_nap_destroy_device_RTN(BD_ADDRESS device_address,
							guint8 service,
							guint8 status)
{
	DBusGMethodInvocation *context = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	context = utils_async_queue_find_list_entry(
						&adapter_object->aql_adapter,
						ID_NAP_Destroy_Device_CFM);

	if (context != NULL) {
		/* found the context info, so send reply */
		dbus_g_method_return(context, service, status);

		return_code = TRUE;
	} else {
		DEBUG_ERROR("No device context so cannot reply on D-bus");
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "RegisterPnPInfo"
 */
gboolean adapter_object_register_pnp_info_req(AdapterObject *object,
						const guint specificationID,
						const guint vendorID,
						const guint productID,
						const guint version,
						const gboolean primaryRecord,
						const guint vendorIDSource,
						const GValueArray* aucURLArray,
						const guint* recordHandle,
						GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	DEBUG_FUNC("Called");
	g_assert(object != NULL);
	g_assert(aucURLArray != NULL);

	//Temporary variables for the URLs
	u8 *cli_exe_url,*serv_des_url,*doc_url;
	u16 string_len;
	guint i; //To copy the urls to the u8 values
	 //Print all the variable value
	
	DEBUG("specificationID = %x", specificationID);
	DEBUG("vendorID = %u", vendorID);
	DEBUG("productID = %u", productID);
	DEBUG("version = %u", version);
	DEBUG("vendorIDSource = %x", vendorIDSource);
	DEBUG("primaryRecord is = %d.", primaryRecord);
	DEBUG("Address of recordHandle = %u", recordHandle);
	//Initialize the above value to the structure
	PnPInfo_AttributePtr_t pnp_info_attribute_ptr = NULL;
	pnp_info_attribute_ptr =
			(PnPInfo_AttributePtr_t)g_malloc0(sizeof(PnPInfo_Attribute_t));
	g_assert(pnp_info_attribute_ptr != NULL);
	pnp_info_attribute_ptr->m_usSpecificationID = specificationID;
	pnp_info_attribute_ptr->m_usVendorID = vendorID;
	pnp_info_attribute_ptr->m_usProductID = productID;
	pnp_info_attribute_ptr->m_usVersion = version;
	pnp_info_attribute_ptr->m_ucPrimaryRecord = primaryRecord;
	pnp_info_attribute_ptr->m_usVendorIDSource = vendorIDSource;

	//Parsing the Optional attributes
	i=0;
	//Store client executable URL
	cli_exe_url = (u8 *)g_value_get_string(g_value_array_get_nth((GValueArray*)aucURLArray,i++));
	string_len = cli_exe_url ? strlen((gchar *)cli_exe_url):0;
	if (string_len > 0)
	{
		if(string_len > CLIENT_EXECUTABLE_URL_MAX_LEN)
			string_len = CLIENT_EXECUTABLE_URL_MAX_LEN;

	g_strlcpy
		(pnp_info_attribute_ptr->m_stOptionalAttribute.m_aucClientExecutableURL,
		cli_exe_url,string_len+1);
	}
	DEBUG("The length of m_usClientExecutableURLLen is %d",string_len);
	pnp_info_attribute_ptr->m_stOptionalAttribute.m_usClientExecutableURLLen
				= string_len;

	//Store Service Description
	serv_des_url = (u8 *)g_value_get_string(g_value_array_get_nth((GValueArray*)aucURLArray,i++));
	string_len = serv_des_url ? strlen((gchar *)serv_des_url):0;
	if (string_len > 0)
	{
	if(string_len > SERVICE_DESCRIPTION_MAX_LEN)
		string_len = SERVICE_DESCRIPTION_MAX_LEN;

	g_strlcpy
		(pnp_info_attribute_ptr->m_stOptionalAttribute.m_aucServiceDescription,
		 serv_des_url,string_len+1);
	}
	DEBUG("The length of m_usServiceDescriptionLen is %d",string_len);
	pnp_info_attribute_ptr->m_stOptionalAttribute.m_usServiceDescriptionLen
					= string_len;

	//Store Documentation URL
	doc_url = (u8 *)g_value_get_string(g_value_array_get_nth((GValueArray*)aucURLArray,i++));
	string_len = doc_url ? strlen((gchar *)doc_url):0;
	if (string_len > 0)
	{
	if(string_len > DOCUMENTATION_URL_MAX_LEN)
		string_len = DOCUMENTATION_URL_MAX_LEN;
	g_strlcpy
			(pnp_info_attribute_ptr->m_stOptionalAttribute.m_aucDocumentationURL,
					doc_url,string_len+1);
	}
	DEBUG("The length of  m_usDocumentationURLLen is %d",string_len);
	pnp_info_attribute_ptr->m_stOptionalAttribute.m_usDocumentationURLLen
						= string_len;

	result = BT_APPL_Register_PnPInfo_REQ(pnp_info_attribute_ptr,(u32 *)recordHandle);
	DEBUG("BT stack function returned");
	return_code = report_any_error_to_dbus(result,
							"BT_APPL_Register_PnPInfo_REQ",
							error);
	DEBUG_FUNC("Exited");
	return return_code;
}

/*
 * Method call for "UnregisterPnPInfo"
 */
gboolean adapter_object_unregister_pnp_info(AdapterObject *object,
						const guint recordHandle,
						GError **error)
{
	BTHRESULT result;
	gboolean return_code = FALSE;
	DEBUG_FUNC("Called");
	g_assert(object != NULL);
	DEBUG("recordHandle = %d", recordHandle);

	result = BT_APPL_Unregister_PnPInfo_REQ(recordHandle);
	DEBUG("BT stack function returned");
	return_code = report_any_error_to_dbus(result,
								"BT_APPL_Register_PnPInfo_REQ",
								error);
	DEBUG_FUNC("Exited");
	return return_code;
}

/*
*   Method call for "WriteInquiryMode"
*/
gboolean adapter_object_write_inquiry_mode(AdapterObject *object,
						const guint8 mode,
						GError **error)
{
   DEBUG_FUNC("Called");
   BTHRESULT result;
   gboolean return_code = FALSE;
   g_assert(NULL != object);
   DEBUG("The inquiry mode is %d",mode);
   result = BT_APPL_Write_InquiryMode_REQ(mode);
   DEBUG("BT stack function returned");
   return_code = report_any_error_to_dbus(result,
						"BT_APPL_Write_InquiryMode_REQ",
						error);
   DEBUG_FUNC("Exited");
   return return_code;
}

/*
 *   Method call for "InquiryRequest"
 */
gboolean adapter_object_inquiry_request(AdapterObject *object,
						guint8 numResponses,
						guint8 inquiryLength,
						GError **error)
{
	DEBUG_FUNC("Called");
	BTHRESULT result;
	gboolean return_code = FALSE;
	g_assert(NULL != object);
	DEBUG("Number of responses is %d",numResponses);
	DEBUG("Amount of time for Inquiry is %d",inquiryLength);
	result = BT_APPL_Inquiry_REQ(numResponses,inquiryLength);
	DEBUG("BT stack function returned");
	return_code = report_any_error_to_dbus(result,
					"BT_APPL_Inquiry_REQ",
					error);
	DEBUG_FUNC("Exited");
	return return_code;
}


/*
 *   Method Call for "InquiryCancelRequest"
 * */

gboolean adapter_object_inquiry_cancel_request(AdapterObject *object,
						GError **error)
{
	DEBUG_FUNC("Called");
	BTHRESULT result;
	gboolean return_code = FALSE;
	g_assert(NULL != object);
	result = BT_APPL_Inquiry_Cancel_REQ();
	DEBUG("BT stack function returned");
	return_code = report_any_error_to_dbus(result,
					"BT_APPL_Inquiry_Cancel_REQ",
					error);
	DEBUG_FUNC("Exited");
	return return_code;
}

/*
*   Method call for "InitiatePair"
*/
gboolean adapter_object_initiate_pair(AdapterObject *object,
						const GByteArray *remote_addr,
						GError **error)
{
	DEBUG_FUNC("Called");
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *address = NULL;
	g_assert(NULL != object);
	DEBUG("Length of BD_Address is %u",remote_addr->len);
	/*
	 * If the requested device address does not match the
	 * length of the BD_ADDRESS then do not send method
	 * request to ALPS
	 * */
	if (remote_addr->len == sizeof(BD_ADDRESS))
	{
		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
				 remote_addr->data[0], remote_addr->data[1],
				 remote_addr->data[2], remote_addr->data[3],
				 remote_addr->data[4], remote_addr->data[5]);
		address = (BD_ADDRESS*)(void *)&remote_addr->data[0];
		result = BT_APPL_Initiate_Pair_REQ(*address);
		DEBUG("BT stack function returned");
	}
	else
	{
		result = BT_APPL_ERROR_PARAMETERS;
	}
	return_code = report_any_error_to_dbus(result,
							"BT_APPL_Initiate_Pair_REQ",
							error);
	DEBUG_FUNC("Exited");
	return return_code;
}

/*
*   Method call for "CancelPair"
*/

gboolean adapter_object_cancel_pair(AdapterObject *object,
						const GByteArray *remote_addr,
						GError **error)
{
	DEBUG_FUNC("Called");
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *address = NULL;
	g_assert(NULL != object);
	DEBUG("Length of BD_Address is %u",remote_addr->len);
	/*
	 * If the requested device address does not match the
	 * length of the BD_ADDRESS then do not send method
	 * request to ALPS
	 * */
	if (remote_addr->len == sizeof(BD_ADDRESS))
	{
		address = (BD_ADDRESS*)(void *)&remote_addr->data[0];
		result = BT_APPL_Cancel_Pair_REQ(*address);
		DEBUG("BT stack function returned");
	}
	else
	{
		DEBUG_ERROR("Input address length is invalid %d",remote_addr->len);
		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
							"BT_APPL_Cancel_Pair_REQ",
							error);
	DEBUG_FUNC("Exited");
	return return_code;
}

/*
 * Method call for "StartAudioPipe"
 */
gboolean adapter_object_start_audio_pipe(AdapterObject *object,
                                         GByteArray *addr,
					 gchar **pipe_name,
					 guint8 *codec_type,
					 guint *bitrate,
					 GError **error)
{
	gboolean return_code = TRUE;
	BTHRESULT result;
	struct btd_audio *aud;
	DeviceObject *device_object;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	aud = &object->aud;

	if (addr->len == sizeof(BD_ADDRESS)) {
		/* find the device object relating to the device address */
		device_object = find_device_object_from_list(addr->data);
		if (device_object && device_object->a2dp_info.connected &&
			device_object->a2dp_info.codec_valid)
		{
			*codec_type = device_object->a2dp_info.codec;
			*bitrate = device_object->a2dp_info.bitrate;
			*pipe_name = btd_audio_get_pipe_name(aud, *codec_type);
			result = btd_audio_start(aud, addr->data, *codec_type,
					*bitrate);
		} else {
			/* spoof an ALPS error code */
 			result = BT_APPL_ERROR_CONNECTION_NOT_EXIST;
			DEBUG("ERROR: No AVP connection or unknown codec");
		}
	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
		DEBUG("ERROR: Device address is not of length 6");
	}

	return_code = report_any_error_to_dbus(result, "StartAudioPipe", error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "StopAudioPipe"
 */
gboolean adapter_object_stop_audio_pipe(AdapterObject *object, GError **error)
{
	gboolean return_code = TRUE;
	struct btd_audio *aud;
	(void)error;
	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	aud = &object->aud;
	btd_audio_stop(aud);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "GetAudioStatus"
 */
gboolean adapter_object_get_audio_status(AdapterObject *object,
					 gboolean *active,
					 GByteArray **addr,
					 gchar **pipe_name,
					 guint8 *codec_type,
					 guint *bitrate,
					 guint *bytes_streamed,
					 GError **error)
{
	gboolean return_code = TRUE;
	struct btd_audio *aud;
	BD_ADDRESS bdaddr;
	(void)error;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	aud = &object->aud;

	*active = aud->started;
	btd_audio_get_status(aud, bdaddr, codec_type, bitrate, bytes_streamed);
	*addr = g_byte_array_new();
	*addr = g_byte_array_append(*addr, bdaddr, sizeof(BD_ADDRESS));
	*pipe_name = btd_audio_get_pipe_name(aud, *codec_type);

	DEBUG_FUNC("Exited");

	return return_code;
}
