/*
 * callbacks.h
 * Handle callbacks from the ALPS Bluetooth stack.
 *
 * Author: Andy Lowe <alowe@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#ifndef _CALLBACKS_H
#define _CALLBACKS_H

/* control the scope of the callback types */
#define INCLUDE_CALLBACK_TYPES	(1)

/*
 * Maximum length (in bytes) of a local or remote Bluetooth device name,
 * including the terminating null.  The name is UTF-8 encoded, so the maximum
 * number of characters in a name varies depending on the number of bytes per
 * character.
 */
#define MAX_NAME_BYTES 256

enum alps_callbacks {
	/* general callbacks */
	BT_APPL_GET_DEVICE_NAME_CFM_CB,
	BT_APPL_ROLE_CHANGE_IND_CB,
	BT_APPL_REQUEST_PIN_IND_CB,
	BT_APPL_SSP_NUMERIC_CONFIRM_IND_CB,
	BT_APPL_SSP_PASSKEY_NOTIFICATION_IND_CB,
	BT_APPL_PAIRING_COMPLETE_IND_CB,
	BT_APPL_SERVICE_TYPE_IND_CB,
	BT_APPL_LINKKEY_REQUEST_IND_CB,
	BT_APPL_CONNECTION_REQUEST_IND_CB,
	BT_APPL_RESET_EVENT_IND,
	BT_APPL_CONNECTION_COMPLETE_IND_CB,
	BT_APPL_DISCONNECTION_IND_CB,
	BT_APPL_AUTHENTICATION_IND_CB,
	BT_APPL_TESTMODE_CFM_CB,
	BT_APPL_TESTMODE_COMPLETE_IND_CB,
	BT_APPL_LINKQUALITY_CFM_CB,
	BT_APPL_SPP_CAPABILITIES_IND_CB,
	
	BT_APPL_PNPINFO_IND_CB,
	BT_APPL_INQUIRY_IND_CB,
	BT_APPL_INQUIRY_COMPLETE_IND_CB,
	BT_APPL_General_Place_Holder_A_CB,
	BT_APPL_General_Place_Holder_B_CB,
	BT_APPL_General_Place_Holder_C_CB,
	/* hfp callbacks */
	BT_APPL_CALL_STATUS_IND_CB,
	BT_APPL_PHONE_STATUS_IND_CB,
	BT_APPL_HF_DIAL_CFM_CB,
	BT_APPL_HF_REDIAL_CFM_CB,
	BT_APPL_HF_MEM_DIAL_CFM_CB,
	BT_APPL_HF_TERMINATE_CALL_CFM_CB,
	BT_APPL_HF_TERMINATE_ALL_CFM_CB,
	BT_APPL_HF_AUDIO_TRANSFER_CFM_CB,
	BT_APPL_HF_SETUP_AUDIO_IND_CB,
	BT_APPL_HF_RELEASE_AUDIO_IND_CB,
	BT_APPL_HF_SEND_DTMF_CFM_CB,
	BT_APPL_HF_INCOMING_CALL_ACT_CFM_CB,
	BT_APPL_HF_RING_IND_CB,
	BT_APPL_HF_RELEASE_INACTIVE_CALL_CFM_CB,
	BT_APPL_HF_RELEASE_ACTIVE_CALL_CFM_CB,
	BT_APPL_HF_SWAP_CALL_CFM_CB,
	BT_APPL_HF_ADD_CALL_CFM_CB,
	BT_APPL_HF_RELEASE_SPEC_CALL_CFM_CB,
	BT_APPL_HF_HOLD_EXCEP_CALL_CFM_CB,
	BT_APPL_HF_EXP_CALL_TRANS_CFM_CB,
	BT_APPL_HF_SUBSCRIBER_CFM_CB,
	BT_APPL_HF_MANUFACTURER_ID_CFM_CB,
	BT_APPL_HF_MODEL_ID_CFM_CB,
	BT_APPL_HF_REVISION_ID_CFM_CB,
	BT_APPL_HF_VOICE_RECOGNITION_ACTIVATION_CFM_CB,
	BT_APPL_HF_VOICE_RECOGNITION_ACTIVATION_IND_CB,
	BT_APPL_HF_BT_APPL_SIRI_XAPL_CFM_CB,
	BT_APPL_HF_BT_APPL_SIRI_NrStat_CFM_CB,
	BT_APPL_HF_BT_APPL_SIRI_NrStat_IND_CB,
	BT_APPL_HF_BT_APPL_SIRI_XAPL_IND_CB,
	BT_APPL_HF_BT_APPL_SIRI_EFM_CFM_CB,
	BT_APPL_HF_RELEASE_WAITING_CALL_CFM_CB,
	BT_APPL_HF_SETUP_AUDIO_WBS_IND_CB,
	BT_APPL_HF_Place_Holder_A_CB,
	BT_APPL_HF_Place_Holder_B_CB,
	BT_APPL_HF_Place_Holder_C_CB,

	/* avp callbacks */
	BT_APPL_AVP_VERSION_IND_CB,
	BT_APPL_AVP_CTRL_CMD_CFM_CB,
	BT_APPL_AVP_CTRL_CMD_IND_CB,
	BT_APPL_AVP_GET_CAPABILITIES_CFM_CB,
	BT_APPL_AVP_LIST_ATTR_CFM_CB,
	BT_APPL_AVP_LIST_VAL_CFM_CB,
	BT_APPL_AVP_SET_VAL_CFM_CB,
	BT_APPL_AVP_PLAY_STATUS_IND_CB,
	BT_APPL_AVP_NOTIFY_SET_COMPLETE_IND_CB,
	BT_APPL_AVP_NOTIFY_CHG_IND_CB,
	BT_APPL_AVP_SET_PLAYER_CFM_CB,
	BT_APPL_AVP_GET_METADATA_ATTR_CFM_CB,
	BT_APPL_AVP_GET_FOLDER_ITEM_CFM_CB,
	BT_APPL_AVP_GET_FOLDER_ITEM_IND_CB,
	BT_APPL_AVP_MEDIA_ELEMENT_IND_CB,
	BT_APPL_AVP_GET_MEDIA_PLAYER_ITEM_CFM_CB,
	BT_APPL_AVP_GET_MEDIA_PLAYER_ITEM_IND_CB,
	BT_APPL_AVP_PLAY_ITEM_CFM_CB,
	BT_APPL_AVP_CHANGE_PATH_CFM_CB,
	BT_APPL_AVP_CODEC_AND_BITRATE_IND_CB,
	BT_APPL_AVP_Place_Holder_A_CB,
	BT_APPL_AVP_Place_Holder_B_CB,
	BT_APPL_AVP_Place_Holder_C_CB,
	/* pbdl callbacks */
	BT_APPL_PBDL_GET_CAPABILITIES_CFM_CB,
	BT_APPL_PBDL_PREPARE_CFM_CB,
	BT_APPL_PBDL_START_CFM_CB,
	BT_APPL_PBDL_STOP_CFM_CB,
	BT_APPL_PBDL_DATA_IND_CB,
	BT_APPL_PBDL_DATA_COMPLETE_IND_CB,
	BT_APPL_PBDL_CALENDAR_COMPLETE_IND_CB,
	BT_APPL_PBDL_COMPLETE_CFM_CB,
	BT_APPL_PBDL_CON_LOST_IND_CB,
	BT_APPL_PBDL_STATUS_CHG_IND_CB,
	BT_APPL_PBDL_Place_Holder_A,
	BT_APPL_PBDL_Place_Holder_B,
	BT_APPL_PBDL_Place_Holder_C,
	/* map callbacks */
	BT_APPL_MAP_GET_CAPABILITIES_CFM_CB,
	BT_APPL_MAP_START_CFM_CB,
	BT_APPL_MAP_ACTIVE_INSTANCE_IND_CB,
	BT_APPL_MAP_NOTIFY_STATUS_IND_CB,
	BT_APPL_MAP_NOTIFY_REGIST_STATUS_IND_CB,
	BT_APPL_MAP_STOP_CFM_CB,
	BT_APPL_MAP_CHG_INSTANCE_CFM_CB,
	BT_APPL_MAP_UPDATE_INBOX_CFM_CB,
	BT_APPL_MAP_LIST_FOLDER_CFM_CB,
	BT_APPL_MAP_LIST_FOLDER_IND_CB,
	BT_APPL_MAP_LIST_FOLDER_COMP_IND_CB,
	BT_APPL_MAP_LIST_MSG_CFM_CB,
	BT_APPL_MAP_LIST_MSG_IND_CB,
	BT_APPL_MAP_LIST_MSG_COMP_IND_CB,
	BT_APPL_MAP_READ_MSG_CFM_CB,
	BT_APPL_MAP_READ_MSG_IND_CB,
	BT_APPL_MAP_READ_MSG_COMP_IND_CB,
	BT_APPL_MAP_SET_MSG_STATUS_CFM_CB,
	BT_APPL_MAP_UPLOAD_CFM_CB,
	BT_APPL_MAP_UPLOAD_IND_CB,
	BT_APPL_MAP_ABORT_CFM_CB,
	BT_APPL_MAP_NOTIFY_IND_CB,
	BT_APPL_MAP_ERROR_REPORT_IND_CB,
	BT_APPL_MSDL_Place_Holder_A,
	BT_APPL_MSDL_Place_Holder_B,
	BT_APPL_MSDL_Place_Holder_C,
	/* internet callbacks */
	BT_APPL_CREATE_DEVICE_CFM_CB,
	BT_APPL_DESTROY_DEVICE_CFM_CB,
	BT_APPL_DESTROY_DEVICE_IND_CB,
	BT_APPL_Device_Place_Holder_A,
	BT_APPL_Device_Place_Holder_B,
	BT_APPL_Device_Place_Holder_C,

	NUM_ALPS_CALLBACKS,
};

/***********/
/* GENERAL */
/***********/
typedef struct general_callback_parms_s {
	u8 name[MAX_NAME_BYTES];
	u32 passkey;
	PairingInfo_t pairing_info;
	BD_ADDRESS bd_addr;
	u8 name_len;
	u8 status;
} general_callback_parms;

typedef struct gen_get_device_name_cfm_s {
	BD_ADDRESS bd_addr;
	gchar *device_name;
} gen_get_device_name_cfm_t;

typedef struct gen_role_change_ind_s {
	BD_ADDRESS bd_addr;
	gboolean status;
	u8 role;
} gen_role_change_ind_t;

typedef struct gen_service_type_ind_s {
	BD_ADDRESS bd_addr;
	u32 service_type;
} gen_service_type_ind_t;

typedef struct gen_connection_complete_ind_s {
	BD_ADDRESS bd_addr;
	gboolean status;
	u8 service;
} gen_connection_complete_ind_t;

typedef struct gen_disconnection_ind_s {
	BD_ADDRESS bd_addr;
	u8 service;
	u8 conn_loss_type;
} gen_disconnection_ind_t;

typedef struct gen_testmode_cfm_s {
	u8 status;
} gen_testmode_cfm_t;

typedef struct gen_testmode_complete_ind_s {
	u8 noti_type;
	u8 status;
} gen_testmode_complete_ind_t;

typedef struct gen_linkqualify_cfm_s {
	u8 status;
	s8 rssi;
	u8 link_quality;
} gen_linkqualify_cfm_t;

typedef struct gen_spp_capabilities_ind_s {
	BD_ADDRESS bd_addr;
	GArray *uuid_index;
	GByteArray *uuid_data;
	u8 service_handle;
	gchar *service_name;
	u8 continue_flag;
} gen_spp_capabilities_ind_t;

typedef struct gen_ssp_numeric_confirm_ind_s {
	BD_ADDRESS bd_addr;
	gchar *name;
	u32 numeric_value;
} gen_ssp_numeric_confirm_ind_t;

typedef struct gen_connection_request_ind_s {
	BD_ADDRESS bd_addr;
	u8 service;
} gen_connection_request_ind_t;

typedef struct gen_pnp_info_attribute_ind_s {
	BD_ADDRESS bd_addr;
	u16 specificationID;
	u16 vendorID;
	u16 productID;
	u16 version;
	u8 primaryRecord;
	u16 vendorIDSource;
	GArray *url_length;
	GByteArray *url_name;
	u8 continue_flag;
}gen_pnp_info_attribute_ind_t;


typedef struct gen_inquiry_complete_ind_s {
	gboolean inquiry_status;
}gen_inquiry_complete_ind_t;

typedef struct codinfo_s {
	u16 major_service_class;
	u8 major_device_class;
	u8 minor_device_class;
}codinfo_t;

typedef struct gen_inquiry_ind_s {
	BD_ADDRESS bd_addr;
	u8 page_scan_rep_mode;
	codinfo_t cod;
	u16 clock_offset;
	s8 rssi;
	gchar* eir_data;
}gen_inquiry_ind_t;

/******/
/* HF */
/******/
typedef struct hf_call_status_ind_s {
	BD_ADDRESS bd_addr;
	u8 call_handle;
	u8 total;
	u8 direction;
	u8 state;
	u8 mode;
	gchar *number;
	u8 type;
} hf_call_status_ind_t;

typedef struct hf_phone_status_ind_s {
	BD_ADDRESS bd_addr;
	u8 service;
	u8 signal;
	u8 roaming;
	u8 battery;
	gchar *operator_name;
	u32 ag_supp_feature;
	u8 AgThreeWayCallSuppFeatures;
} hf_phone_status_ind_t;

typedef struct hf_dial_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_dial_cfm_t;

typedef struct hf_redial_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_redial_cfm_t;

typedef struct hf_mem_dial_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_mem_dial_cfm_t;

typedef struct hf_terminate_call_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_terminate_call_cfm_t;

typedef struct hf_terminate_all_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_terminate_all_cfm_t;

typedef struct hf_audio_transfer_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_audio_transfer_cfm_t;

typedef struct hf_setup_audio_ind_s {
	BD_ADDRESS bd_addr;
} hf_setup_audio_ind_t;

typedef struct hf_release_audio_ind_s {
	BD_ADDRESS bd_addr;
} hf_release_audio_ind_t;

typedef struct hf_send_dtmf_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_send_dtmf_cfm_t;

typedef struct hf_incoming_call_act_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_incoming_call_act_cfm_t;

typedef struct hf_ring_ind_s {
	BD_ADDRESS bd_addr;
	u8 ring_tone;
} hf_ring_ind_t;

typedef struct hf_release_inactive_call_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_release_inactive_call_cfm_t;

typedef struct hf_release_active_call_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_release_active_call_cfm_t;

typedef struct hf_swap_call_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_swap_call_cfm_t;

typedef struct hf_add_call_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_add_call_cfm_t;

typedef struct hf_release_spec_call_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_release_spec_call_cfm_t;

typedef struct hf_hold_excep_call_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_hold_excep_call_cfm_t;

typedef struct hf_exp_call_trans_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
} hf_exp_call_trans_cfm_t;

typedef struct hf_subscriber_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
	GPtrArray *subscriber_info;
} hf_subscriber_cfm_t;

typedef struct hf_manufacturer_id_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
	gchar *manufacturer;
} hf_manufacturer_id_cfm_t;

typedef struct hf_model_id_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
	gchar *model;
} hf_model_id_cfm_t;

typedef struct hf_revision_id_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 code;
	gchar *revision;
} hf_revision_id_cfm_t;

typedef struct hf_voice_recognition_activation_cfm_s {
	BD_ADDRESS bd_addr;
	u8 activation;
	u8 status;
} hf_voice_recognition_activation_cfm_t;

typedef struct hf_voice_recognition_activation_ind_s {
	BD_ADDRESS bd_addr;
	u8 activation;
} hf_voice_recognition_activation_ind_t;

typedef struct appl_xapl_siri_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	gchar *parameter;
} appl_xapl_siri_cfm_t;

typedef struct appl_xapl_siri_ind_s {
	BD_ADDRESS bd_addr;
	u8 status;
} appl_xapl_siri_ind_t;

typedef struct siri_nrstat_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} siri_nrstat_cfm_t;

typedef struct siri_nrstat_ind_s {
	BD_ADDRESS bd_addr;
	u8 ucAplNrStat;
} siri_nrstat_ind_t;

typedef struct siri_efm_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} siri_efm_cfm_t;

typedef struct hf_setup_audio_wbs_ind_s {
	BD_ADDRESS bd_addr;
	u8 audiocodec;
}hf_setup_audio_wbs_ind_t;


/********/
/* PBDL */
/********/
typedef struct pbdl_get_capabilities_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 pb_type;
} pbdl_get_capabilities_cfm_t;

typedef struct pbdl_prepare_cfm_s {
	BD_ADDRESS bd_addr;
	u8 pb_type;
} pbdl_prepare_cfm_t;

typedef struct pbdl_start_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u32 pb_number;
	u32 new_missed_call;
} pbdl_start_cfm_t;

typedef struct pbdl_stop_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} pbdl_stop_cfm_t;

typedef struct pbdl_data_ind_s {
	BD_ADDRESS bd_addr;
	u8 data_type;
	GByteArray *data;
} pbdl_data_ind_t;

typedef struct pbdl_complete_ind_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u32 pb_number;
} pbdl_complete_ind_t;

typedef struct pbdl_calender_complete_ind_s {
	BD_ADDRESS bd_addr;
	u8 status;
} pbdl_calender_complete_ind_t;

typedef struct pbdl_complete_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} pbdl_complete_cfm_t;

typedef struct pbdl_con_lost_ind_s {
	BD_ADDRESS bd_addr;
} pbdl_con_lost_ind_t;

typedef struct pbdl_status_chg_ind_s {
	BD_ADDRESS bd_addr;
	u8 status;
} pbdl_status_chg_ind_t;

/*******/
/* MAP */
/*******/
typedef struct map_get_capabilities_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	GPtrArray *msg_capability;
} map_get_capabilities_cfm_t;

typedef struct map_start_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} map_start_cfm_t;

typedef struct map_active_instance_ind_s {
	BD_ADDRESS bd_addr;
	gchar *sms_name;
	gchar *email_name;
} map_active_instance_ind_t;

typedef struct map_notify_status_ind_s {
	BD_ADDRESS bd_addr;
	u8 status;
} map_notify_status_ind_t;

typedef struct map_notify_regist_status_ind_s {
	BD_ADDRESS bd_addr;
	gchar *service_name;
	u8 status;
} map_notify_regist_status_ind_t;

typedef struct map_stop_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} map_stop_cfm_t;

typedef struct map_chg_instance_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} map_chg_instance_cfm_t;

typedef struct map_update_inbox_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} map_update_inbox_cfm_t;

typedef struct map_list_folder_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 accessible;
	u32 folder_num;
} map_list_folder_cfm_t;

typedef struct map_list_folder_ind_s {
	BD_ADDRESS bd_addr;
	gchar *service_name;
	gchar **folder_listing;
} map_list_folder_ind_t;

typedef struct map_list_folder_comp_ind_s {
	BD_ADDRESS bd_addr;
	gchar *service_name;
	u32 total_size;
	u8 status;
} map_list_folder_comp_ind_t;

typedef struct map_list_msg_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u32 message_num;
} map_list_msg_cfm_t;

typedef struct map_list_msg_data_ind_s {
	BD_ADDRESS bd_addr;
	gchar *service_name;
	GByteArray *handle;
	gchar *msg_subject;
	gchar *msg_date_time;
	gchar *msg_sender_name;
	gchar *msg_sender_address;
	gchar *msg_replyto_address;
	gchar *msg_recipient_name;
	gchar *msg_recipient_address;
	u8 msg_type;
	u32 msg_size;
	u8 msg_text_flag;
	u8 msg_reception_status;
	u32 msg_attach_size;
	u8 msg_priority;
	u8 msg_read;
	u8 msg_sent;
	u8 msg_protect;
} map_list_msg_data_ind_t;

typedef struct map_list_msg_comp_ind_s {
	BD_ADDRESS bd_addr;
	gchar *service_name;
	u32 total_size;
	u8 status;
} map_list_msg_comp_ind_t;

typedef struct map_read_msg_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 read_status;
	u8 type;
	gchar *folder_name;
	GValueArray *sender;
	GPtrArray *recipient;
	u32 msg_len;
	GPtrArray *attachment_list;
	u8 language;
	u8 encoding;
} map_read_msg_cfm_t;

typedef struct map_read_msg_ind_s {
	BD_ADDRESS bd_addr;
	gchar *service_name;
	u8 data_flag;
	GByteArray *msg_data;
} map_read_msg_ind_t;

typedef struct map_read_msg_comp_ind_s {
	BD_ADDRESS bd_addr;
	gchar *service_name;
} map_read_msg_comp_ind_t;

typedef struct map_set_msg_status_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} map_set_msg_status_cfm_t;

typedef struct map_upload_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} map_upload_cfm_t;

typedef struct map_upload_ind_s {
	BD_ADDRESS bd_addr;
	gchar *service_name;
	u8 status;
} map_upload_ind_t;

typedef struct map_abort_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} map_abort_cfm_t;

typedef struct map_notify_ind_s {
	BD_ADDRESS bd_addr;
	gchar *service_name;
	u8 type;
	GByteArray *handle;
	gchar *folder;
	gchar *old_folder;
	u8 msg_type;
} map_notify_ind_t;

typedef struct map_error_report_ind_s {
	BD_ADDRESS bd_addr;
	gchar *service_name;
	u8 operation;
	u8 error_code;
} map_error_report_ind_t;

/*******/
/* AVP */
/*******/
typedef struct avp_version_ind_s {
	BD_ADDRESS bd_addr;
	u8 avrcp_version;
} avp_version_ind_t;

typedef struct avp_ctrl_cmd_cfm_s {
	BD_ADDRESS bd_addr;
	u8 command;
	u8 status_flag;
	u8 status;
} avp_ctrl_cmd_cfm_t;

typedef struct avp_ctrl_cmd_ind_s {
	BD_ADDRESS bd_addr;
	u8 command;
} avp_ctrl_cmd_ind_t;

typedef struct avp_get_capabilities_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 capability_id;
	u16 event_supported;
} avp_get_capabilities_cfm_t;

typedef struct avp_list_attr_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 attributes_bit;
} avp_list_attr_cfm_t;

typedef struct avp_list_val_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	GByteArray *setting_values;
	u8 current_value;
} avp_list_val_cfm_t;

typedef struct avp_set_val_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} avp_set_val_cfm_t;

typedef struct avp_play_status_ind_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u32 song_length;
	u32 song_position;
	u8 play_status;
} avp_play_status_ind_t;

typedef struct avp_notify_set_complete_ind_s {
	BD_ADDRESS bd_addr;
	u32 event_id;
} avp_notify_set_complete_ind_t;

typedef struct avp_notify_chg_ind_s {
	BD_ADDRESS bd_addr;
	u8 event_id;
	GByteArray *data;
} avp_notify_chg_ind_t;

typedef struct avp_set_player_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u32 num_item;
	u8 is_db_aware;
} avp_set_player_cfm_t;

typedef struct avp_get_metadata_attr_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 num_attributes;
	GByteArray *data;
} avp_get_metadata_attr_cfm_t;

typedef struct avp_get_folder_item_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 scope;
	u16 num_items;
} avp_get_folder_item_cfm_t;

typedef struct avp_get_folder_item_ind_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 scope;
	GByteArray *data;
} avp_get_folder_item_ind_t;

typedef struct avp_media_element_ind_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u8 scope;
	GByteArray *data;
} avp_media_element_ind_t;

typedef struct avp_get_media_player_item_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u16 num_items;
} avp_get_media_player_item_cfm_t;

typedef struct avp_get_media_player_ind_s {
	BD_ADDRESS bd_addr;
	u8 status;
	GByteArray *data;
} avp_get_media_player_ind_t;

typedef struct avp_play_item_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
} avp_play_item_cfm_t;

typedef struct avp_change_path_cfm_s {
	BD_ADDRESS bd_addr;
	u8 status;
	u32 num_item;
} avp_change_path_cfm_t;

typedef struct avp_codec_and_bitrate_ind_s {
	BD_ADDRESS bd_addr;
	u8 codec_type;
	u32 bitrate;
} avp_codec_and_bitrate_ind_t;

/*******/
/* NAP */
/*******/
typedef struct nap_create_device_cfm_s {
	BD_ADDRESS bd_addr;
	u8 service;
	u8 status;
	gchar *device_name;
} nap_create_device_cfm_t;

typedef struct nap_destroy_device_cfm_s {
	BD_ADDRESS bd_addr;
	u8 service;
	u8 status;
} nap_destroy_device_cfm_t;

typedef struct nap_destroy_device_ind_s {
	BD_ADDRESS bd_addr;
	u8 service;
} nap_destroy_device_ind_t;

/*************************************/
/* Union of all parameter structures */
/*************************************/
typedef union callback_parms_u {
	/* GENERAL */
	general_callback_parms general;
	gen_get_device_name_cfm_t gen_get_device_name_cfm_parms;
	gen_role_change_ind_t gen_role_change_ind_parms;
	gen_service_type_ind_t gen_service_type_ind_parms;
	gen_connection_complete_ind_t gen_connection_complete_ind_parms;
	gen_disconnection_ind_t gen_disconnection_ind_parms;
	gen_testmode_cfm_t gen_testmode_cfm_parms;
	gen_testmode_complete_ind_t gen_testmode_complete_ind_parms;
	gen_linkqualify_cfm_t gen_linkqualify_cfm_parms;
	gen_spp_capabilities_ind_t gen_spp_capabilities_ind_parms;
	gen_ssp_numeric_confirm_ind_t gen_ssp_numeric_confirm_ind_parms;
	gen_connection_request_ind_t gen_connection_request_ind_parms;
	gen_pnp_info_attribute_ind_t gen_pnp_info_attribute_ind_parms;
	gen_inquiry_complete_ind_t gen_inquiry_complete_ind_parms;
	gen_inquiry_ind_t gen_inquiry_ind_parms;
	/* HF */
	hf_call_status_ind_t hf_call_status_ind_parms;
	hf_phone_status_ind_t hf_phone_status_ind_parms;
	hf_dial_cfm_t hf_dial_cfm_parms;
	hf_redial_cfm_t hf_redial_cfm_parms;
	hf_mem_dial_cfm_t hf_mem_dial_cfm_parms;
	hf_terminate_call_cfm_t hf_terminate_call_cfm_parms;
	hf_terminate_all_cfm_t hf_terminate_all_cfm_parms;
	hf_audio_transfer_cfm_t hf_audio_transfer_cfm_parms;
	hf_setup_audio_ind_t hf_setup_audio_ind_parms;
	hf_setup_audio_wbs_ind_t hf_setup_audio_wbs_ind_parms;
	hf_release_audio_ind_t hf_release_audio_ind_parms;
	hf_send_dtmf_cfm_t hf_send_dtmf_cfm_parms;
	hf_incoming_call_act_cfm_t hf_incoming_call_act_cfm_parms;
	hf_ring_ind_t hf_ring_ind_parms;
	hf_release_inactive_call_cfm_t hf_release_inactive_call_cfm_parms;
	hf_release_active_call_cfm_t hf_release_active_call_cfm_parms;
	hf_swap_call_cfm_t hf_swap_call_cfm_parms;
	hf_add_call_cfm_t hf_add_call_cfm_parms;
	hf_release_spec_call_cfm_t hf_release_spec_call_cfm_parms;
	hf_hold_excep_call_cfm_t hf_hold_excep_call_cfm_parms;
	hf_exp_call_trans_cfm_t hf_exp_call_trans_cfm_parms;
	hf_subscriber_cfm_t hf_subscriber_cfm_parms;
	hf_manufacturer_id_cfm_t hf_manufacturer_id_cfm_parms;
	hf_model_id_cfm_t hf_model_id_cfm_parms;
	hf_revision_id_cfm_t hf_revision_id_cfm_parms;
	hf_voice_recognition_activation_cfm_t hf_voice_recognition_activation_cfm_parms;
	hf_voice_recognition_activation_ind_t hf_voice_recognition_activation_ind_parms;
	appl_xapl_siri_cfm_t appl_xapl_siri_cfm_params;
	appl_xapl_siri_ind_t appl_xapl_siri_ind_params;
	siri_nrstat_cfm_t siri_nrstat_cfm_params;
	siri_nrstat_ind_t siri_nrstat_ind_params;
	siri_efm_cfm_t siri_efm_cfm_params;

	/* PBDL */
	pbdl_get_capabilities_cfm_t pbdl_get_capabilities_cfm_parms;
	pbdl_prepare_cfm_t pbdl_prepare_cfm_parms;
	pbdl_start_cfm_t pbdl_start_cfm_parms;
	pbdl_stop_cfm_t pbdl_stop_cfm_parms;
	pbdl_data_ind_t pbdl_data_ind_parms;
	pbdl_complete_ind_t pbdl_complete_ind_parms;
	pbdl_calender_complete_ind_t pbdl_calender_complete_ind_parms;
	pbdl_complete_cfm_t pbdl_complete_cfm_parms;
	pbdl_con_lost_ind_t pbdl_con_lost_ind_parms;
	pbdl_status_chg_ind_t pbdl_status_chg_ind_parms;

	/* MAP */
	map_get_capabilities_cfm_t map_get_capabilities_cfm_parms;
	map_start_cfm_t map_start_cfm_parms;
	map_active_instance_ind_t map_active_instance_ind_parms;
	map_notify_status_ind_t map_notify_status_ind_parms;
	map_notify_regist_status_ind_t map_notify_regist_status_ind_parms;
	map_stop_cfm_t map_stop_cfm_parms;
	map_chg_instance_cfm_t map_chg_instance_cfm_parms;
	map_update_inbox_cfm_t map_update_inbox_cfm_parms;
	map_list_folder_cfm_t map_list_folder_cfm_parms;
	map_list_folder_ind_t map_list_folder_ind_parms;
	map_list_folder_comp_ind_t map_list_folder_comp_ind_parms;
	map_list_msg_cfm_t map_list_msg_cfm_parms;
	map_list_msg_data_ind_t map_list_msg_data_ind_parms;
	map_list_msg_comp_ind_t map_list_msg_comp_ind_parms;
	map_read_msg_cfm_t map_read_msg_cfm_parms;
	map_read_msg_ind_t map_read_msg_ind_parms;
	map_read_msg_comp_ind_t map_read_msg_comp_ind_parms;
	map_set_msg_status_cfm_t map_set_msg_status_cfm_parms;
	map_upload_cfm_t map_upload_cfm_parms;
	map_upload_ind_t map_upload_ind_parms;
	map_abort_cfm_t map_abort_cfm_parms;
	map_notify_ind_t map_notify_ind_parms;
	map_error_report_ind_t map_error_report_ind_parms;

	/* AVP */
	avp_version_ind_t avp_version_ind_parms;
	avp_ctrl_cmd_cfm_t avp_ctrl_cmd_cfm_parms;
	avp_ctrl_cmd_ind_t avp_ctrl_cmd_ind_parms;
	avp_get_capabilities_cfm_t avp_get_capabilities_cfm_parms;
	avp_list_attr_cfm_t avp_list_attr_cfm_parms;
	avp_list_val_cfm_t avp_list_val_cfm_parms;
	avp_set_val_cfm_t avp_set_val_cfm_parms;
	avp_play_status_ind_t avp_play_status_ind_parms;
	avp_notify_set_complete_ind_t avp_notify_set_complete_ind_parms;
	avp_notify_chg_ind_t avp_notify_chg_ind_parms;
	avp_set_player_cfm_t avp_set_player_cfm_parms;
	avp_get_metadata_attr_cfm_t avp_get_metadata_attr_cfm_parms;
	avp_get_folder_item_cfm_t avp_get_folder_item_cfm_parms;
	avp_get_folder_item_ind_t avp_get_folder_item_ind_parms;
	avp_media_element_ind_t avp_media_element_ind_parms;
	avp_get_media_player_item_cfm_t avp_get_media_player_item_cfm_parms;
	avp_get_media_player_ind_t avp_get_media_player_ind_parms;
	avp_play_item_cfm_t avp_play_item_cfm_parms;
	avp_change_path_cfm_t avp_change_path_cfm_parms;
	avp_codec_and_bitrate_ind_t avp_codec_and_bitrate_ind_parms;

	/* NAP */
	nap_create_device_cfm_t nap_create_device_cfm_parms;
	nap_destroy_device_cfm_t nap_destroy_device_cfm_parms;
	nap_destroy_device_ind_t nap_destroy_device_ind_parms;
} callback_parms;

typedef struct callback_data_s {
	enum alps_callbacks id;
	enum tagBtApplCallbackIdEnum type;
	callback_parms parms;
} callback_data;

typedef void (*do_callback_function) (callback_data *data);

int initialize_alps_callbacks(void);
void shutdown_alps_callbacks(void);
BTHRESULT register_alps_callbacks(void);
void unregister_alps_callbacks(void);

#endif /* ifndef _CALLBACKS_H */
