/*
 * (C) Copyright 2013
 * Carsten Resch Bosch CarMultimedia GmbH, Carsten.Resch@de.bosch.com
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <asm/arch/clock.h>
#include "cmd_mem_ext_test.h"

#define print(a, args...) printf(a, ##args)

U32 PSR_BUF[0x1800];		/* buffer for generated PSR in
				   internal Ram: 24KByte
				 */

void print_ErrorBuffer(Error_Compare_Data * ptr_ErrorBuffer,
		       U32 number_of_errors, int block_nr)
{

	U32 max_prints = number_of_errors;
	U32 Index = 0;

	if (number_of_errors > MAX_SAVED_ERRORS) {
		max_prints = MAX_SAVED_ERRORS;
		printf("Num of error in block (%d) but only %d stored \n",
		       number_of_errors, MAX_SAVED_ERRORS);
	}

	if (number_of_errors > 0) {
		print("RAM errors     : ");
		for (Index = 0; Index < max_prints; Index++) {
			/* Nor Adress or just any expected pattern */
			print("(%04d - 0x%8X     ", block_nr,
			      (U32) ptr_ErrorBuffer[Index].addr_comp);
			print("0x%8X     ",
			      (U32) ptr_ErrorBuffer[Index].addr_ddr3);
			print("0x%8X     ",
			      (U32) ptr_ErrorBuffer[Index].data_comp);
			print("0x%8X     ",
			      (U32) ptr_ErrorBuffer[Index].data_ddr3);
			print("0x%8X     ",
			      ((U32) ptr_ErrorBuffer[Index].data_ddr3) ^
			      ((U32) ptr_ErrorBuffer[Index].data_comp));
			print("\n");
		}
	}
}

/*!
\ AdressPatternTest
\ Writes to each Adress own Adresspattern
\ -> *(U32*)0x80000000 = 0x80000000
*/
void AdressPatternTest(U32 * start_addr, U32 * end_addr,
		       ADDRESS_PATTERN_TEST en_pattern)
{

	U32 banksize = ((U32) end_addr - (U32) start_addr);
	U32 blocksize_int = banksize / 4;
	U32 *ptr_addr_ddr_base = (U32 *) start_addr;
	U32 adressvalue = (U32) start_addr;
	U32 i = 0x0;
	U32 j = 0x0;
	U32 num_of_errors = 0x0;
	U32 error_index = 0x0;
	Error_Compare_Data ErrorBuffer[MAX_SAVED_ERRORS];
	U32 *temp_ptr_dst = 0x0;
	U32 temp = 0x0;

	switch (en_pattern) {
	case (PATTERN_INVERSE_ADDR):
		print("Start AdressPatternTest: Pattern = INVERSE_ADDR ....\n");
		break;
	case (PATTERN_ADDR_XOR_CHESS):
		print
		    ("Start AdressPatternTest: Pattern = PATTERN_ADDR_XOR_CHESS ....\n");
		break;
	case (PATTERN_CHESS_FIELD):
		print
		    ("Start AdressPatternTest: Pattern = PATTERN_CHESS_FIELD ....\n");
		break;
	case (PATTERN_ADDR):
		print("Start AdressPatternTest: Pattern = PATTERN_ADDR ....\n");
		break;
	}

	for (i = 0; i < blocksize_int; i += 8) {

		switch (en_pattern) {
		case (PATTERN_INVERSE_ADDR):
			for (j = 0; j < 8; j++) {
				ptr_addr_ddr_base[i + j] = ~adressvalue;
				adressvalue += sizeof(U32);
			}
			break;
		case (PATTERN_ADDR_XOR_CHESS):
			for (j = 0; j < 8; j++) {
				ptr_addr_ddr_base[i + j] =
				    adressvalue ^ 0xA5A5A5A5;
				adressvalue += sizeof(U32);
			}
			break;
		case (PATTERN_CHESS_FIELD):
			for (j = 0; j < 8; j++) {
				if ((adressvalue % 8) == 0) {
					ptr_addr_ddr_base[i + j] = 0xA5A5A5A5;
				} else {
					ptr_addr_ddr_base[i + j] = 0x5A5A5A5A;
				}
				adressvalue += sizeof(U32);
			}
			break;
		case (PATTERN_ADDR):
			for (j = 0; j < 8; j++) {
				ptr_addr_ddr_base[i + j] = adressvalue;
				adressvalue += sizeof(U32);
			}
			break;
		}
	}

	/* verify */
	adressvalue = (U32) start_addr;

	temp_ptr_dst = (U32 *) adressvalue;

	for (i = 0; i < blocksize_int; i++) {
		temp = *temp_ptr_dst;

		switch (en_pattern) {
		case (PATTERN_INVERSE_ADDR):
			temp = ~temp;
			break;
		case (PATTERN_ADDR_XOR_CHESS):
			temp = temp ^ 0xA5A5A5A5;
			break;
		case (PATTERN_CHESS_FIELD):
			if ((adressvalue % 8) == 0) {
				if (temp == 0xA5A5A5A5) {
					temp = adressvalue;
				}
			} else {
				if (temp == 0x5A5A5A5A) {
					temp = adressvalue;
				}
			}
			break;
		case (PATTERN_ADDR):
			break;
		}

		if (temp != adressvalue) {
			if (num_of_errors < (MAX_SAVED_ERRORS)) {
				ErrorBuffer[error_index].addr_ddr3 =
				    &ptr_addr_ddr_base[i];

				ErrorBuffer[error_index].data_ddr3 = temp;
				error_index++;
			}
			num_of_errors++;
		}

		adressvalue += sizeof(U32);
		temp_ptr_dst++;
	}

	print_ErrorBuffer(ErrorBuffer, num_of_errors, 0);

	/* startaddress for bw verify */
	adressvalue = ((U32) start_addr) + banksize - sizeof(U32);
	error_index = 0x0;
	num_of_errors = 0x0;

	temp_ptr_dst = (U32 *) adressvalue;

	while (temp_ptr_dst >= (U32 *) start_addr) {
		temp = *temp_ptr_dst;
		switch (en_pattern) {
		case (PATTERN_INVERSE_ADDR):
			temp = ~temp;
			break;
		case (PATTERN_ADDR_XOR_CHESS):
			temp = temp ^ 0xA5A5A5A5;
			break;
		case (PATTERN_CHESS_FIELD):
			if ((adressvalue % 8) == 0) {
				if (temp == 0xA5A5A5A5) {
					temp = adressvalue;
				}
			} else {
				if (temp == 0x5A5A5A5A) {
					temp = adressvalue;
				}
			}
			break;
		case (PATTERN_ADDR):
			break;
		}

		if (temp != adressvalue) {
			if (num_of_errors < (MAX_SAVED_ERRORS)) {
				ErrorBuffer[error_index].addr_ddr3 =
				    temp_ptr_dst;
				ErrorBuffer[error_index].data_ddr3 = temp;
				error_index++;
			}
			num_of_errors++;
		}
		temp_ptr_dst--;
		adressvalue -= sizeof(U32);
	}
	print_ErrorBuffer(ErrorBuffer, num_of_errors, 0);
}

extern int do_reset(cmd_tbl_t * cmdtp, int flag, int argc, char *const argv[]);

#define rot(x,k) (((x)<<(k))|((x)>>(32-(k))))

U32 ranval(ranctx * x)
{
	U32 e = x->a - rot(x->b, 27);
	x->a = x->b ^ rot(x->c, 17);
	x->b = x->c + x->d;
	x->c = x->d + e;
	x->d = e + x->a;
	return x->d;
}

void raninit(ranctx * x, U32 seed)
{
	U32 i = 0;

	x->a = 0xf1ea5eed, x->b = seed;
	x->c = seed;
	x->d = seed;

	for (i = 0; i < 20; ++i) {
		(void)ranval(x);
	}
}

/*!
\ 1. CPU_Copy_Word
\
*/
void CPU_Copy_Word(volatile U32 * ptr_src,
		   volatile U32 * ptr_dst,
		   U32 blocksize_byte,
		   DIR_FLAG dir_nor, DIR_FLAG dir_ram, OPERATION_FLAG operation)
{
	U32 i = 0;
	U32 blocksize_U32 = blocksize_byte / 4;
	U32 j = blocksize_U32 - 1;

	if ((dir_nor == FORWARD) && (dir_ram == FORWARD)) {
		for (i = 0; i < blocksize_U32; i++) {
			if (operation == NO_OPERATION) {
				ptr_dst[i] = ptr_src[i];
			} else if (operation == INVERSION) {
				ptr_dst[i] = ~ptr_src[i];
			}
		}
	} else if ((dir_nor == BACKWARD) && (dir_ram == FORWARD)) {
		for (i = 0; i < blocksize_U32; i++) {
			ptr_dst[i] = ptr_src[j - i];
		}
	} else if ((dir_nor == FORWARD) && (dir_ram == BACKWARD)) {
		for (i = 0; i < blocksize_U32; i++) {
			if (operation == NO_OPERATION) {
				ptr_dst[j - i] = ptr_src[i];
			} else if (operation == INVERSION) {
				ptr_dst[j - i] = ~ptr_src[i];
			}
		}
	} else if ((dir_nor == BACKWARD) && (dir_ram == BACKWARD)) {
		for (i = (blocksize_U32); i > 0; i--) {
			ptr_dst[i - 1] = ptr_src[i - 1];
		}
	}
}

/*!
\ 1. fVerify_Word
\ Verify a complete area after copieng
\ loop unrolled to speed up execution time
*/
U32 CPU_Verify_Word(volatile U32 * ptr_src, volatile U32 * ptr_dst, U32 blocksize_byte, Error_Compare_Data * buf,	/* max 30 entries */
		    DIR_FLAG dir_nor,
		    DIR_FLAG dir_ram,
		    OPERATION_FLAG operation, BUFFER_FLAG buffering)
{
	volatile U32 *temp_ptr_dst = (U32 *) 0;
	volatile U32 *temp_ptr_src = (U32 *) 0;
	volatile U8 *temp_ptr_end = (U8 *) 0;

	U32 u32errorexor = 0;

	U32 num_of_errors = 0;
	U32 blocksize_word = blocksize_byte / 4;

	volatile U32 u32_data_src = 0;
	volatile U32 u32_data_dest = 0;

	Error_Compare_Data *temp_Buf = buf;

	if ((dir_nor == FORWARD) &&
	    (dir_ram == FORWARD) && (operation == NO_OPERATION)) {
		temp_ptr_dst = ptr_dst;
		temp_ptr_src = ptr_src;
		temp_ptr_end = ((U8 *) ptr_dst + blocksize_byte);

		while (temp_ptr_dst < (U32 *) temp_ptr_end) {

			u32_data_src = *temp_ptr_src;
			u32_data_dest = *temp_ptr_dst;

			if (u32_data_dest != u32_data_src) {
				if ((num_of_errors < MAX_SAVED_ERRORS) &&
				    (buffering == ERROR_BUF)) {
					temp_Buf->addr_comp = temp_ptr_src;
					temp_Buf->addr_ddr3 = temp_ptr_dst;
					temp_Buf->data_comp = u32_data_src;
					temp_Buf->data_ddr3 = u32_data_dest;
					u32errorexor =
					    (u32_data_src) ^ (u32_data_dest);
				}
				num_of_errors++;
			}
			temp_ptr_dst++;
			temp_ptr_src++;
		}
	} else if ((dir_nor == FORWARD) &&
		   (dir_ram == FORWARD) && (operation == INVERSION)) {
		temp_ptr_dst = ptr_dst;
		temp_ptr_src = ptr_src;
		temp_ptr_end = ((U8 *) ptr_dst + blocksize_byte);

		while (temp_ptr_dst < (U32 *) temp_ptr_end) {
			u32_data_src = *temp_ptr_src;
			u32_data_dest = *temp_ptr_dst;
			if (u32_data_dest != ~(u32_data_src)) {
				if ((num_of_errors < MAX_SAVED_ERRORS) &&
				    (buffering == ERROR_BUF)) {
					temp_Buf->addr_comp = temp_ptr_src;
					temp_Buf->addr_ddr3 = temp_ptr_dst;
					temp_Buf->data_comp = ~u32_data_src;
					temp_Buf->data_ddr3 = u32_data_dest;
					temp_Buf++;
				}
				num_of_errors++;
			}
			temp_ptr_dst++;
			temp_ptr_src++;
		}
	} else if ((dir_nor == FORWARD) &&
		   (dir_ram == BACKWARD) && (operation == NO_OPERATION)) {
		/* 0x80BFFFFF end of block #3 */
		temp_ptr_dst = ptr_dst + (blocksize_word - 1);
		temp_ptr_src = ptr_src;	/* nor start */
		/* 0x80800000 begin of block #3 */
		temp_ptr_end = (U8 *) ptr_dst;

		while (temp_ptr_dst >= (U32 *) temp_ptr_end) {
			u32_data_src = *temp_ptr_src;
			u32_data_dest = *temp_ptr_dst;
			if (u32_data_dest != u32_data_src) {
				if ((num_of_errors < MAX_SAVED_ERRORS) &&
				    (buffering == ERROR_BUF)) {
					temp_Buf->addr_comp = temp_ptr_src;
					temp_Buf->addr_ddr3 = temp_ptr_dst;
					temp_Buf->data_comp = u32_data_dest;
					temp_Buf->data_ddr3 = u32_data_dest;
					temp_Buf++;
				}
				num_of_errors++;
			}
			temp_ptr_dst--;	/* ram pointer */
			temp_ptr_src++;	/* nor pointer */
		}
	} else if ((dir_nor == FORWARD) &&
		   (dir_ram == BACKWARD) && (operation == INVERSION)) {
		temp_ptr_dst = ptr_dst + (blocksize_word - 1);
		temp_ptr_src = ptr_src;
		temp_ptr_end = (U8 *) ptr_dst;

		while (temp_ptr_dst >= (U32 *) temp_ptr_end) {
			u32_data_src = *temp_ptr_src;
			u32_data_dest = *temp_ptr_dst;
			if (u32_data_dest != ~(u32_data_src)) {
				if ((num_of_errors < MAX_SAVED_ERRORS) &&
				    (buffering == ERROR_BUF)) {
					temp_Buf->addr_comp = temp_ptr_src;
					temp_Buf->addr_ddr3 = temp_ptr_dst;
					temp_Buf->data_comp = ~u32_data_src;
					temp_Buf->data_ddr3 = u32_data_dest;
					temp_Buf++;
				}
				num_of_errors++;
			}
			temp_ptr_dst--;	/* ram pointer */
			temp_ptr_src++;	/* nor pointer */
		}
	}
	return num_of_errors;
}

void Create_PSR_Pattern(volatile U32 * intram, U32 size, ranctx * Ran)
{

	U32 i = 0x0;

	for (i = 0; i < size / sizeof(U32); i += 8) {
		intram[i + 0] = ranval(Ran);
		intram[i + 1] = ranval(Ran);
		intram[i + 2] = ranval(Ran);
		intram[i + 3] = ranval(Ran);
		intram[i + 4] = ranval(Ran);
		intram[i + 5] = ranval(Ran);
		intram[i + 6] = ranval(Ran);
		intram[i + 7] = ranval(Ran);
	}
}

static int n_first = 1;

U32 get_seed(void)
{
	unsigned long long curr_us;

	U32 u32_us[2];
	if (n_first == 1) {
		curr_us = get_us();
		u32_us[0] = ((U32) (curr_us >> 32));
		u32_us[1] = ((U32) (curr_us & 0xFFFFFFFF));

		return (u32_us[0] ^ u32_us[1]);
	} else {
		U32 *ptrtest = (U32 *) 0x10000300;
		return *ptrtest;
	}
}

void CompleteMemoryTest_SmallPattern(U32 * start_addr, U32 * end_addr)
{
	volatile U32 *ptr_addr_ddr_base = (U32 *) start_addr;
	volatile U32 *ptr_addr_pattern = PSR_BUF;
	volatile U32 *temp_ptr1_dst = (U32 *) 0x0;
	volatile U32 *temp_ptr2_dst = (U32 *) 0x0;
	U32 BankSize = ((U32) end_addr - (U32) start_addr);
	U32 No64KBLoops = (BankSize / BUFFER_SIZE_16KB) / 4;
	U32 blocksize = (U32) BUFFER_SIZE_16KB;
	U32 BlockIndex = 0x0;
	U32 num_of_errors = 0x0;
	DIR_FLAG direction_nor;
	DIR_FLAG direction_ram;
	OPERATION_FLAG operation;
	Error_Compare_Data ErrorBuffer[MAX_SAVED_ERRORS];	/* max 30 entries */
	U32 error_sum = 0x0;
	U32 seed = 0x0;
	ranctx RandomGenerator;
	int i = 0;

	temp_ptr1_dst = ptr_addr_ddr_base;	/* copy pointer   */
	temp_ptr2_dst = ptr_addr_ddr_base;	/* verify pointer */

	seed = get_seed();
	raninit(&RandomGenerator, seed);
	Create_PSR_Pattern(ptr_addr_pattern, blocksize, &RandomGenerator);

	print("New test round: seed = 0x%08X \n", seed);
	print("PSR Data[0..8]: ");
	for (i = 0; i < 8; i++) {
		print("0x%08X ", ptr_addr_pattern[i]);
	}
	print("\n");

	for (BlockIndex = 0; BlockIndex < No64KBLoops; BlockIndex++) {
		/* forward, forward, no operation */
		direction_nor = FORWARD;
		direction_ram = FORWARD;
		operation = NO_OPERATION;
		CPU_Copy_Word(ptr_addr_pattern, temp_ptr1_dst, blocksize,
			      direction_nor, direction_ram, operation);
		temp_ptr1_dst += BUFFER_SIZE_16KB / 4;

		direction_nor = FORWARD;
		direction_ram = FORWARD;
		operation = INVERSION;
		CPU_Copy_Word(ptr_addr_pattern, temp_ptr1_dst, blocksize,
			      direction_nor, direction_ram, operation);
		temp_ptr1_dst += BUFFER_SIZE_16KB / 4;

		direction_nor = FORWARD;	/* NOR forward  */
		direction_ram = BACKWARD;	/* RAM backward */
		operation = NO_OPERATION;
		CPU_Copy_Word(ptr_addr_pattern, temp_ptr1_dst, blocksize,
			      direction_nor, direction_ram, operation);
		temp_ptr1_dst += BUFFER_SIZE_16KB / 4;

		direction_nor = FORWARD;
		direction_ram = FORWARD;
		operation = NO_OPERATION;
		CPU_Copy_Word(ptr_addr_pattern, temp_ptr1_dst, blocksize,
			      direction_nor, direction_ram, operation);
		temp_ptr1_dst += BUFFER_SIZE_16KB / 4;

		if ((BlockIndex % 500) == 0) {
			print
			    ("Verify	64KB-Block No %04d (addr 0x%08X) ...... (error count=%d) \n",
			     (BlockIndex + 1), (unsigned int)temp_ptr2_dst,
			     error_sum);
		}

		direction_nor = FORWARD;
		direction_ram = FORWARD;
		operation = NO_OPERATION;
		num_of_errors =
		    CPU_Verify_Word(ptr_addr_pattern, temp_ptr2_dst, blocksize,
				    ErrorBuffer, direction_nor, direction_ram,
				    operation, ERROR_BUF);
		error_sum += num_of_errors;
		print_ErrorBuffer(ErrorBuffer, num_of_errors, BlockIndex);
		temp_ptr2_dst += BUFFER_SIZE_16KB / 4;

		direction_nor = FORWARD;
		direction_ram = FORWARD;
		operation = INVERSION;
		num_of_errors =
		    CPU_Verify_Word(ptr_addr_pattern, temp_ptr2_dst, blocksize,
				    ErrorBuffer, direction_nor, direction_ram,
				    operation, ERROR_BUF);
		error_sum += num_of_errors;
		print_ErrorBuffer(ErrorBuffer, num_of_errors, BlockIndex);
		temp_ptr2_dst += BUFFER_SIZE_16KB / 4;

		direction_nor = FORWARD;	/* NOR forward  */
		direction_ram = BACKWARD;	/* RAM backward */
		operation = NO_OPERATION;
		num_of_errors =
		    CPU_Verify_Word(ptr_addr_pattern, temp_ptr2_dst, blocksize,
				    ErrorBuffer, direction_nor, direction_ram,
				    operation, ERROR_BUF);
		error_sum += num_of_errors;
		print_ErrorBuffer(ErrorBuffer, num_of_errors, BlockIndex);
		temp_ptr2_dst += BUFFER_SIZE_16KB / 4;

		direction_nor = FORWARD;
		direction_ram = FORWARD;
		operation = NO_OPERATION;
		num_of_errors =
		    CPU_Verify_Word(ptr_addr_pattern, temp_ptr2_dst, blocksize,
				    ErrorBuffer, direction_nor, direction_ram,
				    operation, ERROR_BUF);
		error_sum += num_of_errors;
		print_ErrorBuffer(ErrorBuffer, num_of_errors, BlockIndex);

		temp_ptr2_dst += BUFFER_SIZE_16KB / 4;

		error_sum = 0;
		num_of_errors = 0;
	}
}

static int do_mem_ext_mtest(cmd_tbl_t * cmdtp, int flag, int argc,
			    char *const argv[])
{
	int i;
	U32 *start_addr = (U32 *) DEFAULT_EXT_MEMTEST_START;
	U32 *end_addr = (U32 *) DEFAULT_EXT_MEMTEST_END;
	int iteration_limit = 1;

	if (argc > 1)
		start_addr = (U32 *) simple_strtoul(argv[1], NULL, 16);

	if (argc > 2)
		end_addr = (U32 *) simple_strtoul(argv[2], NULL, 16);

	if (argc > 3)
		iteration_limit = (ulong) simple_strtoul(argv[3], NULL, 16);

	for (i = 0; i < iteration_limit; i++) {
		CompleteMemoryTest_SmallPattern(start_addr, end_addr);
		AdressPatternTest(start_addr, end_addr, PATTERN_ADDR);
		AdressPatternTest(start_addr, end_addr, PATTERN_INVERSE_ADDR);
		AdressPatternTest(start_addr, end_addr, PATTERN_ADDR_XOR_CHESS);
		AdressPatternTest(start_addr, end_addr, PATTERN_CHESS_FIELD);
	}

	return 0;
}

U_BOOT_CMD(ext_mtest, 4, 1, do_mem_ext_mtest,
	   "extended RAM read/write test", "[start [end [iterations]]]\n"
	   "testable memory range depends on the overall memory size of the board:\n"
	   "All boards:        0x10000000 - 0x17800000, and:\n"
	   "512MB main memory: 0x20000000 - 0x2FC00000\n"
	   "1GB   main memory: 0x20000000 - 0x4FC00000\n"
	   "(last 4MB are used for stack and heap)");
