/*
 * Copyright 2012 Freescale Semiconductor, Inc.
 * Copyright 2012 Linaro Ltd.
 * Copyright 2009 Pengutronix, Sascha Hauer <s.hauer@pengutronix.de>
 *
 * Initial development of this code was funded by
 * Phytec Messtechnik GmbH, http://www.phytec.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/clk.h>
#include <linux/debugfs.h>
#include <linux/err.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/pinctrl/consumer.h>

#include "imx-audmux.h"

#define DRIVER_NAME "imx-audmux"

static struct clk *audmux_clk;
static void __iomem *audmux_base;

#define IMX_AUDMUX_V2_PTCR(x)		((x) * 8)
#define IMX_AUDMUX_V2_PDCR(x)		((x) * 8 + 4)

#define MAX_PORTS 8
#define IOMUXC_NAME "iomuxc"
struct audmux_port {
	struct platform_device *pdev;
	int external;
	int valid;
	int exported;
};

static struct audmux_port ports[MAX_PORTS] = { { 0 } };

#ifdef CONFIG_DEBUG_FS
static struct dentry *audmux_debugfs_root;

/* There is an annoying discontinuity in the SSI numbering with regard
 * to the Linux number of the devices */
static const char *audmux_port_string(int port)
{
	switch (port) {
	case MX31_AUDMUX_PORT1_SSI0:
		return "imx-ssi.0";
	case MX31_AUDMUX_PORT2_SSI1:
		return "imx-ssi.1";
	case MX31_AUDMUX_PORT3_SSI_PINS_3:
		return "AUD3";
	case MX31_AUDMUX_PORT4_SSI_PINS_4:
		return "AUD4";
	case MX31_AUDMUX_PORT5_SSI_PINS_5:
		return "AUD5";
	case MX31_AUDMUX_PORT6_SSI_PINS_6:
		return "AUD6";
	case MX6_AUDMUX_PORT7:
		return "imx-ssi.2";
	default:
		return "UNKNOWN";
	}
}

static ssize_t audmux_read_file(struct file *file, char __user *user_buf,
				size_t count, loff_t *ppos)
{
	ssize_t ret;
	char *buf = kmalloc(PAGE_SIZE, GFP_KERNEL);
	int port = (int)file->private_data;
	u32 pdcr, ptcr;

	if (!buf)
		return -ENOMEM;

	if (audmux_clk)
		clk_prepare_enable(audmux_clk);

	ptcr = readl(audmux_base + IMX_AUDMUX_V2_PTCR(port));
	pdcr = readl(audmux_base + IMX_AUDMUX_V2_PDCR(port));

	if (audmux_clk)
		clk_disable_unprepare(audmux_clk);

	ret = snprintf(buf, PAGE_SIZE, "PDCR: %08x\nPTCR: %08x\n",
		       pdcr, ptcr);

	if (ptcr & IMX_AUDMUX_V2_PTCR_TFSDIR)
		ret += snprintf(buf + ret, PAGE_SIZE - ret,
				"TxFS output from %s, ",
				audmux_port_string((ptcr >> 27) & 0x7));
	else
		ret += snprintf(buf + ret, PAGE_SIZE - ret,
				"TxFS input, ");

	if (ptcr & IMX_AUDMUX_V2_PTCR_TCLKDIR)
		ret += snprintf(buf + ret, PAGE_SIZE - ret,
				"TxClk output from %s",
				audmux_port_string((ptcr >> 22) & 0x7));
	else
		ret += snprintf(buf + ret, PAGE_SIZE - ret,
				"TxClk input");

	ret += snprintf(buf + ret, PAGE_SIZE - ret, "\n");

	if (ptcr & IMX_AUDMUX_V2_PTCR_SYN) {
		ret += snprintf(buf + ret, PAGE_SIZE - ret,
				"Port is symmetric");
	} else {
		if (ptcr & IMX_AUDMUX_V2_PTCR_RFSDIR)
			ret += snprintf(buf + ret, PAGE_SIZE - ret,
					"RxFS output from %s, ",
					audmux_port_string((ptcr >> 17) & 0x7));
		else
			ret += snprintf(buf + ret, PAGE_SIZE - ret,
					"RxFS input, ");

		if (ptcr & IMX_AUDMUX_V2_PTCR_RCLKDIR)
			ret += snprintf(buf + ret, PAGE_SIZE - ret,
					"RxClk output from %s",
					audmux_port_string((ptcr >> 12) & 0x7));
		else
			ret += snprintf(buf + ret, PAGE_SIZE - ret,
					"RxClk input");
	}

	ret += snprintf(buf + ret, PAGE_SIZE - ret,
			"\nData received from %s\n",
			audmux_port_string((pdcr >> 13) & 0x7));

	ret = simple_read_from_buffer(user_buf, count, ppos, buf, ret);

	kfree(buf);

	return ret;
}

static const struct file_operations audmux_debugfs_fops = {
	.open = simple_open,
	.read = audmux_read_file,
	.llseek = default_llseek,
};

static void audmux_debugfs_init(int ports)
{
	int i;
	char buf[20];

	audmux_debugfs_root = debugfs_create_dir("audmux", NULL);
	if (!audmux_debugfs_root) {
		pr_warn("Failed to create AUDMUX debugfs root\n");
		return;
	}

	for (i = 0; i < ports + 1; i++) {
		snprintf(buf, sizeof(buf), "%s", audmux_port_string(i));
		if (!debugfs_create_file(buf, 0444, audmux_debugfs_root,
					 (void *)i, &audmux_debugfs_fops))
			pr_warn("Failed to create AUDMUX port %d debugfs file\n",
				   i);
	}
}

static void audmux_debugfs_remove(void)
{
	debugfs_remove_recursive(audmux_debugfs_root);
}
#else
static inline void audmux_debugfs_init(int)
{
}

static inline void audmux_debugfs_remove(void)
{
}
#endif

enum imx_audmux_type {
	IMX21_AUDMUX,
	IMX31_AUDMUX,
	IMX6_AUDMUX,
} audmux_type;

static struct platform_device_id imx_audmux_ids[] = {
	{
		.name = "imx21-audmux",
		.driver_data = IMX21_AUDMUX,
	}, {
		.name = "imx31-audmux",
		.driver_data = IMX31_AUDMUX,
	}, {
		.name = "imx6-audmux",
		.driver_data = IMX6_AUDMUX,
	}, {
		/* sentinel */
	}
};
MODULE_DEVICE_TABLE(platform, imx_audmux_ids);

static const struct of_device_id imx_audmux_dt_ids[] = {
	{ .compatible = "fsl,imx21-audmux", .data = &imx_audmux_ids[0], },
	{ .compatible = "fsl,imx31-audmux", .data = &imx_audmux_ids[1], },
	{ .compatible = "fsl,imx6-audmux", .data = &imx_audmux_ids[2], },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, imx_audmux_dt_ids);

static const uint8_t port_mapping[] = {
	0x0, 0x4, 0x8, 0x10, 0x14, 0x1c,
};

int imx_audmux_v1_configure_port(unsigned int port, unsigned int pcr)
{
	if (audmux_type != IMX21_AUDMUX)
		return -EINVAL;

	if (!audmux_base)
		return -ENOSYS;

	if (port >= ARRAY_SIZE(port_mapping))
		return -EINVAL;

	writel(pcr, audmux_base + port_mapping[port]);

	return 0;
}
EXPORT_SYMBOL_GPL(imx_audmux_v1_configure_port);

int imx_audmux_v2_configure_port(unsigned int port, unsigned int ptcr,
		unsigned int pdcr)
{
	struct pinctrl *pinctrl;
	int id = port + 1;
	int ret = 0;

	if (audmux_type < IMX31_AUDMUX)
		return -EINVAL;

	if (!audmux_base)
		return -ENOSYS;

	if (audmux_clk)
		clk_prepare_enable(audmux_clk);

	/* write final port settings */
	writel(ptcr, audmux_base + IMX_AUDMUX_V2_PTCR(port));
	writel(pdcr, audmux_base + IMX_AUDMUX_V2_PDCR(port));

	if (ports[id].external && ports[id].valid && !ports[id].exported) {
		pinctrl =
			devm_pinctrl_get_select_default(&(ports[id].pdev->dev));
		if (IS_ERR(pinctrl)) {
			dev_err(&(ports[id].pdev->dev), "setup pinctrl failed!");
			ret = PTR_ERR(pinctrl);
			goto out;
		} else
			ports[id].exported = 1;
	}

out:
	if (audmux_clk)
		clk_disable_unprepare(audmux_clk);

	return ret;
}
EXPORT_SYMBOL_GPL(imx_audmux_v2_configure_port);

static int imx_audmux_probe(struct platform_device *pdev)
{
	struct resource *res;
	struct pinctrl *pinctrl;
	const struct of_device_id *of_id =
			of_match_device(imx_audmux_dt_ids, &pdev->dev);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	audmux_base = devm_request_and_ioremap(&pdev->dev, res);
	if (!audmux_base)
		return -EADDRNOTAVAIL;

	pinctrl = devm_pinctrl_get_select_default(&pdev->dev);
	if (IS_ERR(pinctrl)) {
		audmux_base = NULL;
		dev_err(&pdev->dev, "setup pinctrl failed!");
		return PTR_ERR(pinctrl);
	}

	audmux_clk = clk_get(&pdev->dev, "audmux");
	if (IS_ERR(audmux_clk)) {
		dev_dbg(&pdev->dev, "cannot get clock: %ld\n",
				PTR_ERR(audmux_clk));
		audmux_clk = NULL;
	}

	if (of_id)
		pdev->id_entry = of_id->data;
	audmux_type = pdev->id_entry->driver_data;
	if (audmux_type == IMX31_AUDMUX)
		audmux_debugfs_init(MX31_AUDMUX_PORT6_SSI_PINS_6);
	if (audmux_type == IMX6_AUDMUX)
		audmux_debugfs_init(MX6_AUDMUX_PORT7);

	return 0;
}

static int imx_audmux_remove(struct platform_device *pdev)
{
	if (audmux_type == IMX31_AUDMUX || audmux_type == IMX6_AUDMUX)
		audmux_debugfs_remove();
	clk_put(audmux_clk);

	return 0;
}

#ifdef CONFIG_SND_SOC_IMX_AUDMUX_DEBUG
/* return port number according to Freescale documentation (1,2...)
 * if error returns (MAX_PORTS - 1) which allows access registers
 * without access violation
 */
static int  get_port_number(struct platform_device *pdev)
{
	int i;
	for (i = 1; i < MAX_PORTS; i++)
		if (ports[i].valid && ports[i].pdev == pdev)
			return i-1;

	return MAX_PORTS - 1; /* last elemenet */
}

/* read pdcr value via sysfs file */
static ssize_t pdcr_read(struct device *dev, struct device_attribute *attr,
		char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	uint32_t mode =
		readl(audmux_base + IMX_AUDMUX_V2_PDCR(get_port_number(pdev)));
	return sprintf(buf, "%08x\n", mode);
}

/* write pdcr value via sysfs file */
static ssize_t pdcr_write(struct device *dev, struct device_attribute *attr,
		const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	uint32_t mode;

	if (!sscanf(buf, "%x", &mode))
		return -EINVAL;

	writel(mode, audmux_base + IMX_AUDMUX_V2_PDCR(get_port_number(pdev)));
	return count;
}

static DEVICE_ATTR(pdcr, 0644, pdcr_read, pdcr_write);

/* read ptcr value via sysfs file */
static ssize_t ptcr_read(struct device *dev, struct device_attribute *attr,
		char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	uint32_t mode =
		readl(audmux_base + IMX_AUDMUX_V2_PTCR(get_port_number(pdev)));
	return sprintf(buf, "%08x\n", mode);
}

/* write ptcr value via sysfs file */
static ssize_t ptcr_write(struct device *dev, struct device_attribute *attr,
		const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	uint32_t mode;

	if (!sscanf(buf, "%x", &mode))
		return -EINVAL;

	writel(mode, audmux_base + IMX_AUDMUX_V2_PTCR(get_port_number(pdev)));
	return count;
}

static DEVICE_ATTR(ptcr, 0644, ptcr_read, ptcr_write);

static ssize_t export_read(struct device *dev, struct device_attribute *attr,
		char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	uint32_t mode = ports[get_port_number(pdev)].exported;
	return sprintf(buf, "%08x\n", mode);
}

static ssize_t export_write(struct device *dev, struct device_attribute *attr,
		const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct pinctrl *pinctrl;
	int id = get_port_number(pdev);
	uint32_t mode;

	if (!sscanf(buf, "%x", &mode))
		return -EINVAL;

	if (ports[id].external && ports[id].valid && mode) {
		ports[id].exported = 1;
		pinctrl =
			devm_pinctrl_get_select_default(&(ports[id].pdev->dev));
		if (IS_ERR(pinctrl))
			dev_err(&(ports[id].pdev->dev), "setup pinctrl failed!");
	}
	return count;
}

static DEVICE_ATTR(export, 0644, export_read, export_write);

static ssize_t loopback_read(struct device *dev, struct device_attribute *attr,
			char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	uint32_t mode = ports[get_port_number(pdev)].exported;
	return sprintf(buf, "%08x\n", mode);
}

static ssize_t loopback_write(struct device *dev, struct device_attribute *attr,
		const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct pinctrl *pinctrl;
	int id = get_port_number(pdev);
	int int_port = id;
	int ext_port = id;
	uint32_t mode;
	int ret;

	if (!sscanf(buf, "%x", &mode))
		return -EINVAL;

	/* Async, TX master, RX slave*/
	ret = imx_audmux_v2_configure_port(int_port,
			IMX_AUDMUX_V2_PTCR_RFSEL(ext_port) |
			IMX_AUDMUX_V2_PTCR_RCSEL(ext_port) |
			IMX_AUDMUX_V2_PTCR_RFSDIR |
			IMX_AUDMUX_V2_PTCR_RCLKDIR,
			IMX_AUDMUX_V2_PDCR_RXDSEL(ext_port));
	if (ret) {
		dev_err(&pdev->dev, "audmux loopback setup failed\n");
		return ret;
	}

	if (ports[id].external && ports[id].valid && mode) {
		ports[id].exported = 1;
		pinctrl =
			devm_pinctrl_get_select_default(&(ports[id].pdev->dev));
		if (IS_ERR(pinctrl))
			dev_err(&(ports[id].pdev->dev), "setup pinctrl failed!");
	}
	return count;
}

static DEVICE_ATTR(loopback, 0644, loopback_read, loopback_write);

static const struct attribute *additional_attrs[] = {
	&dev_attr_loopback.attr,
	&dev_attr_pdcr.attr,
	&dev_attr_ptcr.attr,
	NULL,
};

static const struct attribute_group additional_attr_group = {
	.attrs = (struct attribute **)additional_attrs,
};

static const struct attribute *export_attrs[] = {
	&dev_attr_export.attr,
	NULL,
};

static const struct attribute_group export_attr_group = {
	.attrs = (struct attribute **)export_attrs,
};
#endif

#ifdef CONFIG_SND_SOC_IMX_AUDMUX_DEBUG
static inline void imx_audmuxport_create_debug(struct platform_device *pdev,
			int port)
{
	if (ports[port].external)
		if (sysfs_create_group(&pdev->dev.kobj,
				 &export_attr_group))
			dev_err(&pdev->dev,
				"Can't create sysfs entry for export\n");

	if (sysfs_create_group(&pdev->dev.kobj, &additional_attr_group))
		dev_err(&pdev->dev, "Can't create sysfs entries");
}

static inline void imx_audmuxport_remove_debug(struct platform_device *pdev)
{
	sysfs_remove_group(&pdev->dev.kobj, &additional_attr_group);
	if (ports[get_port_number(pdev)].external)
		sysfs_remove_group(&pdev->dev.kobj, &export_attr_group);
}
#else
static inline void imx_audmuxport_create_debug(struct platform_device *pdev,
			int port) {}

static inline void imx_audmuxport_remove_debug(struct platform_device *pdev) {}
#endif

static int imx_audmuxport_probe(struct platform_device *pdev)
{
	struct device_node *connection;
	int i, ret, prev;
	int tfsel = 0;
	int tcsel = 0;
	int rfsel = 0;
	int rcsel = 0;
	int sync = 0;
	int rxdsel = 0;
	int txrxen = 0;
	int inmask = 0xff;
	u32 pdcr, ptcr;

	if (!audmux_base) {
		dev_err(&pdev->dev, "audmux is not probed properly\n");
		return -ENODEV;
	}

	ret = of_property_read_u32(pdev->dev.of_node, "num", &i);
	if (ret) {
		dev_err(&pdev->dev, "num missing or invalid\n");
		return ret;
	} else
	if (i <= 0)
		return -ENODEV;

	connection = of_parse_phandle(pdev->dev.of_node, "connection", 0);
	if (!connection)
		ports[i].external = 1;
	ports[i].pdev = pdev;
	ports[i].valid = 1;

	/* configuration port from DT node */
	ptcr = readl(audmux_base + IMX_AUDMUX_V2_PTCR(i-1));
	pdcr = readl(audmux_base + IMX_AUDMUX_V2_PDCR(i-1));

	/* TFSEL and TCSEL */
	ret = of_property_read_u32(pdev->dev.of_node, "tclkfsy", &tfsel);
	if (!ret && tfsel >= 0) {
		/* set both FTSEL and TCSEL from one param */
		dev_notice(&pdev->dev, "tckfsy = %d", tfsel);
		if (!tfsel)
			ptcr &= ~IMX_AUDMUX_V2_PTCR_TFSDIR;
		else {
			ptcr &= ~(IMX_AUDMUX_V2_PTCR_TFSEL(0xf));
			ptcr |= IMX_AUDMUX_V2_PTCR_TFSEL(tfsel - 1) |
				IMX_AUDMUX_V2_PTCR_TFSDIR;
		}
		if (!tfsel)
			ptcr &= ~IMX_AUDMUX_V2_PTCR_TCLKDIR;
		else {
			ptcr &= ~(IMX_AUDMUX_V2_PTCR_TCSEL(0xf));
			ptcr |= IMX_AUDMUX_V2_PTCR_TCSEL(tfsel - 1) |
				IMX_AUDMUX_V2_PTCR_TCLKDIR;
		}
	}
	prev = ret;
	ret = of_property_read_u32(pdev->dev.of_node, "tclk", &tfsel);
	if (!ret && tfsel >= 0) {
		if (!prev)
			dev_err(&pdev->dev, "both tclkfsy and tclk defined");
		else {
			/* set TFSEL only */
			dev_notice(&pdev->dev, "tclk = %d", tfsel);
			if (!tfsel)
				ptcr &= ~IMX_AUDMUX_V2_PTCR_TFSDIR;
			else {
				ptcr &= ~(IMX_AUDMUX_V2_PTCR_TFSEL(0xf));
				ptcr |= IMX_AUDMUX_V2_PTCR_TFSEL(tfsel - 1) |
					IMX_AUDMUX_V2_PTCR_TFSDIR;
			}
		}
	}
	ret = of_property_read_u32(pdev->dev.of_node, "tfsy", &tcsel);
	if (!ret && tcsel >= 0) {
		if (!prev)
			dev_err(&pdev->dev, "both tclkfsy and tfsy defined");
		else {
			/* set TCSEL only */
			dev_notice(&pdev->dev, "tfsy = %d", tcsel);
			if (!tcsel)
				ptcr &= ~IMX_AUDMUX_V2_PTCR_TCLKDIR;
			else {
				ptcr &= ~(IMX_AUDMUX_V2_PTCR_TCSEL(0xf));
				ptcr |= IMX_AUDMUX_V2_PTCR_TCSEL(tcsel - 1) |
					IMX_AUDMUX_V2_PTCR_TCLKDIR;
			}
		}
	}

	/* SYN */
	ret = of_property_read_u32(pdev->dev.of_node,
					"symmetric", &sync);
	if (!ret) {
		dev_notice(&pdev->dev, "symmetric = %d", sync);
		if (sync)
			ptcr |= IMX_AUDMUX_V2_PTCR_SYN;
		else
			ptcr &= ~IMX_AUDMUX_V2_PTCR_SYN;
	}

	/* RFSEL and RCSEL */
	ret = of_property_read_u32(pdev->dev.of_node, "rclkfsy", &rfsel);
	if (!ret && rfsel >= 0) {
		if (sync)
			dev_err(&pdev->dev, "rclkfsy defined in sync mode");
		/* set both RFSEL and RCSEL form one param */
		dev_notice(&pdev->dev, "rclkfsy = %d", rfsel);
		if (!rfsel)
			ptcr &= ~IMX_AUDMUX_V2_PTCR_RFSDIR;
		else {
			ptcr &= ~(IMX_AUDMUX_V2_PTCR_RFSEL(0xf));
			ptcr |= IMX_AUDMUX_V2_PTCR_RFSEL(rfsel - 1) |
				IMX_AUDMUX_V2_PTCR_RFSDIR;
		}
		if (!rfsel)
			ptcr &= ~IMX_AUDMUX_V2_PTCR_RCLKDIR;
		else {
			ptcr &= ~(IMX_AUDMUX_V2_PTCR_RCSEL(0xf));
			ptcr |= IMX_AUDMUX_V2_PTCR_RCSEL(rfsel - 1) |
				IMX_AUDMUX_V2_PTCR_RCLKDIR;
		}
	}
	prev = ret;
	ret = of_property_read_u32(pdev->dev.of_node, "rclk", &rfsel);
	if (!ret && rfsel >= 0) {
		if (sync)
			dev_err(&pdev->dev, "rclk defined in sync mode");
		if (!prev)
			dev_err(&pdev->dev, "both rclkfsy and rclk defined");
		else {
			/* set RFSEL only */
			dev_notice(&pdev->dev, "rclk = %d", rfsel);
			if (!rfsel)
				ptcr &= ~IMX_AUDMUX_V2_PTCR_RFSDIR;
			else {
				ptcr &= ~(IMX_AUDMUX_V2_PTCR_RFSEL(0xf));
				ptcr |= IMX_AUDMUX_V2_PTCR_RFSEL(rfsel - 1) |
					IMX_AUDMUX_V2_PTCR_RFSDIR;
			}
		}
	}
	ret = of_property_read_u32(pdev->dev.of_node, "rfsy", &rcsel);
	if (!ret && rcsel >= 0) {
		if (sync)
			dev_err(&pdev->dev, "rfsy defined in sync mode");
		if (!prev)
			dev_err(&pdev->dev, "both rclkfsy and rfsy defined");
		else {
			/* set RCSEL only */
			dev_notice(&pdev->dev, "rfsy = %d", rcsel);
			if (!rcsel)
				ptcr &= ~IMX_AUDMUX_V2_PTCR_RCLKDIR;
			else {
				ptcr &= ~(IMX_AUDMUX_V2_PTCR_RCSEL(0xf));
				ptcr |= IMX_AUDMUX_V2_PTCR_RCSEL(rcsel - 1) |
					IMX_AUDMUX_V2_PTCR_RCLKDIR;
			}
		}
	}

	/* RXDSEL */
	ret = of_property_read_u32(pdev->dev.of_node,
					"rdata", &rxdsel);
	if (!ret) {
		dev_notice(&pdev->dev, "rdata = %d", rxdsel);
		if (rxdsel) {
			pdcr &= ~(IMX_AUDMUX_V2_PDCR_RXDSEL(0x7));
			pdcr |= IMX_AUDMUX_V2_PDCR_RXDSEL(rxdsel - 1);
			pdcr &= ~(IMX_AUDMUX_V2_PDCR_MODE(0x3));
			pdcr |= IMX_AUDMUX_V2_PDCR_MODE(0);
		}
	}

	/* INMMASK and MODE */
	prev = ret;
	ret = of_property_read_u32(pdev->dev.of_node,
					"rdata_mask", &inmask);
	if (!ret) {
		if (!prev)
			dev_err(&pdev->dev, "both rdata_mask and rdata defined");
		dev_notice(&pdev->dev, "rdata_mask = %d", inmask);
		pdcr &= ~(IMX_AUDMUX_V2_PDCR_INMMASK(0xff));
		pdcr |= IMX_AUDMUX_V2_PDCR_INMMASK(inmask);
		pdcr &= ~(IMX_AUDMUX_V2_PDCR_MODE(0x3));
		pdcr |= IMX_AUDMUX_V2_PDCR_MODE(1);
	}

	/* TXRXEN */
	ret = of_property_read_u32(pdev->dev.of_node,
					"tx_rx_switch", &txrxen);
	if (!ret) {
		dev_notice(&pdev->dev, "tx_rx_switch = %d", txrxen);
		if (txrxen)
			pdcr |= IMX_AUDMUX_V2_PDCR_TXRXEN;
		else
			pdcr &= ~IMX_AUDMUX_V2_PDCR_TXRXEN;
	}

	ret = imx_audmux_v2_configure_port(i-1, ptcr, pdcr);
	if (ret) {
		dev_err(&pdev->dev,
			"audmux setting port %d failed\n", i);
		return ret;
	}

	/* end port configuration */
	imx_audmuxport_create_debug(pdev, i);
	return 0;
}

static int imx_audmuxport_remove(struct platform_device *pdev)
{
	imx_audmuxport_remove_debug(pdev);
	return 0;
}

static const struct of_device_id imx_audmuxport_dt_ids[] = {
	{ .compatible = "fsl,imx31-audmux-port", },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, imx_audmuxport_dt_ids);

static struct platform_driver imx_audmuxport_driver = {
	.probe		= imx_audmuxport_probe,
	.remove		= imx_audmuxport_remove,
	.driver	= {
		.name	= "audmux-port",
		.owner	= THIS_MODULE,
		.of_match_table = imx_audmuxport_dt_ids,
	}
};

static struct platform_driver imx_audmux_driver = {
	.probe		= imx_audmux_probe,
	.remove		= imx_audmux_remove,
	.id_table	= imx_audmux_ids,
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = imx_audmux_dt_ids,
	}
};

static int __init imx_audmux_init(void)
{
	int ret = platform_driver_register(&imx_audmux_driver);
	if (!ret)
		platform_driver_register(&imx_audmuxport_driver);
	return ret;
}
module_init(imx_audmux_init);

static void __exit imx_audmux_exit(void)
{
	platform_driver_unregister(&imx_audmuxport_driver);
	platform_driver_unregister(&imx_audmux_driver);
}
module_exit(imx_audmux_exit);

MODULE_DESCRIPTION("Freescale i.MX AUDMUX driver");
MODULE_AUTHOR("Sascha Hauer <s.hauer@pengutronix.de>");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:" DRIVER_NAME);
