/*
 * Copyright 2008-2011 Freescale Semiconductor, Inc. All Rights Reserved.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

 /*!
  * @file       fsl_esai.c
  * @brief      this file implements the esai interface
  *             in according to ASoC architecture
  */

#include <linux/init.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/pinctrl/consumer.h>
#include <linux/hrtimer.h>
#include <linux/ktime.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/initval.h>
#include <sound/soc.h>

#include <linux/platform_data/dma-imx.h>

#include "imx-pcm.h"
#include "fsl_esai.h"
#include <linux/mxc_asrc_core_internal.h>
#include <dt-bindings/clock/imx6dq-sysclk-esai.h>

#define MAX_STREAM_NAME_SIZE 64
#define MAX_SLOT_WIDTH_SIZE 32
#define MAX_SAMPLE_SIZE	24
#define MAX_DAI_NAME_SIZE 64
/* defaults settings */
#define ESAI_DEFAULT_SLOTS 2
#define ESAI_DEFAULT_MASK 0xffffffff
#define ESAI_DEFAULT_SLOT_WIDTH 16
#define ESAI_DEFAULT_TX_LINES 0xf
#define ESAI_DEFAULT_RX_LINES 0x3

#define ESAI_DEFAULT_STARTUP_TIMER_TIMEOUT_US 200
#define ESAI_DEFAULT_MAX_STARTUP_RETRIES 25
#define ESAI_DEFAULT_WATERMARK_LEVEL 64
#define ESAI_MAX_TIMER_TIMEOUT 65535
#define ESAI_MAX_TIMER_RETRIES 65535
#define ESAI_MAX_START_FIFO_LEVEL 128
#define ESAI_MIN_WATERMARK_LEVEL 1
#define ESAI_DEFAULT_FIFO_DEPTH 128
#define ESAI_MIN_SDMA_BURST_SIZE 1

#define ESAI_TRANSMIT_ENABLED			0
#define ESAI_TRANSMIT_ERROR_STREAM_NULL		-1
#define ESAI_TRANSMIT_ERROR_TRIGGER_TX_DISABLED -2
#define ESAI_CLK_DEFAULT_RATE 0
#define ESAI_PM_MAX 256
#define ESAI_PSR_MAX 8
#define ESAI_PSR_MIN 1
#define ESAI_PSR_PM_MIN 1
#define ESAI_PSR_PM_MAX (ESAI_PSR_MAX*ESAI_PM_MAX)
#define ESAI_FP_MAX 16
#define ESAI_FP_MIN 1
#define ESAI_DIV_2 2

/*
 * IMX_ESAI_FORMATS and esai_supported_formats[] should correspond
 * to each other otherwise the formats will be constrained incorrectly
 */
#define IMX_ESAI_FORMATS \
	(SNDRV_PCM_FMTBIT_S16_LE | SNDRV_PCM_FMTBIT_S20_3LE | \
	SNDRV_PCM_FMTBIT_S24_LE | SNDRV_PCM_FMTBIT_S32_LE)
static const unsigned int esai_supported_formats[] = {
	SNDRV_PCM_FORMAT_S16_LE,
	SNDRV_PCM_FORMAT_S20_3LE,
	SNDRV_PCM_FORMAT_S24_LE,
	SNDRV_PCM_FORMAT_S32_LE,
};

#define ESAI_SUPPORT_DAI_NUM 2
#define ESAI_PFDRV_NUM_SYNC 1
#define ESAI_PFDRV_NUM_ASYNC 2
#define ESAI_DISCONNECT_FLAG 0
#define ESAI_CONNECT_FLAG 1

#define ESAI_PLAYBACK_USING_GPIO (1 << 0)
#define ESAI_CAPTURE_USING_GPIO (1 << 1)
/**
 * fsl_esai_private: per-ESAI private data
 *
 * @name: name for this device
 */
struct fsl_esai_private {
	spinlock_t lock;
	void __iomem *base;
	dma_addr_t esai_phys;
	unsigned int irq;
	struct snd_pcm_substream *first_stream;
	struct snd_pcm_substream *second_stream;
	struct snd_pcm_substream *playback;
	struct snd_pcm_substream *capture;
	unsigned int fifo_depth;
	struct snd_soc_dai_driver cpu_dai_drv[ESAI_SUPPORT_DAI_NUM];
	struct device_attribute dev_attr;
	struct platform_device *pdev;
#if defined(CONFIG_SND_SOC_IMX_ASRC_DAI_SUPPORT)
	struct fsl_asrc_core_private *asrc_core_private;
	bool ideal_clk_set[2];
#endif

	bool new_binding;
	struct clk *clk;
	struct clk *fsysclk;
	struct clk *ipgclk;
	struct platform_device *imx_pcm_pdev[ESAI_SUPPORT_DAI_NUM];
	struct imx_pcm_dma_params dma_params_tx;
	struct imx_pcm_dma_params dma_params_rx;
	int flags;
	int clk_lines_flag;
	int slot_width[ESAI_SUPPORT_DAI_NUM];
	int slots[ESAI_SUPPORT_DAI_NUM];
	unsigned int tx_mask;
	unsigned int rx_mask;
	unsigned int active_tx_mask;
	unsigned int active_rx_mask;
	unsigned char tx_watermark;
	unsigned char rx_watermark;
	unsigned int tx_lines;
	unsigned int rx_lines;
	/* indicate if dividers were set after open */
	bool div[ESAI_SUPPORT_DAI_NUM];
	char esai_playback_stream_name[MAX_STREAM_NAME_SIZE];
	char esai_capture_stream_name[MAX_STREAM_NAME_SIZE];
	char esai_playback_dai_name[MAX_DAI_NAME_SIZE];
	char esai_capture_dai_name[MAX_DAI_NAME_SIZE];
	struct hrtimer tx_startup_timer;
	long tx_start_timeout;
	int tx_start_retries_cnt;
	int tx_start_max_retries;
	unsigned char tx_start_fifo_level;
	bool tx_trigger_enable;
	struct device_attribute dev_attr_timeout;
	struct device_attribute dev_attr_retries;
	struct device_attribute dev_attr_retry_count;
	struct device_attribute dev_attr_fifo_level;
	int bck_cal_input_freq[ESAI_SUPPORT_DAI_NUM];
	bool hck_divs[ESAI_SUPPORT_DAI_NUM];
	bool sysclk[ESAI_SUPPORT_DAI_NUM];
	unsigned int fmt[ESAI_SUPPORT_DAI_NUM];
	/* indicate if HCKT/HCKR needs to be set */
	bool gpio_hck[ESAI_SUPPORT_DAI_NUM];
	/*
	 * name must be last element for this structure to avoid overwriting,
	 * driver allocating extra memory in probe() for dai name.
	 * esai = kzalloc(sizeof(struct fsl_esai_private) + strlen(p),
	 *             GFP_KERNEL);
	 */
	char name[1];
};

static inline int fsl_esai_active_slots(int slots, unsigned int mask)
{
	return bitmap_weight((const unsigned long *)&mask, slots);
}

static inline int fsl_esai_mask_calc(struct device *dev,
				struct fsl_esai_private *esai,
				int channels, unsigned int *mask, int dir,
				int dai_id)
{
	int used_signal_lines;
	int active_slots;
	int max_signal_lines;
	unsigned int used_mask = 0;

	if (dir == SNDRV_PCM_STREAM_PLAYBACK)
		max_signal_lines = hweight32(esai->tx_lines);
	else
		max_signal_lines = hweight32(esai->rx_lines);

	active_slots = fsl_esai_active_slots(esai->slots[dai_id], *mask);
	if (channels <= active_slots) {
		used_mask = fsl_esai_mask_cut(*mask, channels);
	} else {
		if ((channels % active_slots) == 0) {
			used_mask = *mask &
			BITMAP_LAST_WORD_MASK(esai->slots[dai_id]);
			used_signal_lines = channels / active_slots;
			if (used_signal_lines > max_signal_lines) {
				dev_err(dev, "Channel count %d is not supported\n",
					channels);
				return -EINVAL;
			}
		} else {
			dev_err(dev, "Channel count %d is not supported\n",
				channels);
			return -EINVAL;
		}
	}

	*mask = used_mask;
	return 0;
}

/**
 * flip_number: flips number sideways (MSB <==> LSB)
 *
 * This function flips the lower ESAI_MAX_TX_LINES_COUNT bits of the TX hw
 * lines value read from the DT to match them with HW register bit positions
 * in PRRC/PCRC.
 *
 * For a 'tx-lines=0x14', flip_number(0x14) would produce 0x0A
 *              0x14 : 01 0100
 * flip_number(0x14) : 00 1010
 *
 * @n: the tx-line value
 */
int flip_number(int n)
{
	int r = 0;
	int j = 0;
	while (j < ESAI_MAX_TX_LINES_COUNT) {
		r = (r << 1) | (n & 1);
		n >>= 1;
		j++;
	}

	return r;
}

#if defined(CONFIG_SND_SOC_IMX_ASRC_DAI_SUPPORT)

static int check_for_asrc(struct fsl_esai_private *esai_private,
		struct platform_device *pdev) {

	struct platform_device *asrc_core_pdev;
	struct device_node *asrc_core_np = NULL;

	/* get the parent plaform device private data handle */
	asrc_core_np = of_parse_phandle(pdev->dev.of_node,
			"asrc-core", 0);
	if (asrc_core_np) {
		/* retrieve asrc_core pdev handle */
		asrc_core_pdev = of_find_device_by_node(asrc_core_np);
		if (!asrc_core_pdev) {
			dev_err(&pdev->dev,
			"failed to find ASRC core platform device\n");
			return -EINVAL;
		}

		/* initialize data */
		esai_private->asrc_core_private =
				platform_get_drvdata(asrc_core_pdev);
		dev_info(&pdev->dev, "mxc asrc core found: %p.\n",
				esai_private->asrc_core_private);
	}

	return 0;
}

static struct platform_device *register_named_platform(
		struct fsl_esai_private *esai_private,
		struct platform_device *pdev,
		const char const *pfdrv_name, int platform_id)
{
	return platform_device_register_data(&pdev->dev,
			pfdrv_name, platform_id,
			&esai_private->asrc_core_private,
			sizeof(struct fsl_asrc_core_private *));
}

static int update_ideal_clock(struct fsl_esai_private *esai,
		struct snd_pcm_substream *substream,
		struct snd_pcm_hw_params *hw_params,
		int dai_id)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	char *clock_name = NULL;
	unsigned int bit_clock_rate = 0;
	int ret;

	if (esai->asrc_core_private) {
		/* for ASRC P2P */
		ret = asrc_get_ideal_clkname_from_stream(
			rtd->dai_link->name, &clock_name,
			(substream->stream == SNDRV_PCM_STREAM_PLAYBACK) ?
					ASRC_PLAYBACK : ASRC_CAPTURE);

		if (ret < 0) {
			dev_err(rtd->cpu_dai->dev, "%s: '%s' clk not found",
					__func__, clock_name);
			return ret;
		}

		/* calculate bit clock*/
		bit_clock_rate = params_rate(hw_params)
				* esai->slots[dai_id]
				* esai->slot_width[dai_id];

		ret = asrc_set_ideal_clock_rate(esai->asrc_core_private,
				clock_name, bit_clock_rate);

		kfree(clock_name);
		clock_name = NULL;

		if (ret < 0) {
			dev_err(rtd->cpu_dai->dev, "%s: clk not found or attempting to change rate of already set clock",
					__func__);
			return ret;
		}
		esai->ideal_clk_set[substream->stream] = true;
	}

	return 0;
}

static int invalidate_ideal_clock(struct fsl_esai_private *esai,
		struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	int ret = -EINVAL;


	if (esai->asrc_core_private) {
		/* for ASRC P2P */
		if (esai->ideal_clk_set[substream->stream]) {
			ret = asrc_invalidate_ideal_clock(
				esai->asrc_core_private,
				rtd->dai_link->name,
				(substream->stream ==
						SNDRV_PCM_STREAM_PLAYBACK) ?
						ASRC_PLAYBACK : ASRC_CAPTURE);
			if (ret == 0)
				esai->ideal_clk_set[substream->stream] = false;
		}
	}

	return ret;
}

#else
/**
 * No ASRC integration
 */
static int check_for_asrc(struct fsl_esai_private *esai_private,
		struct platform_device *pdev) { return 0; }

static struct platform_device *register_named_platform(
		struct fsl_esai_private *esai_private,
		struct platform_device *pdev,
		const char const *pfdrv_name, int platform_id)
{
	return platform_device_register_simple(pfdrv_name,
			platform_id, NULL, 0);
}

static int update_ideal_clock(struct fsl_esai_private *esai_private,
		struct snd_pcm_substream *substream,
		struct snd_pcm_hw_params *hw_params,
		int dai_id)
		{ return 0; }

static int invalidate_ideal_clock(struct fsl_esai_private *esai,
		struct snd_pcm_substream *substream)
		{ return 0; }
#endif

/* read tx_start_timeout value via sysfs file */
static ssize_t tx_start_timeout_read(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct fsl_esai_private *esai = container_of(attr,
						     struct fsl_esai_private,
						     dev_attr_timeout);

	return sprintf(buf, "%ld\n", (esai->tx_start_timeout / 1000));
}

/* write tx_start_timeout value via sysfs file */
static ssize_t tx_start_timeout_write(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int n_timeout;
	struct fsl_esai_private *esai = container_of(attr,
						     struct fsl_esai_private,
						     dev_attr_timeout);

	if (esai->tx_trigger_enable)
		return -EBUSY;

	ret = kstrtoint(buf, 0, &n_timeout);
	if (ret)
		return ret;

	if ((n_timeout <= 0) || (n_timeout > ESAI_MAX_TIMER_TIMEOUT))
		return -EINVAL;

	esai->tx_start_timeout = (n_timeout * 1000);

	return count;
}

/* read tx_start_retries value via sysfs file */
static ssize_t tx_start_retries_read(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_esai_private *esai = container_of(attr,
						     struct fsl_esai_private,
						     dev_attr_retries);

	return sprintf(buf, "%d\n", esai->tx_start_max_retries);
}

/* write tx_start_retries value via sysfs file */
static ssize_t tx_start_retries_write(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int n_retries;
	struct fsl_esai_private *esai = container_of(attr,
						     struct fsl_esai_private,
						     dev_attr_retries);

	if (esai->tx_trigger_enable)
		return -EBUSY;

	ret = kstrtoint(buf, 0, &n_retries);
	if (ret)
		return ret;

	if ((n_retries < 0) || (n_retries > ESAI_MAX_TIMER_RETRIES))
		return -EINVAL;

	esai->tx_start_max_retries = n_retries;

	return count;
}

/* read tx_start_retry_count value via sysfs file */
static ssize_t tx_start_retry_count_read(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_esai_private *esai = container_of(attr,
						     struct fsl_esai_private,
						     dev_attr_retry_count);

	return sprintf(buf, "%d\n", esai->tx_start_retries_cnt);
}

/* read tx_start_fifo_level value via sysfs file */
static ssize_t tx_start_fifo_level_read(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct fsl_esai_private *esai = container_of(attr,
						     struct fsl_esai_private,
						     dev_attr_fifo_level);

	return sprintf(buf, "%d\n", esai->tx_start_fifo_level);
}

/* write tx_start_fifo_level value via sysfs file */
static ssize_t tx_start_fifo_level_write(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int n_fifo_level;
	struct fsl_esai_private *esai = container_of(attr,
						     struct fsl_esai_private,
						     dev_attr_fifo_level);

	if (esai->tx_trigger_enable)
		return -EBUSY;

	ret = kstrtoint(buf, 0, &n_fifo_level);
	if (ret)
		return ret;

	if ((n_fifo_level <= 0) || (n_fifo_level > ESAI_MAX_START_FIFO_LEVEL))
		return -EINVAL;

	esai->tx_start_fifo_level = n_fifo_level;

	return count;
}

/*
 * ESAI DAI sysclock configuration.
 *
 * Configures the ESAI's sysclock hardware dividers in a number of
 * different ways depending on whether HCK is used or not.
 */
static int fsl_esai_set_dai_sysclk(struct snd_soc_dai *cpu_dai,
				   int clk_id, unsigned int freq, int dir)
{
	struct fsl_esai_private *esai =
		snd_soc_dai_get_drvdata(cpu_dai);
	unsigned long flags;
	u32 ecr, xccr;
	unsigned int offset;
	int pm = 0, psr = 0;

	if (esai->flags & IMX_ESAI_SYN)
		offset = ESAI_TCCR;
	else
		offset = cpu_dai->id ==
				ID_SYNC_ASYNC_PLAYBACK ?
						ESAI_TCCR : ESAI_RCCR;

	if (!(cpu_dai->rate == 0)) {
		dev_warn(cpu_dai->dev,
				"clock %d is already in use, settings will not be modified",
				clk_id);
		return -EAGAIN;
	}

	spin_lock_irqsave(&esai->lock, flags);
	xccr = readl(esai->base + offset);
	/* Deactivate HCK output */
	xccr &= ~ESAI_TCCR_THCKD;
	writel(xccr, esai->base + offset);
	spin_unlock_irqrestore(&esai->lock, flags);

	if (clk_id == SYSCLK_ESAI_HCK) {
		if (dir == SYSCLK_ESAI_DIR_IN) {
			if (freq == 0) {
				dev_err(cpu_dai->dev,
						"Invalid input freq. HCK input frequency has to be different from 0\n");
				return -EINVAL;
			}
			/*
			 * if clock is HCK input then
			 * store clock frequency input locally
			 */
			esai->bck_cal_input_freq[cpu_dai->id] = freq;
			spin_lock_irqsave(&esai->lock, flags);
			xccr = readl(esai->base + offset);

			xccr &= ~ESAI_TCCR_THCKD;
			writel(xccr, esai->base + offset);
			ESAI_DUMP();
			spin_unlock_irqrestore(&esai->lock, flags);
			esai->sysclk[cpu_dai->id] = true;
			esai->hck_divs[cpu_dai->id] = false;
			esai->gpio_hck[cpu_dai->id] = true;
		} else {
			dev_err(cpu_dai->dev,
					"HCK with SYSCLK_ESAI_DIR_OUT not possible\n");
			return -EINVAL;
		}
	} else {
		unsigned int hck_cal_input_freq = 0;
		switch (clk_id) {
		case SYSCLK_ESAI_EXTAL:
			if (freq != 0) {
				clk_set_rate(esai->clk, freq);
				if (!(esai->flags & IMX_ESAI_SYN))
					dev_warn(cpu_dai->dev, "Common clock EXTAL is updated in ASYNC mode\n");
			}
			esai->bck_cal_input_freq[cpu_dai->id] =
						clk_get_rate(esai->clk);
			esai->hck_divs[cpu_dai->id] = false;
			break;
		case SYSCLK_ESAI_EXTAL_DIVS:
			if (freq == 0) {
				dev_err(cpu_dai->dev, "Unable to configure SYSCLK_ESAI_EXTAL_DIVS. Specified frequency has to be different from 0\n");
				return -EINVAL;
			}
			hck_cal_input_freq = clk_get_rate(esai->clk);
			esai->hck_divs[cpu_dai->id] = true;
			break;
		case SYSCLK_ESAI_FSYS_DIVS:
			if (freq == 0) {
				dev_err(cpu_dai->dev, "Unable to configure SYSCLK_ESAI_FSYS_DIVS. Specified frequency has to be different from 0\n");
				return -EINVAL;
			}
			if (esai->fsysclk == NULL) {
				dev_err(cpu_dai->dev, "FSYS clock is not available\n");
				return -EINVAL;
			}
			hck_cal_input_freq = clk_get_rate(esai->fsysclk);
			esai->hck_divs[cpu_dai->id] = true;
			break;
		case SYSCLK_ESAI_FSYS:
			if (dir == SYSCLK_ESAI_DIR_OUT) {
				dev_err(cpu_dai->dev,
						"FSYS with SYSCLK_ESAI_DIR_OUT not possible\n");
				return -EINVAL;
			}
			if (esai->fsysclk == NULL) {
				dev_err(cpu_dai->dev, "FSYS clock is not available\n");
				return -EINVAL;
			}
			esai->bck_cal_input_freq[cpu_dai->id] =
						clk_get_rate(esai->fsysclk);
			esai->hck_divs[cpu_dai->id] = false;
			break;
		default:
			dev_err(cpu_dai->dev,
					"%d is not a valid clock!\n", clk_id);
			return -EINVAL;
		}

		if (esai->hck_divs[cpu_dai->id] == true) {
			/* HCK target frequency is freq */
			unsigned int value =
					DIV_ROUND_CLOSEST(hck_cal_input_freq,
							ESAI_DIV_2 * freq);
			/* Calculate HCK related divs*/
			if ((value <= ESAI_PSR_PM_MAX)
					&& (value >= ESAI_PSR_PM_MIN)) {
				if (value <= ESAI_PM_MAX) {
					psr = ESAI_PSR_MIN;
					pm = value;
				} else {
					psr = ESAI_PSR_MAX;
					pm = DIV_ROUND_CLOSEST(value,
							ESAI_PSR_MAX);
				}
				/* BCK calculation input FREQ =
				 * EXTAL/FSYS rate divided by HCK divs
				 */
				esai->bck_cal_input_freq[cpu_dai->id] =
						DIV_ROUND_CLOSEST(
						hck_cal_input_freq,
						(psr * pm * ESAI_DIV_2));
			} else {
				dev_err(cpu_dai->dev, "HCK out of range\n");
				esai->hck_divs[cpu_dai->id] = false;
				return -EINVAL;
			}
		}

		esai->sysclk[cpu_dai->id] = true;
		spin_lock_irqsave(&esai->lock, flags);
		ecr = readl(esai->base + ESAI_ECR);

		switch (clk_id) {
		case SYSCLK_ESAI_EXTAL:
			if (cpu_dai->id == ID_SYNC_ASYNC_PLAYBACK) {
				ecr |= ESAI_ECR_ETO;
				ecr |= ESAI_ECR_ETI;
				/* ERI should only be set in sync
				 * mode and not if async and direction is play
				 */
				if (esai->flags & IMX_ESAI_SYN)
					ecr |= ESAI_ECR_ERI;
			} else {
				ecr |= ESAI_ECR_ERO;
				ecr |= ESAI_ECR_ERI;
			}
			break;
		case SYSCLK_ESAI_EXTAL_DIVS:
			if (cpu_dai->id == ID_SYNC_ASYNC_PLAYBACK) {
				ecr &= ~ESAI_ECR_ETO;
				ecr |= ESAI_ECR_ETI;
				/* ERI should only be set in sync
				 * mode and not if async and direction is play
				 */
				if (esai->flags & IMX_ESAI_SYN)
					ecr |= ESAI_ECR_ERI;
			} else {
				ecr &= ~ESAI_ECR_ERO;
				ecr |= ESAI_ECR_ERI;
			}
			break;
		case SYSCLK_ESAI_FSYS_DIVS:
		case SYSCLK_ESAI_FSYS:
			if (cpu_dai->id == ID_SYNC_ASYNC_PLAYBACK) {
				ecr &= ~(ESAI_ECR_ETI | ESAI_ECR_ETO);
				/* ERI should only be set in sync
				 * mode and not if async and direction is play
				 */
				if (esai->flags & IMX_ESAI_SYN)
					ecr &= ~ESAI_ECR_ERI;
			}
			else
				ecr &= ~(ESAI_ECR_ERI | ESAI_ECR_ERO);
		}
		writel(ecr, esai->base + ESAI_ECR);

		xccr = readl(esai->base + offset);
		if (esai->hck_divs[cpu_dai->id] == true) {
			xccr &= ESAI_TCCR_TPM_MASK;
			xccr |= ESAI_TCCR_TPM((pm-1));
			xccr &= ESAI_TCCR_TPSR_MASK;
			if (psr == 1)
				xccr |= ESAI_TCCR_TPSR_BYPASS;
			else
				xccr |= ESAI_TCCR_TPSR_DIV8;
		}

		if (dir == SYSCLK_ESAI_DIR_IN) {
			/*THCKD=1 PRRC5=0, PCRC5=0 */
			esai->gpio_hck[cpu_dai->id] = false;
			xccr |= ESAI_TCCR_THCKD;
		} else {
			/*THCKD=1 PRRC5=1, PCRC5=1 */
			esai->gpio_hck[cpu_dai->id] = true;
			xccr |= ESAI_TCCR_THCKD;
		}
		writel(xccr, esai->base + offset);
		ESAI_DUMP();
		spin_unlock_irqrestore(&esai->lock, flags);
	}
	return 0;
}

static int fsl_esai_set_dai_clkdiv(struct snd_soc_dai *cpu_dai,
				   int div_id, int div)
{
	struct fsl_esai_private *esai =
	    snd_soc_dai_get_drvdata(cpu_dai);
	u32 xccr;
	unsigned long flags;
	unsigned int offset;

	if (esai->flags & IMX_ESAI_SYN)
		offset = ESAI_TCCR;
	else
		offset = cpu_dai->id ==
				ID_SYNC_ASYNC_PLAYBACK ?
						ESAI_TCCR : ESAI_RCCR;

	spin_lock_irqsave(&esai->lock, flags);

	xccr = readl(esai->base + offset);

	switch (div_id) {
	case SYSCLK_ESAI_DIV_PSR:
		xccr &= ESAI_TCCR_TPSR_MASK;
		if (div == SYSCLK_ESAI_DIV_PSR_DIVBY_1)
			xccr |= ESAI_TCCR_TPSR_BYPASS;
		else if (div == SYSCLK_ESAI_DIV_PSR_DIVBY_8)
			xccr |= ESAI_TCCR_TPSR_DIV8;
		break;
	case SYSCLK_ESAI_DIV_PM:
		xccr &= ESAI_TCCR_TPM_MASK;
		xccr |= ESAI_TCCR_TPM(div);
		break;
	case SYSCLK_ESAI_DIV_FP:
		xccr &= ESAI_TCCR_TFP_MASK;
		xccr |= ESAI_TCCR_TFP(div);
		break;
	default:
		spin_unlock_irqrestore(&esai->lock, flags);
		return -EINVAL;
	}
	writel(xccr, esai->base + offset);

	esai->div[cpu_dai->id] = true;

	spin_unlock_irqrestore(&esai->lock, flags);

	return 0;
}

/*
 * ESAI Network Mode or TDM slots configuration.
 */
static int fsl_esai_set_dai_tdm_slot(struct snd_soc_dai *cpu_dai,
				     unsigned int tx_mask, unsigned int rx_mask,
				     int slots, int slot_width)
{
	struct fsl_esai_private *esai =
	    snd_soc_dai_get_drvdata(cpu_dai);
	u32 tccr, rccr;
	unsigned long flags;

	spin_lock_irqsave(&esai->lock, flags);

	if ((slots > 32) || (slots < 0))
		return -EINVAL;
	if ((slot_width > 32) || (slot_width < 0))
		return -EINVAL;
	esai->slot_width[cpu_dai->id] = slot_width ? slot_width :
					esai->slot_width[cpu_dai->id];
	esai->slots[cpu_dai->id] = slots ? slots : esai->slots[cpu_dai->id];

	if (cpu_dai->id == ID_SYNC_ASYNC_PLAYBACK) {
		esai->tx_mask = tx_mask ? tx_mask : esai->tx_mask;
		tccr = readl(esai->base + ESAI_TCCR);

		tccr &= ESAI_TCCR_TDC_MASK;
		tccr |= ESAI_TCCR_TDC(esai->slots[cpu_dai->id] - 1);

		writel(tccr, esai->base + ESAI_TCCR);
	}
	if (cpu_dai->id == ID_ASYNC_CAPTURE ||
		esai->flags & IMX_ESAI_SYN) {
		esai->rx_mask = rx_mask ? rx_mask : esai->rx_mask;
		rccr = readl(esai->base + ESAI_RCCR);

		rccr &= ESAI_RCCR_RDC_MASK;
		rccr |= ESAI_RCCR_RDC(esai->slots[cpu_dai->id] - 1);

		writel(rccr, esai->base + ESAI_RCCR);
	}

	ESAI_DUMP();
	spin_unlock_irqrestore(&esai->lock, flags);
	return 0;
}

/*
 * ESAI DAI format configuration.
 *
 * Configures the ESAI for the clock and frame sync polarity
 *
 * @cpu_dai: the cpu dai
 * @fmt:     the format specifications
 * @_xcr:    pointer to the ESAI Control register(TCR/RCR)
 * @_xccr:   pointer to the ESAI Clock Control register(TCCR/RCCR)
 *
 * Returns error on invalid/unsupported format and 0 on success
 *
 * Note: the TX register bit positions are used as they are same for both TX
 * and RX
 */
static int fsl_esai_set_dai_format(struct snd_soc_dai *cpu_dai,
		unsigned int fmt, u32 *_xcr, u32 *_xccr)
{
	u32 xcr, xccr;

	xcr = *_xcr;
	xccr = *_xccr;

	/* DAI mode */
	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
		/* negative frame sync,
		 * falling edge to clock data & rising edge to latch */
		xccr |= ESAI_TCCR_TFSP | ESAI_TCCR_TCKP;

		/* word-length frame-sync,
		 * data on rising edge of bclk, frame low 1clk before data */
		xcr &= ~ESAI_TCR_TFSL;
		xcr |= ESAI_TCR_TFSR;
		break;

	case SND_SOC_DAIFMT_LEFT_J:
		/* positive frame sync,
		 * falling edge to clock data & rising edge to latch */
		xccr &= ~ESAI_TCCR_TFSP;
		xccr |= ESAI_TCCR_TCKP;

		/* word-length frame-sync,
		 * data on rising edge of bclk, frame high with data */
		xcr &= ~(ESAI_TCR_TFSL | ESAI_TCR_TFSR);
		break;

	case SND_SOC_DAIFMT_DSP_A:
		/* positive frame sync,
		 * falling edge to clock data & rising edge to latch */
		xccr &= ~ESAI_TCCR_TFSP;
		xccr |= ESAI_TCCR_TCKP;

		/* bit-length frame-sync,
		 * data on rising edge of bclk, frame high 1clk before data */
		xcr |= ESAI_TCR_TFSL | ESAI_TCR_TFSR;
		break;

	case SND_SOC_DAIFMT_DSP_B:
		/* positive frame sync,
		 * falling edge to clock data & rising edge to latch */
		xccr &= ~ESAI_TCCR_TFSP;
		xccr |= ESAI_TCCR_TCKP;

		/* bit-length frame-sync,
		 * data on rising edge of bclk, frame high with data */
		xcr &= ~(ESAI_TCR_TFSL | ESAI_TCR_TFSR);
		break;

	default: /* unsupported format */
		dev_err(cpu_dai->dev, "unsupported format (fmt=%X)\n",
				(fmt & SND_SOC_DAIFMT_FORMAT_MASK));
		return -ENOTSUPP;
	}

	/* DAI clock inversion */
	switch (fmt & SND_SOC_DAIFMT_INV_MASK) {
	case SND_SOC_DAIFMT_NB_NF:
		break;

	case SND_SOC_DAIFMT_NB_IF:
		/* inverted frame-sync */
		xccr ^= ESAI_TCCR_TFSP;
		break;

	case SND_SOC_DAIFMT_IB_NF:
		/* inverted bclk */
		xccr ^= ESAI_TCCR_TCKP;
		break;

	case SND_SOC_DAIFMT_IB_IF:
		/* inverted bclk and frame-sync */
		xccr ^= (ESAI_TCCR_TCKP | ESAI_TCCR_TFSP);
		break;

	default: /* invalid format */
		dev_err(cpu_dai->dev, "invalid clock inv format (fmt=%X)\n",
				fmt & SND_SOC_DAIFMT_INV_MASK);
		return -EINVAL;
	}

	/* DAI clock master masks */
	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFM:
		xccr &= ~(ESAI_TCCR_TFSD | ESAI_TCCR_TCKD);
		break;

	case SND_SOC_DAIFMT_CBS_CFM:
		xccr &= ~ESAI_TCCR_TFSD;
		xccr |= ESAI_TCCR_TCKD;
		break;

	case SND_SOC_DAIFMT_CBM_CFS:
		xccr &= ~ESAI_TCCR_TCKD;
		xccr |= ESAI_TCCR_TFSD;
		break;

	case SND_SOC_DAIFMT_CBS_CFS:
		xccr |= (ESAI_TCCR_TFSD | ESAI_TCCR_TCKD);
		break;

	default: /* invalid format */
		dev_err(cpu_dai->dev, "invalid clock master mask (fmt=%X)\n",
				fmt & SND_SOC_DAIFMT_MASTER_MASK);
		return -EINVAL;
	}

	*_xcr = xcr;
	*_xccr = xccr;

	return 0;
}

/*
 * Configures the ESAI for the clock and frame sync polarity
 *
 * @cpu_dai: the cpu dai
 * @fmt: the format specifications
 *
 * NOTE: This function needs to be called during open
 */
static int fsl_esai_set_dai_fmt(struct snd_soc_dai *cpu_dai,
		unsigned int fmt)
{
	struct fsl_esai_private *esai = snd_soc_dai_get_drvdata(cpu_dai);
	u32 tcr, tccr, rcr, rccr;
	int ret;
	unsigned long flags;

	/* restrict format settings if already set */
	if (cpu_dai->active > 0) {
		if (fmt == esai->fmt[cpu_dai->id])
			return 0;

		dev_err(cpu_dai->dev,
		"sync mode - cannot change format another stream active\n");
		dev_err(cpu_dai->dev,
			"requested format: %x, current format: %x\n",
			fmt, esai->fmt[cpu_dai->id]);
		return -EAGAIN;
	}

	esai->fmt[cpu_dai->id] = fmt;

	spin_lock_irqsave(&esai->lock, flags);

	if (cpu_dai->id == ID_SYNC_ASYNC_PLAYBACK) {
		tcr   = readl(esai->base + ESAI_TCR);
		tccr  = readl(esai->base + ESAI_TCCR);

		/* set dai format settings */

		/* configure TX side */
		ret = fsl_esai_set_dai_format(cpu_dai, fmt, &tcr, &tccr);
		if (ret) {
			spin_unlock_irqrestore(&esai->lock, flags);
			return -EINVAL;
		}

		tcr &= ESAI_TCR_TMOD_MASK;
		/* FIXME: bring back normal mode support
		if (esai_private->flags & IMX_ESAI_NET) {
			tcr |= ESAI_TCR_TMOD_NETWORK;
		} else {
			tcr |= ESAI_TCR_TMOD_NORMAL;
		}
		*/
		tcr |= ESAI_TCR_TMOD_NETWORK;

		writel(tcr, esai->base + ESAI_TCR);
		writel(tccr, esai->base + ESAI_TCCR);
	}
	if (cpu_dai->id == ID_ASYNC_CAPTURE ||
		/* FIXME: do we need it in sync mode? */
		esai->flags & IMX_ESAI_SYN) {
		rcr   = readl(esai->base + ESAI_RCR);
		rccr  = readl(esai->base + ESAI_RCCR);

		/* set dai format settings */

		/* configure RX side */
		ret = fsl_esai_set_dai_format(cpu_dai, fmt, &rcr, &rccr);
		if (ret) {
			spin_unlock_irqrestore(&esai->lock, flags);
			return -EINVAL;
		}

		rcr &= ESAI_RCR_RMOD_MASK;
		/* FIXME: bring back normal mode support
		if (esai_private->flags & IMX_ESAI_NET) {
			rcr |= ESAI_RCR_RMOD_NETWORK;
		} else {
			rcr |= ESAI_RCR_RMOD_NORMAL;
		}
		*/
		rcr |= ESAI_RCR_RMOD_NETWORK;

		writel(rcr, esai->base + ESAI_RCR);
		writel(rccr, esai->base + ESAI_RCCR);
	}

	ESAI_DUMP();
	spin_unlock_irqrestore(&esai->lock, flags);
	return 0;
}

static inline void fsl_esai_rise_xrun(struct snd_pcm_substream *substream)
{
	snd_soc_pcm_stop(substream, SNDRV_PCM_STATE_XRUN);
}

/**
 * fsl_esai_isr: ESAI interrupt handler
 *
 * Although it's possible to use the interrupt handler to send and receive
 * data to/from the ESAI, we use the DMA instead.  Programming is more
 * complicated, but the performance is much better.
 *
 * This interrupt handler is used only to gather statistics.
 *
 * @irq: IRQ of the ESAI device
 * @dev_id: pointer to the esai_private structure for this ESAI device
 */
static irqreturn_t fsl_esai_isr(int irq, void *dev_id)
{
	struct fsl_esai_private *esai = dev_id;
	irqreturn_t ret = IRQ_NONE;
	u32 saisr;

	saisr = readl(esai->base + ESAI_SAISR);

	if (saisr & ESAI_SAISR_TUE) {
		fsl_esai_rise_xrun(esai->playback);

		/* In order to avoid channel shift issue caused by HW,
		 * driver has to write data to all transmitters to
		 * ensure TUE flag is cleared
		 */
		writel(ESAI_XRUN_DATA, esai->base + ESAI_TX0);
		writel(ESAI_XRUN_DATA, esai->base + ESAI_TX1);
		writel(ESAI_XRUN_DATA, esai->base + ESAI_TX2);
		writel(ESAI_XRUN_DATA, esai->base + ESAI_TX3);
		writel(ESAI_XRUN_DATA, esai->base + ESAI_TX4);
		writel(ESAI_XRUN_DATA, esai->base + ESAI_TX5);

		ret = IRQ_HANDLED;
	}

	if (saisr & ESAI_SAISR_ROE) {
		fsl_esai_rise_xrun(esai->capture);

		/*
		 * In order to avoid channel shift issue caused by HW,
		 * driver has to read data from all receivers to
		 * ensure ROE flag is cleared
		 */
		readl(esai->base + ESAI_RX0);
		readl(esai->base + ESAI_RX1);
		readl(esai->base + ESAI_RX2);
		readl(esai->base + ESAI_RX3);

		ret = IRQ_HANDLED;
	}

	return ret;
}

static int esai_get_max_channels_per_line(int slots, unsigned int mask)
{
	int max_ch = -EINVAL;

	if (slots < 32)
		max_ch = hweight32(mask & ((1 << slots) - 1));
	else if (slots == 32)
		max_ch = hweight32(mask);

	return max_ch;
}

static int esai_get_channels_range(int slots, unsigned int mask, int lines,
				struct snd_interval *ch,
				struct snd_interval *req_ch)
{
	int ch_line = esai_get_max_channels_per_line(slots, mask);
	int total_ch = ch_line * hweight32(lines);

	if (ch_line <= 0 || !lines)
		return -EINVAL;

	if (req_ch->max < ch_line)
		ch->max = req_ch->max;
	else
		ch->max = (req_ch->max/ch_line) * ch_line;

	if (req_ch->min < ch_line)
		ch->min = req_ch->min;
	else
		ch->min = ((req_ch->min + ch_line - 1) / ch_line) * ch_line;

	if (ch->min > total_ch)
		ch->min = total_ch;

	if (ch->max > total_ch)
		ch->max = total_ch;

	/*
	 * This can happen if range defined by req_ch->min, req_ch->max
	 * does not contain any valid channel configuration.
	 * Setting ch->min = ch->max will result in empty range during
	 * snd_interval_refine as ch->max < req_ch->min in this case.
	 */
	if (ch->min > ch->max)
		ch->min = ch->max;

	ch->integer = 1;
	return 0;
}


static int esai_tx_channels_constraint_cb(struct snd_pcm_hw_params *params,
					       struct snd_pcm_hw_rule *rule)
{
	int ret;
	struct snd_interval ch;
	struct snd_soc_dai *cpu_dai = rule->private;
	struct fsl_esai_private *esai = snd_soc_dai_get_drvdata(cpu_dai);
	struct snd_interval *req_ch = hw_param_interval(params,
						   SNDRV_PCM_HW_PARAM_CHANNELS);
	snd_interval_any(&ch);

	ret = esai_get_channels_range(esai->slots[cpu_dai->id], esai->tx_mask,
					esai->tx_lines, &ch, req_ch);
	if (ret)
		return ret;
	return snd_interval_refine(req_ch, &ch);
}

static int esai_rx_channels_constraint_cb(struct snd_pcm_hw_params *params,
					       struct snd_pcm_hw_rule *rule)
{
	int ret;
	struct snd_interval ch;
	struct snd_soc_dai *cpu_dai = rule->private;
	struct fsl_esai_private *esai = snd_soc_dai_get_drvdata(cpu_dai);
	struct snd_interval *req_ch = hw_param_interval(params,
						   SNDRV_PCM_HW_PARAM_CHANNELS);
	snd_interval_any(&ch);

	ret = esai_get_channels_range(esai->slots[cpu_dai->id], esai->rx_mask,
					esai->rx_lines, &ch, req_ch);
	if (ret)
		return ret;
	return snd_interval_refine(req_ch, &ch);
}

static int esai_format_constraint_cb(struct snd_pcm_hw_params *params,
					      struct snd_pcm_hw_rule *rule)
{
	int i;
	struct snd_soc_dai *cpu_dai = rule->private;
	struct fsl_esai_private *esai = snd_soc_dai_get_drvdata(cpu_dai);
	struct snd_mask fmt;
	snd_mask_none(&fmt);

	for (i = 0; i < ARRAY_SIZE(esai_supported_formats); i++)
		if (esai->slot_width[cpu_dai->id] >=
				snd_pcm_format_width(esai_supported_formats[i]))
			snd_mask_set(&fmt, esai_supported_formats[i]);

	return snd_mask_refine(hw_param_mask(params, SNDRV_PCM_HW_PARAM_FORMAT),
				&fmt);
}

/**
 * configure_clk_lines: connects/disconnects the GPIO lines as per stream
 *
 * @substream: the stream
 * @flag: connect(flag==ESAI_CONNECT_FLAG) or
 *        disconnect(flag==ESAI_DISCONNECT_FLAG)
 */
static void configure_clk_lines(struct snd_pcm_substream *substream, int flag)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_esai_private *esai = snd_soc_dai_get_drvdata(rtd->cpu_dai);
	u32 gpio_mask, pxrc;

	if ((substream->stream == SNDRV_PCM_STREAM_CAPTURE) &&
	    !(esai->flags & IMX_ESAI_SYN)) {
		gpio_mask = ESAI_GPIO_ESAI_FSR | ESAI_GPIO_ESAI_SCKR;
		/*
		 * only connect HCK pin when HCK is used as sysclk,
		 * but always disconnect HCK pin to avoid any issue
		 */
		if (esai->gpio_hck[rtd->cpu_dai->id] ||
		    flag == ESAI_DISCONNECT_FLAG)
			gpio_mask |= ESAI_GPIO_ESAI_HCKR;
	} else {
		gpio_mask = ESAI_GPIO_ESAI_FST | ESAI_GPIO_ESAI_SCKT;
		/*
		 * only connect HCK pin when HCK is used as sysclk,
		 * but always disconnect HCK pin to avoid any issue
		 */
		if (esai->gpio_hck[rtd->cpu_dai->id] ||
		    flag == ESAI_DISCONNECT_FLAG)
			gpio_mask |= ESAI_GPIO_ESAI_HCKT;
	}

	pxrc = readl(esai->base + ESAI_PRRC);

	if (flag == ESAI_CONNECT_FLAG)
		pxrc |= gpio_mask;
	else
		pxrc &= ~gpio_mask;

	writel(pxrc, esai->base + ESAI_PRRC);
	writel(pxrc, esai->base + ESAI_PCRC);
}

/**
 * configure_sdio_lines: connects/disconnests the SDI/SDO lines as per stream
 *
 * @substream: the stream
 * @flag: connect(flag==ESAI_CONNECT_FLAG) or
 *        disconnect(flag==ESAI_DISCONNECT_FLAG)
 */
static void configure_sdio_lines(struct snd_pcm_substream *substream, int flag)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_esai_private *esai = snd_soc_dai_get_drvdata(rtd->cpu_dai);
	int mask, gpio_mask, sdio_mask;

	gpio_mask = readl(esai->base + ESAI_PRRC);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		sdio_mask = flip_number(esai->tx_lines);
	else
		sdio_mask = esai->rx_lines;

	sdio_mask = sdio_mask << ESAI_SDI_SDO_POS;

	if (flag == ESAI_CONNECT_FLAG) {
		/* hw lines connection mask */
		mask = gpio_mask | sdio_mask;
	} else {
		sdio_mask  = ~sdio_mask;

		/* hw lines disconnection mask */
		mask = gpio_mask & sdio_mask;
	}

	writel(mask, esai->base + ESAI_PRRC);
	writel(mask, esai->base + ESAI_PCRC);
}

/**
 * fsl_esai_startup: create a new substream
 *
 * This is the first function called when a stream is opened.
 *
 * If this is the first stream open, then grab the IRQ and program most of
 * the ESAI registers.
 */
static int fsl_esai_startup(struct snd_pcm_substream *substream,
			    struct snd_soc_dai *dai)
{
	int ret;
	snd_pcm_hw_rule_func_t ch_constraint_cb;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_esai_private *esai =
	    snd_soc_dai_get_drvdata(rtd->cpu_dai);
	u32 tcr, rcr;
	unsigned long flags;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		ch_constraint_cb = esai_tx_channels_constraint_cb;
	else
		ch_constraint_cb = esai_rx_channels_constraint_cb;

	/*
	 * Add constraint callback for channels, evaluated when user
	 * tries to set channels
	 */
	ret = snd_pcm_hw_rule_add(substream->runtime, 0,
				  SNDRV_PCM_HW_PARAM_CHANNELS,
				  ch_constraint_cb, dai,
				  SNDRV_PCM_HW_PARAM_CHANNELS, -1);
	if (ret)
		return ret;

	/*
	 * Add constraint callback for formats, evaluated when user
	 * tries to set format
	 */
	ret = snd_pcm_hw_rule_add(substream->runtime, 0,
				  SNDRV_PCM_HW_PARAM_FORMAT,
				  esai_format_constraint_cb,
				  dai,
				  SNDRV_PCM_HW_PARAM_FORMAT, -1);
	if (ret)
		return ret;

	spin_lock_irqsave(&esai->lock, flags);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		esai->playback = substream;
		tcr = readl(esai->base + ESAI_TCR);
		tcr |= ESAI_TCR_TEIE;
		writel(tcr, esai->base + ESAI_TCR);
	} else {
		esai->capture = substream;
		rcr = readl(esai->base + ESAI_RCR);
		rcr |= ESAI_RCR_REIE;
		writel(rcr, esai->base + ESAI_RCR);
	}

	spin_unlock_irqrestore(&esai->lock, flags);

	snd_soc_dai_set_dma_data(dai, substream,
				 (substream->stream ==
				  SNDRV_PCM_STREAM_PLAYBACK) ?
				 &esai->dma_params_tx : &esai->
				 dma_params_rx);

	ESAI_DUMP();
	return 0;
}

/* returns TSWS field value according to slot and sample sizes */
static int calc_tsws(struct snd_soc_dai *cpu_dai, int slot_width,
		     int sample_size)
{
	int shift = ((sample_size - 8) * 2) & 0x1f;
	int index = (slot_width - 8) / 4 + (sample_size == 24 ? 1 : 0);
	int val;
	/*
	 * Every byte in table contains tsws value.
	 * 0xff means incorrect value (error).
	 * TSWS = tsws[f(slot, sample)] >> g(sample)
	 * g(sample) = shift = (sample - 8) * 2
	 * f(slot, sample) = index = (slot - 8)/4 + 1(if sample == 24)
	 */
	static unsigned tsws[] = {
					0xffffff00,/* slot = 8 */
					0xffff0104,/* slot = 12 */
					0xff020508,/* slot = 16 */
					0x0306090c,/* slot = 20 */
					0x070a0b10,/* slot = 24 */
					0xffffff1e,/* slot = 24 & sample = 24 */
					0x0f121518,/* slot = 32 */
					0xffffff1f,/* slot = 32 & sample = 24 */
				};

	if (slot_width > MAX_SLOT_WIDTH_SIZE) {
		dev_err(cpu_dai->dev, "Invalid slot width (%d > %d)",
					slot_width, MAX_SLOT_WIDTH_SIZE);
		return -EINVAL;
	}
	if (sample_size > MAX_SAMPLE_SIZE) {
		dev_err(cpu_dai->dev, "Invalid sample size (%d > %d)",
					sample_size, MAX_SAMPLE_SIZE);
		return -EINVAL;
	}
	if (sample_size > slot_width) {
		dev_err(cpu_dai->dev, "Sample size cannot be greater than slot width (%d > %d)",
					sample_size, slot_width);
		return -EINVAL;
	}
	val = (tsws[index] >> shift) & 0xff;
	if (val == 0xff) {
		dev_err(cpu_dai->dev, "Incorrect sample size(%d) or slot width(%d)",
					sample_size, slot_width);
		return -EINVAL;
	}

	/* check */
	if (!(slot_width == MAX_SAMPLE_SIZE &&
		sample_size == MAX_SAMPLE_SIZE && val == 0x1e) &&
		!(slot_width == MAX_SLOT_WIDTH_SIZE &&
		sample_size == MAX_SAMPLE_SIZE && val == 0x1f) &&
		(val != ((slot_width - sample_size) | ((sample_size - 8) / 4))))
		dev_warn(cpu_dai->dev, "Calculated xSWS mask doesn't match sample size(%d) & slot width(%d)",
					sample_size, slot_width);

	return val;
}

/*
 * Calculate divs for BCK
 */
#define CLK_DIVS(div_psr, div_fp) {    \
	.max = 256*div_psr*div_fp,     \
	.psr = (div_psr == 1) ? 1 : 8, \
	.fp = div_fp - 1,              \
	.step = div_psr*div_fp }

static struct divs_elem {
	int max;  /* PSR * FP * max PM */
	int psr;  /* PSR */
	int fp;   /* FP */
	int step; /* FP * PSR */
} table[] = {
	CLK_DIVS(1, 1),
	CLK_DIVS(1, 2),
	CLK_DIVS(1, 3),
	CLK_DIVS(1, 4),
	CLK_DIVS(1, 5),
	CLK_DIVS(1, 6),
	CLK_DIVS(1, 7),
	CLK_DIVS(1, 8),
	CLK_DIVS(1, 9),
	CLK_DIVS(1, 10),
	CLK_DIVS(1, 11),
	CLK_DIVS(1, 12),
	CLK_DIVS(1, 13),
	CLK_DIVS(1, 14),
	CLK_DIVS(1, 15),
	CLK_DIVS(1, 16),
	CLK_DIVS(8, 2),
	CLK_DIVS(8, 3),
	CLK_DIVS(8, 4),
	CLK_DIVS(8, 5),
	CLK_DIVS(8, 6),
	CLK_DIVS(8, 7),
	CLK_DIVS(8, 8),
	CLK_DIVS(8, 9),
	CLK_DIVS(8, 10),
	CLK_DIVS(8, 11),
	CLK_DIVS(8, 12),
	CLK_DIVS(8, 13),
	CLK_DIVS(8, 14),
	CLK_DIVS(8, 15),
	CLK_DIVS(8, 16),
};

/**
 * fsl_esai_set_divs: Sets clock dividers
 *
 * Uses the bit_clk value passed to calculate the clock divider values and
 * updates the TCCR/RCCR
 *
 * @cpu_dai: the cpu dai
 * @txccr_addr: the clock control register
 * @value: the divider product
 *
 * This function must be called with the esai->lock spinlock held, currently
 * it is called by fsl_esai_set_bck() which called by
 * fsl_esai_hw_tx_params() and fsl_esai_hw_rx_params()
 * which held esai->lock.
 */
static int fsl_esai_set_divs(struct snd_soc_dai *cpu_dai,
		u32 __iomem *txccr_addr, int value)
{
	int i;
	int index = -1;
	int acc1 = value;
	int acc2 = value;
	int acc = value;
	u32 xccr;
	int div;
	int div_fp;

	for (i = 0; i < sizeof(table)/sizeof(table[0]); i++) {
		dev_dbg(cpu_dai->dev, "divs: %d %d %d %d\n",
				table[i].max, table[i].psr,
				table[i].fp, table[i].step);

		if (value > table[i].max + table[i].step)
			continue;

		acc1 = value % table[i].step;

		if (acc1) {
			dev_warn(cpu_dai->dev,
					"bit rate value (%d) value hits "
					"hardware limitation\n", value);
		}

		acc2 = table[i].step - acc1;
		acc1 = acc1 < acc2 ? acc1 : acc2;

		if (acc1 == 0) {
			index = i;
			acc = 0;
			break;
		}

		if (acc > acc1) {
			index = i;
			acc = acc1;
			dev_info(cpu_dai->dev, "acc = %d\n", acc);
		}
	}

	div_fp = table[i].fp;

	if (index >= 0) {
		i = index;
		div = (value + acc)/table[i].step;
		div = div > 256 ? 255 : div-1;

		if (div == -1)
			div = 0;

		/* Avoid condition PSR=1, FP=0, & PM=0 */
		if ((table[i].psr == 1)
				&& (div_fp == 0) && (div == 0)) {
			div_fp = 1;

			dev_err(cpu_dai->dev,
					"Out of range, setting maximum "
					"supported bit clock\n");
		}
	} else {
		i = i-1;
		div = (value + acc)/table[i].step;
		div = div > 256 ? 255 : div-1;

		dev_err(cpu_dai->dev,
				"Out of range, setting minimum supported bit "
				"clock\n");
	}

	dev_dbg(cpu_dai->dev, "divs: %d =  %d * (%d + 1) * (%d + 1) : %d\n",
			value, (table[i].psr == 1) ? 1 : 8,
			div_fp, div, acc);

	/* Macro for Tx and Rx (PSR, FP & PM ) are same, using only tx macros */
	xccr = readl(txccr_addr);

	xccr &= ESAI_TCCR_TPSR_MASK;
	xccr |= table[i].psr ? ESAI_TCCR_TPSR_BYPASS : ESAI_TCCR_TPSR_DIV8;
	xccr &= ESAI_TCCR_TFP_MASK;
	xccr |= ESAI_TCCR_TFP(div_fp);
	xccr &= ESAI_TCCR_TPM_MASK;
	xccr |= ESAI_TCCR_TPM(div);

	writel(xccr, txccr_addr);

	return 0;
}

/**
 * fsl_esai_set_fpdiv: Sets FP clock divider
 *
 * Uses the bit_clk value passed to calculate the clock divider values and
 * updates the TCCR/RCCR
 *
 * @cpu_dai: the cpu dai
 * @xccr_addr: the clock control register
 * @value: the divider product
 *
 * HCK divs are already configured,
 * only need to set bck div (FP)
 *
 * This function must be called with the esai->lock spinlock held, currently
 * it is called by fsl_esai_set_bck() which called by
 * fsl_esai_hw_tx_params() and fsl_esai_hw_rx_params()
 * which held esai->lock.
 */
static int fsl_esai_set_fpdiv(struct snd_soc_dai *cpu_dai,
		u32 __iomem *xccr_addr, int value)
{
	struct fsl_esai_private *esai =
		    snd_soc_dai_get_drvdata(cpu_dai);
	u32 xccr;
	unsigned int fp = DIV_ROUND_CLOSEST(
			esai->bck_cal_input_freq[cpu_dai->id], value);

	if (fp > ESAI_FP_MAX || fp < ESAI_FP_MIN) {
		dev_err(cpu_dai->dev, "unable to set BCK: out of range\n");
		return -EINVAL;
	}

	xccr = readl(xccr_addr);
	xccr &= ESAI_TCCR_TFP_MASK;
	xccr |= ESAI_TCCR_TFP((fp-1));
	writel(xccr, xccr_addr);

	return 0;

}

/**
 * fsl_esai_set_bck: Sets clock dividers
 *
 * Execute the clock dividers functions based on HCK divs.
 *
 * @cpu_dai: the cpu dai
 * @xccr_addr: the clock control register
 * @bck: the divider product
 *
 * This function must be called with the esai->lock spinlock held, currently
 * it is called by fsl_esai_hw_tx_params() and fsl_esai_hw_rx_params()
 * which held esai->lock.
 */
static int fsl_esai_set_bck(struct snd_soc_dai *cpu_dai,
		u32 __iomem *xccr_addr, int bck) {

	struct fsl_esai_private *esai =
		    snd_soc_dai_get_drvdata(cpu_dai);
	int ret = 0;

	if (esai->div[cpu_dai->id] == true) {
		dev_dbg(cpu_dai->dev, "BCK is already configured by setting dividers manually\n");
		return ret;
	} else if (esai->hck_divs[cpu_dai->id] == true) {
		ret = fsl_esai_set_fpdiv(cpu_dai, xccr_addr,
				bck);
	} else {
		/* Set clock dividers */
		ret = fsl_esai_set_divs(cpu_dai, xccr_addr,
				DIV_ROUND_CLOSEST(
					esai->bck_cal_input_freq[cpu_dai->id],
					2 * bck));
	}
	return ret;
}

/*
 * This function is called to initialize the TX port before enable
 * the tx port.
 */
static int fsl_esai_hw_tx_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *params,
				 struct snd_soc_dai *cpu_dai)
{

	struct fsl_esai_private *esai =
	    snd_soc_dai_get_drvdata(cpu_dai);
	u32 tcr, tfcr;
	unsigned int channels;
	unsigned int sample_size = snd_pcm_format_width(params_format(params));
	int ret;
	unsigned long flags;

	/*
	 * The ESAI interface does not support a 32 bit sample size trully but
	 * the FIFOs are 32 bit wide. So, we truncate the least significant byte
	 * by handling the 32 bit format as a 24 bit one and hence we'll be able
	 * to use it with slight precision loss
	 */
	if (sample_size == 32)
		sample_size = MAX_SAMPLE_SIZE;

	spin_lock_irqsave(&esai->lock, flags);

	tcr = readl(esai->base + ESAI_TCR);
	tfcr = readl(esai->base + ESAI_TFCR);

	tfcr |= ESAI_TFCR_TFR;
	writel(tfcr, esai->base + ESAI_TFCR);
	tfcr &= ~ESAI_TFCR_TFR;
	/* DAI data (word) size */
	tfcr &= ESAI_TFCR_TWA_MASK;
	switch (params_format(params)) {
	case SNDRV_PCM_FORMAT_S16_LE:
		tfcr |= ESAI_WORD_LEN_16;
		tcr |= ESAI_TCR_TSHFD_MSB;
		break;
	case SNDRV_PCM_FORMAT_S20_3LE:
		tfcr |= ESAI_WORD_LEN_20;
		tcr |= ESAI_TCR_TSHFD_MSB;
		break;
	case SNDRV_PCM_FORMAT_S24_LE:
		tfcr |= ESAI_WORD_LEN_24;
		tcr |= ESAI_TCR_TSHFD_MSB;
		break;
	case SNDRV_PCM_FORMAT_S32_LE:
		tfcr |= ESAI_WORD_LEN_32;
		tcr |= ESAI_TCR_TSHFD_MSB;
		break;
	}

	channels = params_channels(params);
	tfcr &= ESAI_TFCR_TE_MASK;
	tfcr |=
	    ESAI_TFCR_TE(esai->tx_lines, channels,
			fsl_esai_active_slots(esai->slots[cpu_dai->id],
						esai->tx_mask));
	/*
	 * if cpu_dai->rate = 0 then allow hw_params
	 * to set dividers.
	 */
	if (cpu_dai->rate == 0) {
		ret = calc_tsws(cpu_dai, esai->slot_width[cpu_dai->id],
				sample_size);
		if (ret < 0) {
			spin_unlock_irqrestore(&esai->lock, flags);
			return ret;
		}
		tcr &= ESAI_TCR_TSWS_MASK;
		tcr |= ret << 10;
		if (esai->slot_width[cpu_dai->id] && esai->slots[cpu_dai->id]) {
			unsigned int rate = params_rate(params);
			unsigned int bck =
				rate * esai->slots[cpu_dai->id] *
					esai->slot_width[cpu_dai->id];
			ret = fsl_esai_set_bck(cpu_dai,
					esai->base+ESAI_TCCR, bck);
			if (ret < 0) {
				spin_unlock_irqrestore(&esai->lock, flags);
				return ret;
			}
		}
	} else {
		if (cpu_dai->rate != params_rate(params))
			dev_warn(cpu_dai->dev, "Warning : Rate can not been set as multiple substreams are active, current rate = %d requested rate = %d\n",
				cpu_dai->rate, params_rate(params));
	}

	tfcr |= ESAI_TFCR_TFWM(esai->tx_watermark);

	/* Left aligned, Zero padding */
	tcr |= ESAI_TCR_PADC;
	/* TDR initialized from the FIFO */
	tfcr |= ESAI_TFCR_TIEN;

	writel(tcr, esai->base + ESAI_TCR);
	writel(tfcr, esai->base + ESAI_TFCR);

	ESAI_DUMP();
	spin_unlock_irqrestore(&esai->lock, flags);
	return 0;
}

/*
 * This function is called to initialize the RX port before enable
 * the rx port.
 */
static int fsl_esai_hw_rx_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *params,
				 struct snd_soc_dai *cpu_dai)
{
	struct fsl_esai_private *esai =
	    snd_soc_dai_get_drvdata(cpu_dai);
	u32 rcr, rfcr, tcr;
	unsigned int channels;
	unsigned int sample_size = snd_pcm_format_width(params_format(params));
	int ret, error;
	unsigned long flags;

	/*
	 * The ESAI interface does not support a 32 bit sample size trully but
	 * the FIFOs are 32 bit wide. So, we truncate the least significant byte
	 * by handling the 32 bit format as a 24 bit one and hence we'll be able
	 * to use it with slight precision loss
	 */
	if (sample_size == 32)
		sample_size = MAX_SAMPLE_SIZE;

	spin_lock_irqsave(&esai->lock, flags);

	rcr = readl(esai->base + ESAI_RCR);
	rfcr = readl(esai->base + ESAI_RFCR);

	rfcr |= ESAI_RFCR_RFR;
	writel(rfcr, esai->base + ESAI_RFCR);
	rfcr &= ~ESAI_RFCR_RFR;

	rfcr &= ESAI_RFCR_RWA_MASK;
	rcr &= ESAI_RCR_RSWS_MASK;
	ret = calc_tsws(cpu_dai, esai->slot_width[cpu_dai->id], sample_size);
	if (ret < 0) {
		spin_unlock_irqrestore(&esai->lock, flags);
		return ret;
	}
	rcr |= ret << 10;
	switch (params_format(params)) {
	case SNDRV_PCM_FORMAT_S16_LE:
		rfcr |= ESAI_WORD_LEN_16;
		rcr |= ESAI_RCR_RSHFD_MSB;
		break;
	case SNDRV_PCM_FORMAT_S20_3LE:
		rfcr |= ESAI_WORD_LEN_20;
		rcr |= ESAI_RCR_RSHFD_MSB;
		break;
	case SNDRV_PCM_FORMAT_S24_LE:
		rfcr |= ESAI_WORD_LEN_24;
		rcr |= ESAI_RCR_RSHFD_MSB;
		break;
	case SNDRV_PCM_FORMAT_S32_LE:
		rfcr |= ESAI_WORD_LEN_32;
		rcr |= ESAI_RCR_RSHFD_MSB;
		break;
	}

	channels = params_channels(params);
	rfcr &= ESAI_RFCR_RE_MASK;
	rfcr |= ESAI_RFCR_RE(esai->rx_lines, channels,
			fsl_esai_active_slots(esai->slots[cpu_dai->id],
						esai->rx_mask));
	rfcr |= ESAI_RFCR_RFWM(esai->rx_watermark);

	writel(rcr, esai->base + ESAI_RCR);
	writel(rfcr, esai->base + ESAI_RFCR);
	/*
	 * if cpu_dai->rate = 0 then allow hw_params
	 * to set dividers.
	 */
	if (cpu_dai->rate == 0) {
		if (esai->slot_width[cpu_dai->id] && esai->slots[cpu_dai->id]) {
			unsigned int rate = params_rate(params);
			unsigned int bck =
				rate * esai->slots[cpu_dai->id] *
					esai->slot_width[cpu_dai->id];
			if (esai->flags & IMX_ESAI_SYN) {
				error = fsl_esai_set_bck(cpu_dai, esai->base +
					ESAI_TCCR, bck);
				if (error < 0) {
					spin_unlock_irqrestore(&esai->lock,
							flags);
					return ret;
				}
				tcr = readl(esai->base + ESAI_TCR);
				tcr &= ESAI_TCR_TSWS_MASK;
				tcr |= ret << 10;
				writel(tcr, esai->base + ESAI_TCR);
			} else {
				error = fsl_esai_set_bck(cpu_dai, esai->base +
					ESAI_RCCR, bck);
				if (error < 0) {
					spin_unlock_irqrestore(&esai->lock,
							flags);
					return error;
				}
			}

		}
	} else {
		if (cpu_dai->rate != params_rate(params))
			dev_warn(cpu_dai->dev, "Warning : Rate can not been set as multiple substreams are active, current rate = %d requested rate = %d\n",
				cpu_dai->rate, params_rate(params));
	}

	ESAI_DUMP();
	spin_unlock_irqrestore(&esai->lock, flags);
	return 0;
}

/**
 * fsl_esai_hw_params - program the sample size
 *
 */
static int fsl_esai_hw_params(struct snd_pcm_substream *substream,
			      struct snd_pcm_hw_params *hw_params,
			      struct snd_soc_dai *cpu_dai)
{
	struct fsl_esai_private *esai =
	    snd_soc_dai_get_drvdata(cpu_dai);
	struct imx_pcm_dma_params *dma_data;
	int ret;
	unsigned long flags;

	spin_lock_irqsave(&esai->lock, flags);

	/* disconnect the clk lines if no stream is using it */
	if (!(esai->flags & IMX_ESAI_SYN) || !esai->clk_lines_flag)
		configure_clk_lines(substream, ESAI_DISCONNECT_FLAG);
	/* disconnect the sdi/sdo hw lines */
	configure_sdio_lines(substream, ESAI_DISCONNECT_FLAG);

	spin_unlock_irqrestore(&esai->lock, flags);

	/* update ideal bit clock in case of asrc */
	ret = update_ideal_clock(esai, substream, hw_params, cpu_dai->id);
	if (ret)
		return ret;

	/* Configure default clk, if not done by GMD*/
	if (esai->sysclk[cpu_dai->id] != true)
		fsl_esai_set_dai_sysclk(cpu_dai,
				SYSCLK_ESAI_EXTAL,
				ESAI_CLK_DEFAULT_RATE,
				SYSCLK_ESAI_DIR_IN);

	/* Tx/Rx config */
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		unsigned int mask;

		dma_data = &esai->dma_params_tx;
		snd_soc_dai_set_dma_data(cpu_dai, substream, dma_data);
		mask = esai->tx_mask;
		ret = fsl_esai_mask_calc(cpu_dai->dev, esai,
					params_channels(hw_params),
					&mask,
					SNDRV_PCM_STREAM_PLAYBACK,
					cpu_dai->id);
		if (ret)
			return ret;

		esai->active_tx_mask = mask;
		return fsl_esai_hw_tx_params(substream, hw_params, cpu_dai);
	} else {
		unsigned int mask;

		dma_data = &esai->dma_params_rx;
		snd_soc_dai_set_dma_data(cpu_dai, substream, dma_data);
		mask = esai->rx_mask;
		ret = fsl_esai_mask_calc(cpu_dai->dev, esai,
					params_channels(hw_params),
					&mask,
					SNDRV_PCM_STREAM_CAPTURE,
					cpu_dai->id);
		if (ret)
			return ret;

		esai->active_rx_mask = mask;
		return fsl_esai_hw_rx_params(substream, hw_params, cpu_dai);
	}

	return 0;
}

static int fsl_esai_start_transmit(struct fsl_esai_private *esai)
{
	struct snd_pcm_substream *substream = esai->playback;
	u32 reg = readl(esai->base + ESAI_TCR);

	if (!substream)
		return ESAI_TRANSMIT_ERROR_STREAM_NULL;

	if (!esai->tx_trigger_enable)
		return ESAI_TRANSMIT_ERROR_TRIGGER_TX_DISABLED;

	/*
	 * fsl_esai_start_transmit will be called only for playback stream,
	 * for both SYNC and ASYNC mode, the correct slots for playback is at
	 * index: SNDRV_PCM_STREAM_PLAYBACK
	 */
	reg |= ESAI_TCR_TE(esai->tx_lines,
			   substream->runtime->channels,
			   fsl_esai_active_slots(
			   esai->slots[ID_SYNC_ASYNC_PLAYBACK],
			   esai->tx_mask));
	writel(reg, esai->base + ESAI_TCR);

	/*
	 * First set TSMB, then set TSMA, as TSMA's bit0 is the trigger
	 */
	writel((esai->active_tx_mask & 0xffff), esai->base + ESAI_TSMA);
	writel(((esai->active_tx_mask >> 16) & 0xffff), esai->base + ESAI_TSMB);

	return ESAI_TRANSMIT_ENABLED;
}

/*
* Callback for our startup timer. This will check
* for data count in FIFO before starting up the
* transfers.
* Do not call hrtimer_cancel for this timer with the esai->lock
* held otherwise the system will deadlock if the timer
* is running.
*/
static enum hrtimer_restart tx_startup_timer_callback(struct hrtimer *timer)
{
	int ret = HRTIMER_RESTART;
	unsigned char tx_fifo_cnt;
	struct fsl_esai_private *esai = container_of(timer,
						     struct fsl_esai_private,
						     tx_startup_timer);

	tx_fifo_cnt = ESAI_TFSR_TFCNT(readl(esai->base + ESAI_TFSR));
	esai->tx_start_retries_cnt++;

	if ((esai->tx_start_retries_cnt >= esai->tx_start_max_retries)
	|| (tx_fifo_cnt >= esai->tx_start_fifo_level)) {
		unsigned long flags_spin;

		/*
		* Notify via dmesg if we couldn't
		* fix the underrun by waiting
		*/
		if (!(tx_fifo_cnt >= esai->tx_start_fifo_level)) {
			dev_warn(&esai->pdev->dev,
				"unable to fill TX fifo : trying to start...");
		}

		spin_lock_irqsave(&esai->lock, flags_spin);
		fsl_esai_start_transmit(esai);
		spin_unlock_irqrestore(&esai->lock, flags_spin);

		ret = HRTIMER_NORESTART;
	} else {
		ktime_t ktime_now =
				  hrtimer_cb_get_time(&esai->tx_startup_timer);
		ktime_t ktime_int = ktime_set(0, esai->tx_start_timeout);
		hrtimer_forward(&esai->tx_startup_timer, ktime_now, ktime_int);
	}

	return ret;
}

/**
 * fsl_esai_trigger: start and stop the DMA transfer.
 *
 * This function is called by ALSA to start, stop, pause, and resume the DMA
 * transfer of data.
 *
 * The DMA channel is in external master start and pause mode, which
 * means the ESAI completely controls the flow of data.
 */
static int fsl_esai_trigger(struct snd_pcm_substream *substream, int cmd,
			    struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_esai_private *esai =
	    snd_soc_dai_get_drvdata(rtd->cpu_dai);
	u32 reg, tfcr = 0, rfcr = 0;
	unsigned long flags;

	spin_lock_irqsave(&esai->lock, flags);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		tfcr = readl(esai->base + ESAI_TFCR);
		reg = readl(esai->base + ESAI_TCR);
	} else {
		rfcr = readl(esai->base + ESAI_RFCR);
		reg = readl(esai->base + ESAI_RCR);
	}

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_RESUME:
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			unsigned char tx_fifo_cnt;
			esai->tx_trigger_enable = true;

			tfcr |= ESAI_TFCR_TFEN;
			writel(tfcr, esai->base + ESAI_TFCR);

			esai->tx_start_retries_cnt = 0;
			/* Check if we need to start our startup timer */
			tx_fifo_cnt = ESAI_TFSR_TFCNT(readl(esai->base +
							    ESAI_TFSR));
			if ((esai->tx_start_max_retries == 0) ||
			    (tx_fifo_cnt >= esai->tx_start_fifo_level)) {
				fsl_esai_start_transmit(esai);
			} else {
				ktime_t ktime_int = ktime_set(0,
							esai->tx_start_timeout);
				hrtimer_start(&esai->tx_startup_timer,
					      ktime_int, HRTIMER_MODE_REL);
			}
		} else {
			rfcr |= ESAI_RFCR_RFEN;
			writel(rfcr, esai->base + ESAI_RFCR);
			reg |=
			    ESAI_RCR_RE(esai->rx_lines,
					substream->runtime->channels,
					fsl_esai_active_slots(
							esai->slots[dai->id],
							esai->rx_mask));
			writel(reg, esai->base + ESAI_RCR);
			/*
			 * First set RSMB, then set RSMA, as RSMA's bit0
			 * is the trigger
			 */
			writel((esai->active_rx_mask & 0xffff),
					esai->base + ESAI_RSMA);
			writel(((esai->active_rx_mask >> 16) & 0xffff),
					esai->base + ESAI_RSMB);
		}
		break;
	case SNDRV_PCM_TRIGGER_SUSPEND:
	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			esai->tx_trigger_enable = false;
			reg &= ~ESAI_TCR_TE_MASK;
			writel(reg, esai->base + ESAI_TCR);

			writel(0x0000, esai->base + ESAI_TSMA);
			writel(0x0000, esai->base + ESAI_TSMB);

			tfcr |= ESAI_TFCR_TFR;
			tfcr &= ~ESAI_TFCR_TFEN;
			writel(tfcr, esai->base + ESAI_TFCR);
			tfcr &= ~ESAI_TFCR_TFR;
			writel(tfcr, esai->base + ESAI_TFCR);
			esai->clk_lines_flag &= ~ESAI_PLAYBACK_USING_GPIO;
		} else {
			reg &= ~ESAI_RCR_RE_MASK;
			writel(reg, esai->base + ESAI_RCR);

			writel(0x0000, esai->base + ESAI_RSMA);
			writel(0x0000, esai->base + ESAI_RSMB);

			rfcr |= ESAI_RFCR_RFR;
			rfcr &= ~ESAI_RFCR_RFEN;
			writel(rfcr, esai->base + ESAI_RFCR);
			rfcr &= ~ESAI_RFCR_RFR;
			writel(rfcr, esai->base + ESAI_RFCR);
			esai->clk_lines_flag &= ~ESAI_CAPTURE_USING_GPIO;
		}

		break;
	default:
		spin_unlock_irqrestore(&esai->lock, flags);
		return -EINVAL;
	}

	ESAI_DUMP();

	spin_unlock_irqrestore(&esai->lock, flags);

	return 0;
}

/**
 * fsl_esai_shutdown: shutdown the ESAI
 *
 * Shutdown the ESAI if there are no other substreams open.
 */
static void fsl_esai_shutdown(struct snd_pcm_substream *substream,
			      struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_esai_private *esai =
	    snd_soc_dai_get_drvdata(rtd->cpu_dai);
	u32 tcr, rcr;
	unsigned long flags;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		hrtimer_cancel(&esai->tx_startup_timer);

	spin_lock_irqsave(&esai->lock, flags);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		esai->playback = NULL;
		esai->tx_mask = ESAI_DEFAULT_MASK;
		tcr = readl(esai->base + ESAI_TCR);
		tcr &= ~ESAI_TCR_TEIE;
		writel(tcr, esai->base + ESAI_TCR);
	} else {
		esai->capture = NULL;
		esai->rx_mask = ESAI_DEFAULT_MASK;
		rcr = readl(esai->base + ESAI_RCR);
		rcr &= ~ESAI_RCR_REIE;
		writel(rcr, esai->base + ESAI_RCR);
	}

	/* disconnect the clk lines if no stream is using it */
	if (!(esai->flags & IMX_ESAI_SYN) || !esai->clk_lines_flag)
		configure_clk_lines(substream, ESAI_DISCONNECT_FLAG);

	/* disconnect the sdi/sdo hw lines */
	configure_sdio_lines(substream, ESAI_DISCONNECT_FLAG);

	spin_unlock_irqrestore(&esai->lock, flags);

	if (!dai->active) {
		/* clear setting to avoid persistence */
		esai->slot_width[dai->id] = ESAI_DEFAULT_SLOT_WIDTH;
		esai->slots[dai->id] = ESAI_DEFAULT_SLOTS;
		esai->div[dai->id] = false;
		esai->sysclk[dai->id] = false;
		esai->hck_divs[dai->id] = false;
	}
}

static int fsl_esai_hw_free(struct snd_pcm_substream *substream,
			     struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_esai_private *esai =
				snd_soc_dai_get_drvdata(rtd->cpu_dai);

	/* invalidate ideal bit clock in case of asrc */
	return invalidate_ideal_clock(esai, substream);
}

static int fsl_esai_prepare(struct snd_pcm_substream *substream,
			    struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_esai_private *esai = snd_soc_dai_get_drvdata(rtd->cpu_dai);
	u32 tfcr, rfcr;
	unsigned long flags;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		hrtimer_cancel(&esai->tx_startup_timer);

	spin_lock_irqsave(&esai->lock, flags);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		tfcr = readl(esai->base + ESAI_TFCR);
		tfcr |= ESAI_TFCR_TFR;
		writel(tfcr, esai->base + ESAI_TFCR);
		tfcr &= ~ESAI_TFCR_TFR;
		writel(tfcr, esai->base + ESAI_TFCR);
		esai->clk_lines_flag |= ESAI_PLAYBACK_USING_GPIO;
	} else {
		rfcr = readl(esai->base + ESAI_RFCR);
		rfcr |= ESAI_RFCR_RFR;
		writel(rfcr, esai->base + ESAI_RFCR);
		rfcr &= ~ESAI_RFCR_RFR;
		writel(rfcr, esai->base + ESAI_RFCR);
		esai->clk_lines_flag |= ESAI_CAPTURE_USING_GPIO;
	}

	/* connect the clk lines */
	configure_clk_lines(substream, ESAI_CONNECT_FLAG);
	/* connect the sdi/sdo hw lines */
	configure_sdio_lines(substream, ESAI_CONNECT_FLAG);

	spin_unlock_irqrestore(&esai->lock, flags);

	return 0;
}

static const struct snd_soc_dai_ops fsl_esai_dai_ops = {
	.startup = fsl_esai_startup,
	.hw_params = fsl_esai_hw_params,
	.shutdown = fsl_esai_shutdown,
	.trigger = fsl_esai_trigger,
	.set_sysclk = fsl_esai_set_dai_sysclk,
	.set_clkdiv = fsl_esai_set_dai_clkdiv,
	.set_fmt = fsl_esai_set_dai_fmt,
	.set_tdm_slot = fsl_esai_set_dai_tdm_slot,
	.hw_free = fsl_esai_hw_free,
	.prepare = fsl_esai_prepare,
};

/* Template for the CPU dai driver structure in sync mode*/
static struct snd_soc_dai_driver fsl_esai_dai_template_sync = {
	.id = ID_SYNC_ASYNC_PLAYBACK,
	.playback = {
		     .channels_min = 1,
		     .channels_max = ESAI_MAX_TX_LINES_COUNT *
							ESAI_MAX_SLOTS_PER_LINE,
		     .rates = IMX_ESAI_RATES,
		     .formats = IMX_ESAI_FORMATS,
		     },
	.capture = {
		    .channels_min = 1,
		    .channels_max = ESAI_MAX_RX_LINES_COUNT *
							ESAI_MAX_SLOTS_PER_LINE,
		    .rates = IMX_ESAI_RATES,
		    .formats = IMX_ESAI_FORMATS,
		    },
	.ops = &fsl_esai_dai_ops,
	.symmetric_rates = 1,
};

/* Template for the CPU dai driver structure in async mode*/
static struct snd_soc_dai_driver fsl_esai_dai_template_async[] = {
	{
		.id = ID_SYNC_ASYNC_PLAYBACK,
		.playback = {
			     .channels_min = 1,
			     .channels_max = ESAI_MAX_TX_LINES_COUNT *
						ESAI_MAX_SLOTS_PER_LINE,
			     .rates = IMX_ESAI_RATES,
			     .formats = IMX_ESAI_FORMATS,
			    },
		.ops = &fsl_esai_dai_ops,
		.symmetric_rates = 0,
	},
	{
		.id =  ID_ASYNC_CAPTURE,
		.capture = {
			    .channels_min = 1,
			    .channels_max = ESAI_MAX_RX_LINES_COUNT *
						ESAI_MAX_SLOTS_PER_LINE,
			    .rates = IMX_ESAI_RATES,
			    .formats = IMX_ESAI_FORMATS,
			   },
		.ops = &fsl_esai_dai_ops,
		.symmetric_rates = 0,
	},
};

#define REG_SHOW(name) \
	do { \
		length += sprintf(buf + length, #name "=%x\n", \
			readl(esai->base + name)); \
	} while (0)

/**
 * fsl_sysfs_esai_show: display ESAI statistics
*
 * Display the statistics for the current ESAI device.  To avoid confusion,
 * we only show those counts that are enabled.
 */
static ssize_t fsl_sysfs_esai_show(struct device *dev,
				   struct device_attribute *attr, char *buf)
{
	struct fsl_esai_private *esai =
	    container_of(attr, struct fsl_esai_private, dev_attr);
	ssize_t length = 0;

	REG_SHOW(ESAI_ECR);
	REG_SHOW(ESAI_ESR);
	REG_SHOW(ESAI_TCR);
	REG_SHOW(ESAI_TCCR);
	REG_SHOW(ESAI_TFCR);
	REG_SHOW(ESAI_TFSR);
	REG_SHOW(ESAI_TSMA);
	REG_SHOW(ESAI_TSMB);
	REG_SHOW(ESAI_RCR);
	REG_SHOW(ESAI_RCCR);
	REG_SHOW(ESAI_RFCR);
	REG_SHOW(ESAI_RFSR);
	REG_SHOW(ESAI_RSMA);
	REG_SHOW(ESAI_RSMB);
	REG_SHOW(ESAI_PRRC);
	REG_SHOW(ESAI_PCRC);

	return length;
}

/**
 * Make every character in a string lower-case
 */
static void make_lowercase(char *s)
{
	char *p = s;
	char c;

	while ((c = *p)) {
		if ((c >= 'A') && (c <= 'Z'))
			*p = c + ('a' - 'A');
		p++;
	}
}

static int init_dev_attr(struct device *dev, struct device_attribute *dev_attr,
			const char *name, umode_t mode, void *show, void *store)
{
	sysfs_attr_init(&dev_attr->attr);
	dev_attr->attr.name = name;
	dev_attr->attr.mode = mode;
	dev_attr->show = show;
	dev_attr->store = store;
	return device_create_file(dev, dev_attr);
}

/**
 * read_line_configuration: Reads and verifies hw line settings
 *
 * Reads hw lines settings from the DT and checks them for correctness
 *
 * Example: if tx-lines=0x30, rx-lines=0x03
 *
 *	|  -  |  -  | SDI2| SDI3|
 *	|  -  |  -  |  -  |  -  | SDO1 | SDO0 |
 *	---------------------------------------
 *	| PC6 | PC7 | PC8 | PC9 | PC10 | PC11 |
 *
 * @np: esai device node
 * @tx_lines: pointer to tx lines mask
 * @rx_lines: pointer to rx lines mask
 */
static int read_line_configuration(struct device_node *np, int *tx_lines,
		int *rx_lines)
{
	int ret;
	int common_lines;
	int _tx_lines;

	if (!np)
		return -EINVAL;

	/* read tx-lines */
	ret = of_property_read_u32(np, "tx-lines", tx_lines);

	if (ret) {
		pr_debug("esai: 'tx-lines' setting absent, enabling all (0x%X)\n",
				ESAI_DEFAULT_TX_LINES);
		*tx_lines = ESAI_DEFAULT_TX_LINES;
	} else {
		if (*tx_lines < 0x0 || *tx_lines > ESAI_MAX_TX_LINES) {
			pr_err("esai: error: invalid value(0x%X) for 'tx-lines'\n",
					*tx_lines);

			return -EINVAL;
		}
	}

	/* read rx-lines */
	ret = of_property_read_u32(np, "rx-lines", rx_lines);

	if (ret) {
		pr_debug("esai: 'rx-lines' setting absent, enabling all (0x%X)\n",
				ESAI_DEFAULT_RX_LINES);
		*rx_lines = ESAI_DEFAULT_RX_LINES;
	} else {
		if (*rx_lines < 0x0 || *rx_lines > ESAI_MAX_RX_LINES) {
			pr_err("esai: error: invalid value(0x%X) for 'rx-lines'\n",
					*rx_lines);

			return -EINVAL;
		}
	}

	/* reverse tx_lines bits */
	_tx_lines = flip_number(*tx_lines);

	/* check if lines are being overlapped */
	common_lines = _tx_lines & *rx_lines;

	if (common_lines) {
		pr_err("esai: error: signal lines are being overlapped\n");
		return -EINVAL;
	}

	return 0;
}

static int fsl_esai_probe(struct platform_device *pdev)
{
	struct fsl_esai_private *esai;
	int ret = 0;
	struct device_node *np = pdev->dev.of_node;
	const char *p, *sprop;
	const uint32_t *iprop;
	struct resource res;
	struct pinctrl *pinctrl;
	u32 dma_events[2];
	char name[64];
	const char *pfdrv_name[ESAI_SUPPORT_DAI_NUM]; /* platform driver name */
	unsigned int pfdrv_num = ESAI_PFDRV_NUM_SYNC;
	int platform_id[ESAI_SUPPORT_DAI_NUM] = {-1, -1};
	u32 ecr;
	const char *attr_name;
	int watermark;
	int dma_brst_size;
	int tx_lines, rx_lines;
	int i = 0;
	bool use_default = true;

	/* ESAIs that are not connected on the board should have a
	 *      status = "disabled"
	 * property in their device tree nodes.
	 */
	if (!of_device_is_available(np))
		return -ENODEV;

	pinctrl = devm_pinctrl_get_select_default(&pdev->dev);
	if (IS_ERR(pinctrl)) {
		dev_err(&pdev->dev, "setup pinctrl failed!");
		return PTR_ERR(pinctrl);
	}

	p = of_get_property(np, "dai_name", NULL);
	/* The DAI name is the last part of the full name of the node. */
	if (!p)
		p = strrchr(np->full_name, '/') + 1;
	esai = kzalloc(sizeof(struct fsl_esai_private) + strlen(p),
			       GFP_KERNEL);
	if (!esai) {
		dev_err(&pdev->dev, "could not allocate DAI object\n");
		return -ENOMEM;
	}

	strcpy(esai->name, p);
		/* Assigning stream names */
		snprintf(esai->esai_playback_stream_name,
					MAX_STREAM_NAME_SIZE - 1,
					"%s_playback", p);
		snprintf(esai->esai_capture_stream_name,
					MAX_STREAM_NAME_SIZE - 1,
					"%s_capture", p);

	/* Get the addresses and IRQ */
	ret = of_address_to_resource(np, 0, &res);
	if (ret) {
		dev_err(&pdev->dev, "could not determine device resources\n");
		goto error_kmalloc;
	}
	esai->base = of_iomap(np, 0);
	if (!esai->base) {
		dev_err(&pdev->dev, "could not map device resources\n");
		ret = -ENOMEM;
		goto error_kmalloc;
	}
	esai->esai_phys = res.start;

	esai->irq = irq_of_parse_and_map(np, 0);
	if (esai->irq == NO_IRQ) {
		dev_err(&pdev->dev, "no irq for node %s\n", np->full_name);
		ret = -ENXIO;
		goto error_iomap;
	}

	/* The 'name' should not have any slashes in it. */
	ret =
	    request_irq(esai->irq, fsl_esai_isr, 0, esai->name,
			esai);
	if (ret < 0) {
		dev_err(&pdev->dev, "could not claim irq %u\n",
			esai->irq);
		goto error_irqmap;
	}

	/* reset ESAI flags */
	esai->flags = 0;

	/* Are the RX and the TX clocks locked? */
	if (!of_find_property(np, "fsl,esai-async", NULL)) {
		esai->flags |= IMX_ESAI_SYN;
		pfdrv_num = ESAI_PFDRV_NUM_SYNC;

		/* Initialize this copy of the CPU DAI driver structure */
		memcpy(&esai->cpu_dai_drv, &fsl_esai_dai_template_sync,
			sizeof(fsl_esai_dai_template_sync));
		esai->cpu_dai_drv[ID_SYNC_ASYNC_PLAYBACK].name = esai->name;
		esai->cpu_dai_drv[ID_SYNC_ASYNC_PLAYBACK].playback.stream_name =
				esai->esai_playback_stream_name;
		esai->cpu_dai_drv[ID_SYNC_ASYNC_PLAYBACK].capture.stream_name =
				esai->esai_capture_stream_name;
	} else {
		pfdrv_num = ESAI_PFDRV_NUM_ASYNC;
		/* Initialize this copy of the CPU DAI drivers structure */
		memcpy(&esai->cpu_dai_drv, &fsl_esai_dai_template_async,
			sizeof(fsl_esai_dai_template_async));

		snprintf(esai->esai_playback_dai_name,
					MAX_DAI_NAME_SIZE - 1,
					"%s-tx", esai->name);
		snprintf(esai->esai_capture_dai_name,
					MAX_DAI_NAME_SIZE - 1,
					"%s-rx", esai->name);
		esai->cpu_dai_drv[ID_SYNC_ASYNC_PLAYBACK].name =
			esai->esai_playback_dai_name;
		esai->cpu_dai_drv[ID_ASYNC_CAPTURE].name =
			esai->esai_capture_dai_name;
		esai->cpu_dai_drv[ID_SYNC_ASYNC_PLAYBACK]
			.playback.stream_name =
			esai->esai_playback_stream_name;
		esai->cpu_dai_drv[ID_ASYNC_CAPTURE]
			.capture.stream_name =
			esai->esai_capture_stream_name;
	}

	ret = of_property_read_u32_array(np, "platform-id", platform_id,
					pfdrv_num);
	if (ret) {
		if (esai->flags & IMX_ESAI_SYN)
			dev_warn(&pdev->dev, "platform-id missing or invalid\n");
		else {
			dev_err(&pdev->dev, "require two platform-id in async mode\n");
			goto error_irqmap;
		}
	}

	/* Determine the FIFO depth. */
	iprop = of_get_property(np, "fsl,fifo-depth", NULL);
	if (iprop)
		esai->fifo_depth = be32_to_cpup(iprop);
	else
		/* Older 8610 DTs didn't have the fifo-depth property */
		esai->fifo_depth = ESAI_DEFAULT_FIFO_DEPTH;

	esai->clk = devm_clk_get(&pdev->dev, "clk_esai");
	if (IS_ERR(esai->clk)) {
		ret = PTR_ERR(esai->clk);
		dev_err(&pdev->dev, "could not get clock: %d\n", ret);
		goto error_irq;
	}
	clk_prepare_enable(esai->clk);

/*
 * Example to Activate FSYS:
 * In DT file "arch/arm/boot/dts/imx6qdl.dtsi" under esai: esai@0202400 node,
 * add clocks and clock-names:
 *
 * clocks = <&clks IMX6Q_CLK_ESAI>, <&clks IMX6Q_CLK_AHB>;
 * clock-names = "clk_esai", "fsys";
 */
	esai->fsysclk = devm_clk_get(&pdev->dev, "fsys");
	if (IS_ERR(esai->fsysclk)) {
		dev_warn(&pdev->dev, "could not get FSYS clock\n");
		esai->fsysclk = NULL;
	} else {
		clk_prepare_enable(esai->fsysclk);
	}

	esai->ipgclk = devm_clk_get(&pdev->dev, "ipg");
	if (IS_ERR(esai->ipgclk)) {
		ret = PTR_ERR(esai->ipgclk);
		dev_err(&pdev->dev, "could not get IPG clock: %d\n", ret);
		goto error_clk;
	}
	clk_prepare_enable(esai->ipgclk);

	/* Read the watermarks from DT */
	iprop = of_get_property(np, "tx-wtrmrk", NULL);
	if (iprop)
		watermark = be32_to_cpup(iprop);
	else
		watermark = ESAI_DEFAULT_WATERMARK_LEVEL;
	if ((watermark < ESAI_MIN_WATERMARK_LEVEL) ||
	    (watermark > esai->fifo_depth)) {
		dev_err(&pdev->dev,
		"mis-configured tx-wtrmrk the valid range is %d to %d\n",
		ESAI_MIN_WATERMARK_LEVEL, esai->fifo_depth);
		goto error_clk_ipg;
	}
	esai->tx_watermark = watermark;

	iprop = of_get_property(np, "rx-wtrmrk", NULL);
	if (iprop)
		watermark = be32_to_cpup(iprop);
	else
		watermark = ESAI_DEFAULT_WATERMARK_LEVEL;
	if ((watermark < ESAI_MIN_WATERMARK_LEVEL) ||
	    (watermark > esai->fifo_depth)) {
		dev_err(&pdev->dev,
		"mis-configured rx-wtrmrk the valid range is %d to %d\n",
		ESAI_MIN_WATERMARK_LEVEL, esai->fifo_depth);
		goto error_clk_ipg;
	}
	esai->rx_watermark = watermark;

	/* Read the burst sizes from DT and check sanity against watermarks */
	iprop = of_get_property(np, "tx-burst", NULL);
	if (iprop)
		dma_brst_size = be32_to_cpup(iprop);
	else
		dma_brst_size = esai->tx_watermark;
	if ((dma_brst_size < ESAI_MIN_SDMA_BURST_SIZE) ||
	    (dma_brst_size > esai->tx_watermark)) {
		dev_err(&pdev->dev,
		"mis-configured tx-burst the valid range is %d to %d\n",
		ESAI_MIN_SDMA_BURST_SIZE,
		esai->tx_watermark);
		goto error_clk_ipg;
	}
	esai->dma_params_tx.burstsize = dma_brst_size;

	iprop = of_get_property(np, "rx-burst", NULL);
	if (iprop)
		dma_brst_size = be32_to_cpup(iprop);
	else
		dma_brst_size = esai->rx_watermark;
	if ((dma_brst_size < ESAI_MIN_SDMA_BURST_SIZE) ||
	    (dma_brst_size > esai->rx_watermark)) {
		dev_err(&pdev->dev,
		"mis-configured rx-burst the valid range is %d to %d\n",
		ESAI_MIN_SDMA_BURST_SIZE,
		esai->rx_watermark);
		goto error_clk_ipg;
	}
	esai->dma_params_rx.burstsize = dma_brst_size;

	/* Assign defaults for startup timer functionality */
	esai->tx_start_max_retries = ESAI_DEFAULT_MAX_STARTUP_RETRIES;
	esai->tx_start_timeout = (ESAI_DEFAULT_STARTUP_TIMER_TIMEOUT_US * 1000);
	esai->tx_start_fifo_level = (esai->fifo_depth - esai->tx_watermark + 1);

	/* Init timer attributes for sysfs */
	attr_name = "tx_start_timeout";
	ret = init_dev_attr(&pdev->dev, &esai->dev_attr_timeout,
			    attr_name, (S_IRUGO | S_IWUSR),
			    tx_start_timeout_read, tx_start_timeout_write);
	if (ret) {
		dev_err(&pdev->dev, "could not create sysfs %s file\n",
			attr_name);
		goto error_clk_ipg;
	}

	attr_name = "tx_start_retries";
	ret = init_dev_attr(&pdev->dev, &esai->dev_attr_retries,
			    attr_name, (S_IRUGO | S_IWUSR),
			    tx_start_retries_read, tx_start_retries_write);
	if (ret) {
		dev_err(&pdev->dev, "could not create sysfs %s file\n",
			attr_name);
		goto error_dev_attr_timeout;
	}

	attr_name = "tx_start_retry_count";
	ret = init_dev_attr(&pdev->dev, &esai->dev_attr_retry_count,
			    attr_name, S_IRUGO,
			    tx_start_retry_count_read, NULL);
	if (ret) {
		dev_err(&pdev->dev, "could not create sysfs %s file\n",
			attr_name);
		goto error_dev_attr_retries;
	}

	attr_name = "tx_start_fifo_level";
	ret = init_dev_attr(&pdev->dev, &esai->dev_attr_fifo_level,
			    attr_name, (S_IRUGO | S_IWUSR),
			    tx_start_fifo_level_read,
			    tx_start_fifo_level_write);
	if (ret) {
		dev_err(&pdev->dev, "could not create sysfs %s file\n",
			attr_name);
		goto error_dev_attr_retry_count;
	}

	esai->dma_params_rx.dma_addr = res.start + ESAI_ERDR;
	esai->dma_params_tx.dma_addr = res.start + ESAI_ETDR;

	esai->dma_params_tx.peripheral_type = IMX_DMATYPE_ESAI;
	esai->dma_params_rx.peripheral_type = IMX_DMATYPE_ESAI;
	/*
	 * TODO: This is a temporary solution and should be changed
	 * to use generic DMA binding later when the helplers get in.
	 */
	ret = of_property_read_u32_array(pdev->dev.of_node,
					 "fsl,esai-dma-events", dma_events, 2);
	if (ret) {
		dev_err(&pdev->dev, "could not get dma events\n");
		goto error_dev_attr_flevel;
	}
	esai->dma_params_tx.dma = dma_events[0];
	esai->dma_params_rx.dma = dma_events[1];

	/* Read the line settings */
	ret =  read_line_configuration(np, &tx_lines, &rx_lines);

	if (!ret) {
		esai->tx_lines = tx_lines;
		esai->rx_lines = rx_lines;
	} else {
		goto error_dev_attr_flevel;
	}
	/* Initialize the the device_attribute structure */
	attr_name = "statistics";
	ret = init_dev_attr(&pdev->dev, &esai->dev_attr,
			    attr_name, S_IRUGO,
			    fsl_sysfs_esai_show, NULL);
	if (ret) {
		dev_err(&pdev->dev, "could not create sysfs %s file\n",
			attr_name);
		goto error_dev_attr_flevel;
	}

	esai->tx_trigger_enable = false;

	/* initialize lock */
	spin_lock_init(&esai->lock);

	/* Initialize our hi-res timer and setup callback */
	hrtimer_init(&esai->tx_startup_timer, CLOCK_MONOTONIC,
		     HRTIMER_MODE_REL);
	esai->tx_startup_timer.function = &tx_startup_timer_callback;

	/* Register with ASoC */
	dev_set_drvdata(&pdev->dev, esai);

	for (i = 0; i < ESAI_SUPPORT_DAI_NUM; i++) {
		/* internal interface default */
		esai->slot_width[i] = ESAI_DEFAULT_SLOT_WIDTH;
		esai->slots[i] = ESAI_DEFAULT_SLOTS;
		esai->div[i] = false;
		esai->hck_divs[i] = false;
		esai->sysclk[i] = false;
	}
#if defined(CONFIG_SND_SOC_IMX_ASRC_DAI_SUPPORT)
	esai->ideal_clk_set[SNDRV_PCM_STREAM_PLAYBACK] =
	esai->ideal_clk_set[SNDRV_PCM_STREAM_CAPTURE] = false;
#endif
	esai->tx_mask = ESAI_DEFAULT_MASK;
	esai->rx_mask = ESAI_DEFAULT_MASK;

	writel(ESAI_ECR_ERST, esai->base + ESAI_ECR);
	writel(ESAI_ECR_ESAIEN, esai->base + ESAI_ECR);
	writel(ESAI_GPIO_RESET_ESAI, esai->base + ESAI_PRRC);
	writel(ESAI_GPIO_RESET_ESAI, esai->base + ESAI_PCRC);
	if (esai->flags & IMX_ESAI_SYN) {
		writel(ESAI_SAICR_SYNC, esai->base + ESAI_SAICR);
		ret = snd_soc_register_dai(&pdev->dev,
			&esai->cpu_dai_drv[ID_SYNC_ASYNC_PLAYBACK]);
	} else {
		writel(0x00, esai->base + ESAI_SAICR);
		ret = snd_soc_register_dais(&pdev->dev, esai->cpu_dai_drv,
						ESAI_SUPPORT_DAI_NUM);
	}
	if (ret) {
		dev_err(&pdev->dev, "failed to register DAI: %d\n", ret);
		goto error_dev_attr_stat;
	}

	/* update asrc reference if asrc core available */
	ret = check_for_asrc(esai, pdev);
	if (ret)
		goto error_dai;

	/* get platform driver name from device-tree */
	if (of_find_property(np, "platform-name", NULL)) {
		if (of_property_count_strings(np, "platform-name") !=
			pfdrv_num) {
			dev_err(&pdev->dev, "%d platform-name(s) are required in %s mode\n",
			pfdrv_num,
			esai->flags & IMX_ESAI_SYN ? "sync" : "async");
			goto error_dai;
		} else {
			for (i = 0; i < pfdrv_num; i++) {
				if (of_property_read_string_index(np,
					"platform-name", i,
					&pfdrv_name[i])) {
					dev_err(&pdev->dev, "failed to retrieve platform-name at index %d",
					i);
					goto error_dai;
				}
			}
			use_default = false;
		}
	} else
		dev_warn(&pdev->dev, "platform driver name not found,"
				"using default");

	if (use_default) {
		for (i = 0; i < pfdrv_num; i++)
			esai->imx_pcm_pdev[i] =
				platform_device_register_simple("imx-pcm-audio",
						platform_id[i], NULL, 0);
	} else {
		for (i = 0; i < pfdrv_num; i++)
			esai->imx_pcm_pdev[i] =
				register_named_platform(esai,
					pdev,
					pfdrv_name[i],
					platform_id[i]);
	}

	for (i = 0; i < pfdrv_num; i++)
		if (IS_ERR(esai->imx_pcm_pdev[i])) {
			ret = PTR_ERR(esai->imx_pcm_pdev[i]);
			goto error_platform;
		}

	/*
	 * If codec-handle property is missing from ESAI node, we assume
	 * that the machine driver uses new binding which does not require
	 * ESAI driver to trigger machine driver's probe.
	 */
	if (!of_get_property(np, "codec-handle", NULL)) {
		esai->new_binding = true;
		/*
		* We require the ESAI platform device handle
		* for future reference in case we are using the
		* new binding mechanism. Beware that in the case
		* where new binding is not used, the platform device name
		* will be different.
		*/
		esai->pdev = pdev;
		goto done;
	}

	/* Trigger the machine driver's probe function.  The platform driver
	 * name of the machine driver is taken from /compatible property of the
	 * device tree.  We also pass the address of the CPU DAI driver
	 * structure.
	 */
	sprop = of_get_property(of_find_node_by_path("/"), "compatible", NULL);
	/* Sometimes the compatible name has a "fsl," prefix, so we strip it. */
	p = strrchr(sprop, ',');
	if (p)
		sprop = p + 1;
	snprintf(name, sizeof(name), "snd-soc-%s", sprop);
	make_lowercase(name);

	esai->pdev =
	    platform_device_register_data(&pdev->dev, name, 0, NULL, 0);
	if (IS_ERR(esai->pdev)) {
		ret = PTR_ERR(esai->pdev);
		dev_err(&pdev->dev, "failed to register platform: %d\n", ret);
		goto error_platform;
	}

done:
	return 0;

error_platform:
	for (i = 0; i < pfdrv_num; i++) {
		if (!IS_ERR(esai->imx_pcm_pdev[i]))
			platform_device_unregister(esai->imx_pcm_pdev[i]);
	}

error_dai:
	snd_soc_unregister_dais(&pdev->dev, pfdrv_num);

error_dev_attr_stat:
	ecr = readl(esai->base + ESAI_ECR);
	ecr &= ~(ESAI_ECR_ERST | ESAI_ECR_ESAIEN);
	writel(ecr, esai->base + ESAI_ECR);
	device_remove_file(&pdev->dev, &esai->dev_attr);
	dev_set_drvdata(&pdev->dev, NULL);

error_dev_attr_flevel:
	device_remove_file(&pdev->dev, &esai->dev_attr_fifo_level);

error_dev_attr_retry_count:
	device_remove_file(&pdev->dev, &esai->dev_attr_retry_count);

error_dev_attr_retries:
	device_remove_file(&pdev->dev, &esai->dev_attr_retries);

error_dev_attr_timeout:
	device_remove_file(&pdev->dev, &esai->dev_attr_timeout);

error_clk_ipg:
	clk_disable_unprepare(esai->ipgclk);

error_clk:
	clk_disable_unprepare(esai->clk);
	if (esai->fsysclk != NULL)
		clk_disable_unprepare(esai->fsysclk);

error_irq:
	free_irq(esai->irq, esai);

error_irqmap:
	irq_dispose_mapping(esai->irq);

error_iomap:
	iounmap(esai->base);

error_kmalloc:
	kfree(esai);

	return ret;
}

static int fsl_esai_remove(struct platform_device *pdev)
{
	struct fsl_esai_private *esai = dev_get_drvdata(&pdev->dev);
	int pfdrv_num = esai->flags & IMX_ESAI_SYN ? ESAI_PFDRV_NUM_SYNC :
						     ESAI_PFDRV_NUM_ASYNC;
	int i;

	if (!esai->new_binding)
		platform_device_unregister(esai->pdev);
	for (i = 0; i < pfdrv_num; i++)
		platform_device_unregister(esai->imx_pcm_pdev[i]);
	clk_disable_unprepare(esai->clk);
	if (esai->fsysclk != NULL)
		clk_disable_unprepare(esai->fsysclk);
	clk_disable_unprepare(esai->ipgclk);
	snd_soc_unregister_dais(&pdev->dev, pfdrv_num);
	device_remove_file(&pdev->dev, &esai->dev_attr);
	device_remove_file(&pdev->dev, &esai->dev_attr_timeout);
	device_remove_file(&pdev->dev, &esai->dev_attr_retries);
	device_remove_file(&pdev->dev, &esai->dev_attr_retry_count);
	device_remove_file(&pdev->dev, &esai->dev_attr_fifo_level);

	free_irq(esai->irq, esai);
	irq_dispose_mapping(esai->irq);
	iounmap(esai->base);

	dev_set_drvdata(&pdev->dev, NULL);
	kfree(esai);

	return 0;
}

static const struct of_device_id fsl_esai_ids[] = {
	{.compatible = "fsl,imx6-esai",},
	{}
};

MODULE_DEVICE_TABLE(of, fsl_esai_ids);

static struct platform_driver fsl_esai_driver = {
	.driver = {
		   .name = "fsl-esai-dai",
		   .owner = THIS_MODULE,
		   .of_match_table = fsl_esai_ids,
		   },
	.probe = fsl_esai_probe,
	.remove = fsl_esai_remove,
};

module_platform_driver(fsl_esai_driver);

MODULE_AUTHOR("Freescale Semiconductor, Inc.");
MODULE_DESCRIPTION("i.MX ASoC ESAI driver");
MODULE_LICENSE("GPL");
