/*
 * Copyright (C) 2013 Mentor Graphics, Inc. All Rights Reserved.
 * Copyright 2007-2010 Freescale Semiconductor, Inc.
 *
 * i.MX6 ASRC ALSA SoC Digital Audio Interface (DAI) driver
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/init.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/slab.h>

#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/mxc_asrc_core_internal.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/initval.h>
#include <sound/soc.h>
#include "fsl_asrc.h"


struct fsl_asrc_cpu_private {
	struct snd_pcm_substream *playback;
	struct snd_pcm_substream *capture;
	struct snd_soc_dai_driver cpu_dai_drv;
	struct device_attribute dev_attr;
	struct platform_device *pdev;
	struct platform_device *fsl_asrc_cpu_pdev;
	int idl_clk_idx[2];
	char asrc_playback_stream_name[MAX_STREAM_NAME_SIZE];
	char asrc_capture_stream_name[MAX_STREAM_NAME_SIZE];
	struct soc_enum ideal_clocks_enum;
	struct fsl_asrc_core_private *asrc_core_private;
	/*
	 * For FE in capture, the ASRC pair section is always OUTPUT and for
	 * playback it is always INPUT
	 */
	unsigned int watermark[2];
	unsigned int burstsize[2];
	/*
	*"name" must be last element for the structure to avoid overwriting,
	*driver allocating extra memory in probe() for it.
	*asrc_private = kzalloc(sizeof(struct fsl_asrc_cpu_private) + strlen(p),
			      GFP_KERNEL);
	*/
	char name[0];
};

#define ASRC_DAI_I2S_RATES (SNDRV_PCM_RATE_8000_192000 |\
		SNDRV_PCM_RATE_CONTINUOUS)
#define PLATFORM_NAME "imx-pcm-audio"

#define ASRC_DAI_MAX_CHANNELS 10
#define ASRC_DAI_MIN_CHANNELS 1

static int asrc_ideal_clk_get(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_cpu_private *private = dev_get_drvdata(cpu_dai->dev);
	int dir = kcontrol->id.subdevice;

	ucontrol->value.enumerated.item[0] =
			private->idl_clk_idx[dir];

	return 0;
}

static int asrc_ideal_clk_put(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_cpu_private *private = dev_get_drvdata(cpu_dai->dev);
	int dir = kcontrol->id.subdevice;

	private->idl_clk_idx[dir] =
			ucontrol->value.enumerated.item[0];

	return 0;
}

/**
 * fsl_asrc_cpu_hw_params - program the sample size
 */
static int fsl_asrc_cpu_hw_params(struct snd_pcm_substream *substream,
		struct snd_pcm_hw_params *hw_params,
		struct snd_soc_dai *cpu_dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	enum asrc_section_direction direction;
	struct section_config config;
	int ret;

	direction = FE_STREAM_DIRECTION(substream->stream);

	/* fill in section config */
	config.rate = params_rate(hw_params);
	config.channels = params_channels(hw_params);
	config.format = params_format(hw_params);

	dev_dbg(cpu_dai->dev, "%s rate:%d channels:%d format:%llu params(%p)\n",
			__func__, config.rate,
			config.channels,
			config.format,
			hw_params);

	ret = fsl_asrc_set_setup(rtd->dai_link->name, direction, &config,
			substream->stream);

	return ret;
}

static int fsl_asrc_cpu_prepare(struct snd_pcm_substream *substream,
		struct snd_soc_dai *cpu_dai)
{
	struct fsl_asrc_cpu_private *private = dev_get_drvdata(cpu_dai->dev);
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	enum asrc_section_direction direction;
	int ret;

	direction = FE_STREAM_DIRECTION(substream->stream);

	ret = fsl_asrc_set_fe_clock_reference(rtd->dai_link->name, direction,
		private->idl_clk_idx[substream->stream], substream->stream);
	if (ret)
		return ret;

	ret = fsl_asrc_prepare(rtd->dai_link->name, direction,
			substream->stream);

	return ret;
}

static int fsl_asrc_cpu_trigger(struct snd_pcm_substream *substream, int cmd,
		struct snd_soc_dai *cpu_dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	enum asrc_section_direction direction;

	direction = FE_STREAM_DIRECTION(substream->stream);

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		fsl_asrc_start_conversion(rtd->dai_link->name, direction,
				substream->stream);
		break;

	case SNDRV_PCM_TRIGGER_STOP:
		fsl_asrc_stop_conversion(rtd->dai_link->name, direction,
				substream->stream);
		break;
	default:
		return -EINVAL;
	}

	return 0;
}
/**
 * fsl_asrc_cpu_shutdown: shutdown the ASRC
 *
 * Shutdown the ASRC if there are no other substreams open.
 */
static void fsl_asrc_cpu_shutdown(struct snd_pcm_substream *substream,
			     struct snd_soc_dai *cpu_dai)
{
	enum asrc_section_direction direction;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;

	dev_dbg(cpu_dai->dev, "%s called\n", __func__);

	direction = FE_STREAM_DIRECTION(substream->stream);

	fsl_asrc_clear_xrun_callback(rtd->dai_link->name,
			direction,
			substream->stream);

	fsl_asrc_cleanup(rtd->dai_link->name, direction, substream->stream);
}

static void fsl_asrc_cpu_xrun(void *arg)
{
	struct snd_pcm_substream *substream = (struct snd_pcm_substream *)arg;

	snd_soc_pcm_stop(substream, SNDRV_PCM_STATE_XRUN);
}

static int fsl_asrc_cpu_startup(struct snd_pcm_substream *substream,
			   struct snd_soc_dai *cpu_dai)
{
	int ret;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_asrc_cpu_private *asrc_private =
		snd_soc_dai_get_drvdata(rtd->cpu_dai);
	enum asrc_section_direction direction;
	struct snd_pcm_hardware *pcm_hw;
	struct imx_pcm_dma_params *dma_params;
	unsigned char hw_period[SECTION_CNT];

	dev_dbg(cpu_dai->dev, "%s called\n", __func__);

	direction = FE_STREAM_DIRECTION(substream->stream);

	/*
	 * For FE in capture, the ASRC pair section is always OUTPUT and for
	 * playback it is always INPUT
	 */
	hw_period[SECTION_INPUT] =
			     asrc_private->watermark[SNDRV_PCM_STREAM_PLAYBACK];
	hw_period[SECTION_OUTPUT] =
			      asrc_private->watermark[SNDRV_PCM_STREAM_CAPTURE];

	ret = fsl_asrc_set_period(rtd->dai_link->name, direction,
			hw_period,
			substream->stream);
	if (ret)
		return ret;

	/* p2p_get_fe_dma_conf : dma_params */
	dma_params = fsl_asrc_get_dma_params(rtd->dai_link->name,
			direction,
			substream->stream);

	dev_dbg(cpu_dai->dev, "%s dma_params: 0x%p\n", __func__, dma_params);

	if (dma_params == NULL)
		return -EINVAL;

	dma_params->burstsize =	asrc_private->burstsize[substream->stream];

	/* get fe capabilities : we will only be using the min/max
	 * channels from the pair capabilities to update our cpu dai's hw
	 * structure */
	pcm_hw = fsl_asrc_get_capabilities(
			rtd->dai_link->name,
			direction, substream->stream);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		cpu_dai->playback_dma_data = dma_params;
		/* update driver hw structure for channels in playback
		 * according to the ASRC pair in use */
		asrc_private->cpu_dai_drv.playback.channels_min =
				pcm_hw->channels_min;
		asrc_private->cpu_dai_drv.playback.channels_max =
				pcm_hw->channels_max;
	} else {
		cpu_dai->capture_dma_data = dma_params;
		/* update driver hw structure for channels in capture
		 * according to the ASRC pair in use */
		asrc_private->cpu_dai_drv.capture.channels_min =
				pcm_hw->channels_min;
		asrc_private->cpu_dai_drv.capture.channels_max =
				pcm_hw->channels_max;
	}

	/* register xrun callback */
	fsl_asrc_set_xrun_callback(rtd->dai_link->name,
			direction,
			substream->stream,
			fsl_asrc_cpu_xrun,
			(void *)substream);

	return 0;
}

static int asrc_watermark_get(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_cpu_private *private = dev_get_drvdata(cpu_dai->dev);
	int dir = kcontrol->id.subdevice;

	ucontrol->value.integer.value[0] = private->watermark[dir];

	return 0;
}

static int asrc_watermark_put(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	int section;
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_cpu_private *private = dev_get_drvdata(cpu_dai->dev);
	int dir = kcontrol->id.subdevice;

	/*
	 * For FE in capture, the ASRC pair section is always OUTPUT and for
	 * playback it is always INPUT
	 */
	section = FE_STREAM_DIRECTION(dir);
	private->watermark[dir] = ucontrol->value.integer.value[0];
	private->burstsize[dir] =
		fsl_asrc_adjust_burst_per_watermark(private->burstsize[dir],
						    private->watermark[dir],
						    section);

	return 0;
}

static int asrc_burstsize_get(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_cpu_private *private = dev_get_drvdata(cpu_dai->dev);
	int dir = kcontrol->id.subdevice;

	ucontrol->value.integer.value[0] = private->burstsize[dir];

	return 0;
}

static int asrc_burstsize_put(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	int section;
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_cpu_private *private = dev_get_drvdata(cpu_dai->dev);
	int dir = kcontrol->id.subdevice;

	/*
	 * For FE in capture, the ASRC pair section is always OUTPUT and for
	 * playback it is always INPUT
	 */
	section = FE_STREAM_DIRECTION(dir);
	private->burstsize[dir] =
	   fsl_asrc_adjust_burst_per_watermark(ucontrol->value.integer.value[0],
						private->watermark[dir],
						section);

	return 0;
}

static const struct snd_kcontrol_new  no_pcm_pf_kcontrols[] = {
	FSL_ASRC_VALUE_INT_DIRECTIONAL("capture.asrc_out_watermark",
			MAX_PERIOD,
			asrc_watermark_get,
			asrc_watermark_put,
			SNDRV_PCM_STREAM_CAPTURE),
	FSL_ASRC_VALUE_INT_DIRECTIONAL("playback.asrc_in_watermark",
			MAX_PERIOD,
			asrc_watermark_get,
			asrc_watermark_put,
			SNDRV_PCM_STREAM_PLAYBACK),
	FSL_ASRC_VALUE_INT_DIRECTIONAL("capture.asrc_out_burstsize",
			MAX_PERIOD,
			asrc_burstsize_get,
			asrc_burstsize_put,
			SNDRV_PCM_STREAM_CAPTURE),
	FSL_ASRC_VALUE_INT_DIRECTIONAL("playback.asrc_in_burstsize",
			MAX_PERIOD,
			asrc_burstsize_get,
			asrc_burstsize_put,
			SNDRV_PCM_STREAM_PLAYBACK),
};

static int fsl_asrc_cpu_test_probe(struct snd_soc_dai *cpu_dai)
{
	struct fsl_asrc_cpu_private *private = dev_get_drvdata(cpu_dai->dev);
	struct snd_kcontrol_new new;
	char name[MAX_STREAM_NAME_SIZE], *ideal_name = "ideal.clk";
	int ret, clk_num, i, j;

	struct snd_soc_card *card = cpu_dai->card;
	struct snd_card *snd_card = card->snd_card;
	const char *prefix = NULL;
	struct snd_soc_pcm_runtime *rtd;

	memset(&private->ideal_clocks_enum, 0, sizeof(struct soc_enum));
	memset(&new, 0, sizeof(struct snd_kcontrol_new));

	private->ideal_clocks_enum.texts = fsl_asrc_fetch_clock_list(
			private->asrc_core_private, &clk_num);
	private->ideal_clocks_enum.max = clk_num;

	/* default sound control clock values */
	private->idl_clk_idx[SNDRV_PCM_STREAM_PLAYBACK] =
			clk_num - TABLE_POSITION_CLK_NONE;
	private->idl_clk_idx[SNDRV_PCM_STREAM_CAPTURE] =
			clk_num - TABLE_POSITION_CLK_ASRCK1;

	new.info = snd_soc_info_enum_double;
	new.iface = SNDRV_CTL_ELEM_IFACE_MIXER;
	new.name = name;

	for (j = 0; j < card->num_rtd; j++) {
		rtd = &card->rtd[j];
		prefix = card->dai_link[j].stream_name;

		if (rtd->cpu_dai == cpu_dai) {
			dev_dbg(cpu_dai->dev, "creating cpu_dai->name:%s snd_kcontrols",
					cpu_dai->name);

			for (i = SNDRV_PCM_STREAM_PLAYBACK;
					i <= SNDRV_PCM_STREAM_LAST; i++) {
				new.private_value =
				(unsigned long)&private->ideal_clocks_enum;
				new.put = asrc_ideal_clk_put;
				new.get = asrc_ideal_clk_get;
				new.subdevice = i;
				if (i == SNDRV_PCM_STREAM_PLAYBACK)
					snprintf(name, MAX_STREAM_NAME_SIZE,
						"playback.%s", ideal_name);
				else
					snprintf(name, MAX_STREAM_NAME_SIZE,
						"capture.%s", ideal_name);
				ret = fsl_asrc_add_controls(snd_card,
						cpu_dai->dev,
						&new, 1,
					prefix, cpu_dai);

				if (ret)
					return ret;
			}

			ret = fsl_asrc_add_controls(snd_card, cpu_dai->dev,
						no_pcm_pf_kcontrols,
						ARRAY_SIZE(no_pcm_pf_kcontrols),
						prefix, cpu_dai);
			if (ret)
				return ret;
		}
	}

	/* default sound control watermark/burstsize values */
	private->watermark[SNDRV_PCM_STREAM_CAPTURE] =
	private->watermark[SNDRV_PCM_STREAM_PLAYBACK] = FSL_ASRC_WATERMARK;
	private->burstsize[SNDRV_PCM_STREAM_CAPTURE] =
			       private->watermark[SNDRV_PCM_STREAM_CAPTURE] + 1;
	private->burstsize[SNDRV_PCM_STREAM_PLAYBACK] = (FIFO_DEPTH -
			     private->watermark[SNDRV_PCM_STREAM_PLAYBACK]) + 1;

	return 0;
}

static const struct snd_soc_dai_ops fsl_asrc_cpu_dai_ops = {
	.startup	= fsl_asrc_cpu_startup,
	.hw_params	= fsl_asrc_cpu_hw_params,
	.shutdown	= fsl_asrc_cpu_shutdown,
	.trigger	= fsl_asrc_cpu_trigger,
	.prepare	= fsl_asrc_cpu_prepare,
};

/* Template for the CPU dai driver structure */
static struct snd_soc_dai_driver fsl_asrc_cpu_dai_template = {
	.playback = {
		.channels_min = ASRC_DAI_MIN_CHANNELS,
		.channels_max = ASRC_DAI_MAX_CHANNELS,
		.rates = ASRC_DAI_I2S_RATES,
		.formats = FSL_ASRC_P2P_FORMATS,
	},
	.capture = {
		.channels_min = ASRC_DAI_MIN_CHANNELS,
		.channels_max = ASRC_DAI_MAX_CHANNELS,
		.rates = ASRC_DAI_I2S_RATES,
		.formats = FSL_ASRC_P2P_FORMATS,
	},
	.ops = &fsl_asrc_cpu_dai_ops,
	.probe = fsl_asrc_cpu_test_probe,
};

static int fsl_asrc_cpu_probe(struct platform_device *pdev)
{
	struct fsl_asrc_cpu_private *asrc_private;
	int ret = 0;
	struct device_node *np = pdev->dev.of_node, *asrc_core_np = NULL;
	struct platform_device *asrc_core_pdev;
	const char *p, *sprop;
	char name[MAX_STREAM_NAME_SIZE];
	int platform_id = -1;

	pr_info("kernel probing asrc DAI\n");

	ret = of_property_read_u32(np, "platform-id", &platform_id);
	if (ret) {
		dev_err(&pdev->dev, "platform-id missing or invalid\n");
		return -EINVAL;
	}

	p = of_get_property(np, "dai_name", NULL);
	if (!p)
		p = strrchr(np->full_name, '/') + 1;
	asrc_private = kzalloc(sizeof(struct fsl_asrc_cpu_private) + strlen(p),
			      GFP_KERNEL);
	if (!asrc_private) {
		dev_err(&pdev->dev, "could not allocate DAI object\n");
		return -ENOMEM;
	}

	strcpy(asrc_private->name, p);
	/* Assigning stream names */
	snprintf(asrc_private->asrc_playback_stream_name,
				MAX_STREAM_NAME_SIZE,
				"%s_playback", p);
	snprintf(asrc_private->asrc_capture_stream_name,
				MAX_STREAM_NAME_SIZE,
				"%s_capture", p);


	/* Initialize this copy of the CPU DAI driver structure */
	memcpy(&asrc_private->cpu_dai_drv, &fsl_asrc_cpu_dai_template,
	       sizeof(fsl_asrc_cpu_dai_template));
	asrc_private->cpu_dai_drv.name = asrc_private->name;
	asrc_private->cpu_dai_drv.id = platform_id;
	asrc_private->cpu_dai_drv.playback.stream_name =
			asrc_private->asrc_playback_stream_name;
	asrc_private->cpu_dai_drv.capture.stream_name =
			asrc_private->asrc_capture_stream_name;

	dev_set_drvdata(&pdev->dev, asrc_private);

	/* get the parent platform device private data handle */
	asrc_core_np = of_parse_phandle(pdev->dev.of_node,
			"asrc-core", 0);
	if (asrc_core_np) {
		/* retrieve asrc_core pdev handle */
		asrc_core_pdev = of_find_device_by_node(asrc_core_np);
		if (!asrc_core_pdev) {
			dev_err(&pdev->dev,
			"failed to find ASRC core platform device\n");
			ret = -EINVAL;
			goto error_dev;
		}

		/* initialize data */
		asrc_private->asrc_core_private =
				platform_get_drvdata(asrc_core_pdev);
		dev_info(&pdev->dev, "mxc asrc core found: %p.\n",
				asrc_private->asrc_core_private);
	}

	/* Register with ASoC */
	ret = snd_soc_register_dais(&pdev->dev, &asrc_private->cpu_dai_drv, 1);
	if (ret) {
		dev_err(&pdev->dev, "failed to register DAI: %d\n", ret);
		goto error_dev;
	}

	asrc_private->fsl_asrc_cpu_pdev =
			platform_device_register_simple(PLATFORM_NAME,
					platform_id, NULL, 0);
	if (IS_ERR(asrc_private->fsl_asrc_cpu_pdev)) {
		ret = PTR_ERR(asrc_private->fsl_asrc_cpu_pdev);
			goto error_dai;
	}

	sprop = of_get_property(of_find_node_by_path("/"), "compatible", NULL);
	/* The compatible name has a "adit," prefix, so we strip it. */
	p = strrchr(sprop, ',');
	if (p)
		sprop = p + 1;
	snprintf(name, sizeof(name), "snd-soc-%s", sprop);

	asrc_private->pdev =
		platform_device_register_data(&pdev->dev, name,
				platform_id, NULL, 0);
	if (IS_ERR(asrc_private->pdev)) {
		ret = PTR_ERR(asrc_private->pdev);
		dev_err(&pdev->dev, "failed to register platform: %d\n", ret);
		goto error_platform;
	}

	return 0;

error_platform:
	platform_device_unregister(asrc_private->fsl_asrc_cpu_pdev);
error_dai:
	snd_soc_unregister_dai(&pdev->dev);
error_dev:
	dev_set_drvdata(&pdev->dev, NULL);
	kfree(asrc_private);
	pr_info("error registering asrc DAI\n");
	return ret;
}

static int fsl_asrc_cpu_remove(struct platform_device *pdev)
{
	struct fsl_asrc_cpu_private *asrc_private = dev_get_drvdata(&pdev->dev);

	platform_device_unregister(asrc_private->pdev);
	platform_device_unregister(asrc_private->fsl_asrc_cpu_pdev);
	snd_soc_unregister_dai(&pdev->dev);

	kfree(asrc_private);
	dev_set_drvdata(&pdev->dev, NULL);

	return 0;
}

static const struct of_device_id fsl_asrc_cpu_ids[] = {
	{ .compatible = "adit,imx6-asrc", },
	{}
};
MODULE_DEVICE_TABLE(of, fsl_asrc_cpu_ids);

static struct platform_driver fsl_asrc_cpu_driver = {
	.driver = {
		.name = "fsl-asrc-dai",
		.owner = THIS_MODULE,
		.of_match_table = fsl_asrc_cpu_ids,
	},
	.probe = fsl_asrc_cpu_probe,
	.remove = fsl_asrc_cpu_remove,
};

module_platform_driver(fsl_asrc_cpu_driver);

MODULE_AUTHOR("Joshua Frkuska <joshua_frkuska@mentor.com>");
MODULE_DESCRIPTION("i.MX6 ASRC DAI ASoC Driver");
MODULE_LICENSE("GPL v2");
