/*
 * Copyright (C) 2013 Mentor Graphics, Inc. All Rights Reserved.
 *
 *  fsl_asrc.c  --  ALSA SoC ASRC P2P Platform Driver
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/slab.h>
#include <sound/pcm.h>
#include "fsl_asrc.h"

static struct fsl_asrc_private p2p_private = {
	.section_list = LIST_HEAD_INIT(p2p_private.section_list),
	.data_lock = __SPIN_LOCK_UNLOCKED(p2p_private.data_lock),
};

static struct fsl_asrc_private *fsl_asrc_private = &p2p_private;

static struct section_table_entry *resolve_section_entry(
		const char const *streamname,
		enum asrc_section_direction direction,
		int stream_direction)
{
	struct section_table_entry *section_entry;
	int ret = -1, strlength;
	const char const *basename;

	strlength = asrc_get_base_streamname(streamname, &basename);

	if (strlength < 0)
		return NULL;

	spin_lock(&fsl_asrc_private->data_lock);
	list_for_each_entry(section_entry,
			&fsl_asrc_private->section_list,
			list_item) {

		if (!strncmp(section_entry->stream_name,
				basename,
				strlen(section_entry->stream_name)) &&
				(section_entry->direction == direction) &&
				(section_entry->stream_direction ==
						stream_direction)) {
			ret = 0;
			break;
		}

	}
	spin_unlock(&fsl_asrc_private->data_lock);

	if (!ret)
		return section_entry;
	else
		return NULL;
}

/* search our private table for a matching section */
static struct asrc_section_params *resolve_section_mapping(
		const char const *streamname,
		enum asrc_section_direction direction,
		int stream_direction)
{
	struct section_table_entry *section_entry;

	section_entry = resolve_section_entry(
			streamname, direction,
			stream_direction);

	if (section_entry)
		return section_entry->params;
	else
		return NULL;
}

static int create_section_mapping(
		const char const *streamname,
		struct asrc_section_params *section_params,
		enum asrc_section_direction direction,
		int stream_direction)
{
	struct section_table_entry *section_entry;
	int strlength;
	const char const *basename = NULL;

	/* check table to see if exists*/
	section_entry = resolve_section_entry(streamname, direction,
			stream_direction);
	strlength = asrc_get_base_streamname(streamname, &basename);

	/* do not create a new mapping if one exists */
	if (section_entry || strlength < 0)
		return -EINVAL;

	/* Allocate the work structure */
	section_entry = kmalloc(sizeof(struct section_table_entry), GFP_ATOMIC);
	if (!section_entry)
		return -ENOMEM;

	/* strcpy streamname */
	section_entry->stream_name = kzalloc(strlength + 1, GFP_ATOMIC);
	if (!section_entry->stream_name) {
		kfree(section_entry);
		return -ENOMEM;
	}

	section_entry->direction = direction;
	section_entry->stream_direction = stream_direction;
	section_entry->ideal_clock_name = NULL;
	strncpy(section_entry->stream_name, basename, strlength);
	section_entry->params = section_params;

	spin_lock(&fsl_asrc_private->data_lock);
	/* update section list */
	list_add_tail(&section_entry->list_item,
			&fsl_asrc_private->section_list);
	spin_unlock(&fsl_asrc_private->data_lock);

	pr_debug(
		"%s section_entry[%s]: stream direction:%s name:'%s'(%p)\n",
		__func__,
		(direction == SECTION_INPUT ? "INPUT" : "OUTPUT"),
		(section_entry->stream_direction == 0 ? "Playback" : "Capture"),
		section_entry->stream_name, section_entry);

	return 0;
}

/* search our private table for a matching section */
static int remove_section_mapping(
		const char const *streamname,
		enum asrc_section_direction direction,
		int stream_direction)
{
	struct section_table_entry *temp, *section_entry;
	int ret = -EINVAL, strlength;
	const char const *basename = NULL;

	strlength = asrc_get_base_streamname(streamname, &basename);
	if (strlength < 0)
		return ret;

	spin_lock(&fsl_asrc_private->data_lock);
	list_for_each_entry_safe(section_entry, temp,
			&fsl_asrc_private->section_list,
			list_item) {

		if (!strncmp(section_entry->stream_name,
				basename,
				strlen(section_entry->stream_name)) &&
				(section_entry->direction == direction) &&
				(section_entry->stream_direction ==
							stream_direction)) {
			/* remove from the list */
			list_del(&section_entry->list_item);
			kfree(section_entry->stream_name);
			kfree(section_entry); /* Free the node */
			ret = 0;
			break;
		}

	}
	spin_unlock(&fsl_asrc_private->data_lock);

	if (!ret)
		pr_debug(
			"%s removing section_entry[%s]:"
			"stream direction:%s name:'%s'(%p)\n",
			__func__,
			(direction == SECTION_INPUT ? "INPUT" : "OUTPUT"),
			(stream_direction == ASRC_PLAYBACK ?
					"Playback" : "Capture"),
			streamname, section_entry);

	return ret;
}

int fsl_asrc_request_pair(
		struct fsl_asrc_core_private *asrc_core_private,
		const char const *streamname,
		enum asrc_section_direction direction,
		unsigned char mask,
		int stream_direction)
{
	int i, err;
	struct asrc_section_params *section_params;
	enum asrc_pair_index pair;

	err = asrc_req_pair(asrc_core_private, mask, &pair);
	if (err < 0)
		return err;

	for (i = 0; i < SECTION_CNT; i++) {
		asrc_get_section_from_pair(asrc_core_private,
						pair,
						&section_params,
						i);
		err = create_section_mapping(
				streamname,
				section_params,
				i,
				stream_direction);

		if (err) {
			while (i > 0) {
				i--;
				remove_section_mapping(
						streamname,
						i,
						stream_direction);
				asrc_get_section_from_pair(
					asrc_core_private,
					pair,
					&section_params,
					i);
				asrc_release_pair(section_params);
			}
			return err;
		}
		asrc_open_section(section_params);
	}

	return 0;
}
EXPORT_SYMBOL_GPL(fsl_asrc_request_pair);

int fsl_asrc_clear_xrun_callback(const char const *streamname,
		enum asrc_section_direction direction, int stream_direction)
{
	struct asrc_section_params *section_params;

	section_params = resolve_section_mapping(
			streamname, direction,
			stream_direction);

	return asrc_clear_xrun_callback(section_params);
}
EXPORT_SYMBOL_GPL(fsl_asrc_clear_xrun_callback);

int fsl_asrc_set_xrun_callback(
	const char const *streamname,
	enum asrc_section_direction direction,
	int stream_direction,
	void (*func)(void *), void *arg)
{
	struct asrc_section_params *section_params;

	section_params = resolve_section_mapping(
			streamname, direction,
			stream_direction);

	return asrc_set_xrun_callback(section_params, func, arg);
}
EXPORT_SYMBOL_GPL(fsl_asrc_set_xrun_callback);

int fsl_asrc_set_period(
		const char const *streamname,
		enum asrc_section_direction direction,
		unsigned char *period,
		int stream_direction)
{
	struct asrc_section_params *section_params;
	int err;

	section_params = resolve_section_mapping(
			streamname, direction, stream_direction);

	if (!section_params)
		return -EINVAL;

	err = asrc_set_period(section_params, period[direction]);
	if (err)
		return -EINVAL;

	return 0;
}
EXPORT_SYMBOL_GPL(fsl_asrc_set_period);

int fsl_asrc_set_filters(
		const char const *streamname,
		enum asrc_section_direction direction,
		enum filter_settings settings,
		int stream_direction)
{
	struct asrc_section_params *section_params;
	int err;

	section_params = resolve_section_mapping(
			streamname, direction,
			stream_direction);

	if (!section_params)
		return -EINVAL;

	err = asrc_set_filter_settings(section_params, settings);

	return err;

}
EXPORT_SYMBOL_GPL(fsl_asrc_set_filters);

/* returns valid snd_pcm_hardware pointer or NULL on failure */
struct snd_pcm_hardware *
fsl_asrc_get_capabilities(const char const *streamname,
		enum asrc_section_direction direction,
		int stream_direction)
{
	struct section_table_entry *section_entry =
			resolve_section_entry(streamname, direction,
					stream_direction);
	struct asrc_section_capabilities capabilities;

	if (section_entry == NULL)
		return NULL;

	section_entry->pcm_hw.buffer_bytes_max = FSL_ASRC_DMABUF_SIZE;
	section_entry->pcm_hw.period_bytes_min = FSL_ASRC_PERIOD_BYTES_MIN;
	section_entry->pcm_hw.period_bytes_max = FSL_ASRC_PERIOD_BYTES_MAX;
	section_entry->pcm_hw.periods_min = FSL_ASRC_PERIOD_MIN;
	section_entry->pcm_hw.periods_max = FSL_ASRC_PERIOD_MAX;

	asrc_get_capabilities(section_entry->params, &capabilities);

	/* fill in snd_pcm_hardware struct */
	section_entry->pcm_hw.info = SNDRV_PCM_INFO_INTERLEAVED |
		SNDRV_PCM_INFO_BLOCK_TRANSFER |
		SNDRV_PCM_INFO_MMAP |
		SNDRV_PCM_INFO_MMAP_VALID;
	section_entry->pcm_hw.formats = capabilities.formats;
	section_entry->pcm_hw.rate_min = capabilities.rate_min;
	section_entry->pcm_hw.rate_max = capabilities.rate_max;
	section_entry->pcm_hw.channels_min = capabilities.channels_min;
	section_entry->pcm_hw.channels_max = capabilities.channels_max;

	section_entry->pcm_hw.buffer_bytes_max = FSL_ASRC_DMABUF_SIZE;
	section_entry->pcm_hw.period_bytes_min = FSL_ASRC_PERIOD_BYTES_MIN;
	section_entry->pcm_hw.period_bytes_max = FSL_ASRC_PERIOD_BYTES_MAX;
	section_entry->pcm_hw.periods_min = FSL_ASRC_PERIOD_MIN;
	section_entry->pcm_hw.periods_max = FSL_ASRC_PERIOD_MAX;
	section_entry->pcm_hw.fifo_size = 0;

	return &section_entry->pcm_hw;
}
EXPORT_SYMBOL_GPL(fsl_asrc_get_capabilities);

/* returns valid imx_pcm_dma_params pointer or NULL on failure */
struct imx_pcm_dma_params *
fsl_asrc_get_dma_params(const char const *streamname,
		enum asrc_section_direction direction,
		int stream_direction)
{
	struct section_table_entry *section_entry =
			resolve_section_entry(streamname, direction,
					stream_direction);

	pr_debug("resolve section entry: (%p)\n", section_entry);

	if (section_entry == NULL)
		return NULL;

	section_entry->dma_params.dma =
			asrc_get_dma_request(section_entry->params);
	section_entry->dma_params.dma_addr =
			asrc_get_fifo_addr(section_entry->params);
	section_entry->dma_params.burstsize =
			asrc_get_period(section_entry->params);
	section_entry->dma_params.peripheral_type = IMX_DMATYPE_ASRC;

	pr_debug("\n\tdma_params(dma): %d\n"
			"\tdma_params(dma_addr): 0x%lx\n"
			"\tdma_params(burstsize): %d\n"
			"\tdma_params(peripheral_type): %d\n",
			section_entry->dma_params.dma,
			section_entry->dma_params.dma_addr,
			section_entry->dma_params.burstsize,
			section_entry->dma_params.peripheral_type);

	return &section_entry->dma_params;
}
EXPORT_SYMBOL_GPL(fsl_asrc_get_dma_params);

/* returns 0 on successful update */
int fsl_asrc_set_setup(
		const char const *streamname,
		enum asrc_section_direction direction,
		struct section_config *config,
		int stream_direction)
{
	struct asrc_section_params *section_params =
			resolve_section_mapping(
					streamname, direction,
					stream_direction);

	if (section_params == NULL)
		return -EINVAL;

	return asrc_setup_section(section_params, config);
}
EXPORT_SYMBOL_GPL(fsl_asrc_set_setup);

int fsl_asrc_set_fe_clock_reference(
		const char const *streamname,
		enum asrc_section_direction direction,
		int ideal_clk_idx,
		int stream_direction)
{
	struct asrc_section_params *section_params;

	section_params = resolve_section_mapping(
				streamname, direction,
				stream_direction);

	if (section_params == NULL)
		return -EINVAL;

	if (asrc_set_clock_reference_by_idx(
			section_params, ideal_clk_idx) < 0) {
		pr_err("Selected ideal clock index[%d] is invalid for %s in %s.\n",
				ideal_clk_idx,
				(stream_direction == ASRC_PLAYBACK ?
				"Playback" : "Capture"), streamname);
		return -EINVAL;
	}

	return 0;
}
EXPORT_SYMBOL_GPL(fsl_asrc_set_fe_clock_reference);

const char const **fsl_asrc_fetch_clock_list(
		struct fsl_asrc_core_private *asrc_core_private, int *clk_cnt)
{
	return asrc_fetch_clock_list(asrc_core_private, clk_cnt);
}
EXPORT_SYMBOL_GPL(fsl_asrc_fetch_clock_list);

int fsl_asrc_set_be_clock_reference(
		const char const *streamname,
		enum asrc_section_direction direction,
		int stream_direction)
{
	char *ideal_clkname = NULL;
	int ret, asrc_dir;
	struct asrc_section_params *section_params;

	asrc_dir = (stream_direction == SNDRV_PCM_STREAM_PLAYBACK) ?
			ASRC_PLAYBACK : ASRC_CAPTURE;

	ret = asrc_get_ideal_clkname_from_stream(streamname, &ideal_clkname,
			asrc_dir);

	if (ret) {
		pr_err("%s failed to get ideal clockname from streamname '%s'.",
				__func__, ideal_clkname);
		return ret;
	}

	section_params = resolve_section_mapping(
				streamname, direction,
				stream_direction);

	if (section_params == NULL) {
		pr_err("%s section map couldn't be resolved for '%s' ideal clockname.",
				__func__, ideal_clkname);
		ret = -EINVAL;
		goto fail_clkname;
	}

	if (asrc_set_clock_reference(section_params, ideal_clkname) < 0) {
		pr_err("'%s' does not exist in ideal clock table.\n",
				ideal_clkname);
		ret = -EINVAL;
	}

fail_clkname:
	kfree(ideal_clkname);
	ideal_clkname = NULL;

	return ret;
}
EXPORT_SYMBOL_GPL(fsl_asrc_set_be_clock_reference);

/* returns 0 on successful update */
int fsl_asrc_prepare(
		const char const *streamname,
		enum asrc_section_direction direction,
		int stream_direction)
{
	struct asrc_section_params *section_params =
			resolve_section_mapping(
					streamname, direction,
					stream_direction);

	if (section_params == NULL)
		return -EINVAL;

	return asrc_prepare(section_params);
}
EXPORT_SYMBOL_GPL(fsl_asrc_prepare);

int fsl_asrc_start_conversion(
		const char const *streamname,
		enum asrc_section_direction direction,
		int stream_direction)
{
	struct asrc_section_params *section_params =
			resolve_section_mapping(
					streamname, direction,
					stream_direction);

	if (section_params == NULL)
		return -EINVAL;

	return asrc_start_conversion(section_params);
}
EXPORT_SYMBOL_GPL(fsl_asrc_start_conversion);

int fsl_asrc_stop_conversion(
		const char const *streamname,
		enum asrc_section_direction direction,
		int stream_direction)
{
	struct asrc_section_params *section_params =
			resolve_section_mapping(
					streamname, direction,
					stream_direction);

	if (section_params == NULL)
		return -EINVAL;

	return asrc_stop_conversion(section_params);
}
EXPORT_SYMBOL_GPL(fsl_asrc_stop_conversion);

int fsl_asrc_cleanup(const char const *streamname,
		enum asrc_section_direction direction,
		int stream_direction)
{
	int ret;
	struct asrc_section_params *section_params =
			resolve_section_mapping(
					streamname, direction,
					stream_direction);

	if (section_params == NULL)
		return -EINVAL;

	/* force ourselves to the setup state to cleanup */
	asrc_setup(section_params);

	ret = asrc_release_pair(section_params);
	if (ret)
		return ret;

	return remove_section_mapping(
			streamname, direction, stream_direction);
}
EXPORT_SYMBOL_GPL(fsl_asrc_cleanup);

/* have to do it here as snd_soc_add_controls is not exported */
int fsl_asrc_add_controls(struct snd_card *card, struct device *dev,
	const struct snd_kcontrol_new *controls, int num_controls,
	const char *prefix, void *data)
{
	int err, i;

	for (i = 0; i < num_controls; i++) {
		const struct snd_kcontrol_new *control = &controls[i];
		err = snd_ctl_add(card, snd_soc_cnew(control, data,
						     control->name, prefix));
		if (err < 0) {
			dev_err(dev, "Failed to add %s: %d\n",
					control->name, err);
			return err;
		}
	}

	return 0;
}
EXPORT_SYMBOL_GPL(fsl_asrc_add_controls);

int fsl_asrc_adjust_burst_per_watermark(int user_burst,
					int user_watermark, int section)
{
	int burst = user_burst;
	if ((section == SECTION_OUTPUT) && (user_burst > (user_watermark + 1)))
		burst = user_watermark + 1;
	else if ((section == SECTION_INPUT) &&
			((user_burst + user_watermark) > (FIFO_DEPTH + 1)))
		burst = (FIFO_DEPTH - user_watermark) + 1;

	return burst;
}
EXPORT_SYMBOL_GPL(fsl_asrc_adjust_burst_per_watermark);

int fsl_asrc_no_pcm_info(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_info *uinfo)
{
	int max = kcontrol->private_value;
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;

	uinfo->count = 1;
	uinfo->value.integer.min = 1;
	uinfo->value.integer.max = max;
	return 0;
}
EXPORT_SYMBOL_GPL(fsl_asrc_no_pcm_info);

static int fsl_asrc_probe(struct platform_device *pdev)
{
	pr_debug("imx asoc asrc base platform probed\n");

	return 0;
}

/**
 * mxc_asrc_remove() - Exit asrc device
 * @pdev	Pointer to the registered platform device
 * @return	Error code indicating success or failure
 */
static int fsl_asrc_remove(struct platform_device *pdev)
{
	pr_debug("imx asoc asrc base platform removed\n");

	return 0;
}


static struct platform_driver fsl_asrc_driver = {
	.driver = {
		.name = "imx6-p2p-asrc",
		.owner = THIS_MODULE,
	},
	.probe = fsl_asrc_probe,
	.remove = fsl_asrc_remove,
};
module_platform_driver(fsl_asrc_driver);

MODULE_AUTHOR("Joshua Frkuska <joshua_frkuska@mentor.com>");
MODULE_DESCRIPTION("i.MX6 ASRC P2P Driver");
MODULE_LICENSE("GPL v2");
