/*
 *  This code provides functions to handle gcc's profiling data format
 *  introduced with gcc 3.4. Future versions of gcc may change the gcov
 *  format (as happened before), so all format-specific information needs
 *  to be kept modular and easily exchangeable.
 *
 *  This file is based on gcc-internal definitions. Functions and data
 *  structures are defined to be compatible with gcc counterparts.
 *  For a better understanding, refer to gcc source: gcc/gcov-io.h.
 *
 *    Copyright Mentor Graphics Corp. 2013
 *    Author(s): Jiada Wang <jiada_wang@mentor.com>
 *
 *    Uses gcc-internal data definitions.
 */

#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/seq_file.h>
#include <linux/vmalloc.h>
#include "gcov_4_7.h"

/*
 * Determine whether a counter is active. Based on gcc magic. Doesn't change
 * at run-time.
 */
static int counter_active(struct gcov_info_47 *info, unsigned int type)
{
	return info->merge[type] != NULL;
}

/* Determine number of active counters. Based on gcc magic. */
static unsigned int num_counter_active(struct gcov_info_47 *info)
{
	unsigned int i;
	unsigned int result = 0;

	for (i = 0; i < GCOV_COUNTERS; i++) {
		if (counter_active(info, i))
			result++;
	}
	return result;
}

/**
 * gcov_info_reset_47 - reset profiling data to zero
 * @info: profiling data set
 */
void gcov_info_reset_47(struct gcov_info *info)
{
	struct gcov_info_47 *ginfo = (struct gcov_info_47 *)info;
	unsigned int active = num_counter_active(ginfo);
	unsigned int i, j;

	for (i = 0; i < ginfo->n_functions; i++) {
		for (j = 0; j < active; j++) {
			memset(ginfo->functions[i]->ctrs[j].values, 0,
			       ginfo->functions[i]->ctrs[j].num *
			       sizeof(gcov_type));
		}
	}
}

/**
 * gcov_info_is_compatible_47 - check if profiling data can be added
 * @info1: first profiling data set
 * @info2: second profiling data set
 *
 * Returns non-zero if profiling data can be added, zero otherwise.
 */
int gcov_info_is_compatible_47(struct gcov_info *info1, struct gcov_info *info2)
{
	return (info1->stamp == info2->stamp);
}

/**
 * gcov_info_add_47 - add up profiling data
 * @dest: profiling data set to which data is added
 * @source: profiling data set which is added
 *
 * Adds profiling counts of @source to @dest.
 */
void gcov_info_add_47(struct gcov_info *dest, struct gcov_info *source)
{
	struct gcov_info_47 *gdest = (struct gcov_info_47 *)dest;
	struct gcov_info_47 *gsource = (struct gcov_info_47 *)source;
	unsigned int i, j, k;

	for (i = 0; i < gdest->n_functions; i++) {
		for (j = 0; j < num_counter_active(gdest); j++) {
			for (k = 0; k < gdest->functions[i]->ctrs[j].num; k++) {
				gdest->functions[i]->ctrs[j].values[k] +=
				gsource->functions[i]->ctrs[j].values[k];
			}
		}
	}
}

/* Get size of function info entry. Based on gcc magic. */
static size_t get_fn_size(struct gcov_info_47 *info)
{
	size_t size;

	size = sizeof(struct gcov_fn_info) + num_counter_active(info) *
	       sizeof(struct gcov_ctr_info);
	if (__alignof__(struct gcov_fn_info) > sizeof(struct gcov_ctr_info))
		size = ALIGN(size, __alignof__(struct gcov_fn_info));
	return size;
}

/* Get address of function info entry. Based on gcc magic. */
static struct gcov_fn_info *get_fn_info(struct gcov_info_47 *info,
					   unsigned int fn)
{
	return (struct gcov_fn_info *)
		((char *) info->functions[fn]);
}

/**
 * gcov_info_dup_47 - duplicate profiling data set
 * @info: profiling data set to duplicate
 *
 * Return newly allocated duplicate on success, %NULL on error.
 */
struct gcov_info *gcov_info_dup_47(struct gcov_info *info)
{
	struct gcov_info_47 *ginfo = (struct gcov_info_47 *)info;
	struct gcov_info_47 *dup;
	unsigned int i, j;
	unsigned int active;
	size_t fn_size = 0;

	/* Duplicate gcov_info. */
	active = num_counter_active(ginfo);
	dup = kzalloc(sizeof(struct gcov_info_47), GFP_KERNEL);
	if (!dup)
		return NULL;
	dup->version		= ginfo->version;
	dup->stamp		= ginfo->stamp;
	dup->n_functions	= ginfo->n_functions;
	/* Duplicate filename. */
	dup->filename		= kstrdup(ginfo->filename, GFP_KERNEL);
	if (!dup->filename)
		goto err_free;

	for (i = 0; i < GCOV_COUNTERS; i++)
		dup->merge[i] = ginfo->merge[i];

	/* Duplicate table of functions. */
	fn_size = get_fn_size(ginfo);
	dup->functions = vmalloc(ginfo->n_functions *
				 sizeof(struct gcov_fn_info *));
	if (!dup->functions)
		goto err_free;

	for (i = 0; i < ginfo->n_functions; i++) {
		struct gcov_fn_info *function = ginfo->functions[i];
		dup->functions[i] = vmalloc(fn_size);
		if (!dup->functions[i])
			goto err_free;

		dup->functions[i]->key = dup;
		dup->functions[i]->ident = function->ident;
		dup->functions[i]->lineno_checksum = function->lineno_checksum;
		dup->functions[i]->cfg_checksum = function->cfg_checksum;

		for (j = 0; j < active; j++) {
			size_t size = function->ctrs[j].num * sizeof(gcov_type);
			dup->functions[i]->ctrs[j].values = vmalloc(size);
			if (!dup->functions[i]->ctrs[j].values)
				goto err_free;

			memcpy(dup->functions[i]->ctrs[j].values,
			       function->ctrs[j].values, size);
			dup->functions[i]->ctrs[j].num = function->ctrs[j].num;
		}
	}

	return (struct gcov_info *)dup;

err_free:
	gcov_info_free_47((struct gcov_info *)dup);
	return NULL;
}

/**
 * gcov_info_free_47 - release memory for profiling data set duplicate
 * @info: profiling data set duplicate to free
 */
void gcov_info_free_47(struct gcov_info *info)
{
	struct gcov_info_47 *ginfo = (struct gcov_info_47 *)info;
	unsigned int active = num_counter_active(ginfo);
	unsigned int i, j;

	for (i = 0; i < ginfo->n_functions; i++) {
		struct gcov_fn_info *function = get_fn_info(ginfo, i);

		for (j = 0; j < active ; j++)
			vfree(function->ctrs[j].values);

		vfree(function);
	}

	vfree(ginfo->functions);
	kfree(ginfo->filename);
	kfree(ginfo);
}

static struct gcov_fn_info *get_func(struct gcov_iterator *iter)
{
	return get_fn_info((struct gcov_info_47 *)iter->info, iter->function);
}

static struct type_info *get_type(struct gcov_iterator *iter)
{
	return &iter->type_info[iter->type];
}

/**
 * gcov_iter_new_47 - allocate and initialize profiling data iterator
 * @info: profiling data set to be iterated
 *
 * Return file iterator on success, %NULL otherwise.
 */
struct gcov_iterator *gcov_iter_new_47(struct gcov_info *info)
{
	struct gcov_info_47 *ginfo = (struct gcov_info_47 *)info;
	struct gcov_iterator *iter;

	iter = kzalloc(sizeof(struct gcov_iterator) +
		       num_counter_active(ginfo) * sizeof(struct type_info),
		       GFP_KERNEL);
	if (iter)
		iter->info = info;

	return iter;
}

/**
 * gcov_iter_free_47 - release memory for iterator
 * @iter: file iterator to free
 */
void gcov_iter_free_47(struct gcov_iterator *iter)
{
	kfree(iter);
}

/**
 * gcov_iter_get_info_47 - return profiling data set for given file iterator
 * @iter: file iterator
 */
struct gcov_info *gcov_iter_get_info_47(struct gcov_iterator *iter)
{
	return iter->info;
}

/**
 * gcov_iter_start_47 - reset file iterator to starting position
 * @iter: file iterator
 */
void gcov_iter_start_47(struct gcov_iterator *iter)
{
	int i;

	iter->record = 0;
	iter->function = 0;
	iter->type = 0;
	iter->count = 0;
	iter->num_types = 0;
	for (i = 0; i < GCOV_COUNTERS; i++) {
		if (counter_active((struct gcov_info_47 *)iter->info, i)) {
			iter->type_info[iter->num_types].ctr_type = i;
			iter->type_info[iter->num_types++].offset = 0;
		}
	}
}

/* Mapping of logical record number to actual file content. */
#define RECORD_FILE_MAGIC		0
#define RECORD_GCOV_VERSION		1
#define RECORD_TIME_STAMP		2
#define RECORD_FUNCTION_TAG		3
#define RECORD_FUNCTON_TAG_LEN		4
#define RECORD_FUNCTION_IDENT		5
#define RECORD_FUNCTION_CHECK_LINE	6
#define RECORD_FUNCTION_CHECK_CFG	7
#define RECORD_COUNT_TAG		8
#define RECORD_COUNT_LEN		9
#define RECORD_COUNT			10

/**
 * gcov_iter_next_47 - advance file iterator to next logical record
 * @iter: file iterator
 *
 * Return zero if new position is valid, non-zero if iterator has reached end.
 */
int gcov_iter_next_47(struct gcov_iterator *iter)
{
	switch (iter->record) {
	case RECORD_FILE_MAGIC:
	case RECORD_GCOV_VERSION:
	case RECORD_FUNCTION_TAG:
	case RECORD_FUNCTON_TAG_LEN:
	case RECORD_FUNCTION_IDENT:
	case RECORD_FUNCTION_CHECK_LINE:
	case RECORD_COUNT_TAG:
		/* Advance to next record */
		iter->record++;
		break;
	case RECORD_COUNT:
		/* Advance to next count */
		iter->count++;
		/* fall through */
	case RECORD_COUNT_LEN:
		if (iter->count < get_func(iter)->ctrs[iter->type].num) {
			iter->record = 10;
			break;
		}
		/* Advance to next counter type */

		get_type(iter)->offset += iter->count;
		iter->count = 0;
		iter->type++;
		/* fall through */
	case RECORD_FUNCTION_CHECK_CFG:
		if (iter->type < iter->num_types) {
			iter->record = 8;
			break;
		}
		/* Advance to next function */
		iter->type = 0;
		iter->function++;
		/* fall through */
	case RECORD_TIME_STAMP:
		if (iter->function <
		    ((struct gcov_info_47 *)iter->info)->n_functions)
			iter->record = 3;
		else
			iter->record = -1;
		break;
	}
	/* Check for EOF. */
	if (iter->record == -1)
		return -EINVAL;
	else
		return 0;
}

/**
 * seq_write_gcov_u32 - write 32 bit number in gcov format to seq_file
 * @seq: seq_file handle
 * @v: value to be stored
 *
 * Number format defined by gcc: numbers are recorded in the 32 bit
 * unsigned binary form of the endianness of the machine generating the
 * file.
 */
static int seq_write_gcov_u32(struct seq_file *seq, u32 v)
{
	return seq_write(seq, &v, sizeof(v));
}

/**
 * seq_write_gcov_u64 - write 64 bit number in gcov format to seq_file
 * @seq: seq_file handle
 * @v: value to be stored
 *
 * Number format defined by gcc: numbers are recorded in the 32 bit
 * unsigned binary form of the endianness of the machine generating the
 * file. 64 bit numbers are stored as two 32 bit numbers, the low part
 * first.
 */
static int seq_write_gcov_u64(struct seq_file *seq, u64 v)
{
	u32 data[2];

	data[0] = (v & 0xffffffffUL);
	data[1] = (v >> 32);
	return seq_write(seq, data, sizeof(data));
}

/**
 * gcov_iter_write_47 - write data for current pos to seq_file
 * @iter: file iterator
 * @seq: seq_file handle
 *
 * Return zero on success, non-zero otherwise.
 */
int gcov_iter_write_47(struct gcov_iterator *iter, struct seq_file *seq)
{
	int rc = -EINVAL;

	switch (iter->record) {
	case RECORD_FILE_MAGIC:
		rc = seq_write_gcov_u32(seq, GCOV_DATA_MAGIC);
		break;
	case RECORD_GCOV_VERSION:
		rc = seq_write_gcov_u32(seq, iter->info->version);
		break;
	case RECORD_TIME_STAMP:
		rc = seq_write_gcov_u32(seq, iter->info->stamp);
		break;
	case RECORD_FUNCTION_TAG:
		rc = seq_write_gcov_u32(seq, GCOV_TAG_FUNCTION);
		break;
	case RECORD_FUNCTON_TAG_LEN:
		rc = seq_write_gcov_u32(seq, 3);
		break;
	case RECORD_FUNCTION_IDENT:
		rc = seq_write_gcov_u32(seq, get_func(iter)->ident);
		break;
	case RECORD_FUNCTION_CHECK_LINE:
		rc = seq_write_gcov_u32(seq, get_func(iter)->lineno_checksum);
		break;
	case RECORD_FUNCTION_CHECK_CFG:
		rc = seq_write_gcov_u32(seq, get_func(iter)->cfg_checksum);
		break;
	case RECORD_COUNT_TAG:
		rc = seq_write_gcov_u32(seq,
			GCOV_TAG_FOR_COUNTER(get_type(iter)->ctr_type));
		break;
	case RECORD_COUNT_LEN:
		rc = seq_write_gcov_u32(seq,
				get_func(iter)->ctrs[iter->type].num * 2);
		break;
	case RECORD_COUNT:
		rc = seq_write_gcov_u64(seq,
			get_func(iter)->ctrs[iter->type].values[iter->count]);
		break;
	}
	return rc;
}
