/*
 * Copyright 2005-2009 Freescale Semiconductor, Inc.
 *
 * The code contained herein is licensed under the GNU Lesser General
 * Public License.  You may obtain a copy of the GNU Lesser General
 * Public License Version 2.1 or later at the following locations:
 *
 * http://www.opensource.org/licenses/lgpl-license.html
 * http://www.gnu.org/copyleft/lgpl.html
 */

#ifndef __IMX_IPU_H__
#define __IMX_IPU_H__

#include <linux/types.h>
#include <linux/videodev2.h>
#include <linux/bitmap.h>
#include <linux/fb.h>
#include <linux/videodev2.h>

struct ipu_soc;

enum ipuv3_type {
	IPUV3EX,
	IPUV3M,
	IPUV3H,
	IPUV3HDL,
};

struct ipuv3_platform_data {
	void (*reset) (int ipu_id);
	void (*pg) (int); /* power-gate for IPU, not currently used */
};

/*
 * Bitfield of Display Interface signal polarities.
 */
struct ipu_di_signal_cfg {
	unsigned datamask_en:1;
	unsigned interlaced:1;
	unsigned odd_field_first:1;
	unsigned clksel_en:1;
	unsigned clkidle_en:1;
	unsigned data_pol:1;	/* true = inverted */
	unsigned clk_pol:1;	/* true = rising edge */
	unsigned enable_pol:1;
	unsigned hsync_pol:1;	/* true = active high */
	unsigned vsync_pol:1;

	u16 width;
	u16 height;
	u32 pixel_fmt;
	u16 h_back_porch;
	u16 h_sync_len;
	u16 h_front_porch;
	u16 v_back_porch;
	u16 v_sync_len;
	u16 v_front_porch;
	u32 v_to_h_sync;
	unsigned long pixelclock;
#define IPU_DI_CLKMODE_SYNC	(1 << 0)
#define IPU_DI_CLKMODE_EXT	(1 << 1)
	unsigned long clkflags;
};

/*
 * Bitfield of CSI signal polarities and modes.
 */
struct ipu_csi_signal_cfg {
	unsigned data_width:4;
	unsigned clk_mode:3;
	unsigned ext_vsync:1;
	unsigned vsync_pol:1;
	unsigned hsync_pol:1;
	unsigned pixclk_pol:1;
	unsigned data_pol:1;
	unsigned sens_clksrc:1;
	unsigned pack_tight:1;
	unsigned force_eof:1;
	unsigned data_en_pol:1;

	unsigned data_fmt;
	unsigned mipi_dt;
};

/*
 * Enumeration of CSI data bus widths.
 */
enum ipu_csi_data_width {
	IPU_CSI_DATA_WIDTH_4   = 0,
	IPU_CSI_DATA_WIDTH_8   = 1,
	IPU_CSI_DATA_WIDTH_10  = 3,
	IPU_CSI_DATA_WIDTH_12  = 5,
	IPU_CSI_DATA_WIDTH_16  = 9,
};

/*
 * Enumeration of CSI clock modes.
 */
enum ipu_csi_clk_mode {
	IPU_CSI_CLK_MODE_GATED_CLK,
	IPU_CSI_CLK_MODE_NONGATED_CLK,
	IPU_CSI_CLK_MODE_CCIR656_PROGRESSIVE,
	IPU_CSI_CLK_MODE_CCIR656_INTERLACED,
	IPU_CSI_CLK_MODE_CCIR1120_PROGRESSIVE_DDR,
	IPU_CSI_CLK_MODE_CCIR1120_PROGRESSIVE_SDR,
	IPU_CSI_CLK_MODE_CCIR1120_INTERLACED_DDR,
	IPU_CSI_CLK_MODE_CCIR1120_INTERLACED_SDR,
};

/*
 * Enumeration of CSI destinations
 */
enum ipu_csi_dest {
	IPU_CSI_DEST_IDMAC, /* to memory via SMFC */
	IPU_CSI_DEST_IC,    /* to Image Converter */
	IPU_CSI_DEST_VDIC,  /* to VDIC */
};

enum ipu_color_space {
	IPUV3_COLORSPACE_RGB,
	IPUV3_COLORSPACE_YUV,
	IPUV3_COLORSPACE_UNKNOWN,
};

/*
 * Enumeration of IPU rotation modes
 */
enum ipu_rotate_mode {
	/* Note the enum values correspond to BAM value */
	IPU_ROTATE_NONE = 0,
	IPU_ROTATE_VERT_FLIP = 1,
	IPU_ROTATE_HORIZ_FLIP = 2,
	IPU_ROTATE_180 = 3,
	IPU_ROTATE_90_RIGHT = 4,
	IPU_ROTATE_90_RIGHT_VFLIP = 5,
	IPU_ROTATE_90_RIGHT_HFLIP = 6,
	IPU_ROTATE_90_LEFT = 7,
};

/*
 * Enumeration of VDI MOTION select
 */
enum ipu_motion_sel {
	MOTION_NONE = 0,
	LOW_MOTION,
	MED_MOTION,
	HIGH_MOTION,
};

struct ipuv3_channel;

enum ipu_channel_irq {
	IPU_IRQ_EOF = 0,
	IPU_IRQ_NFACK = 64,
	IPU_IRQ_NFB4EOF = 128,
	IPU_IRQ_EOS = 192,
	IPU_IRQ_EOBND = 320,
	IPU_IRQ_THRESH = 384,
};

enum ipu_err_irq {
	IPU_IRQ_VDI_FIFO1_OVF = 256,
	IPU_IRQ_IC_BAYER_BUF_OVF = 282,
	IPU_IRQ_IC_ENC_BUF_OVF = 283,
	IPU_IRQ_VF_BUF_OVF = 284,
	IPU_IRQ_CSI0_PUPE = 286,
	IPU_IRQ_CSI1_PUPE = 287,
	IPU_IRQ_SMFC0_FRM_LOST = 288,
	IPU_IRQ_SMFC1_FRM_LOST = 289,
	IPU_IRQ_SMFC2_FRM_LOST = 290,
	IPU_IRQ_SMFC3_FRM_LOST = 291,
	IPU_IRQ_DC_TEARING_ERR_1 = 304,
	IPU_IRQ_DC_TEARING_ERR_2 = 305,
	IPU_IRQ_DC_TEARING_ERR_6 = 306,
	IPU_IRQ_DI0_SYNC_DISP_ERR = 307,
	IPU_IRQ_DI1_SYNC_DISP_ERR = 308,
	IPU_IRQ_DI0_TIMEOUT_ERR = 309,
	IPU_IRQ_DI1_TIMEOUT_ERR = 310,
	IPU_IRQ_VF_FRM_LOST = 312,
	IPU_IRQ_ENC_FRM_LOST = 313,
	IPU_IRQ_BAYER_FRM_LOST = 314,
	IPU_IRQ_PRIV_ACCESS_ERR = 316,
	IPU_IRQ_AXIW_ERR = 317,
	IPU_IRQ_AXIR_ERR = 318,
};

enum ipu_gp_irq {
	IPU_IRQ_SNOOPING_1 = 448,
	IPU_IRQ_SNOOPING_2,
	IPU_IRQ_DP_SF_START,
	IPU_IRQ_DP_SF_END,
	IPU_IRQ_DP_ASF_START,
	IPU_IRQ_DP_ASF_END,
	IPU_IRQ_DP_SF_BRAKE,
	IPU_IRQ_DP_ASF_BRAKE,
	IPU_IRQ_DC_FC_0,
	IPU_IRQ_DC_FC_1,
	IPU_IRQ_DC_FC_2,
	IPU_IRQ_DC_FC_3,
	IPU_IRQ_DC_FC_4,
	IPU_IRQ_DC_FC_6,
	IPU_IRQ_DI_VSYNC_PRE_0,
	IPU_IRQ_DI_VSYNC_PRE_1,
	IPU_IRQ_DC_DP_START,
	IPU_IRQ_DC_ASYNC_STOP,
	IPU_IRQ_DI0_CNT_PRE_0,
	IPU_IRQ_DI0_CNT_PRE_1,
	IPU_IRQ_DI0_CNT_PRE_2,
	IPU_IRQ_DI0_CNT_PRE_3,
	IPU_IRQ_DI0_CNT_PRE_4,
	IPU_IRQ_DI0_CNT_PRE_5,
	IPU_IRQ_DI0_CNT_PRE_6,
	IPU_IRQ_DI0_CNT_PRE_7,
	IPU_IRQ_DI0_CNT_PRE_8,
	IPU_IRQ_DI0_CNT_PRE_9,
	IPU_IRQ_DI0_CNT_PRE_10,
	IPU_IRQ_DI1_DISP_CLK_PRE,
	IPU_IRQ_DI1_CNT_PRE_3,
	IPU_IRQ_DI1_CNT_PRE_8,
};

int ipu_idmac_channel_irq(struct ipu_soc *ipu, struct ipuv3_channel *channel,
		enum ipu_channel_irq irq);
int ipu_error_irq(struct ipu_soc *ipu, enum ipu_err_irq err_irq_type);
int ipu_gp_irq(struct ipu_soc *ipu, enum ipu_gp_irq gp_irq_type);

/*
 * IPU Common functions
 */
int ipu_get_num(struct ipu_soc *ipu);
enum ipu_color_space ipu_pixelformat_to_colorspace(u32 pixelformat);
enum ipu_color_space ipu_mbus_code_to_colorspace(u32 mbus_code);
u32 ipu_mbus_code_to_fourcc(u32 mbus_code);
bool ipu_pixelformat_is_planar(u32 pixelformat);
int ipu_bits_per_pixel(u32 pixelformat);
int ipu_stride_to_bytes(u32 pixel_stride, u32 pixelformat);
int ipu_degrees_to_rot_mode(enum ipu_rotate_mode *mode, int degrees,
			    bool hflip, bool vflip);
int ipu_rot_mode_to_degrees(int *degrees, enum ipu_rotate_mode mode,
			    bool hflip, bool vflip);
bool ipu_pixelformat_has_alpha(u32 pixelformat);
void ipu_dump(struct ipu_soc *ipu);

int ipu_link_prp_enc_rot_enc(struct ipu_soc *ipu);
int ipu_link_prpvf_rot_prpvf(struct ipu_soc *ipu);
int ipu_link_pp_rot_pp(struct ipu_soc *ipu);
int ipu_link_csi_prpvf(struct ipu_soc *ipu, int csi_id);
int ipu_unlink_prp_enc_rot_enc(struct ipu_soc *ipu);
int ipu_unlink_prpvf_rot_prpvf(struct ipu_soc *ipu);
int ipu_unlink_pp_rot_pp(struct ipu_soc *ipu);
int ipu_unlink_csi_prpvf(struct ipu_soc *ipu, int csi_id);

/*
 * IPU Image DMA Controller (idmac) functions
 */

/*
 * Enumeration of IDMAC channels
 */
#define IPUV3_CHANNEL_CSI0			 0
#define IPUV3_CHANNEL_CSI1			 1
#define IPUV3_CHANNEL_CSI2			 2
#define IPUV3_CHANNEL_CSI3			 3
#define IPUV3_CHANNEL_VDI_MEM_IC_VF              5
#define IPUV3_CHANNEL_MEM_VDI_P                  8
#define IPUV3_CHANNEL_MEM_VDI                    9
#define IPUV3_CHANNEL_MEM_VDI_N                 10
#define IPUV3_CHANNEL_MEM_IC_PP                 11
#define IPUV3_CHANNEL_MEM_IC_PRP_VF             12
#define IPUV3_CHANNEL_VDI_MEM                   13
#define IPUV3_CHANNEL_G_MEM_IC_PRP_VF           14
#define IPUV3_CHANNEL_G_MEM_IC_PP               15
#define IPUV3_CHANNEL_IC_PRP_ENC_MEM            20
#define IPUV3_CHANNEL_IC_PRP_VF_MEM             21
#define IPUV3_CHANNEL_IC_PP_MEM                 22
#define IPUV3_CHANNEL_MEM_BG_SYNC		23
#define IPUV3_CHANNEL_MEM_BG_ASYNC		24
#define IPUV3_CHANNEL_MEM_FG_SYNC		27
#define IPUV3_CHANNEL_MEM_DC_SYNC		28
#define IPUV3_CHANNEL_MEM_FG_ASYNC		29
#define IPUV3_CHANNEL_MEM_FG_SYNC_ALPHA	        31
#define IPUV3_CHANNEL_MEM_DC_ASYNC		41
#define IPUV3_CHANNEL_MEM_ROT_ENC		45
#define IPUV3_CHANNEL_MEM_ROT_VF		46
#define IPUV3_CHANNEL_MEM_ROT_PP		47
#define IPUV3_CHANNEL_ROT_ENC_MEM	        48
#define IPUV3_CHANNEL_ROT_VF_MEM		49
#define IPUV3_CHANNEL_ROT_PP_MEM		50
#define IPUV3_CHANNEL_MEM_BG_SYNC_ALPHA	        51

struct ipuv3_channel *ipu_idmac_get(struct ipu_soc *ipu, unsigned channel,
				    bool allow_wait);
void ipu_idmac_put(struct ipuv3_channel *);
int ipu_idmac_get_num(struct ipuv3_channel *);
bool ipu_idmac_inuse(struct ipuv3_channel *);

int ipu_idmac_enable_channel(struct ipuv3_channel *channel);
int ipu_idmac_disable_channel(struct ipuv3_channel *channel);

void ipu_idmac_set_double_buffer(struct ipuv3_channel *channel,
				 bool doublebuffer);
void ipu_idmac_set_triple_buffer(struct ipuv3_channel *channel,
				 bool triplebuffer);
bool ipu_idmac_buffer_is_ready(struct ipuv3_channel *channel, u32 buf_num);
int ipu_idmac_current_buffer(struct ipuv3_channel *channel);
void ipu_idmac_reset_current_buffer(struct ipuv3_channel *channel);
void ipu_idmac_clear_buffer_ready(struct ipuv3_channel *channel, u32 buf_num);
void ipu_idmac_select_buffer(struct ipuv3_channel *channel, u32 buf_num);
void ipu_idmac_select_multi_vdi_buffer(struct ipu_soc *ipu, u32 buf_num);
void ipu_idmac_enable_watermark(struct ipuv3_channel *channel, bool enable);
int __ipu_idmac_lock_enable(struct ipuv3_channel *channel, int num_bursts);
int ipu_idmac_lock_enable(struct ipuv3_channel *channel, int num_bursts);

/*
 * IPU Camera Sensor Interface (csi) functions
 */
struct ipu_csi;
int ipu_csi_set_src(struct ipu_csi *csi, u32 vc, bool select_mipi_csi2);
int ipu_csi_mbus_fmt_to_sig_cfg(struct ipu_csi_signal_cfg *cfg,
				u32 mbus_code);
int ipu_csi_init_interface(struct ipu_csi *csi, u16 width, u16 height,
			   struct ipu_csi_signal_cfg *cfg);
int ipu_csi_get_sensor_protocol(struct ipu_csi *csi);
bool ipu_csi_is_interlaced(struct ipu_csi *csi);
void ipu_csi_get_window_size(struct ipu_csi *csi, u32 *width, u32 *height);
void ipu_csi_set_window_size(struct ipu_csi *csi, u32 width, u32 height);
void ipu_csi_set_window_pos(struct ipu_csi *csi, u32 left, u32 top);
void ipu_csi_horizontal_downsize_enable(struct ipu_csi *csi);
void ipu_csi_horizontal_downsize_disable(struct ipu_csi *csi);
void ipu_csi_vertical_downsize_enable(struct ipu_csi *csi);
void ipu_csi_vertical_downsize_disable(struct ipu_csi *csi);
void ipu_csi_set_test_generator(struct ipu_csi *csi, bool active,
				u32 r_value, u32 g_value, u32 b_value,
				u32 pix_clk);
int ipu_csi_set_mipi_datatype(struct ipu_csi *csi, u32 vc,
			      struct ipu_csi_signal_cfg *cfg);
int ipu_csi_set_skip_isp(struct ipu_csi *csi, u32 skip, u32 max_ratio);
int ipu_csi_set_skip_smfc(struct ipu_csi *csi, u32 skip,
			  u32 max_ratio, u32 id);
int ipu_csi_set_dest(struct ipu_csi *csi, enum ipu_csi_dest csi_dest);
int ipu_csi_enable(struct ipu_csi *csi);
int ipu_csi_disable(struct ipu_csi *csi);
int ipu_csi_get_num(struct ipu_csi *csi);
struct ipu_csi *ipu_csi_get(struct ipu_soc *ipu, int id);
void ipu_csi_put(struct ipu_csi *csi);
void ipu_csi_dump(struct ipu_csi *csi);

/*
 * IPU Sensor Multi-Fifo Controller (smfc) functions
 */
struct ipu_smfc;
void ipu_smfc_map(struct ipu_smfc *smfc, struct ipuv3_channel *channel,
		  int csi_id, int mipi_vc);
void ipu_smfc_set_wmc(struct ipu_smfc *smfc,
		      struct ipuv3_channel *channel,
		      bool set, u32 level);
void ipu_smfc_set_burst_size(struct ipu_smfc *smfc,
			     struct ipuv3_channel *channel,
			     u32 burst_size, bool passthrough_16);
int ipu_smfc_enable(struct ipu_smfc *smfc);
int ipu_smfc_disable(struct ipu_smfc *smfc);
struct ipu_smfc *ipu_smfc_get(struct ipu_soc *ipu);
void ipu_smfc_put(struct ipu_smfc *smfc);
void ipu_smfc_dump(struct ipu_smfc *smfc);

/*
 * IPU Image Converter (ic) functions
 */
enum ipu_ic_task {
	IC_TASK_ENCODER,
	IC_TASK_VIEWFINDER,
	IC_TASK_POST_PROCESSOR,
	IC_NUM_TASKS,
};

struct ipu_ic;
int ipu_ic_task_init(struct ipu_ic *ic,
		     int in_width, int in_height,
		     int out_width, int out_height,
		     enum ipu_color_space in_cs,
		     enum ipu_color_space out_cs);
int ipu_ic_task_graphics_init(struct ipu_ic *ic,
			      enum ipu_color_space in_g_cs,
			      bool galpha_en, u32 galpha,
			      bool colorkey_en, u32 colorkey);
void ipu_ic_task_enable(struct ipu_ic *ic);
void ipu_ic_task_disable(struct ipu_ic *ic);
int ipu_ic_task_idma_init(struct ipu_ic *ic, struct ipuv3_channel *channel,
			  u32 width, u32 height, int burst_size,
			  enum ipu_rotate_mode rot);
int ipu_ic_set_src(struct ipu_ic *ic, int csi_id, bool vdi);
int ipu_ic_enable(struct ipu_ic *ic);
int ipu_ic_disable(struct ipu_ic *ic);
struct ipu_ic *ipu_ic_get(struct ipu_soc *ipu, enum ipu_ic_task task);
void ipu_ic_put(struct ipu_ic *ic);
void ipu_ic_dump(struct ipu_ic *ic);

/*
 * IPU Image Rotator (irt) functions
 */
struct ipu_irt;
int ipu_irt_enable(struct ipu_irt *irt);
int ipu_irt_disable(struct ipu_irt *irt);
struct ipu_irt *ipu_irt_get(struct ipu_soc *ipu);
void ipu_irt_put(struct ipu_irt *irt);

/*
 * IPU Video De-Interlacer (vdi) functions
 */
struct ipu_vdi;
void ipu_vdi_set_top_field_man(struct ipu_vdi *vdi, bool top_field_0);
void ipu_vdi_set_motion(struct ipu_vdi *vdi, enum ipu_motion_sel motion_sel);
void ipu_vdi_setup(struct ipu_vdi *vdi,
		   u32 code, int xres, int yres, u32 field,
		   enum ipu_motion_sel motion_sel);
void ipu_vdi_unsetup(struct ipu_vdi *vdi);
void ipu_vdi_toggle_top_field_man(struct ipu_vdi *vdi);
int ipu_vdi_set_src(struct ipu_vdi *vdi, bool csi);
int ipu_vdi_enable(struct ipu_vdi *vdi);
int ipu_vdi_disable(struct ipu_vdi *vdi);
struct ipu_vdi *ipu_vdi_get(struct ipu_soc *ipu);
void ipu_vdi_put(struct ipu_vdi *vdi);

/*
 * IPU Display Controller (dc) functions
 */
#define IPU_DC_CHANNEL_DP_SYNC       5 /* sync main+aux planes */
#define IPU_DC_CHANNEL_SYNC          1 /* sync single plane (no DP) */
#define IPU_DC_CHANNEL_DP_ASYNC      6 /* async main+aux planes */
#define IPU_DC_CHANNEL_ASYNC         2 /* async single plane (no DP) */
#define IPU_DC_CHANNEL_READ          0 /* DC read channel */
#define IPU_DC_NUM_CHANNELS         10

struct ipu_dc;
struct ipu_di;
struct ipu_dc *ipu_dc_get(struct ipu_soc *ipu, int channel);
void ipu_dc_put(struct ipu_dc *dc);
int ipu_dc_init_sync(struct ipu_dc *dc, struct ipu_di *di, bool interlaced,
		     u32 pixel_fmt, u32 *pixel_map, u32 width);
void ipu_dc_init_async(struct ipu_dc *dc, struct ipu_di *di, bool interlaced);
void ipu_dc_uninit(struct ipu_dc *dc);
void ipu_dc_enable_channel(struct ipu_dc *dc);
void ipu_dc_disable_channel(struct ipu_dc *dc);
int ipu_dc_get_num(struct ipu_dc *dc);
void ipu_dc_enable(struct ipu_dc *dc);
void ipu_dc_disable(struct ipu_dc *dc);
/*
 * IPU Display Interface (di) functions
 */
struct ipu_di *ipu_di_get(struct ipu_soc *ipu, int disp);
void ipu_di_put(struct ipu_di *);
int ipu_di_disable(struct ipu_di *);
int ipu_di_disable_clock(struct ipu_di *di);
int ipu_di_enable(struct ipu_di *);
int ipu_di_enable_clock(struct ipu_di *di);
int ipu_di_get_num(struct ipu_di *);
int ipu_di_init_sync_panel(struct ipu_di *, struct ipu_di_signal_cfg *sig);
void ipu_di_uninit_sync_panel(struct ipu_di *di);

/*
 * IPU Display Multi FIFO Controller (dmfc) functions
 */
struct dmfc_channel;
int ipu_dmfc_enable_channel(struct dmfc_channel *dmfc);
void ipu_dmfc_disable_channel(struct dmfc_channel *dmfc);
int ipu_dmfc_alloc_bandwidth(struct dmfc_channel *dmfc,
		unsigned long bandwidth, int width, int burstsize);
void ipu_dmfc_free_bandwidth(struct dmfc_channel *dmfc);
struct dmfc_channel *ipu_dmfc_get(struct ipu_soc *ipu, int ipuv3_channel);
void ipu_dmfc_put(struct dmfc_channel *dmfc);

/*
 * IPU Display Processor (dp) functions
 */
#define IPU_DP_FLOW_SYNC_BG	0
#define IPU_DP_FLOW_SYNC_FG	1
#define IPU_DP_FLOW_ASYNC0_BG	2
#define IPU_DP_FLOW_ASYNC0_FG	3
#define IPU_DP_FLOW_ASYNC1_BG	4
#define IPU_DP_FLOW_ASYNC1_FG	5

struct ipu_dp *ipu_dp_get(struct ipu_soc *ipu, unsigned int flow);
void ipu_dp_put(struct ipu_dp *);
int ipu_dp_enable_channel(struct ipu_dp *dp);
void ipu_dp_disable_channel(struct ipu_dp *dp);
void ipu_dp_enable(struct ipu_dp *dp);
void ipu_dp_disable(struct ipu_dp *dp);
int ipu_dp_setup_channel(struct ipu_dp *dp,
		enum ipu_color_space in, enum ipu_color_space out);
void ipu_dp_uninit_channel(struct ipu_dp *dp);
int ipu_dp_set_window_pos(struct ipu_dp *, u16 x_pos, u16 y_pos);
int ipu_dp_set_global_alpha(struct ipu_dp *dp, bool enable, u8 alpha,
		bool bg_chan);
int ipu_dp_set_chroma_key(struct ipu_dp *dp, bool enable, u32 color_key);
int ipu_dp_set_gamma_correction(struct ipu_dp *dp, bool enable,
				u32 *m, u32 *b);

struct ipu_rgb {
	struct fb_bitfield	red;
	struct fb_bitfield	green;
	struct fb_bitfield	blue;
	struct fb_bitfield	transp;
	int			bits_per_pixel;
};

struct ipu_image {
	struct v4l2_pix_format pix;
	struct v4l2_rect rect;
	dma_addr_t phys0;
	dma_addr_t phys1;
	dma_addr_t phys2;
};

/*
 * IPU Channel Parameter Memory (cpmem) functions
 */

void ipu_cpmem_dump(struct ipuv3_channel *channel);
void ipu_cpmem_zero(struct ipuv3_channel *channel);
void ipu_cpmem_set_resolution(struct ipuv3_channel *channel,
			      int xres, int yres);
void ipu_cpmem_set_stride(struct ipuv3_channel *channel, int stride);
void ipu_cpmem_set_high_priority(struct ipuv3_channel *channel);
int ipu_cpmem_set_format_passthrough(struct ipuv3_channel *channel, int width);
int ipu_cpmem_set_format_rgb(struct ipuv3_channel *channel,
			     struct ipu_rgb *rgb);
void ipu_cpmem_set_yuv_planar(struct ipuv3_channel *channel, u32 pixel_format,
			      int stride, int height);
void ipu_cpmem_set_yuv_planar_full(struct ipuv3_channel *channel,
				   u32 pixel_format, int stride,
				   int u_offset, int v_offset);
int ipu_cpmem_set_fmt(struct ipuv3_channel *channel, u32 pixelformat);
int ipu_cpmem_set_image(struct ipuv3_channel *channel, struct ipu_image *image);
void ipu_cpmem_channel_init(struct ipuv3_channel *channel,
			    u32 pixel_fmt, u32 width,
			    u32 height, u32 stride,
			    u32 u, u32 v,
			    u32 uv_stride, dma_addr_t addr0,
			    dma_addr_t addr1, dma_addr_t addr2);
int ipu_cpmem_get_burst_size(struct ipuv3_channel *channel);
void ipu_cpmem_set_burst_size(struct ipuv3_channel *channel,
				 u16 burst_pixels);
int ipu_cpmem_get_bpp(struct ipuv3_channel *channel);
void ipu_cpmem_set_buffer(struct ipuv3_channel *channel,
			     int bufNum, dma_addr_t phyaddr);
void ipu_cpmem_set_uv_offset(struct ipuv3_channel *channel,
			     u32 u_offset, u32 v_offset);
void ipu_cpmem_set_rotation(struct ipuv3_channel *channel,
			       enum ipu_rotate_mode rot);
void ipu_cpmem_set_block_mode(struct ipuv3_channel *channel);
void ipu_cpmem_set_alpha_use_separate_channel(struct ipuv3_channel *channel,
						 bool option);
void ipu_cpmem_set_alpha_condition_read(struct ipuv3_channel *channel);
void ipu_cpmem_set_alpha_buffer_memory(struct ipuv3_channel *channel);
void ipu_cpmem_set_interlaced_scan(struct ipuv3_channel *channel,
				      int stride);
void ipu_cpmem_set_axi_id(struct ipuv3_channel *channel, u32 id);
void ipu_cpmem_offset_update(struct ipuv3_channel *channel,
			     u32 pixel_fmt,
			     u32 width,
			     u32 height,
			     u32 stride,
			     u32 u,
			     u32 v,
			     u32 uv_stride,
			     u32 vertical_offset,
			     u32 horizontal_offset);
void ipu_cpmem_set_alpha_width(struct ipuv3_channel *channel,
			       int alpha_width);
void ipu_cpmem_set_bandmode(struct ipuv3_channel *channel,
			       u32 band_height);


#endif /* __IMX_IPU_H__ */
