/*
 * SCC RTC driver
 * This driver communicates with the SCC in order to get and set SCC RTC data.
 *
 * Copyright (c) 2013 Robert Bosch Car Multimedia GmbH
 *
 * This file is subject to the terms and conditions of the GNU General
 * Public License. See the file "COPYING" in the main directory of this
 * archive for more details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/err.h>
#include <linux/rtc.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/in.h>
#include <linux/net.h>
#include <linux/inet.h>
#include <linux/dgram_service.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_platform.h>
#include <linux/kthread.h>
#include <linux/printk.h>
#include <linux/inc_ports.h>
#include <linux/jiffies.h>
#include <linux/inc_scc_rtc.h>


#define TIMEOUT (5*(HZ))	/*5 seconds*/
#define RTC_INC_DRV_NAME "rtc-inc"


#ifdef DEBUG
	#define dbg_msg(string, args...) \
		pr_debug("%s:%s:%d:" string, \
			RTC_INC_DRV_NAME, __func__, __LINE__, ##args)
	#define err_msg(string, args...) \
		pr_err("%s:%s:%d:" string, \
			RTC_INC_DRV_NAME, __func__, __LINE__, ##args)
	#define alert_msg(string, args...) \
		pr_alert("%s:%s:%d:" string, \
			RTC_INC_DRV_NAME, __func__, __LINE__, ##args)
	#define warn_msg(string, args...) \
		pr_warn("%s:%s:%d:" string, \
			RTC_INC_DRV_NAME, __func__, __LINE__, ##args)
	#define info_msg(string, args...) \
		pr_info("%s:%s:%d:" string, \
			RTC_INC_DRV_NAME, __func__, __LINE__, ##args)
#else
	#define dbg_msg(string, args...) \
		pr_debug("%s:" string, \
				__func__, ##args)
	#define err_msg(string, args...) \
		pr_err("%s:" string, \
				__func__, ##args)
	#define alert_msg(string, args...) \
		pr_alert("%s:" string, \
				__func__, ##args)
	#define warn_msg(string, args...) \
		pr_warn("%s:" string, \
				__func__, ##args)
	#define info_msg(string, args...) \
		pr_info("%s:" string, \
				__func__, ##args)
#endif


#define DRV_VERSION	"1.0"
#define INC_TASK_NAME	"rtc-inc-task"

#define INC_VERSION	1

#define DGRAMINIT_MAX	256

#define SCC_RTC_R_REJECT_SIZE		3
#define SCC_RTC_R_COMPONENT_STATUS_SIZE	3
#define SCC_RTC_R_GET_DATE_TIME_SIZE	7
#define SCC_RTC_R_SET_DATE_TIME_SIZE	1

#define REJ_NO_REASON			0x00
#define REJ_UNKNOWN_MESSAGE		0x01
#define REJ_INVALID_PARA		0x02
#define REJ_TMP_UNAVAIL			0x03
#define REJ_VERSION_MISMATCH		0x04

#define RTC_ACTIVE			0x01
#define RTC_INACTIVE			0x02



static u32 inet_addr(char *str)
{
	int a, b, c, d;
	char arr[4];
	sscanf(str, "%d.%d.%d.%d", &a, &b, &c, &d);
	arr[0] = a;
	arr[1] = b;
	arr[2] = c;
	arr[3] = d;
	return *(u32 *)arr;

}

/**
 * external configuration of remote port may be required for testing purpose
 */
static ushort remote_port = 0x0000;
module_param(remote_port, ushort, 0);
MODULE_PARM_DESC(remote_port, "INC remote_port");


struct rtc_inc {
	struct device		*dev;
	struct input_dev	*rtcdev;
	struct task_struct	*inc_task;
	struct mutex		inc_thread_data_lock;
	struct mutex		driver_lock;
	struct socket		*socket;
	struct sk_dgram		*dgram;
	struct sockaddr_in	local;
	struct sockaddr_in	remote;
	struct __wait_queue_head data_available;
	bool			error;
	bool			scc_cmd_status_active;
	bool			scc_tm_set;
	bool			scc_tm_valid;
	bool inc_init_done;
	struct rtc_time		scc_tm;
	struct rtc_device *rtc;
};


static int rtc_inc_get_dt_properties(struct rtc_inc *rtc_inc)
{
	struct device_node *rtc_node = rtc_inc->dev->of_node;
	struct device_node *inc_node;
	const char *property_string = NULL, *inc_node_path = NULL;
	const void *prop;

	property_string = "node";
	prop = of_get_property(rtc_node, property_string, NULL);
	if (prop != NULL)
		inc_node_path = (char *) prop;
	else
		goto err;

	inc_node = of_find_node_by_path(inc_node_path);

	if (inc_node == NULL) {
		err_msg("can't find %s node in device tree\n", inc_node_path);
		return -ENODEV;
	}

	dbg_msg("\n");

	rtc_inc->local.sin_family = AF_INC;
	rtc_inc->remote.sin_family = AF_INC;

	property_string = "local-addr";
	prop = of_get_property(inc_node, property_string, NULL);

	if (prop) {
		rtc_inc->local.sin_addr.s_addr =
						inet_addr((char *)prop);
		info_msg("local-addr is: [%s]\n", (char *)prop);
	} else {
		err_msg("Couldnt find local-addr in device tree\n");
		goto err;
	}

	property_string = "remote-addr";
	prop = of_get_property(inc_node, property_string, NULL);

	if (prop) {
		rtc_inc->remote.sin_addr.s_addr =
						inet_addr((char *)prop);
		info_msg("remote-addr is: [%s]\n", (char *)prop);
	} else {
		err_msg("Couldnt find remote-addr in device tree\n");
		goto err;
	}

	if (remote_port) {
		rtc_inc->local.sin_port = htons(RTC_PORT);
		info_msg("local port is: [%d]\n", RTC_PORT);
		rtc_inc->remote.sin_port = htons(remote_port);
		info_msg("remote port is: [%d]\n", remote_port);
	} else {
		rtc_inc->local.sin_port = htons(RTC_PORT);
		rtc_inc->remote.sin_port =
				rtc_inc->local.sin_port;
		info_msg("local and remote ports are: [%d]\n", RTC_PORT);
	}
	return 0;

err:
	err_msg("error in parsing device tree\n");
	return -1;
}

static void inc_sock_close(struct rtc_inc *rtc_inc)
{
	dbg_msg("local %pI4:%d remote %pI4:%d\n",
			&rtc_inc->local.sin_addr.s_addr,
			ntohs(rtc_inc->local.sin_port),
			&rtc_inc->remote.sin_addr.s_addr,
			ntohs(rtc_inc->local.sin_port));

	if (rtc_inc->dgram != NULL) {
		if (rtc_inc->dgram) {
			dbg_msg("dgram_exit\n");
			dgram_exit(rtc_inc->dgram);
		} else {
			err_msg("dgram is null\n");
		}
	}

	if (rtc_inc->socket != NULL) {
		if (rtc_inc->socket) {
			dbg_msg("sock_shutdown\n");
			kernel_sock_shutdown(rtc_inc->socket, SHUT_RDWR);
		} else {
			err_msg("sock is null\n");
		}
	}

	if (rtc_inc->socket != NULL) {
		if (rtc_inc->socket) {
			dbg_msg("sock_release\n");
			sock_release(rtc_inc->socket);
		}
	}

	rtc_inc->dgram = NULL;
	rtc_inc->socket = NULL;
	return;
}

static int inc_sock_open(struct rtc_inc *rtc_inc)
{
	int ret = 0;

	dbg_msg("local %pI4:%d remote %pI4:%d\n",
			&rtc_inc->local.sin_addr.s_addr,
			ntohs(rtc_inc->local.sin_port),
			&rtc_inc->remote.sin_addr.s_addr,
			ntohs(rtc_inc->remote.sin_port));

	ret = sock_create_kern(AF_INC, SOCK_STREAM, 0, &rtc_inc->socket);

	if (ret < 0) {
		err_msg("failed to initialize socket\n");
		goto err;
	} else {
		dbg_msg("done creating socket\n");
	}

	rtc_inc->dgram = dgram_init(rtc_inc->socket, DGRAMINIT_MAX, NULL);

	if (rtc_inc->dgram == NULL) {
		err_msg("dgram_init failed\n");
		goto err;
	}

	ret = kernel_bind(rtc_inc->socket, (struct sockaddr *)&rtc_inc->local,
			sizeof(rtc_inc->local));

	if (ret < 0) {
		err_msg("binding to socket failed ret %d\n", ret);
		goto err;
	} else {
		dbg_msg("done binding to socket\n");
	}

	ret = kernel_connect(rtc_inc->socket,
			(struct sockaddr *)&rtc_inc->remote,
			sizeof(rtc_inc->remote), 0);

	if (ret < 0) {
		err_msg("connecting to socket failed ret %d\n", ret);
		goto err;
	} else {
		dbg_msg("done connecting to socket\n");
	}

	return ret;

err:
	inc_sock_close(rtc_inc);
	return ret;

}


static int inc_recvmsg(struct rtc_inc *rtc_inc, void *data, int len)
{
	int ret = 0, i = 0;

	do {
		ret = dgram_recv(rtc_inc->dgram, data, len);

		if ((ret < 0) && (ret != -EAGAIN))
			alert_msg("recvmsg failed ret: %d\n", ret);

		for (i = 0; i < ret; i++)
			dbg_msg("data[%d]=0x%02X\n", i, ((u_int8_t *)data)[i]);

	} while (ret == -EAGAIN);

	return ret;
}

static int inc_sendmsg(struct rtc_inc *rtc_inc, u_int8_t *data, int len)
{
	int ret = 0 , i = 0;

	dbg_msg("dgram_send to %pI4:%d len:%d data[0]:%d\n",
			&rtc_inc->remote.sin_addr.s_addr,
			ntohs(rtc_inc->remote.sin_port), len, data[0]);

	ret = dgram_send(rtc_inc->dgram, data, len);

	if (ret < 0)
		alert_msg("Could not not send message: %d\n", ret);
	else
		dbg_msg("Send message done return value : %d\n", ret);

	for (i = 0; i < ret; i++)
		dbg_msg("data[%d]=0x%02X\n", i, ((u_int8_t *)data)[i]);

	return ret;
}

static int inc_cmd_status_active(struct rtc_inc *rtc_inc)
{
	u_int8_t data[] = { SCC_RTC_C_COMPONENT_STATUS_MSGID,
				RTC_ACTIVE, INC_VERSION };

	int ret = 0;

	ret = inc_sendmsg(rtc_inc, data, sizeof(data));

	return ret;
}

static int inc_get_datetime(struct rtc_inc *rtc_inc)
{
	u_int8_t data[] = { SCC_RTC_C_GET_DATE_TIME_MSGID };

	int ret = 0;

	ret = inc_sendmsg(rtc_inc, data, sizeof(data));

	return ret;
}

static int inc_set_datetime(struct rtc_inc *rtc_inc, struct rtc_time *tm)
{
	u_int8_t data[7];

	int ret = 0;

	data[0] = SCC_RTC_C_SET_DATE_TIME_MSGID;
	data[1] = tm->tm_hour;
	data[2] = tm->tm_min;
	data[3] = tm->tm_sec;
	data[4] = tm->tm_mday;
	data[5] = tm->tm_mon  + 1;
	data[6] = tm->tm_year - 100;

	ret = inc_sendmsg(rtc_inc, data, sizeof(data));

	return ret;
}

static int inc_handle_msg(struct rtc_inc *rtc_inc, u_int8_t *data, int size)
{

	dbg_msg("inc_handle_msg data[0]:%02Xh\n", data[0]);

	switch (data[0]) {

	case SCC_RTC_R_COMPONENT_STATUS_MSGID:
		if (size != SCC_RTC_R_COMPONENT_STATUS_SIZE) {
			alert_msg("R_COMPONENT_STATUS Message wrong size: %d\n",
					size);

			rtc_inc->error = true;
		} else
		if (data[2] != INC_VERSION) {
			alert_msg("Incorrect version of the INC protocol: %d\n",
					data[2]);
			rtc_inc->error = true;
		} else
			rtc_inc->error = false;

		rtc_inc->scc_cmd_status_active = true;
		wake_up_interruptible(&rtc_inc->data_available);
		break;

	case SCC_RTC_R_REJECT_MSGID:
		if (size != SCC_RTC_R_REJECT_SIZE)
			alert_msg(" R_REJECT Message wrong size: %d\n",
					size);

		else
			alert_msg("R_REJECT Reason:%d MsgID:%d.\n",
				data[1], data[2]);

		rtc_inc->scc_tm_valid = true;
		rtc_inc->scc_tm_set = true;
		rtc_inc->error = true;
		wake_up_interruptible(&rtc_inc->data_available);
		break;

	case SCC_RTC_R_GET_DATE_TIME_MSGID:
		memset(&rtc_inc->scc_tm, 0,
				sizeof(rtc_inc->scc_tm));
		if (size != SCC_RTC_R_GET_DATE_TIME_SIZE) {
			alert_msg("R_GET_DATE_TIME Message wrong size: %d\n",
					size);
			rtc_inc->error = true;
		} else {
			rtc_inc->scc_tm.tm_sec  = data[3];
			rtc_inc->scc_tm.tm_min  = data[2];
			rtc_inc->scc_tm.tm_hour = data[1];
			rtc_inc->scc_tm.tm_wday = 0;
			rtc_inc->scc_tm.tm_mday = data[4];
			rtc_inc->scc_tm.tm_mon  = data[5] - 1;
			rtc_inc->scc_tm.tm_year = data[6] + 100;

			rtc_inc->scc_tm_valid = true;
			rtc_inc->error = false;

			dbg_msg(
			"R_GET_DATE_TIME: %02X-%02X-%02X-%02X-%02X-%02X\n",
			data[1], data[2], data[3], data[4], data[5], data[6]);
		}

		wake_up_interruptible(&rtc_inc->data_available);
		break;

	case SCC_RTC_R_SET_DATE_TIME_MSGID:
		if (size != SCC_RTC_R_SET_DATE_TIME_SIZE) {
			alert_msg("R_SET_DATE_TIME Message wrong size: %d\n",
					size);
			rtc_inc->error = true;
		} else
			rtc_inc->error = false;

		rtc_inc->scc_tm_set = true;
		wake_up_interruptible(&rtc_inc->data_available);
		break;

	default:
		alert_msg("rtc-inc Unknown Message %02Xh.\n", data[0]);
		break;
	}

	if (rtc_inc->error == true)
		return -1;
	else
		return rtc_inc->error;
}

static int inc_handler_thread_function(void *arg)
{
#define RX_BUFFER_SIZE 50

	u_int8_t data[RX_BUFFER_SIZE] = {0};
	int ret = 0;
	struct rtc_inc *rtc_inc = arg;

	dbg_msg("inc_handler_thread_function Started\n");
	do {
		ret = inc_recvmsg(rtc_inc, data, RX_BUFFER_SIZE);

		dbg_msg("Received msg size %d\n", ret);

		if (ret >= 0)
			ret = inc_handle_msg(rtc_inc, data, ret);

	} while (ret >= 0);

	dbg_msg("inc_handler_thread_function Exiting\n");

	return 0;
}

static void inc_cleanup(struct rtc_inc *rtc_inc)
{
	dbg_msg("inc_cleanup\n");

	/**Protecting with lock to protect against race condition
	 * that inc_task is set to NULL in kernel thread*/
	mutex_lock(&rtc_inc->inc_thread_data_lock);
	if (NULL != rtc_inc->inc_task)
		force_sig(SIGKILL, rtc_inc->inc_task);
	mutex_unlock(&rtc_inc->inc_thread_data_lock);

	if (NULL != rtc_inc->inc_task)
		kthread_stop(rtc_inc->inc_task);

	rtc_inc->inc_task = NULL;

	/*Close inc socket*/
	inc_sock_close(rtc_inc);

	dbg_msg("Thread stopped,sock closed\n");

	return;
}

static int inc_init(struct rtc_inc *rtc_inc)
{
	int ret = 0;

	rtc_inc->inc_init_done = false;

	mutex_init(&rtc_inc->driver_lock);
	mutex_init(&rtc_inc->inc_thread_data_lock);

	/* Initialize wait queue */
	init_waitqueue_head(&rtc_inc->data_available);
	dbg_msg("Wait queue initialized\n");

	ret = inc_sock_open(rtc_inc);
	if (ret < 0)
		goto fail;

	rtc_inc->inc_task = kthread_run(inc_handler_thread_function,
						rtc_inc, INC_TASK_NAME);

	if (IS_ERR(rtc_inc->inc_task)) {
		err_msg("failed to create inc handler thread\n");
		goto fail;
	} else {
		dbg_msg("Kthread created\n");
	}

	rtc_inc->error = false;
	rtc_inc->scc_cmd_status_active = false;

	ret = inc_cmd_status_active(rtc_inc);
	if (ret < 0) {
		err_msg("scc RTC not ready.\n");
		goto fail;
	}
	ret = 0;

	dbg_msg("Waiting on the queue\n");

	if (wait_event_interruptible_timeout(rtc_inc->data_available,
			rtc_inc->scc_cmd_status_active, TIMEOUT) < 1) {
		err_msg("inc queue timeout\n");
		ret = -ETIME;
		goto fail;
	}

	dbg_msg("Out of wait queue\n");

	if (rtc_inc->error) {
		ret = -EPROTO;
		goto fail;
	}

	rtc_inc->inc_init_done = true;
	return 0;

fail:
	inc_cleanup(rtc_inc);
	return ret;
}

static int rtc_inc_read_time(struct device *dev, struct rtc_time *tm)
{
	int retval = 0;
	struct rtc_inc *rtc_inc = dev_get_drvdata(dev);

	if (!rtc_inc->inc_init_done)
		return -ENODATA;

	mutex_lock(&rtc_inc->driver_lock);

	dbg_msg("rtc-inc rtc_read_time\n");

	rtc_inc->error = false;
	rtc_inc->scc_tm_valid = false;

	retval = inc_get_datetime(rtc_inc);
	if (retval < 0) {
		err_msg("send msg failed\n");
		goto the_end;
	}
	retval = 0;

	dbg_msg("waiting on the queue\n");

	if (wait_event_interruptible_timeout(rtc_inc->data_available,
			rtc_inc->scc_tm_valid, TIMEOUT) < 1) {
		err_msg("inc queue timeout\n");
		retval = -ETIME;
		goto the_end;
	}

	dbg_msg("Out of wait queue\n");

	if (!rtc_inc->error) {

		mutex_lock(&rtc_inc->inc_thread_data_lock);

		tm->tm_sec  = rtc_inc->scc_tm.tm_sec;
		tm->tm_min  = rtc_inc->scc_tm.tm_min;
		tm->tm_hour = rtc_inc->scc_tm.tm_hour;
		tm->tm_wday = rtc_inc->scc_tm.tm_wday;
		tm->tm_mday = rtc_inc->scc_tm.tm_mday;
		tm->tm_mon  = rtc_inc->scc_tm.tm_mon;
		tm->tm_year = rtc_inc->scc_tm.tm_year;

		mutex_unlock(&rtc_inc->inc_thread_data_lock);
	} else
		retval = -EPROTO;

the_end:
	mutex_unlock(&rtc_inc->driver_lock);
	return retval;
}

static int rtc_inc_set_time(struct device *dev, struct rtc_time *tm)
{
	int retval = 0;
	struct rtc_inc *rtc_inc = dev_get_drvdata(dev);

	if (!rtc_inc->inc_init_done)
		return -ENODATA;

	/* the year in the SCC is from 0 to 99 (2000 to 2099) */
	/* the tm_year field in struct rtc_time in the
	   number of years since 1900, so between 100 and 199 */
	if (!((tm->tm_year >= 100) && (tm->tm_year <= 199)))
		return -EOVERFLOW;

	mutex_lock(&rtc_inc->driver_lock);

	dbg_msg("rtc-inc rtc_set_time\n");

	rtc_inc->error = false;
	rtc_inc->scc_tm_set = false;

	retval = inc_set_datetime(rtc_inc, tm);
	if (retval < 0) {
		err_msg("send msg failed\n");
		goto the_end;
	}
	retval = 0;

	dbg_msg("waiting on the queue\n");

	if (wait_event_interruptible_timeout(rtc_inc->data_available,
			rtc_inc->scc_tm_set, TIMEOUT) < 1) {
		err_msg("inc queue timeout\n");
		retval = -ETIME;
		goto the_end;
	}

	dbg_msg("Out of wait queue\n");

	retval = (!rtc_inc->error) ? 0 : -EPROTO;

the_end:
	mutex_unlock(&rtc_inc->driver_lock);
	return retval;
}


static const struct rtc_class_ops rtc_scc_ops = {
	.read_time        = rtc_inc_read_time,
	.set_time         = rtc_inc_set_time,
};

static ssize_t rtc_inc_irq_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	pr_info("rtc-inc irq_show\n");

	return sprintf(buf, "%d\n", 42);
}

static ssize_t rtc_inc_irq_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t count)
{
	int retval;
	struct rtc_inc *rtc_inc = dev_get_drvdata(dev);
	struct rtc_device *rtc = rtc_inc->rtc;

	pr_info("rtc-inc irq_store\n");

	retval = count;
	if (strncmp(buf, "tick", 4) == 0 && rtc->pie_enabled)
		rtc_update_irq(rtc, 1, RTC_PF | RTC_IRQF);
	else if (strncmp(buf, "alarm", 5) == 0) {
		struct rtc_wkalrm alrm;
		int err = rtc_read_alarm(rtc, &alrm);

		if (!err && alrm.enabled)
			rtc_update_irq(rtc, 1, RTC_AF | RTC_IRQF);

	} else if (strncmp(buf, "update", 6) == 0 && rtc->uie_rtctimer.enabled)
		rtc_update_irq(rtc, 1, RTC_UF | RTC_IRQF);
	else
		retval = -EINVAL;

	return retval;
}

static DEVICE_ATTR(irq, S_IRUGO | S_IWUSR, rtc_inc_irq_show, rtc_inc_irq_store);

static struct of_device_id rtc_inc_match[] = {
	{ .compatible = "rbcm,rtc-inc", },
	{ /* this can be filled with module_param */ },
	{}
};

static int rtc_inc_remove(struct platform_device *plat_dev)
{
	struct rtc_inc *rtc_inc = platform_get_drvdata(plat_dev);
	int err = 0;

	pr_info("rtc-inc rtc_remove\n");

	if (!rtc_inc)
		goto exit;

	rtc_device_unregister(rtc_inc->rtc);

	inc_cleanup(rtc_inc);

	device_remove_file(&plat_dev->dev, &dev_attr_irq);

	platform_set_drvdata(plat_dev, NULL);

	kfree(rtc_inc);

exit:
	return err;
}


static int rtc_inc_probe(struct platform_device *plat_dev)
{
	int err;
	struct rtc_inc *rtc_inc;

	struct device *dev = &plat_dev->dev;
	const struct of_device_id *id =	of_match_device(rtc_inc_match, dev);

	if (!id) {
		dev_err(dev, "%s: failed to find input device\n", __func__);
		err = -EFAULT;
		goto exit;
	}

	rtc_inc = kzalloc(sizeof(*rtc_inc), GFP_KERNEL);
	if (!rtc_inc) {
		dev_err(dev, "%s: failed to allocate memory for device\n",
				__func__);
		err = -ENOMEM;
		goto exit;
	}

	rtc_inc->inc_init_done = false;
	rtc_inc->dev = dev;
	rtc_inc->inc_task = NULL;
	rtc_inc->socket = NULL;
	rtc_inc->dgram = NULL;
	rtc_inc->local.sin_port = htons(RTC_PORT);
	rtc_inc->remote.sin_port = rtc_inc->local.sin_port;

	mutex_init(&rtc_inc->driver_lock);
	mutex_init(&rtc_inc->inc_thread_data_lock);

	err = rtc_inc_get_dt_properties(rtc_inc);
	if (err < 0)
		goto exit;

	err = inc_init(rtc_inc);
	if (err != 0)
		goto exit;

	platform_set_drvdata(plat_dev, rtc_inc);

	rtc_inc->rtc = rtc_device_register("rtc-inc", &plat_dev->dev,
						&rtc_scc_ops, THIS_MODULE);

	pr_info("rtc-inc rtc_probe\n");

	if (IS_ERR(rtc_inc->rtc)) {
		err = PTR_ERR(rtc_inc->rtc);
		goto exit_inc_cleanup;
	}

	err = device_create_file(&plat_dev->dev, &dev_attr_irq);
	if (err)
		goto exit_device_unregister;


	return 0;

exit_device_unregister:
	rtc_device_unregister(rtc_inc->rtc);

exit_inc_cleanup:
	inc_cleanup(rtc_inc);

exit:
	return err;
}



static struct platform_driver rtc_inc_driver = {
	.probe	= rtc_inc_probe,
	.remove = rtc_inc_remove,
	.driver = {
		.name = RTC_INC_DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = rtc_inc_match,
	},
};

static int __init rtc_inc_init(void)
{
	pr_info("rtc-inc initialization\n");

	return platform_driver_register(&rtc_inc_driver);
}

static void __exit rtc_inc_exit(void)
{
	platform_driver_unregister(&rtc_inc_driver);
}

MODULE_AUTHOR("Robert Bosch Car Multimedia GmbH <Eduardo Goncalves external.eduardo.goncalves@de.bosch.com>");
MODULE_DESCRIPTION("RTC SCC driver");
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VERSION);
MODULE_ALIAS("platform:rtc-inc");

module_init(rtc_inc_init);
module_exit(rtc_inc_exit);
