/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_macro.h"
#include "csr_pmem.h"
#include "csr_util.h"
#include "csr_log_text_2.h"

#include "csr_wifi_mib.h"


#define CSR_WIFI_MIB_MORE_MASK   0x80
#define CSR_WIFI_MIB_SIGN_MASK   0x40
#define CSR_WIFI_MIB_TYPE_MASK   0x20
#define CSR_WIFI_MIB_LENGTH_MASK 0x1F


#ifdef CSR_LOG_ENABLE
CsrLogTextHandle *CSR_WIFI_MIB_LOG_ID = NULL;
/* Logging Categories (MUST BE CONTIGUIOUS Starting at 0x0001) */
#define TR_MIB              0x0001
static const CsrCharString *csrWifiMibLogModules[] = {"Mib"};

void CsrWifiMibLogTextRegister(void)
{
    if (CSR_WIFI_MIB_LOG_ID)
    {
        return;
    }
    CsrLogTextRegister2(&CSR_WIFI_MIB_LOG_ID, "CsrWifiMib", CSR_ARRAY_SIZE(csrWifiMibLogModules), csrWifiMibLogModules);
}

#else
#define CsrWifiMibLogTextRegister()
#endif


/*
 * @brief
 *  Append a buffer to an existing buffer.
 *  This will CsrPmemAlloc a new buffer and CsrPmemFree the old one
 */
void CsrWifiMibBufferAppend(CsrWifiDataBlock *dst, CsrSize bufferLength, CsrUint8 *buffer)
{
    CsrUint8 *newBuffer = CsrPmemAlloc(dst->dataLength + bufferLength);

    CsrMemCpy(newBuffer, dst->data, dst->dataLength);
    CsrMemCpy(&newBuffer[dst->dataLength], buffer, bufferLength);

    dst->dataLength += (CsrUint16) bufferLength;
    if (dst->data)
    {
        CsrPmemFree(dst->data);
    }
    dst->data = newBuffer;
}

CsrSize CsrWifiMibEncodeUint32(CsrUint8 *buffer, CsrUint32 value)
{
    CsrUint8 i;
    CsrUint8 writeCount = 0;
    if (value < 64)
    {
        buffer[0] = (CsrUint8) value;
        return 1;
    }

    /* Encode the Integer
     *  0xABFF0055 = [0xAB, 0xFF, 0x00, 0x55]
     *    0xAB0055 = [0xAB, 0x00, 0x55]
     *      0xAB55 = [0xAB, 0x55]
     *        0x55 = [0x55] */
    for (i = 0; i < 4; i++)
    {
        CsrUint8 byteValue = (value & 0xFF000000) >> 24;
        if (byteValue || writeCount)
        {
            buffer[1 + writeCount] = byteValue;
            writeCount++;
        }
        value = value << 8;
    }
    buffer[0] = writeCount | CSR_WIFI_MIB_MORE_MASK; /* vldata Length | more bit */

    return 1 + writeCount;
}

CsrSize CsrWifiMibEncodeInt32(CsrUint8 *buffer, CsrInt32 signedValue)
{
    CsrUint8 i;
    CsrUint8 writeCount = 0;
    CsrUint32 value = (CsrUint32) signedValue;

    if (!(value & 0x10000000))
    {
        /* just use the Unsigned Encoder */
        return CsrWifiMibEncodeUint32(buffer, value);
    }

    if (signedValue >= -64)
    {
        buffer[0] = (CsrUint8) value & 0x7F; /* vldata Length | more bit */
        return 1;
    }

    /* Encode the Negative Integer */
    for (i = 0; i < 4; i++)
    {
        CsrUint8 byteValue = (value & 0xFF000000) >> 24;
        if (!((byteValue == 0xFF) && (value & 0x800000)) || writeCount)
        {
            buffer[1 + writeCount] = byteValue;
            writeCount++;
        }
        value = value << 8;
    }
    buffer[0] = writeCount | CSR_WIFI_MIB_MORE_MASK | CSR_WIFI_MIB_SIGN_MASK; /* vldata Length | more bit | sign bit*/

    return 1 + writeCount;
}

CsrSize CsrWifiMibEncodeOctetString(CsrUint8 *buffer, CsrWifiDataBlock *octetValue)
{
    CsrUint8 i;
    CsrUint8 writeCount = 0;
    CsrSize length = octetValue->dataLength;

    /* Encode the Length (Up to 4 bytes 32 bits worth)
     *  0xABFF0000 = [0xAB, 0xFF, 0x00, 0x00]
     *    0xAB0000 = [0xAB, 0x00, 0x00]
     *      0xAB00 = [0xAB, 0x00]
     *        0x00 = [0x00] */
    for (i = 0; i < 3; i++)
    {
        CsrUint8 byteValue = (length & 0xFF000000) >> 24;
        if (byteValue || writeCount)
        {
            buffer[1 + writeCount] = byteValue;
            writeCount++;
        }
        length = length << 8;
    }

    buffer[0] = (1 + writeCount) | CSR_WIFI_MIB_MORE_MASK | CSR_WIFI_MIB_TYPE_MASK;
    buffer[1 + writeCount] = octetValue->dataLength & 0xFF;
    CsrMemCpy(&buffer[2 + writeCount], octetValue->data, octetValue->dataLength);

    return 2 + writeCount + octetValue->dataLength;
}

CsrSize CsrWifiMibDecodeUint32(CsrUint8 *buffer, CsrUint32 *value)
{
    CsrSize i;
    CsrUint32 v = 0;
    CsrSize length = buffer[0] & 0x1F;
    if (!(buffer[0] & CSR_WIFI_MIB_MORE_MASK))
    {
        *value = buffer[0] & 0x7F;
        return 1;
    }

    for (i = 0; i < length; i++)
    {
        v = (v << 8);
        v |= buffer[1 + i];
    }

    *value = v;

    return 1 + length;
}

CsrSize CsrWifiMibDecodeInt32(CsrUint8 *buffer, CsrInt32 *value)
{
    CsrSize i;
    CsrUint32 v = 0xFFFFFFFF;
    CsrSize length = buffer[0] & 0x1F;

    if (!(buffer[0] & CSR_WIFI_MIB_SIGN_MASK))
    {
        /* just use the Unsigned Decoder */
        return CsrWifiMibDecodeUint32(buffer, (CsrUint32 *) value);
    }

    if (!(buffer[0] & CSR_WIFI_MIB_MORE_MASK))
    {
        *value = (CsrInt32) (0xFFFFFF80 | buffer[0]);
        return 1;
    }

    for (i = 0; i < length; i++)
    {
        v = (v << 8);
        v |= buffer[1 + i];
    }

    *value = (CsrInt32) v;

    return 1 + length;
}

/* Just references the oid in the existing buffer. No new memory is allcated */
CsrSize CsrWifiMibDecodeOctetString(CsrUint8 *buffer, CsrWifiDataBlock *octetValue)
{
    CsrSize i;
    CsrUint32 oidLengthValue = 0;
    CsrSize length = buffer[0] & 0x1F;

    for (i = 0; i < length; i++)
    {
        oidLengthValue = (oidLengthValue << 8);
        oidLengthValue |= buffer[1 + i];
    }

    octetValue->dataLength = oidLengthValue;
    octetValue->data = NULL;
    if (oidLengthValue)
    {
        octetValue->data = &buffer[1 + length];
    }

    return 1 + length + oidLengthValue;
}

static CsrWifiMibType CsrWifiMibDecodeTypeAndLength(CsrUint8 *buffer, CsrSize *length)
{
    *length = 1;
    if (buffer[0] & CSR_WIFI_MIB_MORE_MASK)
    {
        *length = buffer[0] & CSR_WIFI_MIB_LENGTH_MASK;
    }

    if (buffer[0] & CSR_WIFI_MIB_SIGN_MASK)
    {
        return CSR_WIFI_MIB_TYPE_INT;
    }

    if ((buffer[0] & CSR_WIFI_MIB_MORE_MASK) &&
        (buffer[0] & CSR_WIFI_MIB_TYPE_MASK))
    {
        CsrSize i;
        CsrSize oidLengthValue = 0;
        for (i = 0; i < *length; i++)
        {
            oidLengthValue = (oidLengthValue << 8);
            oidLengthValue |= buffer[1 + i];
        }
        *length += oidLengthValue;
        return CSR_WIFI_MIB_TYPE_OCTET;
    }
    return CSR_WIFI_MIB_TYPE_UINT;
}

static CsrSize CsrWifiMibEncodePsidAndIndexs(CsrUint8 *buffer, const CsrWifiMibGetEntry *value)
{
    CsrSize i;
    CSR_COPY_UINT16_TO_LITTLE_ENDIAN(value->psid, &buffer[0]);
    buffer[2] = 0;
    buffer[3] = 0;
    for (i = 0; i < CSR_WIFI_MIB_MAX_INDEXES && value->index[i] != 0; i++)
    {
        buffer[2] += (CsrUint8) CsrWifiMibEncodeUint32(&buffer[4 + buffer[2]], value->index[i]);
    }

    if (buffer[2] % 2 == 1)
    {
        /* Add a padding byte "0x00" to the encoded buffer. The Length value is NOT updated to account for this pad value
         * If the length is an Odd number the Pad values MUST be there if it is Even it will not be.*/
        /*CSR_LOG_TEXT_CRITICAL((CSR_WIFI_MIB_LOG_ID, TR_MIB, "CsrWifiMibEncodePsidAndIndexs(psid:0x%.4X) Odd number of Bytes detected", value->psid));*/
        buffer[4 + buffer[2]] = 0x00;
        return 5 + buffer[2];
    }

    return 4 + buffer[2];
}

CsrResult CsrWifiMibEncode(CsrWifiDataBlock *buffer, CsrWifiMibEntry *value)
{
    CsrSize i;
    CsrSize requiredSize = 5 + (5 * CSR_WIFI_MIB_MAX_INDEXES) +
                           (value->value.type == CSR_WIFI_MIB_TYPE_OCTET ? value->value.u.octetValue.dataLength : 5);

    CsrSize encodedLength = 4;

    CsrUint8 *tmpBuffer = (CsrUint8 *) CsrPmemAlloc(requiredSize);

    CsrWifiMibLogTextRegister();

    CSR_COPY_UINT16_TO_LITTLE_ENDIAN(value->psid, &tmpBuffer[0]);
    tmpBuffer[2] = 0;
    tmpBuffer[3] = 0;
    for (i = 0; i < CSR_WIFI_MIB_MAX_INDEXES && value->index[i] != 0; i++)
    {
        tmpBuffer[2] += (CsrUint8) CsrWifiMibEncodeUint32(&tmpBuffer[4 + tmpBuffer[2]], value->index[i]);
    }
    encodedLength += tmpBuffer[2];

    switch (value->value.type)
    {
        case CSR_WIFI_MIB_TYPE_UINT:
            encodedLength += CsrWifiMibEncodeUint32(&tmpBuffer[encodedLength], value->value.u.uintValue);
            break;
        case CSR_WIFI_MIB_TYPE_INT:
            encodedLength += CsrWifiMibEncodeInt32(&tmpBuffer[encodedLength], value->value.u.intValue);
            break;
        case CSR_WIFI_MIB_TYPE_OCTET:
            encodedLength += CsrWifiMibEncodeOctetString(&tmpBuffer[encodedLength], &value->value.u.octetValue);
            break;
        case CSR_WIFI_MIB_TYPE_BOOL:
            encodedLength += CsrWifiMibEncodeUint32(&tmpBuffer[encodedLength], value->value.u.boolValue ? TRUE : FALSE);
            break;
        case CSR_WIFI_MIB_TYPE_NONE:
            break;
        default:
            CSR_LOG_TEXT_ERROR((CSR_WIFI_MIB_LOG_ID, TR_MIB, "CsrWifiMibEncode() Invalid Type:%d requested", value->value.type));
            CsrPmemFree(tmpBuffer);
            return CSR_WIFI_SME_MIB_STATUS_ERROR;
    }

    CSR_COPY_UINT16_TO_LITTLE_ENDIAN(encodedLength - 4, &tmpBuffer[2]); /* length */

    if (encodedLength % 2 == 1)
    {
        /* Add a padding byte "0x00" to the encoded buffer. The Length value is NOT updated to account for this pad value
         * If the length is an Odd number the Pad values MUST be there if it is Even it will not be.*/
        /*CSR_LOG_TEXT_CRITICAL((CSR_WIFI_MIB_LOG_ID, TR_MIB, "CsrWifiMibEncode(psid:0x%.4X) Odd number of Bytes detected", value->psid));*/
        tmpBuffer[encodedLength] = 0x00;
        encodedLength++;
    }

    CsrWifiMibBufferAppend(buffer, encodedLength, tmpBuffer);
    CsrPmemFree(tmpBuffer);

    return CSR_RESULT_SUCCESS;
}

CsrSize CsrWifiMibDecode(CsrUint8 *buffer, CsrWifiMibEntry *value)
{
    CsrSize indexCount = 0;
    CsrWifiMibLogTextRegister();
    if (buffer && value)
    {
        CsrSize length = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(&buffer[2]);
        CsrSize decodedLength = 4;
        CsrMemSet(value, 0x00, sizeof(CsrWifiMibEntry));
        value->psid = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(buffer);
        value->value.type = CSR_WIFI_MIB_TYPE_NONE;
        CsrMemSet(&value->value.u, 0x00, sizeof(value->value.u));

        while (decodedLength < 4 + length)
        {
            CsrSize nextValueLength;
            CsrWifiMibType type = CsrWifiMibDecodeTypeAndLength(&buffer[decodedLength], &nextValueLength);

            switch (type)
            {
                case CSR_WIFI_MIB_TYPE_UINT:
                {
                    CsrUint32 v;
                    decodedLength += CsrWifiMibDecodeUint32(&buffer[decodedLength], &v);
                    /* If this is that last value then it is the "unitValue" if other values follow it is an Index Value */
                    if ((decodedLength < 4 + length) && (indexCount != CSR_WIFI_MIB_MAX_INDEXES))
                    {
                        value->index[indexCount] = (CsrUint16) v;
                        indexCount++;
                    }
                    else
                    {
                        value->value.type = type;
                        value->value.u.uintValue = v;
                        CSR_LOG_TEXT_CONDITIONAL_ERROR(decodedLength != 4 + length, (CSR_WIFI_MIB_LOG_ID, TR_MIB, "CsrWifiMibDecode() Uint Decode length:%d != expected:%d", (CsrUint32) decodedLength, (CsrUint32) (4 + length)));
                    }
                    break;
                }
                case CSR_WIFI_MIB_TYPE_INT:
                    value->value.type = type;
                    decodedLength += CsrWifiMibDecodeInt32(&buffer[decodedLength], &value->value.u.intValue);
                    CSR_LOG_TEXT_CONDITIONAL_ERROR(decodedLength != 4 + length, (CSR_WIFI_MIB_LOG_ID, TR_MIB, "CsrWifiMibDecode() Int Decode length:%d != expected:%d", (CsrUint32) decodedLength, (CsrUint32) (4 + length)));
                    break;
                case CSR_WIFI_MIB_TYPE_OCTET:
                    value->value.type = type;
                    decodedLength += CsrWifiMibDecodeOctetString(&buffer[decodedLength], &value->value.u.octetValue);
                    CSR_LOG_TEXT_CONDITIONAL_ERROR(decodedLength != 4 + length, (CSR_WIFI_MIB_LOG_ID, TR_MIB, "CsrWifiMibDecode() Octet Decode length:%d != expected:%d", (CsrUint32) decodedLength, (CsrUint32) (4 + length)));
                    break;
                default:
                    /* This cannot happen */
                    break;
            }
        }
        if (length % 2 == 1)
        {
            /* Remove the padding byte "0x00" in the encoded buffer. The Length value does NOT account for this pad value
             * If the length is an Odd number the Pad values MUST be there if it is Even it will not be.*/
            /*CSR_LOG_TEXT_CRITICAL((CSR_WIFI_MIB_LOG_ID, TR_MIB, "CsrWifiMibDecode(psid:0x%.4X) Odd number of Bytes detected", value->psid));*/
            if (buffer[decodedLength] != 0x00)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_MIB_LOG_ID, TR_MIB, "CsrWifiMibDecode(psid:0x%.4X) Padding Not Detected", value->psid));
            }
            length++;
        }
        return 4 + length;
    }
    return 0;
}

void CsrWifiMibEncodeGetList(CsrWifiDataBlock *buffer, CsrUint16 psidsLength, const CsrWifiMibGetEntry *psids)
{
    CsrSize i;
    buffer->dataLength = 0;
    buffer->data = (CsrUint8 *) CsrPmemAlloc((CsrUint32) (psidsLength * 13)); /* 13 Bytes per get will be loads of space for the max 3 indexes */
    for (i = 0; i < psidsLength; i++)
    {
        buffer->dataLength += (CsrUint16) CsrWifiMibEncodePsidAndIndexs(&buffer->data[buffer->dataLength], &psids[i]);
    }
}

void CsrWifiMibEncodeGet(CsrWifiDataBlock *buffer, CsrUint16 psid, CsrUint16 idx)
{
    /* 13 Bytes per get will be loads of space for the max 3 indexes */
    CsrSize size;
    CsrUint8 tmpBuffer[13];
    CsrWifiMibGetEntry entry;
    CsrMemSet(&entry, 0x00, sizeof(CsrWifiMibGetEntry));
    entry.psid = psid;
    entry.index[0] = idx;
    size = CsrWifiMibEncodePsidAndIndexs(tmpBuffer, &entry);
    CsrWifiMibBufferAppend(buffer, size, tmpBuffer);
}

CsrUint8 *CsrWifiMibFind(CsrWifiDataBlock *buffer, const CsrWifiMibGetEntry *entry)
{
    CsrSize bufferLength = buffer->dataLength;
    CsrUint8 *buff = buffer->data;

    CsrWifiMibLogTextRegister();

    if (bufferLength % 2 == 1)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_MIB_LOG_ID, TR_MIB, "CsrWifiMibFind(bufferLength:%d) bufferLength %% 2 != 0 (Invalid Mib data Detected)", (CsrUint32) bufferLength));
        return NULL;
    }
    while (bufferLength >= 4)
    {
        CsrUint16 psid = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(buff);
        CsrSize length = 4 + CSR_GET_UINT16_FROM_LITTLE_ENDIAN(&buff[2]);
        if (entry->psid == psid)
        {
            CsrSize i;
            CsrUint32 idx;
            CsrSize bytesRead = 0;
            for (i = 0; i < CSR_WIFI_MIB_MAX_INDEXES; i++)
            {
                if (!entry->index[i])
                {
                    return buff;
                }
                bytesRead = CsrWifiMibDecodeUint32(&buff[4 + bytesRead], &idx);
                if (entry->index[i] != idx)
                {
                    break;
                }
            }
            if (i == CSR_WIFI_MIB_MAX_INDEXES)
            {
                return buff;
            }
        }
        if (length % 2 == 1)
        {
            /* Remove the padding byte "0x00" in the encoded buffer. The Length value does NOT account for this pad value
             * If the length is an Odd number the Pad values MUST be there if it is Even it will not be.*/
            /* CSR_LOG_TEXT_CRITICAL((CSR_WIFI_MIB_LOG_ID, TR_MIB, "CsrWifiMibFind(psid:0x%.4X) Odd number of Bytes detected", psid)); */
            length++;
        }

        buff += length;
        bufferLength -= length;
    }

    return NULL;
}

CsrWifiMibValue *CsrWifiMibDecodeGetList(CsrWifiDataBlock *buffer, CsrUint16 psidsLength, const CsrWifiMibGetEntry *psids)
{
    CsrWifiMibValue *results = (CsrWifiMibValue *) CsrPmemAlloc(sizeof(CsrWifiMibValue) * psidsLength);
    CsrSize i;
    CsrWifiMibLogTextRegister();

    CSR_LOG_TEXT_BUFFER_DEBUG((CSR_WIFI_MIB_LOG_ID, TR_MIB, buffer->dataLength, buffer->data, "CsrWifiMibDecodeGetList()"));
    for (i = 0; i < psidsLength; i++)
    {
        CsrWifiMibEntry value;
        CsrUint8 *psidPtr = CsrWifiMibFind(buffer, &psids[i]);
        if (psidPtr)
        {
            value.psid = psids[i].psid;
            CsrMemCpy(value.index, psids[i].index, sizeof(value.index));
            (void) CsrWifiMibDecode(psidPtr, &value);

            results[i] = value.value;
        }
        else
        {
            CSR_LOG_TEXT_ERROR((CSR_WIFI_MIB_LOG_ID, TR_MIB, "CsrWifiMibDecodeGetList() Could not find get value psid:0x%.4X ", psids[i].psid));
            results[i].type = CSR_WIFI_MIB_TYPE_NONE;
        }
    }

    return results;
}

CsrResult CsrWifiMibEncodeBool(CsrWifiDataBlock *buffer, CsrUint16 psid, CsrBool value, CsrUint16 idx)
{
    CsrWifiMibEntry v;
    CsrMemSet(&v, 0x00, sizeof(CsrWifiMibEntry));
    v.psid = psid;
    v.index[0] = idx;
    v.value.type = CSR_WIFI_MIB_TYPE_BOOL;
    v.value.u.boolValue = value;
    return CsrWifiMibEncode(buffer, &v);
}

CsrResult CsrWifiMibEncodeInt(CsrWifiDataBlock *buffer, CsrUint16 psid, CsrInt32 value, CsrUint16 idx)
{
    CsrWifiMibEntry v;
    CsrMemSet(&v, 0x00, sizeof(CsrWifiMibEntry));
    v.psid = psid;
    v.index[0] = idx;
    v.value.type = CSR_WIFI_MIB_TYPE_INT;
    v.value.u.intValue = value;
    return CsrWifiMibEncode(buffer, &v);
}

CsrResult CsrWifiMibEncodeUint(CsrWifiDataBlock *buffer, CsrUint16 psid, CsrUint32 value, CsrUint16 idx)
{
    CsrWifiMibEntry v;
    CsrMemSet(&v, 0x00, sizeof(CsrWifiMibEntry));
    v.psid = psid;
    v.index[0] = idx;
    v.value.type = CSR_WIFI_MIB_TYPE_UINT;
    v.value.u.uintValue = value;
    return CsrWifiMibEncode(buffer, &v);
}

CsrResult CsrWifiMibEncodeOctet(CsrWifiDataBlock *buffer, CsrUint16 psid, CsrSize dataLength, const CsrUint8 *data, CsrUint16 idx)
{
    CsrWifiMibEntry v;
    CsrMemSet(&v, 0x00, sizeof(CsrWifiMibEntry));
    v.psid = psid;
    v.index[0] = idx;
    v.value.type = CSR_WIFI_MIB_TYPE_OCTET;
    v.value.u.octetValue.dataLength = (CsrUint32) dataLength;
    v.value.u.octetValue.data = (CsrUint8 *) data;
    return CsrWifiMibEncode(buffer, &v);
}
