/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#ifdef PS_USPACE
#include "csr_wifi_ps_types.h"
#else
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip_card.h"
#include "../hal/csr_wifi_hip_hal_priv.h"
#endif
#include "csr_wifi_ps_q.h"
#include "csr_wifi_ps_circ.h"
#include "csr_wifi_ps_qsig.h"

#include "csr_wifi_ps_if.h"
#include "csr_wifi_ps.h"
#include "csr_wifi_hip_core_trace.h"

#ifdef PS_USPACE
#include "csr_wifi_ps_card.h"
#endif

static CsrResult csrWifiHipPacketSchedulerQueueSetDisassociateUnprotected(card_t *card, CsrInt32 vif_id, CsrInt32 aid);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerVIFAssociate
 *
 *  Create a VIF structure in the Packet Scheduler.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerVIFAssociate(card_t *card, CsrInt32 vif_id, CsrWifiHipPacketSchedulerDiscardCb discard_cb)
{
    CsrInt32 index;
    CsrWifiHipPacketSchedulerVif *vif;
    CsrWifiHipPacketSchedulerMgmtQueue *mgmt_q;
    CsrWifiHipPacketSchedulerMulticastQueue *multicast_q;
    CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;
    CsrWifiHipPacketSchedulerRtxQueue *rtx_q;

    if ((vif_id < PS_MIN_VIF_ID) || (vif_id > PS_MAX_VIF_ID))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFAssociate: vif_id %d out of range\n",
                            card->instance,  vif_id));
        return CSR_RESULT_FAILURE;
    }
    index = MAP_VIFID_TO_INDEX(vif_id);

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFAssociate: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    if (card->fh_buffer_d.packet_scheduler.vifs[index])
    {
        /* already exists */
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFAssociate: vif_id %d already exists\n",
                            card->instance,  vif_id));
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    vif = (CsrWifiHipPacketSchedulerVif *) CsrMemAlloc(sizeof(CsrWifiHipPacketSchedulerVif));
    card->fh_buffer_d.packet_scheduler.vifs[index] = vif;
    if (!vif)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFAssociate:  vif_id %d failed to allocate memory\n", card->instance,  vif_id));
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }

    CsrMemSet((void *) vif, 0, sizeof(CsrWifiHipPacketSchedulerVif));
    vif->discard_cb = discard_cb;
    vif->mcast_complete_cb = 0;

    pending_clr_q = &(vif->pending_clr_q);
    pending_clr_q->common.dequeue = csrWifiHipPacketSchedulerPendingClrDequeue;
    pending_clr_q->common.peek = csrWifiHipPacketSchedulerPendingClrPeek;
    pending_clr_q->common.getlen = csrWifiHipPacketSchedulerPendingClrGetLen;
    pending_clr_q->common.enqueue = csrWifiHipPacketSchedulerPendingClrEnqueue;
    pending_clr_q->common.free = 0;
    pending_clr_q->common.qmod = PS_DEFAULT_QMOD;
    pending_clr_q->common.qcod = 0;
    pending_clr_q->common.vif = vif;
    pending_clr_q->common.qs = 0;
    pending_clr_q->common.pending_clr_q = 0;
    csrWifiHipPacketSchedulerListInit(&(pending_clr_q->pkts));
    pending_clr_q->common.list = 0;

    mgmt_q = &(vif->mgmt_q);
    mgmt_q->common.dequeue = csrWifiHipPacketSchedulerMgmtDequeue;
    mgmt_q->common.peek = csrWifiHipPacketSchedulerMgmtPeek;
    mgmt_q->common.getlen = csrWifiHipPacketSchedulerMgmtGetLen;
    mgmt_q->common.enqueue = csrWifiHipPacketSchedulerMgmtEnqueue;
    mgmt_q->common.free = csrWifiHipPacketSchedulerCommonFree;
    mgmt_q->common.qmod = PS_DEFAULT_QMOD;
    mgmt_q->common.list = &(card->fh_buffer_d.packet_scheduler.sp[1]);
    mgmt_q->common.vif = vif;
    mgmt_q->common.qs = 0;
    mgmt_q->common.pending_clr_q = pending_clr_q;
    ps_circ_init((&(mgmt_q->pkts)), UNIFI_QUEUE_LEN);
    csrWifiHipPacketSchedulerEnableQueue(&(mgmt_q->common));

    multicast_q = &(vif->multicast_q);
    multicast_q->common.dequeue = csrWifiHipPacketSchedulerMulticastDequeue;
    multicast_q->common.peek = csrWifiHipPacketSchedulerMulticastPeek;
    multicast_q->common.getlen = csrWifiHipPacketSchedulerMulticastGetLen;
    multicast_q->common.enqueue = csrWifiHipPacketSchedulerMulticastEnqueue;
    multicast_q->common.free = csrWifiHipPacketSchedulerCommonFree;
    multicast_q->common.qmod = 1;
    multicast_q->common.list = &(card->fh_buffer_d.packet_scheduler.sp[2]);
    multicast_q->common.vif = vif;
    multicast_q->common.qs = 0;
    multicast_q->burstMaximum = multicast_q->burstCurrent = 0;
    multicast_q->common.pending_clr_q = pending_clr_q;
    ps_circ_init((&(multicast_q->pkts)), UNIFI_QUEUE_LEN);

    rtx_q = &(vif->rtx_q);
    rtx_q->pri = 1;
    rtx_q->common.dequeue = csrWifiHipPacketSchedulerRtxDequeue;
    rtx_q->common.peek = csrWifiHipPacketSchedulerRtxPeek;
    rtx_q->common.getlen = csrWifiHipPacketSchedulerRtxGetLen;
    rtx_q->common.enqueue = csrWifiHipPacketSchedulerRtxEnqueue;
    rtx_q->common.free = 0;
    rtx_q->common.list = &(card->fh_buffer_d.packet_scheduler.sp[rtx_q->pri]);
    rtx_q->common.qmod = PS_DEFAULT_QMOD;
    rtx_q->common.qcod = 0;
    rtx_q->common.vif = vif;
    rtx_q->common.qs = 0;
    rtx_q->common.pending_clr_q = pending_clr_q;
    csrWifiHipPacketSchedulerListInit(&(rtx_q->pkts));

    #ifdef PS_USPACE
    csrWifiHipPacketSchedulerListInit(&(multicast_q->sent_pkts));
    #endif

    /* Make available to upper layer. */
    vif->id = vif_id;

    CSR_LOG_TEXT_INFO((
                          CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFAssociate: VIF ID (%d) created\n",
                          card->instance, vif_id));

    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerVIFDisassociate
 *
 *  Destroy a VIF structure (all all associated Queue Set structures).
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerVIFDisassociate(card_t *card, CsrInt32 vif_id)
{
    CsrWifiHipPacketSchedulerMgmtQueue *mgmt_q;
    CsrWifiHipPacketSchedulerMulticastQueue *multicast_q;
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipPacketSchedulerVif *vif;
    CsrInt32 i;

    if ((vif_id < PS_MIN_VIF_ID) || (vif_id > PS_MAX_VIF_ID))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFDisassociate: VIF ID (%d) out of range\n",
                            card->instance, vif_id));
        return CSR_RESULT_FAILURE;
    }

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFDisassociate: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    vif = card->fh_buffer_d.packet_scheduler.vifs[MAP_VIFID_TO_INDEX(vif_id)];
    if (!vif)
    {
        CSR_LOG_TEXT_INFO((
                              CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFDisassociate: VIF ID (%d) instance does not exist\n",
                              card->instance, vif_id));
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    mgmt_q = &(vif->mgmt_q);
    /* Remove management queue for this VIF from the SP list. */
    csrWifiHipPacketSchedulerDisableQueue(&(mgmt_q->common));
    /* Free all signals in the management queue. */
    csrWifiHipPacketSchedulerPurgeQueue(card, &(mgmt_q->common));

    vif->mcast_complete_cb = 0;
    /* Remove the multicast queue for this VIF from the SP list. */
    multicast_q = &(vif->multicast_q);
    csrWifiHipPacketSchedulerDisableQueue(&(multicast_q->common));
    /* Free all signals in the multicast queue. */
    csrWifiHipPacketSchedulerPurgeQueue(card, &(multicast_q->common));
    /* Free all signals in the pending clear queue. */
    csrWifiHipPacketSchedulerPendingClrPurge(card, &(vif->pending_clr_q));
    /* Free all signals held on the VIF rtx queue. */
    csrWifiHipPacketSchedulerDisableQueue(&(vif->rtx_q.common));
    csrWifiHipPacketSchedulerRtxPurge(card, &(vif->rtx_q));

    /* Free all Queue Sets for this VIF. */
    for (i = 0; i < PS_MAX_STA_PER_AP; i++)
    {
        qs = vif->qs[i];
        if (qs)
        {
            (void) csrWifiHipPacketSchedulerQueueSetDisassociateUnprotected(card, vif_id, qs->id);
        }
    }

    /* Only free the VIF memory after all Queue Sets have been destroyed. */
    CsrMemFree(card->fh_buffer_d.packet_scheduler.vifs[MAP_VIFID_TO_INDEX(vif_id)]);

    /* Setting vif pointer to zero will stop more packets being put onto queues. */
    card->fh_buffer_d.packet_scheduler.vifs[MAP_VIFID_TO_INDEX(vif_id)] = 0;

    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);

    CSR_LOG_TEXT_INFO((
                          CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFDisassociate: VIF ID (%d) destroyed\n",
                          card->instance, vif_id));

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetAssociate
 *
 *  Create a Queue Set structure in the Packet Scheduler.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *      pri          Queue Set priority (0 for high, 1 for normal).
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetAssociate(card_t *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerQSPri pri, CsrInt32 qlen, CsrWifiHipFlowControlPauseCb pauseCb, CsrWifiHipFlowControlResumeCb resumeCb, void *pauseResumeContext)
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipPacketSchedulerQsMgmtQueue *mgmt_q;
    CsrWifiHipPacketSchedulerQsMgmtQueue *auth_q;
    CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;
    CsrInt32 ac;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetAssociate: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    vif = csrWifiHipPacketSchedulerGetVif(card, vif_id);
    if (!vif)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetAssociate: VIF ID (%d) instance does not exist\n",
                            card->instance, vif_id));
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    if ((aid < PS_MIN_AID) || (aid > PS_MAX_AID))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetAssociate: AID (%d) out of range\n",
                            card->instance, aid));
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    if ((pri != QS_PRI_HIGH) && (pri != QS_PRI_NORMAL))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetAssociate: pri (%d) out of range\n",
                            card->instance, pri));
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    if (vif->qs[MAP_AID_TO_INDEX(aid)])
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetAssociate: Queue Set VIF %d AID %d already exists\n",
                            card->instance, vif_id, aid));
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    qs = (CsrWifiHipPacketSchedulerQueueSet *) CsrMemAlloc(sizeof(*qs));
    if (!qs)
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    CsrMemSet((void *) qs, 0, sizeof(*qs));
    qs->vif = (void *) vif;
    qs->id = aid;
    qs->pri = pri;
    qs->smod = 4;
    vif->qs[MAP_AID_TO_INDEX(aid)] = qs;

    qs->max_queue_len = qlen;
    qs->resumeCb = resumeCb;
    qs->pauseCb = pauseCb;
    qs->pauseResumeContext = pauseResumeContext;

    qs->station_state = QS_STA_ACTIVE;

    qs->authq_state = QS_AUTHQ_DISABLED;
    qs->acq_state = QS_ACQ_DISABLED;

    pending_clr_q = &(qs->pending_clr_q);
    pending_clr_q->common.dequeue = csrWifiHipPacketSchedulerPendingClrDequeue;
    pending_clr_q->common.peek = csrWifiHipPacketSchedulerPendingClrPeek;
    pending_clr_q->common.getlen = csrWifiHipPacketSchedulerPendingClrGetLen;
    pending_clr_q->common.enqueue = csrWifiHipPacketSchedulerPendingClrEnqueue;
    pending_clr_q->common.free = 0;
    pending_clr_q->common.qmod = PS_DEFAULT_QMOD;
    pending_clr_q->common.qcod = 0;
    pending_clr_q->common.vif = vif;
    pending_clr_q->common.qs = qs;
    pending_clr_q->common.pending_clr_q = 0;
    csrWifiHipPacketSchedulerListInit(&(pending_clr_q->pkts));
    pending_clr_q->common.list = 0;

    mgmt_q = &(qs->mgmt_q);
    mgmt_q->common.dequeue = csrWifiHipPacketSchedulerQsMgmtDequeue;
    mgmt_q->common.peek = csrWifiHipPacketSchedulerQsMgmtPeek;
    mgmt_q->common.getlen = csrWifiHipPacketSchedulerQsMgmtGetLen;
    mgmt_q->common.enqueue = csrWifiHipPacketSchedulerQsMgmtEnqueue;
    mgmt_q->common.free = csrWifiHipPacketSchedulerQsMgmtEnqueueFree;
    mgmt_q->common.qmod = PS_DEFAULT_QMOD;
    mgmt_q->common.qcod = 0;
    mgmt_q->common.vif = vif;
    mgmt_q->common.qs = qs;
    mgmt_q->common.pending_clr_q = pending_clr_q;
    ps_circ_init((&(mgmt_q->pkts)), UNIFI_QUEUE_LEN);
    mgmt_q->common.list = &(card->fh_buffer_d.packet_scheduler.sp[1]);

    auth_q = &(qs->auth_q);
    auth_q->common.dequeue = csrWifiHipPacketSchedulerQsMgmtDequeue;
    auth_q->common.peek = csrWifiHipPacketSchedulerQsMgmtPeek;
    auth_q->common.getlen = csrWifiHipPacketSchedulerQsMgmtGetLen;
    auth_q->common.enqueue = csrWifiHipPacketSchedulerQsMgmtEnqueue;
    auth_q->common.free = csrWifiHipPacketSchedulerQsMgmtEnqueueFree;
    auth_q->common.qmod = PS_DEFAULT_QMOD;
    auth_q->common.qcod = 0;
    auth_q->common.vif = vif;
    auth_q->common.qs = qs;
    auth_q->common.pending_clr_q = pending_clr_q;
    ps_circ_init((&(auth_q->pkts)), UNIFI_QUEUE_LEN);
    auth_q->common.list = &(card->fh_buffer_d.packet_scheduler.sp[1]);

    for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
    {
        CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
        CsrWifiHipPacketSchedulerRtxQueue *rtx_q;

        ac_q = &qs->ac[ac];
        ac_q->pri = ac;
        ac_q->common.qs = qs;

        ac_q->common.dequeue = csrWifiHipPacketSchedulerAcDequeue;
        ac_q->common.peek = csrWifiHipPacketSchedulerAcPeek;
        ac_q->common.getlen = csrWifiHipPacketSchedulerAcGetLen;
        /* To do: Use VO aware enqueue function that can cause a frame to be discarded. */
        ac_q->common.enqueue = csrWifiHipPacketSchedulerAcEnqueue;
        ac_q->common.free = csrWifiHipPacketSchedulerAcFree;
        ac_q->common.list = &(card->fh_buffer_d.packet_scheduler.fairness[qs->pri - 1][ac]);
        ac_q->common.qmod = PS_DEFAULT_QMOD;
        ac_q->common.qcod = 0;
        ac_q->common.vif = vif;
        ac_q->state = AC_QSTATE_NOT_FULL;
        ac_q->common.pending_clr_q = pending_clr_q;
        ps_circ_init((&(ac_q->pkts)), UNIFI_QUEUE_LEN);

        rtx_q = &qs->retransmit[ac];
        rtx_q->pri = ac;
        rtx_q->common.qs = qs;

        rtx_q->common.dequeue = csrWifiHipPacketSchedulerRtxDequeue;
        rtx_q->common.peek = csrWifiHipPacketSchedulerRtxPeek;
        rtx_q->common.getlen = csrWifiHipPacketSchedulerRtxGetLen;
        rtx_q->common.enqueue = csrWifiHipPacketSchedulerRtxEnqueue;
        rtx_q->common.free = 0;
        rtx_q->common.list = &(card->fh_buffer_d.packet_scheduler.sp_retransmit[ac]);
        rtx_q->common.qmod = PS_DEFAULT_QMOD;
        rtx_q->common.qcod = 0;
        rtx_q->common.vif = vif;
        rtx_q->common.pending_clr_q = pending_clr_q;
        csrWifiHipPacketSchedulerListInit(&(rtx_q->pkts));

        csrWifiHipPacketSchedulerListInit(&(qs->hal_queues[ac]));
    }
    CSR_LOG_TEXT_INFO((
                          CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetAssociate: VIF %d QS %d created\n",
                          card->instance, vif_id, aid));

    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerQueueSetDisassociateUnprotected
 *
 *  Destroy a Queue Set structure in the Packet Scheduler.
 *  This function does not take the mutex. The calling function
 *  must take the cfg mutex.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
static CsrResult csrWifiHipPacketSchedulerQueueSetDisassociateUnprotected(card_t *card, CsrInt32 vif_id, CsrInt32 aid)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipPacketSchedulerVif *vif;
    CsrWifiHipPacketSchedulerQsMgmtQueue *mgmt_q;
    CsrWifiHipPacketSchedulerQsMgmtQueue *auth_q;
    CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;

    CsrInt32 ac;

    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetDisassociate: QS (VIF %d AID %d) instance does not exist\n",
                            card->instance, vif_id, aid));
        return CSR_RESULT_FAILURE;
    }

    vif = (CsrWifiHipPacketSchedulerVif *) qs->vif;
    if (!vif)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetDisassociate: VIF linkage error (VIF %d AID %d)\n",
                            card->instance, vif_id, aid));
        return CSR_RESULT_FAILURE;
    }
    /* Disconnect from upper layer i/f. */
    vif->qs[MAP_AID_TO_INDEX(qs->id)] = 0;

    /* Update the counters maintained in the top level packet scheduler structure. */
    if (qs->station_state == QS_STA_POWERSAVE)
    {
        card->fh_buffer_d.packet_scheduler.powersave_qs--;
        qs->vif->powersave_qs--;
    }
    card->fh_buffer_d.packet_scheduler.active_qs--;
    qs->vif->active_qs--;

    mgmt_q = &(qs->mgmt_q);
    /* Remove management queue for this VIF from the SP list. */
    csrWifiHipPacketSchedulerDisableQueue(&(mgmt_q->common));
    /* Free all signals in the management queue. */
    csrWifiHipPacketSchedulerPurgeQueue(card, &(mgmt_q->common));

    auth_q = &(qs->auth_q);
    /* Remove auth queue for this VIF from the SP list. */
    csrWifiHipPacketSchedulerDisableQueue(&(auth_q->common));
    /* Free all signals in the auth queue. */
    csrWifiHipPacketSchedulerPurgeQueue(card, &(auth_q->common));
    /*
     * Reset the flow control callback to zero to prevent the
     * o/s queue from being re-enabled as all the packets are
     * dequeued from it.
     */
    qs->pauseCb = NULL;
    qs->resumeCb = NULL;
    qs->pauseResumeContext = NULL;

    if (qs->max_queue_len)
    {
        for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
        {
            CsrInt32 i, len;
            CsrWifiHipPacketSchedulerQsig *qsig;

            len = qs->hal_queues[ac].entries; /* < qs->max_queue_len ? qs->hal_queues[ac].entries : qs->max_queue_len;*/
            for (i = 0; i < len; i++)
            {
                qsig = (CsrWifiHipPacketSchedulerQsig *) csrWifiHipPacketSchedulerListTakeFirst(&(qs->hal_queues[ac]));
                if (!qsig)
                {
                    CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                                        CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetDisassociate: !qsig after %d processed (len=%d)\n", card->instance, i, len));
                    break;
                }
                if (vif->discard_cb && csrWifiHipPacketSchedulerCfmRqd(qsig))
                {
                    vif->discard_cb(card->ospriv, vif->id, qsig_get_signal_ptr(qsig));
                }
                csrWifiHipPacketSchedulerFreeSignal(card, qsig);
            }
        }
    }

    for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
    {
        CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
        CsrWifiHipPacketSchedulerRtxQueue *rtx_q;

        ac_q = &qs->ac[ac];
        /* Remove queue from scheduler list. */
        csrWifiHipPacketSchedulerDisableQueue(&(ac_q->common));
        csrWifiHipPacketSchedulerPurgeQueue(card, &(ac_q->common));

        rtx_q = &qs->retransmit[ac];
        /* Remove queue from scheduler list. */
        csrWifiHipPacketSchedulerDisableQueue(&(rtx_q->common));
        csrWifiHipPacketSchedulerRtxPurge(card, rtx_q);
    }

    /* Free all signals in the pending clear queue. */
    pending_clr_q = &(qs->pending_clr_q);
    csrWifiHipPacketSchedulerPendingClrPurge(card, pending_clr_q);

    CsrMemFree(qs);

    /* Redistribute the total SMOD amongst the remaining Queue Sets. */
    csrWifiHipPacketSchedulerDistributeSMod(card);

    CSR_LOG_TEXT_INFO((
                          CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetDisassociate: QS instance destroy (VIF %d AID %d)\n",
                          card->instance, vif_id, aid));

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetDisassociate
 *
 *  Destroy a Queue Set structure in the Packet Scheduler.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetDisassociate(card_t *card, CsrInt32 vif_id, CsrInt32 aid)
{
    CsrResult result;
    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetDisassociate: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    result = csrWifiHipPacketSchedulerQueueSetDisassociateUnprotected(card, vif_id, aid);
    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return result;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetActivate
 *
 *  Activate a Queue Set structure in the Packet Scheduler.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetActivate(card_t *card, CsrInt32 vif_id, CsrInt32 aid)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetActivate: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "CsrWifiHipPacketSchedulerQueueSetActivate: QS (VIF %d AID %d) instance does not exist\n", vif_id, aid));
        return CSR_RESULT_FAILURE;
    }
    card->fh_buffer_d.packet_scheduler.active_qs++;
    qs->vif->active_qs++;
    csrWifiHipPacketSchedulerDistributeSMod(card);
    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerVIFACRelativePriorities
 *
 *
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      rpri
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerVIFACRelativePriorities(card_t *card, CsrInt32 vif_id, CsrInt32 rpri[])
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrInt32 tokenBucket, i;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFACRelativePriorities: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    vif = card->fh_buffer_d.packet_scheduler.vifs[MAP_VIFID_TO_INDEX(vif_id)];
    if (!vif)
    {
        CSR_LOG_TEXT_INFO((
                              CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFACRelativePriorities: VIF ID (%d) instance does not exist\n",
                              card->instance, vif_id));
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    for (i = 0; i < PS_MAX_STA_PER_AP; i++)
    {
        qs = vif->qs[i];
        if (qs)
        {
            (void) csrWifiHipPacketSchedulerDisableQueueSet(qs);
            for (tokenBucket = 0; tokenBucket < PS_MAX_FAIRNESS_PRI; tokenBucket++)
            {
                CsrWifiHipPacketSchedulerPriorityQueue *ac_q;

                ac_q = &qs->ac[rpri[tokenBucket]];

                CSR_LOG_TEXT_INFO((
                                      CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerVIFACRelativePriorities: Putting queue %d into token bucket %d\n",
                                      card->instance, rpri[tokenBucket], tokenBucket));
                ac_q->common.list = &(card->fh_buffer_d.packet_scheduler.fairness[qs->pri - 1][tokenBucket]);
            }
            (void) csrWifiHipPacketSchedulerEnableQueueSet(qs);
        }
    }
    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetPriority
 *
 *  Change the Queue Set priority.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *      pri          Queue Set priority (0 for high, 1 for normal).
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetPriority(card_t *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerQSPri pri)
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
    CsrInt32 ac;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetPriority: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    vif = csrWifiHipPacketSchedulerGetVif(card, vif_id);
    if (!vif)
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }

    if ((aid < PS_MIN_AID) || (aid > PS_MAX_AID))
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }

    if ((pri != QS_PRI_HIGH) && (pri != QS_PRI_NORMAL))
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }

    qs = vif->qs[MAP_AID_TO_INDEX(aid)];
    if (!qs)
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }

    if (qs->pri == pri)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetPriority: selected priority (%d) already set\n",
                            card->instance, pri));
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
    {
        ac_q = &qs->ac[ac];
        csrWifiHipPacketSchedulerDisableQueue(&(ac_q->common));
    }

    qs->pri = pri;

    for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
    {
        ac_q = &qs->ac[ac];
        ac_q->common.list = &(card->fh_buffer_d.packet_scheduler.fairness[qs->pri - 1][ac]);

        csrWifiHipPacketSchedulerEnableQueue(&(ac_q->common));
    }

    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetSMod
 *
 *  Change the Queue Set SMOD.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *      smod         Queue Set SMOD.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetSMod(card_t *card, CsrInt32 vif_id, CsrInt32 aid, CsrInt32 smod)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetSMod: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    csrWifiHipPacketSchedulerQueueSetSModHelper(qs, smod);
    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueACQMod
 *
 *  Change the queue QMOD.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *      qmod         Queue QMOD.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueACQMod(card_t *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerACPri prio, CsrInt32 qmod)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetSMod: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: PacketSchedulerQueueACQMod: unable to get VIF %d QS id %d\n",
                            card->instance,  vif_id, aid));
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    if ((prio != AC_PRI_0) && (prio != AC_PRI_1) && (prio != AC_PRI_2) && (prio != AC_PRI_0))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: PacketSchedulerQueueACQMod: priority (%d) out of range (%d..%d)\n",
                            card->instance,  prio, AC_PRI_0, AC_PRI_3));
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    qs->ac[prio].common.qmod = qmod;
    if (MOD_TO_BYTES(qmod) <= qs->ac[prio].common.qcod)
    {
        if (qs->ac[prio].common.active)
        {
            csrWifiHipPacketSchedulerDisableQueue(&qs->ac[prio].common);
        }
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_SUCCESS;
    }
    if (!qs->ac[prio].common.active)
    {
        csrWifiHipPacketSchedulerEnableQueue(&qs->ac[prio].common);
    }
    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetAuthQState
 *
 *  Change the Queue Set auth queue state
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *      state        enable/disable the auth queue (mirrors the 802.1x un controlled port entity)
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetAuthQState(card_t *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerQSAuthQState state)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetAuthQState: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "CsrWifiHipPacketSchedulerQueueSetAuthQState: QS (VIF %d AID %d) instance does not exist\n", vif_id, aid));
        return CSR_RESULT_FAILURE;
    }
    qs->authq_state = state;
    if (state == QS_AUTHQ_ENABLED)
    {
        csrWifiHipPacketSchedulerEnableQueue(&(qs->auth_q.common));
    }
    else
    {
        csrWifiHipPacketSchedulerDisableQueue(&(qs->auth_q.common));
    }
    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetACQState
 *
 *  Change the Queue Set AC queue state
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *      state        enable/disable the AC queues (mirrors the 802.1x controlled port entity)
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetACQState(card_t *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerQSACQState state)
{
    CsrInt32 ac;
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
    CsrWifiHipPacketSchedulerQueueSet *qs;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetACQState: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "CsrWifiHipPacketSchedulerQueueSetACQState: QS (VIF %d AID %d) instance does not exist\n", vif_id, aid));
        return CSR_RESULT_FAILURE;
    }
    qs->acq_state = state;
    if (state == QS_ACQ_ENABLED)
    {
        for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
        {
            ac_q = &qs->ac[ac];
            csrWifiHipPacketSchedulerEnableQueue(&ac_q->common);
        }
    }
    else
    {
        for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
        {
            ac_q = &qs->ac[ac];
            csrWifiHipPacketSchedulerDisableQueue(&ac_q->common);
        }
    }
    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueMulticastBurstEnable
 *
 *  If enabled, set the burst maximum for the queue to defined value and
 *  set the QMOD for the queue to 1. If disabled, set the burst maximum
 *  to zero and the QMOD to the defined QMOD for a multicast queue.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      enable       Flag: not zero to enable, 0 to disable
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueMulticastBurstEnable(card_t *card, CsrInt32 vif_id, CsrInt32 enable)
{
    CsrWifiHipPacketSchedulerVif *vif;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueMulticastBurstMaximum: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    vif = csrWifiHipPacketSchedulerGetVif(card, vif_id);
    if (!vif)
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    if (enable)
    {
        vif->multicast_q.burstMaximum = PS_MCAST_BURST_MAX;
        vif->multicast_q.common.qmod = 1;
    }
    else
    {
        vif->multicast_q.burstMaximum = 0;
        vif->multicast_q.common.qmod = PS_MCAST_QMOD;
    }

    if (MOD_TO_BYTES(vif->multicast_q.common.qmod) <= vif->multicast_q.common.qcod)
    {
        if (vif->multicast_q.common.active)
        {
            csrWifiHipPacketSchedulerDisableQueue(&vif->multicast_q.common);
        }
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_SUCCESS;
    }
    if (!vif->multicast_q.common.active)
    {
        csrWifiHipPacketSchedulerEnableQueue(&vif->multicast_q.common);
    }

    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetBurstMaximum
 *
 *  Change the Queue Set burst maximum.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *      bm           Queue Set burst maximum.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetBurstMaximum(card_t *card, CsrInt32 vif_id, CsrInt32 aid, CsrInt32 bm)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueMulticastBurstMaximum: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        return CSR_RESULT_FAILURE;
    }
    qs->burstMaximum = bm;
    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetAuthQPurge
 *
 *  Free all packets in the Queue Set auth queue only.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetAuthQPurge(card_t *card, CsrInt32 vif_id, CsrInt32 aid)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipPacketSchedulerQsMgmtQueue *auth_q;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetAuthQPurge: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "CsrWifiHipPacketSchedulerQueueSetAuthQPurge: QS (VIF %d AID %d) instance does not exist\n", vif_id, aid));
        return CSR_RESULT_FAILURE;
    }

    auth_q = &(qs->auth_q);
    /* Free all signals in the auth queue. */
    csrWifiHipPacketSchedulerDisableQueue(&(auth_q->common));
    csrWifiHipPacketSchedulerPurgeQueue(card, &(auth_q->common));
    csrWifiHipPacketSchedulerEnableQueue(&(auth_q->common));

    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetACQPurge
 *
 *  Free all packets in the Queue Set AC queues only.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetACQPurge(card_t *card, CsrInt32 vif_id, CsrInt32 aid)
{
    CsrInt32 ac;
    CsrWifiHipPacketSchedulerQueueSet *qs;

    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerQueueSetACQPurge: CsrMutexLock failed\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "CsrWifiHipPacketSchedulerQueueSetACQPurge: QS (VIF %d AID %d) instance does not exist\n", vif_id, aid));
        return CSR_RESULT_FAILURE;
    }

    for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
    {
        CsrWifiHipPacketSchedulerPriorityQueue *ac_q;

        ac_q = &qs->ac[ac];
        csrWifiHipPacketSchedulerDisableQueue(&(ac_q->common));
        csrWifiHipPacketSchedulerPurgeQueue(card, &(ac_q->common));
        csrWifiHipPacketSchedulerEnableQueue(&(ac_q->common));
    }

    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQSigCount
 *
 *  Get current number of free QSIG structures.
 *
 *  Arguments:
 *      hipHandle    Pointer to priv structure.
 *
 *  Returns:
 *      CsrInt32     Number of current free QSIG structures.
 *
 * ---------------------------------------------------------------------------
 */
CsrInt32 CsrWifiHipPacketSchedulerQSigCount(void *hipHandle)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    return csrWifiHipPacketSchedulerFreeSignalCount(priv->card);
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerCtrlStats
 *
 *  Get control queue stats.
 *
 *  Arguments:
 *      hipHandle    Pointer to priv structure.
 *      stats        Pointer to stats structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerCtrlStats(void *hipHandle, CsrWifiHipPacketSchedulerStats *stats)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    (*stats) = priv->card->fh_buffer_d.packet_scheduler.ctrl_q.common.stats;
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerMgmtStats
 *
 *  Get VIF management queue stats.
 *
 *  Arguments:
 *      hipHandle    Pointer to priv structure.
 *      stats        Pointer to stats structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerMgmtStats(void *hipHandle, CsrInt32 vif_id, CsrWifiHipPacketSchedulerStats *stats)
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    vif = csrWifiHipPacketSchedulerGetVif(priv->card, vif_id);
    if (!vif)
    {
        return CSR_RESULT_FAILURE;
    }
    (*stats) = vif->mgmt_q.common.stats;

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerMulticastStats
 *
 *  Get VIF multicast queue stats.
 *
 *  Arguments:
 *      hipHandle    Pointer to priv structure.
 *      stats        Pointer to stats structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerMulticastStats(void *hipHandle, CsrInt32 vif_id, CsrWifiHipPacketSchedulerStats *stats)
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    vif = csrWifiHipPacketSchedulerGetVif(priv->card, vif_id);
    if (!vif)
    {
        return CSR_RESULT_FAILURE;
    }
    (*stats) = vif->multicast_q.common.stats;

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetACStats
 *
 *  Get Queue Set ac queue stats.
 *
 *  Arguments:
 *      hipHandle    Pointer to priv structure.
 *      stats        Pointer to stats structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetACStats(void *hipHandle, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerACPri ac, CsrWifiHipPacketSchedulerStats *stats, CsrWifiHipPacketSchedulerQSACQState *acq_state)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    if ((ac != AC_PRI_0) && (ac != AC_PRI_1) && (ac != AC_PRI_2) && (ac != AC_PRI_3))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: PacketSchedulerQueueSetACStats: priority (%d) out of range (%d..%d)\n",
                            priv->card->instance,  ac, AC_PRI_0, AC_PRI_3));
        return CSR_RESULT_FAILURE;
    }
    qs = csrWifiHipPacketSchedulerGetQueueSet(priv->card, vif_id, aid);
    if (!qs)
    {
        return CSR_RESULT_FAILURE;
    }
    ac_q = &qs->ac[ac];
    (*stats) = ac_q->common.stats;
    (*acq_state) = qs->acq_state;
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetMgmtStats
 *
 *  Get VIF management queue stats.
 *
 *  Arguments:
 *      hipHandle    Pointer to priv structure.
 *      stats        Pointer to stats structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetMgmtStats(void *hipHandle, CsrInt32 vif_id,  CsrInt32 aid, CsrWifiHipPacketSchedulerStats *stats)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    qs = csrWifiHipPacketSchedulerGetQueueSet(priv->card, vif_id, aid);
    if (!qs)
    {
        return CSR_RESULT_FAILURE;
    }
    (*stats) = qs->mgmt_q.common.stats;
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetAuthStats
 *
 *  Get Queue set auth queue stats.
 *
 *  Arguments:
 *      hipHandle    Pointer to priv structure.
 *      stats        Pointer to stats structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetAuthStats(void *hipHandle, CsrInt32 vif_id,  CsrInt32 aid, CsrWifiHipPacketSchedulerStats *stats)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    qs = csrWifiHipPacketSchedulerGetQueueSet(priv->card, vif_id, aid);
    if (!qs)
    {
        return CSR_RESULT_FAILURE;
    }
    (*stats) = qs->auth_q.common.stats;
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerDwrrStatsGet
 *
 *  Get DWRR stats for a specific AC.
 *
 *  Arguments:
 *      hipHandle    Pointer to priv structure.
 *      ac           AC value.
 *      stats        Pointer to stats structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerDwrrStatsGet(void *hipHandle, CsrWifiHipPacketSchedulerQSPri pri, CsrWifiHipPacketSchedulerACPri ac, CsrWifiHipPacketSchedulerDwrrStats *stats)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrUint32 index;

    if (pri == QS_PRI_HIGH)
    {
        index = PS_QS_PRI_HI_INDEX;
    }
    else if (pri == QS_PRI_NORMAL)
    {
        index = PS_QS_PRI_NORM_INDEX;
    }
    else
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "CsrWifiHipPacketSchedulerDwrrStatsGet: QS priority (%d) out of range\n", pri));
        return CSR_RESULT_FAILURE;
    }

    stats->missed = priv->card->fh_buffer_d.packet_scheduler.fairness_algo[index].qs[ac].missed;
    stats->scheduled = priv->card->fh_buffer_d.packet_scheduler.fairness_algo[index].qs[ac].scheduled;
    stats->current_tokens = priv->card->fh_buffer_d.packet_scheduler.fairness_algo[index].qs[ac].tokens;
    stats->accumulated_tokens = priv->card->fh_buffer_d.packet_scheduler.fairness_algo[index].qs[ac].accumulated_tokens;

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetStationStateGet
 *
 *  Get Queue Set station state.
 *
 *  Arguments:
 *      hipHandle    Pointer to priv structure.
 *      stats        Pointer to stats structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetStationStateGet(void *hipHandle, CsrInt32 vif_id,  CsrInt32 aid, CsrInt32 *current_state, CsrUint32 *transitions)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    qs = csrWifiHipPacketSchedulerGetQueueSet(priv->card, vif_id, aid);
    if (!qs)
    {
        return CSR_RESULT_FAILURE;
    }
    if (qs->station_state == QS_STA_ACTIVE)
    {
        *current_state = 1;
    }
    else
    {
        *current_state = 0;
    }
    *transitions = qs->station_state_transitions;
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerQueueSetSMODGet
 *
 *  Get Queue Set station state.
 *
 *  Arguments:
 *      hipHandle    Pointer to priv structure.
 *      vif_id       VIF id.
 *      aid          Association ID
 *      smod         Pointer to integer to hold smod
 *      scod         Pointer to integer to hold scod
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerQueueSetSMODGet(void *hipHandle, CsrInt32 vif_id,  CsrInt32 aid, CsrInt32 *smod, CsrUint32 *scod)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    qs = csrWifiHipPacketSchedulerGetQueueSet(priv->card, vif_id, aid);
    if (!qs)
    {
        return CSR_RESULT_FAILURE;
    }
    *smod = qs->smod;
    *scod = qs->scod;
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  unifiSendCtrlData
 *
 *  Send a control signal.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      signal       Pointer to signal.
 *
 *  Returns:
 *      Success: CSR_RESULT_SUCCESS or CSR_WIFI_PS_RESULT_FULL
 *      Failure: CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerSendCtrlData(card_t *card, card_signal_t *signal)
{
    CsrWifiHipPacketSchedulerQsig *qsig;
    CsrResult r;

    qsig = csrWifiHipPacketSchedulerGetFreeSignal(card);
    if (!qsig)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: unifiSendCtrlData: no signal structures available\n", card->instance));
        card->fh_buffer_d.packet_scheduler.stats.qsig_empty++;
        return CSR_RESULT_FAILURE;
    }
    qsig->cp = &(card->fh_buffer_d.packet_scheduler.ctrl_q.common);
    qsig->ref = 0;
    qsig->data_len = 0;
    if (signal->bulkdata[0].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[0].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }
    if (signal->bulkdata[1].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[1].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }

    CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_count_lock);
    card->fh_signaldata_pending += signal->signal_length;
    card->fh_bulkdata_pending += qsig->data_len;
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_count_lock);

    qsig_set_signal_copy(qsig, signal);
    qsig_set_signal_type(qsig, SIGNAL_CTRL);

    r = card->fh_buffer_d.packet_scheduler.ctrl_q.common.enqueue(card, &card->fh_buffer_d.packet_scheduler.ctrl_q.common, qsig);
    if (r == CSR_RESULT_FAILURE)
    {
        CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendCtrlData: unable to enqueue signal\n", card->instance));
        card->fh_buffer_d.packet_scheduler.ctrl_q.common.stats.discarded_signals++;
        csrWifiHipPacketSchedulerFreeSignal(card, qsig);
    }
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerSendMgmtFrame
 *
 *  Send a management signal for a specified VIF.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       VIF ID.
 *      signal       Pointer to signal.
 *
 *  Returns:
 *      Success: CSR_RESULT_SUCCESS or CSR_WIFI_PS_RESULT_FULL
 *      Failure: CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerSendMgmtFrame(card_t *card, CsrInt32 vif_id, card_signal_t *signal)
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrWifiHipPacketSchedulerQsig *qsig;
    CsrResult r;

    vif = csrWifiHipPacketSchedulerGetVif(card, vif_id);
    if (!vif)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendMgmtFrame: csrWifiHipPacketSchedulerGetVif(, vif_id=%d) failed\n",
                            card->instance,  vif_id));
        return CSR_RESULT_FAILURE;
    }

    qsig = csrWifiHipPacketSchedulerGetFreeSignal(card);
    if (!qsig)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendMgmtFrame: no signal structures available\n", card->instance));
        card->fh_buffer_d.packet_scheduler.stats.qsig_empty++;
        return CSR_RESULT_FAILURE;
    }
    qsig->cp = &(vif->mgmt_q.common);
    qsig->ref = 0;
    qsig->data_len = 0;
    if (signal->bulkdata[0].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[0].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }
    if (signal->bulkdata[1].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[1].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }

    CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_count_lock);
    card->fh_signaldata_pending += signal->signal_length;
    card->fh_bulkdata_pending += qsig->data_len;
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_count_lock);

    qsig_set_signal_copy(qsig, signal);
    qsig_set_signal_type(qsig, SIGNAL_MGMT);

    CSR_LOG_TEXT_DEBUG((
                           CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendMgmtFrame: ref=%d data_length=%d qsig=0x%p\n",
                           card->instance,  qsig->ref, qsig->data_len, (void *) qsig));
    r = vif->mgmt_q.common.enqueue(card, &(vif->mgmt_q.common), qsig);
    if (r == CSR_RESULT_FAILURE)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendMgmtFrame: unable to enqueue signal\n", card->instance));
        vif->mgmt_q.common.stats.discarded_signals++;
        csrWifiHipPacketSchedulerFreeSignal(card, qsig);
    }
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerSendQsMgmtFrame
 *
 *  Send a management signal for a specified Queue Set.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       VIF ID.
 *      aid          Station ID.
 *      signal       Pointer to signal.
 *
 *  Returns:
 *      Success: CSR_RESULT_SUCCESS or CSR_WIFI_PS_RESULT_FULL
 *      Failure: CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerSendQsMgmtFrame(card_t *card, CsrInt32 vif_id, CsrInt32 aid, card_signal_t *signal)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipPacketSchedulerQsig *qsig;
    CsrResult r;

    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendQsMgmtFrame: could not get VIF %d QS %d\n",
                            card->instance, vif_id, aid));
        return CSR_RESULT_FAILURE;
    }

    qsig = csrWifiHipPacketSchedulerGetFreeSignal(card);
    if (!qsig)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "unifi%d: CsrWifiHipPacketSchedulerSendQsMgmtFrame: no signal structures available\n", card->instance));
        card->fh_buffer_d.packet_scheduler.stats.qsig_empty++;
        return CSR_RESULT_FAILURE;
    }
    qsig->cp = &(qs->mgmt_q.common);
    qsig->ref = 0;
    qsig->data_len = 0;
    if (signal->bulkdata[0].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[0].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }
    if (signal->bulkdata[1].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[1].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }

    CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_count_lock);
    card->fh_signaldata_pending += signal->signal_length;
    card->fh_bulkdata_pending += qsig->data_len;
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_count_lock);

    qsig_set_signal_copy(qsig, signal);
    qsig_set_signal_type(qsig, SIGNAL_MGMT);

    CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                        "unifi%d: CsrWifiHipPacketSchedulerSendQsMgmtFrame: ref=%d data_length=%d qsig=0x%p\n",
                        card->instance, qsig->ref, qsig->data_len, (void *) qsig));

    r = qs->mgmt_q.common.enqueue(card, &(qs->mgmt_q.common), qsig);
    if (r == CSR_RESULT_FAILURE)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "unifi%d: CsrWifiHipPacketSchedulerSendQsMgmtFrame: unable to enqueue signa\n", card->instance));
        qs->mgmt_q.common.stats.discarded_signals++;
        csrWifiHipPacketSchedulerFreeSignal(card, qsig);
    }
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerSendQsAuthQueueFrame
 *
 *  Send a auth signal (uncontrolled port) for a specified Queue Set.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       VIF ID.
 *      aid          Station ID.
 *      signal       Pointer to signal.
 *
 *  Returns:
 *      Success: CSR_RESULT_SUCCESS or CSR_WIFI_PS_RESULT_FULL
 *      Failure: CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerSendQsAuthQueueFrame(card_t *card, CsrInt32 vif_id, CsrInt32 aid, card_signal_t *signal)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipPacketSchedulerQsig *qsig;
    CsrResult r;

    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendQsAuthQueueFrame: could not get VIF %d QS %d\n",
                            card->instance, vif_id, aid));
        return CSR_RESULT_FAILURE;
    }

    qsig = csrWifiHipPacketSchedulerGetFreeSignal(card);
    if (!qsig)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "unifi%d: CsrWifiHipPacketSchedulerSendQsAuthQueueFrame: no signal structures available\n", card->instance));
        card->fh_buffer_d.packet_scheduler.stats.qsig_empty++;
        return CSR_RESULT_FAILURE;
    }
    qsig->cp = &(qs->auth_q.common);
    qsig->ref = 0;
    qsig->data_len = 0;
    if (signal->bulkdata[0].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[0].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }
    if (signal->bulkdata[1].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[1].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }

    CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_count_lock);
    card->fh_signaldata_pending += signal->signal_length;
    card->fh_bulkdata_pending += qsig->data_len;
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_count_lock);

    qsig_set_signal_copy(qsig, signal);
    qsig_set_signal_type(qsig, SIGNAL_MGMT);

    CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                        "unifi%d: CsrWifiHipPacketSchedulerSendQsAuthQueueFrame: ref=%d data_length=%d qsig=0x%p\n",
                        card->instance, qsig->ref, qsig->data_len, (void *) qsig));

    r = qs->auth_q.common.enqueue(card, &(qs->auth_q.common), qsig);
    if (r == CSR_RESULT_FAILURE)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "unifi%d: CsrWifiHipPacketSchedulerSendQsAuthQueueFrame: unable to enqueue signa\n", card->instance));
        qs->auth_q.common.stats.discarded_signals++;
        csrWifiHipPacketSchedulerFreeSignal(card, qsig);
    }
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerSendUnicastFrame
 *
 *  Send a unicast frame to a particular destination on a particular AC.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       VIF ID.
 *      aid          Station ID.
 *      ac           AC (0..3)
 *      signal       Pointer to signal.
 *
 *  Returns:
 *      Success: CSR_RESULT_SUCCESS or CSR_WIFI_PS_RESULT_FULL
 *      Failure: CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerSendUnicastFrame(card_t *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerACPri ac, card_signal_t *signal)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
    CsrWifiHipPacketSchedulerQsig *qsig;
    CsrResult r;

    if ((ac != AC_PRI_0) && (ac != AC_PRI_1) && (ac != AC_PRI_2) && (ac != AC_PRI_3))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendUnicastFrame: priority (%d) out of range (%d..%d)\n",
                            card->instance,  ac, AC_PRI_0, AC_PRI_3));
        return CSR_RESULT_FAILURE;
    }
    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendUnicastFrame: could not get VIF %d QS %d\n",
                            card->instance, vif_id, aid));
        return CSR_RESULT_FAILURE;
    }
    if ((signal->bulkdata[0].data_length && !signal->bulkdata[0].os_data_ptr) || (signal->bulkdata[1].data_length && !signal->bulkdata[1].os_data_ptr))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendUnicastFrame: inconsistent bulk data descriptors\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    qsig = csrWifiHipPacketSchedulerGetFreeSignal(card);
    if (!qsig)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendUnicastFrame: no signal structures available\n", card->instance));
        card->fh_buffer_d.packet_scheduler.stats.qsig_empty++;
        return CSR_RESULT_FAILURE;
    }
    qsig_set_signal_copy(qsig, signal);
    qsig_set_signal_type(qsig, SIGNAL_UNICAST);
    qsig_set_signal_pri(qsig, ac);

    ac_q = &qs->ac[ac];
    qsig->cp = &(ac_q->common);
    qsig->ref = 0;
    qsig->data_len = 0;
    if (signal->bulkdata[0].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[0].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }
    if (signal->bulkdata[1].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[1].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }

    /* Definition of the unicast datapath requires a single frame (bulk data) with the signal. */
    if (qsig->ref != 1)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendUnicastFrame: signal with invalid data reference (bulk data count of %d)\n",
                            card->instance,  qsig->ref));
        csrWifiHipPacketSchedulerFreeSignal(card, qsig);
        return CSR_RESULT_FAILURE;
    }


    /*
     * Are there any packets already in the HAL queue? If no, try to add to
     * circular buffer. If the circular buffer is full or there are any
     * packets in the HAL queue, add to HAL queue.
     */
    if (qs->hal_queues[ac].entries == 0)
    {
        /*
         * Place on circular buffer. The enqueue function supports three return codes:
         * CSR_RESULT_SUCCESS: The signal was successfully queued.
         * CSR_WIFI_PS_RESULT_FULL: The signal was successfully queued and the the buffer is now full.
         * CSR_RESULT_FAILURE: The circular buffer was full and the signal could not be enqueued.
         */
        r = ac_q->common.enqueue(card, &(ac_q->common), qsig);
        if (r == CSR_RESULT_SUCCESS)
        {
            CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_count_lock);
            card->fh_signaldata_pending += signal->signal_length;
            card->fh_bulkdata_pending += qsig->data_len;
            CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_count_lock);
            CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                "unifi%d: CsrWifiHipPacketSchedulerSendUnicastFrame: 1 add to circular buffer VIF %d AID %d AC %d signal=0x%p\n",
                                card->instance, vif_id, aid, ac, (void *) qsig));
            return r;
        }
        if (r == CSR_WIFI_PS_RESULT_FULL)
        {
            CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                "unifi%d: CsrWifiHipPacketSchedulerSendUnicastFrame: 2 add to circular buffer VIF %d AID %d AC %d signal=0x%p\n",
                                card->instance, vif_id, aid, ac, (void *) qsig));
            CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_count_lock);
            card->fh_signaldata_pending += signal->signal_length;
            card->fh_bulkdata_pending += qsig->data_len;
            CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_count_lock);
            if (qs->max_queue_len == 0)
            {
                /* Only flow control when there is no list configured. */
                ac_q->state = AC_QSTATE_FULL;
                if (qs->pauseCb)
                {
                    qs->pauseCb(qs->pauseResumeContext, (CsrUint16) qs->vif->id, (CsrUint16) qs->id, (CsrWifiHipTrafficQueue) ac_q->pri);
                    card->hip2Stats.hip2_stats_tx_q_pause[ac_q->pri]++;
                    CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: Cbuf full(1): AC %d PAUSED\n",
                                        card->instance, ac));
                }
                return CSR_WIFI_PS_RESULT_FULL;
            }
            return CSR_RESULT_SUCCESS;
        }
    }
    if ((qs->max_queue_len == 0) || (qs->hal_queues[ac].entries >= qs->max_queue_len))
    {
        /*
         * There is no list above the circular buffer and if we are here, it is because
         * the cirular buffer is full and the caller has not responded to the flow control
         * callback or the CSR_WIFI_PS_RESULT_FULL return code. Reject the signal.
         */
        csrWifiHipPacketSchedulerFreeSignal(card, qsig);
        ac_q->state = AC_QSTATE_FULL;
        if (qs->pauseCb)
        {
            qs->pauseCb(qs->pauseResumeContext, (CsrUint16) qs->vif->id, (CsrUint16) qs->id, (CsrWifiHipTrafficQueue) ac_q->pri);
            card->hip2Stats.hip2_stats_tx_q_pause[ac_q->pri]++;
            CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                "unifi%d: Cbuf full(2): AC %d PAUSED\n",
                                card->instance, ac));
        }
        ac_q->common.stats.rejected_signals++;
        CSR_LOG_TEXT_DEBUG((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendUnicastFrame: REJECT (CBUF full) VIF %d AID %d AC %d signal 0x%p\n",
                            card->instance,  vif_id, aid, ac, (void *) qsig));
        return CSR_RESULT_FAILURE;
    }
    /* Put the signal on the list. */
    CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_count_lock);
    csrWifiHipPacketSchedulerListAddTail(&(qs->hal_queues[ac]), &(qsig->qptr));
    card->fh_signaldata_pending += signal->signal_length;
    card->fh_bulkdata_pending += qsig->data_len;
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_count_lock);
    CSR_LOG_TEXT_INFO((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                       "unifi%d: CsrWifiHipPacketSchedulerSendUnicastFrame: add to HAL queue VIF %d AID %d AC %d signal=0x%p\n",
                       card->instance, vif_id, aid, ac, (void *) qsig));
    if (qs->hal_queues[ac].entries >= qs->max_queue_len)
    {
        /* If the list is now full, set the queue state and signal (if configured) that the queue is full. */
        ac_q->state = AC_QSTATE_FULL;
        if (qs->pauseCb)
        {
            qs->pauseCb(qs->pauseResumeContext, (CsrUint16) qs->vif->id, (CsrUint16) qs->id, (CsrWifiHipTrafficQueue) ac_q->pri);
            card->hip2Stats.hip2_stats_tx_q_pause[ac_q->pri]++;
            CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                "unifi%d: List full: AC %d PAUSED\n",
                                card->instance, ac));
        }
        return CSR_WIFI_PS_RESULT_FULL;
    }
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerSendMulticastFrame
 *
 *  Send a multicast frame to a particular VIF.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       VIF ID.
 *      signal       Pointer to signal.
 *
 *  Returns:
 *      Success: CSR_RESULT_SUCCESS or CSR_WIFI_PS_RESULT_FULL
 *      Failure: CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerSendMulticastFrame(card_t *card, CsrInt32 vif_id, card_signal_t *signal)
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrWifiHipPacketSchedulerQsig *qsig;
    CsrResult r;

    vif = csrWifiHipPacketSchedulerGetVif(card, vif_id);
    if (!vif)
    {
        return CSR_RESULT_FAILURE;
    }
    qsig = csrWifiHipPacketSchedulerGetFreeSignal(card);
    if (!qsig)
    {
        card->fh_buffer_d.packet_scheduler.stats.qsig_empty++;
        return CSR_RESULT_FAILURE;
    }
    qsig->cp = &(vif->multicast_q.common);
    qsig->ref = 0;
    qsig->data_len = 0;
    if (signal->bulkdata[0].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[0].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }
    if (signal->bulkdata[1].data_length)
    {
        qsig->ref++;
        qsig->data_len += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[1].data_length + card->config_data.fh_pushed_block_gap, card->fh_buffer_d.pushed_round_bytes);
    }

    CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_count_lock);
    card->fh_signaldata_pending += signal->signal_length;
    card->fh_bulkdata_pending += qsig->data_len;
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_count_lock);

    qsig_set_signal_copy(qsig, signal);
    qsig_set_signal_type(qsig, SIGNAL_MULTICAST);

    r = vif->multicast_q.common.enqueue(card, &vif->multicast_q.common, qsig);
    if (r == CSR_RESULT_FAILURE)
    {
        CSR_LOG_TEXT_DEBUG((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerSendMulticastFrame: REJECT (CBUF full) VIF %d signal 0x%p\n",
                            card->instance,  vif_id, (void *) qsig));
        vif->multicast_q.common.stats.discarded_signals++;
        csrWifiHipPacketSchedulerFreeSignal(card, qsig);
    }
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerVIFSetMcastCompleteCb
 *
 *  Set the mcast complete callback.
 *
 *  Arguments:
 *      card              Pointer to card structure.
 *      vif_id            Virtual interface ID.
 *      mcast_complete_cb Callback function pointer.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPacketSchedulerVIFSetMcastCompleteCb(struct card *card, CsrInt32 vif_id, CsrWifiHipPacketSchedulerMcastCompleteCb mcast_complete_cb)
{
    CsrWifiHipPacketSchedulerVif *vif;

    vif = csrWifiHipPacketSchedulerGetVif(card, vif_id);
    if (!vif)
    {
        return CSR_RESULT_FAILURE;
    }
    vif->mcast_complete_cb = mcast_complete_cb;
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerVIFGetAllQSState
 *
 *  Collate the configured QS and QS station state into bit maps for the specified VIF.
 *
 *  Arguments:
 *      card              Pointer to card structure.
 *      vif_id            Virtual interface ID.
 *      configured_aid    Pointer to bit map of AID configured on the VIF.
 *      powersave_aid     Pointer to bit map of configured AIDs on the VIF in powersave state.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerVIFGetAllQSState(struct card *card, CsrInt32 vif_id, CsrUint16 *configured_aid, CsrUint16 *powersave_aid)
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrUint16 cfg_aid, ps_aid, i;

    vif = csrWifiHipPacketSchedulerGetVif(card, vif_id);
    if (!vif)
    {
        return CSR_RESULT_FAILURE;
    }
    cfg_aid = 0;
    ps_aid = 0;
    for (i = 0; i < PS_MAX_AID + 1; i++)
    {
        qs = vif->qs[i];
        if (qs && (qs->id >= PS_AP_START_AID))
        {
            cfg_aid |= (1 << (qs->id - PS_AP_START_AID));
            if (qs->station_state == QS_STA_POWERSAVE)
            {
                ps_aid |= (1 << (qs->id - PS_AP_START_AID));
            }
        }
    }
    *configured_aid = cfg_aid;
    *powersave_aid = ps_aid;
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerSuspend
 *
 *  Pauses all the queues
 *
 *  Arguments:
 *      card              Pointer to card structure.
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipPacketSchedulerSuspend(struct card *card)
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrInt32 index;

    for (index = 0; index < PS_MAX_VIFS; index++)
    {
        vif = card->fh_buffer_d.packet_scheduler.vifs[index];
        if (vif)
        {
            CsrWifiHipPacketSchedulerQueueSet *qs;
            CsrInt32 aid;

            /* If the QS has a callback, go through and use the callback to enable or disable queues. */
            for (aid = 0; aid < PS_MAX_AID + 1; aid++)
            {
                qs = vif->qs[aid];
                if (qs && qs->pauseCb)
                {
                    CsrInt32 id;

                    for (id = 0; id < PS_MAX_FAIRNESS_PRI; id++)
                    {
                        CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
                        ac_q = &(qs->ac[id]);
                        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                            "unifi%d: CsrWifiHipPacketSchedulerSuspend: calling pause call back\n",
                                            card->instance));
                        qs->pauseCb(qs->pauseResumeContext, (CsrUint16) qs->vif->id, (CsrUint16) qs->id, (CsrWifiHipTrafficQueue) ac_q->pri);
                    }
                }
            }
        }
    }
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerResume
 *
 *  On system resume this function is called to start the set of transmit queues that should be started.
 *
 *  Arguments:
 *      card              Pointer to card structure.
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipPacketSchedulerResume(struct card *card)
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrInt32 index;

    for (index = 0; index < PS_MAX_VIFS; index++)
    {
        vif = card->fh_buffer_d.packet_scheduler.vifs[index];
        if (vif)
        {
            CsrWifiHipPacketSchedulerQueueSet *qs;
            CsrInt32 aid;

            /* If the QS has a callback, go through and use the callback to enable or disable queues. */
            for (aid = 0; aid < PS_MAX_AID + 1; aid++)
            {
                qs = vif->qs[aid];
                if (qs && qs->pauseCb && qs->resumeCb)
                {
                    CsrInt32 id;

                    for (id = 0; id < PS_MAX_FAIRNESS_PRI; id++)
                    {
                        CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
                        ac_q = &(qs->ac[id]);

                        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                            "unifi%d: CsrWifiHipPacketSchedulerResume: Calling %s callback\n",
                                            card->instance, ac_q->state == AC_QSTATE_FULL ? "pause" : "resume"));

                        if (ac_q->state == AC_QSTATE_FULL)
                        {
                            qs->pauseCb(qs->pauseResumeContext, (CsrUint16) qs->vif->id, (CsrUint16) qs->id, (CsrWifiHipTrafficQueue) ac_q->pri);
                            card->hip2Stats.hip2_stats_tx_q_pause[ac_q->pri]++;
                        }
                        else
                        {
                            qs->resumeCb(qs->pauseResumeContext, (CsrUint16) qs->vif->id, (CsrUint16) qs->id, (CsrWifiHipTrafficQueue) ac_q->pri);
                        }
                    }
                }
            }
        }
    }
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerCapabilitiesQueueLen
 *
 *
 *
 *  Arguments:
 *      card              Pointer to card structure.
 *
 *  Returns:
 *      Queue length or -1 if there is an error
 *
 * ---------------------------------------------------------------------------
 */
CsrInt32 CsrWifiHipPacketSchedulerCapabilitiesQueueLen(struct card *card)
{
    /*
     * The circular buffer implementation leaves a slot empty.
     */
    return UNIFI_QUEUE_LEN - 1;
}
