/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#ifdef PS_USPACE
#include "csr_wifi_ps_types.h"
#else
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip_card.h"
#include "csr_wifi_hip_card_sdio.h"
#endif
#include "csr_wifi_ps_q.h"
#include "csr_wifi_ps_circ.h"
#include "csr_wifi_ps_qsig.h"

#include "csr_wifi_ps_if.h"
#include "csr_wifi_ps.h"
#include "csr_wifi_hip_core_trace.h"

#ifdef PS_USPACE
#include "csr_wifi_ps_card.h"
#endif

static void init_packet_schedulers(card_t *card);
static CsrResult csrWifiHipPacketSchedulerInitSignalList(card_t *card, CsrInt32 qlen);

static void ps_list_add(CsrWifiPacketSchedulerCommonPart *cp);
static void ps_list_remove(CsrWifiPacketSchedulerCommonPart *cp);
static void ps_schedule_re_stock(struct card *card);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerInit
 *
 *  Initialisation routine for the Packet Scheduler.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipPacketSchedulerInit(card_t *card)
{
    CsrWifiHipPacketSchedulerCtrlQueue *ctrl_q;
    CsrInt32 qs_pri, ac, sp_pri, vindex;
    CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;

    CsrMemSet((void *) &(card->fh_buffer_d.packet_scheduler), 0, sizeof(struct packet_scheduler));

    if (CsrMutexCreate(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: packet_scheduler_init: unable to create mutex\n", card->instance));
        return;
    }
    if (csrWifiHipPacketSchedulerInitSignalList(card, UNIFI_QSIG_BUF) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: packet_scheduler_init: unable to allocate memory for signal buffer\n", card->instance));
        return;
    }
    /* Sets up the Strict Priority and DWRR Packet Scheduler function in function pointers. */
    init_packet_schedulers(card);
    /* There are two DWRR instances - one for each QS priority. */
    for (qs_pri = 0; qs_pri < PS_MAX_QS_PRI; qs_pri++)
    {
        csrWifiHipPacketSchedulerDwrrInitInstance(&(card->fh_buffer_d.packet_scheduler.fairness_algo[qs_pri]));
    }
    /* Initialise vif pointers. */
    for (vindex = 0; vindex < PS_MAX_VIFS; vindex++)
    {
        card->fh_buffer_d.packet_scheduler.vifs[vindex] = (CsrWifiHipPacketSchedulerVif *) 0;
    }

    /* Initialise sp lists. */
    for (sp_pri = 0; sp_pri < PS_MAX_SP_PRI; sp_pri++)
    {
        csrWifiHipPacketSchedulerListInit(&(card->fh_buffer_d.packet_scheduler.sp[sp_pri]));
    }
    for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
    {
        csrWifiHipPacketSchedulerListInit(&(card->fh_buffer_d.packet_scheduler.sp_retransmit[ac]));
    }
    /* Initialise fairness lists. */
    for (qs_pri = 0; qs_pri < PS_MAX_QS_PRI; qs_pri++)
    {
        for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
        {
            csrWifiHipPacketSchedulerListInit(&(card->fh_buffer_d.packet_scheduler.fairness[qs_pri][ac]));
        }
    }

    /* Initialise the pending clear list for the signals sent through the control queue. */
    pending_clr_q = &(card->fh_buffer_d.packet_scheduler.pending_clr_q);
    pending_clr_q->common.dequeue = csrWifiHipPacketSchedulerPendingClrDequeue;
    pending_clr_q->common.peek = csrWifiHipPacketSchedulerPendingClrPeek;
    pending_clr_q->common.getlen = csrWifiHipPacketSchedulerPendingClrGetLen;
    pending_clr_q->common.enqueue = csrWifiHipPacketSchedulerPendingClrEnqueue;
    pending_clr_q->common.free = 0;
    pending_clr_q->common.qmod = PS_DEFAULT_QMOD;
    pending_clr_q->common.qcod = 0;
    pending_clr_q->common.vif = 0;
    pending_clr_q->common.qs = 0;
    pending_clr_q->common.pending_clr_q = 0;
    csrWifiHipPacketSchedulerListInit(&(pending_clr_q->pkts));
    pending_clr_q->common.list = 0;

    /* Initialise the control queue. */
    ctrl_q = &(card->fh_buffer_d.packet_scheduler.ctrl_q);
    ctrl_q->common.dequeue = csrWifiHipPacketSchedulerCtrlDequeue;
    ctrl_q->common.peek = csrWifiHipPacketSchedulerCtrlPeek;
    ctrl_q->common.getlen = csrWifiHipPacketSchedulerCtrlGetLen;
    ctrl_q->common.enqueue = csrWifiHipPacketSchedulerCtrlEnqueue;
    ctrl_q->common.free = csrWifiHipPacketSchedulerCommonFree;
    ctrl_q->common.qmod = PS_DEFAULT_QMOD;
    ctrl_q->common.qcod = 0;
    ctrl_q->common.vif = 0;
    ctrl_q->common.qs = 0;
    ctrl_q->common.pending_clr_q = pending_clr_q;
    ps_circ_init((&(ctrl_q->pkts)), UNIFI_QUEUE_LEN);
    ctrl_q->common.list = &(card->fh_buffer_d.packet_scheduler.sp[0]);
    /* Enable the queue by putting it in the SP queue list. */
    csrWifiHipPacketSchedulerEnableQueue(&(ctrl_q->common));

    CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: Packet Scheduler initialised\n", card->instance));
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPacketSchedulerDestroy
 *
 *  Removal routine for the Packet Scheduler.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipPacketSchedulerDestroy(card_t *card)
{
    CsrInt32 vif_id;
    CsrWifiHipPacketSchedulerCtrlQueue *ctrl_q;
    CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;

    CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: Destroying Packet Scheduler...\n", card->instance));
    for (vif_id = PS_MIN_VIF_ID; vif_id <= PS_MAX_VIF_ID; vif_id++)
    {
        /*
         * The mutex is taken in VIFDisassociate.
         */
        if (CsrWifiHipPacketSchedulerVIFDisassociate(card, vif_id) == CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_DEBUG((CsrWifiHipLto,
                                CSR_WIFI_HIP_LTSO_PS, "unifi%d: \tPacket Scheduler VIF %d destroyed\n",
                                card->instance,  vif_id));
        }
    }
    if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: CsrWifiHipPacketSchedulerDestroy: CsrMutexLock failed\n", card->instance));
        /*
         * Fall through here - this is the end of the Packet Scheduler and we need to clean up,
         * even if there appears to be an error in getting the mutex.
         */
    }
    /* Free all signals in the ctrl queue. */
    ctrl_q = &(card->fh_buffer_d.packet_scheduler.ctrl_q);
    csrWifiHipPacketSchedulerDisableQueue(&(ctrl_q->common));
    csrWifiHipPacketSchedulerPurgeQueue(card, &(ctrl_q->common));
    /* Free all signals in the pending clear queue. */
    pending_clr_q = &(card->fh_buffer_d.packet_scheduler.pending_clr_q);
    csrWifiHipPacketSchedulerPendingClrPurge(card, pending_clr_q);

    if (card->fh_buffer_d.packet_scheduler.signal_buf)
    {
        CsrMemFree(card->fh_buffer_d.packet_scheduler.signal_buf);
    }
    CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);

    CsrMutexDestroy(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerInitSignalList
 *
 *  Allocates memory for the QSIG list and initialises the free list.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      qlen         Number of QSIG structures to allocate.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
static CsrResult csrWifiHipPacketSchedulerInitSignalList(card_t *card, CsrInt32 qlen)
{
    CsrInt32 i;
    CsrWifiHipPacketSchedulerQsig *qsig;

    csrWifiHipPacketSchedulerListInit(&(card->fh_buffer_d.packet_scheduler.free_signal_list));
    qsig = (CsrWifiHipPacketSchedulerQsig *) CsrMemAlloc(sizeof(CsrWifiHipPacketSchedulerQsig) * qlen);
    if (!qsig)
    {
        return CSR_RESULT_FAILURE;
    }
    card->fh_buffer_d.packet_scheduler.signal_buf = qsig;
    CsrMemSet(qsig, 0, sizeof(*qsig) * qlen);
    for (i = 0; i < qlen; i++, qsig++)
    {
        csrWifiHipPacketSchedulerListAddTail(&(card->fh_buffer_d.packet_scheduler.free_signal_list), &(qsig->qptr));
    }
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerGetFreeSignal
 *
 *  Take a QSIG structure from the free list.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *
 *  Returns:
 *      Pointer to QSIG structure or zero if none available.
 *
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerGetFreeSignal(card_t *card)
{
    CsrWifiHipPacketSchedulerQsig *qsig;

    CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_count_lock);
    qsig = (CsrWifiHipPacketSchedulerQsig *) csrWifiHipPacketSchedulerListTakeFirst(&(card->fh_buffer_d.packet_scheduler.free_signal_list));
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_count_lock);
    CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerGetFreeSignal qsig=0x%p\n",
                        card->instance, (void *) qsig));
    return qsig;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerFreeSignal
 *
 *  Put a QSIG structure back on the free list.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerFreeSignal(card_t *card, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerFreeSignal: qsig=0x%p\n",
                        card->instance, (void *) qsig));
    CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_count_lock);
    csrWifiHipPacketSchedulerListAddTail(&(card->fh_buffer_d.packet_scheduler.free_signal_list), &(qsig->qptr));
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_count_lock);
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerFreeSignalCount
 *
 *
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      Count of available QSIG structures.
 *
 * ---------------------------------------------------------------------------
 */
CsrInt32 csrWifiHipPacketSchedulerFreeSignalCount(card_t *card)
{
    return card->fh_buffer_d.packet_scheduler.free_signal_list.entries;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerFreeSlot
 *
 *  Called by the HIP when f/w frees a bulk data slot.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerFreeSlot(card_t *card, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CSR_LOG_TEXT_DEBUG((
                           CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerFreeSlot: qsig=%p\n",
                           card->instance,  (void *) qsig));

    if (qsig->ref > 2)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerFreeSlot: qsig has invalid reference count (%d)\n",
                            card->instance,  qsig->ref));
        return CSR_RESULT_FAILURE;
    }
    if (!qsig->cp->free)
    {
        /* Should not happen. */
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerFreeSlot: received an rtx queue pointer\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    if (qsig->ref > 0)
    {
        qsig->ref--;
        if (qsig->ref > 0)
        {
            /* Do nothing - the qsig will be deallocated when the next slot reference is freed by the f/w. */
            return CSR_RESULT_SUCCESS;
        }
    }
    if (qsig->qptr.list)
    {
        /* The qsig is in a pending clear list. */
        CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;

        pending_clr_q = qsig->cp->pending_clr_q;
        if (pending_clr_q)
        {
            pending_clr_q->common.dequeue(card, &(pending_clr_q->common), qsig);
        }
        else
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerFreeSlot: no pending clr q\n", card->instance));
        }
    }
    qsig->cp->free(card, qsig->cp, qsig);
    /* Put qsig back on free list. */
    csrWifiHipPacketSchedulerFreeSignal(card, qsig);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerRtxSlot
 *
 *  Called by the HIP when f/w requests a bulk data slot be retransmitted.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerRtxSlot(card_t *card, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrUint16 priority;
    CsrWifiHipPacketSchedulerRtxQueue *rtx_q;

    CSR_LOG_TEXT_DEBUG((
                           CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerRtxSlot: qsig=%p\n",
                           card->instance,  (void *) qsig));

    if (qsig->ref != 1)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerRtxSlot: qsig has invalid reference count (%d)\n",
                            card->instance,  qsig->ref));
        return CSR_RESULT_FAILURE;
    }
    if (!qsig->qptr.list)
    {
        /* Should not happen. */
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerRtxSlot: received an rtx queue pointer\n", card->instance));
        return CSR_RESULT_FAILURE;
    }
    /* The qsig is in the pending clear list. */
    csrWifiHipPacketSchedulerListTake(qsig->qptr.list, &(qsig->qptr));
    if (!qsig->cp->qs)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerRtxSlot: received rtx for non-ac slot, put in VIF rtx queue\n", card->instance));
        rtx_q = &(qsig->cp->vif->rtx_q);
        rtx_q->common.enqueue(card, &rtx_q->common, qsig);
    }
    else
    {
        priority = qsig_get_signal_pri(qsig);
        if (priority < PS_MAX_FAIRNESS_PRI)
        {
            /* Place the signal on the associated rtx queue. */
            rtx_q = &(qsig->cp->qs->retransmit[priority]);
            rtx_q->common.enqueue(card, &rtx_q->common, qsig);
        }
        else
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                                CSR_WIFI_HIP_LTSO_PS, "unifi%d: QSIG priority out of range - %d\n",
                                card->instance, qsig_get_signal_pri(qsig)));
        }
    }

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerGetVif
 *
 *  Find the VIF structure.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface id (1..PS_MAX_VIFS)
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerVif *csrWifiHipPacketSchedulerGetVif(card_t *card, CsrInt32 vif_id)
{
    if ((vif_id < PS_MIN_VIF_ID) || (vif_id > PS_MAX_VIF_ID))
    {
        return 0;
    }


    return card->fh_buffer_d.packet_scheduler.vifs[MAP_VIFID_TO_INDEX(vif_id)];
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerGetQueueSet
 *
 *  Find the Queue Set structure.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface id (1..PS_MAX_VIFS)
 *      aid          Station ID for the Queue Set.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerQueueSet *csrWifiHipPacketSchedulerGetQueueSet(card_t *card, CsrInt32 vif_id, CsrInt32 aid)
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrInt32 index;

    if ((aid < PS_MIN_AID) || (aid > PS_MAX_AID))
    {
        return 0;
    }

    vif = csrWifiHipPacketSchedulerGetVif(card, vif_id);
    if (!vif)
    {
        return 0;
    }
    index = MAP_AID_TO_INDEX(aid);
    if (index >= PS_MAX_STA_PER_AP)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerGetQueueSet: index out or range: %d (aid=%d)\n", card->instance, index, aid));
        return 0;
    }
    return vif->qs[index];
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerEnableQueueSet
 *
 *  Find the Queue Set structure.
 *
 *  Arguments:
 *      qs           Pointer to Queue Set structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerEnableQueueSet(CsrWifiHipPacketSchedulerQueueSet *qs)
{
    CsrInt32 ac;
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;

    if (qs->active)
    {
        return CSR_RESULT_FAILURE;
    }

    csrWifiHipPacketSchedulerEnableQueue(&(qs->mgmt_q.common));

    if (qs->authq_state == QS_AUTHQ_ENABLED)
    {
        csrWifiHipPacketSchedulerEnableQueue(&(qs->auth_q.common));
    }

    if (qs->acq_state == QS_ACQ_ENABLED)
    {
        for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
        {
            ac_q = &qs->ac[ac];
            if (ac_q->common.qcod < MOD_TO_BYTES(ac_q->common.qmod))
            {
                csrWifiHipPacketSchedulerEnableQueue(&ac_q->common);
            }
        }
    }
    qs->active = 1;
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerDisableQueueSet
 *
 *  Find the Queue Set structure.
 *
 *  Arguments:
 *      qs           Pointer to Queue Set structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerDisableQueueSet(CsrWifiHipPacketSchedulerQueueSet *qs)
{
    CsrInt32 ac;
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;

    if (!qs->active)
    {
        return CSR_RESULT_FAILURE;
    }

    csrWifiHipPacketSchedulerDisableQueue(&(qs->mgmt_q.common));
    csrWifiHipPacketSchedulerDisableQueue(&(qs->auth_q.common));

    for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
    {
        ac_q = &qs->ac[ac];
        csrWifiHipPacketSchedulerDisableQueue(&(ac_q->common));
    }
    qs->active = 0;
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerEnableRtxQueueSet
 *
 *  Add RTX queues with packets to their SP lists.
 *
 *  Arguments:
 *      qs           Pointer to Queue Set structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerEnableRtxQueueSet(CsrWifiHipPacketSchedulerQueueSet *qs)
{
    CsrInt32 ac;
    CsrWifiHipPacketSchedulerRtxQueue *rtx_q;

    for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
    {
        rtx_q = &qs->retransmit[ac];
        csrWifiHipPacketSchedulerListGetFirst(&(rtx_q->pkts));
        csrWifiHipPacketSchedulerEnableQueue(&rtx_q->common);
    }
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerEnableQueue
 *
 *  Add a queue to its list.
 *
 *  Arguments:
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerEnableQueue(CsrWifiPacketSchedulerCommonPart *cp)
{
    if (cp->active)
    {
        return CSR_RESULT_FAILURE;
    }
    ps_list_add(cp);
    cp->active = 1;
    cp->stats.enabled++;
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerDisableQueue
 *
 *  Remove a queue from its list.
 *
 *  Arguments:
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerDisableQueue(CsrWifiPacketSchedulerCommonPart *cp)
{
    if (!cp->active)
    {
        return CSR_RESULT_FAILURE;
    }
    ps_list_remove(cp);
    cp->active = 0;
    cp->stats.disabled++;
    return CSR_RESULT_SUCCESS;
}

#if (!CSR_WIFI_HIP_PSCHED_MOD_ALGO || CSR_WIFI_HIP_PSCHED_MOD_ALGO == 2)
/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerDistributeSMod
 *
 *  Remove a queue from its list.
 *
 *  Arguments:
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerDistributeSMod(card_t *card)
{
    struct packet_scheduler *pscb;
    CsrInt32 total_smod, non_vif_smod;
    CsrWifiHipPacketSchedulerVif *vif;
    CsrInt32 i, smod_active, smod_powersave;
    CsrInt32 active_stations, powersave_stations;
    CsrUint16 vifs_scheduled, vifs_not_scheduled;
    CsrUint16 num_vifs_scheduled, num_vifs_not_scheduled;

    pscb = &card->fh_buffer_d.packet_scheduler;
    if (!pscb->active_qs)
    {
		/*OIL138 suppress the verbose strings when wifi-ON*/
		/*
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "csrWifiHipPacketSchedulerDistributeSMod: no active Queue Sets\n"));*/

		CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "csrWifiHipPacketSchedulerDistributeSMod: no active Queue Sets\n"));
        return;
    }
    total_smod = PS_MAX_SMOD;
    non_vif_smod = 4;

    vifs_scheduled = 0;
    num_vifs_scheduled = 0;
    vifs_not_scheduled = 0;
    num_vifs_not_scheduled = 0;

    /*
     * Figure out the VIFs that are scheduled and unscheduled.
     * The resultant bit map is of Packet Scheduler VIF indices,
     * which map to interface tags in the upper layer s/w.
     */
    for (i = 0; i < PS_MAX_VIFS; i++)
    {
        if (pscb->vifs[i] != 0)
        {
            CsrUint16 vif_index;

            vif_index = CsrWifiHipVifIndexGetReq(card->ospriv, i);
            if (pscb->next_vif & (1 << vif_index))
            {
                vifs_scheduled |= (1 << i);
                num_vifs_scheduled++;
            }
            else
            {
                vifs_not_scheduled |= (1 << i);
                num_vifs_not_scheduled++;
            }
        }
    }

    /*
     * For A11, scheduling of multiple VIFs will occur when the f/w
     * scheduler is idle. For Hydra, it will be possible to schedule
     * multiple VIFs whilst leaving other VIFs unscheduled.
     */
    if ((num_vifs_scheduled > 1) && (num_vifs_not_scheduled > 0))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "csrWifiHipPacketSchedulerDistributeSMod: sched=%d unsched=%D\n",
                            num_vifs_scheduled, num_vifs_not_scheduled));
    }

    /*
     * 1) Allocate a fixed SMOD to all QS in VIFs that are unscheduled.
     * 2) Take what is left and allocate to the QS in the scheduled VIF(s).
     */
    i = 0;
    while (num_vifs_not_scheduled && (i < PS_MAX_VIFS))
    {
        if (vifs_not_scheduled & (1 << i))
        {
            CsrWifiHipPacketSchedulerQueueSet *qs;
            CsrInt32 qsindex;
            CsrInt32 n;

            num_vifs_not_scheduled--;

            vif = pscb->vifs[i];
            if (vif)
            {
                n = vif->active_qs;
                if (n > 0)
                {
                    for (qsindex = 0; qsindex < PS_MAX_STA_PER_AP; qsindex++)
                    {
                        qs = vif->qs[qsindex];
                        if (qs)
                        {
                            csrWifiHipPacketSchedulerQueueSetSModHelper(qs, non_vif_smod);
                            total_smod -= non_vif_smod;
                            if (--n == 0)
                            {
                                break;
                            }
                        }
                    }
                }
            }
        }
        i++;
    }
    if (num_vifs_scheduled == 0)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "csrWifiHipPacketSchedulerDistributeSMod: No VIF scheduled\n"));
        return;
    }
    if (num_vifs_scheduled > 1)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "csrWifiHipPacketSchedulerDistributeSMod: All VIFs scheduled (%d)\n",
                            num_vifs_scheduled));
    }
    /*
     * When more than one VIF is scheduled, this means the f/w is idle in A11
     * and so we are at liberty to decide what balance of traffic to send from the VIFs.
     * The simple approach is to divide the total SMOD equally amongst the
     * VIFs. For a single VIF, this will mean it gets all of remaining SMOD.
     */
    total_smod = total_smod / num_vifs_scheduled;

    i = 0;
    while (num_vifs_scheduled && (i < PS_MAX_VIFS))
    {
        if (vifs_scheduled & (1 << i))
        {
            num_vifs_scheduled--;

            vif = pscb->vifs[i];
            if (vif)
            {
                CsrWifiHipPacketSchedulerQueueSet *qs;
                CsrInt32 qsindex;
                CsrInt32 n;

                if (vif->active_qs > 0)
                {
                    active_stations = vif->active_qs - vif->powersave_qs;
                    powersave_stations = vif->powersave_qs;
                    smod_powersave = PS_POWER_SAVE_SMOD;
                    if (active_stations > 0)
                    {
                        smod_active = (total_smod - (pscb->powersave_qs * PS_POWER_SAVE_SMOD)) / (active_stations);
                        if ((smod_active < 0) || (smod_active > total_smod))
                        {
                            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                                "csrWifiHipPacketSchedulerDistributeSMod: Calculated active smod (%d) is out of range, reverting to default\n", smod_active));

                            smod_active = PS_POWER_SAVE_SMOD;
                        }
                    }
                    else
                    {
                        if (powersave_stations <= 0)
                        {
                            /*
                            * The logic above will preclude this from happening, but code check tools will see the possibility of
                            * a divide by zero.
                            */
                            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                                "csrWifiHipPacketSchedulerDistributeSMod: stations in powersave (%d) out of range, aborting smod redistribution\n", powersave_stations));
                            return;
                        }
                        smod_powersave = total_smod / powersave_stations;
                        smod_active = 0;
                    }

                    n = vif->active_qs;

                    for (qsindex = 0; qsindex < PS_MAX_STA_PER_AP; qsindex++)
                    {
                        qs = vif->qs[qsindex];
                        if (qs)
                        {
                            CsrInt32 smod;
                            if (qs->station_state == QS_STA_ACTIVE)
                            {
                                smod = smod_active;
                                if (smod == 0)
                                {
                                    CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                                        "csrWifiHipPacketSchedulerDistributeSMod: QS is active station state when active smod set to zero, reverting to default\n"));
                                    smod_active = PS_POWER_SAVE_SMOD;
                                }
                            }
                            else
                            {
                                smod = smod_powersave;
                            }
                            csrWifiHipPacketSchedulerQueueSetSModHelper(qs, smod);
                            if (--n == 0)
                            {
                                break;
                            }
                        }
                    }
                }
            }
        }
        i++;
    }
}

#endif /* (CSR_WIFI_HIP_PSCHED_MOD_ALGO == 2) */

#if (CSR_WIFI_HIP_PSCHED_MOD_ALGO == 1)
/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerDistributeSMod
 *
 *  Remove a queue from its list.
 *
 *  Arguments:
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerDistributeSMod(card_t *card)
{
    CsrWifiHipPacketSchedulerVif *vif;
    CsrInt32 i, smod_active, smod_powersave, n;
    CsrInt32 active_stations, powersave_stations;
    struct packet_scheduler *pscb;

    pscb = &card->fh_buffer_d.packet_scheduler;

    if (!pscb->active_qs)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "csrWifiHipPacketSchedulerDistributeSMod: no active Queue Sets\n"));
        return;
    }
    n = pscb->active_qs;
    active_stations = pscb->active_qs - pscb->powersave_qs;
    powersave_stations = pscb->powersave_qs;

    /*
     * 1) Calculate the minimum smod for the stations in powersave.
     * 2) Use what is left to calculate the active station smod.
     * 3) If all stations are in powersave, distribute total smod equally.
     * Examples:
     * 1) powersave_qs=1, active_qs=8 (ie. 7 active stations)
     * smod_powersave=4, smod_active=(32-4)/7=4
     * 2) powersave_qs=4, active_qs=5 (1 active station)
     * smod_powersave=4, smod_active=(32-(4*4))/1=16
     */

    smod_powersave = PS_POWER_SAVE_SMOD;
    if (active_stations > 0)
    {
        smod_active = (PS_MAX_SMOD - (pscb->powersave_qs * PS_POWER_SAVE_SMOD)) / (active_stations);
        if ((smod_active < 0) || (smod_active > PS_MAX_SMOD))
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                "csrWifiHipPacketSchedulerDistributeSMod: Calculated active smod (%d) is out of range, reverting to default\n", smod_active));

            smod_active = PS_POWER_SAVE_SMOD;
        }
    }
    else
    {
        if (powersave_stations <= 0)
        {
            /*
             * The logic above will preclude this from happening, but code check tools will see the possibility of
             * a divide by zero.
             */
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                "csrWifiHipPacketSchedulerDistributeSMod: stations in powersave (%d) out of range, aborting smod redistribution\n", powersave_stations));
            return;
        }
        smod_powersave = PS_MAX_SMOD / powersave_stations;
        smod_active = 0;
    }

    CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                        "csrWifiHipPacketSchedulerDistributeSMod: smod_active=%d smod_powersave=%d\n", smod_active, smod_powersave));
    for (i = 0; i < PS_MAX_VIFS; i++)
    {
        vif = pscb->vifs[i];
        if (vif)
        {
            CsrWifiHipPacketSchedulerQueueSet *qs;
            CsrInt32 qsindex;

            for (qsindex = 0; qsindex < PS_MAX_STA_PER_AP; qsindex++)
            {
                qs = vif->qs[qsindex];
                if (qs)
                {
                    CsrInt32 smod;
                    if (qs->station_state == QS_STA_ACTIVE)
                    {
                        smod = smod_active;
                        if (smod == 0)
                        {
                            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                                "csrWifiHipPacketSchedulerDistributeSMod: QS is active station state when active smod set to zero, reverting to default\n"));
                            smod_active = PS_POWER_SAVE_SMOD;
                        }
                    }
                    else
                    {
                        smod = smod_powersave;
                    }
                    csrWifiHipPacketSchedulerQueueSetSModHelper(qs, smod);
                    if (--n == 0)
                    {
                        return;
                    }
                }
            }
        }
    }
}

#endif /* (CSR_WIFI_HIP_PSCHED_MOD_ALGO == 1) */

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerQueueSetSModHelper
 *
 *  Change the Queue Set SMOD.
 *
 *  Arguments:
 *      qs           Pointer to queue set structure.
 *      smod         Queue Set SMOD.
 *
 *  Returns:
 *      void
 *
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerQueueSetSModHelper(CsrWifiHipPacketSchedulerQueueSet *qs, CsrInt32 smod)
{
    CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "csrWifiHipPacketSchedulerQueueSetSModHelper: QS (VIF %d AID %d) SMOD=%d\n",
                        qs->vif->id, qs->id, smod));
    qs->smod = smod;
    if ((MOD_TO_BYTES(qs->smod) <= qs->scod))
    {
        if (qs->active)
        {
            csrWifiHipPacketSchedulerDisableQueueSet(qs);
        }
        return;
    }
    if (!qs->active)
    {
        csrWifiHipPacketSchedulerEnableQueueSet(qs);
    }
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerQueueSetStateSet
 *
 *  Change the Queue Set station state (active or powersave).
 *
 *  Arguments:
 *      qs           Pointer to queue set structure.
 *      smod         Queue Set SMOD.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS if the state is successfully changed.
 *      CSR_RESULT_FAILURE with no error message if the state is the same as already set.
 *      CSR_RESULT_FAILURE with a log error otherwise.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerQueueSetStateSet(card_t *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerQSState state)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;

    qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
    if (!qs)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerQueueSetStateSet: QS (VIF %d AID %d) instance does not exist\n",
                            card->instance, vif_id, aid));
        return CSR_RESULT_FAILURE;
    }

    if (qs->station_state == state)
    {
        /* Nothing to be done */
        return CSR_RESULT_FAILURE;
    }

    if (qs->station_state == QS_STA_POWERSAVE)
    {
        if (state == QS_STA_ACTIVE)
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "csrWifiHipPacketSchedulerQueueSetStateSet: QS (VIF %d AID %d) new state=active\n",
                                qs->vif->id, qs->id));
            qs->station_state_transitions++;
            qs->vif->powersave_qs--;
            card->fh_buffer_d.packet_scheduler.powersave_qs--;
        }
    }
    else
    {
        if (state == QS_STA_POWERSAVE)
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "csrWifiHipPacketSchedulerQueueSetStateSet: QS (VIF %d AID %d) new state=powersave\n",
                                qs->vif->id, qs->id));
            qs->station_state_transitions++;
            qs->vif->powersave_qs++;
            card->fh_buffer_d.packet_scheduler.powersave_qs++;
        }
    }

    qs->station_state = state;

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerNextVifSet
 *
 *  Change the Queue Set station state (active or powersave).
 *
 *  Arguments:
 *      card         Pointer card structure.
 *      vif_id
 *      changed
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS if the state is successfully changed.
 *      CSR_RESULT_FAILURE with no error message if the state is the same as already set.
 *      CSR_RESULT_FAILURE with a log error otherwise.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerNextVifSet(card_t *card, CsrInt32 vif_id, CsrBool *changed)
{
    if (card->fh_buffer_d.packet_scheduler.next_vif != vif_id)
    {
        *changed = TRUE;
        CSR_LOG_TEXT_DEBUG((CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: csrWifiHipPacketSchedulerNextVifSet: new itag %d (was %d)\n",
                            card->instance, vif_id, card->fh_buffer_d.packet_scheduler.next_vif));
        card->fh_buffer_d.packet_scheduler.next_vif = vif_id;
    }
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerCfmRqd
 *
 *  Figure out whether a confirmation is required by the upper layer.
 *
 *  Arguments:
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
CsrBool csrWifiHipPacketSchedulerCfmRqd(struct csrWifiHipPacketSchedulerQsig *qsig)
{
    card_signal_t *signal;
    CsrUint16 signal_id;
    CsrBool res;

    res = TRUE;
    signal = qsig_get_signal_ptr(qsig);
    signal_id = GET_SIGNAL_ID(signal->sigbuf);
    switch (signal_id)
    {
        case CSR_MA_PACKET_REQUEST_ID:
            if (GET_TRANSMISSIONCONTROL_FROM_MA_PACKET_REQ(signal->sigbuf) & CSR_NO_CONFIRM_REQUIRED)
            {
                res = FALSE;
            }
            break;
        default:
            res = TRUE;
            break;
    }
    return res;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerPurgeQueue
 *
 *  Free all packets in a queue.
 *
 *  Arguments:
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerPurgeQueue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerQsig *qsig;
    CsrInt32 len, i;

    /*
     * Clear qcod for the queue - the peek function for the queue will return zero
     * if qcod exceeds qmod.
     */
    cp->qcod = 0;
    len = cp->getlen(card, cp);

    for (i = 0; i < len; i++)
    {
        qsig = cp->peek(card, cp);
        if (!qsig)
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                                CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerPurgeQueue: !qsig after %d processed\n",
                                card->instance, i));
            break;
        }
        cp->dequeue(card, cp, qsig);
        if (cp->vif && cp->vif->discard_cb && csrWifiHipPacketSchedulerCfmRqd(qsig))
        {
            cp->vif->discard_cb(card->ospriv, cp->vif->id, qsig_get_signal_ptr(qsig));
        }
        csrWifiHipPacketSchedulerFreeSignal(card, qsig);
    }
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  ps_list_add
 *
 *  Add a queue to its list.
 *
 *  Arguments:
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
static void ps_list_add(CsrWifiPacketSchedulerCommonPart *cp)
{
    csrWifiHipPacketSchedulerListAddTail(cp->list, &cp->qptr);
}

/*
 * ---------------------------------------------------------------------------
 *  ps_list_remove
 *
 *  Remove a queue from its list.
 *
 *  Arguments:
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
static void ps_list_remove(CsrWifiPacketSchedulerCommonPart *cp)
{
    csrWifiHipPacketSchedulerListTake(cp->list, &cp->qptr);
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerCommonFree
 *
 *  Dequeue for management queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerCommonFree(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                        "unifi%d: csrWifiHipPacketSchedulerCommonFree: qsig=0x%p\n",
                        card->instance, (void *) qsig));

    cp->qcod -= qsig_get_signal_datalen(qsig);
    if (!cp->active && (cp->qcod < MOD_TO_BYTES(cp->qmod)))
    {
        csrWifiHipPacketSchedulerEnableQueue(cp);
    }
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerCtrlDequeue
 *
 *  Dequeue for control queue.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to signal structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerCtrlDequeue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerCtrlQueue *ctrl_q;

    ctrl_q = (CsrWifiHipPacketSchedulerCtrlQueue *) cp;

    ps_circ_take(&(ctrl_q->pkts), qsig);

    cp->stats.dequeued_signals++;
    cp->stats.dequeued_data += qsig_get_signal_datalen(qsig);

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerCtrlPeek
 *
 *  Peek for control queue.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerCtrlPeek(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerCtrlQueue *ctrl_q;

    ctrl_q = (CsrWifiHipPacketSchedulerCtrlQueue *) cp;

    return ps_circ_peek(&(ctrl_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerCtrlGetLen
 *
 *  Length of control queue.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrInt32 csrWifiHipPacketSchedulerCtrlGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerCtrlQueue *ctrl_q;

    ctrl_q = (CsrWifiHipPacketSchedulerCtrlQueue *) cp;

    return ps_circ_len(&(ctrl_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerCtrlEnqueue
 *
 *  Enqueue for control queue.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerCtrlEnqueue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerCtrlQueue *ctrl_q;

    ctrl_q = (CsrWifiHipPacketSchedulerCtrlQueue *) cp;

    if (ps_circ_add_tail(&(ctrl_q->pkts), qsig) != CSR_RESULT_SUCCESS)
    {
        return CSR_RESULT_FAILURE;
    }

    cp->stats.queued_signals++;
    cp->stats.queued_data += qsig_get_signal_datalen(qsig);

    return ps_circ_free(&(ctrl_q->pkts)) ? CSR_RESULT_SUCCESS : CSR_WIFI_PS_RESULT_FULL;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerMgmtDequeue
 *
 *  Dequeue for management queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerMgmtDequeue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerMgmtQueue *mgmt_q;

    mgmt_q = (CsrWifiHipPacketSchedulerMgmtQueue *) cp;

    ps_circ_take(&(mgmt_q->pkts), qsig);

    cp->stats.dequeued_signals++;
    cp->stats.dequeued_data += qsig_get_signal_datalen(qsig);

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerMgmtPeek
 *
 *  Peek for management queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerMgmtPeek(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerMgmtQueue *mgmt_q;

    mgmt_q = (CsrWifiHipPacketSchedulerMgmtQueue *) cp;

    return ps_circ_peek(&(mgmt_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerMgmtGetLen
 *
 *  Length of management queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrInt32 csrWifiHipPacketSchedulerMgmtGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerMgmtQueue *mgmt_q;

    mgmt_q = (CsrWifiHipPacketSchedulerMgmtQueue *) cp;

    return ps_circ_len(&(mgmt_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerMgmtEnqueue
 *
 *  Enqueue for management queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerMgmtEnqueue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerMgmtQueue *mgmt_q;

    mgmt_q = (CsrWifiHipPacketSchedulerMgmtQueue *) cp;

    if (ps_circ_add_tail(&(mgmt_q->pkts), qsig) != CSR_RESULT_SUCCESS)
    {
        return CSR_RESULT_FAILURE;
    }

    cp->stats.queued_signals++;
    cp->stats.queued_data += qsig_get_signal_datalen(qsig);

    return ps_circ_free(&(mgmt_q->pkts)) ? CSR_RESULT_SUCCESS : CSR_WIFI_PS_RESULT_FULL;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerMulticastDequeue
 *
 *  Dequeue for multicast queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerMulticastDequeue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerMulticastQueue *multicast_q;
    CsrInt32 eosp;

    if (!qsig->ref)
    {
        /* Belts & braces. Datapath should not have signals without frames. */
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerMulticastDequeue: signal without frame\n", card->instance));
        return CSR_RESULT_FAILURE;
    }

    eosp = 0;

    multicast_q = (CsrWifiHipPacketSchedulerMulticastQueue *) cp;

    ps_circ_take(&(multicast_q->pkts), qsig);

    /* Action on last frame in queue: Set EOSP bit. */
    if (!csrWifiHipPacketSchedulerMulticastPeek(card, cp))
    {
        /* If the queue becomes empty, we set the EOSP and the QMOD become 1. */
        if (multicast_q->burstMaximum)
        {
            cp->qmod = 1;
            multicast_q->burstCurrent = 0;
        }
        eosp = 1;

        CSR_LOG_TEXT_DEBUG((
                               CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerMulticastDequeue: Setting multicast EOSP=1\n"
                               , card->instance));
    }
    else
    {
        /* If the queue still has frames, check if we are in burst mode. */
        if (multicast_q->burstMaximum)
        {
            multicast_q->burstCurrent++;
            /* We are in burst mode, have we reached the burstMaximum. */
            if (multicast_q->burstCurrent >= multicast_q->burstMaximum)
            {
                cp->qmod = 1;
                multicast_q->burstCurrent = 0;
                eosp = 1;
                CSR_LOG_TEXT_DEBUG((
                                       CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerMulticastDequeue: BurstMaximum reached for VIF %d\n"
                                       , card->instance, multicast_q->common.vif->id));

                CSR_LOG_TEXT_DEBUG((
                                       CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerMulticastDequeue: Setting multicast EOSP=1\n"
                                       , card->instance));
            }
        }
    }
    if (eosp)
    {
        CsrUint16 tc;

        tc = GET_TRANSMISSIONCONTROL_FROM_MA_PACKET_REQ(qsig->signal.sigbuf);
        tc |= CSR_END_OF_SERVICE;
        SET_TRANSMISSIONCONTROL_FROM_MA_PACKET_REQ(tc, qsig->signal.sigbuf);
        if (cp->vif->mcast_complete_cb)
        {
            cp->vif->mcast_complete_cb(card->ospriv, cp->vif->id);
        }
    }
    CSR_LOG_TEXT_DEBUG((
                           CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerMulticastDequeue: QCOD=%d\n"
                           , card->instance, cp->qcod));

    cp->stats.dequeued_signals++;
    cp->stats.dequeued_data += qsig_get_signal_datalen(qsig);

    #ifdef PS_USPACE
    csrWifiHipPacketSchedulerListAddTail(&(multicast_q->sent_pkts), &(qsig->qptr));
    #endif

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerMulticastPeek
 *
 *  Peek for multicast queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerMulticastPeek(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerMulticastQueue *multicast_q;

    multicast_q = (CsrWifiHipPacketSchedulerMulticastQueue *) cp;

    if (multicast_q->burstMaximum && (multicast_q->burstCurrent >= multicast_q->burstMaximum))
    {
        CSR_LOG_TEXT_INFO((
                              CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: Multicast burstCurrent (%d) >= burstMaximum (%d) for VIF %d\n",
                              card->instance, multicast_q->burstCurrent, multicast_q->burstMaximum, multicast_q->common.vif->id));
        return (CsrWifiHipPacketSchedulerQsig *) 0;
    }

    if (cp->qcod >= MOD_TO_BYTES(cp->qmod))
    {
        CSR_LOG_TEXT_INFO((
                              CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: Multicast QCOD (%d) >= QMOD (%d) for VIF %d\n",
                              card->instance,  multicast_q->common.vif->id, cp->qcod, MOD_TO_BYTES(cp->qmod)));
        return (CsrWifiHipPacketSchedulerQsig *) 0;
    }

    return ps_circ_peek(&(multicast_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerMulticastGetLen
 *
 *  Length of multicast queue.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrInt32 csrWifiHipPacketSchedulerMulticastGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerMulticastQueue *multicast_q;

    multicast_q = (CsrWifiHipPacketSchedulerMulticastQueue *) cp;

    return ps_circ_len(&(multicast_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerMulticastEnqueue
 *
 *  Enqueue for multicast queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerMulticastEnqueue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerMulticastQueue *multicast_q;

    multicast_q = (CsrWifiHipPacketSchedulerMulticastQueue *) cp;

    if (ps_circ_add_tail(&(multicast_q->pkts), qsig) != CSR_RESULT_SUCCESS)
    {
        return CSR_RESULT_FAILURE;
    }

    cp->stats.queued_signals++;
    cp->stats.queued_data += qsig_get_signal_datalen(qsig);
    return ps_circ_free(&(multicast_q->pkts)) ? CSR_RESULT_SUCCESS : CSR_WIFI_PS_RESULT_FULL;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerQueueMulticastBurstTrigger
 *
 *  Change the queue QMOD. The config mutex is not taken in this
 *  code as the calling HIP code takes the mutex.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      vif_id       Virtual interface ID.
 *      aid          Station ID.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerQueueMulticastBurstTrigger(card_t *card, CsrInt32 vif_id)
{
    CsrWifiHipPacketSchedulerVif *vif;

    vif = csrWifiHipPacketSchedulerGetVif(card, vif_id);
    if (!vif)
    {
        return CSR_RESULT_FAILURE;
    }
    if (vif->multicast_q.burstMaximum == 0)
    {
        return CSR_RESULT_FAILURE;
    }
    if (vif->multicast_q.common.getlen(card, &(vif->multicast_q.common)) == 0)
    {
        return CSR_RESULT_FAILURE;
    }

    if (vif->multicast_q.common.qmod > 1)
    {
        CSR_LOG_TEXT_DEBUG((CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,
                            "unifi%d: CsrWifiHipPacketSchedulerQueueMulticastQMod: qmod (%d) already set high\n", card->instance, vif->multicast_q.common.qmod));
        return CSR_RESULT_SUCCESS;
    }
    vif->multicast_q.common.qmod = PS_MCAST_QMOD;

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerAcDequeue
 *
 *  Dequeue for ac queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerAcDequeue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
    CsrWifiHipPacketSchedulerQueueSet *qs;

    ac_q = (CsrWifiHipPacketSchedulerPriorityQueue *) cp;
    qs = cp->qs;


    ps_circ_take(&(ac_q->pkts), qsig);
    if (ps_circ_len(&(ac_q->pkts)) <= CIRC_THRESHOLD)
    {
        if ((qs->max_queue_len > 0) && qs->hal_queues[ac_q->pri].entries)
        {
            card->fh_buffer_d.packet_scheduler.restock = 1;
        }
        else
        {
            if (ac_q->state == AC_QSTATE_FULL)
            {
                ac_q->state = AC_QSTATE_NOT_FULL;
                if (qs->resumeCb)
                {
                    CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                        "unifi%d: csrWifiHipPacketSchedulerAcDequeue: calling resume call back\n",
                                        card->instance));

                    qs->resumeCb(qs->pauseResumeContext, (CsrUint16) qs->vif->id, (CsrUint16) qs->id, (CsrWifiHipTrafficQueue) ac_q->pri);
                }
                CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                    "unifi%d: AC %d RUNNING\n", card->instance, ac_q->pri));
            }
        }
    }
    qs->total++;

    cp->stats.dequeued_signals++;
    cp->stats.dequeued_data += qsig_get_signal_datalen(qsig);

    if (qsig->ref != 1)
    {
        /* Belts & braces. Datapath should not have signals without frames. */
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerAcDequeues: signal without frame: qsig=0x%p ref=%d len=%d\n",
                            card->instance, qsig, qsig->ref, qsig_get_signal_datalen(qsig)));
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerAcPeek
 *
 *  Peek for ac queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerAcPeek(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;

    if (cp->qcod >= MOD_TO_BYTES(cp->qmod))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerAcPeek: qcod=%d\n",
                            card->instance, cp->qcod));
        return (CsrWifiHipPacketSchedulerQsig *) 0;
    }

    ac_q = (CsrWifiHipPacketSchedulerPriorityQueue *) cp;

    return ps_circ_peek(&(ac_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerAcGetLen
 *
 *  Length of ac queue.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrInt32 csrWifiHipPacketSchedulerAcGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;

    ac_q = (CsrWifiHipPacketSchedulerPriorityQueue *) cp;

    return ps_circ_len(&(ac_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerAcEnqueue
 *
 *  Peek for ac queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerAcEnqueue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
    CsrResult r;

    ac_q = (CsrWifiHipPacketSchedulerPriorityQueue *) cp;

    r = ps_circ_add_tail(&(ac_q->pkts), qsig);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "unifi%d: csrWifiHipPacketSchedulerAcEnqueue: Failed to add, %d free\n", card->instance, ps_circ_free(&(ac_q->pkts))));
        return CSR_RESULT_FAILURE;
    }

    cp->qs->total++;

    cp->stats.queued_signals++;
    cp->stats.queued_data += qsig_get_signal_datalen(qsig);

    if (ps_circ_free(&(ac_q->pkts)) == 0)
    {
        r = CSR_WIFI_PS_RESULT_FULL;
    }
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerAcFree
 *
 *  Dequeue for management queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerAcFree(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;

    CSR_LOG_TEXT_DEBUG((
                           CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerAcFree: qsig=%p\n",
                           card->instance,  (void *) qsig));

    qs = cp->qs;

    qs->scod -= qsig_get_signal_datalen(qsig);
    cp->qcod -= qsig_get_signal_datalen(qsig);
    if (qs->active && !cp->active && (cp->qcod < MOD_TO_BYTES(cp->qmod)))
    {
        CSR_LOG_TEXT_INFO((
                              CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: QMOD (mod=%d cod=%d): Enabled Q for VIF %d AID %d AC %d\n",
                              card->instance,
                              MOD_TO_BYTES(cp->qmod), cp->qcod,
                              ((CsrWifiHipPacketSchedulerVif *) qs->vif)->id, qs->id,
                              ((CsrWifiHipPacketSchedulerPriorityQueue *) cp)->pri));
        csrWifiHipPacketSchedulerEnableQueue(cp);
    }
    if (!qs->active)
    {
        CsrUint16 v;

        v = MOD_TO_BYTES(qs->smod);
        #ifdef CSR_WIFI_HIP_PSCHED_SMOD_HIST
        if (qs->smod > PS_POWER_SAVE_SMOD)
        {
            CsrUint16 nv;
            nv = ((v * (100 - CSR_WIFI_HIP_PSCHED_SMOD_HIST)) / 100);
            if (nv < v)
            {
                v = nv;
            }
        }
        #endif
        if (qs->scod < v)
        {
            CSR_LOG_TEXT_INFO((
                                  CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: SMOD (mod=%d cod=%d): Enabled QS for VIF %d AID %d\n",
                                  card->instance,
                                  MOD_TO_BYTES(qs->smod), qs->scod, ((CsrWifiHipPacketSchedulerVif *) qs->vif)->id, qs->id));
            csrWifiHipPacketSchedulerEnableQueueSet(qs);
        }
    }
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerRtxDequeue
 *
 *  Dequeue for RTX queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerRtxDequeue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerRtxQueue *rtx_q;

    /*
     * In single shot mode, the assumption is that the f/w will only ask for a
     * retransmit when it knows it has space to handle the packets. In this case,
     * the packet is removed from the retransmit queue. The packet is not freed -
     * that is still an activity for the HIP (clear slot).
     */

    rtx_q = (CsrWifiHipPacketSchedulerRtxQueue *) cp;

    csrWifiHipPacketSchedulerListTake(&rtx_q->pkts, &(qsig->qptr));
    if (!csrWifiHipPacketSchedulerRtxPeek(card, cp))
    {
        CSR_LOG_TEXT_DEBUG((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: No more retransmit to send, disabling RTX for QS %d AC %d\n",
                            card->instance,  cp->qs->id, rtx_q->pri));
        csrWifiHipPacketSchedulerDisableQueue(cp);
    }

    cp->stats.dequeued_signals++;
    cp->stats.dequeued_data += qsig_get_signal_datalen(qsig);

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerRtxPeek
 *
 *  Peek for RTX queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerRtxPeek(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerRtxQueue *rtx_q;

    rtx_q = (CsrWifiHipPacketSchedulerRtxQueue *) cp;

    return (CsrWifiHipPacketSchedulerQsig *) csrWifiHipPacketSchedulerListPeek(&(rtx_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerRtxGetLen
 *
 *  Length of RTX queue.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrInt32 csrWifiHipPacketSchedulerRtxGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerRtxQueue *rtx_q;

    rtx_q = (CsrWifiHipPacketSchedulerRtxQueue *) cp;

    return rtx_q->pkts.entries;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerRtxEnqueue
 *
 *  Peek for RTX queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerRtxEnqueue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerRtxQueue *rtx_q;

    rtx_q = (CsrWifiHipPacketSchedulerRtxQueue *) cp;

    csrWifiHipPacketSchedulerListAddTail(&rtx_q->pkts, &(qsig->qptr));

    cp->stats.queued_signals++;
    cp->stats.queued_data += qsig_get_signal_datalen(qsig);

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerRtxPurge
 *
 *  Discard all entries in the RTX list.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      rtx_q        Pointer to rtx queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerRtxPurge(card_t *card, CsrWifiHipPacketSchedulerRtxQueue *rtx_q)
{
    CsrWifiHipPacketSchedulerQsig *qsig;
    CsrInt32 len, i, dref;
    CsrUint16 slot, dlen;

    if (rtx_q->common.qs)
    {
        len = rtx_q->common.getlen(card, &(rtx_q->common));
        len = len < rtx_q->common.qs->max_queue_len ? len : rtx_q->common.qs->max_queue_len;
    }
    else
    {
        len = rtx_q->common.getlen(card, &(rtx_q->common));
    }
    for (i = 0; i < len; i++)
    {
        qsig = (CsrWifiHipPacketSchedulerQsig *) csrWifiHipPacketSchedulerListPeek(&(rtx_q->pkts));
        if (!qsig)
        {
            break;
        }
        csrWifiHipPacketSchedulerListTake(&rtx_q->pkts, &(qsig->qptr));
        for (dref = 0; dref < qsig->ref; dref++)
        {
            slot = GET_PACKED_DATAREF_SLOT(qsig->signal.sigbuf, dref);
            dlen = GET_PACKED_DATAREF_LEN(qsig->signal.sigbuf, dref);
            if (dlen > 0)
            {
                CSR_LOG_TEXT_DEBUG((CsrWifiHipLto,
                                    CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerRtxPurge: qsig 0x%p clearing from slot %d\n",
                                    card->instance,  qsig, slot));
                CardClearSlotQsigPtr(card, slot, qsig);
            }
            else
            {
                CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                    "unifi%d: csrWifiHipPacketSchedulerRtxPurge: qsig 0x%p NOT CLEARING ref=%d dref=%d slot=%d len=%d\n",
                                    card->instance, qsig, qsig->ref, dref, slot, dlen));
            }
        }
        if (rtx_q->common.vif->discard_cb && csrWifiHipPacketSchedulerCfmRqd(qsig))
        {
            rtx_q->common.vif->discard_cb(card->ospriv, rtx_q->common.vif->id, qsig_get_signal_ptr(qsig));
        }
        csrWifiHipPacketSchedulerFreeSignal(card, qsig);
    }
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerQsMgmtDequeue
 *
 *  Dequeue for management queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerQsMgmtDequeue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerQsMgmtQueue *mgmt_q;
    CsrWifiHipPacketSchedulerQueueSet *qs;

    mgmt_q = (CsrWifiHipPacketSchedulerQsMgmtQueue *) cp;
    qs = cp->qs;

    ps_circ_take(&(mgmt_q->pkts), qsig);

    qs->scod += qsig_get_signal_datalen(qsig);
    if (qs->scod >= MOD_TO_BYTES(qs->smod))
    {
        /* If the QS exceeds its SMOD, remove it (ie. all of its AC queues) from selection. */
        CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "unifi%d: csrWifiHipPacketSchedulerQsMgmtDequeue: SMOD (mod=%d cod=%d): Disabling QS for VIF %d AID %d\n",
                            card->instance, MOD_TO_BYTES(qs->smod), qs->scod, qs->vif->id, qs->id));
        csrWifiHipPacketSchedulerDisableQueueSet(qs);
    }

    cp->stats.dequeued_signals++;
    cp->stats.dequeued_data += qsig_get_signal_datalen(qsig);

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerQsMgmtPeek
 *
 *  Peek for QS management queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerQsMgmtPeek(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerQsMgmtQueue *mgmt_q;

    mgmt_q = (CsrWifiHipPacketSchedulerQsMgmtQueue *) cp;

    return ps_circ_peek(&(mgmt_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerQsMgmtGetLen
 *
 *  Length of QS management queue.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrInt32 csrWifiHipPacketSchedulerQsMgmtGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerQsMgmtQueue *mgmt_q;

    mgmt_q = (CsrWifiHipPacketSchedulerQsMgmtQueue *) cp;

    return ps_circ_len(&(mgmt_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerQsMgmtEnqueue
 *
 *  Enqueue for management queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerQsMgmtEnqueue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerQsMgmtQueue *mgmt_q;

    mgmt_q = (CsrWifiHipPacketSchedulerQsMgmtQueue *) cp;

    if (ps_circ_add_tail(&(mgmt_q->pkts), qsig) != CSR_RESULT_SUCCESS)
    {
        return CSR_RESULT_FAILURE;
    }

    cp->stats.queued_signals++;
    cp->stats.queued_data += qsig_get_signal_datalen(qsig);

    return ps_circ_free(&(mgmt_q->pkts)) ? CSR_RESULT_SUCCESS : CSR_WIFI_PS_RESULT_FULL;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerQsMgmtEnqueueFree
 *
 *  Dequeue for management queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS | CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerQsMgmtEnqueueFree(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerQueueSet *qs;

    CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                        "unifi%d: csrWifiHipPacketSchedulerQsMgmtEnqueueFree: qsig=%p\n", card->instance, (void *) qsig));

    qs = cp->qs;

    qs->scod -= qsig_get_signal_datalen(qsig);
    cp->qcod -= qsig_get_signal_datalen(qsig);
    if (qs->active && !cp->active && (cp->qcod < MOD_TO_BYTES(cp->qmod)))
    {
        CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "unifi%d: csrWifiHipPacketSchedulerQsMgmtEnqueueFree: QMOD (mod=%d cod=%d): Enabled mgmt Q for VIF %d AID %d\n",
                            card->instance, MOD_TO_BYTES(cp->qmod), cp->qcod, qs->vif->id, qs->id));
        csrWifiHipPacketSchedulerEnableQueue(cp);
    }
    if (!qs->active)
    {
        if (qs->scod < MOD_TO_BYTES(qs->smod))
        {
            CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                "unifi%d: csrWifiHipPacketSchedulerQsMgmtEnqueueFree: SMOD (mod=%d cod=%d): Enabled QS for VIF %d AID %d\n",
                                card->instance, MOD_TO_BYTES(qs->smod), qs->scod, qs->vif->id, qs->id));
            csrWifiHipPacketSchedulerEnableQueueSet(qs);
        }
    }
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerPendingClrDequeue
 *
 *  Dequeue for RTX queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerPendingClrDequeue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;

    pending_clr_q = (CsrWifiHipPacketSchedulerPendingClrQueue *) cp;

    csrWifiHipPacketSchedulerListTake(&pending_clr_q->pkts, &(qsig->qptr));

    cp->stats.dequeued_signals++;
    cp->stats.dequeued_data += qsig_get_signal_datalen(qsig);

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerPendingClrPeek
 *
 *  Peek for RTX queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerPendingClrPeek(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;

    pending_clr_q = (CsrWifiHipPacketSchedulerPendingClrQueue *) cp;

    return (CsrWifiHipPacketSchedulerQsig *) csrWifiHipPacketSchedulerListPeek(&(pending_clr_q->pkts));
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerPendingClrGetLen
 *
 *  Length of pending clear queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrInt32 csrWifiHipPacketSchedulerPendingClrGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;

    pending_clr_q = (CsrWifiHipPacketSchedulerPendingClrQueue *) cp;

    return pending_clr_q->pkts.entries;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerPendingClrEnqueue
 *
 *  Peek for pending clear queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      cp           Pointer to common queue structure.
 *      qsig         Pointer to qsig structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerPendingClrEnqueue(card_t *card, CsrWifiPacketSchedulerCommonPart *cp, CsrWifiHipPacketSchedulerQsig *qsig)
{
    CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;

    pending_clr_q = (CsrWifiHipPacketSchedulerPendingClrQueue *) cp;

    #ifdef LOTS_OF_DEBUG
    if (GET_PACKED_DATAREF_LEN(qsig->signal.sigbuf, 0))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "csrWifiHipPacketSchedulerPendingClrEnqueue: qsig 0x%p slot=%d len=%d QUEUED to 0x%p\n",
                            qsig, GET_PACKED_DATAREF_SLOT(qsig->signal.sigbuf, 0), GET_PACKED_DATAREF_LEN(qsig->signal.sigbuf, 0), &(pending_clr_q->pkts)));
    }
    if (GET_PACKED_DATAREF_LEN(qsig->signal.sigbuf, 1))
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "csrWifiHipPacketSchedulerPendingClrEnqueue: qsig 0x%p slot=%d len=%d QUEUED to 0x%p\n",
                            qsig, GET_PACKED_DATAREF_SLOT(qsig->signal.sigbuf, 1), GET_PACKED_DATAREF_LEN(qsig->signal.sigbuf, 1), &(pending_clr_q->pkts)));
    }
    #endif

    csrWifiHipPacketSchedulerListAddTail(&pending_clr_q->pkts, &(qsig->qptr));

    cp->stats.queued_signals++;
    cp->stats.queued_data += qsig_get_signal_datalen(qsig);

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerPendingClrPurge
 *
 *  Discard all entries in the pending clear list.
 *
 *  Arguments:
 *      card          Pointer to card structure.
 *      pending_clr_q Pointer to pending clr queue structure.
 *
 *  Returns:
 *      Pointer to qsig structure or zero.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerPendingClrPurge(card_t *card, CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q)
{
    CsrWifiHipPacketSchedulerQsig *qsig;
    CsrInt32 len, i, dref;
    CsrUint16 slot, dlen;

    if (!pending_clr_q)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerPendingClrPurge: no clr q\n",
                            card->instance));
        return CSR_RESULT_FAILURE;
    }

    len = pending_clr_q->common.getlen(card, &(pending_clr_q->common));
    for (i = 0; i < len; i++)
    {
        qsig = csrWifiHipPacketSchedulerPendingClrPeek(card, &(pending_clr_q->common));
        if (!qsig)
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                                CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerPendingClrPurge: !qsig after %d processed\n",
                                card->instance, i));
            break;
        }
        (void) pending_clr_q->common.dequeue(card, &(pending_clr_q->common), qsig);
        for (dref = 0; dref < qsig->ref; dref++)
        {
            slot = GET_PACKED_DATAREF_SLOT(qsig->signal.sigbuf, dref);
            dlen = GET_PACKED_DATAREF_LEN(qsig->signal.sigbuf, dref);
            if (dlen > 0)
            {
                CSR_LOG_TEXT_DEBUG((CsrWifiHipLto,
                                    CSR_WIFI_HIP_LTSO_PS, "unifi%d: csrWifiHipPacketSchedulerPendingClrPurge: qsig 0x%p clearing from slot %d\n",
                                    card->instance, qsig, slot));
                CardClearSlotQsigPtr(card, slot, qsig);
            }
            else
            {
                CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                    "unifi%d: csrWifiHipPacketSchedulerPendingClrPurge: qsig 0x%p NOT CLEARING ref=%d dref=%d slot=%d len=%d\n",
                                    card->instance, qsig, qsig->ref, dref, slot, dlen));
            }
        }
        if (pending_clr_q->common.vif && pending_clr_q->common.vif->discard_cb && csrWifiHipPacketSchedulerCfmRqd(qsig))
        {
            pending_clr_q->common.vif->discard_cb(card->ospriv, pending_clr_q->common.vif->id, qsig_get_signal_ptr(qsig));
        }
        csrWifiHipPacketSchedulerFreeSignal(card, qsig);
    }
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  ps_do_re_stock
 *
 *  Re-stock of Packet Scheduler buffers.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *
 *  Returns:
 *      Nothing.
 *
 * ---------------------------------------------------------------------------
 */
static void ps_do_re_stock(card_t *card)
{
    CsrInt32 vif_id, aid, ac, to_do;
    CsrWifiHipPacketSchedulerVif *vif;
    CsrInt32 total;
    CsrWifiHipPacketSchedulerQueueSet *qs;

    total = 0;
    for (vif_id = PS_MIN_VIF_ID; vif_id <= PS_MAX_VIF_ID; vif_id++)
    {
        vif = csrWifiHipPacketSchedulerGetVif(card, vif_id);
        if (vif)
        {
            for (aid = PS_MIN_AID; aid <= PS_MAX_AID; aid++)
            {
                qs = csrWifiHipPacketSchedulerGetQueueSet(card, vif_id, aid);
                if (qs && qs->max_queue_len)
                {
                    for (ac = 0; ac < PS_MAX_FAIRNESS_PRI; ac++)
                    {
                        CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
                        ac_q = &(qs->ac[ac]);
                        to_do = ps_circ_free(&(ac_q->pkts));
                        if (to_do > 0)
                        {
                            CsrInt32 i;
                            for (i = 0; i < to_do; i++)
                            {
                                CsrWifiHipPacketSchedulerQsig *qsig;
                                qsig = (CsrWifiHipPacketSchedulerQsig *) csrWifiHipPacketSchedulerListTakeFirst(&(qs->hal_queues[ac]));
                                if (qsig)
                                {
                                    CsrInt32 discard;

                                    discard = 0;

                                    /*
                                     * Check discard algorithm for AC here.
                                     */
                                    if (ac == UNIFI_TRAFFIC_Q_VO)
                                    {
                                    }
                                    CSR_LOG_TEXT_INFO((
                                                          CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: Taking from HAL queue onto cbuffer VIF %d AID %d AC %d signal=0x%p\n",
                                                          card->instance,  vif_id, aid, ac, (void *) qsig));

                                    if (ac_q->common.enqueue(card, &(ac_q->common), qsig) == CSR_RESULT_FAILURE)
                                    {
                                        CSR_LOG_TEXT_ERROR((CsrWifiHipLto,
                                                            CSR_WIFI_HIP_LTSO_PS, "unifi%d: Failed to restock VIF %d AID %d AC %d\n",
                                                            card->instance,  vif_id, aid, ac));
                                        discard = 1;
                                    }
                                    if (discard)
                                    {
                                        ac_q->common.stats.discarded_signals++;
                                        if (ac_q->common.vif->discard_cb && csrWifiHipPacketSchedulerCfmRqd(qsig))
                                        {
                                            ac_q->common.vif->discard_cb(card->ospriv, ac_q->common.vif->id, qsig_get_signal_ptr(qsig));
                                        }
                                        csrWifiHipPacketSchedulerFreeSignal(card, qsig);
                                    }
                                    else
                                    {
                                        total++;
                                    }
                                    if (ac_q->state == AC_QSTATE_FULL)
                                    {
                                        ac_q->state = AC_QSTATE_NOT_FULL;
                                        if (qs->resumeCb)
                                        {
                                            qs->resumeCb(qs->pauseResumeContext, (CsrUint16) qs->vif->id, (CsrUint16) qs->id, (CsrWifiHipTrafficQueue) ac_q->pri);
                                        }

                                        CSR_LOG_TEXT_INFO((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                                           "unifi%d: AC %d RUNNING\n",
                                                           card->instance, ac_q->pri));
                                    }
                                }
                                else
                                {
                                    break;
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    CSR_LOG_TEXT_INFO((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                       "unifi%d: ps_do_re_stock: %d\n", card->instance, total));
}

/*
 * ---------------------------------------------------------------------------
 *  ps_schedule_re_stock
 *
 *  Schedule re-stock of Packet Scheduler buffers.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *
 *  Returns:
 *      Nothing.
 *
 * ---------------------------------------------------------------------------
 */
static void ps_schedule_re_stock(card_t *card)
{
    ps_do_re_stock(card);
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipSpScheduleSignal
 *
 *  Strict Priority packet selection function.
 *
 *  Arguments:
 *      cp           Pointer to common queue structure.
 *      signal       Pointer to signal structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipSpScheduleSignal(card_t *card, CsrWifiHipPacketSchedulerList *plists, CsrInt32 pri_levels, CsrWifiHipPacketSchedulerQsig **qsig_p, CsrWifiPacketSchedulerCommonPart **cp_p, CsrUint32 window)
{
    CsrInt32 i;
    CsrWifiPacketSchedulerCommonPart *cp, *first;
    CsrWifiHipPacketSchedulerList *ql;

    *qsig_p = 0;
    *cp_p = 0;

    /*
     * Process each priority in order, draining the priority completely before moving
     * to the next priority.
     */
    for (i = 0, ql = plists; i < pri_levels; i++, ql++)
    {
        first = cp = (CsrWifiPacketSchedulerCommonPart *) csrWifiHipPacketSchedulerListGetNext(ql);
        while (cp)
        {
            CsrWifiHipPacketSchedulerQsig *qsig;
            qsig = cp->peek(card, cp);
            if (qsig && (qsig_get_signal_datalen(qsig) <= window))
            {
                *qsig_p = qsig;
                *cp_p = cp;
            }
            cp = (CsrWifiPacketSchedulerCommonPart *) csrWifiHipPacketSchedulerListGetNext(ql);
            if (*qsig_p)
            {
                return CSR_RESULT_SUCCESS;
            }
            if (cp == first)
            {
                /* Get out when we have been round once. */
                break;
            }
        }
    }
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipSpScheduleSignalComplete
 *
 *  Strict Priority packet selection complete. When a packet has been
 *  accepted by the HIP for transmission, this function completes
 *  removal of the packet from the queue and then subsequent test
 *  on the queue QMOD.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *      signal       Pointer to signal structure.
 *      cp           Pointer to common queue structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipSpScheduleSignalComplete(card_t *card, CsrWifiHipPacketSchedulerQsig *qsig, CsrWifiPacketSchedulerCommonPart *cp)
{
    cp->dequeue(card, cp, qsig);

    switch (qsig_get_signal_type(qsig))
    {
        case SIGNAL_CTRL:
            CSR_LOG_TEXT_DEBUG((
                                   CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: Sending CTRL: signal=0x%p len=%d\n",
                                   card->instance,  (void *) qsig, qsig_get_signal_datalen(qsig)));
            break;
        case SIGNAL_MGMT:
            CSR_LOG_TEXT_DEBUG((
                                   CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: Sending MGMT: VIF=%d signal=0x%p len=%d\n",
                                   card->instance, cp->vif->id, (void *) qsig, qsig_get_signal_datalen(qsig)));
            break;
        case SIGNAL_MULTICAST:
            CSR_LOG_TEXT_DEBUG((
                                   CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: Sending MULTICAST: VIF=%d signal=0x%p len=%d\n",
                                   card->instance,  cp->vif->id, (void *) qsig, qsig_get_signal_datalen(qsig)));
            break;
        case SIGNAL_UNICAST:
            CSR_LOG_TEXT_DEBUG((
                                   CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: Resending UNICAST: VIF=%d AID=%d signal=0x%p len=%d ac=%d\n",
                                   card->instance,
                                   cp->vif->id, cp->qs ? cp->qs->id : -1,
                                   (void *) qsig, qsig_get_signal_datalen(qsig), qsig_get_signal_pri(qsig)));
            break;
    }
    cp->qcod += qsig_get_signal_datalen(qsig);
    if (cp->qcod >= MOD_TO_BYTES(cp->qmod))
    {
        csrWifiHipPacketSchedulerDisableQueue(cp);
    }
    if (qsig_get_signal_datalen(qsig))
    {
        /*
         * The signal has bulk data. If it has a pending clear q pointer set (which it should),
         * place the signal on the pending clear queue.
         *
         * [ For retransmissions, if implemented, signals are taken from the pending clear queue
         * (selectively, according to f/w signals) and placed on the appropriate rtx queue ready
         * for scheduling. ]
         *
         * The various pending clear queues (control, VIF, QS) identify the set of signals yet
         * to be cleared by f/w. If a VIF, Queue Set or the entire Packet Scheduler is destroyed,
         * the signals in the relevant pending clear queues are purged, ensuring that any slot
         * clear indications received from f/w after the QS or VIF destruction do not result in
         * incorrect access of destroyed Packet Scheduler queues.
         */
        if (cp->pending_clr_q)
        {
            cp->pending_clr_q->common.enqueue(card, &(cp->pending_clr_q->common), qsig);
        }
        else
        {
            CSR_LOG_TEXT_ERROR((
                                   CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: Unable to add to pending-clr-q (undefined for queue) SP signal=0x%p len=%d\n",
                                   card->instance,  (void *) qsig, qsig_get_signal_datalen(qsig)));
        }
    }
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerAcScheduleComplete
 *
 *  This function is called after the scheduling of packets from AC queues.
 *  It can be used to restart o/s queues for a STA mode QS or to initiate
 *  a re-stocking of the QS circular buffers from HAL queues.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerAcScheduleComplete(card_t *card)
{
    if (card->fh_buffer_d.packet_scheduler.restock)
    {
        ps_schedule_re_stock(card);
        card->fh_buffer_d.packet_scheduler.restock = 0;
    }
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  init_packet_schedulers
 *
 *  Initialise the scheduler structures. There are two types of scheduler:
 *  Strict Priority (SP) and Fairess (DWRR is the implementation). There
 *  are two instances of each scheduler. The first SP priority scheduler
 *  takes care of control and management queues. The second SP priority
 *  scheduler takes care of retransmission of AC traffic. The two Faireness
 *  schedulers are high and normal priority.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS
 *
 * ---------------------------------------------------------------------------
 */
static CsrWifiHipPacketSchedulerPriorityScheduler priority_schedulers[4];
static void init_packet_schedulers(card_t *card)
{
    priority_schedulers[PRIORITY_SCHEDULER_SP_INDEX].schedule_init = 0;
    priority_schedulers[PRIORITY_SCHEDULER_SP_INDEX].schedule_signal = csrWifiHipSpScheduleSignal;
    priority_schedulers[PRIORITY_SCHEDULER_SP_INDEX].schedule_signal_complete = csrWifiHipSpScheduleSignalComplete;
    priority_schedulers[PRIORITY_SCHEDULER_SP_INDEX].schedule_complete = 0;
    priority_schedulers[PRIORITY_SCHEDULER_SP_INDEX].lists = (CsrWifiHipPacketSchedulerList *) (card->fh_buffer_d.packet_scheduler.sp);
    priority_schedulers[PRIORITY_SCHEDULER_SP_INDEX].pri = PS_MAX_SP_PRI;

    priority_schedulers[PRIORITY_SCHEDULER_SP_RTX_INDEX].schedule_init = 0;
    priority_schedulers[PRIORITY_SCHEDULER_SP_RTX_INDEX].schedule_signal = csrWifiHipSpScheduleSignal;
    priority_schedulers[PRIORITY_SCHEDULER_SP_RTX_INDEX].schedule_signal_complete = csrWifiHipSpScheduleSignalComplete;
    priority_schedulers[PRIORITY_SCHEDULER_SP_RTX_INDEX].schedule_complete = 0;
    priority_schedulers[PRIORITY_SCHEDULER_SP_RTX_INDEX].lists = (CsrWifiHipPacketSchedulerList *) (card->fh_buffer_d.packet_scheduler.sp_retransmit);
    priority_schedulers[PRIORITY_SCHEDULER_SP_RTX_INDEX].pri = PS_MAX_FAIRNESS_PRI;

    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_HI_INDEX].schedule_init = csrWifiHipPacketSchedulerDwrrInitTokenMax;
    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_HI_INDEX].schedule_signal = csrWifiHipPacketSchedulerDwrrScheduleSignal;
    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_HI_INDEX].schedule_signal_complete = csrWifiHipPacketSchedulerDwrrScheduleSignalComplete;
    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_HI_INDEX].schedule_complete = csrWifiHipPacketSchedulerAcScheduleComplete;
    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_HI_INDEX].lists = (CsrWifiHipPacketSchedulerList *) &(card->fh_buffer_d.packet_scheduler.fairness[PS_QS_PRI_HI_INDEX]);
    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_HI_INDEX].pri = PS_QS_PRI_HI_INDEX;

    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_NORMAL_INDEX].schedule_init = csrWifiHipPacketSchedulerDwrrInitTokenMax;
    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_NORMAL_INDEX].schedule_signal = csrWifiHipPacketSchedulerDwrrScheduleSignal;
    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_NORMAL_INDEX].schedule_signal_complete = csrWifiHipPacketSchedulerDwrrScheduleSignalComplete;
    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_NORMAL_INDEX].schedule_complete = csrWifiHipPacketSchedulerAcScheduleComplete;
    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_NORMAL_INDEX].lists = (CsrWifiHipPacketSchedulerList *) &(card->fh_buffer_d.packet_scheduler.fairness[PS_QS_PRI_NORM_INDEX]);
    priority_schedulers[PRIORITY_SCHEDULER_FAIRNESS_NORMAL_INDEX].pri = PS_QS_PRI_NORM_INDEX;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerGetScheduler
 *
 *  Get the scheduler structure given the index.
 *
 *  Arguments:
 *      card         Pointer to card structure.
 *
 *  Returns:
 *      Pointer to scheduler structure.
 *
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerPriorityScheduler *csrWifiHipPacketSchedulerGetScheduler(CsrInt32 index)
{
    return &priority_schedulers[index];
}
