/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2012
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

/*
 * ---------------------------------------------------------------------------
 *  FILE:     csr_wifi_hip_card_sdio_intr.c
 *
 *  PURPOSE:
 *      Interrupt processing for the UniFi SDIO driver.
 *
 *      We may need another signal queue of responses to UniFi to hold
 *      bulk data commands generated by read_to_host_signals().
 *
 * ---------------------------------------------------------------------------
 */
#undef CSR_WIFI_HIP_NOISY

#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip_card.h"
#include "csr_wifi_hip_xbv.h"

#include "csr_wifi_ps_qsig.h"
#include "csr_wifi_ps.h"
#include "csr_wifi_hip_log_text.h"


#define FH_PUSHED_HEADER_SIZE 8
#define BULK_DATA_CMD_SIZE 8

/*
 * When sending signals to unifi, the signal data is prepended with a 16 bit length field
 * (the length is that of the signal only) followed by 16 bits set to zero that is used
 * in the f/w for linking signals together.
 */
#define SIGNAL_TX_PREAMBLE_LEN 4
/*
 * When receiving signals from unifi, the signal data is prepended by the 16 bit length
 * field only.
 */
#define SIGNAL_RX_PREAMBLE_LEN 2

/* Driver compile options */

/* RECORD_LAST_TX_SIGNAL_BUFFER: If defined, the signal portion
   of from-host data will be copied to a local buffer. The local
   buffer can be dumped out through the /proc/fwlog file. This
   should not be enabled except for debug. */
/*#define RECORD_LAST_TX_SIGNAL_BUFFER 1*/

#ifdef RECORD_LAST_TX_SIGNAL_BUFFER
#define TX_SIGNAL_BUFFER_MIRROR_MAX_SIZE 512
CsrUint16 tx_signal_buffer_mirror_len = 0;
CsrUint8 tx_signal_buffer_mirror[TX_SIGNAL_BUFFER_MIRROR_MAX_SIZE];
#endif

/*
 * If the SDIO link is idle for this time (in milliseconds),
 * signal UniFi to go into Deep Sleep.
 * Valid return value of unifi_bh().
 */
#define UNIFI_DEFAULT_HOST_IDLE_TIMEOUT 5
/*
 * If the UniFi has not woken up for this time (in milliseconds),
 * signal the bottom half to take action.
 * Valid return value of unifi_bh().
 */
#define UNIFI_DEFAULT_WAKE_TIMEOUT      1000

static CsrResult do_hip(card_t *card);

static CsrResult process_th_bulk_data_command(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks);
static CsrResult process_fh_bulk_data_command_clear(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks);

static CsrResult process_clear_slot_command(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks);
static CsrResult process_clock_request(card_t *card);

static CsrResult process_fh_bulk_data_command(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks);
static CsrResult handle_fh(card_t *card);
static CsrResult handle_fh_segment(card_t *card, CsrUint32 window, CsrUint32 segment, CsrUint32 *sent);
static CsrResult read_th_buffer(card_t *card);
static CsrResult process_th_buffer(card_t *card);
static void unpack_control_buf(card_t *card, ctrl_buffer_t *ctrl, const CsrUint8 *raw);
static CsrUint16 get_chunks_for(const card_t *card, CsrUint16 len);

static CsrResult read_th_bulk_data_zc(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks);
static CsrResult process_th_signal_cmd(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks);
static CsrResult process_pad_cmd(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks);

static void read_unpack_cmd(CsrUint8 *ptr, bulk_data_cmd_t *bulk_data_cmd);

static CsrResult write_ack(card_t *card, CsrUint32 value);

static void calc_bulk_length(card_t *card, card_signal_t *signal, CsrUint32 *bulk_data_length, CsrUint16 *bulk_desc_length);

static CsrResult window_transfer(card_t *card, CsrUint8 *data, CsrUint32 len, CsrUint32 *winp, CsrUint16 *block_ctr, CsrUint32 flag, CsrUint16 handles[], CsrUint32 num_handles, CsrUint32 handle_size);

static CsrUint16 encode_header(card_t *card, CsrUint8 *ptr, CsrUint16 ctrl_size, CsrUint16 bulk_size, CsrUint16 min_win, CsrUint16 req_win);
static CsrUint32 pad_ctrl(card_t *card, CsrUint8 *ptr, CsrUint16 sig_size);

#ifdef CSR_WIFI_HIP_NOISY
CsrInt16 dump_fh_buf = 0;
#endif /* CSR_WIFI_HIP_NOISY */

typedef CsrResult (*HipCmdHandler)(card_t *, CsrUint8 *, CsrUint16 *);

struct hip_cmd_handler
{
    HipCmdHandler function;
};

/* These function pointers should be initialised at runtime
   to correspond with the zero-copy flag. */
struct hip_cmd_handler hip_cmd_handlers[] =
{
    {process_th_signal_cmd},                                   /* SDIO_CMD_SIGNAL - 0x00*/
    {process_th_bulk_data_command /*read_th_bulk_data_zc*/},   /* SDIO_CMD_TO_HOST_TRANSFER - 0x01*/
    {NULL},                                                    /* SDIO_CMD_TO_HOST_TRANSFER_ACK - 0x02 - deprecated */
    {process_fh_bulk_data_command},                            /* SDIO_CMD_FROM_HOST_TRANSFER - 0x03 */
    {NULL},                                                    /* SDIO_CMD_FROM_HOST_TRANSFER_ACK - 0x04 - deprecated */
    {process_clear_slot_command},                              /* SDIO_CMD_CLEAR_SLOT - 0x05 */
    {NULL},                                                    /* SDIO_CMD_OVERLAY_TRANSFER - 0x06 */
    {NULL},                                                    /* SDIO_CMD_OVERLAY_TRANSFER_ACK - 0x07 - deprecated */
    {process_fh_bulk_data_command_clear},                      /* SDIO_CMD_FROM_HOST_AND_CLEAR - 0x08 */
    {NULL},                                                    /* SDIO_CMD_BUFFER_HEADER - 0x09 */
    {NULL},                                                    /* UNUSED - 0x0a */
    {NULL},                                                    /* UNUSED - 0x0b */
    {NULL},                                                    /* UNUSED - 0x0c */
    {NULL},                                                    /* UNUSED - 0x0d */
    {NULL},                                                    /* UNUSED - 0x0e */
    {process_pad_cmd},                                         /* SDIO_CMD_PADDING - 0x0f */
};
#ifdef CALC_WINDOW_16_MACRO
#define calc_window_16(cur, max)           \
    ({                                        \
         CsrUint16 _win;                   \
         if (max >= cur) {                   \
             _win = max - cur;}             \
         else {                              \
             _win = (0x10000 - cur) + max;} \
         _win;                             \
     })
#else
static CsrUint16 calc_window_16(CsrUint16 cur, CsrUint16 max)
{
    CsrUint16 _win;
    if (max >= cur)
    {
        _win = max - cur;
    }
    else
    {
        _win = (0x10000 - cur) + max;
    }
    return _win;
}

#endif
#define calc_window(c) (c->config_data.block_round_size * (calc_window_16(c->fh_buffer_d.fh_pipe_current, c->ctrl_buffer.fh_pipe_limit)))

#define calc_signal_length(c, s) CSR_WIFI_HIP_ROUNDUP(s->signal_length + SIGNAL_TX_PREAMBLE_LEN, c->config_data.signal_chunk_size)

#ifdef CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE

/*
 * The unifi_debug_output buffer can be used to debug the HIP behaviour offline
 * i.e. without using the tracing functions that change the timing.
 *
 * Call unifi_debug_log_to_buf() with printf arguments to store a string into the buffer.
 * When unifi_debug_buf_dump() is called, the contents of the buffer are dumped into the trace.
 * The buffer is circular and writing to it when it is full simply means the oldest string(s)
 * are overwritten by the new one.
 * All these functions are intended to be called only from bh context.
 * They are not safe to call from multiple contexts
 * The exception is unifi_debug_request_mark() which is intended to be called from another context
 *
 * Implementation:
 * Each string of trace is stored in the buffer as a null terminated string
 * (NB this means the buffer contains the null termination for every string, changed from wifi5)
 * At all times unifi_dbgbuf_ptr points at where the next string is about to be written
 * If the wrap flag is set, then the buffer is full and new messages are overwriting older ones
 * The wrap from the top of the buffer back to the start is never done part way through a string.
 * Instead if the string does not fit in the remaining space at the top (highest offset) of the buffer then the whole string
 * is written at the bottom (unifi_debug_output[0]) of the buffer.
 * After the first wrap, the unifi_dbgbuf_ptr is assumed to point part way through an older partially-overwritten string.
 * So for a dump, we search to the end of this partial string (which is discarded) and start printing strings after it.
 *
 * After we have wrapped for the first time unifi_dbgbuf_top is always maintained to point
 * one byte beyond the terminating '\0' of the last string that we wrote before the last time we wrapped.
 * Before the first wrap, contents of unifi_dbgbuf_top are undefined.
 *
 */

#define UNIFI_DEBUG_GBUFFER_SIZE       8192
static CsrCharString unifi_debug_output[UNIFI_DEBUG_GBUFFER_SIZE];
static CsrCharString *unifi_dbgbuf_ptr = unifi_debug_output;
static CsrCharString *unifi_dbgbuf_top; /* marks where we wrapped last time, undefined until unifi_dbgbuf_wrap_flag is set */
static CsrInt16 unifi_dbgbuf_wrap_flag = 0;
static CsrUint8 unifi_dbgbuf_mark_req = 0;

/*
 * ---------------------------------------------------------------------------
 *
 *  Puts "MARK" + the passed id value into the offline trace
 *  but only when a trace function is next called in bh context.
 *  The idea is this allows another context (other than bh) to put something into the trace so
 *  that the relative timing of events can be seen in the log
 *
 *  Arguments:
 *  id                  single byte value to identify a point in the log
 *                      zero is not an allowed value (a request with 0 will do nothing)
 * ---------------------------------------------------------------------------
 */
void unifi_debug_request_mark(CsrUint8 id)
{
    if (id)
    {
        unifi_dbgbuf_mark_req = id;
    }
}

/*
 * ---------------------------------------------------------------------------
 *
 *  puts the contents of the passed buffer into the debug buffer
 * if append_flag is set attempts to append to the previous string
 * (unless too close to end of buffer, in which case, it will be a separate string regardless)
 *
 *  Arguments:
 * hex_flag          flag set to print hex string, clear to print null terminated string.
 * append_flag       if set, attempts to append to last string
 *      buff         buffer to print as hex (if hex_flag is set)
 *                   string to print (if hex_flag is clear)
 *      length       number of hex chars to print (ignored if hex_flag is clear)
 *
 * ---------------------------------------------------------------------------
 */
static void unifi_debug_misc_to_buf(CsrInt16 hex_flag, CsrInt16 append_flag, const CsrCharString *buff, CsrUint16 length)
{
    CsrCharString *dest;
    CsrUint32 string_len = hex_flag ? 2 * length : CsrStrLen(buff);
    CsrUint32 free_space = sizeof(unifi_debug_output) / sizeof(unifi_debug_output[0]) - (unifi_dbgbuf_ptr - unifi_debug_output);

    if (unifi_dbgbuf_mark_req)
    {
        CsrUint8 copy = unifi_dbgbuf_mark_req;

        unifi_dbgbuf_mark_req = 0;
        unifi_debug_log_to_buf(0, "MARK=%u", copy);
    }

    if (free_space >= string_len + 2) /* need one extra byte for unifi_dbgbuf_ptr to point to, one for final '\0' */
    {
        dest = (unifi_dbgbuf_ptr > unifi_debug_output && append_flag) ? unifi_dbgbuf_ptr - 1 : unifi_dbgbuf_ptr;
    }
    else
    {
        /* there is not enough space for the whole string so instead put it at the bottom of buffer */
        /* need one extra byte for unifi_dbgbuf_ptr to point to, one for terminating '\0' */
        if (sizeof(unifi_debug_output) > string_len + 2)
        {
            unifi_dbgbuf_wrap_flag = 1;
            unifi_dbgbuf_top = unifi_dbgbuf_ptr;
            dest = unifi_debug_output;
        }
        else
        {
            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : ignoring offline trace too large at %d chars\n", string_len));
            return;
        }
    }
    unifi_dbgbuf_ptr = dest + string_len + 1; /* advance over string and terminating null */

    if (hex_flag)
    {
        for ( ; length > 0; length--)
        {
            static const CsrCharString *hexstring = "0123456789ABCDEF";

            *dest++ = hexstring[(*buff >> 4) & 0x0f];
            *dest++ = hexstring[*buff++ & 0x0f];
        }
        *dest = '\0';
    }
    else
    {
        CsrStrCpy(dest, buff);
    }
}

/*
 * ---------------------------------------------------------------------------
 *
 * puts the contents of the passed string into the debug buffer
 * as a separate string
 * Arguments:
 *      str         string to print into buffer
 *
 * ---------------------------------------------------------------------------
 */
void unifi_debug_string_to_buf(CsrInt16 append_flag, const CsrCharString *str)
{
    unifi_debug_misc_to_buf(0, 0, str, 0);
}

/*
 * ---------------------------------------------------------------------------
 *
 * puts the contents of the passed string into the debug buffer
 * attempts to append to the previous string
 * (unless too close to end of buffer, in which case, it will be a separate string)
 *
 * Arguments:
 *      str         string to print into buffer
 *
 * ---------------------------------------------------------------------------
 */
void unifi_debug_append_to_buf(const CsrCharString *str)
{
    unifi_debug_misc_to_buf(0, 1, str, 0);
}

/*
 * ---------------------------------------------------------------------------
 * if append_flag is set attempts to append to the previous string
 * (unless too close to end of buffer, in which case, it will be a separate string regardless)
 * on most implementations likely to mean it appears later in trace on the same line rather than a new one
 * ---------------------------------------------------------------------------
 */

void unifi_debug_log_to_buf(CsrInt16 append_flag, const CsrCharString *fmt, ...)
{
    CsrInt32 len;
    CsrInt32 free_space = sizeof(unifi_debug_output) / sizeof(unifi_debug_output[0]) - (unifi_dbgbuf_ptr - unifi_debug_output);
    CsrCharString *dest;
    va_list args;

    if (unifi_dbgbuf_mark_req)
    {
        CsrUint8 copy = unifi_dbgbuf_mark_req;

        unifi_dbgbuf_mark_req = 0;
        unifi_debug_log_to_buf(0, "MARK=%u", copy);
    }

    /* if appending, point ready to overwrite terminating null of previous string.  (ignore beneficial effect on free space) */
    dest = (unifi_dbgbuf_ptr > unifi_debug_output && append_flag) ? unifi_dbgbuf_ptr - 1 : unifi_dbgbuf_ptr;

    va_start(args, fmt);
    len = CsrVsnprintf(dest, free_space, fmt, args);
    va_end(args);

    /* remember vsnprintf does not report space needed for terminating null */
    /* also need one extra byte for unifi_dbgbuf_ptr to point to, one for final '\0' */
    /* following test also copes with non standard vsnprintf, which for example (incorrectly) returns length actually printed or -1 */
    if ((len >= 0) && (free_space >= len + 2))
    {
        unifi_dbgbuf_ptr = dest + len + 1; /* advance over string and terminating null */
    }
    else
    {
        /* there is not enough space for the whole string so instead put it at the bottom of buffer */
        if (len < (CsrInt32) sizeof(unifi_debug_output) - 1) /* need one extra byte for unifi_dbgbuf_ptr to point to */
        {
            unifi_dbgbuf_wrap_flag = 1;
            unifi_dbgbuf_top = unifi_dbgbuf_ptr;
            va_start(args, fmt);
            len = CsrVsnprintf(unifi_debug_output, sizeof(unifi_debug_output), fmt, args);
            va_end(args);

            unifi_dbgbuf_ptr = unifi_debug_output + len + 1; /* advance over string and terminating null */
        }
        else
        {
            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : ignoring offline message because too large\n"));
        }
    }
}

/*
 * ---------------------------------------------------------------------------
 *  unifi_debug_hex_to_buf
 *
 *  puts the contents of the passed buffer into the debug buffer as a hex string
 *  attempts to append to the previous string
 * (unless too close to end of buffer in which case, it will be a separate string)
 *
 *  Arguments:
 *      buff         buffer to print as hex
 *      length       number of chars to print
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
void unifi_debug_hex_to_buf(const CsrUint8 *buff, CsrUint16 length)
{
    unifi_debug_misc_to_buf(1, 1, (CsrCharString *) buff, length);
}

void unifi_debug_buf_dump(void)
{
    CsrCharString *point;

    if (unifi_dbgbuf_wrap_flag)
    {
        if (unifi_dbgbuf_ptr < unifi_dbgbuf_top)
        {
            point = unifi_dbgbuf_ptr + CsrStrLen(unifi_dbgbuf_ptr) + 1; /* discard first, partially overwritten string */
            while (point < unifi_dbgbuf_top)
            {
                CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "%s", point));
                point += (CsrStrLen(point) + 1);
            }
        }
    }

    for (point = unifi_debug_output; point < unifi_dbgbuf_ptr; point += (CsrStrLen(point) + 1))
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "%s", point));
    }
}

#endif /* CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE */


/*
 * ---------------------------------------------------------------------------
 *  unifi_sdio_interrupt_handler
 *
 *      This function should be called by the OS-dependent code to handle
 *      an SDIO interrupt from the UniFi.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      None.
 *
 *  Notes: This function may be called in DRS context. In this case,
 *         tracing with the unifi_trace(), etc, is not allowed.
 * ---------------------------------------------------------------------------
 */
void unifi_sdio_interrupt_handler(card_t *card)
{
    /*
     * Set the flag to say reason for waking was SDIO interrupt.
     * Then ask the OS layer to run the unifi_bh to give attention to the UniFi.
     */
    card->hip2Stats.hip2_stats_rx_interrupts++;
    CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_interrupt_lock);
    card->bh_reason_unifi_int = 1;
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_interrupt_lock);

    if (unifi_run_bh(card->ospriv) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_sdio_interrupt_handler: unifi_run_bh() failed\n",
                               card->instance));
    }
} /*  sdio_interrupt_handler() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_configure_low_power_mode
 *
 *      This function should be called by the OS-dependent when
 *      the deep sleep signaling needs to be enabled or disabled.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *      low_power_mode  Disable/Enable the deep sleep signaling
 *      periodic_wake_mode UniFi wakes host periodically.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success or a CSR error code.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_configure_low_power_mode(card_t                       *card,
                                         enum unifi_low_power_mode     low_power_mode,
                                         enum unifi_periodic_wake_mode periodic_wake_mode)
{
    CsrResult r = CSR_RESULT_SUCCESS;

#ifdef CSR_WIFI_DRIVER_HYDRA
    /* Deep sleep is disabled on Hydra until firmware supports it */
    card->low_power_mode = UNIFI_LOW_POWER_DISABLED;
    card->periodic_wake_mode = UNIFI_PERIODIC_WAKE_HOST_DISABLED;
#else
    card->low_power_mode = low_power_mode;
    card->periodic_wake_mode = periodic_wake_mode;
#endif

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                       "unifi%d: unifi_configure_low_power_mode: new mode = %s, wake_host = %s\n",
                       card->instance,
                       (card->low_power_mode == UNIFI_LOW_POWER_DISABLED) ? "disabled" : "enabled",
                       (card->periodic_wake_mode == UNIFI_PERIODIC_WAKE_HOST_DISABLED) ? "FALSE" : "TRUE"));

    r = unifi_run_bh(card->ospriv);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_configure_low_power_mode: unifi_run_bh() failed\n",
                               card->instance));
        return r;
    }
    return CSR_RESULT_SUCCESS;
} /* unifi_configure_low_power_mode() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_force_low_power_mode
 *
 *      This function should be called by the OS-dependent when
 *      UniFi needs to be set to the low power mode (e.g. on suspend)
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success or a CSR error code.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_force_low_power_mode(card_t *card)
{
    if (card->low_power_mode == UNIFI_LOW_POWER_DISABLED)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Attempt to set mode to TORPID when lower power mode is disabled\n", card->instance));
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    return unifi_set_host_state(card, UNIFI_HOST_STATE_TORPID);
} /* unifi_force_low_power_mode() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_bh
 *
 *      This function should be called by the OS-dependent code when
 *      host and/or UniFi has requested an exchange of messages.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success or a CSR error code.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_bh(card_t *card, CsrUint32 *remaining)
{
    CsrResult r;
    CsrBool pending = FALSE;
    CsrInt32 iostate;
    enum unifi_low_power_mode low_power_mode = card->low_power_mode;
    CsrWifiRouterCtrlTrafficType traffic_type = card->ta_sampling.traffic_type;
    CsrUint32 reason_host;

    reason_host = card->bh_reason_host; /* bh_reason_host may change during unifi_bh(), so cache it */
    card->bh_reason_host = 0;
    card->bh_reason_int_wake_up_only = 0;
    CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_interrupt_lock);
    card->bh_reason_unifi = card->bh_reason_unifi_int;
    card->bh_reason_unifi_int = 0;
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_interrupt_lock);

    if (card->bh_reason_unifi)
    {
        /* This function may be ported to allow for SDIO implementations that
         * call the function driver's interrupt callback without checking INT_PENDING.
         */
        r = CsrWifiHipCheckInterrupt(card->sdio_if, &pending);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Error checking interrupt status \n", card->instance));
            goto exit;
        }
        if (!pending)
        {
            card->hip2Stats.hip2_stats_rx_spurious_interrupts++;
            card->bh_reason_unifi = 0;

            /* Spurious interrupts must be acknowledged */
            CsrSdioInterruptAcknowledge(card->sdio_if);
        }
        else
        {
            card->hip2Stats.hip2_stats_bh_sched[0]++;
        }
    }

    if (reason_host)
    {
        card->hip2Stats.hip2_stats_bh_sched[1]++;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4, "unifi%d: bh state=%d ip=0x%x (u.h=%d.%d)\n",
                        card->instance, card->host_state, pending, card->bh_reason_unifi, reason_host));

    /* Process request to raise the maximum SDIO clock */
    r = process_clock_request(card);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Error setting maximum SDIO clock\n", card->instance));
        goto exit;
    }

    /*
     * Why was the BH thread woken?
     * If it was an SDIO interrupt, UniFi is awake and we need to process it.
     * If it was a host process queueing data, then we need to awaken UniFi.
     *
     * Priority of flags is top down.
     *
     * ----------------------------------------------------------+
     *    \state|   AWAKE      |    DROWSY      |    TORPID      |
     * flag\    |              |                |                |
     * ---------+--------------+----------------+----------------|
     *          | do the host  | go to AWAKE and| go to AWAKE and|
     *   unifi  | protocol     | do the host    | do the host    |
     *          |              | protocol       | protocol       |
     * ---------+--------------+----------------+----------------|
     *          | do the host  |                |                |
     *   host   | protocol     |  do nothing    | go to DROWSY   |
     *          |              |                |                |
     * ---------+--------------+----------------+----------------|
     *          |              |                | should not     |
     *  timeout | go to TORPID | error, unifi   | occur          |
     *          |              | didn't wake up | do nothing     |
     * ----------------------------------------------------------+
     *
     * Note that if we end up in the AWAKE state we always do the host protocol.
     */
    card->prev_host_state = card->host_state;

    do
    {
        /*
         * When the host state is set to DROWSY, then we can not disable the
         * interrupts as UniFi can generate an interrupt even when the INT_ENABLE
         * register has the interrupts disabled. This interrupt will be lost.
         */
        if ((card->host_state == UNIFI_HOST_STATE_DROWSY) || (card->host_state == UNIFI_HOST_STATE_TORPID))
        {
            /*
             * If an interrupt is received set the host state to AWAKE and run the HIP.
             */
            if (pending)
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: UNIFI_HOST_STATE_%s: Set state to AWAKE.\n",
                                    card->instance, (card->host_state == UNIFI_HOST_STATE_TORPID) ? "TORPID" : "DROWSY"));

                r = unifi_set_host_state(card, UNIFI_HOST_STATE_AWAKE);
                if (r == CSR_RESULT_SUCCESS)
                {
                    /* Completed host-initiated wake from deep sleep? */
                    if (card->prev_host_state == UNIFI_HOST_STATE_DROWSY)
                    {
                        /* If the state was previously DROWSY, this interrupt was generated by the chip
                         * hardware signalling it has woken from deep sleep. Now UniFi's deep sleep
                         * control has been set to AWAKE, INT_PENDING changes to indicate whether a
                         * control buffer is pending (e.g. to-host data may be available to read).
                         */
                        r = CardPendingInt(card, &pending);
                        if (r != CSR_RESULT_SUCCESS)
                        {
                            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                   "unifi%d: Error checking awake interrupt status\n", card->instance));
                            goto exit;
                        }
                        if (pending)
                        {
                            card->hip2Stats.hip2_stats_rx_wake_with_data_interrupts++;
                        }
                        else
                        {
                            /* No control buffer pending, but chip is now awake for from-host transfer */
                            card->bh_reason_int_wake_up_only = 1;
                            card->hip2Stats.hip2_stats_rx_wake_only_interrupts++;
                        }
                    }

                    (*remaining) = 0;
                    break;
                }
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: unifi_bh: unifi_set_host_state(, UNIFI_HOST_STATE_AWAKE) failed with %d\n", card->instance,  r));
                goto exit;
            }

            /*
             * If the chip is in TORPID, and the host wants to wake it up,
             * set the host state to DROWSY and wait for the wake-up interrupt.
             */
            if ((card->host_state == UNIFI_HOST_STATE_TORPID) && reason_host)
            {
                r = unifi_set_host_state(card, UNIFI_HOST_STATE_DROWSY);
                if (r == CSR_RESULT_SUCCESS)
                {
                    /*
                     * set the timeout value to UNIFI_DEFAULT_WAKE_TIMEOUT
                     * to capture a wake error.
                     */
                    if (card->bh_reason_unifi)
                    {
                        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_bh: reason_unifi is set, but returning with timeout\n", card->instance));
                    }
                    (*remaining) = UNIFI_DEFAULT_WAKE_TIMEOUT;
                    return CSR_RESULT_SUCCESS;
                }
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_bh: unifi_set_host_state(, UNIFI_HOST_STATE_DROWSY) failed with %d\n", card->instance,  r));

                goto exit;
            }

            /*
             * If the chip is in DROWSY, and the timeout expires,
             * we need to reset the chip. This should never occur.
             * (If it does, check that the calling thread set "remaining"
             * according to the time remaining when unifi_bh() was called).
             */
            if ((card->host_state == UNIFI_HOST_STATE_DROWSY) && ((*remaining) == 0))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UniFi did not wake up on time...\n", card->instance));

                /*
                 * Check if Function1 has gone away or
                 * if we missed an SDIO interrupt.
                 */
                r = unifi_check_io_status(card, &iostate);
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_bh: unifi_check_io_status() returned with %d\n", card->instance,  r));
                if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
                {
                    goto exit;
                }
                /* Need to reset and reboot */
                return CSR_RESULT_FAILURE;
            }
        }
        else /* Currently AWAKE */
        {
            if (card->bh_reason_unifi || reason_host || card->bh_reason_unifi_int)
            {
                break;
            }

            if (((*remaining) == 0) && (low_power_mode == UNIFI_LOW_POWER_ENABLED))
            {
                card->hip2Stats.hip2_stats_state_torpid++;
                r = unifi_set_host_state(card, UNIFI_HOST_STATE_TORPID);
                if (r == CSR_RESULT_SUCCESS)
                {
                    (*remaining) = 0;
                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                       "unifi%d: unifi_bh: entered TORPID\n", card->instance));
                    return CSR_RESULT_SUCCESS;
                }
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_bh: unifi_set_host_state(, UNIFI_HOST_STATE_TORPID) returned with %d\n", card->instance,  r));

                goto exit;
            }
        }
        if (card->bh_reason_unifi)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_bh: reason_unifi is set, but returning before hip\n", card->instance));
            break;
        }
        /* No need to run the host protocol */
        return CSR_RESULT_SUCCESS;
    } while (0);

    if (pending)
    {
        r = CardClearInt(card);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: unifi_bh: couldn't clear interrupt %d\n", card->instance, r));
        }
    }

    /*** Run the host protocol ***/
    r = do_hip(card);

    /* If we handled an interrupt, we must acknowledge */
    if (card->bh_reason_unifi)
    {
        CsrSdioInterruptAcknowledge(card->sdio_if);
        card->bh_reason_unifi = 0;
    }

    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_bh: do_hip returned %d\n", card->instance, r));
        goto exit;
    }

    if (card->bh_reason_unifi_int)
    {
        /* If we have a new interrupt pending from the chip, we will
           come back after poll_period and deal with it. */
        *remaining = card->config_params.poll_period;
    }
    else
    {
        /* If Traffic Analysis says traffic is only occasional and there are
        no outstanding transmissions with the f/w (slots in use) and there
        is no traffic pending to send to the f/w, we can set a timeout for
        idle. If power save is enabled, we can also put the chip into torpid
        state. */
        (*remaining) = UNIFI_DEFAULT_HOST_IDLE_TIMEOUT;
        if ((traffic_type == CSR_WIFI_ROUTER_CTRL_TRAFFIC_TYPE_OCCASIONAL) &&
            (!card->fh_signaldata_pending) &&
            (CardAreAllFromHostDataSlotsEmpty(card) == 1))
        {
            if (low_power_mode == UNIFI_LOW_POWER_ENABLED)
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4,
                                    "unifi%d: Traffic is occasional, set unifi to TORPID immediately\n", card->instance));
                r = unifi_set_host_state(card, UNIFI_HOST_STATE_TORPID);
                if (r != CSR_RESULT_SUCCESS)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                           "unifi%d: unifi_bh: unifi_set_host_state(, UNIFI_HOST_STATE_TORPID) failed with %d\n", card->instance,  r));
                    goto exit;
                }
            }
        }
    }
exit:
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: New state=%d\n", card->instance,  card->host_state));

    /* Acknowledge interrupt on error */
    if (card->bh_reason_unifi)
    {
        CsrSdioInterruptAcknowledge(card->sdio_if);
    }

    return r;
} /* unifi_bh() */

/*
 * ---------------------------------------------------------------------------
 *  process_clock_request
 *
 *      Handle request from the OS layer to increase the SDIO clock speed.
 *      The fast clock is limited until the firmware has indicated that it has
 *      completed initialisation to the OS layer.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success or CSR error code.
 * ---------------------------------------------------------------------------
 */
static CsrResult process_clock_request(card_t *card)
{
    CsrResult r = CSR_RESULT_SUCCESS;
    CsrResult csrResult;

    if (!card->request_max_clock)
    {
        return CSR_RESULT_SUCCESS;   /* No pending request */
    }

    /*
     * The SDIO clock speed request from the OS layer is only acted upon if
     * the UniFi is awake. If it was in any other state, the clock speed will
     * transition through SAFE to MAX while the host wakes it up, and the
     * final speed reached will be UNIFI_SDIO_CLOCK_MAX_HZ.
     * This assumes that the SME never requests low power mode while the f/w
     * initialisation takes place.
     */
#ifndef CSR_WIFI_DRIVER_HYDRA
    if (card->host_state == UNIFI_HOST_STATE_AWAKE)
#endif
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Set SDIO max clock\n"
                              , card->instance));
        csrResult = CsrSdioMaxBusClockFrequencySet(card->sdio_if, UNIFI_SDIO_CLOCK_MAX_HZ);
        if (csrResult != CSR_RESULT_SUCCESS)
        {
            r = ConvertCsrSdioToCsrHipResult(card, csrResult);
        }
        else
        {
            card->sdio_clock_speed = UNIFI_SDIO_CLOCK_MAX_HZ;  /* log the new freq */
        }
    }
#ifndef CSR_WIFI_DRIVER_HYDRA
    else
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Will set SDIO max clock after wakeup\n"
                              , card->instance));
    }
#endif

    /* Cancel the request now that it has been acted upon, or is about to be
     * by the wakeup mechanism
     */
    card->request_max_clock = 0;

    return r;
}

#ifdef CSR_WIFI_HIP_DEFER_SLOT_SUPPORT
/*
 * ---------------------------------------------------------------------------
 *  hip_sig_defer_fh_slot_helper
 *
 *  Process the FH slot deferment signal.
 *  This signal indicates the slots that will be retransmitted to the
 *  f/w at on receipt of a trigger signal.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void hip_sig_defer_fh_slot_helper(card_t *card)
{
    CsrUint16 i;

    for (i = 0; i < (UNIFI_MAX_FH_BULK_SLOTS + 15) / 16; i++)
    {
        CsrUint16 f;
        f = card->ctrl_buffer.fh_defer_slot[i];
        if (f)
        {
            CsrUint16 j;
            for (j = 0; j < 16; j++)
            {
                if (f & (1 << j))
                {
                    CsrUint16 slot;

                    slot = (i * 16) + j;

                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,  "unifi%d: handle_hip_signalling: defer slot %d\n",
                                        card->instance, slot));
                }
            }
        }
    }
}

#endif /* CSR_WIFI_HIP_DEFER_SLOT_SUPPORT */

#ifdef CSR_WIFI_HIP_CLEAR_TH_SLOT_SUPPORT
/*
 * ---------------------------------------------------------------------------
 *  hip_sig_clear_th_slot_helper
 *
 *  Process the TH slot clear signal.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void hip_sig_clear_th_slot_helper(card_t *card)
{
    CsrUint16 i;

    for (i = 0; i < (UNIFI_MAX_FH_BULK_SLOTS + 15) / 16; i++)
    {
        CsrUint16 f;
        f = card->ctrl_buffer.th_clear_slot[i];
        if (f)
        {
            CsrUint16 j;
            for (j = 0; j < 16; j++)
            {
                if (f & (1 << j))
                {
                    CsrUint16 slot;

                    slot = (i * 16) + j;

                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,  "unifi%d: handle_hip_signalling: clear TH slot %d\n",
                                        card->instance, slot));
                }
            }
        }
    }
}

#endif /* CSR_WIFI_HIP_CLEAR_TH_SLOT_SUPPORT */

/*
 * ---------------------------------------------------------------------------
 *  hip_sig_clear_fh_slot_helper
 *
 *  Process the FH slot clear signal.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void hip_sig_clear_fh_slot_helper(card_t *card)
{
    CsrUint16 count, i;

    count = 0;
    for (i = 0; i < (UNIFI_MAX_FH_BULK_SLOTS + 15) / 16; i++)
    {
        CsrUint16 f;
        f = card->ctrl_buffer.fh_clear_slot[i];
        if (f)
        {
            CsrUint16 j;
            for (j = 0; j < 16; j++)
            {
                if (f & (1 << j))
                {
                    CsrUint16 slot;

                    slot = (i * 16) + j;

                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: handle_hip_signalling: clear FH slot %d\n",
                                        card->instance, slot));
                    count++;
                    CardClearFromHostDataSlot(card, slot, TRUE);
                }
            }
        }
    }
    if (count)
    {
        card->hip2Stats.hip2_stats_rx_last_pkts_per_cycle_clrs = count;
        card->hip2Stats.hip2_stats_tx_bulk_data_clrs += count;

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: handle_hip_signalling: clear slot ind count=%d\n",
                            card->instance, count));

        if (card->hip2Stats.hip2_stats_rx_last_pkts_per_cycle_clrs >= sizeof(card->hip2Stats.hip2_stats_rx_pkts_per_cycle_clrs) / sizeof(CsrUint32))
        {
            card->hip2Stats.hip2_stats_rx_pkts_per_cycle_clrs[sizeof(card->hip2Stats.hip2_stats_rx_pkts_per_cycle_clrs) / sizeof(CsrUint32) - 1]++;
        }
        else
        {
            card->hip2Stats.hip2_stats_rx_pkts_per_cycle_clrs[card->hip2Stats.hip2_stats_rx_last_pkts_per_cycle_clrs - 1]++;
        }
    }
}

/*
 * ---------------------------------------------------------------------------
 *  hip_sig_mc_helper
 *
 *  Process the multicast trigger signal.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void hip_sig_mc_helper(card_t *card)
{
    CsrUint16 mc = 0;
    CsrInt32 vif;

    if (card->ctrl_buffer.hip_signals.multicasting != card->th_buffer_d.mc)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                            "unifi%d: handle_hip_signalling: multicast state change mc=0x%04x\n", card->instance, card->ctrl_buffer.hip_signals.multicasting));
    }
    mc = card->ctrl_buffer.hip_signals.multicasting & ~card->th_buffer_d.mc;
    card->th_buffer_d.mc = card->ctrl_buffer.hip_signals.multicasting;
    if (mc)
    {
        /*
         * VIF values range from 1..7
         */
        for (vif = 1; vif < UNIFI_HOST_MAX_VIFS; vif++)
        {
            if (mc & (1 << vif))
            {
                CsrUint16 itag;

                itag = CsrWifiHipInterfaceTagGetReq(card->ospriv, vif);
                if (itag != 0xffff)
                {
                    if (csrWifiHipPacketSchedulerQueueMulticastBurstTrigger(card, itag) == CSR_RESULT_SUCCESS)
                    {
                        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                                            "unifi%d: handle_hip_signalling: VIF %d enabled for multicast\n", card->instance, vif));
                    }
                }
                else
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                                           "unifi%d: handle_hip_signalling: VIF %d not configured (mcast)\n", card->instance, vif));
                }
            }
        }
    }
}

/*
 * ---------------------------------------------------------------------------
 *  hip_sig_sta_state_helper
 *
 *  Process the station state signal.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void hip_sig_sta_state_helper(card_t *card, CsrUint16 *count)
{
    CsrInt32 vif, i;

    *count = 0;

    /*
     * VIF values range from 1..7
     */
    for (vif = 1; vif < UNIFI_HOST_MAX_VIFS; vif++)
    {
        CsrUint32 index, shift;
        CsrUint16 aid, aid_set, cfg_set, ps_set, to_ps_set, to_active_set;
        CsrUint16 itag;
        CsrUint16 last_aid_set;

        itag = CsrWifiHipInterfaceTagGetReq(card->ospriv, vif);
        if (itag == 0xffff)
        {
            continue;
        }

        cfg_set = ps_set = 0;
        /*
         * Get the configured stations and the subset of stations in power save.
         */
        if (csrWifiHipPacketSchedulerVIFGetAllQSState(card, itag, &cfg_set, &ps_set) != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                                   "unifi%d: handle_hip_signalling: csrWifiHipPacketSchedulerVIFGetAllQSState(, itag=%d,) failed\n", card->instance, itag));
            continue;
        }
        /*
         * If no stations are configured on this VIF, go to the next one.
         */
        if (cfg_set == 0)
        {
            continue;
        }
        /*
         * Eight bits per VIF.
         */
        index = (vif * UNIFI_HOST_MAX_PEER) / 16;
        shift = (vif * UNIFI_HOST_MAX_PEER) % 16;

        aid_set = (card->ctrl_buffer.hip_signals.power_saving[index] >> shift) & ((1 << UNIFI_HOST_MAX_PEER) - 1);
        last_aid_set = (card->last_power_saving[index] >> shift) & ((1 << UNIFI_HOST_MAX_PEER) - 1);
        if (aid_set == last_aid_set)
        {
            /* No state change. */
            continue;
        }
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                               "unifi%d: handle_hip_signalling: aid_set changed from 0x%04x to 0x%04x\n", card->instance, last_aid_set, aid_set));
        /*
         * The signalling bits should be the same or a subset of cfg_set.
         * There may be transitory race conditions where a station Queue Set has been
         * destroyed on the host as the result of a disassociation but the f/w sends
         * signal bits for that station for a while until the f/w record is updated.
         * To that end, mask off any bits not in cfg_set.
         */
        if ((aid_set | cfg_set) != cfg_set)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG7,
                                   "unifi%d: handle_hip_signalling: aid_set (0x%04x) != cfg_set (0x%04x)\n", card->instance, aid_set, cfg_set));
        }
        /*
         * Only interested in changes of state, so turn off QS in powersave already.
         */
        to_ps_set = aid_set & cfg_set;
        to_ps_set &= ~ps_set;
        /*
         * Find the subset of QS that were in powersave (ps_set) and are not longer.
         * The bit set in ps_set will not be set in aid_set.
         */
        to_active_set = ((aid_set ^ ps_set) & ps_set);
        /*
         * Process the signal, setting the state in the QS.
         */
        for (aid = 0; aid < UNIFI_HOST_MAX_PEER; aid++)
        {
            if (to_ps_set & (1 << aid))
            {
                if (csrWifiHipPacketSchedulerQueueSetStateSet(card, itag, aid + PS_AP_START_AID, QS_STA_POWERSAVE) == CSR_RESULT_SUCCESS)
                {
                    (*count)++;
                }
            }
            else
            {
                /* Now process the signal, setting the state in the QS. */
                if (to_active_set & (1 << aid))
                {
                    if (csrWifiHipPacketSchedulerQueueSetStateSet(card, itag, aid + PS_AP_START_AID, QS_STA_ACTIVE) == CSR_RESULT_SUCCESS)
                    {
                        (*count)++;
                    }
                }
            }
        }
    }
    /*
     * Save power save signalling bits so that a comparison can be done next time.
     */
    for (i = 0; i < UNIFI_HOST_PS_OCTETS; i++)
    {
        card->last_power_saving[i] = card->ctrl_buffer.hip_signals.power_saving[i];
    }
}

/*
 * ---------------------------------------------------------------------------
 *  handle_hip_signalling
 *
 *  Process the signalling received from the f/w to program the Packet Scheduler.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void handle_hip_signalling(card_t *card)
{
    CsrBool vif_change = FALSE;
    CsrUint16 sta_state_change;

    /*
     * VIF Scheduler:
     * This field indicates the subset of VIFs to schedule.
     */
    if (card->next_vif != card->ctrl_buffer.hip_signals.scheduling)
    {
		/*OIL138 suppress the verbose strings when wifi-ON*/
		/*
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: handle_hip_signalling: next_vif: prev=0x%04x new=0x%04x\n",
                               card->instance, card->next_vif, card->ctrl_buffer.hip_signals.scheduling));*/

		CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: handle_hip_signalling: next_vif: prev=0x%04x new=0x%04x\n",
                               card->instance, card->next_vif, card->ctrl_buffer.hip_signals.scheduling));

        (void) csrWifiHipPacketSchedulerNextVifSet(card, card->ctrl_buffer.hip_signals.scheduling, &vif_change);
        card->next_vif = card->ctrl_buffer.hip_signals.scheduling;
    }

    #ifdef CSR_WIFI_HIP_DEFER_SLOT_SUPPORT
    /*
     * Defer slot:
     * A bit map to indicate which slots to move to the retransmit queues.
     */
    hip_sig_defer_fh_slot_helper(card);
    #endif /* CSR_WIFI_HIP_DEFER_SUPPORT */

    #ifdef CSR_WIFI_HIP_CLEAR_TH_SLOT_SUPPORT
    /*
     * TH clear slot:
     *
     */
    hip_sig_clear_th_slot_helper(card);
    #endif /* CSR_WIFI_HIP_CLEAR_TH_SLOT_SUPPORT */

    /*
     * Clear slot:
     * A bit map to indicate which slots to clear.
     */
    hip_sig_clear_fh_slot_helper(card);

    /*
     * Multicast trigger:
     * This field indicates the VIFs to enable for multicast.
     */
    hip_sig_mc_helper(card);

    /*
     * Station state:
     *     active - Set SMOD to a high value.
     *     legacy power save - Set SMOD to a low value.
     *     wmm-ps - Set SMOD to 1 and set QS burst mode.
     */
    sta_state_change = 0;
    hip_sig_sta_state_helper(card, &sta_state_change);

    /*
     * UAPSD Trigger
     */

    /*
     * Retransmit Trigger
     */

    /*
     * If the state of any station has changed, redistribute the total smod.
     */
    if ((sta_state_change > 0) || (vif_change == TRUE))
    {
        csrWifiHipPacketSchedulerDistributeSMod(card);
    }
}

/*
 * ---------------------------------------------------------------------------
 *  do_hip
 *
 *      Exchange messages with UniFi
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success or CSR error code.
 * ---------------------------------------------------------------------------
 */
static CsrResult do_hip(card_t *card)
{
    CsrResult r = CSR_RESULT_SUCCESS;
    CsrUint16 cbuf_read_flag = 0;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4,
                        "unifi%d: *** START HIP CYCLE\n", card->instance));

    /* If the HIP is to run in response to a firmware interrupt, read the control buffer.
     * Don't do this in response to a wake-up interrupt though as no buffer will be pending.
     */
    if (card->bh_reason_unifi && !card->bh_reason_int_wake_up_only)
    {
        r = unifi_read_control_buffer(card);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: do_hip: unifi_read_control_buffer() failed with %d\n",
                                   card->instance,  r));
            return r;
        }
        cbuf_read_flag = 1;
    }

    card->generate_interrupt = 0;

    if (card->ctrl_buffer.th_blocks > 0)
    {
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: *** TOHOST START\n"
                               , card->instance));
        if (cbuf_read_flag == 0)
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,  "unifi%d: reading TH buffer without having read control first\n"
                                   , card->instance));
        }
        else
        {
            card->ack_outstanding = 0;
        }
        r = read_th_buffer(card);
        #ifndef CSR_WIFI_DRIVER_DEFER_TH_PROCESSING
        if (CSR_RESULT_SUCCESS == r)
        {
            if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: do_hip: CsrMutexLock() failed\n",
                                       card->instance));
                return CSR_RESULT_FAILURE;
            }
            r = process_th_buffer(card);
            CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: do_hip: process_th_buffer() failed with %d\n",
                                   card->instance,  r));
        }

        /* Reset th_blocks now that it has been dealt with. */
        card->ctrl_buffer.th_blocks = 0;

        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: *** TOHOST END\n"
                               , card->instance));
        #endif
    }
    if ((CSR_RESULT_SUCCESS == r) && (card->pending_window_update == 0) && (card->fh_signaldata_pending))
    {
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: *** FROM HOST START\n"
                               , card->instance));

        if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: do_hip: CsrMutexLock() failed\n",
                                   card->instance));
            return CSR_RESULT_FAILURE;
        }
        r = handle_fh(card);
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);

        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: *** FROM HOST END\n"
                               , card->instance));
    }

    if (card->ack_outstanding)
    {
        CsrUint16 ack;
        CSR_COPY_UINT16_TO_LITTLE_ENDIAN(card->Pm, &ack);
        r = write_ack(card, ack);
        if (r == CSR_RESULT_SUCCESS)
        {
            card->ack_outstanding = 0;
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: do_hip: write_ack() failed with %d\n",
                                   card->instance,  r));
        }
        card->hip2Stats.hip2_stats_rx_last_ack_magic = card->Pm;
    }
    else
    {
        card->hip2Stats.hip2_stats_rx_last_implicit_ack_magic = card->Pm;
    }

    #ifdef CSR_WIFI_DRIVER_DEFER_TH_PROCESSING
    if (card->ctrl_buffer.th_blocks > 0)
    {
        if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: do_hip: CsrMutexLock() failed\n",
                                   card->instance));
            return CSR_RESULT_FAILURE;
        }
        r = process_th_buffer(card);
        CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);
        /* Reset th_blocks now that it has been dealt with. */
        card->ctrl_buffer.th_blocks = 0;

        if (card->ack_outstanding)
        {
            CsrUint16 ack;
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,  "unifi%d:writing ack buf after deferred th processing\n", card->instance));
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(card->Pm, &ack);
            r = write_ack(card, ack);
            if (r == CSR_RESULT_SUCCESS)
            {
                card->ack_outstanding = 0;
            }
            else
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: do_hip: write_ack() failed with %d\n",
                                       card->instance,  r));
            }
            card->hip2Stats.hip2_stats_rx_last_ack_magic = card->Pm;
        }
        else
        {
            card->hip2Stats.hip2_stats_rx_last_implicit_ack_magic = card->Pm;
        }
    }
    #endif
    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: END HIP CYCLE ***\n"
                           , card->instance));

    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  get_chunks_for
 *
 *      Rounds the given signal length in bytes to a whole number
 *      of signal_chunk_size.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *      len             The signal length in bytes to convert
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static CsrUint16 get_chunks_for(const card_t *card, CsrUint16 len)
{
    CsrUint16 mul = card->config_data.signal_chunk_size;

    return (len + (mul - 1)) / mul;
} /* get_chunks_for() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_read_control_buffer
 *      read controll information from the hw
 *
 *  Arguments:
 *      card   Pointer to card context structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success otherwise a CSR error code
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_read_control_buffer(card_t *card)
{
    CsrResult r;
    ctrl_buffer_t ctrl_buffer;
    CsrUint16 expected_pm;

    expected_pm = card->Pm + 1;
    if (expected_pm == 0)
    {
        expected_pm = 1;
    }

    r = unifi_bulk_rw(card,
                      card->config_data.ctrl_re_read_buffer_h,
                      card->ctrl_read_buffer,
                      card->ctrl_read_buffer_len,
                      UNIFI_SDIO_READ_CONTROL);
    if (CSR_RESULT_SUCCESS != r)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read control buffer\n", card->instance));
        card->hip2Stats.hip2_stats_rx_ctrl_frames_read_error++;
        return r;
    }

    unpack_control_buf(card, &ctrl_buffer, card->ctrl_read_buffer);

    card->ack_outstanding = 1;
    card->pending_ctrl_buffer = 0;

    /* If magic is zero, or magics don't match or magic is not what we expect, nack. */
    if ((!ctrl_buffer.M1) || (expected_pm != ctrl_buffer.M1) || (ctrl_buffer.M1 != ctrl_buffer.M2))
    {
        CsrUint32 v;
        CsrUint16 ack;

        if (ctrl_buffer.M1 && (card->Pm != ctrl_buffer.M1))
        {
            /* If the magic is non-zero and it isn't the last good one, it probably
            means the f/w has overwritten a control buffer that we have failed to read. */
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read_control_buffer: Expecting magic %d, got %d\n",
                                   card->instance,  expected_pm, ctrl_buffer.M1));
            card->hip2Stats.hip2_stats_rx_ctrl_frames_missed++;
        }

        /* If we read a buffer before it is ready (due to interrupts in power save mode),
        we need to signal the f/w that the read was spurious. We do so by writing the
        last good magic number to the acknowledge handle. This will stop the f/w from assuming the
        host read a good control buffer. */

        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: unifi_read_control_buffer: magic numbers inconsistent (M1=%d M2=%d) writing last M1=%d to ack handle\n",
                              card->instance,
                              ctrl_buffer.M1, ctrl_buffer.M2, card->Pm));

        /*lint -e572*/
        CSR_COPY_UINT16_TO_LITTLE_ENDIAN(1, &ack);
        v = ack << 16;
        CSR_COPY_UINT16_TO_LITTLE_ENDIAN(card->Pm, &ack);
        v |= ack;
        if (write_ack(card, v) != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read_control_buffer: write_ack(,%d) failed\n",
                                   card->instance,  card->Pm));
        }

        card->hip2Stats.hip2_stats_rx_last_nack_magic = card->Pm;
        card->pending_ctrl_buffer = 1;
        card->hip2Stats.hip2_stats_rx_ctrl_frames_invalid++;
        return CSR_RESULT_SUCCESS;
    }


    /*
     * is control block valid
     */
    if (ctrl_buffer.th_ctrl_blocks > ctrl_buffer.th_blocks)
    {
        card->hip2Stats.hip2_stats_rx_ctrl_frames_value_error++;
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read_control_buffer: range error th_ctrl_blocks (%d) > th_blocks (%d)\n",
                               card->instance,
                               ctrl_buffer.th_ctrl_blocks, ctrl_buffer.th_blocks));
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: M1=%u th_h=%d,%d,%d,%d,%d fhp_h=%d,%d,%d,%d,%d th_blocks=%d th_ctrl_blocks=%d,fh_pipe_limit=%d fh_pipe_current=%d M2=%u\n",
                               card->instance,
                               ctrl_buffer.M1,
                               ctrl_buffer.th_handles[0],
                               ctrl_buffer.th_handles[1],
                               ctrl_buffer.th_handles[2],
                               ctrl_buffer.th_handles[3],
                               ctrl_buffer.th_handles[4],
                               ctrl_buffer.fh_pulled_handles[0],
                               ctrl_buffer.fh_pulled_handles[1],
                               ctrl_buffer.fh_pulled_handles[2],
                               ctrl_buffer.fh_pulled_handles[3],
                               ctrl_buffer.fh_pulled_handles[4],
                               ctrl_buffer.th_blocks,
                               ctrl_buffer.th_ctrl_blocks,
                               ctrl_buffer.fh_pipe_limit,
                               ctrl_buffer.fh_pipe_current,
                               ctrl_buffer.M2));
        return CSR_RESULT_SUCCESS;
    }
    else
    {
        /* sanity check values */
        if ((ctrl_buffer.th_blocks * card->config_data.block_round_size) > UNIFI_TH_BUF_SIZE)
        {
            card->hip2Stats.hip2_stats_rx_ctrl_frames_value_error++;
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read_control_buffer: th_blocks larger than buffer (%d > %d)\n",
                                   card->instance,
                                   ctrl_buffer.th_blocks * card->config_data.block_round_size, UNIFI_TH_BUF_SIZE));
        }
        else
        {
            CsrUint16 i;

            if (card->ctrl_buffer.fh_pipe_limit != ctrl_buffer.fh_pipe_limit)
            {
                CsrUint16 stati;
                CsrUint16 win;

                if (card->pending_window_update)
                {
                    card->pending_window_update = 0;
                    CSR_LOG_TEXT_DEBUG((
                                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: unifi_read_control_buffer: PENDING BULK CLEARED\n"
                                           , card->instance));
                }

                win = calc_window_16(card->ctrl_buffer.fh_pipe_limit, ctrl_buffer.fh_pipe_limit);
                stati = win / 16;
                if (stati >= sizeof(card->hip2Stats.hip2_stats_tx_window_rec) / sizeof(card->hip2Stats.hip2_stats_tx_window_rec[0]))
                {
                    stati = sizeof(card->hip2Stats.hip2_stats_tx_window_rec) / sizeof(card->hip2Stats.hip2_stats_tx_window_rec[0]) - 1;
                }
                card->hip2Stats.hip2_stats_tx_window_rec[stati]++;
            }
            else
            {
                if (card->pending_window_update)
                {
                    CSR_LOG_TEXT_DEBUG((
                                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: unifi_read_control_buffer: PENDING BULK CLEAR DEFERRED (last max ptr=%d)\n",
                                           card->instance,
                                           card->ctrl_buffer.fh_pipe_limit));
                    card->hip2Stats.hip2_stats_tx_window_deferred++;
                }
            }
            /* Update the stored control values. */
            card->ctrl_buffer = ctrl_buffer;
            card->Pm = card->ctrl_buffer.M2;
            /* Store the number of handles for to-host and from-host pulled.
            TEMP - arrays are zero terminated by f/w - explore the real need
            to know the number of handles before entering window_transfer(). */
            card->th_buffer_d.num_handles = 0;
            for (i = 0; i < SDIO_FH_PUSHED_HANDLES; i++)
            {
                if (card->ctrl_buffer.th_handles[i] == 0)
                {
                    break;
                }
                card->th_buffer_d.num_handles++;
            }

            card->fh_pulled_buffer_d.num_handles = 0;
            for (i = 0; i < SDIO_FH_PUSHED_HANDLES; i++)
            {
                if (card->ctrl_buffer.fh_pulled_handles[i] == 0)
                {
                    break;
                }
                card->fh_pulled_buffer_d.num_handles++;
            }

            r = CSR_RESULT_SUCCESS;

            card->hip2Stats.hip2_stats_rx_ctrl_bytes += (ctrl_buffer.th_ctrl_blocks * card->config_data.block_round_size);
            card->hip2Stats.hip2_stats_rx_total_bytes += (ctrl_buffer.th_blocks * card->config_data.block_round_size);

            if (CsrMutexLock(&card->fh_buffer_d.packet_scheduler.cfg_mutex) != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read_control_buffer: CsrMutexLock() failed\n",
                                       card->instance));
                return CSR_RESULT_FAILURE;
            }
            /* Now that the cached (ie. card version) of the control buffer has been updated, process the signalling. */
            handle_hip_signalling(card);
            CsrMutexUnlock(&card->fh_buffer_d.packet_scheduler.cfg_mutex);

            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                                   "unifi%d: M1=%u th_h=%d,%d fhp_h=%d,%d th_blocks=%d th_ctrl_blocks=%d,fh_pipe_limit=%d fh_pipe_current=%d M2=%u\n",
                                   card->instance,
                                   card->ctrl_buffer.M1,
                                   card->ctrl_buffer.th_handles[0],
                                   card->ctrl_buffer.th_handles[1],
                                   card->ctrl_buffer.fh_pulled_handles[0],
                                   card->ctrl_buffer.fh_pulled_handles[1],
                                   card->ctrl_buffer.th_blocks,
                                   card->ctrl_buffer.th_ctrl_blocks,
                                   card->ctrl_buffer.fh_pipe_limit,
                                   card->ctrl_buffer.fh_pipe_current,
                                   card->ctrl_buffer.M2));

            card->acks_written = 0;

            card->hip2Stats.hip2_stats_rx_ctrl_frames++;
            card->hip2Stats.pm = card->Pm;
        }
    }
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  unpack_control_buf
 *
 *  Arguments:
 *      ctrl    pointer to control_buffer struct
 *      raw     pointer to CsrUint8 initialised by a previous
 *              call to unifi_read_control_buffer()
 *
 *  Returns:
 *      void
 * ---------------------------------------------------------------------------
 */
static void unpack_control_buf(card_t *card, ctrl_buffer_t *ctrl, const CsrUint8 *raw)
{
    CsrUint16 offset, i;

    offset = 0;

    ctrl->M1 = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
    offset += 2;

    for (i = 0; i < SDIO_TH_HANDLES; i++)
    {
        ctrl->th_handles[i] = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
        offset += 2;
    }

    for (i = 0; i < SDIO_TH_HANDLES; i++)
    {
        ctrl->fh_pulled_handles[i] = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
        offset += 2;
    }
    ctrl->th_blocks = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
    offset += 2;
    ctrl->th_ctrl_blocks = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
    offset += 2;
    ctrl->fh_pipe_limit = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
    offset += 2;
    ctrl->fh_pipe_current = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
    offset += 2;
    for (i = 0; i < (UNIFI_MAX_FH_BULK_SLOTS + 15) / 16; i++)
    {
        ctrl->fh_defer_slot[i] = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
        offset += 2;
    }
    for (i = 0; i < (UNIFI_MAX_FH_BULK_SLOTS + 15) / 16; i++)
    {
        ctrl->fh_clear_slot[i] = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
        offset += 2;
    }
    for (i = 0; i < (UNIFI_MAX_TH_BULK_SLOTS + 15) / 16; i++)
    {
        ctrl->th_clear_slot[i] = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
        offset += 2;
    }
    ctrl->hip_signals.multicasting = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
    offset += 2;
    for (i = 0; i < UNIFI_HOST_PS_OCTETS; i++)
    {
        ctrl->hip_signals.power_saving[i] = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
        offset += 2;
    }
    for (i = 0; i < UNIFI_HOST_PS_OCTETS; i++)
    {
        ctrl->hip_signals.uapsd_trigs[i] = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
        offset += 2;
    }
    for (i = 0; i < UNIFI_HOST_RT_OCTETS; i++)
    {
        ctrl->hip_signals.retransmit[i] = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
        offset += 2;
    }
    ctrl->hip_signals.scheduling = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + offset);
    offset += 2;
    ctrl->M2 = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(raw + (card->config_data.ctrl_info_size - sizeof(ctrl->M2)));
}

/*
 * ---------------------------------------------------------------------------
 *  read_unpack_cmd
 *
 *      Converts a wire-formatted command to the host bulk_data_cmd_t structure.
 *
 *  Arguments:
 *      ptr             Pointer to the command
 *      bulk_data_cmd   Pointer to the host structure
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void read_unpack_cmd(CsrUint8 *ptr, bulk_data_cmd_t *bulk_data_cmd)
{
    CsrInt16 index = 0;
    CsrUint16 cmd_and_slot;

    cmd_and_slot = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(ptr + index);
    bulk_data_cmd->cmd = (cmd_and_slot >> 11) & 0xf;
    bulk_data_cmd->data_slot = cmd_and_slot & 0x7ff;
    index += SIZEOF_UINT16;
    bulk_data_cmd->len = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(ptr + index);
    index += SIZEOF_UINT16;
    bulk_data_cmd->slot_offset = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(ptr + index);
    index += SIZEOF_UINT16;
    bulk_data_cmd->pipe_ptr = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(ptr + index);
}

/*
 * ---------------------------------------------------------------------------
 *  process_th_signal_cmd
 *
 *      Passes signal and associated bulk buffers to higher layer.
 *
 *  Arguments:
 *      card            Pointer to card context struct
 *      th_buffer       Pointer to signal message
 *      chunks          Number of chunks consumed in this function
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR_RESULT_FAILURE.
 * ---------------------------------------------------------------------------
 */
static CsrResult process_th_signal_cmd(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks)
{
    CsrUint16 sig_len;
    CsrUint8 *sig_ptr;
    CsrUint16 i;
    CsrUint16 slot;
    CsrWifiHipBulkDataParam data_ptrs;
    CsrResult r = CSR_RESULT_SUCCESS;

    /* Get sig_len to calculate chunks. */
    sig_len = th_buffer[0] + ((th_buffer[1] & 0x07) << 8); /* 11 bits of length */
    if ((sig_len >= 64) || (sig_len == 0))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: process_th_signal_cmd: ERROR: signal length (%d) is out of range\n",
                               card->instance,  sig_len));
        return CSR_RESULT_FAILURE;
    }
    *chunks = get_chunks_for(card, sig_len + SIGNAL_RX_PREAMBLE_LEN);

    sig_ptr = th_buffer + SIGNAL_RX_PREAMBLE_LEN;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: process_th_signal_cmd: signal 0x%x length=%d chunks=%d\n",
                           card->instance,
                           GET_SIGNAL_ID(sig_ptr), sig_len, *chunks));


    /*
    Protocol comment:

    To pass a frame received from the radio to the host, the f/w will first encode
    a bulk data command (SDIO_CMD_TO_HOST_TRANSFER) in the to host buffer and
    subsequently an ma.data.ind signal. The bulk data command will result in the
    frame being placed into an o/s network buffer and the pointer to that network
    buffer being stored in the "to_host_data" array at index "slot". The subsequent
    ma.data.ind signal will be decoded just here and result in the network buffer
    being passed to the network stack via unifi_receive_event().
    */
    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
    {
        CsrUint16 data_len;

        data_len = GET_PACKED_DATAREF_LEN(sig_ptr, i);
        if (data_len != 0)
        {
            slot = GET_PACKED_DATAREF_SLOT(sig_ptr, i);
            if (slot >= card->config_data.bt_entries)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: process_th_signal_cmd: ERROR: slot number out of range (%d >= %d)\n",
                                       card->instance,  slot, card->config_data.bt_entries));
                return CSR_RESULT_FAILURE;
            }
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: process_th_signal_cmd: slot=%d data_len=%d netbuf=%p\n",
                                   card->instance,  slot, data_len, card->to_host_data[slot].os_net_buf_ptr));
            if (card->to_host_data[slot].os_data_ptr)
            {
                data_ptrs.d[i].os_data_ptr = card->to_host_data[slot].os_data_ptr;
                data_ptrs.d[i].os_net_buf_ptr = card->to_host_data[slot].os_net_buf_ptr;
                data_ptrs.d[i].net_buf_length = card->to_host_data[slot].net_buf_length;
                data_ptrs.d[i].data_length = data_len;
                CSR_LOG_TEXT_BUFFER_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                                           32, (CsrUint8 *) data_ptrs.d[i].os_data_ptr, "TH BULKDATA (first 32):"));
            }
            else
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: process_th_signal_cmd: ERROR: received a slot (%d), but no data in the slot\n",
                                       card->instance,  slot));
                UNIFI_INIT_BULK_DATA(&data_ptrs.d[i]);
            }
        }
        else
        {
            UNIFI_INIT_BULK_DATA(&data_ptrs.d[i]);
        }
    }

    /* Pass event to OS layer. The signal header is decoded in unifi_receive_event. */
    CSR_LOG_TEXT_BUFFER_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, sig_len, sig_ptr, "TH SIGNAL (len=%d):", sig_len));

    /*
    * Log the signal to the UDI, before call unifi_receive_event() as
    * it can modify the bulk data.
    */
    if (card->udi_hook)
    {
        card->udi_hook(card->ospriv, sig_ptr, sig_len, &data_ptrs, CSR_WIFI_HIP_LOG_UDI_DIRECTION_TO_HOST);
    }

    unifi_receive_event(card->ospriv, sig_ptr, sig_len, &data_ptrs);

    card->hip2Stats.hip2_stats_rx_last_pkts_per_cycle++;

    /*
    * Re-initialise the to_host data
    */
    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
    {
        /*
        * The slot is only valid if the length is non-zero.
        */
        if (0 != GET_PACKED_DATAREF_LEN(sig_ptr, i))
        {
            slot = GET_PACKED_DATAREF_SLOT(sig_ptr, i);
            UNIFI_INIT_BULK_DATA(&card->to_host_data[slot]);
        }
    }
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  process_th_buffer
 *
 *  Arguments:
 *      card   Pointer to card context struct
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success otherwise a CSR error code
 * ---------------------------------------------------------------------------
 */
static CsrResult process_th_buffer(card_t *card)
{
    CsrUint16 cmd;
    CsrUint16 c;
    CsrUint16 chunks;
    CsrUint8 *th_buffer = card->th_buffer_d.buffer;
    CsrUint16 chunk_size = card->config_data.signal_chunk_size;
    CsrUint32 th_buffer_chunks;
    CsrResult r = CSR_RESULT_SUCCESS;
    CsrUint32 ctrl_size_bytes;

    ctrl_size_bytes = card->ctrl_buffer.th_ctrl_blocks * card->config_data.block_round_size;
    th_buffer_chunks = ctrl_size_bytes / chunk_size;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: process_th_buffer: proc %d th chunks bulk_to_read=%d\n",
                           card->instance,  th_buffer_chunks, card->bulk_to_read));

    /* th_data needed for copy mode only */
    card->th_buffer_d.th_data = th_buffer + ctrl_size_bytes;

    card->hip2Stats.hip2_stats_rx_last_pkts_per_cycle = 0;

    /* Initialise th_cur_ptr. As bulk data commands are processed,
    th_cur_ptr is updated to the pipe pointer. The offset into the
    th buffer is calculated by subtracting the th_ptr from the pipe
    pointer in the bulk data command. Once the th signals have been
    processed, the th_ptr is updated to th_cur_ptr. */
    card->th_buffer_d.th_cur_ptr = card->th_buffer_d.th_ptr;
    c = 0;
    while (c < th_buffer_chunks)
    {
        card->hip2Stats.hip2_stats_rx_signals++;
        /* The signal header and the bulk data command header only share the first four
        bits (msb) in definition - the command value. */
        cmd = th_buffer[1] >> 4;
        if (cmd >= (sizeof(hip_cmd_handlers) / sizeof(HipCmdHandler)))
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: Unknown th sig/cmd 0x%x\n",
                                   card->instance,  cmd));
            chunks = 1;
            card->hip2Stats.hip2_stats_rx_unknown_signals++;
        }
        else
        {
            if (hip_cmd_handlers[cmd].function)
            {
                (void) hip_cmd_handlers[cmd].function(card, th_buffer, &chunks);
            }
            else
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: Unsupported th sig/cmd 0x%x\n",
                                       card->instance,  cmd));
                chunks = 1;
                card->hip2Stats.hip2_stats_rx_unsupported_signals++;
            }
        }
        th_buffer += (CsrPtrdiff) chunks * chunk_size;
        c += chunks;
    }
    card->hip2Stats.hip2_stats_rx_cycles++;

    card->hip2Stats.hip2_stats_rx_window_efficiency = (((card->ctrl_buffer.th_blocks) * 100) / (128 * card->th_buffer_d.num_handles));

    if (card->hip2Stats.hip2_stats_rx_last_pkts_per_cycle)
    {
        if (!card->hip2Stats.hip2_stats_rx_last_pkts_per_cycle_clrs)
        {
            card->hip2Stats.hip2_stats_rx_cycles_pkts++;
        }
        if (card->hip2Stats.hip2_stats_rx_last_pkts_per_cycle >= sizeof(card->hip2Stats.hip2_stats_rx_pkts_per_cycle) / sizeof(CsrUint32))
        {
            card->hip2Stats.hip2_stats_rx_pkts_per_cycle[sizeof(card->hip2Stats.hip2_stats_rx_pkts_per_cycle) / sizeof(CsrUint32) - 1]++;
        }
        else
        {
            card->hip2Stats.hip2_stats_rx_pkts_per_cycle[card->hip2Stats.hip2_stats_rx_last_pkts_per_cycle - 1]++;
        }
    }

    card->th_buffer_d.th_ptr = card->th_buffer_d.th_cur_ptr;

    if (card->bulk_to_read)
    {
        /* Belts & braces for zero copy mode. Rounding of the individual bulk data portions
           should mean that overall rounding of the bulk data has zero net effect. If that is
           true, then bulk_to_read will be zero at this point and no extra sdio read is executed.
           However, should the overall rounding result in extra padding, then that padding needs
           to be read in order to keep the f/w handle pointer correctly aligned. */
        if (!card->zero_copy)
        {
            /* bulk_to_read should only be non-zero in zero copy mode */
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Illegal bulk data pad read in copy mode\n", card->instance));
        }
        else
        {
            if (card->bulk_to_read > sizeof(card->pad_buf))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Bulk data pad exceeds pad buffer in zero copy mode\n", card->instance));
            }
            else
            {
                r = window_transfer(
                    card,
                    card->pad_buf,
                    card->bulk_to_read,
                    &card->th_buffer_d.winp,
                    0,
                    UNIFI_SDIO_READ_TO_HOST,
                    card->ctrl_buffer.th_handles,
                    card->th_buffer_d.num_handles,
                    card->buff_maximum_size_bytes);
                if (r != CSR_RESULT_SUCCESS)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: process_th_buffer: window_transfer failed with %d\n",
                                           card->instance,  r));
                }
            }
        }
    }

    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  process_clear_slot_command
 *
 *      Process a clear slot command fom the UniFi.
 *
 *  Arguments:
 *   card       Pointer to card context struct
 *   bdcmd      Pointer to bulk-data command msg from UniFi
 *
 *  Returns:
 *      0 on success, CSR error code on error
 * ---------------------------------------------------------------------------
 */
static CsrResult process_clear_slot_command(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks)
{
    bulk_data_cmd_t bdcmd = {0, 0, 0, 0, 0};

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: process_clear_slot_command: %d bytes\n",
                           card->instance,  card->config_data.signal_chunk_size));

    bdcmd.data_slot = card->config_data.bf_entries;
    read_unpack_cmd(th_buffer, &bdcmd);

    *chunks = 1;
    if (bdcmd.data_slot >= card->config_data.bf_entries)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Invalid to-host data slot in SDIO_CMD_CLEAR_SLOT: %d\n",
                               card->instance,  bdcmd.data_slot));
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: process_clear_slot_command: slot %d\n",
                           card->instance,  bdcmd.data_slot));
    CardClearFromHostDataSlot(card, bdcmd.data_slot, TRUE);
    card->hip2Stats.hip2_stats_tx_bulk_data_clrs++;

    card->hip2Stats.hip2_stats_rx_last_pkts_per_cycle_clrs++;

    return CSR_RESULT_SUCCESS;
} /* process_clear_slot_command() */

/*
 * ---------------------------------------------------------------------------
 *  process_bulk_data_command
 *
 *      Process a bulk data request from the UniFi.
 *
 *  Arguments:
 *   card       Pointer to card context struct
 *   bdcmd      Pointer to bulk-data command msg from UniFi
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, CSR error code on error
 * ---------------------------------------------------------------------------
 */
static CsrResult process_th_bulk_data_command(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks)
{
    CsrWifiHipBulkDataDesc *bdslot;
    CsrUint16 slot;
    CsrUint16 len, rounded_len;
    CsrUint16 offset;
    CsrResult r;
    bulk_data_cmd_t bdcmd = {0, 0, 0, 0, 0};
    CsrResult bulk_round;

    if (!card || !th_buffer || !chunks)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : process_th_bulk_data_command: ERROR: th_buffer=%p chunks=%p\n", th_buffer, chunks));
        return CSR_RESULT_FAILURE;
    }

    bulk_round = card->th_buffer_d.bulk_round_bytes;

    if (card->zero_copy)
    {
        return read_th_bulk_data_zc(card, th_buffer, chunks);
    }

    bdcmd.data_slot = card->config_data.bt_entries;
    read_unpack_cmd(th_buffer, &bdcmd);

    *chunks = 1;

    slot = bdcmd.data_slot;
    len = bdcmd.len;

    if (slot >= card->config_data.bt_entries)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: process_th_bulk_data_command: ERROR: slot number out of range (%d >= %d)\n",
                               card->instance,  slot, card->config_data.bt_entries));
        return CSR_RESULT_FAILURE;
    }
    if (len > card->buff_maximum_size_bytes)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: process_th_bulk_data_command: ERROR: slot length out of range (%d >= %d)\n",
                               card->instance,  len, card->buff_maximum_size_bytes));
        return CSR_RESULT_FAILURE;
    }

    /* Allocate memory for card->to_host_data[slot] bulk data here. */
    r = unifi_net_data_malloc(card->ospriv, &card->to_host_data[slot], len);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: malloc failed for th_bulk_data\n", card->instance));
        return CSR_WIFI_HIP_RESULT_NO_MEMORY;
    }

    bdslot = &card->to_host_data[slot];
    /* calculate offset from beginning of th buffer */
    offset = calc_window_16(card->th_buffer_d.th_ptr, bdcmd.pipe_ptr);

    if ((card->th_buffer_d.th_data + offset + len) > (card->th_buffer_d.buffer + ((CsrPtrdiff) card->ctrl_buffer.th_blocks * card->config_data.block_round_size)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: process_th_bulk_data_command: offset (%d) or len (%d) incorrect: bdcmd.pipe_ptr=%d th_ptr=%d\n",
                               card->instance,  offset, len, bdcmd.pipe_ptr, card->th_buffer_d.th_ptr));
        return CSR_RESULT_FAILURE;
    }

    /* Do the transfer */

    CsrMemCpy((void *) bdslot->os_data_ptr,
              card->th_buffer_d.th_data + offset,
              len);

    rounded_len = CSR_WIFI_HIP_ROUNDUP(len + card->config_data.fh_pushed_block_gap, bulk_round);

    card->th_buffer_d.th_cur_ptr += rounded_len;

    bdslot->data_length = len;

    card->hip2Stats.hip2_stats_rx_data_bytes += rounded_len;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: process_th_bulk_data_command: slot=%d netbuf=0x%p r=%d\n",
                           card->instance,  slot, bdslot->os_net_buf_ptr, r));
    return CSR_RESULT_SUCCESS;
}

static CsrResult process_pad_cmd(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks)
{
    *chunks = 1;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: process_pad_cmd: %d bytes\n",
                           card->instance,  card->config_data.signal_chunk_size));
    return CSR_RESULT_SUCCESS;
}

static CsrResult write_ack(card_t *card, CsrUint32 v)
{
    CsrResult r;

    *((CsrUint32 *) (card->ack_buffer_ptr)) = v;
    card->ack_buffer_ptr += card->config_data.ack_record_size;
    card->acks_written++;
    if (card->acks_written >= ((card->config_data.ack_buffer_blocks * card->config_data.block_round_size) / card->config_data.ack_record_size))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: write_ack: ERROR: acks_written=%d max acks=%d\n",
                               card->instance, card->acks_written,
                               ((card->config_data.ack_buffer_blocks * card->config_data.block_round_size) / card->config_data.ack_record_size)));
    }
    if (card->ack_buffer_ptr >= (card->ack_buffer + (card->config_data.ack_buffer_blocks * card->config_data.block_round_size)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_UDBG6, "unifi%d: write_ack: ack_buffer wrapping: ack_buffer=0x%p end=0x%p ptr=0x%p\n",
                               card->instance, card->ack_buffer,
                               (card->ack_buffer + (card->config_data.ack_buffer_blocks * card->config_data.block_round_size)),
                               card->ack_buffer_ptr
                               ));
        card->ack_buffer_ptr = card->ack_buffer;
    }

    CSR_LOG_TEXT_BUFFER_DEBUG((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  32, card->ack_buffer, "unifi%d: write_ack: val=0x%08x: writing %d bytes to handle %d\n",
                                  card->instance,  v, card->config_data.ack_buffer_blocks * card->config_data.block_round_size, card->config_data.ack_buffer_h));

    r = unifi_bulk_rw(card,
                      card->config_data.ack_buffer_h,
                      (void *) card->ack_buffer,
                      card->config_data.ack_buffer_blocks * card->config_data.block_round_size,
                      UNIFI_SDIO_WRITE_ACK);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: write_ack: unifi_bulk_rw failed to write handle %d with %d\n",
                               card->instance,  card->config_data.ack_buffer_h, r));
    }

    card->ack_outstanding = 0;
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  process_fh_bulk_data_command_helper
 *
 *      Process a bulk data pull request from the UniFi.
 *
 *      The data is stored in the to_host_data slot array. It was stored there
 *      as the result of a previous bulk data transfer from the unifi.
 *
 *  Arguments:
 *   card       Pointer to card context struct
 *   bdcmd      Pointer to bulk-data command msg from UniFi
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, CSR error code on error
 * ---------------------------------------------------------------------------
 */
static CsrResult process_fh_bulk_data_command_helper(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks, CsrUint16 *slot)
{
    bulk_data_cmd_t bdcmd = {0, 0, 0, 0, 0};
    CsrWifiHipBulkDataDesc *bulk_data;
    CsrResult r;
    CsrUint32 len;
    const CsrUint8 *p;

    bdcmd.data_slot = card->config_data.bf_entries;
    read_unpack_cmd(th_buffer, &bdcmd);

    if (bdcmd.data_slot >= card->config_data.bf_entries)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: process_fh_bulk_data_command_helper: slot (%d) out of range\n",
                               card->instance,  bdcmd.data_slot));
        return CSR_RESULT_FAILURE;
    }

    *chunks = 1;

    bulk_data = &card->from_host_data[bdcmd.data_slot].bd;
    if (!bulk_data)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: process_fh_bulk_data_command_helper: no data in slot %d for pulled buffer\n",
                               card->instance,  bdcmd.data_slot));
        return CSR_RESULT_FAILURE;
    }
    *slot = bdcmd.data_slot;
    p = bulk_data->os_data_ptr - bdcmd.slot_offset;
    len = bdcmd.slot_offset;
    len = CSR_WIFI_HIP_ROUNDUP(bulk_data->data_length, card->fh_buffer_d.pulled_round_bytes);
    r = window_transfer(
        card,
        (void *) p,
        len,
        &card->fh_pulled_buffer_d.winp,
        0,
        UNIFI_SDIO_WRITE_FROM_HOST_PULLED,
        card->ctrl_buffer.fh_pulled_handles,
        card->fh_pulled_buffer_d.num_handles,
        card->buff_maximum_size_bytes);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: process_fh_bulk_data_command_helper: window_transfer failed with %d\n",
                               card->instance,  r));
    }

    card->fh_pulled_buffer_d.pipe_ptr += (CsrUint16) len;

    card->hip2Stats.hip2_stats_tx_pull_bulk_data_pkts++;
    card->hip2Stats.hip2_stats_tx_pull_bulk_data_bytes += len;
    return r;
}

static CsrResult process_fh_bulk_data_command_clear(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks)
{
    CsrResult r;
    CsrUint16 slot;

    r = process_fh_bulk_data_command_helper(card, th_buffer, chunks, &slot);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }
    CardClearFromHostDataSlot(card, slot, TRUE);

    return CSR_RESULT_SUCCESS;
}

static CsrResult process_fh_bulk_data_command(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks)
{
    CsrUint16 slot;

    return process_fh_bulk_data_command_helper(card, th_buffer, chunks, &slot);
}


/*
 * ---------------------------------------------------------------------------
 *  read_th_buffer
 *
 *  Arguments:
 *      card   Pointer to card context structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success otherwise a CSR error code
 * ---------------------------------------------------------------------------
 */
static CsrResult read_th_buffer(card_t *card)
{
    CsrResult r;
    CsrUint32 ctrl_toread;
    CsrUint32 ctrl_winp;
    static CsrUint16 last_magic_read = 0;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                           "unifi%d: read_th_buffer %u blocks <- %d",
                           card->instance,
                           card->ctrl_buffer.th_blocks,
                           card->ctrl_buffer.th_handles[0]));

    if ((last_magic_read > 0) && (last_magic_read == card->Pm))
    {
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: read_th_buffer: ERROR: reading from ctrl buf %d again\n",
                               card->instance,
                               last_magic_read));
    }
    if (card->ack_outstanding)
    {
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: read_th_buffer: ERROR: ack_outstanding\n",
                               card->instance));
    }

    /* bulk_to_read is the outstanding amount in the handle to read. For copy mode,
    this will be zero. For zero copy mode, this will be set initially to the bulk data
    portion of the handle. Once the bulk data portion has been read frame by frame,
    the bulk_to_read value will the padding. Ideally, this padding wouldn't exist.
    If it does exist, we need to read it because otherwise the handle will still have
    data in it and the firmware expects the host to read all of the data. */

    /* Read the control portion first from the last handle. The f/w has left the
    handle pointer in the correct place. */
    ctrl_toread = (card->ctrl_buffer.th_ctrl_blocks * card->config_data.block_round_size);
    ctrl_toread = CSR_WIFI_HIP_ROUNDUP(ctrl_toread, card->th_buffer_d.ctrl_round_bytes);
    ctrl_winp = 0;
    card->th_buffer_d.block_ctr = 0;
    r = window_transfer(
        card,
        card->th_buffer_d.buffer,
        ctrl_toread,
        &ctrl_winp,
        &card->th_buffer_d.block_ctr,
        UNIFI_SDIO_READ_TO_HOST,
        &card->ctrl_buffer.th_handles[card->th_buffer_d.num_handles - 1],
        1,
        card->buff_maximum_size_bytes);
    if (CSR_RESULT_SUCCESS != r)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read to host buffer\n", card->instance));
    }
    last_magic_read = card->Pm;
    card->ack_outstanding = 1;
    /* After reading the control portion, the f/w will reset the handle pointer
    to the beginning, so we set our window pointer to reflect that. */
    card->th_buffer_d.winp = 0;
    /* The pulled buffer window pointer is also initialised at this point so that
    any commands that result in a pull (ie. a write to unifi) is in line with the
    f/w buffer handle. */
    card->fh_pulled_buffer_d.winp = 0;

    /* th_ctrl_size is the amount of data in control portion - needs rounding up */
    card->bulk_to_read = (CsrUint16) ((card->ctrl_buffer.th_blocks - card->ctrl_buffer.th_ctrl_blocks) * card->config_data.block_round_size);
    if (!card->zero_copy)
    {
        r = window_transfer(
            card,
            card->th_buffer_d.buffer + ctrl_toread,
            card->bulk_to_read,
            &card->th_buffer_d.winp,
            &card->th_buffer_d.block_ctr,
            UNIFI_SDIO_READ_TO_HOST,
            card->ctrl_buffer.th_handles,
            card->th_buffer_d.num_handles,
            card->buff_maximum_size_bytes);
        if (CSR_RESULT_SUCCESS != r)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read to host buffer\n", card->instance));
        }
        card->bulk_to_read = 0;
    }

    CSR_LOG_TEXT_BUFFER_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                               (CsrUint16) (card->ctrl_buffer.th_ctrl_blocks * card->config_data.block_round_size),
                               card->th_buffer_d.buffer, "TH CTRL:"));
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                        "unifi%d: read_th_buffer: after unifi_bulk_rw(ctrl_toread=%d,bulk_to_read=%d): r=%d\n",
                        card->instance,  ctrl_toread, card->bulk_to_read, r));
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  read_th_bulk_data_zc
 *
 *      Process a bulk data request from the UniFi.
 *
 *  Arguments:
 *   card       Pointer to card context struct
 *   bdcmd      Pointer to bulk-data command msg from UniFi
 *   cmd, len   Decoded values of command and length from the msg header
 *              Cmd will only be one of:
 *                      SDIO_CMD_TO_HOST_TRANSFER
 *                      SDIO_CMD_FROM_HOST_TRANSFER
 *                      SDIO_CMD_FROM_HOST_AND_CLEAR
 *                      SDIO_CMD_OVERLAY_TRANSFER
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, CSR error code on error
 * ---------------------------------------------------------------------------
 */

static CsrResult read_th_bulk_data_zc(card_t *card, CsrUint8 *th_buffer, CsrUint16 *chunks)
{
    CsrWifiHipBulkDataDesc *bdslot;
    CsrUint16 slot;
    CsrUint16 toread;
    CsrUint16 len;
    CsrResult r;
    CsrResult bulk_round = card->th_buffer_d.bulk_round_bytes;
    bulk_data_cmd_t bdcmd = {0, 0, 0, 0, 0};

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: read_th_bulk_data_zc: %d bytes\n",
                           card->instance,  card->config_data.signal_chunk_size));

    slot = card->config_data.bt_entries;
    read_unpack_cmd(th_buffer, &bdcmd);

    *chunks = 1;

    slot = bdcmd.data_slot;
    len = bdcmd.len;

    if (slot >= card->config_data.bt_entries)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: read_th_bulk_data_zc: ERROR: slot number out of range (%d >= %d)\n",
                               card->instance,  slot, card->config_data.bt_entries));
        return CSR_RESULT_FAILURE;
    }
    if (len > card->buff_maximum_size_bytes)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: read_th_bulk_data_zc: ERROR: slot length out of range (%d >= %d)\n",
                               card->instance,  len, card->buff_maximum_size_bytes));
        return CSR_RESULT_FAILURE;
    }
    /* Round off what to read using the configured rounding value. */
    toread = CSR_WIFI_HIP_ROUNDUP(len + card->config_data.fh_pushed_block_gap, bulk_round);

    /* Allocate memory for card->to_host_data[slot] bulk data here. */
    r = unifi_net_data_malloc(card->ospriv, &card->to_host_data[slot], toread);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: malloc failed for th_bulk_data\n", card->instance));
        return CSR_WIFI_HIP_RESULT_NO_MEMORY;
    }

    bdslot = &card->to_host_data[slot];
    r = window_transfer(
        card,
        (CsrUint8 *) bdslot->os_data_ptr,
        toread,
        &card->th_buffer_d.winp,
        &card->th_buffer_d.block_ctr,
        UNIFI_SDIO_READ_TO_HOST,
        card->ctrl_buffer.th_handles,
        card->th_buffer_d.num_handles,
        card->buff_maximum_size_bytes);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: read_th_bulk_data_zc: window_transfer failed with %d\n",
                               card->instance,  r));
        return r;
    }

    card->bulk_to_read -= toread;
    card->hip2Stats.hip2_stats_rx_data_bytes += toread;

    /* the length in the host buffer is already set to the correct length.
    The host buffer needs to be aligned with the bulk offset value
    because in zero copy mode, the junk before the data is written
    to the host buffer and the data starts at the calculated relative
    offset. In the copy mode, this is not necessary since the data
    is copied to the right place (ie. the beginning of the host buffer
    data area). */
    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: read_th_bulk_data_zc: slot=%d netbuf=0x%p bulk_to_read=%d r=%d\n",
                           card->instance,  slot, bdslot->os_net_buf_ptr, card->bulk_to_read, r));

    card->th_buffer_d.th_cur_ptr = bdcmd.pipe_ptr + toread;

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  calc_bulk_length
 *
 *
 *  Arguments:
 *      card             Pointer to card context structure.
 *      signal           Pointer to signal structure.
 *      bulk_data_length Calculated length of bulk data associated with signal.
 *      bulk_desc_length Calculated length of bulk descriptors needed for signal.
 *
 *  Returns:
 *      void
 * ---------------------------------------------------------------------------
 */
static void calc_bulk_length(card_t *card, card_signal_t *signal, CsrUint32 *bulk_data_length, CsrUint16 *bulk_desc_length)
{
    CsrUint16 i;

    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
    {
        if (signal->bulkdata[i].data_length && signal->bulkdata[i].os_data_ptr)
        {
            *bulk_data_length += CSR_WIFI_HIP_ROUNDUP(signal->bulkdata[i].data_length + card->config_data.fh_pushed_block_gap, card->config_data.block_round_size);
            *bulk_desc_length += CSR_WIFI_HIP_ROUNDUP(BULK_DATA_CMD_SIZE, card->config_data.signal_chunk_size);
        }
    }
    if (*bulk_data_length != 0)
    {
        CsrUint32 len;
        len = *bulk_data_length;
        *bulk_data_length = CSR_WIFI_HIP_ROUNDUP(len, card->config_data.block_round_size);
    }
}

/*
 * ---------------------------------------------------------------------------
 *  encode_bulk_desc
 *
 *
 *  Arguments:
 *      card        Pointer to card context structure.
 *      ptr         Pointer to buffer.
 *      slot        Slot number to encode in bulk descriptor.
 *      slot_offset Offset to data.
 *      len         Length of data held in slot.
 *      pipe_ptr    Current fh pipe pointer in bytes.
 *
 *  Returns:
 *      Number of bytes written to ptr.
 * ---------------------------------------------------------------------------
 */
static CsrUint16 encode_bulk_desc(card_t *card, CsrUint8 *ptr, CsrUint16 slot, CsrUint16 slot_offset, CsrUint16 len, CsrUint16 pipe_ptr)
{
    CsrUint16 cmd_and_slot;
    CsrUint8 *p;

    p = ptr;
    cmd_and_slot = (SDIO_CMD_FROM_HOST_TRANSFER << 12) | slot;
    CSR_COPY_UINT16_TO_LITTLE_ENDIAN(cmd_and_slot, p);

    p += 2;
    CSR_COPY_UINT16_TO_LITTLE_ENDIAN(len, p);

    p += 2;
    CSR_COPY_UINT16_TO_LITTLE_ENDIAN(slot_offset, p);

    p += 2;
    CSR_COPY_UINT16_TO_LITTLE_ENDIAN(pipe_ptr, p);

    CSR_LOG_TEXT_BUFFER_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, BULK_DATA_CMD_SIZE, ptr, "FH BULK DESC:"));

    return CSR_WIFI_HIP_ROUNDUP(BULK_DATA_CMD_SIZE, card->config_data.signal_chunk_size);
}

/*
 * ---------------------------------------------------------------------------
 *  encode_signal
 *
 *
 *  Arguments:
 *      card   Pointer to card context structure.
 *      ptr    Pointer to buffer.
 *      signal Pointer to signal structure.
 *
 *  Returns:
 *      Number of bytes written to ptr.
 * ---------------------------------------------------------------------------
 */
CsrUint16 encode_signal(card_t *card, CsrUint8 *ptr, card_signal_t *signal)
{
    ptr[0] = signal->signal_length & 0xff;
    ptr[1] = ((signal->signal_length >> 8) & 0xf) | (SDIO_CMD_SIGNAL << 4);
    ptr[2] = 0;
    ptr[3] = 0;
    CsrMemCpy(ptr + SIGNAL_TX_PREAMBLE_LEN, signal->sigbuf, signal->signal_length);

    CSR_LOG_TEXT_BUFFER_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, signal->signal_length + SIGNAL_TX_PREAMBLE_LEN, ptr, "FH SIGNAL:"));

    return CSR_WIFI_HIP_ROUNDUP((signal->signal_length + SIGNAL_TX_PREAMBLE_LEN), card->config_data.signal_chunk_size);
}

/*
 * ---------------------------------------------------------------------------
 *  calc_more_values
 *
 *
 *  Arguments:
 *      card    Pointer to card context structure.
 *      window  Current available window
 *      towrite Size of segment about to be written.
 *      min     Calculated minimum window needed.
 *      max     Calculated requested or maximum window.
 *
 *  Returns:
 *      void
 * ---------------------------------------------------------------------------
 */
#define MIN_TH_WINDOW_SIZE_BYTES (2 * 0x400)

static void calc_more_values(card_t *card, CsrUint32 window, CsrUint32 towrite, CsrUint16 *min, CsrUint16 *max)
{
    CsrUint32 minimum_data_size;
    CsrUint32 required_data_size;
    CsrUint32 whats_left;
    CsrUint32 stati;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4, "unifi%d: calc_more_values: towrite %d sp %d bp %d\n",
                        card->instance, towrite, card->fh_signaldata_pending, card->fh_bulkdata_pending));

    minimum_data_size = MIN_TH_WINDOW_SIZE_BYTES;
    if ((minimum_data_size - card->config_data.block_round_size) <= (window - towrite))
    {
        *min = 0;
        *max = 0;
        return;
    }

    whats_left = (card->fh_buffer_d.max_window - (card->fh_buffer_d.winp + towrite));
    if (whats_left >= card->fh_buffer_d.winp)
    {
        required_data_size = whats_left;
    }
    else
    {
        required_data_size = card->fh_buffer_d.winp;
    }

    if (required_data_size <= minimum_data_size)
    {
        required_data_size = minimum_data_size;
    }

    *min = (minimum_data_size) / card->config_data.block_round_size;
    *max = (required_data_size) / card->config_data.block_round_size;

    stati = (*max) / 16;
    if (stati >= sizeof(card->hip2Stats.hip2_stats_tx_window_req) / sizeof(card->hip2Stats.hip2_stats_tx_window_req[0]))
    {
        stati = sizeof(card->hip2Stats.hip2_stats_tx_window_req) / sizeof(card->hip2Stats.hip2_stats_tx_window_req[0]) - 1;
    }
    card->hip2Stats.hip2_stats_tx_window_req[stati]++;
}

/*
 * ---------------------------------------------------------------------------
 *  encode_header
 *
 *
 *  Arguments:
 *      card      Pointer to card context structure.
 *      ptr       Pointer to a char buffer
 *      ctrl_size Size of the control (signals) portion of the from-host buffer.
 *      bulk_size Size of the bulk data portion of the from-host buffer.
 *      min_win   Minimum window size required in rounds (pages)
 *      req_win   Requested window size in rounds (pages)
 *
 *  Returns:
 *      Number of bytes written to the buffer ptr.
 * ---------------------------------------------------------------------------
 */
static CsrUint16 encode_header(card_t *card, CsrUint8 *ptr, CsrUint16 ctrl_size, CsrUint16 bulk_size, CsrUint16 min_win, CsrUint16 req_win)
{
    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: encode_header: ctrl_size=%d bulk_size=%d total=%d\n",
                           card->instance,  ctrl_size, bulk_size, ctrl_size + bulk_size));
    /*
    * construct header
    */
    ptr[1] = (SDIO_CMD_BUFFER_HEADER << 4);
    if (min_win)
    {
        ptr[0] = min_win & 0xff;
        ptr[1] |= (min_win >> 8) & 0xff;
    }
    else
    {
        ptr[0] = 0;
    }

    /*
    * size of buffer
    */
    ptr[2] = req_win & 0xff;
    ptr[3] = (req_win >> 8) & 0xff;

    /*
    * size of fh pushed signals, HIP cmds, buffer header
    * and required padding
    */
    ptr[4] = ctrl_size & 0xff;
    ptr[5] = ctrl_size >> 8 & 0xff;

    /*
    * size of pushed bulk data and required padding
    */
    ptr[6] = bulk_size & 0xff;
    ptr[7] = (bulk_size >> 8) & 0xff;

    return FH_PUSHED_HEADER_SIZE;
}

/*
 * ---------------------------------------------------------------------------
 *  pad_ctrl
 *
 *
 *  Arguments:
 *      card     Pointer to card context structure.
 *      ptr      Pointer to a char buffer
 *      sig_size Number of bytes already written to buffer.
 *
 *  Returns:
 *      Number of bytes written to the buffer ptr.
 * ---------------------------------------------------------------------------
 */
static CsrUint32 pad_ctrl(card_t *card, CsrUint8 *ptr, CsrUint16 sig_size)
{
    CsrUint16 sig_units;
    CsrUint16 padding_chunks;
    CsrUint8 *pad_ptr;
    CsrUint16 ctrl_size, i;

    pad_ptr = ptr;

    /* Figure out how many pad commands to write */
    sig_units = card->config_data.signal_chunk_size;
    ctrl_size = CSR_WIFI_HIP_ROUNDUP(sig_size, card->config_data.block_round_size);
    padding_chunks = (ctrl_size - sig_size) / sig_units;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: pad_ctrl: sig_size=%d ctrl_size=%d padding_chunks=%d\n",
                           card->instance,  sig_size, ctrl_size, padding_chunks));

    if (padding_chunks)
    {
        CsrMemSet(pad_ptr, 0, (CsrUint16) (padding_chunks * sig_units));

        for (i = 0; i < padding_chunks; i++)
        {
            /* encode little endian for unifi */
            pad_ptr[1] = SDIO_CMD_PADDING << 4;
            pad_ptr += sig_units;
        }
    }
    card->hip2Stats.hip2_stats_tx_pad_bytes += (padding_chunks * sig_units);
    return padding_chunks * sig_units;
}

/*
 * ---------------------------------------------------------------------------
 *  window_transfer
 *
 *  Arguments:
 *      card        Pointer to card context structure.
 *      data        Pointer to the data buffer to either copy to or copy from.
 *      len         Number of bytes to copy.
 *      winp        Pointer to the running window position.
 *      block_ctr   Pointer to block counter.
 *      flag        UNIFI_SDIO_READ or UNIFI_SDIO_WRITE
 *      handles     Pointer to an array of Unifi f/w handles
 *      num_handles Number of elements in handles array.
 *      handle_size Number of bytes in each handle.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success otherwise a CSR error code
 * ---------------------------------------------------------------------------
 */
static CsrResult window_transfer(card_t *card, CsrUint8 *data, CsrUint32 len, CsrUint32 *winp, CsrUint16 *block_ctr, CsrUint32 flag, CsrUint16 handles[], CsrUint32 num_handles, CsrUint32 handle_size)
{
    CsrUint32 to_do;
    CsrUint32 portion, available, hi;
    CsrUint32 handle;
    CsrResult r;
    CsrUint8 *dp;

    if ((*winp) >= (num_handles * handle_size))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: window_transfer: ERROR: %s window already exceeded (winp=%d)\n",
                               card->instance,  flag == UNIFI_SDIO_WRITE ? "tx" : "rx", *winp));
        return CSR_RESULT_FAILURE;
    }

    to_do = len;
    dp = data;
    while (to_do)
    {
        /* Calculate how much data can be written in the selected handle. */
        available = handle_size - (*winp) % handle_size;
        /* Select the handle. */
        hi = (*winp) / handle_size;
        if (hi >= num_handles)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: window_transfer: ERROR: hi=%d winp=%d\n",
                                   card->instance,  hi, *winp));
            return CSR_RESULT_FAILURE;
        }
        handle = handles[hi];
        if (handle == 0)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: window_transfer: ERROR: handle at index %d is zero\n",
                                   card->instance,  hi));
            return CSR_RESULT_FAILURE;
        }
        /* If the available space in this handle is insufficient for the data, two or more
        sdio writes are required. */
        if (to_do <= available)
        {
            portion = to_do;
        }
        else
        {
            portion = available;
        }
        if ((*winp) + portion > (num_handles * handle_size))
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: window_transfer: ERROR: %s window exceeded (winp=%d, portion=%d)\n",
                                   card->instance,  flag == UNIFI_SDIO_WRITE ? "tx" : "rx", *winp, portion));
            return CSR_RESULT_FAILURE;
        }
        r = unifi_bulk_rw(card,
                          handle,
                          (void *) dp,
                          portion,
                          flag);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: window_transfer: unifi_bulk_rw failed with %d\n",
                                   card->instance,  r));
            return r;
        }
        to_do -= portion;
        dp += portion;
        *winp += portion;
        if (block_ctr)
        {
            *block_ctr += portion / card->config_data.block_round_size;
        }
    }
    return CSR_RESULT_SUCCESS;
}

/* TEMP - tx_force_pull_mode is a module variable.
   get_pull_mode() should be redefined to get the
   pull status from a signal. */
#define get_pull_mode(s) (card->config_params.tx_force_pull_mode)

/*
 * ---------------------------------------------------------------------------
 *  handle_fh
 *
 *  Arguments:
 *      card   Pointer to card context structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success otherwise a CSR error code
 * ---------------------------------------------------------------------------
 */

static CsrResult handle_fh(card_t *card)
{
    CsrUint32 window, segment, segment_size;
    CsrUint32 sent;

    func_enter();

    card->tx_pkts_per_cycle = 0;

    segment_size = card->fh_buffer_d.segment_size;

    /* Calculate available window */
    window = calc_window(card);
    if (window < card->fh_buffer_d.pushed_round_bytes)
    {
        /* F/w will send us new window in a subsequent interrupt */
        func_exit_r(CSR_RESULT_SUCCESS);
        return CSR_RESULT_SUCCESS;
    }

    /* If the available window puts the window pointer beyond the end of the buffer, reset the window pointer. */
    if ((card->fh_buffer_d.winp + window) > (CsrUint32) (card->fh_buffer_d.num_handles * card->buff_maximum_size_bytes))
    {
        card->fh_buffer_d.winp = 0;
    }


    sent = 0;
    segment = (window < segment_size) ? window : segment_size;
    return handle_fh_segment(card, window, segment, &sent);
}

/*
 * ---------------------------------------------------------------------------
 *  handle_fh_segment
 *
 *  Arguments:
 *      card   Pointer to card context structure.
 *      window Available window in bytes.
 *      segment Segment size.
 *      sent   Bytes written.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success otherwise a CSR error code
 * ---------------------------------------------------------------------------
 */
static CsrResult handle_fh_segment(card_t *card, CsrUint32 window, CsrUint32 segment, CsrUint32 *sent)
{
    CsrResult r = CSR_RESULT_SUCCESS;
    CsrUint32 ctrl_window, bulk_window;
    CsrUint32 ctrl_window_available = 0, bulk_window_available = 0;
    CsrUint32 signals_encoded_len;
    CsrUint8 *buffer_ptr, *buffer, *ctrl_page_ptr;
    CsrUint16 i;
    CsrUint32 ctrl_size, bulk_size;
    CsrWifiHipBulkDataDesc *bulk_data;
    CsrUint32 ctrl_written = 0;
    CsrUint32 bulk_written = 0;
    CsrUint32 zero_copy;
    CsrUint16 ctrl_size_pages, bulk_size_pages;
    CsrUint16 min_bulk = 0, req_bulk = 0;
    CsrUint32 signaldata_pending;
    CsrWifiHipPacketSchedulerQsig *qsig;
    CsrInt32 sched_index;

    signaldata_pending = card->fh_signaldata_pending;

    #define TRAFFIC_SLOT_THRESHOLD 2

    func_enter();

    if (segment == 0)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: zero segment! win=%u, *sent=%u\n",
                               card->instance, window, *sent));
        func_exit_r(CSR_RESULT_FAILURE);
        return CSR_RESULT_FAILURE;
    }

    ctrl_window = 0;
    bulk_window = 0;
    min_bulk = 0;
    req_bulk = 0;
    zero_copy = card->zero_copy;

    buffer = card->fh_buffer_d.buffer;
    ctrl_page_ptr = buffer;
    buffer_ptr = buffer + FH_PUSHED_HEADER_SIZE;

    ctrl_size = 0;
    bulk_size = 0;
    signals_encoded_len = 0;

    /* F/w may only supply 1 block at start of day. If this happens,
     * ask for more window immediately.
     */
    if (segment <= card->config_data.block_round_size)
    {
        CSR_LOG_TEXT_CRITICAL((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,  "unifi%d: handle_fh: segment=%d jumping to more\n",
                                  card->instance, segment));
        goto more;
    }
    /* Start with a single round for control (signals and bulk descriptors). */
    ctrl_window = card->config_data.block_round_size;
    /* The rest of the window is given over to bulk data initially. */
    bulk_window = segment - ctrl_window;
    /* Initialise control and bulk window availability running counters.*/
    ctrl_window_available = ctrl_window - FH_PUSHED_HEADER_SIZE;
    bulk_window_available = bulk_window;
    /* If there is too little window available for bulk to write a single bulk
    data round, reallocate the bulk window back to the control window. */
    if (bulk_window_available < card->config_data.block_round_size)
    {
        ctrl_window_available += bulk_window_available;
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: handle_fh: bulk_window_available (%d) smaller than bulk block round (%d), ctrl_window_available becomes %d\n",
                              card->instance,
                              bulk_window_available, card->config_data.block_round_size, ctrl_window_available));
        bulk_window_available = 0;
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: handle_fh: entering signal schedule loop: windows: ctrl: %d data:%d ctrl_avail=%d pending=%d\n",
                           card->instance,  ctrl_window, bulk_window, ctrl_window_available, card->fh_signaldata_pending));

    card->fh_buffer_d.fh_pushed_bulk_data_count = 0;
    for (sched_index = 0; sched_index < PRIORITY_SCHEDULER_MAX_INDEX + 1; sched_index++)
    {
        CsrWifiHipPacketSchedulerPriorityScheduler *scheduler;
        CsrWifiPacketSchedulerCommonPart *cp;
        CsrBool exit_schedule_loop;

        exit_schedule_loop = FALSE;

        scheduler = csrWifiHipPacketSchedulerGetScheduler(sched_index);

        card->fh_buffer_d.packet_scheduler.restock = 0;
        if (scheduler->schedule_init)
        {
            r = scheduler->schedule_init(card, scheduler->lists, scheduler->pri, ctrl_window_available + bulk_window_available);
            if (r != CSR_RESULT_SUCCESS)
            {
                if (r != CSR_WIFI_HIP_RESULT_NOT_FOUND)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: schedule_init failed\n", card->instance));
                }
                continue;
            }
        }

        while (scheduler->schedule_signal(card, scheduler->lists, scheduler->pri, &qsig, &cp, ctrl_window_available + bulk_window_available) == CSR_RESULT_SUCCESS)
        {
            card_signal_t *signal;
            CsrUint16 encode_signal_flag;
            CsrUint32 bulk_data_length;
            CsrUint16 bulk_desc_length = 0;
            CsrUint32 rounded_signal_length;
            CsrUint16 bulk_data_slot_list[UNIFI_MAX_DATA_REFERENCES];
            CsrUint16 bulk_data_slot_list_count = 0;

            if (!qsig || !qsig_get_signal_ptr(qsig)->signal_length)
            {
                /* no more signals to send */
                break;
            }

            signal = qsig_get_signal_ptr(qsig);

            if (!cp)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: q not set (signal=%p signal_length=%d)\n",
                                       card->instance,  signal, signal->signal_length));
                func_exit_r(CSR_RESULT_FAILURE);
                return CSR_RESULT_FAILURE;
            }

            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: handle_fh: signal=%p siglen=%d bd[0].data_length=%d bd[1].data_length=%d\n",
                                   card->instance,
                                   signal, signal->signal_length, signal->bulkdata[0].data_length, signal->bulkdata[1].data_length));
            encode_signal_flag = 1;

            /* signal_length is the rounded length of the encoded signal */
            rounded_signal_length = calc_signal_length(card, signal);

            /* bulk_data_length is the aggregate length of all bulk data attached to this signal */
            bulk_data_length = 0;
            /* bulk_desc_length is the aggregate length (rounded appropriately) of the buffer descriptors
            that will be encoded in the control portion of the buffer. */
            bulk_desc_length = 0;
            calc_bulk_length(card, signal, &bulk_data_length, &bulk_desc_length);
            if ((rounded_signal_length + bulk_desc_length) > ctrl_window_available)
            {
                CsrUint32 needed;
                CsrUint16 sig_size;
                CsrBool pad = FALSE;

                sig_size = buffer_ptr - ctrl_page_ptr;

                /*
                 * Reallocate a round from the bulk window to the control window provided
                 * there is space in the bulk portion to do so AND the total control window
                 * will not exceed a single buffer (eg. 8KB).
                 */
                needed = (rounded_signal_length + bulk_desc_length);
                needed = CSR_WIFI_HIP_ROUNDUP(needed, card->config_data.block_round_size);
                if (((card->fh_buffer_d.winp + CSR_WIFI_HIP_ROUNDUP(sig_size, card->config_data.block_round_size)) % card->buff_maximum_size_bytes) == 0)
                {
                    pad = TRUE;
                    /* The boundary conditions of the protocol dictate that we pad out the control page. Therefore,
                     * we need another page after that for the signal.
                     */
                    needed += card->config_data.block_round_size;
                }
                if ((bulk_window_available < (needed + bulk_data_length)) ||
                    (ctrl_window > (card->buff_maximum_size_bytes - needed)))
                {
                    CSR_LOG_TEXT_DEBUG((
                                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: handle_fh: unable to allocate extra ctrl window (%d) needed=%d bulk_w=%d sig bulk data=%d\n",
                                           card->instance, ctrl_window, needed, bulk_window_available, bulk_data_length));

                    card->hip2Stats.hip2_stats_tx_window_congested++;
                    exit_schedule_loop = TRUE;
                    break;
                }
                CSR_LOG_TEXT_DEBUG((
                                       CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: handle_fh: realloc some bulk window to ctrl window: ctrl=%d bulk=%d needed=%d\n",
                                       card->instance,
                                       ctrl_window_available, bulk_window_available, needed));
                ctrl_window += needed;
                ctrl_window_available += needed;
                bulk_window_available -= needed;
                if (pad)
                {
                    CsrUint8 *bp = buffer_ptr;

                    /*
                     * Pad the remaining page. Signals must not straddle two pages in different buffers (handles) because the f/w starts processing the signals
                     * on completion of the sdio command 53. A cmd 53 cannot straddle two f/w buffers.
                     */
                    buffer_ptr += pad_ctrl(card, buffer_ptr, sig_size);
                    /*
                     * Remove the amount padded from the available count.
                     */
                    ctrl_window_available -= (buffer_ptr - bp);
                    CSR_LOG_TEXT_DEBUG((
                                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: handle_fh: ctrl pg at end of buffer (winp=%d, ctrl_size=%d, ctwin=%d), pad to align signal to next buffer\n",
                                           card->instance, card->fh_buffer_d.winp, CSR_WIFI_HIP_ROUNDUP(sig_size, card->config_data.block_round_size), ctrl_window_available));
                }
            }

            if (bulk_data_length)
            {
                if (card->fh_buffer_d.fh_pushed_bulk_data_count < card->config_data.bf_entries - 1)
                {
                    CsrUint32 pull_mode;
                    CsrUint16 slots_available;

                    pull_mode = get_pull_mode(signal);

                    bulk_data_slot_list_count = 0;

                    slots_available = CardGetFreeFromHostDataSlots(card);
                    if ((slots_available <= TRAFFIC_SLOT_THRESHOLD) && (qsig->type != SIGNAL_CTRL) && (qsig->type != SIGNAL_MGMT))
                    {
                        CSR_LOG_TEXT_DEBUG((
                                               CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: handle_fh: hit traffic slot threshold\n"
                                               , card->instance));
                        slots_available = 0;
                    }
                    /* are there slots available to store the bulk data ? */
                    if (slots_available && (CardWriteBulkData(card, qsig, bulk_data_slot_list, &bulk_data_slot_list_count) == CSR_RESULT_SUCCESS))
                    {
                        /* the bulk data pointers are now stored in the fromhost slot array */
                        if (bulk_data_slot_list_count > UNIFI_MAX_DATA_REFERENCES)
                        {
                            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: bulk_data_count (%d) out of range\n",
                                                   card->instance,  bulk_data_slot_list_count));
                            return CSR_RESULT_FAILURE;
                        }
                        /* If pull mode, the signal is encoded. */
                        if (!pull_mode)
                        {
                            /* If push mode, check there is space available in the bulk data portion of the buffer. */
                            if (bulk_data_length <= bulk_window_available)
                            {
                                /* encode the bulk descriptors */
                                for (i = 0; i < bulk_data_slot_list_count; i++)
                                {
                                    CsrUint32 rounded_bulk_data_length;
                                    CsrUint16 slot;

                                    slot = bulk_data_slot_list[i];

                                    if (slot > card->config_params.mparam_slot_count[D4_T_2_MAX_BULK_SLOTS_FROM] - 1)
                                    {
                                        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: invalid slot number %d\n",
                                                               card->instance,  slot));
                                        return CSR_RESULT_FAILURE;
                                    }

                                    bulk_data = &card->from_host_data[slot].bd;

                                    if (!bulk_data->data_length || !bulk_data->os_data_ptr)
                                    {
                                        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: invalid bulk data descriptor in slot %d\n",
                                                               card->instance,  slot));
                                        return CSR_RESULT_FAILURE;
                                    }
                                    rounded_bulk_data_length = CSR_WIFI_HIP_ROUNDUP(bulk_data->data_length + card->config_data.fh_pushed_block_gap, card->config_data.block_round_size);
                                    /* actual length of data encoded in bulk descriptor */
                                    buffer_ptr += encode_bulk_desc(card, buffer_ptr, slot, 0, (CsrUint16) bulk_data->data_length, card->fh_buffer_d.fh_current_data_ptr);
                                    /* window pointer decremented by rounded length */
                                    bulk_window_available -= rounded_bulk_data_length;
                                    bulk_size += rounded_bulk_data_length;
                                    /* pipe pointer incremented by rounded length */
                                    card->fh_buffer_d.fh_current_data_ptr += (CsrUint16) rounded_bulk_data_length;
                                    /* store slot number for processing further down */
                                    CSR_LOG_TEXT_DEBUG((
                                                           CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: handle_fh: adding slot %d to pushed_bulk_data[%d]\n",
                                                           card->instance,  slot, card->fh_buffer_d.fh_pushed_bulk_data_count));
                                    card->fh_buffer_d.pushed_bulk_data[card->fh_buffer_d.fh_pushed_bulk_data_count].slot = slot;
                                    card->fh_buffer_d.fh_pushed_bulk_data_count++;
                                }
                            }
                            else
                            {
                                CsrUint16 j;

                                /* Belts and braces. Free up the slot if we find that there is not space in the buffer
                                at this stage. */
                                for (j = 0; j < bulk_data_slot_list_count; j++)
                                {
                                    CardClearFromHostDataSlot(card, bulk_data_slot_list[j], FALSE);
                                }
                                CSR_LOG_TEXT_DEBUG((
                                                       CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6, "unifi%d: handle_fh: no space in window for bulk data and pull mode not allowed, signal not encoded (%d > %d)\n",
                                                       card->instance,  bulk_data_length, bulk_window_available));
                                encode_signal_flag = 0;
                            }
                        }
                    }
                    else
                    {
                        CSR_LOG_TEXT_DEBUG((
                                               CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: handle_fh: no slots available in fromhost slot array for bulk data, signal not encoded\n"
                                               , card->instance));
                        encode_signal_flag = 0;
                        card->hip2Stats.hip2_stats_tx_slot_congested++;
                    }
                }
                else
                {
                    CSR_LOG_TEXT_DEBUG((
                                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6, "unifi%d: handle_fh: no slots available in buffer for bulk data, signal not encoded\n"
                                           , card->instance));
                    encode_signal_flag = 0;
                }
            }

            if (!encode_signal_flag)
            {
                /*
                 * For some reason, the selected signal cannot be encoded.
                 * This means the end of the bundle of signals, so break out
                 * and continue to the send part.
                 */
                break;
            }
            /*
             * If we get here, there is enough space in the control portion for the signal.
             * Any associated bulk descriptors have been encoded already.
             */

            /*
             * Remove the signal from the queue now that we have it encoded to send.
             * Beware that the dequeue can modify signal flags (like the EOS bit or
             * the more bit).
             */
            r = scheduler->schedule_signal_complete(card, qsig, cp);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: schedule_signal_complete failed\n", card->instance));
                func_exit_r(r);
                return r;
            }
            if (qsig->type == SIGNAL_UNICAST)
            {
                card->hip2Stats.hip2_stats_tx_q_pkts[qsig->pri]++;
            }
            else
            {
                card->hip2Stats.hip2_stats_tx_q_pkts[4]++;
            }

            /*
             * Now encode the signal into the buffer.
             */
            buffer_ptr += encode_signal(card, buffer_ptr, signal);
            ctrl_window_available -= (rounded_signal_length + bulk_desc_length);
            signals_encoded_len += signal->signal_length;

            /* Log the signal to the UDI. */
            /* UDI will get the packed structure */
            /* Can not log the unpacked signal, unless we reconstruct it! */
            if (card->udi_hook)
            {
                CsrWifiHipBulkDataParam data_ptrs;
                CsrMemSet(&data_ptrs, 0, sizeof(CsrWifiHipBulkDataParam));
                for (i = 0; i < bulk_data_slot_list_count; i++)
                {
                    data_ptrs.d[i] = card->from_host_data[bulk_data_slot_list[i]].bd;
                }

                card->udi_hook(card->ospriv, signal->sigbuf, signal->signal_length, &data_ptrs, CSR_WIFI_HIP_LOG_UDI_DIRECTION_FROM_HOST);
            }
            if (!qsig->ref)
            {
                csrWifiHipPacketSchedulerFreeSignal(card, qsig);
            }

            card->tx_pkts_per_cycle++;
        }
        if (scheduler->schedule_complete)
        {
            r = scheduler->schedule_complete(card);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh:schedule_complete failed\n", card->instance));
                func_exit_r(r);
                return r;
            }
        }
        if (exit_schedule_loop)
        {
            break;
        }
    }


    if (card->tx_pkts_per_cycle)
    {
        if (card->tx_pkts_per_cycle >= sizeof(card->hip2Stats.hip2_stats_tx_pkts_per_cycle) / sizeof(CsrUint32))
        {
            card->hip2Stats.hip2_stats_tx_pkts_per_cycle[sizeof(card->hip2Stats.hip2_stats_tx_pkts_per_cycle) / sizeof(CsrUint32) - 1]++;
        }
        else
        {
            card->hip2Stats.hip2_stats_tx_pkts_per_cycle[card->tx_pkts_per_cycle - 1]++;
        }
    }


    if (signals_encoded_len)
    {
        if (bulk_size)
        {
            CsrUint32 total_rounded_bulk_size;

            total_rounded_bulk_size = CSR_WIFI_HIP_ROUNDUP(bulk_size, card->fh_buffer_d.pushed_round_bytes);
            if (bulk_size != total_rounded_bulk_size)
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                                    CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: bulk_size=%d total_rounded_bulk_size=%d\n",
                                    card->instance,  bulk_size, total_rounded_bulk_size));
            }
            bulk_size = total_rounded_bulk_size;
            if (bulk_size > bulk_window)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: ERROR: bulk_size (%d) > bulk_window (%d)\n",
                                       card->instance,  bulk_size, bulk_window));
                func_exit_r(CSR_RESULT_FAILURE);
                return CSR_RESULT_FAILURE;
            }
        }
        /* Adjust running counts - these counters are used in the "more" calculation. These counters
        are incremented in a different context, therefore lock around them here to prevent silly
        values (thereby affecting the more calculation). */
        CSR_WIFI_HIP_SPINLOCK_LOCK(&card->fh_count_lock);
        card->fh_signaldata_pending -= signals_encoded_len;
        card->fh_bulkdata_pending -= bulk_size;
        CSR_WIFI_HIP_SPINLOCK_UNLOCK(&card->fh_count_lock);
    }
    else
    {
        if (signaldata_pending)
        {
            /* Could not send signals, yet data is pending. Set the flag to make sure
            the hip is run again after timeout. */
            /*card->bh_reason_host = 1;*/
            card->hip2Stats.hip2_stats_tx_cycles_starved++;
        }

        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: handle_fh: no signals to send\n"
                              , card->instance));
        func_exit_r(CSR_RESULT_SUCCESS);
        return CSR_RESULT_SUCCESS;
    }
more:
    /* If we get here, signals have been encoded or we need more window, or both. Pad out the page. */
    buffer_ptr += pad_ctrl(card, buffer_ptr, buffer_ptr - ctrl_page_ptr);
    /* buffer_ptr has been incremented to the next unifi page alignment */
    ctrl_size = buffer_ptr - buffer;

    /* keep our copy of the ctrl pipe pointer up-to-date */
    card->fh_buffer_d.fh_current_ctrl_ptr += ctrl_size;

    if (ctrl_size + bulk_size > segment)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: ERROR: ctrl_size=%d bulk_size=%d window=%d ctrl_avail=%d\n",
                               card->instance,  ctrl_size, bulk_size, segment, ctrl_window_available));
        CSR_LOG_TEXT_BUFFER_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, ctrl_size, buffer, "FH CTRL:"));
        func_exit_r(CSR_RESULT_FAILURE);
        return CSR_RESULT_FAILURE;
    }

    calc_more_values(card, window, (ctrl_size + bulk_size), &min_bulk, &req_bulk);
    if (!min_bulk && !signals_encoded_len)
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: handle_fh: no signals to send\n"
                              , card->instance));
        func_exit_r(CSR_RESULT_SUCCESS);
        return CSR_RESULT_SUCCESS;
    }
    /* Now that we know the size of the control and bulk portions, encode the header. */
    ctrl_size_pages = ctrl_size / card->config_data.block_round_size;
    bulk_size_pages = bulk_size / card->config_data.block_round_size;

    card->hip2Stats.hip2_stats_tx_cycles++;

    (void) encode_header(card, buffer, ctrl_size_pages, bulk_size_pages, min_bulk, req_bulk);
    if (min_bulk)
    {
        if (card->fh_signaldata_pending)
        {
            card->hip2Stats.hip2_stats_tx_window_congested++;
        }
        card->pending_window_update++;
        card->hip2Stats.hip2_stats_tx_last_more_req_magic = card->Pm;
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: handle_fh: PENDING BULK SET (pipe limit=%d)\n",
                               card->instance,  card->ctrl_buffer.fh_pipe_limit));
    }

    #ifdef RECORD_LAST_TX_SIGNAL_BUFFER
    tx_signal_buffer_mirror_len = ctrl_size <= TX_SIGNAL_BUFFER_MIRROR_MAX_SIZE ? ctrl_size : TX_SIGNAL_BUFFER_MIRROR_MAX_SIZE;
    CsrMemCpy(tx_signal_buffer_mirror, buffer, tx_signal_buffer_mirror_len);
    #endif

    CSR_LOG_TEXT_BUFFER_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4, ctrl_size, buffer, "FH CTRL:"));
    if (zero_copy)
    {
        /* Write the control portion first and then loop through the list of bulk data, writing those */
        /* Write the control portion first. */
        r = window_transfer(
            card,
            buffer,
            ctrl_size,
            &card->fh_buffer_d.winp,
            &card->fh_buffer_d.fh_pipe_current,
            UNIFI_SDIO_WRITE_FROM_HOST_PUSHED,
            card->fh_buffer_d.handles,
            card->fh_buffer_d.num_handles,
            card->buff_maximum_size_bytes);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: window_transfer failed with %d\n",
                                   card->instance,  r));
            return r;
        }
        ctrl_written += ctrl_size;

        /* Write each bulk data */
        for (i = 0; i < card->fh_buffer_d.fh_pushed_bulk_data_count; i++)
        {
            CsrUint32 len;
            CsrUint16 slot;

            slot = card->fh_buffer_d.pushed_bulk_data[i].slot;
            if (slot >= card->config_data.bf_entries)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: slot %d out of range (index %d)\n",
                                       card->instance,  slot, i));
                return CSR_RESULT_FAILURE;
            }

            bulk_data = &card->from_host_data[slot].bd;

            if (bulk_data->os_data_ptr && (bulk_data->data_length > 0))
            {
                CSR_LOG_TEXT_BUFFER_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, 32, (CsrUint8 *) bulk_data->os_data_ptr, "FH BULKDATA (first 32):"));
                len = CSR_WIFI_HIP_ROUNDUP(bulk_data->data_length + card->config_data.fh_pushed_block_gap, card->config_data.block_round_size);
                if (i + 1 >= card->fh_buffer_d.fh_pushed_bulk_data_count)
                {
                    CsrUint16 diff;
                    diff = len;
                    len = CSR_WIFI_HIP_ROUNDUP(len, card->fh_buffer_d.pushed_round_bytes);
                    if (len > diff)
                    {
                        bulk_window_available -= (len - diff);
                    }
                }
                r = window_transfer(
                    card,
                    (CsrUint8 *) bulk_data->os_data_ptr,
                    len,
                    &card->fh_buffer_d.winp,
                    &card->fh_buffer_d.fh_pipe_current,
                    UNIFI_SDIO_WRITE_FROM_HOST_PUSHED,
                    card->fh_buffer_d.handles,
                    card->fh_buffer_d.num_handles,
                    card->buff_maximum_size_bytes);
                if (r != CSR_RESULT_SUCCESS)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: window_transfer failed with %d\n",
                                           card->instance,  r));
                    return r;
                }
                bulk_written += len;
                card->hip2Stats.hip2_stats_tx_bulk_data_pkts++;
            }
            else
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: bulk slot (%d) incorrect: os_data_ptr=0x%p data_length=%d\n",
                                       card->instance,  slot, bulk_data->os_data_ptr, bulk_data->data_length));
            }
        }
    }
    else
    {
        /* Copy the bulk data to the buffer */
        for (i = 0; i < card->fh_buffer_d.fh_pushed_bulk_data_count; i++)
        {
            CsrUint32 len;
            CsrUint16 slot;

            slot = card->fh_buffer_d.pushed_bulk_data[i].slot;
            if (slot >= card->config_data.bf_entries)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: slot %d out of range\n",
                                       card->instance,  slot));
                return CSR_RESULT_FAILURE;
            }

            bulk_data = &card->from_host_data[slot].bd;
            if (bulk_data->os_data_ptr && (bulk_data->data_length > 0))
            {
                CSR_LOG_TEXT_BUFFER_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, 32, (CsrUint8 *) bulk_data->os_data_ptr, "FH BULKDATA (first 32):"));
                len = CSR_WIFI_HIP_ROUNDUP(bulk_data->data_length + card->config_data.fh_pushed_block_gap, card->config_data.block_round_size);
                if (i + 1 >= card->fh_buffer_d.fh_pushed_bulk_data_count)
                {
                    CsrUint32 diff;
                    diff = len;
                    len = CSR_WIFI_HIP_ROUNDUP(len, card->fh_buffer_d.pushed_round_bytes);
                    if (len > diff)
                    {
                        bulk_window_available -= (len - diff);
                    }
                }
                CsrMemCpy(buffer_ptr, bulk_data->os_data_ptr, bulk_data->data_length);
                buffer_ptr += len;
                card->hip2Stats.hip2_stats_tx_bulk_data_pkts++;
            }
            else
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: bulk slot (%d) incorrect\n",
                                       card->instance,  slot));
            }
        }
        /* Write the entire buffer in one go. */
        r = window_transfer(
            card,
            buffer,
            buffer_ptr - buffer,
            &card->fh_buffer_d.winp,
            &card->fh_buffer_d.fh_pipe_current,
            UNIFI_SDIO_WRITE_FROM_HOST_PUSHED,
            card->fh_buffer_d.handles,
            card->fh_buffer_d.num_handles,
            card->buff_maximum_size_bytes);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: window_transfer failed with %d\n",
                                   card->instance,  r));
            return r;
        }
        ctrl_written += ctrl_size;
        bulk_written += (buffer_ptr - buffer) - ctrl_size;
    }

    if (ctrl_written || bulk_written)
    {
        card->ack_outstanding = 0;
    }

    if (ctrl_written != ctrl_size)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: ERROR: ctrl_written=%d ctrl_size=%d\n",
                               card->instance,  ctrl_written, ctrl_size));
    }
    if (bulk_written != bulk_size)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: handle_fh: ERROR: bulk_written=%d bulk_size=%d\n",
                               card->instance,  bulk_written, bulk_size));
    }
    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: handle_fh: Data written: winp=%d ctrl_size=%d bulk_size=%d\n",
                           card->instance,  card->fh_buffer_d.winp, ctrl_written, bulk_written));


    card->hip2Stats.hip2_stats_tx_ctrl_bytes += ctrl_written;
    card->hip2Stats.hip2_stats_tx_data_bytes += bulk_written;
    card->hip2Stats.hip2_stats_tx_window_efficiency = (card->hip2Stats.hip2_stats_tx_window_efficiency + (CsrUint32) (((ctrl_written + bulk_written) * 100) / segment)) / 2;
    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: push data %u[%u]\n",
                           card->instance,  card->ctrl_buffer.fh_pipe_limit, card->fh_buffer_d.fh_pipe_current));
    func_exit_r(r);

    *sent = (ctrl_written + bulk_written);
    return CSR_RESULT_SUCCESS;
}

void get_signal_counts(card_t *card, CsrInt32 *cmd, CsrInt32 *t0, CsrInt32 *t1, CsrInt32 *t2, CsrInt32 *t3)
{
    *cmd = 0;
    *t0 = 0;
    *t1 = 0;
    *t2 = 0;
    *t3 = 0;
}

CsrInt32 get_signal_count_total(card_t *card)
{
    CsrInt32 cmd, t0, t1, t2, t3;

    get_signal_counts(card, &cmd, &t0, &t1, &t2, &t3);

    return cmd + t1 + t2 + t3;
}

CsrInt32 get_free_slots(card_t *card)
{
    return CardGetFreeFromHostDataSlots(card);
}

CsrInt32 get_pm(card_t *card)
{
    return card->Pm;
}

void get_th_ptrs(card_t *card, CsrUint16 *th_pipe_limit, CsrUint16 *th_pipe_current)
{
    *th_pipe_current = card->fh_buffer_d.fh_pipe_current;
    *th_pipe_limit = card->ctrl_buffer.fh_pipe_limit;
}

void get_pending_data(card_t *card, CsrUint32 *ctrl, CsrUint32 *data)
{
    *ctrl = card->fh_signaldata_pending;
    *data = card->fh_bulkdata_pending;
}

CsrResult unifi_read_fw_control(card_t    *card,
                                CsrUint16 *m1,
                                CsrUint16 *th_handle,
                                CsrUint16 *th_size,
                                CsrUint16 *th_ctrl_size,
                                CsrUint16 *fh_pipe_limit,
                                CsrUint16 *fh_pipe_current)
{
    CsrResult r;
    ctrl_buffer_t ctrl_buffer;
    CsrUint8 buffer[64];

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read_fw_control: \n", card->instance));

    if (card->config_data.ctrl_re_read_buffer_h == 0)
    {
        return CSR_RESULT_FAILURE;
    }

    r = unifi_bulk_rw(card,
                      card->config_data.ctrl_re_read_buffer_h,
                      buffer,
                      64,
                      UNIFI_SDIO_READ_CONTROL);
    if (CSR_RESULT_SUCCESS != r)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read_fw_control: Failed to read log buffer\n", card->instance));
    }
    unpack_control_buf(card, &ctrl_buffer, buffer);

    *m1 = ctrl_buffer.M1;
    *th_handle = ctrl_buffer.th_handles[0];
    *th_size = ctrl_buffer.th_blocks;
    *th_ctrl_size = ctrl_buffer.th_ctrl_blocks;
    *fh_pipe_limit = ctrl_buffer.fh_pipe_limit;
    *fh_pipe_current = ctrl_buffer.fh_pipe_current;

    return r;
}

CsrInt32 unifi_read_last_tx_signal_buffer(card_t *card, CsrUint8 *buffer, CsrInt32 buffer_length)
{
    CsrInt32 len = 0;

    #ifdef RECORD_LAST_TX_SIGNAL_BUFFER
    len = tx_signal_buffer_mirror_len <= buffer_length ? tx_signal_buffer_mirror_len : buffer_length;

    CsrMemCpy(buffer, tx_signal_buffer_mirror, len);
    #endif
    return len;
}

CsrResult get_mbox3(card_t *card, CsrUint16 *mbox3)
{
#ifndef CSR_WIFI_DRIVER_HYDRA
    return unifi_read_direct16(card, ChipHelper_MAILBOX3(card->helper) * 2, mbox3);
#else
    return CSR_RESULT_FAILURE; /* D-30105: Fix this: Hydra */
#endif
}

CsrResult unifi_hip2_stats_get(card_t *card, CsrWifiHipHip2Stats *stats)
{
    *stats = card->hip2Stats;

    card->hip2Stats.fwin = card->pending_window_update;
    card->hip2Stats.ack = card->ack_outstanding;
    card->hip2Stats.cread = card->pending_ctrl_buffer;

    get_signal_counts(card, &card->hip2Stats.cmd_q, &card->hip2Stats.t0_q, &card->hip2Stats.t1_q, &card->hip2Stats.t2_q, &card->hip2Stats.t3_q);
    get_th_ptrs(card, &card->hip2Stats.fh_pipe_limit, &card->hip2Stats.fh_pipe_current);
    get_pending_data(card, &card->hip2Stats.ctrl_data, &card->hip2Stats.bulk_data);
    card->hip2Stats.free_slots = get_free_slots(card);
    card->hip2Stats.packets_pending = 0;
    return CSR_RESULT_SUCCESS;
}
