/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_types.h"
#include "csr_wifi_hip_io.h"
#include "csr_sdio.h"
#include "csr_wifi_hip_hal_priv.h"
#include "csr_wifi_hip.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_card.h"
#include "csr_wifi_hip_chiphelper.h"


static CsrWifiHipHalPriv *csrWifiHipRawSdioPriv = NULL;


#define CSR_WIFI_HIP_RAW_SDIO_STATE_IDLE  0x0
#define CSR_WIFI_HIP_RAW_SDIO_STATE_INIT  0x1
#define CSR_WIFI_HIP_RAW_SDIO_STATE_FW_DL 0x2


CsrResult CsrWifiHipRawSdioInitialise(CsrWifiHipHalPriv *priv, CsrBool noReset)
{
    if (!priv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipRawSdioInitialise: Null priv\n",
                               priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    if (priv->rawSdioState == CSR_WIFI_HIP_RAW_SDIO_STATE_IDLE)
    {
#ifndef CSR_WIFI_DRIVER_HYDRA
        CsrResult r;
#endif
        /* Block unifi_run_bh() until the error has been handled. */
        priv->bh_block_io_thread = 1;

        /* Stop the network traffic */
        CsrWifiHipFlowControlPauseAllInd(priv->osLayerContext);

        csrWifiHipRawSdioPriv = priv;

        /* For coredump, enable raw sdio without causing a chip reset
         * The state of the chip needs to be preserved for the dump. The
         * function driver should have already been registered when the HIP was
         * in stack mode, and HipTerminate never called.
         */
        if (noReset)
        {
            priv->rawSdioState = CSR_WIFI_HIP_RAW_SDIO_STATE_INIT;
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                                "unifi%d: CsrWifiHipRawSdioInitialise: (no reset) was successful \n",
                                priv->instance));
            return CSR_RESULT_SUCCESS;
        }
#ifdef CSR_WIFI_DRIVER_HYDRA
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipRawSdioInitialise: noReset=FALSE not supported for Hydra\n",
                                   priv ? priv->instance : 0));
            return CSR_RESULT_FAILURE;
        }
    }
#else
        CsrSdioClaim(priv->sdio);
        r = CsrSdioMaxBusClockFrequencySet(csrWifiHipRawSdioPriv->sdio, UNIFI_SDIO_CLOCK_INIT_HZ);
        CsrSdioRelease(priv->sdio);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipRawSdioInitialise: CsrSdioMaxBusClockFrequencySet failed, error code %u\n",
                                   priv->instance, r));
            return CSR_RESULT_FAILURE;
        }

        /* Init Unifi */
        CsrSdioClaim(priv->sdio);
        r = unifi_init(priv->card);
        CsrSdioRelease(priv->sdio);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipRawSdioInitialise: Initialisation of UniFi failed, error code %d\n",
                                   priv->instance, r));
            return CSR_RESULT_FAILURE;
        }

        CsrSdioClaim(priv->sdio);
        r = CsrSdioMaxBusClockFrequencySet(csrWifiHipRawSdioPriv->sdio, UNIFI_SDIO_CLOCK_MAX_HZ);
        CsrSdioRelease(priv->sdio);

        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipRawSdioInitialise: CsrSdioMaxBusClockFrequencySet failed, error code %u\n",
                                   priv->instance, r));
            return CSR_RESULT_FAILURE;
        }

        priv->rawSdioState = CSR_WIFI_HIP_RAW_SDIO_STATE_INIT;

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                            "unifi%d: CsrWifiHipRawSdioInitialise was successful\n",
                            priv->instance));
        return CSR_RESULT_SUCCESS;
    }
#endif
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipRawSdioInitialise: HIP not ready, error code %u\n",
                               priv->instance, priv->rawSdioState));
        return CSR_RESULT_FAILURE;
    }
}

void CsrWifiHipRawSdioDeinitialise(void)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: CsrWifiHipRawSdioDeinitialise: Not implemented\n"));
    return;
#else
    if (!csrWifiHipRawSdioPriv)
    {
        return;
    }

    if (csrWifiHipRawSdioPriv->rawSdioState >= CSR_WIFI_HIP_RAW_SDIO_STATE_INIT)
    {
        CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
        (void) CsrSdioFunctionDisable(csrWifiHipRawSdioPriv->sdio);
        CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);

        csrWifiHipRawSdioPriv->rawSdioState = CSR_WIFI_HIP_RAW_SDIO_STATE_IDLE;
        csrWifiHipRawSdioPriv = NULL;

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                            "unifi : CsrWifiHipRawSdioDeinitialise was successful\n"));
    }
#endif
}

CsrResult CsrWifiHipRawSdioReset(void)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: CsrWifiHipRawSdioReset: Not implemented\n"));
    return CSR_RESULT_FAILURE;
#else
    CsrResult r;

    if (!csrWifiHipRawSdioPriv)
    {
        return CSR_RESULT_FAILURE;
    }

    CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
    r = unifi_init(csrWifiHipRawSdioPriv->card);
    CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipRawSdioInitialise: Initialisation of UniFi failed, error code %u \n",
                               csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0,  r));
        return CSR_RESULT_FAILURE;
    }

    CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
    r = CsrSdioMaxBusClockFrequencySet(csrWifiHipRawSdioPriv->sdio, UNIFI_SDIO_CLOCK_MAX_HZ);
    CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipRawSdioInitialise: CsrSdioMaxBusClockFrequencySet failed. error code %u \n",
                               csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0,  r));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: CsrWifiHipRawSdioInitialise was successful \n"
                           , csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0));
    return CSR_RESULT_SUCCESS;
#endif
}

CsrResult CsrWifiHipRawSdioByteWrite(CsrUint8 function, CsrUint32 address, CsrUint8 data)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: CsrWifiHipRawSdioByteWrite: Not implemented\n"));
    return CSR_RESULT_FAILURE;
#else
    CsrResult r;

    if (!csrWifiHipRawSdioPriv)
    {
        return CSR_RESULT_FAILURE;
    }

    if (function == 0)
    {
        CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
        r = CsrSdioF0Write8(csrWifiHipRawSdioPriv->sdio, address, data);
        CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
    }
    else
    {
#ifdef CSR_WIFI_TRANSPORT_CSPI
        CsrUint16 wData = 0;
        wData = data;

        CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
        r = CsrSdioWrite16(csrWifiHipRawSdioPriv->sdio, address, wData);
        CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
#else
        CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
        r = CsrSdioWrite8(csrWifiHipRawSdioPriv->sdio, address, data);
        CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
#endif
    }

    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

    return CSR_RESULT_SUCCESS;
#endif
}

CsrResult CsrWifiHipRawSdioByteRead(CsrUint8 function, CsrUint32 address, CsrUint8 *pdata)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi : CsrWifiHipRawSdioByteRead: Not implemented\n"));
    return CSR_RESULT_FAILURE;
#else
    CsrResult r;

    if (!csrWifiHipRawSdioPriv)
    {
        return CSR_RESULT_FAILURE;
    }

    if (function == 0)
    {
        CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
        r = CsrSdioF0Read8(csrWifiHipRawSdioPriv->sdio, address, pdata);
        CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
    }
    else
    {
#ifdef CSR_WIFI_TRANSPORT_CSPI
        CsrUint16 w;

        CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
        r = CsrSdioRead16(csrWifiHipRawSdioPriv->sdio, address, &w);
        CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
        *pdata = (CsrUint8) (w & 0xFF);
#else
        CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
        r = CsrSdioRead8(csrWifiHipRawSdioPriv->sdio, address, pdata);
        CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
#endif
    }

    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

    return CSR_RESULT_SUCCESS;
#endif
}

CsrResult CsrWifiHipRawSdioFirmwareDownload(CsrUint32 length, const CsrUint8 *pdata)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: CsrWifiHipRawSdioFirmwareDownload: Not implemented\n"));
    return CSR_RESULT_FAILURE;
#else
    if (!csrWifiHipRawSdioPriv)
    {
        return CSR_RESULT_FAILURE;
    }

    if (csrWifiHipRawSdioPriv->rawSdioState == CSR_WIFI_HIP_RAW_SDIO_STATE_INIT)
    {
        CsrResult r;

        if ((length == 0) || (pdata == NULL) || (length > 0xfffffff))
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipRawSdioFirmwareDownload: No data or invalid data length \n", csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0));
            return CSR_RESULT_FAILURE;
        }

        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: CsrWifiHipRawSdioFirmwareDownload: Firmware: size = %u\n",
                              csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0,  length));


        /* Setting fw_desc NULL indicates to the core that no f/w file was loaded
         * via the kernel request_firmware() mechanism. This indicates to the core
         * that it shouldn't call release_firmware() after the download is done.
         */
        csrWifiHipRawSdioPriv->fw_sta.fw_desc = NULL;            /* No OS f/w resource */
        csrWifiHipRawSdioPriv->fw_sta.dl_data = pdata;
        csrWifiHipRawSdioPriv->fw_sta.dl_len = length;


        /* Application may have stopped the XAPs, but they are needed for reset */
        CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
        r = unifi_start_processors(csrWifiHipRawSdioPriv->card);
        CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipRawSdioFirmwareDownload: Failed to start XAPs. Hard reset required, error code %d \n",
                                   csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0,  r));
        }


        /* Download the f/w. On UF6xxx this will cause the f/w file to convert
         * into patch format and download via the ROM boot loader
         */
        CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
        r = unifi_download(csrWifiHipRawSdioPriv->card, 0x0c00);
        CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipRawSdioFirmwareDownload: Failed to download the f/w, error code %d \n",
                                   csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0,  r));

            return CSR_RESULT_FAILURE;
        }

        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: CsrWifiHipRawSdioFirmwareDownload was successful \n"
                               , csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0));
        return CSR_RESULT_SUCCESS;
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipRawSdioFirmwareDownload: Invalid state to download firmware. RAW SDIO state %u \n",
                               csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0,
                               csrWifiHipRawSdioPriv->rawSdioState));

        return CSR_RESULT_FAILURE;
    }
#endif
}

#ifndef CSR_WIFI_DRIVER_HYDRA
static CsrResult CsrWifiHipRawSdioRead16(CsrUint8 function, CsrUint32 address, CsrUint16 *pdata)
{
    CsrResult r;
    CsrUint8 b0, b1;

    r = CsrWifiHipRawSdioByteRead(function, address, &b0);
    if (r != CSR_RESULT_SUCCESS)
    {
        return CSR_RESULT_FAILURE;
    }

    r = CsrWifiHipRawSdioByteRead(function, address + 1, &b1);
    if (r != CSR_RESULT_SUCCESS)
    {
        return CSR_RESULT_FAILURE;
    }

    *pdata = ((CsrUint16) b1 << 8) | b0;

    return CSR_RESULT_SUCCESS;
} /* CsrSdioRead16() */

#endif

CsrResult CsrWifiHipRawSdioCoreDumpPrepare(CsrBool suspendSme)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    return CSR_RESULT_SUCCESS;
#else
    CsrUint16 data_u16 = 0;
    CsrInt32 i;
    CsrResult r = CSR_RESULT_FAILURE;
    CsrWifiHipHalPriv *priv;
    CsrInt32 ret;


    if (!csrWifiHipRawSdioPriv)
    {
        return CSR_RESULT_FAILURE;
    }

    priv = csrWifiHipRawSdioPriv;

    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: Start...\n", priv ? priv->instance : 0));
    CsrSdioClaim(priv->sdio);
    for (i = 0; i < 3; i++)
    {
        ret = CsrWifiHipRawSdioRead16(1, CHIP_HELPER_UNIFI_GBL_CHIP_VERSION * 2, &data_u16);
        if ((ret != CSR_RESULT_SUCCESS) || (data_u16 != priv->card->chip_version))
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: Failed to read valid chip version! r=%d (0x%04x) %d\n",
                                   priv ? priv->instance : 0,  ret, data_u16, i));
            /* Set clock speed low */
            r = CsrSdioMaxBusClockFrequencySet(priv->sdio, UNIFI_SDIO_CLOCK_SAFE_HZ);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: CsrSdioMaxBusClockFrequencySet failed %d\n",
                                       priv ? priv->instance : 0,  r));
            }

            /* First try re-enabling function in case a f/w watchdog reset disabled it */
            if (i == 0)
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Try function enable\n", priv ? priv->instance : 0));
                r = CsrSdioFunctionEnable(priv->card->sdio_if);
                if (r != CSR_RESULT_SUCCESS)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrSdioFunctionEnable failed %d\n",
                                           priv ? priv->instance : 0,  r));
                }
                continue;
            }

            /* Second  try, just set chip awake */
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Try wake up\n", priv ? priv->instance : 0));

            /* Ensure chip is awake */
            r = unifi_set_host_state(priv->card, UNIFI_HOST_STATE_AWAKE);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_set_host_state() failed %d\n",
                                       priv ? priv->instance : 0,  r));
            }

            /* Set clock speed low in case setting the host state raised it, which
             * would only happen if host state was previously TORPID
             */
            r = CsrSdioMaxBusClockFrequencySet(priv->sdio, UNIFI_SDIO_CLOCK_SAFE_HZ);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrSdioMaxBusClockFrequencySet() failed2 %d\n",
                                       priv ? priv->instance : 0,  r));
            }

            if (i == 1)
            {
                continue;
            }

            /* Subsequent tries - do a s/w reset of the chip */
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Try s/w reset\n", priv ? priv->instance : 0));

            r = unifi_card_hard_reset(priv->card);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: CardHardReset failed %d\n",
                                       priv ? priv->instance : 0,  r));
            }
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: Chip version 0x%04x\n",
                                   priv ? priv->instance : 0,  data_u16));
            r = CSR_RESULT_SUCCESS;
            break;
        }
    }


    if (r != CSR_RESULT_SUCCESS)
    {
        CsrSdioRelease(priv->sdio);
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: Failed to prepare chip\n", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    /* Stop processors for coredump. They must be resumed in raw sdio deinit */
    r = unifi_card_stop_processor(priv->card, UNIFI_PROC_BOTH);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: Failed to stop processors %d\n",
                               priv ? priv->instance : 0,  r));
    }

    ret = CsrWifiHipRawSdioRead16(1, CHIP_HELPER_UNIFI_GBL_CHIP_VERSION * 2, &data_u16);
    if (ret != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: Failed to read chip version! Try reset...\n", priv ? priv->instance : 0));
        /* Set clock speed low */
        r = CsrSdioMaxBusClockFrequencySet(priv->sdio, UNIFI_SDIO_CLOCK_SAFE_HZ);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: CsrSdioMaxBusClockFrequencySet failed %d\n",
                                   priv ? priv->instance : 0,  r));
        }

        r = unifi_card_hard_reset(priv->card);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: CardHardReset failed %d\n",
                                   csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0,  r));
        }
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: Chip version 0x%04x again\n",
                               priv ? priv->instance : 0,  data_u16));
    }
    CsrSdioRelease(priv->sdio);

    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CoreDumpPrepare: Start success\n", priv ? priv->instance : 0));
    return CSR_RESULT_SUCCESS;
#endif
}

CsrResult CsrWifiHipRawSdioByteBlockRead(CsrUint8 function, CsrUint32 address,
                                         CsrUint8 *pdata, CsrUint32 length)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: CsrWifiHipRawSdioByteBlockRead: Not implemented\n"));
    return CSR_RESULT_FAILURE;
#else
    CsrResult r;

    CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
    r = unifi_card_readn(csrWifiHipRawSdioPriv->card, address, (void *) pdata, (CsrUint16) length);
    CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: ByteBlockRead: Failed to read data block at 0x%x, len 0x%x\n",
                               csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0,  address, length));
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
#endif
}

CsrResult CsrWifiHipRawSdioGpRead16(CsrUint8 function, CsrUint32 address,
                                    CsrUint16 *pdata)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: CsrWifiHipRawSdioGpRead: Not implemented\n"));
    return CSR_RESULT_FAILURE;
#else
    CsrResult r;

    if (!csrWifiHipRawSdioPriv)
    {
        return CSR_RESULT_FAILURE;
    }

    CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
    r = unifi_card_read16(csrWifiHipRawSdioPriv->card, address, pdata);
    CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipRawSdioGpRead16: Failed to read register 0x%x\n",
                               csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0,  address));
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
#endif
}

CsrResult CsrWifiHipRawSdioGpWrite16(CsrUint8 function, CsrUint32 address,
                                     CsrUint16 data)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: CsrWifiHipRawSdioGpWrite: Not implemented\n"));
    return CSR_RESULT_FAILURE;
#else
    CsrResult r;

    if (!csrWifiHipRawSdioPriv)
    {
        return CSR_RESULT_FAILURE;
    }

    CsrSdioClaim(csrWifiHipRawSdioPriv->sdio);
    r = unifi_card_write16(csrWifiHipRawSdioPriv->card, address, data);
    CsrSdioRelease(csrWifiHipRawSdioPriv->sdio);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipRawSdioGpWrite16: Failed to write register 0x%x\n",
                               csrWifiHipRawSdioPriv ? csrWifiHipRawSdioPriv->instance : 0,  address));
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
#endif
}
