/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#ifndef CSR_WIFI_HIP_HAL_PRIV_H__
#define CSR_WIFI_HIP_HAL_PRIV_H__

#include "csr_synergy.h"

#include "csr_framework_ext.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_card_sdio.h"
#include "csr_wifi_common.h"
#include "csr_wifi_hip.h"
#include "csr_wifi_hip_list.h"
#include "csr_wifi_hip_spinlock.h"

#include "csr_wifi_vif_utils.h"

#ifdef CSR_LOG_ENABLE
#include "csr_log_text_2.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifndef CSR_HAVE_TIMER
#error The Wi-Fi driver requires timer support. Please ensure that timer support is enabled in the Framwork BSP.
#endif


#ifdef CSR_LOG_ENABLE

CSR_LOG_TEXT_HANDLE_DECLARE(CsrWifiHalLto);

/* These values must match the array positions below */
#define CSR_WIFI_HAL_LTSO_BA 1
#define CSR_WIFI_HAL_LTSO_BH 2
#define CSR_WIFI_HAL_LTSO_CHIP 3
#define CSR_WIFI_HAL_LTSO_CONFIG 4
#define CSR_WIFI_HAL_LTSO_PORT 5
#define CSR_WIFI_HAL_LTSO_RAWSDIO 6
#define CSR_WIFI_HAL_LTSO_SDIO 7
/* messages that are known to report only control packets */
#define CSR_WIFI_HAL_LTSO_RXC 8
#define CSR_WIFI_HAL_LTSO_TXC 9
/* messages that are known to report only data (MA) packets */
#define CSR_WIFI_HAL_LTSO_RXD 10
#define CSR_WIFI_HAL_LTSO_TXD 11
/* messages that report both data and control packets */
#define CSR_WIFI_HAL_LTSO_RX 13
#define CSR_WIFI_HAL_LTSO_TX 14
#define CSR_WIFI_HAL_LTSO_DEF 15
#define CSR_WIFI_HAL_LTSO_INIT 16

/* Positions must match the index values defined above.  But beware +1 offset */
#define CSR_WIFI_HAL_SUBORIGINS {"BA", "BH", "CHIP", "CONFIG", "PORT", "RAWSDIO", "SDIO", "RXC", "TXC", "RXD", "TXD", "RX", "TX", "DEF", "INIT"}

#endif

struct dlpriv
{
    const CsrUint8 *dl_data;
    CsrUint32       dl_len;
    void           *fw_desc;
};


/******************************** Block Ack *********************************/

#define CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_TX_MAX   1
#define CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX   4

#define CSR_WIFI_HIP_BA_BUFFER_SIZE_MAX 64
#define CSR_WIFI_HIP_BA_TRAFFIC_STREAM_MAX 15

/* Define for age (in us) value for frames in MPDU reorder buffer */
#define CSR_WIFI_HIP_BA_MPDU_FRAME_AGE_TIMEOUT  30000 /* 30 milli seconds */

typedef struct
{
    CsrBool                 active;
    CsrWifiHipBulkDataParam bulkdata;
    CSR_SIGNAL              signal;
    CsrUint16               sn;
} CsrWifiHipBaFrameDesc;


typedef struct
{
    void *vif;

    CsrWifiHipBaFrameDesc *buffer;
    CsrUint16              bufferSize;
    CsrUint16              occupiedSlots;

    CsrUint16         expectedSn;
    CsrUint16         startSn;
    CsrBool           triggerBaAfterSsn;
    CsrUint8          trafficStreamId;
    CsrWifiMacAddress macAddress;
    CsrTimerHandle    baAgingTimer;
    CsrBool           timerRunning;
    CsrBool           handleTimeout;
    CsrMutexHandle    baAgingMutex;
} CsrWifiHipBaSessionRx;


typedef struct
{
    CsrUint8          trafficStreamId;
    CsrWifiMacAddress macAddress;
} CsrWifiHipBaSessionTx;


/***************************** Controlled port ******************************/

#define CSR_WIFI_HIP_QUEUE_UNCONTROLLED_MAX  1
#define CSR_WIFI_HIP_QUEUE_CONTROLLED_MAX   50

typedef struct
{
    struct listHeader header;

    void                   *vif;
    CSR_SIGNAL              signal;
    CsrWifiHipBulkDataParam bulkdata;
    CsrUint8                macHeaderOffset;
    CsrWifiMacAddress       dMac;
    CsrWifiMacAddress       sMac;
} CsrWifiHipCtrlPortRxElement;

typedef CsrUint8 CsrWifiHipDirection;
#define CSR_WIFI_HIP_DIRECTION_TX  ((CsrWifiHipDirection) 0)
#define CSR_WIFI_HIP_DIRECTION_RX  ((CsrWifiHipDirection) 1)

typedef struct unifi_port_cfg
{
    CsrBool                     in_use;
    CsrWifiRouterCtrlPortAction port_action;
    CsrWifiMacAddress           mac_address;
} unifi_port_cfg_t;


#define CSR_WIFI_HIP_PEER_CONNECTIONS_MAX    8
#define UF_DATA_PORT_NOT_OVERIDE    0
#define UF_DATA_PORT_OVERIDE        1

typedef struct unifi_port_config
{
    CsrUint8         entries_in_use;
    CsrUint8         overide_action;
    unifi_port_cfg_t port_cfg[CSR_WIFI_HIP_PEER_CONNECTIONS_MAX];
} unifi_port_config_t;


#define ETH_ALEN  6
#define ETH_P_PAE 0x888E
#define ETH_P_WAI 0x88b4
#define IP4_OFFSET_TO_TOS_FIELD 1

/* This client tag shall only be used to construct frames,
   which have to be generated in HAL. If a MaPacketCfm is
   requested with this client tag, it shall not leave HAL. */
#define CSR_WIFI_HIP_HAL_INTERNAL_CLIENT_TAG 0xC999

#ifdef CSR_WIFI_AP_ENABLE
/************************** WMMPS helpers ***************************/
#define CSR_WIFI_HIP_MAX_ACCESS_CATOGORY              4

typedef CsrUint8 CsrWifiAcPowersaveMode;
#define CSR_WIFI_HIP_AC_TRIGGER_ONLY_ENABLED            ((CsrWifiAcPowersaveMode) 0x00)
#define CSR_WIFI_HIP_AC_DELIVERY_ONLY_ENABLE            ((CsrWifiAcPowersaveMode) 0x01)
#define CSR_WIFI_HIP_AC_LEGACY_POWER_SAVE               ((CsrWifiAcPowersaveMode) 0x02)
#define CSR_WIFI_HIP_AC_TRIGGER_AND_DELIVERY_ENABLED    ((CsrWifiAcPowersaveMode) 0x03)

#define CSR_WIFI_HIP_IS_DELIVERY_ENABLED(mode) ((mode & CSR_WIFI_HIP_AC_DELIVERY_ONLY_ENABLE) ? 1 : 0)
#define CSR_WIFI_HIP_IS_DELIVERY_AND_TRIGGER_ENABLED(mode) (((mode & CSR_WIFI_HIP_AC_DELIVERY_ONLY_ENABLE) || (mode & CSR_WIFI_HIP_AC_TRIGGER_AND_DELIVERY_ENABLED)) ? 1 : 0)

/**************** Defines for STA inactivity detection *****************/
#define     CSR_WIFI_HIP_STA_INACTIVE_DETECTION_TRIGGER_THRESHOLD  1                    /* in number of stations */
#define     CSR_WIFI_HIP_STA_INACTIVE_DETECTION_TIMER_INTERVAL     (30 * 1000 * 1000)   /* 30 seconds in micro seconds */
#define     CSR_WIFI_HIP_STA_INACTIVE_TIMEOUT_VAL                  (120 * 1000 * 1000)  /* 120 seconds */
#endif

/**************************** Virtual interfaces ****************************/
#define CSR_WIFI_HIP_MAX_VIFINDEX                     (8 + 1)

typedef CsrUint8 CsrWifiHipiVifIndexAction;
#define CSR_WIFI_HIP_VIFINDEX_ADD             ((CsrWifiHipiVifIndexAction) 0)
#define CSR_WIFI_HIP_VIFINDEX_DEL             ((CsrWifiHipiVifIndexAction) 1)
#define CSR_WIFI_HIP_VIFINDEX_FLUSH_IFACE_DB  ((CsrWifiHipiVifIndexAction) 2) /* Flush interface specific database mapping */


typedef struct
{
    CsrBool           entryInUse;
    CsrUint32         assignedHandle;
    CsrWifiMacAddress peerMacAddress;
    CsrBool           wmmOrQosEnabled;

    /* Router/Driver database */
    unifi_port_cfg_t *peerControlledPort;
    unifi_port_cfg_t *peerUnControlledPort;

    CsrWifiRouterCtrlPeerStatus currentPeerState;

    CsrBool   protection;
    CsrUint16 associationId;
#ifdef CSR_WIFI_AP_ENABLE
    /* WiFiH-204: these power save parameters are kept for the time being
     * If there is a need to inform packet scheduler regarding this, it
     * can be used
     */
    CsrUint16              maxSpLength;
    CsrWifiAcPowersaveMode powersaveMode[CSR_WIFI_HIP_MAX_ACCESS_CATOGORY];

    /* STA Inactivity detection parameters */
    CsrBool        active;
    CsrTime        lastActivityTime;
    CsrUint16      listenIntervalInTUs;
    CSR_CLIENT_TAG hostTagNullFrame;
#endif
} CsrWifiHipStaPeerInfo;


typedef struct
{
    CsrWifiMacAddress sta_mac_addresses;
    CsrWifiMacAddress bssid;
    CsrBool           protect;
    CsrUint32         wmmControl;
    CsrUint8          queueEnabled[UNIFI_NO_OF_TX_QS];

    unifi_port_config_t controlled_data_port;
    unifi_port_config_t uncontrolled_data_port;

#if defined(CSR_WIFI_SECURITY_WAPI_ENABLE)
    CsrBool wapi_multicast_mic_filter;
    CsrBool wapi_unicast_mic_filter;
#endif
} CsrWifiHipStaInfo;

typedef struct
{
    void                 *priv;
    CsrUint16             interfaceTag;
    CsrWifiRouterCtrlMode interfaceMode;

#ifdef CSR_WIFI_AP_ENABLE
    CsrBool  intraBssEnabled;
    CsrUint8 numPeers;

    /* Timer for detecting station inactivity */
    CsrBool        inactiveDetectEnabled;
    CsrTimerHandle inActivityTimerHandle;
#endif

    CsrWifiHipStaInfo     staInfo;
    CsrWifiHipStaPeerInfo staPeerInfo[CSR_WIFI_HIP_PEER_CONNECTIONS_MAX];

    /* Block Ack */
    CsrWifiHipBaSessionRx *baSessionRx[CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX];
    CsrWifiHipBaSessionTx *baSessionTx[CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_TX_MAX];

    CsrWifiHipBaFrameDesc baComplete[CSR_WIFI_HIP_BA_BUFFER_SIZE_MAX];
    CsrUint8              baCompleteIndex;

    CsrWifiHipReceiveStat receiveStats;
    CsrUint32             hostTag;

    CsrUint32         numOfUncontrolledFrames;
    CsrUint32         numOfControlledFrames;
    struct listHeader uncontrolledPortQueue;
    struct listHeader controlledPortQueue;
} CsrWifiHipVifInstance;


/*************************** Unifi priv structure ***************************/

/* Event group */
#define CSR_WIFI_HIP_EVENT_MASK_CLOSED_BH   0x1

/* bh event group */
#define CSR_WIFI_HIP_EVENT_MASK_BH_RUN      0x1
#define CSR_WIFI_HIP_EVENT_MASK_BH_SUSPEND  0x2
#define CSR_WIFI_HIP_EVENT_MASK_BH_SUS_RESP 0x4
#define CSR_WIFI_HIP_EVENT_MASK_BH_RESUME   0x8
#define CSR_WIFI_HIP_EVENT_MASK_BH_CLOSE    0x10
#define CSR_WIFI_HIP_EVENT_MASK_STA_INACT   0x20
#define CSR_WIFI_HIP_EVENT_MASK_BA_TIMEOUT   0x40

/* Wi-Fi on state */
#define CSR_WIFI_HIP_WIFI_ON_IN_PROGRESS 0
#define CSR_WIFI_HIP_WIFI_ON_DONE        1

typedef struct
{
    card_t          *card;
    CsrSdioFunction *sdio;

    void *osLayerContext;            /* The OS layer context */

    CsrBool blockingTerminate;       /* Indicates if the termination will be blocking */

    CsrInt16              instance;
    CsrWifiHipVifInstance vif[CSR_WIFI_MAX_INTERFACES];

    /* Fast fetching of primary vifIndex from interfaceTag mapping table */
    CsrUint16 primaryVifIndexMap[CSR_WIFI_MAX_INTERFACES];
    /* Fast fetching of interfaceTag from primary vifIndex mapping table */
    CsrUint16 interfaceTagMap[CSR_WIFI_HIP_MAX_VIFINDEX];
    /* protection for interfaceTag/VifIndex mapping database */
    CsrWifiHipSpinlock vifMapLock;

    CsrInt16 fw_init;
    CsrUint8 hipMode;                /* 0 indicates that HIP is in WiFi stack mode
                                        1 indicates that HIP is in Raw SDIO API mode */

    CsrSchedQid smeAppHandle;

    /* Bottom-half related parameters */
    CsrUint8                      bh_thread_terminate;
    CsrUint8                      bhThreadStarted;
    CsrUint8                      bh_thread_before_suspend; /* saves bhThreadStarted across hard suspend */
    CsrUint8                      bh_block_io_thread;
    card_info_t                   card_info;
    CsrWifiHipCardParams          cardParamsCache; /* not used in HAL, stored here on its way to lib hip */
    CsrWifiHipFlowControlPauseCb  pauseCb;
    CsrWifiHipFlowControlResumeCb resumeCb;

    /* Firmware images */
    struct dlpriv fw_sta;
    struct dlpriv fw_conv;  /* used for conversion of production test image */
    CsrBool       haveRequestedFirmware;

    /* Debug only */
    CsrWifiHipDriverInfoDebugInfo debugInfo;

    /* Mutex settings */
    CsrWifiHipSpinlock peerInfoLock;
    CsrMutexHandle     configurationMutex;

    /* Events settings */
    CsrEventHandle eventHandle;
    CsrUint32      eventBits;

    CsrEventHandle bhEventHandle;

    /* Raw SDIO API */
    CsrUint8 rawSdioState;

    /* Thread */
    CsrThreadHandle bhThreadHandle;

    /* Suspend type */
    CsrBool suspend;          /* Indicates if HIP is currently in suspended state */
    CsrBool resumeIrq;        /* Indicates that an SDIO interrupt occurred during resume */

    CsrUint32 wifiOnState;

    /* Allocate and free bulkdata functions */
    CsrWifiHipBulkDataAlloc allocFrame;
    CsrWifiHipBulkDataFree  freeFrame;

    /* UDI hook */
    udi_func_t udiProxyHook;
    CsrUint8   retryCount;
    CsrBool    cmanrTestMode;
    CSR_RATE   cmanrTestModeTransmitRate;
} CsrWifiHipHalPriv;


/************************ Quality of Service ************************/

#define CSR_WIFI_HIP_QOS_CAPABILITY_WMM_ENABLED      0x0001
#define CSR_WIFI_HIP_QOS_CAPABILITY_WMM_UAPSD        0x0002
#define CSR_WIFI_HIP_QOS_CAPABILITY_ACM_BE_ENABLED   0x0010
#define CSR_WIFI_HIP_QOS_CAPABILITY_ACM_BK_ENABLED   0x0020
#define CSR_WIFI_HIP_QOS_CAPABILITY_ACM_VI_ENABLED   0x0040
#define CSR_WIFI_HIP_QOS_CAPABILITY_ACM_VO_ENABLED   0x0080
#define CSR_WIFI_HIP_QOS_CAPABILITY_TS_BE_ENABLED    0x0100
#define CSR_WIFI_HIP_QOS_CAPABILITY_TS_BK_ENABLED    0x0200
#define CSR_WIFI_HIP_QOS_CAPABILITY_TS_VI_ENABLED    0x0400
#define CSR_WIFI_HIP_QOS_CAPABILITY_TS_VO_ENABLED    0x0800

/*
 * Upper Edge Initialisation functions
 */
CsrResult csrWifiHipBhInit(CsrWifiHipHalPriv *priv);
CsrResult uf_init_hw(CsrWifiHipHalPriv *priv);

/* Controlled port */
CsrWifiRouterCtrlPortAction csrWifiHipControlledPortState(CsrWifiHipHalPriv     *priv,
                                                          CsrWifiHipVifInstance *vif,
                                                          CsrUint8              *address,
                                                          CsrWifiHipPortType     queue);

unifi_port_cfg_t *CsrWifiHipControlledPortHandle(CsrWifiHipHalPriv     *priv,
                                                 CsrWifiHipVifInstance *vif,
                                                 CsrUint8              *address,
                                                 CsrUint32              queue);
void unifi_debug_buf_dump(void);

CsrResult csrWifiHipPackAndSendFrame(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, CsrUint32 aid,
                                     CsrWifiHipPortType portType,
                                     CsrWifiHipFrameType frameType, CsrWifiHipPacketType packetType,
                                     CSR_SIGNAL *signal, CsrWifiHipBulkDataParam *bulkdata);

CsrResult csrWifiHipProcessMaPacketReq(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif,
                                       CsrWifiHipPortType portType,
                                       CsrWifiHipFrameType frameType,
                                       CsrUint8 *peerMacAddress, CSR_PRIORITY priority,
                                       CSR_SIGNAL *signal, CsrWifiHipBulkDataParam *bulkdata,
                                       CsrWifiHipStaPeerInfo *dstStaInfo);


CsrWifiHipStaPeerInfo *csrWifiHipGetStationRecordFromPeerMacAddress(CsrWifiHipHalPriv     *priv,
                                                                    CsrWifiHipVifInstance *vif,
                                                                    const CsrUint8        *peerMacAddress);

CsrResult csrWifiHipPortConfigure(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif,
                                  CsrWifiRouterCtrlPortAction port_action,
                                  CsrWifiMacAddress *macAddress, CsrWifiHipPortType queue);

void csrWifiHipCtrlPortQueueProcess(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, CsrWifiHipPortType portType,
                                    CsrWifiMacAddress *macAddress, CsrBool indicate);
void csrWifiHipCtrlPortQueueFreeAll(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif);
void csrWifiHipCtrlPortQueueEntryAdd(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, CsrWifiHipPortType portType,
                                     CSR_SIGNAL *signal, CsrWifiHipBulkDataParam *bulkdata, CsrUint8 *dMac,
                                     CsrUint8 *sMac, CsrUint8 macHeaderOffset);

CsrResult csrWifiHipPeerAssociateQueueSetReq(CsrWifiHipHalPriv *priv, CsrUint16 interfaceTag, CsrUint16 associationId);

/* Inactivity timer handler function */
void csrWifiHipTimerInactivityHandler(CsrWifiHipHalPriv *priv);
void CsrWifiHipTimerExpInactivity(void *pointer);
void csrWifiHipBaAgingTimeoutHandler(CsrWifiHipHalPriv *priv);

#ifdef __cplusplus
}
#endif

#endif /* CSR_WIFI_HIP_HAL_PRIV_H__ */
