/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_types.h"
#include "csr_result.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_hal_priv.h"
#include "csr_wifi_hip_util.h"


#define SN_TO_INDEX(__baSession, __sn) (((__sn - __baSession->startSn) & 0xFFF) % __baSession->bufferSize)

#define ADVANCE_EXPECTED_SN(__baSession) \
    { \
        __baSession->expectedSn++; \
        __baSession->expectedSn &= 0xFFF; \
    }

#define FREE_BUFFER_SLOT(__baSession, __index) \
    { \
        __baSession->occupiedSlots--; \
        __baSession->buffer[__index].active = FALSE; \
        ADVANCE_EXPECTED_SN(__baSession); \
    }


static void baAddFrameToBaComplete(CsrWifiHipHalPriv     *priv,
                                   CsrWifiHipVifInstance *vif,
                                   CsrWifiHipBaFrameDesc *frameDesc)
{
    vif->baComplete[vif->baCompleteIndex] = *frameDesc;
    vif->baCompleteIndex++;
}

static void baUpdateExpectedSn(CsrWifiHipHalPriv     *priv,
                               CsrWifiHipVifInstance *vif,
                               CsrWifiHipBaSessionRx *baSession,
                               CsrUint16              sn)
{
    CsrUint32 i, j;
    CsrUint16 gap;

    gap = (sn - baSession->expectedSn) & 0xFFF;
    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: baUpdateExpectedSn: Proccess the frames up to new_expectedSn = %d gap = %d\n",
                           priv ? priv->instance : 0,  sn, gap));

    for (j = 0; j < gap && j < baSession->bufferSize; j++)
    {
        i = SN_TO_INDEX(baSession, baSession->expectedSn);
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: baUpdateExpectedSn: Proccess the slot index = %d\n",
                               priv ? priv->instance : 0,  i));
        if (baSession->buffer[i].active)
        {
            baAddFrameToBaComplete(priv, vif, &baSession->buffer[i]);
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: baUpdateExpectedSn: Proccess the frame at index = %d expectedSn = %d\n",
                                   priv ? priv->instance : 0,  i, baSession->expectedSn));
            FREE_BUFFER_SLOT(baSession, i);
        }
        else
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: baUpdateExpectedSn: Empty slot at index = %d\n",
                                   priv ? priv->instance : 0,  i));
            ADVANCE_EXPECTED_SN(baSession);
        }
    }

    baSession->expectedSn = sn;
}

static void baCompleteReadySequence(CsrWifiHipHalPriv     *priv,
                                    CsrWifiHipVifInstance *vif,
                                    CsrWifiHipBaSessionRx *baSession)
{
    CsrInt32 i;

    i = SN_TO_INDEX(baSession, baSession->expectedSn);
    while (baSession->buffer[i].active)
    {
        baAddFrameToBaComplete(priv, vif, &baSession->buffer[i]);
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                            "unifi%d: baCompleteReadySequence: Completed stored frame (expectedSn=%d) at i = %d\n",
                            priv ? priv->instance : 0, baSession->expectedSn, i));
        FREE_BUFFER_SLOT(baSession, i);
        i = SN_TO_INDEX(baSession, baSession->expectedSn);
    }
}

static void baScrollWindow(CsrWifiHipHalPriv     *priv,
                           CsrWifiHipVifInstance *vif,
                           CsrWifiHipBaSessionRx *baSession,
                           CsrUint16              sn)
{
    if (((sn - baSession->expectedSn) & 0xFFF) <= 2048)
    {
        baUpdateExpectedSn(priv, vif, baSession, sn);
        baCompleteReadySequence(priv, vif, baSession);
    }
}

static CsrInt32 baConsumeFrameOrGetBufferIndex(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif,
                                               CsrWifiHipBaSessionRx *baSession,
                                               CsrUint16 sn, CsrWifiHipBaFrameDesc *frameDesc)
{
    CsrInt32 i;
    CsrUint16 snTemp;
    CsrBool stopTimer = FALSE;
    CsrBool locked = FALSE;

    if (baSession->timerRunning == TRUE)
    {
        CsrMutexLock(&baSession->baAgingMutex);
        locked = TRUE;
    }
    if (((sn - baSession->expectedSn) & 0xFFF) <= 2048)
    {
        /* Once we are in BA window, set the flag for BA trigger */
        if (!baSession->triggerBaAfterSsn)
        {
            baSession->triggerBaAfterSsn = TRUE;
        }

        snTemp = baSession->expectedSn + baSession->bufferSize;
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: baConsumeFrameOrGetBufferIndex: New frame: sn=%d\n",
                               priv ? priv->instance : 0,  sn));
        if (!(((sn - snTemp) & 0xFFF) > 2048))
        {
            CsrUint16 newExpectedSn;
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: baConsumeFrameOrGetBufferIndex: Frame is out of window\n"
                                   , priv ? priv->instance : 0));
            snTemp = (sn - baSession->bufferSize) & 0xFFF;

            stopTimer = TRUE;
            newExpectedSn = (snTemp + 1) & 0xFFF;
            baUpdateExpectedSn(priv, vif, baSession, newExpectedSn);
        }

        i = -1;
        if (sn == baSession->expectedSn)
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: baConsumeFrameOrGetBufferIndex: sn = baSession->expectedSn = %d\n",
                                   priv ? priv->instance : 0,  sn));

            stopTimer = TRUE;
            ADVANCE_EXPECTED_SN(baSession);
            baAddFrameToBaComplete(priv, vif, frameDesc);
        }
        else
        {
            i = SN_TO_INDEX(baSession, sn);
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: baConsumeFrameOrGetBufferIndex: sn (%d) != baSession->expectedSn(%d), i = %d\n",
                                   priv ? priv->instance : 0,  sn, baSession->expectedSn, i));
            if (baSession->buffer[i].active)
            {
                CSR_LOG_TEXT_DEBUG((
                                       CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: baConsumeFrameOrGetBufferIndex: free frame at i = %d\n",
                                       priv ? priv->instance : 0,  i));
                i = -1;
                priv->freeFrame(priv->osLayerContext, &frameDesc->bulkdata.d[0]);
            }
        }
    }
    else
    {
        i = -1;
        if (!baSession->triggerBaAfterSsn)
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: baConsumeFrameOrGetBufferIndex: frame before ssn, pass it up: sn=%d\n",
                                   priv ? priv->instance : 0,  sn));
            baAddFrameToBaComplete(priv, vif, frameDesc);
        }
        else
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: baConsumeFrameOrGetBufferIndex: old frame, drop: sn=%d, expectedSn=%d\n",
                                   priv ? priv->instance : 0,  sn, baSession->expectedSn));
            priv->freeFrame(priv->osLayerContext, &frameDesc->bulkdata.d[0]);
        }
    }
    if (locked)
    {
        CsrMutexUnlock(&baSession->baAgingMutex);
    }

    if ((stopTimer == TRUE) && (baSession->timerRunning == TRUE))
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                            "unifi%d: baConsumeFrameOrGetBufferIndex Timer stop\n", priv->instance));
        CsrTimerStop(&baSession->baAgingTimer);
        baSession->timerRunning = FALSE;
    }
    return i;
}

void csrWifiHipBaAgingTimeoutHandler(CsrWifiHipHalPriv *priv)
{
    CsrUint8 i, j, k;
    CsrUint8 gap = 1;
    CsrUint16 tempSn;
    CsrUint8 baSessionIdx;
    CsrWifiHipVifInstance *vif;
    CsrWifiHipBaSessionRx *baSession = NULL;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6, "unifi%d: csrWifiHipBaAgingTimeoutHandler\n", priv ? priv->instance : 0));

    for (k = 0; k < CSR_WIFI_MAX_INTERFACES; k++)
    {
        vif = &priv->vif[k];
        for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX; baSessionIdx++)
        {
            if ((vif->baSessionRx[baSessionIdx]) && (vif->baSessionRx[baSessionIdx]->handleTimeout == TRUE))
            {
                baSession = vif->baSessionRx[baSessionIdx];
                CsrMutexLock(&baSession->baAgingMutex);

                baSession->timerRunning = FALSE;
                baSession->handleTimeout = FALSE;

                if (baSession->occupiedSlots)
                {
                    /* expected sequence has not arrived so start searching from next
                     * sequence number until a frame is available and determine the gap.
                     * Release all the frames upto next hole from the reorder buffer.
                     */
                    tempSn = (baSession->expectedSn + 1) & 0xFFF;

                    for (j = 0; j < baSession->bufferSize; j++)
                    {
                        i = SN_TO_INDEX(baSession, tempSn);

                        if (baSession->buffer[i].active)
                        {
                            while (gap--)
                            {
                                ADVANCE_EXPECTED_SN(baSession);
                            }
                            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                                                "unifi%d: csrWifiHipBaAgingTimeoutHandler: Completed stored frame (expectedSn=%d) at i = %d\n", priv ? priv->instance : 0, baSession->expectedSn, i));
                            baAddFrameToBaComplete(priv, vif, &baSession->buffer[i]);
                            FREE_BUFFER_SLOT(baSession, i);
                            baCompleteReadySequence(priv, vif, baSession);
                            break;
                        }
                        else
                        {
                            /* advance temp sequence number and frame gap */
                            tempSn = (tempSn + 1) & 0xFFF;
                            gap++;
                        }
                    }

                    /* Process the data now marked as completed. */
                    csrWifiHipBaProcessComplete(priv, vif);

                    /* Check for next hole in the buffer, if hole exists create the timer for next missing frame */
                    if (baSession->occupiedSlots != 0)
                    {
                        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                                            "unifi%d: csrWifiHipBaAgingTimeoutHandler: Timer start\n", priv->instance));
                        CsrTimerStart(&baSession->baAgingTimer, CSR_WIFI_HIP_BA_MPDU_FRAME_AGE_TIMEOUT);
                        baSession->timerRunning = TRUE;
                    }
                }
                CsrMutexUnlock(&baSession->baAgingMutex);
            }
        }
    }
}

void csrWifiHipBaProcessFrame(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif,
                              CsrWifiHipBaSessionRx *baSession, CsrWifiHipBaFrameDesc *frameDesc)
{
    CsrInt32 i;
    CsrUint16 sn = frameDesc->sn;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: csrWifiHipBaProcessFrame: Got frame(sn=%d)\n",
                           priv ? priv->instance : 0,  sn));

    i = baConsumeFrameOrGetBufferIndex(priv, vif, baSession, sn, frameDesc);
    if (i >= 0)
    {
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: csrWifiHipBaProcessFrame: Store frame(sn=%d) at i = %d\n",
                               priv ? priv->instance : 0,  sn, i));
        CsrMutexLock(&baSession->baAgingMutex);
        baSession->buffer[i] = *frameDesc;
        baSession->occupiedSlots++;
        CsrMutexUnlock(&baSession->baAgingMutex);
    }
    else
    {
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: csrWifiHipBaProcessFrame: Frame consumed - sn = %d\n",
                               priv ? priv->instance : 0,  sn));
    }

    baCompleteReadySequence(priv, vif, baSession);
    /* Start timer if the hole exists */
    if ((baSession->timerRunning == FALSE) && baSession->occupiedSlots)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6, "unifi%d: csrWifiHipBaProcessFrame: Timer start\n", priv ? priv->instance : 0));
        CsrTimerStart(&baSession->baAgingTimer, CSR_WIFI_HIP_BA_MPDU_FRAME_AGE_TIMEOUT);
        baSession->timerRunning = TRUE;
    }
}

void csrWifiHipAmsduProcess(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, CSR_SIGNAL *signal, CsrWifiHipBulkDataParam *bulkdata)
{
    CsrUint32 subframeLength, subframeBodyLength;
    CsrUint8 *ptr;
    CsrWifiHipBulkDataParam subframeBulkdata;
    CsrUint8 *dot11HdrPtr = (CsrUint8 *) bulkdata->d[0].os_data_ptr;
    CsrResult result;
    CsrUint8 macHeaderOffset = 0;
    CsrUint32 offset = 0;
    CsrUint8 *dMac = NULL, *sMac = NULL;
    CsrUint8 toDs = 0, fromDs = 0;
    CsrResult receptionsStatus = CSR_RESULT_SUCCESS;
    CsrUint16 frameControl = 0;
    CsrUint32 length = bulkdata->d[0].data_length;

    frameControl = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(dot11HdrPtr);

    if (CSR_WIFI_HIP_IEEE80211_HAS_A4(frameControl))
    {
        /* 4 MAC addresses */
        macHeaderOffset = MAC_HEADER_SIZE + ETH_ALEN;
    }
    else
    {
        /* 3 MAC addresses header */
        macHeaderOffset = MAC_HEADER_SIZE;
    }

    toDs = CSR_WIFI_HIP_IEEE80211_HAS_TO_DS(frameControl);
    fromDs = CSR_WIFI_HIP_IEEE80211_HAS_FROM_DS(frameControl);

    dMac = (CsrUint8 *) bulkdata->d[0].os_data_ptr + 4 + toDs * 12;                 /* Address 1 or 3 */
    sMac = (CsrUint8 *) bulkdata->d[0].os_data_ptr + 10 + fromDs * (6 + toDs * 8);  /* Address 2, 3 or 4 */

    if (signal->u.MaPacketIndication.ReceptionStatus != CSR_RX_SUCCESS)
    {
        receptionsStatus = CSR_RESULT_FAILURE;
    }

    if (!(*(dot11HdrPtr + macHeaderOffset) & CSR_WIFI_HIP_IEEE80211_QOS_CONTROL_A_MSDU_PRESENT))
    {
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: csrWifiHipAmsduProcess: NO aggregation, calling CsrWifiHipProcessDataRx\n"
                               , priv ? priv->instance : 0));
        macHeaderOffset += QOS_CONTROL_HEADER_SIZE;
        CsrWifiHipProcessDataRx(priv,
                                vif,
                                vif->interfaceTag,
                                signal,
                                receptionsStatus,
                                bulkdata,
                                frameControl,
                                dMac,
                                sMac);
        return;
    }

    /* Clear the A_MSDU bit */
    *(dot11HdrPtr + macHeaderOffset) &= ~CSR_WIFI_HIP_IEEE80211_QOS_CONTROL_A_MSDU_PRESENT;

    macHeaderOffset += QOS_CONTROL_HEADER_SIZE;
    ptr = (CsrUint8 *) (dot11HdrPtr + macHeaderOffset);

    while (length > (offset + macHeaderOffset + (2 * ETH_ALEN + 2) + CSR_WIFI_HIP_IEEE80211_LLC_HEADER_LEN))
    {
        subframeBodyLength = ptr[2 * ETH_ALEN] << 8 | ptr[(2 * ETH_ALEN) + 1];
        if (subframeBodyLength > CSR_WIFI_HIP_IEEE80211_MAX_DATA_LEN)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipAmsduProcess: Wrong subframeBodyLength %d",
                                   priv ? priv->instance : 0,  subframeBodyLength));
            break;
        }

        subframeLength = subframeBodyLength + (2 * ETH_ALEN) + 2;
        CsrMemSet(&subframeBulkdata, 0, sizeof(CsrWifiHipBulkDataParam));

        result = priv->allocFrame(priv->osLayerContext, &subframeBulkdata.d[0], macHeaderOffset + subframeBodyLength);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipAmsduProcess: priv->allocFrame failed\n", priv ? priv->instance : 0));
            break;
        }

        CsrMemCpy((CsrUint8 *) subframeBulkdata.d[0].os_data_ptr, dot11HdrPtr, macHeaderOffset);

        /* When toDS = 0 and fromDS = 0, address 3 will already have BSSID so no need to re-program */
        if (toDs && !(fromDs))
        {
            CsrMemCpy(((CsrWifiHipIeee80211Hdr *) subframeBulkdata.d[0].os_data_ptr)->addr3.a,
                      dMac,
                      ETH_ALEN);
        }
        else if (!(toDs) && fromDs)
        {
            CsrMemCpy(((CsrWifiHipIeee80211Hdr *) subframeBulkdata.d[0].os_data_ptr)->addr3.a,
                      sMac,
                      ETH_ALEN);
        }


        CsrMemCpy((CsrUint8 *) (subframeBulkdata.d[0].os_data_ptr + macHeaderOffset),
                  (CsrUint8 *) (ptr + ((2 * ETH_ALEN) + 2)),
                  subframeBodyLength);

        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: csrWifiHipAmsduProcess: calling CsrWifiHipProcessDataRx. length %d subframeBodyLength %d\n",
                               priv ? priv->instance : 0,  length, subframeBodyLength));

        CsrWifiHipProcessDataRx(priv,
                                vif,
                                vif->interfaceTag,
                                signal,
                                receptionsStatus,
                                &subframeBulkdata,
                                frameControl,
                                dMac,
                                sMac);

        subframeLength = (subframeLength + 3) & (~0x3);
        ptr += subframeLength;
        offset += subframeLength;
    }

    priv->freeFrame(priv->osLayerContext, &bulkdata->d[0]);
}

CsrResult csrWifiHipBaSessionStop(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif,
                                  CsrWifiRouterCtrlBlockAckRole role,
                                  CsrWifiRouterCtrlTrafficStreamId trafficStreamID,
                                  CsrWifiMacAddress macAddress)
{
    CsrWifiHipBaSessionRx *baSessionRx = NULL;
    CsrWifiHipBaSessionTx *baSessionTx = NULL;
    CsrUint8 baSessionIdx = 0;
    CsrUint32 i;

    if (trafficStreamID > CSR_WIFI_HIP_BA_TRAFFIC_STREAM_MAX)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipBaSessionStop: Invalid trafficStreamID = %d\n",
                               priv ? priv->instance : 0,  trafficStreamID));
        return CSR_RESULT_FAILURE;
    }

    if ((role != CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_ORIGINATOR) &&
        (role != CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_RECIPIENT))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipBaSessionStop: Invalid role = %d\n",
                               priv ? priv->instance : 0,  role));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                          CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipBaSessionStop: Stopping baSession for peer = %02x:%02x:%02x:%02x:%02x:%02x role = %d tID = %d\n",
                          priv ? priv->instance : 0,
                          macAddress.a[0], macAddress.a[1], macAddress.a[2],
                          macAddress.a[3], macAddress.a[4], macAddress.a[5],
                          role,
                          trafficStreamID));

    /* Find the appropriate ba session (/station /tid /role) for which stop is requested */
    if (role == CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_RECIPIENT)
    {
        for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX; baSessionIdx++)
        {
            baSessionRx = vif->baSessionRx[baSessionIdx];

            if (baSessionRx)
            {
                if ((!CsrMemCmp(baSessionRx->macAddress.a, macAddress.a, ETH_ALEN)) &&
                    (baSessionRx->trafficStreamId == trafficStreamID))
                {
                    break;
                }
            }
        }

        if (!baSessionRx || (baSessionIdx == CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX))
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipBaSessionStop: Invalid baSession for Rx [trafficStreamID %d]\n",
                                   priv ? priv->instance : 0,  trafficStreamID));
            return CSR_RESULT_FAILURE;
        }

        for (i = 0; i < baSessionRx->bufferSize; i++)
        {
            if (baSessionRx->buffer[i].active)
            {
                CsrWifiHipBaFrameDesc *frameDesc = &baSessionRx->buffer[i];
                priv->freeFrame(priv->osLayerContext, &frameDesc->bulkdata.d[0]);
            }
        }
        CsrTimerDestroy(&baSessionRx->baAgingTimer);
        CsrMutexDestroy(&baSessionRx->baAgingMutex);
        CsrMemFree(baSessionRx->buffer);

        vif->baSessionRx[baSessionIdx] = NULL;
        CsrMemFree(baSessionRx);
    }
    else if (role == CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_ORIGINATOR)
    {
        for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_TX_MAX; baSessionIdx++)
        {
            baSessionTx = vif->baSessionTx[baSessionIdx];
            if (baSessionTx)
            {
                if ((!CsrMemCmp(baSessionTx->macAddress.a, macAddress.a, ETH_ALEN)) && (baSessionTx->trafficStreamId == trafficStreamID))
                {
                    break;
                }
            }
        }

        if (!baSessionTx || (baSessionIdx == CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_TX_MAX))
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipBaSessionStop: Invalid baSession for Tx [trafficStreamID %d]\n",
                                   priv ? priv->instance : 0,  trafficStreamID));
            return CSR_RESULT_FAILURE;
        }

        vif->baSessionTx[baSessionIdx] = NULL;
        CsrMemFree(baSessionTx);
    }

    return CSR_RESULT_SUCCESS;
}

static void CsrWifiHipBaTimeout(void *data)
{
    CsrWifiHipBaSessionRx *baSession = (CsrWifiHipBaSessionRx *) data;
    CsrWifiHipVifInstance *vif = (CsrWifiHipVifInstance *) baSession->vif;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) vif->priv;

    baSession->handleTimeout = TRUE;
    (void) CsrEventSet(&priv->bhEventHandle, CSR_WIFI_HIP_EVENT_MASK_BA_TIMEOUT);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                        "unifi : CsrWifiHipTimerBaTimeout: \n"));
}

CsrResult csrWifiHipBaSessionStart(CsrWifiHipHalPriv               *priv,
                                   CsrWifiHipVifInstance           *vif,
                                   CsrWifiRouterCtrlTrafficStreamId trafficStreamID,
                                   CsrUint16                        timeout,
                                   CsrWifiRouterCtrlBlockAckRole    role,
                                   CsrUint16                        bufferSize,
                                   CsrUint16                        ssn,
                                   CsrWifiMacAddress                macAddress)
{
    CsrWifiHipBaSessionRx *baSessionRx = NULL;
    CsrWifiHipBaSessionTx *baSessionTx = NULL;
    CsrUint8 baSessionIdx = 0;


    if (trafficStreamID > CSR_WIFI_HIP_BA_TRAFFIC_STREAM_MAX)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: Invalid trafficStream ID %d\n",
                               priv ? priv->instance : 0,  trafficStreamID));
        return CSR_RESULT_FAILURE;
    }

    if (bufferSize > CSR_WIFI_HIP_BA_BUFFER_SIZE_MAX)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: Invalid wind size %d\n",
                               priv ? priv->instance : 0,  ssn));
        return CSR_RESULT_FAILURE;
    }

    if ((role != CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_ORIGINATOR) &&
        (role != CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_RECIPIENT))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: Invalid role %d\n",
                               priv ? priv->instance : 0,  role));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: BA session with peer= (%02x:%02x:%02x:%02x:%02x:%02x) \n",
                        priv ? priv->instance : 0,
                        macAddress.a[0], macAddress.a[1], macAddress.a[2],
                        macAddress.a[3], macAddress.a[4], macAddress.a[5]));

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: BA session for trafficStreamId=%d timeout=%d role=%d bufferSize=%d startSn=%d\n",
                        priv ? priv->instance : 0,
                        trafficStreamID,
                        timeout,
                        role,
                        bufferSize,
                        ssn));


    /* Check if BA session exists for per station, per trafficStreamId, per role or not.
       If BA session exists update parameters and if it does not exist
       create a new BA session */
    if (role == CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_ORIGINATOR)
    {
        for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_TX_MAX; baSessionIdx++)
        {
            baSessionTx = vif->baSessionTx[baSessionIdx];
            if (baSessionTx)
            {
                if ((!CsrMemCmp(baSessionTx->macAddress.a, macAddress.a, ETH_ALEN)) && (baSessionTx->trafficStreamId == trafficStreamID))
                {
                    CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: baSession for Tx already exists\n", priv ? priv->instance : 0));
                    return CSR_RESULT_SUCCESS;
                }
            }
        }

        /* We have to create new baSessionTx struct */
        baSessionTx = NULL;

        /* Loop through until an empty BA session slot is there and save the session there */
        for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_TX_MAX; baSessionIdx++)
        {
            if (!(vif->baSessionTx[baSessionIdx]))
            {
                break;
            }
        }

        if (baSessionIdx == CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_TX_MAX)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: All baSession used for Tx, NO free session available\n", priv ? priv->instance : 0));
            return CSR_RESULT_FAILURE;
        }

        /* Create and populate the new BA session structure */
        baSessionTx = CsrMemAlloc(sizeof(CsrWifiHipBaSessionTx));
        if (!baSessionTx)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: Memory allocation failed for baSessionTx\n", priv ? priv->instance : 0));
            return CSR_RESULT_FAILURE;
        }

        CsrMemSet(baSessionTx, 0, sizeof(CsrWifiHipBaSessionTx));

        baSessionTx->trafficStreamId = trafficStreamID;
        baSessionTx->macAddress = macAddress;

        vif->baSessionTx[baSessionIdx] = baSessionTx;
    }
    else if (role == CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_RECIPIENT)
    {
        CsrResult ret;
        for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX; baSessionIdx++)
        {
            baSessionRx = (CsrWifiHipBaSessionRx *) vif->baSessionRx[baSessionIdx];
            if (baSessionRx)
            {
                if ((!CsrMemCmp(baSessionRx->macAddress.a, macAddress.a, ETH_ALEN)) &&
                    (baSessionRx->trafficStreamId == trafficStreamID))
                {
                    CsrResult r;

                    CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                                          CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: ba session for Rx[traffic Id %d] already exists\n",
                                          priv ? priv->instance : 0,  trafficStreamID));

                    if ((baSessionRx->bufferSize == bufferSize) &&
                        (baSessionRx->expectedSn == ssn))
                    {
                        return CSR_RESULT_SUCCESS;
                    }

                    if (baSessionRx->bufferSize != bufferSize)
                    {
                        r = csrWifiHipBaSessionStop(priv, vif, role, trafficStreamID, macAddress);
                        if (r != CSR_RESULT_SUCCESS)
                        {
                            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: Failed to to stop session, trafficStreamId %u \n",
                                                   priv ? priv->instance : 0,
                                                   trafficStreamID));
                        }
                    }
                    else
                    {
                        /*
                         * The starting sequence number shall remain same if the BA
                         * enable request is issued to update BA parameters only. If
                         * it is not same, then we scroll our window to the new starting
                         * sequence number. This could happen if the DELBA frame from
                         * originator is lost and then we receive ADDBA frame with new SSN.
                        */
                        if (baSessionRx->startSn != ssn)
                        {
                            baScrollWindow(priv, vif, baSessionRx, ssn);
                        }
                        return CSR_RESULT_SUCCESS;
                    }
                }
            }
        }

        /* we could have a valid BA session pointer here or un-initialized
        ba session pointer. but in any case we have to create a new session.
        so re-initialize the baSession pointer */
        baSessionRx = NULL;

        /* Loop through until an empty BA session slot is there and save the session there */
        for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX; baSessionIdx++)
        {
            if (!(vif->baSessionRx[baSessionIdx]))
            {
                break;
            }
        }

        if (baSessionIdx == CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: All baSession used for Rx, NO free session available\n", priv ? priv->instance : 0));
            return CSR_RESULT_FAILURE;
        }


        baSessionRx = CsrMemAlloc(sizeof(CsrWifiHipBaSessionRx));
        if (!baSessionRx)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: Memory allocation failed for baSessionRx\n", priv ? priv->instance : 0));
            return CSR_RESULT_FAILURE;
        }

        CsrMemSet(baSessionRx, 0, sizeof(CsrWifiHipBaSessionRx));

        baSessionRx->bufferSize = bufferSize;
        baSessionRx->startSn = baSessionRx->expectedSn = ssn;
        baSessionRx->triggerBaAfterSsn = FALSE;

        baSessionRx->buffer = CsrMemAlloc(baSessionRx->bufferSize * sizeof(CsrWifiHipBaFrameDesc));
        if (!baSessionRx->buffer)
        {
            CsrMemFree(baSessionRx);
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: Memory allocation failed for buffer\n", priv ? priv->instance : 0));
            return CSR_RESULT_FAILURE;
        }

        CsrMemSet(baSessionRx->buffer, 0, baSessionRx->bufferSize * sizeof(CsrWifiHipBaFrameDesc));

        baSessionRx->vif = vif;
        baSessionRx->trafficStreamId = trafficStreamID;
        baSessionRx->macAddress = macAddress;

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                            "unifi%d: CsrWifiHipBlockAckEnableReq: create BA aging timer\n",
                            priv->instance));

        CsrMemSet(&baSessionRx->baAgingTimer, 0, sizeof(baSessionRx->baAgingTimer));
        ret = CsrTimerCreate(CsrWifiHipBaTimeout, (void *) baSessionRx, &baSessionRx->baAgingTimer);
        if (ret != CSR_RESULT_SUCCESS)
        {
            CsrMemFree(baSessionRx->buffer);
            CsrMemFree(baSessionRx);
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipBlockAckEnableReq: CsrTimerCreate failed\n",
                                   priv->instance));
            return CSR_RESULT_FAILURE;
        }
        ret = CsrMutexCreate(&baSessionRx->baAgingMutex);
        if (ret != CSR_RESULT_SUCCESS)
        {
            CsrMemFree(baSessionRx->buffer);
            CsrTimerDestroy(&baSessionRx->baAgingTimer);
            CsrMemFree(baSessionRx);
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq:  CSR_WIFI_HIP_SPINLOCK_REATE failed\n", priv ? priv->instance : 0));
            return CSR_RESULT_FAILURE;
        }

        baSessionRx->handleTimeout = FALSE;
        vif->baSessionRx[baSessionIdx] = baSessionRx;
    }

    return CSR_RESULT_SUCCESS;
}

void csrWifiHipBaProcessComplete(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif)
{
    CsrWifiHipBaFrameDesc *frameDesc;
    CsrUint8 i;

    for (i = 0; i < vif->baCompleteIndex; i++)
    {
        frameDesc = &vif->baComplete[i];
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: csrWifiHipBaProcessComplete: calling process_amsdu()"
                               , priv ? priv->instance : 0));
        csrWifiHipAmsduProcess(priv, vif, &frameDesc->signal, &frameDesc->bulkdata);
    }

    vif->baCompleteIndex = 0;
}

void csrWifiHipProcessMaPacketErrorInd(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, CSR_SIGNAL *signal)
{
    const CSR_MA_PACKET_ERROR_INDICATION *pktErrInd = &signal->u.MaPacketErrorIndication;
    CsrWifiHipBaSessionRx *baSession;
    CsrUint8 baSessionIdx = 0;
    CSR_PRIORITY userPriority;
    CSR_SEQUENCE_NUMBER sn;

    func_enter();

    userPriority = pktErrInd->UserPriority;
    if (userPriority > CSR_WIFI_HIP_BA_TRAFFIC_STREAM_MAX)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipProcessMaPacketErrorInd: Invalid userPriority %d\n",
                               priv ? priv->instance : 0,  userPriority));
        func_exit();
        return;
    }

    sn = pktErrInd->SequenceNumber;

    (void) CsrMutexLock(&priv->configurationMutex);
    /* To find the right baSession loop through the BA sessions, compare MAC address and tID */
    for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX; baSessionIdx++)
    {
        baSession = vif->baSessionRx[baSessionIdx];
        if (baSession)
        {
            if ((!CsrMemCmp(baSession->macAddress.a, pktErrInd->PeerQstaAddress.x, ETH_ALEN)) &&
                (baSession->trafficStreamId == (userPriority & CSR_WIFI_HIP_IEEE80211_QOS_CTL_TID_MASK)))
            {
                baScrollWindow(priv, vif, baSession, sn);
                break;
            }
        }
    }
    (void) CsrMutexUnlock(&priv->configurationMutex);

    csrWifiHipBaProcessComplete(priv, vif);
    func_exit();
}
