/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_hal_priv.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip.h"
#include "csr_sched.h"
#include "csr_pmem.h"
#include "csr_types.h"
#include "csr_sdio.h"
#include "csr_wifi_hip_raw_sdio.h"
#include "csr_wifi_common.h"
#include "csr_wifi_hip_util.h"
#include "csr_wifi_hip_list.h"
#include "csr_wifi_hip_io.h"
#include "csr_wifi_router_ctrl_lib.h"
#include "csr_wifi_hip_ap_util.h"
#include "csr_wifi_ps_if.h"

#ifdef CSR_LOG_ENABLE
CsrLogTextHandle *CSR_WIFI_HIP_LOG_ID = NULL;
CSR_LOG_TEXT_HANDLE_DEFINE(CsrWifiHalLto);
static const CsrCharString *legacy_subOrigins[] = CSR_WIFI_HIP_LOG_LEGACY_SUBORIGINS;
static const CsrCharString *subOrigins[] = CSR_WIFI_HAL_SUBORIGINS;
#endif

CsrResult CsrWifiHipPostInitReq(void *hipHandle, CsrBool scheduledInterrupt)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrUint32 interruptMode;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPostInitReq: Invalid priv\n", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    /* Configure interruption method */
    if (scheduledInterrupt)
    {
        interruptMode = CSR_WIFI_INTMODE_RUN_BH_ONCE;
    }
    else
    {
        interruptMode = CSR_WIFI_INTMODE_DEFAULT;
    }
    unifi_set_interrupt_mode(priv->card, interruptMode);

    unifi_request_max_sdio_clock(priv->card);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                        "unifi%d: HIP post init was successful\n", priv->instance));
    return CSR_RESULT_SUCCESS;
}

void CsrWifiHipSuspendRes(void *hipHandle, CsrResult status)
{
#ifdef CSR_WIFI_HIP_SUSPEND_ENABLE
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
#ifdef CSR_WIFI_HIP_SUSPEND_WOL
    CsrResult result;

    /* Signal to bh thread */
    result = CsrEventSet(&priv->bhEventHandle, CSR_WIFI_HIP_EVENT_MASK_BH_SUS_RESP);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipSuspendRes: Fail to Set event error code %u\n",
                               priv ? priv->instance : 0,  result));
    }
#else
    /* Send suspend ack here because bh thread cannot because bh has terminated because wifi is off */
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: CsrWifiHipSuspendRes: UniFi unpowered", priv->instance));
    CsrSdioSuspendAcknowledge(priv->sdio, status);
#endif
#endif
}

void CsrWifiHipResumeRes(void *hipHandle, CsrResult status)
{
#ifdef CSR_WIFI_HIP_SUSPEND_ENABLE
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrSdioResumeAcknowledge(priv->sdio, status);
#endif
}

CsrResult CsrWifiHipLowPowerModeReq(void *hipHandle, CsrWifiRouterCtrlLowPowerMode lowPowerMode, CsrBool wakeHost)
{
    CsrResult r;
    enum unifi_low_power_mode low_power_mode;
    enum unifi_periodic_wake_mode periodic_wake_mode;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    if (hipHandle == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipLowPowerModeReq: The hipHandle is invalid \n", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    if (lowPowerMode == CSR_WIFI_ROUTER_CTRL_LOW_POWER_MODE_DISABLED)
    {
        low_power_mode = UNIFI_LOW_POWER_DISABLED;
    }
    else if (lowPowerMode == CSR_WIFI_ROUTER_CTRL_LOW_POWER_MODE_ENABLED)
    {
        low_power_mode = UNIFI_LOW_POWER_ENABLED;
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Periodic_wake_host doesn't contain a valid argument \n", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    if (wakeHost)
    {
        periodic_wake_mode = UNIFI_PERIODIC_WAKE_HOST_ENABLED;
    }
    else
    {
        periodic_wake_mode = UNIFI_PERIODIC_WAKE_HOST_DISABLED;
    }

    r = unifi_configure_low_power_mode(priv->card, low_power_mode, periodic_wake_mode);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipLowPowerModeReq: Could not set low power mode, error code: %u \n",
                               priv ? priv->instance : 0,  r));
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
}

CsrResult CsrWifiHipPriorityGetReq(void                       *hipHandle,
                                   CsrUint16                   interfaceTag,
                                   CsrUint16                   protocol,
                                   CsrUint8                   *frame,
                                   CsrUint8                   *dMac,
                                   CsrWifiHipMaPacketPriority *priority,
                                   CsrInt32                   *aidp)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrWifiHipVifInstance *vif = NULL;
    CsrWifiHipStaPeerInfo *dstStaInfo = NULL;

    func_enter();

    if (interfaceTag < CSR_WIFI_MAX_INTERFACES)
    {
        vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPriorityGetReq: Invalid interfaceTag: %u \n", priv->instance, interfaceTag));
        return CSR_RESULT_FAILURE;
    }


    /* Priority Mapping for all the Modes */
    switch (vif->interfaceMode)
    {
        case CSR_WIFI_ROUTER_CTRL_MODE_STA:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI:
            #ifdef CSR_NATIVE_SOFTMAC
            *priority = csrWifiHipPacketPriorityGet(priv, vif, frame, protocol);
            #else
            if ((vif->staInfo.wmmControl & CSR_WIFI_HIP_QOS_CAPABILITY_WMM_ENABLED) == 1)
            {
                *priority = csrWifiHipPacketPriorityGet(priv, vif, frame, protocol);
            }
            else
            {
                *priority = CSR_WIFI_HIP_MA_PACKET_PRIORITY_CONTENTION;
            }
            #endif
            if (aidp)
            {
                *aidp = 0;
            }
            break;

        case CSR_WIFI_ROUTER_CTRL_MODE_AMP:
        {
            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                "unifi%d: CsrWifiHipPriorityGet: Unexpectedly entered CsrWifiHipPriorityGetReq in AMP mode. CsrWifiHipPriorityGet will return failure, since this is not suppose to happen\n",
                                priv->instance));
            return CSR_RESULT_FAILURE;
        }

        case CSR_WIFI_ROUTER_CTRL_MODE_AP:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PGO:
        case CSR_WIFI_ROUTER_CTRL_MODE_IBSS:
        {
            dstStaInfo = csrWifiHipGetStationRecordFromPeerMacAddress(priv,
                                                                      vif,
                                                                      dMac);
            if (!(*dMac & 0x01) && dstStaInfo && dstStaInfo->wmmOrQosEnabled)
            {
                /* If packet is not Broadcast/multicast and Association is WMM */
                *priority = csrWifiHipPacketPriorityGet(priv, vif, frame, protocol);
            }
            else
            {
                /* Since packet destination is not QSTA, set priority to CONTENTION */
                *priority = CSR_WIFI_HIP_MA_PACKET_PRIORITY_CONTENTION;
            }
            if (aidp && dstStaInfo)
            {
                *aidp = dstStaInfo->associationId;
            }
            break;
        }
        default:
        {
            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPriorityGetReq: Unknown mode = 0x%X\n",
                                priv->instance, vif->interfaceMode));
            return CSR_RESULT_FAILURE;
        }
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: CsrWifiHipPriorityGetReq: Priority = %x\n", priv->instance, *priority));

    func_exit();
    return CSR_RESULT_SUCCESS;
}

static CsrUint8 calculateAllocOffset(CsrWifiHipQosType qosFrame, CsrWifiHipVifInstance *vif)
{
    CsrUint8 offset = MAC_HEADER_SIZE;

    if (qosFrame == CSR_WIFI_HIP_QOS_TYPE_QOS_CONTROL)
    {
        offset += QOS_CONTROL_HEADER_SIZE;
    }

    if (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_AMP)
    {
        offset += ETH_ALEN;
    }

    return offset;
}

void CsrWifiHipFrameBufferSizeReq(void *hipHandle, CsrUint16 interfaceTag, CsrWifiHipQosType qosType, CsrUint8 *offset)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrWifiHipVifInstance *vif = NULL;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipFrameBufferSizeReq: Invalid priv\n", priv ? priv->instance : 0));
        *offset = 0;
        return;
    }

    if (interfaceTag < CSR_WIFI_MAX_INTERFACES)
    {
        vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipFrameBufferSizeReq: Invalid interfaceTag: %u \n",
                               priv ? priv->instance : 0,  interfaceTag));
        *offset = 0;
        return;
    }

    *offset = calculateAllocOffset(qosType, vif);
}

CsrResult csrWifiHipPackAndSendFrame(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, CsrUint32 aid,
                                     CsrWifiHipPortType portType,
                                     CsrWifiHipFrameType frameType,
                                     CsrWifiHipPacketType packetType,
                                     CSR_SIGNAL *signal,
                                     CsrWifiHipBulkDataParam *bulkdata)
{
    CsrUint8 sigbuf[UNIFI_PACKED_SIGBUF_SIZE];
    CsrUint16 packed_siglen;
    CsrResult r;

    r = write_pack(signal, sigbuf, &packed_siglen);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipPackAndSendFrame: Failed to pack HIP signal \n", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    r = unifi_send_signal(priv->card, vif->interfaceTag, aid, portType, frameType, packetType, sigbuf, packed_siglen, bulkdata);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: csrWifiHipPackAndSendFrame: Failed to queue the frame for transmission \n"
                               , priv ? priv->instance : 0));
    }

    return r;
}

void CsrWifiHipConstructMaPacketReqSignal(void *hipHandle, CsrUint16 interfaceTag,
                                          CsrUint32 priority, CsrUint16 transmitRate, CsrUint32 hostTag,
                                          CsrUint16 transmissionControl, CsrUint16 leSenderProcessId,
                                          CsrUint8 *dMac, CsrUint16 vifIndex, CsrUint8 *sigData, CsrWifiHipFrameType frameType, CsrUint16 minVifDuration)
{
    CSR_SIGNAL *signal = (CSR_SIGNAL *) sigData;
    CsrUint8 baSessionIdx = 0;
    CsrUint8 *baAddr = NULL;
    CsrWifiHipBaSessionTx *baSessionTx = NULL;
    CsrUint16 convertedLePriority;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrWifiHipVifInstance *vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];

    CSR_MA_PACKET_REQUEST *req = &signal->u.MaPacketRequest;

    CsrMemSet(signal, 0, sizeof(CSR_SIGNAL));
    signal->SignalPrimitiveHeader.SignalId = CSR_MA_PACKET_REQUEST_ID;
    signal->SignalPrimitiveHeader.ReceiverProcessId = 0;
    signal->SignalPrimitiveHeader.SenderProcessId = leSenderProcessId;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: CsrWifiHipConstructMaPacketReqSignal: MaPacketReq signal with Priority: 0x%x, Rate: %d\n",
                           priv ? priv->instance : 0,
                           priority,
                           transmitRate));
    /* Fill the MA-PACKET.req */
    req->Priority = (CSR_PRIORITY) priority;

    /* A value of 0 for rate is used for auto selection of rates. But for P2P GO case
     * for action frames the rate is governed by SME. Hence instead of 0,
     * the rate is filled in with the value passed here
     */
    req->TransmitRate = transmitRate;

    if (frameType == CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA)
    {
        if (priv->cmanrTestMode)
        {
            /* overwrite the requested rate */
            req->TransmitRate = priv->cmanrTestModeTransmitRate;
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                "unifi%d: CsrWifiHipConstructMaPacketReqSignal: cmanrTestModeTransmitRate = %d TransmitRate=%d\n",
                                priv ? priv->instance : 0, priv->cmanrTestModeTransmitRate, req->TransmitRate));
        }
    }


    /* packets from netdev then no confirm required but packets from
     * Nme/Sme eapol data frames requires the confirmation
     */
    req->TransmissionControlBitmap = transmissionControl;
    req->MinVifDuration = minVifDuration;
    req->VirtualInterfaceIndex = vifIndex;

    if (hostTag == CSR_WIFI_HIP_HOST_TAG_INVALID)
    {
        req->HostTag = vif->hostTag++;
        req->HostTag |= 0x40000000;
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: CsrWifiHipConstructMaPacketReqSignal: New Host tag assigned = 0x%x\n",
                               priv ? priv->instance : 0,  req->HostTag));
        vif->hostTag &= 0x0FFFFFFF;
    }
    else
    {
        req->HostTag = hostTag;
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: CsrWifiHipConstructMaPacketReqSignal: HostTag from SME 0x%x\n",
                               priv ? priv->instance : 0,  req->HostTag));
    }

    CsrMemCpy(req->Ra.x, dMac, ETH_ALEN);

    /* Get the BA recipient address */
    if ((vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_AP) ||
        (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PGO))
    {
        baAddr = dMac;
    }
    else
    {
        baAddr = vif->staInfo.bssid.a;
    }

    /* Check if BA session exists for the peer on same trafficStreamId
     * and if it exists set the transmission control bit for ALLOW_BA
     */
    for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_TX_MAX; baSessionIdx++)
    {
        baSessionTx = vif->baSessionTx[baSessionIdx];
        if (baSessionTx)
        {
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(priority, &convertedLePriority);

            if ((!CsrMemCmp(baSessionTx->macAddress.a, baAddr, ETH_ALEN))
                && (baSessionTx->trafficStreamId == (convertedLePriority & CSR_WIFI_HIP_IEEE80211_QOS_CTL_TID_MASK)))
            {
                req->TransmissionControlBitmap |= CSR_ALLOW_BA;
                break;
            }
        }
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: CsrWifiHipConstructMaPacketReqSignal: with TransmissionControl: 0x%x\n",
                           priv ? priv->instance : 0,  req->TransmissionControlBitmap));
}

/* D-29955: This function has lot of formal paramters, has to be reduced */
CsrResult CsrWifiHipMaPacketReq(void                    *hipHandle,
                                CsrUint8                *dMac,
                                CsrUint8                *sMac,
                                CsrUint16                senderProcessId,
                                CsrUint16                interfaceTag,
                                CsrWifiHipFrameType      frameType,
                                CsrUint16                protocol,
                                CsrWifiHipBulkDataParam *bulkdata,
                                CsrUint16                priority,
                                CsrUint16                transmitRate,
                                CsrUint32                hostTag,
                                CsrBool                  requestCfm,
                                CsrUint8                 allocedMacHdrMem,
                                CsrUint16                vifIndex,
                                CsrUint16                minVifDuration)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrResult r;
    CSR_SIGNAL signal;
    CSR_MA_PACKET_REQUEST *req = &signal.u.MaPacketRequest;
    CsrWifiRouterCtrlPortAction cpAction;
    CsrWifiHipVifInstance *vif = NULL;
    CsrUint16 transmissionControl = 0;
    CsrWifiHipPortType port = CSR_WIFI_HIP_PORT_TYPE_CONTROLLED;
    CsrWifiHipStaPeerInfo *dstStaInfo = NULL;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipMaPacketReq: Invalid hipHandle. Can't free frame\n"));
        return CSR_RESULT_FAILURE;
    }

    if ((bulkdata->d[0].os_data_ptr == NULL) ||
        (priv->bh_thread_terminate == 1) || (priv->bh_block_io_thread == 1))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipMaPacketReq: Could not transmit data due no bulkdata or HIP is about to terminate \n", priv ? priv->instance : 0));
        priv->freeFrame(priv->osLayerContext, &bulkdata->d[0]);
        return CSR_RESULT_FAILURE;
    }

    vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];

    if (!vifIndex || (vifIndex > (CSR_WIFI_HIP_MAX_VIFINDEX - 1)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipMaPacketReq: vifIndex is %x\n",
                               priv->instance, vifIndex));
        priv->freeFrame(priv->osLayerContext, &bulkdata->d[0]);
        return CSR_RESULT_FAILURE;
    }

    if (requestCfm)
    {
        transmissionControl &= ~(CSR_NO_CONFIRM_REQUIRED);
    }
    else
    {
        transmissionControl |= CSR_NO_CONFIRM_REQUIRED;
    }


    if ((vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_AP)
        || (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PGO)
        || (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_IBSS))
    {
        dstStaInfo = csrWifiHipGetStationRecordFromPeerMacAddress(priv, vif, dMac);
    }

    CsrWifiHipConstructMaPacketReqSignal(priv, interfaceTag, priority, transmitRate, hostTag, transmissionControl,
                                         senderProcessId, dMac, vifIndex,
                                         (CsrUint8 *) &signal, frameType, minVifDuration);


    if (frameType == CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA)
    {
        /* Add MAC header to bulk data */
        r = csrWifiHipPrepareAndAddMacheader(priv,
                                             vif,
                                             req->Priority,
                                             protocol,
                                             dMac,
                                             sMac,
                                             allocedMacHdrMem,
                                             req->HostTag,
                                             (CsrWifiHipBulkDataParam *) bulkdata,
                                             dstStaInfo);

        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipMaPacketReq: Could not add MAC header to bulkdata \n",
                                   priv ? priv->instance : 0));
            priv->freeFrame(priv->osLayerContext, &bulkdata->d[0]);
            return r;
        }
    }

    CsrMemCpy(req->Ra.x, (bulkdata->d[0].os_data_ptr + 4), ETH_ALEN);

    /* Management frames shouldn't be restricted by the controlled port */
    if (frameType == CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA)
    {
        if ((protocol != ETH_P_PAE) && (protocol != ETH_P_WAI))
        {
            port = CSR_WIFI_HIP_PORT_TYPE_CONTROLLED;
        }
        else
        {
            port = CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED;
        }

        cpAction = csrWifiHipControlledPortState(priv, vif, dMac, port);
        if ((port == CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED) &&
            ((cpAction == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN) ||
             (cpAction == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_BLOCK)))
        {
            /* Continue transmit: the packet scheduler implements the port rules */
        }
        else if ((port == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED) &&
                 (cpAction == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN))
        {
            /* Continue transmit: the packet scheduler implements the port rules */
        }
        else
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,
                                "unifi%d: CsrWifiHipMaPacketReq: Frame dropped. Port was closed",
                                priv ? priv->instance : 0));

            priv->freeFrame(priv->osLayerContext, &bulkdata->d[0]);

            func_exit();
            return CSR_WIFI_HIP_CP_DROPPED_PACKET;
        }
    }

    /* Notify the TA module about the Tx frame */
    if ((vif->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_AP) &&
        (vif->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_P2PGO))
    {
        CsrWifiHipBulkDataParam bulkdataNoMacHeader;
        CsrUint32 taMinVifDuration = 0;

        bulkdataNoMacHeader.d[0].data_length = bulkdata->d[0].data_length - allocedMacHdrMem;
        bulkdataNoMacHeader.d[0].net_buf_length = bulkdata->d[0].net_buf_length - allocedMacHdrMem;
        bulkdataNoMacHeader.d[0].os_data_ptr = bulkdata->d[0].os_data_ptr + allocedMacHdrMem;
        bulkdataNoMacHeader.d[0].os_net_buf_ptr = bulkdata->d[0].os_net_buf_ptr;

        unifi_ta_sample_tx(priv->card, interfaceTag,
                           &bulkdataNoMacHeader.d[0],
                           (const CsrUint8 *) sMac,
                           (const CsrUint8 *) vif->staInfo.sta_mac_addresses.a,
                           CsrTimeGet(NULL),
                           &taMinVifDuration);
        if (!minVifDuration)
        {
            req->MinVifDuration = taMinVifDuration;
        }
    }

    r = csrWifiHipProcessMaPacketReq(priv,
                                     vif,
                                     port,
                                     frameType,
                                     dMac,
                                     req->Priority,
                                     &signal,
                                     bulkdata,
                                     dstStaInfo);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipMaPacketReq: Failed to send/enqueue MaPacketReq, error code %u\n",
                               priv ? priv->instance : 0,  r));
        priv->freeFrame(priv->osLayerContext, &bulkdata->d[0]);

        if (r == CSR_WIFI_HIP_RESULT_NO_SPACE)
        {
            return CSR_WIFI_HIP_TX_PAUSE;
        }

        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
}

void CsrWifiHipMaPacketCancelReq(void *hipHandle, CsrUint16 senderProcessId, CsrUint16 interfaceTag, CsrUint32 hostTag)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipMaPacketCancelReq: Invalid hipHandle \n", priv ? priv->instance : 0));
        return;
    }

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipMaPacketCancelReq: Invalid interfaceTag \n", priv ? priv->instance : 0));
        return;
    }
    else
    {
        CSR_SIGNAL signal;
        CSR_MA_PACKET_CANCEL_REQUEST *request = &signal.u.MaPacketCancelRequest;
        CsrWifiHipBulkDataParam *bulkdata = NULL;
        CsrResult r;
        CsrWifiHipVifInstance *vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag & 0xFF];

        signal.SignalPrimitiveHeader.SignalId = CSR_MA_PACKET_CANCEL_REQUEST_ID;
        signal.SignalPrimitiveHeader.ReceiverProcessId = 0;
        signal.SignalPrimitiveHeader.SenderProcessId = senderProcessId;

        request->VirtualInterfaceIndex = CsrWifiHipVifIndexGetReq(hipHandle, interfaceTag);

        request->HostTag = hostTag;

        /* Packing MA Packet Cancel request */
        r = csrWifiHipPackAndSendFrame(priv, vif, 0,
                                       CSR_WIFI_HIP_PORT_TYPE_CONTROLLED,
                                       CSR_WIFI_HIP_FRAMETYPE_IEEE80211_RESERVED,
                                       CSR_WIFI_HIP_PKT_TYPE_RESERVED, &signal, bulkdata);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipMaPacketCancelReq: unifi_send_signal failed, error code %u \n",
                                   priv ? priv->instance : 0,  r));
        }
    }
}

/* Checks whether a signal is a Data MA-PACKET.req, and updates the passed in frameType and
 * packetType accordingly if so. The types are left unchanged otherwise.
 */
static CsrResult updateMaPacketReqDataTypes(const CSR_SIGNAL              *sigPtr,
                                            const CsrWifiHipBulkDataParam *bulkdata,
                                            CsrWifiHipFrameType           *frameType,
                                            CsrWifiHipPacketType          *packetType)
{
    /* Check for data frames to be routed via an AC queue */
    if (sigPtr->SignalPrimitiveHeader.SignalId == CSR_MA_PACKET_REQUEST_ID)
    {
        const CsrUint8 *peerMacAddress = &sigPtr->u.MaPacketRequest.Ra.x[0];
        const CsrUint16 frameCtrl = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(bulkdata->d[0].os_data_ptr);

        if ((frameCtrl & CSR_WIFI_HIP_IEEE80211_FCTL_TYPE_MASK) == CSR_WIFI_HIP_IEEE80211_FCTL_TYPE_DATA)
        {
            *frameType = CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA;

            /* Determine destination type (unicast or multicast/broadcast) */
            if (*peerMacAddress & 0x01)
            {
                *packetType = CSR_WIFI_HIP_MULTICAST_PDU;
            }
            else
            {
                *packetType = CSR_WIFI_HIP_UNICAST_PDU;
            }
        }

        /* It was an MA_PACKET_REQ, frameType and packetType are updated if necessary */
        return CSR_RESULT_SUCCESS;
    }

    /* Not an MA_PACKET_REQ */
    return CSR_RESULT_FAILURE;
}

CsrResult CsrWifiHipUnpackedSignalReq(void *hipHandle, CsrUint16 interfaceTag, CsrUint32 aId, CsrUint16 sigLength,
                                      CsrUint8 *sigData, CsrWifiHipBulkDataParam *bulkdata)
{
    CsrWifiHipHalPriv *priv;
    CSR_SIGNAL *sigPtr = (CSR_SIGNAL *) sigData;
    CsrUint8 sigbuf[UNIFI_PACKED_SIGBUF_SIZE];
    CsrUint16 packed_siglen;
    CsrResult csrResult;
    CsrWifiHipFrameType frameType = CSR_WIFI_HIP_FRAMETYPE_IEEE80211_RESERVED; /* default to the control queue */
    CsrWifiHipPacketType packetType = CSR_WIFI_HIP_PKT_TYPE_RESERVED;
    CsrWifiHipPortType portType = CSR_WIFI_HIP_PORT_TYPE_CONTROLLED; /* default */

    if (hipHandle == NULL)
    {
        return CSR_RESULT_FAILURE;
    }

    priv = (CsrWifiHipHalPriv *) hipHandle;

    if ((interfaceTag & 0xFF) >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipUnpackedSignalReq: Invalid interfaceTag %u\n", priv ? priv->instance : 0, interfaceTag));
        return CSR_RESULT_FAILURE;
    }

    csrResult = write_pack(sigPtr, sigbuf, &packed_siglen);
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Malformed HIP signal in CsrWifiHipUnpackedSignalReq()\n", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

#ifndef CSR_WIFI_DRIVER_UDI_SINGLE_QUEUE
    /* Update the frameType and packetType fields based on the contents of the bulkdata */
    (void) updateMaPacketReqDataTypes(sigPtr, bulkdata, &frameType, &packetType);
#endif

    csrResult = unifi_send_signal(priv->card, interfaceTag, aId,
                                  portType, frameType, packetType,
                                  sigbuf, packed_siglen, bulkdata);

    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4,
                            "unifi%d: CsrWifiHipUnpackedSignalReq: Failed to queue the frame for transmission \n",
                            priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
}

CsrResult CsrWifiHipPackedSignalReq(void *hipHandle, CsrUint16 interfaceTag, CsrUint16 sigLength, CsrUint8 *sigData, CsrWifiHipBulkDataParam *bulkdata)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrResult r;
#ifndef CSR_NATIVE_LINUX
    CSR_SIGNAL *signal = (CSR_SIGNAL *) sigData;
    CsrUint8 counter;
#endif

    if (priv == NULL)
    {
        return CSR_RESULT_FAILURE;
    }

#ifdef CSR_NATIVE_LINUX
    /* AId 0 is fine for management frames - also for the AP case */
    r = unifi_send_signal(priv->card, interfaceTag, 0,
                          CSR_WIFI_HIP_PORT_TYPE_CONTROLLED,
                          CSR_WIFI_HIP_FRAMETYPE_IEEE80211_RESERVED,
                          CSR_WIFI_HIP_PKT_TYPE_RESERVED, sigData, sigLength,
                          bulkdata);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4,
                            "unifi%d: CsrWifiHipPackedSignalReq: Failed to queue the frame for transmission \n",
                            priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
#else
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                        "unifi%d: CsrWifiHipPackedSignalReq: >>>\n", priv ? priv->instance : 0));

    /* Check that we have received a valid MLME command */
    if ((sigData == NULL) ||
        (priv->bh_thread_terminate == 1) || (priv->bh_block_io_thread == 1))
    {
        if (sigData != NULL)
        {
            CsrMemFree(sigData);
            sigData = NULL;
        }

        for (counter = 0; counter < CSR_WIFI_HIP_MAX_DATA_REFERENCES; counter++)
        {
            priv->freeFrame(priv->osLayerContext, &bulkdata->d[counter]);
        }

        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPackedSignalReq: Invalid hipHandle (%p), no data or HIP is about to terminate \n", priv ? priv->instance : 0, priv));
        return CSR_RESULT_FAILURE;
    }

    if (signal->SignalPrimitiveHeader.SignalId == CSR_MA_PACKET_REQUEST_ID)
    {
        CSR_SIGNAL unpackedSignalHeader;
        CsrUint8 *dMac, *sMac;
        CSR_MA_PACKET_REQUEST request;
        CsrBool confirmRequired = FALSE;
        CsrWifiHipFrameType frameType = CSR_WIFI_HIP_FRAMETYPE_IEEE80211_MANAGEMENT;
        CsrWifiHipPacketType dummy;

        /* Use the Ma Packet transmission mechanism */

        r = read_unpack_signal(sigData, &unpackedSignalHeader);
        if (r)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPackedSignalReq: Fail to unpack signal \n", priv ? priv->instance : 0));
            for (counter = 0; counter < CSR_WIFI_HIP_MAX_DATA_REFERENCES; counter++)
            {
                priv->freeFrame(priv->osLayerContext, &bulkdata->d[counter]);
            }
            CsrMemFree(sigData);
            return r;
        }

        request = unpackedSignalHeader.u.MaPacketRequest;

        if ((interfaceTag = CsrWifiHipInterfaceTagGetReq(hipHandle, request.VirtualInterfaceIndex)) == 0xFFFF)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPackedSignalReq: Invalid interfaceTag\n", priv ? priv->instance : 0));
            for (counter = 0; counter < CSR_WIFI_HIP_MAX_DATA_REFERENCES; counter++)
            {
                priv->freeFrame(priv->osLayerContext, &bulkdata->d[counter]);
            }
            CsrMemFree(sigData);
            return CSR_RESULT_FAILURE;
        }

        if (!(request.TransmissionControlBitmap & CSR_NO_CONFIRM_REQUIRED))
        {
            confirmRequired = TRUE;
        }

        dMac = (CsrUint8 *) request.Ra.x;
        sMac = (CsrUint8 *) priv->vif[interfaceTag].staInfo.sta_mac_addresses.a;

#ifndef CSR_WIFI_DRIVER_UDI_SINGLE_QUEUE
        /* Update the frameType field based on the contents of the bulkdata */
        (void) updateMaPacketReqDataTypes(&unpackedSignalHeader, bulkdata, &frameType, &dummy);
#endif
        r = CsrWifiHipMaPacketReq(hipHandle,
                                  dMac,
                                  sMac,
                                  unpackedSignalHeader.SignalPrimitiveHeader.SenderProcessId,
                                  interfaceTag,
                                  frameType,
                                  0, /* no protocol for management frame */
                                  bulkdata,
                                  (CsrUint16) request.Priority,
                                  (CsrUint16) request.TransmitRate,
                                  request.HostTag,
                                  confirmRequired,
                                  0,
                                  request.VirtualInterfaceIndex,
                                  request.MinVifDuration);
    }
    else
    {
        /* Ensure that the data slot number is initialised since the upper byte is used
           to signal bulkdata offset in case the platform sets alignment requirements */
        sigData[SIZEOF_SIGNAL_HEADER + 1] = 0x0;
        sigData[SIZEOF_SIGNAL_HEADER + SIZEOF_DATAREF + 1] = 0x0;

        r = unifi_send_signal(priv->card, interfaceTag, 0,
                              CSR_WIFI_HIP_PORT_TYPE_CONTROLLED,
                              CSR_WIFI_HIP_FRAMETYPE_IEEE80211_RESERVED,
                              CSR_WIFI_HIP_PKT_TYPE_RESERVED, sigData, sigLength,
                              bulkdata);
        if (r)
        {
            for (counter = 0; counter < CSR_WIFI_HIP_MAX_DATA_REFERENCES; counter++)
            {
                priv->freeFrame(priv->osLayerContext, &bulkdata->d[counter]);
            }
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPackedSignalReq: Failed to send MLME signal, error code %d \n",
                                   priv ? priv->instance : 0,  r));
        }
    }

    CsrMemFree(sigData);
    sigData = NULL;

    return r;
#endif
}

CsrResult CsrWifiHipActivateReq(void)
{
#ifdef CSR_LOG_ENABLE
    CsrLogTextRegister2(&CSR_WIFI_HIP_LOG_ID, "CsrWifiHip", CSR_ARRAY_SIZE(legacy_subOrigins), legacy_subOrigins);
    CsrLogTextRegister2(&CsrWifiHalLto, "HAL", CSR_ARRAY_SIZE(subOrigins), subOrigins);
#endif
    CSR_LOG_TEXT_DEBUG((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_INIT,
                        "unifi : CsrWifiHipActivateReq\n"));

    return csrWifiHipSdioRegister();
}

CsrResult CsrWifiHipDeactivateReq(void)
{
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                        "unifi : CsrWifiHipDeactivateReq\n"));

    csrWifiHipSdioUnregister();
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipWifiOnReq
 *
 *      This function is called from non-blocking Synergy context; hence, start
 *      BH thread and return.
 *
 *
 *  Arguments:
 *      void
 *
 *  Returns:
 *      Status
 *
 *  Notes:
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipWifiOnReq(void *hipHandle, void *osLayerContext, CsrSchedQid smeAppHandle, CsrWifiHipInstanceConfig config)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrResult result;

    if ((hipHandle == NULL) || (osLayerContext == NULL))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipWifiOnReq: Invalid hipHandle or invalid osLayerContext pointer \n"));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d : CsrWifiHipWifiOnReq\n", priv->instance));

#ifndef CSR_WIFI_DRIVER_HYDRA
    /* Power on UniFi */
    CsrSdioClaim(priv->sdio);
    result = CsrSdioPowerOn(priv->sdio);
    CsrSdioRelease(priv->sdio);
    if ((result != CSR_RESULT_SUCCESS) && (result != CSR_SDIO_RESULT_NOT_RESET))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipWifiOnReq: Failed to power on UniFi\n", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }
#endif

    priv->fw_init = config.fw_init;
    priv->cardParamsCache = config.cardParams;
    priv->allocFrame = config.allocFunc;
    priv->freeFrame = config.freeFunc;
    priv->hipMode = config.hipMode;
    priv->smeAppHandle = smeAppHandle;
    priv->wifiOnState = CSR_WIFI_HIP_WIFI_ON_IN_PROGRESS;
    priv->osLayerContext = osLayerContext;
    priv->blockingTerminate = FALSE;
    priv->cmanrTestMode = config.cmanrTestMode;

    if ((config.flowControlResumeCb == NULL) || (config.flowControlPauseCb == NULL))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipWifiOnReq: The flow control %s callback is NULL. Can not complete CsrWifiHipWifiOnReq\n",
                               config.flowControlPauseCb == NULL ? "pause" : "resume"));
        return CSR_RESULT_FAILURE;
    }
    else
    {
        priv->resumeCb = config.flowControlResumeCb;
        priv->pauseCb = config.flowControlPauseCb;
    }

    /* Create event group. This event group will be used to terminate HIP */
    result = CsrEventCreate(&priv->eventHandle);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipWifiOnReq: CsrEventCreate return error code %u \n", result));
        return CSR_RESULT_FAILURE;
    }

    if (priv->hipMode == CSR_WIFI_HIP_WIFI_STACK_MODE)
    {
        /* WiFi stack mode */

        /* Start the bh thread */
        result = csrWifiHipBhInit(priv);
        if (result != CSR_RESULT_SUCCESS)
        {
            /* Thread creation error */
            CsrEventDestroy(&priv->eventHandle);

            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi : CsrWifiHipWifiOnReq: Could not start BH thread in WIFI mode, error code %u \n", result));
            return CSR_RESULT_FAILURE;
        }
    }
    else
    {
        /* Raw SDIO API */
        result = CsrWifiHipRawSdioInitialise(priv, FALSE); /* Initialise and reset */
        if (result != CSR_RESULT_SUCCESS)
        {
            CsrEventDestroy(&priv->eventHandle);

            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi : CsrWifiHipWifiOnReq: Could not initialise RAW SDIO, error code %u \n", result));
            return CSR_RESULT_FAILURE;
        }
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                        "unifi : CsrWifiHipWifiOnReq was successful\n"));
    return CSR_RESULT_SUCCESS;
}

void CsrWifiHipWifiOffReq(CsrBool blockingTerminate, void *priv_unifi)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) priv_unifi;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi: CsrWifiHipWifiOffReq: No HIP instances are running!\n"));
        return;
    }

    if (priv->hipMode == CSR_WIFI_HIP_WIFI_STACK_MODE)
    {
        CsrResult result;
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE) && defined(CSR_LOG_ENABLE)
        unifi_debug_buf_dump();
#endif
        if (blockingTerminate)
        {
            priv->blockingTerminate = TRUE;
        }
        else
        {
            priv->blockingTerminate = FALSE;
        }

        /* Signal to bh thread that it shall terminate */
        result = CsrEventSet(&priv->bhEventHandle, CSR_WIFI_HIP_EVENT_MASK_BH_CLOSE);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipWifiOffReq: Fail to Set event error code %u \n",
                                   priv ? priv->instance : 0,  result));
            priv->eventBits = 0;
            CsrEventDestroy(&priv->eventHandle);
            return;
        }

        /* Wait for BH thread to terminate and signal its closed event.
         * In the event of a card initialisation failure, the BH thread won't have
         * started, so check it is running before waiting.
         */
        if (blockingTerminate && (priv->bhThreadStarted == 1))
        {
            while ((priv->eventBits & CSR_WIFI_HIP_EVENT_MASK_CLOSED_BH) !=
                   CSR_WIFI_HIP_EVENT_MASK_CLOSED_BH)
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                    "unifi%d: CsrWifiHipWifiOffReq: Wait for BH to terminate\n", priv ? priv->instance : 0));

                result = CsrEventWait(&priv->eventHandle, CSR_EVENT_WAIT_INFINITE, &priv->eventBits);
                if (result != CSR_RESULT_SUCCESS)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                           "unifi%d: CsrWifiHipWifiOffReq: Waiting for BH to terminate failed, error code %u \n",
                                           priv ? priv->instance : 0,  result));
                    break;
                }

                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                    "unifi%d: CsrWifiHipWifiOffReq: Wait for BH to terminate, got event 0x%x\n",
                                    priv ? priv->instance : 0, priv->eventBits));
            }
        }

        /* The MMC driver requires that the interrupt handler is unregistered explicitly.
         * CSR_WIFI_HIP_LINUX_REMOVE_IRQ has to be called from this context, and not the
         * BH thread, since the MMC/SDIO driver tries to kill its' interrupt thread,
         * which may cause problem when the BH thread is terminating as well. */
        result = CSR_WIFI_HIP_LINUX_REMOVE_IRQ(priv->sdio);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipWifiOffReq: Could not remove the IRQ handler %u\n",
                                   priv ? priv->instance : 0, result));
        }
    }
    else
    {
        /* Raw SDIO API */
        CsrWifiHipRawSdioDeinitialise();
    }

#ifndef CSR_WIFI_DRIVER_HYDRA
    if (csrWifiHipPowerChipOff(priv))
    {
        CsrSdioClaim(priv->sdio);
        CsrSdioPowerOff(priv->sdio);
        CsrSdioRelease(priv->sdio);
    }
#endif
    priv->eventBits = 0;
    CsrEventDestroy(&priv->eventHandle);
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipStartCoreDumpReq
 *
 *  Enable Raw SDIO on an already-started HIP process for capturing coredump
 *
 *  Arguments:
 *      hipHandle
 *
 *  Returns:
 *      Status
 *
 *  Notes:
 *      This allows the raw SDIO functions to be used without resetting the
 *      UniFi which is required to capture its state.
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipStartCoreDumpReq(void *hipHandle)
{
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                        "unifi : CsrWifiHipStartCoreDumpReq\n"));

#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE)
    unifi_debug_buf_dump();
#endif
    return CsrWifiHipRawSdioInitialise(hipHandle, TRUE);    /* Don't reset */
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipStopCoreDumpReq
 *
 *  Finish HIP coredump over raw SDIO
 *
 *  Arguments:
 *      hipHandle
 *
 *  Returns:
 *      None
 *
 *  Notes:
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipStopCoreDumpReq(void *hipHandle)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrResult r;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF,
                        "unifi : CsrWifiHipStopCoreDumpReq\n"));

    if (hipHandle)
    {
        /* Ensure the XAPs are left running when coredump completes */
        CsrSdioClaim(priv->sdio);
        r = unifi_start_processors(priv->card);
        CsrSdioRelease(priv->sdio);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to restart XAPs. Hard reset may be required.\n", priv ? priv->instance : 0));
        }
    }
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipWapiMicFilterReq
 *
 *  Set/Reset the filters used for special MIC error handling in WAPI
 *
 *  Arguments:
 *       hipHandle
 *       interfaceTag
 *       status
 *       isUnicast
 *  Returns:
 *       None
 *
 *  Notes:
 *---------------------------------------------------------------------------
 */
void CsrWifiHipWapiMicFilterReq(void     *hipHandle,
                                CsrUint16 interfaceTag,
                                CsrBool   status,
                                CsrBool   isUnicast)
{
#if defined(CSR_WIFI_SECURITY_WAPI_ENABLE)
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipWapiMicFilterReq: Invalid hipHandle\n"));
        return;
    }
    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipWapiMicFilterReq: Invalid interfaceTag\n",
                               priv->instance));
        return;
    }
    else
    {
        CsrWifiHipVifInstance *vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag & 0xFF];
        if (isUnicast)
        {
            vif->staInfo.wapi_unicast_mic_filter = status;
        }
        else
        {
            vif->staInfo.wapi_multicast_mic_filter = status;
        }
    }
#else
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi : CsrWifiHipWapiMicFilterReq: called when WAPI isn't enabled\n"));
#endif
}

#ifdef CSR_LOG_ENABLE
/*
 * ---------------------------------------------------------------------------
 *  debug_string_indication
 *  debug_word16_indication
 *
 *      Handlers for debug indications.
 *
 *  Arguments:
 *      priv            Pointer to private context structure.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void debug_string_indication(CsrWifiHipHalPriv *priv, const void *extra, CsrUint32 extralen)
{
    const CsrUint32 maxlen = CSR_WIFI_HIP_LAST_DEBUG_STRING_LENGTH - 1;

    if (extralen > maxlen)
    {
        extralen = maxlen;
    }

    if (!(CsrStrNCpy(priv->debugInfo.last_debug_string, (const CsrCharString *) extra, extralen)))
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: string copy failed in debug_string_indication\n", priv ? priv->instance : 0));
        return;
    }

    /* Make sure the string is terminated */
    priv->debugInfo.last_debug_string[maxlen] = '\0';

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                        "unifi%d: unifi debug: %s\n",
                        priv->instance, priv->debugInfo.last_debug_string));
}

static void debug_word16_indication(CsrWifiHipHalPriv *priv, const CSR_SIGNAL *sigptr)
{
    CsrUint8 i;

    for (i = 0; i < 16; i++)
    {
        priv->debugInfo.last_debug_word16[i] =
            sigptr->u.DebugWord16Indication.DebugWords[i];
    }

    if (priv->debugInfo.last_debug_word16[0] == 0xFA11)
    {
        CsrUint32 ts;
        ts = (priv->debugInfo.last_debug_word16[6] << 16) | priv->debugInfo.last_debug_word16[5];
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi%d:  %10lu: %s fault %04x, arg %04x (x%d)\n",
                            priv->instance,
                            ts,
                            priv->debugInfo.last_debug_word16[3] == 0x8000 ? "MAC" :
                            priv->debugInfo.last_debug_word16[3] == 0x4000 ? "PHY" :
                            "???",
                            priv->debugInfo.last_debug_word16[1],
                            priv->debugInfo.last_debug_word16[2],
                            priv->debugInfo.last_debug_word16[4]));
    }
    else if (priv->debugInfo.last_debug_word16[0] != 0xDBAC)
    {
        /* suppress SDL Trace output (note: still available to unicli). */
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi%d: unifi debug: %04X %04X %04X %04X %04X %04X %04X %04X\n",
                            priv->instance,
                            priv->debugInfo.last_debug_word16[0], priv->debugInfo.last_debug_word16[1],
                            priv->debugInfo.last_debug_word16[2], priv->debugInfo.last_debug_word16[3],
                            priv->debugInfo.last_debug_word16[4], priv->debugInfo.last_debug_word16[5],
                            priv->debugInfo.last_debug_word16[6], priv->debugInfo.last_debug_word16[7]));
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi%d:              %04X %04X %04X %04X %04X %04X %04X %04X\n",
                            priv->instance,
                            priv->debugInfo.last_debug_word16[8], priv->debugInfo.last_debug_word16[9],
                            priv->debugInfo.last_debug_word16[10], priv->debugInfo.last_debug_word16[11],
                            priv->debugInfo.last_debug_word16[12], priv->debugInfo.last_debug_word16[13],
                            priv->debugInfo.last_debug_word16[14], priv->debugInfo.last_debug_word16[15]));
    }
}

static void debug_generic_indication(CsrWifiHipHalPriv *priv, const CSR_SIGNAL *sigptr)
{
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                        "unifi%d: debug: %04X %04X %04X %04X %04X %04X %04X %04X\n",
                        priv->instance,
                        sigptr->u.DebugGenericIndication.DebugWords[0],
                        sigptr->u.DebugGenericIndication.DebugWords[1],
                        sigptr->u.DebugGenericIndication.DebugWords[2],
                        sigptr->u.DebugGenericIndication.DebugWords[3],
                        sigptr->u.DebugGenericIndication.DebugWords[4],
                        sigptr->u.DebugGenericIndication.DebugWords[5],
                        sigptr->u.DebugGenericIndication.DebugWords[6],
                        sigptr->u.DebugGenericIndication.DebugWords[7]));
}

#endif /* #ifdef CSR_LOG_ENABLE */

void CsrWifiHipProcessDataRx(CsrWifiHipHalPriv       *priv,
                             CsrWifiHipVifInstance   *vif,
                             CsrUint16                interfaceTag,
                             CSR_SIGNAL              *sigptr,
                             CsrResult                receptionResult,
                             CsrWifiHipBulkDataParam *bulkdata,
                             CsrUint16                frameControl,
                             CsrUint8                *dMac,
                             CsrUint8                *sMac)
{
    CsrUint8 macHeaderLength = MAC_HEADER_SIZE;
    CsrUint8 toDS = 0, fromDS = 0;
    CsrUint16 protocol = 0;
    CsrWifiHipPortType queue;
    CsrWifiRouterCtrlPortAction portAction;
    CsrWifiHipBulkDataDesc bulkDataPayload;

    toDS = CSR_WIFI_HIP_IEEE80211_HAS_TO_DS(frameControl);
    fromDS = CSR_WIFI_HIP_IEEE80211_HAS_FROM_DS(frameControl);

    /* Calculate MAC header size */
    switch ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK))
    {
        case CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_DATA:
        case CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_NULL:

            /* If both toDs and fromDs are set then the Address 4 exists (only for AP) */
            if (fromDS & toDS)
            {
                /* ETH_ALEN is the size of Address 4 field */
                macHeaderLength += (QOS_CONTROL_HEADER_SIZE + ETH_ALEN);
            }
            else
            {
                macHeaderLength += QOS_CONTROL_HEADER_SIZE;
            }

            /* If order bit is set, then HT control field is the part of MAC header */
            if ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_ORDER))
            {
                macHeaderLength += HT_CONTROL_HEADER_SIZE;
            }

            break;

        default:

            if (fromDS & toDS)
            {
                macHeaderLength += ETH_ALEN;
            }
    }

    switch ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK))
    {
        case CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_NULL:
        case CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_NULL:
            protocol = 0;
            break;
        default:
        {
            protocol = bulkdata->d[0].os_data_ptr[macHeaderLength + CSR_WIFI_LLC_SNAP_HDR_PROT_OFFSET] << 8 |
                       bulkdata->d[0].os_data_ptr[macHeaderLength + CSR_WIFI_LLC_SNAP_HDR_PROT_OFFSET + 1];
        }
    }

    if ((protocol != ETH_P_PAE) && (protocol != ETH_P_WAI))
    {
        queue = CSR_WIFI_HIP_PORT_TYPE_CONTROLLED;
    }
    else
    {
        queue = CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED;
    }

    portAction = csrWifiHipControlledPortState(priv, vif, sMac, queue);

    bulkDataPayload.os_data_ptr = (CsrUint8 *) bulkdata->d[0].os_data_ptr + macHeaderLength;
    bulkDataPayload.data_length = bulkdata->d[0].data_length - macHeaderLength;
    bulkDataPayload.net_buf_length = bulkdata->d[0].net_buf_length - macHeaderLength;
    bulkDataPayload.os_net_buf_ptr = bulkdata->d[0].os_net_buf_ptr;

    if ((vif->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_AP) &&
        (vif->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_P2PGO))
    {
        /* checking here for NULL Data, before sending for traffic analysis */
        if (bulkDataPayload.data_length > 0)
        {
            /* Notify TA about the RX frame */
            unifi_ta_sample_rx(priv->card, interfaceTag,
                               &bulkDataPayload,
                               sMac,
                               (const CsrUint8 *) vif->staInfo.sta_mac_addresses.a,
                               CsrTimeGet(NULL),
                               sigptr->u.MaPacketIndication.ReceivedRate
                               );
        }
    }


#ifndef CSR_NATIVE_LINUX
#ifdef CSR_WIFI_AP_ENABLE
    else
    {
        /* For AP/P2P GO mode, get the peer station record */
        CsrWifiHipStaPeerInfo *staPeerInfo = csrWifiHipGetStationRecordFromPeerMacAddress(priv,
                                                                                          vif,
                                                                                          sMac);

        if (staPeerInfo == NULL)
        {
            CsrWifiMacAddress sMacAddress;

            CsrMemCpy(sMacAddress.a, sMac, ETH_ALEN);

            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                               "unifi%d: CsrWifiHipProcessDataRx: Unexpected Frame Indication STA = (%02x:%02x:%02x:%02x:%02x:%02x)\n",
                               priv->instance, sMac[0], sMac[1], sMac[2], sMac[3], sMac[4], sMac[5]));

            CsrWifiRouterCtrlUnexpectedFrameIndSend(priv->smeAppHandle,
                                                    0,
                                                    interfaceTag,
                                                    sMacAddress);

            priv->vif[interfaceTag].receiveStats.rxDropped++;
            goto free_now;
        }

        /* Update station activity flag */
        staPeerInfo->active = TRUE;

        if (portAction != CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN)
        {
            priv->vif[interfaceTag].receiveStats.rxDropped++;
            goto free_now;
        }

        /* null-data frames are dropped here */
        if (((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK) == CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_NULL)
            || ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK) == CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_NULL))
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,
                               "unifi%d: CsrWifiHipProcessDataRx: NULL frame received, Dropped\n",
                               priv->instance));
            priv->vif[interfaceTag].receiveStats.rxDropped++;
            goto free_now;
        }

        /* This function takes care of appropriate routing for AP/P2PGO case.
         * The function handles mainly
         * - intra BSS relay for group addressed frames
         * - unicast frames destined for other stations in same BSS
         */
        if (csrWifiHipApProcessDataPdu(priv,
                                       vif,
                                       dMac,
                                       staPeerInfo,
                                       sigptr,
                                       bulkdata,
                                       macHeaderLength) != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                                "unifi%d: CsrWifiHipProcessDataRx: AP consumed the frame, no need to pass it to L3\n",
                                priv->instance));
            func_exit();
            return;
        }

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                            "unifi%d: CsrWifiHipProcessDataRx: No specific AP handling process as normal frame\n",
                            priv->instance));
    }
#endif /* CSR_WIFI_AP_ENABLE */
#endif /* !CSR_NATIVE_LINUX */

    /* Now that the MAC header is removed, null-data frames have zero length
     * and can be dropped
     */
    if (bulkDataPayload.data_length == 0)
    {
        if (((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK) != CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_NULL) &&
            ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK) != CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_NULL))
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                               "unifi%d: CsrWifiHipProcessDataRx: Zero length frame, but not null-data %04x\n",
                               priv->instance, frameControl));
            priv->vif[interfaceTag].receiveStats.rxDropped++;
        }

        goto free_now;
    }

    /* Apply (un)controlled port handling */
    if (portAction != CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN)
    {
        if (portAction == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_BLOCK)
        {
            /* Queue most recent EAPOL frame */
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4,
                                "unifi%d: CsrWifiHipProcessDataRx: Have to buffer frame\n",
                                priv->instance));

            /* To avoid any races between the port being opened and queuing of frames
               the configurationMutex will be claimed and the port state will be re-read */
            (void) CsrMutexLock(&priv->configurationMutex);
            portAction = csrWifiHipControlledPortState(priv, vif, sMac, queue);
            if (portAction == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_BLOCK)
            {
                csrWifiHipCtrlPortQueueEntryAdd(priv,
                                                vif,
                                                queue,
                                                sigptr,
                                                (CsrWifiHipBulkDataParam *) bulkdata,
                                                dMac,
                                                sMac,
                                                macHeaderLength);

                (void) CsrMutexUnlock(&priv->configurationMutex);
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                   "unifi%d: CsrWifiHipProcessDataRx: Frame queued since port is blocked\n",
                                   priv->instance));
                return;
            }
            else if (portAction == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN)
            {
                (void) CsrMutexUnlock(&priv->configurationMutex);
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                   "unifi%d: CsrWifiHipProcessDataRx: Continue. Port has been opened since the first check\n",
                                   priv->instance));
                goto forward_frame;
            }
            else
            {
                (void) CsrMutexUnlock(&priv->configurationMutex);
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipProcessDataRx: Dropping packet, proto=0x%04x, port=%u. Port has been closed since first check\n",
                                   priv->instance, protocol, portAction));

                priv->vif[interfaceTag].receiveStats.rxDropped++;
                goto free_now;
            }
        }
        else
        {
            /* Drop the packet and return */
            if (queue == CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED)
            {
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipProcessDataRx: Uncontrolled port, dropping packet, proto=0x%04x, port=%u\n",
                                   priv->instance, protocol, portAction));
            }
            else
            {
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipProcessDataRx: Controlled port, dropping packet, proto=0x%04x, port=%u\n",
                                   priv->instance, protocol, portAction));
            }

            priv->vif[interfaceTag].receiveStats.rxDropped++;
            goto free_now;
        }
    }

forward_frame:
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                            "unifi%d: CsrWifiHipProcessDataRx: Transfering data to layer 3\n",
                            priv->instance));

        /* Send the frame to router */
        priv->vif[interfaceTag].receiveStats.rxBytes += bulkdata->d[0].data_length;
        priv->vif[interfaceTag].receiveStats.rxPackets++;

        if (priv->cmanrTestMode)
        {
            priv->cmanrTestModeTransmitRate = sigptr->u.MaPacketIndication.ReceivedRate;
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                "unifi%d: CsrWifiHipProcessDataRx: cmanrTestModeTransmitRate=%d\n",
                                priv->instance, priv->cmanrTestModeTransmitRate));
        }

        CsrWifiHipMaPacketInd(priv->osLayerContext,
                              interfaceTag,
                              sigptr->SignalPrimitiveHeader.ReceiverProcessId,
                              sizeof(CSR_SIGNAL),
                              (CsrUint8 *) sigptr,
                              dMac,
                              sMac,
                              CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA,
                              macHeaderLength,
                              (CsrWifiHipBulkDataParam *) bulkdata,
                              receptionResult,
                              sigptr->u.MaPacketIndication.Rssi,
                              sigptr->u.MaPacketIndication.Snr,
                              sigptr->u.MaPacketIndication.ReceivedRate);
        return;
    }

free_now:
    {
        CsrUint8 i;

        for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
        {
            if (bulkdata->d[i].data_length != 0)
            {
                priv->freeFrame(priv->osLayerContext, (void *) &bulkdata->d[i]);
            }
        }
    }
}

void unifi_receive_event(void *ospriv, CsrUint8 *sigdata, CsrUint32 siglen, CsrWifiHipBulkDataParam *bulkdata)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) ospriv;
    CsrUint16 signalId;

    func_enter();

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : unifi_receive_event: Invalid priv. Can't free frame since priv is invalid\n"));
        return;
    }

    /* Unpack signal id */
    signalId = GET_SIGNAL_ID(sigdata);

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2, "unifi%d: unifi_receive_event: Process signal 0x%.4X\n", priv->instance, signalId));

    /**************************************
     **** Send MA_PACKET_IND to router ****
     **************************************/
    if (signalId == CSR_MA_PACKET_INDICATION_ID)
    {
        CsrResult receptionResult;
        CsrUint16 interfaceTag;
        CSR_SIGNAL sigptr;
        CsrUint8 *dMac = NULL, *sMac = NULL, *bssid = NULL;
        CsrBool routeDataInd = FALSE;
        CsrWifiHipVifInstance *vif = NULL;
        CsrUint16 frameControl = 0;
        CsrUint16 receptionStatus = 0;
        CsrUint8 toDS = 0, fromDS = 0;
        CsrBool micError = FALSE;
        CsrResult r;

        frameControl = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(bulkdata->d[0].os_data_ptr);

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,
                            "unifi%d: unifi_receive_event: Sequence = %d\n", priv->instance,
                            (CSR_GET_UINT16_FROM_LITTLE_ENDIAN(bulkdata->d[0].os_data_ptr + CSR_WIFI_HIP_MAC_HEADER_SEQ_CTRL_OFFSET)) >> 4 & 0xfff));


        /* Unpack signal header */
        r = read_unpack_signal(sigdata, &sigptr);
        if (r)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_receive_event: Unpack failed (%u), Frame dropped\n",
                                   priv->instance, r));
            goto free_now;
        }

        toDS = CSR_WIFI_HIP_IEEE80211_HAS_TO_DS(frameControl);
        fromDS = CSR_WIFI_HIP_IEEE80211_HAS_FROM_DS(frameControl);
        dMac = (CsrUint8 *) bulkdata->d[0].os_data_ptr + 4 + (toDS * 12);                  /* Address 1 or 3 */
        sMac = (CsrUint8 *) bulkdata->d[0].os_data_ptr + 10 + (fromDS * (6 + (toDS * 8))); /* Address 2, 3 or 4 */

        /* Fetch and validate interface tag */
        if ((interfaceTag = CsrWifiHipInterfaceTagGetReq(ospriv, sigptr.u.MaPacketIndication.VirtualInterfaceIndex)) == 0xFFFF)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_receive_event: Received MaPacketInd with invalid interfaceTag %u\n",
                                   priv->instance,  interfaceTag));
            goto free_now;
        }

        vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];


        /* Verify whether packet to be routed to SME or to driver/Network stack*/

        receptionStatus = sigptr.u.MaPacketIndication.ReceptionStatus;
        if (receptionStatus == CSR_MIC_ERROR)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_receive_event: MIC failure", priv->instance));
            micError = TRUE;
            routeDataInd = TRUE;
        }
        else if ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_TYPE_MASK) == CSR_WIFI_HIP_IEEE80211_FCTL_TYPE_MGMT)
        {
            micError = FALSE;
            routeDataInd = TRUE;
        }

#ifdef CSR_WIFI_AP_ENABLE
        if ((receptionStatus == CSR_NO_TEMPORAL_KEY_AVAILABLE) && ((vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_AP) ||
                                                                   (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PGO)))
        {
            CsrWifiHipStaPeerInfo *staPeerInfo = csrWifiHipGetStationRecordFromPeerMacAddress(priv,
                                                                                              vif,
                                                                                              sMac);

            if (staPeerInfo == NULL)
            {
                CsrWifiMacAddress sMacAddress;

                CsrMemCpy(sMacAddress.a, sMac, ETH_ALEN);

                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                   "unifi%d: unifi_receive_event: Unexpected Frame Indication STA = (%02x:%02x:%02x:%02x:%02x:%02x)\n",
                                   priv->instance, sMac[0], sMac[1], sMac[2], sMac[3], sMac[4], sMac[5]));

                CsrWifiRouterCtrlUnexpectedFrameIndSend(priv->smeAppHandle,
                                                        0,
                                                        interfaceTag,
                                                        sMacAddress);

                priv->vif[interfaceTag].receiveStats.rxDropped++;
                goto free_now;
            }
        }
#endif

        if (routeDataInd)
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                                "unifi%d: unifi_receive_event: Routing MaPacketInd to the SME\n",
                                priv->instance));

            /* Message should go to SME */
            if (micError)
            {
                CsrBool unicastPdu = TRUE;
                CsrWifiMacAddress peerMacAddress;
                CsrUint8 *macHdrLocation;
                CsrUint8 *raddr = NULL, *taddr = NULL;

                macHdrLocation = (CsrUint8 *) bulkdata->d[0].os_data_ptr;

                /* Point to the addresses */
                raddr = macHdrLocation + CSR_WIFI_HIP_MAC_HEADER_ADDR1_OFFSET;
                taddr = macHdrLocation + CSR_WIFI_HIP_MAC_HEADER_ADDR2_OFFSET;

                CsrMemCpy(peerMacAddress.a, taddr, ETH_ALEN);

                if (*raddr & 0x1)
                {
                    unicastPdu = FALSE;
                }

#if defined(CSR_WIFI_SECURITY_WAPI_ENABLE)
                /* STA (WAPI) - Special handling of MIC errors under the certain scenarios */
                if ((vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_STA) &&
                    (!unicastPdu) &&
                    (vif->staInfo.wapi_multicast_mic_filter))
                {
                    /* Forward the multicast PDU to the host for MIC validation when the filter has been set by the host */
                    CsrWifiRouterCtrlWapiRxMicCheckIndSend(priv->smeAppHandle,
                                                           0,
                                                           interfaceTag,
                                                           unicastPdu,
                                                           siglen,
                                                           sigdata,
                                                           bulkdata->d[0].data_length,
                                                           (CsrUint8 *) bulkdata->d[0].os_data_ptr);
                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                        "unifi%d: unifi_receive_event: calling CsrWifiRouterCtrlWapiRxMicCheckIndSend() interfaceTag=0x%.4X data_length=%d\n",
                                        priv ? priv->instance : 0, interfaceTag, bulkdata->d[0].data_length));
                    goto free_now;
                }
#endif
                if ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_TYPE_MASK) != CSR_WIFI_HIP_IEEE80211_FCTL_TYPE_DATA)
                {
                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                        "unifi%d: unifi_receive_event: Has MIC error but not a data frame. Frame dropped\n", priv ? priv->instance : 0));
                    goto free_now;
                }

                if (((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK) != CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_DATA)
                    && ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK) != CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_DATA))
                {
                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                        "unifi%d: unifi_receive_event: Has MIC error but data frame with no data. Frame dropped\n", priv ? priv->instance : 0));
                    goto free_now;
                }

                if ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_PROTO_VERSION_MASK) != 0x00)
                {
                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                        "unifi%d: unifi_receive_event: Has MIC error but bad protocol version. Frame dropped\n", priv ? priv->instance : 0));
                    goto free_now;
                }
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                    "unifi%d: unifi_receive_event: calling CsrWifiRouterCtrlMicFailureIndSend()\n",
                                    priv ? priv->instance : 0));
                CsrWifiRouterCtrlMicFailureIndSend(priv->smeAppHandle,
                                                   0,
                                                   interfaceTag,
                                                   peerMacAddress,
                                                   unicastPdu);

                goto free_now;
            }
#ifdef CSR_WIFI_AP_ENABLE
            /* In P2P device mode, drop the legacy AP beacons */
            if ((vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2P) && \
                ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK) == CSR_WIFI_HIP_IEEE80211_FRAME_SUBTYPE_BEACON))
            {
                CsrUint8 *pSsid, *pSsidLen;
                static const CsrUint8 p2pWildCardSsid[CSR_WIFI_HIP_IEEE80211_P2P_WILDCARD_SSID_LENGTH] = {'D', 'I', 'R', 'E', 'C', 'T', '-'};

                pSsidLen = ((CsrUint8 *) bulkdata->d[0].os_data_ptr) + MAC_HEADER_SIZE + CSR_WIFI_HIP_IEEE80211_BEACON_SSID_OFFSET;
                pSsid = pSsidLen + 2;

                if (*(pSsidLen + 1) >= CSR_WIFI_HIP_IEEE80211_P2P_WILDCARD_SSID_LENGTH)
                {
                    if (CsrMemCmp(pSsid, p2pWildCardSsid, CSR_WIFI_HIP_IEEE80211_P2P_WILDCARD_SSID_LENGTH) != 0)
                    {
                        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                                            "unifi%d: unifi_receive_event: Received a Legacy AP beacon in P2P mode, drop it\n",
                                            priv ? priv->instance : 0));
                        goto free_now;
                    }
                }
                else
                {
                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                                        "unifi%d: unifi_receive_event: Received a Legacy AP beacon in P2P mode, drop it\n",
                                        priv ? priv->instance : 0));
                    goto free_now;
                }
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                                    "unifi%d: unifi_receive_event: Received a P2P Beacon, pass it to SME\n",
                                    priv ? priv->instance : 0));
            }
#endif
            CsrWifiHipPackedSignalInd(priv->osLayerContext,
                                      (CsrUint16) siglen,
                                      sigdata,
                                      (CsrWifiHipBulkDataParam *) bulkdata);
            return;
        }

        /* Convert reception status */
        if (sigptr.u.MaPacketIndication.ReceptionStatus == CSR_RX_SUCCESS)
        {
            receptionResult = CSR_RESULT_SUCCESS;
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG7,
                                "unifi%d: unifi_receive_event: SNR = %d, RSSI = %d dBm, Rate = %d\n",
                                priv->instance,
                                sigptr.u.MaPacketIndication.Snr,
                                sigptr.u.MaPacketIndication.Rssi,
                                sigptr.u.MaPacketIndication.ReceivedRate));
        }
        else
        {
            receptionResult = CSR_RESULT_FAILURE;
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: unifi_receive_event: Received MaPacketInd with status %d, Frame dropped\n",
                                   priv ? priv->instance : 0,  sigptr.u.MaPacketIndication.ReceptionStatus));

            vif->receiveStats.rxDropped++;
            goto free_now;
        }


        if (toDS && fromDS)
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                                "unifi%d: unifi_receive_event: 4 address frame: Don't try to find BSSID\n",
                                priv ? priv->instance : 0));
            bssid = NULL;
        }
        else
        {
            bssid = (CsrUint8 *) ((bulkdata->d[0].os_data_ptr + 4 + 12 - (fromDS * 6)) - (toDS * 12));
        }

        /* Identify received frame */
        if ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_TYPE_MASK) == CSR_WIFI_HIP_IEEE80211_FCTL_TYPE_CTRL)
        {
            CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                  "unifi%d: unifi_receive_event: Received Control Frame (frame control=%x)\n",
                                  priv ? priv->instance : 0, frameControl));
            vif->receiveStats.rxDropped++;
            goto free_now;
        }

        if ((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_TYPE_MASK) != CSR_WIFI_HIP_IEEE80211_FCTL_TYPE_DATA)
        {
            CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                  "unifi%d: unifi_receive_event: Not a control or data frame received, Frame dropped\n",
                                  priv->instance));
            vif->receiveStats.rxDropped++;
            goto free_now;
        }

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                            "unifi%d: unifi_receive_event: Received a data frame\n",
                            priv ? priv->instance : 0));


        /********** MPDU Re-order and A-MSDU de-aggregation **********/

        /* AMP traffic has the WMM header but no BA session. */
        if (((frameControl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK) == CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_DATA) &&
            (vif->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_AMP))
        {
            CsrWifiHipIeee80211Hdr hdr;
            CsrWifiRouterCtrlTrafficStreamId trafficStreamId;
            CsrWifiHipBaSessionRx *baSession;
            CsrUint8 baSessionIdx = 0;
            CsrUint8 *baAddr = NULL;

            hdr.frameControl = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(bulkdata->d[0].os_data_ptr);
            hdr.durationId = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(bulkdata->d[0].os_data_ptr + 2);
            hdr.seqCtrl = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(bulkdata->d[0].os_data_ptr + MAC_HEADER_SIZE - 2);


            if (CSR_WIFI_HIP_IEEE80211_HAS_A4((hdr.frameControl)))
            {
                /* In case the MAC header contains 4 MAC adddresses */
                trafficStreamId = (*(bulkdata->d[0].os_data_ptr + MAC_HEADER_SIZE + ETH_ALEN) & CSR_WIFI_HIP_IEEE80211_QOS_CTL_TID_MASK);
            }
            else
            {
                /* If the MAC header contains 3 MAC addresses */
                trafficStreamId = (*(bulkdata->d[0].os_data_ptr + MAC_HEADER_SIZE) & CSR_WIFI_HIP_IEEE80211_QOS_CTL_TID_MASK);
            }

            /* Get the BA originator address */
            if ((vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_AP) ||
                (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PGO))
            {
                baAddr = sMac;
            }
            else
            {
                baAddr = bssid;
            }

            if (baAddr)
            {
                (void) CsrMutexLock(&priv->configurationMutex);
                for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX; baSessionIdx++)
                {
                    baSession = vif->baSessionRx[baSessionIdx];
                    if (baSession)
                    {
                        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: unifi_receive_event: Found baSession 0x%x baSessionIdx=%d \n",
                                            priv ? priv->instance : 0, baSession, baSessionIdx));
                        if ((!CsrMemCmp(baSession->macAddress.a, baAddr, ETH_ALEN)) && (baSession->trafficStreamId == trafficStreamId))
                        {
                            CsrWifiHipBaFrameDesc frameDesc;
                            frameDesc.bulkdata = *bulkdata;
                            frameDesc.signal = sigptr;
                            frameDesc.sn = (hdr.seqCtrl & CSR_WIFI_HIP_IEEE80211_SCTL_SEQ) >> 4;
                            frameDesc.active = TRUE;
                            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: unifi_receive_event: calling csrWifiHipBaProcessFrame (session=%d)\n",
                                                priv ? priv->instance : 0, baSessionIdx));
                            csrWifiHipBaProcessFrame(priv, vif, baSession, &frameDesc);
                            (void) CsrMutexUnlock(&priv->configurationMutex);
                            csrWifiHipBaProcessComplete(priv, vif);
                            break;
                        }
                    }
                }

                if (baSessionIdx == CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX)
                {
                    (void) CsrMutexUnlock(&priv->configurationMutex);
                    csrWifiHipAmsduProcess(priv, vif, &sigptr, bulkdata);
                }
            }
            else
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: unifi_receive_event: Dropping frame. WMM frame but couldn't identify baAddr\n",
                                    priv ? priv->instance : 0));
                vif->receiveStats.rxDropped++;
                goto free_now;
            }
        }
        else
        {
            CsrWifiHipProcessDataRx(priv, vif, interfaceTag,
                                    &sigptr, receptionResult,
                                    bulkdata,
                                    frameControl, dMac, sMac);
        }

        return;
    }


    /**************************************
     ****** Send MA PACKET ERROR ind ******
     **************************************/
    if (signalId == CSR_MA_PACKET_ERROR_INDICATION_ID)
    {
        CSR_SIGNAL sigptr;
        CsrUint16 interfaceTag;
        CsrWifiHipVifInstance *vif = NULL;

        /* Unpack signal header */
        (void) read_unpack_signal(sigdata, &sigptr);

        /* Fetch and validate interface tag */
        if ((interfaceTag = CsrWifiHipInterfaceTagGetReq(ospriv, sigptr.u.MaPacketErrorIndication.VirtualInterfaceIndex)) == 0xFFFF)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_receive_event: Received MaPacketErrorInd with invalid interfaceTag %u\n",
                                   priv ? priv->instance : 0,  sigptr.u.MaPacketConfirm.VirtualInterfaceIndex));
            goto free_now;
        }

        vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];

        csrWifiHipProcessMaPacketErrorInd(priv, vif, &sigptr);
        goto free_now;
    }


    /**************************************
     **** Send MA Packet cfm to router ****
     **************************************/
    if (signalId == CSR_MA_PACKET_CONFIRM_ID)
    {
        CSR_SIGNAL sigptr;
        CsrUint16 interfaceTag;
        CsrResult transmissionResult;

        /* Unpack signal header */
        (void) read_unpack_signal(sigdata, &sigptr);

        /* Convert transmission status */
        if (sigptr.u.MaPacketConfirm.TransmissionStatus == CSR_TX_SUCCESSFUL)
        {
            transmissionResult = CSR_RESULT_SUCCESS;
        }
        else
        {
            transmissionResult = CSR_RESULT_FAILURE;
        }
        /* Fetch and validate interface tag */
        if ((interfaceTag = CsrWifiHipInterfaceTagGetReq(ospriv, sigptr.u.MaPacketConfirm.VirtualInterfaceIndex)) == 0xFFFF)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_receive_event: Received MaPacketCfm with invalid interfaceTag %u\n",
                                   priv ? priv->instance : 0,  sigptr.u.MaPacketConfirm.VirtualInterfaceIndex));
            goto free_now;
        }

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                            "unifi%d: unifi_receive_event: Received MaPacketCfm HostTag=%x status=%u\n",
                            priv ? priv->instance : 0,  sigptr.u.MaPacketConfirm.HostTag, sigptr.u.MaPacketConfirm.TransmissionStatus));

#ifndef CSR_NATIVE_LINUX
#ifdef CSR_WIFI_AP_ENABLE
        if (!csrWifiHipProcessMaPacketConfirm(priv, interfaceTag, &sigptr))
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, "unifi%d: Received MaPacketCfm is consumed by HAL\n"
                                   , priv ? priv->instance : 0));
            goto free_now;
        }
#endif
#endif
        CsrWifiHipMaPacketCfm(priv->osLayerContext,
                              interfaceTag,
                              (CsrUint16) sigptr.SignalPrimitiveHeader.ReceiverProcessId,
                              transmissionResult,
                              sigptr.u.MaPacketConfirm.Rate,
                              sigptr.u.MaPacketConfirm.HostTag,
                              (CsrUint16) siglen,
                              sigdata,
                              (CsrWifiHipBulkDataParam *) bulkdata);

        return;
    }


    /*******************************
     **** Display debug signals ****
     *******************************/
    if ((signalId == CSR_DEBUG_GENERIC_CONFIRM_ID) || (signalId == CSR_DEBUG_GENERIC_INDICATION_ID) ||
        (signalId == CSR_DEBUG_STRING_INDICATION_ID) || (signalId == CSR_DEBUG_WORD16_INDICATION_ID))
    {
#ifdef CSR_LOG_ENABLE
        CSR_SIGNAL sigptr;
        CsrResult r;

        r = read_unpack_signal(sigdata, &sigptr);
        if (r)
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d:  ***************************************************************** \n", priv ? priv->instance : 0));
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d:  * A debug signal has been received - but it hasn't been unpacked * \n", priv ? priv->instance : 0));
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d:  ***************************************************************** \n", priv ? priv->instance : 0));
        }
        else if (signalId == CSR_DEBUG_WORD16_INDICATION_ID)
        {
            debug_word16_indication(priv, &sigptr);
        }
        else if (signalId == CSR_DEBUG_STRING_INDICATION_ID)
        {
            debug_string_indication(priv, bulkdata->d[0].os_data_ptr, bulkdata->d[0].data_length);
        }
        else if ((signalId == CSR_DEBUG_GENERIC_CONFIRM_ID) || (signalId == CSR_DEBUG_GENERIC_INDICATION_ID))
        {
            debug_generic_indication(priv, &sigptr);
        }
        else
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d:  ***************************************************************** \n", priv ? priv->instance : 0));
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d:  * A debug signal has been received - but it hasn't been decoded * \n", priv ? priv->instance : 0));
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d:  ***************************************************************** \n", priv ? priv->instance : 0));
        }
#endif  /* CSR_LOG_ENABLE */
    }


    /********************************************
     **** Send hip indications to the router ****
     ********************************************/

    CsrWifiHipPackedSignalInd(priv->osLayerContext,
                              (CsrUint16) siglen,
                              sigdata,
                              (CsrWifiHipBulkDataParam *) bulkdata);
    return;


free_now:
    {
        CsrUint8 i;

        for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
        {
            if (bulkdata->d[i].data_length != 0)
            {
                priv->freeFrame(priv->osLayerContext, (void *) &bulkdata->d[i]);
            }
        }
    }

    func_exit();
}

#if defined(CSR_WIFI_SECURITY_WAPI_ENABLE)
static CsrResult wapiMaPacketIndPackFunction(const CSR_SIGNAL *sig, CsrUint8 *ptr, CsrUint16 *sig_len)
{
    CsrInt16 index = 0;

    CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->SignalPrimitiveHeader.SignalId, ptr + index);
    index += SIZEOF_UINT16;

    CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->SignalPrimitiveHeader.ReceiverProcessId, ptr + index);
    index += SIZEOF_UINT16;

    CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->SignalPrimitiveHeader.SenderProcessId, ptr + index);
    index += SIZEOF_UINT16;

    switch (sig->SignalPrimitiveHeader.SignalId)
    {
        case CSR_MA_PACKET_INDICATION_ID:
        {
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->u.MaPacketIndication.Data.SlotNumber, ptr + index);
            index += SIZEOF_UINT16;
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->u.MaPacketIndication.Data.DataLength, ptr + index);
            index += SIZEOF_UINT16;
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->u.MaPacketIndication.Dummydataref2.SlotNumber, ptr + index);
            index += SIZEOF_UINT16;
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->u.MaPacketIndication.Dummydataref2.DataLength, ptr + index);
            index += SIZEOF_UINT16;
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->u.MaPacketIndication.VirtualInterfaceIndex, ptr + index);
            index += SIZEOF_UINT16;
            CsrMemCpy(ptr + index, sig->u.MaPacketIndication.LocalTime.x, 64 / 8);
            index += 64 / 8;
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->u.MaPacketIndication.ChannelFrequency, ptr + index);
            index += SIZEOF_UINT16;
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->u.MaPacketIndication.ChannelInformation, ptr + index);
            index += SIZEOF_UINT16;
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->u.MaPacketIndication.ReceptionStatus, ptr + index);
            index += SIZEOF_UINT16;
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->u.MaPacketIndication.Rssi, ptr + index);
            index += SIZEOF_UINT16;
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->u.MaPacketIndication.Snr, ptr + index);
            index += SIZEOF_UINT16;
            CSR_COPY_UINT16_TO_LITTLE_ENDIAN(sig->u.MaPacketIndication.ReceivedRate, ptr + index);
            index += SIZEOF_UINT16;
            break;
        }

        default:
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi : wapiMaPacketIndPackFunction: Unknown signal ID\n"));
            return CSR_RESULT_FAILURE;
        }
    }

    return CSR_RESULT_SUCCESS;
}

#endif

void CsrWifiHipWapiMicVerifiedFrameRxReq(void *hipHandle, CsrUint16 interfaceTag,
                                         CsrUint32 signalLength, CsrUint8 *signal,
                                         CsrWifiHipBulkDataParam *bulkdata)
{
#if defined(CSR_WIFI_SECURITY_WAPI_ENABLE)
    CSR_SIGNAL updatedSignal;
    CsrResult result;
    CsrUint8 sigPack[UNIFI_PACKED_SIGBUF_SIZE];

    if (hipHandle == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipWapiMicVerifiedFrameRxReq: Invalid priv. Can't free frame since priv is invalid\n"));
        return;
    }

    /* Update reception status since the MIC is OK. This requires
       that the signal is unpacked, updated and packed again. */
    result = read_unpack_signal(signal, &updatedSignal);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipWapiMicVerifiedFrameRxReq: Could not unpack signal\n"));
        return;
    }

    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi : CsrWifiHipWapiMicVerifiedFrameRxReq: Unpacked signal %u\n", updatedSignal.SignalPrimitiveHeader.SignalId));

    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi : CsrWifiHipWapiMicVerifiedFrameRxReq: ReceptionsStatus  %u\n", updatedSignal.u.MaPacketIndication.ReceptionStatus));

    updatedSignal.u.MaPacketIndication.ReceptionStatus = CSR_RX_SUCCESS;

    result = wapiMaPacketIndPackFunction((const CSR_SIGNAL *) &updatedSignal, sigPack, (CsrUint16 *) &signalLength);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipWapiMicVerifiedFrameRxReq: Could not pack signal\n"));
        return;
    }

    unifi_receive_event(hipHandle, sigPack, signalLength, bulkdata);

    /* The signal pointer is freed when this function return,
       nothing more to do here. */
#elif defined(CSR_LOG_ENABLE)
    CsrWifiHipHalPriv *priv = hipHandle;

    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: CsrWifiHipWapiMicVerifiedFrameRxReq: called when WAPI isn't enabled\n",
                           priv ? priv->instance : 0));
#endif
}
