/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"

#include <linux/moduleparam.h>
#include <linux/module.h>
#include <linux/init.h>

#include "os_linux_priv.h"

#include "sdio_freescale/fs_sdio_api.h"

static CsrSdioFunctionDriver *sdio_func_drv;


/* MMC uses ENOMEDIUM to indicate card gone away */
static CsrResult ConvertSdioToCsrSdioResult(int r)
{
    CsrResult csrResult = CSR_RESULT_FAILURE;

    switch (r)
    {
        case 0:
            csrResult = CSR_RESULT_SUCCESS;
            break;
        case -EIO:
        case -EILSEQ:
            csrResult = CSR_SDIO_RESULT_CRC_ERROR;
            break;
        /* Timeout errors */
        case -ETIMEDOUT:
        case -EBUSY:
            csrResult = CSR_SDIO_RESULT_TIMEOUT;
            break;
        case -ENODEV:
        case -ENOMEDIUM:
            csrResult = CSR_SDIO_RESULT_NO_DEVICE;
            break;
        case -EINVAL:
            csrResult = CSR_SDIO_RESULT_INVALID_VALUE;
            break;
        case -ENOMEM:
        case -ENOSYS:
        case -ERANGE:
        case -ENXIO:
            csrResult = CSR_RESULT_FAILURE;
            break;
        default:
            CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                                  CSR_WIFI_HIP_LOG_DEF,
                                  "unifi : Unrecognised SDIO error code: %d\n", r));
            break;
    }

    return csrResult;
}

CsrResult CsrSdioRead8(CsrSdioFunction *function, CsrUint32 address, CsrUint8 *data)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int err = 0;

    err = fs_sdio_readb(fdev, 1, address, data);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }
    return CSR_RESULT_SUCCESS;
} /* CsrSdioRead8() */

CsrResult CsrSdioWrite8(CsrSdioFunction *function, CsrUint32 address, CsrUint8 data)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int err = 0;

    err = fs_sdio_writeb(fdev, 1, address, data);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }
    return CSR_RESULT_SUCCESS;
} /* CsrSdioWrite8() */

CsrResult CsrSdioRead16(CsrSdioFunction *function, CsrUint32 address, CsrUint16 *data)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int err;
    CsrUint8 b0, b1;

    err = fs_sdio_readb(fdev, 1, address, &b0);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }

    err = fs_sdio_readb(fdev, 1, address + 1, &b1);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }

    *data = ((CsrUint16) b1 << 8) | b0;

    return CSR_RESULT_SUCCESS;
} /* CsrSdioRead16() */

CsrResult CsrSdioWrite16(CsrSdioFunction *function, CsrUint32 address, CsrUint16 data)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int err;
    uint8_t b0, b1;

    b1 = (data >> 8) & 0xFF;
    err = fs_sdio_writeb(fdev, 1, address + 1, b1);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }

    b0 = data & 0xFF;
    err = fs_sdio_writeb(fdev, 1, address, b0);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }

    return CSR_RESULT_SUCCESS;
} /* CsrSdioWrite16() */

CsrResult CsrSdioF0Read8(CsrSdioFunction *function, CsrUint32 address, CsrUint8 *data)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int err = 0;

    err = fs_sdio_readb(fdev, 0, address, data);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }
    return CSR_RESULT_SUCCESS;
} /* CsrSdioF0Read8() */

CsrResult CsrSdioF0Write8(CsrSdioFunction *function, CsrUint32 address, CsrUint8 data)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int err = 0;

    err = fs_sdio_writeb(fdev, 0, address, data);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }
    return CSR_RESULT_SUCCESS;
} /* CsrSdioF0Write8() */

CsrResult CsrSdioRead(CsrSdioFunction *function, CsrUint32 address, void *data, CsrUint32 length)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int err;

    err = fs_sdio_block_rw(fdev, 1, address, data, length, 0);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }
    return CSR_RESULT_SUCCESS;
} /* CsrSdioRead() */

CsrResult CsrSdioWrite(CsrSdioFunction *function, CsrUint32 address, const void *data, CsrUint32 length)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int err;

    err = fs_sdio_block_rw(fdev, 1, address, (unsigned char *) data, length, 1);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }
    return CSR_RESULT_SUCCESS;
} /* CsrSdioWrite() */

CsrResult CsrSdioInterruptEnable(CsrSdioFunction *function)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int err = 0;

    err = fs_sdio_enable_interrupt(fdev, 1);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }
    return CSR_RESULT_SUCCESS;
} /* CsrSdioInterruptEnable() */

CsrResult CsrSdioInterruptDisable(CsrSdioFunction *function)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int err = 0;

    err = fs_sdio_enable_interrupt(fdev, 0);
    if (err)
    {
        return ConvertSdioToCsrSdioResult(err);
    }
    return CSR_RESULT_SUCCESS;
} /* CsrSdioInterruptDisable() */

void CsrSdioInterruptAcknowledge(CsrSdioFunction *function)
{
}

CsrResult CsrSdioFunctionEnable(CsrSdioFunction *function)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int err;

    err = fs_sdio_enable(fdev);

    return ConvertSdioToCsrSdioResult(err);
} /* CsrSdioFunctionEnable() */

CsrResult CsrSdioFunctionDisable(CsrSdioFunction *function)
{
    return CSR_RESULT_FAILURE;
} /* CsrSdioFunctionDisable() */

void CsrSdioFunctionActive(CsrSdioFunction *function)
{
} /* CsrSdioFunctionActive() */

void CsrSdioFunctionIdle(CsrSdioFunction *function)
{
} /* CsrSdioFunctionIdle() */

CsrResult CsrSdioBlockSizeSet(CsrSdioFunction *function, CsrUint16 blockSize)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int r = 0;

    /* Module parameter overrides */
    if (sdio_block_size > -1)
    {
        blockSize = sdio_block_size;
    }

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi : Set SDIO function block size to %d\n", blockSize));
    r = fs_sdio_set_block_size(fdev, blockSize);
    if (r)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : Error %d setting block size\n", r));
    }

    function->blockSize = fdev->max_blocksize;

    return ConvertSdioToCsrSdioResult(r);
} /* CsrSdioBlockSizeSet() */

CsrResult CsrSdioPowerOn(CsrSdioFunction *function)
{
    return CSR_RESULT_SUCCESS;
} /* CsrSdioPowerOn() */

void CsrSdioPowerOff(CsrSdioFunction *function)
{
} /* CsrSdioPowerOff() */

CsrResult CsrSdioHardReset(CsrSdioFunction *function)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int r;

    /* Hard reset can be disabled by a module parameter */
    r = 1;
    if (disable_hw_reset != 1)
    {
        r = fs_sdio_hard_reset(fdev); /* may return 1 if can't reset */
        if (r < 0)
        {
            return ConvertSdioToCsrSdioResult(r);   /* fatal error */
        }
    }

    if (r == 1)
    {
        return CSR_SDIO_RESULT_NOT_RESET;
    }

    return ConvertSdioToCsrSdioResult(r);
} /* CsrSdioHardReset() */

CsrResult CsrSdioMaxBusClockFrequencySet(CsrSdioFunction *function, CsrUint32 maxFrequency)
{
    struct sdio_dev *fdev = (struct sdio_dev *) function->priv;
    int r;
    CsrUint32 max_khz = maxFrequency / 1000;

    /* Respect the max set by the module parameter. */
    if (!max_khz || (max_khz > sdio_clock))
    {
        max_khz = sdio_clock;
    }

    r = fs_sdio_set_max_clock_speed(fdev, max_khz);
    if (r < 0)
    {
        return ConvertSdioToCsrSdioResult(r);
    }

    return CSR_RESULT_SUCCESS;
} /* CsrSdioMaxBusClockFrequencySet() */

#ifdef CONFIG_PM
static void uf_glue_sdio_suspend(struct sdio_dev *fdev, pm_message_t state)
{
    CsrSdioFunction *sdio_ctx = (CsrSdioFunction *) fdev->drv_data;

    func_enter();

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi : System Suspend...\n"));

    /* Clean up the SDIO function driver */
    if (sdio_func_drv && sdio_func_drv->suspend)
    {
        sdio_func_drv->suspend(sdio_ctx);
    }

    func_exit();
} /* uf_glue_sdio_suspend */

static void uf_glue_sdio_resume(struct sdio_dev *fdev)
{
    CsrSdioFunction *sdio_ctx = (CsrSdioFunction *) fdev->drv_data;

    func_enter();

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi : System Resume...\n"));

    /* Clean up the SDIO function driver */
    if (sdio_func_drv && sdio_func_drv->resume)
    {
        sdio_func_drv->resume(sdio_ctx);
    }

    func_exit();
} /* uf_glue_sdio_resume */

#else
#define uf_glue_sdio_suspend  NULL
#define uf_glue_sdio_resume   NULL
#endif

static void uf_glue_sdio_int_handler(struct sdio_dev *fdev)
{
    CsrSdioInterruptDsrCallback func_dsr_callback;
    CsrSdioFunction *sdio_ctx;

    sdio_ctx = (CsrSdioFunction *) fdev->drv_data;

    /* If the function driver has registered a handler, call it */
    if (sdio_func_drv && sdio_func_drv->intr)
    {
        func_dsr_callback = sdio_func_drv->intr(sdio_ctx);

        /* If interrupt handle returns a DSR handle, call it */
        if (func_dsr_callback)
        {
            func_dsr_callback(sdio_ctx);
        }
    }
} /* uf_glue_sdio_int_handler() */

static int uf_glue_sdio_probe(struct sdio_dev *fdev)
{
    int instance = 0;
    CsrSdioFunction *sdio_ctx;

    func_enter();

    /* Assumes one card per host, which is true for SDIO */
    printk("UniFi card 0x%X inserted\n", instance);

    /* Allocate context */
    sdio_ctx = (CsrSdioFunction *) kmalloc(sizeof(CsrSdioFunction),
                                           GFP_KERNEL);
    if (sdio_ctx == NULL)
    {
        return -ENOMEM;
    }

    /* Initialise the context */
    sdio_ctx->sdioId.manfId = fdev->vendor_id;
    sdio_ctx->sdioId.cardId = fdev->device_id;
    sdio_ctx->sdioId.sdioFunction = 1;
    sdio_ctx->sdioId.sdioInterface = 0;
    sdio_ctx->blockSize = fdev->max_blocksize;
    sdio_ctx->priv = (void *) fdev;
    sdio_ctx->features = 0;

    /* Module parameter enables byte mode */
    if (sdio_byte_mode)
    {
        sdio_ctx->features |= CSR_SDIO_FEATURE_BYTE_MODE;
    }

    /* Pass context to the SDIO driver */
    fdev->drv_data = sdio_ctx;

    /* Register this device with the SDIO function driver */
    /* Call the main UniFi driver inserted handler */
    if (sdio_func_drv && sdio_func_drv->inserted)
    {
        uf_add_os_device(instance, NULL);
        sdio_func_drv->inserted(sdio_ctx);
    }

    func_exit();
    return 0;
} /* uf_glue_sdio_probe() */

static void uf_glue_sdio_remove(struct sdio_dev *fdev)
{
    CsrSdioFunction *sdio_ctx = (CsrSdioFunction *) fdev->drv_data;
    func_enter();

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF,
                        "unifi : UniFi card removed\n"));

    /* Clean up the SDIO function driver */
    if (sdio_func_drv && sdio_func_drv->removed)
    {
        uf_remove_os_device(0);
        sdio_func_drv->removed(sdio_ctx);
    }

    kfree(sdio_ctx);

    func_exit();
} /* uf_glue_sdio_remove */

static struct fs_driver fs_driver =
{
    .name = "fs_driver",
    .probe = uf_glue_sdio_probe,
    .remove = uf_glue_sdio_remove,
    .card_int_handler = uf_glue_sdio_int_handler,
    .suspend = uf_glue_sdio_suspend,
    .resume = uf_glue_sdio_resume,
};


CsrResult CsrSdioFunctionDriverRegister(CsrSdioFunctionDriver *sdio_drv)
{
    int r;

    printk("Unifi: Using MMC/SDIO glue driver\n");

    if (sdio_func_drv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : sdio_mmc_fs: UniFi driver already registered\n"));
        return CSR_SDIO_RESULT_INVALID_VALUE;
    }

    /* Save the registered driver description */
    /*
     * Alternatively could have a table here to handle a call to register for just one function.
     * mmc only allows us to register for the whole device
     */
    sdio_func_drv = sdio_drv;

    /* Register ourself with mmc_core */
    r = fs_sdio_register_driver(&fs_driver);
    if (r)
    {
        printk(KERN_ERR "unifi_sdio: Failed to register UniFi SDIO driver: %d\n", r);
        return ConvertSdioToCsrSdioResult(r);
    }

    return CSR_RESULT_SUCCESS;
} /* CsrSdioFunctionDriverRegister() */

void CsrSdioFunctionDriverUnregister(CsrSdioFunctionDriver *sdio_drv)
{
    printk(KERN_INFO "UniFi: unregister from sdio\n");

    fs_sdio_unregister_driver(&fs_driver);

    sdio_func_drv = NULL;
} /* CsrSdioFunctionDriverUnregister() */
