/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"

#include <linux/kmod.h>
#include <linux/init.h>
#include "csr_wifi_hip_unifi.h"
#include "os_linux_priv.h"

#include "sdioemb/sdio_api.h"

/*
 * These glue functions are required because the os_linux layer doesn't
 * directly call CsrSdioInterrupt*(). It could.
 * (However this does allow us to print the SDIO driver that is installed!)
 */
int csr_sdio_linux_remove_irq(CsrSdioFunction *function)
{
    CsrResult csrResult = CsrSdioInterruptDisable(function);

    return csrResult != CSR_RESULT_SUCCESS;
}

int csr_sdio_linux_install_irq(CsrSdioFunction *function)
{
    CsrResult csrResult;

    printk("Unifi: Using CSR SDIO API on CSR embedded SDIO driver\n");

    csrResult = CsrSdioInterruptEnable(function);

    return csrResult != CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipCheckInterrupt
 *
 *      Check for an interrupt on the Wi-Fi SDIO function.
 *
 *      This implemented in the glue layer to allow optimisation for the
 *      platform - some implementations (e.g. Linux SDIO/MMC) will have already
 *      checked the source by the time the Wi-Fi function driver's
 *      interrupt handler function has been called.
 *
 *  Arguments:
 *      function            SDIO context pointer
 *      pending             Location into which 1 will be written if an interrupt
 *                          was pending, 0 if not.
 *
 *  Returns:
 *      Zero on success or a UniFi driver error code.
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipCheckInterrupt(CsrSdioFunction *function, CsrUint8 *pending)
{
    CsrUint8 v;
    CsrResult r;

    /* Note - It may be possible to eliminate the interrupt status r/w here.
     * This may vary depending upon the underlying driver being sdioemb or MMC,
     * in which case a build config option would be needed
     */

    r = CsrSdioF0Read8(function, 0x5 /*SDIO_INT_PENDING*/, &v);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipCheckInterrupt: CsrSdioF0Read8(..., SDIO_INT_PENDING,..) failed with %d\n", r));
        return r;
    }

    *pending = (v & (1 << function->sdioId.sdioFunction)) ? TRUE : FALSE;

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipAddOsDevice
 *  CsrWifiHipRemoveOsDevice
 *
 *      Register this CSR SDIO function context as an "OS device".
 *      The CSR SDIO API provides no means of identifying an instance or
 *      bus ID, so we use the CsrSdioFunction pointer as a key.
 *
 *  Arguments:
 *      function            SDIO context pointer
 *
 *  Returns:
 *      None
 * ---------------------------------------------------------------------------
 */
static CsrSdioFunction *csr_sdio_emb_inst[MAX_UNIFI_DEVS];

void CsrWifiHipAddOsDevice(CsrSdioFunction *function)
{
    int i;

    for (i = 0; i < MAX_UNIFI_DEVS; i++)
    {
        if (csr_sdio_emb_inst[i] == NULL)
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                "unifi : add OS instance %d : %p\n", i, function->device));
            uf_add_os_device(i, NULL);
            csr_sdio_emb_inst[i] = function;
            return;
        }
    }

    CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                        "unifi: No more card instances free\n"));
}

void CsrWifiHipRemoveOsDevice(CsrSdioFunction *function)
{
    int i;

    for (i = 0; i < MAX_UNIFI_DEVS; i++)
    {
        if (csr_sdio_emb_inst[i] == function)
        {
            uf_remove_os_device(i);
            return;
        }
    }

    CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                        "unifi: Card instance not found\n"));
}
