/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"

#include <linux/vmalloc.h>
#include <linux/firmware.h>

#include "os_linux_priv.h"
#include "csr_wifi_hip.h"


int unifi_putest_cmd52_read(os_linux_priv_t *priv, unsigned char *arg)
{
    struct unifi_putest_cmd52 cmd52_params;
    CsrUint8 *arg_pos;
    unsigned int cmd_param_size;
    int r;
    CsrResult csrResult;
    unsigned char ret_buffer[32];
    CsrUint8 *ret_buffer_pos;
    CsrUint8 retries;

    arg_pos = (CsrUint8 *) (((unifi_putest_command_t *) arg) + 1);
    if (get_user(cmd_param_size, (int *) arg_pos))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_cmd52_read: Failed to get the argument\n",
                               priv->instance));
        return -EFAULT;
    }

    if (cmd_param_size != sizeof(struct unifi_putest_cmd52))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_cmd52_read: cmd52 struct mismatch\n",
                               priv->instance));
        return -EINVAL;
    }

    arg_pos += sizeof(unsigned int);
    if (copy_from_user(&cmd52_params,
                       (void *) arg_pos,
                       sizeof(struct unifi_putest_cmd52)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_cmd52_read: Failed to get the cmd52 params\n",
                               priv->instance));
        return -EFAULT;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: cmd52r: func=%d addr=0x%x ", priv->instance,
                       cmd52_params.funcnum, cmd52_params.addr));

    retries = 3;
    do
    {
        csrResult = CsrWifiHipRawSdioByteRead(cmd52_params.funcnum, cmd52_params.addr, &cmd52_params.data);
    } while (--retries && ((csrResult == CSR_SDIO_RESULT_CRC_ERROR) || (csrResult == CSR_SDIO_RESULT_TIMEOUT)));

    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: \nunifi_putest_cmd52_read: CsrWifiHipRawSdioByteRead() failed (csrResult=0x%x)\n",
                               priv->instance, csrResult));
        return -EFAULT;
    }
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: data=%d\n", priv->instance, cmd52_params.data));

    /* Copy the info to the out buffer */
    *(unifi_putest_command_t *) ret_buffer = UNIFI_PUTEST_CMD52_READ;
    ret_buffer_pos = (CsrUint8 *) (((unifi_putest_command_t *) ret_buffer) + 1);
    *(unsigned int *) ret_buffer_pos = sizeof(struct unifi_putest_cmd52);
    ret_buffer_pos += sizeof(unsigned int);
    memcpy(ret_buffer_pos, &cmd52_params, sizeof(struct unifi_putest_cmd52));
    ret_buffer_pos += sizeof(struct unifi_putest_cmd52);

    r = copy_to_user((void *) arg,
                     ret_buffer,
                     ret_buffer_pos - ret_buffer);
    if (r)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_cmd52_read: Failed to return the data\n", priv->instance));
        return -EFAULT;
    }

    return 0;
}

int unifi_putest_cmd52_write(os_linux_priv_t *priv, unsigned char *arg)
{
    struct unifi_putest_cmd52 cmd52_params;
    CsrUint8 *arg_pos;
    unsigned int cmd_param_size;
    CsrResult csrResult;
    CsrUint8 retries;

    arg_pos = (CsrUint8 *) (((unifi_putest_command_t *) arg) + 1);
    if (get_user(cmd_param_size, (int *) arg_pos))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_cmd52_write: Failed to get the argument\n", priv->instance));
        return -EFAULT;
    }

    if (cmd_param_size != sizeof(struct unifi_putest_cmd52))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_cmd52_write: cmd52 struct mismatch\n", priv->instance));
        return -EINVAL;
    }

    arg_pos += sizeof(unsigned int);
    if (copy_from_user(&cmd52_params,
                       (void *) (arg_pos),
                       sizeof(struct unifi_putest_cmd52)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_cmd52_write: Failed to get the cmd52 params\n", priv->instance));
        return -EFAULT;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: cmd52w: func=%d addr=0x%x data=%d\n", priv->instance,
                       cmd52_params.funcnum, cmd52_params.addr, cmd52_params.data));

    retries = 3;
    do
    {
        csrResult = CsrWifiHipRawSdioByteWrite(cmd52_params.funcnum, cmd52_params.addr, cmd52_params.data);
    } while (--retries && ((csrResult == CSR_SDIO_RESULT_CRC_ERROR) || (csrResult == CSR_SDIO_RESULT_TIMEOUT)));

    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_cmd52_write: CsrWifiHipRawSdioByteWrite() failed (csrResult=0x%x)\n",
                               priv->instance, csrResult));
        return -EFAULT;
    }

    return 0;
}

int unifi_putest_gp_read16(os_linux_priv_t *priv, unsigned char *arg)
{
    struct unifi_putest_gp_rw16 gp_r16_params;
    CsrUint8 *arg_pos;
    unsigned int cmd_param_size;
    int r;
    CsrResult csrResult;
    unsigned char ret_buffer[32];
    CsrUint8 *ret_buffer_pos;

    arg_pos = (CsrUint8 *) (((unifi_putest_command_t *) arg) + 1);
    if (get_user(cmd_param_size, (int *) arg_pos))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_gp_read16: Failed to get the argument\n", priv->instance));
        return -EFAULT;
    }

    if (cmd_param_size != sizeof(struct unifi_putest_gp_rw16))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_gp_read16: struct mismatch\n", priv->instance));
        return -EINVAL;
    }

    arg_pos += sizeof(unsigned int);
    if (copy_from_user(&gp_r16_params,
                       (void *) arg_pos,
                       sizeof(struct unifi_putest_gp_rw16)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_gp_read16: Failed to get the params\n", priv->instance));
        return -EFAULT;
    }
    csrResult = CsrWifiHipRawSdioGpRead16(1, gp_r16_params.addr, &gp_r16_params.data);
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_gp_read16: unifi_card_read16() GP=0x%x failed (csrResult=0x%x)\n",
                               priv->instance, gp_r16_params.addr, csrResult));
        return -EFAULT;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: gp_r16: GP=0x%08x, data=0x%04x\n",
                       priv->instance, gp_r16_params.addr, gp_r16_params.data));

    /* Copy the info to the out buffer */
    *(unifi_putest_command_t *) ret_buffer = UNIFI_PUTEST_GP_READ16;
    ret_buffer_pos = (CsrUint8 *) (((unifi_putest_command_t *) ret_buffer) + 1);
    *(unsigned int *) ret_buffer_pos = sizeof(struct unifi_putest_gp_rw16);
    ret_buffer_pos += sizeof(unsigned int);
    memcpy(ret_buffer_pos, &gp_r16_params, sizeof(struct unifi_putest_gp_rw16));
    ret_buffer_pos += sizeof(struct unifi_putest_gp_rw16);

    r = copy_to_user((void *) arg,
                     ret_buffer,
                     ret_buffer_pos - ret_buffer);
    if (r)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_gp_read16: Failed to return the data\n", priv->instance));
        return -EFAULT;
    }

    return 0;
}

int unifi_putest_gp_write16(os_linux_priv_t *priv, unsigned char *arg)
{
    struct unifi_putest_gp_rw16 gp_w16_params;
    CsrUint8 *arg_pos;
    unsigned int cmd_param_size;
    CsrResult csrResult;

    arg_pos = (CsrUint8 *) (((unifi_putest_command_t *) arg) + 1);
    if (get_user(cmd_param_size, (int *) arg_pos))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_gp_write16: Failed to get the argument\n", priv->instance));
        return -EFAULT;
    }

    if (cmd_param_size != sizeof(struct unifi_putest_gp_rw16))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_gp_write16: struct mismatch\n", priv->instance));
        return -EINVAL;
    }

    arg_pos += sizeof(unsigned int);
    if (copy_from_user(&gp_w16_params,
                       (void *) (arg_pos),
                       sizeof(struct unifi_putest_gp_rw16)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_gp_write16: Failed to get the params\n", priv->instance));
        return -EFAULT;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: gp_w16: GP=0x%08x, data=0x%04x\n",
                       priv->instance, gp_w16_params.addr, gp_w16_params.data));

    csrResult = CsrWifiHipRawSdioGpWrite16(1, gp_w16_params.addr, gp_w16_params.data);
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_gp_write16: CsrWifiHipRawSdioGpRead16() GP=%x failed (csrResult=0x%x)\n",
                               priv->instance, gp_w16_params.addr, csrResult));
        return -EFAULT;
    }

    return 0;
}

int unifi_putest_set_sdio_clock(os_linux_priv_t *priv, unsigned char *arg)
{
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: unifi_putest_set_sdio_clock: Trying to set sdio clock but this should not be used anymore. Doing nothing.\n",
                           priv->instance));

    return -EFAULT;
}

int unifi_putest_start(os_linux_priv_t *priv, unsigned char *arg)
{
    int r;
    CsrResult csrResult;
    int already_in_test = priv->ptest_mode;
    CsrWifiHipInstanceConfig config;

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: unifi_putest_start", priv->instance));

    priv->ptest_mode = 1;

    if (already_in_test)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                           "unifi%d: unifi_putest_start: Already in test - perform reset\n", priv->instance));

        csrResult = CsrWifiHipRawSdioReset();
        if (csrResult != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: unifi_putest_start: Failed to reset SDIO\n", priv->instance));
            return CsrHipResultToStatus(csrResult);
        }
    }
    else
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                           "unifi%d: unifi_putest_start: First time - perform WifiOn procedure\n", priv->instance));

        if (priv->sme_cli)
        {
            /* Send a fake cold suspend ind to SME.  SME will respond with ctrl router wifi off */
            r = sme_sys_suspend(priv);
            if (r)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_putest_start: failed to suspend UniFi", priv->instance));
                return r;
            }
        }

        /* Set up the config struct */
        config.hipMode = CSR_WIFI_HIP_RAW_SDIO_MODE;
        config.allocFunc = os_linux_net_data_malloc;
        config.freeFunc = os_linux_net_data_free;
        config.flowControlPauseCb = os_linux_flow_control_pause_cb;
        config.flowControlResumeCb = os_linux_flow_control_resume_cb;
        if ((priv->instance >= 0) && (priv->instance < MAX_UNIFI_DEVS))
        {
            config.fw_init = fw_init[priv->instance];
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: unifi_putest_start: The instance number in os_linux priv is invalid (%u). Defaulting to instance 0\n",
                                   priv->instance, priv->instance));
            config.fw_init = fw_init[0];
        }
        config.cardParams = config_params;
        config.cmanrTestMode = FALSE;

        csrResult = CsrWifiHipWifiOnReq(priv->hip_handle, priv, priv->sme_synergy_sched_queue, config);
        if (csrResult != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: unifi_putest_start: failed to init UniFi\n", priv->instance));
            return CsrHipResultToStatus(csrResult);
        }
    }

    return 0;
}

int unifi_putest_stop(os_linux_priv_t *priv, unsigned char *arg)
{
    int r = 0;

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                       "unifi%d: unifi_putest_stop\n", priv->instance));

    /* The CsrWifiHipStopCoreDumpReq function tries to start the XAPs,
       which is harmless, so we'll always try to do that - regardless
       of mode. */
    CsrWifiHipStopCoreDumpReq(priv->hip_handle);

    /* PUTEST_STOP is also used to resume the XAPs after SME coredump.
     * Don't power off the chip, leave that to the normal wifi-off which is
     * about to carry on. No need to resume the SME either, as it wasn't suspended.
     */
    if (priv->coredump_mode)
    {
        priv->coredump_mode = 0;
        return 0;
    }

    /* Terminate HAL - it is in Raw SDIO mode */
    CsrWifiHipWifiOffReq(TRUE, priv->hip_handle);

    /* Resume the SME and UniFi */
    if (priv->sme_cli)
    {
        r = sme_sys_resume(priv);
        if (r)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: unifi_putest_stop: failed to resume SME\n", priv->instance));
        }
    }

    priv->ptest_mode = 0;
    return r;
}

int unifi_putest_dl_fw(os_linux_priv_t *priv, unsigned char *arg)
{
#define UF_PUTEST_MAX_FW_FILE_NAME      16
#define UNIFI_MAX_FW_PATH_LEN           32
    unsigned int fw_name_length;
    unsigned char fw_name[UF_PUTEST_MAX_FW_FILE_NAME + 1];
    unsigned char *name_buffer;
    int postfix;
    char fw_path[UNIFI_MAX_FW_PATH_LEN];
    const struct firmware *fw_entry;
    int r;
    CsrResult csrResult;

    /* Get the f/w file name length */
    if (get_user(fw_name_length, (unsigned int *) (((unifi_putest_command_t *) arg) + 1)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_dl_fw: Failed to get the length argument\n",
                               priv ? priv->instance : 0));
        return -EFAULT;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: unifi_putest_dl_fw: file name size = %d\n",
                       priv ? priv->instance : 0, fw_name_length));

    /* Sanity check for the f/w file name length */
    if (fw_name_length > UF_PUTEST_MAX_FW_FILE_NAME)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_dl_fw: F/W file name is too long\n",
                               priv ? priv->instance : 0));
        return -EINVAL;
    }

    /* Get the f/w file name */
    name_buffer = ((unsigned char *) arg) + sizeof(unifi_putest_command_t) + sizeof(unsigned int);
    if (copy_from_user(fw_name, (void *) name_buffer, fw_name_length))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_dl_fw: Failed to get the file name\n",
                               priv ? priv->instance : 0));
        return -EFAULT;
    }
    fw_name[fw_name_length] = '\0';
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: unifi_putest_dl_fw: file = %s\n",
                       priv->instance, fw_name));

    /* Get the putest f/w */
    postfix = priv->instance;
    scnprintf(fw_path, UNIFI_MAX_FW_PATH_LEN, "unifi-sdio-%d/%s",
              postfix, fw_name);
    r = request_firmware(&fw_entry, fw_path, priv->unifi_device);
    if (r != 0)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: Firmware file not available\n", priv->instance));
        return -EINVAL;
    }

    csrResult = CsrWifiHipRawSdioFirmwareDownload(fw_entry->size, fw_entry->data);
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_putest_dl_fw: failed to download the f/w\n", priv->instance));
    }

    release_firmware((const struct firmware *) fw_entry);

    return CsrHipResultToStatus(csrResult);
}

int unifi_putest_dl_fw_buff(os_linux_priv_t *priv, unsigned char *arg)
{
    unsigned int fw_length;
    unsigned char *fw_buf = NULL;
    unsigned char *fw_user_ptr;
    CsrResult csrResult;

    /* Get the f/w buffer length */
    if (get_user(fw_length, (unsigned int *) (((unifi_putest_command_t *) arg) + 1)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_dl_fw_buff: Failed to get the length arg\n", priv ? priv->instance : 0));
        return -EFAULT;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: unifi_putest_dl_fw_buff: size = %d\n",
                       priv ? priv->instance : 0, fw_length));

    /* Sanity check for the buffer length */
    if ((fw_length == 0) || (fw_length > 0xfffffff))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_dl_fw_buff: buffer length bad %u\n",
                               priv ? priv->instance : 0, fw_length));
        return -EINVAL;
    }

    /* Buffer for kernel copy of the f/w image */
    fw_buf = CsrPmemAlloc(fw_length);
    if (!fw_buf)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_dl_fw_buff: malloc fail\n", priv ? priv->instance : 0));
        return -ENOMEM;
    }

    /* Get the f/w image */
    fw_user_ptr = ((unsigned char *) arg) + sizeof(unifi_putest_command_t) + sizeof(unsigned int);
    if (copy_from_user(fw_buf, (void *) fw_user_ptr, fw_length))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unifi_putest_dl_fw_buff: Failed to get the buffer\n", priv ? priv->instance : 0));
        CsrPmemFree(fw_buf);
        return -EFAULT;
    }

    csrResult = CsrWifiHipRawSdioFirmwareDownload(fw_length, fw_buf);
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipRawSdioFirmwareDownload: failed\n", priv ? priv->instance : 0));
    }

    CsrPmemFree(fw_buf);

    return CsrHipResultToStatus(csrResult);
}

int unifi_putest_coredump_prepare(os_linux_priv_t *priv, unsigned char *arg)
{
    CsrResult r;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                        "unifi%d: Preparing for SDIO coredump\n", priv->instance));
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE)
    unifi_debug_buf_dump();
#endif

    /* Set coredump mode flag so that deinit won't power off the chip */
    priv->coredump_mode = 1;

    r = CsrWifiHipStartCoreDumpReq(priv->hip_handle);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipStartCoreDumpReq failed\n", priv->instance));
    }

    r = CsrWifiHipRawSdioCoreDumpPrepare(FALSE);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipRawSdioCoreDumpPrepare failed\n", priv->instance));
    }

    return 0;
}

int unifi_putest_cmd52_block_read(os_linux_priv_t *priv, unsigned char *arg)
{
    struct unifi_putest_block_cmd52_r block_cmd52;
    CsrUint8 *arg_pos;
    unsigned int cmd_param_size;
    CsrResult r;
    CsrUint8 *block_local_buffer;

    arg_pos = (CsrUint8 *) (((unifi_putest_command_t *) arg) + 1);
    if (get_user(cmd_param_size, (int *) arg_pos))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: cmd52r_block: Failed to get the argument\n", priv->instance));
        return -EFAULT;
    }

    if (cmd_param_size != sizeof(struct unifi_putest_block_cmd52_r))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: cmd52r_block: cmd52 struct mismatch\n", priv->instance));
        return -EINVAL;
    }

    arg_pos += sizeof(unsigned int);
    if (copy_from_user(&block_cmd52,
                       (void *) arg_pos,
                       sizeof(struct unifi_putest_block_cmd52_r)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: cmd52r_block: Failed to get the cmd52 params\n", priv->instance));
        return -EFAULT;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: cmd52r_block: func=%d addr=0x%x len=0x%x ",
                       priv->instance, block_cmd52.funcnum, block_cmd52.addr, block_cmd52.length));

    block_local_buffer = vmalloc(block_cmd52.length);
    if (block_local_buffer == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: cmd52r_block: Failed to allocate buffer\n", priv->instance));
        return -ENOMEM;
    }
    r = CsrWifiHipRawSdioByteBlockRead(1, block_cmd52.addr, block_local_buffer, block_cmd52.length);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: cmd52r_block: CsrWifiHipRawSdioByteBlockRead failed\n", priv->instance));
        return -EIO;
    }

    if (copy_to_user((void *) block_cmd52.data,
                     block_local_buffer,
                     block_cmd52.length))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: cmd52r_block: Failed to return the data\n", priv->instance));
        return -EFAULT;
    }

    return 0;
}
