/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip.h"
#include "csr_result.h"
#include "os_linux_priv.h"

#include <linux/init.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/ctype.h>
#include <asm/uaccess.h>
#include <linux/jiffies.h>
#include <linux/netdevice.h>
#include <net/sch_generic.h>
#include <asm/byteorder.h>
#include <linux/workqueue.h>
#include <linux/wait.h>

#include "os_linux_priv.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip_chiphelper.h"

#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
#include <hydra/service.h>
#include <hydra/service_driver.h>
#include <hydra/sdio.h>
#endif

/* Module parameter variables */
int buswidth = 0;               /* 0 means use default, values 1,4 */
int sdio_clock = 50000;         /* kHz */
int unifi_debug = 0;
#ifdef CSR_LOG_ENABLE
static char *trace_levels = "";
#endif
/* fw_init prevents f/w initialisation on error. */
/*ALSI_DL 20140206*/
/*int fw_init[MAX_UNIFI_DEVS] = {-1, -1};*/
int fw_init[MAX_UNIFI_DEVS] = {-1, -1, -1};
int no_helper = 0;
int use_5g = 0;
int disable_hw_reset = 0;
int disable_power_control = 0;
int sdio_block_size = -1;      /* Override SDIO block size */
int sdio_byte_mode = 0;        /* 0 for block mode + padding, 1 for byte mode */
int run_bh_once = -1;          /* Set for scheduled interrupt mode, -1 = default */
int default_vif = 1;
#ifdef CSR_NATIVE_LINUX
int zero_copy_mode = 0; /* copy by default */
int mparam_slot_count[CSR_WIFI_HIP_PARAM_SLOT_COUNT] = {64, 120};
int poll_period = 1;
int tx_force_pull_mode = 0;
int tx_window_segment_size = 0;
#endif
CsrWifiHipCardParams config_params;

MODULE_DESCRIPTION("CSR UniFi (SDIO)");

module_param(buswidth,    int, S_IRUGO | S_IWUSR);
module_param(sdio_clock,  int, S_IRUGO | S_IWUSR);
module_param(unifi_debug, int, S_IRUGO | S_IWUSR);
#ifdef UNIFI_DEBUG
#ifdef CSR_LOG_ENABLE
module_param(trace_levels, charp, S_IRUGO);
#endif
#endif
module_param_array(fw_init, int, NULL, S_IRUGO | S_IWUSR);
module_param(no_helper,   int, S_IRUGO | S_IWUSR);
module_param(use_5g,      int, S_IRUGO | S_IWUSR);
module_param(disable_hw_reset,  int, S_IRUGO | S_IWUSR);
module_param(disable_power_control,  int, S_IRUGO | S_IWUSR);
module_param(sdio_block_size, int, S_IRUGO | S_IWUSR);
module_param(sdio_byte_mode, int, S_IRUGO | S_IWUSR);
module_param(run_bh_once, int, S_IRUGO | S_IWUSR);
#ifdef CSR_NATIVE_LINUX
module_param(zero_copy_mode, int, S_IRUGO | S_IWUSR);
module_param_array(mparam_slot_count, int, NULL, S_IRUGO | S_IWUSR);
module_param(poll_period, int, S_IRUGO | S_IWUSR);
module_param(tx_force_pull_mode, int, S_IRUGO | S_IWUSR);
module_param(tx_window_segment_size, int, S_IRUGO | S_IWUSR);
#endif

MODULE_PARM_DESC(buswidth, "SDIO bus width (0=default), set 1 for 1-bit or 4 for 4-bit mode");
MODULE_PARM_DESC(sdio_clock, "SDIO bus frequency in kHz, (default = 50 MHz)");
MODULE_PARM_DESC(unifi_debug, "Diagnostic reporting level");
MODULE_PARM_DESC(fw_init, "Set to 0 to prevent f/w initialization on error");
MODULE_PARM_DESC(no_helper, "Set to 1 to prevent userspace helper from starting");
MODULE_PARM_DESC(use_5g, "Use the 5G (802.11a) radio band");
MODULE_PARM_DESC(disable_hw_reset, "Set to 1 to disable hardware reset");
#ifdef CSR_WIFI_DRIVER_HYDRA
MODULE_PARM_DESC(disable_power_control, "Set to 1 to leave Hydra service enabled on exit");
#else
MODULE_PARM_DESC(disable_power_control, "Set to 1 to disable SDIO power control");
#endif
MODULE_PARM_DESC(sdio_block_size, "Set to override SDIO block size");
MODULE_PARM_DESC(sdio_byte_mode, "Set to 1 for byte mode SDIO");
MODULE_PARM_DESC(run_bh_once, "Run BH only when firmware interrupts");
#ifdef UNIFI_DEBUG
#ifdef CSR_LOG_ENABLE
MODULE_PARM_DESC(trace_levels, "List of trace levels for each origin and sub-origin");
#endif
#endif

#ifdef UNIFI_DEBUG
#ifdef CSR_LOG_ENABLE
CSR_LOG_TEXT_HANDLE_DEFINE(CsrWifiOsLto);
static const CsrCharString *subOrigins[] = CSR_WIFI_OS_SUBORIGINS;
#endif
#endif

/* Callback for event logging to UDI clients */
static void udi_log_event(ul_client_t *client,
                          const u8 *signal, int signal_len,
                          const CsrWifiHipBulkDataParam *bulkdata,
                          int dir);

static void udi_set_log_filter(ul_client_t      *pcli,
                               unifiio_filter_t *udi_filter);


/* Mutex to protect access to  priv->sme_cli */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 37)
DEFINE_SEMAPHORE(udi_mutex);
#else
DECLARE_MUTEX(udi_mutex);
#endif


CsrInt32 CsrHipResultToStatus(CsrResult csrResult)
{
    CsrInt32 r = -EIO;

    switch (csrResult)
    {
        case CSR_RESULT_SUCCESS:
            r = 0;
            break;
        case CSR_WIFI_HIP_RESULT_RANGE:
            r = -ERANGE;
            break;
        case CSR_WIFI_HIP_RESULT_NO_DEVICE:
            r = -ENODEV;
            break;
        case CSR_WIFI_HIP_RESULT_INVALID_VALUE:
            r = -EINVAL;
            break;
        case CSR_WIFI_HIP_RESULT_NOT_FOUND:
            r = -ENOENT;
            break;
        case CSR_WIFI_HIP_RESULT_NO_SPACE:
            r = -ENOSPC;
            break;
        case CSR_WIFI_HIP_RESULT_NO_MEMORY:
            r = -ENOMEM;
            break;
        case CSR_RESULT_FAILURE:
            r = -EIO;
            break;
        default:
            /*unifi_warning(card->ospriv, "CsrHipResultToStatus: Unrecognised csrResult error code: %d\n", csrResult);*/
            r = -EIO;
    }
    return r;
}

#ifdef CSR_LOG_ENABLE
/* This function is only called from within a trace macro. */
static const char *trace_putest_cmdid(unifi_putest_command_t putest_cmd)
{
    switch (putest_cmd)
    {
        case UNIFI_PUTEST_START:
            return "START";
        case UNIFI_PUTEST_STOP:
            return "STOP";
        case UNIFI_PUTEST_SET_SDIO_CLOCK:
            return "SET CLOCK";
        case UNIFI_PUTEST_CMD52_READ:
            return "CMD52R";
        case UNIFI_PUTEST_CMD52_BLOCK_READ:
            return "CMD52BR";
        case UNIFI_PUTEST_CMD52_WRITE:
            return "CMD52W";
        case UNIFI_PUTEST_DL_FW:
            return "D/L FW";
        case UNIFI_PUTEST_DL_FW_BUFF:
            return "D/L FW BUFFER";
        case UNIFI_PUTEST_COREDUMP_PREPARE:
            return "PREPARE COREDUMP";
        case UNIFI_PUTEST_GP_READ16:
            return "GP16R";
        case UNIFI_PUTEST_GP_WRITE16:
            return "GP16W";
        default:
            return "ERROR: unrecognised command";
    }
}

#endif /* CSR_LOG_ENABLE */

/*
 * ---------------------------------------------------------------------------
 *  unifi_open
 *  unifi_release
 *
 *      Open and release entry points for the UniFi debug driver.
 *
 *  Arguments:
 *      Normal linux driver args.
 *
 *  Returns:
 *      Linux error code.
 * ---------------------------------------------------------------------------
 */
static int unifi_open(struct inode *inode, struct file *file)
{
    int devno;
    os_linux_priv_t *priv;
    ul_client_t *udi_cli;

    func_enter();

    devno = MINOR(inode->i_rdev) >> 1;

    /*
     * Increase the ref_count for the char device clients.
     * Make sure you call uf_put_instance() to decreace it if
     * unifi_open returns an error.
     */
    priv = uf_get_instance(devno);
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : unifi_open: No device present\n"));
        func_exit();
        return -ENODEV;
    }

    /* Register this instance in the client's list. */
    /* The minor number determines the nature of the client (Unicli or SME). */
    if (MINOR(inode->i_rdev) & 0x1)
    {
        udi_cli = ul_register_client(priv, CLI_USING_WIRE_FORMAT, udi_log_event);
        if (udi_cli == NULL)
        {
            /* Too many clients already using this device */
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Too many clients already open\n", priv ? priv->instance : 0));
            uf_put_instance(devno);
            func_exit();
            return -ENOSPC;
        }
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Client is registered to /dev/unifiudi%d\n",
                              priv ? priv->instance : 0,  devno));
    }
    else
    {
        /*
         * Even-numbered device nodes are the control application.
         * This is the userspace helper containing SME or
         * unifi_manager.
         */

        down(&udi_mutex);

#ifdef CSR_SME_USERSPACE
        /* Check if a config client is already attached */
        if (priv->sme_cli)
        {
            up(&udi_mutex);
            uf_put_instance(devno);

            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: There is already a configuration client using the character device\n", priv ? priv->instance : 0));
            func_exit();
            return -EBUSY;
        }
#endif /* CSR_SME_USERSPACE */

#ifdef CSR_SUPPORT_SME
        /* The callback is not used in the SME set up - handling of received signals
           is done in unifi_event.c in the handler functions for the signals. */
        udi_cli = ul_register_client(priv,
                                     CLI_USING_WIRE_FORMAT | CLI_SME_USERSPACE,
                                     NULL);
#else
        /* Config client for native driver */
        udi_cli = ul_register_client(priv,
                                     0,
                                     sme_native_log_event);
#endif
        if (udi_cli == NULL)
        {
            /* Too many clients already using this device */
            up(&udi_mutex);
            uf_put_instance(devno);

            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Too many clients already open\n", priv ? priv->instance : 0));
            func_exit();
            return -ENOSPC;
        }

        /*
         * Fill-in the pointer to the configuration client.
         * This is the SME userspace helper or unifi_manager.
         * Not used in the SME embedded version.
         */
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: SME client (id:%d s:0x%X) is registered\n",
                              priv ? priv->instance : 0,
                              udi_cli->client_id, udi_cli->sender_id));
        /* Store the SME UniFi Linux Client */
        if (priv->sme_cli == NULL)
        {
            priv->sme_cli = udi_cli;
        }

        up(&udi_mutex);
    }


    /*
     * Store the pointer to the client.
     * All char driver's entry points will pass this pointer.
     */
    file->private_data = udi_cli;

    func_exit();
    return 0;
} /* unifi_open() */

static int unifi_release(struct inode *inode, struct file *filp)
{
    ul_client_t *udi_cli = (void *) filp->private_data;
    int devno;
    os_linux_priv_t *priv;

    func_enter();

    priv = uf_find_instance(udi_cli->instance);
    if (!priv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_close: instance for device not found\n", priv ? priv->instance : 0));
        return -ENODEV;
    }

    devno = MINOR(inode->i_rdev) >> 1;

    /* Even device nodes are the config client (i.e. SME or unifi_manager) */
    if ((MINOR(inode->i_rdev) & 0x1) == 0)
    {
        if (priv->sme_cli != udi_cli)
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Surprise closing config device: not the sme client\n", priv ? priv->instance : 0));
        }
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: SME client close (unifi%d)\n",
                           priv ? priv->instance : 0,  devno));

        /*
         * Clear sme_cli before calling unifi_sys_... so it doesn't try to
         * queue a reply to the (now gone) SME.
         */
        down(&udi_mutex);
        priv->sme_cli = NULL;
        up(&udi_mutex);

#ifdef CSR_SME_USERSPACE
        /* Power-down when config client closes - if the driver hasn't been de-init'ed already */
        if ((priv->state != OS_LINUX_STATE_IDLE) && (priv->state != OS_LINUX_STATE_ACTIVATED))
        {
            CsrWifiRouterCtrlWifiOffReq req = {{CSR_WIFI_ROUTER_CTRL_HIP_REQ, 0, 0, 0, NULL}};
            CsrWifiRouterCtrlWifiOffReqHandler(priv, &req.common);
        }

        uf_sme_deinit(priv);

        /* It is possible that a blocking SME request was made from another process
         * which did not get read by the SME before the WifiOffReq.
         * So check for a pending request which will go unanswered and cancel
         * the wait for event. As only one blocking request can be in progress at
         * a time, up to one event should be completed.
         */
        uf_sme_cancel_request(priv, 0);
#endif /* CSR_SME_USERSPACE */


#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
        uf_stop_hydra_wlan_service(priv);
#endif
    }
    else
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UDI client close (unifiudi%d)\n",
                           priv ? priv->instance : 0,  devno));

        /* If the pointer matches the logging client, stop logging. */
        down(&priv->udi_logging_mutex);
        if (udi_cli == priv->logging_client)
        {
            CsrResult ret;

            priv->logging_client = NULL;
            ret = CsrWifiHipLogUdiUnregistrationReq(priv->hip_handle);
            if (ret != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : unifi_release: Failed to remove UDI log hook\n"));
            }
        }
        up(&priv->udi_logging_mutex);
    }

    /* Deregister this instance from the client's list. */
    ul_deregister_client(udi_cli);
    uf_put_instance(devno);

    return 0;
} /* unifi_release() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_read
 *
 *      The read() driver entry point.
 *
 *  Arguments:
 *      filp        The file descriptor returned by unifi_open()
 *      p           The user space buffer to copy the read data
 *      len         The size of the p buffer
 *      poff
 *
 *  Returns:
 *      number of bytes read or an error code on failure
 * ---------------------------------------------------------------------------
 */
static ssize_t unifi_read(struct file *filp, char *p, size_t len, loff_t *poff)
{
    ul_client_t *pcli = (void *) filp->private_data;
    os_linux_priv_t *priv;
    udi_log_t *logptr = NULL;
    udi_msg_t *msgptr;
    struct list_head *l;
    int msglen;

    func_enter();

    priv = uf_find_instance(pcli->instance);

    if (!priv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : invalid priv\n"));
        return -ENODEV;
    }

    if (!pcli->udi_enabled)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read: unknown client.", priv ? priv->instance : 0));
        return -EINVAL;
    }

    if (list_empty(&pcli->udi_log))
    {
        if (filp->f_flags & O_NONBLOCK)
        {
            /* Non-blocking - just return if the udi_log is empty */
            return 0;
        }
        else
        {
            /* Blocking - wait on the UDI wait queue */
            if (wait_event_interruptible(pcli->udi_wq,
                                         !list_empty(&pcli->udi_log)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read: wait_event_interruptible failed.", priv ? priv->instance : 0));
                return -ERESTARTSYS;
            }
        }
    }

    /* Read entry from list head and remove it from the list */
    if (down_interruptible(&pcli->udi_sem))
    {
        return -ERESTARTSYS;
    }
    l = pcli->udi_log.next;
    list_del(l);
    up(&pcli->udi_sem);

    /* Get a pointer to whole struct */
    logptr = list_entry(l, udi_log_t, q);
    if (logptr == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read: failed to get event.\n", priv ? priv->instance : 0));
        return -EINVAL;
    }

    /* Get the real message */
    msgptr = &logptr->msg;
    msglen = msgptr->length;
    if (msglen > len)
    {
        printk(KERN_WARNING "truncated read to %d actual msg len is %lu\n", msglen, (long unsigned int) len);
        msglen = len;
    }

    /* and pass it to the client (SME or Unicli). */
    if (copy_to_user(p, msgptr, msglen))
    {
        printk(KERN_ERR "Failed to copy UDI log to user\n");
        kfree(logptr);
        return -EFAULT;
    }

    /* It is our resposibility to free the message buffer. */
    kfree(logptr);

    func_exit_r(msglen);
    return msglen;
} /* unifi_read() */

/*
 * ---------------------------------------------------------------------------
 * udi_send_signal_unpacked
 *
 *      Sends an unpacked signal to UniFi.
 *
 * Arguments:
 *      priv            Pointer to private context struct
 *      data            Pointer to request structure and data to send
 *      data_len        Length of data in data pointer.
 *
 * Returns:
 *      Number of bytes written, error otherwise.
 *
 * Notes:
 *      All clients that use this function to send a signal to the unifi
 *      must use the host formatted structures.
 * ---------------------------------------------------------------------------
 */
static int udi_send_signal_unpacked(netInterface_priv_t *interfacePriv, unsigned char *data, uint data_len)
{
    os_linux_priv_t *priv = interfacePriv->privPtr;
    CSR_SIGNAL *sigptr = (CSR_SIGNAL *) data;
    CSR_DATAREF *datarefptr;
    CsrWifiHipBulkDataParam bulk_data;
    uint signal_size, i;
    uint bulk_data_offset = 0;
    int bytecount;
    CsrResult csrResult;
    CsrUint32 aId = 0;

    /* Number of bytes in the signal */
    signal_size = SigGetSize(sigptr);
    if (!signal_size || (signal_size > data_len))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_sme_mlme_req - Invalid signal 0x%x size should be %d bytes\n",
                               priv ? priv->instance : 0,
                               sigptr->SignalPrimitiveHeader.SignalId,
                               signal_size));
        return -EINVAL;
    }
    bytecount = signal_size;

    /* Get a pointer to the information of the first data reference */
    datarefptr = (CSR_DATAREF *) &sigptr->u;

    /* Initialize the offset in the data buffer, bulk data is right after the signal. */
    bulk_data_offset = signal_size;

    /* store the references and the size of the bulk data to the bulkdata structure */
    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
    {
        /* the length of the bulk data is in the signal */
        if ((datarefptr + i)->DataLength)
        {
            void *dest;

            csrResult = os_linux_net_data_malloc(priv, (CsrWifiHipBulkDataDesc *) &bulk_data.d[i], (datarefptr + i)->DataLength);
            if (csrResult != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: udi_send_signal_unpacked: failed to allocate request_data.\n", priv ? priv->instance : 0));
                return -EIO;
            }

            dest = (void *) bulk_data.d[i].os_data_ptr;
            memcpy(dest, data + bulk_data_offset, bulk_data.d[i].data_length);
        }
        else
        {
            bulk_data.d[i].data_length = 0;
        }

        bytecount += bulk_data.d[i].data_length;
        /* advance the offset, to point the next bulk data */
        bulk_data_offset += bulk_data.d[i].data_length;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, "unifi%d: SME Send: signal 0x%.4X\n",
                        priv ? priv->instance : 0, sigptr->SignalPrimitiveHeader.SignalId));

#ifdef CSR_NATIVE_LINUX
    /* Send the signal. */
    if (interfacePriv->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_AP)
    {
        if (sigptr->SignalPrimitiveHeader.SignalId == CSR_MA_PACKET_REQUEST_ID)
        {
            CsrUint8 toDS, *dMac;
            CsrUint16 frameControl;

            frameControl = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(bulk_data.d[0].os_data_ptr);
            toDS = CSR_WIFI_HIP_IEEE80211_HAS_TO_DS(frameControl);
            dMac = (CsrUint8 *) bulk_data.d[0].os_data_ptr + 4 + (toDS * 12); /* Address 1 or 3 */

            for (i = 0; i < interfacePriv->peer_sta_count; i++)
            {
                if (!memcmp(interfacePriv->peer_sta_info[i].peer_macaddr, dMac, ETH_ALEN))
                {
                    aId = interfacePriv->peer_sta_info[i].association_id;
                    break;
                }
            }
        }
        else
        {
            /* None data frame - use aId 0 */
            aId = 0;
        }
    }
#endif

    csrResult = CsrWifiHipUnpackedSignalReq(priv->hip_handle, interfacePriv->InterfaceTag, aId, signal_size,
                                            (CsrUint8 *) sigptr, (CsrWifiHipBulkDataParam *) &bulk_data);
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: udi_send_signal_unpacked: send failed (%d)\n",
                               priv ? priv->instance : 0, csrResult));
        for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
        {
            if (bulk_data.d[i].data_length != 0)
            {
                os_linux_net_data_free(priv, (CsrWifiHipBulkDataDesc *) &bulk_data.d[i]);
            }
        }
        func_exit();
        return -EIO;
    }

    return bytecount;
} /* udi_send_signal_unpacked() */

/*
 * ---------------------------------------------------------------------------
 * udi_send_signal_raw
 *
 *      Sends a packed signal to UniFi.
 *
 * Arguments:
 *      priv            Pointer to private context struct
 *      buf             Pointer to request structure and data to send
 *      buflen          Length of data in data pointer.
 *
 * Returns:
 *      Number of bytes written, error otherwise.
 *
 * Notes:
 *      All clients that use this function to send a signal to the unifi
 *      must use the wire formatted structures.
 * ---------------------------------------------------------------------------
 */
static int udi_send_signal_raw(os_linux_priv_t *priv, unsigned char *buf, int buflen)
{
    int signal_size;
    int sig_id;
    CsrWifiHipBulkDataParam data_ptrs;
    int i, r;
    unsigned int num_data_refs;
    int bytecount;
    CsrResult csrResult;

    func_enter();

    /*
     * The signal is the first thing in buf, the signal id is the
     * first 16 bits of the signal.
     */
    /* Number of bytes in the signal */
    sig_id = GET_SIGNAL_ID(buf);
    signal_size = buflen;
    signal_size -= GET_PACKED_DATAREF_LEN(buf, 0);
    signal_size -= GET_PACKED_DATAREF_LEN(buf, 1);
    if ((signal_size <= 0) || (signal_size > buflen))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: udi_send_signal_raw - Couldn't find length of signal 0x%x\n",
                               priv ? priv->instance : 0,
                               sig_id));
        func_exit();
        return -EINVAL;
    }
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: udi_send_signal_raw: signal 0x%.4X len:%d\n",
                          priv ? priv->instance : 0,
                          sig_id, signal_size));
    /* Zero the data ref arrays */
    memset(&data_ptrs, 0, sizeof(data_ptrs));

    /*
     * Find the number of associated bulk data packets.  Scan through
     * the data refs to check that we have enough data and pick out
     * pointers to appended bulk data.
     */
    num_data_refs = 0;
    bytecount = signal_size;

    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; ++i)
    {
        unsigned int len = GET_PACKED_DATAREF_LEN(buf, i);
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: udi_send_signal_raw: data_ref length = %d\n",
                               priv ? priv->instance : 0,  len));

        if (len != 0)
        {
            void *dest;

            csrResult = os_linux_net_data_malloc(priv, &data_ptrs.d[i], len);
            if (csrResult != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: udi_send_signal_raw: failed to allocate request_data.\n", priv ? priv->instance : 0));
                return -EIO;
            }

            dest = (void *) data_ptrs.d[i].os_data_ptr;
            memcpy(dest, buf + bytecount, len);

            bytecount += len;
            num_data_refs++;
        }
        data_ptrs.d[i].data_length = len;
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: Queueing signal 0x%.4X from UDI with %u data refs\n",
                           priv ? priv->instance : 0,
                           sig_id,
                           num_data_refs));

    if (bytecount > buflen)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: udi_send_signal_raw: Not enough data (%d instead of %d)\n",
                               priv ? priv->instance : 0,  buflen, bytecount));
        func_exit();
        return -EINVAL;
    }

    /* Send the signal calling the function that uses the wire-formatted signals. */
    r = ul_send_signal_raw(priv, buf, signal_size, (CsrWifiHipBulkDataParam *) &data_ptrs);
    if (r < 0)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: udi_send_signal_raw: send failed (%d)\n",
                               priv ? priv->instance : 0,  r));
        func_exit();
        return -EIO;
    }

#ifdef CSR_NATIVE_LINUX
    if (sig_id == CSR_MLME_POWERMGT_REQUEST_ID)
    {
        CsrUint8 i;
        int any_connected;
        int power_mode = CSR_GET_UINT16_FROM_LITTLE_ENDIAN((buf +
                                                            SIZEOF_SIGNAL_HEADER + (UNIFI_MAX_DATA_REFERENCES * SIZEOF_DATAREF)));
        /* Configure deep sleep signaling */
        any_connected = 0;
        for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
        {
            if (priv->interfacePriv[i]->connected != UnifiNotConnected)
            {
                any_connected = 1;
            }
        }
        if (power_mode || !any_connected)
        {
            csrResult = CsrWifiHipLowPowerModeReq(priv->hip_handle,
                                                  CSR_WIFI_ROUTER_CTRL_LOW_POWER_MODE_ENABLED,
                                                  FALSE);
            if (csrResult != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: udi_send_signal_raw: Failed to set power mode to enable.\n", priv ? priv->instance : 0));
            }
        }
        else
        {
            csrResult = CsrWifiHipLowPowerModeReq(priv->hip_handle,
                                                  CSR_WIFI_ROUTER_CTRL_LOW_POWER_MODE_DISABLED,
                                                  FALSE);
            if (csrResult != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: udi_send_signal_raw: Failed to set power mode to disable.\n", priv ? priv->instance : 0));
            }
        }
    }
#endif

    func_exit_r(bytecount);

    return bytecount;
} /* udi_send_signal_raw */

/*
 * ---------------------------------------------------------------------------
 *  unifi_write
 *
 *      The write() driver entry point.
 *      A UniFi Debug Interface client such as unicli can write a signal
 *      plus bulk data to the driver for sending to the UniFi chip.
 *
 *      Only one signal may be sent per write operation.
 *
 *  Arguments:
 *      filp        The file descriptor returned by unifi_open()
 *      p           The user space buffer to get the data from
 *      len         The size of the p buffer
 *      poff
 *
 *  Returns:
 *      number of bytes written or an error code on failure
 * ---------------------------------------------------------------------------
 */
static ssize_t unifi_write(struct file *filp, const char *p, size_t len, loff_t *poff)
{
    ul_client_t *pcli = (ul_client_t *) filp->private_data;
    os_linux_priv_t *priv;
    unsigned char *buf;
    unsigned char *bufptr;
    int remaining, bytes_written;

    func_enter();

    priv = uf_find_instance(pcli->instance);
    if (!priv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : invalid priv\n"));
        return -ENODEV;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: unifi_write: len = %d\n",
                        priv ? priv->instance : 0,  len));

    if (!pcli->udi_enabled)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: udi disabled\n", priv ? priv->instance : 0));
        return -EINVAL;
    }


    buf = kmalloc(len, GFP_KERNEL);
    if (!buf)
    {
        return -ENOMEM;
    }

    /* Get the data from the client (SME or Unicli). */
    if (copy_from_user((void *) buf, p, len))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: copy from user failed\n", priv->instance));
        kfree(buf);
        return -EFAULT;
    }

    /*
     * In SME userspace build read() contains a SYS or MGT message.
     * Note that even though the SME sends one signal at a time, we can not
     * use unifi_net_data_malloc because in the early stages, before having
     * initialised the core, it will fail since the I/O block size is unknown.
     */
#ifdef CSR_SME_USERSPACE
    if (pcli->configuration & CLI_SME_USERSPACE)
    {
        CsrWifiRouterTransportRecv(priv, buf, len);
        kfree(buf);
        return len;
    }
#endif

    /* ul_send_signal_raw will  do a sanity check of len against signal content */

    /*
     * udi_send_signal_raw() and udi_send_signal_unpacked() return the number of bytes consumed.
     * A write call can pass multiple signal concatenated together.
     */
    bytes_written = 0;
    remaining = len;
    bufptr = buf;
    while (remaining > 0)
    {
        int r;

        /*
         * Set the SenderProcessId.
         * The SignalPrimitiveHeader is the first 3 16-bit words of the signal,
         * the SenderProcessId is bytes 4,5.
         * The MSB of the sender ID needs to be set to the client ID.
         * The LSB is controlled by the SME.
         */
        bufptr[5] = (pcli->sender_id >> 8) & 0xff;

        /* use the appropriate interface, depending on the clients' configuration */
        if (pcli->configuration & CLI_USING_WIRE_FORMAT)
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,
                                "unifi%d: unifi_write: call udi_send_signal().\n", priv->instance));
            r = udi_send_signal_raw(priv, bufptr, remaining);
        }
        else
        {
            CsrResult csrResult;
            CsrUint16 interfaceTag = 0, interfaceIndex = default_vif;
            CsrUint8 sigPack[UNIFI_PACKED_SIGBUF_SIZE];
            CsrUint16 sigPackLen;

            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,
                                "unifi%d: unifi_write: call udi_send_signal_unpacked().\n", priv->instance));

            /* Try to find the interfaceIndex/interfaceTag from the encoded signal.
             * To do this, we need to pack the signal to use read_vif_from_packed()!
             * This is sub-optimal, as it happens again later on, but this as interface
             * is only used for low bandwidth traffic from debug tools, it's preferable
             * in terms of code size to implementing read_vif_from_unpacked().
             * Default to interfaceTag 0.
             */
            csrResult = write_pack((const CSR_SIGNAL *) bufptr, sigPack, &sigPackLen);
            if (csrResult != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: Unknown unpacked HIP signal in unifi_write()\n", priv->instance));
                bytes_written = -EINVAL; /* Set the return value to the error code */
                break;
            }

            csrResult = read_vif_from_packed(sigPack, &interfaceIndex);
            if (csrResult != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: Could not find vifIndex - use default interfaceIndex=%u\n", priv->instance, interfaceIndex));
                interfaceIndex = default_vif;
            }

            interfaceTag = CsrWifiHipInterfaceTagGetReq(priv->hip_handle, interfaceIndex);
            if (interfaceTag >= priv->totalInterfaceCount)
            {
                interfaceTag = 0;
            }

            /* Debug signals go via the magic default device */
            r = udi_send_signal_unpacked(priv->interfacePriv[interfaceTag], bufptr, remaining);
        }
        if (r < 0)
        {
            /* Set the return value to the error code */
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: unifi_write: (udi or sme)_send_signal() returns %d\n", priv->instance,  r));
            bytes_written = r;
            break;
        }
        bufptr += r;
        remaining -= r;
        bytes_written += r;
    }

    kfree(buf);

    func_exit_r(bytes_written);

    return bytes_written;
} /* unifi_write() */

#ifdef CSR_LOG_ENABLE
/* This function is only called from within a trace macro. */
static const char *build_type_to_string(unsigned char build_type)
{
    switch (build_type)
    {
        case UNIFI_BUILD_NME:
            return "NME";
        case UNIFI_BUILD_WEXT:
            return "WEXT";
        case UNIFI_BUILD_AP:
            return "AP";
    }
    return "unknown";
}

#endif /* CSR_LOG_ENABLE */

/*
 * ----------------------------------------------------------------
 *  unifi_ioctl
 *
 *      Ioctl handler for unifi driver.
 *
 * Arguments:
 *  inodep          Pointer to inode structure.
 *  filp            Pointer to file structure.
 *  cmd             Ioctl cmd passed by user.
 *  arg             Ioctl arg passed by user.
 *
 * Returns:
 *      0 on success, -ve error code on error.
 * ----------------------------------------------------------------
 */
static long unifi_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
    ul_client_t *pcli = (ul_client_t *) filp->private_data;
    os_linux_priv_t *priv;
    struct net_device *dev;
    int r = 0;
    int int_param, i;
#if (defined CSR_SUPPORT_SME)
    unifi_cfg_command_t cfg_cmd;
#if (defined CSR_SUPPORT_WEXT)
    unsigned char uchar_param;
    CsrWifiDataBlock mibData = {0, NULL};
    CsrUint32 mibDataOutLength;
#endif
#endif
    unifi_putest_command_t putest_cmd;


    priv = uf_find_instance(pcli->instance);

    if (!priv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: ioctl error: unknown instance=%d\n",
                               priv ? priv->instance : 0,  pcli->instance));
        r = -ENODEV;
        goto out;
    }
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                        "unifi%d: unifi_ioctl: cmd=0x%X, arg=0x%lX\n", priv ? priv->instance : 0,  cmd, arg));

    switch (cmd)
    {
        case UNIFI_GET_UDI_ENABLE:
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                                "unifi%d: UniFi Get UDI Enable\n", priv ? priv->instance : 0));

            down(&priv->udi_logging_mutex);
            int_param = (priv->logging_client == NULL) ? 0 : 1;
            up(&priv->udi_logging_mutex);

            if (put_user(int_param, (int *) arg))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_GET_UDI_ENABLE: Failed to copy to user\n", priv ? priv->instance : 0));
                r = -EFAULT;
                goto out;
            }
            break;

        case UNIFI_SET_UDI_ENABLE:
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                "unifi%d: UniFi Set UDI Enable\n", priv ? priv->instance : 0));
            if (get_user(int_param, (int *) arg))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_SET_UDI_ENABLE: Failed to copy from user\n", priv ? priv->instance : 0));
                r = -EFAULT;
                goto out;
            }

            down(&priv->udi_logging_mutex);
            if (int_param)
            {
                CsrResult retUdi;
                pcli->event_hook = udi_log_event;

                retUdi = CsrWifiHipLogUdiRegistrationReq(priv->hip_handle, (CsrWifiHipLogUdiFunc) logging_handler);
                if (retUdi != CSR_RESULT_SUCCESS)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to register UDI hook in HAL\n", priv ? priv->instance : 0));
                }

                /* Log all signals by default */
                for (i = 0; i < SIG_FILTER_SIZE; i++)
                {
                    pcli->signal_filter[i] = 0xFFFF;
                }
                if (int_param == 2)
                {
                    /* Advanced Logging to enable Logging of Host signals */
                    pcli->configuration |= CLI_LOG_HOST;
                }
                if (int_param == 3)
                {
                    /* Advanced Logging to enable Logging of Host signals (Including CsrWifiRouterCtrlHip Signals) */
                    pcli->configuration |= CLI_LOG_HOST;
                    pcli->configuration |= CLI_LOG_HOST_ALL;
                }
                priv->logging_client = pcli;
            }
            else
            {
                priv->logging_client = NULL;
                pcli->event_hook = NULL;
            }
            up(&priv->udi_logging_mutex);

            break;

        case UNIFI_SET_MIB:
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                "unifi : UniFi Set MIB\n"));
#if (defined CSR_SUPPORT_SME) && (defined CSR_SUPPORT_WEXT)
            /* Read first 4 bytes as length */
            if (copy_from_user((void *) &mibData.dataLength, (void *) arg, 4))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF,
                                       "unifi : UNIFI_SET_MIB: Failed to copy in Mib Data Length\n"));
                r = -EFAULT;
                goto out;
            }
            /* Read first 2nd 4 bytes as out buffer length */
            if (copy_from_user((void *) &mibDataOutLength, (void *) (arg + 4), 4))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF,
                                       "unifi : UNIFI_SET_MIB: Failed to copy in Mib Data Out Length\n"));
                r = -EFAULT;
                goto out;
            }

            mibData.data = kmalloc(mibData.dataLength, GFP_KERNEL);

            /* Read the rest of the Mib Data */
            if (copy_from_user((void *) mibData.data, (void *) (arg + 8), mibData.dataLength))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF,
                                       "unifi : UNIFI_SET_MIB: Failed to copy in Mib Data\n"));
                r = -EFAULT;
                kfree(mibData.data);
                goto out;
            }

            r = sme_mgt_mib_set(priv, 0 /* InterfaceTag */, &mibData);
            kfree(mibData.data);
            if (r)
            {
                goto out;
            }
#else
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : UNIFI_SET_MIB: Unsupported.\n"));
#endif /* CSR_SUPPORT_WEXT */
            break;

        case UNIFI_GET_MIB:
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                "unifi%d: UniFi Get MIB\n", priv ? priv->instance : 0));
#if (defined CSR_SUPPORT_SME) && (defined CSR_SUPPORT_WEXT)
            /* Read first 4 bytes as length */
            if (copy_from_user((void *) &mibData.dataLength, (void *) arg, 4))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_GET_MIB: Failed to copy in varbind header\n", priv ? priv->instance : 0));
                r = -EFAULT;
                goto out;
            }
            /* Read first 2nd 4 bytes as out buffer length */
            if (copy_from_user((void *) &mibDataOutLength, (void *) (arg + 4), 4))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to copy in Mib Data Out Length\n",
                                       priv ? priv->instance : 0));
                r = -EFAULT;
                goto out;
            }

            mibData.data = kmalloc(mibData.dataLength, GFP_KERNEL);

            /* Read the rest of the Mib Data */
            if (copy_from_user((void *) mibData.data, (void *) (arg + 8), mibData.dataLength))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_GET_MIB: Failed to copy in Mib Data\n", priv ? priv->instance : 0));
                r = -EFAULT;
                kfree(mibData.data);
                goto out;
            }

            r = sme_mgt_mib_get(priv, 0 /* InterfaceTag */, &mibData);
            if (r)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_GET_MIB: sme_mgt_mib_get() Failed\n", priv ? priv->instance : 0));
                goto out;
            }
            /* copy out Mib Data */
            if (mibData.dataLength > mibDataOutLength)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: unifi_ioctl: UNIFI_GET_MIB. Mib Data result too long (%d, limit %d)\n",
                                       priv->instance, mibData.dataLength, mibDataOutLength));
                r = -EINVAL;
                goto out;
            }
            if (copy_to_user((void *) arg, (void *) &mibData.dataLength, 4))
            {
                r = -EFAULT;
                goto out;
            }
            if (copy_to_user((void *) (arg + 4), mibData.data, mibData.dataLength))
            {
                r = -EFAULT;
                goto out;
            }
            CsrPmemFree(mibData.data);
#else
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: UNIFI_GET_MIB: Unsupported.\n", priv->instance));
#endif /* CSR_SUPPORT_WEXT */
            break;
        case UNIFI_CFG:
#if (defined CSR_SUPPORT_SME)
            if (get_user(cfg_cmd, (unifi_cfg_command_t *) arg))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to get the command\n", priv ? priv->instance : 0));
                r = -EFAULT;
                goto out;
            }

            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: UNIFI_CFG: Command is %d (t=%u) sz=%d\n",
                                  priv ? priv->instance : 0,
                                  cfg_cmd, jiffies_to_msecs(jiffies), sizeof(unifi_cfg_command_t)));
            switch (cfg_cmd)
            {
                case UNIFI_CFG_POWER:
                    r = unifi_cfg_power(priv, (unsigned char *) arg);
                    break;
                case UNIFI_CFG_POWERSAVE:
                    r = unifi_cfg_power_save(priv, 0 /* InterfaceTag */, (unsigned char *) arg);
                    break;
                case UNIFI_CFG_POWERSUPPLY:
                    r = unifi_cfg_power_supply(priv, 0 /* InterfaceTag */, (unsigned char *) arg);
                    break;
                case UNIFI_CFG_FILTER:
                    r = unifi_cfg_packet_filters(priv, 0 /* InterfaceTag */, (unsigned char *) arg);
                    break;
                case UNIFI_CFG_GET:
                    r = unifi_cfg_get_info(priv, 0 /* InterfaceTag */, (unsigned char *) arg);
                    break;
                case UNIFI_CFG_WMM_QOSINFO:
                    r = unifi_cfg_wmm_qos_info(priv, 0 /* InterfaceTag */, (unsigned char *) arg);
                    break;
                case UNIFI_CFG_WMM_ADDTS:
                    r = unifi_cfg_wmm_addts(priv, 0 /* InterfaceTag */, (unsigned char *) arg);
                    break;
                case UNIFI_CFG_WMM_DELTS:
                    r = unifi_cfg_wmm_delts(priv, 0 /* InterfaceTag */, (unsigned char *) arg);
                    break;
                case UNIFI_CFG_STRICT_DRAFT_N:
                    r = unifi_cfg_strict_draft_n(priv, 0 /* InterfaceTag */, (unsigned char *) arg);
                    break;

#ifdef CSR_SUPPORT_SME
                case UNIFI_CFG_CORE_DUMP:
                    CsrWifiRouterCtrlWifiOffIndSend(priv->CSR_WIFI_SME_IFACEQUEUE, 0, CSR_WIFI_SME_CONTROL_INDICATION_ERROR);
                    CSR_LOG_TEXT_INFO((
                                          CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2, "unifi : UNIFI_CFG_CORE_DUMP: sent wifi off indication\n"));
                    break;
#endif
#ifdef CSR_SUPPORT_WEXT_AP
                case UNIFI_CFG_SET_AP_CONFIG:
                    r = unifi_cfg_set_ap_config(priv, (unsigned char *) arg);
                    break;
#endif
                default:
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Unknown Command (%d)\n",
                                           priv ? priv->instance : 0,  cfg_cmd));
                    r = -EINVAL;
                    goto out;
            }
#endif

            break;
        case UNIFI_PUTEST:
            if (get_user(putest_cmd, (unifi_putest_command_t *) arg))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_PUTEST: Failed to get the command\n", priv ? priv->instance : 0));
                r = -EFAULT;
                goto out;
            }

            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: UNIFI_PUTEST: Command is %s\n",
                                  priv ? priv->instance : 0,
                                  trace_putest_cmdid(putest_cmd)));

            switch (putest_cmd)
            {
                case UNIFI_PUTEST_START:
                    r = unifi_putest_start(priv, (unsigned char *) arg);
                    break;
                case UNIFI_PUTEST_STOP:
                    r = unifi_putest_stop(priv, (unsigned char *) arg);
                    break;
                case UNIFI_PUTEST_SET_SDIO_CLOCK:
                    r = unifi_putest_set_sdio_clock(priv, (unsigned char *) arg);
                    break;
                case UNIFI_PUTEST_CMD52_READ:
                    r = unifi_putest_cmd52_read(priv, (unsigned char *) arg);
                    break;
                case UNIFI_PUTEST_CMD52_BLOCK_READ:
                    r = unifi_putest_cmd52_block_read(priv, (unsigned char *) arg);
                    break;
                case UNIFI_PUTEST_CMD52_WRITE:
                    r = unifi_putest_cmd52_write(priv, (unsigned char *) arg);
                    break;
                case UNIFI_PUTEST_DL_FW:
                    r = unifi_putest_dl_fw(priv, (unsigned char *) arg);
                    break;
                case UNIFI_PUTEST_DL_FW_BUFF:
                    r = unifi_putest_dl_fw_buff(priv, (unsigned char *) arg);
                    break;
                case UNIFI_PUTEST_COREDUMP_PREPARE:
                    r = unifi_putest_coredump_prepare(priv, (unsigned char *) arg);
                    break;
                case UNIFI_PUTEST_GP_READ16:
                    r = unifi_putest_gp_read16(priv, (unsigned char *) arg);
                    break;
                case UNIFI_PUTEST_GP_WRITE16:
                    r = unifi_putest_gp_write16(priv, (unsigned char *) arg);
                    break;
                default:
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_PUTEST: Unknown Command (%d)\n",
                                           priv ? priv->instance : 0,  putest_cmd));
                    r = -EINVAL;
                    goto out;
            }

            break;
        case UNIFI_BUILD_TYPE:
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                               "unifi%d: UNIFI_BUILD_TYPE userspace=%s\n",
                               priv ? priv->instance : 0,  build_type_to_string(*(unsigned char *) arg)));

#ifndef CSR_SUPPORT_WEXT_AP
            if (UNIFI_BUILD_AP == *(unsigned char *) arg)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Userspace has AP support, which is incompatible\n", priv ? priv->instance : 0));
            }
#endif

#ifndef CSR_SUPPORT_WEXT
            if (UNIFI_BUILD_WEXT == *(unsigned char *) arg)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Userspace has WEXT support, which is incompatible\n", priv ? priv->instance : 0));
            }
#endif
            break;
        case UNIFI_INIT_HW:
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                               "unifi%d: UNIFI_INIT_HW.\n", priv ? priv->instance : 0));
#ifndef CSR_NATIVE_LINUX
            priv->init_progress = UNIFI_INIT_NONE;
#endif

#if defined(CSR_SUPPORT_WEXT) || defined(CSR_NATIVE_LINUX)
            /* At this point we are ready to start the SME. */
            r = sme_mgt_wifi_on(priv);
            if (r)
            {
                goto out;
            }
#endif

            break;

        case UNIFI_INIT_NETDEV:
        {
            /* D-29894 get the proper interface. For now, use the default netdevice i.e 'netdev 0' */
            CsrUint16 interfaceTag = 0;
#ifdef CSR_NATIVE_LINUX_DEFAULT_INTERFACE
            netInterface_priv_t *interfacePriv = priv->interfacePriv[interfaceTag];
#endif
            dev = priv->netdev[interfaceTag];

            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                               "unifi%d: UNIFI_INIT_NETDEV.\n", priv ? priv->instance : 0));

            if (copy_from_user((void *) dev->dev_addr, (void *) arg, 6))
            {
                r = -EFAULT;
                goto out;
            }

#ifdef CSR_NATIVE_LINUX_DEFAULT_INTERFACE
            /* Attach the network device to the stack */
            if (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_UNREGISTERED)
            {
                r = uf_register_netdev(priv, interfaceTag);
                if (r)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to register the network device.\n", priv ? priv->instance : 0));
                    goto out;
                }
            }

            priv->init_progress = UNIFI_INIT_COMPLETED;

#ifdef CSR_SUPPORT_WEXT
            /* Notify the Android wpa_supplicant that we are ready */
            wext_send_started_event(priv, interfacePriv->InterfaceTag);
#endif

            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: UniFi ready\n", priv ? priv->instance : 0));

#ifdef CSR_NATIVE_SOFTMAC /* For softmac dev, force-enable the network interface rather than wait for a connected-ind */
            {
                netif_carrier_on(dev);
            }
#endif
#endif

            break;
        }
        case UNIFI_GET_INIT_STATUS:
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                               "unifi%d: UNIFI_GET_INIT_STATUS.\n", priv ? priv->instance : 0));
            if (put_user(priv->init_progress, (int *) arg))
            {
                printk(KERN_ERR "UNIFI_GET_INIT_STATUS: Failed to copy to user\n");
                r = -EFAULT;
                goto out;
            }
            break;

        case UNIFI_SET_DEBUG:
            unifi_debug = arg;
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                "unifi%d: unifi_debug set to %d\n",
                                priv ? priv->instance : 0,  unifi_debug));
            break;

        case UNIFI_SET_TRACE:
            /* no longer supported */
            r = -EINVAL;
            break;

        case UNIFI_SOFTMAC_CFG:
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_SOFTMAC_CFG: Not supported in this build\n", priv ? priv->instance : 0));
            r = -EINVAL;
            break;
        }

        case UNIFI_SET_UDI_LOG_MASK:
        {
            unifiio_filter_t udi_filter;
            uint16_t *sig_ids_addr;
#define UF_MAX_SIG_IDS  128     /* Impose a sensible limit */

            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                "unifi%d: UNIFI_SET_UDI_LOG_MASK\n", priv ? priv->instance : 0));

            if (copy_from_user((void *) (&udi_filter), (void *) arg, sizeof(udi_filter)))
            {
                r = -EFAULT;
                goto out;
            }
            if ((udi_filter.action < UfSigFil_AllOn) ||
                (udi_filter.action > UfSigFil_SelectOff))
            {
                printk(KERN_WARNING
                       "UNIFI_SET_UDI_LOG_MASK: Bad action value: %d\n",
                       udi_filter.action);
                r = -EINVAL;
                goto out;
            }
            /* No signal list for "All" actions */
            if ((udi_filter.action == UfSigFil_AllOn) ||
                (udi_filter.action == UfSigFil_AllOff))
            {
                udi_filter.num_sig_ids = 0;
            }

            if (udi_filter.num_sig_ids > UF_MAX_SIG_IDS)
            {
                printk(KERN_WARNING
                       "UNIFI_SET_UDI_LOG_MASK: too many signal ids (%d, max %d)\n",
                       udi_filter.num_sig_ids, UF_MAX_SIG_IDS);
                r = -EINVAL;
                goto out;
            }

            /* Copy in signal id list if given */
            if (udi_filter.num_sig_ids > 0)
            {
                /* Preserve userspace address of sig_ids array */
                sig_ids_addr = udi_filter.sig_ids;
                /* Allocate kernel memory for sig_ids and copy to it */
                udi_filter.sig_ids =
                    kmalloc(udi_filter.num_sig_ids * sizeof(uint16_t), GFP_KERNEL);
                if (!udi_filter.sig_ids)
                {
                    r = -ENOMEM;
                    goto out;
                }
                if (copy_from_user((void *) udi_filter.sig_ids,
                                   (void *) sig_ids_addr,
                                   udi_filter.num_sig_ids * sizeof(uint16_t)))
                {
                    kfree(udi_filter.sig_ids);
                    r = -EFAULT;
                    goto out;
                }
            }

            udi_set_log_filter(pcli, &udi_filter);

            if (udi_filter.num_sig_ids > 0)
            {
                kfree(udi_filter.sig_ids);
            }
            break;
        }

        case UNIFI_SME_PRESENT:
        {
            u8 ind;
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                   "unifi%d: UniFi SME Present IOCTL.\n", priv ? priv->instance : 0));
            if (copy_from_user((void *) (&int_param), (void *) arg, sizeof(int)))
            {
                printk(KERN_ERR "UNIFI_SME_PRESENT: Failed to copy from user\n");
                r = -EFAULT;
                goto out;
            }

            if (int_param == 1)
            {
                ind = CONFIG_SME_PRESENT;
            }
            else
            {
                ind = CONFIG_SME_NOT_PRESENT;
            }
            /* Send an indication to the helper app. */
            ul_log_config_ind(priv, &ind, sizeof(u8));
            break;
        }

        case UNIFI_CFG_PERIOD_TRAFFIC:
        {
#if (defined CSR_SUPPORT_SME) && (defined CSR_SUPPORT_WEXT)
            CsrWifiDataBlock mibData = {0, NULL};
            CsrWifiMibEntry mibValue;
            CsrSize offset = 0;
            CsrUint8 *data;
            csr_wifi_cfg_coex_config_t coexConfig;

            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                "unifi%d: UniFi Configure Periodic Traffic.\n", priv ? priv->instance : 0));
            if (copy_from_user((void *) (&uchar_param), (void *) arg, sizeof(unsigned char)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG_PERIOD_TRAFFIC: Failed to copy from user\n", priv ? priv->instance : 0));
                r = -EFAULT;
                goto out;
            }

            if (uchar_param == 0)
            {
#ifdef CSR_WIFI_COEX_ENABLE
                CsrWifiMibEncodeGet(&mibData, CSR_WIFI_SME_PSID_COEX_CFG_ENABLE_SCHEME_MANAGEMENT, 0);
#endif
                CsrWifiMibEncodeGet(&mibData, CSR_WIFI_SME_PSID_PERIODIC_WAKE_HOST, 0);

                data = mibData.data;
                r = sme_mgt_mib_get(priv, 0 /* InterfaceTag */, &mibData);
                CsrPmemFree(data);

                if (r)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG_PERIOD_TRAFFIC: Get unifi_CoexInfoValue failed.\n", priv ? priv->instance : 0));
                    goto out;
                }
#ifdef CSR_WIFI_COEX_ENABLE
                offset = CsrWifiMibDecode(mibData.data, &mibValue);
                coexConfig.coexEnableSchemeManagement = mibValue.value.u.uintValue;
#endif
                CsrWifiMibDecode(&mibData.data[offset], &mibValue);
                coexConfig.coexPeriodicWakeHost = mibValue.value.u.uintValue;

                CsrPmemFree(mibData.data);

                if (copy_to_user((void *) (arg + 1),
                                 (void *) &coexConfig,
                                 sizeof(csr_wifi_cfg_coex_config_t)))
                {
                    r = -EFAULT;
                    goto out;
                }
                goto out;
            }

            if (copy_from_user((void *) (&coexConfig), (void *) (arg + 1), sizeof(csr_wifi_cfg_coex_config_t)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG_PERIOD_TRAFFIC: Failed to copy from user\n", priv ? priv->instance : 0));
                r = -EFAULT;
                goto out;
            }

#ifdef CSR_WIFI_COEX_ENABLE
            CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_COEX_CFG_ENABLE_SCHEME_MANAGEMENT, coexConfig.coexEnableSchemeManagement, 0);
#endif
            CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_PERIODIC_WAKE_HOST, coexConfig.coexPeriodicWakeHost, 0);

            data = mibData.data;
            r = sme_mgt_mib_set(priv, 0 /* InterfaceTag */, &mibData);
            CsrPmemFree(data);

            if (r)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG_PERIOD_TRAFFIC: Set unifi_CoexInfoValue failed.\n", priv ? priv->instance : 0));
                goto out;
            }
#endif
            break;
        }
        case UNIFI_CFG_UAPSD_TRAFFIC:
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                "unifi%d: UniFi Configure U-APSD Mask.\n", priv ? priv->instance : 0));
#if (defined CSR_SUPPORT_SME) && (defined CSR_SUPPORT_WEXT)
            if (copy_from_user((void *) (&uchar_param), (void *) arg, sizeof(unsigned char)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG_UAPSD_TRAFFIC: Failed to copy from user\n", priv ? priv->instance : 0));
                r = -EFAULT;
                goto out;
            }
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: New U-APSD Mask: 0x%x\n",
                                   priv ? priv->instance : 0,  uchar_param));
#endif /* CSR_SUPPORT_SME && CSR_SUPPORT_WEXT */
            break;

#ifdef CSR_NATIVE_LINUX
        case UNIFI_NETDEV_CFG:
        {
            unifiio_netdev_cfg_t cfg_req;
            netInterface_priv_t *interfacePriv = NULL;

            if (copy_from_user((void *) (&cfg_req), (void *) arg, sizeof(cfg_req)))
            {
                r = -EFAULT;
                goto out;
            }
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2, "unifi%d:%s: UNIFI_NETDEV_CFG ifTag %d, Op %d\n",
                               priv ? priv->instance : 0, __FUNCTION__, cfg_req.interfaceTag, cfg_req.operation));

            /* Make damn sure that from-user string is null-terminated */
            cfg_req.name[IFNAMSIZ - 1] = '\0';

            switch (cfg_req.operation)
            {
                case 0: /* Configure operation */
                {
                    CsrUint8 i;
                    CsrWifiMacAddress macAddr;

                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                       "%s UNIFI_NETDEV_CFG 0 - Configure %d\n",
                                       __FUNCTION__, cfg_req.interfaceTag));

                    /* Copy the Mac address to internal structure */
                    memcpy(macAddr.a, cfg_req.macaddr, ETH_ALEN);

                    if (cfg_req.interfaceTag < 0)
                    {
                        /* New interface  Creation request */
                        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: %s: Create ifTag=%d \n",
                                            priv ? priv->instance : 0, __FUNCTION__, cfg_req.interfaceTag));

                        if ((cfg_req.interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_NONE) || (cfg_req.interfaceMode > CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI))
                        {
                            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                               "%s UNIFI_NETDEV_CFG CMD:0 creating interface with mode 0 is illegal\n",
                                               __FUNCTION__));
                            r = -EINVAL;
                            goto out;
                        }

                        for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
                        {
                            CsrUint16 vif;

                            vif = CsrWifiHipVifIndexGetReq(priv->hip_handle, i);
                            if ((vif == 0xffff) || (vif == 0))
                            {
                                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                                   "%s UNIFI_NETDEV_CFG CMD:0 using CsrWifiHipVifIndexGetReq() to find free priv %d\n",
                                                   __FUNCTION__, i));
                                interfacePriv = priv->interfacePriv[i];
                                break;
                            }
                        }
                        if (!interfacePriv)
                        {
                            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                "%s: No free netdev slot\n", __FUNCTION__));
                            r = -EINVAL;
                            goto out;
                        }

                        /* Validate the vifIndex range & already existing */
                        if (!cfg_req.vif_index || (cfg_req.vif_index > (CSR_WIFI_HIP_MAX_VIFINDEX - 1)) ||
                            (CsrWifiHipInterfaceTagGetReq(priv->hip_handle, cfg_req.vif_index) != 0XFFFF))
                        {
                            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                "UNIFI_NETDEV_CFG(new): Bad vif_index %d\n",
                                                cfg_req.vif_index));
                            r = -EINVAL;
                            goto out;
                        }


                        /* Check if a netdev already has been registered */
                        if (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_UNREGISTERED)
                        {
                            /* Registers the statically created netdevice interface with network sub system */
                            if (CsrWifiHipInterfaceAddReq(priv->hip_handle, interfacePriv->InterfaceTag, macAddr))
                            {
                                CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                    "%s: unifi(%d): netdev registering failed\n",
                                                    __FUNCTION__, cfg_req.interfaceTag));
                                r = -EINVAL;
                                goto out;
                            }
                        }

                        CsrWifiHipVifModeSetReq(priv->hip_handle, interfacePriv->InterfaceTag,
                                                cfg_req.interfaceMode, macAddr, 0, 0,
                                                cfg_req.vif_index);

                        /* Store the interface mode */
                        interfacePriv->interfaceMode = cfg_req.interfaceMode;

                        if ((cfg_req.interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_STA) ||
                            (cfg_req.interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI) ||
                            (cfg_req.interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_IBSS) ||
                            (cfg_req.interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_AMP))
                        {
                            /* In these modes, a Packet Scheduler Queue Set is created by default.
                             * The Queue Set implements the transmit controlled port and it will
                             * be closed on crated. Open it.
                             */
                            CsrWifiHipPortConfigureReq(priv->hip_handle, interfacePriv->InterfaceTag,
                                                       CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN,
                                                       CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN,
                                                       &macAddr, FALSE);
                        }
                        interfacePriv->peer_sta_count = 0;
                        interfacePriv->tx_rate = 0;
                        /* Copy back device name/interfaceTag on successfull netdev registration */
                        memcpy(cfg_req.name, priv->netdev[interfacePriv->InterfaceTag]->name, IFNAMSIZ);
                        cfg_req.interfaceTag = interfacePriv->InterfaceTag;

                        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                           "unifi(%d): %s UNIFI_NETDEV_CFG(%d) - add: %s ready on VIF %d\n",
                                           interfacePriv->InterfaceTag, __FUNCTION__,
                                           cfg_req.interfaceTag,
                                           priv->netdev[interfacePriv->InterfaceTag]->name,
                                           cfg_req.vif_index));
                        netif_carrier_on(priv->netdev[interfacePriv->InterfaceTag]);
                        if ((interfacePriv->netdev_state != OS_LINUX_NETDEV_STATE_UNREGISTERED) && (priv->netdev[interfacePriv->InterfaceTag]->flags & IFF_UP))
                        {
                            /* It is stopping all the queues in interface unregister request. And the queues
                             * are restarted in uf_net_open. But if there is no transition in interface state
                             * (UP/DOWN) then uf_net_open is not invoked. So if the state is UP alreay start
                             * all the queues here itself
                             */
                            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                               "unifi(%d): unifi_ioctl: UNIFI_NETDEV_CFG: Interface already UP. Start all the queues\n",
                                               priv ? priv->instance : 0));
                            UF_NETIF_TX_START_ALL_QUEUES(priv->netdev[interfacePriv->InterfaceTag]);
                        }
                    }
                    else
                    {
                        /* Action on existing interface requested */

                        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: %s: Update ifTag=%d \n",
                                            priv ? priv->instance : 0, __FUNCTION__, cfg_req.interfaceTag));

                        /* Validate the input interfaceTag */
                        if (cfg_req.interfaceTag >= CSR_WIFI_MAX_INTERFACES)
                        {
                            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                "%s: unifi(%d): Bad interfaceTag\n",
                                                __FUNCTION__, cfg_req.interfaceTag));
                            r = -EINVAL;
                            goto out;
                        }
                        /* Fetch the corresponding interfacePriv from interfaceTag */
                        if (!(interfacePriv = priv->interfacePriv[cfg_req.interfaceTag]))
                        {
                            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                "%s: unifi(%d): interfacePriv=NULL\n",
                                                __FUNCTION__, cfg_req.interfaceTag));
                            r = -EINVAL;
                            goto out;
                        }

                        /* Ensure that we issue an error if someone is trying to modify a none registered netdev */
                        if (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_UNREGISTERED)
                        {
                            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                "%s: unifi(%d): Trying to modify an interface, which hasn't been added\n",
                                                __FUNCTION__, cfg_req.interfaceTag));
                            r = -EINVAL;
                            goto out;
                        }

                        /* Copy data for clean return in case of delete */
                        memcpy(cfg_req.name, priv->netdev[interfacePriv->InterfaceTag]->name, IFNAMSIZ);

                        /* Delete existing interface request i.e. vif_index = 0 */
                        if (!cfg_req.vif_index)
                        {
                            if (cfg_req.interfaceTag == 0)
                            {
                                /* Taken care by SME but for safety sake */
                                CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                    "UNIFI_NETDEV_CFG(%d): refusing delete of default IF\n",
                                                    cfg_req.interfaceTag));
                                r = -EINVAL;
                                goto out;
                            }
                            /* Copy in what it used to be, in case someone needs it */
                            cfg_req.vif_index = CsrWifiHipVifIndexGetReq(priv->hip_handle, interfacePriv->InterfaceTag);
                            cfg_req.interfaceMode = interfacePriv->interfaceMode;
                            memcpy(cfg_req.macaddr, priv->netdev[interfacePriv->InterfaceTag]->dev_addr, ETH_ALEN);

                            /* Set mode to NONE to cleart database corresponding to interfaceTag/mode */
                            CsrWifiHipVifModeSetReq(priv->hip_handle, interfacePriv->InterfaceTag,
                                                    CSR_WIFI_ROUTER_CTRL_MODE_NONE, macAddr,
                                                    0, 0, cfg_req.vif_index);

                            /* Unregisters the statically created netdevice interface with network sub system */
                            if (CsrWifiHipInterfaceDelReq(priv->hip_handle, interfacePriv->InterfaceTag))
                            {
                                CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                    "%s: unifi(%d): netdev Unregistering failed\n",
                                                    __FUNCTION__, interfacePriv->InterfaceTag));
                                r = -EINVAL;
                                goto out;
                            }
                            interfacePriv->tx_rate = 0;
                            interfacePriv->peer_sta_count = 0;
                            break; /* Break out of this case - goes to Success for UNIFI_NETDEV_CFG. */
                        }

                        /* Set New values on existing(registered) interface:
                         * -> Handle mode change scenario
                         * -> If mode is same, check for Mac address change scenario
                         * -> Mode, MacAddress both are same, check for vifIndex update scenario
                         */
                        if (interfacePriv->interfaceMode != cfg_req.interfaceMode)
                        {
                            /* Handles Mode change Req as well Mac address change also */
                            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                               "unifi(%d): Mode change Req: prevMode=%x, newMode=%x\n",
                                               interfacePriv->InterfaceTag, interfacePriv->interfaceMode,
                                               cfg_req.interfaceMode));

                            /* Primarily stop the existing mode on the interface */
                            CsrWifiHipVifModeSetReq(priv->hip_handle, interfacePriv->InterfaceTag,
                                                    CSR_WIFI_ROUTER_CTRL_MODE_NONE, macAddr,
                                                    0, 0, 0);

                            /* Stop any traffic flow from the network stack */
                            if (CsrWifiHipInterfaceDelReq(priv->hip_handle, interfacePriv->InterfaceTag))
                            {
                                CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                    "%s: unifi(%d): netdev Unregistering failed\n",
                                                    __FUNCTION__, interfacePriv->InterfaceTag));
                                r = -EINVAL;
                                goto out;
                            }

                            /* Since operation on existing interface, if needed updates MacAddress
                             * change & enables the carrier
                             */
                            if (CsrWifiHipInterfaceAddReq(priv->hip_handle, interfacePriv->InterfaceTag, macAddr))
                            {
                                CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                    "%s: unifi(%d): netdev registering failed\n",
                                                    __FUNCTION__, cfg_req.interfaceTag));
                                r = -EINVAL;
                                goto out;
                            }

                            /* Set mode to requested mode for corresponding interface */
                            CsrWifiHipVifModeSetReq(priv->hip_handle, interfacePriv->InterfaceTag,
                                                    cfg_req.interfaceMode, macAddr, 0, 0,
                                                    cfg_req.vif_index);
                            /* Store the interface mode */
                            interfacePriv->interfaceMode = cfg_req.interfaceMode;
                            interfacePriv->tx_rate = 0;
                            interfacePriv->peer_sta_count = 0;
                            if ((cfg_req.interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_STA) ||
                                (cfg_req.interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI) ||
                                (cfg_req.interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_IBSS) ||
                                (cfg_req.interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_AMP))
                            {
                                /*
                                 * In these modes, a Packet Scheduler Queue Set is created by default.
                                 * The Queue Set implements the transmit controlled port and it will
                                 * be closed on crated. Open it.
                                 */
                                CsrWifiHipPortConfigureReq(priv->hip_handle, interfacePriv->InterfaceTag,
                                                           CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN,
                                                           CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN,
                                                           &macAddr, FALSE);
                            }
                        }
                        else if (memcmp(macAddr.a, priv->netdev[interfacePriv->InterfaceTag]->dev_addr, ETH_ALEN))
                        {
                            /* Mode still the same, handles the Mac address change */

                            /* Updates netdev struct with the new MacAdress provided */
                            if (CsrWifiHipInterfaceAddReq(priv->hip_handle, interfacePriv->InterfaceTag, macAddr))
                            {
                                CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                    "%s: unifi(%d): netdev registering failed\n",
                                                    __FUNCTION__, cfg_req.interfaceTag));
                                r = -EINVAL;
                                goto out;
                            }
                        }
                        else
                        {
                            CsrUint16 old_vif = 0;
                            /* Mode & Mac address are same but only "VifIndex" to be updated
                             * (if vifIndex not in database already) for already existing interface
                             *  (interfaceTag) so last arg(i.e isPrimaryVifIndex) set to TRUE here
                             */
                            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: %s: Update ifTag=%d vifIndex=%d\n",
                                                priv ? priv->instance : 0, __FUNCTION__, cfg_req.interfaceTag, cfg_req.vif_index));
                            if ((old_vif = CsrWifiHipVifIndexGetReq(priv->hip_handle, interfacePriv->InterfaceTag)) != cfg_req.vif_index)
                            {
                                if (CsrWifiHipVifAddReq(priv->hip_handle, interfacePriv->InterfaceTag, cfg_req.vif_index, TRUE))
                                {
                                    CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                        "%s: unifi(%d): Failed to update vifIndex=%x\n",
                                                        __FUNCTION__, cfg_req.interfaceTag, cfg_req.vif_index));
                                    r = -EINVAL;
                                    goto out;
                                }
                                if (CsrWifiHipVifDelReq(priv->hip_handle, interfacePriv->InterfaceTag, old_vif, cfg_req.vif_index))
                                {
                                    CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                        "%s: unifi(%d): CsrWifiHipVifDelReq Failed to remove vifIndex=%x\n",
                                                        __FUNCTION__, cfg_req.interfaceTag, old_vif));
                                    r = -EINVAL;
                                    goto out;
                                }
                            }
                        }
                        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                           "unifi(%d): %s UNIFI_NETDEV_CFG(%d) - update: %s ready on VIF %d\n",
                                           interfacePriv->InterfaceTag, __FUNCTION__,
                                           cfg_req.interfaceTag,
                                           priv->netdev[interfacePriv->InterfaceTag]->name,
                                           cfg_req.vif_index));

                        netif_carrier_on(priv->netdev[interfacePriv->InterfaceTag]);
                        if ((interfacePriv->netdev_state != OS_LINUX_NETDEV_STATE_UNREGISTERED) && (priv->netdev[interfacePriv->InterfaceTag]->flags & IFF_UP))
                        {
                            /* It is stopping all the queues in interface unregister request. And the queues
                             * are restarted in uf_net_open. But if there is no transition in interface state
                             * (UP/DOWN) then uf_net_open is not invoked. So if the state is UP alreay start
                             * all the queues here itself
                             */
                            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                               "unifi(%d): unifi_ioctl: UNIFI_NETDEV_CFG: Interface already UP. Start all the queues\n",
                                               priv ? priv->instance : 0));
                            UF_NETIF_TX_START_ALL_QUEUES(priv->netdev[interfacePriv->InterfaceTag]);
                        }
                        /* Store the interface mode */
                        interfacePriv->interfaceMode = cfg_req.interfaceMode;

                        memcpy(cfg_req.name, priv->netdev[interfacePriv->InterfaceTag]->name, IFNAMSIZ);
                    }
                    break;
                }

                case 1: /* Query operation */
                {
                    static const CsrWifiMacAddress broadcast_address = {{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}};
                    CsrUint16 interfaceTag;
                    CsrUint8 i;

                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                       "%s UNIFI_NETDEV_CFG - Query %d\n",
                                       __FUNCTION__, cfg_req.interfaceTag));

                    if ((cfg_req.interfaceTag >= 0) && (cfg_req.interfaceTag < CSR_WIFI_MAX_INTERFACES))   /* Find by interfaceTag */
                    {   /* Find by interfaceTag */
                        if (!(interfacePriv = priv->interfacePriv[cfg_req.interfaceTag]))
                        {
                            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                "UNIFI_NETDEV_CFG, queried non-existent interfaceTag(%d)\n",
                                                cfg_req.interfaceTag));
                            r = -EINVAL;
                            goto out;
                        }
                    }

                    if (!interfacePriv && (cfg_req.vif_index > 0))   /* Not found and find by vif_index */
                    {
                        if ((interfaceTag = CsrWifiHipInterfaceTagGetReq(priv->hip_handle, cfg_req.vif_index)) == 0xFFFF)
                        {
                            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                "UNIFI_NETDEV_CFG, queried non-existent vif_index (%x)\n",
                                                cfg_req.vif_index));
                            r = -EINVAL;
                            goto out;
                        }
                        else
                        {
                            interfacePriv = priv->interfacePriv[interfaceTag];
                        }
                    }

                    if (!interfacePriv && (cfg_req.name[0] != '\0'))
                    {
                        for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
                        {
                            if (!memcmp(cfg_req.name, priv->netdev[i]->name, sizeof(priv->netdev[i]->name)))
                            {
                                break;
                            }
                        }
                        if (i < CSR_WIFI_MAX_INTERFACES)
                        {
                            if (!(interfacePriv = priv->interfacePriv[i]))
                            {
                                CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                    "UNIFI_NETDEV_CFG, queried non-existent device name(%s)\n",
                                                    cfg_req.name));
                                r = -EINVAL;
                                goto out;
                            }
                        }
                    }

                    if (!interfacePriv && memcmp(cfg_req.macaddr, broadcast_address.a, ETH_ALEN))
                    {
                        for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
                        {
                            if (!memcmp(cfg_req.macaddr, priv->netdev[i]->dev_addr, ETH_ALEN))
                            {
                                break;
                            }
                        }
                        if (i < CSR_WIFI_MAX_INTERFACES)
                        {
                            if (!(interfacePriv = priv->interfacePriv[i]))
                            {
                                CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                    "UNIFI_NETDEV_CFG, queried non-existent address %x-%x-%x-%x-%x-%x\n",
                                                    cfg_req.macaddr[0], cfg_req.macaddr[1], cfg_req.macaddr[2],
                                                    cfg_req.macaddr[3], cfg_req.macaddr[4], cfg_req.macaddr[5]));
                                r = -EINVAL;
                                goto out;
                            }
                        }
                    }

                    if (!interfacePriv)
                    {
                        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                            "UNIFI_NETDEV_CFG, NULL query (no valid fields)\n"));
                        r = -EINVAL;
                        goto out;
                    }

                    /* We have an if_info. Copy stuff and drop through to success */
                    cfg_req.interfaceTag = interfacePriv->InterfaceTag;
                    cfg_req.vif_index = CsrWifiHipVifIndexGetReq(priv->hip_handle, interfacePriv->InterfaceTag);
                    cfg_req.interfaceMode = interfacePriv->interfaceMode;
                    memcpy(cfg_req.macaddr, priv->netdev[interfacePriv->InterfaceTag]->dev_addr, ETH_ALEN);
                    memcpy(cfg_req.name, priv->netdev[interfacePriv->InterfaceTag]->name, IFNAMSIZ);
                    break;
                }

                case 2: /* Add peer operation */
                {
                    CsrWifiRouterCtrlStaInfo staInfo;
                    CsrWifiMacAddress macaddr;
                    CsrUint32 index = 0;

                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                       "%s UNIFI_NETDEV_CFG - Add Peer %d, AID %d\n",
                                       __FUNCTION__, cfg_req.interfaceTag, cfg_req.association_id));

                    if ((cfg_req.interfaceTag >= 0) && (cfg_req.interfaceTag < CSR_WIFI_MAX_INTERFACES))
                    {
                        /* Find by interfaceTag */
                        if (!(interfacePriv = priv->interfacePriv[cfg_req.interfaceTag]))
                        {
                            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                "UNIFI_NETDEV_CFG, configuring non-existent interfaceTag(%d)\n",
                                                cfg_req.interfaceTag));
                            r = -EINVAL;
                            goto out;
                        }
                    }
                    else
                    {
                        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                            "%s: unifi: Add peer to a invalid interfaceTag (%u)\n",
                                            __FUNCTION__, cfg_req.interfaceTag));
                        r = -EINVAL;
                        goto out;
                    }


                    if (interfacePriv->peer_sta_count >= 8)
                    {
                        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                            "UNIFI_NETDEV_CFG, reached max peer count, interfaceTag %d\n",
                                            cfg_req.interfaceTag));
                        r = -EINVAL;
                        goto out;
                    }

                    for (index = 0; index < interfacePriv->peer_sta_count; index++)
                    {
                        if (interfacePriv->peer_sta_info[index].association_id == cfg_req.association_id)
                        {
                            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                                "UNIFI_NETDEV_CFG, peer sta (index=%d, aid=%d) already exists - deleting\n",
                                                index, cfg_req.association_id));
                            if (CsrWifiHipPeerDelReq(priv->hip_handle, cfg_req.interfaceTag, interfacePriv->peer_sta_info[index].peerHandle) != CSR_RESULT_SUCCESS)
                            {
                                CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                    "UNIFI_NETDEV_CFG Failed to del peer aid %d, ifTag %d from PS queues\n",
                                                    cfg_req.association_id, cfg_req.interfaceTag));
                                r = -EINVAL;
                                goto out;
                            }
                            interfacePriv->peer_sta_count--;
                        }
                    }

                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                        "UNIFI_NETDEV_CFG, Adding peer sta (index=%d, aid=%d, addr=%02X:%02X:%02X:%02X:%02X:%02X\n",
                                        index, cfg_req.association_id,
                                        cfg_req.peer_macaddr[0], cfg_req.peer_macaddr[1],
                                        cfg_req.peer_macaddr[2], cfg_req.peer_macaddr[3],
                                        cfg_req.peer_macaddr[4], cfg_req.peer_macaddr[5]));

                    interfacePriv->peer_sta_info[index].association_id = cfg_req.association_id;
                    memcpy(interfacePriv->peer_sta_info[index].peer_macaddr,
                           cfg_req.peer_macaddr, ETH_ALEN);

                    memcpy(cfg_req.name, priv->netdev[interfacePriv->InterfaceTag]->name, IFNAMSIZ);
                    memcpy(cfg_req.macaddr, priv->netdev[interfacePriv->InterfaceTag]->dev_addr, ETH_ALEN);
                    cfg_req.vif_index = CsrWifiHipVifIndexGetReq(priv->hip_handle, interfacePriv->InterfaceTag);
                    cfg_req.interfaceMode = interfacePriv->interfaceMode;

                    /* Store this for use in controlled port api */
                    memcpy(macaddr.a, cfg_req.peer_macaddr, ETH_ALEN);

                    staInfo.wmmOrQosEnabled = TRUE;
                    /* peerHandle not used. When the mode is changed from AP mode or equivalent to another mode, the sta record list is cleared. */
                    if (CsrWifiHipPeerAddReq(priv->hip_handle, cfg_req.interfaceTag, macaddr, cfg_req.association_id, &staInfo, &interfacePriv->peer_sta_info[index].peerHandle) != CSR_RESULT_SUCCESS)
                    {
                        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                            "UNIFI_NETDEV_CFG Failed to add peer aid %d, ifTag %d to PS queues\n",
                                            cfg_req.association_id, cfg_req.interfaceTag));
                        r = -EINVAL;
                        goto out;
                    }
                    interfacePriv->peer_sta_count++;

                    /* Ensure that controlled port is open */
                    CsrWifiHipPortConfigureReq(priv->hip_handle, cfg_req.interfaceTag,
                                               CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN,
                                               CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN,
                                               &macaddr, FALSE);
                    break;
                }

                case 3: /* Delete interface */
                {
                    CsrWifiMacAddress macAddr = {{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}};

                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                       "%s UNIFI_NETDEV_CFG - Delete Interface %d\n",
                                       __FUNCTION__, cfg_req.interfaceTag));

                    if ((cfg_req.interfaceTag >= 0) && (cfg_req.interfaceTag < CSR_WIFI_MAX_INTERFACES))
                    {
                        /* Find by interfaceTag */
                        if (!(interfacePriv = priv->interfacePriv[cfg_req.interfaceTag]))
                        {
                            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                "UNIFI_NETDEV_CFG, Deleting non-existent interfaceTag(%d)\n",
                                                cfg_req.interfaceTag));
                            r = -EINVAL;
                            goto out;
                        }
                    }
                    else
                    {
                        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                            "%s: unifi: Invalid interfaceTag (%u)\n",
                                            __FUNCTION__, cfg_req.interfaceTag));
                        r = -EINVAL;
                        goto out;
                    }

                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                       "%s Try to delete interface %u (%u)\n",
                                       __FUNCTION__, interfacePriv->InterfaceTag, cfg_req.interfaceTag));


                    if ((interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED) ||
                        (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED_STOPPED))
                    {
                        /* Trying to remove existing interfaces */

                        /* Copy in what it used to be, in case someone needs it */
                        memcpy(cfg_req.name, priv->netdev[interfacePriv->InterfaceTag]->name, IFNAMSIZ);
                        cfg_req.vif_index = CsrWifiHipVifIndexGetReq(priv->hip_handle, interfacePriv->InterfaceTag);
                        cfg_req.interfaceMode = interfacePriv->interfaceMode;
                        memcpy(cfg_req.macaddr, priv->netdev[interfacePriv->InterfaceTag]->dev_addr, ETH_ALEN);

                        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                           "Set mode to none and turn off the carrier for(%s)\n",
                                           priv->netdev[interfacePriv->InterfaceTag]->name));

                        /* Primarily stop the existing mode on the interface */
                        CsrWifiHipVifModeSetReq(priv->hip_handle, interfacePriv->InterfaceTag,
                                                CSR_WIFI_ROUTER_CTRL_MODE_NONE, macAddr,
                                                0, 0, 0);
                        interfacePriv->interfaceMode = 0;

                        /* Unregisters the statically created netdevice interface with network sub system */
                        if (CsrWifiHipInterfaceDelReq(priv->hip_handle, interfacePriv->InterfaceTag))
                        {
                            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                "%s: unifi(%d): netdev Unregistering failed\n",
                                                __FUNCTION__, interfacePriv->InterfaceTag));
                            r = -EINVAL;
                            goto out;
                        }
                    }
                    break;
                }

                default:
                    CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                        "UNIFI_NETDEV_CFG, bad operation %d\n",
                                        cfg_req.operation));
                    r = -EINVAL;
                    goto out;
            }
            /* Success for UNIFI_NETDEV_CFG */
            /* Copy to return values */
            if (copy_to_user((void *) arg, (void *) &cfg_req, sizeof(cfg_req)))
            {
                r = -EFAULT;
                goto out;
            }
            break;
        }
#endif
        default:
            r = -EINVAL;
    }

out:
    return (long) r;
} /* unifi_ioctl() */

static unsigned int unifi_poll(struct file *filp, poll_table *wait)
{
    ul_client_t *pcli = (ul_client_t *) filp->private_data;
    unsigned int mask = 0;
    int ready;

    func_enter();

    ready = !list_empty(&pcli->udi_log);

    poll_wait(filp, &pcli->udi_wq, wait);

    if (ready)
    {
        mask |= POLLIN | POLLRDNORM;    /* readable */
    }

    func_exit();

    return mask;
} /* unifi_poll() */

/*
 * ---------------------------------------------------------------------------
 *  udi_set_log_filter
 *
 *      Configure the bit mask that determines which signal primitives are
 *      passed to the logging process.
 *
 *  Arguments:
 *      pcli            Pointer to the client to configure.
 *      udi_filter      Pointer to a unifiio_filter_t containing instructions.
 *
 *  Returns:
 *      None.
 *
 *  Notes:
 *      SigGetFilterPos() returns a 32-bit value that contains an index and a
 *      mask for accessing a signal_filter array. The top 16 bits specify an
 *      index into a signal_filter, the bottom 16 bits specify a mask to
 *      apply.
 * ---------------------------------------------------------------------------
 */
static void udi_set_log_filter(ul_client_t *pcli, unifiio_filter_t *udi_filter)
{
    CsrUint32 filter_pos;
    int i;

    if (udi_filter->action == UfSigFil_AllOn)
    {
        for (i = 0; i < SIG_FILTER_SIZE; i++)
        {
            pcli->signal_filter[i] = 0xFFFF;
        }
    }
    else if (udi_filter->action == UfSigFil_AllOff)
    {
        for (i = 0; i < SIG_FILTER_SIZE; i++)
        {
            pcli->signal_filter[i] = 0;
        }
    }
    else if (udi_filter->action == UfSigFil_SelectOn)
    {
        for (i = 0; i < udi_filter->num_sig_ids; i++)
        {
            filter_pos = SigGetFilterPos(udi_filter->sig_ids[i]);
            if (filter_pos == 0xFFFFFFFF)
            {
                printk(KERN_WARNING
                       "Unrecognised signal id (0x%X) specifed in logging filter\n",
                       udi_filter->sig_ids[i]);
            }
            else
            {
                pcli->signal_filter[filter_pos >> 16] |= (filter_pos & 0xFFFF);
            }
        }
    }
    else if (udi_filter->action == UfSigFil_SelectOff)
    {
        for (i = 0; i < udi_filter->num_sig_ids; i++)
        {
            filter_pos = SigGetFilterPos(udi_filter->sig_ids[i]);
            if (filter_pos == 0xFFFFFFFF)
            {
                printk(KERN_WARNING
                       "Unrecognised signal id (0x%X) specifed in logging filter\n",
                       udi_filter->sig_ids[i]);
            }
            else
            {
                pcli->signal_filter[filter_pos >> 16] &= ~(filter_pos & 0xFFFF);
            }
        }
    }
} /* udi_set_log_filter() */

/*
 * ---------------------------------------------------------------------------
 *  udi_log_event
 *
 *      Callback function to be registered as the UDI hook callback.
 *      Copies the signal content into a new udi_log_t struct and adds
 *      it to the read queue for this UDI client.
 *
 *  Arguments:
 *      pcli            A pointer to the client instance.
 *      signal          Pointer to the received signal.
 *      signal_len      Size of the signal structure in bytes.
 *      bulkdata        Pointers to any associated bulk data.
 *      dir             Direction of the signal. Zero means from host,
 *                      non-zero means to host.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void udi_log_event(ul_client_t *pcli,
                   const u8 *signal, int signal_len,
                   const CsrWifiHipBulkDataParam *bulkdata,
                   int dir)
{
    udi_log_t *logptr;
    u8 *p;
    int i;
    int total_len;
    udi_msg_t *msgptr;
    CsrUint32 filter_pos;
#ifdef OMNICLI_LINUX_EXTRA_LOG
    static volatile unsigned int printk_cpu = UINT_MAX;
    unsigned long long t;
    unsigned long nanosec_rem;
    unsigned long n_1000;
#endif

    func_enter();

    /* Just a sanity check */
    if ((signal == NULL) || (signal_len <= 0))
    {
        return;
    }

#ifdef CSR_NATIVE_LINUX
    uf_native_process_udi_signal(pcli, signal, signal_len, bulkdata, dir);
#endif

    /*
     * Apply the logging filter - only report signals that have their
     * bit set in the filter mask.
     */
    filter_pos = SigGetFilterPos(GET_SIGNAL_ID(signal));

    if ((filter_pos != 0xFFFFFFFF) &&
        ((pcli->signal_filter[filter_pos >> 16] & (filter_pos & 0xFFFF)) == 0))
    {
        /* Signal is not wanted by client */
        return;
    }


    /* Calculate the buffer we need to store signal plus bulk data */
    total_len = signal_len;
    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
    {
        total_len += bulkdata->d[i].data_length;
    }

    /* Allocate log structure plus actual signal. */
    logptr = (udi_log_t *) kmalloc(sizeof(udi_log_t) + total_len, GFP_KERNEL);

    if (logptr == NULL)
    {
        printk(KERN_ERR
               "Failed to allocate %lu bytes for a UDI log record\n",
               (long unsigned int) (sizeof(udi_log_t) + total_len));
        return;
    }

    /* Fill in udi_log struct */
    INIT_LIST_HEAD(&logptr->q);
    msgptr = &logptr->msg;
    msgptr->length = sizeof(udi_msg_t) + total_len;
#ifdef OMNICLI_LINUX_EXTRA_LOG
    t = cpu_clock(printk_cpu);
    nanosec_rem = do_div(t, 1000000000);
    n_1000 = nanosec_rem / 1000;
    msgptr->timestamp = (t << 10) | ((unsigned long) (n_1000 >> 10) & 0x3ff);
#else
    msgptr->timestamp = jiffies_to_msecs(jiffies);
#endif
    msgptr->direction = dir;
    msgptr->signal_length = signal_len;

    /* Copy signal and bulk data to the log */
    p = (u8 *) (msgptr + 1);
    memcpy(p, signal, signal_len);
    p += signal_len;

    /* Append any bulk data */
    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
    {
        int len = bulkdata->d[i].data_length;

        /*
         * Len here might not be the same as the length in the bulk data slot.
         * The slot length will always be even, but len could be odd.
         */
        if (len > 0)
        {
            if (bulkdata->d[i].os_data_ptr)
            {
                memcpy(p, bulkdata->d[i].os_data_ptr, len);
            }
            else
            {
                memset(p, 0, len);
            }
            p += len;
        }
    }

    /* Add to tail of log queue */
    if (down_interruptible(&pcli->udi_sem))
    {
        printk(KERN_WARNING "udi_log_event_q: Failed to get udi sem\n");
        kfree(logptr);
        func_exit();
        return;
    }
    list_add_tail(&logptr->q, &pcli->udi_log);
    up(&pcli->udi_sem);

    /* Wake any waiting user process */
    wake_up_interruptible(&pcli->udi_wq);

    func_exit();
} /* udi_log_event() */

#ifdef CSR_SME_USERSPACE
int uf_sme_queue_message(os_linux_priv_t *priv, u8 *buffer, int length)
{
    udi_log_t *logptr;
    udi_msg_t *msgptr;
    u8 *p;

    func_enter();

    /* Just a sanity check */
    if ((buffer == NULL) || (length <= 0))
    {
        return -EINVAL;
    }

    /* Allocate log structure plus actual signal. */
    logptr = (udi_log_t *) kmalloc(sizeof(udi_log_t) + length, GFP_ATOMIC);
    if (logptr == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to allocate %d bytes for an SME message\n",
                               priv ? priv->instance : 0,
                               sizeof(udi_log_t) + length));
        CsrPmemFree(buffer);
        return -ENOMEM;
    }

    /* Fill in udi_log struct */
    INIT_LIST_HEAD(&logptr->q);
    msgptr = &logptr->msg;
    msgptr->length = sizeof(udi_msg_t) + length;
    msgptr->signal_length = length;

    /* Copy signal and bulk data to the log */
    p = (u8 *) (msgptr + 1);
    memcpy(p, buffer, length);

    /* Add to tail of log queue */
    down(&udi_mutex);
    if (priv->sme_cli == NULL)
    {
        kfree(logptr);
        CsrPmemFree(buffer);
        up(&udi_mutex);
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Message for the SME dropped, SME has gone away\n", priv ? priv->instance : 0));
        return 0;
    }

    down(&priv->sme_cli->udi_sem);
    list_add_tail(&logptr->q, &priv->sme_cli->udi_log);
    up(&priv->sme_cli->udi_sem);

    /* Wake any waiting user process */
    wake_up_interruptible(&priv->sme_cli->udi_wq);
    up(&udi_mutex);

    /* It is our responsibility to free the buffer allocated in build_packed_*() */
    CsrPmemFree(buffer);

    func_exit();

    return 0;
} /* uf_sme_queue_message() */

#endif


#define UF_DEVICE_CREATE(_class, _parent, _devno, _priv, _fmt, _args)       \
    device_create(_class, _parent, _devno, _priv, _fmt, _args)

/*
 ****************************************************************************
 *
 *      Driver instantiation
 *
 ****************************************************************************
 */
static struct file_operations unifi_fops =
{
    .owner = THIS_MODULE,
    .open = unifi_open,
    .release = unifi_release,
    .read = unifi_read,
    .write = unifi_write,
    .unlocked_ioctl = unifi_ioctl,
    .poll = unifi_poll,
};

#define UF_DEVICE_CREATE(_class, _parent, _devno, _priv, _fmt, _args)       \
    device_create(_class, _parent, _devno, _priv, _fmt, _args)


static dev_t unifi_first_devno;
static struct class *unifi_class;


int uf_create_device_nodes(os_linux_priv_t *priv, int bus_id)
{
    dev_t devno;
    int r;

    cdev_init(&priv->unifi_cdev, &unifi_fops);

    /* cdev_init() should set the cdev owner, but it does not */
    priv->unifi_cdev.owner = THIS_MODULE;

    devno = MKDEV(MAJOR(unifi_first_devno),
                  MINOR(unifi_first_devno) + (bus_id * 2));
    r = cdev_add(&priv->unifi_cdev, devno, 1);
    if (r)
    {
        return r;
    }

#ifdef SDIO_EXPORTS_STRUCT_DEVICE
    if (!UF_DEVICE_CREATE(unifi_class, priv->unifi_device,
                          devno, priv, "unifi%d", bus_id))
    {
#else
    priv->unifi_device = UF_DEVICE_CREATE(unifi_class, NULL,
                                          devno, priv, "unifi%d", bus_id);
    if (priv->unifi_device == NULL)
    {
#endif /* SDIO_EXPORTS_STRUCT_DEVICE */

        cdev_del(&priv->unifi_cdev);
        return -EINVAL;
    }

    cdev_init(&priv->unifiudi_cdev, &unifi_fops);

    /* cdev_init() should set the cdev owner, but it does not */
    priv->unifiudi_cdev.owner = THIS_MODULE;

    devno = MKDEV(MAJOR(unifi_first_devno),
                  MINOR(unifi_first_devno) + (bus_id * 2) + 1);
    r = cdev_add(&priv->unifiudi_cdev, devno, 1);
    if (r)
    {
        device_destroy(unifi_class, priv->unifi_cdev.dev);
        cdev_del(&priv->unifi_cdev);
        return r;
    }

    if (!UF_DEVICE_CREATE(unifi_class,
#ifdef SDIO_EXPORTS_STRUCT_DEVICE
                          priv->unifi_device,
#else
                          NULL,
#endif /* SDIO_EXPORTS_STRUCT_DEVICE */
                          devno, priv, "unifiudi%d", bus_id))
    {
        device_destroy(unifi_class, priv->unifi_cdev.dev);
        cdev_del(&priv->unifiudi_cdev);
        cdev_del(&priv->unifi_cdev);
        return -EINVAL;
    }

    return 0;
}

void uf_destroy_device_nodes(os_linux_priv_t *priv)
{
    device_destroy(unifi_class, priv->unifiudi_cdev.dev);
    device_destroy(unifi_class, priv->unifi_cdev.dev);
    cdev_del(&priv->unifiudi_cdev);
    cdev_del(&priv->unifi_cdev);
}

/*
 * ----------------------------------------------------------------
 *  uf_create_debug_device
 *
 *      Allocates device numbers for unifi character device nodes
 *      and creates a unifi class in sysfs
 *
 * Arguments:
 *  fops          Pointer to the char device operations structure.
 *
 * Returns:
 *      0 on success, -ve error code on error.
 * ----------------------------------------------------------------
 */
static int uf_create_debug_device(struct file_operations *fops)
{
    int ret;

    /* Allocate two device numbers for each device. */
    ret = alloc_chrdev_region(&unifi_first_devno, 0, MAX_UNIFI_DEVS * 2, UNIFI_NAME);
    if (ret)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : Failed to add alloc dev numbers: %d\n", ret));
        return ret;
    }

    /* Create a UniFi class */
    unifi_class = class_create(THIS_MODULE, UNIFI_NAME);
    if (IS_ERR(unifi_class))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : Failed to create UniFi class\n"));

        /* Release device numbers */
        unregister_chrdev_region(unifi_first_devno, MAX_UNIFI_DEVS * 2);
        unifi_first_devno = 0;
        return -EINVAL;
    }

    return 0;
} /* uf_create_debug_device() */

/*
 * ----------------------------------------------------------------
 *  uf_remove_debug_device
 *
 *      Destroys the unifi class and releases the allocated
 *      device numbers for unifi character device nodes.
 *
 * Arguments:
 *
 * Returns:
 * ----------------------------------------------------------------
 */
static void uf_remove_debug_device(void)
{
    /* Destroy the UniFi class */
    class_destroy(unifi_class);

    /* Release device numbers */
    unregister_chrdev_region(unifi_first_devno, MAX_UNIFI_DEVS * 2);
    unifi_first_devno = 0;
} /* uf_remove_debug_device() */

#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
/*
 * Helpers to store a Hydra service object
 */
#define UNIFI_MAX_SVCS  16

struct unifi_hydra_service
{
    struct hydra_service *service;
    void *uf_priv;
    wait_queue_head_t event;
};


static struct unifi_hydra_service uf_hysvcs[UNIFI_MAX_SVCS];
static spinlock_t uf_hysv_lock;
wait_queue_head_t uf_hydra_wq;

/* Add a Hydra WLAN service instance */
static void uf_add_hydra_service(struct hydra_service *hysv)
{
    unsigned long state;
    int sid = hydra_service_get_sid(hysv);

    if (sid >= UNIFI_MAX_SVCS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: add sid range! %d\n", sid));
        return;
    }

    spin_lock_irqsave(&uf_hysv_lock, state);
    uf_hysvcs[sid].service = hysv;
    spin_unlock_irqrestore(&uf_hysv_lock, state);
}

/* Remove a Hydra WLAN service instance */
static void uf_remove_hydra_service(struct hydra_service *hysv)
{
    unsigned long state;
    int sid = hydra_service_get_sid(hysv);

    if (sid >= UNIFI_MAX_SVCS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: remove sid range! %d\n", sid));
        return;
    }

    spin_lock_irqsave(&uf_hysv_lock, state);
    uf_hysvcs[sid].service = NULL;
    uf_hysvcs[sid].uf_priv = NULL;
    spin_unlock_irqrestore(&uf_hysv_lock, state);
}

/* Claim a Hydra service for use by this priv */
static struct hydra_service *uf_get_hydra_service(os_linux_priv_t *priv)
{
    unsigned long state;
    int i;
    struct hydra_service *res = NULL;

    spin_lock_irqsave(&uf_hysv_lock, state);

    for (i = 0; i < UNIFI_MAX_SVCS; i++)
    {
        /* Search for a service already claimed by this priv */
        if ((uf_hysvcs[i].service != NULL) && (uf_hysvcs[i].uf_priv == priv))
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                "unifi%d: Using Hydra S%d\n", priv->instance, hydra_service_get_sid(uf_hysvcs[i].service)));
            res = uf_hysvcs[i].service;
            goto done;
        }
    }
    for (i = 0; i < UNIFI_MAX_SVCS; i++)
    {
        /* Find an unclaimed service */
        if ((uf_hysvcs[i].service != NULL) && (uf_hysvcs[i].uf_priv == NULL))
        {
            uf_hysvcs[i].uf_priv = priv;
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                "unifi%d: Claimed Hydra S%d\n", priv->instance, hydra_service_get_sid(uf_hysvcs[i].service)));
            res = uf_hysvcs[i].service;
            goto done;
        }
    }

done:
    spin_unlock_irqrestore(&uf_hysv_lock, state);
    return res;
}

/* Find a Hydra service already claimed by this priv */
static struct hydra_service *uf_find_hydra_service(os_linux_priv_t *priv)
{
    unsigned long state;
    int i;
    struct hydra_service *res = NULL;

    spin_lock_irqsave(&uf_hysv_lock, state);

    for (i = 0; i < UNIFI_MAX_SVCS; i++)
    {
        /* Search for a service already claimed by this priv */
        if ((uf_hysvcs[i].service != NULL) && (uf_hysvcs[i].uf_priv == priv))
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4,
                                "unifi%d: Found Hydra S%d\n", priv->instance, hydra_service_get_sid(uf_hysvcs[i].service)));

            res = uf_hysvcs[i].service;
            break;
        }
    }

    spin_unlock_irqrestore(&uf_hysv_lock, state);
    return res;
}

/* Find a driver priv used by this service */
static os_linux_priv_t *uf_find_hydra_priv(struct hydra_service *svc)
{
    unsigned long state;
    int i;
    os_linux_priv_t *res = NULL;

    spin_lock_irqsave(&uf_hysv_lock, state);

    for (i = 0; i < UNIFI_MAX_SVCS; i++)
    {
        /* Search for the os_linux priv related to this service */
        if ((uf_hysvcs[i].service != NULL) && (uf_hysvcs[i].service == svc))
        {
            res = uf_hysvcs[i].uf_priv;

            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4,
                                "unifi%d: Found Hydra S%d priv=%p\n", res ? res->instance : 0, hydra_service_get_sid(svc), res));

            break;
        }
    }

    spin_unlock_irqrestore(&uf_hysv_lock, state);
    return res;
}

/*
 * ----------------------------------------------------------------
 *  wlan_hydra_service_entry
 *
 *  The Hydra Driver has found a WLAN service available on chip.
 *
 * Arguments:
 *      service     - context for the service driver
 *
 * Returns:
 * ----------------------------------------------------------------
 */
static void wlan_hydra_service_entry(struct hydra_service *service)
{
    int sid = hydra_service_get_sid(service);

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                       "unifi : wlan-service: S%d entry\n", sid));

    uf_add_hydra_service(service);
    wake_up_interruptible(&uf_hydra_wq);
}

/*
 * ----------------------------------------------------------------
 *  wlan_hydra_service_exit
 *
 *  The Hydra Driver has found that the WLAN service is now unavailable.
 *
 *  The service driver should close down the service.
 *
 * Arguments:
 *      service     - context for the service driver
 *
 * Returns:
 * ----------------------------------------------------------------
 */
static void wlan_hydra_service_exit(struct hydra_service *service)
{
    int sid = hydra_service_get_sid(service);
    os_linux_priv_t *priv = uf_find_hydra_priv(service);

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                       "unifi%d: wlan-service: S%d exit\n", priv ? priv->instance : 0, sid));


    /* Stop the WLAN subsystem */
#ifdef CSR_SME_USERSPACE
    if (priv)
    {
        /* Power-down when config client closes - if the driver hasn't been de-init'ed already */
        if ((priv->state != OS_LINUX_STATE_IDLE) && (priv->state != OS_LINUX_STATE_ACTIVATED))
        {
            CsrWifiRouterCtrlWifiOffReq req = {{CSR_WIFI_ROUTER_CTRL_HIP_REQ, 0, 0, 0, NULL}};
            CsrWifiRouterCtrlWifiOffReqHandler(priv, &req.common);
        }

        uf_sme_deinit(priv);
        uf_sme_cancel_request(priv, 0);
    }
#else
    hydra_service_disable(service);
#endif
    uf_remove_hydra_service(service);
}

/*
 * ----------------------------------------------------------------
 *  wlan_hydra_service_event
 *
 *
 *
 * Arguments:
 *      service     - context for the service driver
 *
 * Returns:
 * ----------------------------------------------------------------
 */
static void wlan_hydra_service_event(struct hydra_service *service, enum hydra_service_event_type event)
{
    int sid = hydra_service_get_sid(service);

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                       "unifi : wlan_hydra_service_event: sid=%d event=%d\n", sid, event));
}

/* Hydra Service Driver definition */
struct hydra_service_driver wlan_hydra_service_driver =
{
    .name = "wlan-service",
    .class = HYDRA_SERVICE_CLASS_WLAN,
    .entry = wlan_hydra_service_entry,
    .exit = wlan_hydra_service_exit,
    .event = wlan_hydra_service_event,
};

/* Utility functions to start/stop the Hydra WLAN service */
CsrResult uf_start_hydra_wlan_service(os_linux_priv_t *linux_priv)
{
    int r;
    struct hydra_service *hysv = uf_get_hydra_service(linux_priv);

    if (linux_priv->hydra_service_enabled)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: Hydra service already started\n", linux_priv->instance));
        return CSR_RESULT_SUCCESS;
    }

    if (hysv == NULL)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: Awaiting Hydra service event\n", linux_priv->instance));

        r = wait_event_interruptible_timeout(uf_hydra_wq, uf_get_hydra_service(linux_priv), msecs_to_jiffies(20000));
        if (r == 0)
        {
            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                "unifi%d: Wait for Hydra service event timed out\n", linux_priv->instance));
            return CSR_RESULT_FAILURE;
        }
        else if (r == -ERESTARTSYS)
        {
            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                "unifi%d: Wait for Hydra service event terminated by signal\n", linux_priv->instance));
            return CSR_RESULT_FAILURE;
        }
        else
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: Got Hydra service event\n", linux_priv->instance));

            /* Object is now available */
            hysv = uf_get_hydra_service(linux_priv);
            if (hysv == NULL)
            {
                CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                    "unifi%d: No service object\n", linux_priv->instance));
                return CSR_RESULT_FAILURE;
            }
        }
    }

    /* Start the firmware */
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                       "unifi%d: Starting Hydra service S%d\n",  linux_priv->instance, hydra_service_get_sid(hysv)));

    r = hydra_service_enable(hysv);
    if (r)
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "Failed to start WLAN service S%d (%d)\n", hydra_service_get_sid(hysv), r));
        return CSR_RESULT_FAILURE;
    }

    linux_priv->hydra_service_enabled = TRUE;

    return CSR_RESULT_SUCCESS;
}

static void _uf_stop_hydra_wlan_service(os_linux_priv_t *linux_priv)
{
    struct hydra_service *hysv = uf_find_hydra_service(linux_priv);


    if (hysv && linux_priv->hydra_service_enabled)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: Stopping Hydra service S%d\n",
                           linux_priv->instance, hydra_service_get_sid(hysv)));

        hydra_service_disable(hysv);
    }
    linux_priv->hydra_service_enabled = FALSE;
}

/*
 * Workaround for hydra_service_disable terminating early, leaving the Hydra
 * drivers out of sync if the userspace helper is terminated by a signal.
 */
struct hysv_work_info
{
    struct workqueue_struct *workqueue;
    struct work_struct work;
    wait_queue_head_t wait;
    os_linux_priv_t *linux_priv;
};

static void do_stop_hydra_wlan_service(struct work_struct *work)
{
    struct hysv_work_info *pwq = container_of(work, struct hysv_work_info, work);

    _uf_stop_hydra_wlan_service(pwq->linux_priv);
    wake_up(&pwq->wait);
}

void uf_stop_hydra_wlan_service(os_linux_priv_t *linux_priv)
{
    struct hysv_work_info wq;

    wq.linux_priv = linux_priv;

    init_waitqueue_head(&wq.wait);
    wq.workqueue = create_singlethread_workqueue("uf_hysv_work");
    INIT_WORK(&wq.work, do_stop_hydra_wlan_service);

    /* Stop the service in a workqueue, use a non-interruptible wait */
    queue_work(wq.workqueue, &wq.work);
    wait_event(wq.wait, linux_priv->hydra_service_enabled == FALSE);

    destroy_workqueue(wq.workqueue);
}

/* Hydra HIF driver callbacks. Most are stubs.
 */
static int wlan_hif_sdio_setup(struct hydra_sdio_func *func, struct hydra_service *service,
                               const uint8_t *hif_info, size_t hif_info_len)
{
    /* Indicates that the SDIO HIF function is ready for use */
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                       "hif: F%d: setup S%d", func->function->sdioId.sdioFunction, hydra_service_get_sid(service)));
    ++func->users;

    return 0;
}

static void wlan_hif_sdio_shutdown(struct hydra_sdio_func *func, struct hydra_service *service)
{
    /* Indicates that the SDIO HIF function is shut down */

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                       "hif: F%d: shutdown S%d", func->function->sdioId.sdioFunction, hydra_service_get_sid(service)));
    --func->users;
}

static bool wlan_hif_sdio_enqueue(struct hydra_hif *hif, struct hydra_q *q,
                                  struct hydra_mbuf *mbuf)
{
    return false;
}

static bool wlan_hif_sdio_cancel(struct hydra_hif *hif, struct hydra_q *q,
                                 struct hydra_mbuf *mbuf, enum hydra_mbuf_status status)
{
    return false;
}

static int wlan_hif_sdio_map_q(struct hydra_sdio_func *func, struct hydra_q *q)
{
    int ret;

    ret = hydra_sdio_func_map_q(func, q);
    if (ret < 0)
    {
        return ret;
    }

    q->enqueue = wlan_hif_sdio_enqueue;
    q->cancel = wlan_hif_sdio_cancel;

    return 0;
}

static void wlan_hif_sdio_unmap_q(struct hydra_sdio_func *func, struct hydra_q *q)
{
    hydra_sdio_func_unmap_q(func, q);
}

/* Register with the Hydra driver as an SDIO HIF function */
void CsrWifiHipHydraRegister(void *osLayerContext, void *sdioContext)
{
    CsrSdioFunction *function = (CsrSdioFunction *) sdioContext;
    struct hydra_sdio_func *func;
    int ret;
    os_linux_priv_t *priv = osLayerContext;

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                       "hif: F%d: inserted", function->sdioId.sdioFunction));

    func = hydra_sdio_func_create(function);
    if (func == NULL)
    {
        goto err;
    }

    func->is_curator = FALSE;
    func->setup = wlan_hif_sdio_setup;
    func->shutdown = wlan_hif_sdio_shutdown;
    func->map_q = wlan_hif_sdio_map_q;
    func->unmap_q = wlan_hif_sdio_unmap_q;

    func->interrupt_handler = NULL;  /* Interrupts managed locally */

    priv->hyfn = func;

    ret = hydra_sdio_card_add_func(func);
    if (ret < 0)
    {
        goto err;
    }

    return;

err:
    CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                        "hif: F%d: inserted error", func->function->sdioId.sdioFunction));
    kfree(func);
}

/* Unregister with the Hydra driver as an SDIO HIF function */
void CsrWifiHipHydraUnregister(void *osLayerContext, void *sdioContext)
{
    CsrSdioFunction *function = (CsrSdioFunction *) sdioContext;
    os_linux_priv_t *priv = osLayerContext;
    struct hydra_sdio_func *func = NULL;

    func = priv->hyfn;
    if (!func)
    {
        return;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                       "hif: F%d: removed", function->sdioId.sdioFunction));

    hydra_sdio_card_del_func(func);

    /* The HIF callbacks must be cleared as the func object can persist
     * (due to refcount) after the HIF module has been unloaded.
     */
    func->map_q = NULL;
    func->unmap_q = NULL;
    func->setup = NULL;
    func->shutdown = NULL;

    hydra_sdio_func_destroy(func);
}

#endif  /* CSR_WIFI_DRIVER_HYDRA */

#include <linux/module.h>       /* Specifically, a module */
#include <linux/kernel.h>       /* We're doing kernel work */
#include <linux/proc_fs.h>      /* Necessary because we use the proc fs */
#include <asm/uaccess.h>        /* for copy_from_user */


/*
 * ---------------------------------------------------------------------------
 *
 *      Module loading.
 *
 * ---------------------------------------------------------------------------
 */
int __init unifi_load(void)
{
    int r;
    CsrResult result;
#ifdef CSR_NATIVE_LINUX
    int i;
#endif

    printk("UniFi SDIO Driver: %s %s %s\n",
           CSR_WIFI_VERSION,
           __DATE__, __TIME__);

#ifdef CSR_WIFI_DRIVER_HYDRA
#ifdef CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
    printk("Hydra support, using Hydra drivers\n"); /* Hydra drivers in auto mode */
#else
    printk("Hydra support, manual mode\n"); /* Hydra drivers in manual mode */
#endif
#endif

#ifdef CSR_SME_USERSPACE
#ifdef CSR_SUPPORT_WEXT
    printk("CSR SME with WEXT support\n");
#else
    printk("CSR SME no WEXT support\n");
#endif /* CSR_SUPPORT_WEXT */
#endif /* CSR_SME_USERSPACE */

#ifdef CSR_NATIVE_LINUX
#ifdef CSR_SUPPORT_WEXT
#error WEXT unsupported in the native driver
#endif
    printk("CSR native no WEXT support\n");
#endif

#ifdef UNIFI_DEBUG
#ifdef CSR_LOG_ENABLE
    init_log_text(trace_levels, unifi_debug);
    CsrLogTextRegister2(&CsrWifiOsLto, "OS", CSR_ARRAY_SIZE(subOrigins), subOrigins);
#endif
#endif

    printk("Kernel %d.%d.%d\n",
           ((LINUX_VERSION_CODE) >> 16) & 0xff,
           ((LINUX_VERSION_CODE) >> 8) & 0xff,
           (LINUX_VERSION_CODE) & 0xff);

#ifdef CSR_NATIVE_LINUX
    config_params.zero_copy = zero_copy_mode;
    for (i = 0; i < CSR_WIFI_HIP_PARAM_SLOT_COUNT; i++)
    {
        config_params.mparam_slot_count[i] = mparam_slot_count[i];
    }
    config_params.poll_period = (CsrUint32) poll_period;
    config_params.tx_force_pull_mode = (CsrUint32) tx_force_pull_mode;
    config_params.tx_window_segment_size = (CsrUint32) tx_window_segment_size;
#endif

    /*
     * Instantiate the /dev/unifi* device nodes.
     * We must do this before registering with the SDIO driver because it
     * will immediately call the "insert" callback if the card is
     * already present.
     */
    r = uf_create_debug_device(&unifi_fops);
    if (r)
    {
        return r;
    }

#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
    /* Register as a Hydra Service Driver so that the chip's WLAN service
     * can be started.
     */
    init_waitqueue_head(&uf_hydra_wq);
    spin_lock_init(&uf_hysv_lock);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF,
                        "unifi : Registering Hydra service driver\n"));

    r = hydra_service_driver_register(&wlan_hydra_service_driver);
    if (r)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi: service driver registration failed (%d)\n", r));
        uf_remove_debug_device();
        return r;
    }
#endif
    /* Now activate HAL, which will register with the SDIO driver etc. */
    result = CsrWifiHipActivateReq();
    if (result != CSR_RESULT_SUCCESS)
    {
        uf_remove_debug_device();
        return result;
    }

    if (sdio_block_size > -1)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi : sdio_block_size %d\n", sdio_block_size));
    }

    if (sdio_byte_mode)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi : sdio_byte_mode\n"));
    }
    if (disable_power_control)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi : disable_power_control\n"));
    }

    if (disable_hw_reset)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi : disable_hw_reset\n"));
    }

    if (run_bh_once != -1)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi : run_bh_once %d\n", run_bh_once));
    }

    return 0;
} /* unifi_load() */

void CsrWifiHipInsertedInd(void *hipHandle)
{
    CsrResult r;
    CsrUint32 instanceNumber;

    r = uf_add_instance(&instanceNumber);
    if (r != CSR_RESULT_SUCCESS)
    {
        printk("CsrWifiHipInsertedInd: Failed to add os_linux instance (%u)\n", instanceNumber);
        CsrWifiHipInsertedRes(hipHandle, NULL, CSR_RESULT_FAILURE, 0);
        return;
    }

    r = register_unifi_linux(hipHandle, instanceNumber);
    if (r != CSR_RESULT_SUCCESS)
    {
        uf_release_instance(instanceNumber);
        CsrWifiHipInsertedRes(hipHandle, NULL, CSR_RESULT_FAILURE, 0);
        printk("UniFi SDIO Driver: Failed register Linux devices (%u)\n", r);
        return;
    }

    CsrWifiHipInsertedRes(hipHandle, (void *) uf_find_instance(instanceNumber), CSR_RESULT_SUCCESS, instanceNumber);
}

void CsrWifiHipRemovedInd(void *osLayerContext, CsrUint32 instance)
{
    os_linux_priv_t *os_linux = uf_find_instance(instance);
    if (os_linux != NULL)
    {
        unregister_unifi_linux(instance);
    }
}

void __exit unifi_unload(void)
{
    CsrResult result;

    /* Deactivate HAL (and unregister SDIO). */
    result = CsrWifiHipDeactivateReq();
    if (result != CSR_RESULT_SUCCESS)
    {
        printk("Failed to deactivate HAL (%u)\n", result);
    }

#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                        "unifi : Unregistering Hydra service driver\n"));

    hydra_service_driver_unregister(&wlan_hydra_service_driver);
#endif

    uf_remove_debug_device();

#ifdef UNIFI_DEBUG
#ifdef CSR_LOG_ENABLE
    deinit_log_text();
#endif
#endif
    printk("UniFi SDIO Driver: Unloaded\n");
} /* unifi_unload() */

module_init(unifi_load);
module_exit(unifi_unload);

MODULE_DESCRIPTION("UniFi Device driver");
MODULE_AUTHOR("Cambridge Silicon Radio Ltd.");
MODULE_LICENSE("GPL and additional rights");
