/*
 * Copyright (C) 2012-2013 CETITEC GmbH. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <linux/module.h>
#include <linux/dma-mapping.h>

#include "mxc_mlb_ext.h"

enum MLB_CTYPE {
	MLB_CTYPE_SYNC,
	MLB_CTYPE_CTRL,
	MLB_CTYPE_ASYNC,
	MLB_CTYPE_ISOC,
};

static u32 mlb_sync_channel_addresses[MLB_MAX_SYNC_DEVICES] = {
/*       Tx           Rx */
	((8 << 16) | 30),
	((9 << 16) | 30),
	((10 << 16) | 30),
	((11 << 16) | 30),
	((16 << 16) | 30),
	((30 << 16) | 17)
};

static struct mlb_data *drvdata;

static void (*alsa_capture_callback)(u32 *, struct snd_pcm_substream *);
static void (*alsa_playback_callback) (u32 *, struct snd_pcm_substream *);

struct syncsound_substream_info {
	enum channelmode mode;
	struct snd_pcm_substream *substream;
};

static struct syncsound_substream_info
	syncsound_substreams[MLB_MAX_SYNC_DEVICES];

#define SYNCINDEX(syncminor) (syncminor - MLB_STATIC_MINOR_DEVS)
#define SYNCMINOR(syncindex) (MLB_STATIC_MINOR_DEVS + syncindex)


/* called in mxc_mlb150_probe() (i.e. device init) */
void ext_set_drvdata(struct mlb_data *ext_drvdata)
{
	drvdata = ext_drvdata;
}

/* called on startup and irq */
void ext_start_rx(unsigned int minor, unsigned int channel_type, u32 *rpos)
{
	struct snd_pcm_substream *substream = NULL;

	if (SYNCINDEX(minor) >= 0)
		substream = syncsound_substreams[SYNCINDEX(minor)].substream;

	if ((channel_type == MLB_CTYPE_SYNC) && (substream != NULL))
		alsa_capture_callback(rpos, substream);
}

/* called on irq */
void ext_start_tx(unsigned int minor, unsigned int channel_type)
{
	struct snd_pcm_substream *substream = NULL;
	u32 phy_addr = 0;

	if (SYNCINDEX(minor) >= 0)
		substream = syncsound_substreams[SYNCINDEX(minor)].substream;

	if ((channel_type == MLB_CTYPE_SYNC) && (substream != NULL)) {
		alsa_playback_callback(&phy_addr, substream);
		if (phy_addr != 0)
			mlb_start_tx(drvdata, minor, phy_addr);
	}
}

int
register_syncsound_callbacks(void (*playback_callback) (u32 *,
							struct snd_pcm_substream
							*),
			     void (*capture_callback) (u32 *,
							struct snd_pcm_substream
							*))
{
	if ((alsa_playback_callback != NULL) ||
	    (alsa_capture_callback != NULL)) {
		pr_info
		    ("mxc_mlb: alsa callbacks already registered\n");
		return -1;
	}

	alsa_playback_callback = playback_callback;
	alsa_capture_callback = capture_callback;

	return 0;
}
EXPORT_SYMBOL(register_syncsound_callbacks);

void unregister_syncsound_callbacks(void)
{
	alsa_playback_callback = NULL;
	alsa_capture_callback = NULL;
}
EXPORT_SYMBOL(unregister_syncsound_callbacks);

int register_syncsound_substream(int sync_device_index,
				 struct snd_pcm_substream *substream)
{
	if (syncsound_substreams[sync_device_index].substream) {
		pr_info
		    ("mxc_mlb: alsa substream %d already registered\n",
		     sync_device_index);
	}
	syncsound_substreams[sync_device_index].substream = substream;
	return 0;
}
EXPORT_SYMBOL(register_syncsound_substream);

int unregister_syncsound_substream(int sync_device_index)
{
	syncsound_substreams[sync_device_index].substream = NULL;
	return 0;
}
EXPORT_SYMBOL(unregister_syncsound_substream);

int syncsound_open(int sync_device_index, enum channelmode mode,
			u8 **buf_virt, u32 *buf_phys)
{
	int ret = mxc_mlb150_do_open(drvdata, SYNCMINOR(sync_device_index),
				 buf_virt, buf_phys);
	/* zero signals success, a negative number failure */
	if (ret == 0) {
		syncsound_substreams[sync_device_index].mode = mode;
		mxc_mlb150_chan_setaddr(drvdata, SYNCMINOR(sync_device_index),
			mlb_sync_channel_addresses[sync_device_index]);
	}
	return ret;
}
EXPORT_SYMBOL(syncsound_open);

int syncsound_release(int sync_device_index)
{
	return mxc_mlb150_do_release(drvdata, SYNCMINOR(sync_device_index),
				syncsound_substreams[sync_device_index].mode);
}
EXPORT_SYMBOL(syncsound_release);

int syncsound_chan_startup(int sync_device_index, unsigned int audio_channels)
{
	return mxc_mlb150_chan_startup(drvdata, SYNCMINOR(sync_device_index),
				syncsound_substreams[sync_device_index].mode,
				audio_channels);
}
EXPORT_SYMBOL(syncsound_chan_startup);

int syncsound_start_playback(int sync_device_index, u32 phy_addr)
{
	struct snd_pcm_substream *substream =
		syncsound_substreams[sync_device_index].substream;
	if (substream == NULL)
		return -ENODEV;
	if (!phy_addr)
		return -EFAULT;
	mlb_start_tx(drvdata, SYNCMINOR(sync_device_index), phy_addr);
	return 0;
}
EXPORT_SYMBOL(syncsound_start_playback);

int syncsound_chan_shutdown(int sync_device_index)
{
	return mxc_mlb150_chan_shutdown(drvdata, SYNCMINOR(sync_device_index),
				syncsound_substreams[sync_device_index].mode);
}
EXPORT_SYMBOL(syncsound_chan_shutdown);

void *alloc_ext_mem(size_t buf_size, dma_addr_t *dma_handle)
{
	if (!drvdata)
		return NULL;

	return dma_alloc_coherent(drvdata->dev, buf_size,
				  dma_handle, GFP_KERNEL);
}
EXPORT_SYMBOL(alloc_ext_mem);

int mmap_ext_mem(struct vm_area_struct *vma, void *cpu_addr,
		 dma_addr_t dma_addr, size_t size)
{
	if (!drvdata)
		return -EINVAL;
	return dma_mmap_coherent(drvdata->dev, vma,
				 cpu_addr, dma_addr, size);
}
EXPORT_SYMBOL(mmap_ext_mem);

void free_ext_mem(size_t buf_size, void *buf, dma_addr_t dma_handle)
{
	if (!drvdata)
		return;

	dma_free_coherent(drvdata->dev, buf_size, buf, dma_handle);
}
EXPORT_SYMBOL(free_ext_mem);

u32 syncsound_get_num_devices(void)
{
	return number_sync_channels;
}
EXPORT_SYMBOL(syncsound_get_num_devices);
