/*
 * Copyright (C) 2011-2012 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/* don't use IRAM on imx6 */
#ifdef MLB_USE_IRAM
#undef MLB_USE_IRAM
#endif

#include <linux/slab.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/cdev.h>
#include <linux/clk.h>
#include <linux/interrupt.h>
#include <linux/errno.h>
#include <linux/platform_device.h>
#include <linux/regulator/consumer.h>
#include <linux/uaccess.h>

#ifdef MLB_USE_IRAM
#include <linux/iram_alloc.h>
#else
#include <linux/dma-mapping.h>
#endif

#include <linux/fsl_devices.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
#include <linux/sched.h>

#include <linux/of_device.h>
#include <linux/of_platform.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>

#include "mxc_mlb.h"
#include "mxc_mlb_ext.h"

#include <linux/clkdev.h>

#include <linux/jiffies.h>
#define DRIVER_NAME "mlb150"

/*!
 * MLB module memory map registers define
 */
#define REG_MLBC0		0x0
#define MLBC0_MLBEN		(0x1)
#define MLBC0_MLBCLK_MASK	(0x7 << 2)
#define MLBC0_MLBCLK_SHIFT	(2)
#define MLBC0_MLBPEN		(0x1 << 5)
#define MLBC0_MLBLK		(0x1 << 7)
#define MLBC0_ASYRETRY		(0x1 << 12)
#define MLBC0_CTLRETRY		(0x1 << 12)
#define MLBC0_FCNT_MASK		(0x7 << 15)
#define MLBC0_FCNT_SHIFT	(15)

#define REG_MLBPC0		0x8
#define MLBPC0_MCLKHYS		(0x1 << 11)

#define REG_MS0			0xC
#define REG_MS1			0x14

#define REG_MSS			0x20
#define MSS_RSTSYSCMD		(0x1)
#define MSS_LKSYSCMD		(0x1 << 1)
#define MSS_ULKSYSCMD		(0x1 << 2)
#define MSS_CSSYSCMD		(0x1 << 3)
#define MSS_SWSYSCMD		(0x1 << 4)
#define MSS_SERVREQ		(0x1 << 5)

#define REG_MSD			0x24

#define REG_MIEN		0x2C
#define MIEN_ISOC_PE		(0x1)
#define MIEN_ISOC_BUFO		(0x1 << 1)
#define MIEN_SYNC_PE		(0x1 << 16)
#define MIEN_ARX_DONE		(0x1 << 17)
#define MIEN_ARX_PE		(0x1 << 18)
#define MIEN_ARX_BREAK		(0x1 << 19)
#define MIEN_ATX_DONE		(0x1 << 20)
#define MIEN_ATX_PE		(0x1 << 21)
#define MIEN_ATX_BREAK		(0x1 << 22)
#define MIEN_CRX_DONE		(0x1 << 24)
#define MIEN_CRX_PE		(0x1 << 25)
#define MIEN_CRX_BREAK		(0x1 << 26)
#define MIEN_CTX_DONE		(0x1 << 27)
#define MIEN_CTX_PE		(0x1 << 28)
#define MIEN_CTX_BREAK		(0x1 << 29)

#define REG_MLBPC2		0x34
#define REG_MLBPC1		0x38
#define MLBPC1_VAL		(0x00000888)

#define REG_MLBC1		0x3C
#define MLBC1_LOCK		(0x1 << 6)
#define MLBC1_CLKM		(0x1 << 7)
#define MLBC1_NDA_MASK		(0xFF << 8)
#define MLBC1_NDA_SHIFT		(8)

#define REG_HCTL		0x80
#define HCTL_RST0		(0x1)
#define HCTL_RST1		(0x1 << 1)
#define HCTL_EN			(0x1 << 15)

#define REG_HCMR0		0x88
#define REG_HCMR1		0x8C
#define REG_HCER0		0x90
#define REG_HCER1		0x94
#define REG_HCBR0		0x98
#define REG_HCBR1		0x9C

#define REG_MDAT0		0xC0
#define REG_MDAT1		0xC4
#define REG_MDAT2		0xC8
#define REG_MDAT3		0xCC

#define REG_MDWE0		0xD0
#define REG_MDWE1		0xD4
#define REG_MDWE2		0xD8
#define REG_MDWE3		0xDC

#define REG_MCTL		0xE0
#define MCTL_XCMP		(0x1)

#define REG_MADR		0xE4
#define MADR_WNR		(0x1 << 31)
#define MADR_TB			(0x1 << 30)
#define MADR_ADDR_MASK		(0x7f << 8)
#define MADR_ADDR_SHIFT		(0)

#define REG_ACTL		0x3C0
#define ACTL_MPB		(0x1 << 4)
#define ACTL_DMAMODE		(0x1 << 2)
#define ACTL_SMX		(0x1 << 1)
#define ACTL_SCE		(0x1)

#define REG_ACSR0		0x3D0
#define REG_ACSR1		0x3D4
#define REG_ACMR0		0x3D8
#define REG_ACMR1		0x3DC

#define LOGIC_CH_NUM		(64)
#define BUF_CDT_OFFSET		(0x0)
#define BUF_ADT_OFFSET		(0x40)
#define BUF_CAT_MLB_OFFSET	(0x80)
#define BUF_CAT_HBI_OFFSET	(0x88)
#define BUF_CTR_END_OFFSET	(0x8F)

#define CAT_MODE_RX		(0x1 << 0)
#define CAT_MODE_TX		(0x1 << 1)
#define CAT_MODE_INBOUND_DMA	(0x1 << 8)
#define CAT_MODE_OUTBOUND_DMA	(0x1 << 9)

#define CH_SYNC_BUF_DEP		SYNC_BUFFER_DEP
#define CH_CTRL_BUF_DEP		(64)
#define CH_ASYNC_BUF_DEP		(2048)
#define CH_ISOC_BLK_SIZE		(196)
#define CH_ISOC_BLK_NUM		(3)
#define CH_ISOC_BUF_DEP		\
	(CH_ISOC_BLK_SIZE * CH_ISOC_BLK_NUM)

#define CH_CTRL_DBR_BUF_OFFSET	(0x0)
#define CH_ASYNC_DBR_BUF_OFFSET	\
	(CH_CTRL_DBR_BUF_OFFSET + 2 * CH_CTRL_BUF_DEP)
#define CH_ISOC_DBR_BUF_OFFSET	\
	(CH_ASYNC_DBR_BUF_OFFSET + 2 * CH_ASYNC_BUF_DEP)
#define CH_SYNC0_DBR_BUF_OFFSET	\
	(CH_ISOC_DBR_BUF_OFFSET + 2 * CH_ISOC_BUF_DEP)
#define CH_SYNCN_DBR_BUF_OFFSET(n)\
	(CH_SYNC0_DBR_BUF_OFFSET + n * 2 * CH_SYNC_BUF_DEP)

static u32 mlb150_ch_packet_buf_size[4] = {
	CH_CTRL_BUF_DEP,
	CH_ASYNC_BUF_DEP,
	CH_ISOC_BUF_DEP,
	CH_SYNC_BUF_DEP
};

#define CH_PACKET_BUF_SIZE(minor)                  \
	((minor < MLB_STATIC_MINOR_DEVS) ?                \
	  mlb150_ch_packet_buf_size[minor] :              \
	  mlb150_ch_packet_buf_size[MLB_STATIC_MINOR_DEVS])

#define DBR_BUF_START 0x00000

#define CDT_LEN			(16)
#define ADT_LEN			(16)
#define CAT_LEN			(2)

#define CDT_SZ			(CDT_LEN * LOGIC_CH_NUM)
#define ADT_SZ			(ADT_LEN * LOGIC_CH_NUM)
#define CAT_SZ			(CAT_LEN * LOGIC_CH_NUM * 2)

#define CDT_BASE(base)		(base + BUF_CDT_OFFSET)
#define ADT_BASE(base)		(base + BUF_ADT_OFFSET)
#define CAT_MLB_BASE(base)	(base + BUF_CAT_MLB_OFFSET)
#define CAT_HBI_BASE(base)	(base + BUF_CAT_HBI_OFFSET)

#define CAT_CL_SHIFT		(0x0)
#define CAT_CT_SHIFT		(8)
#define CAT_CE			(0x1 << 11)
#define CAT_RNW			(0x1 << 12)
#define CAT_MT			(0x1 << 13)
#define CAT_FCE			(0x1 << 14)
#define CAT_MFE			(0x1 << 14)

#define CDT_WSBC_SHIFT		(14)
#define CDT_WPC_SHIFT		(11)
#define CDT_RSBC_SHIFT		(30)
#define CDT_RPC_SHIFT		(27)
#define CDT_WPC_1_SHIFT		(12)
#define CDT_RPC_1_SHIFT		(28)
#define CDT_WPTR_SHIFT		(0)
#define CDT_SYNC_WSTS_MASK	(0x0000f000)
#define CDT_SYNC_WSTS_SHIFT	(12)
#define CDT_CTRL_ASYNC_WSTS_MASK	(0x0000f000)
#define CDT_CTRL_ASYNC_WSTS_SHIFT	(12)
#define CDT_ISOC_WSTS_MASK	(0x0000e000)
#define CDT_ISOC_WSTS_SHIFT	(13)
#define CDT_RPTR_SHIFT		(16)
#define CDT_SYNC_RSTS_MASK	(0xf0000000)
#define CDT_SYNC_RSTS_SHIFT	(28)
#define CDT_CTRL_ASYNC_RSTS_MASK	(0xf0000000)
#define CDT_CTRL_ASYNC_RSTS_SHIFT	(28)
#define CDT_ISOC_RSTS_MASK	(0xe0000000)
#define CDT_ISOC_RSTS_SHIFT	(29)
#define CDT_CTRL_ASYNC_WSTS_1	(0x1 << 14)
#define CDT_CTRL_ASYNC_RSTS_1	(0x1 << 15)
#define CDT_BD_SHIFT		(0)
#define CDT_BA_SHIFT		(16)
#define CDT_BS_SHIFT		(0)
#define CDT_BF_SHIFT		(31)

#define ADT_PG			(0x1 << 13)
#define ADT_LE			(0x1 << 14)
#define ADT_CE			(0x1 << 15)
#define ADT_BD1_SHIFT		(0)
#define ADT_ERR1		(0x1 << 13)
#define ADT_DNE1		(0x1 << 14)
#define ADT_RDY1		(0x1 << 15)
#define ADT_BD2_SHIFT		(16)
#define ADT_ERR2		(0x1 << 29)
#define ADT_DNE2		(0x1 << 30)
#define ADT_RDY2		(0x1 << 31)
#define ADT_BA1_SHIFT		(0x0)
#define ADT_BA2_SHIFT		(0x0)
#define ADT_PS1			(0x1 << 12)
#define ADT_PS2			(0x1 << 28)
#define ADT_MEP1		(0x1 << 11)
#define ADT_MEP2		(0x1 << 27)

#define MLB_CONTROL_DEV_NAME	"ctrl"
#define MLB_ASYNC_DEV_NAME	"async"
#define MLB_SYNC_DEV_NAME	"sync"
#define MLB_ISOC_DEV_NAME	"isoc"

#define TX_CHANNEL		0
#define RX_CHANNEL		1
#define PING_BUF_MAX_SIZE	(2 * 1024)
#define PONG_BUF_MAX_SIZE	(2 * 1024)
/* max package data size */
#define ASYNC_PACKET_SIZE	1024
#define CTRL_PACKET_SIZE	64
#define TRANS_RING_NODES	BUF_RING_NODES

enum MLB_FIX_MINOR_DEVICES {
	MLB_MINOR_CTRL,
	MLB_MINOR_ASYNC,
	MLB_MINOR_ISOC,
};

enum MLB_CTYPE {
	MLB_CTYPE_SYNC,
	MLB_CTYPE_CTRL,
	MLB_CTYPE_ASYNC,
	MLB_CTYPE_ISOC,
};

static const char *ctype_names[4] = {
	MLB_SYNC_DEV_NAME,
	MLB_CONTROL_DEV_NAME,
	MLB_ASYNC_DEV_NAME,
	MLB_ISOC_DEV_NAME
};

enum CLK_SPEED {
	CLK_256FS,
	CLK_512FS,
	CLK_1024FS,
	CLK_2048FS,
	CLK_3072FS,
	CLK_4096FS,
	CLK_6144FS,
	CLK_8192FS,
};

struct mlb_ringbuf {
	u32 wpos;
	u32 rpos;
	spinlock_t pos_lock; /* protecting wpos and rpos */
	u32 size;
	/* Last buffer is for package drop */
	u8 *virt_bufs[TRANS_RING_NODES + 1];
	dma_addr_t phy_addrs[TRANS_RING_NODES + 1];
};

struct mlb_channel_info {

	/* channel address */
	u32 address;
	/* DBR buf head */
	u32 dbr_buf_head;
	/* channel buffer size */
	u32 buf_size;
	/* channel buffer current ptr */
	void *buf_ptr;
	/* channel buffer phy addr */
	dma_addr_t buf_phy_addr;
};

struct mlb_dev_info {
	/* device node name */
	char dev_name[20];
	/* channel type */
	unsigned int channel_type;
	/* ch fps */
	enum CLK_SPEED fps;
	/* channel info for tx/rx */
	struct mlb_channel_info channels[2];
	/* rx ring buffer */
	struct mlb_ringbuf rx_bufs;
	/* exception event */
	unsigned long ex_event;
	/* tx busy indicator */
	unsigned long tx_busy;
	/* spinlock for tx_busy access */
	spinlock_t tx_busy_lock;
	/* channel started up or not */
	atomic_t on;
	/* device open count */
	atomic_t opencnt;
	/* wait queue head for channel */
	wait_queue_head_t rx_wq;
	wait_queue_head_t tx_wq;
	/* spinlock for event access */
	spinlock_t event_lock;
};

static struct mlb_dev_info mlb_devinfo[MLB_MAX_DEVICES] = {
	{
	 .dev_name = MLB_CONTROL_DEV_NAME,
	 .channel_type = MLB_CTYPE_CTRL,
	 .channels = {
		      [TX_CHANNEL] = {
				      .buf_size = CH_CTRL_BUF_DEP,
				      .dbr_buf_head =
				      CH_CTRL_DBR_BUF_OFFSET,
				      },
		      [RX_CHANNEL] = {
				      .buf_size = CH_CTRL_BUF_DEP,
				      .dbr_buf_head =
				      CH_CTRL_DBR_BUF_OFFSET +
				      CH_CTRL_BUF_DEP,
				      },
		      },
	 .on = ATOMIC_INIT(0),
	 .opencnt = ATOMIC_INIT(0),
	 .rx_wq =
	 __WAIT_QUEUE_HEAD_INITIALIZER(mlb_devinfo[MLB_MINOR_CTRL].rx_wq),
	 .tx_wq =
	 __WAIT_QUEUE_HEAD_INITIALIZER(mlb_devinfo[MLB_MINOR_CTRL].tx_wq),
	 .event_lock =
	 __SPIN_LOCK_UNLOCKED(mlb_devinfo[MLB_MINOR_CTRL].event_lock),
	 .tx_busy_lock =
	 __SPIN_LOCK_UNLOCKED(mlb_devinfo[MLB_MINOR_CTRL].tx_busy_lock),
	 .rx_bufs = {
		.pos_lock =
		__SPIN_LOCK_UNLOCKED(mlb_devinfo[MLB_MINOR_CTRL]
					.rx_bufs.pos_lock),
	     },
	},
	{
	 .dev_name = MLB_ASYNC_DEV_NAME,
	 .channel_type = MLB_CTYPE_ASYNC,
	 .channels = {
		      [TX_CHANNEL] = {
				      .buf_size = CH_ASYNC_BUF_DEP,
				      .dbr_buf_head =
				      CH_ASYNC_DBR_BUF_OFFSET,
				      },
		      [RX_CHANNEL] = {
				      .buf_size = CH_ASYNC_BUF_DEP,
				      .dbr_buf_head =
				      CH_ASYNC_DBR_BUF_OFFSET +
				      CH_ASYNC_BUF_DEP,
				      },
		      },
	 .on = ATOMIC_INIT(0),
	 .opencnt = ATOMIC_INIT(0),
	 .rx_wq =
	 __WAIT_QUEUE_HEAD_INITIALIZER(mlb_devinfo[MLB_MINOR_ASYNC].rx_wq),
	 .tx_wq =
	 __WAIT_QUEUE_HEAD_INITIALIZER(mlb_devinfo[MLB_MINOR_ASYNC].tx_wq),
	 .event_lock =
	 __SPIN_LOCK_UNLOCKED(mlb_devinfo[MLB_MINOR_ASYNC].event_lock),
	 .tx_busy_lock =
	 __SPIN_LOCK_UNLOCKED(mlb_devinfo[MLB_MINOR_ASYNC].tx_busy_lock),
	 .rx_bufs = {
		.pos_lock =
		__SPIN_LOCK_UNLOCKED(mlb_devinfo[MLB_MINOR_ASYNC]
					.rx_bufs.pos_lock),
	     },
	},

	{
	 .dev_name = MLB_ISOC_DEV_NAME,
	 .channel_type = MLB_CTYPE_ISOC,
	 .channels = {
		      [TX_CHANNEL] = {
				      .buf_size = CH_ISOC_BUF_DEP,
				      .dbr_buf_head =
				      CH_ISOC_DBR_BUF_OFFSET,
				      },
		      [RX_CHANNEL] = {
				      .buf_size = CH_ISOC_BUF_DEP,
				      .dbr_buf_head =
				      CH_ISOC_DBR_BUF_OFFSET +
				      CH_ISOC_BUF_DEP,
				      },
		      },
	 .on = ATOMIC_INIT(0),
	 .opencnt = ATOMIC_INIT(0),
	 .rx_wq =
	 __WAIT_QUEUE_HEAD_INITIALIZER(mlb_devinfo[MLB_MINOR_ISOC].rx_wq),
	 .tx_wq =
	 __WAIT_QUEUE_HEAD_INITIALIZER(mlb_devinfo[MLB_MINOR_ISOC].tx_wq),
	 .event_lock =
	 __SPIN_LOCK_UNLOCKED(mlb_devinfo[MLB_MINOR_ISOC].event_lock),
	 .tx_busy_lock =
	 __SPIN_LOCK_UNLOCKED(mlb_devinfo[MLB_MINOR_ISOC].tx_busy_lock),
	 .rx_bufs = {
		.pos_lock =
		__SPIN_LOCK_UNLOCKED(mlb_devinfo[MLB_MINOR_ISOC]
					.rx_bufs.pos_lock),
	     },
	},

	/*
	 * Sync channels will be filled in function init_sync_devices()
	 */
};

static void __iomem *mlb_base;		/* mlb module base address */

static u32 used_minor_devices = MLB_STATIC_MINOR_DEVS;

#define READ_MLB_REG(mlb_reg) \
	(__raw_readl(mlb_base + (mlb_reg)))

#define WRITE_MLB_REG(value, mlb_reg) \
	(__raw_writel((value), mlb_base + (mlb_reg)))


/* default number of sync channels which is used
   if module is loaded without parameters. */
u32 number_sync_channels = 3;
module_param(number_sync_channels, int, S_IRUGO);

static DEFINE_SPINLOCK(ctr_lock);

static void lld_dump(const char *channel, const u8 *buf, size_t count)
{
	unsigned i;
	unsigned length = 0x10;
	pr_debug(DRIVER_NAME": %s[%u] ", channel, count);

	if (count < 0x10)
		length = count;

	for (i = 0; i < length; ++i)
		pr_debug("%02X ", buf[i]);

	pr_debug("\n");
}

#ifdef DEBUG

#define DUMP_REG(reg) pr_debug(#reg": 0x%08x\n", READ_MLB_REG(reg))

static void mlb150_dev_dump_reg(void)
{
	pr_debug(DRIVER_NAME": Dump registers:\n");
	DUMP_REG(REG_MLBC0);
	DUMP_REG(REG_MLBPC0);
	DUMP_REG(REG_MS0);
	DUMP_REG(REG_MS1);
	DUMP_REG(REG_MSS);
	DUMP_REG(REG_MSD);
	DUMP_REG(REG_MIEN);
	DUMP_REG(REG_MLBPC2);
	DUMP_REG(REG_MLBPC1);
	DUMP_REG(REG_MLBC1);
	DUMP_REG(REG_HCTL);
	DUMP_REG(REG_HCMR0);
	DUMP_REG(REG_HCMR1);
	DUMP_REG(REG_HCER0);
	DUMP_REG(REG_HCER1);
	DUMP_REG(REG_HCBR0);
	DUMP_REG(REG_HCBR1);
	DUMP_REG(REG_MDAT0);
	DUMP_REG(REG_MDAT1);
	DUMP_REG(REG_MDAT2);
	DUMP_REG(REG_MDAT3);
	DUMP_REG(REG_MDWE0);
	DUMP_REG(REG_MDWE1);
	DUMP_REG(REG_MDWE2);
	DUMP_REG(REG_MDWE3);
	DUMP_REG(REG_MCTL);
	DUMP_REG(REG_MADR);
	DUMP_REG(REG_ACTL);
	DUMP_REG(REG_ACSR0);
	DUMP_REG(REG_ACSR1);
	DUMP_REG(REG_ACMR0);
	DUMP_REG(REG_ACMR1);
}

static void mlb150_dev_dump_hex(const u8 *buf, u32 len)
{
	print_hex_dump(KERN_DEBUG, "CTR DUMP:",
			DUMP_PREFIX_OFFSET, 8, 1, buf, len, 0);
}
#endif

static inline void mlb150_dev_enable_ctr_write(u32 mdat0_bits_en,
					       u32 mdat1_bits_en,
					       u32 mdat2_bits_en,
					       u32 mdat3_bits_en)
{
	WRITE_MLB_REG(mdat0_bits_en, REG_MDWE0);
	WRITE_MLB_REG(mdat1_bits_en, REG_MDWE1);
	WRITE_MLB_REG(mdat2_bits_en, REG_MDWE2);
	WRITE_MLB_REG(mdat3_bits_en, REG_MDWE3);
}

/* ctr_lock MUST be held when calling this */
static s32 __mlb150_dev_ctr_read(u32 ctr_offset, u32 *ctr_val)
{
	u32 timeout = 1000;

	WRITE_MLB_REG(ctr_offset, REG_MADR);

	while ((!(READ_MLB_REG(REG_MCTL) & MCTL_XCMP))
		&& --timeout)
		;

	if (timeout == 0) {
		pr_warn(DRIVER_NAME": Read CTR timeout\n");
		return -ETIME;
	}

	ctr_val[0] = READ_MLB_REG(REG_MDAT0);
	ctr_val[1] = READ_MLB_REG(REG_MDAT1);
	ctr_val[2] = READ_MLB_REG(REG_MDAT2);
	ctr_val[3] = READ_MLB_REG(REG_MDAT3);

	WRITE_MLB_REG(0, REG_MCTL);

	return 0;
}

static s32 mlb150_dev_ctr_read(u32 ctr_offset, u32 *ctr_val)
{
	s32 ret;
	unsigned long flags;

	spin_lock_irqsave(&ctr_lock, flags);

	ret = __mlb150_dev_ctr_read(ctr_offset, ctr_val);

	spin_unlock_irqrestore(&ctr_lock, flags);

	return ret;
}

/* ctr_lock MUST be held when calling this */
static s32 __mlb150_dev_ctr_write(u32 ctr_offset, const u32 *ctr_val)
{
	u32 timeout = 1000;

	WRITE_MLB_REG(ctr_val[0], REG_MDAT0);
	WRITE_MLB_REG(ctr_val[1], REG_MDAT1);
	WRITE_MLB_REG(ctr_val[2], REG_MDAT2);
	WRITE_MLB_REG(ctr_val[3], REG_MDAT3);

	WRITE_MLB_REG(MADR_WNR | ctr_offset, REG_MADR);

	while ((!(READ_MLB_REG(REG_MCTL) & MCTL_XCMP))
		&& --timeout)
		;

	if (timeout == 0) {
		pr_warn(DRIVER_NAME": Write CTR timeout\n");
		return -ETIME;
	}

	WRITE_MLB_REG(0, REG_MCTL);

	return 0;
}

/* this function will erase the given part of the DBR */
/* ctr_lock MUST be held when calling this */
static s32 __mlb150_dev_dbr_reset(u32 dbr_offset, u32 dbr_size)
{
	u32 i;
	for (i = 0; i < dbr_size; i++) {
		u32 timeout = 1000;

		WRITE_MLB_REG(0, REG_MDAT0);

		WRITE_MLB_REG(MADR_WNR | MADR_TB |
			(dbr_offset + i), REG_MADR);

		while ((!(READ_MLB_REG(REG_MCTL) & MCTL_XCMP))
			&& --timeout)
			;

		if (timeout == 0) {
			pr_warn(DRIVER_NAME": Write CTR timeout\n");
			return -ETIME;
		}

		WRITE_MLB_REG(0, REG_MCTL);
	}

	return 0;
}

static s32 mlb150_dev_ctr_write(u32 ctr_offset, const u32 *ctr_val)
{
	s32 ret;
	unsigned long flags;

	spin_lock_irqsave(&ctr_lock, flags);

	ret = __mlb150_dev_ctr_write(ctr_offset, ctr_val);

	spin_unlock_irqrestore(&ctr_lock, flags);

#ifdef DEBUG_CTR
	{
		u32 ctr_rd[4] = { 0 };

		if (!mlb150_dev_ctr_read(ctr_offset, ctr_rd)) {
			if (ctr_val[0] == ctr_rd[0] &&
			    ctr_val[1] == ctr_rd[1] &&
			    ctr_val[2] == ctr_rd[2] && ctr_val[3] == ctr_rd[3])
				ret = 0;
			else {
				pr_debug(DRIVER_NAME": ctr write failed\n");
				ret = -EBADE;
			}
		} else {
			pr_debug(DRIVER_NAME": ctr read failed\n");
			ret = -EBADE;
		}
	}
#endif

	return ret;
}

#ifdef DEBUG
static s32 mlb150_dev_cat_read(u32 ctr_offset, u32 ch, u16 *cat_val)
{
	/* TODO: Alignment! */
	u16 ctr_val[8] = { 0 };

	if (mlb150_dev_ctr_read(ctr_offset, (u32 *)ctr_val))
		return -ETIME;

	/*
	 * Use u16 array to get u32 array value,
	 * need to convert
	 * */
	*cat_val = ctr_val[ch % 8];

	 return 0;
}
#endif

/* ctr_lock MUST be held when calling this */
static s32 __mlb150_dev_cat_write(u32 ctr_offset, u32 ch, const u16 cat_val)
{
	/* TODO: Alignment! */
	u16 ctr_val[8] = { 0 };

	if (__mlb150_dev_ctr_read(ctr_offset, (u32 *)ctr_val))
		return -ETIME;

	ctr_val[ch % 8] = cat_val;
	if (__mlb150_dev_ctr_write(ctr_offset, (u32 *)ctr_val))
		return -ETIME;

	return 0;
}

#define mlb150_dev_cat_mlb_read(ch, cat_val)	\
	mlb150_dev_cat_read(BUF_CAT_MLB_OFFSET + ((ch) >> 3), ch, cat_val)
#define mlb150_dev_cat_mlb_write(ch, cat_val)	\
	mlb150_dev_cat_write(BUF_CAT_MLB_OFFSET + ((ch) >> 3), ch, cat_val)
#define mlb150_dev_cat_hbi_read(ch, cat_val)	\
	mlb150_dev_cat_read(BUF_CAT_HBI_OFFSET + ((ch) >> 3), ch, cat_val)
#define mlb150_dev_cat_hbi_write(ch, cat_val)	\
	mlb150_dev_cat_write(BUF_CAT_HBI_OFFSET + ((ch) >> 3), ch, cat_val)

#define mlb150_dev_cdt_read(ch, cdt_val)	\
	mlb150_dev_ctr_read(BUF_CDT_OFFSET + (ch), cdt_val)
#define mlb150_dev_cdt_write(ch, cdt_val)	\
	mlb150_dev_ctr_write(BUF_CDT_OFFSET + (ch), cdt_val)
#define mlb150_dev_adt_read(ch, adt_val)	\
	mlb150_dev_ctr_read(BUF_ADT_OFFSET + (ch), adt_val)
#define mlb150_dev_adt_write(ch, adt_val)	\
	mlb150_dev_ctr_write(BUF_ADT_OFFSET + (ch), adt_val)

static s32 mlb150_dev_get_adt_sts(u32 ch)
{
	u32 timeout = 1000;
	unsigned long flags;
	u32 reg;

	spin_lock_irqsave(&ctr_lock, flags);
	WRITE_MLB_REG(BUF_ADT_OFFSET + ch, REG_MADR);

	while ((!(READ_MLB_REG(REG_MCTL)
		  & MCTL_XCMP)) && --timeout)
		;

	if (timeout == 0) {
		pr_debug(DRIVER_NAME": Read CTR timeout\n");
		spin_unlock_irqrestore(&ctr_lock, flags);
		return -ETIME;
	}

	reg = READ_MLB_REG(REG_MDAT1);

	WRITE_MLB_REG(0, REG_MCTL);
	spin_unlock_irqrestore(&ctr_lock, flags);

#ifdef DEBUG_ADT
	pr_debug(DRIVER_NAME": Get ch %d adt sts: 0x%08x\n", ch, reg);
#endif

	return reg;
}

#ifdef DEBUG
static void mlb150_dev_dump_ctr_tbl(u32 ch_start, u32 ch_end)
{
	u32 i = 0;
	u32 ctr_val[4] = { 0 };

	pr_debug(DRIVER_NAME": CDT Table");
	for (i = BUF_CDT_OFFSET + ch_start;
	     i < BUF_CDT_OFFSET + ch_end; ++i) {
		mlb150_dev_ctr_read(i, ctr_val);
		pr_debug("CTR 0x%02x: 0x%08x, 0x%08x, 0x%08x, 0x%08x\n",
			 i, ctr_val[3], ctr_val[2], ctr_val[1], ctr_val[0]);
	}

	pr_debug(DRIVER_NAME": ADT Table");
	for (i = BUF_ADT_OFFSET + ch_start;
	     i < BUF_ADT_OFFSET + ch_end; ++i) {
		mlb150_dev_ctr_read(i, ctr_val);
		pr_debug("CTR 0x%02x: 0x%08x, 0x%08x, 0x%08x, 0x%08x\n",
			 i, ctr_val[3], ctr_val[2], ctr_val[1], ctr_val[0]);
	}

	pr_debug(DRIVER_NAME": CAT MLB Table");
	for (i = BUF_CAT_MLB_OFFSET + (ch_start >> 3);
			i <= BUF_CAT_MLB_OFFSET + ((ch_end + 8) >> 3);
			++i) {
		mlb150_dev_ctr_read(i, ctr_val);
		pr_debug("CTR 0x%02x: 0x%08x, 0x%08x, 0x%08x, 0x%08x\n",
			i, ctr_val[3], ctr_val[2], ctr_val[1], ctr_val[0]);
	}

	pr_debug(DRIVER_NAME": CAT HBI Table");
	for (i = BUF_CAT_HBI_OFFSET + (ch_start >> 3);
			i <= BUF_CAT_HBI_OFFSET + ((ch_end + 8) >> 3);
			++i) {
		mlb150_dev_ctr_read(i, ctr_val);
		pr_debug("CTR 0x%02x: 0x%08x, 0x%08x, 0x%08x, 0x%08x\n",
			i, ctr_val[3], ctr_val[2], ctr_val[1], ctr_val[0]);
	}
}
#endif

/*!
 * Initial the MLB module device
 */
static inline void mlb150_dev_enable_dma_irq(u32 enable)
{
	u32 value = 0;
	if (enable)
		value = 0xffffffff;

	WRITE_MLB_REG(value, REG_ACMR0);
	WRITE_MLB_REG(value, REG_ACMR1);
}

/* channel specific irq enable */
static inline void mlb150_dev_enable_dma_irq_ch(u32 ch, u32 enable)
{
	u32 acmr_val;
	u32 acmr_adr;

	if (ch < 32)
		acmr_adr = REG_ACMR0;
	else
		acmr_adr = REG_ACMR1;

	acmr_val = READ_MLB_REG(acmr_adr);

	if (enable)
		acmr_val |= 1 << (ch % 32);
	else
		acmr_val &= ~(1 << (ch % 32));

	WRITE_MLB_REG(acmr_val, acmr_adr);
}

static s32 mlb150_dev_init_ir_amba_ahb(void)
{
	u32 reg = 0;

	/* Step 1. Program the ACMRn registers to enable interrupts from all
	 * active DMA channels */
	/* don't do this here but in mlb_channel_enable for each channel */
	/* mlb150_dev_enable_dma_irq(1); */

	/* Step 2. Select the status clear method:
	 * ACTL.SCE = 0, hardware clears on read
	 * We only support DMA MODE 1 */
	reg = READ_MLB_REG(REG_ACTL);
	reg |= ACTL_DMAMODE;
#ifdef MULTIPLE_PACKAGE_MODE
	reg |= REG_ACTL_MPB;
#endif

	/* Step 3. Select 1 or 2 interrupt signals:
	 * ACTL.SMX = 0: one interrupt for channels 0 - 31 on ahb_init[0]
	 *      and another interrupt for channels 32 - 63 on ahb_init[1]
	 * ACTL.SMX = 1: singel interrupt all channels on ahb_init[0]
	 * */
	reg &= ~ACTL_SMX;


	WRITE_MLB_REG(reg, REG_ACTL);
	return 0;
}

static inline s32 mlb150_dev_enable_ir_mlb(u32 enable)
{
	/* Step 1, Select the MSn to be cleared by software,
	 * writing a '0' to the appropriate bits */
	WRITE_MLB_REG(0, REG_MS0);
	WRITE_MLB_REG(0, REG_MS1);

	/* Step 1, Program MIEN to enable protocol error
	 * interrupts for all active MLB channels */
	if (enable)
		WRITE_MLB_REG(MIEN_CTX_PE |
			     MIEN_CRX_PE | MIEN_ATX_PE |
			     MIEN_ARX_PE | MIEN_SYNC_PE |
			     MIEN_ISOC_PE,
			     REG_MIEN);
	else
		WRITE_MLB_REG(0, REG_MIEN);

	return 0;
}

/* complete CDT reset */
static s32 mlb150_dev_reset_cdt(void)
{
	int i = 0;
	u32 ctr_val[4] = { 0 };

	for (i = 0; i < (LOGIC_CH_NUM); ++i)
		mlb150_dev_ctr_write(BUF_CDT_OFFSET + i, ctr_val);

	return 0;
}

/* ctr_lock MUST be held when calling this */
static s32 mlb150_dev_init_ch_cdt(struct mlb_dev_info *pdevinfo, u32 ch,
				  enum MLB_CTYPE ctype, u32 ch_func,
				  unsigned int audio_channels)
{
	u32 cdt_val[4] = { 0 };

	/* a. Set the 14-bit base address (BA) */
	pr_debug(DRIVER_NAME": ctype: %d, ch: %d, dbr_buf_head: 0x%08x",
		ctype, ch, pdevinfo->channels[ch_func].dbr_buf_head);
	cdt_val[3] = (pdevinfo->channels[ch_func].dbr_buf_head)
			<< CDT_BA_SHIFT;

	/* b. Set the 12-bit or 13-bit buffer depth (BD)
	 * BD = buffer depth in bytes - 1 */

	switch (ctype) {
	case MLB_CTYPE_SYNC:
		/* For synchronous channels: (BD + 1) = 4 * m * bpf */
	  cdt_val[3] |= ((SYNC_BUFFER_DEP_MIN * audio_channels - 1)
			 << CDT_BD_SHIFT);
		break;
	case MLB_CTYPE_CTRL:
		/* For control channels: (BD + 1) >= max packet length (64) */
		/* BD */
		cdt_val[3] |= ((CH_CTRL_BUF_DEP - 1) << CDT_BD_SHIFT);
		break;
	case MLB_CTYPE_ASYNC:
		/* For asynchronous channels: (BD + 1) >= max packet length
		 * 1024 for a MOST Data packet (MDP);
		 * 1536 for a MOST Ethernet Packet (MEP) */
		cdt_val[3] |= ((CH_ASYNC_BUF_DEP - 1) << CDT_BD_SHIFT);
		break;
	case MLB_CTYPE_ISOC:
		/* For isochronous channels: (BD + 1) mod (BS + 1) = 0 */
		/* BS */
		cdt_val[1] |= (CH_ISOC_BLK_SIZE - 1);
		/* BD */
		cdt_val[3] |= (CH_ISOC_BUF_DEP - 1)
		    << CDT_BD_SHIFT;
		break;
	default:
		break;
	}

	pr_debug(DRIVER_NAME": Set CDT val of channel %d, ctype: %d: 0x%08x 0x%08x 0x%08x 0x%08x\n",
		 ch, ctype, cdt_val[3], cdt_val[2], cdt_val[1], cdt_val[0]);

	if (__mlb150_dev_ctr_write(BUF_CDT_OFFSET + ch, cdt_val))
		return -ETIME;

#ifdef DEBUG_CTR
	{
		u32 cdt_rd[4] = { 0 };
		if (!mlb150_dev_cdt_read(ch, cdt_rd)) {
			pr_debug(DRIVER_NAME": CDT val of channel %d: 0x%08x 0x%08x 0x%08x 0x%08x\n",
				 ch, cdt_rd[3], cdt_rd[2], cdt_rd[1],
				 cdt_rd[0]);
			if (cdt_rd[3] == cdt_val[3] &&
			    cdt_rd[2] == cdt_val[2] &&
			    cdt_rd[1] == cdt_val[1] &&
			    cdt_rd[0] == cdt_val[0]) {
				pr_debug(DRIVER_NAME": set cdt succeed!\n");
				return 0;
			} else {
				pr_debug(DRIVER_NAME": set cdt failed!\n");
				return -EBADE;
			}
		} else {
			pr_debug
			    (DRIVER_NAME": Read CDT val of channel %d failed\n",
			     ch);
			return -EBADE;
		}
	}
#endif

	return 0;
}

/* ctr_lock MUST be held when calling this */
static s32 mlb150_dev_init_ch_cat(u32 ch,
		u32 cat_mode, enum MLB_CTYPE ctype)
{
	u16 cat_val = 0;
#ifdef DEBUG_CTR
	u16 cat_rd = 0;
#endif

	cat_val = CAT_CE | (ctype << CAT_CT_SHIFT) | ch;

	if (cat_mode & CAT_MODE_OUTBOUND_DMA)
		cat_val |= CAT_RNW;

	if (MLB_CTYPE_SYNC == ctype) {
		cat_val |= CAT_MT;
#ifdef MLB_USE_MFE
		/* enable multi-frame sub-buffering */
		cat_val |= CAT_MFE;
#endif
	}

	switch (cat_mode) {
	case CAT_MODE_RX | CAT_MODE_INBOUND_DMA:
	case CAT_MODE_TX | CAT_MODE_OUTBOUND_DMA:
		pr_debug(DRIVER_NAME": set CAT val of channel %d, type: %d: 0x%04x\n",
			ch, ctype, cat_val);

		if (__mlb150_dev_cat_write
		     (BUF_CAT_MLB_OFFSET + (ch >> 3), ch, cat_val))
			return -ETIME;
#ifdef DEBUG_CTR
		if (!mlb150_dev_cat_mlb_read(ch, &cat_rd))
			pr_debug
			    (DRIVER_NAME": CAT val of mlb channel %d: 0x%04x",
			     ch, cat_rd);
		else {
			pr_debug
			    (DRIVER_NAME": Read CAT of mlb channel %d failed\n",
			     ch);
			return -EBADE;
		}
#endif
		break;
	case CAT_MODE_TX | CAT_MODE_INBOUND_DMA:
	case CAT_MODE_RX | CAT_MODE_OUTBOUND_DMA:
		pr_debug(DRIVER_NAME": set CAT val of channel %d, type: %d: 0x%04x\n",
			ch, ctype, cat_val);

		if (__mlb150_dev_cat_write
		     (BUF_CAT_HBI_OFFSET + (ch >> 3), ch, cat_val))
			return -ETIME;
#ifdef DEBUG_CTR
		if (!mlb150_dev_cat_hbi_read(ch, &cat_rd))
			pr_debug
			    (DRIVER_NAME": CAT val of hbi channel %d: 0x%04x",
			     ch, cat_rd);
		else {
			pr_debug
			    (DRIVER_NAME": Read CAT of hbi channel %d failed\n",
			     ch);
			return -EBADE;
		}
#endif
		break;
	default:
		return -EBADRQC;
	}

#ifdef DEBUG_CTR
	{
		if (cat_val == cat_rd) {
			pr_debug(DRIVER_NAME": set cat succeed!\n");
			return 0;
		} else {
			pr_debug(DRIVER_NAME": set cat failed!\n");
			return -EBADE;
		}
	}
#endif
	return 0;
}

/* complete CAT reset */
static s32 mlb150_dev_reset_cat(void)
{
	int i = 0;
	u32 ctr_val[4] = { 0 };

	for (i = 0; i < (LOGIC_CH_NUM >> 3); ++i) {
		mlb150_dev_ctr_write(BUF_CAT_MLB_OFFSET + i, ctr_val);
		mlb150_dev_ctr_write(BUF_CAT_HBI_OFFSET + i, ctr_val);
	}

	return 0;
}

/* channel specific ctr reset */
/* ctr_lock MUST be held when calling this; */
/* ctr write must be unmasked */
static inline s32 mlb150_dev_reset_ch(u32 ch)
{
	u32 ctr_val[4] = { 0 };
	u16 ctr_val_16 = 0;

	__mlb150_dev_cat_write(BUF_CAT_MLB_OFFSET + (ch >> 3), ch,
			       ctr_val_16);
	__mlb150_dev_cat_write(BUF_CAT_HBI_OFFSET + (ch >> 3), ch,
			       ctr_val_16);
	/* no need to reset cdt */

	/* amba */
	__mlb150_dev_ctr_write(BUF_ADT_OFFSET + ch, ctr_val);

	return 0;
}

/* ctr_lock MUST be held when calling this; *
 * ctr write must be completely unmasked    */
static s32 mlb150_dev_init_rfb_rx(struct mlb_dev_info *pdevinfo, u32 rx_ch,
				  enum MLB_CTYPE ctype,
				  unsigned int audio_channels)
{
	/* Step 1, Initialize all bits of CAT to '0' */
	/*mlb150_dev_reset_ch_rfb(rx_ch, tx_ch); */
	/* not necessary here; done on probe/channel disable */

	/* Step 2, Initialize logical channel */
	/* Step 3, Program the CDT for channel N */
	mlb150_dev_init_ch_cdt(pdevinfo, rx_ch, ctype, RX_CHANNEL
				, audio_channels);

	/* Step 4&5, Program the CAT for the inbound and outbound DMA */
	mlb150_dev_init_ch_cat(rx_ch,
			       CAT_MODE_RX | CAT_MODE_INBOUND_DMA,
			       ctype);
	mlb150_dev_init_ch_cat(rx_ch,
			       CAT_MODE_RX |
			       CAT_MODE_OUTBOUND_DMA,
			       ctype);
	return 0;
}

/* ctr_lock MUST be held when calling this; *
 * ctr write must be completely unmasked    */
static s32 mlb150_dev_init_rfb_tx(struct mlb_dev_info *pdevinfo, u32 tx_ch,
				  enum MLB_CTYPE ctype,
				  unsigned int audio_channels)
{
	/* Step 1, Initialize all bits of CAT to '0' */
	/*mlb150_dev_reset_ch_rfb(rx_ch, tx_ch); */
	/* not necessary here; done on probe/channel disable */

	/* Step 2, Initialize logical channel */
	/* Step 3, Program the CDT for channel N */
	mlb150_dev_init_ch_cdt(pdevinfo, tx_ch, ctype, TX_CHANNEL,
						   audio_channels);

	/* Step 4&5, Program the CAT for the inbound and outbound DMA */
	mlb150_dev_init_ch_cat(tx_ch,
			       CAT_MODE_TX | CAT_MODE_INBOUND_DMA,
			       ctype);
	mlb150_dev_init_ch_cat(tx_ch,
			       CAT_MODE_TX |
			       CAT_MODE_OUTBOUND_DMA,
			       ctype);
	return 0;
}

/* complete ADT reset */
static s32 mlb150_dev_reset_adt(void)
{
	int i = 0;
	u32 ctr_val[4] = { 0 };

	for (i = 0; i < (LOGIC_CH_NUM); ++i)
		mlb150_dev_ctr_write(BUF_ADT_OFFSET + i, ctr_val);

	return 0;
}

/* complete CTR reset; only useful for init */
static void mlb150_dev_reset_whole_ctr(void)
{
	mlb150_dev_enable_ctr_write(0xffffffff, 0xffffffff,
				    0xffffffff, 0xffffffff);
	mlb150_dev_reset_cdt();
	mlb150_dev_reset_adt();
	mlb150_dev_reset_cat();
}

#define CLR_REG(reg)  WRITE_MLB_REG(0x0, reg)

static void mlb150_dev_reset_all_regs(void)
{
	CLR_REG(REG_MLBC0);
	CLR_REG(REG_MLBPC0);
	CLR_REG(REG_MS0);
	CLR_REG(REG_MS1);
	CLR_REG(REG_MSS);
	CLR_REG(REG_MSD);
	CLR_REG(REG_MIEN);
	CLR_REG(REG_MLBPC2);
	CLR_REG(REG_MLBPC1);
	CLR_REG(REG_MLBC1);
	CLR_REG(REG_HCTL);
	CLR_REG(REG_HCMR0);
	CLR_REG(REG_HCMR1);
	CLR_REG(REG_HCER0);
	CLR_REG(REG_HCER1);
	CLR_REG(REG_HCBR0);
	CLR_REG(REG_HCBR1);
	CLR_REG(REG_MDAT0);
	CLR_REG(REG_MDAT1);
	CLR_REG(REG_MDAT2);
	CLR_REG(REG_MDAT3);
	CLR_REG(REG_MDWE0);
	CLR_REG(REG_MDWE1);
	CLR_REG(REG_MDWE2);
	CLR_REG(REG_MDWE3);
	CLR_REG(REG_MCTL);
	CLR_REG(REG_MADR);
	CLR_REG(REG_ACTL);
	CLR_REG(REG_ACSR0);
	CLR_REG(REG_ACSR1);
	CLR_REG(REG_ACMR0);
	CLR_REG(REG_ACMR1);
}

static inline s32 mlb150_dev_set_ch_amba_ahb(u32 ch, enum MLB_CTYPE ctype,
					     u32 dne_sts, u32 buf_addr)
{
	u32 ctr_val[4] = { 0 };

	if (MLB_CTYPE_ASYNC == ctype || MLB_CTYPE_CTRL == ctype) {
		ctr_val[1] |= ADT_PS1;
		ctr_val[1] |= ADT_PS2;
	}

	/* Clear DNE1 and ERR1 */
	/* Set the page ready bit (RDY1) */
	if (dne_sts & ADT_DNE1) {
		/* ctr_val[0] = 0; (per initialization) */
		ctr_val[1] |= ADT_RDY2;
		/* ctr_val[2] = 0; (per initialization) */
		ctr_val[3] = buf_addr;	/* BA2 */
	} else {
		/* ctr_val[2] = 0; (per initialization) */
		ctr_val[1] |= ADT_RDY1;
		ctr_val[2] = buf_addr;	/* BA1 */
		/* ctr_val[3] = 0; (per initialization) */
	}

	if (!(dne_sts & (ADT_DNE1 | ADT_DNE2))) {
		pr_debug
		    (DRIVER_NAME": AHB error status in channel %d, ctype %d\n",
		     ch, ctype);
	}
#ifdef DEBUG_ADT
	pr_debug(DRIVER_NAME": Set ADT val of channel %d, ctype: %d: 0x%08x 0x%08x 0x%08x 0x%08x\n",
		 ch, ctype, ctr_val[3], ctr_val[2], ctr_val[1], ctr_val[0]);
#endif

	if (mlb150_dev_adt_write(ch, ctr_val))
		return -ETIME;

#ifdef DEBUG_ADT_N
	{
		u32 ctr_rd[4] = { 0 };
		if (likely(!mlb150_dev_adt_read(ch, ctr_rd))) {
			pr_debug(DRIVER_NAME": ADT val of channel %d: 0x%08x 0x%08x 0x%08x 0x%08x\n",
				 ch, ctr_rd[3], ctr_rd[2],
				 ctr_rd[1], ctr_rd[0]);
			if (ctr_rd[3] == ctr_val[3] &&
			    ctr_rd[2] == ctr_val[2] &&
			    ctr_rd[1] == ctr_val[1] &&
			    ctr_rd[0] == ctr_val[0]) {
				pr_debug(DRIVER_NAME": set adt succeed!\n");
				return 0;
			} else {
				pr_debug(DRIVER_NAME": set adt failed!\n");
				return -EBADE;
			}
		} else {
			pr_debug
			    (DRIVER_NAME": Read ADT val of channel %d failed\n",
			     ch);
			return -EBADE;
		}
	}
#endif

	return 0;
}

static s32 mlb150_dev_init_ch_amba_ahb(struct mlb_channel_info *chinfo,
				       enum MLB_CTYPE ctype,
				       unsigned int audio_channels)
{
	u32 ctr_val[4] = { 0 };

	/* a. Set the 32-bit base address (BA1) */
	/* no need to do this; RDY is not set either */

	ctr_val[1] = (chinfo->buf_size - 1) << ADT_BD1_SHIFT;
	ctr_val[1] |= (chinfo->buf_size - 1) << ADT_BD2_SHIFT;

	if (MLB_CTYPE_ASYNC == ctype || MLB_CTYPE_CTRL == ctype) {
		ctr_val[1] |= ADT_PS1;
		ctr_val[1] |= ADT_PS2;
	}

	ctr_val[0] |= (ADT_LE | ADT_CE);

	pr_debug(DRIVER_NAME": Set ADT val of channel %d, ctype: %s: 0x%08x 0x%08x 0x%08x 0x%08x\n",
		 chinfo->address, ctype_names[ctype], ctr_val[3], ctr_val[2],
		 ctr_val[1], ctr_val[0]);

	if (__mlb150_dev_ctr_write
	     (BUF_ADT_OFFSET + chinfo->address, ctr_val))
		return -ETIME;

#ifdef DEBUG_CTR
	{
		u32 ctr_rd[4] = { 0 };
		if (!mlb150_dev_adt_read(chinfo->address, ctr_rd)) {
			pr_debug(DRIVER_NAME": init ADT val of channel %d: 0x%08x 0x%08x 0x%08x 0x%08x\n",
				 chinfo->address, ctr_rd[3], ctr_rd[2],
				 ctr_rd[1], ctr_rd[0]);
			if (ctr_rd[3] == ctr_val[3] &&
			    ctr_rd[2] == ctr_val[2] &&
			    ctr_rd[1] == ctr_val[1] &&
			    ctr_rd[0] == ctr_val[0]) {
				pr_debug(DRIVER_NAME": init adt succeeded!\n");
				return 0;
			} else {
				pr_debug(DRIVER_NAME": init adt failed!\n");
				return -EBADE;
			}
		} else {
			pr_debug
			    (DRIVER_NAME": Read ADT val of channel %d failed\n",
			     chinfo->address);
			return -EBADE;
		}
	}
#endif

	return 0;
}

static void mlb150_dev_exit(void)
{
	mlb150_dev_enable_dma_irq(0);
	mlb150_dev_enable_ir_mlb(0);

	WRITE_MLB_REG(0, REG_HCTL);
	WRITE_MLB_REG(0, REG_MLBC0);

	WRITE_MLB_REG(0x0, REG_HCMR0);
	WRITE_MLB_REG(0x0, REG_HCMR1);
}

static void mlb150_dev_init(void)
{
	u32 c0_val = 0;

	/* reset the CTR (here!!) */
	mlb150_dev_reset_whole_ctr();

	/* reset all registers */
	mlb150_dev_reset_all_regs();

	/* Step 1, Configure the MediaLB interface */

	c0_val |= MLBC0_MLBEN;

#ifdef MLB_USE_MFE
	/* set FCNT to 64 frames per sub-buffer (b110) */
	c0_val |= FCNT_VALUE << MLBC0_FCNT_SHIFT;
#endif

	WRITE_MLB_REG(c0_val, REG_MLBC0);

	/* Step 2, Configure the HBI interface */
	WRITE_MLB_REG(0xffffffff, REG_HCMR0);
	WRITE_MLB_REG(0xffffffff, REG_HCMR1);
	WRITE_MLB_REG(HCTL_EN, REG_HCTL);

	/* setup ahb channel interrupts (but don't enable yet) */
	mlb150_dev_init_ir_amba_ahb();

	/* enable protocol error interrupts for all channels */
	mlb150_dev_enable_ir_mlb(1);
}

/* ctr_lock MUST be held when calling this */
static s32 mlb150_dev_unmute_syn_ch(u32 rx_ch, u32 tx_ch, enum channelmode mode)
{
	u32 timeout = 10000;
	u32 mfe;

	/* Check that MediaLB clock is running (MLBC1.CLKM = 0)
	 * If MLBC1.CLKM = 1, clear the register bit, wait one
	 * APB or I/O clock cycle and repeat the check */
	while ((READ_MLB_REG(REG_MLBC1) & MLBC1_CLKM)
		&& --timeout)
		WRITE_MLB_REG(~MLBC1_CLKM, REG_MLBC1);

	if (0 == timeout)
		return -ETIME;

	timeout = 10000;
	/* Poll for MLB lock (MLBC0.MLBLK = 1) */
	while (!(READ_MLB_REG(REG_MLBC0) & MLBC0_MLBLK)
		&& --timeout)
		;

	if (0 == timeout)
		return -ETIME;

	mfe = 0;
#ifdef MLB_USE_MFE
	/* enable multi-frame sub-buffering */
	mfe = CAT_MFE;
#endif

	/* Unmute synchronous channel(s) */
	if (mode == MLB_RDONLY || mode == MLB_RDWR) {
		__mlb150_dev_cat_write(BUF_CAT_MLB_OFFSET + (rx_ch >> 3),
				       rx_ch, mfe | CAT_CE | rx_ch);
		__mlb150_dev_cat_write(BUF_CAT_HBI_OFFSET + (rx_ch >> 3),
				       rx_ch, mfe | CAT_CE | rx_ch | CAT_RNW);
	}
	if (mode == MLB_WRONLY || mode == MLB_RDWR) {
		__mlb150_dev_cat_write(BUF_CAT_MLB_OFFSET + (tx_ch >> 3),
				       tx_ch, mfe | CAT_CE | tx_ch | CAT_RNW);
		__mlb150_dev_cat_write(BUF_CAT_HBI_OFFSET + (tx_ch >> 3),
				       tx_ch, mfe | CAT_CE | tx_ch);
	}

	return 0;
}

static void set_tx_busy(struct mlb_dev_info *dev_info,
	unsigned long value)
{
	unsigned long flags;
	spin_lock_irqsave(&dev_info->tx_busy_lock, flags);
	dev_info->tx_busy = value;
	spin_unlock_irqrestore(&dev_info->tx_busy_lock, flags);
}

static unsigned long get_tx_busy(struct mlb_dev_info *dev_info)
{
	unsigned long ret, flags;

	spin_lock_irqsave(&dev_info->tx_busy_lock, flags);
	ret = dev_info->tx_busy;
	spin_unlock_irqrestore(&dev_info->tx_busy_lock, flags);

	return ret;
}

/*!
 * MLB receive start function
 *
 * load phy_head to next buf register to start next rx
 * here use single-packet buffer, set start=end
 */
static
void mlb_start_rx(u32 ch, enum MLB_CTYPE ctype, u32 dne_sts, u32 buf_addr)
{
	/*  Set ADT for RX */
	mlb150_dev_set_ch_amba_ahb(ch, ctype, dne_sts, buf_addr);
}

/*!
 * MLB transmit start function
 */
s32 mlb_start_tx(struct mlb_data *drvdata, int minor, u32 buf_addr)
{
	u32 adt_sts = 0;
	struct mlb_dev_info *pdevinfo = &drvdata->devinfo[minor];
	struct mlb_channel_info *pchinfo = &pdevinfo->channels[TX_CHANNEL];

	adt_sts = mlb150_dev_get_adt_sts(pchinfo->address);
	set_tx_busy(pdevinfo, 1);

	/*  Set ADT for TX */
	return mlb150_dev_set_ch_amba_ahb(pchinfo->address,
			pdevinfo->channel_type, adt_sts, buf_addr);
}

/*!
 * Enable the MLB channel
 */
/* ext parameter */
static void mlb_channel_enable(struct mlb_data *drvdata, int minor,
			       enum channelmode mode, int on,
			       unsigned int audio_channels)
{
	struct mlb_dev_info *pdevinfo = &drvdata->devinfo[minor];
	struct mlb_channel_info *tx_chinfo = &pdevinfo->channels[TX_CHANNEL];
	struct mlb_channel_info *rx_chinfo = &pdevinfo->channels[RX_CHANNEL];
	u32 tx_ch = tx_chinfo->address;
	u32 rx_ch = rx_chinfo->address;
	u32 ctype = pdevinfo->channel_type;
	unsigned long flags;

	pr_debug(DRIVER_NAME": channel_enable(%d) on minor %d\n", on, minor);
	/*!
	 * setup the direction, enable, channel type,
	 * mode select, channel address and mask buf start
	 */
	if (on) {
		pr_debug(DRIVER_NAME": mlb_channel_enable on minor %d: rx %d, tx %d, ctype %d, mode %d, audio_channels %d\n",
			minor, rx_ch, tx_ch, ctype, mode,
			audio_channels);
		spin_lock_irqsave(&ctr_lock, flags);
		mlb150_dev_enable_ctr_write(0xffffffff, 0xffffffff,
					    0xffffffff, 0xffffffff);

		if (mode == MLB_RDONLY || mode == MLB_RDWR) {
			/* reset DBR as to not receive trash on start */
			__mlb150_dev_dbr_reset(rx_chinfo->dbr_buf_head,
					       rx_chinfo->buf_size);
			mlb150_dev_init_rfb_rx(pdevinfo, rx_ch, ctype,
						audio_channels);
			mlb150_dev_init_ch_amba_ahb(rx_chinfo, ctype,
						    audio_channels);
		}

		if (mode == MLB_WRONLY || mode == MLB_RDWR) {
			/* reset DBR as to not transmit trash on start */
			__mlb150_dev_dbr_reset(tx_chinfo->dbr_buf_head,
					       tx_chinfo->buf_size);
			mlb150_dev_init_rfb_tx(pdevinfo, tx_ch, ctype,
					       audio_channels);
			mlb150_dev_init_ch_amba_ahb(tx_chinfo, ctype,
						    audio_channels);
		}

		/* Synchronize and unmute synchrouous channel */
		if (MLB_CTYPE_SYNC == ctype)
			mlb150_dev_unmute_syn_ch(rx_ch, tx_ch, mode);

		mlb150_dev_enable_ctr_write(0x0,
				ADT_RDY1 | ADT_DNE1 | ADT_ERR1 |
				ADT_PS1 | ADT_MEP1 | ADT_RDY2 |
				ADT_DNE2 | ADT_ERR2 | ADT_PS2 |
				ADT_MEP2, 0xffffffff, 0xffffffff);

		spin_unlock_irqrestore(&ctr_lock, flags);

		if (pdevinfo->fps >= CLK_2048FS)
			pr_err("6 pin MLB is not supported\n");

		atomic_set(&pdevinfo->on, 1);

		if (mode == MLB_RDONLY || mode == MLB_RDWR)
			mlb150_dev_enable_dma_irq_ch(rx_ch, 1);

		if (mode == MLB_WRONLY || mode == MLB_RDWR)
			mlb150_dev_enable_dma_irq_ch(tx_ch, 1);

#ifdef DEBUG_CHANNEL_ENABLE
		mlb150_dev_dump_reg();
		mlb150_dev_dump_ctr_tbl(0, tx_chinfo->address + 1);
#endif
		if (mode == MLB_RDONLY || mode == MLB_RDWR)
			mlb_start_rx(rx_ch, ctype, ADT_DNE2,
				     pdevinfo->rx_bufs.phy_addrs[0]);
	} else {
		/* reset/disable this channel */
		if (mode == MLB_WRONLY || mode == MLB_RDWR)
			mlb150_dev_enable_dma_irq_ch(tx_ch, 0);
		if (mode == MLB_RDONLY || mode == MLB_RDWR)
			mlb150_dev_enable_dma_irq_ch(rx_ch, 0);

		spin_lock_irqsave(&ctr_lock, flags);
		mlb150_dev_enable_ctr_write(0xffffffff, 0xffffffff,
					    0xffffffff, 0xffffffff);

		if (mode == MLB_WRONLY || mode == MLB_RDWR) {
			mlb150_dev_reset_ch(tx_ch);
			set_tx_busy(pdevinfo, 0);
		}
		if (mode == MLB_RDONLY || mode == MLB_RDWR)
			mlb150_dev_reset_ch(rx_ch);

		mlb150_dev_enable_ctr_write(0x0,
					    ADT_RDY1 | ADT_DNE1 |
					    ADT_ERR1 | ADT_PS1 |
					    ADT_MEP1 | ADT_RDY2 |
					    ADT_DNE2 | ADT_ERR2 |
					    ADT_PS2 | ADT_MEP2,
					    0xffffffff, 0xffffffff);
		spin_unlock_irqrestore(&ctr_lock, flags);

		atomic_set(&pdevinfo->on, 0);
	}
}

/*!
 * MLB interrupt handler
 */
static void mlb_tx_isr(struct mlb_dev_info *pdevinfo, int minor)
{
	set_tx_busy(pdevinfo, 0);
	wake_up_interruptible(&pdevinfo->tx_wq);

	/* call ext. IRQ handler (syncsound) */
	ext_start_tx(minor, pdevinfo->channel_type);
}

static void mlb_rx_isr(struct mlb_dev_info *pdevinfo, int minor)
{
	struct mlb_channel_info *pchinfo = &pdevinfo->channels[RX_CHANNEL];
	struct mlb_ringbuf *rx_rbuf = &pdevinfo->rx_bufs;
	unsigned long flags;
	u32 wpos, wpos_new, rpos;
	s32 adt_sts;
	dma_addr_t rx_ring_buf = 0;
	u32 ch_addr = pchinfo->address;
	u32 ctype = pdevinfo->channel_type;

#ifdef DEBUG_RX
	pr_debug(DRIVER_NAME": mlb_rx_isr\n");
#endif

	lld_dump((minor == 0) ? "R(c):" : "R(a):",
		rx_rbuf->virt_bufs[rx_rbuf->wpos], pchinfo->buf_size);

	spin_lock_irqsave(&rx_rbuf->pos_lock, flags);
	rpos = rx_rbuf->rpos;
	wpos = rx_rbuf->wpos;

#ifdef DEBUG_RX
	pr_debug("adt_buf_ptr: 0x%08x\n", (u32) adt_buf_ptr);
#endif

	/*!
	 * Check if we have space in the ring buf
	 * if the ++wpos == rpos, drop this packet
	 */
	wpos_new = (wpos + 1) % TRANS_RING_NODES;
	if (wpos_new != rpos) {
		/* update the ring wpos */
		rx_rbuf->wpos = wpos_new;

		/* wake up the reader */
		wake_up_interruptible(&pdevinfo->rx_wq);

		rx_ring_buf = rx_rbuf->phy_addrs[wpos_new];
	}
	spin_unlock_irqrestore(&rx_rbuf->pos_lock, flags);

	if (wpos_new != rpos) {
		/* ext_start_rx() writes rx_rbuf->rpos in case of syncsound.
		 * However, in this case there are no other asynchronous
		 * readers or writers of rx_rbuf->rpos, so no lock needed.
		 * In case of char dev usage, ext_start_rx() doesn't change
		 * rx_rbuf->rpos, so no race-condition with readers/writers
		 * in poll() and do_read().
		 */
		ext_start_rx(minor, pdevinfo->channel_type, &rx_rbuf->rpos);

#ifdef DEBUG_RX
		pr_debug(DRIVER_NAME": recv package, rx_rpos: %d, rx_wpos: %d\n",
			 rpos, wpos_new);
#endif
	} else {
		/* Last buffer is for package drop */
		rx_ring_buf = pdevinfo->rx_bufs.phy_addrs[TRANS_RING_NODES];

		pr_debug(DRIVER_NAME": drop package, due to no space\n");
	}

	adt_sts = mlb150_dev_get_adt_sts(ch_addr);
	mlb_start_rx(ch_addr, ctype, adt_sts, rx_ring_buf);
}

static irqreturn_t mlb_ahb_isr(int irq, void *dev_id)
{
	struct mlb_data *drvdata = dev_id;
	u32 rx_int_sts, tx_int_sts, acsr0, acsr1, rx_err, tx_err, hcer0, hcer1;
	struct mlb_dev_info *pdev;
	struct mlb_channel_info *ptxchinfo, *prxchinfo;
	int minor;

	/* Step 5, Read the ACSRn registers to determine which channel or
	 * channels are causing the interrupt */
	acsr0 = READ_MLB_REG(REG_ACSR0);
	acsr1 = READ_MLB_REG(REG_ACSR1);

	hcer0 = READ_MLB_REG(REG_HCER0);
	hcer1 = READ_MLB_REG(REG_HCER1);

	/* Step 6, If ACTL.SCE = 1, write the result of step 5 back to ACSR0
	 * and ACSR1 to clear the interrupt */
	if (ACTL_SCE & READ_MLB_REG(REG_ACTL)) {
		WRITE_MLB_REG(acsr0, REG_ACSR0);
		WRITE_MLB_REG(acsr1, REG_ACSR1);
	}
#ifdef DEBUG_IRQ
	pr_info_ratelimited("mlb_ahb_isr(): irq=%i, ACSR0=0x%08X, ACSR1=0x%08X, HCER0=0x%08X, HCER1=0x%08X\n",
	     irq, acsr0, acsr1, hcer0, hcer1);
#endif

	for (minor = 0; minor < used_minor_devices; minor++) {
		pdev = &drvdata->devinfo[minor];
		prxchinfo = &pdev->channels[RX_CHANNEL];
		ptxchinfo = &pdev->channels[TX_CHANNEL];

		rx_int_sts = (prxchinfo->address < 32) ? acsr0 : acsr1;
		tx_int_sts = (ptxchinfo->address < 32) ? acsr0 : acsr1;
		rx_err = (prxchinfo->address < 32) ? hcer0 : hcer1;
		tx_err = (ptxchinfo->address < 32) ? hcer0 : hcer1;

		/* get tx channel interrupt status */
		if (tx_int_sts & (1 << (ptxchinfo->address % 32))) {
			if (!(tx_err &
				(1 << (ptxchinfo->address % 32)))) {
				mlb_tx_isr(pdev, minor);
			} else {
				pr_info("tx channel %d encountered an AHB error!\n",
					 ptxchinfo->address);
			}
		}

		/* get rx channel interrupt status */
		if (rx_int_sts & (1 << (prxchinfo->address % 32))) {
			if (!(rx_err
				& (1 << (prxchinfo->address % 32)))) {
				mlb_rx_isr(pdev, minor);
			} else {
				pr_info("rx channel %d encountered an AHB error!\n",
						prxchinfo->address);
			}
		}
	}

	return IRQ_HANDLED;
}

/* error interrupt */
static irqreturn_t mlb_isr(int irq, void *dev_id)
{
	struct mlb_data *drvdata = dev_id;
	u32 rx_int_sts, tx_int_sts, ms0, ms1, tx_cis, rx_cis, ctype;
	int minor;
	u32 cdt_val[4] = { 0 };

	/* Step 4, Read the MSn register to determine which channel(s)
	 * are causing the interrupt */
	ms0 = READ_MLB_REG(REG_MS0);
	ms1 = READ_MLB_REG(REG_MS1);

#ifdef DEBUG_IRQ
	pr_info_ratelimited("mlb_isr(): irq=%i, MS0=0x%08X, MS1=0x%08X\n",
		irq, ms0, ms1);
#endif

	for (minor = 0; minor < used_minor_devices; minor++) {
		struct mlb_dev_info *pdevinfo = &drvdata->devinfo[minor];
		u32 rx_mlb_ch = pdevinfo->channels[RX_CHANNEL].address;
		u32 tx_mlb_ch = pdevinfo->channels[TX_CHANNEL].address;
		tx_cis = 0;
		rx_cis = 0;

		ctype = pdevinfo->channel_type;
		rx_int_sts = (rx_mlb_ch < 32) ? ms0 : ms1;
		tx_int_sts = (tx_mlb_ch < 32) ? ms0 : ms1;

		/* Get tx channel interrupt status */
		if (tx_int_sts & (1 << (tx_mlb_ch % 32))) {
			mlb150_dev_cdt_read(tx_mlb_ch, cdt_val);
			pr_info_ratelimited(DRIVER_NAME": TX_CH: %d, cdt_val[3]: 0x%08x, cdt_val[2]: 0x%08x, cdt_val[1]: 0x%08x, cdt_val[0]: 0x%08x\n",
				tx_mlb_ch, cdt_val[3], cdt_val[2], cdt_val[1],
				cdt_val[0]);
			switch (ctype) {
			case MLB_CTYPE_SYNC:
				tx_cis = (cdt_val[2] & ~CDT_SYNC_WSTS_MASK)
				    >> CDT_SYNC_WSTS_SHIFT;
				/* Clear RSTS/WSTS errors to resume
				 * channel operation */
				/* a. For synchronous channels: WSTS[3] = 0 */
				cdt_val[2] &= ~(0x8 << CDT_SYNC_WSTS_SHIFT);
				break;
			case MLB_CTYPE_CTRL:
			case MLB_CTYPE_ASYNC:
				tx_cis = (cdt_val[2] &
					  ~CDT_CTRL_ASYNC_WSTS_MASK)
				    >> CDT_CTRL_ASYNC_WSTS_SHIFT;
				tx_cis = (cdt_val[3] & CDT_CTRL_ASYNC_WSTS_1) ?
				    (tx_cis | (0x1 << 4)) : tx_cis;
				/* b. For async and ctrl channels:
				 * RSTS[4]/WSTS[4] = 0
				 * and RSTS[2]/WSTS[2] = 0 */
				cdt_val[3] &= ~CDT_CTRL_ASYNC_WSTS_1;
				cdt_val[2] &=
				    ~(0x4 << CDT_CTRL_ASYNC_WSTS_SHIFT);
				break;
			case MLB_CTYPE_ISOC:
				tx_cis = (cdt_val[2] & ~CDT_ISOC_WSTS_MASK)
				    >> CDT_ISOC_WSTS_SHIFT;
				/* c. For isoc channels: WSTS[2:1] = 0x00 */
				cdt_val[2] &= ~(0x6 << CDT_ISOC_WSTS_SHIFT);
				break;
			default:
				break;
			}
			mlb150_dev_cdt_write(tx_mlb_ch, cdt_val);
		}

		/* Get rx channel interrupt status */
		if (rx_int_sts & (1 << (rx_mlb_ch % 32))) {
			mlb150_dev_cdt_read(rx_mlb_ch, cdt_val);
			pr_info_ratelimited(DRIVER_NAME": RX_CH: %d, cdt_val[3]: 0x%08x, cdt_val[2]: 0x%08x, cdt_val[1]: 0x%08x, cdt_val[0]: 0x%08x\n",
				rx_mlb_ch, cdt_val[3], cdt_val[2], cdt_val[1],
				cdt_val[0]);
			switch (ctype) {
			case MLB_CTYPE_SYNC:
				rx_cis = (cdt_val[2] & ~CDT_SYNC_RSTS_MASK)
				    >> CDT_SYNC_RSTS_SHIFT;
				cdt_val[2] &= ~(0x8 << CDT_SYNC_WSTS_SHIFT);
				break;
			case MLB_CTYPE_CTRL:
			case MLB_CTYPE_ASYNC:
				rx_cis =
				    (cdt_val[2] & ~CDT_CTRL_ASYNC_RSTS_MASK)
				    >> CDT_CTRL_ASYNC_RSTS_SHIFT;
				rx_cis = (cdt_val[3] & CDT_CTRL_ASYNC_RSTS_1) ?
				    (rx_cis | (0x1 << 4)) : rx_cis;
				cdt_val[3] &= ~CDT_CTRL_ASYNC_RSTS_1;
				cdt_val[2] &=
				    ~(0x4 << CDT_CTRL_ASYNC_RSTS_SHIFT);
				break;
			case MLB_CTYPE_ISOC:
				rx_cis = (cdt_val[2] & ~CDT_ISOC_RSTS_MASK)
				    >> CDT_ISOC_RSTS_SHIFT;
				cdt_val[2] &= ~(0x6 << CDT_ISOC_WSTS_SHIFT);
				break;
			default:
				break;
			}
			mlb150_dev_cdt_write(rx_mlb_ch, cdt_val);
		}

		if (!tx_cis && !rx_cis)
			continue;

		/* fill exception event */
		spin_lock(&pdevinfo->event_lock);
		pdevinfo->ex_event |= (rx_cis << 16) | tx_cis;
		spin_unlock(&pdevinfo->event_lock);
	}

	return IRQ_HANDLED;
}

int mxc_mlb150_do_open(struct mlb_data *drvdata, int minor,
			u8 **buf_virt, u32 *buf_phys)
{
	int ring_buf_size, buf_size, j, ret;
	void *buf_addr;
	dma_addr_t phyaddr;

	struct mlb_dev_info *pdevinfo = NULL;
	struct mlb_channel_info *pchinfo = NULL;

	if (!drvdata)
		return -ENODEV;

	pdevinfo = &drvdata->devinfo[minor];
	pchinfo = &pdevinfo->channels[TX_CHANNEL];

	if (minor < 0 || minor >= used_minor_devices) {
		pr_err("no device\n");
		return -ENODEV;
	}

	/* open for each channel device */
	if (atomic_cmpxchg(&pdevinfo->opencnt, 0, 1) != 0) {
		pr_err("busy\n");
		return -EBUSY;
	}

	ring_buf_size = CH_PACKET_BUF_SIZE(minor);
	buf_size = ring_buf_size * (TRANS_RING_NODES + 1) + PING_BUF_MAX_SIZE;
#ifdef MLB_USE_IRAM
	buf_addr = iram_alloc(buf_size, &phyaddr);
#else
	ret = dma_set_mask(drvdata->dev, DMA_BIT_MASK(32));
	if (ret) {
		ret = dma_set_coherent_mask(drvdata->dev, DMA_BIT_MASK(32));
		if (ret) {
			dev_err(drvdata->dev, DRIVER_NAME": No usable DMA configuration, aborting\n");
			return -ENOMEM;
		}
	}

	buf_addr =
	    dma_alloc_coherent(drvdata->dev, buf_size,
			       &phyaddr, GFP_KERNEL);
#endif
	if (unlikely(buf_addr == NULL)) {
		ret = -ENOMEM;
		dev_err(drvdata->dev, "can not alloc rx buffers\n");
		return ret;
	}
	memset(buf_addr, 0, buf_size);

	dev_dbg(drvdata->dev, DRIVER_NAME": ch_type: %s, RX ring buf virt base: 0x%p phy base: 0x%08x\n",
		ctype_names[pdevinfo->channel_type], buf_addr,
		phyaddr);

	if (buf_virt)
		*buf_virt = buf_addr;

	if (buf_phys)
		*buf_phys = phyaddr;

	for (j = 0; j < TRANS_RING_NODES + 1;
	     ++j, buf_addr += ring_buf_size, phyaddr += ring_buf_size) {
		pdevinfo->rx_bufs.virt_bufs[j] = buf_addr;
		pdevinfo->rx_bufs.phy_addrs[j] = phyaddr;
		pdevinfo->rx_bufs.size = pchinfo->buf_size;
	}

	/* set the virtual and physical buf head address */
	pchinfo->buf_ptr = buf_addr;
	pchinfo->buf_phy_addr = phyaddr;

	dev_dbg(drvdata->dev, "ctype: %s, tx phy_head: 0x%08x, buf_head: 0x%p\n",
		ctype_names[pdevinfo->channel_type],
		pchinfo->buf_phy_addr, pchinfo->buf_ptr);

	/* reset the buffer read/write ptr */
	pdevinfo->rx_bufs.rpos = 0;
	pdevinfo->rx_bufs.wpos = 0;
	pdevinfo->ex_event = 0;

	set_tx_busy(pdevinfo, 0);

	return 0;
}

static int mxc_mlb150_open(struct inode *inode, struct file *filp)
{
	int minor;
	struct mlb_data *drvdata;

	minor = MINOR(inode->i_rdev);
	drvdata = container_of(inode->i_cdev, struct mlb_data, cdev);

	filp->private_data = drvdata;

	return mxc_mlb150_do_open(drvdata, minor, NULL, NULL);
}

int mxc_mlb150_do_release(struct mlb_data *drvdata, int minor,
			  enum channelmode mode)
{
	u32 buf_size;
	struct mlb_dev_info *pdevinfo;

	if (!drvdata)
		return -ENODEV;

	pdevinfo = &drvdata->devinfo[minor];

#ifdef DEBUG
	mlb150_dev_dump_reg();
	mlb150_dev_dump_ctr_tbl(0, pdevinfo->channels[TX_CHANNEL].address + 1);
	mlb150_dev_dump_hex((const u8 *)pdevinfo->rx_bufs.virt_bufs[0],
			    pdevinfo->rx_bufs.size);
#endif

	/* clear channel settings and info */
	mlb_channel_enable(drvdata, minor, mode, 0, 0);

	buf_size =
	    CH_PACKET_BUF_SIZE(minor) * (TRANS_RING_NODES + 1) +
	    PING_BUF_MAX_SIZE;

#ifdef MLB_USE_IRAM
	iram_free(pdevinfo->rx_bufs.phy_addrs[0], buf_size);
#else
	dma_free_coherent(drvdata->dev, buf_size,
			  pdevinfo->rx_bufs.virt_bufs[0],
			  pdevinfo->rx_bufs.phy_addrs[0]);
#endif
	/* decrease the open count */
	atomic_set(&pdevinfo->opencnt, 0);

	return 0;
}

static int mxc_mlb150_release(struct inode *inode, struct file *filp)
{
	int minor;
	struct mlb_data *drvdata = filp->private_data;

	minor = MINOR(inode->i_rdev);

	return mxc_mlb150_do_release(drvdata, minor, MLB_RDWR);
}

/* caddr: tx/rx channel address: 0xttttrrrr */
void mxc_mlb150_chan_setaddr(struct mlb_data *drvdata, int minor,
			     unsigned int caddr)
{
	struct mlb_dev_info *pdevinfo = &drvdata->devinfo[minor];
	struct mlb_channel_info *tx_chinfo = &pdevinfo->channels[TX_CHANNEL];
	struct mlb_channel_info *rx_chinfo = &pdevinfo->channels[RX_CHANNEL];

	tx_chinfo->address = (caddr >> 16) & 0xFFFF;
	rx_chinfo->address = caddr & 0xFFFF;

	pr_debug
	    (DRIVER_NAME": set ch addr for channel type: %s, tx: %d, rx: %d\n",
	     ctype_names[pdevinfo->channel_type],
	     tx_chinfo->address, rx_chinfo->address);
}

int mxc_mlb150_chan_startup(struct mlb_data *drvdata, int minor,
			    enum channelmode mode,
			    unsigned int audio_channels)
{
	struct mlb_dev_info *pdevinfo;
	struct mlb_channel_info *tx_chinfo, *rx_chinfo;
	u32 ctype;
	int j;
	void *buf_addr;
	ulong phyaddr;
	unsigned int buf_size_new;

	if (!drvdata)
		return -ENODEV;

	pdevinfo = &drvdata->devinfo[minor];
	tx_chinfo = &pdevinfo->channels[TX_CHANNEL];
	rx_chinfo = &pdevinfo->channels[RX_CHANNEL];

	ctype = pdevinfo->channel_type;

	if (atomic_read(&mlb_devinfo[minor].on)) {
		pr_info(DRIVER_NAME": channel already started up\n");
		return 0;
	}

	if (ctype != MLB_CTYPE_SYNC) {
		if (audio_channels != 0) {
			pr_err(DRIVER_NAME": audio_channels must be 0 for channel type %d\n",
				ctype);
			return -EFAULT;
		}
	} else {
		if ((audio_channels < 1) || (audio_channels > 2)) {
			pr_err(DRIVER_NAME": audio_channels must be 1 or 2 for sync channels\n");
			return -EFAULT;
		}

		/* fix buffer size for sync audio channels */
		buf_size_new = SYNC_BUFFER_DEP_MIN * audio_channels;
		tx_chinfo->buf_size = buf_size_new;
		rx_chinfo->buf_size = buf_size_new;

		buf_addr = pdevinfo->rx_bufs.virt_bufs[0];
		phyaddr = pdevinfo->rx_bufs.phy_addrs[0];
		for (j = 0; j < TRANS_RING_NODES + 1;
		     ++j, buf_addr += buf_size_new, phyaddr += buf_size_new) {
			pdevinfo->rx_bufs.virt_bufs[j] = buf_addr;
			pdevinfo->rx_bufs.phy_addrs[j] = phyaddr;
			pdevinfo->rx_bufs.size = buf_size_new;
		}
	}

	pr_debug(DRIVER_NAME": start channel on minor %d, tx: %d, rx: %d\n",
		 minor, tx_chinfo->address, rx_chinfo->address);
	mlb_channel_enable(drvdata, minor, mode, 1, audio_channels);
	return 0;
}

int mxc_mlb150_chan_shutdown(struct mlb_data *drvdata, int minor,
			     enum channelmode mode)
{
	struct mlb_dev_info *pdevinfo;
	struct mlb_channel_info *tx_chinfo, *rx_chinfo;

	if (!drvdata)
		return -ENODEV;

	pdevinfo = &drvdata->devinfo[minor];
	tx_chinfo = &pdevinfo->channels[TX_CHANNEL];
	rx_chinfo = &pdevinfo->channels[RX_CHANNEL];

	if (atomic_read(&pdevinfo->on) == 0) {
		pr_debug(DRIVER_NAME": channel already shut down\n");
		return -1;
	}

	pr_debug(DRIVER_NAME": shutdown channel, tx: %d, rx: %d\n",
		 tx_chinfo->address, rx_chinfo->address);
	mlb_channel_enable(drvdata, minor, mode, 0, 0);
	return 0;
}

static long mxc_mlb150_ioctl(struct file *filp,
			     unsigned int cmd, unsigned long arg)
{
	struct inode *inode = filp->f_dentry->d_inode;
	struct mlb_data *drvdata = filp->private_data;
	struct mlb_dev_info *pdevinfo = NULL;
	void __user *argp = (void __user *)arg;
	unsigned long flags, event;
	int minor;

	minor = MINOR(inode->i_rdev);
	pdevinfo = &drvdata->devinfo[minor];

	switch (cmd) {
	case MLB_CHAN_SETADDR:
		{
			unsigned int caddr;
			/* get channel address from user space */
			if (copy_from_user(&caddr, argp, sizeof(caddr))) {
				pr_err(DRIVER_NAME": copy from user failed\n");
				return -EFAULT;
			}
			mxc_mlb150_chan_setaddr(drvdata, minor, caddr);
			break;
		}
	case MLB_CHAN_STARTUP:
		mxc_mlb150_chan_startup(drvdata, minor, MLB_RDWR, 0);
		break;
	case MLB_SYNC_CHAN_STARTUP:
		{
			unsigned int audio_channels;
			int ret;
			/* get channel address from user space */
			if (copy_from_user(&audio_channels, argp,
					   sizeof(audio_channels))) {
				pr_err(DRIVER_NAME": copy from user failed\n");
				return -EFAULT;
			}
			ret = mxc_mlb150_chan_startup(drvdata, minor, MLB_RDWR,
						      audio_channels);
			if (ret)
				return ret;
			break;
		}
	case MLB_CHAN_SHUTDOWN:
		mxc_mlb150_chan_shutdown(drvdata, minor, MLB_RDWR);
		break;
	case MLB_CHAN_GETEVENT:
		/* get and clear the ex_event */
		spin_lock_irqsave(&pdevinfo->event_lock, flags);
		event = pdevinfo->ex_event;
		pdevinfo->ex_event = 0;
		spin_unlock_irqrestore(&pdevinfo->event_lock, flags);

		pr_debug(DRIVER_NAME": get event\n");
		if (event) {
			if (copy_to_user(argp, &event, sizeof(event))) {
				pr_err(DRIVER_NAME": copy to user failed\n");
				return -EFAULT;
			}
		} else {
			pr_debug(DRIVER_NAME": no exception event now\n");
			return -EAGAIN;
		}
		break;
	case MLB_SET_FPS:
		{
			u32 fps, c0_val;

			/* get fps from user space */
			if (copy_from_user(&fps, argp, sizeof(fps))) {
				pr_err(DRIVER_NAME": copy from user failed\n");
				return -EFAULT;
			}

			c0_val = READ_MLB_REG(REG_MLBC0);
			c0_val &= ~MLBC0_MLBCLK_MASK;

			/* check fps value */
			switch (fps) {
			case 256:
			case 512:
			case 1024:
				pdevinfo->fps = fps >> 9;
				c0_val &= ~MLBC0_MLBPEN;
				c0_val |= (fps >> 9)
				    << MLBC0_MLBCLK_SHIFT;
				break;
			case 2048:
			case 3072:
			case 4096:
				pdevinfo->fps = (fps >> 10) + 1;
				c0_val |= ((fps >> 10) + 1)
				    << MLBC0_MLBCLK_SHIFT;
				break;
			case 6144:
				pdevinfo->fps = fps >> 10;
				c0_val |= ((fps >> 10) + 1)
				    << MLBC0_MLBCLK_SHIFT;
				break;
			case 8192:
				pdevinfo->fps = (fps >> 10) - 1;
				c0_val |= ((fps >> 10) - 1)
				    << MLBC0_MLBCLK_SHIFT;
				break;
			default:
				pr_debug
				    (DRIVER_NAME": invalid fps argument: %d\n",
				     fps);
				return -EINVAL;
			}

			WRITE_MLB_REG(c0_val, REG_MLBC0);

			pr_info(DRIVER_NAME": set fps to %d, MLBC0: 0x%08x\n",
				fps,
				(u32) READ_MLB_REG(REG_MLBC0));

			break;
		}

	case MLB_GET_VER:
		{
			u32 version;

			/* get MLB device module version */
			version = 0x03030003;

#ifdef DEBUG_MLBLOCK
			version |=
			    READ_MLB_REG(REG_MLBC0) & MLBC0_MLBLK;
#endif

			pr_debug(DRIVER_NAME": get version: 0x%08x\n", version);

			if (copy_to_user(argp, &version, sizeof(version))) {
				pr_err(DRIVER_NAME": copy to user failed\n");
				return -EFAULT;
			}
			break;
		}

	case MLB_SET_DEVADDR:
		{
			u32 c1_val;
			u8 devaddr;

			/* get MLB device address from user space */
			if (copy_from_user
				     (&devaddr, argp, sizeof(unsigned char))) {
				pr_err(DRIVER_NAME": copy from user failed\n");
				return -EFAULT;
			}

			c1_val = READ_MLB_REG(REG_MLBC1);
			c1_val &= ~MLBC1_NDA_MASK;
			c1_val |= devaddr << MLBC1_NDA_SHIFT;
			WRITE_MLB_REG(c1_val, REG_MLBC1);
			pr_info(DRIVER_NAME": set dev addr, dev addr: %d, MLBC1: 0x%08x\n",
				devaddr, (u32) READ_MLB_REG(REG_MLBC1));

			break;
		}
	default:
		pr_info(DRIVER_NAME": Invalid ioctl command\n");
		return -EINVAL;
	}

	return 0;
}

/*!
 * MLB read routine
 *
 * Read the current received data from queued buffer,
 * and free this buffer for hw to fill ingress data.
 */
static ssize_t mxc_mlb150_read(struct file *filp, char __user *buf,
			       size_t count, loff_t *f_pos)
{
	int minor, ret;
	int size;
	u32 rpos, wpos;
	unsigned long flags;
	struct mlb_data *drvdata = filp->private_data;
	struct mlb_dev_info *pdevinfo;
	struct mlb_ringbuf *rx_rbuf;
	struct mlb_channel_info *pchinfo;

#ifdef DEBUG_RX
	pr_debug(DRIVER_NAME": mxc_mlb150_read\n");
#endif

	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
	pdevinfo = &drvdata->devinfo[minor];
	pchinfo = &pdevinfo->channels[RX_CHANNEL];

	rx_rbuf = &pdevinfo->rx_bufs;

	spin_lock_irqsave(&rx_rbuf->pos_lock, flags);
	rpos = rx_rbuf->rpos;
	wpos = rx_rbuf->wpos;
	spin_unlock_irqrestore(&rx_rbuf->pos_lock, flags);

	/* check the current rx buffer is available or not */
	if (rpos == wpos) {
		if (filp->f_flags & O_NONBLOCK)
			return -EAGAIN;
		/* if !O_NONBLOCK, we wait for recv packet */
		ret = wait_event_interruptible(pdevinfo->rx_wq,
					       (rx_rbuf->wpos != rpos));
		if (ret < 0)
			return ret;
	}

	size = pchinfo->buf_size;
	if (size > count) {
		/* the user buffer is too small */
		pr_warn(DRIVER_NAME": received data size is bigger than size: %d, count: %d\n",
			    size, count);
		return -EINVAL;
	}

	/* copy rx buffer data to user buffer */
	if (copy_to_user(buf, rx_rbuf->virt_bufs[rpos], size)) {
		pr_err(DRIVER_NAME": copy to userspace failed\n");
		return -EFAULT;
	}

	/* update the read ptr after having copied the data */
	spin_lock_irqsave(&rx_rbuf->pos_lock, flags);
	rx_rbuf->rpos = (rpos + 1) % TRANS_RING_NODES;
	spin_unlock_irqrestore(&rx_rbuf->pos_lock, flags);

	*f_pos = 0;

	return size;

}

static
ssize_t mxc_mlb150_do_write(struct mlb_data *drvdata, int minor,
			    const char __user *buf, size_t count,
			    int nonblock, int user)
{
	int ret = 0;
	struct mlb_channel_info *pchinfo = NULL;
	struct mlb_dev_info *pdevinfo = NULL;

	pdevinfo = &drvdata->devinfo[minor];
	pchinfo = &pdevinfo->channels[TX_CHANNEL];

	/* check the current tx buffer is used or not */
	if (nonblock) {
		if (get_tx_busy(pdevinfo) != 0)
			return -EAGAIN;
	} else {
		ret = wait_event_interruptible_timeout(pdevinfo->tx_wq,
			0 == get_tx_busy(pdevinfo), msecs_to_jiffies(50));

		if (ret <= 0) {
			/* Write adt entry to 0 in this case.
			 * This way, the next write will change RDY from 0 to 1.
			 */
			u32 val = 0;
			mlb150_dev_adt_write(pchinfo->address, &val);

			if (ret == 0)
				set_tx_busy(pdevinfo, 0);

			pr_debug(DRIVER_NAME": wait_event_interruptible ret = %d\n",
				ret);
			goto out;
		}
	}

	if (copy_from_user(pchinfo->buf_ptr, buf, count)) {
		pr_err(DRIVER_NAME": copy from user failed\n");
		ret = -EFAULT;
		goto out;
	}

	lld_dump((minor == 0) ? "T(c):" : "T(a):",
		(void *)pchinfo->buf_ptr, count);

	if (mlb_start_tx(drvdata, minor, pchinfo->buf_phy_addr) != 0)
		pr_err(DRIVER_NAME": mlb_start_tx returns false!\n");

	ret = count;

out:
	return ret;
}

/*!
 * MLB write routine
 *
 * Copy the user data to tx channel buffer,
 * and prepare the channel current/next buffer ptr.
 */
static ssize_t mxc_mlb150_write(struct file *filp, const char __user *buf,
				size_t count, loff_t *f_pos)
{
	int minor, nonblock;
	struct mlb_data *drvdata = filp->private_data;
	struct mlb_dev_info *pdevinfo;
	struct mlb_channel_info *pchinfo;

	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
	pdevinfo = &drvdata->devinfo[minor];
	pchinfo = &pdevinfo->channels[TX_CHANNEL];

	nonblock = filp->f_flags & O_NONBLOCK;

	if (count > pchinfo->buf_size) {
		/* too many data to write */
		pr_warn(DRIVER_NAME": overflow write data\n");
		return -EFBIG;
	}

	*f_pos = 0;

	return mxc_mlb150_do_write(drvdata, minor, buf, count, nonblock, 1);
}

static unsigned int mxc_mlb150_poll(struct file *filp,
				    struct poll_table_struct *wait)
{
	int minor;
	unsigned int ret = 0;
	unsigned long flags;
	struct mlb_data *drvdata = filp->private_data;
	struct mlb_dev_info *pdevinfo;

	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
	pdevinfo = &drvdata->devinfo[minor];

	pr_debug(DRIVER_NAME": mxc_mlb150_poll tx ch %d, rx ch %d\n",
		 pdevinfo->channels[TX_CHANNEL].address,
		 pdevinfo->channels[RX_CHANNEL].address);

	poll_wait(filp, &pdevinfo->rx_wq, wait);
	poll_wait(filp, &pdevinfo->tx_wq, wait);

	/* check the tx buffer is avaiable or not */
	if (get_tx_busy(pdevinfo) == 0)
		ret |= POLLOUT | POLLWRNORM;

	/* check the rx buffer filled or not */
	spin_lock_irqsave(&pdevinfo->rx_bufs.pos_lock, flags);
	if (pdevinfo->rx_bufs.rpos != pdevinfo->rx_bufs.wpos)
		ret |= POLLIN | POLLRDNORM;
	spin_unlock_irqrestore(&pdevinfo->rx_bufs.pos_lock, flags);

	/* check the exception event */
	if (pdevinfo->ex_event)
		ret |= POLLIN | POLLRDNORM;

	return ret;
}

/*!
 * char dev file operations structure
 */
static const struct file_operations mxc_mlb150_fops = {

	.owner = THIS_MODULE,
	.open = mxc_mlb150_open,
	.release = mxc_mlb150_release,
	.unlocked_ioctl = mxc_mlb150_ioctl,
	.poll = mxc_mlb150_poll,
	.read = mxc_mlb150_read,
	.write = mxc_mlb150_write,
};

/*!
 * platform driver structure for MLB
 */
static struct of_device_id mxc_mlb150_of_device_ids[] = {
	{.compatible = "smsc,mxc_mlb150"},
	{},
};

/*!
 * This function is called whenever the MLB device is detected.
 */
static int mxc_mlb150_probe(struct platform_device *pdev)
{
	int ret, mlb_major, i;
	struct mlb_data *drvdata;
	const struct of_device_id *match;

	match = of_match_device(mxc_mlb150_of_device_ids, &pdev->dev);

	if (!match) {
		dev_dbg(&pdev->dev, "mlb device does not match driver\n");
		return -1;
	}

	drvdata = kzalloc(sizeof(struct mlb_data), GFP_KERNEL);
	if (!drvdata) {
		dev_err(&pdev->dev, "can't allocate enough memory\n");
		return -ENOMEM;
	}

	drvdata->devinfo = mlb_devinfo;
	drvdata->dev = &pdev->dev;

	/* give platform data pointer to ext sync stuff */
	ext_set_drvdata(drvdata);

	/**
	 * Register MLB lld as four character devices
	 */
	ret = alloc_chrdev_region(&drvdata->firstdev, 0,
			used_minor_devices, "mxc_mlb150");
	mlb_major = MAJOR(drvdata->firstdev);
	dev_dbg(&pdev->dev, "MLB device major: %d\n", mlb_major);

	if (ret < 0) {
		dev_err(&pdev->dev, "can't get major %d\n", mlb_major);
		goto err_reg; /*2;*/
	}

	cdev_init(&drvdata->cdev, &mxc_mlb150_fops);
	drvdata->cdev.owner = THIS_MODULE;

	ret = cdev_add(&drvdata->cdev, drvdata->firstdev, used_minor_devices);
	if (ret) {
		dev_err(&pdev->dev, "can't add cdev\n");
		goto err_reg; /*2;*/
	}

	/* create class and device for udev information */
	drvdata->class = class_create(THIS_MODULE, DRIVER_NAME);
	if (IS_ERR(drvdata->class)) {
		dev_err(&pdev->dev, "failed to create device class\n");
		ret = -ENOMEM;
		goto err_reg; /*2;*/
	}

	for (i = 0; i < used_minor_devices; i++) {
		struct device *class_dev;
		class_dev = device_create(drvdata->class, NULL,
				MKDEV(mlb_major, i),
				NULL, mlb_devinfo[i].dev_name);
		if (IS_ERR(class_dev)) {
			dev_err(&pdev->dev, "failed to create mlb150 %s class device\n",
					mlb_devinfo[i].dev_name);
			ret = -ENOMEM;
			goto err_dev; /*1;*/
		}
	}

	/* get irq */
	/* ahb0 irq */
	drvdata->irq_ahb0 = irq_of_parse_and_map(pdev->dev.of_node, 1);
	dev_dbg(&pdev->dev, "ahb0_irq: %d\n", drvdata->irq_ahb0);
	if (request_irq(drvdata->irq_ahb0, mlb_ahb_isr, IRQF_TRIGGER_HIGH,
			"mlb_ahb0", drvdata)) {
		dev_err(&pdev->dev, "can't claim irq %d\n",
			drvdata->irq_ahb0);
		drvdata->irq_ahb0 = 0;
		goto err_irq; /*0;*/
	}

	/* ahb1 irq */
	drvdata->irq_ahb1 = irq_of_parse_and_map(pdev->dev.of_node, 2);
	dev_dbg(&pdev->dev, "ahb1_irq: %d\n", drvdata->irq_ahb1);
	if (request_irq(drvdata->irq_ahb1, mlb_ahb_isr, IRQF_TRIGGER_HIGH,
			"mlb_ahb1", drvdata)) {
		dev_err(&pdev->dev, "can't claim irq %d\n",
			drvdata->irq_ahb1);
		drvdata->irq_ahb1 = 0;
		goto err_irq; /*0;*/
	}

	/* mlb irq */
	drvdata->irq_mlb = irq_of_parse_and_map(pdev->dev.of_node, 0);
	dev_dbg(&pdev->dev, "mlb_irq: %d\n", drvdata->irq_mlb);
	if (request_irq(drvdata->irq_mlb, mlb_isr, IRQF_TRIGGER_HIGH,
			"mlb", drvdata)) {
		dev_err(&pdev->dev, "can't claim irq %d\n",
			drvdata->irq_mlb);
		drvdata->irq_mlb = 0;
		goto err_irq; /*0;*/
	}

	/* ioremap from phy mlb to kernel space */
	mlb_base = of_iomap(pdev->dev.of_node, 0);
	dev_dbg(&pdev->dev, "mapped base address: 0x%p\n", mlb_base);
	if (!mlb_base) {
		dev_err(&pdev->dev,
			"failed to get ioremap base\n");
		goto err_irq; /* 0*/
	}
	drvdata->membase = mlb_base;

	/* enable clock */
	drvdata->clk_mlb3p = devm_clk_get(&pdev->dev, "mlb");
	if (IS_ERR(drvdata->clk_mlb3p)) {
		dev_err(&pdev->dev, "unable to get mlb clock\n");
		ret = PTR_ERR(drvdata->clk_mlb3p);
		goto err_clk; /*0;*/
	}
	clk_prepare_enable(drvdata->clk_mlb3p);

	platform_set_drvdata(pdev, drvdata);

	mlb150_dev_init();

	return 0;

err_clk:
	clk_disable(drvdata->clk_mlb3p);
err_irq: /*0:*/
	if (drvdata->irq_ahb0) {
		free_irq(drvdata->irq_ahb0, drvdata);
		drvdata->irq_ahb0 = 0;
	}
	if (drvdata->irq_ahb1) {
		free_irq(drvdata->irq_ahb1, drvdata);
		drvdata->irq_ahb1 = 0;
	}
	if (drvdata->irq_mlb) {
		free_irq(drvdata->irq_mlb, drvdata);
		drvdata->irq_mlb = 0;
	}
	if (drvdata->membase)
		iounmap(drvdata->membase);
err_dev: /*1:*/
	for (--i; i >= 0; i--)
		device_destroy(drvdata->class, MKDEV(mlb_major, i));

	class_destroy(drvdata->class);
	drvdata->class = NULL;
err_reg: /*2:*/
	unregister_chrdev_region(drvdata->firstdev, used_minor_devices);

	kfree(drvdata);

	return ret;
}

static int mxc_mlb150_remove(struct platform_device *pdev)
{
	struct mlb_data *drvdata = platform_get_drvdata(pdev);

	mlb150_dev_exit();

	/* disable mlb clock */
	clk_disable_unprepare(drvdata->clk_mlb3p);
	clk_put(drvdata->clk_mlb3p);

	/* iounmap */
	if (drvdata->membase) {
		iounmap(drvdata->membase);
		drvdata->membase = NULL;
	}

	if (drvdata->irq_ahb0)
		free_irq(drvdata->irq_ahb0, drvdata);
	if (drvdata->irq_ahb1)
		free_irq(drvdata->irq_ahb1, drvdata);
	if (drvdata->irq_mlb)
		free_irq(drvdata->irq_mlb, drvdata);
	drvdata->irq_ahb0 = 0;
	drvdata->irq_ahb1 = 0;
	drvdata->irq_mlb = 0;

	/* destroy mlb device class */
	if (drvdata->class) {
		int i;
		for (i = used_minor_devices - 1; i >= 0; i--)
			device_destroy(drvdata->class,
				MKDEV(MAJOR(drvdata->firstdev), i));
		class_destroy(drvdata->class);
		drvdata->class = NULL;
	}

	cdev_del(&drvdata->cdev);

	/* Unregister the two MLB devices */
	unregister_chrdev_region(drvdata->firstdev, used_minor_devices);

	kfree(drvdata);

	return 0;
}

#ifdef CONFIG_PM
static int mxc_mlb150_suspend(struct platform_device *pdev, pm_message_t state)
{
	return 0;
}

static int mxc_mlb150_resume(struct platform_device *pdev)
{
	return 0;
}
#else
#define mxc_mlb150_suspend NULL
#define mxc_mlb150_resume NULL
#endif

/* fill in requested number of sync minor devices */
static void init_sync_devices(void)
{
	u32 i;
	u32 index = 0;

	if (number_sync_channels > MLB_MAX_SYNC_DEVICES) {
		number_sync_channels = MLB_MAX_SYNC_DEVICES;
		pr_warn("Requested number of sync channels too large; limited to %d.\n",
				MLB_MAX_SYNC_DEVICES);
	}

	used_minor_devices = MLB_STATIC_MINOR_DEVS + number_sync_channels;

	for (i = MLB_STATIC_MINOR_DEVS; i < used_minor_devices; i++) {

		sprintf(mlb_devinfo[i].dev_name, "%s%d", MLB_SYNC_DEV_NAME,
			index);
		mlb_devinfo[i].channel_type = MLB_CTYPE_SYNC;

		mlb_devinfo[i].channels[TX_CHANNEL].buf_size =
		    CH_SYNC_BUF_DEP;
		mlb_devinfo[i].channels[TX_CHANNEL].dbr_buf_head =
		    CH_SYNCN_DBR_BUF_OFFSET(index);
		mlb_devinfo[i].channels[RX_CHANNEL].buf_size =
		    CH_SYNC_BUF_DEP;
		mlb_devinfo[i].channels[RX_CHANNEL].dbr_buf_head =
		    CH_SYNCN_DBR_BUF_OFFSET(index) +
		    CH_SYNC_BUF_DEP;

		atomic_set(&mlb_devinfo[i].on, 0);
		atomic_set(&mlb_devinfo[i].opencnt, 0);
		init_waitqueue_head(&mlb_devinfo[i].rx_wq);
		init_waitqueue_head(&mlb_devinfo[i].tx_wq);
		spin_lock_init(&mlb_devinfo[i].event_lock);
		spin_lock_init(&mlb_devinfo[i].tx_busy_lock);
		spin_lock_init(&mlb_devinfo[i].rx_bufs.pos_lock);
		index++;
	}
}

/*!
 * platform driver structure for MLB
 */
static struct platform_driver mxc_mlb150_driver = {
	.driver = {
		.name = DRIVER_NAME,
		.owner = THIS_MODULE,
		.of_match_table = mxc_mlb150_of_device_ids,
	},
	.probe = mxc_mlb150_probe,
	.remove = mxc_mlb150_remove,
	.suspend = mxc_mlb150_suspend,
	.resume = mxc_mlb150_resume,
};

static int __init mxc_mlb150_init(void)
{
	init_sync_devices();

	return platform_driver_register(&mxc_mlb150_driver);
}

static void __exit mxc_mlb150_exit(void)
{
	platform_driver_unregister(&mxc_mlb150_driver);
}

module_init(mxc_mlb150_init);
module_exit(mxc_mlb150_exit);

MODULE_AUTHOR("Freescale Semiconductor, Inc.");
MODULE_DESCRIPTION("MLB150 low level driver");
MODULE_LICENSE("GPL");
