/*
 * Copyright (C) 2012 Mentor Graphics Inc.
 * Copyright (C) 2005-2009 Freescale Semiconductor, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */
#include <linux/export.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/err.h>
#include <linux/platform_device.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/clkdev.h>

#include "ipu-prv.h"

struct ipu_smfc {
	void __iomem *base;
	u32 module;
	spinlock_t lock;
	int use_count;
	struct ipu_soc *ipu;
};


/* SMFC Register Offsets */
#define SMFC_MAP                0x0000
#define SMFC_WMC                0x0004
#define SMFC_BS                 0x0008

/* SMFC Register Fields */
#define SMFC_MAP_CH0_MASK                  0x00000007L
#define SMFC_MAP_CH0_SHIFT                 0
#define SMFC_MAP_CH1_MASK                  0x00000038L
#define SMFC_MAP_CH1_SHIFT                 3
#define SMFC_MAP_CH2_MASK                  0x000001C0L
#define SMFC_MAP_CH2_SHIFT                 6
#define SMFC_MAP_CH3_MASK                  0x00000E00L
#define SMFC_MAP_CH3_SHIFT                 9

#define SMFC_WM0_SET_MASK                  0x00000007L
#define SMFC_WM0_SET_SHIFT                 0
#define SMFC_WM1_SET_MASK                  0x000001C0L
#define SMFC_WM1_SET_SHIFT                 6
#define SMFC_WM2_SET_MASK                  0x00070000L
#define SMFC_WM2_SET_SHIFT                 16
#define SMFC_WM3_SET_MASK                  0x01C00000L
#define SMFC_WM3_SET_SHIFT                 22

#define SMFC_WM0_CLR_MASK                  0x00000038L
#define SMFC_WM0_CLR_SHIFT                 3
#define SMFC_WM1_CLR_MASK                  0x00000E00L
#define SMFC_WM1_CLR_SHIFT                 9
#define SMFC_WM2_CLR_MASK                  0x00380000L
#define SMFC_WM2_CLR_SHIFT                 19
#define SMFC_WM3_CLR_MASK                  0x0E000000L
#define SMFC_WM3_CLR_SHIFT                 25

#define SMFC_BS0_MASK                      0x0000000FL
#define SMFC_BS0_SHIFT                     0
#define SMFC_BS1_MASK                      0x000000F0L
#define SMFC_BS1_SHIFT                     4
#define SMFC_BS2_MASK                      0x00000F00L
#define SMFC_BS2_SHIFT                     8
#define SMFC_BS3_MASK                      0x0000F000L
#define SMFC_BS3_SHIFT                     12

static inline u32 ipu_smfc_read(struct ipu_smfc *smfc, unsigned offset)
{
	return readl(smfc->base + offset);
}

static inline void ipu_smfc_write(struct ipu_smfc *smfc, u32 value,
				 unsigned offset)
{
	writel(value, smfc->base + offset);
}

void ipu_smfc_dump(struct ipu_smfc *smfc)
{
	dev_dbg(smfc->ipu->dev, "SMFC_MAP: %08x\n",
		ipu_smfc_read(smfc, SMFC_MAP));
	dev_dbg(smfc->ipu->dev, "SMFC_WMC: %08x\n",
		ipu_smfc_read(smfc, SMFC_WMC));
	dev_dbg(smfc->ipu->dev, "SMFC_BS: %08x\n",
		ipu_smfc_read(smfc, SMFC_BS));
}
EXPORT_SYMBOL_GPL(ipu_smfc_dump);

/*
 * ipu_smfc_map
 *	Maps frames from given CSI and MIPI CSI2 virtual channel to given
 *      IDMAC channel.
 *
 * @param	channel		IDMAC channel, 0-3
 * @param	csi_id		CSI id, 0 or 1
 * @param	mipi_vc		virtual channel, 0-3
 */
void ipu_smfc_map(struct ipu_smfc *smfc, struct ipuv3_channel *channel,
		  int csi_id, int mipi_vc)
{
	unsigned long flags;
	u32 temp;

	spin_lock_irqsave(&smfc->lock, flags);

	temp = ipu_smfc_read(smfc, SMFC_MAP);

	switch (channel->num) {
	case IPUV3_CHANNEL_CSI0:
		temp &= ~SMFC_MAP_CH0_MASK;
		temp |= ((csi_id << 2) | mipi_vc) << SMFC_MAP_CH0_SHIFT;
		break;
	case IPUV3_CHANNEL_CSI1:
		temp &= ~SMFC_MAP_CH1_MASK;
		temp |= ((csi_id << 2) | mipi_vc) << SMFC_MAP_CH1_SHIFT;
		break;
	case IPUV3_CHANNEL_CSI2:
		temp &= ~SMFC_MAP_CH2_MASK;
		temp |= ((csi_id << 2) | mipi_vc) << SMFC_MAP_CH2_SHIFT;
		break;
	case IPUV3_CHANNEL_CSI3:
		temp &= ~SMFC_MAP_CH3_MASK;
		temp |= ((csi_id << 2) | mipi_vc) << SMFC_MAP_CH3_SHIFT;
		break;
	default:
		goto out;
	}

	ipu_smfc_write(smfc, temp, SMFC_MAP);
out:
	spin_unlock_irqrestore(&smfc->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_smfc_map);

/*
 * ipu_smfc_set_wmc
 *	Caution: The number of required channels,  the enabled channels
 *	and the FIFO size per channel are configured restrictedly.
 *
 * @param	channel		IDMAC channel 0-3
 * @param	set		set 1 or clear 0
 * @param	level		water mark level when FIFO is on the
 *				relative size
 */
void ipu_smfc_set_wmc(struct ipu_smfc *smfc,
		      struct ipuv3_channel *channel,
		      bool set, u32 level)
{
	unsigned long flags;
	u32 temp;

	spin_lock_irqsave(&smfc->lock, flags);

	temp = ipu_smfc_read(smfc, SMFC_WMC);

	switch (channel->num) {
	case IPUV3_CHANNEL_CSI0:
		if (set == true) {
			temp &= ~SMFC_WM0_SET_MASK;
			temp |= level << SMFC_WM0_SET_SHIFT;
		} else {
			temp &= ~SMFC_WM0_CLR_MASK;
			temp |= level << SMFC_WM0_CLR_SHIFT;
		}
		break;
	case IPUV3_CHANNEL_CSI1:
		if (set == true) {
			temp &= ~SMFC_WM1_SET_MASK;
			temp |= level << SMFC_WM1_SET_SHIFT;
		} else {
			temp &= ~SMFC_WM1_CLR_MASK;
			temp |= level << SMFC_WM1_CLR_SHIFT;
		}
		break;
	case IPUV3_CHANNEL_CSI2:
		if (set == true) {
			temp &= ~SMFC_WM2_SET_MASK;
			temp |= level << SMFC_WM2_SET_SHIFT;
		} else {
			temp &= ~SMFC_WM2_CLR_MASK;
			temp |= level << SMFC_WM2_CLR_SHIFT;
		}
		break;
	case IPUV3_CHANNEL_CSI3:
		if (set == true) {
			temp &= ~SMFC_WM3_SET_MASK;
			temp |= level << SMFC_WM3_SET_SHIFT;
		} else {
			temp &= ~SMFC_WM3_CLR_MASK;
			temp |= level << SMFC_WM3_CLR_SHIFT;
		}
		break;
	default:
		goto out;
	}

	ipu_smfc_write(smfc, temp, SMFC_WMC);
out:
	spin_unlock_irqrestore(&smfc->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_smfc_set_wmc);

/*
 * ipu_smfc_set_burst_size
 *
 * @param	channel		IDMAC channel 0-3
 * @param	pixfmt		pixel format
 */
void ipu_smfc_set_burst_size(struct ipu_smfc *smfc,
			     struct ipuv3_channel *channel,
			     u32 burst_size, bool passthrough_16)
{
	unsigned long flags;
	u32 fifo_acc_per_burst;
	u32 temp, bs;

	/*
	 * The imx6 TRM states that SMFC_BS register
	 * should be set to NPB[6:4] for 8 and 16 bit-per-pixel,
	 * when generic pixel format is used (PFS=6).
	 *
	 * The TRM goes on to say that the register is
	 * "the number of IDMAC's active accesses that will done
	 * for each IDMAC's burst.".
	 *
	 * It's not too clear what that means, but an educated
	 * guess is that the SMFC_BS register holds the
	 * number of SMFC FIFO lines accessed per burst from
	 * the IDMAC.
	 *
	 * If true, and assuming a FIFO line holds 128 bits,
	 * then SMFC_BS register should be set to
	 *
	 *     NPB * bpp / 128
	 *
	 * So for 16 bpp, that would mean NPB[6:3], not NPB[6:4].
	 * Only for 8 bpp, would it correctly be NPB[6:4].
	 *
	 * We don't support 8-bit generic, so ignore that.
	 */

	if (passthrough_16)
		fifo_acc_per_burst = burst_size >> 3;
	else
		fifo_acc_per_burst = burst_size >> 2;

	bs = fifo_acc_per_burst - 1;

	spin_lock_irqsave(&smfc->lock, flags);

	temp = ipu_smfc_read(smfc, SMFC_BS);

	switch (channel->num) {
	case IPUV3_CHANNEL_CSI0:
		temp &= ~SMFC_BS0_MASK;
		temp |= bs << SMFC_BS0_SHIFT;
		break;
	case IPUV3_CHANNEL_CSI1:
		temp &= ~SMFC_BS1_MASK;
		temp |= bs << SMFC_BS1_SHIFT;
		break;
	case IPUV3_CHANNEL_CSI2:
		temp &= ~SMFC_BS2_MASK;
		temp |= bs << SMFC_BS2_SHIFT;
		break;
	case IPUV3_CHANNEL_CSI3:
		temp &= ~SMFC_BS3_MASK;
		temp |= bs << SMFC_BS3_SHIFT;
		break;
	default:
		goto out;
	}

	ipu_smfc_write(smfc, temp, SMFC_BS);
out:
	spin_unlock_irqrestore(&smfc->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_smfc_set_burst_size);

int ipu_smfc_enable(struct ipu_smfc *smfc)
{
	unsigned long flags;

	spin_lock_irqsave(&smfc->lock, flags);

	if (!smfc->use_count)
		ipu_module_enable(smfc->ipu, smfc->module);

	smfc->use_count++;

	spin_unlock_irqrestore(&smfc->lock, flags);

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_smfc_enable);

int ipu_smfc_disable(struct ipu_smfc *smfc)
{
	unsigned long flags;

	spin_lock_irqsave(&smfc->lock, flags);

	smfc->use_count--;

	if (!smfc->use_count)
		ipu_module_disable(smfc->ipu, smfc->module);

	if (smfc->use_count < 0)
		smfc->use_count = 0;

	spin_unlock_irqrestore(&smfc->lock, flags);

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_smfc_disable);

struct ipu_smfc *ipu_smfc_get(struct ipu_soc *ipu)
{
	return ipu->smfc_priv;
}
EXPORT_SYMBOL_GPL(ipu_smfc_get);

void ipu_smfc_put(struct ipu_smfc *smfc)
{
}
EXPORT_SYMBOL_GPL(ipu_smfc_put);

int ipu_smfc_init(struct ipu_soc *ipu, struct device *dev,
		  unsigned long base, u32 module)
{
	struct ipu_smfc *smfc;

	smfc = devm_kzalloc(dev, sizeof(*smfc), GFP_KERNEL);
	if (!smfc)
		return -ENOMEM;

	ipu->smfc_priv = smfc;

	spin_lock_init(&smfc->lock);
	smfc->module = module;
	smfc->base = devm_ioremap(dev, base, PAGE_SIZE);
	if (!smfc->base)
		return -ENOMEM;

	dev_dbg(dev, "SMFC base: 0x%08lx remapped to %p\n", base, smfc->base);
	smfc->ipu = ipu;

	return 0;
}

void ipu_smfc_exit(struct ipu_soc *ipu)
{
}
