/*
 * Copyright (c) 2010 Sascha Hauer <s.hauer@pengutronix.de>
 * Copyright (C) 2005-2009 Freescale Semiconductor, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */
#ifndef __IPU_PRV_H__
#define __IPU_PRV_H__

struct ipu_soc;

#include <linux/types.h>
#include <linux/device.h>
#include <linux/clk.h>
#include <linux/platform_device.h>
#include <linux/platform_data/imx-ipu-v3.h>
#include <linux/regmap.h>
#include <linux/mfd/syscon.h>
#include <linux/mfd/syscon/imx6q-iomuxc-gpr.h>

#define IPU_NUM 2

#define IPU_MCU_T_DEFAULT	8
#define IPU_CM_IDMAC_REG_OFS	0x00008000
#define IPU_CM_ISP_REG_OFS	0x00010000
#define IPU_CM_DP_REG_OFS	0x00018000
#define IPU_CM_IC_REG_OFS	0x00020000
#define IPU_CM_IRT_REG_OFS	0x00028000
#define IPU_CM_CSI0_REG_OFS	0x00030000
#define IPU_CM_CSI1_REG_OFS	0x00038000
#define IPU_CM_SMFC_REG_OFS	0x00050000
#define IPU_CM_DC_REG_OFS	0x00058000
#define IPU_CM_DMFC_REG_OFS	0x00060000
#define IPU_CM_LUT_REG_OFS	0x01020000
#define IPU_CM_ISP_TBPR_REG_OFS	0x010c0000
#define IPU_CM_VDI_REG_OFS	0x00068000

/* Register addresses */
/* IPU Common registers */
#define IPU_CM_REG(offset)	(offset)

#define IPU_CONF			IPU_CM_REG(0)

#define IPU_SRM_PRI1			IPU_CM_REG(0x00a0)
#define IPU_SRM_PRI2			IPU_CM_REG(0x00a4)
#define IPU_FS_PROC_FLOW1		IPU_CM_REG(0x00a8)
#define IPU_FS_PROC_FLOW2		IPU_CM_REG(0x00ac)
#define IPU_FS_PROC_FLOW3		IPU_CM_REG(0x00b0)
#define IPU_FS_DISP_FLOW1		IPU_CM_REG(0x00b4)
#define IPU_FS_DISP_FLOW2		IPU_CM_REG(0x00b8)
#define IPU_SKIP			IPU_CM_REG(0x00bc)
#define IPU_DISP_ALT_CONF		IPU_CM_REG(0x00c0)
#define IPU_DISP_GEN			IPU_CM_REG(0x00c4)
#define IPU_DISP_ALT1			IPU_CM_REG(0x00c8)
#define IPU_DISP_ALT2			IPU_CM_REG(0x00cc)
#define IPU_DISP_ALT3			IPU_CM_REG(0x00d0)
#define IPU_DISP_ALT4			IPU_CM_REG(0x00d4)
#define IPU_SNOOP			IPU_CM_REG(0x00d8)
#define IPU_MEM_RST			IPU_CM_REG(0x00dc)
#define IPU_PM				IPU_CM_REG(0x00e0)
#define IPU_GPR				IPU_CM_REG(0x00e4)
#define IPU_CHA_DB_MODE_SEL(ch)		IPU_CM_REG(0x0150 + 4 * ((ch) / 32))
#define IPU_ALT_CHA_DB_MODE_SEL(ch)	IPU_CM_REG(0x0168 + 4 * ((ch) / 32))
#define IPU_CHA_CUR_BUF(ch)		IPU_CM_REG(0x023C + 4 * ((ch) / 32))
#define IPU_ALT_CUR_BUF0		IPU_CM_REG(0x0244)
#define IPU_ALT_CUR_BUF1		IPU_CM_REG(0x0248)
#define IPU_SRM_STAT			IPU_CM_REG(0x024C)
#define IPU_PROC_TASK_STAT		IPU_CM_REG(0x0250)
#define IPU_DISP_TASK_STAT		IPU_CM_REG(0x0254)
#define IPU_CHA_BUF0_RDY(ch)		IPU_CM_REG(0x0268 + 4 * ((ch) / 32))
#define IPU_CHA_BUF1_RDY(ch)		IPU_CM_REG(0x0270 + 4 * ((ch) / 32))
#define IPU_CHA_BUF2_RDY(ch)		IPU_CM_REG(0x0288 + 4 * ((ch) / 32))
#define IPU_ALT_CHA_BUF0_RDY(ch)	IPU_CM_REG(0x0278 + 4 * ((ch) / 32))
#define IPU_ALT_CHA_BUF1_RDY(ch)	IPU_CM_REG(0x0280 + 4 * ((ch) / 32))
#define IPU_CHA_TRB_MODE_SEL(ch)        IPU_CM_REG(0x0178 + 4 * ((ch) / 32))
#define IPU_CHA_TRIPLE_CUR_BUF(ch)      IPU_CM_REG(0x0258 + 4 * ((ch) / 32))

#define IPU_INT_CTRL(n)		IPU_CM_REG(0x003C + 4 * (n))
#define IPU_INT_STAT(n)		IPU_CM_REG(0x0200 + 4 * (n))

#define IPUIRQ_2_STATREG(irq)   (IPU_INT_STAT(0) + ((irq) / 32))
#define IPUIRQ_2_CTRLREG(irq)   (IPU_INT_CTRL(0) + ((irq) / 32))
#define IPUIRQ_2_MASK(irq)      (1UL << ((irq) & 0x1F))

#define IPU_DI0_COUNTER_RELEASE			(1 << 24)
#define IPU_DI1_COUNTER_RELEASE			(1 << 25)

#define FS_PRPENC_ROT_SRC_SEL_MASK     (0xf << 0)
#define FS_PRPENC_ROT_SRC_SEL_OFFSET   0
#define FS_PRPVF_ROT_SRC_SEL_MASK      (0xf << 8)
#define FS_PRPVF_ROT_SRC_SEL_OFFSET    8
#define FS_PP_ROT_SRC_SEL_MASK         (0xf << 16)
#define FS_PP_ROT_SRC_SEL_OFFSET       16
#define FS_PP_SRC_SEL_MASK             (0xf << 12)
#define FS_PP_SRC_SEL_OFFSET           12
#define FS_PP_SRC_SEL_VDOA             (1 << 15)
#define FS_PRP_SRC_SEL_MASK            (0xf << 24)
#define FS_PRP_SRC_SEL_OFFSET          24
#define FS_VF_IN_VALID                 (1 << 31)
#define FS_ENC_IN_VALID                (1 << 30)
#define FS_VDI_SRC_SEL_MASK            (0x3 << 28)
#define FS_VDI_SRC_SEL_VDOA            (0x2 << 28)
#define FS_VDOA_DEST_SEL_MASK          (0x3 << 16)
#define FS_VDOA_DEST_SEL_VDI           (0x2 << 16)
#define FS_VDOA_DEST_SEL_IC            (0x1 << 16)
#define FS_VDI_SRC_SEL_OFFSET          28

#define FS_PRPENC_DEST_SEL_MASK        (0xf << 0)
#define FS_PRPENC_DEST_SEL_OFFSET      0
#define FS_PRPVF_DEST_SEL_MASK         (0xf << 4)
#define FS_PRPVF_DEST_SEL_OFFSET       4
#define FS_PRPVF_ROT_DEST_SEL_MASK     (0xf << 8)
#define FS_PRPVF_ROT_DEST_SEL_OFFSET   8
#define FS_PP_DEST_SEL_MASK            (0xf << 12)
#define FS_PP_DEST_SEL_OFFSET          12
#define FS_PP_ROT_DEST_SEL_MASK        (0xf << 16)
#define FS_PP_ROT_DEST_SEL_OFFSET      16
#define FS_PRPENC_ROT_DEST_SEL_MASK    (0xf << 20)
#define FS_PRPENC_ROT_DEST_SEL_OFFSET  20

#define FS_SMFC0_DEST_SEL_MASK         (0xf << 0)
#define FS_SMFC0_DEST_SEL_OFFSET       0
#define FS_SMFC1_DEST_SEL_MASK         (0x7 << 4)
#define FS_SMFC1_DEST_SEL_OFFSET       4
#define FS_SMFC2_DEST_SEL_MASK         (0xf << 7)
#define FS_SMFC2_DEST_SEL_OFFSET       7
#define FS_SMFC3_DEST_SEL_MASK         (0x7 << 11)
#define FS_SMFC3_DEST_SEL_OFFSET       11

#define FS_DC1_SRC_SEL_MASK            (0xf << 20)
#define FS_DC1_SRC_SEL_OFFSET          20
#define FS_DC2_SRC_SEL_MASK            (0xf << 16)
#define FS_DC2_SRC_SEL_OFFSET          16
#define FS_DP_SYNC0_SRC_SEL_MASK       (0xf << 0)
#define FS_DP_SYNC0_SRC_SEL_OFFSET     0
#define FS_DP_SYNC1_SRC_SEL_MASK       (0xf << 4)
#define FS_DP_SYNC1_SRC_SEL_OFFSET     4
#define FS_DP_ASYNC0_SRC_SEL_MASK      (0xf << 8)
#define FS_DP_ASYNC0_SRC_SEL_OFFSET    8
#define FS_DP_ASYNC1_SRC_SEL_MASK      (0xf << 12)
#define FS_DP_ASYNC1_SRC_SEL_OFFSET    12

#define FS_AUTO_REF_PER_MASK           0
#define FS_AUTO_REF_PER_OFFSET         16

#define IPU_IDMAC_REG(offset)	(offset)

#define IDMAC_CONF			IPU_IDMAC_REG(0x0000)
#define IDMAC_CHA_EN(ch)		IPU_IDMAC_REG(0x0004 + 4 * ((ch) / 32))
#define IDMAC_SEP_ALPHA			IPU_IDMAC_REG(0x000c)
#define IDMAC_ALT_SEP_ALPHA		IPU_IDMAC_REG(0x0010)
#define IDMAC_CHA_PRI(ch)		IPU_IDMAC_REG(0x0014 + 4 * ((ch) / 32))
#define IDMAC_WM_EN(ch)			IPU_IDMAC_REG(0x001c + 4 * ((ch) / 32))
#define IDMAC_CH_LOCK_EN_1		IPU_IDMAC_REG(0x0024)
#define IDMAC_CH_LOCK_EN_2		IPU_IDMAC_REG(0x0028)
#define IDMAC_SUB_ADDR_0		IPU_IDMAC_REG(0x002c)
#define IDMAC_SUB_ADDR_1		IPU_IDMAC_REG(0x0030)
#define IDMAC_SUB_ADDR_2		IPU_IDMAC_REG(0x0034)
#define IDMAC_SUB_ADDR_3		IPU_IDMAC_REG(0x0038)
#define IDMAC_SUB_ADDR_4		IPU_IDMAC_REG(0x003c)
#define IDMAC_BAND_EN(ch)		IPU_IDMAC_REG(0x0040 + 4 * ((ch) / 32))
#define IDMAC_CHA_BUSY(ch)		IPU_IDMAC_REG(0x0100 + 4 * ((ch) / 32))

#define NONRT_REQ_W_LIM_MASK            (0x7 << 17)
#define NONRT_REQ_W_LIM_OFFSET          17
#define NONRT_REQ_W_LIM_EN              (1 << 20)
#define NONRT_REQ_R_LIM_MASK            (0xf << 21)
#define NONRT_REQ_R_LIM_OFFSET          21
#define NONRT_REQ_R_LIM_EN              (1 << 25)

#define IPU_GPR_QOS_W_ID_MASK(id)       (0xf << (4 * (id)))
#define IPU_GPR_QOS_W_ID(id, val)       ((val) << (4 * (id)))
#define IPU_GPR_QOS_R_ID_MASK(id)       (0xf << (16 + 4 * (id)))
#define IPU_GPR_QOS_R_ID(id, val)       ((val) << (16 + 4 * (id)))

#define IPU_NUM_IRQS	(15 * 8 * sizeof(u32))

#define NO_DMA -1
#define idma_is_valid(ch)	(ch && ch->num != NO_DMA)
#define idma_mask(ch)		(1 << (ch & 0x1f))
#define idma_is_set(ipu, reg, ch) \
	(ch && (ipu_idmac_read(ipu, reg(ch->num)) & idma_mask(ch->num)))
#define tri_cur_buf_mask(ch)    (idma_mask((ch) * 2) * 3)
#define tri_cur_buf_shift(ch)   (ffs(idma_mask((ch) * 2)) - 1)

enum ipu_modules {
	IPU_CONF_CSI0_EN		= (1 << 0),
	IPU_CONF_CSI1_EN		= (1 << 1),
	IPU_CONF_IC_EN			= (1 << 2),
	IPU_CONF_ROT_EN			= (1 << 3),
	IPU_CONF_ISP_EN			= (1 << 4),
	IPU_CONF_DP_EN			= (1 << 5),
	IPU_CONF_DI0_EN			= (1 << 6),
	IPU_CONF_DI1_EN			= (1 << 7),
	IPU_CONF_SMFC_EN		= (1 << 8),
	IPU_CONF_DC_EN			= (1 << 9),
	IPU_CONF_DMFC_EN		= (1 << 10),

	IPU_CONF_VDI_EN			= (1 << 12),

	IPU_CONF_IDMAC_DIS		= (1 << 22),

	IPU_CONF_IC_DMFC_SEL		= (1 << 25),
	IPU_CONF_IC_DMFC_SYNC		= (1 << 26),
	IPU_CONF_VDI_DMFC_SYNC		= (1 << 27),

	IPU_CONF_CSI0_DATA_SOURCE	= (1 << 28),
	IPU_CONF_CSI1_DATA_SOURCE	= (1 << 29),
	IPU_CONF_IC_INPUT		= (1 << 30),
	IPU_CONF_CSI_SEL		= (1 << 31),
};

struct ipu_cpmem;

/* IPU Driver IDMA channel definition.	*/
struct ipuv3_channel {
	unsigned int num;
	int alt_num; /* alternative cpmem index for this channel */

	atomic_t inuse;
	bool enabled;
	bool interlaced;

	wait_queue_head_t inuse_wait;

	struct ipu_soc *ipu;
	struct ipu_cpmem *cpmem;
};

struct ipu_dc_priv;
struct ipu_dmfc_priv;
struct ipu_csi;
struct ipu_smfc;
struct ipu_ic_priv;
struct ipu_irt;
struct ipu_vdi;
struct ipu_di;
struct ipu_devtype;

struct ipu_soc {
	struct device		*dev;
	const struct ipu_devtype	*devtype;
	enum ipuv3_type		ipu_type;
	spinlock_t		lock;

	struct regmap           *gp_reg; /* regmap to IOMUXC GPR registers */
	void __iomem		*cm_reg;
	void __iomem		*idmac_reg;

	int                     id;
	int		        usecount;

	struct clk		*clk;

	struct ipuv3_channel	channel[64];

	int			irq_start;
	int			irq_sync;
	int			irq_err;

	struct ipu_cpmem        *cpmem_priv;
	struct ipu_dc_priv	*dc_priv;
	struct ipu_dp_priv	*dp_priv;
	struct ipu_dmfc_priv	*dmfc_priv;
	struct ipu_di		*di_priv[2];
	struct ipu_csi		*csi_priv[2];
	struct ipu_smfc         *smfc_priv;
	struct ipu_ic_priv      *ic_priv;
	struct ipu_irt          *irt_priv;
	struct ipu_vdi          *vdi_priv;
};

static inline u32 ipu_cm_read(struct ipu_soc *ipu, unsigned offset)
{
	return readl(ipu->cm_reg + offset);
}

static inline void ipu_cm_write(struct ipu_soc *ipu, u32 value, unsigned offset)
{
	writel(value, ipu->cm_reg + offset);
}

static inline u32 ipu_idmac_read(struct ipu_soc *ipu, unsigned offset)
{
	return readl(ipu->idmac_reg + offset);
}

static inline void ipu_idmac_write(struct ipu_soc *ipu, u32 value,
		unsigned offset)
{
	writel(value, ipu->idmac_reg + offset);
}

void ipu_srm_dp_sync_update(struct ipu_soc *ipu);
void __ipu_idmac_clear_buffer_ready(struct ipuv3_channel *channel,
				    u32 buf_num);
void __ipu_idmac_reset_current_buffer(struct ipuv3_channel *channel);
int __ipu_idmac_enable_channel(struct ipuv3_channel *channel);
int __ipu_idmac_disable_channel(struct ipuv3_channel *channel);

void ipu_set_csi_src_mux(struct ipu_soc *ipu, int csi_id, bool mipi_csi2);
void ipu_set_ic_src_mux(struct ipu_soc *ipu, int csi_id, bool vdi);
void ipu_set_vdi_src_mux(struct ipu_soc *ipu, bool csi);

int ipu_module_enable(struct ipu_soc *ipu, u32 mask);
int ipu_module_disable(struct ipu_soc *ipu, u32 mask);

int ipu_csi_init(struct ipu_soc *ipu, struct device *dev, int id,
		 unsigned long base, u32 module, struct clk *clk_ipu);
void ipu_csi_exit(struct ipu_soc *ipu, int id);

int ipu_smfc_init(struct ipu_soc *ipu, struct device *dev,
		  unsigned long base, u32 module);
void ipu_smfc_exit(struct ipu_soc *ipu);

int ipu_ic_init(struct ipu_soc *ipu, struct device *dev,
		unsigned long base, unsigned long tpmem_base,
		u32 module);
void ipu_ic_exit(struct ipu_soc *ipu);

int ipu_irt_init(struct ipu_soc *ipu, struct device *dev, u32 module);
void ipu_irt_exit(struct ipu_soc *ipu);

int ipu_vdi_init(struct ipu_soc *ipu, struct device *dev,
		 unsigned long base, u32 module);
void ipu_vdi_exit(struct ipu_soc *ipu);

int ipu_di_init(struct ipu_soc *ipu, struct device *dev, int id,
		unsigned long base, u32 module, struct clk *ipu_clk);
void ipu_di_exit(struct ipu_soc *ipu, int id);

int ipu_dmfc_init(struct ipu_soc *ipu, struct device *dev, unsigned long base,
		struct clk *ipu_clk);
void ipu_dmfc_exit(struct ipu_soc *ipu);

int ipu_dp_init(struct ipu_soc *ipu, struct device *dev, unsigned long base);
void ipu_dp_exit(struct ipu_soc *ipu);

int ipu_dc_init(struct ipu_soc *ipu, struct device *dev, unsigned long base,
		unsigned long template_base);
void ipu_dc_exit(struct ipu_soc *ipu);

int ipu_cpmem_init(struct ipu_soc *ipu, struct device *dev, unsigned long base);
void ipu_cpmem_exit(struct ipu_soc *ipu);
int ipu_cpmem_get_third_buf_cpmem_num(struct ipuv3_channel *channel);

#endif				/* __IPU_PRV_H__ */
