/*
 * Copyright (C) 2012 Mentor Graphics Inc.
 * Copyright 2005-2012 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */
#include <linux/types.h>
#include <linux/bitrev.h>
#include <linux/io.h>
#include "ipu-prv.h"

struct ipu_cpmem_word {
	u32 data[5];
	u32 res[3];
};

struct ipu_ch_param {
	struct ipu_cpmem_word word[2];
};

struct ipu_cpmem {
	struct ipu_ch_param __iomem *base;
	u32 module;
	spinlock_t lock;
	int use_count;
	struct ipu_soc *ipu;
};

#define IPU_CPMEM_WORD(word, ofs, size)			\
	((((word) * 160 + (ofs)) << 8) | (size))

#define IPU_FIELD_UBO		IPU_CPMEM_WORD(0, 46, 22)
#define IPU_FIELD_VBO		IPU_CPMEM_WORD(0, 68, 22)
#define IPU_FIELD_IOX		IPU_CPMEM_WORD(0, 90, 4)
#define IPU_FIELD_RDRW		IPU_CPMEM_WORD(0, 94, 1)
#define IPU_FIELD_SO		IPU_CPMEM_WORD(0, 113, 1)
#define IPU_FIELD_SLY		IPU_CPMEM_WORD(1, 102, 14)
#define IPU_FIELD_SLUV		IPU_CPMEM_WORD(1, 128, 14)

#define IPU_FIELD_XV		IPU_CPMEM_WORD(0, 0, 10)
#define IPU_FIELD_YV		IPU_CPMEM_WORD(0, 10, 9)
#define IPU_FIELD_XB		IPU_CPMEM_WORD(0, 19, 13)
#define IPU_FIELD_YB		IPU_CPMEM_WORD(0, 32, 12)
#define IPU_FIELD_NSB_B		IPU_CPMEM_WORD(0, 44, 1)
#define IPU_FIELD_CF		IPU_CPMEM_WORD(0, 45, 1)
#define IPU_FIELD_SX		IPU_CPMEM_WORD(0, 46, 12)
#define IPU_FIELD_SY		IPU_CPMEM_WORD(0, 58, 11)
#define IPU_FIELD_NS		IPU_CPMEM_WORD(0, 69, 10)
#define IPU_FIELD_SDX		IPU_CPMEM_WORD(0, 79, 7)
#define IPU_FIELD_SM		IPU_CPMEM_WORD(0, 86, 10)
#define IPU_FIELD_SCC		IPU_CPMEM_WORD(0, 96, 1)
#define IPU_FIELD_SCE		IPU_CPMEM_WORD(0, 97, 1)
#define IPU_FIELD_SDY		IPU_CPMEM_WORD(0, 98, 7)
#define IPU_FIELD_SDRX		IPU_CPMEM_WORD(0, 105, 1)
#define IPU_FIELD_SDRY		IPU_CPMEM_WORD(0, 106, 1)
#define IPU_FIELD_BPP		IPU_CPMEM_WORD(0, 107, 3)
#define IPU_FIELD_DEC_SEL	IPU_CPMEM_WORD(0, 110, 2)
#define IPU_FIELD_DIM		IPU_CPMEM_WORD(0, 112, 1)
#define IPU_FIELD_BNDM		IPU_CPMEM_WORD(0, 114, 3)
#define IPU_FIELD_BM		IPU_CPMEM_WORD(0, 117, 2)
#define IPU_FIELD_ROT		IPU_CPMEM_WORD(0, 119, 1)
#define IPU_FIELD_HF		IPU_CPMEM_WORD(0, 120, 1)
#define IPU_FIELD_VF		IPU_CPMEM_WORD(0, 121, 1)
#define IPU_FIELD_ROT_HF_VF	IPU_CPMEM_WORD(0, 119, 3)
#define IPU_FIELD_THE		IPU_CPMEM_WORD(0, 122, 1)
#define IPU_FIELD_CAP		IPU_CPMEM_WORD(0, 123, 1)
#define IPU_FIELD_CAE		IPU_CPMEM_WORD(0, 124, 1)
#define IPU_FIELD_FW		IPU_CPMEM_WORD(0, 125, 13)
#define IPU_FIELD_FH		IPU_CPMEM_WORD(0, 138, 12)
#define IPU_FIELD_EBA0		IPU_CPMEM_WORD(1, 0, 29)
#define IPU_FIELD_EBA1		IPU_CPMEM_WORD(1, 29, 29)
#define IPU_FIELD_ILO		IPU_CPMEM_WORD(1, 58, 20)
#define IPU_FIELD_NPB		IPU_CPMEM_WORD(1, 78, 7)
#define IPU_FIELD_PFS		IPU_CPMEM_WORD(1, 85, 4)
#define IPU_FIELD_ALU		IPU_CPMEM_WORD(1, 89, 1)
#define IPU_FIELD_ALBM		IPU_CPMEM_WORD(1, 90, 3)
#define IPU_FIELD_ID		IPU_CPMEM_WORD(1, 93, 2)
#define IPU_FIELD_TH		IPU_CPMEM_WORD(1, 95, 7)
#define IPU_FIELD_SL		IPU_CPMEM_WORD(1, 102, 14)
#define IPU_FIELD_WID0		IPU_CPMEM_WORD(1, 116, 3)
#define IPU_FIELD_WID1		IPU_CPMEM_WORD(1, 119, 3)
#define IPU_FIELD_WID2		IPU_CPMEM_WORD(1, 122, 3)
#define IPU_FIELD_WID3		IPU_CPMEM_WORD(1, 125, 3)
#define IPU_FIELD_OFS0		IPU_CPMEM_WORD(1, 128, 5)
#define IPU_FIELD_OFS1		IPU_CPMEM_WORD(1, 133, 5)
#define IPU_FIELD_OFS2		IPU_CPMEM_WORD(1, 138, 5)
#define IPU_FIELD_OFS3		IPU_CPMEM_WORD(1, 143, 5)
#define IPU_FIELD_SXYS		IPU_CPMEM_WORD(1, 148, 1)
#define IPU_FIELD_CRE		IPU_CPMEM_WORD(1, 149, 1)
#define IPU_FIELD_DEC_SEL2	IPU_CPMEM_WORD(1, 150, 1)


static inline struct ipu_ch_param __iomem *
ipu_get_cpmem(struct ipuv3_channel *channel)
{
	struct ipu_cpmem *cpmem = channel->cpmem;
	return cpmem->base + channel->num;
}

static inline struct ipu_ch_param __iomem *
ipu_get_alt_cpmem(struct ipuv3_channel *channel)
{
	if (channel->alt_num < 0)
		return NULL;

	return channel->cpmem->base + channel->alt_num;
}

static void ipu_ch_param_write_field(struct ipu_ch_param __iomem *base,
				     u32 wbs, u32 v)
{
	u32 bit = (wbs >> 8) % 160;
	u32 size = wbs & 0xff;
	u32 word = (wbs >> 8) / 160;
	u32 i = bit / 32;
	u32 ofs = bit % 32;
	u32 mask = (1 << size) - 1;
	u32 val;

	pr_debug("%s %d %d %d\n", __func__, word, bit , size);

	val = readl(&base->word[word].data[i]);
	val &= ~(mask << ofs);
	val |= v << ofs;
	writel(val, &base->word[word].data[i]);

	if ((bit + size - 1) / 32 > i) {
		val = readl(&base->word[word].data[i + 1]);
		val &= ~(mask >> (ofs ? (32 - ofs) : 0));
		val |= v >> (ofs ? (32 - ofs) : 0);
		writel(val, &base->word[word].data[i + 1]);
	}
}

static u32 ipu_ch_param_read_field(struct ipu_ch_param __iomem *base, u32 wbs)
{
	u32 bit = (wbs >> 8) % 160;
	u32 size = wbs & 0xff;
	u32 word = (wbs >> 8) / 160;
	u32 i = bit / 32;
	u32 ofs = bit % 32;
	u32 mask = (1 << size) - 1;
	u32 val = 0;

	pr_debug("%s %d %d %d\n", __func__, word, bit , size);

	val = (readl(&base->word[word].data[i]) >> ofs) & mask;

	if ((bit + size - 1) / 32 > i) {
		u32 tmp;
		tmp = readl(&base->word[word].data[i + 1]);
		tmp &= mask >> (ofs ? (32 - ofs) : 0);
		val |= tmp << (ofs ? (32 - ofs) : 0);
	}

	return val;
}

static void ipu_ch_param_set_field_io(struct ipu_ch_param __iomem *base,
				      u32 wbs, u32 v)
{
	u32 bit = (wbs >> 8) % 160;
	u32 size = wbs & 0xff;
	u32 word = (wbs >> 8) / 160;
	u32 i = bit / 32;
	u32 ofs = bit % 32;
	u32 tmp;

	tmp = readl(&base->word[word].data[i]);
	tmp |= (v << ofs);
	writel(tmp, &base->word[word].data[i]);

	if ((bit + size - 1) / 32 > i) {
		tmp = readl(&base->word[word].data[i + 1]);
		tmp |= (v >> (ofs ? (32 - ofs) : 0));
		writel(tmp, &base->word[word].data[i + 1]);
	}
}

static void ipu_ch_param_set_field(struct ipu_ch_param *base,
				   u32 wbs, u32 v)
{
	u32 bit = (wbs >> 8) % 160;
	u32 size = wbs & 0xff;
	u32 word = (wbs >> 8) / 160;
	u32 i = bit / 32;
	u32 ofs = bit % 32;

	base->word[word].data[i] |= (v << ofs);

	if ((bit + size - 1) / 32 > i)
		base->word[word].data[i + 1] |= (v >> (ofs ? (32 - ofs) : 0));
}

static void ipu_ch_param_mod_field_io(struct ipu_ch_param __iomem *base,
				      u32 wbs, u32 v)
{
	u32 bit = (wbs >> 8) % 160;
	u32 size = wbs & 0xff;
	u32 word = (wbs >> 8) / 160;
	u32 i = bit / 32;
	u32 ofs = bit % 32;
	u32 mask = (1 << size) - 1;
	u32 temp;

	temp = readl(&base->word[word].data[i]);
	temp &= ~(mask << ofs);
	temp |= (v << ofs);
	writel(temp, &base->word[word].data[i]);

	if ((bit + size - 1) / 32 > i) {
		temp = readl(&base->word[word].data[i + 1]);
		temp &= ~(mask >> (32 - ofs));
		temp |= (v >> (ofs ? (32 - ofs) : 0));
		writel(temp, &base->word[word].data[i + 1]);
	}
}

void ipu_cpmem_zero(struct ipuv3_channel *channel)
{
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);
	void __iomem *base = p;
	int i;

	for (i = 0; i < sizeof(*p) / sizeof(u32); i++)
		writel(0, base + i * sizeof(u32));
}
EXPORT_SYMBOL_GPL(ipu_cpmem_zero);

void ipu_cpmem_set_resolution(struct ipuv3_channel *channel,
			      int xres, int yres)
{
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);
	ipu_ch_param_write_field(p, IPU_FIELD_FW, xres - 1);
	ipu_ch_param_write_field(p, IPU_FIELD_FH, yres - 1);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_resolution);

void ipu_cpmem_set_stride(struct ipuv3_channel *channel, int stride)
{
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);
	ipu_ch_param_write_field(p, IPU_FIELD_SLY, stride - 1);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_stride);

void ipu_cpmem_set_high_priority(struct ipuv3_channel *channel)
{
	struct ipu_soc *ipu = channel->ipu;
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);
	u32 val;

	if (ipu->ipu_type == IPUV3EX)
		ipu_ch_param_write_field(p, IPU_FIELD_ID, 1);

	val = ipu_idmac_read(ipu, IDMAC_CHA_PRI(channel->num));
	val |= 1 << (channel->num % 32);
	ipu_idmac_write(ipu, val, IDMAC_CHA_PRI(channel->num));
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_high_priority);

int ipu_cpmem_set_format_rgb(struct ipuv3_channel *channel,
			     struct ipu_rgb *rgb)
{
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);
	int bpp = 0, npb = 0, ro, go, bo, to;

	ro = rgb->bits_per_pixel - rgb->red.length - rgb->red.offset;
	go = rgb->bits_per_pixel - rgb->green.length - rgb->green.offset;
	bo = rgb->bits_per_pixel - rgb->blue.length - rgb->blue.offset;
	to = rgb->bits_per_pixel - rgb->transp.length - rgb->transp.offset;

	ipu_ch_param_write_field(p, IPU_FIELD_WID0, rgb->red.length - 1);
	ipu_ch_param_write_field(p, IPU_FIELD_OFS0, ro);
	ipu_ch_param_write_field(p, IPU_FIELD_WID1, rgb->green.length - 1);
	ipu_ch_param_write_field(p, IPU_FIELD_OFS1, go);
	ipu_ch_param_write_field(p, IPU_FIELD_WID2, rgb->blue.length - 1);
	ipu_ch_param_write_field(p, IPU_FIELD_OFS2, bo);

	if (rgb->transp.length) {
		ipu_ch_param_write_field(p, IPU_FIELD_WID3,
				rgb->transp.length - 1);
		ipu_ch_param_write_field(p, IPU_FIELD_OFS3, to);
	} else {
		ipu_ch_param_write_field(p, IPU_FIELD_WID3, 7);
		ipu_ch_param_write_field(p, IPU_FIELD_OFS3,
				rgb->bits_per_pixel);
	}

	switch (rgb->bits_per_pixel) {
	case 32:
		bpp = 0;
		npb = 15;
		break;
	case 24:
		bpp = 1;
		npb = 19;
		break;
	case 16:
		bpp = 3;
		npb = 31;
		break;
	case 8:
		bpp = 5;
		npb = 63;
		break;
	default:
		return -EINVAL;
	}
	ipu_ch_param_write_field(p, IPU_FIELD_BPP, bpp);
	ipu_ch_param_write_field(p, IPU_FIELD_NPB, npb);
	ipu_ch_param_write_field(p, IPU_FIELD_PFS, 7); /* rgb mode */

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_format_rgb);

int ipu_cpmem_set_format_passthrough(struct ipuv3_channel *channel,
				     int width)
{
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);
	int bpp = 0, npb = 0;

	switch (width) {
	case 32:
		bpp = 0;
		npb = 15;
		break;
	case 24:
		bpp = 1;
		npb = 19;
		break;
	case 16:
		bpp = 3;
		npb = 31;
		break;
	case 8:
		bpp = 5;
		npb = 63;
		break;
	default:
		return -EINVAL;
	}

	ipu_ch_param_write_field(p, IPU_FIELD_BPP, bpp);
	ipu_ch_param_write_field(p, IPU_FIELD_NPB, npb);
	ipu_ch_param_write_field(p, IPU_FIELD_PFS, 6); /* raw mode */

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_format_passthrough);

void ipu_cpmem_set_yuv_planar_full(struct ipuv3_channel *channel,
				   u32 pixel_format, int stride,
				   int u_offset, int v_offset)
{
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);

	switch (pixel_format) {
	case V4L2_PIX_FMT_YUV420:
	case V4L2_PIX_FMT_YUV422P:
		ipu_ch_param_write_field(p, IPU_FIELD_SLUV, (stride / 2) - 1);
		ipu_ch_param_write_field(p, IPU_FIELD_UBO, u_offset / 8);
		ipu_ch_param_write_field(p, IPU_FIELD_VBO, v_offset / 8);
		break;
	case V4L2_PIX_FMT_YVU420:
	case V4L2_PIX_FMT_YVU422P:
		ipu_ch_param_write_field(p, IPU_FIELD_SLUV, (stride / 2) - 1);
		ipu_ch_param_write_field(p, IPU_FIELD_UBO, v_offset / 8);
		ipu_ch_param_write_field(p, IPU_FIELD_VBO, u_offset / 8);
		break;
	case V4L2_PIX_FMT_NV12:
	case V4L2_PIX_FMT_NV16:
		ipu_ch_param_write_field(p, IPU_FIELD_SLUV, stride - 1);
		ipu_ch_param_write_field(p, IPU_FIELD_UBO, u_offset / 8);
		ipu_ch_param_write_field(p, IPU_FIELD_VBO, u_offset / 8);
		break;
	case V4L2_PIX_FMT_NV21:
	case V4L2_PIX_FMT_NV61:
		ipu_ch_param_write_field(p, IPU_FIELD_SLUV, stride - 1);
		ipu_ch_param_write_field(p, IPU_FIELD_UBO, v_offset / 8);
		ipu_ch_param_write_field(p, IPU_FIELD_VBO, v_offset / 8);
		break;
	}
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_yuv_planar_full);

void ipu_cpmem_set_yuv_planar(struct ipuv3_channel *channel, u32 pixel_format,
			      int stride, int height)
{
	int u_offset, v_offset;
	int uv_stride = 0;

	switch (pixel_format) {
	case V4L2_PIX_FMT_YUV420:
	case V4L2_PIX_FMT_YVU420:
		uv_stride = stride / 2;
		u_offset = stride * height;
		v_offset = u_offset + (uv_stride * height / 2);
		ipu_cpmem_set_yuv_planar_full(channel,
					      pixel_format, stride,
					      u_offset, v_offset);
		break;
	case V4L2_PIX_FMT_YUV422P:
	case V4L2_PIX_FMT_YVU422P:
		uv_stride = stride / 2;
		u_offset = stride * height;
		v_offset = u_offset + (uv_stride * height);
		ipu_cpmem_set_yuv_planar_full(channel,
					      pixel_format, stride,
					      u_offset, v_offset);
		break;
	case V4L2_PIX_FMT_NV12:
	case V4L2_PIX_FMT_NV16:
		u_offset = stride * height;
		ipu_cpmem_set_yuv_planar_full(channel,
					      pixel_format, stride,
					      u_offset, 0);
		break;
	case V4L2_PIX_FMT_NV21:
	case V4L2_PIX_FMT_NV61:
		v_offset = stride * height;
		ipu_cpmem_set_yuv_planar_full(channel,
					      pixel_format, stride,
					      0, v_offset);
		break;
	}
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_yuv_planar);

static struct ipu_rgb def_rgb_32 = {
	.red	= { .offset = 16, .length = 8, },
	.green	= { .offset =  8, .length = 8, },
	.blue	= { .offset =  0, .length = 8, },
	.transp = { .offset = 24, .length = 8, },
	.bits_per_pixel = 32,
};

static struct ipu_rgb def_bgr_32 = {
	.red	= { .offset =  0, .length = 8, },
	.green	= { .offset =  8, .length = 8, },
	.blue	= { .offset = 16, .length = 8, },
	.transp = { .offset = 24, .length = 8, },
	.bits_per_pixel = 32,
};

static struct ipu_rgb def_rgb_24 = {
	.red	= { .offset = 16, .length = 8, },
	.green	= { .offset =  8, .length = 8, },
	.blue	= { .offset =  0, .length = 8, },
	.transp = { .offset =  0, .length = 0, },
	.bits_per_pixel = 24,
};

static struct ipu_rgb def_bgr_24 = {
	.red	= { .offset =  0, .length = 8, },
	.green	= { .offset =  8, .length = 8, },
	.blue	= { .offset = 16, .length = 8, },
	.transp = { .offset =  0, .length = 0, },
	.bits_per_pixel = 24,
};

static struct ipu_rgb def_rgb_16 = {
	.red	= { .offset = 11, .length = 5, },
	.green	= { .offset =  5, .length = 6, },
	.blue	= { .offset =  0, .length = 5, },
	.transp = { .offset =  0, .length = 0, },
	.bits_per_pixel = 16,
};

#define Y_OFFSET(pix, x, y)	((x) + pix->width * (y))
#define U_OFFSET(pix, x, y)	((pix->width * pix->height) + \
					(pix->width * (y) / 4) + (x) / 2)
#define V_OFFSET(pix, x, y)	((pix->width * pix->height) + \
					(pix->width * pix->height / 4) + \
					(pix->width * (y) / 4) + (x) / 2)
#define U2_OFFSET(pix, x, y)	((pix->width * pix->height) + \
					(pix->width * (y) / 2) + (x) / 2)
#define V2_OFFSET(pix, x, y)	((pix->width * pix->height) + \
					(pix->width * pix->height / 2) + \
					(pix->width * (y) / 2) + (x) / 2)
#define UV_OFFSET(pix, x, y)	((pix->width * pix->height) + \
					(pix->width * (y) / 2) + (x))
#define UV2_OFFSET(pix, x, y)	((pix->width * pix->height) + \
					(pix->width * y) + (x))

int ipu_cpmem_set_fmt(struct ipuv3_channel *channel, u32 pixelformat)
{
	struct ipu_ch_param __iomem *cpmem = ipu_get_cpmem(channel);

	switch (pixelformat) {
	case V4L2_PIX_FMT_YUV420:
	case V4L2_PIX_FMT_YVU420:
		/* pix format */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_PFS, 2);
		/* burst size */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_NPB, 31);
		break;
	case V4L2_PIX_FMT_YUV422P:
	case V4L2_PIX_FMT_YVU422P:
		/* pix format */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_PFS, 1);
		/* burst size */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_NPB, 31);
		break;
	case V4L2_PIX_FMT_NV12:
	case V4L2_PIX_FMT_NV21:
		/* pix format */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_PFS, 4);
		/* burst size */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_NPB, 31);
		break;
	case V4L2_PIX_FMT_NV16:
	case V4L2_PIX_FMT_NV61:
		/* pix format */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_PFS, 3);
		/* burst size */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_NPB, 31);
		break;
	case V4L2_PIX_FMT_UYVY:
		/* bits/pixel */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_BPP, 3);
		/* pix format */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_PFS, 0xA);
		/* burst size */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_NPB, 31);
		break;
	case V4L2_PIX_FMT_YUYV:
		/* bits/pixel */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_BPP, 3);
		/* pix format */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_PFS, 0x8);
		/* burst size */
		ipu_ch_param_write_field(cpmem, IPU_FIELD_NPB, 31);
		break;
	case V4L2_PIX_FMT_RGB32:
		ipu_cpmem_set_format_rgb(channel, &def_rgb_32);
		break;
	case V4L2_PIX_FMT_RGB565:
		ipu_cpmem_set_format_rgb(channel, &def_rgb_16);
		break;
	case V4L2_PIX_FMT_BGR32:
		ipu_cpmem_set_format_rgb(channel, &def_bgr_32);
		break;
	case V4L2_PIX_FMT_RGB24:
		ipu_cpmem_set_format_rgb(channel, &def_rgb_24);
		break;
	case V4L2_PIX_FMT_BGR24:
		ipu_cpmem_set_format_rgb(channel, &def_bgr_24);
		break;
	case V4L2_PIX_FMT_GENERIC16:
		ipu_cpmem_set_format_passthrough(channel, 16);
		break;
	default:
		return -EINVAL;
	}

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_fmt);

int ipu_cpmem_set_image(struct ipuv3_channel *channel,
			struct ipu_image *image)
{
	struct v4l2_pix_format *pix = &image->pix;
	int offset, y_offset, u_offset, v_offset;

	pr_debug("%s: resolution: %dx%d stride: %d\n",
			__func__, pix->width, pix->height,
			pix->bytesperline);

	ipu_cpmem_set_resolution(channel, image->rect.width,
			image->rect.height);
	ipu_cpmem_set_stride(channel, pix->bytesperline);

	ipu_cpmem_set_fmt(channel, pix->pixelformat);

	switch (pix->pixelformat) {
	case V4L2_PIX_FMT_YUV420:
	case V4L2_PIX_FMT_YVU420:
		y_offset = Y_OFFSET(pix, image->rect.left, image->rect.top);
		u_offset = U_OFFSET(pix, image->rect.left,
				image->rect.top) - y_offset;
		v_offset = V_OFFSET(pix, image->rect.left,
				image->rect.top) - y_offset;

		ipu_cpmem_set_yuv_planar_full(channel, pix->pixelformat,
				pix->bytesperline, u_offset, v_offset);
		ipu_cpmem_set_buffer(channel, 0, image->phys0 + y_offset);
		ipu_cpmem_set_buffer(channel, 1, image->phys1 + y_offset);
		ipu_cpmem_set_buffer(channel, 2, image->phys2 + y_offset);
		break;
	case V4L2_PIX_FMT_YUV422P:
	case V4L2_PIX_FMT_YVU422P:
		y_offset = Y_OFFSET(pix, image->rect.left, image->rect.top);
		u_offset = U2_OFFSET(pix, image->rect.left,
				image->rect.top) - y_offset;
		v_offset = V2_OFFSET(pix, image->rect.left,
				image->rect.top) - y_offset;

		ipu_cpmem_set_yuv_planar_full(channel, pix->pixelformat,
				pix->bytesperline, u_offset, v_offset);
		ipu_cpmem_set_buffer(channel, 0, image->phys0 + y_offset);
		ipu_cpmem_set_buffer(channel, 1, image->phys1 + y_offset);
		ipu_cpmem_set_buffer(channel, 2, image->phys2 + y_offset);
		break;
	case V4L2_PIX_FMT_NV12:
		y_offset = Y_OFFSET(pix, image->rect.left, image->rect.top);
		u_offset = UV_OFFSET(pix, image->rect.left,
				image->rect.top) - y_offset;
		v_offset = 0;

		ipu_cpmem_set_yuv_planar_full(channel, pix->pixelformat,
				pix->bytesperline, u_offset, v_offset);
		ipu_cpmem_set_buffer(channel, 0, image->phys0 + y_offset);
		ipu_cpmem_set_buffer(channel, 1, image->phys1 + y_offset);
		ipu_cpmem_set_buffer(channel, 2, image->phys2 + y_offset);
		break;
	case V4L2_PIX_FMT_NV21:
		y_offset = Y_OFFSET(pix, image->rect.left, image->rect.top);
		u_offset = 0;
		v_offset = UV_OFFSET(pix, image->rect.left,
				image->rect.top) - y_offset;

		ipu_cpmem_set_yuv_planar_full(channel, pix->pixelformat,
				pix->bytesperline, u_offset, v_offset);
		ipu_cpmem_set_buffer(channel, 0, image->phys0 + y_offset);
		ipu_cpmem_set_buffer(channel, 1, image->phys1 + y_offset);
		ipu_cpmem_set_buffer(channel, 2, image->phys2 + y_offset);
		break;
	case V4L2_PIX_FMT_NV16:
		y_offset = Y_OFFSET(pix, image->rect.left, image->rect.top);
		u_offset = UV2_OFFSET(pix, image->rect.left,
				image->rect.top) - y_offset;
		v_offset = 0;

		ipu_cpmem_set_yuv_planar_full(channel, pix->pixelformat,
				pix->bytesperline, u_offset, v_offset);
		ipu_cpmem_set_buffer(channel, 0, image->phys0 + y_offset);
		ipu_cpmem_set_buffer(channel, 1, image->phys1 + y_offset);
		ipu_cpmem_set_buffer(channel, 2, image->phys2 + y_offset);
		break;
	case V4L2_PIX_FMT_NV61:
		y_offset = Y_OFFSET(pix, image->rect.left, image->rect.top);
		u_offset = 0;
		v_offset = UV2_OFFSET(pix, image->rect.left,
				image->rect.top) - y_offset;

		ipu_cpmem_set_yuv_planar_full(channel, pix->pixelformat,
				pix->bytesperline, u_offset, v_offset);
		ipu_cpmem_set_buffer(channel, 0, image->phys0 + y_offset);
		ipu_cpmem_set_buffer(channel, 1, image->phys1 + y_offset);
		ipu_cpmem_set_buffer(channel, 2, image->phys2 + y_offset);
		break;
	case V4L2_PIX_FMT_UYVY:
	case V4L2_PIX_FMT_YUYV:
	case V4L2_PIX_FMT_RGB565:
		offset = image->rect.left * 2 +
			image->rect.top * pix->bytesperline;
		ipu_cpmem_set_buffer(channel, 0, image->phys0 + offset);
		ipu_cpmem_set_buffer(channel, 1, image->phys1 + offset);
		ipu_cpmem_set_buffer(channel, 2, image->phys2 + offset);
		break;
	case V4L2_PIX_FMT_RGB32:
	case V4L2_PIX_FMT_BGR32:
		offset = image->rect.left * 4 +
			image->rect.top * pix->bytesperline;
		ipu_cpmem_set_buffer(channel, 0, image->phys0 + offset);
		ipu_cpmem_set_buffer(channel, 1, image->phys1 + offset);
		ipu_cpmem_set_buffer(channel, 2, image->phys2 + offset);
		break;
	case V4L2_PIX_FMT_RGB24:
	case V4L2_PIX_FMT_BGR24:
		offset = image->rect.left * 3 +
			image->rect.top * pix->bytesperline;
		ipu_cpmem_set_buffer(channel, 0, image->phys0 + offset);
		ipu_cpmem_set_buffer(channel, 1, image->phys1 + offset);
		ipu_cpmem_set_buffer(channel, 2, image->phys2 + offset);
		break;
	default:
		return -EINVAL;
	}

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_image);

int ipu_cpmem_get_third_buf_cpmem_num(struct ipuv3_channel *channel)
{
	switch (channel->num) {
	case 8:
		return 64;
	case 9:
		return 65;
	case 10:
		return 66;
	case 13:
		return 67;
	case 21:
		return 68;
	case 23:
		return 69;
	case 27:
		return 70;
	case 28:
		return 71;
	}

	return NO_DMA;
}

static void ipu_ch_param_set_packing(struct ipu_ch_param *p,
				     int red_width, int red_offset,
				     int green_width, int green_offset,
				     int blue_width, int blue_offset,
				     int alpha_width, int alpha_offset)
{
	/* Setup red width and offset */
	ipu_ch_param_set_field(p, IPU_FIELD_WID0, red_width - 1);
	ipu_ch_param_set_field(p, IPU_FIELD_OFS0, red_offset);
	/* Setup green width and offset */
	ipu_ch_param_set_field(p, IPU_FIELD_WID1, green_width - 1);
	ipu_ch_param_set_field(p, IPU_FIELD_OFS1, green_offset);
	/* Setup blue width and offset */
	ipu_ch_param_set_field(p, IPU_FIELD_WID2, blue_width - 1);
	ipu_ch_param_set_field(p, IPU_FIELD_OFS2, blue_offset);
	/* Setup alpha width and offset */
	ipu_ch_param_set_field(p, IPU_FIELD_WID3, alpha_width - 1);
	ipu_ch_param_set_field(p, IPU_FIELD_OFS3, alpha_offset);
}

static void fill_cpmem(struct ipu_ch_param __iomem *p,
		       struct ipu_ch_param *params)
{
	void __iomem *addr = p;
	int i, w;

	/* 2 words, 5 valid data */
	for (w = 0; w < 2; w++) {
		for (i = 0; i < 5; i++) {
			writel(params->word[w].data[i], addr);
			addr += 4;
		}
		addr += 12;
	}
}

void ipu_cpmem_channel_init(struct ipuv3_channel *channel,
			    u32 pixel_fmt, u32 width,
			    u32 height, u32 stride,
			    u32 u, u32 v,
			    u32 uv_stride, dma_addr_t addr0,
			    dma_addr_t addr1, dma_addr_t addr2)
{
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);
	struct ipu_soc *ipu = channel->ipu;
	struct ipu_ch_param params;
	int ch = channel->num;
	u32 u_offset = 0;
	u32 v_offset = 0;

	memset(&params, 0, sizeof(params));

	ipu_ch_param_set_field(&params, IPU_FIELD_FW, width - 1);

	if (ch == 8 || ch == 9 || ch == 10) {
		ipu_ch_param_set_field(&params, IPU_FIELD_FH, (height / 2) - 1);
		ipu_ch_param_set_field(&params, IPU_FIELD_SL, (stride * 2) - 1);
	} else {
		/* note: for vdoa+vdi- ch8/9/10, always use band mode */
		ipu_ch_param_set_field(&params, IPU_FIELD_FH, height - 1);
		ipu_ch_param_set_field(&params, IPU_FIELD_SL, stride - 1);
	}

	/* EBA is 8-byte aligned */
	ipu_ch_param_set_field(&params, IPU_FIELD_EBA0, addr0 >> 3);
	ipu_ch_param_set_field(&params, IPU_FIELD_EBA1, addr1 >> 3);
	if (addr0 % 8)
		dev_warn(ipu->dev, "ch%d EBA0 %08x is not 8-byte aligned\n",
			 ch, addr0);
	if (addr1 % 8)
		dev_warn(ipu->dev, "ch%d EBA1 %08x is not 8-byte aligned\n",
			 ch, addr1);

	switch (pixel_fmt) {
	case V4L2_PIX_FMT_RGB565:
		ipu_ch_param_set_field(&params, IPU_FIELD_BPP, 3);
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 7);
		ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);
		ipu_ch_param_set_packing(&params, 5, 0, 6, 5, 5, 11, 8, 16);
		break;
	case V4L2_PIX_FMT_BGR24:
		ipu_ch_param_set_field(&params, IPU_FIELD_BPP, 1);
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 7);
		ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 19);
		ipu_ch_param_set_packing(&params, 8, 0, 8, 8, 8, 16, 8, 24);
		break;
	case V4L2_PIX_FMT_RGB24:
	case V4L2_PIX_FMT_YUV444:
		ipu_ch_param_set_field(&params, IPU_FIELD_BPP, 1);
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 7);
		ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 19);
		ipu_ch_param_set_packing(&params, 8, 16, 8, 8, 8, 0, 8, 24);
		break;
	case V4L2_PIX_FMT_BGR32:
		ipu_ch_param_set_field(&params, IPU_FIELD_BPP, 0);
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 7);
		ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 15);
		ipu_ch_param_set_packing(&params, 8, 8, 8, 16, 8, 24, 8, 0);
		break;
	case V4L2_PIX_FMT_RGB32:
		ipu_ch_param_set_field(&params, IPU_FIELD_BPP, 0);
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 7);
		ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 15);
		ipu_ch_param_set_packing(&params, 8, 24, 8, 16, 8, 8, 8, 0);
		break;
	case V4L2_PIX_FMT_UYVY:
		ipu_ch_param_set_field(&params, IPU_FIELD_BPP, 3);
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 0xA);
		if ((ch == 8) || (ch == 9) || (ch == 10))
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 15);
		else
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);
		break;
	case V4L2_PIX_FMT_YUYV:
		ipu_ch_param_set_field(&params, IPU_FIELD_BPP, 3);
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 0x8);
		if ((ch == 8) || (ch == 9) || (ch == 10))
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 15);
		else
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);
		break;
	case V4L2_PIX_FMT_YUV420:
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 2);

		if (uv_stride < stride / 2)
			uv_stride = stride / 2;

		u_offset = stride * height;
		v_offset = u_offset + (uv_stride * height / 2);
		if ((ch == 8) || (ch == 9) || (ch == 10)) {
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 15);
			uv_stride = uv_stride*2;
		} else {
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);
		}
		break;
	case V4L2_PIX_FMT_YVU420:
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 2);

		if (uv_stride < stride / 2)
			uv_stride = stride / 2;

		v_offset = stride * height;
		u_offset = v_offset + (uv_stride * height / 2);
		if ((ch == 8) || (ch == 9) || (ch == 10)) {
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 15);
			uv_stride = uv_stride*2;
		} else {
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);
		}
		break;
	case V4L2_PIX_FMT_YUV422P:
		/* BPP & pixel format */
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 1);
		ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);

		if (uv_stride < stride / 2)
			uv_stride = stride / 2;

		u_offset = (u == 0) ? stride * height : u;
		v_offset = (v == 0) ? u_offset + u_offset / 2 : v;
		break;
	case V4L2_PIX_FMT_YVU422P:
		/* BPP & pixel format */
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 1);
		ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);

		if (uv_stride < stride / 2)
			uv_stride = stride / 2;

		v_offset = (v == 0) ? stride * height : v;
		u_offset = (u == 0) ? v_offset + v_offset / 2 : u;
		break;
	case V4L2_PIX_FMT_NV12:
		/* BPP & pixel format */
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 4);
		uv_stride = stride;
		u_offset = (u == 0) ? stride * height : u;
		if ((ch == 8) || (ch == 9) || (ch == 10)) {
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 15);
			/* top/bottom field in one buffer*/
			uv_stride = uv_stride*2;
		} else {
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);
		}
		break;
	case V4L2_PIX_FMT_NV21:
		/* BPP & pixel format */
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 4);
		uv_stride = stride;
		v_offset = (v == 0) ? stride * height : v;
		if ((ch == 8) || (ch == 9) || (ch == 10)) {
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 15);
			/* top/bottom field in one buffer*/
			uv_stride = uv_stride*2;
		} else {
			ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);
		}
		break;
	case V4L2_PIX_FMT_NV16:
		/* BPP & pixel format */
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 3);
		ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);
		uv_stride = stride;
		u_offset = (u == 0) ? stride * height : u;
		break;
	case V4L2_PIX_FMT_NV61:
		/* BPP & pixel format */
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 3);
		ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);
		uv_stride = stride;
		v_offset = (v == 0) ? stride * height : v;
		break;
	case V4L2_PIX_FMT_GENERIC16:
		/* Raw 16-bit generic */
		ipu_ch_param_set_field(&params, IPU_FIELD_BPP, 3);
		ipu_ch_param_set_field(&params, IPU_FIELD_NPB, 31);
		ipu_ch_param_set_field(&params, IPU_FIELD_PFS, 6);
		break;
	default:
		dev_err(ipu->dev, "mxc ipu: unimplemented pixel format\n");
		break;
	}
	/*set burst size to 16*/


	if (uv_stride)
		ipu_ch_param_set_field(&params, IPU_FIELD_SLUV, uv_stride - 1);

	/* Get the uv offset from user when need cropping */
	if (u || v) {
		u_offset = u;
		v_offset = v;
	}

	/* UBO and VBO are 22-bit and 8-byte aligned */
	if (u_offset/8 > 0x3fffff)
		dev_warn(ipu->dev,
			 "IDMAC%d's U offset exceeds IPU limitation\n", ch);
	if (v_offset/8 > 0x3fffff)
		dev_warn(ipu->dev,
			 "IDMAC%d's V offset exceeds IPU limitation\n", ch);
	if (u_offset % 8)
		dev_warn(ipu->dev, "ch%d U offset %08x is not 8-byte aligned\n",
			 ch, u_offset);
	if (v_offset % 8)
		dev_warn(ipu->dev, "ch%d V offset %08x is not 8-byte aligned\n",
			 ch, v_offset);

	ipu_ch_param_set_field(&params, IPU_FIELD_UBO, u_offset / 8);
	ipu_ch_param_set_field(&params, IPU_FIELD_VBO, v_offset / 8);

	dev_dbg(ipu->dev, "initializing idma ch %d @ %p\n", channel->num, p);
	fill_cpmem(p, &params);

	if (addr2) {
		struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);

		ipu_ch_param_set_field(&params, IPU_FIELD_EBA0, addr2 >> 3);
		ipu_ch_param_set_field(&params, IPU_FIELD_EBA1, 0);

		if (alt_p) {
			dev_dbg(ipu->dev,
				"initializing idma ch %d @ %p sub cpmem\n",
				channel->num, alt_p);
			fill_cpmem(alt_p, &params);
		}
	}
}
EXPORT_SYMBOL_GPL(ipu_cpmem_channel_init);

void ipu_cpmem_set_burst_size(struct ipuv3_channel *channel, u16 burst_pixels)
{
	struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);

	ipu_ch_param_mod_field_io(ipu_get_cpmem(channel), IPU_FIELD_NPB,
				  burst_pixels - 1);

	if (alt_p)
		ipu_ch_param_mod_field_io(alt_p, IPU_FIELD_NPB,
					  burst_pixels - 1);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_burst_size);

int ipu_cpmem_get_burst_size(struct ipuv3_channel *channel)
{
	return ipu_ch_param_read_field(ipu_get_cpmem(channel),
				       IPU_FIELD_NPB) + 1;
}
EXPORT_SYMBOL_GPL(ipu_cpmem_get_burst_size);

int ipu_cpmem_get_bpp(struct ipuv3_channel *channel)
{
	return ipu_ch_param_read_field(ipu_get_cpmem(channel), IPU_FIELD_BPP);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_get_bpp);

void ipu_cpmem_set_buffer(struct ipuv3_channel *channel,
			  int bufNum, dma_addr_t phyaddr)
{
	struct ipu_ch_param __iomem *cpmem;
	struct ipu_soc *ipu = channel->ipu;

	if (phyaddr % 8)
		dev_warn(ipu->dev, "ch%d EBA %08x is not 8-byte aligned\n",
			 channel->num, phyaddr);

	if (bufNum == 2) {
		cpmem = ipu_get_alt_cpmem(channel);
		if (!cpmem)
			return;
		ipu_ch_param_mod_field_io(cpmem, IPU_FIELD_EBA0, phyaddr / 8);
	} else {
		cpmem = ipu_get_cpmem(channel);
		ipu_ch_param_mod_field_io(
			cpmem, bufNum ? IPU_FIELD_EBA1 : IPU_FIELD_EBA0,
			phyaddr / 8);
	}
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_buffer);

void ipu_cpmem_set_uv_offset(struct ipuv3_channel *channel,
			     u32 u_offset, u32 v_offset)
{
	struct ipu_ch_param __iomem *cpmem = ipu_get_cpmem(channel);
	struct ipu_soc *ipu = channel->ipu;

	if (u_offset % 8)
		dev_warn(ipu->dev, "ch%d U offset %08x is not 8-byte aligned\n",
			 channel->num, u_offset);
	if (v_offset % 8)
		dev_warn(ipu->dev, "ch%d V offset %08x is not 8-byte aligned\n",
			 channel->num, v_offset);

	ipu_ch_param_write_field(cpmem, IPU_FIELD_UBO, u_offset / 8);
	ipu_ch_param_write_field(cpmem, IPU_FIELD_VBO, v_offset / 8);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_uv_offset);

void ipu_cpmem_set_rotation(struct ipuv3_channel *channel,
			    enum ipu_rotate_mode rot)
{
	struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);
	u32 temp_rot = bitrev8(rot) >> 5;

	ipu_ch_param_mod_field_io(ipu_get_cpmem(channel),
				  IPU_FIELD_ROT_HF_VF, temp_rot);
	if (alt_p)
		ipu_ch_param_mod_field_io(alt_p, IPU_FIELD_ROT_HF_VF, temp_rot);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_rotation);

void ipu_cpmem_set_block_mode(struct ipuv3_channel *channel)
{
	struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);

	ipu_ch_param_mod_field_io(ipu_get_cpmem(channel), IPU_FIELD_BM, 1);

	if (alt_p)
		ipu_ch_param_mod_field_io(alt_p, IPU_FIELD_BM, 1);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_block_mode);

void ipu_cpmem_set_alpha_use_separate_channel(struct ipuv3_channel *channel,
					      bool option)
{
	struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);

	ipu_ch_param_mod_field_io(ipu_get_cpmem(channel), IPU_FIELD_ALU,
				  option);
	if (alt_p)
		ipu_ch_param_mod_field_io(alt_p, IPU_FIELD_ALU, option);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_alpha_use_separate_channel);

void ipu_cpmem_set_alpha_condition_read(struct ipuv3_channel *channel)
{
	struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);

	ipu_ch_param_mod_field_io(ipu_get_cpmem(channel), IPU_FIELD_CRE, 1);

	if (alt_p)
		ipu_ch_param_mod_field_io(alt_p, IPU_FIELD_CRE, 1);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_alpha_condition_read);

void ipu_cpmem_set_alpha_buffer_memory(struct ipuv3_channel *channel)
{
	struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);
	struct ipu_soc *ipu = channel->ipu;
	int alp_mem_idx;

	switch (channel->num) {
	case 14: /* PRP graphic */
		alp_mem_idx = 0;
		break;
	case 15: /* PP graphic */
		alp_mem_idx = 1;
		break;
	case 23: /* DP BG SYNC graphic */
		alp_mem_idx = 4;
		break;
	case 27: /* DP FG SYNC graphic */
		alp_mem_idx = 2;
		break;
	default:
		dev_err(ipu->dev, "unsupported correlative channel of local " \
				"alpha channel\n");
		return;
	}

	ipu_ch_param_mod_field_io(ipu_get_cpmem(channel),
				  IPU_FIELD_ALBM, alp_mem_idx);

	if (alt_p)
		ipu_ch_param_mod_field_io(alt_p, IPU_FIELD_ALBM, alp_mem_idx);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_alpha_buffer_memory);

void ipu_cpmem_set_interlaced_scan(struct ipuv3_channel *channel, int stride)
{
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);
	struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);
	struct ipu_soc *ipu = channel->ipu;

	ipu_ch_param_set_field_io(p, IPU_FIELD_SO, 1);
	if (alt_p)
		ipu_ch_param_set_field_io(alt_p, IPU_FIELD_SO, 1);
	if (!stride)
		stride = ipu_ch_param_read_field(p, IPU_FIELD_SLY) + 1;
	/* ILO is 20-bit and 8-byte aligned */
	if (stride/8 > 0xfffff)
		dev_warn(ipu->dev, "IDMAC%d's ILO exceeds IPU limitation\n",
			 channel->num);
	if (stride % 8)
		dev_warn(ipu->dev, "ch%d ILO %08x is not 8-byte aligned\n",
			 channel->num, stride);
	ipu_ch_param_mod_field_io(p, IPU_FIELD_ILO, stride / 8);
	if (alt_p)
		ipu_ch_param_mod_field_io(alt_p, IPU_FIELD_ILO,
					  stride / 8);
	stride *= 2;
	ipu_ch_param_mod_field_io(p, IPU_FIELD_SLY, stride - 1);
	if (alt_p)
		ipu_ch_param_mod_field_io(alt_p, IPU_FIELD_SLY, stride - 1);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_interlaced_scan);

void ipu_cpmem_set_axi_id(struct ipuv3_channel *channel, u32 id)
{
	struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);

	id %= 4;

	ipu_ch_param_mod_field_io(ipu_get_cpmem(channel), IPU_FIELD_ID, id);

	if (alt_p)
		ipu_ch_param_mod_field_io(alt_p, IPU_FIELD_ID, id);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_axi_id);

/* IDMAC U/V offset changing support */
/* U and V input is not affected, */
/* the update is done by new calculation according to */
/* vertical_offset and horizontal_offset */
void ipu_cpmem_offset_update(struct ipuv3_channel *channel,
			     u32 pixel_fmt,
			     u32 width,
			     u32 height,
			     u32 stride,
			     u32 u,
			     u32 v,
			     u32 uv_stride,
			     u32 vertical_offset,
			     u32 horizontal_offset)
{
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);
	struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);
	struct ipu_soc *ipu = channel->ipu;
	u32 u_offset = 0;
	u32 v_offset = 0;
	u32 old_offset = 0;
	u32 u_fix = 0;
	u32 v_fix = 0;

	switch (pixel_fmt) {
	case V4L2_PIX_FMT_RGB565:
	case V4L2_PIX_FMT_BGR24:
	case V4L2_PIX_FMT_RGB24:
	case V4L2_PIX_FMT_YUV444:
	case V4L2_PIX_FMT_BGR32:
	case V4L2_PIX_FMT_RGB32:
	case V4L2_PIX_FMT_UYVY:
	case V4L2_PIX_FMT_YUYV:
	case V4L2_PIX_FMT_GENERIC16:
		break;

	case V4L2_PIX_FMT_YUV420:
		if (uv_stride < stride / 2)
			uv_stride = stride / 2;

		u_offset = stride * (height - vertical_offset - 1) +
			(stride - horizontal_offset) +
			(uv_stride * vertical_offset / 2) +
			horizontal_offset / 2;
		v_offset = u_offset + (uv_stride * height / 2);
		u_fix = u ? (u + (uv_stride * vertical_offset / 2) +
			     (horizontal_offset / 2) -
			     (stride * vertical_offset) -
			     (horizontal_offset)) :
			u_offset;
		v_fix = v ? (v + (uv_stride * vertical_offset / 2) +
			     (horizontal_offset / 2) -
			     (stride * vertical_offset) -
			     (horizontal_offset)) :
			v_offset;
		break;
	case V4L2_PIX_FMT_YUV422P:
		if (uv_stride < stride / 2)
			uv_stride = stride / 2;

		u_offset = stride * (height - vertical_offset - 1) +
			(stride - horizontal_offset) +
			(uv_stride * vertical_offset) +
			horizontal_offset / 2;
		v_offset = u_offset + uv_stride * height;
		u_fix = u ? (u + (uv_stride * vertical_offset) +
			     horizontal_offset / 2 -
			     (stride * vertical_offset) - (horizontal_offset)) :
			u_offset;
		v_fix = v ? (v + (uv_stride * vertical_offset) +
			     horizontal_offset / 2 -
			     (stride * vertical_offset) - (horizontal_offset)) :
			v_offset;
		break;

	case V4L2_PIX_FMT_NV12:
		uv_stride = stride;
		u_offset = stride * (height - vertical_offset - 1) +
			(stride - horizontal_offset) +
			(uv_stride * vertical_offset / 2) +
			horizontal_offset;
		u_fix = u ? (u + (uv_stride * vertical_offset / 2) +
			     horizontal_offset -
			     (stride * vertical_offset) - (horizontal_offset)) :
			u_offset;
		break;
	case V4L2_PIX_FMT_NV16:
		uv_stride = stride;
		u_offset = stride * (height - vertical_offset - 1) +
			(stride - horizontal_offset) +
			(uv_stride * vertical_offset) +
			horizontal_offset;
		u_fix = u ? (u + (uv_stride * vertical_offset) +
			     horizontal_offset -
			     (stride * vertical_offset) - (horizontal_offset)) :
			u_offset;
		break;
	default:
		dev_err(ipu->dev, "mxc ipu: unimplemented pixel format\n");
		break;
	}

	if (u_fix > u_offset)
		u_offset = u_fix;

	if (v_fix > v_offset)
		v_offset = v_fix;

	/* UBO and VBO are 22-bit and 8-byte aligned */
	if (u_offset/8 > 0x3fffff)
		dev_warn(ipu->dev,
			 "IDMAC%d's U offset exceeds IPU limitation\n",
			 channel->num);
	if (v_offset/8 > 0x3fffff)
		dev_warn(ipu->dev,
			"IDMAC%d's V offset exceeds IPU limitation\n",
			 channel->num);
	if (u_offset % 8)
		dev_warn(ipu->dev, "ch%d U offset %08x is not 8-byte aligned\n",
			 channel->num, u_offset);
	if (v_offset % 8)
		dev_warn(ipu->dev, "ch%d V offset %08x is not 8-byte aligned\n",
			 channel->num, v_offset);

	old_offset = ipu_ch_param_read_field(p, IPU_FIELD_UBO);
	if (old_offset != u_offset / 8)
		ipu_ch_param_mod_field_io(p, IPU_FIELD_UBO, u_offset / 8);
	old_offset = ipu_ch_param_read_field(p, IPU_FIELD_VBO);
	if (old_offset != v_offset / 8)
		ipu_ch_param_mod_field_io(p, IPU_FIELD_VBO, v_offset / 8);

	if (alt_p) {
		old_offset = ipu_ch_param_read_field(alt_p, IPU_FIELD_UBO);
		if (old_offset != u_offset / 8)
			ipu_ch_param_mod_field_io(alt_p, IPU_FIELD_UBO,
						  u_offset / 8);
		old_offset = ipu_ch_param_read_field(alt_p, IPU_FIELD_VBO);
		if (old_offset != v_offset / 8)
			ipu_ch_param_mod_field_io(alt_p, IPU_FIELD_VBO,
						  v_offset / 8);
	}
}
EXPORT_SYMBOL_GPL(ipu_cpmem_offset_update);

void ipu_cpmem_set_alpha_width(struct ipuv3_channel *channel, int alpha_width)
{
	struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);

	ipu_ch_param_set_field_io(ipu_get_cpmem(channel),
				  IPU_FIELD_WID3, alpha_width - 1);
	if (alt_p)
		ipu_ch_param_set_field_io(alt_p, IPU_FIELD_WID3,
					  alpha_width - 1);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_alpha_width);

void ipu_cpmem_set_bandmode(struct ipuv3_channel *channel, u32 band_height)
{
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);
	struct ipu_ch_param __iomem *alt_p = ipu_get_alt_cpmem(channel);
	struct ipu_soc *ipu = channel->ipu;

	ipu_ch_param_set_field_io(p, IPU_FIELD_BNDM, band_height - 1);

	dev_dbg(ipu->dev, "BNDM 0x%x, ",
		ipu_ch_param_read_field(p, IPU_FIELD_BNDM));

	if (alt_p)
		ipu_ch_param_set_field_io(alt_p, IPU_FIELD_BNDM,
					  band_height - 1);
}
EXPORT_SYMBOL_GPL(ipu_cpmem_set_bandmode);

void ipu_cpmem_dump(struct ipuv3_channel *channel)
{
	struct ipu_ch_param __iomem *p = ipu_get_cpmem(channel);
	struct ipu_soc *ipu = channel->ipu;
	int ch = channel->num;

	dev_dbg(ipu->dev, "ch %d word 0 - %08X %08X %08X %08X %08X\n", ch,
		readl(&p->word[0].data[0]),
		readl(&p->word[0].data[1]),
		readl(&p->word[0].data[2]),
		readl(&p->word[0].data[3]),
		readl(&p->word[0].data[4]));
	dev_dbg(ipu->dev, "ch %d word 1 - %08X %08X %08X %08X %08X\n", ch,
		readl(&p->word[1].data[0]),
		readl(&p->word[1].data[1]),
		readl(&p->word[1].data[2]),
		readl(&p->word[1].data[3]),
		readl(&p->word[1].data[4]));
	dev_dbg(ipu->dev, "PFS 0x%x, ",
		 ipu_ch_param_read_field(p, IPU_FIELD_PFS));
	dev_dbg(ipu->dev, "BPP 0x%x, ",
		ipu_ch_param_read_field(p, IPU_FIELD_BPP));
	dev_dbg(ipu->dev, "NPB 0x%x\n",
		 ipu_ch_param_read_field(p, IPU_FIELD_NPB));

	dev_dbg(ipu->dev, "FW %d, ",
		 ipu_ch_param_read_field(p, IPU_FIELD_FW));
	dev_dbg(ipu->dev, "FH %d, ",
		 ipu_ch_param_read_field(p, IPU_FIELD_FH));
	dev_dbg(ipu->dev, "EBA0 0x%x\n",
		 ipu_ch_param_read_field(p, IPU_FIELD_EBA0) << 3);
	dev_dbg(ipu->dev, "EBA1 0x%x\n",
		 ipu_ch_param_read_field(p, IPU_FIELD_EBA1) << 3);
	dev_dbg(ipu->dev, "Stride %d\n",
		 ipu_ch_param_read_field(p, IPU_FIELD_SL));
	dev_dbg(ipu->dev, "scan_order %d\n",
		 ipu_ch_param_read_field(p, IPU_FIELD_SO));
	dev_dbg(ipu->dev, "uv_stride %d\n",
		 ipu_ch_param_read_field(p, IPU_FIELD_SLUV));
	dev_dbg(ipu->dev, "u_offset 0x%x\n",
		 ipu_ch_param_read_field(p, IPU_FIELD_UBO) << 3);
	dev_dbg(ipu->dev, "v_offset 0x%x\n",
		 ipu_ch_param_read_field(p, IPU_FIELD_VBO) << 3);

	dev_dbg(ipu->dev, "Width0 %d+1, ",
		 ipu_ch_param_read_field(p, IPU_FIELD_WID0));
	dev_dbg(ipu->dev, "Width1 %d+1, ",
		 ipu_ch_param_read_field(p, IPU_FIELD_WID1));
	dev_dbg(ipu->dev, "Width2 %d+1, ",
		 ipu_ch_param_read_field(p, IPU_FIELD_WID2));
	dev_dbg(ipu->dev, "Width3 %d+1, ",
		 ipu_ch_param_read_field(p, IPU_FIELD_WID3));
	dev_dbg(ipu->dev, "Offset0 %d, ",
		 ipu_ch_param_read_field(p, IPU_FIELD_OFS0));
	dev_dbg(ipu->dev, "Offset1 %d, ",
		 ipu_ch_param_read_field(p, IPU_FIELD_OFS1));
	dev_dbg(ipu->dev, "Offset2 %d, ",
		 ipu_ch_param_read_field(p, IPU_FIELD_OFS2));
	dev_dbg(ipu->dev, "Offset3 %d\n",
		 ipu_ch_param_read_field(p, IPU_FIELD_OFS3));
}
EXPORT_SYMBOL_GPL(ipu_cpmem_dump);

int ipu_cpmem_init(struct ipu_soc *ipu, struct device *dev, unsigned long base)
{
	struct ipu_cpmem *cpmem;
	int i;

	cpmem = devm_kzalloc(dev, sizeof(*cpmem), GFP_KERNEL);
	if (!cpmem)
		return -ENOMEM;

	ipu->cpmem_priv = cpmem;

	spin_lock_init(&cpmem->lock);
	cpmem->base = devm_ioremap(dev, base, SZ_128K);
	if (!cpmem->base)
		return -ENOMEM;

	dev_dbg(dev, "CPMEM base: 0x%08lx remapped to %p\n",
		base, cpmem->base);
	cpmem->ipu = ipu;

	for (i = 0; i < 64; i++)
		ipu->channel[i].cpmem = cpmem;

	return 0;
}

void ipu_cpmem_exit(struct ipu_soc *ipu)
{
}

