/*
 * Copyright 2014 Robert Bosch Car Multimedia GmbH
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/kernel.h>
#include <linux/bitops.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/notifier.h>
#include <linux/module.h>
#include <linux/of_gpio.h>
#include <linux/platform_device.h>
#include <linux/reboot.h>
#include <linux/workqueue.h>

#define UNDERVOLTAGE_NAME	"undervoltage"
#define WQ_NAME			"hpi-pon-qw"

#define CPU_RUN		BIT(0)
#define EMMC_SHUTDOWN	BIT(1)

ATOMIC_NOTIFIER_HEAD(undervoltage_notifier_list);

void register_undervoltage_notifier(struct notifier_block *nb)
{
	atomic_notifier_chain_register(&undervoltage_notifier_list, nb);
}

void unregister_undervoltage_notifier(struct notifier_block *nb)
{
	atomic_notifier_chain_unregister(&undervoltage_notifier_list, nb);
}

struct undervoltage_gpio {
	int nr;
	char *name;
	int irq;
	int edge;
};
struct uv_work_t {
	struct work_struct my_work;
	struct undervoltage_data *uvdata;
};

struct undervoltage_data {
	struct device *dev;
	int udrop_gpio;
	int udrop_irq;
	int rstwrn_gpio;
	int rstwrn_irq;
	int cpurun_gpio;
	bool cpurun_inactive;
	bool irq_done;
	bool irq_disabled;
	unsigned int cpurun_value;
	struct uv_work_t *uv_work;
	struct workqueue_struct *notify_wq;
};

static void undervoltage_worker(struct work_struct *ptr)
{
	struct uv_work_t *work_ptr = container_of(
					ptr,  struct uv_work_t, my_work);
	struct undervoltage_data *uvdata  = work_ptr->uvdata;

	atomic_notifier_call_chain(&undervoltage_notifier_list, 0, NULL);

	/* Notify the SCC about the shut down */
	gpio_set_value(uvdata->cpurun_gpio, uvdata->cpurun_inactive);

	/*
	 * Point of no return
	 *
	 * After notifying all sub systems about the shut down above,
	 * system power off is expected after 30 ms done by the
	 * SCC.
	 */

	/*
	 * This is expected to NEVER run!
	 * Just in case the SCC fails to do the shut down,
	 * reset the system after 300ms
	 */
	msleep(300);

	pr_alert("Error: SCC reset failed after 300ms. Restarting the system\n");
	kernel_restart(NULL);
}

static irqreturn_t undervoltage_irq(int irq, void *data)
{
	struct undervoltage_data *uvdata = data;

	/* Ensure we are running this handler only once */
	if ((uvdata->irq_done) || (uvdata->irq_disabled))
		return IRQ_HANDLED;

	uvdata->irq_done = 1;

	queue_work(uvdata->notify_wq, &uvdata->uv_work->my_work);

	return IRQ_HANDLED;
}

ssize_t undervoltage_trigger(struct device *dev, struct device_attribute *attr,
			     const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct undervoltage_data *uvdata = platform_get_drvdata(pdev);
	int val, ret;

	ret = kstrtoint(buf, 10, &val);
	if (ret)
		return ret;

	/*
	 * valid values:
	 * 0: CPURUN GPIO: inactive, shutdown notifier not triggered
	 * 1: CPURUN GPIO: active,   shutdown notifier not triggered
	 * 2: CPURUN GPIO: inactive, shutdown notifier     triggered
	 * 3: CPURUN GPIO: active,   shutdown notifier     triggered
	 */
	if ((val < 0) || (val > 3)) {
		dev_err(dev,
			"invalid value %i. Use values between 0 and 3!\n",
			val);
		return -EINVAL;
	}

	if (val & EMMC_SHUTDOWN) {
		queue_work(uvdata->notify_wq, &uvdata->uv_work->my_work);
		dev_info(dev, "eMMC shutdown triggered");
	}

	val = (val & CPU_RUN) ? !uvdata->cpurun_inactive :
				uvdata->cpurun_inactive;

	gpio_set_value(uvdata->cpurun_gpio, val);
	dev_info(dev, "CPU RUN GPIO set to %d\n", val);

	return count;
}
DEVICE_ATTR(trigger, S_IWUSR, NULL, undervoltage_trigger);

ssize_t undervoltage_disable_show(struct device *dev,
				  struct device_attribute *attr,
				  char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct undervoltage_data *uvdata = platform_get_drvdata(pdev);

	return scnprintf(buf, 4, "%i\n", uvdata->irq_disabled);
}

ssize_t undervoltage_disable_store(struct device *dev,
				   struct device_attribute *attr,
				   const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct undervoltage_data *uvdata = platform_get_drvdata(pdev);
	int val, ret;

	ret = kstrtoint(buf, 10, &val);
	if (ret)
		return ret;

	/*
	 * valid values:
	 * 0: disable == 0 (false) -> enable undervoltage IRQ
	 * 1: disable == 1 (true)  -> disable undervoltage IRQ
	 */
	if ((val < 0) || (val > 1)) {
		dev_err(dev,
			"invalid value %i. Use 0 (enable) or 1 (disable)!\n",
			val);
		return -EINVAL;
	}

	/* Disable only if the IRQ has not happened, yet */
	if ((val == 1) && (uvdata->irq_done == 0)) {
		uvdata->irq_disabled = 1;
		dev_info(dev, "undervoltage IRQ disabled\n");
	}

	/*
	 * Enable only if the IRQ has been disabled
	 */
	if ((val == 0) && (uvdata->irq_disabled == 1)) {
		uvdata->irq_disabled = 0;
		dev_info(dev, "undervoltage IRQ enabled\n");
	}

	return count;
}
DEVICE_ATTR(disable, S_IRUGO | S_IWUSR, undervoltage_disable_show,
	    undervoltage_disable_store);

static int request_gpio_irq(struct device *dev,
			    struct undervoltage_gpio *gpio,
			    struct undervoltage_data *uvdata)
{
	int err;

	err = gpio_request(gpio->nr, gpio->name);
	if (err < 0) {
		dev_err(dev, "failed to request gpio (%s)\n", gpio->name);
		goto err2;
	}

	err = gpio_direction_input(gpio->nr);
	if (err < 0) {
		dev_err(dev, "failed to set gpio as input (%s)\n", gpio->name);
		goto err1;
	}

	gpio->irq = gpio_to_irq(gpio->nr);
	if (gpio->irq < 0) {
		dev_err(dev, "no irq associated with gpio (%d)\n", gpio->nr);
		goto err1;
	}

	err = request_irq(gpio->irq, undervoltage_irq, gpio->edge,
			  gpio->name, (void *)uvdata);

	if (err) {
		dev_err(dev, "irq could not be requested (%d)\n", gpio->irq);
		goto err1;
	}

	return 0;

err1:
	gpio_free(gpio->nr);
err2:
	return -EINVAL;
}

static int undervoltage_probe(struct platform_device *pdev)
{
	struct device_node *uv_node = pdev->dev.of_node;
	struct device_node *dt_node = NULL;
	struct undervoltage_data *uvdata;
	struct undervoltage_gpio gpio;
	const char *edge;
	bool gpio_val;
	int err = 0;

	uvdata = devm_kzalloc(&pdev->dev, sizeof(struct undervoltage_data),
			      GFP_KERNEL);
	if (!uvdata) {
		dev_err(&pdev->dev, "error: can't allocate enough memory forthe undervoltage data\n");
		err = -ENOMEM;
		goto out1;
	}

	uvdata->uv_work = devm_kzalloc(&pdev->dev, sizeof(struct uv_work_t),
					GFP_KERNEL);
	if (!uvdata->uv_work) {
		dev_err(&pdev->dev, "error: can't allocate enough memory for the workqueue\n");
		err = -ENOMEM;
		goto out1;
	}

	uvdata->notify_wq = create_singlethread_workqueue(WQ_NAME);
	if (!uvdata->notify_wq) {
		dev_err(&pdev->dev, "error: can't create workqueue\n");
		err = -ENOMEM;
		goto out1;
	}

	uvdata->uv_work->uvdata = uvdata;
	uvdata->dev = &pdev->dev;

	/* CPU RUN GPIO -> output */
	dt_node = of_get_child_by_name(uv_node, "cpu-run");
	if (!dt_node) {
		dev_err(&pdev->dev, "Could not find cpu-run node in device tree\n");
		err = -ENODEV;
		goto out1;
	}

	uvdata->cpurun_gpio = of_get_named_gpio(dt_node, "gpios", 0);
	if (!gpio_is_valid(uvdata->cpurun_gpio)) {
		dev_err(&pdev->dev, "Invalid CPU RUN GPIO\n");
		of_node_put(dt_node);
		err = -EINVAL;
		goto out1;
	}

	if (!of_find_property(dt_node, "value", NULL)) {
		dev_err(&pdev->dev, "No CPU RUN GPIO value found\n");
		of_node_put(dt_node);
		err = -EINVAL;
		goto out1;
	}

	err = of_property_read_u32(dt_node, "value", &uvdata->cpurun_value);
	if ((err < 0) || (uvdata->cpurun_value > 1)) {
		dev_err(&pdev->dev, "Wrong value for the CPU RUN GPIO (%d)\n",
			uvdata->cpurun_value);
		goto out1;
	}

	/* Get back TRUE (1) if active-low is defined */
	uvdata->cpurun_inactive = of_property_read_bool(dt_node, "active-low");

	of_node_put(dt_node);

	err = gpio_request(uvdata->cpurun_gpio, "CPU_HMI_RUN_GPIO");
	if (err < 0) {
		dev_err(&pdev->dev, "faild to request gpio CPU_HMI_RUN_GPIO (%d)\n",
			uvdata->cpurun_gpio);
		goto out1;
	}

	err = gpio_direction_output(uvdata->cpurun_gpio, uvdata->cpurun_value);
	if (err < 0) {
		dev_err(&pdev->dev, "failed to set CPU_HMI_RUN_GPIO as output\n");
		goto out2;
	}

	/* RST WARN GPIO -> IRQ */
	dt_node = of_get_child_by_name(uv_node, "scc-rstwarn-cpu");
	if (!dt_node) {
		dev_err(&pdev->dev, "Could not find rstwarn node in device tree\n");
		err = -ENODEV;
		goto out2;
	}

	uvdata->rstwrn_gpio = of_get_named_gpio(dt_node, "gpios", 0);
	if (!gpio_is_valid(uvdata->rstwrn_gpio)) {
		dev_err(&pdev->dev, "Invalid RST WARN GPIO\n");
		err = -EINVAL;
		of_node_put(dt_node);
		goto out2;
	}

	err = of_property_read_string(dt_node, "edge", &edge);
	of_node_put(dt_node);
	if (err) {
		dev_err(&pdev->dev, "RST WARN GPIO: no edge in device tree\n");
		goto out2;
	}

	if (strncmp(edge, "falling", 7) == 0)
		gpio.edge = IRQF_TRIGGER_FALLING;
	else if (strncmp(edge, "rising", 6) == 0)
		gpio.edge = IRQF_TRIGGER_RISING;
	else {
		dev_err(&pdev->dev, "RST WARN GPIO: unsupported edge %s\n",
			edge);
		err = -EINVAL;
		goto out2;
	}

	INIT_WORK(&uvdata->uv_work->my_work, undervoltage_worker);

	uvdata->irq_done = 0;

	gpio.nr		= uvdata->rstwrn_gpio;
	gpio.name	= "SCC_RSTWARN_CPU_GPIO";
	gpio.irq	= -1;
	err = request_gpio_irq(&pdev->dev, &gpio, uvdata);
	if (err < 0) {
		dev_err(&pdev->dev, "failed to request irq %d on %s (%d)\n",
			gpio.irq, gpio.name, gpio.nr);
		goto out2;
	}
	uvdata->rstwrn_irq = gpio.irq;

	/* Check if the GPIO is already active at init time */
	if (gpio.edge == IRQF_TRIGGER_FALLING)
		gpio_val = 0;
	else
		gpio_val = 1;

	if (!gpio_get_value(uvdata->rstwrn_gpio) == !gpio_val) {
		queue_work(uvdata->notify_wq, &uvdata->uv_work->my_work);
		uvdata->irq_done = 1;
	}

	platform_set_drvdata(pdev, uvdata);

	err = device_create_file(&pdev->dev, &dev_attr_trigger);
	if (err) {
		dev_err(&pdev->dev, "create sysfs file trigger failed\n");
		goto out3;
	}

	uvdata->irq_disabled = 0;

	err = device_create_file(&pdev->dev, &dev_attr_disable);
	if (err) {
		dev_err(&pdev->dev, "create sysfs file disable failed\n");
		goto out3;
	}

	/* Some boards also have 3 volt warning */
	dt_node = of_get_child_by_name(uv_node, "pwr-udrop-30");
	if (!dt_node) {
		uvdata->udrop_gpio = -1;
		goto skip;
	}

	uvdata->udrop_gpio = of_get_named_gpio(dt_node, "gpios", 0);
	if (!gpio_is_valid(uvdata->udrop_gpio)) {
		dev_err(&pdev->dev, "Invalid UDROP 30 GPIO\n");
		of_node_put(dt_node);
		goto skip;
	}

	err = of_property_read_string(dt_node, "edge", &edge);
	of_node_put(dt_node);
	if (err) {
		dev_err(&pdev->dev, "UDROP 30 GPIO: no edge in device tree\n");
		goto skip;
	}

	if (strncmp(edge, "falling", 7) == 0)
		gpio.edge = IRQF_TRIGGER_FALLING;
	else if (strncmp(edge, "rising", 6) == 0)
		gpio.edge = IRQF_TRIGGER_RISING;
	else {
		dev_err(&pdev->dev, "UDROP 30: unsupported edge %s\n", edge);
		goto skip;
	}

	gpio.nr		= uvdata->udrop_gpio;
	gpio.name	= "PWR_UDROP_30_GPIO";
	gpio.irq	= -1;
	err = request_gpio_irq(&pdev->dev, &gpio, uvdata);
	if (err < 0)
		dev_err(&pdev->dev, "failed to request irq %d on %s (%d)\n",
			gpio.irq, gpio.name, gpio.nr);

	uvdata->udrop_irq = gpio.irq;

	/* Check if the GPIO is already active at init time */
	if (gpio.edge == IRQF_TRIGGER_FALLING)
		gpio_val = 0;
	else
		gpio_val = 1;

	if ((!gpio_get_value(uvdata->udrop_gpio) == !gpio_val) &&
	    (!uvdata->irq_done)) {
		queue_work(uvdata->notify_wq, &uvdata->uv_work->my_work);
		uvdata->irq_done = 1;
	}

skip:
	return 0;

out3:
	free_irq(uvdata->rstwrn_irq, (void *)uvdata);
	gpio_free(uvdata->rstwrn_gpio);
out2:
	gpio_free(uvdata->cpurun_gpio);
out1:
	return err;
}

static int undervoltage_remove(struct platform_device *pdev)
{
	struct undervoltage_data *uvdata = platform_get_drvdata(pdev);

	device_remove_file(&pdev->dev, &dev_attr_trigger);

	if (gpio_is_valid(uvdata->udrop_gpio)) {
		free_irq(uvdata->udrop_irq, (void *)uvdata);
		gpio_free(uvdata->udrop_gpio);
	}

	free_irq(uvdata->rstwrn_irq, (void *)uvdata);
	gpio_free(uvdata->rstwrn_gpio);

	gpio_free(uvdata->cpurun_gpio);

	flush_workqueue(uvdata->notify_wq);

	destroy_workqueue(uvdata->notify_wq);

	return 0;
}

static struct of_device_id undervoltage_of_device_ids[] = {
	{.compatible = "Bosch,undervoltage"},
	{},
};
MODULE_DEVICE_TABLE(of, undervoltage_of_device_ids);

static struct platform_driver undervoltage_driver = {
	.driver = {
		.name = UNDERVOLTAGE_NAME,
		.owner = THIS_MODULE,
		.of_match_table = undervoltage_of_device_ids,
	},
	.probe = undervoltage_probe,
	.remove = undervoltage_remove
};

static int __init undervoltage_init(void)
{
	return platform_driver_register(&undervoltage_driver);
}
late_initcall(undervoltage_init);

static void __exit undervoltage_exit(void)
{
	platform_driver_unregister(&undervoltage_driver);
}
module_exit(undervoltage_exit);

MODULE_AUTHOR("Robert Bosch Car Multimedia GmbH, Dirk Behme");
MODULE_AUTHOR("Bosch SofTec GmbH, Stefan Steingraeber");
MODULE_DESCRIPTION("Undervoltage driver");
MODULE_LICENSE("GPL");
