/*
 * linux/drivers/misc/tzasc/tzasc.c
 *
 * Copyright (C) 2014 Advanced Driver Information Technology GmbH
 * Written by Martin Herzog (mherzog@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.   See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/err.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/tzasc.h>
#include <linux/errmem.h>

struct tzasc_struct {
	struct		module *owner;
	struct		device dev;
	int		nr;
	char		name[48];
	void __iomem	*base;
};

/*
 * This parameter can be used to apply a different control setting for the
 * speculative access.
 */
static int spec_ctl = -1;
module_param(spec_ctl, int, 0);
MODULE_PARM_DESC(spec_ctl,
	"Speculative access: 0 = rw, 1 = w, 2 = r, 3 = none, -1 = no change");

static struct platform_device_id tzasc_devtype[] = {
	{
		.name = "tzasc",
	},
	{ }
};
MODULE_DEVICE_TABLE(platform, tzasc_devtype);

static const struct of_device_id tzasc_ids[] = {
	{
		.compatible = "arm,tzc380",
		.data	    = &tzasc_devtype[0]
	},
	{ }
};
MODULE_DEVICE_TABLE(of, tzasc_ids);

#if defined(CONFIG_TZASC_PANIC_ON_IRQ)
static void tzasc_logger(const struct device *dev, const char *logstr)
{
	char errmemstr[256];

	sprintf(errmemstr, "%s %s: %s",
			dev_driver_string(dev), dev_name(dev), logstr);
	panic(errmemstr);
}
#elif defined(CONFIG_ERRMEM)
static void tzasc_logger(const struct device *dev, const char *logstr)
{
	char errmemstr[256];

	sprintf(errmemstr, "%s %s: %s",
			dev_driver_string(dev), dev_name(dev), logstr);
	errmem_store(0, strlen(errmemstr), errmemstr);
}
#else
static void tzasc_logger(const struct device *dev, const char *logstr)
{
	dev_err(dev, logstr);
}
#endif

static irqreturn_t tzasc_isr(int irq, void *dev_id)
{
	struct tzasc_struct *tzasc = dev_id;
	u32 fail_adr, fail_ctrl, fail_id;
	char logstr[256];

	fail_adr  = readl(tzasc->base + TZASC_FAIL_ADR_LOW);
	fail_ctrl = readl(tzasc->base + TZASC_FAIL_CTRL);
	fail_id   = readl(tzasc->base + TZASC_FAIL_ID);

	sprintf(logstr,
		"failed %sprivileged %ssecure %s access to 0x%08x by AXI ID 0x%02x\n",
		(fail_ctrl & (1 << 20)) ? "" : "un",
		(fail_ctrl & (1 << 19)) ? "non-" : "",
		(fail_ctrl & (1 << 24)) ? "write" : "read",
		fail_adr, fail_id & 0xff);
	tzasc_logger(&tzasc->dev, logstr);

	writel(0, tzasc->base + TZASC_INT_CLR);

	return IRQ_HANDLED;
}

static int tzasc_probe(struct platform_device *pdev)
{
	const struct of_device_id *of_id = of_match_device(tzasc_ids,
							   &pdev->dev);
	struct tzasc_struct *tzasc;
	struct resource *res;
	void __iomem *base;
	u32 reg_spec_ctl;
	int irq, ret;

	dev_dbg(&pdev->dev, "<%s>\n", __func__);
	dev_info(&pdev->dev, "version %x.%02x\n",
			TZASC_VERSION >> 8, TZASC_VERSION & 0xff);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(&pdev->dev, "can't get device resources\n");
		return -ENOENT;
	}
	irq = platform_get_irq(pdev, 0);
	if (irq < 0) {
		dev_err(&pdev->dev, "can't get irq number\n");
		return -ENOENT;
	}

	base = devm_request_and_ioremap(&pdev->dev, res);
	if (!base)
		return -EBUSY;

	tzasc = devm_kzalloc(&pdev->dev, sizeof(struct tzasc_struct),
				GFP_KERNEL);
	if (!tzasc) {
		dev_err(&pdev->dev, "can't allocate interface\n");
		return -ENOMEM;
	}

	if (of_id)
		pdev->id_entry = of_id->data;

	/* Setup tzasc driver structure */
	strlcpy(tzasc->name, pdev->name, sizeof(tzasc->name));
	tzasc->owner	= THIS_MODULE;
	tzasc->dev	= pdev->dev;
	tzasc->nr	= pdev->id;
	tzasc->base	= base;

	/* Request IRQ */
	ret = devm_request_irq(&pdev->dev, irq, tzasc_isr, 0,
				pdev->name, tzasc);
	if (ret) {
		dev_err(&pdev->dev, "can't claim irq %d\n", irq);
		return ret;
	}

	/* Set up platform driver data */
	platform_set_drvdata(pdev, tzasc);

	if ((spec_ctl >= 0) && (spec_ctl <= 3)) {
		reg_spec_ctl = readl(tzasc->base + TZASC_SPEC_CTL);
		dev_info(&tzasc->dev,
			"got valid module param spec_ctl=%d, old setting=%d\n",
			spec_ctl, reg_spec_ctl & 0x3);
		reg_spec_ctl = (reg_spec_ctl & 0xfffffffc) | (spec_ctl & 0x3);
		writel(reg_spec_ctl, tzasc->base + TZASC_SPEC_CTL);
	}

	dev_dbg(&tzasc->dev, "claimed irq %d\n", irq);
	dev_dbg(&tzasc->dev, "device resources from 0x%x to 0x%x\n",
		res->start, res->end);
	dev_dbg(&tzasc->dev, "allocated %d bytes at 0x%x\n",
		resource_size(res), res->start);
	dev_dbg(&tzasc->dev, "adapter name: \"%s\"\n",
		tzasc->name);
	dev_dbg(&tzasc->dev, "initialized\n");

	return 0;
}

static int tzasc_remove(struct platform_device *pdev)
{
	platform_set_drvdata(pdev, NULL);
	dev_info(&pdev->dev, "removed\n");
	return 0;
}

static struct platform_driver tzasc_driver = {
	.driver	= {
		.name		= "tzasc",
		.of_match_table = tzasc_ids,
		.owner		= THIS_MODULE,
	},
	.probe	= tzasc_probe,
	.remove	= tzasc_remove
};
module_platform_driver(tzasc_driver);

MODULE_AUTHOR("Martin Herzog <mherzog@de.adit-jv.com>");
MODULE_DESCRIPTION("TZASC IRQ handler");
MODULE_LICENSE("GPL v2");
MODULE_VERSION("1.00");
