/*
 * Copyright (C) 2013 Robert Bosch Car Multimedia GmbH
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/err.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include "ti_ds90ub92x.h"

/* ds90ub927q serializer registers */
#define GEN_CONFIG 0x03
#define GEN_CONFIG_I2C_PASS	(1 << 3)

#define MODE_SELECT 0x04
#define MODE_SELECT_DE_GATE_RGB (1<<4)
#define MODE_SELECT_BC_OVERRIDE (1<<3)
#define MODE_SELECT_BC (1<<2)
#define MODE_SELECT_LFMODE_OVERRIDE	(1<<1)
#define MODE_SELECT_LFMODE (1<<0)

#define DES_ID	0x06

#define SLAVE_ID 0x07

#define GPIO_MAX_PINS	9
#define GPIO0_CFG 0x0D
#define GPIO_CFG_MASK_LSB	0x0F
#define GPIO_CFG_MASK_MSB	0xF0
#define GPIO_CFG_INPUT_LSB	0x03
#define GPIO_CFG_INPUT_MSB	0x30
#define GPIO_CFG_REMOTE_LSB	0x05
#define GPIO_CFG_REMOTE_MSB	0x50

#define DATAPATH 0x12
#define DATAPATH_18BITS		(1 << 2)

#define GEN_CTRL 0x13
#define GEN_CTRL_MAPSEL_OVERRIDE (1 << 6)
#define GEN_CTRL_MAPSEL_MSB	(1 << 5)

#define SLAVE_ID_1 0x70
#define SLAVE_ALIAS_1 0x77

#define ICR 0xC6
#define ICR_RX_INT (1 << 5)
#define ICR_INT_EN (1 << 0)

#define ISR 0xC7
#define ISR_IS_RX_INT (1 << 5)
#define ISR_IS_INT_EN (1 << 0)

#define DEVICE_ID 0xF0
#define DEVICE_ID_LEN 6

#define UH926_GPIO0_CFG 0x1D

#define OF_NAME_SIZE	32

static struct i2c_client *master;
static int ds90ub927_init_serializer(struct i2c_client *client,
				     struct ds90ub927q_data *pdata);

int lvds_serializer_ack_irq(void)
{
	unsigned int val;
	/* clear interrupt status */
	if (master == NULL)
		return -ENODEV;

	val = i2c_smbus_read_byte_data(master, ISR);
	if (val & (ISR_IS_RX_INT | ISR_IS_INT_EN))
		return 1;
	return 0;
}
EXPORT_SYMBOL_GPL(lvds_serializer_ack_irq);

static int ds90ub927_i2c_init(
		struct i2c_client *client, struct i2c_bridge *pi2c)
{
	int i, ret;

	for (i = 0; i < pi2c->num; i++) {
		if (pi2c->address[i] == 0)
			break;
		ret = i2c_smbus_write_byte_data(client, SLAVE_ID_1 + i,
							pi2c->address[i] << 1);
		if (ret < 0) {
			dev_err(&client->dev, "Write I2C map address fail\n");
			return ret;
		}
		ret = i2c_smbus_write_byte_data(client, SLAVE_ALIAS_1 + i,
							pi2c->alias[i] << 1);
		if (ret < 0) {
			dev_err(&client->dev, "Write I2C map alias fail\n");
			return ret;
		}
	}
	return 0;
}

static int ds90ub92x_gpio_init(
		struct i2c_client *client, struct gpio_bridge *gpio)
{
	int i, reg, val;

	for (i = 0; i < gpio->pin_num; i++) {
		reg = gpio->base + ((i + 1) / 2);
		val = i2c_smbus_read_byte_data(client, reg);
		switch (gpio->gpio_direction[i]) {
		case GPIO_INPUT:
			if ((i & 1)  || (i == 0))
				val = (val & GPIO_CFG_MASK_MSB) |
					GPIO_CFG_INPUT_LSB;
			else
				val = (val & GPIO_CFG_MASK_LSB) |
					GPIO_CFG_INPUT_MSB;
			break;
		case GPIO_OUTPUT_REMOTE:
			if ((i & 1)  || (i == 0))
				val = (val & GPIO_CFG_MASK_MSB) |
					GPIO_CFG_REMOTE_LSB;
			else
				val = (val & GPIO_CFG_MASK_LSB) |
					GPIO_CFG_REMOTE_MSB;
			break;
		default:
			if ((i & 1)  || (i == 0))
				val = (val & GPIO_CFG_MASK_MSB);
			else
				val = (val & GPIO_CFG_MASK_LSB);
			break;
		}
		i2c_smbus_write_byte_data(client, reg, val);
	}
	return 0;
}

static int ds90ub927_video_init(
		struct i2c_client *client, struct video_bridge *pvideo)
{
	int val, reg_val;

	/* set "Video Color Depth Mode" if necessary */
	if (pvideo->bitfmt != VIDEO_BITFMT_INVALID) {
		val = reg_val = i2c_smbus_read_byte_data(client, DATAPATH);

		if (pvideo->bitfmt == VIDEO_BITFMT_18BIT)
			val |= DATAPATH_18BITS;
		else
			val &= ~DATAPATH_18BITS;

		if (val != reg_val)
			i2c_smbus_write_byte_data(client, DATAPATH, val);
	}

	/* set "FPD-Link Map Select" if necessary */
	val = reg_val = i2c_smbus_read_byte_data(client, GEN_CTRL);
	switch (pvideo->bitmsb) {
	case VIDEO_BITMSB_LSB:
		val &= ~GEN_CTRL_MAPSEL_MSB;
		val |= GEN_CTRL_MAPSEL_OVERRIDE;
		break;
	case VIDEO_BITMSB_MSB:
		val |= (GEN_CTRL_MAPSEL_OVERRIDE | GEN_CTRL_MAPSEL_MSB);
		break;
	default:
		val &= ~GEN_CTRL_MAPSEL_OVERRIDE;
	}

	if (val != reg_val)
		i2c_smbus_write_byte_data(client, GEN_CTRL, val);

	return 0;
}

static void ds90ub927_set_bcmode(struct i2c_client *client,
				 struct ds90ub927q_data *pdata)
{
	int val;
	int reg_val = i2c_smbus_read_byte_data(client, MODE_SELECT);

	val = reg_val;

	/* backward compatibility override */
	dev_dbg(&client->dev, "bcmode: %d\n", pdata->bcmode);
	switch (pdata->bcmode) {
	case 0: /* backward compatibility off */
		val |= MODE_SELECT_BC_OVERRIDE;
		val &= ~MODE_SELECT_BC;
		/* Workaround for Alpine 10 inch display.
		 * TI recommended to set "DE Gates RGB" to avoid
		 * display fault. Actual only Alpine display is
		 * without backward compatibility so it is
		 * correct to connect this normally unrelated
		 * configurations, but this may change with
		 * additional supported displays in the future */
		/* set DE Gate RGB */
		val |= MODE_SELECT_DE_GATE_RGB;
		break;
	case 1: /* backward compatibility on */
		val |= (MODE_SELECT_BC_OVERRIDE | MODE_SELECT_BC);
		/* reset DE Gate RGB */
		val &= ~MODE_SELECT_DE_GATE_RGB;
		break;
	default: /* no backward compatibility override */
		val &= ~MODE_SELECT_BC_OVERRIDE;
		/* set DE Gate RGB */
		val |= MODE_SELECT_DE_GATE_RGB;
		break;
	}

	if (val != reg_val)
		i2c_smbus_write_byte_data(client, MODE_SELECT, val);
}

static void ds90ub927_set_lfmode(struct i2c_client *client,
				 struct ds90ub927q_data *pdata)
{
	int val;
	int reg_val = i2c_smbus_read_byte_data(client, MODE_SELECT);

	val = reg_val;

	/* Low frequency mode override */
	dev_dbg(&client->dev, "lfmode: %d\n", pdata->lfmode);
	switch (pdata->lfmode) {
	case 0: /* Low frequency mode off */
		val |= (MODE_SELECT_LFMODE_OVERRIDE | MODE_SELECT_LFMODE);
		break;
	case 1: /* Low frequency mode on */
		val |= MODE_SELECT_LFMODE_OVERRIDE;
		val &= ~MODE_SELECT_LFMODE;
		break;
	default: /* no Low frequency mode override */
		val &= ~MODE_SELECT_LFMODE_OVERRIDE;
		break;
	}

	if (val != reg_val)
		i2c_smbus_write_byte_data(client, MODE_SELECT, val);
}

ssize_t show_bcmode(struct device *dev, struct device_attribute *attr,
		    char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct ds90ub92x_device *dsdev = platform_get_drvdata(pdev);
	struct ds90ub927q_data *dsdata = dsdev->chip_data;

	return scnprintf(buf, PAGE_SIZE, "%d\n", dsdata->bcmode);
}

ssize_t store_bcmode(struct device *dev, struct device_attribute *attr,
		     const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct ds90ub92x_device *dsdev = platform_get_drvdata(pdev);
	struct ds90ub927q_data *dsdata = dsdev->chip_data;
	struct i2c_client *client = to_i2c_client(dev);
	int ret, val;

	ret = kstrtoint(buf, 10, &val);
	if (ret)
		return ret;

	/* valid values: -1, 0, 1 */
	if ((val < -1) || (val > 1)) {
		dev_err(dev,
			"invalid bcmode: %i, use 0 (off), 1 (on) or -1 "
			      "(no override)\n", val);
		return -EINVAL;
	}

	dsdata->bcmode = val;

	dev_dbg(dev, "bcmode: %i\n", dsdata->bcmode);

	ds90ub927_set_bcmode(client, dsdata);

	return count;
}
DEVICE_ATTR(bcmode, S_IRUGO | S_IWUSR, show_bcmode, store_bcmode);

ssize_t show_lfmode(struct device *dev, struct device_attribute *attr,
		    char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct ds90ub92x_device *dsdev = platform_get_drvdata(pdev);
	struct ds90ub927q_data *dsdata = dsdev->chip_data;

	return scnprintf(buf, PAGE_SIZE, "%d\n", dsdata->lfmode);
}

ssize_t store_lfmode(struct device *dev, struct device_attribute *attr,
		     const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct ds90ub92x_device *dsdev = platform_get_drvdata(pdev);
	struct ds90ub927q_data *dsdata = dsdev->chip_data;
	struct i2c_client *client = to_i2c_client(dev);
	int ret, val;

	ret = kstrtoint(buf, 10, &val);
	if (ret)
		return ret;

	/* valid values: -1, 0, 1 */
	if ((val < -1) || (val > 1)) {
		dev_err(dev,
			"invalid lfmode: %i, use 0 (off), 1 (on) or -1 "
			"(no override)\n", val);
		return -EINVAL;
	}

	dsdata->lfmode = val;

	dev_dbg(dev, "lfmode: %i\n", dsdata->lfmode);

	ds90ub927_set_lfmode(client, dsdata);

	return count;
}
DEVICE_ATTR(lfmode, S_IRUGO | S_IWUSR, show_lfmode, store_lfmode);

ssize_t show_reinit(struct device *dev, struct device_attribute *attr,
		    char *buf)
{
	/* This file is only used to trigger a re-init */
	/* No need to read-out */
	return scnprintf(buf,
			 PAGE_SIZE,
			 "Write to this file to trigger a reinit\n");
}

ssize_t store_init_ds90ub927q(struct device *dev, struct device_attribute *attr,
		     const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct ds90ub92x_device *dsdev = platform_get_drvdata(pdev);
	struct ds90ub927q_data *dsdata = dsdev->chip_data;
	struct i2c_client *client = to_i2c_client(dev);
	u32 addr, of_addr;

	of_addr = i2c_smbus_read_byte_data(client, GEN_CONFIG);
	i2c_smbus_write_byte_data(client, GEN_CONFIG,
				of_addr | GEN_CONFIG_I2C_PASS);

	addr = dsdata->des_slave_addr;
	i2c_smbus_write_byte_data(client, SLAVE_ID, addr << 1);

	dev_dbg(dev, "re-init serializer ...\n");
	ds90ub927_init_serializer(client, dsdata);

	dev_info(&client->dev,
		 "store_init_ds90ub927q: Reinit done\n");

	return count;
}
DEVICE_ATTR(reinit_ds90ub927q, S_IRUGO | S_IWUSR,
	    show_reinit,
	    store_init_ds90ub927q);

ssize_t store_init_ds90uh926(struct device *dev, struct device_attribute *attr,
		     const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct ds90ub92x_device *dsdev = platform_get_drvdata(pdev);
	struct ds90uh926q_data *dsdata = dsdev->chip_data;
	struct i2c_client *client = to_i2c_client(dev);

	dev_info(dev, "store_init_ds90uh926: Reinit done\n");
	ds90ub92x_gpio_init(client, &dsdata->gpio);

	return count;
}
DEVICE_ATTR(reinit_ds90uh926, S_IRUGO | S_IWUSR,
	    show_reinit,
	    store_init_ds90uh926);

static int ds90uh926_init(
		struct i2c_client *client, struct ds90uh926q_data *pdata)
{
	int err;

	err = device_create_file(&client->dev, &dev_attr_reinit_ds90uh926);
	if (err) {
		dev_err(&client->dev, "create file reinit failed\n");
		return err;
	}
	ds90ub92x_gpio_init(client, &pdata->gpio);
	return 0;
}

static int ds90ub927_init_serializer(struct i2c_client *client,
				 struct ds90ub927q_data *pdata)
{
	int val;

	ds90ub927_set_bcmode(client, pdata);
	ds90ub927_set_lfmode(client, pdata);

	if (pdata->rx_irq) {
		/* enable interrupt */
		val = i2c_smbus_read_byte_data(client, ICR);
		val |= ICR_RX_INT | ICR_INT_EN;
		i2c_smbus_write_byte_data(client, ICR, val);
	}

	ds90ub927_i2c_init(client, &pdata->i2c);
	ds90ub92x_gpio_init(client, &pdata->gpio);
	ds90ub927_video_init(client, &pdata->video);

	return 0;
}

static int ds90ub927_init(
		struct i2c_client *client, struct ds90ub927q_data *pdata)
{
	int err;

	/* add sysfs files */
	err = device_create_file(&client->dev, &dev_attr_lfmode);
	if (err) {
		dev_err(&client->dev, "create file lfmode failed\n");
		return err;
	}
	err = device_create_file(&client->dev, &dev_attr_bcmode);
	if (err) {
		dev_err(&client->dev, "create file bcmode failed\n");
		return err;
	}
	err = device_create_file(&client->dev, &dev_attr_reinit_ds90ub927q);
	if (err) {
		dev_err(&client->dev, "create file reinit failed\n");
		return err;
	}

	ds90ub927_init_serializer(client, pdata);
	return 0;
}

static int of_video_bridge_probe(
		struct device_node *np, struct video_bridge *vg)
{
	struct property *prop;

	vg->bitfmt = VIDEO_BITFMT_INVALID;
	vg->bitmsb = VIDEO_BITMSB_INVALID;

	prop = of_find_property(np, "24bits", NULL);
	if (prop)
		vg->bitfmt = VIDEO_BITFMT_24BIT;
	prop = of_find_property(np, "18bits", NULL);
	if (prop)
		vg->bitfmt = VIDEO_BITFMT_18BIT;

	prop = of_find_property(np, "msb", NULL);
	if (prop)
		vg->bitmsb = VIDEO_BITMSB_MSB;
	prop = of_find_property(np, "lsb", NULL);
	if (prop)
		vg->bitmsb = VIDEO_BITMSB_LSB;

	return 0;
}

static int of_i2c_bridge_probe(struct device_node *np, struct i2c_bridge *ig)
{
	int i, num;
	struct property *prop;
	const	__be32 *val;

	prop = of_find_property(np, "addr-maps", NULL);
	if (prop == NULL)
		return -EINVAL;

	num = prop->length / sizeof(u32);
	if (num > (2 * TI_MAX_I2C_SLAVES))
		num = 2 * TI_MAX_I2C_SLAVES;

	ig->num = num / 2 ;
	val = prop->value;
	for (i = 0; i < num; i += 2) {
		ig->address[i / 2] = be32_to_cpup(val++);
		ig->alias[i / 2] = be32_to_cpup(val++);
	}
	return 0;
}

static int of_gpio_bridge_probe(struct device_node *np, struct gpio_bridge *gp)
{
	int pin;
	const char *s;
	char buf[OF_NAME_SIZE];

	for (pin = 0; pin < gp->pin_num; pin++) {
		sprintf(buf, "gpio%d", pin);
		s = of_get_property(np, buf, NULL);
		if (s == NULL) {
			gp->gpio_direction[pin] = GPIO_FUNCTION;
			continue;
		}
		switch (s[0]) {
		case 'i':
			if (!strcmp(s + 1, "nput"))
				gp->gpio_direction[pin] = GPIO_INPUT;
			break;
		case 'o':
			if (!strcmp(s + 1, "utput"))
				gp->gpio_direction[pin] = GPIO_OUTPUT_LOCAL;
			break;
		case 'r':
			if (!strcmp(s + 1, "emote"))
				gp->gpio_direction[pin] = GPIO_OUTPUT_REMOTE;
			break;
		default:
			if (!strcmp(s, "function"))
				gp->gpio_direction[pin] = GPIO_FUNCTION;
			break;
		}
	}
	return 0;
}

static int of_deserializer_probe(
		struct device_node *np, struct ds90ub92x_device *pdev)
{
	const __be32 *val;
	u32 addr, of_addr;
	struct i2c_client *clt, *client = pdev->client;
	struct i2c_adapter *adap = client->adapter;
	struct i2c_board_info info;
	struct ds90ub927q_data *pdata = pdev->chip_data;

	addr = i2c_smbus_read_byte_data(client, DES_ID);
	if (addr < 0) {
		dev_err(&client->dev, "read deserializer address fail\n");
		return addr;
	}

	val = of_get_property(np, "addr", NULL);
	if (val) {
		of_addr = be32_to_cpup(val) << 1;
		if (addr != of_addr) {
			dev_warn(&client->dev,
				"deserializer address conflict!\n");
			addr = of_addr;
		}
	}

	addr = (addr & 0xFF) >> 1;

	of_addr = i2c_smbus_read_byte_data(client, GEN_CONFIG);
	i2c_smbus_write_byte_data(client, GEN_CONFIG,
				of_addr | GEN_CONFIG_I2C_PASS);
	i2c_smbus_write_byte_data(client, SLAVE_ID, addr << 1);

	/* Store slave addr of deserializer */
	/* It is required for re-init sequence */
	pdata->des_slave_addr = addr;

	memset(&info, 0, sizeof(struct i2c_board_info));
	info.addr = addr;
	info.of_node = of_node_get(np);
	strlcpy(info.type, np->name, I2C_NAME_SIZE);
	clt = i2c_new_device(adap, &info);
	if (!clt) {
		of_node_put(np);
		dev_err(&client->dev, "create deserializer failed\n");
		return -ENOMEM;
	}
	return 0;
}

static int ds90uh926_probe(
		struct device_node *np, struct ds90ub92x_device *pdev)
{
	struct device_node *cn;
	struct ds90uh926q_data *pdata;

	pdata = kzalloc(sizeof(struct ds90uh926q_data), GFP_KERNEL);
	if (pdata == NULL)
		return -ENOMEM;

	pdev->chip_data = pdata;
	for_each_child_of_node(np, cn) {
		if (!strcmp(cn->name, "gpio-bridge")) {
			pdata->gpio.base = UH926_GPIO0_CFG;
			pdata->gpio.pin_num = GPIO_MAX_PINS;
			of_gpio_bridge_probe(cn, &pdata->gpio);
			continue;
		}
	}
	return 0;
}

static int ds90ub927_probe(
		struct device_node *np, struct ds90ub92x_device *pdev)
{
	struct device_node *cn;
	struct property *prop;
	struct ds90ub927q_data *pdata;

	pdata = kzalloc(sizeof(struct ds90ub927q_data), GFP_KERNEL);
	if (pdata == NULL)
		return -ENOMEM;

	master = pdev->client;

	pdev->chip_data = pdata;

	/* backward compatibility */
	pdata->bcmode = -1; /* preset no override */
	prop = of_find_property(np, "bcmode", NULL);
	if (prop) {
		if (prop->value && !strcmp(prop->value, "enable"))
			pdata->bcmode = 1;
		else
			pdata->bcmode = 0;
	}

	/* low frequency */
	pdata->lfmode = -1; /* preset no override */
	prop = of_find_property(np, "lfmode", NULL);
	if (prop) {
		if (prop->value && !strcmp(prop->value, "enable"))
			pdata->lfmode = 1;
		else
			pdata->lfmode = 0;
	}

	/* use interrupt */
	prop = of_find_property(np, "rx-irq", NULL);
	if (prop) {
		if (prop->value && !strcmp(prop->value, "enable"))
			pdata->rx_irq = 1;
		else
			pdata->rx_irq = 0;
	}

	for_each_child_of_node(np, cn) {
		if (!strcmp(cn->name, "i2c-bridge")) {
			of_i2c_bridge_probe(cn, &pdata->i2c);
			continue;
		}
		if (!strcmp(cn->name, "gpio-bridge")) {
			pdata->gpio.base = GPIO0_CFG;
			pdata->gpio.pin_num = GPIO_MAX_PINS;
			of_gpio_bridge_probe(cn, &pdata->gpio);
			continue;
		}
		if (!strcmp(cn->name, "video-bridge")) {
			of_video_bridge_probe(cn, &pdata->video);
			continue;
		}
		of_deserializer_probe(cn, pdev);
	}
	return 0;
}

static int ds90ub92x_probe(
		struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret = 0;
	int retries = 30;
	struct ds90ub92x_device *pdev;
	struct device_node *np = client->dev.of_node;
	u8	ident[DEVICE_ID_LEN];

	pdev = kzalloc(sizeof(struct ds90ub92x_device), GFP_KERNEL);
	if (!pdev) {
		dev_err(&client->dev, "No enough memory\n");
		return -ENOMEM;
	}

	pdev->client = client;
	i2c_set_clientdata(client, pdev);

	/* get and initialize LVDS Power/Enable GPIO */
	pdev->power_pin = of_get_named_gpio_flags(np, "power-gpios", 0,
						  &pdev->power_active);
	if (pdev->power_pin == -EPROBE_DEFER) {
		ret = -EPROBE_DEFER;
		goto free_mem;
	}

	if (gpio_is_valid(pdev->power_pin)) {
		dev_dbg(&client->dev, "power_pin: %d active %s\n",
			pdev->power_pin, pdev->power_active ? "high" : "low");

		ret = gpio_request_one(pdev->power_pin, GPIOF_DIR_OUT |
			((pdev->power_active == OF_GPIO_ACTIVE_LOW) ?
				GPIOF_INIT_LOW : GPIOF_INIT_HIGH),
			client->name);
		if (ret < 0) {
			dev_err(&client->dev, "request gpio failed!\n");
			goto free_mem;
		}
	}

	/* read device ID and probe device */
	while (retries--) {
		ret = i2c_smbus_read_i2c_block_data(client, DEVICE_ID,
			DEVICE_ID_LEN, ident);

		if (ret >= 0)
			break;
	}

	if (ret < 0) {
		dev_err(&client->dev, "read device id failed!");
		goto gpio_free;
	}

	if (!strncmp(ident, "_UB927", sizeof(ident)) ||
				!strncmp(ident, "_UH927", sizeof(ident))) {
		ret = ds90ub927_probe(np, pdev);
		if (ret == 0)
			ret = ds90ub927_init(client, pdev->chip_data);
	}

	if (!strncmp(ident, "_UB926", sizeof(ident)) ||
				!strncmp(ident, "_UH926", sizeof(ident))) {
		ret = ds90uh926_probe(np, pdev);
		if (ret == 0)
			ret = ds90uh926_init(client, pdev->chip_data);
	}

	if (ret < 0)
		goto gpio_free;

	return 0;

gpio_free:
	if (gpio_is_valid(pdev->power_pin))
		gpio_free(pdev->power_pin);
	pdev->power_pin = -1;

free_mem:
	kfree(pdev->chip_data);
	kfree(pdev);
	return ret;
}

static int ds90ub92x_remove(struct i2c_client *client)
{
	struct ds90ub92x_device *pdev;
	pdev = i2c_get_clientdata(client);
	if (pdev == NULL)
		return 0;

	device_remove_file(&client->dev, &dev_attr_lfmode);
	device_remove_file(&client->dev, &dev_attr_bcmode);

	if (gpio_is_valid(pdev->power_pin)) {
		gpio_set_value(pdev->power_pin, pdev->power_active ? 0 : 1);
		gpio_free(pdev->power_pin);
	}
	pdev->power_pin = -1;
	kfree(pdev->chip_data);
	kfree(pdev);
	return 0;
}

static const struct i2c_device_id ds90ub92x_ids[] = {
	{"ds90ub927q", 0},
	{"ds90uh926q", 0},
	{ },
};

static const struct of_device_id ds90ub92x_dt_ids[] = {
	{.compatible = "ti,ds90ub92x",},
	{ },
};

MODULE_DEVICE_TABLE(i2c, ds90ub92x_dt_ids);

static struct i2c_driver ds90ub92x_driver = {
	.driver = {
		.name = "ds90ub92x",
		.owner = THIS_MODULE,
		.of_match_table = ds90ub92x_dt_ids,
	},
	.probe = ds90ub92x_probe,
	.remove = ds90ub92x_remove,
	.id_table = ds90ub92x_ids,
};

module_i2c_driver(ds90ub92x_driver);

MODULE_AUTHOR("Fred Fan <yefeng.fan2@de.bosch.com>");
MODULE_DESCRIPTION("TI DS90UB92X module");
MODULE_LICENSE("GPL v2");
MODULE_VERSION("0001");
