/*
 * Copyright (C) 2014 Robert Bosch Car Multimedia GmbH
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/kernel.h>
#include <linux/io.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/uaccess.h>

#define PRAM_NAME	"pram"

struct pram_data {
	struct cdev cdev;
	struct class *class;
	struct device *dev;
	struct mutex lock;
	dev_t first;
	void __iomem *virt_base;
	unsigned int size;
};

static int pram_open(struct inode *inode, struct file *f)
{
	int minor;
	struct pram_data *pramdata;

	minor = MINOR(inode->i_rdev);
	pramdata = container_of(inode->i_cdev, struct pram_data, cdev);

	f->private_data = pramdata;

	return 0;
}

static ssize_t pram_read(struct file *f, char __user *buf, size_t len,
			 loff_t *off)
{
	struct pram_data *pramdata = f->private_data;
	int ret;

	if (unlikely(*off >= pramdata->size))
		return 0;
	if ((*off + len) > pramdata->size)
		len = pramdata->size - *off;
	if (unlikely(!len))
		return len;

	mutex_lock(&pramdata->lock);

	/* copy PRAM data to user buffer */
	if (copy_to_user(buf, pramdata->virt_base + *off, len)) {
		dev_err(pramdata->dev, "error: copy to userspace failed\n");
		ret = -EFAULT;
		goto out;
	}

	ret = len;
	*off += len;
out:
	mutex_unlock(&pramdata->lock);

	return ret;
}

static ssize_t pram_write(struct file *f, const char __user *buf, size_t len,
			  loff_t *off)
{
	struct pram_data *pramdata = f->private_data;
	int ret;

	if (unlikely(*off >= pramdata->size))
		return -EFBIG;
	if ((*off + len) > pramdata->size)
		len = pramdata->size - *off;
	if (unlikely(!len))
		return len;

	mutex_lock(&pramdata->lock);

	if (copy_from_user((void *)pramdata->virt_base + *off, buf, len)) {
		dev_err(pramdata->dev, "error: copy from userspace failed\n");
		ret = -EFAULT;
		goto out;
	}
	ret = len;
	*off += len;
out:
	mutex_unlock(&pramdata->lock);

	return ret;
}

static const struct file_operations pram_fops = {
	.owner = THIS_MODULE,
	.open = pram_open,
	.read = pram_read,
	.write = pram_write,
	.llseek = default_llseek,
};

static int pram_probe(struct platform_device *pdev)
{

	struct pram_data *pramdata;
	struct device *dev;
	struct resource *res;
	int ret = 0;

	pramdata = devm_kzalloc(&pdev->dev, sizeof(struct pram_data),
				GFP_KERNEL);
	if (!pramdata) {
		dev_err(&pdev->dev, "error: can't allocate enough memory\n");
		return -ENOMEM;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (res == NULL) {
		dev_err(&pdev->dev, "error: invalid resource\n");
		return -ENODEV;
	}

	pramdata->dev = &pdev->dev;

	pramdata->virt_base = devm_ioremap_resource(&pdev->dev, res);
	if (!pramdata->virt_base) {
		dev_err(&pdev->dev, "error: failed to ioremap\n");
		return -ENOMEM;
	}

	pramdata->size = resource_size(res);

	/* register our character device */
	ret = alloc_chrdev_region(&pramdata->first, 0, 1, PRAM_NAME);
	if (ret) {
		dev_err(&pdev->dev, "error: failed to register chardev\n");
		return ret;
	}

	/* create class and device for udev information */
	pramdata->class = class_create(THIS_MODULE, PRAM_NAME);
	if (IS_ERR(pramdata->class)) {
		dev_err(&pdev->dev, "error: failed to create class\n");
		ret = PTR_ERR(pramdata->class);
		goto out1;
	}

	dev = device_create(pramdata->class, NULL, pramdata->first, NULL,
			    PRAM_NAME);
	if (IS_ERR(dev)) {
		dev_err(&pdev->dev, "error: failed to create pram device\n");
		ret = PTR_ERR(dev);
		goto out2;
	}

	cdev_init(&pramdata->cdev, &pram_fops);
	pramdata->cdev.owner = THIS_MODULE;
	ret = cdev_add(&pramdata->cdev, pramdata->first, 1);
	if (ret) {
		dev_err(&pdev->dev, "error: failed to add cdev\n");
		goto out3;
	}

	mutex_init(&pramdata->lock);

	platform_set_drvdata(pdev, pramdata);

	return 0;

out3:
	device_destroy(pramdata->class, pramdata->first);
out2:
	class_destroy(pramdata->class);
out1:
	unregister_chrdev_region(pramdata->first, 1);
	return ret;
}

static int pram_remove(struct platform_device *pdev)
{
	struct pram_data *pramdata = platform_get_drvdata(pdev);

	cdev_del(&pramdata->cdev);
	device_destroy(pramdata->class, pramdata->first);
	unregister_chrdev_region(pramdata->first, 1);
	class_destroy(pramdata->class);

	return 0;
}

static struct of_device_id pram_of_device_ids[] = {
	{.compatible = "Bosch,pram"},
	{},
};
MODULE_DEVICE_TABLE(of, pram_of_device_ids);

static struct platform_driver pram_driver = {
	.driver = {
		.name = PRAM_NAME,
		.owner = THIS_MODULE,
		.of_match_table = pram_of_device_ids,
	},
	.probe = pram_probe,
	.remove = pram_remove
};

static int __init pram_init(void)
{
	return platform_driver_register(&pram_driver);
}

static void __exit pram_exit(void)
{
	platform_driver_unregister(&pram_driver);
}

module_init(pram_init);
module_exit(pram_exit);

MODULE_AUTHOR("Robert Bosch Car Multimedia GmbH, Dirk Behme");
MODULE_DESCRIPTION("PRAM driver");
MODULE_LICENSE("GPL");
