/*
 * Module for Reading I2S data from SSI3 channel in MASCA
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
#include <linux/module.h>
#include "masca_trace.h"
#include "masca_ssi.h"
#include "masca_dma.h"

#define READ32(x) ioread32(ssiptr+x);  /*Read value from register*/

static void *ssiptr;
static bool gssi_on;

/*Writes a mask to a SSI register*/
static void writemask32(int mask, int addr, int flag)
{
	unsigned int value = READ32(addr);
	if (flag == 1) /*flag=1:Bit Set*/
		iowrite32(value | mask, ssiptr+addr);
	else if (flag == 0) /*flag=0:Bit Set*/
		iowrite32(value & (~mask), ssiptr+addr);
}

void masca_ssi_on(void)
{
	if (!gssi_on) {
		gssi_on = true;
		writemask32(SSI_SCR_SSIEN, SSI_SCR, 1);
		masca_dma_start();
		SSI_TD_TRACE("ssi is on\n");
	}
}
EXPORT_SYMBOL(masca_ssi_on);

void masca_ssi_off(void)
{
	if (gssi_on) {
		gssi_on = false;
		masca_dma_stop();
		writemask32(SSI_SCR_SSIEN, SSI_SCR, 0);
		SSI_TD_TRACE("ssi is off\n");
	}
}
EXPORT_SYMBOL(masca_ssi_off);

bool masca_ssi_is_on(void)
{
	return gssi_on;
}
EXPORT_SYMBOL(masca_ssi_is_on);

static int __init masca_ssi_init(void)
{
	unsigned int ret = 0;

	/*Map the Clock Controller Module(CCM)registers*/
	ssiptr = ioremap(CCM_BASE_ADDR, CCM_CCGR5_OFFET);

	/*Writing to CCM_CCGR5 bits(23-22)-> ssi3_clk_enable*/
	writemask32(SSI3_CLK_EN, CCM_CCGR5_OFFET, 1);

	iounmap(ssiptr);

	/*SSI3 register configuration*/
	/*Map the SSI3 register address*/
	ssiptr = ioremap(SSI3_BASE_ADDR, SSI3_OFFSET);

	/*SSI Control Register(SCR) Configuration*/
	writemask32(SSI_SCR_SYN | SSI_SCR_I2S_MODE_SLAVE, SSI_SCR, 1);

	/*SSI Interrupt Enable Register(SIER) Configuration*/
	writemask32(SSI_SIER_RFF0_EN | SSI_SIER_RIE, SSI_SIER, 0);
	writemask32(SSI_SIER_RDMAE, SSI_SIER, 1);

	/*SSI Receive Configuration Register(SRCR) Configuration*/
	writemask32(SSI_SRCR_RFEN0 | SSI_SRCR_RXBIT0 | SSI_SRCR_RSCKP,
							SSI_SRCR, 1);
	writemask32(SSI_STCR_TXBIT0 | SSI_STCR_TSCKP, SSI_STCR, 1);

	/*SSI Receive Clock Control Register(SRCCR) Configuration*/
	/*WL3-WL0(16-13)=0111 DC=00001*/ /*16 Bits/Word 2 channels */
	writemask32(0xE100, SSI_SRCCR, 1);
	/* we need STCCR for common RX TX clock (SSI_SCR_SYN) */
	writemask32(0xE100, SSI_STCCR, 1);

	/*SSI FIFO Control/Status Register(SFCSR) Configuration*/
	writemask32(0x80, SSI_SFCSR, 1); /*RFWM0[3:0]=1000*/

	/*SSI Control Register(SCR) Configuration*/
	writemask32(SSI_SCR_RE, SSI_SCR, 1);
	ret = masca_dma_init();
	SSI_TD_TRACE("masca_ssi_init ret:%d!!\n", ret);
	return ret;
}

static void __exit masca_ssi_exit(void)
{
	writemask32(SSI_SCR_SSIEN, SSI_SCR, 0); /*Disable SSI on exit*/
	masca_dma_stop();
	masca_dma_deinit();
	iounmap(ssiptr);
}

module_init(masca_ssi_init);
module_exit(masca_ssi_exit);

MODULE_DESCRIPTION("GEN3 MASCA SSI DRIVER");
MODULE_LICENSE("GPL v2");
