/*
 * MASCA State Machine File
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_helper.h"
#include "masca_drv_mngr.h"
#include "masca_sm.h"
#include "masca_event_config.h"
#include "masca_scsi_handler.h"
#include "masca_blkdev.h"

enum masca_cd_load_state {
	TOC_READ_INITIATE,
	TOC_READ_WAIT,
	SESSION_INFO_INITIATE,
	SESSION_INFO_WAIT,
	CD_TEXT_INITIATE,
	CD_TEXT_WAIT,
	COMPUTE_HASH_ID_ROM,
	LOAD_COMPLETE,
	LOAD_FAILED
};

enum masca_hash_state {
	ROM_HASH_START,
	ROM_HASH_READ,
	ROM_HASH_WAIT,
	ROM_HASH_COMPLETE,
	ROM_HASH_FAILED
};

#define DATA_TRACK_BIT		0x40
#define HASH_START_SECTOR	16
#define HASH_END_SECTOR		19
/* Offset value of CD DATA START address */
#define CD_DATA_START_OFFSET	150

/* Read subchannel related */
#define CD_CURRENT_POS		0x01
#define SUB_Q_BIT_SET		0x40
#define GET_MSF_DATA		0x02
#define Q_SUBCHANNEL_CONTROL	0x11
#define SUBCHANNEL_RESP_SIZE	0x0c
#define PLAY_IN_PROGRESS	0x11
#define PLAY_OPR_PAUSED		0x12
#define PLAY_OPR_COMPLETED	0x13
#define PLAY_OPR_STOPPED_ERR	0x14
#define PLAY_NO_AUDIO_STATUS	0x15
#define BCD_DATA_LENGTH		0x0a

static enum masca_sm_voltage_state	voltage_state;
static enum masca_sm_pwr_state		power_status;
static enum masca_last_sess last_sess_status;
static enum masca_drive_status drive_status = DRV_MSG_MAX;
static enum masca_play_mode play_mode = PLAY_MODE_NON_CONTINUOUS;

/*This temperature state is as informed by the
hardware based on its own specification*/
static enum masca_sm_temperature_state	hw_temp_state;

struct masca_cd_toc			cd_toc;
static enum masca_sm_drive_state	drive_state;
static enum masca_sm_module_state	driver_state;
static enum masca_sm_disc_state		disc_state;
static enum masca_cd_load_state		load_state;
static struct masca_cd_text		cd_text;
static struct masca_session_info	session_info;
static struct masca_session_info	cdrom_session;
static unsigned int			hash_cd;
static unsigned int			present_hash_id;
enum masca_media_type			type_of_disc;
enum masca_media_type			present_cd_type;
static enum masca_hash_state		rom_hash_read_state;
static unsigned char			hash_sector_buf[2048];
static unsigned int			sectors_to_read;
static bool				is_disc_load_allowed;
/*status indicating signalling of bad media*/
static bool			is_bad_media_signalled;
static char user_media_evt = 0x00;
/* Power event field */
atomic_t drv_pwr_state = ATOMIC_INIT(0);
static char fwrk_media_evt	= 0x00;

/*media status according to ATAPI specifications*/
enum masca_media_status	media_status;
static enum masca_ign_aud_status curr_ign_status;
static bool curr_ign_cmd_bit;

char usr_pause_cmd;

static void masca_load_cd_info(
		const struct masca_drv_response_info * const p_response_info,
		enum masca_media_type * const p_current_cd_type,
		unsigned int * const p_current_hash_id);
static void masca_exec_drive_sm(const struct masca_drv_response_info *
			const p_response_info,
			const struct masca_drv_event_info * const p_drv_event,
			struct masca_event_info * const p_evt_info);

static
void masca_exec_disc_sm(const struct masca_drv_event_info * const p_drv_event,
			struct masca_event_info * const p_evt_info);

static enum masca_error masca_snd_cmd_drv(const enum masca_cmd command_to_send);

static void masca_compute_media_type(
				enum masca_media_type * const p_current_cd_type,
				unsigned int * const p_current_hash_id);
static void masca_compute_hash_id_cdrom(unsigned int * const p_current_hash_id,
					const struct masca_drv_response_info *
					const p_resp_info);
static void masca_updt_state_machines(
		const struct masca_drv_response_info * const p_response_info,
		const struct masca_drv_event_info * const p_drv_event,
		struct masca_event_info * const p_evt_info);

static void masca_disable_disc_load(void);
static void masca_enable_disc_load(void);

void masca_sm_init(const struct masca_configuration * const p_config)
{
	load_state = TOC_READ_INITIATE;
	rom_hash_read_state = ROM_HASH_START;
	voltage_state = NORMAL_VOLTAGE;
	hw_temp_state = NORMAL_TEMPERATURE;
	drive_state = SM_NO_CD;
	driver_state = NORMAL;
	disc_state = SM_MAX_DISC_STATE;
	load_state = TOC_READ_INITIATE;
	hash_cd = 0;
	type_of_disc = DISC_MAX;
	is_disc_load_allowed = TRUE;
	is_bad_media_signalled = FALSE;
	present_hash_id = 0;
	present_cd_type = DISC_MAX;
	media_status = MASCA_NO_MEDIA;
	curr_ign_status = IGN_AUD_DISABLE;
	curr_ign_cmd_bit = FALSE;
}

void masca_update_drive_temperature(void)
{
	enum masca_error ret_err =  masca_check_async_sense();

	hw_temp_state = NORMAL_TEMPERATURE;
	if (ret_err == MASCA_SE_OVR_TEMP)
		hw_temp_state = OVER_TEMPERATURE;
}

enum masca_error masca_sm_cmd_reply(
		const struct masca_drv_response_info * const p_response_info,
		struct masca_event_info * const p_evt_info)
{
	short temperature = 0;
	p_evt_info->evt = NO_EVENT;

	if (DIAG_GET_TEMPERATURE == p_response_info->drv_cmd_response) {
		if ((MASCA_OK == p_response_info->response_error) ||
		 (MASCA_PROCESSED == p_response_info->response_error)) {
			temperature = (short)
			(p_response_info->response_param.cd_diag_params.
				temperature[0] << BYTE_2_START) |
			(p_response_info->response_param.cd_diag_params.
				temperature[1]);
		} else {
			SM_TD_TRACE
			("masca_sm_cmd_reply:0x%x\n",
					p_response_info->response_error);
		}
	}
	masca_updt_state_machines(p_response_info, NULL, p_evt_info);

	return MASCA_OK;
}


enum masca_error masca_sm_event(const unsigned int evt_pattern,
			const struct masca_drv_event_info * const p_drv_event,
			struct masca_event_info * const p_evt_info)
{
	enum masca_error err = MASCA_PTR_NULL;
	enum masca_sm_module_state new_driver_state = driver_state;
	enum masca_sm_voltage_state new_voltage_state = voltage_state;

	if ((evt_pattern & SM_EVENT_SUSPEND) == SM_EVENT_SUSPEND)
		new_driver_state = SUSPENDED;
	else if ((evt_pattern & SM_EVENT_RESUME) == SM_EVENT_RESUME)
		new_driver_state = NORMAL;

	p_evt_info->evt = NO_EVENT;
	err = MASCA_OK;
	if (driver_state != new_driver_state) {
		driver_state = new_driver_state;
		if (SUSPENDED == driver_state)
			/*suspend the driver*/
			/*put the drive to sleep mode*/
			masca_drive_mngr_sleep();
		else
			masca_drive_mngr_wakeup();
	} else if (NULL != p_drv_event) {
		if (p_drv_event->drv_event != DRV_NO_EVENT) {
			if (DRV_UNDER_VOLT == p_drv_event->drv_event)
				new_voltage_state = UNDER_VOLTAGE;
			else if (DRV_NORMAL_VOLT == p_drv_event->drv_event)
				new_voltage_state = NORMAL_VOLTAGE;
		}
	}

	/*we never get both voltage and temperature state
	change at the same point of time*/
	if (new_voltage_state != voltage_state) {
		voltage_state = new_voltage_state;
		if (UNDER_VOLTAGE == new_voltage_state)
			p_evt_info->evt = DRIVE_UNDER_VOLT;
		else if (NORMAL_VOLTAGE == new_voltage_state)
			p_evt_info->evt = DRIVE_NORMAL_VOLT;

		p_evt_info->event_info.cd_identify.cd_type = present_cd_type;
		p_evt_info->event_info.cd_identify.unique_id = present_hash_id;
	}
	masca_updt_state_machines(NULL, p_drv_event, p_evt_info);

	return err;
}

/* Power status field */
void masca_sm_update_power_status(char pwr_status)
{
	power_status = pwr_status;
}

/* Last session readability status field
 * Set Last session readability status field */
void masca_sm_update_last_sess_state(enum masca_last_sess status)
{
	last_sess_status = status;
}

enum masca_last_sess masca_sm_get_last_sess_state(void)
{
	return last_sess_status;
}

/* Drive status */
void masca_update_drive_status(enum masca_drive_status stat)
{
	drive_status = stat;
}

enum masca_drive_status masca_get_drive_status(void)
{
	return drive_status;
}

/* play mode */
void masca_update_play_mode(enum masca_play_mode mode)
{
	play_mode = mode;
}

enum masca_play_mode masca_get_play_mode(void)
{
	return play_mode;
}

void masca_sm_get_state(struct masca_sm_state * const p_sm_state)
{
	p_sm_state->sm_disc_state = disc_state;
	p_sm_state->sm_drive_state = drive_state;
	p_sm_state->sm_driver_state = driver_state;
	p_sm_state->sm_temp_state = hw_temp_state;
	p_sm_state->sm_volt_state = voltage_state;
	p_sm_state->sm_pwr_state = power_status;
}

enum masca_error
masca_sm_get_session_info(struct masca_session_info **const p_info_session)
{
	enum masca_error err = MASCA_INVALID_STATE;
	(void)memset(&cdrom_session, 0, sizeof(struct masca_session_info));

	if (SM_ACCESSIBLE == drive_state) {
		err = MASCA_OK;
		if (present_cd_type == DISC_CDROM) {
			memcpy(&cdrom_session, &session_info,
				sizeof(struct masca_session_info));
			cdrom_session.last_sess_trk_info
				.track_start_lba = session_info
					.last_sess_trk_info
					.track_start_lba
						- CD_DATA_START_OFFSET;
			*p_info_session = &cdrom_session;
		} else {
			*p_info_session = &session_info;
		}
	}
	return err;
}

enum masca_error masca_sm_get_toc(struct masca_cd_toc ** const p_info_toc)
{
	enum masca_error err = MASCA_INVALID_STATE;
	if (SM_ACCESSIBLE == drive_state) {
		err = MASCA_OK;
		*p_info_toc = &cd_toc;
	}
	return err;
}

enum masca_error masca_sm_get_cd_text(struct masca_cd_text ** const p_info_text)
{
	enum masca_error err = MASCA_INVALID_STATE;
	if (SM_ACCESSIBLE == drive_state) {
		err = MASCA_OK;
		*p_info_text = &cd_text;
	}
	return err;
}

void masca_sm_get_media_type(enum masca_media_type * const p_mda_type)
{
	if (SM_ACCESSIBLE == drive_state)
		*p_mda_type = present_cd_type;
}

enum masca_error masca_sm_get_cd_capacity(unsigned int * const p_cd_capacity)
{
	enum masca_error err = MASCA_INVALID_STATE;
	if (SM_ACCESSIBLE == drive_state) {
		err = MASCA_OK;
		*p_cd_capacity = GET_LBA(cd_toc.cd_max_min,
						cd_toc.cd_max_sec, 0);
	}
	return err;
}

/* curr_ign_status used to maintain the current status of ignore audio bit
 * of CD drive */
void masca_sm_set_ign_aud_status(enum masca_ign_aud_status ign_status)
{
	curr_ign_status = ign_status;
}

enum masca_ign_aud_status masca_sm_get_ign_aud_status(void)
{
	return curr_ign_status;
}

/* curr_ign_cmd_bit used to enable/disable ignore audio bit on every status
 * command to CD drive depend on the last command */
void masca_set_last_cmd_ign_bit(bool cmd_ign_bit)
{
	curr_ign_cmd_bit = cmd_ign_bit;
}

bool masca_get_last_cmd_ign_bit(void)
{
	return curr_ign_cmd_bit;
}

/* masca_drive_send_ignore_audio_cmd function will send ignore audio
* command to switch between audio track and data track on mixed mode CD */
enum masca_error masca_drive_send_ignore_audio_cmd(bool ign_audio)
{
	enum masca_error err = MASCA_BUSY;

	if ((curr_ign_status == IGN_AUD_DISABLE_TOC_READY) &&
			(ign_audio == TRUE)) {
		err = masca_snd_cmd_drv(EN_IGN_AUDIO);
	} else if ((curr_ign_status == IGN_AUD_ENABLE_TOC_READY) &&
			(ign_audio == FALSE)) {
		err = masca_snd_cmd_drv(DIS_IGN_AUDIO);
	}

	return err;

}

enum masca_media_status masca_sm_get_media_status(void)
{
	enum masca_media_status ret_media_stat = media_status;

	if ((MASCA_MEDIA_INSERTED == media_status) ||
			(MASCA_MEDIA_EJECT_PROCESSED == media_status))
		media_status = MASCA_NO_UPDATE;

	return ret_media_stat;
}

void masca_conv_readsub_data_bcd(unsigned char *val, unsigned char len)
{
	unsigned char cnt = 0;

	while (cnt < len) {
		/* Convert hex value to decimal */
		val[cnt] = ((val[cnt] >> 4) * 16) + (val[cnt] & (0x0f));
		/* convert Decimal value to BCD value */
		val[cnt] = (val[cnt]/10*16) + (val[cnt]%10);
		cnt++;
	}
}

enum masca_error masca_read_subchannel(
				const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				const unsigned int buf_length)
{
	int play_addr = 0;
	struct masca_sm_state sm_state;
	enum masca_error err = MASCA_IR_PARAM_NOT_SUPP;
	struct masca_read_subchannel_cmd_resp sb_resp;
	unsigned char *cmd = p_output->cmdbuf;
	memset(&(sb_resp), 0, sizeof(struct masca_read_subchannel_cmd_resp));
	if ((cmd[READSUBCH_REQ_SUBQ] & SUB_Q_BIT_SET) == SUB_Q_BIT_SET) {
		/* Get current state of state machine */
		if ((cmd[READSUBCH_REQ_SUBCH_LIST] & CD_CURRENT_POS) ==
							CD_CURRENT_POS) {
			masca_sm_get_state(&sm_state);
		/* Audio Status Codes will Take from State Machine */
		if (out_event_global.evt == CD_DEFECTIVE) {
			sb_resp.subchannel_header[READSUBCH_HDR_AUD_STAT] =
							PLAY_OPR_STOPPED_ERR;
			out_event_global.evt = NO_EVENT;
		} else if ((out_event_global.evt == CD_PLAY_COMPLETE)) {
			sb_resp.subchannel_header[READSUBCH_HDR_AUD_STAT] =
							PLAY_OPR_COMPLETED;
		} else if ((out_event_global.evt == CD_PLAY_INFO) &&
			(sm_state.sm_disc_state == SM_PLAYING)) {
			sb_resp.subchannel_header[READSUBCH_HDR_AUD_STAT] =
							PLAY_IN_PROGRESS;
		} else if ((sm_state.sm_disc_state == SM_PAUSED) ||
			(sm_state.sm_disc_state == SM_STOPPED)) {
			if (usr_pause_cmd & USR_PAUSE_CMD)
				sb_resp.
				subchannel_header[READSUBCH_HDR_AUD_STAT] =
								PLAY_OPR_PAUSED;
			else
				sb_resp.
				subchannel_header[READSUBCH_HDR_AUD_STAT] =
							PLAY_NO_AUDIO_STATUS;
		} else {
			sb_resp.subchannel_header[READSUBCH_HDR_AUD_STAT] =
							PLAY_NO_AUDIO_STATUS;
		}

		/*12 Bytes of Subchannel Data to follow, Time=0*/
		sb_resp.subchannel_header[READSUBCH_HDR_SUBCH_DATLEN+0x01] =
							SUBCHANNEL_RESP_SIZE;
		/*Sub Channel Data Format Code 0x01*/
		sb_resp.subchannel_data[READSUBCH_RESP_SUBCH_CODE] =
							CD_CURRENT_POS;
		/*ADR Q Sub channel field = 1h,control field*/
		sb_resp.subchannel_data[READSUBCH_RESP_ADR_CTRL] =
							Q_SUBCHANNEL_CONTROL;
		sb_resp.subchannel_data[READSUBCH_RESP_TRK_NUM] =
			out_event_global.event_info.play_info.trk_playing;
		sb_resp.subchannel_data[READSUBCH_RESP_INDX_NUM] =
			out_event_global.event_info.play_info.index;
		masca_conv_readsub_data_bcd((unsigned char *)
				&sb_resp.subchannel_data
					[READSUBCH_RESP_TRK_NUM],
					READSUBCH_RESP_TRK_NUM);
		/*Time bit = 0*/
		if ((cmd[READSUBCH_REQ_TIME] & GET_MSF_DATA) == false) {
			play_addr = GET_LBA(out_event_global.event_info.
							play_info.abs_min,
			/* Absolute CD address */
			out_event_global.event_info.play_info.abs_sec, 0);
			sb_resp.subchannel_data[READSUBCH_RESP_ABS_TIME+0x00] =
						(play_addr>>BYTE_4_START);
			sb_resp.subchannel_data[READSUBCH_RESP_ABS_TIME+0x01] =
						(play_addr>>BYTE_3_START);
			sb_resp.subchannel_data[READSUBCH_RESP_ABS_TIME+0x02] =
						(play_addr>>BYTE_2_START);
			sb_resp.subchannel_data[READSUBCH_RESP_ABS_TIME+0x03] =
								play_addr;
			play_addr = GET_LBA(out_event_global.event_info.
							play_info.rel_min,
			/* Track rel. CD address */
			out_event_global.event_info.play_info.rel_sec, 0);
			sb_resp.subchannel_data[READSUBCH_RESP_RELAT_TIME+0x00]
						= (play_addr>>BYTE_4_START);
			sb_resp.subchannel_data[READSUBCH_RESP_RELAT_TIME+0x01]
						= (play_addr>>BYTE_3_START);
			sb_resp.subchannel_data[READSUBCH_RESP_RELAT_TIME+0x02]
						= (play_addr>>BYTE_2_START);
			sb_resp.subchannel_data[READSUBCH_RESP_RELAT_TIME+0x03]
						= play_addr;
		} else { /*Time bit = 1*/
			/* Absolute CD address */
			sb_resp.subchannel_data[READSUBCH_RESP_ABS_TIME+0x01]
						= out_event_global.
					event_info.play_info.abs_min;
			sb_resp.subchannel_data[READSUBCH_RESP_ABS_TIME+0x02]
						= out_event_global.
					event_info.play_info.abs_sec;
			/* Track rel. CD address */
			sb_resp.subchannel_data[READSUBCH_RESP_RELAT_TIME+0x01]
						= out_event_global.
					event_info.play_info.rel_min;
			sb_resp.subchannel_data[READSUBCH_RESP_RELAT_TIME+0x02]
						= out_event_global.
					event_info.play_info.rel_sec;
			/* As per MMC3 Spec fields like TNO, INDEX, HMSF,
			 * AHMSF should be in BCD format.*/
			masca_conv_readsub_data_bcd((unsigned char *)
			&sb_resp.subchannel_data[READSUBCH_RESP_ABS_TIME],
				READSUBCH_RESP_RELAT_TIME);
		}
		}
		err = MASCA_OK;
	}
	memcpy(p_buffer, &(sb_resp), buf_length);

	return err;
}

/*EVENT NOTIFICATION IMPLEMENTATION*/
void masca_update_evt_stat(char evnt_stat)
{
	user_media_evt = evnt_stat;
	if ((evnt_stat == MASCA_NEW_MEDIA) || (evnt_stat == MASCA_EJECT_REQ))
		fwrk_media_evt = evnt_stat;
}

void masca_update_power_evt(char pwr_stat)
{
	atomic_set(&drv_pwr_state, pwr_stat);
}

#define VOLATGE_RES			0x02
#define MEDIA_RES			0x04
#define VOLT_REQ			0x04
#define MEDIA_REQ			0x10
#define DATA_LEN			0x04
#define POLL_MODE			0x01
#define NO_MEDIA_TRAY_CLOSE	0x00
#define NO_MEDIA_TRAY_OPEN	0x01
#define MEDIA_TRAY_CLOSE	0x02
#define MEDIA_TRAY_OPEN		0x03
#define APPLICATION_REQ		0x80

void masca_get_evt_reply(unsigned char event_stat,
		struct get_evt_status *evnt_resp, bool is_user)
{
	switch (event_stat) {
	/* MediaChanged - Insert in progress */
	case MASCA_MEDIA_CHANGED:
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_EVT] =
							MASCA_MEDIA_CHANGED;
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_STAT] =
							NO_MEDIA_TRAY_OPEN;
		break;
	/* MediaChanged - Inserted */
	case MASCA_MEDCHG_INSERTED:
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_EVT] =
							MASCA_MEDIA_CHANGED;
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_STAT] =
							MEDIA_TRAY_CLOSE;
		break;
	/* NewMedia - CD in play position/Accessible */
	case MASCA_NEW_MEDIA:
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_EVT] =
							MASCA_NEW_MEDIA;
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_STAT] =
							MEDIA_TRAY_CLOSE;
		if (is_user == FALSE)
			fwrk_media_evt = MASCA_CD_PREVINSERT;
		break;
	/* EjectRequest - Eject in progress */
	case MASCA_EJECT_REQ:
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_EVT] =
							MASCA_EJECT_REQ;
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_STAT] =
							MEDIA_TRAY_OPEN;
		if (is_user == FALSE)
			fwrk_media_evt = MASCA_CD_PREVEJECT;
		break;
	/* MediaRemoval - CD is in slot */
	case MASCA_MEDIA_REM_SLT:
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_EVT] =
							MASCA_MEDIA_REMOVAL;
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_STAT] =
							MEDIA_TRAY_OPEN;
		break;
	/* MediaRemoval - CD is ejected */
	case MASCA_MEDIA_REMOVAL:
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_EVT] =
							MASCA_MEDIA_REMOVAL;
		evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_STAT] =
							NO_MEDIA_TRAY_CLOSE;
		masca_update_evt_stat(MASCA_CD_PREVEJECT);
		break;
	/* Nochg */
	default:
		if ((MASCA_CD_PREVINSERT == event_stat)
						&& (is_user == FALSE)) {
			evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_EVT] =
							MASCA_NO_CHANGE;
			evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_STAT] =
							MEDIA_TRAY_CLOSE;

		} else if (MASCA_CD_PREVEJECT == event_stat) {
			evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_EVT] =
							MASCA_NO_CHANGE;
			evnt_resp->evt_data[GETEVT_MEDIA_RESP_MEDIA_STAT] =
							NO_MEDIA_TRAY_CLOSE;
		}
		break;
	}
}

enum masca_error masca_get_evt_notification(
				const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				const unsigned int buf_length)
{
	enum masca_error err = MASCA_OK;
	char *cmd = p_output->cmdbuf;
	unsigned char evnt_stat;
	bool is_userapp = FALSE;
	char pwr_evt;
	struct get_evt_status evt_resp;

	memset(&evt_resp, 0, sizeof(struct get_evt_status));
	err = masca_check_async_sense();
	/*Preparing Event Header */
	evt_resp.evt_header[GETEVT_HDR_DATA_LEN+0x00] = 0x00;
	evt_resp.evt_header[GETEVT_HDR_DATA_LEN+0x01] = DATA_LEN;
	evt_resp.evt_header[GETEVT_HDR_NEA_SUPP_CLASS] = MEDIA_REQ | VOLT_REQ;

	if ((cmd[GETEVT_REQ_IMMED] & POLL_MODE) == POLL_MODE) {
		/* Response for power states */
		if ((cmd[GETEVT_REQ_NOTY_REQ] & VOLT_REQ) == VOLT_REQ) {
			evt_resp.evt_header[GETEVT_HDR_NEA_NOTY_CLASS] =
								VOLATGE_RES;
			pwr_evt = atomic_read(&drv_pwr_state);
			evt_resp.evt_data[GETEVT_POWR_RESP_PWR_EVT] = pwr_evt;
			evt_resp.evt_data[GETEVT_POWR_RESP_PWR_STAT] =
								power_status;

			if (pwr_evt != MASCA_NO_CHANGE)
				masca_update_power_evt(MASCA_NO_CHANGE);

		} else if ((cmd[GETEVT_REQ_NOTY_REQ] &
						MEDIA_REQ) == MEDIA_REQ) {
			/* Response for Media status */
			evt_resp.evt_header[GETEVT_HDR_NEA_NOTY_CLASS] =
								MEDIA_RES;
			evnt_stat = fwrk_media_evt;
			if ((APPLICATION_REQ & cmd[GETEVT_REQ_CTRL]) ==
							APPLICATION_REQ) {
				is_userapp = TRUE;
				evnt_stat = user_media_evt;
			}
			masca_get_evt_reply(evnt_stat, &evt_resp, is_userapp);
		} else {
			/* Other notification classes not supported */
			if (MASCA_OK == err)
				err = MASCA_IR_PARAM_NOT_SUPP;
		}
	} else {
		/* Asyncronous mode not supporetd,Error Handling
		 * need to be done */
		if (MASCA_OK == err)
			err = MASCA_IR_PARAM_NOT_SUPP;
	}
	memcpy(p_buffer, &(evt_resp), buf_length);

	return err;
}

/*===========================================================================*/
/*===================static FUNCTIONS=========================================*/
/*===========================================================================*/
static void masca_exec_drive_sm_load(
		const struct masca_drv_response_info * const p_response_info,
		struct masca_event_info * const p_evt_info,
		enum masca_media_type * const current_cd_type,
		unsigned int * const current_hash_id)
{
	struct masca_cd_identification *cd_identify =
		&(p_evt_info->event_info.cd_identify);

	*current_hash_id = 0;
	*current_cd_type = DISC_BAD_MEDIA;
	/* get the toc, session and cd text
	 * information from drive manager*/
	if (TRUE == is_disc_load_allowed) {
		if (LOAD_COMPLETE == load_state) {
			/* Now we are inserting the same CD for which mount
			 * event is already given.So now provide
			 * an auto re-insert event */
			p_evt_info->evt =  CD_ACCESSIBLE;
			cd_identify->cd_type = type_of_disc;
			cd_identify->unique_id = hash_cd;
			cd_identify->same_as_previous = TRUE;
			cd_identify->auto_reinsert = TRUE;
			drive_state = SM_ACCESSIBLE;
			disc_state = SM_STOPPED;
		} else {
			masca_load_cd_info(p_response_info, current_cd_type,
			current_hash_id);
			if (LOAD_COMPLETE == load_state) {
				p_evt_info->evt = CD_ACCESSIBLE;
				cd_identify->cd_type = *current_cd_type;
				cd_identify->unique_id = *current_hash_id;
				cd_identify->same_as_previous = FALSE;
				cd_identify->auto_reinsert = FALSE;
				/* Now we have read all necessary info from CD*/
				if ((*current_cd_type == type_of_disc) &&
						(hash_cd == *current_hash_id)) {
					/* provide an information that same CD
					 * is inserted again*/
					cd_identify->same_as_previous = TRUE;
				}
				type_of_disc = *current_cd_type;
				hash_cd = *current_hash_id;
				drive_state = SM_ACCESSIBLE;
			} else if (LOAD_FAILED == load_state) {
				if (FALSE == is_bad_media_signalled) {
					is_bad_media_signalled = TRUE;
					present_cd_type = DISC_BAD_MEDIA;
					present_hash_id = 0;
					p_evt_info->evt = CD_LOAD_ERROR;
					cd_identify->cd_type = DISC_BAD_MEDIA;
				}
			}
		}
	} else {
		SM_TD_TRACE("disc loading not allowed %d\n",
				is_disc_load_allowed);
		/*disc loading not allowed*/
	}
}

static void masca_exec_drive_sm(
		const struct masca_drv_response_info * const p_response_info,
		const struct masca_drv_event_info * const p_drv_event,
		struct masca_event_info * const p_evt_info)
{
	enum masca_drv_event drv_mngr_event = DRV_NO_EVENT;
	enum masca_media_type	current_cd_type;
	unsigned int		current_hash_id;
	bool			is_imm_transition = TRUE;

	if (p_drv_event != NULL)
		drv_mngr_event = p_drv_event->drv_event;

	p_evt_info->evt = NO_EVENT;
	while (TRUE == is_imm_transition) {
		is_imm_transition = FALSE;
		switch (drive_state) {
		case SM_NO_CD:
			media_status = MASCA_NO_MEDIA;
			disc_state = SM_MAX_DISC_STATE;
			is_bad_media_signalled = FALSE;
			load_state = TOC_READ_INITIATE;
			rom_hash_read_state = ROM_HASH_START;
			if (DRV_DISC_INSERTING == drv_mngr_event) {
				drive_state = SM_INSERTING;
				p_evt_info->evt = CD_INSERTING;
			} else if (DRV_DISC_INSERTED == drv_mngr_event) {
				drive_state = SM_INSERTED;
				p_evt_info->evt = CD_INSERTED;
				media_status = MASCA_MEDIA_INSERTED;
			} else if (DRV_DISC_ABSENT == drv_mngr_event) {
				p_evt_info->evt = CD_ABSENT;
			}
			break;
		case SM_INSERTING:
			disc_state = SM_MAX_DISC_STATE;
			if (DRV_DISC_INSERTED == drv_mngr_event) {
				drive_state = SM_INSERTED;
				p_evt_info->evt = CD_INSERTED;
				media_status = MASCA_MEDIA_INSERTED;
			} else if (DRV_DISC_EJECTING == drv_mngr_event) {
				p_evt_info->event_info.cd_identify.
					was_cd_accessible = FALSE;
				drive_state = SM_EJECTING;
				p_evt_info->evt = CD_EJECTING;
			} else if (DRV_DISC_ABSENT == drv_mngr_event) {
				/*Just a manual trigger of the drive
				insertion without inserting CD can
				bring us to this state*/
				drive_state = SM_NO_CD;
				p_evt_info->evt = CD_ABSENT;
			}
			break;
		case SM_INSERTED:
			disc_state = SM_MAX_DISC_STATE;
			if (DRV_DISC_TOC_READY == drv_mngr_event) {
				drive_state = SM_LOADING;
				is_imm_transition = TRUE;
			} else if (DRV_DISC_TOC_ERROR == drv_mngr_event) {
				p_evt_info->evt = CD_LOAD_ERROR;
				p_evt_info->event_info.cd_identify
					.cd_type = DISC_BAD_MEDIA;
			} else if (DRV_DISC_EJECTING == drv_mngr_event) {
				p_evt_info->event_info.cd_identify.
					was_cd_accessible = FALSE;
				drive_state = SM_EJECTING;
				p_evt_info->evt = CD_EJECTING;
				media_status =
					MASCA_MEDIA_EJECT_PROCESSED;
			}
			break;
		case SM_LOADING:
			if (DRV_DISC_READING_TOC == drv_mngr_event) {
				/*Might be TOC reading is re-initiated
				* so go back to the previous state
				* till the TOC is ready*/
				drive_state = SM_INSERTED;
			} else if (DRV_DISC_EJECTING == drv_mngr_event) {
				p_evt_info->event_info.cd_identify.
					was_cd_accessible = FALSE;
				drive_state = SM_EJECTING;
				p_evt_info->evt =  CD_EJECTING;
				media_status = MASCA_MEDIA_EJECT_PROCESSED;
			} else if (DRV_DISC_PAUSED == drv_mngr_event) {
				disc_state = SM_PAUSED;
			} else if (DRV_DISC_STOPPED == drv_mngr_event) {
				disc_state = SM_STOPPED;
			} else if (DRV_DISC_PLAYING == drv_mngr_event) {
				disc_state = SM_PLAYING;
			} else {
				masca_exec_drive_sm_load(p_response_info,
						p_evt_info, &current_cd_type,
						&current_hash_id);
			}
			break;
		case SM_ACCESSIBLE:
			if (DRV_DISC_EJECTING == drv_mngr_event) {
				drive_state = SM_EJECTING;
				p_evt_info->event_info.cd_identify.
					was_cd_accessible = TRUE;
				p_evt_info->evt =  CD_EJECTING;
				media_status = MASCA_MEDIA_EJECT_PROCESSED;
			} else {
				/*execute the disc state machine*/
				masca_exec_disc_sm(p_drv_event, p_evt_info);
			}
			/*No Additional Sense Information*/
			masca_update_global_sense_buffer(0x00, 0x00, 0x00);
			break;
		case SM_EJECTING:
			disc_state = SM_MAX_DISC_STATE;
			is_bad_media_signalled = FALSE;
			if (DRV_DISC_INSERTING == drv_mngr_event) {
				drive_state = SM_INSERTING;
				p_evt_info->evt = CD_AUTO_REINSERTING;
			} else if (DRV_DISC_ABSENT == drv_mngr_event) {
				drive_state = SM_NO_CD;
				p_evt_info->evt = CD_ABSENT;
			} else if (DRV_DISC_IN_SLOT == drv_mngr_event) {
				p_evt_info->evt = CD_IN_SLOT;
			} else if (DRV_DISC_INSERTED == drv_mngr_event) {
				drive_state = SM_INSERTED;
				p_evt_info->evt = CD_INSERTED;
				media_status = MASCA_MEDIA_INSERTED;
			}
			break;
		default:
			SM_TI_TRACE("SM default drive state %x\n", drive_state);
			break;
		}
	}

	if (drive_state > SM_NO_CD) {
		if (DRV_DISC_DEFECTIVE == drv_mngr_event)
			p_evt_info->evt = CD_DEFECTIVE;
		else if (DRV_MECHANIC_LOAD_ERR == drv_mngr_event)
			p_evt_info->evt = CD_MECHA_LOAD_ERR;
	}

	if (SM_ACCESSIBLE == drive_state) {
		p_evt_info->event_info.cd_identify.cd_type = present_cd_type;
		p_evt_info->event_info.cd_identify.unique_id = present_hash_id;
	}
}

static void
masca_load_cd_info(const struct masca_drv_response_info * const p_response_info,
			enum masca_media_type * const p_current_cd_type,
			unsigned int * const p_current_hash_id)
{
	enum masca_cmd		responded_cmd = MAX_COMMAND;
	enum masca_error	response = MASCA_PTR_NULL;
	enum masca_error	err;
	bool			imm_state_transition = TRUE;
	unsigned char		trk_num = 0;

	if (p_response_info != NULL) {
		responded_cmd = p_response_info->drv_cmd_response;
		response = p_response_info->response_error;
	}

	while (TRUE == imm_state_transition) {
		imm_state_transition = FALSE;
		switch (load_state) {
		case TOC_READ_INITIATE:
			err = masca_snd_cmd_drv(GET_TOC);
			if ((err != MASCA_OK) && (err != MASCA_PROCESSED)) {
				load_state = LOAD_FAILED;
				SM_TE_DRIVEE02(
				   "%x during TOC_READ_INITIATE\n", load_state);
			} else if (MASCA_PROCESSED == err) {
				load_state = SESSION_INFO_INITIATE;
				imm_state_transition = TRUE;
			} else {
				load_state = TOC_READ_WAIT;
			}
			break;
		case TOC_READ_WAIT:
			if ((GET_TOC == responded_cmd) &&
			((MASCA_OK == response) ||
			(MASCA_PROCESSED == response))) {
				/*toc is read successfully, compute media type*/
				/*here hash id is computed for only audio CD*/
				masca_compute_media_type(&present_cd_type,
							   &present_hash_id);
				if (DISC_BAD_MEDIA == present_cd_type) {
					load_state = LOAD_FAILED;
				} else {
					load_state = SESSION_INFO_INITIATE;
					imm_state_transition = TRUE;
				}
			} else if (GET_TOC == responded_cmd) {
				/*TOC read failed*/
				load_state = LOAD_FAILED;
			}
			break;
		case SESSION_INFO_INITIATE:
			/*initiate session info read*/
#ifdef SESSION_INFO_NO_SUPPORT
			session_info.first_complete_session = 1;
			if (DISC_CDDA == present_cd_type) {
				/*It will be treated as just one single session
				even if there are multiple sessions as hw does
				not say how many sessions are present*/
				trk_num = cd_toc.first_track_number;
				session_info.last_complete_session = 1;
			} else if (DISC_CDROM == present_cd_type) {
				/*disc type CD ROM*/
				trk_num = cd_toc.last_track_number;
				session_info.last_complete_session = trk_num -
					cd_toc.first_track_number + 1;
			} else if (DISC_MIXED_MODE == present_cd_type) {
				session_info.first_complete_session =
						cd_toc.first_data_trk_num;
				session_info.last_complete_session =
						cd_toc.last_data_trk_num -
						cd_toc.first_data_trk_num + 1;
			}
			session_info.last_sess_trk_info.track_number = trk_num;
			session_info.last_sess_trk_info.adr_ctrl =
				cd_toc.track_info[trk_num].adr_ctrl;
			memcpy(&session_info.last_sess_trk_info.track_start.min,
				&cd_toc.track_info[trk_num].track_start.min,
				sizeof(struct track_msf));
			session_info.last_sess_trk_info.track_start_lba =
				cd_toc.track_info[trk_num].track_start_lba;
			load_state = CD_TEXT_INITIATE;
			imm_state_transition = TRUE;

#else
			err = masca_snd_cmd_drv(GET_SESSION_INFO);
			if ((err != MASCA_OK) && (err != MASCA_PROCESSED)) {
				load_state = LOAD_FAILED;
				SM_TE_TRACE("%x during SESSION_INFO_INITIATE\n",
								load_state);
			} else if (MASCA_PROCESSED == err) {
				load_state = CD_TEXT_INITIATE;
				imm_state_transition = TRUE;
			} else {
				load_state = SESSION_INFO_WAIT;
			}
#endif
			break;
		case SESSION_INFO_WAIT:
			if ((GET_SESSION_INFO == responded_cmd)
				&& ((MASCA_OK == response)
				|| (MASCA_PROCESSED == response))) {
				load_state = CD_TEXT_INITIATE;
				imm_state_transition = TRUE;
			} else if (GET_SESSION_INFO == responded_cmd) {
				/*TOC read failed*/
				load_state = LOAD_FAILED;
			}
			break;
		case CD_TEXT_INITIATE:
			/*now initiate read of CD text*/
			if ((DISC_CDDA == present_cd_type) ||
					(DISC_MIXED_MODE == present_cd_type)) {
				/*cd text is necessary only for audio CDs*/
				err = masca_snd_cmd_drv(GET_TITLE_ALBUM_AUTHOR);
				if ((err != MASCA_OK)
					&& (err != MASCA_PROCESSED)) {
					load_state = LOAD_FAILED;
					SM_TE_TRACE
					("%x during CD_TEXT_INITIATE\n",
						load_state);
				} else if (MASCA_PROCESSED == err) {
					load_state = LOAD_COMPLETE;
				} else {
					load_state = CD_TEXT_WAIT;
				}
			} else {
				/*cd text reading is not applicable for CD ROM*/
				load_state = COMPUTE_HASH_ID_ROM;
				imm_state_transition = TRUE;
			}
			break;
		case CD_TEXT_WAIT:
			if ((GET_TITLE_ALBUM_AUTHOR == responded_cmd)
				&& ((MASCA_OK == response)
				|| (MASCA_PROCESSED == response))) {
				load_state = LOAD_COMPLETE;
			} else if (GET_TITLE_ALBUM_AUTHOR == responded_cmd)
				/*TOC read failed*/
				load_state = LOAD_FAILED;

			break;
		case COMPUTE_HASH_ID_ROM:
			/*hash ID in case of CDROM is computed
			 * This should happen only during cmd flow
			 * and not during event flow */
			if (NULL != p_response_info) {
				masca_compute_hash_id_cdrom(&present_hash_id,
						p_response_info);
				if (ROM_HASH_FAILED == rom_hash_read_state)
					load_state = LOAD_FAILED;
				else if (ROM_HASH_COMPLETE
						== rom_hash_read_state)
					load_state = LOAD_COMPLETE;
			}
			break;
		default:
			load_state = LOAD_FAILED;
			break;
		}
	}
	if (LOAD_COMPLETE == load_state) {
		*p_current_cd_type = present_cd_type;
		*p_current_hash_id = present_hash_id;
	}

}
static void masca_computehash_romread(struct masca_drv_response_info *
		manager_response, unsigned int *current_hash_id,
		bool *imm_state_transition)
{
	unsigned short byte_count;

	if ((MAX_COMMAND != manager_response->drv_cmd_response) &&
		(READ_SECTORS_HASH == manager_response->drv_cmd_response)) {
		if ((MASCA_OK == manager_response->response_error) ||
			(MASCA_PROCESSED == manager_response->response_error)) {
			for (byte_count = 0; byte_count < 2048; byte_count++)
				*current_hash_id += hash_sector_buf[byte_count];
			sectors_to_read++;
			if (sectors_to_read > HASH_END_SECTOR)
				rom_hash_read_state = ROM_HASH_COMPLETE;
			else
				*imm_state_transition = TRUE;
		} else {
			rom_hash_read_state = ROM_HASH_FAILED;
			SM_TE_DRIVEE03("%x ROM_HASH_READ\n",
			rom_hash_read_state);
		}
	} else {
		rom_hash_read_state = ROM_HASH_WAIT;
	}
}

static void
masca_compute_hash_id_cdrom(unsigned int * const p_current_hash_id,
				const struct masca_drv_response_info *
				const p_resp_info)
{
	unsigned int current_hash_id = *p_current_hash_id;
	unsigned short byte_count;
	enum masca_error		err = MASCA_OK;
	struct masca_drv_cmd_info	manager_cmd;
	struct masca_drv_response_info manager_response;
	enum masca_cmd		responded_cmd = p_resp_info->drv_cmd_response;
	enum masca_error		response = p_resp_info->response_error;
	bool			imm_state_transition = TRUE;

	while (TRUE == imm_state_transition) {
		imm_state_transition = FALSE;
		switch (rom_hash_read_state) {
		case ROM_HASH_START:
			sectors_to_read = HASH_START_SECTOR;
			rom_hash_read_state = ROM_HASH_READ;
			imm_state_transition = TRUE;
			break;

		case ROM_HASH_READ:
			masca_drv_mngr_flush_buffers();
			manager_cmd.command = READ_SECTORS_HASH;
				manager_cmd.command_param.drv_param.
				block_read_info.number_of_sectors = 1;
			manager_cmd.command_param.drv_param.
				block_read_info.p_buffer = hash_sector_buf;
			manager_response.drv_cmd_response = MAX_COMMAND;
			manager_cmd.command_param.drv_param.
				block_read_info.sector_start_address =
							sectors_to_read;
			err = masca_drive_mngr_cmd(&manager_cmd,
						   &manager_response);
			if (err >= MASCA_OK)
				masca_computehash_romread(&manager_response,
						&current_hash_id,
						&imm_state_transition);
			break;

		case ROM_HASH_WAIT:
			if ((READ_SECTORS_HASH == responded_cmd)
				&& ((MASCA_OK == response)
				|| (MASCA_PROCESSED == response))) {
				for (byte_count = 0;
					byte_count < 2048;
					byte_count++) {
					current_hash_id +=
						hash_sector_buf[byte_count];
				}
				sectors_to_read++;
				if (sectors_to_read > HASH_END_SECTOR) {
					rom_hash_read_state =
						ROM_HASH_COMPLETE;
				} else {
					rom_hash_read_state =
						ROM_HASH_READ;
					imm_state_transition = TRUE;
				}
			} else if (READ_SECTORS_HASH == responded_cmd) {
				rom_hash_read_state = ROM_HASH_FAILED;
				SM_TE_TRACE("%x during ROM_HASH_WAIT\n",
						rom_hash_read_state);
			}
			break;

		default:
			rom_hash_read_state = ROM_HASH_FAILED;
			break;
		}
	}
	*p_current_hash_id = current_hash_id;
}

static void
masca_compute_media_type(enum masca_media_type * const p_current_cd_type,
				unsigned int * const p_current_hash_id)
{

	unsigned int		current_hash_id = 0;
	enum masca_media_type	current_cd_type = DISC_BAD_MEDIA;
	unsigned char		trk_num = cd_toc.first_track_number;
	unsigned char		last_trk_num = cd_toc.last_track_number;
	unsigned char		trk_adr_ctrl;
	const unsigned char	data_trk_bit = 0x40;
	bool		is_audio_trk_present = FALSE;
	bool		is_data_trk_present = FALSE;

	/*we will compute hash ID only for CDDA as it is
	straight forward through TOC*/
	while ((last_trk_num <= MAX_TRACK) && (trk_num <= last_trk_num)) {
		trk_adr_ctrl = cd_toc.track_info[trk_num].adr_ctrl;
		if ((unsigned char)(trk_adr_ctrl & data_trk_bit) ==
					data_trk_bit) {
			/*it is a data track*/
			is_data_trk_present = TRUE;
		} else {
			/*it is an audio track*/
			is_audio_trk_present = TRUE;

			current_hash_id +=
			cd_toc.track_info[trk_num].track_start_lba * 19;
		}
		trk_num++;
	}

	if ((TRUE == is_data_trk_present)
		&& (TRUE == is_audio_trk_present)) {
		/* It is a mixed mode CD */
		current_cd_type = DISC_MIXED_MODE;
	} else if (TRUE == is_audio_trk_present) {
		/* It is an audio CD */
		current_cd_type = DISC_CDDA;
		/* reset audio ignore bit */
		masca_set_last_cmd_ign_bit(FALSE);
	} else if (TRUE == is_data_trk_present) {
		/* It is a CD-ROM */
		current_cd_type = DISC_CDROM;
		/* reset audio ignore bit */
		masca_set_last_cmd_ign_bit(FALSE);
	}

	*p_current_cd_type = current_cd_type;
	*p_current_hash_id = current_hash_id;

}

static void masca_exec_disc_sm(
			const struct masca_drv_event_info * const p_drv_event,
			struct masca_event_info * const p_evt_info)
{
	if ((NULL != p_drv_event) && (NULL != p_evt_info)) {
		p_evt_info->evt = NO_EVENT;
		/*assumed that null check is done outside of this function*/
		switch (disc_state) {
		case SM_PLAYING:
			if (DRV_DISC_PAUSED == p_drv_event->drv_event) {
				disc_state = SM_PAUSED;
			} else if (DRV_DISC_STOPPED == p_drv_event->drv_event) {
				disc_state = SM_STOPPED;
			} else if (DRV_DISC_PLAY_INFO ==
						p_drv_event->drv_event) {
				if ((p_drv_event->play_position
						.abs_min != 0) ||
					(p_drv_event->
						play_position.abs_sec != 0)) {
					p_evt_info->evt = CD_PLAY_INFO;
					memcpy(
					&p_evt_info->event_info.play_info,
					&p_drv_event->play_position,
					sizeof(struct masca_drv_play_info));
				}
			} else if (DRV_TRACK_END == p_drv_event->drv_event) {
				p_evt_info->evt = CD_TRACK_END;
			} else if (DRV_DISC_END == p_drv_event->drv_event) {
				p_evt_info->evt = CD_END;
			}
			break;

		case SM_STOPPED:
			if (DRV_DISC_PLAYING == p_drv_event->drv_event)
				disc_state = SM_PLAYING;

			break;

		case SM_PAUSED:
			if (DRV_DISC_PLAYING == p_drv_event->drv_event)
				disc_state = SM_PLAYING;
			else if (DRV_DISC_STOPPED == p_drv_event->drv_event)
				disc_state =  SM_STOPPED;

			break;
		default:
			disc_state = SM_MAX_DISC_STATE;
			SM_TE_TRACE(
			    "SM - Disc state %x during ROM_HASH_WAIT\n",
						  rom_hash_read_state);
			break;
		}
		if (DRV_PLAY_RANGE_END == p_drv_event->drv_event)
			/*Play end is respected in all states because
			according to ATAPI specifications a play range of 0
			also should give out play complete event eventhough
			hardware does not go into play state*/
			p_evt_info->evt = CD_PLAY_COMPLETE;

	}
}

static enum masca_error masca_snd_cmd_drv(const enum masca_cmd command_to_send)
{
	enum masca_error err = MASCA_OK;

	struct masca_drv_cmd_info	manager_cmd;
	struct masca_drv_response_info	manager_response;

	manager_cmd.command = command_to_send;
	manager_response.drv_cmd_response = MAX_COMMAND;
	manager_response.response_error = MASCA_OK;

	if (GET_TOC == command_to_send) {
		manager_cmd.command_param.p_cd_toc = &cd_toc;
	} else if (GET_SESSION_INFO == command_to_send) {
		manager_cmd.command_param.p_cd_session_info = &session_info;
	} else if (GET_TITLE_ALBUM_AUTHOR == command_to_send) {
		manager_cmd.command_param.p_cd_toc = &cd_toc;
		manager_cmd.command_param.p_cd_text = &cd_text;
	} else if ((EN_IGN_AUDIO == command_to_send) ||
			(DIS_IGN_AUDIO == command_to_send) ||
			(PAUSE == command_to_send)) {
		/* Send command to drive for mixed mode CD switching */
	} else {
		err = MASCA_INVALID_PARAM;
	}

	if (MASCA_OK == err) {
		masca_drive_mngr_cmd(&manager_cmd, &manager_response);
		err = manager_response.response_error;
	}

	if (MASCA_OK == err) {
		if ((MAX_COMMAND != manager_response.drv_cmd_response) &&
			(command_to_send == manager_response.
			 drv_cmd_response)) {
			if ((MASCA_OK == manager_response.response_error)
				|| (MASCA_PROCESSED ==
					manager_response.response_error))
				err = MASCA_PROCESSED;

		} else {
			err = MASCA_OK;
		}
	}
	return err;
}

static void masca_updt_state_machines(
		const struct masca_drv_response_info * const p_response_info,
		const struct masca_drv_event_info * const p_drv_event,
		struct masca_event_info * const p_evt_info)
{
	enum masca_event out_evt = p_evt_info->evt;

	/*Now we have to control our drive state machine
	depending on the temperature state, voltage state and
	driver state*/
	/*We should always execute the drive state machine
	otherwise updates of disc state from drive manager does
	not happen*/
	if ((SUSPENDED == driver_state)
		|| (OVER_TEMPERATURE == hw_temp_state)
		|| (UNDER_VOLTAGE == voltage_state)) {
		masca_disable_disc_load();
		/*If we are not giving out any voltage or temperature
		related events then only execute the state machine
		as there is some other event to be processed*/
		if (NO_EVENT == out_evt) {
			masca_exec_drive_sm(p_response_info,
						p_drv_event,
						p_evt_info);
			/*We consider only the following events given
			  out by drive sm*/
			out_evt = p_evt_info->evt;
			if ((CD_INSERTING != out_evt) &&
				(CD_AUTO_REINSERTING != out_evt) &&
				(CD_INSERTED != out_evt) &&
				(CD_EJECTING != out_evt) &&
				(CD_IN_SLOT != out_evt) &&
				(CD_DEFECTIVE != out_evt) &&
				(CD_MECHA_LOAD_ERR != out_evt)) {
				p_evt_info->evt = NO_EVENT;
			}
		} else {
			masca_util_set_event(SM_EVENT_UPDATE);
		}
	} else {
		masca_enable_disc_load();
		if (NO_EVENT == out_evt)
			masca_exec_drive_sm(p_response_info,
					    p_drv_event, p_evt_info);
		else
			/*update to state machine is necessary as
			because at this point of time we have either a
			voltage/temperature and hence we will not be
			able to update state machine as p_evt_info is
			already filled */
			masca_util_set_event(SM_EVENT_UPDATE);

	}
}

static void masca_disable_disc_load(void)
{
	is_disc_load_allowed = FALSE;
	/*now revert the load state to a triggering
	state, not a wait state*/
	if (TOC_READ_WAIT == load_state)
		load_state = TOC_READ_INITIATE;
	else if (SESSION_INFO_WAIT == load_state)
		load_state = SESSION_INFO_INITIATE;
	else if (CD_TEXT_WAIT == load_state)
		load_state = CD_TEXT_INITIATE;
	else if (COMPUTE_HASH_ID_ROM == load_state)
		rom_hash_read_state = ROM_HASH_START;
}

static void masca_enable_disc_load(void)
{
	is_disc_load_allowed = TRUE;
}
