/*
 * MASCA Drive Manager block read, serves block read request
 * READ(10) along with masca blockreader and masca cache files.
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_helper.h"
#include "masca_interpreter.h"
#include "masca_block_reader.h"
#include "masca_event_config.h"
#include "masca_drv_mngr.h"
#include "masca_drv_mngr_slot.h"
#include "masca_drv_mngr_cnfg.h"
#include "masca_drv_mngr_internal.h"
#include "masca_hal.h"

static void masca_blkrd_response(
			struct masca_drv_response_info * const p_response_info);

static struct masca_cmd_slot blk_read;
static enum masca_intrptr_cmd active_cmd;
static bool is_pause_cmd_pending;
/*Default cd-rom read speed*/
static enum masca_intrptr_cmd play_command = CMD_PLAY_CONTINUOUS_HALFMAX;
static bool blkrd_disable_cache;

void masca_init_blkrd(const unsigned int retry_cnt,
				const unsigned int reset_cnt,
				const unsigned int response_tmout)
{
	blkrd_disable_cache = FALSE;
	blk_read.slot_stat = SLOT_PAUSED;
	blk_read.app_cmd_info.command = MAX_COMMAND;
	blk_read.hw_response = MASCA_OK;
	blk_read.slot_tmout = response_tmout;
	active_cmd = CMD_MAX;
	is_pause_cmd_pending = FALSE;
	masca_init_tx_rx(SLOT_BLK_RD, retry_cnt, reset_cnt);
}

enum masca_cmd masca_get_blkrd_cmd(void)
{
	return blk_read.app_cmd_info.command;
}

void masca_blkrd_set_spin_speed(enum masca_spin_speed spin_speed)
{
	if (SPIN_SPEED_HALFMAX == spin_speed) {
		play_command = CMD_PLAY_CONTINUOUS_HALFMAX;
		drive_speed = 2; /* 2x speed - Max. 150 sectors/sec */
	} else if (SPIN_SPEED_MAX == spin_speed) {
		play_command = CMD_PLAY_CONTINUOUS_MAX;
		drive_speed = 4; /* 4x speed - Max. 300 sectors/sec */
	} else {
		play_command = CMD_PLAY_CONTINUOUS;
		drive_speed = 1; /* 1x speed - Max. 75 sectors/sec */
	}
}

void masca_blkrd_alm_cbk(void)
{
	/*This timeout just indicates that there is no response for the play
	 *command sent to the drive*/
	masca_util_set_event(SLOT_BLOCK_READ_TMOUT);
}

/* masca_blkrd_calc_cd_bounds is used to calculate
 * the start and end address of data tracks*/
static void masca_blkrd_calc_cd_bounds(struct masca_cd_toc *p_toc,
					unsigned int *start_addr,
					unsigned int *end_addr)
{
	unsigned char	first_trk = p_toc->first_data_trk_num;
	unsigned char	last_trk = p_toc->last_data_trk_num;
	if (p_toc->first_audio_trk_num != p_toc->first_track_number) {
		*start_addr = p_toc->track_info[first_trk].track_start_lba;
		if ((last_trk != p_toc->last_track_number) &&
			(p_toc->first_track_number !=
					p_toc->last_track_number)) {
			*end_addr = p_toc->track_info[last_trk + 1].
							track_start_lba - 1;
		} else {
			*end_addr = GET_LBA(p_toc->cd_max_min,
						p_toc->cd_max_sec, 0);
		}
	} else {
		*start_addr = p_toc->track_info[p_toc->first_track_number].
								track_start_lba;
		*end_addr = GET_LBA(p_toc->cd_max_min, p_toc->cd_max_sec, 0);
	}

}

enum masca_error masca_tx_blkrd(
			const struct masca_drv_cmd_info * const p_cmd_info,
			struct masca_drv_response_info * const p_response_info)
{
	enum masca_error ret_err = MASCA_BUSY;
	enum masca_cmd app_cmd;
	unsigned int sect_start_addr;
	unsigned int sect_count;
	unsigned int cd_strt_addr;
	unsigned int cd_end_addr;
	unsigned char *p_buf;
	struct masca_cd_toc *p_toc_info = NULL;

	masca_get_cd_toc(&p_toc_info);
	app_cmd = blk_read.app_cmd_info.command;

	masca_blkrd_calc_cd_bounds(p_toc_info, &cd_strt_addr, &cd_end_addr);
	masca_blkrdr_set_cd_bounds(cd_strt_addr, cd_end_addr);
	sect_start_addr = p_cmd_info->command_param.drv_param.block_read_info.
							sector_start_address;
	sect_start_addr = GET_PHY_ADDR(sect_start_addr);
	sect_count = p_cmd_info->command_param.drv_param.block_read_info.
							number_of_sectors;
	p_buf = p_cmd_info->command_param.drv_param.block_read_info.p_buffer;

	if ((sect_start_addr < cd_strt_addr) ||
			((sect_start_addr + sect_count - 1) > cd_end_addr)) {
		ret_err = MASCA_OK;
		p_response_info->drv_cmd_response = p_cmd_info->command;
		p_response_info->response_error = MASCA_IR_LBA_OUT_OF_RANGE;
	} else if (0 == sect_count) {
		ret_err = MASCA_PROCESSED;
		p_response_info->drv_cmd_response = p_cmd_info->command;
		p_response_info->response_error = MASCA_PROCESSED;
	} else if (SLOT_PAUSED == blk_read.slot_stat) {
		if (MAX_COMMAND == app_cmd) {
			memcpy((&(blk_read.app_cmd_info)), p_cmd_info,
				sizeof(struct masca_drv_cmd_info));
			ret_err = MASCA_OK;
		}
	} else if ((READ_SECTORS == p_cmd_info->command) ||
				(READ_SECTORS_HASH == p_cmd_info->command)) {
		ret_err = MASCA_OK;
		if (MAX_COMMAND == app_cmd) {
			/*We do not have any requests to
			 * process,so add this*/
			memcpy((&(blk_read.app_cmd_info)), p_cmd_info,
				sizeof(struct masca_drv_cmd_info));
			ret_err = masca_blkrdr_read_blocks(sect_start_addr,
						sect_count, p_buf);
			DRV_MNGR_TD_TRACE("tx_blkrd:appcmd:%d\n",
					blk_read.app_cmd_info.command);
			if ((MASCA_OK != ret_err) &&
					(MASCA_ACCEPTED != ret_err)) {
				blk_read.hw_response = ret_err;
				masca_blkrd_response(p_response_info);
			}
		} else {
			/*We are already executing a block read
			 * now just check if data is available
			 * in cache by querying block reader*/
			ret_err = masca_blkrdr_read_blocks(sect_start_addr,
						sect_count, p_buf);
			if ((MASCA_PROCESSED == ret_err)
					|| (MASCA_IO_ERR == ret_err)) {
				/*fill up response info*/
				p_response_info->drv_cmd_response =
							p_cmd_info->command;
				p_response_info->response_error = ret_err;
			}
		}
	}

	return ret_err;
}

void masca_rx_blkrd(const struct masca_intrptr_msg_info_list * const p_list,
		const struct masca_intrptr_rjt_cmd_list * const p_rjt_cmds,
		const struct masca_intrptr_msg_info * const p_msg,
		struct masca_drv_response_info * const p_response_info)
{
	/*In the paused state also the responses from the hardware
	 * are respected*/
	enum masca_error ret_err;

	if (SLOT_PAUSED != blk_read.slot_stat) {
		/*Finding out a response is received or not is a generic
		 * functionality. Hence it is implemented in a common place*/
		ret_err = masca_rcv_response(SLOT_BLK_RD, p_list, p_rjt_cmds);
		if ((MASCA_OK != ret_err) && (MASCA_REJECTED != ret_err)) {
			/*We got response to the slot, so clear the timeout*/
			if ((play_command == active_cmd) &&
					(TRUE == is_pause_cmd_pending) &&
					(MASCA_PROCESSED == ret_err)) {
				/*We received response for play.
				 * Send pause command now*/
				active_cmd = CMD_PAUSE;
				is_pause_cmd_pending = FALSE;
				masca_send_cmd(SLOT_BLK_RD, CMD_PAUSE, 0, 0, 0);
				masca_util_clr_event(SLOT_BLOCK_READ_TMOUT);
				masca_util_sta_alm(blk_read.slot_tmout,
								ALM_BLK_RD);
			} else {
				active_cmd = CMD_MAX;
				masca_util_stp_alm(ALM_BLK_RD);
				masca_util_clr_event(SLOT_BLOCK_READ_TMOUT);
			}
		}
	}
}

void masca_pause_blkrd(void)
{
	blk_read.slot_stat = SLOT_PAUSED;
	/*we are pausing so don't allow timeouts*/
	if (MAX_COMMAND > blk_read.app_cmd_info.command) {
		masca_util_stp_alm(ALM_BLK_RD);
		masca_util_clr_event(SLOT_BLOCK_READ_TMOUT);
		/*stop the respective unit so that no retry happens*/
		masca_abort_unit(SLOT_BLK_RD);
		/*Also abort the block reader*/
		masca_blkrdr_abort();
	}
}

void masca_resume_blkrd(struct masca_drv_response_info * const p_response_info)
{
	unsigned int sect_start_addr;
	unsigned int sect_count;
	unsigned char *p_buf;
	enum masca_error ret_err;
	unsigned int cd_strt_addr;
	unsigned int cd_end_addr;
	struct masca_cd_toc *p_toc_info = NULL;

	blk_read.slot_stat = SLOT_IDLE;
	if (MAX_COMMAND > blk_read.app_cmd_info.command) {
		masca_get_cd_toc(&p_toc_info);
		masca_blkrd_calc_cd_bounds(p_toc_info, &cd_strt_addr,
								&cd_end_addr);
		masca_blkrdr_set_cd_bounds(cd_strt_addr, cd_end_addr);
		sect_start_addr = blk_read.app_cmd_info.command_param.
				drv_param.block_read_info.sector_start_address;
		sect_start_addr = GET_PHY_ADDR(sect_start_addr);
		sect_count = blk_read.app_cmd_info.command_param.drv_param.
					block_read_info.number_of_sectors;
		p_buf = blk_read.app_cmd_info.command_param.drv_param.
					block_read_info.p_buffer;
		ret_err = masca_blkrdr_read_blocks(sect_start_addr, sect_count,
							p_buf);
		if ((MASCA_PROCESSED == ret_err) || (MASCA_IO_ERR == ret_err)) {
			/*fill up response info*/
			p_response_info->drv_cmd_response
					= blk_read.app_cmd_info.command;
			p_response_info->response_error = ret_err;
		}
	}
}

void masca_abort_blkrd(struct masca_drv_response_info * const p_response_info,
			enum masca_error abort_reason)
{
	blk_read.hw_response = abort_reason;
	if (MAX_COMMAND == p_response_info->drv_cmd_response) {
		blk_read.abort_slot = FALSE;
		masca_blkrd_response(p_response_info);
	} else {
		/* set an abort update so that through event
		 * interface of this slot we can abort the command*/
		blk_read.abort_slot = TRUE;
		masca_util_set_event(SLOT_ABORT_UPDATE);
	}
	masca_blkrdr_abort();
	/*We need to pause the disk whenever we abort*/
	if (play_command == active_cmd) {
		is_pause_cmd_pending = TRUE;
	} else {
		/*we have not sent play command to drive, so no
		 need to pause*/
		/*stop the respective unit so that no retry happens*/
		masca_abort_unit(SLOT_BLK_RD);
		masca_util_stp_alm(ALM_BLK_RD);
		masca_util_clr_event(SLOT_BLOCK_READ_TMOUT);
	}
}

bool masca_evt_blkrd(unsigned int evt_ptn_slot,
		struct masca_drv_response_info * const p_response_info)
{
	bool is_rst_required = FALSE;
	enum masca_error ret_err;
	unsigned int play_lba_strt;
	unsigned char block_cnt;

	if (SLOT_ABORT_UPDATE == evt_ptn_slot) {
		if (TRUE == blk_read.abort_slot) {
			masca_abort_blkrd(p_response_info,
							blk_read.hw_response);
		}
	} else if (BLOCK_READ_COMPLETE == evt_ptn_slot) {
		blk_read.hw_response = MASCA_PROCESSED;
		masca_blkrd_response(p_response_info);
		DRV_MNGR_TD_TRACE("In DRV MGR:BLOCK_READ_COMPLETE\n");
	} else if (BLOCK_READ_FAILED == evt_ptn_slot) {
		blk_read.hw_response = MASCA_ME_IDCRC_CDROM;
		masca_blkrd_response(p_response_info);
		DRV_MNGR_TD_TRACE("BLOCK_READ_FAILED\n");
	} else if (SLOT_PAUSED != blk_read.slot_stat) {
		if (SLOT_BLOCK_READ_TMOUT == evt_ptn_slot) {
			/*Now we need to do retries. These are
			 * just retries to send play/pause command
			 * and not for actual block read*/
			ret_err = masca_evt_tmout(SLOT_BLK_RD,
							&is_rst_required);
			if (MASCA_OK != ret_err) {
				blk_read.hw_response = ret_err;
				if (ret_err != MASCA_PROCESSED) {
					/*abort the block reader from
					 * processing further  as this
					 * command to drive itself is not
					 * successful*/
					masca_blkrdr_abort();
					masca_blkrd_response(p_response_info);
				}
			} else {
				masca_util_sta_alm(blk_read.slot_tmout,
								ALM_BLK_RD);
			}
		} else if (BLOCK_READ_REQ == evt_ptn_slot) {
			/*Now send out a play command*/
			active_cmd = play_command;
			is_pause_cmd_pending = FALSE;
			(void) masca_blkrdr_get_play_pos(&play_lba_strt,
								&block_cnt);
			masca_send_cmd(SLOT_BLK_RD, play_command, play_lba_strt,
								0, block_cnt);
			masca_util_sta_alm(blk_read.slot_tmout, ALM_BLK_RD);
		} else if (BLOCK_READ_RESUME == evt_ptn_slot) {
			struct masca_intrptr_cmd_info cmd;
			unsigned char cmd_stream[COMMAND_MAX_LENGTH];
			unsigned char stream_lng = 0;
			BLKRDR_TR_D("In case BLOCK_READ_RESUME\n");
			cmd.command = CMD_RESUME;
			masca_util_stp_alm(ALM_BLK_RD);
			if (MASCA_OK == masca_intrptr_encode_cmd(&cmd,
						cmd_stream, &stream_lng)) {
				masca_prepare_sndbuf(cmd_stream, stream_lng);
				BLKRDR_TR_D("resume cmd sent!!\n");
			}
		} else if (BLOCK_READER_IDLE == evt_ptn_slot) {
			if (active_cmd < CMD_MAX) {
				/*There is no response to play command yet from
				 the drive. So wait*/
				is_pause_cmd_pending = TRUE;
			} else {
				active_cmd = CMD_PAUSE;
				is_pause_cmd_pending = FALSE;
				masca_send_cmd(SLOT_BLK_RD, CMD_PAUSE, 0, 0, 0);
				masca_util_sta_alm(blk_read.slot_tmout,
								ALM_BLK_RD);
			}
		}
	}

	return is_rst_required;
}

static void masca_blkrd_response(struct masca_drv_response_info *
					const p_response_info)
{
	if (MASCA_OK != blk_read.hw_response) {
		p_response_info->drv_cmd_response
				= blk_read.app_cmd_info.command;
		DRV_MNGR_TD_TRACE("masca_blkrd_response: app cmd %d\n",
				blk_read.app_cmd_info.command);

		p_response_info->response_error = blk_read.hw_response;
		masca_util_stp_alm(ALM_BLK_RD);
		masca_util_clr_event(SLOT_BLOCK_READ_TMOUT);
		blk_read.app_cmd_info.command = MAX_COMMAND;
		blk_read.hw_response = MASCA_OK;

		if (SLOT_PAUSED != blk_read.slot_stat)
			blk_read.slot_stat = SLOT_IDLE;

		if (blkrd_disable_cache != FALSE)
			masca_blkrdr_flush();
	}
}

/*
 * masca_drv_mngr_disable_cache
 *
 * Bocke read cache enable/diaable
 *
 * drv_mngr_disable_cache = TRUE : Disable cache
 * drv_mngr_disable_cache = FALSE : Enable cache
 *
 */
extern void masca_drv_mngr_blkrd_cache_disable(bool drv_mngr_disable_cache)
{
	blkrd_disable_cache = drv_mngr_disable_cache;
	   masca_blkrdr_cache_disable(drv_mngr_disable_cache);
}

