/*
 * MASCA SCSI Implementation File
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/cdrom.h>
#include <target/target_core_base.h>
#include "masca_blkdev.h"
#include "masca_scsi_handler.h"
#include "masca_atapi_bridge.h"
#include "masca_sysfs.h"
#include "masca_sm.h"

#define MASCA_HOST_ID		-1
#define MASCA_CMDQ_LEN		1
#define MASCA_CMD_LUN		1
#define MASCA_MAX_SECTOR	128
#define MASCA_SG_TABLE_SIZE	0x0f
#define MASCA_MAX_DEV		1

/* COMMANDS */
#define ATAPI_RESET		0x08

struct device *scsi_dev;
struct masca_host_info *hostinfo;
static char masca_proc_name[] = COMPONENT_NAME_MASCA;
static DEFINE_MUTEX(scsicmdq_lock);
atomic_t async_sense = ATOMIC_INIT(0);

struct masca_host_info {
	struct Scsi_Host *shost;
	struct device dev;
};

struct masca_drive_info version_info = {
	.peripheral_device_type = PHERIPHERAL_TYPE_CDROM, /*CDROM*/
	.rmb                    = 0x80,
	.iso_version            = 0x00,
	.atapi_version          = 0x02,
	.additional_length      = 0x1f,
};

static unsigned char mode_sense_10_data[] = {0x00, 0x26, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x2a, 0x1e,
						0x03, 0x00, 0x71, 0x01, 0x09,
						0x00, 0x02, 0xc0, 0x00, 0x02,
						0x00, 0x80, 0x02, 0xc0, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00,
						0x01, 0x00, 0x00, 0x00, 0x00};

void masca_set_async_sense(int sense)
{
	struct masca_sense_info si = masca_get_sense_value(sense);
	int loc_sense = atomic_read(&async_sense);
	loc_sense |= sense;
	atomic_set(&async_sense, loc_sense);
	masca_update_global_sense_buffer(si.key, si.asc, si.ascq);
}

void masca_clr_async_sense(int sense)
{
	int loc_sense = atomic_read(&async_sense);
	loc_sense &= ~sense;
	atomic_set(&async_sense, loc_sense);
	masca_update_global_sense_buffer(MASCA_OK, MASCA_OK, MASCA_OK);
}

enum masca_error masca_resp_sense(struct scsi_cmnd *scp)
{
	struct masca_sense_info sinfo;
	enum masca_error err = MASCA_OK;

	err = masca_check_async_sense();
	if (MASCA_OK != err) {
		sinfo = masca_get_sense_value(err);
		masca_update_global_sense_buffer(sinfo.key, sinfo.asc,
						sinfo.ascq);
		masca_fill_sense_buffer(scp, sinfo);
		scp->result = SAM_STAT_CHECK_CONDITION;
	}

	return err;
}

void masca_reply_sense(struct scsi_cmnd *scp, short err)
{
	struct masca_sense_info si = masca_get_sense_value(err);

	masca_update_global_sense_buffer(si.key, si.asc, si.ascq);
	masca_fill_sense_buffer(scp, si);
	scp->result = SAM_STAT_CHECK_CONDITION;
}

enum masca_error masca_check_async_sense(void)
{
	int sense = atomic_read(&async_sense);
	/* Sense keys are prioritized here be cautious while
	 * changing the below order of if, elseif */
	if (0 < sense) {
		if ((sense & SENSE_DRV_DISCONNECT) == SENSE_DRV_DISCONNECT)
			sense = MASCA_HE_COMM_FAIL;
		else if ((sense & SENSE_VOLT_ERR) == SENSE_VOLT_ERR)
			sense = MASCA_HE_VOLTAGE_FAULT;
		else if ((sense & SENSE_LOAD_ERR) == SENSE_LOAD_ERR)
			sense = MASCA_HE_LOAD_EJECT_FAIL;
		else if ((sense & SENSE_UNKNOWNFORM_ERR) ==
				SENSE_UNKNOWNFORM_ERR)
			sense = MASCA_NR_UNKNOWN_FORMAT;
		else if ((sense & SENSE_TOC_ERR) == SENSE_TOC_ERR)
			sense = MASCA_ME_TOC_ERROR;
		else if ((sense & SENSE_CDEND_ERR) == SENSE_CDEND_ERR)
			sense = MASCA_IR_TRK_END_MSG;
		else if ((sense & SENSE_TEMP_ERR) == SENSE_TEMP_ERR)
			sense = MASCA_SE_OVR_TEMP;
	} else {
		sense = MASCA_OK;
	}

	return sense;
}

void masca_fill_scsi_buffer(struct scsi_cmnd *scp, void *arr, int arr_len)
{
	int resid = 0;
	if ((0 != arr_len) && (NULL != arr)) {
		resid = scsi_bufflen(scp)
				- sg_copy_from_buffer(scsi_sglist(scp),
						scsi_sg_count(scp), arr,
						arr_len);
		scsi_set_resid(scp, resid);
	} else
		/* No data transfer */
		scsi_set_resid(scp, scsi_bufflen(scp));
}

static void masca_release_adapter(struct device *dev)
{
}

static int masca_driver_remove(struct device *dev)
{
	struct masca_host_info *hostinfo;

	hostinfo = container_of(dev, struct masca_host_info, dev);
	scsi_remove_host(hostinfo->shost);
	scsi_host_put(hostinfo->shost);
	return 0;
}

static int masca_abort(struct scsi_cmnd *scpnt)
{
	struct masca_rcvr *rcvr = NULL;
	int err = FAILED;

	rcvr = masca_get_aborted_cmd_from_que(scpnt);
	if ((NULL != rcvr) && (rcvr->a_cmnd == scpnt)) {
		/* Handle abort */
		masca_dispatch_abort_req(rcvr->replyid);
		/* Supply proper sense key for user */
		masca_reply_sense(scpnt, MASCA_HE_TIMEOUT_ON_LOGUNIT);
		scpnt->scsi_done(scpnt);
		/* Remove this command from local queue */
		masca_rmv_rcvr_que(rcvr);
		err = SUCCESS;
	}

	return err;
}

#define GCONFIG_HDSIZE			8
#define GCONF_LIST_SIZE			8
#define GCONF_CORE_SIZE			8
#define GCONF_RMMEDIA_SIZE		8
#define GCONF_APLAY_SIZE		8
#define GCONF_RANDRD_SIZE		12
#define GCONF_LISTTOT_SIZE		52
static unsigned char config_header[] = {0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00};
static unsigned char config_data[] = {	0x00, 0x00, 0x03, 0x04,
					0x00, 0x08, 0x01, 0x00,
					0x00, 0x01, 0x03, 0x04,
					0x00, 0x00, 0xFF, 0xFF,
					0x00, 0x03, 0x03, 0x04,
					0x28, 0x00, 0x00, 0x00,
					0x01, 0x03, 0x00, 0x04,
					0x04, 0x00, 0x00, 0x02,
					0x00, 0x10, 0x00, 0x08,
					0x00, 0x00, 0x08, 0x00,
					0x00, 0x0a, 0x00, 0x00};
static unsigned char config_response[GCONF_LISTTOT_SIZE];
void masca_get_config(struct scsi_cmnd *scp)
{
	unsigned char *cmd = (unsigned char *) scp->cmnd;
		unsigned short fcode = (unsigned short)
				(cmd[GETCONFIG_REQ_FEAT_NUM+0x00] << 8) +
				cmd[GETCONFIG_REQ_FEAT_NUM+0x01];
		unsigned int buf_length = 0;
		unsigned int offset = 0;

		if ((cmd[GETCONFIG_REQ_RT] & 0x03) == 0) {
			/* Prepare Feature Descriptor */
			if (fcode == 0x00) { /* list Feature */
				buf_length =	GCONF_LIST_SIZE +
						GCONF_CORE_SIZE +
						GCONF_RMMEDIA_SIZE +
						GCONF_APLAY_SIZE +
						GCONF_RANDRD_SIZE;
			} else if (fcode == 0x01) { /* core Feat */
				buf_length = GCONF_CORE_SIZE;
				offset = GCONF_LIST_SIZE;
			} else if (fcode == 0x03) { /* Removable media */
				buf_length = GCONF_RMMEDIA_SIZE;
				offset =	GCONF_LIST_SIZE +
						GCONF_CORE_SIZE;
			} else if (fcode == 0x103) { /*Analog Audio play */
				buf_length = GCONF_APLAY_SIZE;
				offset =	GCONF_LIST_SIZE +
						GCONF_CORE_SIZE +
						GCONF_RMMEDIA_SIZE;
			} else if (fcode == 0x10) { /* Random read */
				buf_length = GCONF_RANDRD_SIZE;
				offset =	GCONF_LIST_SIZE +
						GCONF_CORE_SIZE +
						GCONF_RMMEDIA_SIZE +
						GCONF_APLAY_SIZE;
			} else {
				SCSI_BLK_TD_TRACE(
				"Get_config cmd: Unhandled Feature Code = 0x%x\n",
									fcode);
			}
		} else {
			SCSI_BLK_TD_TRACE(
			"Get_config cmd: Unhandled Request Type	= 0x%x\n",
						(cmd[GETCONFIG_REQ_RT] & 0x03));
		}



		memcpy(&config_response[GCONFIG_HDSIZE],
				&config_data[offset], buf_length);

		/* set current profile depending on media status */
		if (media_status == MASCA_NO_MEDIA)
			/* no current profile */
			config_header[GETCONFIG_RESP_HDR_CUR_PRFL+0x01] = 0x00;
		else
			/* CD-ROM profile */
			config_header[GETCONFIG_RESP_HDR_CUR_PRFL+0x01] = 0x08;

		buf_length += GCONFIG_HDSIZE;
		config_header[GETCONFIG_RESP_HDR_DAT_LEN+0x00] =
						(buf_length >> 24) & 0xff;
		config_header[GETCONFIG_RESP_HDR_DAT_LEN+0x01] =
						(buf_length >> 16) & 0xff;
		config_header[GETCONFIG_RESP_HDR_DAT_LEN+0x02] =
						(buf_length >> 8) & 0xff;
		config_header[GETCONFIG_RESP_HDR_DAT_LEN+0x03] =
							buf_length & 0xFF;

		memcpy(&config_response[GETCONFIG_RESP_HDR_DAT_LEN],
		&config_header[GETCONFIG_RESP_HDR_DAT_LEN], GCONFIG_HDSIZE);

		masca_fill_scsi_buffer(scp,
		&config_response[GETCONFIG_RESP_HDR_DAT_LEN], buf_length);

}

void masca_mode_sense10_response(struct scsi_cmnd *scp)
{
	if (scp->cmnd[MODESENSE10_REQ_PAGE_CODE] == 0x2A) {
		if (media_status == MASCA_NO_MEDIA)
			mode_sense_10_data[MODESENSE10_RESP_MEDIUM_TYPE] = 0;
		else {
			if (type_of_disc == DISC_CDDA)
				mode_sense_10_data[MODESENSE10_RESP_MEDIUM_TYPE]
									= 2;
			else if (type_of_disc == DISC_CDROM)
				mode_sense_10_data[MODESENSE10_RESP_MEDIUM_TYPE]
									= 1;
			else if (type_of_disc == DISC_MIXED_MODE)
				mode_sense_10_data[MODESENSE10_RESP_MEDIUM_TYPE]
									= 3;
		}

		masca_fill_scsi_buffer(scp, &mode_sense_10_data,
			sizeof(mode_sense_10_data));
	} else{
		SCSI_BLK_TE_TRACE("unsupported MODE SENSE page\n");
	}
}

static int masca_queuecommand_lck(struct scsi_cmnd *scp,
					void (*done)(struct scsi_cmnd *))
{
	unsigned char *cmd = NULL;
	bool add_to_que = false;
	cmd = (unsigned char *) scp->cmnd;
	scp->result = 0;
	switch (*cmd) {
	case REQUEST_SENSE:	/*mandatory, ignore unit attention*/
		masca_reply_sense(scp, MASCA_OUT_OF_BOUND);
		break;
	case MODE_SENSE_10:
		masca_mode_sense10_response(scp);
		break;
	case GPCMD_GET_CONFIGURATION:
		masca_get_config(scp);
	break;
	case GPCMD_READ_SUBCHANNEL:
	case GET_EVENT_STATUS_NOTIFICATION:
	case GPCMD_READ_DISC_INFO:
	case LOG_SENSE:
	case INQUIRY:
	case RESERVE:
	case RELEASE:
	case SEND_DIAGNOSTIC:
	case RECEIVE_DIAGNOSTIC:
	case ATAPI_RESET:
	case START_STOP:
	case GPCMD_PLAY_AUDIO_MSF:
	case GPCMD_PLAY_AUDIO_10:
	case GPCMD_PLAY_AUDIO_TI:
	case GPCMD_PAUSE_RESUME:
	case GPCMD_STOP_PLAY_SCAN:
	case READ_TOC:
	case GPCMD_SCAN:
	case READ_CAPACITY:
	case TEST_UNIT_READY:
	case READ_10:
	case GPCMD_SET_SPEED:
		add_to_que = true;
		break;
	default:
		SCSI_BLK_TD_TRACE("Unhandled command %x:\n", *cmd);
		masca_reply_sense(scp, MASCA_IR_CMD_NOT_SUPP);
		break;
	}
	if (true == add_to_que) {
			/* TODO: Need to check, if add to queue is
			 * success or not */
			masca_add_rcvr_que(scp, false);
			masca_set_recvr_waitcondition(MASCA_COMMAND);
	} else {
		scp->scsi_done(scp);
	}
	return 0;
}

static DEF_SCSI_QCMD(masca_queuecommand)

static struct scsi_host_template masca_driver_template = {
	.proc_name	=	masca_proc_name,
	.name		=	"MASCA SCSI",
	.queuecommand	=	masca_queuecommand,
	.eh_abort_handler =	masca_abort,
	.can_queue	=	MASCA_CMDQ_LEN,
	.this_id	=	MASCA_HOST_ID,
	.sg_tablesize	=	MASCA_SG_TABLE_SIZE,
	.cmd_per_lun	=	MASCA_CMD_LUN,
	.max_sectors	=	MASCA_MAX_SECTOR,
	.use_clustering	=	DISABLE_CLUSTERING,
	.module		=	THIS_MODULE,
};

static int masca_driver_probe(struct device *dev)
{
	struct Scsi_Host *hpnt;
	struct masca_host_info *masca_host;
	int error;
	masca_host = container_of(dev, struct masca_host_info, dev);
	hpnt = scsi_host_alloc(&masca_driver_template, sizeof(struct
							      masca_host_info));
	if (NULL == hpnt) {
		SCSI_BLK_TA_TRACE("%s: scsi_register failed\n", __func__);
		error = -ENODEV;
		return error;
	}
	masca_host->shost = hpnt;
	hpnt->max_id = MASCA_MAX_DEV;
	hpnt->max_lun = 0;
	*((struct masca_host_info **)hpnt->hostdata) = masca_host;

	error = scsi_add_host(hpnt, &masca_host->dev);
	if (error) {
		SCSI_BLK_TA_TRACE("%s: scsi_add_host failed\n", __func__);
		error = -ENODEV;
		scsi_host_put(hpnt);
	} else
		scsi_scan_host(hpnt);
	return error;
}

static struct bus_type masca_lld_bus = {
	.name = "masca",
	.probe = masca_driver_probe,
	.remove = masca_driver_remove,
};

struct device_driver masca_devdrv = {
	.name	= masca_proc_name,
	.bus	= &masca_lld_bus,
};

int masca_scsi_init(void)
{
	int ret = 0;
	hostinfo = kzalloc(sizeof(struct masca_host_info), GFP_KERNEL);
	if (NULL == hostinfo) {
		SCSI_BLK_TA_TRACE("%s: out of memory at line %d\n",
				  __func__, __LINE__);
		return -ENOMEM;
	}
	scsi_dev = root_device_register(ROOT_DEVICE_NAME);
	if (IS_ERR(scsi_dev)) {
		SCSI_BLK_TA_TRACE("masca: root_device_register() error\n");
		ret = PTR_ERR(scsi_dev);
		goto free_vm;
	}
	ret = bus_register(&masca_lld_bus);
	if (ret < 0) {
		SCSI_BLK_TA_TRACE("masca: bus_register error: %d\n",
				  ret);
		goto dev_unreg;
	}
	ret = driver_register(&masca_devdrv);
	if (ret < 0) {
		SCSI_BLK_TA_TRACE("masca: driver_register error: %d\n",
				  ret);
		goto bus_unreg;
	}
	hostinfo->dev.bus = &masca_lld_bus;
	hostinfo->dev.parent  = scsi_dev;
	hostinfo->dev.release = &masca_release_adapter;
	dev_set_name(&hostinfo->dev, "adapter%d", 1);
	ret = configure_sysfs_files(&masca_devdrv);
	if (!ret)
		ret = device_register(&hostinfo->dev);
	if (ret == 0)
		return ret;

	driver_unregister(&masca_devdrv);
bus_unreg:
	bus_unregister(&masca_lld_bus);
dev_unreg:
	root_device_unregister(scsi_dev);
free_vm:
	kfree(hostinfo);
	return ret;
}

int masca_scsi_deinit(void)
{
	remove_sysfs_files(&masca_devdrv);
	device_unregister(&hostinfo->dev);
	driver_unregister(&masca_devdrv);
	bus_unregister(&masca_lld_bus);
	root_device_unregister(scsi_dev);
	kfree(hostinfo);
	return 0;
}
