/*
 * linux/drivers/char/exchnd/exchnd_filter.c
 *
 * Copyright (C) 2013 Advanced Driver Information Technology GmbH
 * Written by Matthias Weise (mweise@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * Process filter
 *
 * The process filter implement a mechanism to allow to check if a process is
 * in a given set of processes. The processes are identified by the pathname of
 * the executable or their name
 */
#define pr_fmt(fmt) "exchnd: " fmt

#include <linux/exchnd.h>
#include <linux/jhash.h>
#include <linux/kernel.h>
#include <linux/sched.h>

#include "exchnd_internal.h"

/* Maximum number of process filters */
#define MAX_REG_PROCS 64
/* Helper macros to calculate the INDEX_SHIFT */
#define NBITS2(n) ((n&2) ? 1 : 0)
#define NBITS4(n) ((n&(0xC)) ? (2+NBITS2(n>>2)) : (NBITS2(n)))
#define NBITS8(n) ((n&0xF0) ? (4+NBITS4(n>>4)) : (NBITS4(n)))
#define NBITS16(n) ((n&0xFF00) ? (8+NBITS8(n>>8)) : (NBITS8(n)))
#define NBITS32(n) ((n&0xFFFF0000) ? (16+NBITS16(n>>16)) : (NBITS16(n)))
#define NBITS(n) (n == 0 ? 0 : NBITS32(n)+1)
/* Number of bits  to shift to get index in bitfield array */
#define INDEX_SHIFT (2 + NBITS(sizeof(atomic_t)))
/* Bitfield defining the free/used filters */
static atomic_t reg_bitfield[MAX_REG_PROCS >> INDEX_SHIFT];
/* Filter item structure */
struct exchnd_process_filter {
	struct rb_node node;
	u32 hash;
	unsigned int kernel:1;
	u32 path_hash;
};
/* List of the registered process filters */
static struct exchnd_process_filter filters[MAX_REG_PROCS];
/* Root of filter red/black tree */
static struct rb_root filter_root = RB_ROOT;

/**
 * exchnd_get_free_filter - Get free filter entry
 *
 * Return:  Number of entry in the filter array or ENOMEM if array is full
 *
 * This function looks for a free entry in the filter array. It is based on a
 * bitfield and thread safe. It defines if an entry in the filter array is in
 * use or not with  the help of a bitfield.
 */
static int exchnd_get_free_filter(void)
{
	int loop = 0;

	while (loop < (MAX_REG_PROCS >> 5)) {
		int local = atomic_read(&reg_bitfield[loop]);
		if (local != -1) {
			unsigned int revert = ~local;
			unsigned int first_free = __builtin_ctz(revert);
			unsigned int new_val = local | (1 << first_free);
			int old = atomic_cmpxchg(&reg_bitfield[loop],
			   local, new_val);
			if (old == local)
				return (loop << INDEX_SHIFT) + first_free;
		} else
			loop++;
	}

	return -ENOMEM;
}

/**
 * exchnd_remove_filter_reg - Remove a filter from the bitfield
 * @index: Index of filter to remove
 *
 * Return: 0 if removal was successful, an error code otherwise
 *
 * This function removes a filter from the bitfield. The filter is free and
 * can be used again.
 */
static int exchnd_remove_filter_reg(unsigned int index)
{
	unsigned int array_index = index >> INDEX_SHIFT;
	unsigned int bit_no = (~(array_index << INDEX_SHIFT)) & index;
	int local;
	int new_val;
	int old;

	if (index >=  MAX_REG_PROCS)
		return -EINVAL;

	do {
		local = atomic_read(&reg_bitfield[array_index]);
		new_val = local & (~(1 << bit_no));
		old = atomic_cmpxchg(&reg_bitfield[array_index],
			   new_val, local);
	} while (old != local);

	return 0;
}

/**
 * exchnd_compare_filter - Compares two filters
 * @left: First filter to compare
 * @right: Second filter to compare
 *
 * Return: -1 if left filter is "smaller" than right filter
 *          0 if both filters are equal
 *          1 if left filter is "bigger" than right filter
 *
 * This function compares two process filters. First the hashes are compared.
 * If they are equal, a process filter is "bigger" than a kernel filter.
 * Kernel filters are equal if their hashes are equal. For process filters the
 * path hashes are compared if the hashes are equal. Only if they are equal
 * the process filters are equal. Otherwise the order of the path_hashes
 * determines the complete order.
 */
static int exchnd_compare_filter(struct exchnd_process_filter *left,
	struct exchnd_process_filter *right)
{
	if (left->hash > right->hash)
		return 1;
	if (left->hash < right->hash)
		return -1;
	/* Hashes are equal */
	if ((left->kernel == 1) && (right->kernel == 1))
		return 0;
	/* Process is "greater" than kernel */
	if (left->kernel != right->kernel)
		return (left->kernel == 1) ? -1 : 1;
	/* Process names -> now compare path hashes) */
	if (left->path_hash == right->path_hash)
		return 0;

	return (left->path_hash > right->path_hash) ? 1 : -1;
}

/**
 * exchnd_find_in_rbtree - Looks for a filter in the tree
 * @new: Filter to search in the tree
 * @parent: Found parent
 * @link: Found filter
 *
 * Return: -1 if insertion position was found left of the parent
 *          0 if filter was found
 *          1 if insertion position was found right of the parent
 *
 * This function searches for a filer in the tree. If it does not find the
 * tree in returns the position where the tree would be inserted via its
 * output parameters.
 */
static int exchnd_find_in_rbtree(struct exchnd_process_filter *new,
	struct rb_node **parent, struct rb_node ***link)
{
	int cmp = 1;
	struct exchnd_process_filter *filter;

	*link = &filter_root.rb_node;
	*parent = NULL;

	while (*(*link)) {
		*parent = *(*link);
		filter = rb_entry(*parent, struct exchnd_process_filter, node);
		cmp = exchnd_compare_filter(filter, new);
		if (cmp == 0)
			return 0;

		if (cmp == 1)
			*link = &(*(*link))->rb_left;
		else
			*link = &(*(*link))->rb_right;
	}

	return cmp;
}

/**
 * exchnd_process_filter_add - Adds new process filter
 * @id: ID for the process. The complete path name for a user space process
 *      and the thread name for kernel processes.
 * @kernel: 1 if it is a kernel thread, 0 for user space processes
 *
 * Return: 0 if addition was successful, error code otherwise
 *
 * This function adds a process filter to the system. If the filter already
 * exists it returns an error.
 */
int exchnd_process_filter_add(const char *id, unsigned int kernel)
{
	int filter_id;
	struct exchnd_process_filter *new_filter;

	struct rb_node **link;
	struct rb_node *parent;

	if (exchnd_get_debug() & EXCHND_DEBUG_FILTER)
		pr_info("Adding %s to list (k: %d).\n", id, kernel);

	filter_id = exchnd_get_free_filter();
	if (filter_id < 0)
		return filter_id;
	new_filter = &filters[filter_id];
	/* Prepare filter for inserting */
	if (kernel == 0) {
		new_filter->hash = jhash(kbasename(id),
		 strlen(kbasename(id)), 0);
		new_filter->path_hash = jhash(id, strlen(id), 0);
		new_filter->kernel = 0;
	} else {
		new_filter->hash = jhash(id, strlen(id), 0);
		new_filter->path_hash = 0;
		new_filter->kernel = 1;
	}
	/* Insert into red/black tree */
	if (exchnd_find_in_rbtree(new_filter, &parent, &link) == 0)
		return -EEXIST;
	rb_link_node(&new_filter->node, parent, link);
	rb_insert_color(&new_filter->node, &filter_root);

	return 0;
}

/**
 * exchnd_process_filter_remove - Remove a process filter
 * @id: ID for the process. The complete path name for a user space process
 *      and the thread name for kernel processes.
 * @kernel: 1 if it is a kernel thread, 0 for user space processes
 *
 * Return: 0 if removal was successful, error code otherwise
 *
 * This function removes a process filter from the system. If the filter
 * does not exist it returns an error.
 */
int exchnd_process_filter_remove(const char *id, unsigned int kernel)
{
	struct exchnd_process_filter sample;
	struct rb_node **link;
	struct rb_node *parent;
	int index;

	if (exchnd_get_debug() & EXCHND_DEBUG_FILTER)
		pr_info("Removing %s from list (k: %d).\n", id, kernel);

	if (kernel == 0) {
		sample.hash = jhash(kbasename(id),
		 strlen(kbasename(id)), 0);
		sample.path_hash = jhash(id, strlen(id), 0);
		sample.kernel = 0;
	} else {
		sample.hash = jhash(id, strlen(id), 0);
		sample.path_hash = 0;
		sample.kernel = 1;
	}
	if (exchnd_find_in_rbtree(&sample, &parent, &link) != 0)
		return -ENOENT;

	/* Calculate index in filter array */
	index = rb_entry(*link, struct exchnd_process_filter, node) - filters;
	rb_erase(*link, &filter_root);
	exchnd_remove_filter_reg(index);

	return 0;
}

/**
 * exchnd_is_filtered - Check if process is filtered
 * @task: The task that shall be checked if it is filtered
 *
 * Return: 1 if task is filtered, 0 otherwise
 *
 * This function checks if the given task is in the "list" of configured
 * tasks. The list is used to find out which tasks get a special behavior
 * in case of an exception /(e.g. process exit, signals, etc.)
 */
int exchnd_is_filtered(struct task_struct *task)
{
	/* Spinlock locks protects static path buffer */
	static DEFINE_SPINLOCK(buffer_lock);
	struct exchnd_process_filter search;
	struct rb_node *node;
	struct exchnd_process_filter *filter_node;
	static char path_buffer[1024];
	char *path;

	if (task->mm != NULL) {
		const char *fname =
			task->mm->exe_file->f_path.dentry->d_name.name;
		search.hash = jhash(fname, strlen(fname), 0);
		search.path_hash = 0;
		search.kernel = 0;

		if (exchnd_get_debug() & EXCHND_DEBUG_FILTER)
			pr_info("Checking if %s is filtered.\n", fname);
	} else {
		search.hash = jhash(task->comm, strlen(task->comm), 0);
		search.path_hash = 0;
		search.kernel = 1;

		if (exchnd_get_debug() & EXCHND_DEBUG_FILTER)
			pr_info("Checking if %s is filtered.\n", task->comm);
	}

	node = filter_root.rb_node;
	while (node != NULL) {
		filter_node = rb_entry(node, struct exchnd_process_filter,
			node);
		if (filter_node->hash == search.hash)
			break;
		if (filter_node->hash < search.hash)
			node = node->rb_right;
		else
			node = node->rb_left;
	}
	/* Not found */
	if (node == NULL)
		return 0;
	/* Must be the same type, to be comparable */
	if (filter_node->kernel != search.kernel)
		return 0;
	/* For a kernel thread we are finished */
	if (search.kernel == 1)
		return 1;
	/* Get path from current process */
	spin_lock(&buffer_lock);
	path_get(&task->mm->exe_file->f_path);
	path = d_path(&task->mm->exe_file->f_path, &path_buffer[0],
	  sizeof(path_buffer));
	path_put(&task->mm->exe_file->f_path);
	search.path_hash = jhash(path, strlen(path), 0);
	spin_unlock(&buffer_lock);
	if (rb_entry(node, struct exchnd_process_filter, node)->path_hash ==
		 search.path_hash)
		return 1;
	else
		return 0;
}
