/*
 * (c) 2013 Advanced Driver Information Technology GmbH
 *          Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * llrb - lock-less ring buffer
 *
 * llrb provides a lock-less, thread-safe ring buffer with a fixed
 * number of slots. Atomics are used to modify the read and write
 * index into the buffer as well as each slot's state.
 *
 * Indexes increase monotonically and the read index is always equal
 * to or smaller than the write index. Access to the ring buffer's
 * slots is done index modulo number of slots.
 *
 * Each slot can be in any of the following states:
 *  FREE    - the slot is not in use
 *  WRITING - the slot is in use but data has not been written yet
 *  WRITTEN - the slot is in use and data has been written
 *  READING - the slot is in use and being read
 *
 * WRITING and READING are transient states. However, due to preemptive
 * scheduling a slot can stay in these states for any amount of
 * time. During that time, the slot is excluded from ring buffer
 * processing by other threads of execution.
 *
 * If data is put into the buffer and the buffer is full, either the
 * new data is discarded, or the oldest data in the buffer is
 * automatically discarded if LLRB_DROP_OLD == 1.
 *
 * If a slot in state WRITING or READING is to be discarded, the slot
 * is silently skipped and the next slot is discarded.
 *
 * Each allocated slot has a sequence number and flags. The sequence
 * number is the write index when allocating the slot. No two slots
 * can have the same sequence number, but sequence numbers can be
 * non-consecutive due to slots being skipped during allocation.
 * Flags are used to indicate error conditions, e.g. if data had to
 * be dropped from the ring buffer when allocating a slot.
 *
 * Slots can hold so called binary data (data has as its first byte
 * a null byte (\0) or ASCII data (data doesn't start with a null byte.
 * Binary data can have a maximum size of 256 bytes. If a write
 * request for binary data exceeds this limit the overrun will be
 * discarded. In case of ASCII data it is allowed to have an arbitrary
 * number of bytes in a write request. One slot can hold at a maximum
 * 254 bytes payload plus a newline and a null byte. If a write
 * request exceeds the maximum data size which can be stored in one
 * slot, the driver splits the data stream and generates independent
 * messages.
 */

#ifndef _LLRB_H
#define _LLRB_H

#include <linux/atomic.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/time.h>

#include "llrb_data.h"

/* number of slots in the ring buffer */
#define LLRB_SIZE                16

/* drop old or new data when the buffer is full */
#define LLRB_DROP_OLD             1

/* status flags of slots in the ring buffer */
#define LLRB_SLOT_OK          0x000
#define LLRB_SLOT_DROPPED     0x001 /* some data has been dropped */
#define LLRB_SLOT_BUSY        0x002 /* multiple readers for a slot */
#define LLRB_SLOT_INCOMPLETE  0x004 /* incomplete message */
#define LLRB_SLOT_OUTDATED    0x008 /* outdated (out-of-order) message */
#define LLRB_SLOT_CRC_ERROR   0x010 /* message with bad crc */
#define LLRB_SLOT_ANY_ERROR   0x01f /* any of the above flags */
#define LLRB_SLOT_RESTART     0x020 /* ring buffer has been restarted */
#define LLRB_SLOT_REBOOT      0x020 /* ring buffer has been restarted */
#define LLRB_SLOT_INITIALIZED 0x040 /* ring buffer has been initialized */
#define LLRB_SLOT_POWER       0x040 /* ring buffer has been initialized */
#define LLRB_SLOT_STATUS      0x07f /* all status flags */
#define LLRB_SLOT_READ        0x080 /* slot has been read */
#define LLRB_SLOT_ALL_FLAGS   0x0ff /* all flags */

/* bit position of status flags of slots in the ring buffer */
#define LLRB_SLOT_DROPPED_        0 /* some data has been dropped */
#define LLRB_SLOT_BUSY_           1 /* multiple readers for a slot */
#define LLRB_SLOT_INCOMPLETE_     2 /* incomplete message */
#define LLRB_SLOT_OUTDATED_       3 /* outdated (out-of-order) message */
#define LLRB_SLOT_CRC_ERROR_      4 /* message with bad crc */
#define LLRB_SLOT_RESTART_        5 /* ring buffer has been restarted */
#define LLRB_SLOT_REBOOT_         5 /* ring buffer has been restarted */
#define LLRB_SLOT_INITIALIZED_    6 /* ring buffer has been initialized */
#define LLRB_SLOT_POWER_          6 /* ring buffer has been initialized */
#define LLRB_SLOT_READ_           7 /* slot has been read */

/* magic numbers */
#define LLRB_MAGIC_START      0xdeadbeef
#define LLRB_MAGIC_END        0xfeebdaed

#define MILLISECONDS_TO_NANOSECONDS 1000000

/* a slot in the ring buffer */
struct llrb_slot {
	atomic_t state;           /* state of the slot */
	/* --- crc start --- */
	unsigned seqnum;          /* unique sequence number of the slot */
	unsigned long long time;  /* write kernel time */
	struct llrb_data data;    /* data */
	/* --- crc end --- */
	unsigned crc;            /* crc over seqnum,time and data */
	unsigned flags;          /* slot status flags LLRB_SLOT_... */
};

/* the ring buffer */
struct llrb {
	unsigned magic_start;         /* magic number start of ring buffer */
	unsigned size;                /* number of slots in the ring buffer */
	unsigned low, high;           /* read water marks */
	atomic_t rd;                  /* read index */
	atomic_t wr;                  /* write index, rd<=wr */
	atomic_t free;                /* free index, free<rd */
	unsigned long flags;          /* status flags LLRB_SLOT_... */
	struct llrb_slot slots[LLRB_SIZE]; /* slots in the ring buffer */
	unsigned magic_end;           /* magic number end of ring buffer */
};

/* llrb_init_mem - initialize the ring buffer in a piece of memory */
/* calculates the size of the ring buffer from the amount of the memory */
extern struct llrb *llrb_init_mem(
	void*, unsigned amount, /* memory to use for the ring buffer */
	int reuse);             /* if set re-use existing error memory */

/* llrb_init - initialize the ring buffer */
/* uses the given number of slots */
extern struct llrb *llrb_init(
	struct llrb*,
	unsigned size); /* number of ring buffer slots */

/* llrb_store - store an arbitrary large block of data in the ring buffer */
/* returns the sequence number of the first entry in the ring buffer or ~0 */
extern unsigned llrb_store(
	struct llrb*,
	unsigned,          /* sequence number to append to or 0 */
	struct llrb_iterator *);      /* data to store */

/* llrb_retrieve - retrieve a block of data from the ring buffer */
/* returns the sequence number of the next entry in the ring buffer or 0 */
extern unsigned llrb_retrieve(
	struct llrb*,
	unsigned,          /* sequence number to retrieve */
	struct llrb_iterator*,  /* location for retrieved data */
	unsigned long long*,           /* out: write kernel time */
	unsigned *flags);       /* out: slot status flags LLRB_SLOT_... */

/* llrb_free - free ring buffer slots */
/* removes packets with the given sequence number range from the ring buffer */
/* returns the sequence number of the next available slot */
extern unsigned llrb_free(
	struct llrb*,
	unsigned,  /* first packet to remove */
	unsigned); /* last packet to remove (inclusive) */

/* llrb_can_write - determine if and how many slots can be written */
extern unsigned llrb_can_write(struct llrb *);

/* llrb_can_read - determine if and which slot can be read */
extern unsigned llrb_can_read(struct llrb*, unsigned, int skip);

/* llrb_get_slotsize - get the ring buffer's slotsize */
static inline unsigned llrb_get_slotsize(struct llrb *llrb)
{
	return LLRB_DATA_SIZE;
}

/* llrb_get_size - get the ring buffer's size */
static inline unsigned llrb_get_size(struct llrb *llrb)
{
	return llrb->size;
}

/* llrb_get_watermark_low - get the ring buffer's low water mark */
static inline unsigned llrb_get_watermark_low(struct llrb *llrb)
{
	return llrb->low;
}

/* llrb_get_watermark_high - get the ring buffer's high water mark */
static inline unsigned llrb_get_watermark_high(struct llrb *llrb)
{
	return llrb->high;
}

/* llrb_set_watermark_low - set the ring buffer's low water mark */
/* changes the behavior of llrb_can_read: */
/* if the number of allocated slots is below this watermark and */
/* the oldest slot has not been fully written to, llrb_can_read assumes */
/* there are no slots to read; defaults to 0 */
static inline void llrb_set_watermark_low(struct llrb *llrb, unsigned low)
{
	llrb->low = low;
}

/* llrb_set_watermark_high - set the ring buffer's high water mark */
/* changes the behavior of llrb_can_read: */
/* if the number of allocated slots is above this watermark and */
/* the oldest slot has not been fully written to, llrb_can_read looks */
/* for other slots to read; defaults to the size of the ring buffer */
static inline void llrb_set_watermark_high(struct llrb *llrb, unsigned high)
{
	llrb->high = high;
}

/* llrb_blocking_read - check for blocking read */
/* reading may only block if a sequence number hasn't been written yet */
static inline int llrb_blocking_read(struct llrb *llrb, unsigned rd)
{
	return rd >= atomic_read(&llrb->wr);
}

/* llrb_get_rd - get the ring buffer's read index */
static inline unsigned llrb_get_rd(struct llrb *llrb)
{
	return atomic_read(&llrb->rd);
}

/* llrb_get_wr - get the ring buffer's write index */
static inline unsigned llrb_get_wr(struct llrb *llrb)
{
	return atomic_read(&llrb->wr);
}

/* llrb_get_free - get the ring buffer's free index */
static inline unsigned llrb_get_free(struct llrb *llrb)
{
	return atomic_read(&llrb->free);
}

/* llrb_get_state - get the state of a slot */
unsigned llrb_get_state(struct llrb *llrb, unsigned);

/* llrb_set_real_world_time - set the RWT value in ns */
extern long llrb_set_real_world_time(unsigned long long epoch_time);

#endif /* ifndef _LLRB_H */
