/*
 * This is a v4l2 output driver for the Freescale i.MX6 SOC. It carries out
 * color-space conversion, downsizing, resizing, and rotation transformations
 * on input buffers using the IPU Image Converter's Post-Processing task.
 *
 * Copyright (c) 2012 Mentor Graphics Inc.
 * Steve Longerbeam <steve_longerbeam@mentor.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version
 */
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/slab.h>

#include <linux/platform_device.h>
#include <media/v4l2-device.h>
#include <media/v4l2-ioctl.h>
#include <media/videobuf2-dma-contig.h>
#include <linux/platform_data/imx-ipu-v3.h>
#include <media/imx6.h>

MODULE_DESCRIPTION("i.MX6 v4l2 output device");
MODULE_AUTHOR("Steve Longerbeam <steve_longerbeam@mentor.com>");
MODULE_LICENSE("GPL");
MODULE_VERSION("0.1");

#define MIN_W        32
#define MIN_H        32
#define MAX_W      4096
#define MAX_H      4096
#define MAX_W_CROP 1024
#define MAX_H_CROP 1024

#define H_ALIGN    1 /* multiple of 2 */
#define S_ALIGN    1 /* multiple of 2 */

#define DEVICE_NAME "imx6-v4l2-output"

/* In bytes, per queue */
#define VID_MEM_LIMIT	SZ_64M

#define dprintk(dev, fmt, arg...) \
	v4l2_dbg(1, 1, &dev->v4l2_dev, "%s: " fmt, __func__, ## arg)

struct mx6vout_buffer {
	struct vb2_buffer vb; /* v4l buffer must be first */
	struct list_head list;
};

static inline struct mx6vout_buffer *to_mx6vout_vb(struct vb2_buffer *vb)
{
	return container_of(vb, struct mx6vout_buffer, vb);
}

struct mx6vout_pixfmt {
	char	*name;
	u32	fourcc;
	int     depth;   /* total bpp */
	int     y_depth; /* depth of first Y plane for planar formats */
};

struct mx6vout_format {
	unsigned int      width;
	unsigned int      height;
	unsigned int      bytesperline;
	unsigned int      stride;
	unsigned int      rot_stride; /* only applies to dest */
	unsigned int      sizeimage;
	struct mx6vout_pixfmt *fmt;
};

struct mx6vout_dma_buf {
	void          *virt;
	dma_addr_t    phys;
	unsigned long len;
};

struct mx6vout_dev {
	struct v4l2_device	v4l2_dev;
	struct video_device	*vfd;

	int                     use_count;
	struct mutex		mutex;
	spinlock_t		irqlock;

	struct list_head        active_q;
	bool                    stop;
	struct completion       comp;
	struct timer_list       start_frame_timer;
	unsigned long           frame_counter;
	unsigned long           streamon_jiffies;

	/* intermediate buffer for rotation */
	struct mx6vout_dma_buf  rot_intermediate;

	/* The rotation controls */
	int                     rotation; /* degrees */
	bool                    hflip;
	bool                    vflip;

	/* derived from rotation, hflip, vflip controls */
	enum ipu_rotate_mode    rot_mode;

	bool                    global_alpha_enable;
	bool                    chromakey_enable;

	/* the source format from userland */
	struct v4l2_pix_format  pix;

	/*
	 * win (from s_fmt_vid_out_overlay) holds global alpha, chromakey,
	 * and interlace info for the overlay. Width and height info is
	 * ignored, width and height of target overlay is set with s_crop.
	 */
	struct v4l2_window      win;

	/*
	 * the crop rectangle (from s_crop) specifies the crop dimensions
	 * and position of the overlay within the overlay framebuffer.
	 */
	struct v4l2_rect        crop;

	/*
	 * info about the overlay framebuffer (base address, width/height,
	 * pix format).
	 */
	struct v4l2_framebuffer fbuf;

	/* translated source and destination formats */
	struct mx6vout_format   src;
	struct mx6vout_format   dest;

	/* IPU resources */
	struct ipuv3_channel    *ipu_mem_pp_ch;
	struct ipuv3_channel    *ipu_pp_mem_ch;
	struct ipuv3_channel    *ipu_mem_rot_pp_ch;
	struct ipuv3_channel    *ipu_rot_pp_mem_ch;
	struct ipu_soc          *ipu;
	struct ipu_ic           *ic;
	struct ipu_irt          *irt;

	/* the IPU end-of-frame irqs */
	int                     pp_mem_irq;
	int                     rot_pp_mem_irq;

	/* buffer queue used in videobuf2 */
	struct vb2_queue        buffer_queue;
	struct vb2_alloc_ctx    *alloc_ctx;
};

static struct mx6vout_pixfmt mx6vout_pixformats[] = {
	{
		.name	= "RGB565",
		.fourcc	= V4L2_PIX_FMT_RGB565,
		.depth  = 16,
	}, {
		.name	= "RGB24",
		.fourcc	= V4L2_PIX_FMT_RGB24,
		.depth  = 24,
	}, {
		.name	= "BGR24",
		.fourcc	= V4L2_PIX_FMT_BGR24,
		.depth  = 24,
	}, {
		.name	= "RGB32",
		.fourcc	= V4L2_PIX_FMT_RGB32,
		.depth  = 32,
	}, {
		.name	= "BGR32",
		.fourcc	= V4L2_PIX_FMT_BGR32,
		.depth  = 32,
	}, {
		.name	= "4:2:2 packed, YUYV",
		.fourcc	= V4L2_PIX_FMT_YUYV,
		.depth  = 16,
	}, {
		.name	= "4:2:2 packed, UYVY",
		.fourcc	= V4L2_PIX_FMT_UYVY,
		.depth  = 16,
	}, {
		.name	= "4:2:0 planar, YUV",
		.fourcc	= V4L2_PIX_FMT_YUV420,
		.depth  = 12,
		.y_depth = 8,
	}, {
		.name   = "4:2:0 planar, YVU",
		.fourcc = V4L2_PIX_FMT_YVU420,
		.depth  = 12,
		.y_depth = 8,
	}, {
		.name   = "4:2:2 planar, YUV",
		.fourcc = V4L2_PIX_FMT_YUV422P,
		.depth  = 16,
		.y_depth = 8,
	}, {
		.name   = "4:2:0 planar, Y/CbCr",
		.fourcc = V4L2_PIX_FMT_NV12,
		.depth  = 12,
		.y_depth = 8,
	},
};

#define NUM_FORMATS ARRAY_SIZE(mx6vout_pixformats)

/* Controls */
static struct v4l2_queryctrl mx6vout_ctrls[] = {
	{
		.id		= V4L2_CID_HFLIP,
		.type		= V4L2_CTRL_TYPE_BOOLEAN,
		.name		= "Horizontal Flip",
		.minimum	= 0,
		.maximum	= 1,
		.step		= 1,
		.default_value	= 0,
	}, {
		.id		= V4L2_CID_VFLIP,
		.type		= V4L2_CTRL_TYPE_BOOLEAN,
		.name		= "Vertical Flip",
		.minimum	= 0,
		.maximum	= 1,
		.step		= 1,
		.default_value	= 0,
	}, {
		.id		= V4L2_CID_ROTATE,
		.type		= V4L2_CTRL_TYPE_INTEGER,
		.name		= "Rotation",
		.minimum	= 0,
		.maximum	= 270,
		.step		= 90,
		.default_value	= 0,
	},
};

static struct mx6vout_pixfmt *mx6vout_get_format(struct v4l2_pix_format *pix)
{
	struct mx6vout_pixfmt *ret = NULL;
	int i;

	for (i = 0; i < NUM_FORMATS; i++) {
		if (mx6vout_pixformats[i].fourcc == pix->pixelformat) {
			ret = &mx6vout_pixformats[i];
			break;
		}
	}

	return ret;
}

static struct v4l2_queryctrl *mx6vout_get_ctrl(int id)
{
	struct v4l2_queryctrl *ret = NULL;
	int i;

	for (i = 0; i < ARRAY_SIZE(mx6vout_ctrls); i++) {
		if (id == mx6vout_ctrls[i].id) {
			ret = &mx6vout_ctrls[i];
			break;
		}
	}

	return ret;
}

/* irqlock held when calling */
static void init_idmac_channel(struct mx6vout_dev *dev,
			       struct ipuv3_channel *channel,
			       int width, int height, int stride,
			       u32 fourcc, enum ipu_rotate_mode rot_mode,
			       dma_addr_t physbuf)
{
	unsigned int burst_size;

	ipu_cpmem_channel_init(channel, fourcc, width, height, stride, 0, 0, 0,
			       physbuf, 0, 0);

	if (rot_mode)
		ipu_cpmem_set_rotation(channel, rot_mode);

	if (channel == dev->ipu_mem_rot_pp_ch ||
	    channel == dev->ipu_rot_pp_mem_ch) {
		burst_size = 8;
		ipu_cpmem_set_block_mode(channel);
	} else
		burst_size = (width % 16) ? 8 : 16;

	ipu_cpmem_set_burst_size(channel, burst_size);

	ipu_ic_task_idma_init(dev->ic, channel, width, height,
			      burst_size, rot_mode);

	ipu_cpmem_set_axi_id(channel, 1);

	ipu_idmac_set_double_buffer(channel, false);
	ipu_idmac_set_triple_buffer(channel, false);
}

/* irqlock held when calling */
static int start_norotate(struct mx6vout_dev *dev,
			  dma_addr_t p_in, dma_addr_t p_out)
{
	struct mx6vout_format *src = &dev->src;
	struct mx6vout_format *dest = &dev->dest;
	enum ipu_color_space src_cs, dest_cs;
	int ret;

	src_cs = ipu_pixelformat_to_colorspace(src->fmt->fourcc);
	dest_cs = ipu_pixelformat_to_colorspace(dest->fmt->fourcc);

	/* setup the IC resizer and CSC */
	ret = ipu_ic_task_init(dev->ic,
			       src->width, src->height,
			       dest->width, dest->height,
			       src_cs, dest_cs);
	if (ret) {
		v4l2_err(&dev->v4l2_dev, "ipu_ic_task_init failed, %d\n", ret);
		return ret;
	}

	/* init the source MEM-->IC PP IDMAC channel */
	init_idmac_channel(dev, dev->ipu_mem_pp_ch,
			   src->width, src->height,
			   src->stride, src->fmt->fourcc,
			   IPU_ROTATE_NONE, p_in);

	/* init the destination IC PP-->MEM IDMAC channel */
	init_idmac_channel(dev, dev->ipu_pp_mem_ch,
			   dest->width, dest->height,
			   dest->stride, dest->fmt->fourcc,
			   dev->rot_mode, p_out);

	/* enable the IC */
	ipu_ic_enable(dev->ic);

	/* set buffers ready */
	ipu_idmac_select_buffer(dev->ipu_mem_pp_ch, 0);
	ipu_idmac_select_buffer(dev->ipu_pp_mem_ch, 0);

	/* enable the channels! */
	ipu_idmac_enable_channel(dev->ipu_mem_pp_ch);
	ipu_idmac_enable_channel(dev->ipu_pp_mem_ch);

	ipu_ic_task_enable(dev->ic);

	ipu_cpmem_dump(dev->ipu_mem_pp_ch);
	ipu_cpmem_dump(dev->ipu_pp_mem_ch);
	ipu_ic_dump(dev->ic);
	ipu_dump(dev->ipu);

	return 0;
}

/* irqlock held when calling */
static int start_rotate(struct mx6vout_dev *dev,
			dma_addr_t p_in, dma_addr_t p_out)
{
	struct mx6vout_format *src = &dev->src;
	struct mx6vout_format *dest = &dev->dest;
	enum ipu_color_space src_cs, dest_cs;
	int ret;

	src_cs = ipu_pixelformat_to_colorspace(src->fmt->fourcc);
	dest_cs = ipu_pixelformat_to_colorspace(dest->fmt->fourcc);

	/* setup the IC resizer and CSC (swap dest width/height) */
	ret = ipu_ic_task_init(dev->ic,
			       src->width, src->height,
			       dest->height, dest->width,
			       src_cs, dest_cs);
	if (ret) {
		v4l2_err(&dev->v4l2_dev, "ipu_ic_task_init failed, %d\n", ret);
		return ret;
	}

	/* init the source MEM-->IC PP IDMAC channel */
	init_idmac_channel(dev, dev->ipu_mem_pp_ch,
			   src->width, src->height,
			   src->stride, src->fmt->fourcc,
			   IPU_ROTATE_NONE, p_in);

	/* init the IC PP-->MEM IDMAC channel */
	init_idmac_channel(dev, dev->ipu_pp_mem_ch,
			   dest->height, dest->width,
			   dest->rot_stride, dest->fmt->fourcc,
			   IPU_ROTATE_NONE, dev->rot_intermediate.phys);

	/* init the MEM-->IC PP ROT IDMAC channel */
	init_idmac_channel(dev, dev->ipu_mem_rot_pp_ch,
			   dest->height, dest->width,
			   dest->rot_stride, dest->fmt->fourcc,
			   dev->rot_mode, dev->rot_intermediate.phys);

	/* init the destination IC PP ROT-->MEM IDMAC channel */
	init_idmac_channel(dev, dev->ipu_rot_pp_mem_ch,
			   dest->width, dest->height,
			   dest->stride, dest->fmt->fourcc,
			   IPU_ROTATE_NONE, p_out);

	/* now link IC PP-->MEM to MEM-->IC PP ROT */
	ipu_link_pp_rot_pp(dev->ipu);

	/* enable the IC and IRT */
	ipu_ic_enable(dev->ic);
	ipu_irt_enable(dev->irt);

	/* set buffers ready */
	ipu_idmac_select_buffer(dev->ipu_mem_pp_ch, 0);
	ipu_idmac_select_buffer(dev->ipu_pp_mem_ch, 0);
	ipu_idmac_select_buffer(dev->ipu_rot_pp_mem_ch, 0);

	/* enable the channels! */
	ipu_idmac_enable_channel(dev->ipu_mem_pp_ch);
	ipu_idmac_enable_channel(dev->ipu_pp_mem_ch);
	ipu_idmac_enable_channel(dev->ipu_mem_rot_pp_ch);
	ipu_idmac_enable_channel(dev->ipu_rot_pp_mem_ch);

	ipu_ic_task_enable(dev->ic);

	ipu_cpmem_dump(dev->ipu_mem_pp_ch);
	ipu_cpmem_dump(dev->ipu_pp_mem_ch);
	ipu_cpmem_dump(dev->ipu_mem_rot_pp_ch);
	ipu_cpmem_dump(dev->ipu_rot_pp_mem_ch);
	ipu_ic_dump(dev->ic);
	ipu_dump(dev->ipu);

	return 0;
}

/*
 * Start the next frame in the active queue now.
 * irqlock held when calling.
 */
static void start_next_frame(struct mx6vout_dev *dev)
{
	struct mx6vout_format *dest = &dev->dest;
	struct mx6vout_buffer *frame;
	dma_addr_t p_in, p_out;

	frame = list_entry(dev->active_q.next, struct mx6vout_buffer, list);

	p_in = vb2_dma_contig_plane_dma_addr(&frame->vb, 0);
	p_out = (dma_addr_t)dev->fbuf.base;

	if (!p_in || !p_out) {
		v4l2_err(&dev->v4l2_dev,
			 "Acquiring kernel pointers to buffers failed\n");
		return;
	}

#if 0
	/* TODO: double buffering in the DRM overlay plane doesn't work yet */
	if (dev->frame_counter & 1)
		p_out += dest->sizeimage;
#endif
	p_out += dest->stride * dev->crop.top;
	if (dest->fmt->y_depth)
		p_out += ((dev->crop.left * dest->fmt->y_depth) >> 3);
	else
		p_out += ((dev->crop.left * dest->fmt->depth) >> 3);

	if (dev->rot_mode >= IPU_ROTATE_90_RIGHT)
		start_rotate(dev, p_in, p_out);
	else
		start_norotate(dev, p_in, p_out);
}

/* timer function to start a frame at the right timestamp */
static void start_frame_timer_func(unsigned long data)
{
	struct mx6vout_dev *dev = (struct mx6vout_dev *)data;
	unsigned long flags;

	spin_lock_irqsave(&dev->irqlock, flags);
	start_next_frame(dev);
	spin_unlock_irqrestore(&dev->irqlock, flags);
}

/* irqlock held when calling */
static void schedule_next_frame(struct mx6vout_dev *dev)
{
	struct mx6vout_buffer *frame;
	unsigned long next_frame_j, cur_j;
	struct timeval cur_time, next_frame_time;

	frame = list_entry(dev->active_q.next, struct mx6vout_buffer, list);
	next_frame_time = frame->vb.v4l2_buf.timestamp;

	do_gettimeofday(&cur_time);

	if (next_frame_time.tv_sec == 0 &&
	    next_frame_time.tv_usec == 0)
		next_frame_time = cur_time;

	next_frame_j = timeval_to_jiffies(&next_frame_time);
	cur_j = timeval_to_jiffies(&cur_time);

	if (next_frame_j <= cur_j) {
		start_next_frame(dev);
	} else {
		next_frame_j -= cur_j;
		next_frame_j += jiffies;
		mod_timer(&dev->start_frame_timer, next_frame_j);
	}
}

/* irqlock held when calling */
static void stop_norotate(struct mx6vout_dev *dev)
{
	/* disable IC tasks and the channels */
	ipu_ic_task_disable(dev->ic);
	ipu_idmac_disable_channel(dev->ipu_mem_pp_ch);
	ipu_idmac_disable_channel(dev->ipu_pp_mem_ch);

	ipu_ic_disable(dev->ic);
}

/* irqlock held when calling */
static void stop_rotate(struct mx6vout_dev *dev)
{
	/* disable IC tasks and the channels */
	ipu_ic_task_disable(dev->ic);
	ipu_idmac_disable_channel(dev->ipu_mem_pp_ch);
	ipu_idmac_disable_channel(dev->ipu_pp_mem_ch);
	ipu_idmac_disable_channel(dev->ipu_mem_rot_pp_ch);
	ipu_idmac_disable_channel(dev->ipu_rot_pp_mem_ch);

	ipu_unlink_pp_rot_pp(dev->ipu);

	ipu_ic_disable(dev->ic);
	ipu_irt_disable(dev->irt);
}

/* irqlock held when calling */
static void mx6vout_doirq(struct mx6vout_dev *dev)
{
	struct mx6vout_buffer *frame;
	struct timeval cur_time;

	if (!list_empty(&dev->active_q)) {
		do_gettimeofday(&cur_time);
		frame = list_entry(dev->active_q.next,
				   struct mx6vout_buffer, list);
		frame->vb.v4l2_buf.timestamp = cur_time;
		vb2_buffer_done(&frame->vb, VB2_BUF_STATE_DONE);
		list_del(&frame->list);

		dev->frame_counter++;
	}

	if (dev->stop)
		complete(&dev->comp);
	else if (!list_empty(&dev->active_q))
		schedule_next_frame(dev);
}

static irqreturn_t mx6vout_norotate_irq(int irq, void *data)
{
	struct mx6vout_dev *dev = (struct mx6vout_dev *)data;
	unsigned long flags;

	spin_lock_irqsave(&dev->irqlock, flags);

	if (dev->rot_mode >= IPU_ROTATE_90_RIGHT) {
		/*
		 * this is a rotation operation, this is expected,
		 * just ignore and let the rotation irq handler deal
		 * with it
		 */
		spin_unlock_irqrestore(&dev->irqlock, flags);
		return IRQ_HANDLED;
	}

	stop_norotate(dev);

	mx6vout_doirq(dev);

	spin_unlock_irqrestore(&dev->irqlock, flags);

	return IRQ_HANDLED;
}

static irqreturn_t mx6vout_rotate_irq(int irq, void *data)
{
	struct mx6vout_dev *dev = (struct mx6vout_dev *)data;
	unsigned long flags;

	spin_lock_irqsave(&dev->irqlock, flags);

	stop_rotate(dev);

	if (dev->rot_mode < IPU_ROTATE_90_RIGHT) {
		/* this was NOT a rotation operation, shouldn't happen */
		v4l2_err(&dev->v4l2_dev, "Unexpected rotation interrupt\n");
		spin_unlock_irqrestore(&dev->irqlock, flags);
		return IRQ_HANDLED;
	}

	mx6vout_doirq(dev);

	spin_unlock_irqrestore(&dev->irqlock, flags);

	return IRQ_HANDLED;
}

static void mx6vout_release_ipu_resources(struct mx6vout_dev *dev)
{
	if (dev->pp_mem_irq >= 0)
		devm_free_irq(dev->v4l2_dev.dev, dev->pp_mem_irq, dev);
	if (dev->rot_pp_mem_irq >= 0)
		devm_free_irq(dev->v4l2_dev.dev, dev->rot_pp_mem_irq, dev);

	if (!IS_ERR_OR_NULL(dev->ipu_mem_pp_ch))
		ipu_idmac_put(dev->ipu_mem_pp_ch);
	if (!IS_ERR_OR_NULL(dev->ipu_pp_mem_ch))
		ipu_idmac_put(dev->ipu_pp_mem_ch);
	if (!IS_ERR_OR_NULL(dev->ipu_mem_rot_pp_ch))
		ipu_idmac_put(dev->ipu_mem_rot_pp_ch);
	if (!IS_ERR_OR_NULL(dev->ipu_rot_pp_mem_ch))
		ipu_idmac_put(dev->ipu_rot_pp_mem_ch);

	if (!IS_ERR_OR_NULL(dev->ic))
		ipu_ic_put(dev->ic);
	if (!IS_ERR_OR_NULL(dev->irt))
		ipu_irt_put(dev->irt);

	dev->ipu_mem_pp_ch = dev->ipu_pp_mem_ch = dev->ipu_mem_rot_pp_ch =
		dev->ipu_rot_pp_mem_ch = NULL;
	dev->ic = NULL;
	dev->irt = NULL;
	dev->pp_mem_irq = dev->rot_pp_mem_irq = -1;
}

static int mx6vout_get_ipu_resources(struct mx6vout_dev *dev,
				     bool nonblocking)
{
	int ret;

	dev->ic = ipu_ic_get(dev->ipu, IC_TASK_POST_PROCESSOR);
	if (IS_ERR(dev->ic)) {
		v4l2_err(&dev->v4l2_dev, "could not get IC PP\n");
		ret = PTR_ERR(dev->ic);
		goto err;
	}

	dev->irt = ipu_irt_get(dev->ipu);
	if (IS_ERR(dev->irt)) {
		v4l2_err(&dev->v4l2_dev, "could not get IRT\n");
		ret = PTR_ERR(dev->irt);
		goto err;
	}

	/* get our IDMAC channels */
	dev->ipu_mem_pp_ch = ipu_idmac_get(dev->ipu,
					   IPUV3_CHANNEL_MEM_IC_PP,
					   !nonblocking);
	dev->ipu_pp_mem_ch = ipu_idmac_get(dev->ipu,
					   IPUV3_CHANNEL_IC_PP_MEM,
					   !nonblocking);
	dev->ipu_mem_rot_pp_ch = ipu_idmac_get(dev->ipu,
					       IPUV3_CHANNEL_MEM_ROT_PP,
					       !nonblocking);
	dev->ipu_rot_pp_mem_ch = ipu_idmac_get(dev->ipu,
					       IPUV3_CHANNEL_ROT_PP_MEM,
					       !nonblocking);
	if (IS_ERR(dev->ipu_mem_pp_ch) ||
	    IS_ERR(dev->ipu_pp_mem_ch) ||
	    IS_ERR(dev->ipu_mem_rot_pp_ch) ||
	    IS_ERR(dev->ipu_rot_pp_mem_ch)) {
		v4l2_err(&dev->v4l2_dev, "could not acquire IDMAC channels\n");
		ret = nonblocking ? -EBUSY : -ERESTARTSYS;
		goto err;
	}

	/* acquire the EOF interrupts */
	dev->pp_mem_irq = ipu_idmac_channel_irq(dev->ipu,
						dev->ipu_pp_mem_ch,
						IPU_IRQ_EOF);
	dev->rot_pp_mem_irq = ipu_idmac_channel_irq(dev->ipu,
						    dev->ipu_rot_pp_mem_ch,
						    IPU_IRQ_EOF);

	ret = devm_request_irq(dev->v4l2_dev.dev, dev->pp_mem_irq,
			       mx6vout_norotate_irq, 0, DEVICE_NAME, dev);
	if (ret < 0) {
		v4l2_err(&dev->v4l2_dev, "could not acquire irq %d\n",
			 dev->pp_mem_irq);
		goto err;
	}

	ret = devm_request_irq(dev->v4l2_dev.dev, dev->rot_pp_mem_irq,
			       mx6vout_rotate_irq, 0, DEVICE_NAME, dev);
	if (ret < 0) {
		v4l2_err(&dev->v4l2_dev, "could not acquire irq %d\n",
			 dev->rot_pp_mem_irq);
		goto err;
	}

	return 0;
err:
	mx6vout_release_ipu_resources(dev);
	return ret;
}

static void free_rot_intermediate_buffer(struct mx6vout_dev *dev)
{
	if (dev->rot_intermediate.virt) {
		dma_free_coherent(dev->v4l2_dev.dev,
				  dev->rot_intermediate.len,
				  dev->rot_intermediate.virt,
				  dev->rot_intermediate.phys);
		dev->rot_intermediate.virt = NULL;
	}
}

static int alloc_rot_intermediate_buffer(struct mx6vout_dev *dev)
{
	struct mx6vout_format *dest = &dev->dest;
	unsigned long newlen = PAGE_ALIGN(dest->sizeimage);

	if (dev->rot_intermediate.virt) {
		if (dev->rot_intermediate.len == newlen)
			return 0;

		free_rot_intermediate_buffer(dev);
	}

	dev->rot_intermediate.len = newlen;
	dev->rot_intermediate.virt =
		(void *)dma_alloc_coherent(dev->v4l2_dev.dev,
					   dev->rot_intermediate.len,
					   &dev->rot_intermediate.phys,
					   GFP_DMA | GFP_KERNEL);
	if (!dev->rot_intermediate.virt) {
		v4l2_err(&dev->v4l2_dev, "failed to alloc rotation "	\
			 "intermedia buffer\n");
		return -ENOMEM;
	}

	return 0;
}

/*
 * video ioctls
 */
static int vidioc_querycap(struct file *file, void *priv,
			   struct v4l2_capability *cap)
{
	strncpy(cap->driver, DEVICE_NAME, sizeof(cap->driver) - 1);
	strncpy(cap->card, DEVICE_NAME, sizeof(cap->card) - 1);
	cap->bus_info[0] = 0;
	cap->capabilities = V4L2_CAP_STREAMING |
		V4L2_CAP_VIDEO_OUTPUT |
		V4L2_CAP_VIDEO_OUTPUT_OVERLAY;

	return 0;
}

static int vidioc_enum_fmt_vid_out(struct file *file, void *priv,
				   struct v4l2_fmtdesc *f)
{
	struct mx6vout_pixfmt *fmt;
	int i, num = 0;

	for (i = 0; i < NUM_FORMATS; ++i) {
		/* index-th format of type type found ? */
		if (num == f->index)
			break;
		/* Correct type but haven't reached our index yet,
		 * just increment per-type index */
		++num;
	}

	if (i < NUM_FORMATS) {
		/* Format found */
		fmt = &mx6vout_pixformats[i];
		strncpy(f->description, fmt->name, sizeof(f->description) - 1);
		f->pixelformat = fmt->fourcc;
		return 0;
	}

	/* Format not found */
	return -EINVAL;
}


static int vidioc_g_fmt_vid_out(struct file *file, void *priv,
				struct v4l2_format *f)
{
	struct mx6vout_dev *dev = video_drvdata(file);

	f->fmt.pix = dev->pix;
	return 0;
}

static int vidioc_try_fmt_vid_out(struct file *file, void *priv,
				  struct v4l2_format *f)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	unsigned int width_align;
	enum v4l2_field field;
	struct mx6vout_pixfmt *fmt;

	fmt = mx6vout_get_format(&f->fmt.pix);
	if (!fmt) {
		v4l2_err(&dev->v4l2_dev,
			 "Fourcc format (0x%08x) invalid.\n",
			 f->fmt.pix.pixelformat);
		return -EINVAL;
	}

	field = f->fmt.pix.field;

	if (field == V4L2_FIELD_ANY)
		field = V4L2_FIELD_NONE;
	else if (V4L2_FIELD_NONE != field) {
		v4l2_err(&dev->v4l2_dev, "Interlaced not supported\n");
		return -EINVAL;
	}

	/* V4L2 specification suggests the driver corrects the format struct
	 * if any of the dimensions is unsupported */
	f->fmt.pix.field = field;

	/*
	 * We have to adjust the width such that the physaddrs and U and
	 * U and V plane offsets are multiples of 8 bytes as required by
	 * the IPU DMA Controller. For the planar formats, this corresponds
	 * to a pixel alignment of 16. For all the packed formats, 8 is
	 * good enough.
	 */
	width_align = fmt->y_depth ? 4 : 3;

	v4l_bound_align_image(&f->fmt.pix.width, MIN_W, MAX_W,
			      width_align, &f->fmt.pix.height,
			      MIN_H, MAX_H, H_ALIGN, S_ALIGN);

	f->fmt.pix.bytesperline = (f->fmt.pix.width * fmt->depth) >> 3;
	f->fmt.pix.sizeimage = f->fmt.pix.height * f->fmt.pix.bytesperline;

	return 0;
}

static int vidioc_s_fmt_vid_out(struct file *file, void *priv,
				struct v4l2_format *f)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct mx6vout_format *src = &dev->src;
	int ret;

	ret = vidioc_try_fmt_vid_out(file, priv, f);
	if (ret)
		return ret;

	if (vb2_is_busy(&dev->buffer_queue)) {
		v4l2_err(&dev->v4l2_dev, "%s queue busy\n", __func__);
		return -EBUSY;
	}

	src->fmt          = mx6vout_get_format(&f->fmt.pix);
	src->width        = f->fmt.pix.width;
	src->height       = f->fmt.pix.height;
	src->bytesperline = f->fmt.pix.bytesperline;
	if (src->fmt->y_depth)
		src->stride  = (src->fmt->y_depth * src->width) >> 3;
	else
		src->stride  = src->bytesperline;

	src->sizeimage = src->bytesperline * src->height;

	dev->pix = f->fmt.pix;

	dprintk(dev, "%dx%d, %s\n", src->width, src->height, src->fmt->name);
	return 0;
}

static int vidioc_enum_fmt_vid_out_overlay(struct file *file, void *priv,
					   struct v4l2_fmtdesc *fmt)
{
	return vidioc_enum_fmt_vid_out(file, priv, fmt);
}

static int vidioc_g_fmt_vid_out_overlay(struct file *file, void *priv,
					struct v4l2_format *f)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct v4l2_window *win = &f->fmt.win;

	win->w            = dev->win.w;
	win->global_alpha = dev->win.global_alpha;
	win->chromakey    = dev->win.chromakey;
	win->field        = dev->win.field;

	return 0;
}

static int vidioc_try_fmt_vid_out_overlay(struct file *file, void *priv,
					  struct v4l2_format *f)
{
	struct v4l2_window *win = &f->fmt.win;

	win->w.top = win->w.left = 0;
	win->global_alpha = 255; /* TODO */
	win->chromakey    = 0;   /* TODO */
	win->field        = V4L2_FIELD_NONE; /* TODO: interlaced */

	return 0;
}

static int vidioc_s_fmt_vid_out_overlay(struct file *file, void *priv,
					struct v4l2_format *f)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct v4l2_window *win = &f->fmt.win;
	int ret;

	if (vb2_is_streaming(&dev->buffer_queue)) {
		v4l2_err(&dev->v4l2_dev, "%s: not allowed while streaming\n",
			 __func__);
		return -EBUSY;
	}

	ret = vidioc_try_fmt_vid_out_overlay(file, priv, f);
	if (ret)
		return ret;

	dev->win = *win;

	return 0;
}

static int vidioc_cropcap(struct file *file, void *priv,
			  struct v4l2_cropcap *cropcap)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct v4l2_rect *defrect = &cropcap->defrect;
	struct v4l2_rect *bounds = &cropcap->bounds;

	if (cropcap->type != V4L2_BUF_TYPE_VIDEO_OUTPUT)
		return -EINVAL;

	bounds->width  = defrect->width  = dev->fbuf.fmt.width;
	bounds->height = defrect->height = dev->fbuf.fmt.height;
	defrect->left = defrect->top = 0;

	cropcap->pixelaspect.numerator = 1;
	cropcap->pixelaspect.denominator = 1;
	return 0;
}

static int vidioc_g_crop(struct file *file, void *priv,
			 struct v4l2_crop *crop)
{
	struct mx6vout_dev *dev = video_drvdata(file);

	if (crop->type != V4L2_BUF_TYPE_VIDEO_OUTPUT)
		return -EINVAL;
	crop->c = dev->crop;
	return 0;
}

static int vidioc_s_crop(struct file *file, void *priv,
			 const struct v4l2_crop *crop)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct mx6vout_format *dest = &dev->dest;
	struct v4l2_rect *ovrect = (struct v4l2_rect *)&crop->c;
	unsigned int width_align;

	if (crop->type != V4L2_BUF_TYPE_VIDEO_OUTPUT)
		return -EINVAL;

	if (vb2_is_streaming(&dev->buffer_queue)) {
		v4l2_err(&dev->v4l2_dev, "%s: not allowed while streaming\n",
			 __func__);
		return -EBUSY;
	}

	if (!dest->fmt) {
		v4l2_err(&dev->v4l2_dev,
			 "%s: call S_FBUF first to set overlay "	\
			 "framebuffer config\n", __func__);
		return -EAGAIN;
	}

	/*
	 * We have to adjust the width such that the physaddrs and U and
	 * U and V plane offsets are multiples of 8 bytes as required by
	 * the IPU DMA Controller. For the planar formats, this corresponds
	 * to a pixel alignment of 16. For all the packed formats, 8 is
	 * good enough.
	 */
	width_align = dest->fmt->y_depth ? 4 : 3;

	/*
	 * first align to framebuffer dimensions, then to IPU IC resizer
	 * restrictions.
	 */
	v4l_bound_align_image(&ovrect->width, MIN_W, dev->fbuf.fmt.width,
			      width_align,
			      &ovrect->height, MIN_H, dev->fbuf.fmt.height,
			      H_ALIGN, S_ALIGN);

	v4l_bound_align_image(&ovrect->width, MIN_W, MAX_W_CROP,
			      width_align, &ovrect->height,
			      MIN_H, MAX_H_CROP, H_ALIGN, S_ALIGN);

	/* now make sure position is within fbuf bounds */
	if (ovrect->top < 0)
		ovrect->top = 0;
	if (ovrect->left < 0)
		ovrect->left = 0;
	if (ovrect->left + ovrect->width > dev->fbuf.fmt.width)
		ovrect->left = dev->fbuf.fmt.width - ovrect->width;
	if (ovrect->top + ovrect->height > dev->fbuf.fmt.height)
		ovrect->top = dev->fbuf.fmt.height - ovrect->height;

	dest->width  = ovrect->width;
	dest->height = ovrect->height;

	dest->bytesperline = dev->fbuf.fmt.bytesperline;
	if (dest->fmt->y_depth) {
		dest->stride  = (dest->fmt->y_depth *
				 dev->fbuf.fmt.width) >> 3;
		dest->rot_stride = (dest->fmt->y_depth * dest->height) >> 3;
	} else {
		dest->stride  = dest->bytesperline;
		dest->rot_stride = (dest->fmt->depth * dest->height) >> 3;
	}

	dest->sizeimage = (dest->width * dest->height * dest->fmt->depth) >> 3;

	dev->crop = *ovrect;

	dprintk(dev, "%dx%d, %s\n", dest->width, dest->height, dest->fmt->name);

	return 0;
}

static int vidioc_s_fbuf(struct file *file, void *priv,
			 const struct v4l2_framebuffer *fbuf)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct mx6vout_format *dest = &dev->dest;
	struct mx6vout_pixfmt *fmt;

	if (fbuf->flags != V4L2_FBUF_FLAG_OVERLAY || !fbuf->base)
		return -EINVAL;

	if (vb2_is_streaming(&dev->buffer_queue)) {
		v4l2_err(&dev->v4l2_dev, "%s: not allowed while streaming\n",
			 __func__);
		return -EBUSY;
	}

	fmt = mx6vout_get_format((struct v4l2_pix_format *)&fbuf->fmt);
	if (!fmt) {
		v4l2_err(&dev->v4l2_dev,
			 "Fourcc format (0x%08x) invalid.\n",
			 fbuf->fmt.pixelformat);
		return -EINVAL;
	}
	dest->fmt = fmt;

	dev->global_alpha_enable =
		(fbuf->flags & V4L2_FBUF_FLAG_GLOBAL_ALPHA) != 0;
	dev->chromakey_enable =
		(fbuf->flags & V4L2_FBUF_FLAG_CHROMAKEY) != 0;

	dev->fbuf = *fbuf;

	return 0;
}

static int vidioc_g_fbuf(struct file *file, void *priv,
			 struct v4l2_framebuffer *fbuf)
{
	struct mx6vout_dev *dev = video_drvdata(file);

	fbuf->capability = V4L2_FBUF_CAP_CHROMAKEY |
		V4L2_FBUF_CAP_LOCAL_ALPHA |
		V4L2_FBUF_CAP_GLOBAL_ALPHA;

	fbuf->flags = V4L2_FBUF_FLAG_OVERLAY;
	if (dev->global_alpha_enable)
		fbuf->flags |= V4L2_FBUF_FLAG_GLOBAL_ALPHA;
	if (dev->chromakey_enable)
		fbuf->flags |= V4L2_FBUF_FLAG_CHROMAKEY;

	fbuf->base = dev->fbuf.base;
	fbuf->fmt = dev->fbuf.fmt;

	return 0;
}

static int vidioc_reqbufs(struct file *file, void *priv,
			  struct v4l2_requestbuffers *reqbufs)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct vb2_queue *vq = &dev->buffer_queue;

	if (vb2_is_busy(vq))
		return -EBUSY;

	return vb2_reqbufs(vq, reqbufs);
}

static int vidioc_querybuf(struct file *file, void *priv,
			   struct v4l2_buffer *buf)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct vb2_queue *vq = &dev->buffer_queue;

	return vb2_querybuf(vq, buf);
}

static long vidioc_default(struct file *file, void *fh, bool valid_prio,
			   int cmd, void *arg)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct vb2_queue *vq = &dev->buffer_queue;
	struct v4l2_buffer *buf;
	struct vb2_buffer *vb;

	switch (cmd) {
	case VIDIOC_IMX6_GET_PHYS:
		/*
		 * returns the physical address of a v4l2 buffer in
		 * buf->m.offset. The mfw_v4lsink gstreamer plugin uses
		 * this to share h/w buffers with upstream elements,
		 * such as vpudec. Note that for USERPTR type buffers,
		 * a valid physaddr will only be available after it is
		 * queued!
		 */
		buf = (struct v4l2_buffer *)arg;
		if (buf->type != vq->type) {
			v4l2_err(&dev->v4l2_dev,
				 "%s: wrong buffer type\n", __func__);
			return -EINVAL;
		}

		if (buf->index >= vq->num_buffers) {
			v4l2_err(&dev->v4l2_dev,
				 "%s: buffer index out of range\n", __func__);
			return -EINVAL;
		}

		vb = vq->bufs[buf->index];
		buf->m.offset = vb2_dma_contig_plane_dma_addr(vb, 0);
		break;
	default:
		return -ENOTTY;
	}

	return 0;
}

static int vidioc_qbuf(struct file *file, void *priv, struct v4l2_buffer *buf)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct vb2_queue *vq = &dev->buffer_queue;

	return vb2_qbuf(vq, buf);
}

static int vidioc_dqbuf(struct file *file, void *priv, struct v4l2_buffer *buf)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct vb2_queue *vq = &dev->buffer_queue;

	return vb2_dqbuf(vq, buf, file->f_flags & O_NONBLOCK);
}

static int vidioc_expbuf(struct file *file, void *priv,
			 struct v4l2_exportbuffer *eb)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct vb2_queue *vq = &dev->buffer_queue;

	return vb2_expbuf(vq, eb);
}

static int vidioc_streamon(struct file *file, void *priv,
			   enum v4l2_buf_type type)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct vb2_queue *vq = &dev->buffer_queue;
	int ret;

	ret = mx6vout_get_ipu_resources(dev, file->f_flags & O_NONBLOCK);
	if (ret < 0)
		return ret;

	return vb2_streamon(vq, type);
}

static int vidioc_streamoff(struct file *file, void *priv,
			    enum v4l2_buf_type type)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct vb2_queue *vq = &dev->buffer_queue;
	int ret;

	ret = vb2_streamoff(vq, type);

	return ret;
}

static int vidioc_queryctrl(struct file *file, void *priv,
			    struct v4l2_queryctrl *qc)
{
	struct v4l2_queryctrl *c;

	c = mx6vout_get_ctrl(qc->id);
	if (!c)
		return -EINVAL;

	*qc = *c;
	return 0;
}

static int vidioc_g_ctrl(struct file *file, void *priv,
			 struct v4l2_control *ctrl)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	int ret = 0;

	switch (ctrl->id) {
	case V4L2_CID_HFLIP:
		ctrl->value = dev->hflip ? 1 : 0;
		break;
	case V4L2_CID_VFLIP:
		ctrl->value = dev->vflip ? 1 : 0;
		break;
	case V4L2_CID_ROTATE:
		ctrl->value = dev->rotation;
		break;
	default:
		v4l2_err(&dev->v4l2_dev, "Invalid control\n");
		ret = -EINVAL;
		break;
	}

	return ret;
}

static int check_ctrl_val(struct mx6vout_dev *dev, struct v4l2_control *ctrl)
{
	struct v4l2_queryctrl *c;

	c = mx6vout_get_ctrl(ctrl->id);
	if (!c)
		return -EINVAL;

	if (ctrl->value < c->minimum || ctrl->value > c->maximum) {
		v4l2_err(&dev->v4l2_dev, "Value out of range\n");
		return -ERANGE;
	}

	return 0;
}

static int vidioc_s_ctrl(struct file *file, void *priv,
			 struct v4l2_control *ctrl)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	enum ipu_rotate_mode rot_mode;
	bool hflip, vflip;
	int rotation;
	int ret = 0;

	/* can't change rotation mid-streaming */
	if (vb2_is_streaming(&dev->buffer_queue)) {
		v4l2_err(&dev->v4l2_dev, "%s: not allowed while streaming\n",
			 __func__);
		return -EBUSY;
	}

	ret = check_ctrl_val(dev, ctrl);
	if (ret != 0)
		return ret;

	rotation = dev->rotation;
	hflip = dev->hflip;
	vflip = dev->vflip;

	switch (ctrl->id) {
	case V4L2_CID_HFLIP:
		hflip = (ctrl->value == 1);
		break;
	case V4L2_CID_VFLIP:
		vflip = (ctrl->value == 1);
		break;
	case V4L2_CID_ROTATE:
		rotation = ctrl->value;
		break;
	default:
		v4l2_err(&dev->v4l2_dev, "Invalid control\n");
		return -EINVAL;
	}

	ret = ipu_degrees_to_rot_mode(&rot_mode, rotation, hflip, vflip);
	if (ret)
		return ret;

	dev->rotation = rotation;
	dev->hflip = hflip;
	dev->vflip = vflip;
	dev->rot_mode = rot_mode;

	return 0;
}


static const struct v4l2_ioctl_ops mx6vout_ioctl_ops = {
	.vidioc_querycap	= vidioc_querycap,

	.vidioc_enum_fmt_vid_out        = vidioc_enum_fmt_vid_out,
	.vidioc_g_fmt_vid_out           = vidioc_g_fmt_vid_out,
	.vidioc_try_fmt_vid_out         = vidioc_try_fmt_vid_out,
	.vidioc_s_fmt_vid_out           = vidioc_s_fmt_vid_out,

	.vidioc_enum_fmt_vid_overlay    = vidioc_enum_fmt_vid_out_overlay,
	.vidioc_g_fmt_vid_out_overlay	= vidioc_g_fmt_vid_out_overlay,
	.vidioc_try_fmt_vid_out_overlay	= vidioc_try_fmt_vid_out_overlay,
	.vidioc_s_fmt_vid_out_overlay	= vidioc_s_fmt_vid_out_overlay,

	.vidioc_s_fbuf          = vidioc_s_fbuf,
	.vidioc_g_fbuf          = vidioc_g_fbuf,

	.vidioc_cropcap         = vidioc_cropcap,
	.vidioc_g_crop          = vidioc_g_crop,
	.vidioc_s_crop          = vidioc_s_crop,

	.vidioc_queryctrl	= vidioc_queryctrl,
	.vidioc_g_ctrl		= vidioc_g_ctrl,
	.vidioc_s_ctrl		= vidioc_s_ctrl,

	.vidioc_reqbufs		= vidioc_reqbufs,
	.vidioc_querybuf	= vidioc_querybuf,
	.vidioc_qbuf		= vidioc_qbuf,
	.vidioc_dqbuf		= vidioc_dqbuf,
	.vidioc_expbuf		= vidioc_expbuf,

	.vidioc_streamon	= vidioc_streamon,
	.vidioc_streamoff	= vidioc_streamoff,

	.vidioc_default         = vidioc_default,
};


/*
 * Queue operations
 */

static int mx6vout_queue_setup(struct vb2_queue *vq,
				const struct v4l2_format *fmt,
				unsigned int *nbuffers, unsigned int *nplanes,
				unsigned int sizes[], void *alloc_ctxs[])
{
	struct mx6vout_dev *dev = vb2_get_drv_priv(vq);
	struct mx6vout_format *src = &dev->src;
	unsigned int count = *nbuffers;

	if (vq->type != V4L2_BUF_TYPE_VIDEO_OUTPUT)
		return -EINVAL;

	while (src->sizeimage * count > VID_MEM_LIMIT)
		count--;

	*nplanes = 1;
	*nbuffers = count;
	sizes[0] = src->sizeimage;

	alloc_ctxs[0] = dev->alloc_ctx;

	dprintk(dev, "get %d buffer(s) of size %d each.\n",
		count, src->sizeimage);

	return 0;
}

static int mx6vout_buf_init(struct vb2_buffer *vb)
{
	struct mx6vout_buffer *buf = to_mx6vout_vb(vb);
	INIT_LIST_HEAD(&buf->list);
	return 0;
}

static int mx6vout_buf_prepare(struct vb2_buffer *vb)
{
	struct mx6vout_dev *dev = vb2_get_drv_priv(vb->vb2_queue);
	struct mx6vout_format *src = &dev->src;

	if (vb2_plane_size(vb, 0) < src->sizeimage) {
		v4l2_err(&dev->v4l2_dev,
			 "data will not fit into plane (%lu < %lu)\n",
			 vb2_plane_size(vb, 0), (long)src->sizeimage);
		return -EINVAL;
	}

	vb2_set_plane_payload(vb, 0, src->sizeimage);

	return 0;
}

static void mx6vout_buf_queue(struct vb2_buffer *vb)
{
	struct mx6vout_dev *dev = vb2_get_drv_priv(vb->vb2_queue);
	struct mx6vout_buffer *buf = to_mx6vout_vb(vb);
	unsigned long flags;

	spin_lock_irqsave(&dev->irqlock, flags);

	list_add_tail(&buf->list, &dev->active_q);

	/* kickstart DMA chain if the active queue underrun */
	if (vb2_is_streaming(vb->vb2_queue) &&
	    list_is_singular(&dev->active_q))
		schedule_next_frame(dev);

	spin_unlock_irqrestore(&dev->irqlock, flags);
}

static void mx6vout_lock(struct vb2_queue *vq)
{
	struct mx6vout_dev *dev = vb2_get_drv_priv(vq);
	mutex_lock(&dev->mutex);
}

static void mx6vout_unlock(struct vb2_queue *vq)
{
	struct mx6vout_dev *dev = vb2_get_drv_priv(vq);
	mutex_unlock(&dev->mutex);
}

static int mx6vout_start_streaming(struct vb2_queue *vq, unsigned int count)
{
	struct mx6vout_dev *dev = vb2_get_drv_priv(vq);
	unsigned long flags;
	int ret;

	ret = alloc_rot_intermediate_buffer(dev);
	if (ret)
		return ret;

	INIT_COMPLETION(dev->comp);
	dev->stop = false;

	spin_lock_irqsave(&dev->irqlock, flags);

	dev->frame_counter = 0;
	dev->streamon_jiffies = jiffies;

	if (!list_empty(&dev->active_q))
		schedule_next_frame(dev);

	spin_unlock_irqrestore(&dev->irqlock, flags);
	return ret;
}

static int mx6vout_stop_streaming(struct vb2_queue *vq)
{
	struct mx6vout_dev *dev = vb2_get_drv_priv(vq);
	struct mx6vout_buffer *frame;
	unsigned long timeout, flags;

	if (!vb2_is_streaming(vq))
		return 0;

	spin_lock_irqsave(&dev->irqlock, flags);

	dev->stop = true;

	if (!list_empty(&dev->active_q)) {
		spin_unlock_irqrestore(&dev->irqlock, flags);
		timeout = wait_for_completion_timeout(&dev->comp,
						      msecs_to_jiffies(1000));
		spin_lock_irqsave(&dev->irqlock, flags);

		if (timeout == 0) {
			v4l2_err(&dev->v4l2_dev, "stop timeout\n");
			if (dev->rot_mode >= IPU_ROTATE_90_RIGHT)
				stop_rotate(dev);
			else
				stop_norotate(dev);
		}

		/* release any remaining buffers */
		while (!list_empty(&dev->active_q)) {
			frame = list_entry(dev->active_q.next,
					   struct mx6vout_buffer, list);
			list_del(&frame->list);
			vb2_buffer_done(&frame->vb, VB2_BUF_STATE_ERROR);
		}
	}

	spin_unlock_irqrestore(&dev->irqlock, flags);

	mx6vout_release_ipu_resources(dev);
	free_rot_intermediate_buffer(dev);
	return 0;
}

static struct vb2_ops mx6vout_qops = {
	.queue_setup	 = mx6vout_queue_setup,
	.buf_init        = mx6vout_buf_init,
	.buf_prepare	 = mx6vout_buf_prepare,
	.buf_queue	 = mx6vout_buf_queue,
	.wait_prepare	 = mx6vout_unlock,
	.wait_finish	 = mx6vout_lock,
	.start_streaming = mx6vout_start_streaming,
	.stop_streaming  = mx6vout_stop_streaming,
};

/*
 * File operations
 */
static int mx6vout_open(struct file *file)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	int ret = 0;

	if (mutex_lock_interruptible(&dev->mutex))
		return -ERESTARTSYS;

	ret = v4l2_fh_open(file);
	if (ret) {
		v4l2_err(&dev->v4l2_dev, "v4l2_fh_open failed\n");
		goto unlock;
	}

	if (!v4l2_fh_is_singular_file(file))
		goto unlock;

	ret = vb2_queue_init(&dev->buffer_queue);
	if (ret) {
		v4l2_err(&dev->v4l2_dev, "vb2_queue_init failed\n");
		goto unlock;
	}

	INIT_LIST_HEAD(&dev->active_q);
unlock:
	mutex_unlock(&dev->mutex);
	return ret;
}

static int mx6vout_release(struct file *file)
{
	struct mx6vout_dev *dev = video_drvdata(file);

	mutex_lock(&dev->mutex);

	if (v4l2_fh_is_singular_file(file))
		vb2_queue_release(&dev->buffer_queue);

	v4l2_fh_release(file);

	mutex_unlock(&dev->mutex);
	return 0;
}

static unsigned int mx6vout_poll(struct file *file,
				 struct poll_table_struct *wait)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct vb2_queue *vq = &dev->buffer_queue;
	int ret;

	if (mutex_lock_interruptible(&dev->mutex))
		return -ERESTARTSYS;

	ret = vb2_poll(vq, file, wait);

	mutex_unlock(&dev->mutex);
	return ret;
}

static int mx6vout_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct mx6vout_dev *dev = video_drvdata(file);
	struct vb2_queue *vq = &dev->buffer_queue;
	int ret;

	if (mutex_lock_interruptible(&dev->mutex))
		return -ERESTARTSYS;

	ret = vb2_mmap(vq, vma);

	mutex_unlock(&dev->mutex);
	return ret;
}

static const struct v4l2_file_operations mx6vout_fops = {
	.owner		= THIS_MODULE,
	.open		= mx6vout_open,
	.release	= mx6vout_release,
	.poll		= mx6vout_poll,
	.unlocked_ioctl	= video_ioctl2,
	.mmap		= mx6vout_mmap,
};

static struct video_device mx6vout_videodev = {
	.name		= DEVICE_NAME,
	.fops		= &mx6vout_fops,
	.ioctl_ops	= &mx6vout_ioctl_ops,
	.minor		= -1,
	.release	= video_device_release,
	.vfl_dir	= VFL_DIR_TX,
};

static int mx6vout_probe(struct platform_device *pdev)
{
	struct mx6vout_dev *dev;
	struct video_device *vfd;
	struct vb2_queue *vq;
	int ret;

	dev = kzalloc(sizeof *dev, GFP_KERNEL);
	if (!dev)
		return -ENOMEM;

	spin_lock_init(&dev->irqlock);

	ret = v4l2_device_register(&pdev->dev, &dev->v4l2_dev);
	if (ret)
		goto free_dev;

	pdev->dev.coherent_dma_mask = DMA_BIT_MASK(32);

	/* get our parent IPU and the Image Converter */
	dev->ipu = dev_get_drvdata(pdev->dev.parent);
	if (IS_ERR_OR_NULL(dev->ipu)) {
		v4l2_err(&dev->v4l2_dev, "could not get parent ipu\n");
		ret = -ENODEV;
		goto unreg_dev;
	}

	dev->pp_mem_irq = dev->rot_pp_mem_irq = -1;

	mutex_init(&dev->mutex);
	init_completion(&dev->comp);

	init_timer(&dev->start_frame_timer);
	dev->start_frame_timer.function = start_frame_timer_func;
	dev->start_frame_timer.data     = (unsigned long)dev;

	vq = &dev->buffer_queue;
	vq->type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
	vq->io_modes = VB2_MMAP | VB2_USERPTR | VB2_DMABUF;
	vq->drv_priv = dev;
	vq->buf_struct_size = sizeof(struct mx6vout_buffer);
	vq->ops = &mx6vout_qops;
	vq->mem_ops = &vb2_dma_contig_memops;

	vfd = video_device_alloc();
	if (!vfd) {
		v4l2_err(&dev->v4l2_dev, "Failed to allocate video device\n");
		ret = -ENOMEM;
		goto unreg_dev;
	}

	*vfd = mx6vout_videodev;
	vfd->lock = &dev->mutex;
	vfd->v4l2_dev = &dev->v4l2_dev;

	ret = video_register_device(vfd, VFL_TYPE_GRABBER, 0);
	if (ret) {
		v4l2_err(&dev->v4l2_dev, "Failed to register video device\n");
		video_device_release(vfd);
		goto unreg_dev;
	}

	video_set_drvdata(vfd, dev);
	snprintf(vfd->name, sizeof(vfd->name), "%s", mx6vout_videodev.name);
	dev->vfd = vfd;
	v4l2_info(&dev->v4l2_dev,
		  "Device registered as /dev/video%d, parent is ipu%d\n",
		  vfd->num, ipu_get_num(dev->ipu));

	platform_set_drvdata(pdev, dev);

	dev->alloc_ctx = vb2_dma_contig_init_ctx(&pdev->dev);
	if (IS_ERR(dev->alloc_ctx)) {
		v4l2_err(&dev->v4l2_dev, "Failed to alloc vb2 context\n");
		ret = PTR_ERR(dev->alloc_ctx);
		goto unreg_vdev;
	}

	return 0;

unreg_vdev:
	video_unregister_device(dev->vfd);
unreg_dev:
	v4l2_device_unregister(&dev->v4l2_dev);
free_dev:
	kfree(dev);

	return ret;
}

static int mx6vout_remove(struct platform_device *pdev)
{
	struct mx6vout_dev *dev =
		(struct mx6vout_dev *)platform_get_drvdata(pdev);

	v4l2_info(&dev->v4l2_dev, "Removing " DEVICE_NAME "\n");
	vb2_dma_contig_cleanup_ctx(dev->alloc_ctx);
	video_unregister_device(dev->vfd);
	v4l2_device_unregister(&dev->v4l2_dev);
	kfree(dev);

	return 0;
}

static struct of_device_id mx6vout_dt_ids[] = {
	{ .compatible = "fsl,imx6-v4l2-output" },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, mx6vout_dt_ids);

static struct platform_driver mx6vout_pdrv = {
	.probe		= mx6vout_probe,
	.remove		= mx6vout_remove,
	.driver		= {
		.name	= DEVICE_NAME,
		.owner	= THIS_MODULE,
		.of_match_table	= mx6vout_dt_ids,
	},
};

module_platform_driver(mx6vout_pdrv);
