/*
 * Copyright (c) 2012-2014 Mentor Graphics Inc.
 * Copyright 2004-2012 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

/*!
 * @file drivers/media/platform/mxc/capture/mxc_v4l2_capture.c
 *
 * @brief Mxc Video For Linux 2 driver
 *
 * @ingroup MXC_V4L2_CAPTURE
 */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/pinctrl/consumer.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/ctype.h>
#include <linux/io.h>
#include <linux/mutex.h>
#include <linux/pagemap.h>
#include <linux/vmalloc.h>
#include <linux/types.h>
#include <linux/dma-mapping.h>
#include <linux/of_platform.h>
#include <media/v4l2-chip-ident.h>
#include <media/v4l2-ioctl.h>

#include "mxc_v4l2_capture.h"
#include "ipu_prp_sw.h"

#define MX6_CAM_DRV_NAME "imx6-v4l2-cap"

/*
 * The Gstreamer v4l2src plugin appears to have a bug, it doesn't handle
 * frame sizes of type V4L2_FRMSIZE_TYPE_STEPWISE correctly. Set this
 * module param to get around this bug. We can remove once v4l2src handles
 * stepwise frame sizes correctly.
 */
static int v4l2src_compat = 1;
module_param(v4l2src_compat, int, 0644);
MODULE_PARM_DESC(v4l2src_compat,
		 "Gstreamer v4l2src plugin compatibility (default: 1)");

/*! List of TV input video format framing supported. The video formats
 * correspond to the v4l2_id in video_fmt_t. Currently, only PAL and NTSC
 * framing is supported. Needs to be expanded in the future.
 */
enum video_fmt_idx {
	TV_NTSC = 0,		/*!< Locked on (M) NTSC video signal. */
	TV_PAL,			/*!< (B, G, H, I, N)PAL video signal. */
	TV_UNSUPPORTED,         /* Unknown or unsupported standard */
};

/*! Number of video standards supported */
#define TV_STD_MAX		TV_UNSUPPORTED

/*
 * Min/Max supported width and heights.
 */
#define MIN_W       176
#define MIN_H       144
#define MAX_W      4096
#define MAX_H      4096
#define MAX_W_IC   1024
#define MAX_H_IC   1024
#define H_ALIGN       1 /* multiple of 2 */
#define S_ALIGN       1 /* multiple of 2 */

/*! Video format structure. */
struct video_fmt_t {
	struct v4l2_standard standard;
	u16 active_width;	/*!< Active width. */
	u16 active_height;	/*!< Active height. */
	u16 active_top;		/*!< Active top. */
	u16 active_left;	/*!< Active left. */
};

static enum video_fmt_idx mxc_get_std_index(v4l2_std_id std)
{
	enum video_fmt_idx std_idx;

	switch (std) {
	case V4L2_STD_PAL:
	case V4L2_STD_PAL_N:
		std_idx = TV_PAL;
		break;
	case V4L2_STD_NTSC:
	case V4L2_STD_NTSC_443:
	case V4L2_STD_PAL_M:
	case V4L2_STD_PAL_60:
		std_idx = TV_NTSC;
		break;
	default:
		std_idx = TV_UNSUPPORTED;
		break;
	}

	return std_idx;
}

/*!
 * Description of video formats supported.
 *
 *  PAL: active=720x576.
 *  NTSC: active=720x480.
 */
static struct video_fmt_t video_fmts[] = {
	{			/*! NTSC */
		.standard = {
			.index = TV_NTSC,
			.id = V4L2_STD_NTSC,
			.name = "NTSC",
			.frameperiod = {1001, 30000},
			.framelines = 525,
		},
		.active_width = 720,		/* ACT_FRM_WIDTH */
		.active_height = 480,		/* ACT_FRM_HEIGHT */
		.active_top = 13,
		.active_left = 0,
	},
	{			/*! (B, G, H, I, N) PAL */
		.standard = {
			.index = TV_PAL,
			.id = V4L2_STD_PAL,
			.name = "PAL",
			.frameperiod = {1, 25},
			.framelines = 625
		},
		.active_width = 720,
		.active_height = 576,
		.active_top = 0,
		.active_left = 0,
	},
};

static u32 pixel_fmts[] = {
	V4L2_PIX_FMT_RGB565,
	V4L2_PIX_FMT_BGR24,
	V4L2_PIX_FMT_RGB24,
	V4L2_PIX_FMT_BGR32,
	V4L2_PIX_FMT_RGB32,
	V4L2_PIX_FMT_YUV422P,
	V4L2_PIX_FMT_UYVY,
	V4L2_PIX_FMT_YUYV,
	V4L2_PIX_FMT_YUV420,
	V4L2_PIX_FMT_YVU420,
	V4L2_PIX_FMT_NV12,
	V4L2_PIX_FMT_GENERIC16,
};

/* Our controls */
static struct v4l2_queryctrl mxc_ctrls[] = {
	{
		.id		= V4L2_CID_HFLIP,
		.type		= V4L2_CTRL_TYPE_BOOLEAN,
		.name		= "Horizontal Flip",
		.minimum	= 0,
		.maximum	= 1,
		.step		= 1,
		.default_value	= 0,
	}, {
		.id		= V4L2_CID_VFLIP,
		.type		= V4L2_CTRL_TYPE_BOOLEAN,
		.name		= "Vertical Flip",
		.minimum	= 0,
		.maximum	= 1,
		.step		= 1,
		.default_value	= 0,
	}, {
		.id		= V4L2_CID_ROTATE,
		.type		= V4L2_CTRL_TYPE_INTEGER,
		.name		= "Rotation",
		.minimum	= 0,
		.maximum	= 270,
		.step		= 90,
		.default_value	= 0,
	}, {
		.id		= V4L2_CID_MXC_VF_ROT,
		.type		= V4L2_CTRL_TYPE_INTEGER,
		.name		= "VF Rotation",
		.minimum	= 0,
		.maximum	= 270,
		.step		= 90,
		.default_value	= 0,
	},
};

static struct v4l2_queryctrl *mxc_get_ctrl(int id)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(mxc_ctrls); i++) {
		if (id == mxc_ctrls[i].id)
			return &mxc_ctrls[i];
	}

	return NULL;
}

/*!
 * Indicates whether the pixel format is supported.
 *
 * @return false if failed
 */
static bool mxc_pixfmt_supported(u32 pixfmt)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(pixel_fmts); i++) {
		if (pixfmt == pixel_fmts[i])
			return true;
	}

	return false;
}

static int mxc_v4l2_master_attach(struct v4l2_int_device *slave);
static void mxc_v4l2_master_detach(struct v4l2_int_device *slave);
static int start_preview(struct mx6_camera_dev *cam);
static int stop_preview(struct mx6_camera_dev *cam);

/*! Information about this driver. */
static struct v4l2_int_master mxc_v4l2_master = {
	.attach = mxc_v4l2_master_attach,
	.detach = mxc_v4l2_master_detach,
};

/***************************************************************************
 * Functions for handling Frame buffers.
 **************************************************************************/

/*!
 * Free frame buffers
 *
 * @param cam      Structure struct mx6_camera_dev *
 *
 * @return status  0 success.
 */
static int mxc_free_frame_buf(struct mx6_camera_dev *cam)
{
	int i;

	dev_dbg(cam->dev, "MVC: In mxc_free_frame_buf\n");

	for (i = 0; i < FRAME_NUM; i++) {
		if (cam->frame[i].vaddress != NULL) {
			dma_free_coherent(cam->dev, cam->frame[i].buffer.length,
					  cam->frame[i].vaddress,
					  cam->frame[i].paddress);
			cam->frame[i].vaddress = NULL;
		}
	}

	return 0;
}

/*!
 * Allocate frame buffers
 *
 * @param cam      Structure struct mx6_camera_dev*
 * @param count    int number of buffer need to allocated
 *
 * @return status  -0 Successfully allocated a buffer, -ENOBUFS	failed.
 */
static int mxc_allocate_frame_buf(struct mx6_camera_dev *cam, int count)
{
	int i;

	dev_dbg(cam->dev, "In MVC:mxc_allocate_frame_buf - size=%d\n",
		cam->user_fmt.fmt.pix.sizeimage);

	for (i = 0; i < count; i++) {
		cam->frame[i].vaddress =
		    dma_alloc_coherent(
			    cam->dev,
			    PAGE_ALIGN(cam->user_fmt.fmt.pix.sizeimage),
			    &cam->frame[i].paddress,
			    GFP_DMA | GFP_KERNEL | __GFP_NOINIT);
		if (cam->frame[i].vaddress == NULL) {
			dev_err(cam->dev, "ERROR: v4l2 capture: " \
					"mxc_allocate_frame_buf failed.\n");
			mxc_free_frame_buf(cam);
			return -ENOBUFS;
		}
		cam->frame[i].buffer.index = i;
		cam->frame[i].buffer.flags = V4L2_BUF_FLAG_MAPPED;
		cam->frame[i].buffer.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		cam->frame[i].buffer.length =
		    PAGE_ALIGN(cam->user_fmt.fmt.pix.sizeimage);
		cam->frame[i].buffer.memory = V4L2_MEMORY_MMAP;
		cam->frame[i].buffer.m.offset = cam->frame[i].paddress;
		cam->frame[i].index = i;
	}

	return 0;
}

/*!
 * Free frame buffers status
 *
 * @param cam    Structure struct mx6_camera_dev *
 *
 * @return none
 */
static void mxc_free_frames(struct mx6_camera_dev *cam)
{
	int i;

	dev_dbg(cam->dev, "In MVC:mxc_free_frames\n");

	mutex_lock(&cam->irq_lock);

	for (i = 0; i < FRAME_NUM; i++)
		cam->frame[i].buffer.flags = V4L2_BUF_FLAG_MAPPED;

	cam->enc_counter = 0;
	INIT_LIST_HEAD(&cam->ready_q);
	INIT_LIST_HEAD(&cam->working_q);
	INIT_LIST_HEAD(&cam->done_q);

	mutex_unlock(&cam->irq_lock);
}

/*!
 * Return the buffer status
 *
 * @param cam      Structure struct mx6_camera_dev *
 * @param buf      Structure v4l2_buffer *
 *
 * @return status  0 success, EINVAL failed.
 */
static int mxc_v4l2_buffer_status(
	struct mx6_camera_dev *cam,
	struct v4l2_buffer *buf)
{
	dev_dbg(cam->dev, "In MVC:mxc_v4l2_buffer_status\n");

	if (buf->index < 0 || buf->index >= FRAME_NUM) {
		dev_err(cam->dev, "ERROR: v4l2 capture: " \
				"mxc_v4l2_buffer_status buffers " \
				"not allocated\n");
		return -EINVAL;
	}

	memcpy(buf, &(cam->frame[buf->index].buffer), sizeof(*buf));
	return 0;
}

static int mxc_v4l2_release_bufs(struct mx6_camera_dev *cam)
{
	dev_dbg(cam->dev, "In MVC:mxc_v4l2_release_bufs\n");
	return 0;
}

static int mxc_v4l2_prepare_bufs(
	struct mx6_camera_dev *cam,
	struct v4l2_buffer *buf)
{
	dev_dbg(cam->dev, "In MVC:mxc_v4l2_prepare_bufs\n");

	if (buf->index < 0 || buf->index >= FRAME_NUM ||
	    buf->length < PAGE_ALIGN(cam->user_fmt.fmt.pix.sizeimage)) {
		dev_err(cam->dev, "ERROR: v4l2 capture: " \
				"mxc_v4l2_prepare_bufs buffers " \
				"not allocated,index=%d, length=%d\n",
				buf->index, buf->length);
		return -EINVAL;
	}

	cam->frame[buf->index].buffer.index = buf->index;
	cam->frame[buf->index].buffer.flags = V4L2_BUF_FLAG_MAPPED;
	cam->frame[buf->index].buffer.length = buf->length;
	cam->frame[buf->index].buffer.m.offset =
		cam->frame[buf->index].paddress = buf->m.offset;
	cam->frame[buf->index].buffer.type = buf->type;
	cam->frame[buf->index].buffer.memory = V4L2_MEMORY_USERPTR;
	cam->frame[buf->index].index = buf->index;

	return 0;
}

/***************************************************************************
 * Functions for handling the video stream.
 **************************************************************************/

/*
 * Query sensor and update signal lock status. Returns true if lock
 * status has changed.
 */
static bool mxc_update_signal_lock_status(struct mx6_camera_dev *cam)
{
	struct v4l2_input input;
	int current_input;
	bool locked, lsc, changed;

	vidioc_int_g_input(cam->ep->sensor, &current_input);

	input.index = current_input;
	vidioc_int_enum_input(cam->ep->sensor, &input);

	locked = ((input.status & V4L2_IN_ST_NO_SYNC) == 0);

	/*
	 * _g_signal_lsc returns -EIO if there was a lock status
	 * change since last call
	 */
	lsc = (vidioc_int_g_signal_lsc(cam->ep->sensor) == -EIO);

	changed = lsc || (cam->signal_locked != locked);
	cam->signal_locked = locked;

	return changed;
}

/*
 * This function is called in the streaming and preview EOF handlers.
 *
 * The sensor is polled for the current detected video standard and signal
 * lock status. If the standard changed from the time of streaming/preview
 * on, or if there was a lock status change, schedule a restart.
 *
 * Restarts are needed on video standard changes in order to setup new
 * CSI cropping windows which depend on the sensor's standard, and
 * also to re-init resizing.
 *
 * Restarts are needed on lock status changes to resolve vertical sync
 * issues with some TVin sensors (ADV7180).
 *
 * The irq_lock must be held when calling.
 */
void mxc_v4l2_poll_std_and_lock_status(struct mx6_camera_dev *cam)
{
	bool lock_status_change;
	bool restart = false;
	v4l2_std_id std;

	lock_status_change = mxc_update_signal_lock_status(cam);
	vidioc_int_querystd(cam->ep->sensor, &std);

	if (std != cam->current_std && cam->signal_locked) {
		dev_warn(cam->dev, "TVin standard changed to %s\n",
			 v4l2_norm_to_name(std));
		cam->current_std = std;
		restart = true;
	} else if (lock_status_change) {
		dev_warn(cam->dev, "TVin lock status change (%s)\n",
			 cam->signal_locked ? "locked" : "unlocked");
		restart = true;
	}

	if (restart) {
		cam->status_change = true;
		mod_timer(&cam->restart_timer, jiffies +
			  msecs_to_jiffies(MXC_RESTART_DELAY));
	}
}

/*
 * Start the encoder job. There must be at least two frames in the
 * ready queue.
 */
static int start_encoder(struct mx6_camera_dev *cam)
{
	struct mxc_v4l_frame *frame;
	dma_addr_t eba0, eba1;
	int ret;

	if (!cam->enc_enable || !cam->enc_update_eba)
		return -EINVAL;

	if (cam->encoder_on)
		return 0;

	mutex_lock(&cam->irq_lock);

	frame = list_entry(cam->ready_q.next, struct mxc_v4l_frame, queue);
	list_del(cam->ready_q.next);
	list_add_tail(&frame->queue, &cam->working_q);
	frame->ipu_buf_num = 0;
	eba0 = frame->buffer.m.offset;

	frame = list_entry(cam->ready_q.next, struct mxc_v4l_frame, queue);
	list_del(cam->ready_q.next);
	list_add_tail(&frame->queue, &cam->working_q);
	frame->ipu_buf_num = 1;
	eba1 = frame->buffer.m.offset;

	cam->current_buf_num = 0;

	mutex_unlock(&cam->irq_lock);

	/* encoder on! */
	ret = cam->enc_enable(cam, eba0, eba1);
	if (ret)
		return ret;

	if (cam->enc_enable_csi) {
		ret = cam->enc_enable_csi(cam);
		if (ret) {
			cam->enc_disable(cam);
			return ret;
		}
	}

	/* start the EOF timeout timer */
	mod_timer(&cam->eof_timer,
		  jiffies + msecs_to_jiffies(MXC_EOF_TIMEOUT));

	cam->encoder_on = true;

	return 0;
}

/*
 * Shut down the encoder job
 */
static int stop_encoder(struct mx6_camera_dev *cam)
{
	int ret;

	if (!cam->encoder_on)
		return 0;

	/* stop the EOF timeout timer */
	del_timer_sync(&cam->eof_timer);

	if (cam->enc_disable_csi) {
		ret = cam->enc_disable_csi(cam);
		if (ret)
			return ret;
	}
	if (cam->enc_disable) {
		ret = cam->enc_disable(cam);
		if (ret)
			return ret;
	}

	cam->encoder_on = false;

	return 0;
}

/*
 * Return true if the current capture parameters require the use of
 * the Image Converter. We need the IC for scaling, colorspace conversion,
 * preview, and rotation.
 */
static bool need_ic(struct mx6_camera_dev *cam,
		    struct v4l2_format *sf,
		    struct v4l2_format *uf,
		    struct v4l2_rect *crop)
{
	struct v4l2_pix_format *sensor_fmt = &sf->fmt.pix;
	struct v4l2_pix_format *user_fmt = &uf->fmt.pix;
	enum ipu_color_space sensor_cs, user_cs;
	bool ret;

	sensor_cs = ipu_mbus_code_to_colorspace(sensor_fmt->pixelformat);
	user_cs = ipu_pixelformat_to_colorspace(user_fmt->pixelformat);

	ret = (user_fmt->width != crop->width ||
	       user_fmt->height != crop->height ||
	       user_cs != sensor_cs ||
	       cam->overlay_on ||
	       cam->rot_mode != IPU_ROTATE_NONE);

	dev_dbg(cam->dev, "user %dx%d, crop window %dx%d\n",
		user_fmt->width, user_fmt->height,
		crop->width, crop->height);
	dev_dbg(cam->dev, "user CS %s, sensor CS %s\n",
		user_cs == IPUV3_COLORSPACE_YUV ? "YUV" : "RGB",
		sensor_cs == IPUV3_COLORSPACE_YUV ? "YUV" : "RGB");
	dev_dbg(cam->dev, "overlay_on %s\n",
		cam->overlay_on ? "true" : "false");
	dev_dbg(cam->dev, "rotation %d\n", cam->rot_mode);
	dev_dbg(cam->dev, "need IC = %s\n", ret ? "true" : "false");

	return ret;
}

/*
 * Return true if user and sensor formats currently meet the IC
 * restrictions:
 *     o the pixel format from the sensor cannot be 16-bit generic.
 *     o the endpoint id must be 0 (for MIPI CSI2, the endpoint id is the
 *       virtual channel number, and only VC0 can pass through the IC).
 *     o the resizer output size must be at or below 1024x1024.
 */
static bool can_use_ic(struct mx6_camera_dev *cam,
		       struct v4l2_format *sf,
		       struct v4l2_format *uf)
{
	struct v4l2_pix_format *sensor_fmt = &sf->fmt.pix;
	struct v4l2_pix_format *user_fmt = &uf->fmt.pix;
	struct ipu_csi_signal_cfg csicfg;

	ipu_csi_mbus_fmt_to_sig_cfg(&csicfg, sensor_fmt->pixelformat);

	return ((cam->ep->ep.bus_type == V4L2_MBUS_CSI2 ||
		 csicfg.data_width != IPU_CSI_DATA_WIDTH_16) &&
		cam->ep->ep.id == 0 &&
		user_fmt->width <= MAX_W_IC &&
		user_fmt->height <= MAX_H_IC);
}

/*
 * Adjusts passed width and height to meet IC resizer limits.
 */
static void adjust_to_resizer_limits(struct mx6_camera_dev *cam,
				     struct v4l2_format *uf,
				     struct v4l2_rect *crop)
{
	u32 *width, *height;

	if (uf->type == V4L2_BUF_TYPE_VIDEO_CAPTURE) {
		width = &uf->fmt.pix.width;
		height = &uf->fmt.pix.height;
	} else {
		width = &uf->fmt.win.w.width;
		height = &uf->fmt.win.w.height;
	}

	/* output of resizer can't be above 1024x1024 */
	*width = min_t(__u32, *width, MAX_W_IC);
	*height = min_t(__u32, *height, MAX_H_IC);

	/* resizer cannot downsize more than 8:1 */
	if (cam->rot_mode >= IPU_ROTATE_90_RIGHT) {
		*height = max_t(__u32, *height, crop->width / 8);
		*width = max_t(__u32, *width, crop->height / 8);
	} else {
		*width = max_t(__u32, *width, crop->width / 8);
		*height = max_t(__u32, *height, crop->height / 8);
	}
}

static void adjust_user_fmt(struct mx6_camera_dev *cam,
			    struct v4l2_format *sf,
			    struct v4l2_format *uf,
			    struct v4l2_rect *crop)
{
	int bpp;

	/*
	 * Make sure resolution is within IC resizer limits
	 * if we need the Image Converter.
	 */
	if (need_ic(cam, sf, uf, crop))
		adjust_to_resizer_limits(cam, uf, crop);

	/*
	 * Force the resolution to match crop window if
	 * we can't use the Image Converter.
	 */
	if (!can_use_ic(cam, sf, uf)) {
		uf->fmt.pix.width = crop->width;
		uf->fmt.pix.height = crop->height;
	}

	bpp = ipu_bits_per_pixel(uf->fmt.pix.pixelformat);
	uf->fmt.pix.bytesperline = (uf->fmt.pix.width * bpp) >> 3;
	uf->fmt.pix.sizeimage = uf->fmt.pix.height * uf->fmt.pix.bytesperline;
}

/*
 * Calculate what the default active crop window should be. For sensors
 * that output a standard video format, the default crop window is
 * the active window for that standard, otherwise it is just the
 * raw sensor window. This crop window will be stored to
 * cam->crop_current.
 */
static void mxc_calc_default_crop(struct mx6_camera_dev *cam,
				  struct v4l2_rect *crop,
				  struct v4l2_format *sf)
{
	struct v4l2_pix_format *sensor_fmt = &sf->fmt.pix;
	enum video_fmt_idx std_idx;

	if (cam->current_std == V4L2_STD_ALL) {
		/*
		 * sensor does not conform to a specific video standard,
		 * so set crop active window to sensor format.
		 */
		crop->top = crop->left = 0;
		crop->width = sensor_fmt->width;
		crop->height = sensor_fmt->height;
	} else {
		std_idx = mxc_get_std_index(cam->current_std);
		if (std_idx == TV_UNSUPPORTED) {
			dev_warn(cam->dev,
				 "Warning: Unsupported video standard 0x%llx, "
				 "setting cropping for PAL\n",
				 cam->current_std);
			std_idx = TV_PAL;
		}

		crop->top = video_fmts[std_idx].active_top;
		crop->left = video_fmts[std_idx].active_left;
		crop->width = video_fmts[std_idx].active_width;
		crop->height = video_fmts[std_idx].active_height;
		cam->standard = video_fmts[std_idx].standard;
	}

	/* adjust crop window to h/w alignment restrictions */
	crop->width &= ~0x7;
	crop->left &= ~0x3;
}

/*
 * Use the parsed endpoint info and sensor format to fill
 * ipu_csi_signal_cfg.
 */
static void fill_csi_signal_cfg(struct mx6_camera_dev *cam)
{
	struct ipu_csi_signal_cfg *csicfg = &cam->ep->csi_sig_cfg;
	struct v4l2_of_endpoint *ep = &cam->ep->ep;

	memset(csicfg, 0, sizeof(*csicfg));

	ipu_csi_mbus_fmt_to_sig_cfg(csicfg,
				    cam->sensor_fmt.fmt.pix.pixelformat);

	switch (ep->bus_type) {
	case V4L2_MBUS_PARALLEL:
		csicfg->ext_vsync = 1;
		csicfg->vsync_pol = (ep->bus.parallel.flags &
				     V4L2_MBUS_VSYNC_ACTIVE_LOW) ? 1 : 0;
		csicfg->hsync_pol = (ep->bus.parallel.flags &
				     V4L2_MBUS_HSYNC_ACTIVE_LOW) ? 1 : 0;
		csicfg->pixclk_pol = (ep->bus.parallel.flags &
				      V4L2_MBUS_PCLK_SAMPLE_FALLING) ? 1 : 0;
		csicfg->clk_mode = IPU_CSI_CLK_MODE_GATED_CLK;
		break;
	case V4L2_MBUS_BT656:
		csicfg->ext_vsync = 0;
		if (cam->sensor_fmt.fmt.pix.field == V4L2_FIELD_INTERLACED)
			csicfg->clk_mode = IPU_CSI_CLK_MODE_CCIR656_INTERLACED;
		else
			csicfg->clk_mode = IPU_CSI_CLK_MODE_CCIR656_PROGRESSIVE;
		break;
	case V4L2_MBUS_CSI2:
		/*
		 * MIPI CSI-2 requires non gated clock mode, all other
		 * parameters are not applicable for MIPI CSI-2 bus.
		 */
		csicfg->clk_mode = IPU_CSI_CLK_MODE_NONGATED_CLK;
		break;
	default:
		/* will never get here, keep compiler quiet */
		break;
	}
}

/*
 * Fill in struct ipu_csi_signal_cfg and struct v4l2_format
 * from info provided by the sensor.
 */
static void mxc_update_sensor_fmt(struct mx6_camera_dev *cam)
{
	cam->sensor_fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	vidioc_int_g_fmt_cap(cam->ep->sensor, &cam->sensor_fmt);

	/* update cropping bounds with new raw sensor window */
	cam->crop_bounds.top = cam->crop_bounds.left = 0;
	cam->crop_bounds.width = cam->sensor_fmt.fmt.pix.width;
	cam->crop_bounds.height = cam->sensor_fmt.fmt.pix.height;
	cam->crop_defrect = cam->crop_bounds;

	fill_csi_signal_cfg(cam);
}

/*!
 * Start streaming.
 *
 * @param cam      structure struct mx6_camera_dev *
 *
 * @return status  0 Success
 */
static int mxc_streamon(struct mx6_camera_dev *cam)
{
	int err = 0;

	dev_dbg(cam->dev, "In MVC:mxc_streamon\n");

	if (cam->streaming_on)
		return 0;

	mutex_lock(&cam->irq_lock);

	/* Get current standard and lock status if needed */
	if (cam->status_change) {
		mxc_update_sensor_fmt(cam);
		vidioc_int_querystd(cam->ep->sensor, &cam->current_std);
		mxc_calc_default_crop(cam, &cam->crop_current,
				      &cam->sensor_fmt);
		mxc_update_signal_lock_status(cam);
		cam->status_change = false;
		dev_info(cam->dev, "at stream on: %s, %s\n",
			 v4l2_norm_to_name(cam->current_std),
			 cam->signal_locked ? "signal locked" : "no signal");
	}

	mutex_unlock(&cam->irq_lock);

	cam->using_ic = (need_ic(cam, &cam->sensor_fmt, &cam->user_fmt,
				 &cam->crop_current) &&
			 can_use_ic(cam, &cam->sensor_fmt, &cam->user_fmt));
	if (cam->using_ic)
		prp_enc_select(cam);
	else
		csi_enc_select(cam);

	if (cam->overlay_on == true)
		stop_preview(cam);

	cam->dummy_frame.vaddress =
		dma_alloc_coherent(cam->dev,
				   PAGE_ALIGN(cam->user_fmt.fmt.pix.sizeimage),
				   &cam->dummy_frame.paddress,
				   GFP_DMA | GFP_KERNEL | __GFP_NOINIT);
	if (cam->dummy_frame.vaddress == NULL) {
		dev_err(cam->dev, "ERROR: %s: Allocate dummy frame failed.\n",
			__func__);
		return -ENOBUFS;
	}

	cam->dummy_frame.buffer.type = V4L2_BUF_TYPE_PRIVATE;
	cam->dummy_frame.buffer.length =
		PAGE_ALIGN(cam->user_fmt.fmt.pix.sizeimage);
	cam->dummy_frame.buffer.m.offset = cam->dummy_frame.paddress;

	/*
	 * If there are two or more frames in the ready queue, we can
	 * start the encoding now. Otherwise the encoding will start
	 * once two frames have been queued.
	 *
	 * We don't need to hold irq_lock here b/c streaming is stopped
	 * and we hold the busy_lock so no new activity can be started
	 * by another thread.
	 */
	if (!list_empty(&cam->ready_q) && !list_is_singular(&cam->ready_q))
		err = start_encoder(cam);

	if (cam->overlay_on == true)
		start_preview(cam);

	cam->streaming_on = true;

	return err;
}


/*!
 * Stop streaming.
 *
 * @param cam      structure struct mx6_camera_dev *
 *
 * @return status  0 Success
 */
static int mxc_streamoff(struct mx6_camera_dev *cam)
{
	int err = 0;

	dev_dbg(cam->dev, "In MVC:mxc_streamoff\n");

	if (!cam->streaming_on)
		return 0;

	err = stop_encoder(cam);

	if (cam->dummy_frame.vaddress) {
		dma_free_coherent(cam->dev, cam->dummy_frame.buffer.length,
				  cam->dummy_frame.vaddress,
				  cam->dummy_frame.paddress);
		cam->dummy_frame.vaddress = NULL;
	}

	mxc_free_frames(cam);

	if (cam->using_ic)
		err = prp_enc_deselect(cam);
	else
		err = csi_enc_deselect(cam);

	cam->streaming_on = false;

	return err;
}

/*!
 * Validate and adjust the overlay window size, position
 *
 * @param cam      structure struct mx6_camera_dev *
 * @param f        struct v4l2_format  *
 *
 * @return 0
 */
static int verify_preview(struct mx6_camera_dev *cam, struct v4l2_format *f)
{
	struct v4l2_window *win = &f->fmt.win;
	unsigned int width_align;

	dev_dbg(cam->dev, "In MVC: verify_preview\n");

	// FIXME: implement real checking!

	width_align = ipu_pixelformat_is_planar(cam->v4l2_fb.fmt.pixelformat) ?
		4 : 3;

	v4l_bound_align_image(&win->w.width, MIN_W, MAX_W_IC,
			      width_align, &win->w.height,
			      MIN_H, MAX_H_IC, H_ALIGN, S_ALIGN);

	if (win->w.width == 0 || win->w.height == 0) {
		dev_err(cam->dev, "ERROR: v4l2 capture: " \
			"width or height too small.\n");
		return -EINVAL;
	}

	adjust_to_resizer_limits(cam, f, &cam->crop_current);

	return 0;
}

/*!
 * start the viewfinder job
 *
 * @param cam      structure struct mx6_camera_dev *
 *
 * @return status  0 Success
 */
static int start_preview(struct mx6_camera_dev *cam)
{
	int err = 0;

	dev_dbg(cam->dev, "MVC: start_preview\n");

	mutex_lock(&cam->irq_lock);

	/* Get current standard and lock status if needed */
	if (cam->status_change) {
		mxc_update_sensor_fmt(cam);
		vidioc_int_querystd(cam->ep->sensor, &cam->current_std);
		mxc_calc_default_crop(cam, &cam->crop_current,
				      &cam->sensor_fmt);
		mxc_update_signal_lock_status(cam);
		cam->status_change = false;
		dev_info(cam->dev, "at preview on: %s, %s\n",
			 v4l2_norm_to_name(cam->current_std),
			 cam->signal_locked ? "signal locked" : "no signal");
	}

	mutex_unlock(&cam->irq_lock);

	err = prp_vf_sdc_select(cam);
	if (err != 0)
		return err;

	if (cam->vf_start_sdc) {
		err = cam->vf_start_sdc(cam);
		if (err != 0)
			return err;
	}

	if (cam->vf_enable_csi)
		err = cam->vf_enable_csi(cam);

	/* start the VF EOF timeout timer */
	mod_timer(&cam->eof_vf_timer,
		  jiffies + msecs_to_jiffies(MXC_EOF_TIMEOUT));

	return err;
}

/*!
 * shut down the viewfinder job
 *
 * @param cam      structure struct mx6_camera_dev *
 *
 * @return status  0 Success
 */
static int stop_preview(struct mx6_camera_dev *cam)
{
	int err = 0;

	/* stop the VF EOF timeout timer */
	del_timer_sync(&cam->eof_vf_timer);

	if (cam->vf_disable_csi) {
		err = cam->vf_disable_csi(cam);
		if (err != 0)
			return err;
	}

	if (cam->vf_stop_sdc) {
		err = cam->vf_stop_sdc(cam);
		if (err != 0)
			return err;
	}

	err = prp_vf_sdc_deselect(cam);

	return err;
}

/***************************************************************************
 * VIDIOC Functions.
 **************************************************************************/

/*!
 * V4L2 - mxc_v4l2_g_fmt function
 *
 * @param cam         structure struct mx6_camera_dev *
 *
 * @param f           structure v4l2_format *
 *
 * @return  status    0 success, EINVAL failed
 */
static int mxc_v4l2_g_fmt(struct mx6_camera_dev *cam, struct v4l2_format *f)
{
	int retval = 0;

	dev_dbg(cam->dev, "In MVC: mxc_v4l2_g_fmt type=%d\n", f->type);

	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		dev_dbg(cam->dev, "   type is V4L2_BUF_TYPE_VIDEO_CAPTURE\n");
		f->fmt.pix = cam->user_fmt.fmt.pix;
		break;
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
		dev_dbg(cam->dev, "   type is V4L2_BUF_TYPE_VIDEO_OVERLAY\n");
		f->fmt.win = cam->win;
		break;
	default:
		dev_dbg(cam->dev, "   type is invalid\n");
		retval = -EINVAL;
	}

	return retval;
}

/*!
 * V4L2 - mxc_v4l2_try_fmt_cap function
 *
 * @param cam         structure struct mx6_camera_dev *
 *
 * @param sf          structure v4l2_format * for sensor format
 * @param f           structure v4l2_format * for user format
 *
 * @return  status    0 success, EINVAL
 */
static int mxc_v4l2_try_fmt_cap(struct mx6_camera_dev *cam,
				struct v4l2_format *sf,
				struct v4l2_format *f)
{
	unsigned int width_align;
	struct v4l2_rect crop;
	int ret;

	if (!mxc_pixfmt_supported(f->fmt.pix.pixelformat)) {
		dev_info(cam->dev, "%s: pixel format not supported\n",
			 __func__);
		return -EINVAL;
	}

	ret = vidioc_int_try_fmt_cap(cam->ep->sensor, sf);
	if (ret)
		return ret;

	width_align = ipu_pixelformat_is_planar(f->fmt.pix.pixelformat) ? 4 : 3;

	v4l_bound_align_image(&f->fmt.pix.width, MIN_W, MAX_W,
			      width_align, &f->fmt.pix.height,
			      MIN_H, MAX_H, H_ALIGN, S_ALIGN);

	if (f->fmt.pix.width == 0 || f->fmt.pix.height == 0) {
		dev_info(cam->dev, "%s: width or height too small\n",
			 __func__);
		return -EINVAL;
	}

	/*
	 * calculate what the active crop window will be for this
	 * sensor format.
	 */
	mxc_calc_default_crop(cam, &crop, sf);

	adjust_user_fmt(cam, sf, f, &crop);

	return 0;
}

/*!
 * V4L2 - mxc_v4l2_try_fmt function
 *
 * @param cam         structure struct mx6_camera_dev *
 *
 * @param f           structure v4l2_format *
 *
 * @return  status    0 success, EINVAL
 */
static int mxc_v4l2_try_fmt(struct mx6_camera_dev *cam,
			    struct v4l2_format *f)
{
	struct v4l2_format sensor_fmt;
	int ret;

	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		sensor_fmt = *f;
		ret = mxc_v4l2_try_fmt_cap(cam, &sensor_fmt, f);
		break;
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
		ret = verify_preview(cam, f);
		break;
	default:
		ret = -EINVAL;
		break;
	}

	return ret;
}

/*!
 * V4L2 - mxc_v4l2_s_fmt function
 *
 * @param cam         structure struct mx6_camera_dev *
 *
 * @param f           structure v4l2_format *
 *
 * @return  status    0 success, EINVAL, EBUSY failed
 */
static int mxc_v4l2_s_fmt(struct mx6_camera_dev *cam, struct v4l2_format *f)
{
	struct v4l2_format sensor_fmt;
	int retval = 0;

	dev_dbg(cam->dev, "In MVC: mxc_v4l2_s_fmt\n");

	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		dev_dbg(cam->dev, "   type=V4L2_BUF_TYPE_VIDEO_CAPTURE\n");

		if (cam->streaming_on) {
			dev_err(cam->dev,
				"ERROR: %s: not allowed while streaming\n",
				__func__);
			return -EBUSY;
		}

		sensor_fmt = *f;
		retval = mxc_v4l2_try_fmt_cap(cam, &sensor_fmt, f);
		if (retval)
			return retval;

		retval = vidioc_int_s_fmt_cap(cam->ep->sensor, &sensor_fmt);
		if (retval)
			return retval;

		/* format change successful, so update our copy */
		cam->user_fmt.fmt.pix = f->fmt.pix;

		mxc_update_sensor_fmt(cam);
		/* calculate new crop_current */
		mxc_calc_default_crop(cam, &cam->crop_current,
				      &cam->sensor_fmt);

		if (cam->user_fmt.fmt.pix.priv != 0) {
			if (copy_from_user(
				    &cam->offset,
				    (void __user *)cam->user_fmt.fmt.pix.priv,
				    sizeof(cam->offset))) {
				retval = -EFAULT;
				break;
			}
		}

		dev_dbg(cam->dev, "%dx%d, stride %d, sizeimage %d\n",
			f->fmt.pix.width, f->fmt.pix.height,
			f->fmt.pix.bytesperline, f->fmt.pix.sizeimage);
		break;
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
		dev_dbg(cam->dev, "   type=V4L2_BUF_TYPE_VIDEO_OVERLAY\n");
		retval = verify_preview(cam, f);
		if (!retval) {
			if (cam->overlay_on == true)
				stop_preview(cam);
			cam->win = f->fmt.win;
			if (cam->overlay_on == true)
				start_preview(cam);
		}
		break;
	default:
		retval = -EINVAL;
	}

	return retval;
}

/*!
 * V4L2 - mxc_v4l2_reqbufs function
 *
 * @param cam         structure struct mx6_camera_dev *
 *
 * @param req         structure v4l2_requestbuffers *
 *
 * @return  status    0 success, EINVAL, EBUSY failed
 */
static int mxc_v4l2_reqbufs(struct mx6_camera_dev *cam,
			    struct v4l2_requestbuffers *req)
{
	int ret = 0;

	if ((req->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)) {
		dev_err(cam->dev, "ERROR: %s: wrong buffer type\n", __func__);
		return -EINVAL;
	}

	if (cam->streaming_on) {
		dev_err(cam->dev, "ERROR: %s: not allowed while streaming\n",
			__func__);
		return -EBUSY;
	}

	if (req->count > FRAME_NUM) {
		req->count = FRAME_NUM;
		dev_warn(cam->dev, "%s: not enough buffers, set to %d\n",
			 __func__, req->count);
	}

	if (req->memory & V4L2_MEMORY_MMAP) {
		mxc_free_frame_buf(cam);
		ret = mxc_allocate_frame_buf(cam, req->count);
	}

	return ret;
}

/*!
 * V4L2 - mxc_v4l2_querybuf function
 *
 * @param cam         structure struct mx6_camera_dev *
 *
 * @param buf         structure v4l2_buffer *
 *
 * @return  status    0 success, EINVAL, EBUSY failed
 */
static int mxc_v4l2_querybuf(struct mx6_camera_dev *cam,
			     struct v4l2_buffer *buf)
{
	int ret = 0;

	if (buf->type != V4L2_BUF_TYPE_VIDEO_CAPTURE) {
		dev_err(cam->dev, "ERROR: %s: wrong buffer type\n",  __func__);
		return -EINVAL;
	}

	if (cam->streaming_on) {
		dev_err(cam->dev, "ERROR: %s: not allowed while streaming\n",
			__func__);
		return -EBUSY;
	}

	mutex_lock(&cam->param_lock);

	if (buf->memory & V4L2_MEMORY_USERPTR) {
		mxc_v4l2_release_bufs(cam);
		ret = mxc_v4l2_prepare_bufs(cam, buf);
	} else if (buf->memory & V4L2_MEMORY_MMAP)
		ret = mxc_v4l2_buffer_status(cam, buf);

	mutex_unlock(&cam->param_lock);

	return ret;
}

/*!
 * V4L2 - mxc_v4l2_s_crop function
 *
 * @param cam         structure struct mx6_camera_dev *
 *
 * @param crop        structure v4l2_crop *
 *
 * @return  status    0 success, EINVAL, EBUSY failed
 */
static int mxc_v4l2_s_crop(struct mx6_camera_dev *cam,
			   struct v4l2_crop *crop)
{
	struct v4l2_rect *b = &cam->crop_bounds;

	if (crop->type != V4L2_BUF_TYPE_VIDEO_CAPTURE &&
	    crop->type != V4L2_BUF_TYPE_VIDEO_OVERLAY) {
		dev_err(cam->dev, "ERROR: %s: wrong buffer type\n",  __func__);
		return -EINVAL;
	}

	if (cam->streaming_on) {
		dev_err(cam->dev, "ERROR: %s: not allowed while streaming\n",
			__func__);
		return -EBUSY;
	}

	/*
	 * The IPU currently does not setup the CCIR code registers
	 * properly to handle arbitrary crop windows. So ignore this
	 * request if the sensor bus is BT.656.
	 */
	if (cam->ep->ep.bus_type == V4L2_MBUS_BT656 &&
			(crop->c.top != cam->crop_current.top ||
			 crop->c.height != cam->crop_current.height)) {
		dev_err(cam->dev,
			"vertical crop is not supported for this sensor!\n");
		return -EINVAL;
	}

	crop->c.top = (crop->c.top < b->top) ? b->top : crop->c.top;
	if (crop->c.top > b->top + b->height)
		crop->c.top = b->top + b->height - 1;
	if (crop->c.height > b->top + b->height - crop->c.top)
		crop->c.height = b->top + b->height - crop->c.top;

	crop->c.left = (crop->c.left < b->left) ? b->left : crop->c.left;
	if (crop->c.left > b->left + b->width)
		crop->c.left = b->left + b->width - 1;
	if (crop->c.width > b->left - crop->c.left + b->width)
		crop->c.width = b->left - crop->c.left + b->width;

	crop->c.width &= ~0x7;
	crop->c.left &= ~0x3;
	cam->crop_current = crop->c;

	dev_dbg(cam->dev, "%s: crop window = %dx%d\n", __func__,
		cam->crop_current.width, cam->crop_current.height);

	/*
	 * Crop window has changed, we need to adjust the user
	 * width/height to meet new IC resizer restrictions or to
	 * match the new crop window if the IC can't be used.
	 */
	adjust_user_fmt(cam, &cam->sensor_fmt, &cam->user_fmt,
			&cam->crop_current);

	return 0;
}

/*!
 * V4L2 - mxc_v4l2_overlay function
 *
 * @param cam         structure struct mx6_camera_dev *
 *
 * @param enable      bool, whether to enable overlay
 *
 * @return  status    0 success, EBUSY failed
 */
static int mxc_v4l2_overlay(struct mx6_camera_dev *cam, bool enable)
{
	int ret = 0;

	if (enable && !cam->overlay_on) {
		if (cam->streaming_on && !cam->using_ic) {
			dev_err(cam->dev,
				"ERROR: %s: not allowed while streaming " \
				"without IC\n", __func__);
			return -EBUSY;
		}

		if (!can_use_ic(cam, &cam->sensor_fmt, &cam->user_fmt)) {
			dev_err(cam->dev,
				"ERROR: %s: current format does not "	\
				"allow preview\n", __func__);
			return -EINVAL;
		}

		cam->overlay_on = true;
		cam->overlay_pid = current->tgid;
		ret = start_preview(cam);
	} else if (!enable && cam->overlay_on) {
		ret = stop_preview(cam);
		cam->overlay_on = false;
	}

	return ret;
}

/*
 * Update rotation controls.
 */
static int mxc_set_rotations(struct mx6_camera_dev *cam,
			     int rotation, int vf_rotation,
			     bool hflip, bool vflip)
{
	enum ipu_rotate_mode rot_mode, vf_rot_mode;
	int ret;

	ret = ipu_degrees_to_rot_mode(&rot_mode, rotation,
				      hflip, vflip);
	if (ret)
		return ret;
	ret = ipu_degrees_to_rot_mode(&vf_rot_mode, vf_rotation,
				      hflip, vflip);
	if (ret)
		return ret;

	if (rot_mode != cam->rot_mode) {
		if (cam->streaming_on) {
			dev_err(cam->dev,
				"ERROR: %s: not allowed while streaming\n",
				__func__);
			return -EBUSY;
		}

		if (rot_mode != IPU_ROTATE_NONE &&
		    !can_use_ic(cam, &cam->sensor_fmt, &cam->user_fmt)) {
			dev_err(cam->dev,
				"ERROR: %s: current format does not "   \
				"allow rotation\n", __func__);
			return -EINVAL;
		}
	}

	cam->rot_mode = rot_mode;
	cam->rotation = rotation;
	cam->vf_rotation = vf_rotation;
	cam->hflip = hflip;
	cam->vflip = vflip;

	if (vf_rot_mode != cam->vf_rot_mode) {
		if (cam->overlay_on == true)
			stop_preview(cam);
		cam->vf_rot_mode = vf_rot_mode;
		if (cam->overlay_on == true)
			start_preview(cam);
	}

	return 0;
}

/*!
 * get control param
 *
 * @param cam         structure struct mx6_camera_dev *
 *
 * @param c           structure v4l2_control *
 *
 * @return  status    0 success, EINVAL failed
 */
static int mxc_v4l2_g_ctrl(struct mx6_camera_dev *cam, struct v4l2_control *c)
{
	int ret = 0;

	dev_dbg(cam->dev, "In MVC:mxc_v4l2_g_ctrl\n");

	switch (c->id) {
	case V4L2_CID_HFLIP:
		c->value = cam->hflip ? 1 : 0;
		break;
	case V4L2_CID_VFLIP:
		c->value = cam->vflip ? 1 : 0;
		break;
	case V4L2_CID_ROTATE:
		c->value = cam->rotation;
		break;
	case V4L2_CID_MXC_VF_ROT:
		c->value = cam->vf_rotation;
		break;
	default:
		ret = vidioc_int_g_ctrl(cam->ep->sensor, c);
		break;
	}

	return ret;
}

/*!
 * V4L2 - set_control function
 * @param cam         structure struct mx6_camera_dev *
 *
 * @param c           structure v4l2_control *
 *
 * @return            0 success, EINVAL, EBUSY failed
 */
static int mxc_v4l2_s_ctrl(struct mx6_camera_dev *cam, struct v4l2_control *c)
{
	int rotation, vf_rotation;
	bool hflip, vflip;

	dev_dbg(cam->dev, "In MVC:mxc_v4l2_s_ctrl\n");

	rotation = cam->rotation;
	vf_rotation = cam->vf_rotation;
	hflip = cam->hflip;
	vflip = cam->vflip;

	switch (c->id) {
	case V4L2_CID_HFLIP:
		hflip = (c->value == 1);
		break;
	case V4L2_CID_VFLIP:
		vflip = (c->value == 1);
		break;
	case V4L2_CID_ROTATE:
		rotation = c->value;
		break;
	case V4L2_CID_MXC_VF_ROT:
		vf_rotation = c->value;
		break;
	default:
		return vidioc_int_s_ctrl(cam->ep->sensor, c);
	}

	return mxc_set_rotations(cam, rotation, vf_rotation, hflip, vflip);
}

static int mxc_v4l2_queryctrl(struct mx6_camera_dev *cam,
			      struct v4l2_queryctrl *qc)
{
	struct v4l2_queryctrl *c;

	c = mxc_get_ctrl(qc->id);
	if (c) {
		/* it's our control, return it */
		*qc = *c;
		return 0;
	}

	/* it's not our control, try the sensor */
	return vidioc_int_queryctrl(cam->ep->sensor, qc);
}

static int mxc_v4l2_querymenu(struct mx6_camera_dev *cam,
			      struct v4l2_querymenu *qm)
{
	/* there are currently no menu controls */
	return -EINVAL;
}


/*!
 * V4L2 - mxc_v4l2_s_param function
 * Allows setting of capturemode and frame rate.
 *
 * @param cam         structure struct mx6_camera_dev *
 * @param parm        structure v4l2_streamparm *
 *
 * @return  status    0 success, EINVAL, EBUSY failed
 */
static int mxc_v4l2_s_param(struct mx6_camera_dev *cam,
			    struct v4l2_streamparm *parm)
{
	struct v4l2_streamparm currentparm;
	u32 current_fps, parm_fps;
	int err = 0;

	dev_dbg(cam->dev, "In mxc_v4l2_s_param\n");

	if (parm->type != V4L2_BUF_TYPE_VIDEO_CAPTURE) {
		dev_err(cam->dev, KERN_ERR "mxc_v4l2_s_param invalid type\n");
		return -EINVAL;
	}

	if (cam->streaming_on) {
		dev_err(cam->dev, "ERROR: %s: not allowed while streaming\n",
			__func__);
		return -EBUSY;
	}

	/* Stop the viewfinder */
	if (cam->overlay_on == true)
		stop_preview(cam);

	currentparm.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

	/* First check that this device can support the changes requested. */
	err = vidioc_int_g_parm(cam->ep->sensor, &currentparm);
	if (err) {
		dev_err(cam->dev, "%s: vidioc_int_g_parm returned an error %d\n",
			__func__, err);
		goto exit;
	}

	current_fps = currentparm.parm.capture.timeperframe.denominator
			/ currentparm.parm.capture.timeperframe.numerator;
	parm_fps = parm->parm.capture.timeperframe.denominator
			/ parm->parm.capture.timeperframe.numerator;

	dev_dbg(cam->dev, "   Current capabilities are %x\n",
			currentparm.parm.capture.capability);
	dev_dbg(cam->dev, "   Current capturemode is %d  change to %d\n",
			currentparm.parm.capture.capturemode,
			parm->parm.capture.capturemode);
	dev_dbg(cam->dev, "   Current framerate is %d  change to %d\n",
			current_fps, parm_fps);

	/* This will change any camera settings needed. */
	err = vidioc_int_s_parm(cam->ep->sensor, parm);
	if (err) {
		dev_err(cam->dev,
			"%s: vidioc_int_s_parm returned an error %d\n",
			__func__, err);
	}

exit:
	if (cam->overlay_on == true)
		start_preview(cam);

	return err;
}

/*!
 * V4L2 - mxc_v4l2_s_std function
 *
 * Sets the TV standard to be used.
 *
 * @param cam	      structure struct mx6_camera_dev *
 * @param parm	      structure v4l2_streamparm *
 *
 * @return  status    0 success, EINVAL, EBUSY failed
 */
static int mxc_v4l2_s_std(struct mx6_camera_dev *cam, v4l2_std_id e)
{
	int ret;

	dev_dbg(cam->dev, "In mxc_v4l2_s_std %Lx\n", e);

	if (cam->streaming_on) {
		dev_err(cam->dev, "ERROR: %s: not allowed while streaming\n",
			__func__);
		return -EBUSY;
	}

	ret = vidioc_int_s_std(cam->ep->sensor, &e);
	if (ret)
		return ret;

	cam->current_std = e;

	/*
	 * the sensor's video standard output has changed, we need
	 * to update sensor's format.
	 */
	mxc_update_sensor_fmt(cam);

	return 0;
}

/*!
 * V4L2 - mxc_v4l2_g_std function
 *
 * Gets the TV standard from the TV input device.
 *
 * @param cam	      structure struct mx6_camera_dev *
 *
 * @param e	      structure v4l2_streamparm *
 *
 * @return  status    0 success, EINVAL failed
 */
static int mxc_v4l2_g_std(struct mx6_camera_dev *cam, v4l2_std_id *e)
{
	dev_dbg(cam->dev, "In mxc_v4l2_g_std\n");

	*e = cam->current_std;

	return 0;
}

/* find the endpoint that is handling this input index */
static struct mxc_v4l2_endpoint *find_ep_by_input_index(
	struct mx6_camera_dev *cam, int input_idx)
{
	struct mxc_v4l2_endpoint *ep;
	int i;

	for (i = 0; i < cam->num_eps; i++) {
		ep = &cam->all_eps[i];
		if (!ep->sensor)
			continue;

		if (input_idx >= ep->sensor_input.first &&
		    input_idx <= ep->sensor_input.last)
			break;
	}

	return (i < cam->num_eps) ? ep : NULL;
}

/* find the endpoint that is attached to the given remote slave node */
static struct mxc_v4l2_endpoint *find_ep_by_remote_node(
	struct mx6_camera_dev *cam, struct device_node *node)
{
	struct mxc_v4l2_endpoint *ep;
	int i;

	for (i = 0; i < cam->num_eps; i++) {
		ep = &cam->all_eps[i];

		if (node == ep->remote_node)
			break;
	}

	return (i < cam->num_eps) ? ep : NULL;
}

/*!
 * V4L2 - mxc_v4l2_enum_input function
 *
 * Enumerates the requested input at input->index.
 *
 * @param cam	      structure struct mx6_camera_dev *
 *
 * @param input	      structure v4l2_input *
 *
 * @return  status    0 success, EINVAL failed
 */
static int mxc_v4l2_enum_input(struct mx6_camera_dev *cam,
			       struct v4l2_input *input)
{
	struct mxc_v4l2_endpoint *ep;
	int ret;

	/* find the endpoint that is handling this input */
	ep = find_ep_by_input_index(cam, input->index);
	if (!ep || !ep->sensor)
		return -EINVAL;

	/*
	 * Call into the sensors enum_input. The sensor's input index
	 * starts from 0, so remove our offset index first.
	 */
	input->index -= ep->sensor_input.first;
	ret = vidioc_int_enum_input(ep->sensor, input);
	input->index += ep->sensor_input.first;

	return ret;
}

/*!
 * V4L2 - mxc_v4l2_s_input function
 *
 * Sets the requested input at index.
 *
 * @param cam	      structure struct mx6_camera_dev *
 *
 * @param index	      the input index
 *
 * @return  status    0 success, EINVAL, EBUSY failed
 */
static int mxc_v4l2_s_input(struct mx6_camera_dev *cam, int index)
{
	struct mxc_v4l2_endpoint *ep;
	int i, sensor_input;

	if (index == cam->current_input)
		return 0;

	/* find the endpoint that is handling this input */
	ep = find_ep_by_input_index(cam, index);
	if (!ep || !ep->sensor)
		return -EINVAL;

	if (cam->ep != ep) {
		if (cam->streaming_on || cam->overlay_on) {
			dev_err(cam->dev,
				"%s: disable streaming and preview before " \
				"switching sensors\n", __func__);
			return -EBUSY;
		}

		/* set new endpoint */
		cam->ep = ep;

		/* power down other sensors before enable new one */
		for (i = 0; i < cam->num_eps; i++) {
			if (cam->all_eps[i].sensor &&
			    &cam->all_eps[i] != cam->ep) {
				vidioc_int_dev_exit(cam->all_eps[i].sensor);
				vidioc_int_s_power(cam->all_eps[i].sensor, 0);
			}
		}

		/* power-up the new sensor */
		vidioc_int_s_power(cam->ep->sensor, 1);
		vidioc_int_reset(cam->ep->sensor);
		vidioc_int_dev_init(cam->ep->sensor);

		if (cam->ep->ep.bus_type == V4L2_MBUS_CSI2 && cam->csi2) {
			/* power-up the mipi csi-2 receiver */
			vidioc_int_s_power(cam->csi2, 1);
		}
	}

	/* finally select the sensor's input */
	sensor_input = index - cam->ep->sensor_input.first;
	vidioc_int_s_input(cam->ep->sensor, &sensor_input);

	cam->current_input = index;

	/*
	 * sometimes on switching video input on TVin devices no lock
	 * status change event is generated, but vertical sync is
	 * messed up nevertheless. So schedule a restart to correct it.
	 */
	mod_timer(&cam->restart_timer,
		  jiffies + msecs_to_jiffies(MXC_RESTART_DELAY));

	return 0;
}

/*!
 * V4L2 - mxc_v4l2_enum_fmt function
 *
 * Enumerate the supported pixel formats.
 *
 * @param cam	      structure struct mx6_camera_dev *
 *
 * @param fmt	      struct v4l2_fmtdesc *
 *
 * @return  status    0 success, EINVAL
 */
static int mxc_v4l2_enum_fmt(struct mx6_camera_dev *cam,
			     struct v4l2_fmtdesc *fmt)
{
	if (fmt->index >= ARRAY_SIZE(pixel_fmts))
		return -EINVAL;

	fmt->pixelformat = pixel_fmts[fmt->index];

	return 0;
}

/*!
 * V4L2 - mxc_v4l2_enum_framesizes function
 *
 * Enumerate the supported frame sizes.
 *
 * @param cam	      structure struct mx6_camera_dev *
 *
 * @param fsize	      struct v4l2_frmsizeenum *
 *
 * @return  status    0 success, EINVAL
 */
static int mxc_v4l2_enum_framesizes(struct mx6_camera_dev *cam,
				    struct v4l2_frmsizeenum *fsize)
{
	struct v4l2_format uf;
	int ret = 0;

	if (!mxc_pixfmt_supported(fsize->pixel_format))
		return -EINVAL;

	if (!v4l2src_compat) {
		if (fsize->index)
			return -EINVAL;

		fsize->type = V4L2_FRMSIZE_TYPE_STEPWISE;
		fsize->stepwise.min_width = MIN_W;
		fsize->stepwise.step_width =
			ipu_pixelformat_is_planar(fsize->pixel_format) ? 16 : 8;
		fsize->stepwise.min_height = MIN_H;
		fsize->stepwise.step_height = 1 << H_ALIGN;

		uf = cam->user_fmt;
		uf.fmt.pix.pixelformat = fsize->pixel_format;

		if (need_ic(cam, &cam->sensor_fmt, &uf, &cam->crop_current)) {
			fsize->stepwise.max_width = MAX_W_IC;
			fsize->stepwise.max_height = MAX_H_IC;
		} else {
			fsize->stepwise.max_width = MAX_W;
			fsize->stepwise.max_height = MAX_H;
		}
	} else {
		ret = vidioc_int_enum_framesizes(cam->ep->sensor, fsize);
	}

	return ret;
}

/*!
 * V4L2 - mxc_v4l2_enum_frameintervals function
 *
 * Enumerate the supported frame intervals (frame rates).
 *
 * @param cam	      structure struct mx6_camera_dev *
 *
 * @param fival	      struct v4l2_frmivalenum *
 *
 * @return            0 success, EINVAL
 */
static int mxc_v4l2_enum_frameintervals(struct mx6_camera_dev *cam,
					struct v4l2_frmivalenum *fival)
{
	if (!mxc_pixfmt_supported(fival->pixel_format))
		return -EINVAL;

	return vidioc_int_enum_frameintervals(cam->ep->sensor, fival);
}


/*!
 * Dequeue one V4L capture buffer
 *
 * @param cam         structure struct mx6_camera_dev *
 * @param buf         structure v4l2_buffer *
 *
 * @return  status    0 success, EINVAL invalid frame number,
 *                    ETIME timeout, ERESTARTSYS interrupted by user
 */
static int mxc_v4l_dqueue(struct mx6_camera_dev *cam, struct v4l2_buffer *buf)
{
	int retval = 0;
	struct mxc_v4l_frame *frame;

	dev_dbg(cam->dev, "In MVC:mxc_v4l_dqueue\n");

retry:
	if (!wait_event_interruptible_timeout(
		    cam->enc_queue, cam->enc_counter != 0,
		    msecs_to_jiffies(MXC_DQ_TIMEOUT))) {
		dev_err(cam->dev, "ERROR: v4l2 capture: "		\
				"mxc_v4l_dqueue timeout enc_counter %x\n",
		       cam->enc_counter);
		return -ETIME;
	} else if (signal_pending(current)) {
		dev_dbg(cam->dev, "%s: interrupt received\n", __func__);
		return -ERESTARTSYS;
	}

	if (mutex_lock_interruptible(&cam->busy_lock))
		return -EBUSY;

	mutex_lock(&cam->irq_lock);

	/*
	 * we can race with the restart/stop handlers here, make sure a
	 * restart/stop didn't clear the done queue just after our
	 * wait_event above.
	 */
	if (!cam->enc_counter || list_empty(&cam->done_q)) {
		dev_dbg(cam->dev, "%s: race with restart\n", __func__);
		mutex_unlock(&cam->irq_lock);
		mutex_unlock(&cam->busy_lock);
		goto retry;
	}

	cam->enc_counter--;

	frame = list_entry(cam->done_q.next, struct mxc_v4l_frame, queue);
	list_del(cam->done_q.next);
	if (frame->buffer.flags & V4L2_BUF_FLAG_DONE) {
		frame->buffer.flags &= ~V4L2_BUF_FLAG_DONE;
	} else if (frame->buffer.flags & V4L2_BUF_FLAG_QUEUED) {
		dev_err(cam->dev, "ERROR: v4l2 capture: VIDIOC_DQBUF: " \
				"Buffer not filled.\n");
		frame->buffer.flags &= ~V4L2_BUF_FLAG_QUEUED;
		retval = -EINVAL;
	} else if ((frame->buffer.flags & 0x7) == V4L2_BUF_FLAG_MAPPED) {
		dev_err(cam->dev, "ERROR: v4l2 capture: VIDIOC_DQBUF: " \
				"Buffer not queued.\n");
		retval = -EINVAL;
	}

	buf->bytesused = cam->user_fmt.fmt.pix.sizeimage;
	buf->index = frame->index;
	buf->flags = frame->buffer.flags;
	buf->length = frame->buffer.length;
	buf->m = cam->frame[frame->index].buffer.m;
	buf->timestamp = cam->frame[frame->index].buffer.timestamp;

	mutex_unlock(&cam->irq_lock);

	mutex_unlock(&cam->busy_lock);
	return retval;
}

/*!
 * Queue one V4L capture buffer
 *
 * @param cam         structure struct mx6_camera_dev *
 * @param buf         structure v4l2_buffer *
 *
 * @return  status    0 success, EINVAL invalid requested buffer
 */
static int mxc_v4l_queue(struct mx6_camera_dev *cam, struct v4l2_buffer *buf)
{
	struct mxc_v4l_frame *frame = &cam->frame[buf->index];
	bool was_singular = false;
	int retval = 0;

	mutex_lock(&cam->irq_lock);

	if ((frame->buffer.flags & 0x7) == V4L2_BUF_FLAG_MAPPED) {
		frame->buffer.flags |= V4L2_BUF_FLAG_QUEUED;
		frame->buffer.flags &= ~V4L2_BUF_FLAG_ERROR;
		was_singular = list_is_singular(&cam->ready_q);
		list_add_tail(&frame->queue, &cam->ready_q);
	} else if (frame->buffer.flags & V4L2_BUF_FLAG_QUEUED) {
		dev_err(cam->dev, "ERROR: v4l2 capture: VIDIOC_QBUF: " \
			"buffer already queued\n");
		retval = -EINVAL;
	} else if (frame->buffer.flags & V4L2_BUF_FLAG_DONE) {
		dev_err(cam->dev, "ERROR: v4l2 capture: VIDIOC_QBUF: " \
			"overwrite done buffer.\n");
		frame->buffer.flags &= ~V4L2_BUF_FLAG_DONE;
		frame->buffer.flags |= V4L2_BUF_FLAG_QUEUED;
		retval = -EINVAL;
	}

	buf->flags = frame->buffer.flags;

	mutex_unlock(&cam->irq_lock);

	/*
	 * kickstart encoder if the ready queue now has two frames
	 * and streaming is enabled
	 */
	if (cam->streaming_on && was_singular)
		retval = start_encoder(cam);

	return retval;
}

/*!
 * V4L interface - open function
 *
 * @param file         structure file *
 *
 * @return  status    0 success, ENODEV invalid device instance,
 *                    ENODEV timeout, ERESTARTSYS interrupted by user
 */
static int mxc_v4l_open(struct file *file)
{
	struct video_device *dev = video_devdata(file);
	struct mx6_camera_dev *cam = video_get_drvdata(dev);
	int err = 0;

	dev_dbg(cam->dev, "\nIn MVC: mxc_v4l_open\n");
	dev_dbg(cam->dev, "   device name is %s\n", dev->name);

	if (!cam) {
		dev_err(cam->dev, "ERROR: v4l2 capture: Internal error, " \
				"struct mx6_camera_dev not found!\n");
		return -EBADF;
	}

	mutex_lock(&cam->busy_lock);
	err = 0;
	if (signal_pending(current))
		goto out;

	if (cam->ep == NULL || cam->ep->sensor == NULL ||
	    cam->ep->sensor->type != v4l2_int_type_slave) {
		dev_dbg(cam->dev, "%s: no slave attached yet\n", __func__);
		err = -EAGAIN;
		goto out;
	}

	if (cam->open_count++ == 0) {
		wait_event_interruptible(cam->power_queue,
					 cam->low_power == false);

		cam->enc_counter = 0;
		INIT_LIST_HEAD(&cam->ready_q);
		INIT_LIST_HEAD(&cam->working_q);
		INIT_LIST_HEAD(&cam->done_q);

		mxc_update_sensor_fmt(cam);
		vidioc_int_querystd(cam->ep->sensor, &cam->current_std);
		mxc_update_signal_lock_status(cam);

		vidioc_int_s_power(cam->ep->sensor, 1);
		vidioc_int_init(cam->ep->sensor);
		vidioc_int_dev_init(cam->ep->sensor);

		if (cam->ep->ep.bus_type == V4L2_MBUS_CSI2 && cam->csi2) {
			/* power-up the mipi csi-2 receiver */
			vidioc_int_s_power(cam->csi2, 1);
		}
	}

	file->private_data = dev;
out:
	mutex_unlock(&cam->busy_lock);
	return err;
}

/*!
 * V4L interface - close function
 *
 * @param file     struct file *
 *
 * @return         0 success
 */
static int mxc_v4l_close(struct file *file)
{
	struct video_device *dev = video_devdata(file);
	int err = 0;
	struct mx6_camera_dev *cam = video_get_drvdata(dev);

	dev_dbg(cam->dev, "In MVC:mxc_v4l_close\n");

	if (!cam) {
		dev_err(cam->dev, "ERROR: v4l2 capture: Internal error, " \
				"struct mx6_camera_dev not found!\n");
		return -EBADF;
	}

	mutex_lock(&cam->busy_lock);

	if (cam->ep == NULL || cam->ep->sensor == NULL) {
		dev_dbg(cam->dev, "%s: lost the slave?\n", __func__);
		goto out;
	}

	/* for the case somebody hit the ctrl C */
	if (cam->overlay_pid == current->tgid && cam->overlay_on) {
		err = stop_preview(cam);
		cam->overlay_on = false;
	}
	if (cam->capture_pid == current->tgid) {
		err |= mxc_streamoff(cam);
		wake_up_interruptible(&cam->enc_queue);
	}

	if (--cam->open_count == 0) {
		vidioc_int_s_power(cam->ep->sensor, 0);

		wait_event_interruptible(cam->power_queue,
					 cam->low_power == false);
		dev_dbg(cam->dev, "mxc_v4l_close: release resource\n");

		mxc_free_frame_buf(cam);
		file->private_data = NULL;

		/* capture off */
		wake_up_interruptible(&cam->enc_queue);
		mxc_free_frames(cam);
		cam->enc_counter++;
	}

out:
	mutex_unlock(&cam->busy_lock);
	return err;
}

/*
 * V4L interface - read function
 *
 * @param file       struct file *
 * @param read buf   char *
 * @param count      size_t
 * @param ppos       structure loff_t *
 *
 * @return           bytes read
 */
static ssize_t mxc_v4l_read(struct file *file, char __user *buf, size_t count,
			    loff_t *ppos)
{
	int err = 0;
	struct video_device *dev = video_devdata(file);
	struct mx6_camera_dev *cam = video_get_drvdata(dev);

	if (mutex_lock_interruptible(&cam->busy_lock))
		return -EINTR;

	if (cam->ep == NULL || cam->ep->sensor == NULL) {
		dev_err(cam->dev, "%s: no slave attached!\n", __func__);
		err = -ENODEV;
		goto out_unlock;
	}

	if (cam->streaming_on) {
		dev_err(cam->dev, "ERROR: %s: not allowed while streaming\n",
			__func__);
		err = -EBUSY;
		goto out_unlock;
	}

	/* Stop the viewfinder */
	if (cam->overlay_on == true)
		stop_preview(cam);

	err = prp_still_select(cam);
	if (err != 0) {
		err = -EIO;
		goto exit;
	}

	err = cam->csi_start(cam, buf);
	if (err != 0) {
		prp_still_deselect(cam);
		goto exit;
	}

	prp_still_deselect(cam);

exit:
	if (cam->overlay_on == true)
		start_preview(cam);

out_unlock:
	mutex_unlock(&cam->busy_lock);
	if (err < 0)
		return err;

	return cam->user_fmt.fmt.pix.sizeimage - err;
}

/*!
 * V4L interface - ioctl function
 *
 * @param file       struct file*
 *
 * @param ioctlnr    unsigned int
 *
 * @param arg        void*
 *
 * @return           0 for success, or appropriate error code.
 */
static long mxc_v4l_do_ioctl(struct file *file,
			    unsigned int ioctlnr, void *arg)
{
	struct video_device *dev = video_devdata(file);
	struct mx6_camera_dev *cam = video_get_drvdata(dev);
	int retval = 0;

	dev_dbg(cam->dev, "In MVC: mxc_v4l_do_ioctl %x\n", ioctlnr);
	wait_event_interruptible(cam->power_queue, cam->low_power == false);
	/* make this _really_ smp-safe */
	if (ioctlnr != VIDIOC_DQBUF)
		if (mutex_lock_interruptible(&cam->busy_lock))
			return -EBUSY;

	if (cam->ep == NULL || cam->ep->sensor == NULL) {
		dev_err(cam->dev, "ERROR: %s: no slave attached!\n", __func__);
		retval = -ENODEV;
		goto out;
	}

	switch (ioctlnr) {
	/*!
	 * V4l2 VIDIOC_QUERYCAP ioctl
	 */
	case VIDIOC_QUERYCAP: {
		struct v4l2_capability *cap = arg;
		dev_dbg(cam->dev, "   case VIDIOC_QUERYCAP\n");
		strcpy(cap->driver, "mxc_v4l2");
		cap->version = KERNEL_VERSION(0, 1, 11);
		cap->capabilities = V4L2_CAP_VIDEO_CAPTURE |
				    V4L2_CAP_VIDEO_OVERLAY |
				    V4L2_CAP_STREAMING |
				    V4L2_CAP_READWRITE;
		cap->card[0] = '\0';
		cap->bus_info[0] = '\0';
		break;
	}

	/*!
	 * V4l2 VIDIOC_G_FMT ioctl
	 */
	case VIDIOC_G_FMT: {
		struct v4l2_format *gf = arg;
		dev_dbg(cam->dev, "   case VIDIOC_G_FMT\n");
		retval = mxc_v4l2_g_fmt(cam, gf);
		break;
	}

	/*!
	 * V4l2 VIDIOC_TRY_FMT ioctl
	 */
	case VIDIOC_TRY_FMT: {
		struct v4l2_format *f = arg;
		retval = mxc_v4l2_try_fmt(cam, f);
		break;
	}

	/*!
	 * V4l2 VIDIOC_S_FMT ioctl
	 */
	case VIDIOC_S_FMT: {
		struct v4l2_format *sf = arg;
		dev_dbg(cam->dev, "   case VIDIOC_S_FMT\n");
		retval = mxc_v4l2_s_fmt(cam, sf);
		break;
	}

	/*!
	 * V4l2 VIDIOC_REQBUFS ioctl
	 */
	case VIDIOC_REQBUFS: {
		struct v4l2_requestbuffers *req = arg;
		dev_dbg(cam->dev, "   case VIDIOC_REQBUFS\n");
		retval = mxc_v4l2_reqbufs(cam, req);
		break;
	}

	/*!
	 * V4l2 VIDIOC_QUERYBUF ioctl
	 */
	case VIDIOC_QUERYBUF: {
		struct v4l2_buffer *buf = arg;
		dev_dbg(cam->dev, "   case VIDIOC_QUERYBUF\n");
		retval = mxc_v4l2_querybuf(cam, buf);
		break;
	}

	/*!
	 * V4l2 VIDIOC_QBUF ioctl
	 */
	case VIDIOC_QBUF: {
		struct v4l2_buffer *buf = arg;
		dev_dbg(cam->dev, "   case VIDIOC_QBUF\n");
		retval = mxc_v4l_queue(cam, buf);
		break;
	}

	/*!
	 * V4l2 VIDIOC_DQBUF ioctl
	 */
	case VIDIOC_DQBUF: {
		struct v4l2_buffer *buf = arg;
		dev_dbg(cam->dev, "   case VIDIOC_DQBUF\n");

		if ((cam->enc_counter == 0) &&
			(file->f_flags & O_NONBLOCK)) {
			retval = -EAGAIN;
			break;
		}

		retval = mxc_v4l_dqueue(cam, buf);
		break;
	}

	/*!
	 * V4l2 VIDIOC_STREAMON ioctl
	 */
	case VIDIOC_STREAMON: {
		dev_dbg(cam->dev, "   case VIDIOC_STREAMON\n");
		retval = mxc_streamon(cam);
		if (!retval)
			cam->capture_pid = current->tgid;
		break;
	}

	/*!
	 * V4l2 VIDIOC_STREAMOFF ioctl
	 */
	case VIDIOC_STREAMOFF: {
		dev_dbg(cam->dev, "   case VIDIOC_STREAMOFF\n");
		retval = mxc_streamoff(cam);
		break;
	}

	/*!
	 * V4l2 VIDIOC_G_CTRL ioctl
	 */
	case VIDIOC_G_CTRL: {
		dev_dbg(cam->dev, "   case VIDIOC_G_CTRL\n");
		retval = mxc_v4l2_g_ctrl(cam, arg);
		break;
	}

	/*!
	 * V4l2 VIDIOC_S_CTRL ioctl
	 */
	case VIDIOC_S_CTRL: {
		dev_dbg(cam->dev, "   case VIDIOC_S_CTRL\n");
		retval = mxc_v4l2_s_ctrl(cam, arg);
		break;
	}

	/*!
	 * V4l2 VIDIOC_CROPCAP ioctl
	 */
	case VIDIOC_CROPCAP: {
		struct v4l2_cropcap *cap = arg;
		dev_dbg(cam->dev, "   case VIDIOC_CROPCAP\n");
		if (cap->type != V4L2_BUF_TYPE_VIDEO_CAPTURE &&
		    cap->type != V4L2_BUF_TYPE_VIDEO_OVERLAY) {
			retval = -EINVAL;
			break;
		}
		cap->bounds = cam->crop_bounds;
		cap->defrect = cam->crop_defrect;
		cap->pixelaspect.numerator = 1;
		cap->pixelaspect.denominator = 1;
		break;
	}

	/*!
	 * V4l2 VIDIOC_G_CROP ioctl
	 */
	case VIDIOC_G_CROP: {
		struct v4l2_crop *crop = arg;
		dev_dbg(cam->dev, "   case VIDIOC_G_CROP\n");

		if (crop->type != V4L2_BUF_TYPE_VIDEO_CAPTURE &&
		    crop->type != V4L2_BUF_TYPE_VIDEO_OVERLAY) {
			retval = -EINVAL;
			break;
		}
		crop->c = cam->crop_current;
		break;
	}

	/*!
	 * V4l2 VIDIOC_S_CROP ioctl
	 */
	case VIDIOC_S_CROP: {
		struct v4l2_crop *crop = arg;
		dev_dbg(cam->dev, "   case VIDIOC_S_CROP\n");
		retval = mxc_v4l2_s_crop(cam, crop);
		break;
	}

	/*!
	 * V4l2 VIDIOC_OVERLAY ioctl
	 */
	case VIDIOC_OVERLAY: {
		int *on = arg;
		dev_dbg(cam->dev, "   VIDIOC_OVERLAY on=%d\n", *on);
		retval = mxc_v4l2_overlay(cam, *on ? true : false);
		break;
	}

	/*!
	 * V4l2 VIDIOC_G_FBUF ioctl
	 */
	case VIDIOC_G_FBUF: {
		struct v4l2_framebuffer *fb = arg;
		dev_dbg(cam->dev, "   case VIDIOC_G_FBUF\n");
		*fb = cam->v4l2_fb;
//		fb->capability = V4L2_FBUF_CAP_EXTERNOVERLAY;
		break;
	}

	/*!
	 * V4l2 VIDIOC_S_FBUF ioctl
	 */
	case VIDIOC_S_FBUF: {
		struct v4l2_framebuffer *fb = arg;
		dev_dbg(cam->dev, "   case VIDIOC_S_FBUF\n");
		if (cam->overlay_on == true)
			stop_preview(cam);
		cam->v4l2_fb = *fb;
		if (cam->overlay_on == true)
			start_preview(cam);
		break;
	}

	case VIDIOC_G_PARM: {
		struct v4l2_streamparm *parm = arg;
		dev_dbg(cam->dev, "   case VIDIOC_G_PARM\n");
		retval = vidioc_int_g_parm(cam->ep->sensor, parm);
		break;
	}

	case VIDIOC_S_PARM:  {
		struct v4l2_streamparm *parm = arg;
		dev_dbg(cam->dev, "   case VIDIOC_S_PARM\n");
		retval = mxc_v4l2_s_param(cam, parm);
		break;
	}

	/* linux v4l2 bug, kernel c0485619 user c0405619 */
	case VIDIOC_ENUMSTD: {
		struct v4l2_standard *e = arg;
		dev_dbg(cam->dev, "   case VIDIOC_ENUMSTD\n");
		if (e->index >= TV_STD_MAX) {
			retval = -EINVAL;
			break;
		}
		*e = video_fmts[e->index].standard;
		break;
	}

	case VIDIOC_G_STD: {
		v4l2_std_id *e = arg;
		dev_dbg(cam->dev, "   case VIDIOC_G_STD\n");
		retval = mxc_v4l2_g_std(cam, e);
		break;
	}

	case VIDIOC_S_STD: {
		v4l2_std_id *e = arg;
		dev_dbg(cam->dev, "   case VIDIOC_S_STD\n");
		retval = mxc_v4l2_s_std(cam, *e);

		break;
	}

	case VIDIOC_ENUMINPUT: {
		struct v4l2_input *input = arg;
		dev_dbg(cam->dev, "   case VIDIOC_ENUMINPUT\n");
		retval = mxc_v4l2_enum_input(cam, input);
		break;
	}

	case VIDIOC_G_INPUT: {
		int *index = arg;
		dev_dbg(cam->dev, "   case VIDIOC_G_INPUT\n");
		*index = cam->current_input;
		break;
	}

	case VIDIOC_S_INPUT: {
		int *index = arg;
		dev_dbg(cam->dev, "   case VIDIOC_S_INPUT\n");
		retval = mxc_v4l2_s_input(cam, *index);
		break;
	}

	case VIDIOC_ENUM_FMT: {
		struct v4l2_fmtdesc *f = arg;
		retval = mxc_v4l2_enum_fmt(cam, f);
		break;
	}
	case VIDIOC_ENUM_FRAMESIZES: {
		struct v4l2_frmsizeenum *fsize = arg;
		retval = mxc_v4l2_enum_framesizes(cam, fsize);
		break;
	}
	case VIDIOC_ENUM_FRAMEINTERVALS: {
		struct v4l2_frmivalenum *fival = arg;
		retval = mxc_v4l2_enum_frameintervals(cam, fival);
		break;
	}
	case VIDIOC_DBG_G_CHIP_IDENT: {
		struct v4l2_dbg_chip_ident *p = arg;
		p->ident = V4L2_IDENT_NONE;
		p->revision = 0;
		retval = vidioc_int_g_chip_ident(cam->ep->sensor, (int *)p);
		break;
	}
	case VIDIOC_QUERYCTRL: {
		struct v4l2_queryctrl *qc = arg;
		retval = mxc_v4l2_queryctrl(cam, qc);
		break;
	}
	case VIDIOC_QUERYMENU: {
		struct v4l2_querymenu *qm = arg;
		retval = mxc_v4l2_querymenu(cam, qm);
		break;
	}
	case VIDIOC_G_TUNER:
	case VIDIOC_S_TUNER:
	case VIDIOC_G_FREQUENCY:
	case VIDIOC_S_FREQUENCY:
	default:
		dev_dbg(cam->dev, "   case default or not supported\n");
		retval = -EINVAL;
		break;
	}

out:
	if (ioctlnr != VIDIOC_DQBUF)
		mutex_unlock(&cam->busy_lock);
	return retval;
}

/*
 * V4L interface - ioctl function
 *
 * @return  None
 */
static long mxc_v4l_ioctl(struct file *file, unsigned int cmd,
			 unsigned long arg)
{
	return video_usercopy(file, cmd, arg, mxc_v4l_do_ioctl);
}

/*!
 * V4L interface - mmap function
 *
 * @param file        structure file *
 *
 * @param vma         structure vm_area_struct *
 *
 * @return status     0 Success, EINTR busy lock error, ENOBUFS remap_page error
 */
static int mxc_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct video_device *dev = video_devdata(file);
	unsigned long size;
	int res = 0;
	struct mx6_camera_dev *cam = video_get_drvdata(dev);

	dev_dbg(cam->dev, "In MVC:mxc_mmap\n");
	dev_dbg(cam->dev, "   pgoff=0x%lx, start=0x%lx, end=0x%lx\n",
		 vma->vm_pgoff, vma->vm_start, vma->vm_end);

	/* make this _really_ smp-safe */
	if (mutex_lock_interruptible(&cam->busy_lock))
		return -EINTR;

	if (cam->ep == NULL || cam->ep->sensor == NULL) {
		dev_err(cam->dev, "%s: no slave attached!\n", __func__);
		res = -ENODEV;
		goto mxc_mmap_exit;
	}

	size = vma->vm_end - vma->vm_start;
	vma->vm_page_prot = pgprot_writecombine(vma->vm_page_prot);

	if (remap_pfn_range(vma, vma->vm_start,
			    vma->vm_pgoff, size, vma->vm_page_prot)) {
		dev_err(cam->dev, "ERROR: v4l2 capture: mxc_mmap: " \
				"remap_pfn_range failed\n");
		res = -ENOBUFS;
		goto mxc_mmap_exit;
	}

	vma->vm_flags &= ~VM_IO;	/* using shared anonymous pages */

mxc_mmap_exit:
	mutex_unlock(&cam->busy_lock);
	return res;
}

/*!
 * V4L interface - poll function
 *
 * @param file       structure file *
 *
 * @param wait       structure poll_table_struct *
 *
 * @return  status   POLLIN | POLLRDNORM
 */
static unsigned int mxc_poll(struct file *file, struct poll_table_struct *wait)
{
	struct video_device *dev = video_devdata(file);
	struct mx6_camera_dev *cam = video_get_drvdata(dev);
	wait_queue_head_t *queue = NULL;
	int res = POLLIN | POLLRDNORM;

	dev_dbg(cam->dev, "In MVC:mxc_poll\n");

	if (mutex_lock_interruptible(&cam->busy_lock))
		return -EINTR;

	if (cam->ep == NULL || cam->ep->sensor == NULL) {
		dev_err(cam->dev, "%s: no slave attached!\n", __func__);
		res = -ENODEV;
		goto out;
	}

	queue = &cam->enc_queue;
	poll_wait(file, queue, wait);
out:
	mutex_unlock(&cam->busy_lock);
	return res;
}

/*!
 * This structure defines the functions to be called in this driver.
 */
static struct v4l2_file_operations mxc_v4l_fops = {
	.owner = THIS_MODULE,
	.open = mxc_v4l_open,
	.release = mxc_v4l_close,
	.read = mxc_v4l_read,
	.ioctl = mxc_v4l_ioctl,
	.mmap = mxc_mmap,
	.poll = mxc_poll,
};

static struct video_device mxc_v4l_template = {
	.name = "Mxc Camera",
	.fops = &mxc_v4l_fops,
	.release = video_device_release,
	.vfl_dir = VFL_DIR_RX,
};

/*!
 * This function can be used to release any platform data on closing.
 */
static void camera_platform_release(struct device *device)
{
}

/*!
 * Camera V4l2 callback function.
 *
 * @param mask      u32
 *
 * @param cam       structure struct mx6_camera_dev *
 *
 * @return status
 */
static void camera_callback(struct mx6_camera_dev *cam, u32 mask)
{
	struct mxc_v4l_frame *done_frame;
	struct mxc_v4l_frame *ready_frame;
	struct timeval cur_time;

	if (cam == NULL)
		return;

	dev_dbg(cam->dev, "In MVC:camera_callback\n");

	mutex_lock(&cam->irq_lock);

	/*
	 * if we are in the process of restarting, abort any work
	 * and exit.
	 */
	if (cam->restarting)
		goto unlock;

	mxc_v4l2_poll_std_and_lock_status(cam);

	/* bump the EOF timeout timer */
	mod_timer(&cam->eof_timer,
		  jiffies + msecs_to_jiffies(MXC_EOF_TIMEOUT));

	if (!list_empty(&cam->working_q)) {
		do_gettimeofday(&cur_time);

		done_frame = list_entry(cam->working_q.next,
					struct mxc_v4l_frame,
					queue);

		if (done_frame->ipu_buf_num != cam->current_buf_num)
			goto next;

		/*
		 * Set the current time to done frame buffer's
		 * timestamp. Users can use this information to judge
		 * the frame's usage.
		 */
		done_frame->buffer.timestamp = cur_time;

		if (done_frame->buffer.flags & V4L2_BUF_FLAG_QUEUED) {
			done_frame->buffer.flags |= V4L2_BUF_FLAG_DONE;
			done_frame->buffer.flags &= ~V4L2_BUF_FLAG_QUEUED;
			if (!cam->signal_locked)
				done_frame->buffer.flags |= V4L2_BUF_FLAG_ERROR;

			/* Added to the done queue */
			list_del(cam->working_q.next);
			list_add_tail(&done_frame->queue, &cam->done_q);

			/* Wake up the queue */
			cam->enc_counter++;
			wake_up_interruptible(&cam->enc_queue);
		} else
			dev_err(cam->dev, "ERROR: v4l2 capture: " \
					"camera_callback: buffer not queued\n");
	}

next:
	if (!list_empty(&cam->ready_q)) {
		ready_frame = list_entry(cam->ready_q.next,
					 struct mxc_v4l_frame,
					 queue);
		ready_frame->ipu_buf_num = cam->current_buf_num;
		if (cam->enc_update_eba) {
			dma_addr_t eba = ready_frame->buffer.m.offset;
			if (cam->enc_update_eba(cam, eba,
						cam->current_buf_num) == 0) {
				list_del(cam->ready_q.next);
				list_add_tail(&ready_frame->queue,
					      &cam->working_q);
			}
		}
	} else {
		dev_dbg(cam->dev, "underrun!\n");
		if (cam->enc_update_eba)
			cam->enc_update_eba(
				cam, cam->dummy_frame.buffer.m.offset,
				cam->current_buf_num);
	}

	cam->current_buf_num ^= 1;

unlock:
	mutex_unlock(&cam->irq_lock);
}

/*!
 * Restart work handler. This is called in two cases: NFB4EOF errors,
 * and when the TVin signal lock status changes mid-streaming and/or
 * mid-preview.
 */
static void restart_work_handler(struct work_struct *w)
{
	struct mx6_camera_dev *cam = container_of(w, struct mx6_camera_dev,
						  restart_work);
	struct list_head tmp_queue;
	struct mxc_v4l_frame *frame;

	INIT_LIST_HEAD(&tmp_queue);

	mutex_lock(&cam->busy_lock);

	if (!cam->streaming_on) {
		/* just restart preview if on */
		if (cam->overlay_on == true) {
			dev_warn(cam->dev, "restarting preview\n");
			stop_preview(cam);
			start_preview(cam);
		}
		goto out_unlock;
	}

	dev_warn(cam->dev, "restarting\n");

	mutex_lock(&cam->irq_lock);

	/*
	 * inform camera_callback that we are restarting so that it will
	 * abort any work on the queues
	 */
	cam->restarting = true;

	/* collect buffers from all queues and store them in tmp_queue */
	list_splice(&cam->working_q, &tmp_queue);
	list_splice(&cam->ready_q, &tmp_queue);
	list_splice(&cam->done_q, &tmp_queue);

	mutex_unlock(&cam->irq_lock);

	/* stop capturing */
	mxc_streamoff(cam);

	/*
	 * reset the buffer flags and enqueue them back to ready_q
	 *
	 * We don't need to hold irq_lock here b/c streaming is stopped
	 * and we hold the busy_lock so no new activity can be started
	 * by another thread.
	 */
	list_for_each_entry(frame, &tmp_queue, queue) {
		frame->buffer.flags |= V4L2_BUF_FLAG_QUEUED;
		frame->buffer.flags &= ~V4L2_BUF_FLAG_ERROR;
		frame->buffer.flags &= ~V4L2_BUF_FLAG_DONE;
	}
	list_splice(&tmp_queue, &cam->ready_q);

	cam->restarting = false;

	/* start capturing (this also restarts preview) */
	mxc_streamon(cam);

out_unlock:
	mutex_unlock(&cam->busy_lock);
}

/*!
 * Stop work handler. This used to be called when the TVin video
 * standard changed mid-streaming and/or mid-preview, but no longer.
 * Keep it around in case it comes in handy later.
 */
static void stop_work_handler(struct work_struct *w)
{
	struct mx6_camera_dev *cam = container_of(w, struct mx6_camera_dev,
						  stop_work);

	mutex_lock(&cam->busy_lock);

	if (cam->overlay_on == true) {
		dev_err(cam->dev, "stopping preview\n");
		stop_preview(cam);
		cam->overlay_on = false;
	}

	if (cam->streaming_on) {
		dev_err(cam->dev, "stopping\n");
		mxc_streamoff(cam);
	}

	mutex_unlock(&cam->busy_lock);
}

/*!
 * Lock status change and NFB4EOF timeout timer function.
 * Schedules a restart.
 */
static void mxc_lsc_nfb4eof_timeout(unsigned long data)
{
	struct mx6_camera_dev *cam = (struct mx6_camera_dev *)data;

	schedule_work(&cam->restart_work);
}

/*!
 * EOF timeout timer function. Schedules a restart.
 */
static void mxc_eof_timeout(unsigned long data)
{
	struct mx6_camera_dev *cam = (struct mx6_camera_dev *)data;

	dev_err(cam->dev, "ERROR: EOF timeout\n");
	/*
	 * cancel a running restart timer from a LSC or NFB4EOF
	 * since we are restarting now anyway.
	 */
	del_timer_sync(&cam->restart_timer);

	schedule_work(&cam->restart_work);
}

static int mxc_parse_endpoints(struct mx6_camera_dev *cam,
			       struct device_node *node)
{
	struct device_node *remote, *epnode = NULL;
	struct v4l2_of_endpoint ep;
	int err, csi_id, i;

	for (i = 0; i < MXC_MAX_ENDPOINTS; i++) {
		epnode = v4l2_of_get_next_endpoint(node, epnode);
		if (!epnode)
			break;

		v4l2_of_parse_endpoint(epnode, &ep);

		csi_id = ep.port;
		if (csi_id > 1) {
			dev_err(cam->dev, "ERROR: invalid port %d\n", csi_id);
			err = -EINVAL;
			goto out;
		}

		dev_dbg(cam->dev, "found %s EP%d: port %d, id %d\n",
			ep.bus_type == V4L2_MBUS_CSI2 ? "CSI-2" : "Parallel",
			i, ep.port, ep.id);

		cam->all_eps[i].ep = ep;

		remote = v4l2_of_get_remote_port_parent(epnode);
		if (!remote) {
			dev_err(cam->dev,
				"ERROR: no remote endpoint defined\n");
			err = -EINVAL;
			goto out;
		}

		cam->all_eps[i].remote_node = remote;

		of_node_put(remote);
		of_node_put(epnode);
	}

	cam->num_eps = i;
	if (!cam->num_eps) {
		dev_err(cam->dev, "ERROR: no endpoints defined\n");
		return -EINVAL;
	}

	cam->ep = &cam->all_eps[0];
	return 0;
out:
	of_node_put(epnode);
	return err;
}

/*!
 * initialize struct mx6_camera_dev structure
 *
 * @param cam      structure struct mx6_camera_dev *
 *
 * @return status  0 Success
 */
static int init_camera_struct(struct mx6_camera_dev *cam,
			      struct platform_device *pdev)
{
	struct device_node *node = pdev->dev.of_node;
	struct pinctrl *pinctrl;
	int ret = 0;

	/* Default everything to 0 */
	memset(cam, 0, sizeof(struct mx6_camera_dev));

	cam->dev = &pdev->dev;
	dev_dbg(cam->dev, "In MVC: init_camera_struct\n");
	cam->ipu = dev_get_drvdata(cam->dev->parent);
	if (IS_ERR_OR_NULL(cam->ipu)) {
		dev_err(cam->dev,
			"ERROR: v4l2 capture: failed to get parent ipu\n");
		return -ENODEV;
	}

	mutex_init(&cam->param_lock);
	mutex_init(&cam->busy_lock);

	cam->video_dev = video_device_alloc();
	if (cam->video_dev == NULL) {
		dev_err(cam->dev,
			"ERROR: v4l2 capture: video device alloc failed\n");
		return -ENOMEM;
	}

	*(cam->video_dev) = mxc_v4l_template;

	video_set_drvdata(cam->video_dev, cam);
	dev_set_drvdata(&pdev->dev, (void *)cam);
	cam->video_dev->minor = -1;

	init_waitqueue_head(&cam->enc_queue);
	INIT_WORK(&cam->restart_work, restart_work_handler);
	INIT_WORK(&cam->stop_work, stop_work_handler);

	init_timer(&cam->restart_timer);
	init_timer(&cam->eof_timer);
	init_timer(&cam->eof_vf_timer);
	cam->restart_timer.data = (unsigned long)cam;
	cam->restart_timer.function = mxc_lsc_nfb4eof_timeout;
	cam->eof_timer.data = (unsigned long)cam;
	cam->eof_timer.function = mxc_eof_timeout;
	cam->eof_vf_timer.data = (unsigned long)cam;
	cam->eof_vf_timer.function = mxc_eof_timeout;

	/* Get the CSI pins */
	pinctrl = devm_pinctrl_get_select_default(&pdev->dev);
	if (IS_ERR(pinctrl)) {
		dev_err(cam->dev, "ERROR: failed to acquire CSI pins\n");
		ret = PTR_ERR(pinctrl);
		goto out;
	}

	ret = mxc_parse_endpoints(cam, node);
	if (ret)
		goto out;

	/* init cropping with some default values */
	cam->crop_bounds.left = 0;
	cam->crop_bounds.width = 640;
	cam->crop_bounds.top = 0;
	cam->crop_bounds.height = 480;
	cam->crop_current = cam->crop_defrect = cam->crop_bounds;

	cam->standard.index = 0;
	cam->standard.id = V4L2_STD_UNKNOWN;
	cam->standard.frameperiod.denominator = 30;
	cam->standard.frameperiod.numerator = 1;
	cam->standard.framelines = 480;
	cam->streamparm.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	cam->streamparm.parm.capture.capturemode = 0;
	cam->streamparm.parm.capture.timeperframe = cam->standard.frameperiod;
	cam->streamparm.parm.capture.capability = V4L2_CAP_TIMEPERFRAME;
	cam->overlay_on = false;
	cam->streaming_on = false;
	cam->v4l2_fb.flags = V4L2_FBUF_FLAG_OVERLAY;

	cam->user_fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	cam->user_fmt.fmt.pix.sizeimage = 640 * 480 * 3 / 2;
	cam->user_fmt.fmt.pix.bytesperline = 640 * 3 / 2;
	cam->user_fmt.fmt.pix.width = 640;
	cam->user_fmt.fmt.pix.height = 480;
	cam->user_fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_YUV420;
	cam->win.w.width = 160;
	cam->win.w.height = 160;
	cam->win.w.left = 0;
	cam->win.w.top = 0;

	cam->enc_callback = camera_callback;
	init_waitqueue_head(&cam->power_queue);
	mutex_init(&cam->irq_lock);

	return 0;

out:
	video_device_release(cam->video_dev);
	return ret;
}

static ssize_t show_streaming(struct device *dev,
			struct device_attribute *attr, char *buf)
{
	struct video_device *video_dev = container_of(dev,
						struct video_device, dev);
	struct mx6_camera_dev *cam = video_get_drvdata(video_dev);

	if (cam->streaming_on)
		return sprintf(buf, "stream on\n");
	else
		return sprintf(buf, "stream off\n");
}
static DEVICE_ATTR(fsl_v4l2_capture_property, S_IRUGO, show_streaming, NULL);

static ssize_t show_overlay(struct device *dev,
			struct device_attribute *attr, char *buf)
{
	struct video_device *video_dev = container_of(dev,
						struct video_device, dev);
	struct mx6_camera_dev *cam = video_get_drvdata(video_dev);

	if (cam->overlay_on)
		return sprintf(buf, "overlay on\n");
	else
		return sprintf(buf, "overlay off\n");
}
static DEVICE_ATTR(fsl_v4l2_overlay_property, S_IRUGO, show_overlay, NULL);

/*!
 * This function is called to probe the devices if registered.
 *
 * @param   pdev  the device structure used to give information on which device
 *                to probe
 *
 * @return  The function returns 0 on success or error code
 */
static int mxc_v4l2_probe(struct platform_device *pdev)
{
	struct mx6_camera_dev *cam;
	int err;

	cam = kmalloc(sizeof(struct mx6_camera_dev), GFP_KERNEL);
	if (!cam)
		return -ENOMEM;

	err = init_camera_struct(cam, pdev);
	if (err)
		goto out_free;

	pdev->dev.release = camera_platform_release;
	pdev->dev.coherent_dma_mask = DMA_BIT_MASK(32);

	/* init the IPU encoder submodules */
	err = prp_still_init(cam);
	if (err)
		goto out_release;
	err = csi_enc_init(cam);
	if (err)
		goto out_free_still;
	err = prp_enc_init(cam);
	if (err)
		goto out_free_csi_enc;
	err = prp_vf_sdc_init(cam);
	if (err)
		goto out_free_prpenc;

	/* register v4l video device */
	cam->video_dev->parent = &pdev->dev;
	err = video_register_device(cam->video_dev, VFL_TYPE_GRABBER, 0);
	if (err) {
		dev_err(cam->dev,
			"ERROR: v4l2 capture: video_register_device failed\n");
		goto out_free_prpvf;
	}

	/* Set up our master internal device and register it */
	cam->this.module = THIS_MODULE;
	sprintf(cam->this.name, "%s%d", MX6_CAM_DRV_NAME, cam->video_dev->num);
	cam->this.type = v4l2_int_type_master;
	cam->this.u.master = &mxc_v4l2_master;
	cam->this.priv = cam;

	v4l2_int_device_register(&cam->this);

	if (device_create_file(&cam->video_dev->dev,
			&dev_attr_fsl_v4l2_capture_property))
		dev_err(&pdev->dev, "Error on creating sysfs file" \
				" for capture\n");

	if (device_create_file(&cam->video_dev->dev,
			&dev_attr_fsl_v4l2_overlay_property))
		dev_err(&pdev->dev, "Error on creating sysfs file" \
				" for overlay\n");

	dev_info(cam->dev,
		 "Device registered as /dev/video%d, parent is ipu%d\n",
		 cam->video_dev->num, ipu_get_num(cam->ipu));
	return 0;

out_free_prpvf:
	prp_vf_sdc_exit(cam);
out_free_prpenc:
	prp_enc_exit(cam);
out_free_csi_enc:
	csi_enc_exit(cam);
out_free_still:
	prp_still_exit(cam);
out_release:
	video_device_release(cam->video_dev);
out_free:
	kfree(cam);
	return err;
}

/*!
 * This function is called to remove the devices when device unregistered.
 *
 * @param   pdev  the device structure used to give information on which device
 *                to remove
 *
 * @return  The function returns 0 on success or error code
 */
static int mxc_v4l2_remove(struct platform_device *pdev)
{
	struct mx6_camera_dev *cam = platform_get_drvdata(pdev);

	if (cam->open_count) {
		dev_err(cam->dev, "ERROR: v4l2 capture:camera open " \
				"-- setting ops to NULL\n");
		return -EBUSY;
	} else {
		device_remove_file(&cam->video_dev->dev,
			&dev_attr_fsl_v4l2_capture_property);
		device_remove_file(&cam->video_dev->dev,
			&dev_attr_fsl_v4l2_overlay_property);

		dev_info(cam->dev, "V4L2 freeing image input device\n");
		v4l2_int_device_unregister(&cam->this);
		video_unregister_device(cam->video_dev);

		prp_vf_sdc_exit(cam);
		prp_enc_exit(cam);
		csi_enc_exit(cam);
		prp_still_exit(cam);

		mxc_free_frame_buf(cam);
		kfree(cam);
	}

	dev_info(cam->dev, "V4L2 unregistering video\n");
	return 0;
}

/*!
 * This function is called to put the sensor in a low power state.
 * Refer to the document driver-model/driver.txt in the kernel source tree
 * for more information.
 *
 * @param   pdev  the device structure used to give information on which I2C
 *                to suspend
 * @param   state the power state the device is entering
 *
 * @return  The function returns 0 on success and -ENODEV on failure.
 */
static int mxc_v4l2_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct mx6_camera_dev *cam = platform_get_drvdata(pdev);
	int ret = 0;

	dev_dbg(cam->dev, "In MVC:mxc_v4l2_suspend\n");

	mutex_lock(&cam->busy_lock);

	if (cam == NULL || cam->ep == NULL || cam->ep->sensor == NULL) {
		ret = -ENODEV;
		goto out;
	}

	cam->low_power = true;

	if (cam->overlay_on == true)
		stop_preview(cam);
	if (cam->streaming_on == true)
		stop_encoder(cam);

	if (cam->open_count)
		vidioc_int_s_power(cam->ep->sensor, 0);
out:
	mutex_unlock(&cam->busy_lock);
	return ret;
}

/*!
 * This function is called to bring the sensor back from a low power state.
 * Refer to the document driver-model/driver.txt in the kernel source tree
 * for more information.
 *
 * @param   pdev   the device structure
 *
 * @return  The function returns 0 on success and -ENODEV on failure
 */
static int mxc_v4l2_resume(struct platform_device *pdev)
{
	struct mx6_camera_dev *cam = platform_get_drvdata(pdev);
	int ret = 0;

	dev_dbg(cam->dev, "In MVC:mxc_v4l2_resume\n");

	mutex_lock(&cam->busy_lock);

	if (cam == NULL || cam->ep == NULL || cam->ep->sensor == NULL) {
		ret = -ENODEV;
		goto out;
	}

	cam->low_power = false;
	wake_up_interruptible(&cam->power_queue);

	if (cam->open_count)
		vidioc_int_s_power(cam->ep->sensor, 1);

	if (cam->overlay_on == true)
		start_preview(cam);
	if (cam->streaming_on && !list_empty(&cam->ready_q) &&
	    !list_is_singular(&cam->ready_q))
		start_encoder(cam);
out:
	mutex_unlock(&cam->busy_lock);
	return ret;
}

static void recalc_input_indexes(struct mx6_camera_dev *cam)
{
	struct mxc_v4l2_endpoint *ep, *prev_ep = NULL;
	int i, first = 0, last;

	for (i = 0; i < cam->num_eps; i++) {
		ep = &cam->all_eps[i];
		if (!ep->sensor)
			continue;

		first += prev_ep ? prev_ep->sensor_input.num : 0;
		last = first + ep->sensor_input.num - 1;

		ep->sensor_input.first = first;
		ep->sensor_input.last = last;
		prev_ep = ep;
	}
}

/*!
 * Initializes the camera driver.
 */
static int mxc_v4l2_master_attach(struct v4l2_int_device *slave)
{
	struct mx6_camera_dev *cam = slave->u.slave->master->priv;
	struct mxc_v4l2_endpoint *ep;
	struct device *slave_dev;
	struct v4l2_input input;
	int err, i, ret = 0;

	mutex_lock(&cam->busy_lock);
	mutex_lock(&cam->irq_lock);

	dev_info(cam->dev, "%s: enter\n", __func__);
	dev_info(cam->dev, "   slave.name = %s\n",
			 slave->name);
	dev_info(cam->dev, "   master.name = %s\n",
			 slave->u.slave->master->name);

	if (slave == NULL) {
		dev_err(cam->dev,
			"ERROR: v4l2 capture: slave parameter not valid.\n");
		ret = -EINVAL;
		goto out;
	}

	vidioc_int_g_dev(slave, &slave_dev);

	if (of_device_is_compatible(slave_dev->of_node,
				    "fsl,imx6-mipi-csi2")) {
		dev_info(cam->dev, "Attaching MIPI CSI-2 Receiver\n");
		cam->csi2 = slave;
		goto out;
	}

	/* find the endpoint attached to this remote slave */
	ep = find_ep_by_remote_node(cam, slave_dev->of_node);
	if (!ep) {
		dev_err(cam->dev, "ERROR: endpoint for this slave not found\n");
		ret = -EINVAL;
		goto out;
	}

	/* select this endpoint as current active endpoint */
	cam->ep = ep;
	cam->ep->sensor = slave;

	/* how many inputs does this sensor have? */
	for (i = 0;; i++) {
		input.index = i;
		err = vidioc_int_enum_input(cam->ep->sensor, &input);
		if (err)
			break;
	}
	if (!i) {
		dev_err(cam->dev,
			"ERROR: v4l2 capture: slave has no inputs!\n");
		ret = -ENODEV;
		goto out;
	}

	dev_info(cam->dev, "   slave num inputs = %d\n", i);
	dev_info(cam->dev, "   %s endpoint, port = %d, id = %d\n",
		 ep->ep.bus_type == V4L2_MBUS_CSI2 ? "CSI-2" : "Parallel",
		 ep->ep.port, ep->ep.id);

	cam->ep->sensor_input.num = i;

	recalc_input_indexes(cam);

	for (i = 0; i < cam->num_eps; i++) {
		if (!cam->all_eps[i].sensor)
			continue;
		vidioc_int_dev_exit(cam->all_eps[i].sensor);
		vidioc_int_s_power(cam->all_eps[i].sensor, 0);
	}

	mxc_update_sensor_fmt(cam);
	vidioc_int_querystd(cam->ep->sensor, &cam->current_std);
	mxc_calc_default_crop(cam, &cam->crop_current,
			      &cam->sensor_fmt);

out:
	dev_info(cam->dev, "%s: exit\n", __func__);
	mutex_unlock(&cam->irq_lock);
	mutex_unlock(&cam->busy_lock);
	return ret;
}

/*!
 * Disconnects the camera driver.
 */
static void mxc_v4l2_master_detach(struct v4l2_int_device *slave)
{
	struct mx6_camera_dev *cam = slave->u.slave->master->priv;
	struct mxc_v4l2_endpoint *ep;
	struct device *slave_dev;

	mutex_lock(&cam->busy_lock);
	mutex_lock(&cam->irq_lock);

	dev_dbg(cam->dev, "In MVC:mxc_v4l2_master_detach\n");

	vidioc_int_g_dev(slave, &slave_dev);

	if (of_device_is_compatible(slave_dev->of_node,
				    "fsl,imx6-mipi-csi2")) {
		dev_info(cam->dev, "Detaching MIPI CSI-2 Receiver\n");
		cam->csi2 = NULL;
		goto out;
	}

	/* find the endpoint attached to this remote slave */
	ep = find_ep_by_remote_node(cam, slave_dev->of_node);
	if (!ep) {
		dev_err(cam->dev, "ERROR: %s: slave not found!\n", __func__);
		goto out;
	}

	if (ep->ep.bus_type == V4L2_MBUS_CSI2 && cam->csi2) {
		/* power-down the mipi csi-2 receiver */
		vidioc_int_s_power(cam->csi2, 0);
	}

	ep->sensor = NULL;
	recalc_input_indexes(cam);

	/* Point current endpoint to new input 0 */
	cam->ep = find_ep_by_input_index(cam, 0);
	cam->current_input = 0;

	vidioc_int_dev_exit(slave);
out:
	mutex_unlock(&cam->irq_lock);
	mutex_unlock(&cam->busy_lock);
}

static struct of_device_id mxc_v4l2_dt_ids[] = {
	{ .compatible = "fsl,imx6-v4l2-capture" },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, mxc_v4l2_dt_ids);

static struct platform_driver mxc_v4l2_driver = {
	.driver = {
		.name  = MX6_CAM_DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table	= mxc_v4l2_dt_ids,
	},
	.probe    = mxc_v4l2_probe,
	.remove   = mxc_v4l2_remove,
	.suspend  = mxc_v4l2_suspend,
	.resume   = mxc_v4l2_resume,
	.shutdown = NULL,
};
module_platform_driver(mxc_v4l2_driver);

MODULE_AUTHOR("Freescale Semiconductor, Inc.");
MODULE_DESCRIPTION("V4L2 Camera Interface driver for i.MX6");
MODULE_LICENSE("GPL");
MODULE_SUPPORTED_DEVICE("video");
