/*
 * Copyright 2005-2012 Freescale Semiconductor, Inc. All Rights Reserved.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

/*!
 * @file adv7180.c
 *
 * @brief Analog Device ADV7180 video decoder functions
 *
 * @ingroup Camera
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/ctype.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/semaphore.h>
#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/wait.h>
#include <linux/videodev2.h>
#include <linux/workqueue.h>
#include <linux/regulator/consumer.h>
#include <linux/fsl_devices.h>
#include <media/v4l2-chip-ident.h>
#include <media/v4l2-int-device.h>
#include "mxc_v4l2_capture.h"

/*!
 * Maintains the information on the current state of the sensor.
 */
struct adv7180_dev {
	struct i2c_client *i2c_client;
	struct v4l2_of_bus_parallel bus_cfg;
	struct v4l2_pix_format pix;
	struct v4l2_captureparm streamcap;
	int rev_id;
	bool on;

	bool locked;             /* locked to signal */
	bool lock_status_change; /* there was a lock status change */

	/* control settings */
	int brightness;
	int hue;
	int contrast;
	int saturation;
	int red;
	int green;
	int blue;
	int ae_mode;

	struct regulator *dvddio;
	struct regulator *dvdd;
	struct regulator *avdd;
	struct regulator *pvdd;
	int pwdn_gpio;

	v4l2_std_id std_id;

	/* Standard index of ADV7180. */
	int video_idx;

	/* Current analog input mux */
	int current_input;

	struct mutex mutex;
};

/*! List of input video formats supported. The video formats is corresponding
 * with v4l2 id in video_fmt_t
 */
enum {
	ADV7180_NTSC = 0,	/*!< Locked on (M) NTSC video signal. */
	ADV7180_PAL,		/*!< (B, G, H, I, N)PAL video signal. */
};

/*! Number of video standards supported (including 'not locked' signal). */
#define ADV7180_STD_MAX		(ADV7180_PAL + 1)

/*! Video format structure. */
struct video_fmt_t {
	int v4l2_id;		/*!< Video for linux ID. */
	char name[16];		/*!< Name (e.g., "NTSC", "PAL", etc.) */
	u16 raw_width;		/*!< Raw width. */
	u16 raw_height;		/*!< Raw height. */
	u16 active_width;	/*!< Active width. */
	u16 active_height;	/*!< Active height. */
};

/*! Description of video formats supported.
 *
 *  PAL: raw=720x625, active=720x576.
 *  NTSC: raw=720x525, active=720x480.
 */
static struct video_fmt_t video_fmts[] = {
	{			/*! NTSC */
	 .v4l2_id = V4L2_STD_NTSC,
	 .name = "NTSC",
	 .raw_width = 720,	/* SENS_FRM_WIDTH */
	 .raw_height = 525,	/* SENS_FRM_HEIGHT */
	 .active_width = 720,	/* ACT_FRM_WIDTH plus 1 */
	 .active_height = 480,	/* ACT_FRM_WIDTH plus 1 */
	 },
	{			/*! (B, G, H, I, N) PAL */
	 .v4l2_id = V4L2_STD_PAL,
	 .name = "PAL",
	 .raw_width = 720,
	 .raw_height = 625,
	 .active_width = 720,
	 .active_height = 576,
	 },
};

#define IF_NAME                    "adv7180"
#define ADV7180_INPUT_CTL              0x00	/* Input Control */
#define ADV7180_STATUS_1               0x10	/* Status #1 */
#define   ADV7180_IN_LOCK              (1 << 0)
#define   ADV7180_LOST_LOCK            (1 << 1)
#define   ADV7180_FSC_LOCK             (1 << 2)
#define   ADV7180_AD_RESULT_BIT        4
#define   ADV7180_AD_RESULT_MASK       (0x7 << ADV7180_AD_RESULT_BIT)
#define   ADV7180_AD_NTSC              0
#define   ADV7180_AD_NTSC_4_43         1
#define   ADV7180_AD_PAL_M             2
#define   ADV7180_AD_PAL_60            3
#define   ADV7180_AD_PAL               4
#define   ADV7180_AD_SECAM             5
#define   ADV7180_AD_PAL_N             6
#define   ADV7180_AD_SECAM_525         7
#define ADV7180_CONTRAST               0x08	/* Contrast */
#define ADV7180_BRIGHTNESS             0x0a	/* Brightness */
#define ADV7180_HUE_REG                0x0b	/* Signed, inverted */
#define ADV7180_IDENT                  0x11	/* IDENT */
#define ADV7180_VSYNC_FIELD_CTL_1      0x31	/* VSYNC Field Control #1 */
#define ADV7180_MANUAL_WIN_CTL         0x3d	/* Manual Window Control */
#define ADV7180_SD_SATURATION_CB       0xe3	/* SD Saturation Cb */
#define ADV7180_SD_SATURATION_CR       0xe4	/* SD Saturation Cr */
#define ADV7180_PWR_MNG                0x0f     /* Power Management */
#define ADV7180_INT_CONFIG_1           0x40     /* Interrupt Config 1 */
#define ADV7180_INT_STATUS_1           0x42     /* Interrupt Status 1 (r/o) */
#define   ADV7180_INT_SD_LOCK          (1 << 0)
#define   ADV7180_INT_SD_UNLOCK        (1 << 1)
#define ADV7180_INT_CLEAR_1            0x43     /* Interrupt Clear 1 (w/o) */
#define ADV7180_INT_MASK_1             0x44     /* Interrupt Mask 1 */
#define ADV7180_INT_STATUS_2           0x46     /* Interrupt Status 2 (r/o) */
#define ADV7180_INT_CLEAR_2            0x47     /* Interrupt Clear 2 (w/o) */
#define ADV7180_INT_MASK_2             0x48     /* Interrupt Mask 2 */
#define ADV7180_INT_RAW_STATUS_3       0x49   /* Interrupt Raw Status 3 (r/o) */
#define   ADV7180_INT_SD_V_LOCK        (1 << 1)
#define ADV7180_INT_STATUS_3           0x4a   /* Interrupt Status 3 (r/o) */
#define   ADV7180_INT_SD_V_LOCK_CHNG   (1 << 1)
#define   ADV7180_INT_SD_AD_CHNG       (1 << 3)
#define ADV7180_INT_CLEAR_3            0x4b     /* Interrupt Clear 3 (w/o) */
#define ADV7180_INT_MASK_3             0x4c     /* Interrupt Mask 3 */

/* supported controls */
/* This hasn't been fully implemented yet.
 * This is how it should work, though. */
static struct v4l2_queryctrl adv7180_qctrl[] = {
	{
	.id = V4L2_CID_BRIGHTNESS,
	.type = V4L2_CTRL_TYPE_INTEGER,
	.name = "Brightness",
	.minimum = 0,		/* check this value */
	.maximum = 255,		/* check this value */
	.step = 1,		/* check this value */
	.default_value = 0,	/* check this value */
	.flags = 0,
	}, {
	.id = V4L2_CID_SATURATION,
	.type = V4L2_CTRL_TYPE_INTEGER,
	.name = "Saturation",
	.minimum = 0,		/* check this value */
	.maximum = 255,		/* check this value */
	.step = 0x1,		/* check this value */
	.default_value = 128,	/* check this value */
	.flags = 0,
	}, {
	.id = V4L2_CID_CONTRAST,
	.type = V4L2_CTRL_TYPE_INTEGER,
	.name = "Contrast",
	.minimum = 0,
	.maximum = 255,
	.step = 0x1,
	.default_value = 128,
	.flags = 0,
	}, {
	.id = V4L2_CID_HUE,
	.type = V4L2_CTRL_TYPE_INTEGER,
	.name = "Hue",
	.minimum = -127,
	.maximum = 128,
	.step = 0x1,
	.default_value = 0,
	.flags = 0,
	}
};

struct adv7180_inputs_t {
	const char *desc;   /* Analog input description */
	u8 insel;           /* insel bits to select this input */
};

/* Analog Inputs on 64-Lead and 48-Lead LQFP */
static const struct adv7180_inputs_t adv7180_inputs_64_48[] = {
	{ .insel = 0x00, .desc = "ADV7180 Composite on Ain1" },
	{ .insel = 0x01, .desc = "ADV7180 Composite on Ain2" },
	{ .insel = 0x02, .desc = "ADV7180 Composite on Ain3" },
	{ .insel = 0x03, .desc = "ADV7180 Composite on Ain4" },
	{ .insel = 0x04, .desc = "ADV7180 Composite on Ain5" },
	{ .insel = 0x05, .desc = "ADV7180 Composite on Ain6" },
	{ .insel = 0x06, .desc = "ADV7180 Y/C on Ain1/4" },
	{ .insel = 0x07, .desc = "ADV7180 Y/C on Ain2/5" },
	{ .insel = 0x08, .desc = "ADV7180 Y/C on Ain3/6" },
	{ .insel = 0x09, .desc = "ADV7180 YPbPr on Ain1/4/5" },
	{ .insel = 0x0a, .desc = "ADV7180 YPbPr on Ain2/3/6" },
};
#define NUM_INPUTS_64_48 ARRAY_SIZE(adv7180_inputs_64_48)

#if 0
/*
 * FIXME: there is no way to distinguish LQFP vs LFCSP chips, so
 * we will just have to assume LQFP.
 */
/* Analog Inputs on 40-Lead and 32-Lead LFCSP */
static const struct adv7180_inputs_t adv7180_inputs_40_32[] = {
	{ .insel = 0x00, .desc = "ADV7180 Composite on Ain1" },
	{ .insel = 0x03, .desc = "ADV7180 Composite on Ain2" },
	{ .insel = 0x04, .desc = "ADV7180 Composite on Ain3" },
	{ .insel = 0x06, .desc = "ADV7180 Y/C on Ain1/2" },
	{ .insel = 0x09, .desc = "ADV7180 YPbPr on Ain1/2/3" },
};
#define NUM_INPUTS_40_32 ARRAY_SIZE(adv7180_inputs_40_32)
#endif

#define ADV7180_VOLTAGE_ANALOG               1800000
#define ADV7180_VOLTAGE_DIGITAL_CORE         1800000
#define ADV7180_VOLTAGE_DIGITAL_IO           3300000
#define ADV7180_VOLTAGE_PLL                  1800000

static int adv7180_regulator_enable(struct adv7180_dev *sensor)
{
	struct device *dev = &sensor->i2c_client->dev;
	int ret = 0;

	sensor->dvddio = devm_regulator_get(dev, "DOVDD");
	if (!IS_ERR(sensor->dvddio)) {
		regulator_set_voltage(sensor->dvddio,
				      ADV7180_VOLTAGE_DIGITAL_IO,
				      ADV7180_VOLTAGE_DIGITAL_IO);
		ret = regulator_enable(sensor->dvddio);
		if (ret) {
			dev_err(dev, "set io voltage failed\n");
			return ret;
		} else {
			dev_dbg(dev, "set io voltage ok\n");
		}
	} else {
		dev_warn(dev, "cannot get io voltage\n");
	}

	sensor->dvdd = devm_regulator_get(dev, "DVDD");
	if (!IS_ERR(sensor->dvdd)) {
		regulator_set_voltage(sensor->dvdd,
				      ADV7180_VOLTAGE_DIGITAL_CORE,
				      ADV7180_VOLTAGE_DIGITAL_CORE);
		ret = regulator_enable(sensor->dvdd);
		if (ret) {
			dev_err(dev, "set core voltage failed\n");
			return ret;
		} else {
			dev_dbg(dev, "set core voltage ok\n");
		}
	} else {
		dev_warn(dev, "cannot get core voltage\n");
	}

	sensor->avdd = devm_regulator_get(dev, "AVDD");
	if (!IS_ERR(sensor->avdd)) {
		regulator_set_voltage(sensor->avdd,
				      ADV7180_VOLTAGE_ANALOG,
				      ADV7180_VOLTAGE_ANALOG);
		ret = regulator_enable(sensor->avdd);
		if (ret) {
			dev_err(dev, "set analog voltage failed\n");
			return ret;
		} else {
			dev_dbg(dev, "set analog voltage ok\n");
		}
	} else {
		dev_warn(dev, "cannot get analog voltage\n");
	}

	sensor->pvdd = devm_regulator_get(dev, "PVDD");
	if (!IS_ERR(sensor->pvdd)) {
		regulator_set_voltage(sensor->pvdd,
				      ADV7180_VOLTAGE_PLL,
				      ADV7180_VOLTAGE_PLL);
		ret = regulator_enable(sensor->pvdd);
		if (ret) {
			dev_err(dev, "set pll voltage failed\n");
			return ret;
		} else {
			dev_dbg(dev, "set pll voltage ok\n");
		}
	} else {
		dev_warn(dev, "cannot get pll voltage\n");
	}

	return ret;
}

static void adv7180_regulator_disable(struct adv7180_dev *sensor)
{
	if (sensor->dvddio)
		regulator_disable(sensor->dvddio);

	if (sensor->dvdd)
		regulator_disable(sensor->dvdd);

	if (sensor->avdd)
		regulator_disable(sensor->avdd);

	if (sensor->pvdd)
		regulator_disable(sensor->pvdd);
}

/***********************************************************************
 * I2C transfert.
 ***********************************************************************/

/*! Read one register from a ADV7180 i2c slave device.
 *
 *  @param *reg		register in the device we wish to access.
 *
 *  @return		       0 if success, an error code otherwise.
 */
static int adv7180_read(struct adv7180_dev *sensor, u8 reg)
{
	int ret = i2c_smbus_read_byte_data(sensor->i2c_client, reg);
	if (ret < 0)
		dev_err(&sensor->i2c_client->dev,
			"%s: read reg error: reg=%2x\n", __func__, reg);
	return ret;
}

/*! Write one register of a ADV7180 i2c slave device.
 *
 *  @param *reg		register in the device we wish to access.
 *
 *  @return		       0 if success, an error code otherwise.
 */
static int adv7180_write_reg(struct adv7180_dev *sensor, u8 reg, u8 val)
{
	int ret = i2c_smbus_write_byte_data(sensor->i2c_client, reg, val);
	if (ret < 0)
		dev_err(&sensor->i2c_client->dev,
			"%s: write reg error:reg=%2x,val=%2x\n", __func__,
			reg, val);
	return ret;
}

/* Read AD_RESULT to get the autodetected video standard */
static void adv7180_get_autodetect_std(struct adv7180_dev *sensor)
{
	int stat1, ad_result, idx = ADV7180_PAL;
	v4l2_std_id std = V4L2_STD_PAL;

	/*
	 * When the chip loses lock, it continues to send data at whatever
	 * standard was detected before, so leave the standard at the last
	 * detected standard.
	 */
	if (!sensor->locked) {
		dev_dbg(&sensor->i2c_client->dev, "No signal lock\n");
		return;
	}

	stat1 = adv7180_read(sensor, ADV7180_STATUS_1);
	ad_result = (stat1 & ADV7180_AD_RESULT_MASK) >> ADV7180_AD_RESULT_BIT;

	switch (ad_result) {
	case ADV7180_AD_PAL:
		std = V4L2_STD_PAL;
		idx = ADV7180_PAL;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected PAL standard\n");
		break;
	case ADV7180_AD_PAL_M:
		std = V4L2_STD_PAL_M;
		/* PAL M is very similar to NTSC (same lines/field) */
		idx = ADV7180_NTSC;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected PAL M standard\n");
		break;
	case ADV7180_AD_PAL_N:
		std = V4L2_STD_PAL_N;
		idx = ADV7180_PAL;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected PAL N standard\n");
		break;
	case ADV7180_AD_PAL_60:
		std = V4L2_STD_PAL_60;
		/* PAL 60 has same lines as NTSC */
		idx = ADV7180_NTSC;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected PAL 60 standard\n");
		break;
	case ADV7180_AD_NTSC:
		std = V4L2_STD_NTSC;
		idx = ADV7180_NTSC;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected NTSC standard\n");
		break;
	case ADV7180_AD_NTSC_4_43:
		std = V4L2_STD_NTSC_443;
		idx = ADV7180_NTSC;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected NTSC 4.43 standard\n");
		break;
	case ADV7180_AD_SECAM:
		std = V4L2_STD_SECAM;
		idx = ADV7180_PAL;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected SECAM standard\n");
		break;
	case ADV7180_AD_SECAM_525:
		/*
		 * FIXME: could not find any info on "SECAM 525", assume
		 * it is SECAM but with NTSC line standard.
		 */
		std = V4L2_STD_SECAM;
		idx = ADV7180_NTSC;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected SECAM 525 standard\n");
		break;
	}

	if (std != sensor->std_id) {
		dev_dbg(&sensor->i2c_client->dev,
			"changing std: 0x%08x --> 0x%08x\n",
			(u32)sensor->std_id, (u32)std);
		sensor->video_idx = idx;
		sensor->std_id = std;
		sensor->pix.width = video_fmts[sensor->video_idx].raw_width;
		sensor->pix.height = video_fmts[sensor->video_idx].raw_height;
	}
}

/* Update lock status */
static void adv7180_update_lock_status(struct adv7180_dev *sensor)
{
	int stat1, int_stat1, int_stat3, int_raw_stat3;

	stat1 = adv7180_read(sensor, ADV7180_STATUS_1);

	/* Switch to interrupt register map */
	adv7180_write_reg(sensor, 0x0E, 0x20);

	int_stat1 = adv7180_read(sensor, ADV7180_INT_STATUS_1);
	int_stat3 = adv7180_read(sensor, ADV7180_INT_STATUS_3);
	/* clear the interrupts */
	adv7180_write_reg(sensor, ADV7180_INT_CLEAR_1, int_stat1);
	adv7180_write_reg(sensor, ADV7180_INT_CLEAR_3, int_stat3);

	int_raw_stat3 = adv7180_read(sensor, ADV7180_INT_RAW_STATUS_3);

	/* Switch back to normal register map */
	adv7180_write_reg(sensor, 0x0E, 0x00);

	dev_dbg(&sensor->i2c_client->dev,
		"stat1=0x%02x, int_stat 1/3/raw3 = 0x%02x/0x%02x/0x%02x\n",
		stat1, int_stat1, int_stat3, int_raw_stat3);

	if ((int_stat1 & ADV7180_INT_SD_LOCK) ||
	    (int_stat1 & ADV7180_INT_SD_UNLOCK) ||
	    (int_stat3 & ADV7180_INT_SD_V_LOCK_CHNG))
		sensor->lock_status_change = true;

	sensor->locked = ((stat1 & ADV7180_IN_LOCK) &&
			  (stat1 & ADV7180_FSC_LOCK) &&
			  (int_raw_stat3 & ADV7180_INT_SD_V_LOCK));

	dev_dbg(&sensor->i2c_client->dev, "lsc %s: %s\n",
		sensor->lock_status_change ? "true" : "false",
		sensor->locked ? "locked" : "unlocked");
}

static void adv7180_power(struct adv7180_dev *sensor, bool enable)
{
	if (enable && !sensor->on) {
		if (gpio_is_valid(sensor->pwdn_gpio))
			gpio_set_value_cansleep(sensor->pwdn_gpio, 1);

		usleep_range(5000, 5001);
		adv7180_write_reg(sensor, ADV7180_PWR_MNG, 0);
	} else if (!enable && sensor->on) {
		adv7180_write_reg(sensor, ADV7180_PWR_MNG, 0x24);

		if (gpio_is_valid(sensor->pwdn_gpio))
			gpio_set_value_cansleep(sensor->pwdn_gpio, 0);
	}

	sensor->on = enable;
}

/*
 * Enable the SD_UNLOCK and SD_AD_CHNG interrupts.
 */
static void adv7180_enable_interrupts(struct adv7180_dev *sensor)
{
	mutex_lock(&sensor->mutex);

	/* Switch to interrupt register map */
	adv7180_write_reg(sensor, 0x0E, 0x20);
	/* INTRQ active low, active until cleared */
	adv7180_write_reg(sensor, ADV7180_INT_CONFIG_1, 0xd1);
	/* unmask SD_UNLOCK and SD_LOCK */
	adv7180_write_reg(sensor, ADV7180_INT_MASK_1,
			  ADV7180_INT_SD_UNLOCK | ADV7180_INT_SD_LOCK);
	/* unmask SD_AD_CHNG and SD_V_LOCK_CHNG */
	adv7180_write_reg(sensor, ADV7180_INT_MASK_3,
			  ADV7180_INT_SD_AD_CHNG | ADV7180_INT_SD_V_LOCK_CHNG);
	/* Switch back to normal register map */
	adv7180_write_reg(sensor, 0x0E, 0x00);

	mutex_unlock(&sensor->mutex);
}

/* threaded irq handler */
static irqreturn_t adv7180_interrupt(int irq, void *dev_id)
{
	struct adv7180_dev *sensor = dev_id;

	mutex_lock(&sensor->mutex);

	adv7180_update_lock_status(sensor);
	adv7180_get_autodetect_std(sensor);

	mutex_unlock(&sensor->mutex);

	return IRQ_HANDLED;
}

/***********************************************************************
 * mxc_v4l2_capture interface.
 ***********************************************************************/

/***********************************************************************
 * IOCTL Functions from v4l2_int_ioctl_desc.
 ***********************************************************************/

/*!
 * ioctl_querystd - V4L2 sensor interface handler for vidioc_int_querystd_num.
 * Return current video standard. This device autodetects the current
 * standard, so this function also sets the values that need to be changed
 * if the standard changes. There is no set std equivalent function.
 */
static int ioctl_querystd(struct v4l2_int_device *s, v4l2_std_id *std)
{
	struct adv7180_dev *sensor = s->priv;

	mutex_lock(&sensor->mutex);

	/*
	 * If we have the ADV7180 irq, we can just return the currently
	 * detected standard. Otherwise we have to poll the AD_RESULT
	 * bits every time ioctl_querystd() is called.
	 */
	if (!sensor->i2c_client->irq) {
		adv7180_update_lock_status(sensor);
		adv7180_get_autodetect_std(sensor);
	}

	*std = sensor->std_id;

	mutex_unlock(&sensor->mutex);

	return 0;
}

/*!
 * ioctl_g_ifparm - V4L2 sensor interface handler for vidioc_int_g_ifparm_num
 * s: pointer to standard V4L2 device structure
 * p: pointer to standard V4L2 vidioc_int_g_ifparm_num ioctl structure
 *
 * Gets slave interface parameters.
 * Calculates the required xclk value to support the requested
 * clock parameters in p.  This value is returned in the p
 * parameter.
 *
 * vidioc_int_g_ifparm returns platform-specific information about the
 * interface settings used by the sensor.
 *
 * Called on open.
 */
static int ioctl_g_ifparm(struct v4l2_int_device *s, struct v4l2_ifparm *p)
{
	struct adv7180_dev *sensor = s->priv;

	dev_dbg(&sensor->i2c_client->dev, "%s\n", __func__);

	if (s == NULL) {
		dev_err(&sensor->i2c_client->dev,
			"ERROR!! no slave device set!\n");
		return -ENODEV;
	}

	/* Initialize structure to 0s then set any non-0 values. */
	memset(p, 0, sizeof(*p));
	p->if_type = V4L2_IF_TYPE_BT656; /* This is the only possibility. */
	p->u.bt656.mode = V4L2_IF_TYPE_BT656_MODE_BT_8BIT;
	p->u.bt656.bt_sync_correct = 1; /* use bt656 sync codes */
	p->u.bt656.clock_curr = 0;

	/* ADV7180 has a dedicated clock so no clock settings needed. */

	return 0;
}

/*!
 * Sets the camera power.
 *
 * s  pointer to the camera device
 * on if 1, power is to be turned on.  0 means power is to be turned off
 *
 * ioctl_s_power - V4L2 sensor interface handler for vidioc_int_s_power_num
 * @s: pointer to standard V4L2 device structure
 * @on: power state to which device is to be set
 *
 * Sets devices power state to requrested state, if possible.
 * This is called on open, close, suspend and resume.
 */
static int ioctl_s_power(struct v4l2_int_device *s, int on)
{
	struct adv7180_dev *sensor = s->priv;

	dev_dbg(&sensor->i2c_client->dev, "%s\n", __func__);

	return 0;
}

/*!
 * ioctl_g_parm - V4L2 sensor interface handler for VIDIOC_G_PARM ioctl
 * @s: pointer to standard V4L2 device structure
 * @a: pointer to standard V4L2 VIDIOC_G_PARM ioctl structure
 *
 * Returns the sensor's video CAPTURE parameters.
 */
static int ioctl_g_parm(struct v4l2_int_device *s, struct v4l2_streamparm *a)
{
	struct adv7180_dev *sensor = s->priv;
	struct v4l2_captureparm *cparm = &a->parm.capture;

	dev_dbg(&sensor->i2c_client->dev, "%s\n", __func__);

	switch (a->type) {
	/* These are all the possible cases. */
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		dev_dbg(&sensor->i2c_client->dev,
			"   type is V4L2_BUF_TYPE_VIDEO_CAPTURE\n");
		memset(a, 0, sizeof(*a));
		a->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		cparm->capability = sensor->streamcap.capability;
		cparm->timeperframe = sensor->streamcap.timeperframe;
		cparm->capturemode = sensor->streamcap.capturemode;
		break;

	case V4L2_BUF_TYPE_VIDEO_OUTPUT:
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	case V4L2_BUF_TYPE_VBI_CAPTURE:
	case V4L2_BUF_TYPE_VBI_OUTPUT:
	case V4L2_BUF_TYPE_SLICED_VBI_CAPTURE:
	case V4L2_BUF_TYPE_SLICED_VBI_OUTPUT:
		break;

	default:
		dev_dbg(&sensor->i2c_client->dev,
			"%s: type is unknown %d\n", __func__, a->type);
		break;
	}

	return 0;
}

/*!
 * ioctl_s_parm - V4L2 sensor interface handler for VIDIOC_S_PARM ioctl
 * @s: pointer to standard V4L2 device structure
 * @a: pointer to standard V4L2 VIDIOC_S_PARM ioctl structure
 *
 * Configures the sensor to use the input parameters, if possible.  If
 * not possible, reverts to the old parameters and returns the
 * appropriate error code.
 *
 * This driver cannot change these settings.
 */
static int ioctl_s_parm(struct v4l2_int_device *s, struct v4l2_streamparm *a)
{
	struct adv7180_dev *sensor = s->priv;

	dev_dbg(&sensor->i2c_client->dev, "%s\n", __func__);

	switch (a->type) {
	/* These are all the possible cases. */
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
	case V4L2_BUF_TYPE_VIDEO_OUTPUT:
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	case V4L2_BUF_TYPE_VBI_CAPTURE:
	case V4L2_BUF_TYPE_VBI_OUTPUT:
	case V4L2_BUF_TYPE_SLICED_VBI_CAPTURE:
	case V4L2_BUF_TYPE_SLICED_VBI_OUTPUT:
		break;

	default:
		dev_dbg(&sensor->i2c_client->dev,
			"   type is unknown - %d\n", a->type);
		break;
	}

	return 0;
}

/*!
 * ioctl_g_fmt_cap - V4L2 sensor interface handler for ioctl_g_fmt_cap
 * @s: pointer to standard V4L2 device structure
 * @f: pointer to standard V4L2 v4l2_format structure
 *
 * Returns the sensor's current pixel format in the v4l2_format
 * parameter.
 */
static int ioctl_g_fmt_cap(struct v4l2_int_device *s, struct v4l2_format *f)
{
	struct adv7180_dev *sensor = s->priv;

	dev_dbg(&sensor->i2c_client->dev, "%s\n", __func__);

	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		dev_dbg(&sensor->i2c_client->dev,
			"   Returning size of %dx%d\n",
			 sensor->pix.width, sensor->pix.height);
		f->fmt.pix = sensor->pix;
		break;
	default:
		f->fmt.pix = sensor->pix;
		break;
	}

	return 0;
}

/*!
 * ioctl_try_fmt_cap - V4L2 sensor interface handler for VIDIOC_TRY_FMT
 * @s: pointer to standard V4L2 device structure
 * @fmt: pointer to standard V4L2 fmt description structure
 *
 * This driver autodetects a standard video mode, so we don't allow
 * setting a mode, just return the current autodetected mode.
 *
 * Return 0.
 */
static int ioctl_try_fmt_cap(struct v4l2_int_device *s,
			      struct v4l2_format *f)
{
	struct adv7180_dev *sensor = s->priv;

	f->fmt.pix = sensor->pix;
	return 0;
}

/*!
 * ioctl_s_fmt_cap - V4L2 sensor interface handler for VIDIOC_S_FMT
 * @s: pointer to standard V4L2 device structure
 * @fmt: pointer to standard V4L2 fmt description structure
 *
 * This driver autodetects a standard video mode, so we don't allow
 * setting a mode, just return the current autodetected mode.
 *
 * Return 0.
 */
static int ioctl_s_fmt_cap(struct v4l2_int_device *s, struct v4l2_format *f)
{
	struct adv7180_dev *sensor = s->priv;

	f->fmt.pix = sensor->pix;
	return 0;
}

/*!
 * ioctl_queryctrl - V4L2 sensor interface handler for VIDIOC_QUERYCTRL ioctl
 * @s: pointer to standard V4L2 device structure
 * @qc: standard V4L2 VIDIOC_QUERYCTRL ioctl structure
 *
 * If the requested control is supported, returns the control information
 * from the video_control[] array.  Otherwise, returns -EINVAL if the
 * control is not supported.
 */
static int ioctl_queryctrl(struct v4l2_int_device *s,
			   struct v4l2_queryctrl *qc)
{
	struct adv7180_dev *sensor = s->priv;
	int i;

	dev_dbg(&sensor->i2c_client->dev, "%s\n", __func__);

	for (i = 0; i < ARRAY_SIZE(adv7180_qctrl); i++)
		if (qc->id && qc->id == adv7180_qctrl[i].id) {
			memcpy(qc, &(adv7180_qctrl[i]),
				sizeof(*qc));
			return 0;
		}

	return -EINVAL;
}

/*!
 * ioctl_g_ctrl - V4L2 sensor interface handler for VIDIOC_G_CTRL ioctl
 * @s: pointer to standard V4L2 device structure
 * @vc: standard V4L2 VIDIOC_G_CTRL ioctl structure
 *
 * If the requested control is supported, returns the control's current
 * value from the video_control[] array.  Otherwise, returns -EINVAL
 * if the control is not supported.
 */
static int ioctl_g_ctrl(struct v4l2_int_device *s, struct v4l2_control *vc)
{
	struct adv7180_dev *sensor = s->priv;
	int ret = 0;

	dev_dbg(&sensor->i2c_client->dev, "%s\n", __func__);

	switch (vc->id) {
	case V4L2_CID_BRIGHTNESS:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_BRIGHTNESS\n");
		sensor->brightness = adv7180_read(sensor, ADV7180_BRIGHTNESS);
		vc->value = sensor->brightness;
		break;
	case V4L2_CID_CONTRAST:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_CONTRAST\n");
		sensor->contrast = adv7180_read(sensor, ADV7180_CONTRAST);
		vc->value = sensor->contrast;
		break;
	case V4L2_CID_SATURATION:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_SATURATION\n");
		sensor->saturation =
			adv7180_read(sensor, ADV7180_SD_SATURATION_CB);
		vc->value = sensor->saturation;
		break;
	case V4L2_CID_HUE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_HUE\n");
		vc->value = sensor->hue;
		break;
	case V4L2_CID_AUTO_WHITE_BALANCE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_AUTO_WHITE_BALANCE\n");
		break;
	case V4L2_CID_DO_WHITE_BALANCE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_DO_WHITE_BALANCE\n");
		break;
	case V4L2_CID_RED_BALANCE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_RED_BALANCE\n");
		vc->value = sensor->red;
		break;
	case V4L2_CID_BLUE_BALANCE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_BLUE_BALANCE\n");
		vc->value = sensor->blue;
		break;
	case V4L2_CID_GAMMA:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_GAMMA\n");
		break;
	case V4L2_CID_EXPOSURE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_EXPOSURE\n");
		vc->value = sensor->ae_mode;
		break;
	case V4L2_CID_AUTOGAIN:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_AUTOGAIN\n");
		break;
	case V4L2_CID_GAIN:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_GAIN\n");
		break;
	case V4L2_CID_HFLIP:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_HFLIP\n");
		break;
	case V4L2_CID_VFLIP:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_VFLIP\n");
		break;
	default:
		dev_dbg(&sensor->i2c_client->dev,
			"   Default case\n");
		vc->value = 0;
		ret = -EPERM;
		break;
	}

	return ret;
}

/*!
 * ioctl_s_ctrl - V4L2 sensor interface handler for VIDIOC_S_CTRL ioctl
 * @s: pointer to standard V4L2 device structure
 * @vc: standard V4L2 VIDIOC_S_CTRL ioctl structure
 *
 * If the requested control is supported, sets the control's current
 * value in HW (and updates the video_control[] array).  Otherwise,
 * returns -EINVAL if the control is not supported.
 */
static int ioctl_s_ctrl(struct v4l2_int_device *s, struct v4l2_control *vc)
{
	struct adv7180_dev *sensor = s->priv;
	int retval = 0;
	u8 tmp;

	dev_dbg(&sensor->i2c_client->dev, "%s\n", __func__);

	switch (vc->id) {
	case V4L2_CID_BRIGHTNESS:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_BRIGHTNESS\n");
		tmp = vc->value;
		adv7180_write_reg(sensor, ADV7180_BRIGHTNESS, tmp);
		sensor->brightness = vc->value;
		break;
	case V4L2_CID_CONTRAST:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_CONTRAST\n");
		tmp = vc->value;
		adv7180_write_reg(sensor, ADV7180_CONTRAST, tmp);
		sensor->contrast = vc->value;
		break;
	case V4L2_CID_SATURATION:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_SATURATION\n");
		tmp = vc->value;
		adv7180_write_reg(sensor, ADV7180_SD_SATURATION_CB, tmp);
		adv7180_write_reg(sensor, ADV7180_SD_SATURATION_CR, tmp);
		sensor->saturation = vc->value;
		break;
	case V4L2_CID_HUE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_HUE\n");
		tmp = vc->value;
		/* Hue is inverted according to HSL chart */
		adv7180_write_reg(sensor, ADV7180_HUE_REG, -tmp);
		sensor->hue = vc->value;
		break;
	case V4L2_CID_AUTO_WHITE_BALANCE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_AUTO_WHITE_BALANCE\n");
		break;
	case V4L2_CID_DO_WHITE_BALANCE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_DO_WHITE_BALANCE\n");
		break;
	case V4L2_CID_RED_BALANCE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_RED_BALANCE\n");
		break;
	case V4L2_CID_BLUE_BALANCE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_BLUE_BALANCE\n");
		break;
	case V4L2_CID_GAMMA:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_GAMMA\n");
		break;
	case V4L2_CID_EXPOSURE:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_EXPOSURE\n");
		break;
	case V4L2_CID_AUTOGAIN:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_AUTOGAIN\n");
		break;
	case V4L2_CID_GAIN:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_GAIN\n");
		break;
	case V4L2_CID_HFLIP:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_HFLIP\n");
		break;
	case V4L2_CID_VFLIP:
		dev_dbg(&sensor->i2c_client->dev,
			"   V4L2_CID_VFLIP\n");
		break;
	default:
		dev_dbg(&sensor->i2c_client->dev,
			"   Default case\n");
		retval = -EPERM;
		break;
	}

	return retval;
}

/*!
 * ioctl_enum_framesizes - V4L2 sensor interface handler for
 *			   VIDIOC_ENUM_FRAMESIZES ioctl
 * @s: pointer to standard V4L2 device structure
 * @fsize: standard V4L2 VIDIOC_ENUM_FRAMESIZES ioctl structure
 *
 * Return 0 if successful, otherwise -EINVAL.
 */
static int ioctl_enum_framesizes(struct v4l2_int_device *s,
				 struct v4l2_frmsizeenum *fsize)
{
	struct adv7180_dev *sensor = s->priv;

	if (fsize->index > 0)
		return -EINVAL;

	fsize->pixel_format = sensor->pix.pixelformat;
	fsize->discrete.width = video_fmts[sensor->video_idx].active_width;
	fsize->discrete.height = video_fmts[sensor->video_idx].active_height;
	dev_dbg(&sensor->i2c_client->dev, "Framesize %d %d\n",
		fsize->discrete.width, fsize->discrete.height);
	return 0;
}

/*!
 * ioctl_enum_input - V4L2 sensor interface handler for VIDIOC_ENUMINPUT ioctl
 * @s: pointer to standard V4L2 device structure
 * @input: standard V4L2 VIDIOC_ENUMINPUT ioctl structure
 *
 * Return 0 if successful, otherwise -EINVAL.
 */
static int ioctl_enum_input(struct v4l2_int_device *s,
			    struct v4l2_input *input)
{
	struct adv7180_dev *sensor = s->priv;
	const struct adv7180_inputs_t *advinput;

	if (input->index >= NUM_INPUTS_64_48)
		return -EINVAL;

	advinput = &adv7180_inputs_64_48[input->index];

	input->type = V4L2_INPUT_TYPE_CAMERA;
	input->std = V4L2_STD_NTSC | V4L2_STD_PAL | V4L2_STD_SECAM;
	input->status = 0;

	mutex_lock(&sensor->mutex);

	if (input->index == sensor->current_input) {
		if (sensor->on) {
			if (!sensor->locked)
				input->status = V4L2_IN_ST_NO_SIGNAL |
					V4L2_IN_ST_NO_SYNC;
		} else
			input->status = V4L2_IN_ST_NO_POWER;
	}

	mutex_unlock(&sensor->mutex);

	input->capabilities = 0;
	strcpy(input->name, advinput->desc);

	return 0;
}

static int ioctl_g_signal_lsc(struct v4l2_int_device *s)
{
	struct adv7180_dev *sensor = s->priv;
	int ret = 0;

	mutex_lock(&sensor->mutex);

	if (sensor->lock_status_change) {
		sensor->lock_status_change = false;
		ret = -EIO;
	}

	mutex_unlock(&sensor->mutex);

	return ret;
}

/*!
 * ioctl_g_input - V4L2 sensor interface handler for VIDIOC_G_INPUT ioctl
 * @s: pointer to standard V4L2 device structure
 * @index: pointer to return index number
 *
 * Returns 0.
 */
static int ioctl_g_input(struct v4l2_int_device *s, int *index)
{
	struct adv7180_dev *sensor = s->priv;

	*index = sensor->current_input;
	return 0;
}

/*!
 * ioctl_s_input - V4L2 sensor interface handler for VIDIOC_S_INPUT ioctl
 * @s: pointer to standard V4L2 device structure
 * @index: pointer to index number to set
 *
 * Return 0 if successful, otherwise -EINVAL.
 */
static int ioctl_s_input(struct v4l2_int_device *s, int *index)
{
	struct adv7180_dev *sensor = s->priv;
	const struct adv7180_inputs_t *advinput;

	if (*index >= NUM_INPUTS_64_48)
		return -EINVAL;

	advinput = &adv7180_inputs_64_48[*index];

	mutex_lock(&sensor->mutex);

	adv7180_write_reg(sensor, ADV7180_INPUT_CTL, advinput->insel);

	sensor->current_input = *index;

	mutex_unlock(&sensor->mutex);

	return 0;
}

/*!
 * ioctl_g_chip_ident - V4L2 sensor interface handler for
 *			VIDIOC_DBG_G_CHIP_IDENT ioctl
 * @s: pointer to standard V4L2 device structure
 * @id: pointer to int
 *
 * Return 0.
 */
static int ioctl_g_chip_ident(struct v4l2_int_device *s, int *id)
{
	((struct v4l2_dbg_chip_ident *)id)->match.type =
					V4L2_CHIP_MATCH_I2C_DRIVER;
	strcpy(((struct v4l2_dbg_chip_ident *)id)->match.name,
						"adv7180_decoder");
	((struct v4l2_dbg_chip_ident *)id)->ident = V4L2_IDENT_ADV7180;

	return 0;
}

/*!
 * ioctl_init - V4L2 sensor interface handler for VIDIOC_INT_INIT
 * @s: pointer to standard V4L2 device structure
 */
static int ioctl_init(struct v4l2_int_device *s)
{
	struct adv7180_dev *sensor = s->priv;

	dev_dbg(&sensor->i2c_client->dev, "%s\n", __func__);
	return 0;
}

/*!
 * ioctl_enum_fmt_cap - V4L2 sensor interface handler for VIDIOC_ENUM_FMT
 * @s: pointer to standard V4L2 device structure
 * @fmt: pointer to standard V4L2 fmt description structure
 *
 * Return 0.
 */
static int ioctl_enum_fmt_cap(struct v4l2_int_device *s,
			      struct v4l2_fmtdesc *fmt)
{
	struct adv7180_dev *sensor = s->priv;
	if (fmt->index > 0)
		return -EINVAL;

	fmt->pixelformat = sensor->pix.pixelformat;
	return 0;
}


/*!
 * ioctl_dev_init - V4L2 sensor interface handler for vidioc_int_dev_init_num
 * @s: pointer to standard V4L2 device structure
 *
 * Initialise the device when slave attaches to the master.
 */
static int ioctl_dev_init(struct v4l2_int_device *s)
{
	struct adv7180_dev *sensor = s->priv;

	dev_dbg(&sensor->i2c_client->dev, "%s\n", __func__);

	return 0;
}

static int ioctl_g_dev(struct v4l2_int_device *s, struct device **dev)
{
	struct adv7180_dev *sensor = s->priv;

	*dev = &sensor->i2c_client->dev;
	return 0;
}

/*!
 * This structure defines all the ioctls for this module.
 */
static struct v4l2_int_ioctl_desc adv7180_ioctl_desc[] = {

	{vidioc_int_dev_init_num, (v4l2_int_ioctl_func*)ioctl_dev_init},
	{vidioc_int_g_dev_num, (v4l2_int_ioctl_func *)ioctl_g_dev},
	/*!
	 * Delinitialise the dev. at slave detach.
	 * The complement of ioctl_dev_init.
	 */
/*	{vidioc_int_dev_exit_num, (v4l2_int_ioctl_func *)ioctl_dev_exit}, */

	{vidioc_int_s_power_num, (v4l2_int_ioctl_func*)ioctl_s_power},
	{vidioc_int_g_ifparm_num, (v4l2_int_ioctl_func*)ioctl_g_ifparm},
/*	{vidioc_int_g_needs_reset_num,
				(v4l2_int_ioctl_func *)ioctl_g_needs_reset}, */
/*	{vidioc_int_reset_num, (v4l2_int_ioctl_func *)ioctl_reset}, */
	{vidioc_int_init_num, (v4l2_int_ioctl_func*)ioctl_init},

	/*!
	 * VIDIOC_ENUM_FMT ioctl for the CAPTURE buffer type.
	 */
	{vidioc_int_enum_fmt_cap_num, (v4l2_int_ioctl_func*)ioctl_enum_fmt_cap},

	/*!
	 * VIDIOC_TRY_FMT ioctl for the CAPTURE buffer type.
	 * This ioctl is used to negotiate the image capture size and
	 * pixel format without actually making it take effect.
	 */
	{vidioc_int_try_fmt_cap_num,
				(v4l2_int_ioctl_func *)ioctl_try_fmt_cap},

	{vidioc_int_g_fmt_cap_num, (v4l2_int_ioctl_func*)ioctl_g_fmt_cap},

	/*!
	 * If the requested format is supported, configures the HW to use that
	 * format, returns error code if format not supported or HW can't be
	 * correctly configured.
	 */
	{vidioc_int_s_fmt_cap_num, (v4l2_int_ioctl_func *)ioctl_s_fmt_cap},

	{vidioc_int_g_parm_num, (v4l2_int_ioctl_func*)ioctl_g_parm},
	{vidioc_int_s_parm_num, (v4l2_int_ioctl_func*)ioctl_s_parm},
	{vidioc_int_queryctrl_num, (v4l2_int_ioctl_func*)ioctl_queryctrl},
	{vidioc_int_g_ctrl_num, (v4l2_int_ioctl_func*)ioctl_g_ctrl},
	{vidioc_int_s_ctrl_num, (v4l2_int_ioctl_func*)ioctl_s_ctrl},
	{vidioc_int_enum_framesizes_num, (v4l2_int_ioctl_func*)ioctl_enum_framesizes},
	{vidioc_int_enum_input_num, (v4l2_int_ioctl_func*)ioctl_enum_input},
	{vidioc_int_g_input_num, (v4l2_int_ioctl_func*)ioctl_g_input},
	{vidioc_int_s_input_num, (v4l2_int_ioctl_func*)ioctl_s_input},
	{vidioc_int_g_chip_ident_num, (v4l2_int_ioctl_func*)ioctl_g_chip_ident},
	{vidioc_int_querystd_num, (v4l2_int_ioctl_func *)ioctl_querystd},
	{vidioc_int_g_signal_lsc_num, (v4l2_int_ioctl_func *)ioctl_g_signal_lsc},
};

static struct v4l2_int_slave adv7180_slave = {
	.ioctls = adv7180_ioctl_desc,
	.num_ioctls = ARRAY_SIZE(adv7180_ioctl_desc),
};

static struct v4l2_int_device adv7180_int_device = {
	.module = THIS_MODULE,
	.name = "adv7180",
	.type = v4l2_int_type_slave,
	.u = {
		.slave = &adv7180_slave,
	},
};


/***********************************************************************
 * I2C client and driver.
 ***********************************************************************/

/*! ADV7180 Reset function.
 *
 *  @return		None.
 */
static void adv7180_hard_reset(struct adv7180_dev *sensor)
{
	dev_dbg(&sensor->i2c_client->dev, "%s\n", __func__);

	/* assert reset bit */
	adv7180_write_reg(sensor, ADV7180_PWR_MNG, 0x80);
	usleep_range(5000, 5001);

	/* Set analog mux for Composite Ain1 */
	adv7180_write_reg(sensor, ADV7180_INPUT_CTL, 0x00);

	/* Datasheet recommends */
	adv7180_write_reg(sensor, 0x01, 0xc8);
	adv7180_write_reg(sensor, 0x02, 0x04);
	adv7180_write_reg(sensor, 0x03, 0x00);
	adv7180_write_reg(sensor, 0x04, 0x45);
	adv7180_write_reg(sensor, 0x05, 0x00);
	adv7180_write_reg(sensor, 0x06, 0x02);
	adv7180_write_reg(sensor, 0x07, 0x7F);
	adv7180_write_reg(sensor, 0x08, 0x80);
	adv7180_write_reg(sensor, 0x0A, 0x00);
	adv7180_write_reg(sensor, 0x0B, 0x00);
	adv7180_write_reg(sensor, 0x0C, 0x36);
	adv7180_write_reg(sensor, 0x0D, 0x7C);
	adv7180_write_reg(sensor, 0x0E, 0x00);
	adv7180_write_reg(sensor, 0x0F, 0x00);
	adv7180_write_reg(sensor, 0x13, 0x00);
	adv7180_write_reg(sensor, 0x14, 0x12);
	adv7180_write_reg(sensor, 0x15, 0x00);
	adv7180_write_reg(sensor, 0x16, 0x00);
	adv7180_write_reg(sensor, 0x17, 0x01);
	adv7180_write_reg(sensor, 0x18, 0x93);
	adv7180_write_reg(sensor, 0xF1, 0x19);
	adv7180_write_reg(sensor, 0x1A, 0x00);
	adv7180_write_reg(sensor, 0x1B, 0x00);
	adv7180_write_reg(sensor, 0x1C, 0x00);
	adv7180_write_reg(sensor, 0x1D, 0x40);
	adv7180_write_reg(sensor, 0x1E, 0x00);
	adv7180_write_reg(sensor, 0x1F, 0x00);
	adv7180_write_reg(sensor, 0x20, 0x00);
	adv7180_write_reg(sensor, 0x21, 0x00);
	adv7180_write_reg(sensor, 0x22, 0x00);
	adv7180_write_reg(sensor, 0x23, 0xC0);
	adv7180_write_reg(sensor, 0x24, 0x00);
	adv7180_write_reg(sensor, 0x25, 0x00);
	adv7180_write_reg(sensor, 0x26, 0x00);
	adv7180_write_reg(sensor, 0x27, 0x58);
	adv7180_write_reg(sensor, 0x28, 0x00);
	adv7180_write_reg(sensor, 0x29, 0x00);
	adv7180_write_reg(sensor, 0x2A, 0x00);
	adv7180_write_reg(sensor, 0x2B, 0xE1);
	adv7180_write_reg(sensor, 0x2C, 0xAE);
	adv7180_write_reg(sensor, 0x2D, 0xF4);
	adv7180_write_reg(sensor, 0x2E, 0x00);
	adv7180_write_reg(sensor, 0x2F, 0xF0);
	adv7180_write_reg(sensor, 0x30, 0x00);
	adv7180_write_reg(sensor, 0x31, 0x12);
	adv7180_write_reg(sensor, 0x32, 0x41);
	adv7180_write_reg(sensor, 0x33, 0x84);
	adv7180_write_reg(sensor, 0x34, 0x00);
	adv7180_write_reg(sensor, 0x35, 0x02);
	adv7180_write_reg(sensor, 0x36, 0x00);
	adv7180_write_reg(sensor, 0x37, 0x01);
	adv7180_write_reg(sensor, 0x38, 0x80);
	adv7180_write_reg(sensor, 0x39, 0xC0);
	adv7180_write_reg(sensor, 0x3A, 0x10);
	adv7180_write_reg(sensor, 0x3B, 0x05);
	adv7180_write_reg(sensor, 0x3C, 0x58);
	adv7180_write_reg(sensor, 0x3D, 0xB2);
	adv7180_write_reg(sensor, 0x3E, 0x64);
	adv7180_write_reg(sensor, 0x3F, 0xE4);
	adv7180_write_reg(sensor, 0x40, 0x90);
	adv7180_write_reg(sensor, 0x41, 0x01);
	adv7180_write_reg(sensor, 0x42, 0x7E);
	adv7180_write_reg(sensor, 0x43, 0xA4);
	adv7180_write_reg(sensor, 0x44, 0xFF);
	adv7180_write_reg(sensor, 0x45, 0xB6);
	adv7180_write_reg(sensor, 0x46, 0x12);
	adv7180_write_reg(sensor, 0x48, 0x00);
	adv7180_write_reg(sensor, 0x49, 0x00);
	adv7180_write_reg(sensor, 0x4A, 0x00);
	adv7180_write_reg(sensor, 0x4B, 0x00);
	adv7180_write_reg(sensor, 0x4C, 0x00);
	adv7180_write_reg(sensor, 0x4D, 0xEF);
	adv7180_write_reg(sensor, 0x4E, 0x08);
	adv7180_write_reg(sensor, 0x4F, 0x08);
	adv7180_write_reg(sensor, 0x50, 0x08);
	adv7180_write_reg(sensor, 0x51, 0xA4);
	adv7180_write_reg(sensor, 0x52, 0x0B);
	adv7180_write_reg(sensor, 0x53, 0x4E);
	adv7180_write_reg(sensor, 0x54, 0x80);
	adv7180_write_reg(sensor, 0x55, 0x00);
	adv7180_write_reg(sensor, 0x56, 0x10);
	adv7180_write_reg(sensor, 0x57, 0x00);
	adv7180_write_reg(sensor, 0x58, 0x00);
	adv7180_write_reg(sensor, 0x59, 0x00);
	adv7180_write_reg(sensor, 0x5A, 0x00);
	adv7180_write_reg(sensor, 0x5B, 0x00);
	adv7180_write_reg(sensor, 0x5C, 0x00);
	adv7180_write_reg(sensor, 0x5D, 0x00);
	adv7180_write_reg(sensor, 0x5E, 0x00);
	adv7180_write_reg(sensor, 0x5F, 0x00);
	adv7180_write_reg(sensor, 0x60, 0x00);
	adv7180_write_reg(sensor, 0x61, 0x00);
	adv7180_write_reg(sensor, 0x62, 0x20);
	adv7180_write_reg(sensor, 0x63, 0x00);
	adv7180_write_reg(sensor, 0x64, 0x00);
	adv7180_write_reg(sensor, 0x65, 0x00);
	adv7180_write_reg(sensor, 0x66, 0x00);
	adv7180_write_reg(sensor, 0x67, 0x03);
	adv7180_write_reg(sensor, 0x68, 0x01);
	adv7180_write_reg(sensor, 0x69, 0x00);
	adv7180_write_reg(sensor, 0x6A, 0x00);
	adv7180_write_reg(sensor, 0x6B, 0xC0);
	adv7180_write_reg(sensor, 0x6C, 0x00);
	adv7180_write_reg(sensor, 0x6D, 0x00);
	adv7180_write_reg(sensor, 0x6E, 0x00);
	adv7180_write_reg(sensor, 0x6F, 0x00);
	adv7180_write_reg(sensor, 0x70, 0x00);
	adv7180_write_reg(sensor, 0x71, 0x00);
	adv7180_write_reg(sensor, 0x72, 0x00);
	adv7180_write_reg(sensor, 0x73, 0x10);
	adv7180_write_reg(sensor, 0x74, 0x04);
	adv7180_write_reg(sensor, 0x75, 0x01);
	adv7180_write_reg(sensor, 0x76, 0x00);
	adv7180_write_reg(sensor, 0x77, 0x3F);
	adv7180_write_reg(sensor, 0x78, 0xFF);
	adv7180_write_reg(sensor, 0x79, 0xFF);
	adv7180_write_reg(sensor, 0x7A, 0xFF);
	adv7180_write_reg(sensor, 0x7B, 0x1E);
	adv7180_write_reg(sensor, 0x7C, 0xC0);
	adv7180_write_reg(sensor, 0x7D, 0x00);
	adv7180_write_reg(sensor, 0x7E, 0x00);
	adv7180_write_reg(sensor, 0x7F, 0x00);
	adv7180_write_reg(sensor, 0x80, 0x00);
	adv7180_write_reg(sensor, 0x81, 0xC0);
	adv7180_write_reg(sensor, 0x82, 0x04);
	adv7180_write_reg(sensor, 0x83, 0x00);
	adv7180_write_reg(sensor, 0x84, 0x0C);
	adv7180_write_reg(sensor, 0x85, 0x02);
	adv7180_write_reg(sensor, 0x86, 0x03);
	adv7180_write_reg(sensor, 0x87, 0x63);
	adv7180_write_reg(sensor, 0x88, 0x5A);
	adv7180_write_reg(sensor, 0x89, 0x08);
	adv7180_write_reg(sensor, 0x8A, 0x10);
	adv7180_write_reg(sensor, 0x8B, 0x00);
	adv7180_write_reg(sensor, 0x8C, 0x40);
	adv7180_write_reg(sensor, 0x8D, 0x00);
	adv7180_write_reg(sensor, 0x8E, 0x40);
	adv7180_write_reg(sensor, 0x8F, 0x00);
	adv7180_write_reg(sensor, 0x90, 0x00);
	adv7180_write_reg(sensor, 0x91, 0x50);
	adv7180_write_reg(sensor, 0x92, 0x00);
	adv7180_write_reg(sensor, 0x93, 0x00);
	adv7180_write_reg(sensor, 0x94, 0x00);
	adv7180_write_reg(sensor, 0x95, 0x00);
	adv7180_write_reg(sensor, 0x96, 0x00);
	adv7180_write_reg(sensor, 0x97, 0xF0);
	adv7180_write_reg(sensor, 0x98, 0x00);
	adv7180_write_reg(sensor, 0x99, 0x00);
	adv7180_write_reg(sensor, 0x9A, 0x00);
	adv7180_write_reg(sensor, 0x9B, 0x00);
	adv7180_write_reg(sensor, 0x9C, 0x00);
	adv7180_write_reg(sensor, 0x9D, 0x00);
	adv7180_write_reg(sensor, 0x9E, 0x00);
	adv7180_write_reg(sensor, 0x9F, 0x00);
	adv7180_write_reg(sensor, 0xA0, 0x00);
	adv7180_write_reg(sensor, 0xA1, 0x00);
	adv7180_write_reg(sensor, 0xA2, 0x00);
	adv7180_write_reg(sensor, 0xA3, 0x00);
	adv7180_write_reg(sensor, 0xA4, 0x00);
	adv7180_write_reg(sensor, 0xA5, 0x00);
	adv7180_write_reg(sensor, 0xA6, 0x00);
	adv7180_write_reg(sensor, 0xA7, 0x00);
	adv7180_write_reg(sensor, 0xA8, 0x00);
	adv7180_write_reg(sensor, 0xA9, 0x00);
	adv7180_write_reg(sensor, 0xAA, 0x00);
	adv7180_write_reg(sensor, 0xAB, 0x00);
	adv7180_write_reg(sensor, 0xAC, 0x00);
	adv7180_write_reg(sensor, 0xAD, 0x00);
	adv7180_write_reg(sensor, 0xAE, 0x60);
	adv7180_write_reg(sensor, 0xAF, 0x00);
	adv7180_write_reg(sensor, 0xB0, 0x00);
	adv7180_write_reg(sensor, 0xB1, 0x60);
	adv7180_write_reg(sensor, 0xB2, 0x1C);
	adv7180_write_reg(sensor, 0xB3, 0x54);
	adv7180_write_reg(sensor, 0xB4, 0x00);
	adv7180_write_reg(sensor, 0xB5, 0x00);
	adv7180_write_reg(sensor, 0xB6, 0x00);
	adv7180_write_reg(sensor, 0xB7, 0x13);
	adv7180_write_reg(sensor, 0xB8, 0x03);
	adv7180_write_reg(sensor, 0xB9, 0x33);
	adv7180_write_reg(sensor, 0xBF, 0x02);
	adv7180_write_reg(sensor, 0xC0, 0x00);
	adv7180_write_reg(sensor, 0xC1, 0x00);
	adv7180_write_reg(sensor, 0xC2, 0x00);
	adv7180_write_reg(sensor, 0xC3, 0x00);
	adv7180_write_reg(sensor, 0xC4, 0x00);
	adv7180_write_reg(sensor, 0xC5, 0x81);
	adv7180_write_reg(sensor, 0xC6, 0x00);
	adv7180_write_reg(sensor, 0xC7, 0x00);
	adv7180_write_reg(sensor, 0xC8, 0x00);
	adv7180_write_reg(sensor, 0xC9, 0x04);
	adv7180_write_reg(sensor, 0xCC, 0x69);
	adv7180_write_reg(sensor, 0xCD, 0x00);
	adv7180_write_reg(sensor, 0xCE, 0x01);
	adv7180_write_reg(sensor, 0xCF, 0xB4);
	adv7180_write_reg(sensor, 0xD0, 0x00);
	adv7180_write_reg(sensor, 0xD1, 0x10);
	adv7180_write_reg(sensor, 0xD2, 0xFF);
	adv7180_write_reg(sensor, 0xD3, 0xFF);
	adv7180_write_reg(sensor, 0xD4, 0x7F);
	adv7180_write_reg(sensor, 0xD5, 0x7F);
	adv7180_write_reg(sensor, 0xD6, 0x3E);
	adv7180_write_reg(sensor, 0xD7, 0x08);
	adv7180_write_reg(sensor, 0xD8, 0x3C);
	adv7180_write_reg(sensor, 0xD9, 0x08);
	adv7180_write_reg(sensor, 0xDA, 0x3C);
	adv7180_write_reg(sensor, 0xDB, 0x9B);
	adv7180_write_reg(sensor, 0xDC, 0xAC);
	adv7180_write_reg(sensor, 0xDD, 0x4C);
	adv7180_write_reg(sensor, 0xDE, 0x00);
	adv7180_write_reg(sensor, 0xDF, 0x00);
	adv7180_write_reg(sensor, 0xE0, 0x14);
	adv7180_write_reg(sensor, 0xE1, 0x80);
	adv7180_write_reg(sensor, 0xE2, 0x80);
	adv7180_write_reg(sensor, 0xE3, 0x80);
	adv7180_write_reg(sensor, 0xE4, 0x80);
	adv7180_write_reg(sensor, 0xE5, 0x25);
	adv7180_write_reg(sensor, 0xE6, 0x44);
	adv7180_write_reg(sensor, 0xE7, 0x63);
	adv7180_write_reg(sensor, 0xE8, 0x65);
	adv7180_write_reg(sensor, 0xE9, 0x14);
	adv7180_write_reg(sensor, 0xEA, 0x63);
	adv7180_write_reg(sensor, 0xEB, 0x55);
	adv7180_write_reg(sensor, 0xEC, 0x55);
	adv7180_write_reg(sensor, 0xEE, 0x00);
	adv7180_write_reg(sensor, 0xEF, 0x4A);
	adv7180_write_reg(sensor, 0xF0, 0x44);
	adv7180_write_reg(sensor, 0xF1, 0x0C);
	adv7180_write_reg(sensor, 0xF2, 0x32);
	adv7180_write_reg(sensor, 0xF3, 0x00);
	adv7180_write_reg(sensor, 0xF4, 0x3F);
	adv7180_write_reg(sensor, 0xF5, 0xE0);
	adv7180_write_reg(sensor, 0xF6, 0x69);
	adv7180_write_reg(sensor, 0xF7, 0x10);
	adv7180_write_reg(sensor, 0xF8, 0x00);
	adv7180_write_reg(sensor, 0xF9, 0x03);
	adv7180_write_reg(sensor, 0xFA, 0xFA);
	adv7180_write_reg(sensor, 0xFB, 0x40);
}

/*! ADV7180 I2C attach function.
 *
 *  @param *adapter	struct i2c_adapter *.
 *
 *  @return		Error code indicating success or failure.
 */

/*!
 * ADV7180 I2C probe function.
 * Function set in i2c_driver struct.
 * Called by insmod.
 *
 *  @param *adapter	I2C adapter descriptor.
 *
 *  @return		Error code indicating success or failure.
 */
static int adv7180_probe(struct i2c_client *client,
			 const struct i2c_device_id *id)
{
	struct v4l2_of_endpoint bus_cfg;
	struct device_node *endpoint;
	struct adv7180_dev *sensor;
	const char *norm = "pal";
	int ret = 0;

	dev_dbg(&client->dev, "%s\n", __func__);

	sensor = devm_kzalloc(&client->dev, sizeof(struct adv7180_dev),
			      GFP_KERNEL);
	if (!sensor)
		return -ENOMEM;

	ret = of_property_read_string(client->dev.of_node, "default-std",
			&norm);
	if (ret < 0 && ret != -EINVAL) {
		dev_err(&client->dev, "error reading default-std property!\n");
		return ret;
	}
	if (!strcasecmp(norm, "pal")) {
		sensor->std_id = V4L2_STD_PAL;
		sensor->video_idx = ADV7180_PAL;
		dev_info(&client->dev, "defaulting to PAL!\n");
	} else if (!strcasecmp(norm, "ntsc")) {
		sensor->std_id = V4L2_STD_NTSC;
		sensor->video_idx = ADV7180_NTSC;
		dev_info(&client->dev, "defaulting to NTSC!\n");
	} else {
		dev_err(&client->dev, "invalid default-std value: '%s'!\n",
				norm);
		return -EINVAL;
	}

	/* Set initial values for the sensor struct. */
	sensor->i2c_client = client;
	sensor->streamcap.timeperframe.denominator = 30;
	sensor->streamcap.timeperframe.numerator = 1;
	sensor->pix.width = video_fmts[sensor->video_idx].raw_width;
	sensor->pix.height = video_fmts[sensor->video_idx].raw_height;
	sensor->pix.pixelformat = V4L2_MBUS_FMT_UYVY8_2X8;
	sensor->pix.field = V4L2_FIELD_INTERLACED;

	mutex_init(&sensor->mutex);

	endpoint = v4l2_of_get_next_endpoint(client->dev.of_node, NULL);
	if (!endpoint) {
		dev_err(&client->dev, "endpoint node not found\n");
		return -EINVAL;
	}

	v4l2_of_parse_endpoint(endpoint, &bus_cfg);
	if (bus_cfg.bus_type != V4L2_MBUS_BT656) {
		dev_err(&client->dev, "invalid bus type, must be bt.656\n");
		return -EINVAL;
	}
	of_node_put(endpoint);

	sensor->bus_cfg = bus_cfg.bus.parallel;

	ret = of_get_named_gpio(client->dev.of_node, "pwdn-gpio", 0);
	if (gpio_is_valid(ret)) {
		sensor->pwdn_gpio = ret;
		ret = gpio_request_one(sensor->pwdn_gpio, GPIOF_OUT_INIT_HIGH,
				       "adv7180_pwdn");
		if (ret < 0) {
			dev_err(&client->dev,
				"request for power down gpio failed\n");
			return ret;
		}
	} else {
		if (ret == -EPROBE_DEFER)
			return ret;
		/* assume a power-down gpio is not required */
		sensor->pwdn_gpio = -1;
	}

	adv7180_regulator_enable(sensor);

	/* Power on the tvin chip */
	adv7180_power(sensor, true);

	dev_dbg(&sensor->i2c_client->dev,
		"%s: adv7180 probe i2c address is 0x%02X\n",
		__func__, sensor->i2c_client->addr);

	/*! ADV7180 initialization. */
	adv7180_hard_reset(sensor);

	/*! Read the revision ID of the tvin chip */
	sensor->rev_id = adv7180_read(sensor, ADV7180_IDENT);
	if (sensor->rev_id < 0) {
		dev_err(&sensor->i2c_client->dev,
			"failed to read ADV7180 IDENT register!\n");
		ret = -ENODEV;
		goto cleanup;
	}

	dev_info(&sensor->i2c_client->dev,
		 "Analog Devices ADV7180 Rev 0x%02X detected!\n",
		 sensor->rev_id);

	dev_dbg(&sensor->i2c_client->dev, "   type is %d (expect %d)\n",
		adv7180_int_device.type, v4l2_int_type_slave);
	dev_dbg(&sensor->i2c_client->dev, "   num ioctls is %d\n",
		adv7180_int_device.u.slave->num_ioctls);

	/* This function attaches this structure to the /dev/video0 device.
	 * The pointer in priv points to the mt9v111_data structure here.*/
	adv7180_int_device.priv = sensor;
	ret = v4l2_int_device_register(&adv7180_int_device);

	/* see if there is a signal lock already */
	adv7180_update_lock_status(sensor);
	adv7180_get_autodetect_std(sensor);

	if (sensor->i2c_client->irq) {
		ret = request_threaded_irq(sensor->i2c_client->irq,
					   NULL, adv7180_interrupt,
					   IRQF_TRIGGER_LOW | IRQF_ONESHOT,
					   IF_NAME, sensor);
		if (ret < 0) {
			dev_err(&sensor->i2c_client->dev,
				"Failed to register irq %d\n",
				sensor->i2c_client->irq);
			goto unregister;
		}

		adv7180_enable_interrupts(sensor);

		dev_info(&sensor->i2c_client->dev, "Registered irq %d\n",
			 sensor->i2c_client->irq);
	}

	return 0;

unregister:
	v4l2_int_device_unregister(&adv7180_int_device);
cleanup:
	gpio_free(sensor->pwdn_gpio);
	adv7180_regulator_disable(sensor);
	return ret;
}

/*!
 * ADV7180 I2C detach function.
 * Called on rmmod.
 *
 *  @param *client	struct i2c_client*.
 *
 *  @return		Error code indicating success or failure.
 */
static int adv7180_detach(struct i2c_client *client)
{
	struct adv7180_dev *sensor =
		(struct adv7180_dev *)adv7180_int_device.priv;

	dev_dbg(&sensor->i2c_client->dev,
		"%s: Removing %s video decoder @ 0x%02X from adapter %s\n",
		__func__, IF_NAME, client->addr << 1, client->adapter->name);

	if (sensor->i2c_client->irq)
		free_irq(sensor->i2c_client->irq, sensor);

	/* Power off the tvin chip */
	adv7180_power(sensor, false);

	if (gpio_is_valid(sensor->pwdn_gpio))
		gpio_free(sensor->pwdn_gpio);

	adv7180_regulator_disable(sensor);

	v4l2_int_device_unregister(&adv7180_int_device);

	return 0;
}

static const struct i2c_device_id adv7180_id[] = {
	{ "adv7180", 0 },
	{}
};
MODULE_DEVICE_TABLE(i2c, adv7180_id);

static struct of_device_id adv7180_dt_ids[] = {
	{ .compatible = "adi,adv7180" },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, adv7180_dt_ids);

static struct i2c_driver adv7180_driver = {
	.driver = {
		.name	= "adv7180",
		.owner	= THIS_MODULE,
		.of_match_table	= adv7180_dt_ids,
	},
	.id_table	= adv7180_id,
	.probe		= adv7180_probe,
	.remove		= adv7180_detach,
};

module_i2c_driver(adv7180_driver);

MODULE_AUTHOR("Freescale Semiconductor");
MODULE_DESCRIPTION("Anolog Device ADV7180 video decoder driver");
MODULE_LICENSE("GPL");
