/*
 * Copyright 2012 Robert Bosch Car Multimedia GmbH. All Rights Reserved.
 */
/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */
/*!
 * @file tv_fpga.c
 *
 * @brief TV_FPGA video decoder functions
 *
 * @ingroup Camera
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/of_device.h>
#include <linux/wait.h>
#include <media/v4l2-device.h>
#include <media/v4l2-ioctl.h>
#include <media/v4l2-chip-ident.h>
#include <media/v4l2-subdev.h>
#include <media/v4l2-of.h>

/*! List of input video formats supported. The video formats is corresponding
 * with v4l2 id in video_fmt_t
 */
enum tv_fpga_fmt_idx {
	TV_FPGA_5120_960 = 0,	/* Format 1) 5120 x 960 pixel */
	TV_FPGA_1280_3840,	/* Format 2) 1280 x 3840 pixel */
};
/*! Number of video standards supported (including 'not locked' signal). */
#define TV_FPGA_STD_MAX		(TV_FPGA_1280_3840)
/*! Video format structure. */
struct video_fmt_t {
	int v4l2_id;		/*!< Video for linux ID. */
	char name[16];		/*!< Name (e.g., "NTSC", "PAL", etc.) */
	struct v4l2_rect raw;   /* raw sensor frame */
};

/*! Description of video formats supported.
 *
 *  5120_960 --> 5120 x 960  pixel active and raw
 *  1280_3840--> 1280 x 3840 pixel active and raw
 */
static struct video_fmt_t video_fmts[] = {
	{
		.name = "5120_960",
		.raw = {
			.width = 5120,
			.height = 960,
		},
	}, {
		.name = "1280_3840",
		.raw = {
			.width = 1280,
			.height = 3840,
		},
	},
};

struct tv_fpga_dev {
	struct i2c_client *i2c_client;
	struct device *dev;
	struct v4l2_subdev sd;
	struct v4l2_of_endpoint ep; /* the parsed DT endpoint info */
	struct v4l2_mbus_framefmt fmt;
	struct v4l2_captureparm streamcap;
};

static inline struct tv_fpga_dev *to_tv_fpga_dev(struct v4l2_subdev *sd)
{
	return container_of(sd, struct tv_fpga_dev, sd);
}

/*!* Standard index of TV_FPGA. */
static enum tv_fpga_fmt_idx video_idx = TV_FPGA_5120_960;
#define IF_NAME                    "tv_fpga"


/* --------------- Subdev Operations --------------- */

/*!
 * tv_fpga_querystd - V4L2 sensor interface handler for VIDIOC_QUERYSTD
 * @sd: pointer to standard V4L2 subdevice structure
 * @std: v4l2_std_id *
 */
static int tv_fpga_querystd(struct v4l2_subdev *sd, v4l2_std_id *std)
{
	*std = V4L2_STD_ALL;
	return 0;
}

/*!
 * tv_fpga_g_parm - V4L2 sensor interface handler for VIDIOC_G_PARM ioctl
 * @sd: pointer to standard V4L2 subdevice structure
 * @a: pointer to standard V4L2 VIDIOC_G_PARM ioctl structure
 *
 * Returns the sensor's video CAPTURE parameters.
 */
static int tv_fpga_g_parm(struct v4l2_subdev *sd, struct v4l2_streamparm *a)
{
	struct tv_fpga_dev *sensor = to_tv_fpga_dev(sd);
	struct v4l2_captureparm *cparm = &a->parm.capture;

	if (a->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
		return -EINVAL;

	memset(a, 0, sizeof(*a));
	a->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	cparm->capability = sensor->streamcap.capability;
	cparm->timeperframe = sensor->streamcap.timeperframe;
	cparm->capturemode = sensor->streamcap.capturemode;

	return 0;
}

/*!
 * tv_fpga_s_parm - V4L2 sensor interface handler for VIDIOC_S_PARM ioctl
 * @sd: pointer to standard V4L2 subdevice structure
 * @a: pointer to standard V4L2 VIDIOC_S_PARM ioctl structure
 *
 * Configures the sensor to use the input parameters, if possible.  If
 * not possible, reverts to the old parameters and returns the
 * appropriate error code.
 *
 * This driver cannot change these settings.
 */
static int tv_fpga_s_parm(struct v4l2_subdev *sd, struct v4l2_streamparm *a)
{
	return 0;
}

static int tv_fpga_enum_mbus_fmt(struct v4l2_subdev *sd, unsigned index,
				enum v4l2_mbus_pixelcode *code)
{
	struct tv_fpga_dev *sensor = to_tv_fpga_dev(sd);

	if (index != 0)
		return -EINVAL;

	*code = sensor->fmt.code;

	return 0;
}

static int tv_fpga_g_mbus_fmt(struct v4l2_subdev *sd,
			      struct v4l2_mbus_framefmt *fmt)

{
	struct tv_fpga_dev *sensor = to_tv_fpga_dev(sd);

	*fmt = sensor->fmt;
	return 0;
}

static int tv_fpga_try_mbus_fmt(struct v4l2_subdev *sd,
				struct v4l2_mbus_framefmt *fmt)
{
	struct tv_fpga_dev *sensor = to_tv_fpga_dev(sd);

	*fmt = sensor->fmt;
	return 0;
}

static int tv_fpga_s_mbus_fmt(struct v4l2_subdev *sd,
			      struct v4l2_mbus_framefmt *fmt)
{
	struct tv_fpga_dev *sensor = to_tv_fpga_dev(sd);

	*fmt = sensor->fmt;
	return 0;
}

/*!
 * tv_fpga_enum_framesizes - V4L2 sensor interface handler for
 *			   VIDIOC_ENUM_FRAMESIZES ioctl
 * @sd: pointer to standard V4L2 subdevice structure
 * @fsize: standard V4L2 VIDIOC_ENUM_FRAMESIZES ioctl structure
 *
 * Return 0 if successful, otherwise -EINVAL.
 */
static int tv_fpga_enum_framesizes(struct v4l2_subdev *sd,
				 struct v4l2_frmsizeenum *fsize)
{
	if (fsize->index > TV_FPGA_STD_MAX)
		return -EINVAL;

	fsize->type = V4L2_FRMSIZE_TYPE_DISCRETE;
	fsize->discrete.width = video_fmts[fsize->index].raw.width;
	fsize->discrete.height = video_fmts[fsize->index].raw.height;

	return 0;
}

static int tv_fpga_g_input_status(struct v4l2_subdev *sd, u32 *status)
{
	*status = 0;
	return 0;
}

static int tv_fpga_s_routing(struct v4l2_subdev *sd, u32 input,
			     u32 output, u32 config)
{
	return (input != 0) ? -EINVAL : 0;
}

/*!
 * tv_fpga_g_chip_ident - V4L2 sensor interface handler for
 *			VIDIOC_DBG_G_CHIP_IDENT ioctl
 * @sd: pointer to standard V4L2 subdevice structure
 * @id: pointer to int
 *
 * Return 0.
 */
static int tv_fpga_g_chip_ident(struct v4l2_subdev *sd,
				struct v4l2_dbg_chip_ident *chip)
{
	chip->match.type = V4L2_CHIP_MATCH_HOST;
	strcpy(chip->match.name, "top_view_fpga");

	return 0;
}

static int tv_fpga_g_mbus_config(struct v4l2_subdev *sd,
				struct v4l2_mbus_config *cfg)
{
	struct tv_fpga_dev *sensor = to_tv_fpga_dev(sd);

	cfg->type = V4L2_MBUS_PARALLEL;
	cfg->flags = sensor->ep.bus.parallel.flags;

	return 0;
}

static int tv_fpga_s_stream(struct v4l2_subdev *sd, int enable)
{
	return 0;
}

static struct v4l2_subdev_core_ops tv_fpga_core_ops = {
	.g_chip_ident = tv_fpga_g_chip_ident,
};

static struct v4l2_subdev_video_ops tv_fpga_video_ops = {
	.enum_mbus_fmt = tv_fpga_enum_mbus_fmt,
	.try_mbus_fmt = tv_fpga_try_mbus_fmt,
	.g_mbus_fmt = tv_fpga_g_mbus_fmt,
	.s_mbus_fmt = tv_fpga_s_mbus_fmt,
	.s_parm = tv_fpga_s_parm,
	.g_parm = tv_fpga_g_parm,
	.enum_framesizes = tv_fpga_enum_framesizes,
	.g_input_status = tv_fpga_g_input_status,
	.s_routing = tv_fpga_s_routing,
	.querystd = tv_fpga_querystd,
	.g_mbus_config  = tv_fpga_g_mbus_config,
	.s_stream = tv_fpga_s_stream,
};

static struct v4l2_subdev_ops tv_fpga_subdev_ops = {
	.core = &tv_fpga_core_ops,
	.video = &tv_fpga_video_ops,
};

/*!
 * TV_FPGA init function.
 * Called on insmod.
 *
 * @return    Error code indicating success or failure.
 */
static int tv_fpga_probe(struct platform_device *pdev)
{
	struct device_node *endpoint;
	struct tv_fpga_dev *sensor;

	pr_debug("In tv_fpga_probe\n");

	sensor = devm_kzalloc(&pdev->dev, sizeof(struct tv_fpga_dev),
			      GFP_KERNEL);
	if (!sensor)
		return -ENOMEM;

	sensor->streamcap.timeperframe.denominator = 30;
	sensor->streamcap.timeperframe.numerator = 1;
	video_idx = TV_FPGA_5120_960;
	sensor->fmt.width = video_fmts[video_idx].raw.width;
	sensor->fmt.height = video_fmts[video_idx].raw.height;
	sensor->fmt.code = V4L2_MBUS_FMT_UYVY8_1X16;
	sensor->fmt.field = V4L2_FIELD_NONE;
	sensor->dev = &pdev->dev;

	endpoint = v4l2_of_get_next_endpoint(pdev->dev.of_node, NULL);
	if (!endpoint) {
		dev_err(&pdev->dev, "endpoint node not found\n");
		return -EINVAL;
	}

	v4l2_of_parse_endpoint(endpoint, &sensor->ep);
	if (sensor->ep.bus_type != V4L2_MBUS_PARALLEL) {
		dev_err(&pdev->dev, "invalid bus type, must be parallel\n");
		return -EINVAL;
	}
	of_node_put(endpoint);

	v4l2_subdev_init(&sensor->sd, &tv_fpga_subdev_ops);

	dev_set_drvdata(sensor->dev, &sensor->sd);

	return 0;
}

static int tv_fpga_remove(struct platform_device *pdev)
{
	return 0;
}

static struct of_device_id tv_fpga_dt_ids[] = {
	{ .compatible = "bosch,tv-fpga" },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, tv_fpga_dt_ids);

static struct platform_driver tv_fpga_driver = {
	.driver = {
		.name  = "tv-fpga-sensor",
		.owner = THIS_MODULE,
		.of_match_table = tv_fpga_dt_ids,
	},
	.probe    = tv_fpga_probe,
	.remove   = tv_fpga_remove,
};
module_platform_driver(tv_fpga_driver);

MODULE_AUTHOR("Robert Bosch Car Multimedia GmbH");
MODULE_DESCRIPTION("TV_FPGA video decoder driver");
MODULE_LICENSE("GPL");
