 /* Neonode Zforce touch device HW check interfaces.
 *
 * Copyright (c) 2014 Robert Bosch Car Multimedia GmbH
 *
 * This source code is provided as a sample only and has not gone
 * through full internal testing. This sample code is provided "as is",
 * with no  warranties for the functionality of the code, nor
 * any warranties concerning side effects when using the code. The
 * intention of this code is reference example for internal evaluation only.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the term of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/module.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/cdev.h>
#include <linux/io.h>
#include <linux/uaccess.h>
#include <linux/slab.h>
#include <linux/device.h>

#include "zforce_hwcheck.h"
#include "zforce_ioctl.h"

#define WAIT_CMD_TOUT_MS 1500
#define RX_DATA_BLOCK_LEN_OFFSET 1

struct response_buf {
	unsigned char len;
	unsigned char buf[ZFORCE_BUF_LEN];
};

struct hwcheck_chrdev {
	struct cdev cdev;
	struct class *hwcheck_class;
	int major_no;
	int minor_no;
	void *zforce_dev;
	struct response_buf buf;
};

static struct hwcheck_chrdev *hwcheckdev;

DECLARE_COMPLETION(wait_cmd_complete);

#ifdef CONFIG_COMPAT
static long hwcheck_chrdev_compat_ioctl(struct file *filp,
		unsigned int cmd, unsigned long arg)
{
	return hwcheck_chrdev_ioctl(filp, cmd, (unsigned long)compat_ptr(arg));
}
#else
#define hwcheck_chrdev_compat_ioctl NULL
#endif /* CONFIG_COMPAT */

/* boot loader requests */
#define BL_HEADER 0x80

/* ERASE_SEGMENT:
 * A memory segment (512B in Flash) addressed by ADDR is erased. */
static u_int8_t bl_ERASE_SEGMENT[] = {BL_HEADER, 3, 0x12 /*ADDR[L], ADDR[H]*/};
/* ERASE_APP:
 * The entire application area is erased. */
static u_int8_t bl_ERASE_APP[] = {BL_HEADER, 1, 0x15, 0, 0};
/* RX_DATA_BLOCK:
 * n Bytes of data are programmed starting at address ADDR */
static u_int8_t bl_RX_DATA_BLOCK[] = {BL_HEADER, 19, 0x10 /*ADDR[L], ADDR[H], DATA0-15*/};
/* TX_VERSION:
 * Requests the boot loader version from the target */
static u_int8_t bl_TX_VERSION[] = {BL_HEADER, 1, 0x19, 0, 0};
/* JUMP2APP:
 * Target resets and jumps to zForce FW if validation is OK */
static u_int8_t bl_JUMP2APP[] = {BL_HEADER, 1, 0x1C, 0, 0};
/* VALIDATEAPP:
 * Validates the checksum of the App, does not jump to app after */
static u_int8_t bl_VALIDATEAPP[] = {BL_HEADER, 1, 0x1D, 0, 0};

/* calculate CCITT CRC16 */
static unsigned short crc16(const unsigned char *buf, int len)
{
	#define CRC_POLYNOM 0x1021
	unsigned short crc = 0xFFFF;
	int byte;
	unsigned char bit;

	for (byte = 0; byte < len; ++byte)
	{
		crc ^= (buf[byte] << 8);
		for (bit = 8; bit > 0; --bit)
		{
			if (crc & 0x8000)
				crc = (crc << 1) ^ CRC_POLYNOM;
			else
				crc = (crc << 1);
		}
	}
	return crc;
}

/* insert checksum at the end off buffer */
static void insert_crc(unsigned char *buf, int len)
{
	unsigned short crc = crc16((char*)buf + 2, len - 4);
	buf[len-1] = (unsigned char)((crc >> 8) & 0x00FF);
	buf[len-2] = (unsigned char)(crc & 0x00FF);
}

/* send function request to zForce I2C bootloader */
#define SEND_BOOTLOADER_REQUEST(buf) \
	hwcheck_send_bootloader_request(hwcheckdev->zforce_dev, buf, sizeof(buf))

static int hwcheck_send_bootloader_request(void *zforce_dev,
	u_int8_t *data, int len)
{
	insert_crc(data, len);
	return zforce_ts_hwcheck_send_bootloader_req(zforce_dev, data, len);
}

/* erase 512 byte segment */
static int hwcheck_send_bootloader_erase_segment(void *zforce_dev,
	u_int8_t *data, int len)
{
	u_int8_t buf[ZFORCE_BUF_LEN];
	/* number of bytes to send = address + header + CRC */
	int send_len = len + sizeof(bl_ERASE_SEGMENT) + 2;

	/* copy header to transfer buffer */
	memcpy(buf, bl_ERASE_SEGMENT, sizeof(bl_ERASE_SEGMENT));

	/* copy address behind header */
	memcpy(buf + sizeof(bl_ERASE_SEGMENT), data, len);

	/* add CRC */
	insert_crc(buf, send_len);

	return zforce_ts_hwcheck_send_bootloader_req(zforce_dev, buf, send_len);
}

/* send data to zForce I2C boot loader */
static int hwcheck_send_bootloader_data(void *zforce_dev,
	u_int8_t *data, int len)
{
	u_int8_t buf[ZFORCE_BUF_LEN];
	/* number of bytes to send = data + header + CRC */
	int send_len = len + sizeof(bl_RX_DATA_BLOCK) + 2;

	/* copy header to transfer buffer */
	memcpy(buf, bl_RX_DATA_BLOCK, sizeof(bl_RX_DATA_BLOCK));

	/* correct header length information (data + command) */
	buf[RX_DATA_BLOCK_LEN_OFFSET] = len + 1;

	/* copy address and data behind header */
	memcpy(buf + sizeof(bl_RX_DATA_BLOCK), data, len);

	/* add CRC */
	insert_crc(buf, send_len);

	return zforce_ts_hwcheck_send_bootloader_req(zforce_dev, buf, send_len);
}


static int hwcheck_chrdev_open(struct inode *inode, struct file *filp)
{
	struct hwcheck_chrdev *dev;

	dev = container_of(inode->i_cdev, struct hwcheck_chrdev, cdev);
	filp->private_data = dev;
	return 0;
}

static int hwcheck_chrdev_deinit(struct hwcheck_chrdev *dev)
{
	device_destroy(dev->hwcheck_class, MKDEV(dev->major_no, dev->minor_no));
	class_destroy(dev->hwcheck_class);
	cdev_del(&dev->cdev);
	kfree(dev);
	unregister_chrdev_region(MKDEV(dev->major_no, dev->minor_no), 1);
	return 0;
}

static long hwcheck_chrdev_ioctl(struct file *filp,
		unsigned int cmd, unsigned long arg)
{
	struct hwcheck_chrdev *dev;
	struct response_buf *resp_buf;
	unsigned long tout;
	int reset, axis, ret = -EFAULT;

	INIT_COMPLETION(wait_cmd_complete);

	dev = filp->private_data;
	resp_buf = &dev->buf;
	memset(resp_buf->buf, 0, ZFORCE_BUF_LEN);

	switch (cmd) {
	/* firmware ioctls */
	case ZFORCE_HWCHECKDEV_IOC_WR_GVER:
		ret = zforce_ts_hwcheck_send_status_req(hwcheckdev->zforce_dev);
	break;
	case ZFORCE_HWCHECKDEV_IOC_WR_LEDL:
		if (get_user(axis, (int __user *)arg))
			break;
		ret = zforce_ts_hwcheck_send_led_levels_req(
				hwcheckdev->zforce_dev, axis);
		break;
	case ZFORCE_HWCHECKDEV_IOC_WR_OPNS:
		if (get_user(axis, (int __user *)arg))
			break;
		ret = zforce_ts_hwcheck_send_send_openshort_req(
				hwcheckdev->zforce_dev, axis);
		break;
	case ZFORCE_HWCHECKDEV_IOC_WR_LOWS:
		if (get_user(axis, (int __user *)arg))
			break;
		ret = zforce_ts_hwcheck_send_low_signals_req(
				hwcheckdev->zforce_dev, axis);
		break;
	case ZFORCE_HWCHECKDEV_IOC_WR_BOOTMODE:
		ret = zforce_ts_hwcheck_send_bootmode_req(hwcheckdev->zforce_dev);
		break;

	/* bootloader ioctl */
	case ZFORCE_HWCHECKDEV_IOC_WR_BL_TX_VERSION:
	case ZFORCE_HWCHECKDEV_IOC_WR_BL_ERASE_APP:
	case ZFORCE_HWCHECKDEV_IOC_WR_BL_JUMP2APP:
	case ZFORCE_HWCHECKDEV_IOC_WR_BL_VALIDATEAPP:
	case ZFORCE_HWCHECKDEV_IOC_WR_BL_RX_DATA_BLOCK:
	case ZFORCE_HWCHECKDEV_IOC_WR_BL_ERASE_SEGMENT:
		/* assume zForce boot loader mode to handle response */
		zforce_bootloader = 1;
		switch (cmd) {
		case ZFORCE_HWCHECKDEV_IOC_WR_BL_TX_VERSION:
			ret = SEND_BOOTLOADER_REQUEST(bl_TX_VERSION);
			break;
		case ZFORCE_HWCHECKDEV_IOC_WR_BL_ERASE_APP:
			ret = SEND_BOOTLOADER_REQUEST(bl_ERASE_APP);
			break;
		case ZFORCE_HWCHECKDEV_IOC_WR_BL_JUMP2APP:
			ret = SEND_BOOTLOADER_REQUEST(bl_JUMP2APP);
			/* leave boot loader mode */
			zforce_bootloader = 0;
			break;
		case ZFORCE_HWCHECKDEV_IOC_WR_BL_VALIDATEAPP:
			ret = SEND_BOOTLOADER_REQUEST(bl_VALIDATEAPP);
			break;
		case ZFORCE_HWCHECKDEV_IOC_WR_BL_RX_DATA_BLOCK:
			if (copy_from_user(resp_buf->buf, (void __user *)arg, ZFORCE_BUF_LEN))
				break;
			ret = hwcheck_send_bootloader_data(
				hwcheckdev->zforce_dev, resp_buf->buf + 1, resp_buf->buf[0]);
			break;
		case ZFORCE_HWCHECKDEV_IOC_WR_BL_ERASE_SEGMENT:
			if (copy_from_user(resp_buf->buf, (void __user *)arg, ZFORCE_BUF_LEN))
				break;
			ret = hwcheck_send_bootloader_erase_segment(
				hwcheckdev->zforce_dev, resp_buf->buf + 1, resp_buf->buf[0]);
			break;
		}
		break;

	/* set reset signal */
	case ZFORCE_HWCHECKDEV_IOC_WR_SET_RESET:
		if (get_user(reset, (int __user *)arg))
			break;
		ret = zforce_ts_hwcheck_set_reset(hwcheckdev->zforce_dev, reset);
		break;

	default:
		ret = -EINVAL;
		break;
	}

	if (ret <= 0)
		return ret;

	tout = wait_for_completion_killable_timeout(
			&wait_cmd_complete,
			msecs_to_jiffies(WAIT_CMD_TOUT_MS));

	if (tout > 0)
		ret = copy_to_user((char __user *)arg,
				(char *)resp_buf,
				sizeof(resp_buf->len)+resp_buf->len);
	else
		ret = -EFAULT;

	return ret;
}

static int hwcheck_chrdev_release(struct inode *inode, struct file *filp)
{
	return 0;
}

static const struct file_operations hwcheck_chrdev_fops = {
	.owner     = THIS_MODULE,
	.open        = hwcheck_chrdev_open,
	.unlocked_ioctl	= hwcheck_chrdev_ioctl,
	.compat_ioctl = hwcheck_chrdev_compat_ioctl,
	.release   = hwcheck_chrdev_release,
};

static struct hwcheck_chrdev *hwcheck_chrdev_init(void)
{
	dev_t dev_no;
	int ret;
	struct hwcheck_chrdev *dev;

	dev = kzalloc(sizeof(struct hwcheck_chrdev), GFP_KERNEL);
	if (dev == NULL)
		return dev;

	ret = alloc_chrdev_region(&dev_no, dev->minor_no, 1, ZFORCE_IOCTLDEV_NAME);
	dev->major_no = MAJOR(dev_no);
	if (ret) {
		pr_alert("Failed to alloc_chrdev_region\n");
		goto err_malloc;
	}

	cdev_init(&dev->cdev, &hwcheck_chrdev_fops);
	dev->cdev.owner = THIS_MODULE;
	dev->cdev.ops = &hwcheck_chrdev_fops;
	ret = cdev_add(&dev->cdev, dev_no, 1);
	if (ret) {
		pr_alert("Failed to cdev_add\n");
		goto err_cdev_add;
	}
	/*autocreate device inode file*/
	dev->hwcheck_class = class_create(THIS_MODULE, ZFORCE_IOCTLDEV_NAME);
	if (IS_ERR(dev->hwcheck_class)) {
		pr_alert("Failed to class_create\n");
		goto err_class_crt;
	}
	device_create(dev->hwcheck_class, NULL,
		MKDEV(dev->major_no, dev->minor_no), 0, ZFORCE_IOCTLDEV_NAME);

	return dev;

err_class_crt:
	cdev_del(&dev->cdev);
err_cdev_add:
	unregister_chrdev_region(MKDEV(dev->major_no, dev->minor_no), 1);
err_malloc:
	kfree(dev);
	dev = NULL;
	return dev;
}

/*
 * Zforce HW check interface functions
 * */
int zf_hwcheck_init(void *zforce_dev)
{
	/* Init the HW check device and register the driver functions */
	hwcheckdev = hwcheck_chrdev_init();

	if (hwcheckdev && zforce_dev)
		hwcheckdev->zforce_dev = zforce_dev;
	else
		return -ENODEV;

	return 0;
}

void zf_hwcheck_deinit(void)
{
	if (hwcheckdev) {
		hwcheck_chrdev_deinit(hwcheckdev);
		hwcheckdev = NULL;
	}
}

void zf_hwcheck_handle_response(char *data, int len)
{

	if (hwcheckdev && (len > 0)) {
		int cpylen = (len > ZFORCE_BUF_LEN) ? ZFORCE_BUF_LEN : len;
		hwcheckdev->buf.len = cpylen;
		memcpy(hwcheckdev->buf.buf, data, cpylen);
	}
	complete(&wait_cmd_complete);
}
