
/*
 * CAAM Secure Memory/Keywrap API Definitions
 * Copyright (C) 2008-2013 Freescale Semiconductor, Inc.
 */

#ifndef SM_H
#define SM_H

#include <uapi/linux/sm.h>

/* Storage access permissions */
#define SM_PERM_READ 0x01
#define SM_PERM_WRITE 0x02
#define SM_PERM_BLOB 0x03

/*Defines*/
#define KEY32_BB_SIZE		80
#define KEY_SIZE		32 /* Size of key to be used */
#define SM_DATA_BANK		2 /* Data Page for wrap/unwrap */
#define KID_TABLE_LEN		28 /* 28 keys for loading */

/* KHA driver ERROR*/
/* KHA structures */
struct skey_prepare {
	u32  flags;
	u32  uid;	/* UID of key to install */
	u32  gid;	/* GID of key to install */
	u32  enc_keyid; /* Key ID used to encrypt the key in buffer */
	u32  in_kid;    /* Key ID of key to be prepared */
	u32  in_len;    /* Size of input key */
	u8  *in_buf;    /* Key data to prepare */
	u32  out_len;   /* Length of output buffer */
	u8  *out_buf;   /* Buffer for prepared key */
};

struct skey_load {
	u32  kid; /* Key id to install */
	u32  uid; /* UID of key to install */
	u32  gid; /* GID of key to install */
	u32  len; /* Length of encapsulated key */
	u8  *buf; /* Encapsulated key */
};

/* Release key structure */
struct skey_remove {
	u32  keyid;
	u32  uid;	/* UID of key to remove */
	u32  gid;	/* GID of key to remove */
};

/* Table for kid to slot conversion */
struct keyidmapper {
	u32	state;
	u32	kid;
	u32	slot;
};

/*sm_store test function*/
int caam_sm_test_case(struct platform_device *pdev);

/* Keystore maintenance functions */
void sm_init_keystore(struct device *dev);
u32 sm_detect_keystore_units(struct device *dev);
int sm_establish_keystore(struct device *dev, u32 unit);
void sm_release_keystore(struct device *dev, u32 unit);
void caam_sm_shutdown(struct platform_device *pdev);
int caam_sm_example_init(struct platform_device *pdev);

int caam_create_hash(u8 *bbk, u32 len, u8 *hash);

/* KHA API */
int caam_sm_ksm_relslot_key(u32 slot);
int caam_sm_ksm_readblob(u32 kid, u8 *smbkey);
int caam_sm_ksm_prepareblob(u8 *sm_key);
int caam_sm_ksm_unprepareblob(u8 *sm_blob);
int caam_sm_ksm_init(struct platform_device *pdev);
int caam_sm_ksm_prepare(struct skey_prepare *keyprepare);
int caam_sm_ksm_load(struct skey_load *skeyload);
int caam_sm_ksm_loadblob(u32 kid, u32 uid, u32 gid, u8 *blob);
u8 *caam_kha_readkey(u32 keyid);
int get_jr_device(struct device **ksdev);

/* Keystore accessor functions */
extern int sm_keystore_slot_alloc(struct device *dev, u32 unit, u32 size,
				  u32 *slot);
extern int sm_keystore_slot_dealloc(struct device *dev, u32 unit, u32 slot);
extern int sm_keystore_slot_load(struct device *dev, u32 unit, u32 slot,
				 const u8 *key_data, u32 key_length,
				 u32 kid, u32 uid, u32 gid);
extern int sm_keystore_slot_byid(struct device *dev, u32 unit, u32 kid,
				 u32 *slot);
extern int sm_keystore_slot_read(struct device *dev, u32 unit, u32 slot,
				 u32 key_length, u8 *key_data);
extern int sm_keystore_slot_encapsulate(struct device *dev, u32 unit,
					u32 inslot, u32 outslot, u16 secretlen,
					u8 *keymod, u16 keymodlen);
extern int sm_keystore_slot_decapsulate(struct device *dev, u32 unit,
					u32 inslot, u32 outslot, u16 secretlen,
					u8 *keymod, u16 keymodlen);
extern int sm_keystore_wrap_unwrap_data(struct device *dev, u32 unit,
					u8 *inbuf, u8 *outbuf, u16 secretlen,
					u8 *keymod, u16 keymodlen, u32 optype,
					u8 type);
extern int sm_keystore_get_permission(struct device *dev, u32 unit, u32 slot,
				u32 *uid, u32 *gid);

/* Data structure to hold per-slot information */
struct keystore_data_slot_info {
	u32	allocated;	/* Track slot assignments */
	u32	key_length;	/* Size of the key */
	u32	uid;		/* user id of slot owner */
	u32	gid;		/* Group id of slot owner */
	u32	kid;		/* Key id of key stored in slot */
};

/* Data structure to hold keystore information */
struct keystore_data {
	void	*base_address;	/* Base of the Secure Partition */
	u32	slot_count;	/* Number of slots in the keystore */
	struct keystore_data_slot_info *slot; /* Per-slot information */
};

/* store the detected attributes of a secure memory page */
struct sm_page_descriptor {
	u16 phys_pagenum;	/* may be discontiguous */
	u16 own_part;		/* Owning partition */
	void *page_phy_addr;	/* Page physical addr */
	void *pg_base;		/* Calculated virtual address */
	struct keystore_data *ksdata;
};

struct caam_drv_private_sm {
	struct device *parentdev;	/* this ends up as the controller */
	struct device *smringdev;	/* ring that owns this instance */
	spinlock_t kslock ____cacheline_aligned;

	/* Default parameters for geometry */
	u32 max_pages;		/* maximum pages this instance can support */
	u32 top_partition;	/* highest partition number in this instance */
	u32 top_page;		/* highest page number in this instance */
	u32 page_size;		/* page size */
	u32 slot_size;		/* selected size of each storage block */

	/* Partition/Page Allocation Map */
	u32 localpages;		/* Number of pages we can access */
	struct sm_page_descriptor *pagedesc;	/* Allocated per-page */

	/* Installed handlers for keystore access */
	int (*data_init)(struct device *dev, u32 unit);
	void (*data_cleanup)(struct device *dev, u32 unit);
	int (*slot_alloc)(struct device *dev, u32 unit, u32 size, u32 *slot);
	int (*slot_dealloc)(struct device *dev, u32 unit, u32 slot);
	void *(*slot_get_address)(struct device *dev, u32 unit, u32 handle);
	u32 (*slot_get_base)(struct device *dev, u32 unit, u32 handle);
	u32 (*slot_get_offset)(struct device *dev, u32 unit, u32 handle);
	u32 (*slot_get_slot_size)(struct device *dev, u32 unit, u32 handle);
};

#endif /* SM_H */
