/*
 * Copyright 2011 Freescale Semiconductor, Inc.
 * Copyright 2011 Linaro Ltd.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/init.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/reset-controller.h>
#include <linux/smp.h>
#include <linux/delay.h>
#include <asm/smp_plat.h>
#include "common.h"

#define SRC_SCR				0x000
#define SRC_SIMR			0x018
#define SRC_GPR1			0x020
#define BP_SRC_SCR_WARM_RESET_ENABLE	0
#define BP_SRC_SCR_SW_GPU_RST		1
#define BP_SRC_SCR_SW_VPU_RST		2
#define BP_SRC_SCR_SW_IPU1_RST		3
#define BP_SRC_SCR_SW_OPEN_VG_RST	4
#define BP_SRC_SCR_SW_IPU2_RST		12
#define BP_SRC_SCR_CORE1_RST		14
#define BP_SRC_SCR_CORE1_ENABLE		22
#define BP_SRC_SCR_VPU_RST		2
#define BP_SRC_SCR_IPU1_RST		3
#define BP_SRC_SCR_IPU2_RST		12
#define BP_SRC_SCR_WDOG			15
#define BP_SRC_SCR_WDOG_SHIFT		7
#define BP_SRC_SCR_WDOG_MASK		5
#define BP_SRC_SCR_WDOG_UNMASK		10
#define BP_SRC_SIMR_VPU_MASK		1
#define BP_SRC_SIMR_IPU1_MASK		2
#define BP_SRC_SIMR_IPU2_MASK		4

static void __iomem *src_base;
static DEFINE_SPINLOCK(scr_lock);

static const int sw_reset_bits[6] = {
	BP_SRC_SCR_SW_GPU_RST,
	BP_SRC_SCR_SW_VPU_RST,
	BP_SRC_SCR_SW_IPU1_RST,
	BP_SRC_SCR_SW_OPEN_VG_RST,
	BP_SRC_SCR_SW_IPU2_RST,
	BP_SRC_SCR_WDOG_SHIFT
};

static int imx_src_reset_module(struct reset_controller_dev *rcdev,
		unsigned long sw_reset_idx)
{
	unsigned long timeout;
	unsigned long flags;
	int bit;
	u32 val;

	if (!src_base)
		return -ENODEV;

	if (sw_reset_idx >= ARRAY_SIZE(sw_reset_bits))
		return -EINVAL;

	bit = 1 << sw_reset_bits[sw_reset_idx];

	spin_lock_irqsave(&scr_lock, flags);
	val = readl_relaxed(src_base + SRC_SCR);
	val |= bit;
	writel_relaxed(val, src_base + SRC_SCR);
	spin_unlock_irqrestore(&scr_lock, flags);

	timeout = jiffies + msecs_to_jiffies(1000);
	while (readl(src_base + SRC_SCR) & bit) {
		if (time_after(jiffies, timeout))
			return -ETIME;
		cpu_relax();
	}

	return 0;
}

static int wdog_reset_mask(struct reset_controller_dev *rcdev,
						unsigned long id)
{
	u32 val;
	int shift;
	unsigned long flags;

	if (id >= ARRAY_SIZE(sw_reset_bits))
		return -EINVAL;

	shift = sw_reset_bits[id];

	spin_lock_irqsave(&scr_lock, flags);
	val = readl_relaxed(src_base + SRC_SCR);
	val &= ~(BP_SRC_SCR_WDOG << shift);
	val |= (BP_SRC_SCR_WDOG_MASK << shift);
	writel_relaxed(val, src_base + SRC_SCR);
	spin_unlock_irqrestore(&scr_lock, flags);

	return 0;
}

static int wdog_reset_unmask(struct reset_controller_dev *rcdev,
						unsigned long id)
{
	u32 val;
	int shift;
	unsigned long flags;

	if (id >= ARRAY_SIZE(sw_reset_bits))
		return -EINVAL;

	shift = sw_reset_bits[id];

	spin_lock_irqsave(&scr_lock, flags);
	val = readl_relaxed(src_base + SRC_SCR);
	val &= ~(BP_SRC_SCR_WDOG << shift);
	val |= (BP_SRC_SCR_WDOG_UNMASK << shift);
	writel_relaxed(val, src_base + SRC_SCR);
	spin_unlock_irqrestore(&scr_lock, flags);

	return 0;
}

static struct reset_control_ops imx_src_ops = {
	.reset = imx_src_reset_module,
	.mask = wdog_reset_mask,
	.unmask = wdog_reset_unmask,
};

static struct reset_controller_dev imx_reset_controller = {
	.ops = &imx_src_ops,
	.nr_resets = ARRAY_SIZE(sw_reset_bits),
};


void imx_reset_vpu(void)
{
	u32 val;

	/* mask interrupt due to vpu passed reset */
	val = readl_relaxed(src_base + SRC_SIMR);
	val |= (1 << BP_SRC_SIMR_VPU_MASK);
	writel_relaxed(val, src_base + SRC_SIMR);

	val = readl_relaxed(src_base + SRC_SCR);
	val |= (1 << BP_SRC_SCR_VPU_RST);    /* reset vpu */
	writel_relaxed(val, src_base + SRC_SCR);
	while (readl_relaxed(src_base + SRC_SCR) &
		(1 << BP_SRC_SCR_VPU_RST))
		;
}

void imx_src_reset_ipu(int ipu_id)
{
	int timeout = 50000; /* usec */
	u32 val, reset;

	reset = (ipu_id == 0) ?
		(1 << BP_SRC_SCR_IPU1_RST) : (1 << BP_SRC_SCR_IPU2_RST);

	val = readl_relaxed(src_base + SRC_SCR);
	val |= reset;
	writel_relaxed(val, src_base + SRC_SCR);

	/* the IPU reset bit is self-clearing */
	while (readl_relaxed(src_base + SRC_SCR) & reset) {
		usleep_range(10, 11);
		timeout -= 10;
		if (timeout <= 0) {
			pr_err("ipu%d reset timeout!\n", ipu_id);
			break;
		}
	}
}

void imx_enable_cpu(int cpu, bool enable)
{
	u32 mask, val;

	cpu = cpu_logical_map(cpu);
	mask = 1 << (BP_SRC_SCR_CORE1_ENABLE + cpu - 1);
	spin_lock(&scr_lock);
	val = readl_relaxed(src_base + SRC_SCR);
	val = enable ? val | mask : val & ~mask;
	val |= 1 << (BP_SRC_SCR_CORE1_RST + cpu - 1);
	writel_relaxed(val, src_base + SRC_SCR);
	spin_unlock(&scr_lock);
}

void imx_set_cpu_jump(int cpu, void *jump_addr)
{
	cpu = cpu_logical_map(cpu);
	writel_relaxed(virt_to_phys(jump_addr),
		       src_base + SRC_GPR1 + cpu * 8);
}

u32 imx_get_cpu_arg(int cpu)
{
	cpu = cpu_logical_map(cpu);
	return readl_relaxed(src_base + SRC_GPR1 + cpu * 8 + 4);
}

void imx_set_cpu_arg(int cpu, u32 arg)
{
	cpu = cpu_logical_map(cpu);
	writel_relaxed(arg, src_base + SRC_GPR1 + cpu * 8 + 4);
}

void imx_disable_watchdog_mask(void)
{
	u32 val;
	unsigned long flags;

	if (!src_base)
		return;

	spin_lock_irqsave(&scr_lock, flags);

	/* To stay in defined state unmask watchdog signal */
	val = readl_relaxed(src_base + SRC_SCR);
	val &= ~(BP_SRC_SCR_WDOG << BP_SRC_SCR_WDOG_SHIFT);
	val |= (BP_SRC_SCR_WDOG_UNMASK << BP_SRC_SCR_WDOG_SHIFT);
	writel_relaxed(val, src_base + SRC_SCR);

	spin_unlock_irqrestore(&scr_lock, flags);
}

void __init imx_src_init(void)
{
	struct device_node *np;
	u32 val;

	np = of_find_compatible_node(NULL, NULL, "fsl,imx6q-src");
	src_base = of_iomap(np, 0);
	WARN_ON(!src_base);

	imx_reset_controller.of_node = np;
	reset_controller_register(&imx_reset_controller);

	/*
	 * force warm reset sources to generate cold reset
	 * for a more reliable restart
	 */
	spin_lock(&scr_lock);
	val = readl_relaxed(src_base + SRC_SCR);
	val &= ~(1 << BP_SRC_SCR_WARM_RESET_ENABLE);
	writel_relaxed(val, src_base + SRC_SCR);
	spin_unlock(&scr_lock);
}
